﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using Blocks.Core;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using Renderer2D.Core;
using EffectDefinitions;
using System.Collections.Generic;

namespace EffectCombiner.Primitives.Operations
{
    /// <summary>
    /// ブロックエレメントを削除するコマンドです。
    /// </summary>
    public class RemoveBlockOperation : OperationBase
    {
        /// <summary>
        /// 処理対象のブロックエレメントの情報です。
        /// </summary>
        private readonly BlockElementInfo[] blockElementInfos;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="blockManager">ブロックマネージャ</param>
        /// <param name="blockElement">ブロックエレメント</param>
        public RemoveBlockOperation(BlockManagerBase blockManager, EffectBlockElementBase blockElement)
            : this(blockManager, new EffectBlockElementBase[] { blockElement })
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="blockManager">ブロックマネージャ</param>
        /// <param name="blockElement">ブロックエレメントリスト</param>
        public RemoveBlockOperation(BlockManagerBase blockManager, IEnumerable<EffectBlockElementBase> blockElements)
            : base(OperationType.RemoveBlock, blockManager)
        {
            if (blockElements == null)
                throw new ArgumentNullException("blockElements");

            this.blockElementInfos = new BlockElementInfo[blockElements.Count()];

            int blockElementIndex = 0;

            foreach (var blockElement in blockElements)
            {
                BlockElementInfo info = new BlockElementInfo();

                info.InstanceIdentifier = blockElement.InstanceIdentifier;
                info.BlockDefinition = blockElement.BlockDefinition;
                info.Position = new Point(blockElement.Left, blockElement.Top);

                var constElement = blockElement as ConstantBlockElement;
                if (constElement != null) info.ConstValue = constElement.Values;

                var commentElement = blockElement as CommentBlockElement;
                if (commentElement != null)
                {
                    info.CommentText = commentElement.Text;
                    info.Size = new Size(blockElement.Width, blockElement.Height);
                }
                else
                {
                    info.Size = null;
                }

                // 入力接続情報を列挙
                {
                    // 入力接続情報を列挙する
                    var inputConnections = blockElement.WorkflowItem.InputPlugs  // 入力プラグリストについて
                        .Cast<EffectInputPlug>()                                 // IEnumerable<EffectInputPlug> にキャスト
                        .Where(p => p.BlockDefinitionPlug.Semantic == null)      // セマンティックが未指定のものを列挙して
                        .Select((p, i) => Tuple.Create(p, i))                    // この並びでインデックスを付ける
                        .Where(t => t.Item1.RemoteOutputPlug != null);           // 出力プラグに未接続のものは省く (インデックスにはカウントする)

                    // 列挙した入力接続情報を InpugPlugInfo にまとめる
                    info.InputConnections = inputConnections.Select(t => new InputPlugInfo(
                        t.Item2,
                        new PlugInfo(
                            t.Item1.RemoteOutputPlug.Index,
                            ((EffectWorkflowItem)t.Item1.RemoteWorkflowItem).BlockElement.InstanceIdentifier
                        )
                    )).ToArray();
                }

                // 出力接続情報を列挙
                {
                    // 出力接続情報を列挙する
                    var outputConnections = blockElement.WorkflowItem.OutputPlugs  // 出力プラグリストについて
                        .Cast<EffectOutputPlug>()                                  // IEnumerable<EffectOutputPlug> にキャスト
                        .Where(p => p.Plug.Semantic == null)                       // セマンティックが未指定のものを列挙して
                        .Select((p, i) => Tuple.Create(p, i))                      // この並びでインデックスを付ける
                        .Where(t => t.Item1.RemoteInputPlugs.Length > 0);          // 入力プラグに未接続のものは省く (インデックスにはカウントする)

                    // 列挙した出力接続情報を OutputPlugInfo にまとめる
                    info.OutputConnections = outputConnections.Select(t => new OutputPlugInfo(
                        t.Item2,
                        t.Item1.RemoteInputPlugs.Cast<EffectInputPlug>().Select(r => new PlugInfo(
                            r.Index,
                            r.BlockElement.InstanceIdentifier
                        )).ToArray())
                    ).ToArray();
                }

                this.blockElementInfos[blockElementIndex] = info;
                ++blockElementIndex;
            }
        }

        /// <summary>
        /// 編集操作をアンドゥします。
        /// </summary>
        public override void Rollback()
        {
            this.BlockManager.ClearSelections();

            var manager = this.BlockManager as CompositionBlockManager;

            foreach (var info in this.blockElementInfos)
            {
                var newBlockElement = (EffectBlockElementBase)BlockManager.CreateBlockElement(info.BlockDefinition);
                newBlockElement.SetNewInstanceIdentifier(info.InstanceIdentifier);

                var element = newBlockElement as ConstantBlockElement;
                if (element != null) element.Values = info.ConstValue;

                var commentElement = newBlockElement as CommentBlockElement;
                if (commentElement != null)
                {
                    commentElement.Text = info.CommentText;
                    commentElement.SetComputedSize(info.Size.Width, info.Size.Height);
                }

                // セマンティックを展開する
                if (manager != null)
                {
                    var expandedBlocks = SemanticHelper.ExpandSemantics(this.BlockManager, Generation.Globals.BlockDefinitions.Values, newBlockElement);

                    foreach (var eb in expandedBlocks)
                    {
                        this.BlockManager.AddBlock(eb);
                    }
                }

                newBlockElement.SetPosition(info.Position.X, info.Position.Y);

                // 復元したブロックエレメントについて、削除前の接続状態を復元する
                ConnectionUtility.RestoreInputConnections(newBlockElement, info.InputConnections, this.FindBlock);
                ConnectionUtility.RestoreOutputConnections(newBlockElement, info.OutputConnections, this.FindBlock);

                this.BlockManager.AddBlock(newBlockElement);
                newBlockElement.IsSelected = true;
            }
        }

        /// <summary>
        /// 編集操作を実行します。
        /// </summary>
        public override void Execute()
        {
            this.BlockManager.ClearSelections();

            foreach (var info in this.blockElementInfos)
            {
                var blockElement = FindBlock(info.InstanceIdentifier);
                this.BlockManager.RemoveBlock(blockElement);
            }
        }

        /// <summary>
        /// 処理対象のブロックエレメントの情報です。
        /// </summary>
        private class BlockElementInfo
        {
            /// <summary>
            /// 識別子を取得または設定します。
            /// </summary>
            public uint InstanceIdentifier { get; set; }

            /// <summary>
            /// 位置を取得または設定します。
            /// </summary>
            public IPoint Position { get; set; }

            /// <summary>
            /// サイズを取得または設定します。
            /// </summary>
            public ISize Size { get; set; }

            /// <summary>
            /// ブロック定義データを取得または設定します。
            /// </summary>
            public BlockDefinition BlockDefinition;

            /// <summary>
            /// constant 値を取得または設定します。
            /// </summary>
            public string[,] ConstValue { get; set; }

            /// <summary>
            /// コメントを取得または設定します。
            /// </summary>
            public string CommentText { get; set; }

            /// <summary>
            /// 入力プラグ情報を取得または設定します。
            /// </summary>
            public InputPlugInfo[] InputConnections { get; set; }

            /// <summary>
            /// 出力プラグ情報を取得または設定します。
            /// </summary>
            public OutputPlugInfo[] OutputConnections {get; set; }
        }
    }
}
