﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Linq;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using Workflow.Core;

namespace EffectCombiner.Primitives.Generation
{
    public class RegularEffectBlockElement : EffectBlockElementBase
    {
        public RegularEffectBlockElement(BlockDefinition blockDefinition)
            : base(
            blockDefinition,
            new BlockRenderInfo())
        {
            BuildWorkflowItemPlugs(blockDefinition);
        }

        private void BuildWorkflowItemPlugs(BlockDefinition blockDefinition)
        {
            foreach (var inputPlug in blockDefinition.InputPlugs)
            {
                var workflowInputPlug = new EffectInputPlug(inputPlug, this);
                WorkflowItem.AddInputPlug(workflowInputPlug);
            }

            foreach (var outputPlug in blockDefinition.OutputPlugs)
            {
                var workflowOutputPlug = new EffectOutputPlug(outputPlug, this);
                WorkflowItem.AddOutputPlug(workflowOutputPlug);
            }

            BlockRenderInfo.Initialize(blockDefinition);
        }

        public override bool IsSelected
        {
            get
            {
                return base.IsSelected;
            }
            set
            {
                BlockRenderInfo.IsSelected = value;
                base.IsSelected = value;
            }
        }

        public override bool IsInvertingSelection
        {
            get
            {
                return base.IsInvertingSelection;
            }
            set
            {
                BlockRenderInfo.IsInvertingSelection = value;
                base.IsInvertingSelection = value;
            }
        }

        public override bool IsDragging
        {
            get
            {
                return base.IsDragging;
            }
            set
            {
                BlockRenderInfo.IsDragging = value;
                base.IsDragging = value;
            }
        }

        protected override void OnBlockDefinitionChanged(BlockDefinition oldDefinition, BlockDefinition newDefinition)
        {
            // TODO: find a way to solve the shader generation locking problem

            //Globals.WorkspaceManager.BlockManager.BeginShaderCodeGenerationLock();

            //try
            //{
                TransplantBlockDefinition(oldDefinition, newDefinition);
            //}
            //finally
            //{
            //    Globals.WorkspaceManager.BlockManager.EndShaderCodeGenerationLock();
            //}

            base.OnBlockDefinitionChanged(oldDefinition, newDefinition);
        }

        private void TransplantBlockDefinition(BlockDefinition oldDefinition, BlockDefinition newDefinition)
        {
            if (oldDefinition is ConstantBlockDefinition && newDefinition is ConstantBlockDefinition)
                return;

            if (newDefinition.AreInputPlugsSameAs(oldDefinition) == false)
            {
                foreach (var inputPlug in WorkflowItem.InputPlugs)
                {
                    ConnectionManager.Disconnect(inputPlug);
                }
            }
            else
            {
                for (int i = 0; i < newDefinition.InputPlugs.Length; ++i)
                {
                    if (string.IsNullOrEmpty(newDefinition.InputPlugs[i].Semantic) &&
                        ((EffectWorkflowItem)WorkflowItem.InputPlugs[i].RemoteWorkflowItem).BlockElement is GhostBlockElement)
                    {
                        ConnectionManager.Disconnect(WorkflowItem.InputPlugs[i]);
                    }
                }
            }

            if (newDefinition.AreOutputPlugsSameAs(oldDefinition) == false)
            {
                foreach (var outputPlug in WorkflowItem.OutputPlugs)
                {
                    ConnectionManager.Disconnect(outputPlug);
                }
            }
            else
            {
                for (int i = 0; i < newDefinition.OutputPlugs.Length; ++i)
                {
                    if (oldDefinition.OutputPlugs[i].Semantic != newDefinition.OutputPlugs[i].Semantic)
                    {
                        ConnectionManager.Disconnect(WorkflowItem.OutputPlugs[i]);
                    }
                }
            }

            var inputConnections = WorkflowItem.InputPlugs
                .Select(x => x.RemoteOutputPlug)
                .ToArray();
            var outputConnections = WorkflowItem.OutputPlugs
                .SelectMany((x, i) => x.RemoteInputPlugs.Select(y => Tuple.Create(i, y)))
                .ToArray();

            foreach (var inputPlug in WorkflowItem.InputPlugs)
            {
                ConnectionManager.Disconnect(inputPlug);
                WorkflowItem.RemoveInputPlug(inputPlug);
            }
            foreach (var outputPlug in WorkflowItem.OutputPlugs)
            {
                ConnectionManager.Disconnect(outputPlug);
                WorkflowItem.RemoveOutputPlug(outputPlug);
            }

            BuildWorkflowItemPlugs(newDefinition);

            for (int i = 0; i < inputConnections.Length; ++i)
            {
                if (inputConnections[i] != null)
                {
                    ConnectionManager.Connect(inputConnections[i], WorkflowItem.InputPlugs[i]);
                }
                else if (string.IsNullOrEmpty(newDefinition.InputPlugs[i].Semantic) == false)
                {
                    var expandedBlocks = SemanticHelper.ExpandSemantics(BlockManager, Globals.BlockDefinitions.Values, this);
                    foreach (var eb in expandedBlocks)
                        BlockManager.AddBlock(eb);
                }
            }
            foreach (var connection in outputConnections)
            {
                ConnectionManager.Connect(WorkflowItem.OutputPlugs[connection.Item1], connection.Item2);
            }

            BlockRenderInfo.Initialize(newDefinition);
        }
    }
}
