﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using EffectDefinitions;

namespace EffectCombiner.Primitives.Generation
{
    /// <summary>
    /// constant ブロックエレメントです。
    /// </summary>
    public class ConstantBlockElement : RegularEffectBlockElement
    {
        /// <summary>
        /// constant 値です。
        /// </summary>
        private string[,] values;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ConstantBlockElement()
            : base(new ConstantBlockDefinition(new ShaderTypeDefinition("float")))
        {
            this.UpdateData();
        }

        /// <summary>
        /// constant 値を取得または設定します。
        /// </summary>
        public string[,] Values
        {
            get
            {
                return this.values;
            }

            set
            {
                string[,] oldValue = this.values;

                if (this.values == value)
                {
                    return;
                }

                this.values = value;
                this.UpdateParameterText();

                ConstantValuesChangedEventArgs args = new ConstantValuesChangedEventArgs(this, oldValue, value);
                this.OnValuesChanged(args);
            }
        }

        public void SetupConstantBlockElement(string constType, string constValues)
        {
            if (string.IsNullOrWhiteSpace(constType))
            {
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "constType"), "constType");
            }

            if (string.IsNullOrWhiteSpace(constValues))
            {
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "constValues"), "constValues");
            }

            var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(constType);
            var dimensionX = GlslTypingUtility.GetDimensionX(shaderType);
            var dimensionY = GlslTypingUtility.GetDimensionY(shaderType);

            this.UpdateBlockDefinition(new ConstantBlockDefinition(new ShaderTypeDefinition(constType)));
            this.UpdateData();

            var array = Core.CoreUtility.StringToMultiDimArray(constValues, dimensionX, dimensionY);

            // 旧バージョンでデフォルト値を保存したときに "0.0f" と f 付きで保存されることがあったため "0.0" に書き換える
            for (int i = 0; i < dimensionX; ++i)
            {
                for (int j = 0; j < dimensionY; ++j)
                {
                    if (array[i, j] == "0.0f")
                    {
                        array[i, j] = "0.0";
                    }
                }
            }

            Core.CoreUtility.ArrayCopy(this.Values, array);
            GlslTypingUtility.ConvertShaderTypeValue(array, shaderType);
            this.Values = array;
        }

        /// <summary>
        /// パラメータを更新します。
        /// </summary>
        public void UpdateData()
        {
            var shaderType = (ShaderTypeDefinition)BlockDefinition.CurrentFunctionDefinition.ReturnType;
            var glslShaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(shaderType.TypeString);

            var constantBlock = BlockDefinition as ConstantBlockDefinition;
            constantBlock.SubName = glslShaderType.ToString();

            var array = GlslTypingUtility.CreateDefaultValue(glslShaderType);
            Core.CoreUtility.ArrayCopy(array, this.values);
            GlslTypingUtility.ConvertShaderTypeValue(array, glslShaderType);
            this.Values = array;
        }

        /// <summary>
        /// パラメータのテキストを更新します。
        /// </summary>
        public void UpdateParameterText()
        {
            var shaderType = (ShaderTypeDefinition)BlockDefinition.CurrentFunctionDefinition.ReturnType;
            var glslShaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(shaderType.TypeString);

            var constantBlock = BlockDefinition as ConstantBlockDefinition;

            GlslTypingUtility.ConvertShaderTypeValue(this.Values, glslShaderType);

            if (BlockDefinition is ConstantBlockDefinition)
            {
                var constantDefinition = BlockDefinition as ConstantBlockDefinition;
                constantDefinition.ParameterText = GlslTypingUtility.GetShaderParamFromShaderType(glslShaderType, this.Values, null);
            }
        }

        /// <summary>
        /// constant 値が変更されたときに発生するイベントです。
        /// </summary>
        public event EventHandler<ConstantValuesChangedEventArgs> ValuesChanged;

        /// <summary>
        /// constant 値が変更されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnValuesChanged(ConstantValuesChangedEventArgs e)
        {
            this.ValuesChanged?.Invoke(this, e);
        }
    }
}
