﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectDefinitions;
using ShaderGenerator.GLSL;

namespace EffectCombiner.Primitives.Generation.AutoGen.Operators
{
    internal static class RangeConvertionCommon
    {
        internal static string CreateShaderCode(FunctionDefinition func, string customCode)
        {
            var typeString = ((ShaderTypeDefinition)func.Parameters[0].Type).TypeString;

            var sb = new StringBuilder();

            sb.AppendFormat("{0} {1}(", ((ShaderTypeDefinition)func.ReturnType).TypeString, func.Name);
            sb.Append(func.Parameters[0].GetPrototype());
            sb.AppendLine(")");

            sb.AppendLine("{");
            sb.Append("  return ");

            sb.Append(customCode);

            sb.AppendLine(";");
            sb.AppendLine("}");

            return sb.ToString();
        }
    }

    public class ZeroOneToMinusOneOneOperatorDefinitionSet : IAutoDefinitionSet
    {
        public ZeroOneToMinusOneOneOperatorDefinitionSet()
        {
            var paramNames = new[] { "n" };

            var supportedFunctionPrototypes = new[]
            {
                new[] { "int", "int" },
                new[] { "uint", "uint" },
                new[] { "float", "float" },
                new[] { "double", "double" },
                new[] { "ivec2", "ivec2" },
                new[] { "ivec3", "ivec3" },
                new[] { "ivec4", "ivec4" },
                new[] { "uvec2", "uvec2" },
                new[] { "uvec3", "uvec3" },
                new[] { "uvec4", "uvec4" },
                new[] { "vec2", "vec2" },
                new[] { "vec3", "vec3" },
                new[] { "vec4", "vec4" },
                new[] { "dvec2", "dvec2" },
                new[] { "dvec3", "dvec3" },
                new[] { "dvec4", "dvec4" },
            };

            FunctionDefinitions = OperatorDefinitionSetUtility.CreateFunctions(this, supportedFunctionPrototypes, paramNames);
            BlockDefinition = new PolymorphicBlockDefinition(BlockName, Guid, FunctionDefinitions);
            ShaderCodes = FunctionDefinitions.Select(CreateShaderCode).ToArray();
        }

        public string Guid
        {
            get { return "{006F1F69-EF73-4E39-8792-4593C2DE3B52}"; }
        }

        public string FunctionName
        {
            get { return "Zero_One_To_MinusOne_One"; }
        }

        public string BlockName
        {
            get { return "[0, 1] => [-1, 1]"; }
        }

        public PolymorphicBlockDefinition BlockDefinition { get; private set; }
        public FunctionDefinition[] FunctionDefinitions { get; private set; }
        public string[] ShaderCodes { get; private set; }

        private static string CreateShaderCode(FunctionDefinition func)
        {
            var typeString = ((ShaderTypeDefinition)func.Parameters[0].Type).TypeString;

            var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(typeString);
            var primitiveShaderType = GlslTypingUtility.GetPrimitiveShaderType(shaderType);

            var typedOne = GlslTypingUtility.FormatNumericValue(primitiveShaderType, 1);
            var typedTwo = GlslTypingUtility.FormatNumericValue(primitiveShaderType, 2);

            var customCode = string.Format("({0} * {1}) - {2}", func.Parameters[0].Name, typedTwo, typedOne);

            return RangeConvertionCommon.CreateShaderCode(func, customCode);
        }
    }

    public class MinusOneOneToZeroOneOperatorDefinitionSet : IAutoDefinitionSet
    {
        public MinusOneOneToZeroOneOperatorDefinitionSet()
        {
            var paramNames = new[] { "n" };

            var supportedFunctionPrototypes = new[]
            {
                new[] { "int", "int" },
                new[] { "uint", "uint" },
                new[] { "float", "float" },
                new[] { "double", "double" },
                new[] { "ivec2", "ivec2" },
                new[] { "ivec3", "ivec3" },
                new[] { "ivec4", "ivec4" },
                new[] { "uvec2", "uvec2" },
                new[] { "uvec3", "uvec3" },
                new[] { "uvec4", "uvec4" },
                new[] { "vec2", "vec2" },
                new[] { "vec3", "vec3" },
                new[] { "vec4", "vec4" },
                new[] { "dvec2", "dvec2" },
                new[] { "dvec3", "dvec3" },
                new[] { "dvec4", "dvec4" },
            };

            FunctionDefinitions = OperatorDefinitionSetUtility.CreateFunctions(this, supportedFunctionPrototypes, paramNames);
            BlockDefinition = new PolymorphicBlockDefinition(BlockName, Guid, FunctionDefinitions);
            ShaderCodes = FunctionDefinitions.Select(CreateShaderCode).ToArray();
        }

        public string Guid
        {
            get { return "{8F63C932-9F03-45B9-9A2D-B6593BA31D02}"; }
        }

        public string FunctionName
        {
            get { return "MinusOne_One_To_Zero_One"; }
        }

        public string BlockName
        {
            get { return "[-1, 1] => [0, 1]"; }
        }

        public PolymorphicBlockDefinition BlockDefinition { get; private set; }
        public FunctionDefinition[] FunctionDefinitions { get; private set; }
        public string[] ShaderCodes { get; private set; }

        private static string CreateShaderCode(FunctionDefinition func)
        {
            var typeString = ((ShaderTypeDefinition)func.Parameters[0].Type).TypeString;

            var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(typeString);
            var primitiveShaderType = GlslTypingUtility.GetPrimitiveShaderType(shaderType);

            var typedOne = GlslTypingUtility.FormatNumericValue(primitiveShaderType, 1);
            var typedTwo = GlslTypingUtility.FormatNumericValue(primitiveShaderType, 2);

            var customCode = string.Format("({0} + {1}) / {2}", func.Parameters[0].Name, typedOne, typedTwo);

            return RangeConvertionCommon.CreateShaderCode(func, customCode);
        }
    }
}
