﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace EffectCombiner.PreBuild
{
    class Program
    {
        private const bool FailOnError = false;

        static int Main(string[] args)
        {
            var errorCode = 0;

            if (args.Length == 0)
            {
                Console.WriteLine(Messages.INVALID_ARGUMENT_COUNT);
                errorCode = -1;
            }
            else
            {
                try
                {
                    if (Subversion.IsAvailable == false)
                    {
                        Console.WriteLine(Messages.SUBVERSION_NOT_AVAILABLE);
                        return 0; // silently fail if user do not have the versioning system installed
                    }

                    errorCode = Run(args[0].Trim().Trim('"'));
                }
                catch (Exception ex)
                {
                    Common.DumpException(ex);
                    errorCode = -2;
                }
            }

            return FailOnError ? errorCode : 0;
        }

        private static int Run(string solutionDir)
        {
            var revision = Subversion.GetCurrentRevision(Path.GetFullPath(Path.Combine(solutionDir, "..")));

            if (revision == null)
            {
                Console.WriteLine(Messages.CANNOT_GET_REVISION_INFO);
                return 1;
            }

            var combinerProjectDir = Path.GetFullPath(Path.Combine(solutionDir, "EffectCombiner.Editor"));
            var filename = Path.GetFullPath(Path.Combine(combinerProjectDir, "MainForm.g.cs"));

            Console.WriteLine(Messages.GENERATE_FILE, filename);

            using (var fs = new FileStream(filename, FileMode.Create, FileAccess.Write))
            {
                WriteToAutoGeneratedMainForm(fs, revision);
            }

            return 0;
        }

        private static void WriteToAutoGeneratedMainForm(Stream stream, string revision)
        {
            var writer = new StreamWriter(stream);

            BeginMainForm(writer);

            writer.WriteLine(@"        public static readonly string RepositoryRevision = ""{0}"";", revision);
            // add more info here

            EndMainForm(writer);
        }

        private static void BeginMainForm(StreamWriter writer)
        {
            writer.WriteLine(@"namespace EffectCombiner.Editor");
            writer.WriteLine(@"{");
            writer.WriteLine(@"    public partial class MainForm");
            writer.WriteLine(@"    {");
        }

        private static void EndMainForm(StreamWriter writer)
        {
            writer.WriteLine(@"    }");
            writer.WriteLine(@"}");

            writer.Flush();
        }
    }
}
