﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using EffectCombiner.Generator;
using ShaderGenerator.GLSL;

namespace EffectCombiner.GeneratorTests
{
    /// <summary>
    /// Summary description for UnitTest1
    /// </summary>
    [TestClass]
    public class GenerationTest
    {
        private static string basePath = "../../../../../TestData/UnitTests/ShadersGraphs/";

        private static string[] ProduceShaderCodes(string filename)
        {
            var settings = new ShaderGenSettings
            {
                GeneratedCodeDependencyMode = DependencyMode.Link,
            };
            var defPaths = new string[] { basePath };
            foreach (var path in defPaths)
            {
                settings.EffectDefinitionsPaths.Add(path,
                    SearchOption.AllDirectories, "*.glsl", "*.edml");
            }

            var shaderGenerator = new ShaderGen(settings);
            shaderGenerator.Initialize();

            var codes = shaderGenerator.ProduceShaderCodes(filename);
            return codes;
        }

        private void TestShader(string inputFilename, string reference)
        {
            var codes = ProduceShaderCodes(inputFilename);

            if (codes == null || codes.Length == 0)
            {
                throw new Exception("No code produced.");
            }
            else if (codes.Length == 1)
            {
                string expectedCode = string.Empty;
                try
                {
                    using (var reader = new StreamReader(reference))
                    {
                        expectedCode = reader.ReadToEnd();
                    }
                }
                catch
                {
                    throw new Exception(string.Format("Could not read reference file: {0}", reference));
                }
                if (codes[0] != expectedCode)
                {
                    throw new Exception(string.Format("Generated code should be:\r\n{0}\r\nbut is:\r\n{1}", expectedCode, codes[0]));
                }
            }
            else
            {
                throw new Exception(string.Format("{0} codes produced, instead of 1", codes.Length));
            }
        }

        [TestMethod]
        public void TestBlueOutputSemanticToSemantic()
        {
            var inputFilename = Path.GetFullPath(basePath + "blueOutputSemanticToSemanticTest.xml");
            var reference = Path.GetFullPath(basePath + "blueOutputSemanticToSemanticTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestBlueOutputSemanticToVec4()
        {
            var inputFilename = Path.GetFullPath(basePath + "blueOutputSemanticToVec4Test.xml");
            var reference = Path.GetFullPath(basePath + "blueOutputSemanticToVec4Test.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestBlueOutput()
        {
            var inputFilename = Path.GetFullPath(basePath + "blueOutputTest.xml");
            var reference = Path.GetFullPath(basePath + "blueOutputTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestBlueOutputVec4ToSemantic()
        {
            var inputFilename = Path.GetFullPath(basePath + "blueOutputVec4ToSemanticTest.xml");
            var reference = Path.GetFullPath(basePath + "blueOutputVec4ToSemanticTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestGreenOutputSemanticToVec4ToVec4()
        {
            var inputFilename = Path.GetFullPath(basePath + "greenOutputSemanticToVec4ToVec4Test.xml");
            var reference = Path.GetFullPath(basePath + "greenOutputSemanticToVec4ToVec4Test.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestGreenOutput()
        {
            var inputFilename = Path.GetFullPath(basePath + "greenOutputTest.xml");
            var reference = Path.GetFullPath(basePath + "greenOutputTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestGreenOutputVec4ToSemanticToVec4()
        {
            var inputFilename = Path.GetFullPath(basePath + "greenOutputVec4ToSemanticToVec4Test.xml");
            var reference = Path.GetFullPath(basePath + "greenOutputVec4ToSemanticToVec4Test.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestGreenOutputVec4ToVec4ToSemantic()
        {
            var inputFilename = Path.GetFullPath(basePath + "greenOutputVec4ToVec4ToSemanticTest.xml");
            var reference = Path.GetFullPath(basePath + "greenOutputVec4ToVec4ToSemanticTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestOrangeBGRABGRA()
        {
            var inputFilename = Path.GetFullPath(basePath + "orangeBGRA-BGRA.xml");
            var reference = Path.GetFullPath(basePath + "orangeBGRA-BGRA.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestOrangeBGRARGBA()
        {
            var inputFilename = Path.GetFullPath(basePath + "orangeBGRA-RGBA.xml");
            var reference = Path.GetFullPath(basePath + "orangeBGRA-RGBA.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestOrangeRGBABGRA()
        {
            var inputFilename = Path.GetFullPath(basePath + "orangeRGBA-BGRA.xml");
            var reference = Path.GetFullPath(basePath + "orangeRGBA-BGRA.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestOrangeRGBARGBA()
        {
            var inputFilename = Path.GetFullPath(basePath + "orangeRGBA-RGBA.xml");
            var reference = Path.GetFullPath(basePath + "orangeRGBA-RGBA.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestOrangeOutputBGRAToSemantic()
        {
            var inputFilename = Path.GetFullPath(basePath + "orangeOutputBGRAToSemanticTest.xml");
            var reference = Path.GetFullPath(basePath + "orangeOutputBGRAToSemanticTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestOrangeOutputRGBAToSemantic()
        {
            var inputFilename = Path.GetFullPath(basePath + "orangeOutputRGBAToSemanticTest.xml");
            var reference = Path.GetFullPath(basePath + "orangeOutputRGBAToSemanticTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestOutputOrangeAndWhiteConvoluted()
        {
            var inputFilename = Path.GetFullPath(basePath + "outputOrangeAndWhiteConvolutedTest.xml");
            var reference = Path.GetFullPath(basePath + "outputOrangeAndWhiteConvolutedTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestRedOutputRgbAndAToRgbAndA()
        {
            var inputFilename = Path.GetFullPath(basePath + "redOutputRgbAndAToRgbAndATest.xml");
            var reference = Path.GetFullPath(basePath + "redOutputRgbAndAToRgbAndATest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestRedOutputRGBAToRGBA()
        {
            var inputFilename = Path.GetFullPath(basePath + "redOutputRGBAToRGBATest.xml");
            var reference = Path.GetFullPath(basePath + "redOutputRGBAToRGBATest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestRedOutputSemanticToSemantic()
        {
            var inputFilename = Path.GetFullPath(basePath + "redOutputSemanticToSemanticTest.xml");
            var reference = Path.GetFullPath(basePath + "redOutputSemanticToSemanticTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestRedOutputSemanticToVec4()
        {
            var inputFilename = Path.GetFullPath(basePath + "redOutputSemanticToVec4Test.xml");
            var reference = Path.GetFullPath(basePath + "redOutputSemanticToVec4Test.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestRedOutput()
        {
            var inputFilename = Path.GetFullPath(basePath + "redOutputTest.xml");
            var reference = Path.GetFullPath(basePath + "redOutputTest.expected.frag");
            TestShader(inputFilename, reference);
        }

        [TestMethod]
        public void TestRedOutputVec4ToSemantic()
        {
            var inputFilename = Path.GetFullPath(basePath + "redOutputVec4ToSemanticTest.xml");
            var reference = Path.GetFullPath(basePath + "redOutputVec4ToSemanticTest.expected.frag");
            TestShader(inputFilename, reference);
        }
    }
}
