﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Primitives;
using ShaderGenerator.CafeCompiler;
using System.Text.RegularExpressions;

namespace EffectCombiner.Editor
{
    public partial class ShaderCodeForm : Form
    {
        public ShaderCodeForm()
        {
            InitializeComponent();

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    var title = Localization.Controls.FORM_SHADER_CODE_TITLE;

                    if (wentThroughCompilation)
                        title += " - " + compilationResult.ToMessage();

                    Text = title;
                });

            FormClosed += (ss, ee) => localizationSubscription.Dispose();
        }

        private readonly string shaderCode;
        private readonly IEnumerable<string> errors;
        private readonly CompilationResultCode compilationResult;
        private readonly bool wentThroughCompilation;

        public ShaderCodeForm(string shaderCode, IEnumerable<string> errors,
            CompilationResultCode compilationResult,
            bool wentThroughCompilation)
            : this()
        {
            this.shaderCode = shaderCode;
            this.errors = errors;
            this.compilationResult = compilationResult;
            this.wentThroughCompilation = wentThroughCompilation;

            txtCode.KeyDown += txtCode_KeyDown;
            txtCode.MouseDown += txtCode_MouseDown;
            txtCode.MouseMove += txtCode_MouseMove;

            txtLog.DoubleClick += txtLog_DoubleClick;
        }

        private static readonly Regex errorRegex = new Regex(@"^ERROR: (?<col>\d+):(?<row>\d+):", RegexOptions.Compiled);

        private void txtLog_DoubleClick(object sender, EventArgs e)
        {
            var lineNum = txtLog.GetLineFromCharIndex(txtLog.SelectionStart);
            var line = errors.ElementAt(lineNum);

            var m = errorRegex.Match(line);

            if (m.Success)
            {
                int row;
                int col;

                if (int.TryParse(m.Groups["row"].Value, out row) && int.TryParse(m.Groups["col"].Value, out col))
                {
                    row--; // compiler starts line at 1

                    var index = txtCode.GetFirstCharIndexFromLine(row);
                    var length = txtCode.GetFirstCharIndexFromLine(row + 1) - 2 - index;

                    index += col;
                    txtCode.SelectionStart = index;
                    txtCode.SelectionLength = length;

                    SetCaretPositionInfo();

                    txtCode.ScrollToCaret();
                    txtCode.Focus();
                }
            }

            Console.WriteLine(line);
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            BeginInvoke((Action)Initialize);
        }

        private void txtCode_KeyDown(object sender, KeyEventArgs e)
        {
            BeginInvoke((Action)SetCaretPositionInfo);
        }

        private void txtCode_MouseDown(object sender, MouseEventArgs e)
        {
            SetCaretPositionInfo();
        }

        private void txtCode_MouseMove(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
                SetCaretPositionInfo();
        }

        private void Initialize()
        {
            txtCode.Text = shaderCode;

            txtCode.SelectionStart = shaderCode.Length;
            txtCode.SelectionLength = 0;
            txtCode.ScrollToCaret();

            if (wentThroughCompilation)
            {
                txtCode.BackColor = compilationResult == CompilationResultCode.Success
                    ? VisualResourceSet.ShaderOKBackgroundColor
                    : VisualResourceSet.ShaderNotOKBackgroundColor;
            }

            if (errors != null && errors.Any())
            {
                txtLog.Text = string.Join(Environment.NewLine, errors);
                pnlRoot.Panel2Collapsed = false;
            }
            else
                pnlRoot.Panel2Collapsed = true;
        }

        private void SetCaretPositionInfo()
        {
            if (txtCode.SelectionLength == 0)
            {
                var position = txtCode.SelectionStart + txtCode.SelectionLength;

                var currentLine = txtCode.GetLineFromCharIndex(position);
                var temp = txtCode.GetFirstCharIndexOfCurrentLine();
                var currentColumn = position - temp;

                lblInfo.Text = string.Format(Localization.Controls.LINE_POSITION, currentLine + 1, currentColumn + 1);
            }
            else
            {
                var startPosition = txtCode.SelectionStart;
                var endPosition = txtCode.SelectionStart + txtCode.SelectionLength;

                var startLine = txtCode.GetLineFromCharIndex(startPosition);
                var endLine = txtCode.GetLineFromCharIndex(endPosition);

                var temp1 = txtCode.GetFirstCharIndexFromLine(startLine);
                var temp2 = txtCode.GetFirstCharIndexFromLine(endLine);

                var startColumn = startPosition - temp1;
                var endColumn = endPosition - temp2;

                lblInfo.Text = string.Format(Localization.Controls.LINE_POSITION_SELECT,
                    startLine + 1, startColumn + 1,
                    endLine + 1, endColumn + 1,
                    txtCode.SelectionLength);
            }
        }
    }
}
