﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Primitives;
using EffectDefinitions;
using System.IO;
using System.Text.RegularExpressions;
using ShaderGenerator.Core;

namespace EffectCombiner.Editor
{
    public partial class EventsReportForm : Form
    {
        public EventsReportForm()
        {
            InitializeComponent();

            lstEvents.SelectedIndexChanged += lstEvents_SelectedIndexChanged;
            lstEvents.MouseDoubleClick += lstEvents_MouseDoubleClick;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    Text = Localization.Controls.FORM_EVENT_REPORTS_TITLE;
                    btnClear.Text = Localization.Controls.FORM_EVENT_REPORTS_CLEAR_BUTTON;
                    colDescription.Text = Localization.Controls.FORM_EVENT_REPORTS_COLUMN_DESCRIPTION;
                    colCategory.Text = Localization.Controls.FORM_EVENT_REPORTS_COLUMN_CATEGORY;
                    colIssueDate.Text = Localization.Controls.FORM_EVENT_REPORTS_COLUMN_ISSUE_DATE;
                    colSource.Text = Localization.Controls.FORM_EVENT_REPORTS_COLUMN_SOURCE;
                    colComment.Text = Localization.Controls.FORM_EVENT_REPORTS_COLUMN_COMMENT;

                    EvaluateSelectedReport();
                });

            FormClosed += (ss, ee) => localizationSubscription.Dispose();
        }

        private readonly Regex sourceWithLocationRegex = new Regex(@"(?<file>.*):\((?<line>\d+),(?<pos>\d+)\)", RegexOptions.Compiled);

        private void lstEvents_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (lstEvents.SelectedItems.Count != 1)
                return;

            var item = lstEvents.SelectedItems[0] as EventReportListViewItem;
            if (item == null)
                return;

            if (string.IsNullOrWhiteSpace(item.Report.Source))
                return;

            var match = sourceWithLocationRegex.Match(item.Report.Source);
            if (match.Success == false)
                return;

            var uri = match.Groups["file"].Value;

            uint line;
            if (uint.TryParse(match.Groups["line"].Value, out line) == false)
                return;

            uint pos;
            if (uint.TryParse(match.Groups["pos"].Value, out pos) == false)
                return;

            string content;
            try
            {
                content = Globals.ContentRepository.GetContent(new Uri(uri), Encoding.UTF8);
            }
            catch
            {
                return;
            }

            var f = new SourceCodeViewForm();
            f.SetInfo(uri, content, (int)line, (int)pos, 0);
            f.Show();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            OptionsExtensions.ApplyRecordedFormWindowState(Globals.Options.EnvironmentSettings.EventsWindow, this);
        }

        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            if (e.CloseReason == CloseReason.UserClosing)
            {
                OptionsExtensions.RecordFormWindowState(Globals.Options.EnvironmentSettings.EventsWindow, this);
                Hide();
                e.Cancel = true;
            }
        }

        private void btnClearLog_Click(object sender, EventArgs e)
        {
            foreach (var item in lstEvents.Items.Cast<EventReportListViewItem>())
                item.Dispose();

            lstEvents.Items.Clear();
            pnlRootContainer.Panel2Collapsed = true;
        }

        private void lstEvents_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Delete)
            {
                foreach (var item in lstEvents.SelectedItems.Cast<EventReportListViewItem>().ToArray())
                {
                    lstEvents.Items.Remove(item);
                    item.Dispose();
                }
            }
        }

        private void lstEvents_SelectedIndexChanged(object sender, EventArgs e)
        {
            EvaluateSelectedReport();
        }

        private void EvaluateSelectedReport()
        {
            if (lstEvents.SelectedItems.Count == 1)
            {
                var item = lstEvents.SelectedItems[0] as EventReportListViewItem;
                if (item != null && item.Report.Exception != null)
                {
                    txtDetails.Text = Core.CoreUtility.DumpException(item.Report.Exception);
                    pnlRootContainer.Panel2Collapsed = false;
                    return;
                }
            }

            pnlRootContainer.Panel2Collapsed = true;
        }

        internal void Report(EventReport eventReport)
        {
            if (eventReport == null)
                return;

            lstEvents.Items.Add(new EventReportListViewItem(eventReport));
        }

        private void lstEvents_ColumnWidthChanging(object sender, ColumnWidthChangingEventArgs e)
        {
            if (e.ColumnIndex == 0)
                e.Cancel = true;
        }

        private void lstEvents_ColumnWidthChanged(object sender, ColumnWidthChangedEventArgs e)
        {
            lstEvents.ColumnWidthChanging -= lstEvents_ColumnWidthChanging;
            lstEvents.ColumnWidthChanged -= lstEvents_ColumnWidthChanged;

            if (e.ColumnIndex == 0)
                colType.Width = 20;

            lstEvents.ColumnWidthChanged += lstEvents_ColumnWidthChanged;
            lstEvents.ColumnWidthChanging += lstEvents_ColumnWidthChanging;
        }

        private void lstEvents_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            var order = lstEvents.Sorting == SortOrder.Ascending;
            lstEvents.ListViewItemSorter = new EventReportSorter(e.Column, order);
            lstEvents.Sorting = order ? SortOrder.Descending : SortOrder.Ascending;

        }

        private class EventReportSorter : System.Collections.IComparer
        {
            private readonly int column;
            private readonly bool order;

            public EventReportSorter(int column, bool order)
            {
                if (column < 0 || column > 5)
                    throw new ArgumentOutOfRangeException("column");

                this.column = column;
                this.order = order;
            }

            public int Compare(object x, object y)
            {
                var ctrlA = order ? x as EventReportListViewItem : y as EventReportListViewItem;
                var ctrlB = order ? y as EventReportListViewItem : x as EventReportListViewItem;

                if (ctrlA == null || ctrlB == null)
                    return 0;

                var a = ctrlA.Report;
                var b = ctrlB.Report;

                if (a == null || b == null)
                    return 0;

                switch (column)
                {
                    case 0:
                        return a.Level.CompareTo(b.Level);
                    case 1:
                        return string.Compare(a.Message, b.Message, StringComparison.InvariantCultureIgnoreCase);
                    case 2:
                        return a.Category.CompareTo(b.Category);
                    case 3:
                        return a.IssueDate.CompareTo(b.IssueDate);
                    case 4:
                        return string.Compare(a.Source, b.Source, StringComparison.InvariantCultureIgnoreCase);
                    case 5:
                        return string.Compare(a.Comment, b.Comment, StringComparison.InvariantCultureIgnoreCase);
                }

                return 0;
            }
        }
    }

    public class EventReportListViewItem : ListViewItem, IDisposable
    {
        public EventReport Report { get; private set; }

        public EventReportListViewItem(EventReport eventReport)
        {
            if (eventReport == null)
                throw new ArgumentNullException("eventReport");

            Report = eventReport;

            ImageIndex = (int)eventReport.Level;
            SubItems[0].Text = "";
            SubItems/*[1]*/.Add(eventReport.Message);
            SubItems/*[2]*/.Add(eventReport.Category.ToString());
            SubItems/*[3]*/.Add(eventReport.IssueDate.ToString(Localization.Messages.DATE_FORMAT));
            SubItems/*[4]*/.Add(eventReport.Source ?? string.Empty);
            SubItems/*[5]*/.Add(eventReport.Comment ?? string.Empty);

            if (Report.Exception != null)
                BackColor = Color.FromArgb(255, 230, 230);

            Globals.Localization.CultureChanged += LocalizationCultureChanged;
        }

        private void LocalizationCultureChanged(object sender, EventArgs e)
        {
            SubItems[1].Text = Report.Message;
            SubItems[3].Text = Report.IssueDate.ToString(Localization.Messages.DATE_FORMAT);
            SubItems[5].Text = Report.Comment;
        }

        public void Dispose()
        {
            Globals.Localization.CultureChanged -= LocalizationCultureChanged;
            Report.Dispose();
        }
    }
}
