﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Editor;
using EffectCombiner.Editor.Localization;
using EffectCombiner.Primitives;

namespace EffectCombiner.Primitives
{
    /// <summary>
    /// The event command.
    /// </summary>
    public enum EventCommand
    {
        /// <summary>
        /// The pre save.
        /// </summary>
        PreSave,

        /// <summary>
        /// The post save.
        /// </summary>
        PostSave,

        /// <summary>
        /// The pre open.
        /// </summary>
        PreOpen,
    }

    /// <summary>
    /// The event command executer.
    /// </summary>
    public static class EventCommandExecuter
    {
        /// <summary>
        /// The execute.
        /// </summary>
        /// <param name="eventType">
        /// The event Type.
        /// </param>
        /// <param name="targetList">
        /// The target List.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static int Execute(EventCommand eventType, List<string> targetList)
        {
            if (targetList.Count == 0)
            {
                return 0;
            }

            var commandPath = string.Empty;
            switch (eventType)
            {
                case EventCommand.PreSave:
                    commandPath = Globals.Options.PreSaveCommandPath;
                    break;
                case EventCommand.PostSave:
                    commandPath = Globals.Options.PostSaveCommandPath;
                    break;
                case EventCommand.PreOpen:
                    commandPath = Globals.Options.PreOpenCommandPath;
                    break;
                default:
                    Debug.Assert(false, "Undefined event command.");
                    break;
            }

            if (string.IsNullOrEmpty(commandPath))
            {
                return 0;
            }

            // 指定されたスクリプトが存在しない。
            if (!File.Exists(commandPath))
            {
                // 通知して中断
                MessageBox.Show(
                    string.Format(
                        Messages.ERRMSG_COMMAND_NOT_FOUND_WITH_NAME,
                        commandPath),
                    Messages.ERRMSG_COMMAND_CAPTION,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                Reporting.Report(
                    new EventReport(
                        () => Messages.ERRMSG_COMMAND_NOT_FOUND,
                        ReportLevel.Error,
                        ReportCategory.Application,
                        Messages.ERRMSG_EXECUTING_AT + commandPath,
                        () => Messages.ERRMSG_COMMAND_NOT_FOUND_DESC,
                        null));
                return 1;
            }

            var listFilePath = MakeTempList(targetList);

            var startInfo = new ProcessStartInfo
            {
                FileName = System.Environment.GetEnvironmentVariable("ComSpec"),
                CreateNoWindow = true,
                UseShellExecute = false,
                Arguments = string.Format(@"/c " + commandPath + " {0}", listFilePath),
                RedirectStandardError = true,
                RedirectStandardOutput = true
            };

            Process process;
            string errMsg;
            try
            {
                process = Process.Start(startInfo);
                errMsg = process.StandardError.ReadToEnd();
                process.WaitForExit();
            }
            catch (Exception e)
            {
                MessageBox.Show(
                    Messages.ERRMSG_COMMAND_EXCEPTION + ":\r\n" + e.ToString(),
                    Messages.ERRMSG_COMMAND_CAPTION,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);

                Reporting.Report(
                    new EventReport(
                        () => Messages.ERRMSG_COMMAND_EXCEPTION,
                        ReportLevel.Error,
                        ReportCategory.Application,
                        Messages.ERRMSG_EXECUTING_AT + commandPath,
                        () => Messages.ERRMSG_COMMAND_EXCEPTION_DESC,
                        null));

                return 1;
            }

            File.Delete(listFilePath);

            if (process.ExitCode != 0)
            {
                MessageBox.Show(
                    Messages.ERRMSG_COMMNAD_ERROR + ":\r\n" + errMsg,
                    Messages.ERRMSG_COMMAND_CAPTION,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);

                Reporting.Report(
                    new EventReport(
                        () => Messages.ERRMSG_COMMNAD_ERROR,
                        ReportLevel.Error,
                        ReportCategory.Application,
                        Messages.ERRMSG_EXECUTING_AT + commandPath,
                        () => Messages.ERRMSG_COMMAND_ERROR_DESC,
                        null));
            }

            return process.ExitCode;
        }

        /// <summary>
        /// The make temp list.
        /// </summary>
        /// <param name="targetList">
        /// The target list.
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        private static string MakeTempList(IEnumerable<string> targetList)
        {
            var tempFile = System.IO.Path.GetTempFileName();

            var sw = new System.IO.StreamWriter(
                tempFile,
                false,
                new System.Text.UTF8Encoding(false));

            foreach (var path in targetList)
            {
                sw.WriteLine(path);
            }

            sw.Close();

            return tempFile;
        }
    }
}
