﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using Nintendo.Foundation.IO;
using EffectCombiner.Editor.Log;

namespace EffectCombiner.Editor.CommandLine
{
    /// <summary>
    /// コマンドライン処理を管理するクラスです。
    /// </summary>
    public static class CommandLineApplication
    {
        /// <summary>
        /// 終了コードの定義です。
        /// </summary>
        public enum ExitCodeKind : int
        {
            NoErrors,
            WarningOnConvertProcess,
            InvalidArguments,
            ExceptionOccured,
        }

        /// <summary>
        /// 引数をチェックしてコマンドラインモードのON/OFFを取得します。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>コマンドラインモードのON/OFF</returns>
        public static bool ShouldEnableCommandLine(string[] args)
        {
            // 引数が無い場合はGUIモードとする
            if (!args.Any())
            {
                return false;
            }

            // コマンドライン引数を解析
            CommandLineParams parameters = ParseArgs(args, true);

            // 解析に失敗した場合、全ての引数が有効なファイルパスであればGUIモードにする
            // -? ヘルプ呼び出しのみの場合も解析失敗扱いになりCUIでヘルプが表示される
            if (parameters == null)
            {
                bool isValidFilePath = args.Aggregate(true, (current, arg) => current & File.Exists(arg));
                return !isValidFilePath;
            }

            // コンバートする場合はCUIモードとする
            if (parameters.InputFile != null)
            {
                return true;
            }

            // その他の場合はGUIモードとする
            return false;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>リザルトコードを返します。</returns>
        public static int Run(string[] args)
        {
            // コンソールの自動割り当てを無効化するかどうか取得
            bool disableAllocConsole = args.Any(x => x == "--disable-alloc-console");

            // コンソール画面を初期化
            ConsoleDisplay consoleDisplay = new ConsoleDisplay();
            consoleDisplay.Initialize(!disableAllocConsole);

            Logger.RegisterLogHandler(consoleDisplay);

            // コマンドライン引数を解析
            CommandLineParams parameters = ParseArgs(args, false);

            if (parameters == null)
            {
                // コンソール画面を破棄
                Logger.UnregisterLogHandler(consoleDisplay);
                consoleDisplay.Release();
                consoleDisplay = null;

                return (int)ExitCodeKind.InvalidArguments;
            }

            // サイレントモードのオプションを適用
            consoleDisplay.IsSilentMode = parameters.IsSilentMode;

            CommandLineShaderCodeGenerator shaderCodeGenerator = new CommandLineShaderCodeGenerator();

            // シェーダコードの生成処理を実行
            int resGenerate = shaderCodeGenerator.Generate(parameters);

            // コンソール画面を破棄
            Logger.UnregisterLogHandler(consoleDisplay);
            consoleDisplay.Release();
            consoleDisplay = null;

            return resGenerate;
        }

        /// <summary>
        /// コマンドライン引数をパースします。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <param name="silent">出力を無効にするか</param>
        /// <returns>解析結果を返します。</returns>
        public static CommandLineParams ParseArgs(string[] args, bool silent)
        {
            // バイナリ情報を取得
            System.Reflection.Assembly assembly = System.Reflection.Assembly.GetExecutingAssembly();
            System.Diagnostics.FileVersionInfo assemblyInfo = System.Diagnostics.FileVersionInfo.GetVersionInfo(assembly.Location);

            // パーサーを作成
            StringWriter helpWriter = new StringWriter();
            StringWriter errorWriter = new StringWriter();

            var settings = new CommandLineParserSettings()
            {
                HelpWriter = helpWriter.Write,
                ErrorWriter = errorWriter.Write,
                ApplicationName = assemblyInfo.ProductName,
                Version = string.Format("{0}.{1}.{2}", assemblyInfo.ProductMajorPart, assemblyInfo.ProductMinorPart, assemblyInfo.ProductBuildPart),
                ApplicationDescription = Properties.Resources.ConsoleDescApplicationDescription
            };

            var commandLineParser = new CommandLineParser(settings);

            bool resParse = false;
            CommandLineParams parameters = null;

            // パース処理を実行
            try
            {
                //// ParseArgs()の挙動
                //// パース成功     : params = input, return = true , throw = false
                //// パースエラー   : params = null , return = false, throw = true
                //// ヘルプ表示     : params = null , return = false, throw = false
                //// バージョン表示 : params = null , return = false, throw = false
                resParse = commandLineParser.ParseArgs(args, out parameters);
            }
            catch (Exception e)
            {
                // エラーメッセージを表示
                string errorText = errorWriter.ToString();

                if (!silent)
                {
                    if (errorText.Length != 0)
                    {
                        // パースエラーを表示
                        Logger.Log("Console", LogLevels.Error, errorText);
                    }
                    else if (!silent)
                    {
                        // DLLが読み込めないときなどのアプリケーションエラーを表示
                        Logger.Log("Console", LogLevels.Error, e.Message);
                    }
                }

                return null;
            }

            // ヘルプ情報またはバージョン情報を表示
            if (resParse == false)
            {
                if (!silent)
                {
                    string helpText = helpWriter.ToString();
                    Logger.Log("Console", LogLevels.Information, helpText);
                }

                return null;
            }

            return parameters;
        }
    }
}
