﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using EffectDefinitions;

namespace EffectCombiner.BlockAssistant
{
    public class ParseInfo
    {
        public FunctionDefinition Definition { get; private set; }
        public int Start { get; private set; }
        public int Length { get; private set; }

        public ParseInfo(FunctionDefinition definition, int start, int length)
        {
            if (definition == null)
                throw new ArgumentNullException("definition");
            if (start < 0)
                throw new ArgumentException(string.Format(Localization.Messages.EXCEPTION_MUST_BE_GREATER_THAN_OR_EQUAL_TO, "start", 0), "start");
            if (length <= 0)
                throw new ArgumentException(string.Format(Localization.Messages.EXCEPTION_MUST_BE_GREATER_THAN, "length", 0), "length");

            Definition = definition;
            Start = start;
            Length = length;
        }
    }

    public class PrototypeParser
    {
        private static readonly Regex prototypeRegex = new Regex(@"\s*(?<ret>[_a-zA-Z0-9\s]+)\s+(?<fname>[^\s]+)\s*\((?<params>[^\)]*)\)", RegexOptions.Compiled);
        private static readonly Regex identifierRegex = new Regex(Utility.ValidIdentifierRegEx, RegexOptions.Compiled);

        private static readonly string[] knownShaderTypes =
        {
            "void",
            "bool", "int", "uint", "float", "double",
            "bvec2", "bvec3", "bvec4",
            "ivec2", "ivec3", "ivec4",
            "uvec2", "uvec3", "uvec4",
            "vec2", "vec3", "vec4",
            "dvec2", "dvec3", "dvec4",
            "mat2x2", "mat3x2", "mat4x2",
            "mat2x3", "mat3x3", "mat4x3",
            "mat2x4", "mat3x4", "mat4x4",
        };

        public event EventHandler<ReportEventArgs> Report;

        protected virtual void OnReport(Report report)
        {
            var handler = Report;
            if (handler != null)
                handler(this, new ReportEventArgs(report));
        }

        public event EventHandler<ParsingEventArgs> ParsingDone;

        protected virtual void OnParsingDone(ParseInfo parseInfo)
        {
            var handler = ParsingDone;
            if (handler != null)
                handler(this, new ParsingEventArgs(parseInfo));
        }

        public ParseInfo Parse(string text)
        {
            if (string.IsNullOrWhiteSpace(text))
                return null;

            var m = prototypeRegex.Match(text);
            if (m.Success == false)
            {
                OnReport(new Report(() => Localization.Messages.PARSING_NO_MATCHING_CODE, ReportLevel.Warning, null));
                OnParsingDone(null);
                return null;
            }

            var returnGroup = m.Groups["ret"];

            var start = returnGroup.Index;
            var length = m.Length;

            var isValid = true;

            var returnType = CleanupSpaces(returnGroup.Value);
            if (knownShaderTypes.Contains(returnType) == false)
            {
                OnReport(new Report(() => string.Format(Localization.Messages.PARSING_UNKNOWN_RETURN_TYPE, returnType), ReportLevel.Warning, null));
                isValid = false;
            }

            var funcName = m.Groups["fname"].Value.Trim();
            var rawParams = m.Groups["params"].Value.Trim();

            if (identifierRegex.IsMatch(funcName) == false)
            {
                OnReport(new Report(() => string.Format(Localization.Messages.PARSING_FUNCTION_NAME_INVALID, funcName), ReportLevel.Error, null));
                isValid = false;
            }

            var parameters = new List<ParameterDefinition>();
            if (string.IsNullOrWhiteSpace(rawParams) == false)
            {
                var i = -1;
                foreach (var rawParam in rawParams.Split(',').Select(s => s.Trim()))
                {
                    i++;
                    var localIndex = i;

                    if (string.IsNullOrWhiteSpace(rawParam))
                    {
                        OnReport(new Report(() => string.Format(Localization.Messages.PARSING_PARAMETER_N_IS_MISSING, localIndex + 1), ReportLevel.Error, null));
                        isValid = false;
                        continue;
                    }

                    var splittedRawParam = rawParam.Split(new[] { ' ', '\t', '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);

                    var hasDirection = false;
                    var argDirection = ShaderTyping.ParameterDirection.Input;
                    if (string.Equals(splittedRawParam[0], "in", StringComparison.OrdinalIgnoreCase))
                        hasDirection = true;
                    else if (string.Equals(splittedRawParam[0], "out", StringComparison.OrdinalIgnoreCase))
                    {
                        argDirection = ShaderTyping.ParameterDirection.Output;
                        hasDirection = true;
                    }
                    else if (string.Equals(splittedRawParam[0], "inout", StringComparison.OrdinalIgnoreCase))
                    {
                        argDirection = ShaderTyping.ParameterDirection.Reference;
                        hasDirection = true;
                    }

                    string argType = null;
                    string argName = null;

                    if (hasDirection)
                    {
                        if (splittedRawParam.Length == 2)
                            argType = splittedRawParam[1];
                        else if (splittedRawParam.Length >= 3)
                        {
                            argType = string.Join(" ", splittedRawParam.Skip(1).Take(splittedRawParam.Length - 2));
                            argName = splittedRawParam.Last();
                        }
                    }
                    else
                    {
                        argType = splittedRawParam[0];
                        if (splittedRawParam.Length == 2)
                            argName = splittedRawParam[1];
                        else if (splittedRawParam.Length >= 3)
                        {
                            argType = string.Join(" ", splittedRawParam.Take(splittedRawParam.Length - 1));
                            argName = splittedRawParam.Last();
                        }
                    }

                    if (argType == null)
                    {
                        OnReport(new Report(() => string.Format(Localization.Messages.PARSING_ARGUMENT_N_TYPE_MISSING, localIndex + 1), ReportLevel.Error, null));
                        isValid = false;
                    }
                    else if (knownShaderTypes.Contains(argType) == false)
                        OnReport(new Report(() => string.Format(Localization.Messages.PARSING_UNKNOWN_ARGUMENT_N_TYPE, argType, localIndex + 1), ReportLevel.Warning, null));

                    if (string.Equals(argType, "void", StringComparison.OrdinalIgnoreCase))
                    {
                        OnReport(new Report(() => string.Format(Localization.Messages.PARSING_ARGUMENT_N_TYPE_INVALID, argType, localIndex + 1), ReportLevel.Error, null));
                        isValid = false;
                    }

                    if (argName == null)
                    {
                        OnReport(new Report(() => string.Format(Localization.Messages.PARSING_ARGUMENT_N_NAME_MISSING, localIndex + 1), ReportLevel.Error, null));
                        isValid = false;
                    }
                    else if (identifierRegex.IsMatch(argName) == false)
                    {
                        OnReport(new Report(() => string.Format(Localization.Messages.PARSING_ARGUMENT_N_NAME_INVALID, argName, localIndex + 1), ReportLevel.Error, null));
                        isValid = false;
                    }

                    if (isValid == false)
                        continue; // useless to add parameter if the definition is invalid

                    parameters.Add(new ParameterDefinition(
                        null,
                        argName,
                        new ShaderTypeDefinition(argType),
                        argDirection,
                        null,
                        null));
                }
            }

            if (isValid == false)
            {
                OnParsingDone(null);
                return null;
            }

            var funcDef = new FunctionDefinition(
                null,
                funcName,
                funcName,
                new ShaderTypeDefinition(returnType),
                null,
                null,
                null,
                parameters.ToArray());

            var parseInfo = new ParseInfo(funcDef, start, length);

            OnParsingDone(parseInfo);

            return parseInfo;
        }

        private static string CleanupSpaces(string input)
        {
            return string.Join(" ", input.Split(new[] { ' ', '\t', '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries));
        }
    }
}
