﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DoOptionDialog OptionDialogProc InitOptionDialog UpdateOptionDialog
// UpdatePreviewBitmap DoLighting SaveSettings LoadSettings
// DoAboutDialog

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
#define MOVE_LIGHT_ON_CLICK // クリックした位置にライトを直接移動するなら定義します。

//=============================================================================
// include
//=============================================================================
#include "NintendoNormalMapFilter.h"
#include "NintendoNormalMapFilterUI.h"
#include "../NintendoFtx/NpsVersion.h"
#include "../NintendoFtx/NpsPreview.h"
#include "Resource.h"
#include <commctrl.h> // for slider & tooltip

using namespace nn::gfx::tool::nps;

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// filter type
static const int s_FilterTypeIds[RParameters::FilterType_Count] =
{
    IDC_FILTER_TYPE_4,
    IDC_FILTER_TYPE_3X3,
    IDC_FILTER_TYPE_5X5,
    IDC_FILTER_TYPE_7X7,
    IDC_FILTER_TYPE_9X9,
    IDC_FILTER_TYPE_13X13,
    IDC_FILTER_TYPE_17X17,
    IDC_FILTER_TYPE_21X21,
    IDC_FILTER_TYPE_25X25,
};

//-----------------------------------------------------------------------------
// settings file
#define SETTINGS_FILE_FILTER "Settings files (*.xml)\0*.xml\0" \
                             "All files (*.*)\0*.*\0\0"
#define SETTINGS_FILE_EXT   "xml"

static const char* const s_SettingsFilterTypeValues[RParameters::FilterType_Count] =
{
    "4",
    "3x3",
    "5x5",
    "7x7",
    "9x9",
    "13x13",
    "17x17",
    "21x21",
    "25x25",
};

//=============================================================================
// variables
//=============================================================================

//-----------------------------------------------------------------------------
// preview
static RPreview s_Preview; // プレビュー表示です。

static RImage s_OtherImage; // プレビューで合成する他の法線マップの画像データです。

static uint8_t* s_pLightingPixels = nullptr; // ライティング結果のピクセルデータです。
static float s_LightVec[3]; // ライトベクトルです。
static bool s_IsLightMoving = false; // マウスでライト移動中なら true です。
static float s_MoveStartLightVec[3] = { 0.0f, 0.0f, 0.0f }; // 移動開始時のライトベクトルです。

//-----------------------------------------------------------------------------
//! @brief ライティング結果のピクセルデータを更新します。
//!
//! @param[in] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoLighting(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // パラメーターを設定します。
    const float DIFFUSE   = 200.0f;
    const float SPECULAR  = 150.0f;
    const float SHININESS =  30.0f;

    uint8_t* pDst = s_pLightingPixels;
    const uint8_t* pSrc = globals->m_pDstPixels;

    const int curW = globals->m_ImageW;
    const int curH = globals->m_ImageH;
    const int fullW = curW;

    const float* litVec = s_LightVec;
    float halfVec[3] =
    {
        litVec[0], litVec[1], litVec[2] + 1.0f,
    };
    RNormalizeNormal(halfVec, false);

    //-----------------------------------------------------------------------------
    // 更新領域を計算します。
    int ofsX = 0;
    int ofsY = 0;
    int upW = curW;
    int upH = curH;
    int stepPix = 1;

    if (s_Preview.m_Zoom >= 1)
    {
        ofsX = s_Preview.m_Ix / s_Preview.m_Zoom;
        ofsY = s_Preview.m_Iy / s_Preview.m_Zoom;
        upW = (s_Preview.GetItemW() + s_Preview.m_Zoom - 1) / s_Preview.m_Zoom;
        upH = (s_Preview.GetItemH() + s_Preview.m_Zoom - 1) / s_Preview.m_Zoom;
        if (s_Preview.m_Zoom >= 2)
        {
            if (ofsX + upW + 1 < curW) ++upW;
            if (ofsY + upH + 1 < curH) ++upH;
        }
    }
    else
    {
        stepPix = 1 << (1 - s_Preview.m_Zoom);
        ofsX = s_Preview.m_Ix * stepPix;
        ofsY = s_Preview.m_Iy * stepPix;
        upW = RMin(s_Preview.GetItemW(), curW) * stepPix;
        upH = RMin(s_Preview.GetItemH(), curH) * stepPix;
    }
    if (ofsX + upW > curW) upW = curW - ofsX;
    if (ofsY + upH > curH) upH = curH - ofsY;

    const int startOfs = (ofsX + ofsY * fullW) * R_RGBA_BYTES;
    pSrc += startOfs;
    pDst += startOfs;
    const int colOfs = stepPix * R_RGBA_BYTES;
    const int lineOfs = colOfs * fullW;

    //-----------------------------------------------------------------------------
    // ライティング計算をします。
    RTimeMeasure tm1;
    const uint8_t* pSrcLine = pSrc;
    uint8_t* pDstLine = pDst;
    for (int iy = 0; iy < upH; iy += stepPix)
    {
        pSrc = pSrcLine;
        pDst = pDstLine;
        for (int ix = 0; ix < upW; ix += stepPix)
        {
            float n[3];
            RGetNormalFromColor(n, pSrc);
            const float dotLN = RMax(litVec[0] * n[0] + litVec[1] * n[1] + litVec[2] * n[2], 0.0f);
            int color = static_cast<int>(dotLN * DIFFUSE);
            if (dotLN > 0.0f)
            {
                const float dotHN = RMax(halfVec[0] * n[0] + halfVec[1] * n[1] + halfVec[2] * n[2], 0.0f);
                color += static_cast<int>(powf(dotHN, SHININESS) * SPECULAR);
            }
            const uint8_t colorU8 = static_cast<uint8_t>(RClampValue(0x00, 0xff, color));
            if (stepPix == 1)
            {
                pDst[0] = pDst[1] = pDst[2] = colorU8;
            }
            else // stepPix x stepPix の領域にコピーします。
            {
                int copyOfs = 0;
                for (int by = 0; by < stepPix; ++by)
                {
                    if (iy + by >= upH)
                    {
                        break;
                    }
                    for (int bx = 0; bx < stepPix; ++bx)
                    {
                        if (ix + bx < upW)
                        {
                            pDst[copyOfs] = pDst[copyOfs + 1] = pDst[copyOfs + 2] = colorU8;
                        }
                        copyOfs += R_RGBA_BYTES;
                    }
                    copyOfs += (fullW - stepPix) * R_RGBA_BYTES;
                }
            }
            pSrc += colOfs;
            pDst += colOfs;
        }
        pSrcLine += lineOfs;
        pDstLine += lineOfs;
    }
    //RNoteTrace("lighting: %6.2f: %6.2f %6.2f %6.2f", tm1.GetMilliSec(), litVec[0], litVec[1], litVec[2]);
    //RNoteTrace("lighting: %6.2f: %4d, %4d: %4d x %4d", tm1.GetMilliSec(), ofsX, ofsY, upW, upH);
}

//-----------------------------------------------------------------------------
//! @brief プレビュー画像を更新します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] lightingOnly ライティング結果のみ強制的に更新するなら true を指定します。
//-----------------------------------------------------------------------------
static void UpdatePreviewBitmap(GPtr globals, HWND hDlg, const bool lightingOnly)
{
    //-----------------------------------------------------------------------------
    // プレビューが無効なら何もしません。
    s_Preview.m_UpdateTimerCount = 0;
    if (!globals->m_DisplaysPreview)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // フィルター後のピクセルデータを更新します。
    bool dstChanged = false;
    if (!lightingOnly && globals->m_PreviewParams != *gParams)
    {
        //RNoteTrace("update preview bitmap: %d", globals->m_PreviewParams != *gParams);
        bool success = true;

        //-----------------------------------------------------------------------------
        // 合成する他の法線マップをリードします。
        if (gParams->m_Operation == RParameters::Operation_CombineFile)
        {
            if (s_OtherImage.GetFilePath() != gParams->m_OtherPath ||
                s_OtherImage.GetImagePtr() == nullptr)
            {
                success = s_OtherImage.ReadFile(gParams->m_OtherPath);
                //RNoteTrace("read other: %d: %s", success, gParams->m_OtherPath);
                if (!success)
                {
                    const size_t pixelCount = static_cast<size_t>(globals->m_ImageW) * globals->m_ImageH;
                    memset(globals->m_pDstPixels, 0x00, pixelCount * R_RGBA_BYTES);
                }
            }
        }

        //-----------------------------------------------------------------------------
        // フィルター処理します。
        if (success)
        {
            sPSUIHooks->SetCursor(kPICursorWatch);
            DoFilter(globals, s_OtherImage);
            sPSUIHooks->SetCursor(kPICursorArrow);
        }

        globals->m_IsPreviewFiltered = success;
        if (success)
        {
            globals->m_PreviewParams = *gParams;
            dstChanged = true;
        }
    }

    //-----------------------------------------------------------------------------
    // ライティング結果のピクセルデータを更新します。
    if (globals->m_LightingPreview && (dstChanged || lightingOnly))
    {
        DoLighting(globals);
    }

    //-----------------------------------------------------------------------------
    // プレビュー画像を更新します。
    if (dstChanged || lightingOnly)
    {
        //RTimeMeasure tm1;
        s_Preview.UpdateWinBmp(hDlg,
            (globals->m_LightingPreview) ? s_pLightingPixels : globals->m_pDstPixels);
        //RNoteTrace("up win bmp: %6.2f", tm1.GetMilliSec());
    }
}

//-----------------------------------------------------------------------------
//! @brief ライトベクトルを変更します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void ChangeLightVec(
    GPtr globals,
    HWND hDlg,
    const POINTS& pos,
    const POINTS& startPos
)
{
    float* v = s_LightVec;

    const float moveScale = RMin(s_Preview.GetItemW(), s_Preview.GetItemH()) * 0.5f;
    #ifdef MOVE_LIGHT_ON_CLICK
    v[0] =  (pos.x - (s_Preview.m_ItemRect.left + s_Preview.m_ItemRect.right ) / 2) / moveScale;
    v[1] = -(pos.y - (s_Preview.m_ItemRect.top  + s_Preview.m_ItemRect.bottom) / 2) / moveScale;
    R_UNUSED_VARIABLE(startPos);
    #else
    float* sv = s_MoveStartLightVec;
    v[0] = sv[0] + (pos.x - startPos.x) / moveScale;
    v[1] = sv[1] - (pos.y - startPos.y) / moveScale;
    #endif
    const float xxyy = v[0] * v[0] + v[1] * v[1];
    if (xxyy <= 1.0f)
    {
        v[2] = sqrtf(1.0f - xxyy);
    }
    else
    {
        const float scale = 1.0f / sqrtf(xxyy);
        v[0] *= scale;
        v[1] *= scale;
        v[2] = 0.0f;
    }

    UpdatePreviewBitmap(globals, hDlg, true);
}

//-----------------------------------------------------------------------------
//! @brief ライトベクトルを Z 軸を中心に回転します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] angle 角度（単位は度数）です。
//-----------------------------------------------------------------------------
void RotatePreviewLight(GPtr globals, HWND hDlg, const float angle)
{
    float* v = s_LightVec;

    const float angRad = static_cast<float>(angle * R_M_DEG_TO_RAD);
    const float sinR = sinf(angRad);
    const float cosR = cosf(angRad);
    const float x = v[0];
    const float y = v[1];
    v[0] = cosR * x - sinR * y;
    v[1] = sinR * x + cosR * y;
    RNormalizeNormal(v, true);

    const int updateTimerCountBak = s_Preview.m_UpdateTimerCount;
    UpdatePreviewBitmap(globals, hDlg, true);
    s_Preview.m_UpdateTimerCount = updateTimerCountBak;
}

//-----------------------------------------------------------------------------
//! @brief スライダーのつまみの位置を設定します。
//!        値がスライダーの範囲外なら範囲を拡張します。
//!
//! @param[in] slider スライダーのウィンドウハンドルです。
//! @param[in] pos つまみの位置です。
//-----------------------------------------------------------------------------
static void SetSliderPosAndRange(HWND slider, const int pos)
{
    if (pos < static_cast<int>(SendMessage(slider, TBM_GETRANGEMIN, 0, 0)))
    {
        SendMessage(slider, TBM_SETRANGEMIN, FALSE, pos);
    }
    if (pos > static_cast<int>(SendMessage(slider, TBM_GETRANGEMAX, 0, 0)))
    {
        SendMessage(slider, TBM_SETRANGEMAX, FALSE, pos);
    }
    SendMessage(slider, TBM_SETPOS, TRUE, pos);
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログのエディットテキストに値を設定します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void SetOptionEditText(GPtr globals, HWND hDlg)
{
    SetDlgItemInt(hDlg, IDC_HEIGHT_SCALE , gParams->m_HeightScale , FALSE);
    SetDlgItemInt(hDlg, IDC_SLOPE_SCALE  , gParams->m_SlopeScale  , FALSE);
    SetDlgItemInt(hDlg, IDC_COMBINE_SCALE, gParams->m_CombineScale, FALSE);
    SetWindowText(GetDlgItem(hDlg, IDC_OTHER_NORMAL_MAP), gParams->m_OtherPath);
    for (int layerIdx = 1; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
    {
        SetDlgItemInt(hDlg, IDC_LAYER_SCALE1 + (layerIdx - 1),
            gParams->m_CombineLayerScales[layerIdx], FALSE);
    }
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログのエディットテキストから値を取得します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//!
//! @return すべての値が有効なら true を返します。
//-----------------------------------------------------------------------------
static bool GetOptionEditText(GPtr globals, HWND hDlg)
{
    const int otherPathLen = GetWindowText(GetDlgItem(hDlg, IDC_OTHER_NORMAL_MAP),
        gParams->m_OtherPath, sizeof(gParams->m_OtherPath));
    if (gParams->m_Operation == RParameters::Operation_CombineFile)
    {
        if (otherPathLen == 0)
        {
            RShowError(globals, "Other Normal Map is empty");
            return false;
        }

        const std::string ext = RGetExtensionFromFilePath(gParams->m_OtherPath);
        if (ext != "psd" && ext != "tga")
        {
            RShowError(globals, "Other Normal Map is wrong type");
            return false;
        }
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログを更新します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void UpdateOptionDialog(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // common
    for (int iOpe = 0; iOpe < RParameters::Operation_Count; ++iOpe)
    {
        if (gParams->m_Operation == iOpe)
        {
            SendMessage(GetDlgItem(hDlg, IDC_OPERATION), CB_SETCURSEL, iOpe, 0L);
            break;
        }
    }

    EnableWindow(GetDlgItem(hDlg, IDC_POSITIVE_Z),
        (gParams->m_Operation != RParameters::Operation_ConvertFromHeightMap));
    CheckDlgButton(hDlg, IDC_POSITIVE_Z,
        (gParams->m_PositiveZ) ? BST_CHECKED : BST_UNCHECKED);

    //-----------------------------------------------------------------------------
    // convert from height map
    const bool heigtMapEnable = (gParams->m_Operation == RParameters::Operation_ConvertFromHeightMap);
    EnableWindow(GetDlgItem(hDlg, IDC_HEIGT_MAP_GRP      ), heigtMapEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT_SCALE_LBL   ), heigtMapEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT_SCALE       ), heigtMapEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT_SCALE_SLIDER), heigtMapEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_FILTER_TYPE_LBL    ), heigtMapEnable);
    for (int iFilterType = 0; iFilterType < RParameters::FilterType_Count; ++iFilterType)
    {
        EnableWindow(GetDlgItem(hDlg, s_FilterTypeIds[iFilterType]), heigtMapEnable);
    }
    EnableWindow(GetDlgItem(hDlg, IDC_EDGE_WRAP          ), heigtMapEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_MULTIPLY_ALPHA  ),
        (heigtMapEnable && globals->m_pSrcAlphas != nullptr));

    CheckRadioButton(hDlg, s_FilterTypeIds[0], s_FilterTypeIds[RParameters::FilterType_Count - 1],
        s_FilterTypeIds[gParams->m_FilterType]);
    CheckDlgButton(hDlg, IDC_EDGE_WRAP,
        (gParams->m_EdgeWrap) ? BST_CHECKED : BST_UNCHECKED);
    CheckDlgButton(hDlg, IDC_MULTIPLY_ALPHA,
        (gParams->m_MultiplyAlpha) ? BST_CHECKED : BST_UNCHECKED);

    //-----------------------------------------------------------------------------
    // scale slope
    const bool scaleSlopeEnable = (gParams->m_Operation == RParameters::Operation_ScaleSclope);
    EnableWindow(GetDlgItem(hDlg, IDC_SCALE_SLOPE_GRP   ), scaleSlopeEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_SLOPE_SCALE_LBL   ), scaleSlopeEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_SLOPE_SCALE       ), scaleSlopeEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_SLOPE_SCALE_SLIDER), scaleSlopeEnable);

    //-----------------------------------------------------------------------------
    // ファイル合成
    const bool combineEnable = (gParams->m_Operation == RParameters::Operation_CombineFile);
    EnableWindow(GetDlgItem(hDlg, IDC_COMBINE_GRP            ), combineEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_OTHER_NORMAL_MAP_LBL   ), combineEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_OTHER_NORMAL_MAP       ), combineEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_OTHER_NORMAL_MAP_BROWSE), combineEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_COMBINE_SCALE_LBL      ), combineEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_COMBINE_SCALE          ), combineEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_COMBINE_SCALE_SLIDER   ), combineEnable);

    //-----------------------------------------------------------------------------
    // レイヤー合成
    const bool isCombineLayerEnabled = (gParams->m_Operation == RParameters::Operation_CombineLayer);
    const RStringArray& layerNames = *globals->m_pLayerNameArray;
    const int activeLayerNameIdx = static_cast<int>(layerNames.size());
    EnableWindow(GetDlgItem(hDlg, IDC_COMBINE_LAYER_GRP), isCombineLayerEnabled);
    for (int layerIdx = 0; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
    {
        HWND layerNameList = GetDlgItem(hDlg, IDC_LAYER_NAME0 + layerIdx);
        EnableWindow(GetDlgItem(hDlg, IDC_LAYER_NAME0_LBL + layerIdx), isCombineLayerEnabled);
        EnableWindow(layerNameList, isCombineLayerEnabled);
        if (layerIdx >= 1)
        {
            EnableWindow(GetDlgItem(hDlg, IDC_LAYER_SCALE1_LBL    + (layerIdx - 1)), isCombineLayerEnabled);
            EnableWindow(GetDlgItem(hDlg, IDC_LAYER_SCALE1        + (layerIdx - 1)), isCombineLayerEnabled);
            EnableWindow(GetDlgItem(hDlg, IDC_LAYER_SCALE1_SLIDER + (layerIdx - 1)), isCombineLayerEnabled);
        }

        int layerSelIdx = (layerIdx == 0) ? activeLayerNameIdx : 0;
        const std::string layerName = gParams->m_CombineLayerNames[layerIdx];
        const int layerNameIdx =
            (layerName.empty()                        ) ? -1                 :
            (layerName == RParameters::ActiveLayerName) ? activeLayerNameIdx :
            RFindValueInArray(layerNames, layerName);
        if (layerNameIdx != -1)
        {
            layerSelIdx = ((layerIdx == 0) ? 0 : 1) + layerNameIdx;
        }
        SendMessage(layerNameList, CB_SETCURSEL, layerSelIdx, 0L);
    }

    //-----------------------------------------------------------------------------
    // preview
    s_Preview.UpdateControl(hDlg);
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログの各 UI にツールチップを設定します。
//!
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void SetOptionDialogToolTip(HWND hDlg)
{
    const bool isJp = RIsJapaneseUI();
    RAddToolTip(hDlg, IDOK, (isJp) ? "フィルターを実行します" : "Applies the filter");
    RAddToolTip(hDlg, IDCANCEL, (isJp) ? "キャンセルします" : "Cancels");

    RAddToolTip(hDlg, IDC_ZOOM_OUT, (isJp) ? "プレビュー画像を縮小します" : "Zooms out on the preview image");
    RAddToolTip(hDlg, IDC_ZOOM_IN, (isJp) ? "プレビュー画像を拡大します" : "Zooms in on the preview image");
    RAddToolTip(hDlg, IDC_LIGHTING_PREVIEW, (isJp) ? "ライティングした状態をプレビューします" : "Previews with lighting");
    RAddToolTip(hDlg, IDC_ROTATE_LIGHT, (isJp) ? "ライト方向を自動的に回転します" : "Rotates the direction of lighting automatically");

    const char* pOperationTip = (isJp) ?
        "フィルター処理の種類を指定します。\n"
        "Convert from Height Map : 画像の輝度を高さとみなして、その凹凸から法線マップを作成。\n"
        "Normalize : 法線マップの各ピクセルの法線を正規化。\n"
        "Scale Slope : 接空間法線マップの各ピクセルの法線と (0, 0, 1) との角度をスケール。\n"
        "Combine : 選択したレイヤーの接空間法線マップに別ファイルの法線マップを合成。\n"
        "Combine Layer : 複数レイヤーの接空間法線マップを合成" :
        "Specifies the type of filtering. \n"
        "Convert from Height Map : Creates normal maps from bumps and depressions detected in an image by assuming brightness values indicate height. \n"
        "Normalize : Normalizes the normal vector for each pixel in a normal map. \n"
        "Scale Slope : Scales the angle between the normal and (0, 0, 1) for each pixel in a tangent space normal map. \n"
        "Combine : Combines the normal map of another file to the tangent space normal map of the selected layer. \n"
        "Combine Layer : Combines tangent space normal maps of multiple layers";
    RAddToolTip(hDlg, IDC_OPERATION_LBL, pOperationTip);
    RAddToolTip(hDlg, IDC_OPERATION    , pOperationTip);
    RAddToolTip(hDlg, IDC_POSITIVE_Z, (isJp) ? "Z 成分が 0 以上になるように調整するなら ON にします" :
        "Processes normals so the Z component is greater than zero if ON is selected");

    const char* pHeightScaleTip = (isJp) ? "高さに乗算するスケール値をパーセントで指定します" :
        "Specifies the scale to be applied to the heights as a percent";
    RAddToolTip(hDlg, IDC_HEIGHT_SCALE_LBL   , pHeightScaleTip);
    RAddToolTip(hDlg, IDC_HEIGHT_SCALE       , pHeightScaleTip);
    RAddToolTip(hDlg, IDC_HEIGHT_SCALE_SLIDER, pHeightScaleTip);
    const char* pFilterTypeTip = (isJp) ?
        "サンプリングする範囲を指定します。\n値が大きくなるほど広範囲をサンプリングします。" :
        "Specifies the range to be sampled. \nIncreasing this value samples a wider range";
    RAddToolTip(hDlg, IDC_FILTER_TYPE_LBL, pFilterTypeTip);
    for (int iFilterType = 0; iFilterType < RParameters::FilterType_Count; ++iFilterType)
    {
        RAddToolTip(hDlg, s_FilterTypeIds[iFilterType], pFilterTypeTip);
    }
    RAddToolTip(hDlg, IDC_EDGE_WRAP, (isJp) ?
        "画像が繰り返しているとして端のピクセルを処理するなら ON にします。\n"
        "画像の端のピクセルを引き伸ばして処理するなら OFF にします" :
        "Processes edge pixels in the case of a repeating height map if ON is selected. \n"
        "Extends the edge pixels of an image if OFF is selected");
    RAddToolTip(hDlg, IDC_MULTIPLY_ALPHA, (isJp) ?
        "高さマップにアルファチャンネルの値を乗算して法線を計算するなら ON にします" :
        "Calculates normals by multiplying a height map by the alpha value if ON is selected");

    const char* pSlopeScaleTip = (isJp) ?
        "法線とベクトル (0, 0, 1) との角度に乗算するスケール値をパーセントで指定します" :
        "Specifies the scale to be multiplied against the angle between the normal and vector (0, 0, 1) as a percent";
    RAddToolTip(hDlg, IDC_SLOPE_SCALE_LBL   , pSlopeScaleTip);
    RAddToolTip(hDlg, IDC_SLOPE_SCALE       , pSlopeScaleTip);
    RAddToolTip(hDlg, IDC_SLOPE_SCALE_SLIDER, pSlopeScaleTip);

    const char* pOtherNormalMapTip = (isJp) ?
        "合成する他の接空間法線マップ（PSD または TGA ファイル）をフルパスで指定します" :
        "Specifies the full path to the other tangent space normal map (PSD or TGA file) to be combined";
    RAddToolTip(hDlg, IDC_OTHER_NORMAL_MAP_LBL, pOtherNormalMapTip);
    RAddToolTip(hDlg, IDC_OTHER_NORMAL_MAP    , pOtherNormalMapTip);
    RAddToolTip(hDlg, IDC_OTHER_NORMAL_MAP_BROWSE, (isJp) ? "合成する他の接空間法線マップを選択するダイアログを表示します" :
        "Open the dialog box for selecting the other tangent space normal map to be combined");
    const char* pCombineScaleTip = (isJp) ?
        "合成する他の接空間法線マップの影響の強さをパーセントで指定します" :
        "Specifies the effect intensity of the other tangent space normal map to be combined as a percent";
    RAddToolTip(hDlg, IDC_COMBINE_SCALE_LBL   , pCombineScaleTip);
    RAddToolTip(hDlg, IDC_COMBINE_SCALE       , pCombineScaleTip);
    RAddToolTip(hDlg, IDC_COMBINE_SCALE_SLIDER, pCombineScaleTip);

    static const char* const LayerStrss[RParameters::CombineLayerCountMax][2] =
    {
        { "ベースのレイヤー"        , "base layer"                  },
        { "最初に合成するレイヤー"  , "first layer to be combined"  },
        { "2 番目に合成するレイヤー", "second layer to be combined" },
        { "3 番目に合成するレイヤー", "third layer to be combined"  },
        { "4 番目に合成するレイヤー", "fourth layer to be combined" },
    };
    for (int layerIdx = 0; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
    {
        const std::string layerStr = LayerStrss[layerIdx][(isJp) ? 0 : 1];
        const std::string layerNameTip = (isJp) ?
            layerStr + "を指定します。\n<Active> : レイヤーパネルで選択中のレイヤー" :
            "Specifies the " + layerStr + ". \n<Active>: Layer selected in Layer panel";
        RAddToolTip(hDlg, IDC_LAYER_NAME0_LBL + layerIdx, layerNameTip.c_str());
        RAddToolTip(hDlg, IDC_LAYER_NAME0     + layerIdx, layerNameTip.c_str());
        if (layerIdx >= 1)
        {
            const std::string layerScaleTip = (isJp) ?
                layerStr + "の影響の強さをパーセントで指定します" :
                "Specifies the effect intensity of the " + layerStr + " as a percent";
            RAddToolTip(hDlg, IDC_LAYER_SCALE1_LBL    + (layerIdx - 1), layerScaleTip.c_str());
            RAddToolTip(hDlg, IDC_LAYER_SCALE1        + (layerIdx - 1), layerScaleTip.c_str());
            RAddToolTip(hDlg, IDC_LAYER_SCALE1_SLIDER + (layerIdx - 1), layerScaleTip.c_str());
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログを初期化します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void InitOptionDialog(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // center dialog
    CenterDialog(hDlg);

    //-----------------------------------------------------------------------------
    // operation
    static const char* const OperationNames[] =
    {
        "Convert from Height Map",
        "Normalize",
        "Scale Slope",
        "Combine",
        "Combine Layer",
    };

    HWND operationLst = GetDlgItem(hDlg, IDC_OPERATION);
    for (int iOpe = 0; iOpe < RParameters::Operation_Count; ++iOpe)
    {
        SendMessage(operationLst, CB_ADDSTRING, 0, reinterpret_cast<LPARAM>(OperationNames[iOpe]));
    }

    //-----------------------------------------------------------------------------
    // convert from height map
    HWND heightScaleSlider = GetDlgItem(hDlg, IDC_HEIGHT_SCALE_SLIDER);
    SendMessage(heightScaleSlider, TBM_SETRANGE   , TRUE, MAKELPARAM(50, 400));
    SendMessage(heightScaleSlider, TBM_SETPAGESIZE, TRUE, 10);
    SetSliderPosAndRange(heightScaleSlider, gParams->m_HeightScale);

    //-----------------------------------------------------------------------------
    // slope scale
    HWND slopeScaleSlider = GetDlgItem(hDlg, IDC_SLOPE_SCALE_SLIDER);
    SendMessage(slopeScaleSlider, TBM_SETRANGE   , TRUE, MAKELPARAM(50, 200));
    SendMessage(slopeScaleSlider, TBM_SETPAGESIZE, TRUE, 10);
    SetSliderPosAndRange(slopeScaleSlider, gParams->m_SlopeScale);

    //-----------------------------------------------------------------------------
    // combine scale
    HWND combineScaleSlider = GetDlgItem(hDlg, IDC_COMBINE_SCALE_SLIDER);
    SendMessage(combineScaleSlider, TBM_SETRANGE   , TRUE, MAKELPARAM(50, 200));
    SendMessage(combineScaleSlider, TBM_SETPAGESIZE, TRUE, 10);
    SetSliderPosAndRange(combineScaleSlider, gParams->m_CombineScale);

    //-----------------------------------------------------------------------------
    // レイヤー合成
    const RStringArray& layerNames = *globals->m_pLayerNameArray;
    for (int layerIdx = 0; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
    {
        // Layer 0   の名前リストは layerNames[0], layerNames[1], ..., <Active>
        // Layer 1-4 の名前リストは 空白, layerNames[0], layerNames[1], ..., <Active>
        HWND layerNameList = GetDlgItem(hDlg, IDC_LAYER_NAME0 + layerIdx);
        if (layerIdx >= 1)
        {
            SendMessage(layerNameList, CB_ADDSTRING, 0, reinterpret_cast<LPARAM>(""));
        }
        for (const std::string& layerName : layerNames)
        {
            SendMessage(layerNameList, CB_ADDSTRING, 0, reinterpret_cast<LPARAM>(layerName.c_str()));
        }
        SendMessage(layerNameList, CB_ADDSTRING, 0,
            reinterpret_cast<LPARAM>(RParameters::ActiveLayerName.c_str()));
    }

    for (int layerIdx = 1; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
    {
        HWND layerScaleSlider = GetDlgItem(hDlg, IDC_LAYER_SCALE1_SLIDER + (layerIdx - 1));
        SendMessage(layerScaleSlider, TBM_SETRANGE   , TRUE, MAKELPARAM(50, 200));
        SendMessage(layerScaleSlider, TBM_SETPAGESIZE, TRUE, 10);
        SetSliderPosAndRange(layerScaleSlider, gParams->m_CombineLayerScales[layerIdx]);
    }

    //-----------------------------------------------------------------------------
    // 現状 Load/Save Settings ボタンを非表示にします。
    ShowWindow(GetDlgItem(hDlg, IDC_LOAD_SETTINGS), SW_HIDE);
    ShowWindow(GetDlgItem(hDlg, IDC_SAVE_SETTINGS), SW_HIDE);

    //-----------------------------------------------------------------------------
    // preview
    static const int PREVIEW_W = 256 + 208 + 2;
    static const int PREVIEW_H = 256 + 2;

    const size_t pixelCount = static_cast<size_t>(globals->m_ImageW) * globals->m_ImageH;
    s_pLightingPixels = new uint8_t[pixelCount * R_RGBA_BYTES];
    s_LightVec[0] = -0.5f;
    s_LightVec[1] =  0.5f;
    const float xxyy = s_LightVec[0] * s_LightVec[0] + s_LightVec[1] * s_LightVec[1];
    s_LightVec[2] = sqrtf(1.0f - RMin(xxyy, 1.0f));
    s_IsLightMoving = false;

    CheckDlgButton(hDlg, IDC_LIGHTING_PREVIEW,
        (globals->m_LightingPreview) ? BST_CHECKED : BST_UNCHECKED);
    CheckDlgButton(hDlg, IDC_ROTATE_LIGHT,
        (globals->m_RotatesLightingPreview) ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(GetDlgItem(hDlg, IDC_ROTATE_LIGHT), globals->m_LightingPreview);

    RECT previewRect;
    GetWindowRect(GetDlgItem(hDlg, IDC_PREVIEW_AREA), &previewRect);
    ScreenToClient(hDlg, reinterpret_cast<LPPOINT>(&previewRect.left));
    previewRect.right  = previewRect.left + PREVIEW_W;
    previewRect.bottom = previewRect.top  + PREVIEW_H;

    s_Preview.InitMemory(hDlg, previewRect, globals->m_ImageW, globals->m_ImageH);
    s_Preview.CenterScroll();
    s_Preview.SetControlId(IDC_ZOOM_IN, IDC_ZOOM_OUT, IDC_ZOOM_VALUE);

    //-----------------------------------------------------------------------------
    // update dialog
    SetOptionEditText(globals, hDlg);
    UpdateOptionDialog(globals, hDlg);

    //-----------------------------------------------------------------------------
    // focus OK button
    SetFocus(GetDlgItem(hDlg, IDOK));

    //-----------------------------------------------------------------------------
    // 各 UI にツールチップを追加します。
    SetOptionDialogToolTip(hDlg);
}

//-----------------------------------------------------------------------------
//! @brief 他の接空間法線マップのファイルパスをブラウザで選択します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
static bool BrowseOtherPath(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // ファイル選択ダイアログの初期フォルダを決定します。
    char initialBuf[MAX_PATH] = { 0 };
    GetWindowText(GetDlgItem(hDlg, IDC_OTHER_NORMAL_MAP), initialBuf, sizeof(initialBuf));
    std::string initialPath(initialBuf);
    if (!initialPath.empty())
    {
        initialPath = RGetFolderFromFilePath(RGetWindowsFilePath(initialPath));
    }
    if (initialPath.empty() || !RFolderExists(initialPath))
    {
        initialPath = RGetFolderFromFilePath(RGetDocumentFilePath(globals));
    }
    strncpy_s(initialBuf, initialPath.c_str(), sizeof(initialBuf));
    //RNoteTrace("initial path: [%s]",initialBuf);

    //-----------------------------------------------------------------------------
    // ファイル選択ダイアログを表示します。
    char pathBuf[MAX_PATH] = { 0 };
    OPENFILENAME ofn;
    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.hwndOwner = hDlg;
    ofn.lpstrFilter = "Image files (*.psd;*.tga)\0*.psd;*.tga\0\0";
    ofn.lpstrFile = pathBuf;
    ofn.nMaxFile = MAX_PATH;
    ofn.lpstrInitialDir = initialBuf;
    ofn.Flags = OFN_FILEMUSTEXIST | OFN_ENABLESIZING;
    if (!GetOpenFileName(&ofn) || (strlen(ofn.lpstrFile) == 0))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // パスを取得します。
    strncpy_s(gParams->m_OtherPath, ofn.lpstrFile, sizeof(gParams->m_OtherPath));
    SetOptionEditText(globals, hDlg);

    return true;
}

//-----------------------------------------------------------------------------
//! @brief オプション設定を xml ファイルにセーブします。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
static bool SaveSettings(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // セーブファイル選択ダイアログを表示します。
    char pathBuf[MAX_PATH] = { 0 };
    OPENFILENAME ofn;
    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.hwndOwner = hDlg;
    ofn.lpstrFilter = SETTINGS_FILE_FILTER;
    ofn.lpstrFile = pathBuf;
    ofn.nMaxFile = MAX_PATH;
    ofn.Flags = OFN_OVERWRITEPROMPT | OFN_ENABLESIZING;
    ofn.lpstrDefExt = SETTINGS_FILE_EXT; // 拡張子（.xml）が付いていない場合付加します。
    if (!GetSaveFileName(&ofn) || strlen(ofn.lpstrFile) == 0)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // open file
    std::ofstream ofs(ofn.lpstrFile, ios_base::out | ios_base::binary);
    if (!ofs)
    {
        RShowError(globals, "Can't open the file: %s", ofn.lpstrFile);
        return false;
    }
    std::ostream& os = ofs;
    ROutUtf8Bom(os);

    //-----------------------------------------------------------------------------
    // xml header
    os << RTab(0) << "<?xml version=\"1.0\" encoding=\"utf-8\"?>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // begin root
    os << RTab(0) << "<nw4f_normal_map_filter_settings version=\"1.0.0\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // convert from height map
    os << RTab(1) << "<convert_from_height_map" << R_ENDL;
    os << RTab(2) << "height_scale=\"" << gParams->m_HeightScale << "\"" << R_ENDL;
    os << RTab(2) << "filter_type=\"" << s_SettingsFilterTypeValues[gParams->m_FilterType] << "\"" << R_ENDL;

    os << RTab(2) << "edge_wrap=\"" << RBoolStr(gParams->m_EdgeWrap) << "\"" << R_ENDL;
    os << RTab(2) << "multiply_alpha=\"" << RBoolStr(gParams->m_MultiplyAlpha) << "\"" << R_ENDL;

    os << RTab(1) << "/>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end root
    os << RTab(0) << "</nw4f_normal_map_filter_settings>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // close file
    ofs.close();

    return true;
}

//-----------------------------------------------------------------------------
//! @brief オプション設定を xml ファイルからロードします。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
static bool LoadSettings(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // ロードファイル選択ダイアログを表示します。
    char pathBuf[MAX_PATH] = { 0 };
    OPENFILENAME ofn;
    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.hwndOwner = hDlg;
    ofn.lpstrFilter = SETTINGS_FILE_FILTER;
    ofn.lpstrFile = pathBuf;
    ofn.nMaxFile = MAX_PATH;
    ofn.Flags = OFN_FILEMUSTEXIST | OFN_ENABLESIZING;
    if (!GetOpenFileName(&ofn) || (strlen(ofn.lpstrFile) == 0))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // read file
    RFileBuf fileBuf(ofn.lpstrFile);
    if (!fileBuf)
    {
        RShowError(globals, "Can't open the file: %s", ofn.lpstrFile);
        return false;
    }

    //-----------------------------------------------------------------------------
    // parse XML
    RXMLElement xml;
    xml.LoadDocument(nullptr, std::string(reinterpret_cast<const char*>(fileBuf.GetBuf()), fileBuf.GetSize()));

    bool success = false;
    for (;;) // エラー時に break するための for で、ループではありません。
    {
        const RXMLElement* rootElem = xml.FindElement("nw4f_normal_map_filter_settings", false);
        if (rootElem == nullptr) break;

        const RXMLElement* heightMapElem = rootElem->FindElement("convert_from_height_map", false);
        if (heightMapElem == nullptr) break;

        //-----------------------------------------------------------------------------
        // convert from height map
        gParams->m_HeightScale = atol(heightMapElem->GetAttribute("height_scale").c_str());
        const std::string filterType = heightMapElem->GetAttribute("filter_type");
        for (int iFilterType = 0; iFilterType < RParameters::FilterType_Count; ++iFilterType)
        {
            if (filterType == s_SettingsFilterTypeValues[iFilterType])
            {
                gParams->m_FilterType = static_cast<RParameters::FilterType>(iFilterType);
            }
        }
        gParams->m_EdgeWrap      = (heightMapElem->GetAttribute("edge_wrap"     ) == "true");
        gParams->m_MultiplyAlpha = (heightMapElem->GetAttribute("multiply_alpha") == "true");

        success = true;
        break;
    }

    if (!success)
    {
        RShowError(globals, "Settings file is wrong: %s", ofn.lpstrFile);
        return false;
    }

    //-----------------------------------------------------------------------------
    // update dialog
    SetOptionEditText(globals, hDlg);
    UpdateOptionDialog(globals, hDlg);

    return true;
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログのカーソルを設定します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] pos カーソル座標です。
//-----------------------------------------------------------------------------
static void SetOptionDialogCursor(GPtr globals, const POINTS& pos)
{
    if (s_IsLightMoving)
    {
        sPSUIHooks->SetCursor(kPICursorPathArrow);
    }
    else if (globals->m_DisplaysPreview &&
        (s_Preview.m_IsDragging        ||
         s_Preview.m_IsShowingOriginal ||
         RIsPointInRect(s_Preview.m_ItemRect, pos)))
    {
        sPSUIHooks->SetCursor(kPICursorHand);
    }
    else
    {
        sPSUIHooks->SetCursor(kPICursorArrow);
    }
}

//-----------------------------------------------------------------------------
//! @brief プレビューの拡大率を変更します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] zoomStep 拡大率の増分です。縮小なら負の値を指定します。
//-----------------------------------------------------------------------------
static void ChangePreviewZoom(GPtr globals, HWND hDlg, const int zoomStep)
{
    const int newZoom = s_Preview.m_Zoom + zoomStep;
    if (s_Preview.m_MinZoom <= newZoom && newZoom <= s_Preview.m_MaxZoom)
    {
        s_Preview.ChangeZoom(hDlg, zoomStep);
        if (globals->m_LightingPreview)
        {
            UpdatePreviewBitmap(globals, hDlg, true);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログのダイアログプロシージャです。
//!
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] uMsg メッセージです。
//! @param[in] wParam メッセージの付加情報です。
//! @param[in] lParam メッセージの付加情報です。
//!
//! @return 処理した場合は TRUE を返します。
//-----------------------------------------------------------------------------
static BOOL WINAPI OptionDialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    static GPtr globals = nullptr; // ダイアログ初期化時に保持するので static にする必要があります。
    static RParameters paramsBak; // キャンセル時にパラメーターを元に戻すために保持します。
    static POINTS dragStartPos; // ドラッグ開始時の座標です。
    static int dragStartIx; // ドラッグ開始時のプレビューの水平方向のスクロール値です。
    static int dragStartIy; // ドラッグ開始時のプレビューの垂直方向のスクロール値です。
    const int UPDATE_TIMER_ID = 1000; // プレビュー更新用タイマーの ID です。

    char strBuf[MAX_PATH] = { 0 }; // エディットテキストの値を取得するためのバッファです。

    switch (uMsg)
    {
    case  WM_INITDIALOG:
        globals = reinterpret_cast<GPtr>(lParam);
        paramsBak = *gParams;
        InitOptionDialog(globals, hDlg);
        if (gResult != noErr)
        {
            return FALSE;
        }
        SetTimer(hDlg, UPDATE_TIMER_ID, 50, nullptr); // 50 [msec] ごとに WM_TIMER を発生させます。
        //break; // break せずに WM_PAINT の処理もします。

    case WM_PAINT:
        if (globals->m_DisplaysPreview)
        {
            //RNoteTrace("paint");
            s_Preview.Paint(hDlg);
        }
        return FALSE;

    case WM_DESTROY:
        KillTimer(hDlg, UPDATE_TIMER_ID);
        s_Preview.FreeMemory();
        RFreeAndClearArray(s_pLightingPixels);
        s_OtherImage.FreeMemory();
        break;

    case WM_TIMER:
        if (wParam == UPDATE_TIMER_ID)
        {
            if (s_Preview.m_UpdateTimerCount > 0)
            {
                //RNoteTrace("timer: %d", s_Preview.m_UpdateTimerCount);
                --s_Preview.m_UpdateTimerCount;
                if (s_Preview.m_UpdateTimerCount == 0)
                {
                    //RNoteTrace("update by timer");
                    UpdatePreviewBitmap(globals, hDlg, false);
                }
            }

            if (globals->m_DisplaysPreview        &&
                globals->m_LightingPreview        &&
                globals->m_RotatesLightingPreview &&
                !s_Preview.m_IsShowingOriginal    &&
                !s_IsLightMoving)
            {
                RotatePreviewLight(globals, hDlg, -8.0f);
            }
        }
        break;

    case WM_COMMAND:
        {
            const int id     = LOWORD(wParam);
            const int notify = HIWORD(wParam);
            switch (id)
            {
            case IDOK:
                if (GetOptionEditText(globals, hDlg))
                {
                    EndDialog(hDlg, id);
                }
                break;

            case IDCANCEL:
                *gParams = paramsBak;
                EndDialog(hDlg, id);
                break;

            case IDC_OPERATION:
                if (notify == CBN_SELCHANGE)
                {
                    const int sel = static_cast<int>(SendMessage(GetDlgItem(hDlg, id),
                        CB_GETCURSEL, 0, 0));
                    gParams->m_Operation = static_cast<RParameters::Operation>(sel);
                    UpdateOptionDialog(globals, hDlg);
                    UpdatePreviewBitmap(globals, hDlg, false);
                }
                break;

            case IDC_POSITIVE_Z:
                gParams->m_PositiveZ = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                UpdatePreviewBitmap(globals, hDlg, false);
                break;

            case IDC_HEIGHT_SCALE:
                if (notify == EN_CHANGE || notify == EN_KILLFOCUS)
                {
                    GetWindowText(GetDlgItem(hDlg, id), strBuf, sizeof(strBuf));
                    if (strBuf[0] != 0)
                    {
                        gParams->m_HeightScale = atoi(strBuf);
                        SetSliderPosAndRange(GetDlgItem(hDlg, IDC_HEIGHT_SCALE_SLIDER), gParams->m_HeightScale);
                        if (s_Preview.m_UpdateTimerCount == 0)
                        {
                            UpdatePreviewBitmap(globals, hDlg, false);
                        }
                    }
                }
                break;

            case IDC_FILTER_TYPE_4:
            case IDC_FILTER_TYPE_3X3:
            case IDC_FILTER_TYPE_5X5:
            case IDC_FILTER_TYPE_7X7:
            case IDC_FILTER_TYPE_9X9:
            case IDC_FILTER_TYPE_13X13:
            case IDC_FILTER_TYPE_17X17:
            case IDC_FILTER_TYPE_21X21:
            case IDC_FILTER_TYPE_25X25:
                for (int iFilterType = 0; iFilterType < RParameters::FilterType_Count; ++iFilterType)
                {
                    if (s_FilterTypeIds[iFilterType] == id)
                    {
                        gParams->m_FilterType = static_cast<RParameters::FilterType>(iFilterType);
                        UpdatePreviewBitmap(globals, hDlg, false);
                        break;
                    }
                }
                break;

            case IDC_EDGE_WRAP:
                gParams->m_EdgeWrap = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                UpdatePreviewBitmap(globals, hDlg, false);
                break;

            case IDC_MULTIPLY_ALPHA:
                gParams->m_MultiplyAlpha = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                UpdatePreviewBitmap(globals, hDlg, false);
                break;

            case IDC_SLOPE_SCALE:
                if (notify == EN_CHANGE || notify == EN_KILLFOCUS)
                {
                    GetWindowText(GetDlgItem(hDlg, id), strBuf, sizeof(strBuf));
                    if (strBuf[0] != 0)
                    {
                        gParams->m_SlopeScale = atoi(strBuf);
                        SetSliderPosAndRange(GetDlgItem(hDlg, IDC_SLOPE_SCALE_SLIDER), gParams->m_SlopeScale);
                        if (s_Preview.m_UpdateTimerCount == 0)
                        {
                            UpdatePreviewBitmap(globals, hDlg, false);
                        }
                    }
                }
                break;

            case IDC_OTHER_NORMAL_MAP:
                if (notify == EN_KILLFOCUS)
                {
                    const int otherPathLen = GetWindowText(GetDlgItem(hDlg, id),
                        gParams->m_OtherPath, sizeof(gParams->m_OtherPath));
                    if (otherPathLen != 0)
                    {
                        UpdatePreviewBitmap(globals, hDlg, false);
                    }
                }
                break;

            case IDC_OTHER_NORMAL_MAP_BROWSE:
                if (notify == BN_CLICKED)
                {
                    if (BrowseOtherPath(globals, hDlg))
                    {
                        UpdatePreviewBitmap(globals, hDlg, false);
                    }
                }
                break;

            case IDC_COMBINE_SCALE:
                if (notify == EN_CHANGE || notify == EN_KILLFOCUS)
                {
                    GetWindowText(GetDlgItem(hDlg, id), strBuf, sizeof(strBuf));
                    if (strBuf[0] != 0)
                    {
                        gParams->m_CombineScale = atoi(strBuf);
                        SetSliderPosAndRange(GetDlgItem(hDlg, IDC_COMBINE_SCALE_SLIDER), gParams->m_CombineScale);
                        if (s_Preview.m_UpdateTimerCount == 0)
                        {
                            UpdatePreviewBitmap(globals, hDlg, false);
                        }
                    }
                }
                break;

            case IDC_LAYER_NAME0:
            case IDC_LAYER_NAME1:
            case IDC_LAYER_NAME2:
            case IDC_LAYER_NAME3:
            case IDC_LAYER_NAME4:
                if (notify == CBN_SELCHANGE)
                {
                    const int layerIdx = id - IDC_LAYER_NAME0;
                    char* nameBuf = gParams->m_CombineLayerNames[layerIdx];
                    const size_t nameBufSize = sizeof(gParams->m_CombineLayerNames[layerIdx]);
                    memset(nameBuf, 0x00, nameBufSize);
                    GetWindowText(GetDlgItem(hDlg, id), nameBuf, nameBufSize);
                    UpdateOptionDialog(globals, hDlg);
                    UpdatePreviewBitmap(globals, hDlg, false);
                }
                break;

            case IDC_LAYER_SCALE1:
            case IDC_LAYER_SCALE2:
            case IDC_LAYER_SCALE3:
            case IDC_LAYER_SCALE4:
                if (notify == EN_CHANGE || notify == EN_KILLFOCUS)
                {
                    GetWindowText(GetDlgItem(hDlg, id), strBuf, sizeof(strBuf));
                    if (strBuf[0] != 0)
                    {
                        const int layerIdx = id - IDC_LAYER_SCALE1 + 1;
                        gParams->m_CombineLayerScales[layerIdx] = atoi(strBuf);
                        SetSliderPosAndRange(GetDlgItem(hDlg, IDC_LAYER_SCALE1_SLIDER + (layerIdx - 1)),
                            gParams->m_CombineLayerScales[layerIdx]);
                        if (s_Preview.m_UpdateTimerCount == 0)
                        {
                            UpdatePreviewBitmap(globals, hDlg, false);
                        }
                    }
                }
                break;

            case IDC_LOAD_SETTINGS:
                if (notify == BN_CLICKED)
                {
                    LoadSettings(globals, hDlg);
                }
                break;

            case IDC_SAVE_SETTINGS:
                if (notify == BN_CLICKED)
                {
                    if (GetOptionEditText(globals, hDlg))
                    {
                        SaveSettings(globals, hDlg);
                    }
                }
                break;

            case IDC_ZOOM_IN:
                ChangePreviewZoom(globals, hDlg, 1);
                break;

            case IDC_ZOOM_OUT:
                ChangePreviewZoom(globals, hDlg, -1);
                break;

            case IDC_LIGHTING_PREVIEW:
                globals->m_LightingPreview = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                EnableWindow(GetDlgItem(hDlg, IDC_ROTATE_LIGHT), globals->m_LightingPreview);
                UpdatePreviewBitmap(globals, hDlg, true);
                break;

            case IDC_ROTATE_LIGHT:
                globals->m_RotatesLightingPreview = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                break;

            case IDM_LOCAL_HELP:
                RShowPluginHelp(NPS_PLUGIN_HELP_URL);
                break;

            case IDM_HELP_INDEX:
                RShowPluginHelp(NPS_HELP_INDEX_URL);
                break;

            default:
                break;
            }
        }
        break;

    case WM_HSCROLL:
        {
            const int code = LOWORD(wParam);
            //const int pos = HIWORD(wParam); // SB_THUMBPOSITION と SB_THUMBTRACK のみ有効
            const int id = GetWindowLong(reinterpret_cast<HWND>(lParam), GWL_ID);
            if (id == IDC_HEIGHT_SCALE_SLIDER  ||
                id == IDC_SLOPE_SCALE_SLIDER   ||
                id == IDC_COMBINE_SCALE_SLIDER ||
                id == IDC_LAYER_SCALE1_SLIDER  ||
                id == IDC_LAYER_SCALE2_SLIDER  ||
                id == IDC_LAYER_SCALE3_SLIDER  ||
                id == IDC_LAYER_SCALE4_SLIDER)
            {
                //RNoteTrace("hscroll: %d %d", code, HIWORD(wParam));
                if (code != SB_ENDSCROLL)
                {
                    // スライダーをドラッグ中は一定時間ごとにプレビューを更新します。
                    // ドラッグ終了時はすぐに更新します。
                    s_Preview.m_UpdateTimerCount = (code == SB_THUMBTRACK) ?
                        RPreview::TIMER_DRAG : 0;
                    int editId;
                    switch (id)
                    {
                    default:
                    case IDC_HEIGHT_SCALE_SLIDER : editId = IDC_HEIGHT_SCALE ; break;
                    case IDC_SLOPE_SCALE_SLIDER  : editId = IDC_SLOPE_SCALE  ; break;
                    case IDC_COMBINE_SCALE_SLIDER: editId = IDC_COMBINE_SCALE; break;
                    case IDC_LAYER_SCALE1_SLIDER : editId = IDC_LAYER_SCALE1 ; break;
                    case IDC_LAYER_SCALE2_SLIDER : editId = IDC_LAYER_SCALE2 ; break;
                    case IDC_LAYER_SCALE3_SLIDER : editId = IDC_LAYER_SCALE3 ; break;
                    case IDC_LAYER_SCALE4_SLIDER : editId = IDC_LAYER_SCALE4 ; break;
                    }
                    const int posVal = static_cast<int>(SendDlgItemMessage(hDlg, id, TBM_GETPOS, 0, 0));
                    SetDlgItemInt(hDlg, editId, posVal, FALSE);
                }
            }
        }
        break;

    case WM_LBUTTONDOWN:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (globals->m_DisplaysPreview &&
                !s_IsLightMoving           &&
                RIsPointInRect(s_Preview.m_ItemRect, pos))
            {
                // フォーカスをプレビューに移してホイールで拡大縮小ができるようにします。
                SetFocus(GetDlgItem(hDlg, IDC_PREVIEW_AREA));
                s_Preview.m_IsDragging = s_Preview.IsScrollEnable();
                s_Preview.m_IsShowingOriginal = true;
                const uint8_t* pOrgPixels =
                    (gParams->m_Operation == RParameters::Operation_ConvertFromHeightMap &&
                     globals->m_pGrayMatPixels != nullptr) ?
                    globals->m_pGrayMatPixels : globals->m_pSrcPixels;
                s_Preview.UpdateWinBmp(hDlg, pOrgPixels);

                SetOptionDialogCursor(globals, pos);
                if (s_Preview.m_IsDragging)
                {
                    // start drag
                    dragStartPos = pos;
                    dragStartIx = s_Preview.m_Ix;
                    dragStartIy = s_Preview.m_Iy;
                    SetCapture(hDlg);
                    return FALSE;
                }
            }
        }
        break;

    case WM_RBUTTONDOWN:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (globals->m_DisplaysPreview &&
                globals->m_LightingPreview &&
                !s_Preview.m_IsDragging    &&
                RIsPointInRect(s_Preview.m_ItemRect, pos))
            {
                s_IsLightMoving = true;
                memcpy(s_MoveStartLightVec, s_LightVec, sizeof(s_MoveStartLightVec));
                dragStartPos = pos;
                #ifdef MOVE_LIGHT_ON_CLICK
                ChangeLightVec(globals, hDlg, pos, dragStartPos);
                #endif
                SetOptionDialogCursor(globals, pos);
                SetCapture(hDlg);
                return FALSE;
            }
        }
        break;

    case WM_MOUSEMOVE:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (s_Preview.m_IsDragging)
            {
                s_Preview.ChangeScroll(hDlg,
                    dragStartIx - (pos.x - dragStartPos.x),
                    dragStartIy - (pos.y - dragStartPos.y));
            }
            else if (s_IsLightMoving)
            {
                ChangeLightVec(globals, hDlg, pos, dragStartPos);
            }
            SetOptionDialogCursor(globals, pos);
        }
        break;

    case WM_LBUTTONUP:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (s_Preview.m_IsDragging)
            {
                s_Preview.m_IsDragging = false;
                ReleaseCapture();
            }
            if (s_Preview.m_IsShowingOriginal)
            {
                s_Preview.m_IsShowingOriginal = false;
                if (globals->m_LightingPreview)
                {
                    UpdatePreviewBitmap(globals, hDlg, true);
                }
                else
                {
                    s_Preview.UpdateWinBmp(hDlg, globals->m_pDstPixels);
                }
            }
            SetOptionDialogCursor(globals, pos);
        }
        break;

    case WM_RBUTTONUP:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (s_IsLightMoving)
            {
                s_IsLightMoving = false;
                ReleaseCapture();
            }
            SetOptionDialogCursor(globals, pos);
        }
        break;

    case WM_MOUSEWHEEL:
        {
            POINTS pos = MAKEPOINTS(lParam); // スクリーン座標
            POINT pnt; // クライアント座標に変換するために POINT (LONG x 2) を使用します。
            pnt.x = pos.x;
            pnt.y = pos.y;
            ScreenToClient(hDlg, &pnt);
            pos.x = static_cast<short>(pnt.x);
            pos.y = static_cast<short>(pnt.y);
            const short delta = static_cast<short>(HIWORD(wParam));
            if (globals->m_DisplaysPreview)
            {
                if (delta > 0)
                {
                    ChangePreviewZoom(globals, hDlg, 1);
                }
                else if (delta < 0)
                {
                    ChangePreviewZoom(globals, hDlg, -1);
                }
            }
            SetOptionDialogCursor(globals, pos);
            //RNoteTrace("wheel: %d %d\n", pos.x, pos.y);
        }
        break;

    default:
        return FALSE;
    }
    return TRUE;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief オプションダイアログを表示します。
//!
//! @param[in,out] globals グローバルデータです。
//!
//! @return OK ボタンがクリックされたら true、
//!         Cancel ボタンがクリックされたら false を返します。
//-----------------------------------------------------------------------------
bool DoOptionDialog(GPtr globals)
{
    HINSTANCE instance = GetDLLInstance();
    //RNoteTrace("dll path: %s", RGetModuleFileName(instance).c_str());
    PlatformData* platform = reinterpret_cast<PlatformData*>(gStuff->platformData);
    INT_PTR result = DialogBoxParam(instance,
        MAKEINTRESOURCE(IDD_OPTION), reinterpret_cast<HWND>(platform->hwnd),
        reinterpret_cast<DLGPROC>(OptionDialogProc), reinterpret_cast<LPARAM>(globals));
    return (result == IDOK);
}

//-----------------------------------------------------------------------------
//! @brief アバウトダイアログを表示します。
//!
//! @param[in] about アバウトレコードです。
//-----------------------------------------------------------------------------
void DoAboutDialog(AboutRecordPtr about)
{
    HINSTANCE instance = GetDLLInstance();
    PlatformData* platform = reinterpret_cast<PlatformData*>(about->platformData);
    RAboutParam aboutParam =
    {
        NPS_ABOUT_TITLE, NPS_ABOUT_MESSAGE,
        IDC_ABOUT_TEXT, IDC_HELP_INDEX, NPS_HELP_INDEX_URL
    };
    DialogBoxParam(instance,
        MAKEINTRESOURCE(IDD_ABOUT), reinterpret_cast<HWND>(platform->hwnd),
        reinterpret_cast<DLGPROC>(RAboutDialogProc), reinterpret_cast<LPARAM>(&aboutParam));
}

