﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// variables
// RDoFormatDialog FormatDialogProc InitFormatDialog EndFormatDialog
// GetFormatPresetInfos ConfigChangedCB FormatDialogFormatCB
// SetFormatDialogSize
// RDoAboutDialog

//=============================================================================
// include
//=============================================================================
#ifndef NPS_CLONE_SOURCE
    #include "Globals.h"
    #include "NpsFormatUI.h"
    #include "NpsVersion.h"
    #include "Resource.h"
#endif
#include "NpsPreview.h"

#include <commctrl.h> // for slider

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
//#define USE_3D_AND_ARRAY // 3D および配列テクスチャに対応するなら定義します。

//=============================================================================
//! @brief フォーマットプリセット情報のクラスです。
//=============================================================================
class RFormatPresetInfo
{
public:
    std::string m_Name; //!< プリセット名です。
    std::string m_Label; //!< ラベルです。
    FtxFormat m_Format; //!< フォーマットです。
    FtxCompSel m_CompSel; //!< 成分選択です。
    std::string m_Hint; //!< ヒント情報です。
    int m_LinearFlag; //!< リニア変換フラグです。
    int m_InitialSwizzle; //!< 初期スウィズル値です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] name プリセット名です。
    //! @param[in] format フォーマットです。
    //! @param[in] pCompSel 成分選択オプション文字列へのポインタです。
    //!                     NULL ならフォーマットに対するデフォルトの成分選択で初期化します。
    //! @param[in] pHint ヒント情報文字列へのポインタです。
    //!                  NULL なら指定なしです。
    //! @param[in] linearFlag リニア変換フラグです。
    //! @param[in] initialSwizzle 初期スウィズル値です。
    //!
    RFormatPresetInfo(
        const std::string& name,
        const FtxFormat format,
        const char* pCompSel = NULL,
        const char* pHint = NULL,
        const int linearFlag = RLinearFlag::LINEAR_DEFAULT,
        const int initialSwizzle = RAddInfo::SWIZZLE_DEFAULT
    )
    : m_Name(name),
      m_Label(name),
      m_Format(format),
      m_LinearFlag(linearFlag),
      m_InitialSwizzle(initialSwizzle)
    {
        m_Hint = (pHint != NULL) ? std::string(pHint) : HintValueDefault;
        m_CompSel = (pCompSel != NULL) ?
            RGetCompSelFromOptionString(pCompSel) :
            RGetDefaultCompSel(format, m_Hint);
    }
};

//! @brief フォーマットプリセット情報配列の定義です。
typedef std::vector<RFormatPresetInfo> RFormatPresetInfoArray;

//=============================================================================
// variables
//=============================================================================

//-----------------------------------------------------------------------------
// format dialog
static HWND s_FormatDialogHandle = NULL; // オプションダイアログのダイアログハンドルです。
static HFONT s_FormatFont = NULL; // フォーマットプリセットのフォントです。

static RStringArray s_AllConfigNames; //!< コンフィグ名一覧です。
static std::string s_ConfigName; //!< 現在のコンフィグ名です。
static RFormatPresetInfoArray s_FormatPresetInfos; // フォーマットプリセット情報配列です。
static int s_FormatPresetIdx = 0; // 最後に選択したフォーマットプリセットのインデックスです。
static bool s_IsAllFormat = true; // 全フォーマットから指定するなら true です。
static bool s_EnablesWeightedCompress = false; //!< 重み付け圧縮有効フラグです。
static RStringArray s_HintPresets; //!< ヒント情報プリセット配列です。

static int s_FormatDialogNoPreviewW; // プレビューなしの場合のオプションダイアログの幅です。
static int s_FormatDialogNoPreviewH; // プレビューなしの場合のオプションダイアログの高さです。
static int s_FormatDialogPreviewMinW; // プレビューありの場合のオプションダイアログの幅です。
static int s_FormatDialogPreviewMinH; // プレビューありの場合のオプションダイアログの高さです。

//-----------------------------------------------------------------------------
// preview
static RPreview s_Preview; // プレビュー表示です。
static int s_PreviewRgbPosOfsY = 0; // RGB / Alpha ボタンのオプションダイアログ下端からのオフセットです。
static int s_PreviewZoomBtnPosOfsY = 0; // ズームイン／アウトボタンのオプションダイアログ下端からのオフセットです。
static int s_PreviewZoomTextPosOfsY = 0; // 拡大率テキストのオプションダイアログ下端からのオフセットです。

//-----------------------------------------------------------------------------
// format list
static const FtxFormat s_AllIntegerFormats[] = //!< 整数型から変換可能な全フォーマットの配列です。
{
    FtxFormat_Unorm_5_6_5,

    FtxFormat_Unorm_5_5_5_1,
    FtxFormat_Unorm_4_4_4_4,

    FtxFormat_Unorm_8,
    FtxFormat_Unorm_4_4,
    FtxFormat_Unorm_8_8,
    FtxFormat_Unorm_8_8_8_8,

    FtxFormat_Snorm_8,
    FtxFormat_Snorm_8_8,
    FtxFormat_Snorm_8_8_8_8,
    FtxFormat_Srgb_8_8_8_8,

    FtxFormat_Unorm_Bc1,
    FtxFormat_Unorm_Bc2,
    FtxFormat_Unorm_Bc3,
    FtxFormat_Unorm_Bc4,
    FtxFormat_Unorm_Bc5,
    FtxFormat_Snorm_Bc4,
    FtxFormat_Snorm_Bc5,
    FtxFormat_Srgb_Bc1,
    FtxFormat_Srgb_Bc2,
    FtxFormat_Srgb_Bc3,

    FtxFormat_Uint_8,
    FtxFormat_Uint_8_8,
    FtxFormat_Uint_8_8_8_8,
    FtxFormat_Sint_8,
    FtxFormat_Sint_8_8,
    FtxFormat_Sint_8_8_8_8,

    FtxFormat_Unorm_Bc7,
    FtxFormat_Srgb_Bc7,

    FtxFormat_Unorm_Etc1,
    FtxFormat_Unorm_Etc2,
    FtxFormat_Srgb_Etc2,
    FtxFormat_Unorm_Etc2_Mask,
    FtxFormat_Srgb_Etc2_Mask,
    FtxFormat_Unorm_Etc2_Alpha,
    FtxFormat_Srgb_Etc2_Alpha,

    FtxFormat_Unorm_Eac_11,
    FtxFormat_Snorm_Eac_11,
    FtxFormat_Unorm_Eac_11_11,
    FtxFormat_Snorm_Eac_11_11,

    FtxFormat_Unorm_Pvrtc1_2Bpp,
    FtxFormat_Srgb_Pvrtc1_2Bpp,
    FtxFormat_Unorm_Pvrtc1_4Bpp,
    FtxFormat_Srgb_Pvrtc1_4Bpp,
    FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp,
    FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp,
    FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp,
    FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp,
    FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp,
    FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp,
    FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp,
    FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp,

    FtxFormat_Unorm_Astc_4x4,
    FtxFormat_Srgb_Astc_4x4,
    FtxFormat_Unorm_Astc_5x4,
    FtxFormat_Srgb_Astc_5x4,
    FtxFormat_Unorm_Astc_5x5,
    FtxFormat_Srgb_Astc_5x5,
    FtxFormat_Unorm_Astc_6x5,
    FtxFormat_Srgb_Astc_6x5,
    FtxFormat_Unorm_Astc_6x6,
    FtxFormat_Srgb_Astc_6x6,
    FtxFormat_Unorm_Astc_8x5,
    FtxFormat_Srgb_Astc_8x5,
    FtxFormat_Unorm_Astc_8x6,
    FtxFormat_Srgb_Astc_8x6,
    FtxFormat_Unorm_Astc_8x8,
    FtxFormat_Srgb_Astc_8x8,
    FtxFormat_Unorm_Astc_10x5,
    FtxFormat_Srgb_Astc_10x5,
    FtxFormat_Unorm_Astc_10x6,
    FtxFormat_Srgb_Astc_10x6,
    FtxFormat_Unorm_Astc_10x8,
    FtxFormat_Srgb_Astc_10x8,
    FtxFormat_Unorm_Astc_10x10,
    FtxFormat_Srgb_Astc_10x10,
    FtxFormat_Unorm_Astc_12x10,
    FtxFormat_Srgb_Astc_12x10,
    FtxFormat_Unorm_Astc_12x12,
    FtxFormat_Srgb_Astc_12x12,
};

static const FtxFormat s_AllRealNumberFormats[] = //!< 実数型から変換可能な全フォーマットの配列です。
{
    FtxFormat_Unorm_8_8_8_8,
    FtxFormat_Snorm_8_8_8_8,
    FtxFormat_Srgb_8_8_8_8,

    FtxFormat_Unorm_16,
    FtxFormat_Unorm_16_16,
    FtxFormat_Unorm_16_16_16_16,
    FtxFormat_Snorm_16,
    FtxFormat_Snorm_16_16,
    FtxFormat_Snorm_16_16_16_16,

    FtxFormat_Float_16,
    FtxFormat_Float_32,
    FtxFormat_Float_16_16,
    FtxFormat_Float_32_32,
    FtxFormat_Float_11_11_10,
    FtxFormat_Float_16_16_16_16,
    FtxFormat_Float_32_32_32_32,

    FtxFormat_Unorm_Bc1,
    FtxFormat_Unorm_Bc2,
    FtxFormat_Unorm_Bc3,
    FtxFormat_Unorm_Bc4,
    FtxFormat_Unorm_Bc5,
    FtxFormat_Snorm_Bc4,
    FtxFormat_Snorm_Bc5,
    FtxFormat_Srgb_Bc1,
    FtxFormat_Srgb_Bc2,
    FtxFormat_Srgb_Bc3,

    FtxFormat_Ufloat_Bc6,
    FtxFormat_Float_Bc6,
    FtxFormat_Unorm_Bc7,
    FtxFormat_Srgb_Bc7,

    FtxFormat_Unorm_Astc_4x4,
    FtxFormat_Srgb_Astc_4x4,
    FtxFormat_Unorm_Astc_5x4,
    FtxFormat_Srgb_Astc_5x4,
    FtxFormat_Unorm_Astc_5x5,
    FtxFormat_Srgb_Astc_5x5,
    FtxFormat_Unorm_Astc_6x5,
    FtxFormat_Srgb_Astc_6x5,
    FtxFormat_Unorm_Astc_6x6,
    FtxFormat_Srgb_Astc_6x6,
    FtxFormat_Unorm_Astc_8x5,
    FtxFormat_Srgb_Astc_8x5,
    FtxFormat_Unorm_Astc_8x6,
    FtxFormat_Srgb_Astc_8x6,
    FtxFormat_Unorm_Astc_8x8,
    FtxFormat_Srgb_Astc_8x8,
    FtxFormat_Unorm_Astc_10x5,
    FtxFormat_Srgb_Astc_10x5,
    FtxFormat_Unorm_Astc_10x6,
    FtxFormat_Srgb_Astc_10x6,
    FtxFormat_Unorm_Astc_10x8,
    FtxFormat_Srgb_Astc_10x8,
    FtxFormat_Unorm_Astc_10x10,
    FtxFormat_Srgb_Astc_10x10,
    FtxFormat_Unorm_Astc_12x10,
    FtxFormat_Srgb_Astc_12x10,
    FtxFormat_Unorm_Astc_12x12,
    FtxFormat_Srgb_Astc_12x12,
};

//-----------------------------------------------------------------------------
// dimension list
static const FtxDimension s_AllDimensions[] = //!< 全次元の配列です。
{
    FtxDimension_1d,
    FtxDimension_2d,
    FtxDimension_CubeMap,
    #ifdef USE_3D_AND_ARRAY
    FtxDimension_3d,
    FtxDimension_1dArray,
    FtxDimension_2dArray,
    #endif
};

static std::vector<FtxDimension> s_Dimensions; //!< 選択可能な次元の配列です。

//-----------------------------------------------------------------------------
// comp sel control
static const int s_CompSelControls[R_RGBA_BYTES] = //!< 成分選択のコントロール ID 配列です。
{
    IDC_COMP_SEL_R,
    IDC_COMP_SEL_G,
    IDC_COMP_SEL_B,
    IDC_COMP_SEL_A,
};

//-----------------------------------------------------------------------------
//! @brief プレビュー画像を更新します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void UpdatePreviewBitmap(GPtr globals, HWND hDlg)
{
    if (globals->m_DisplaysPreview)
    {
        //-----------------------------------------------------------------------------
        // エンコードします。
        if (!globals->m_PreviewDataSetFlag ||
            globals->m_PreviewFormat != g_AddInfo.m_Format)
        {
            RUpdatePreviewDataBuf(globals);
        }

        //-----------------------------------------------------------------------------
        // プレビュー画像を更新します。
        s_Preview.UpdateWinBmp(hDlg, globals->m_pBitmapData);
    }
}

//-----------------------------------------------------------------------------
//! @brief 有効なヒント情報フラグなら true を返します。
//!
//! @param[in] hint ヒント情報フラグです。
//!
//! @return 有効なヒント情報フラグなら true を返します。
//-----------------------------------------------------------------------------
static bool IsValidHint(const std::string& hint)
{
    const int count = static_cast<int>(hint.size());
    for (int ic = 0; ic < count; ++ic)
    {
        const char c = hint[ic];
        if (
            ('0' <= c && c <= '9') ||
            ('A' <= c && c <= 'Z') ||
            ('a' <= c && c <= 'z') ||
            c == '_')
        {
            // OK
        }
        else
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 有効な成分選択なら true を返します。
//!
//! @param[in] compSel 成分選択文字列です。
//!
//! @return 有効な成分選択なら true を返します。
//-----------------------------------------------------------------------------
static bool IsValidCompSel(const std::string& compSel)
{
    if (compSel.size() == 4)
    {
        for (int iRgba = 0; iRgba < 4; ++iRgba)
        {
            const char c = compSel[iRgba];
            if (c != 'r' &&
                c != 'g' &&
                c != 'b' &&
                c != 'a' &&
                c != '0' &&
                c != '1')
            {
                return false;
            }
        }
        return true;
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定した名前のフォーマットプリセット情報を検索します。
//!
//! @param[in] name フォーマットプリセット名です。
//!
//! @return フォーマットプリセット情報のインデックスを返します。
//!         見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindFormatPresetInfoByName(const std::string& name)
{
    const int presetCount = static_cast<int>(s_FormatPresetInfos.size());
    for (int iPreset = 0; iPreset < presetCount; ++iPreset)
    {
        const RFormatPresetInfo& presetInfo = s_FormatPresetInfos[iPreset];
        if (presetInfo.m_Name == name)
        {
            return iPreset;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief 指定したテクスチャフォーマット、成分選択、ヒント情報、
//!        リニア変換フラグ、初期スウィズル値に対応するフォーマットプリセット
//!        情報を検索します。
//!
//! @param[in] format テクスチャフォーマットです。
//! @param[in] compSel 成分選択です。
//! @param[in] hint ヒント情報です。
//! @param[in] linearFlag リニア変換フラグです。
//! @param[in] initialSwizzle 初期スウィズル値です。
//!
//! @return フォーマットプリセット情報のインデックスを返します。
//!         見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindFormatPresetInfoByFormat(
    const FtxFormat format,
    const FtxCompSel compSel,
    const std::string& hint,
    const int linearFlag,
    const int initialSwizzle
)
{
    const int presetCount = static_cast<int>(s_FormatPresetInfos.size());
    for (int iPreset = 0; iPreset < presetCount; ++iPreset)
    {
        const RFormatPresetInfo& presetInfo = s_FormatPresetInfos[iPreset];
        if (presetInfo.m_Format == format)
        {
            const int presetLinearFlag = (presetInfo.m_LinearFlag != RLinearFlag::LINEAR_DEFAULT) ?
                presetInfo.m_LinearFlag : RGetDefaultLinearFlag(presetInfo.m_Format);
            if (presetInfo.m_CompSel == compSel &&
                (presetInfo.m_Hint == HintValueDefault || presetInfo.m_Hint == hint) &&
                presetLinearFlag == linearFlag &&
                (presetInfo.m_InitialSwizzle == RAddInfo::SWIZZLE_DEFAULT || presetInfo.m_InitialSwizzle == initialSwizzle))
            {
                return iPreset;
            }
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief 変換可能な全テクスチャフォーマットの配列を取得します。
//!
//! @param[out] pAllFormatCount 配列中のフォーマット数を格納します。nullptr なら格納しません。
//! @param[in] isFromRealNumber データが実数型なら true、整数型なら false を指定します。
//!
//! @return 変換可能な全テクスチャフォーマットの配列へのポインタを返します。
//-----------------------------------------------------------------------------
static const FtxFormat* GetAllFormatArray(int* pAllFormatCount, const bool isFromRealNumber)
{
    if (pAllFormatCount != nullptr)
    {
        *pAllFormatCount = (!isFromRealNumber) ?
            sizeof(s_AllIntegerFormats   ) / sizeof(s_AllIntegerFormats[0]   ) :
            sizeof(s_AllRealNumberFormats) / sizeof(s_AllRealNumberFormats[0]);
    }
    return (!isFromRealNumber) ? s_AllIntegerFormats : s_AllRealNumberFormats;
}

//-----------------------------------------------------------------------------
//! @brief 変換可能な全テクスチャフォーマットの配列から指定したフォーマットを検索します。
//!
//! @param[in] format テクスチャフォーマットです。
//! @param[in] isFromRealNumber データが実数型なら true、整数型なら false を指定します。
//!
//! @return 全フォーマットの配列中のインデックスを返します。
//!         見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindFormatFromAllFormatArray(const FtxFormat format, const bool isFromRealNumber)
{
    int allFormatCount;
    const FtxFormat* allFormats = GetAllFormatArray(&allFormatCount, isFromRealNumber);
    for (int iFormat = 0; iFormat < allFormatCount; ++iFormat)
    {
        if (allFormats[iFormat] == format)
        {
            return iFormat;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報プリセット配列に追加します。
//!
//! @param[in] hintPresets ヒント情報プリセット配列です。
//-----------------------------------------------------------------------------
static void AddHintPresets(const RStringArray& hintPresets)
{
    for (int iHint = 0; iHint < static_cast<int>(hintPresets.size()); ++iHint)
    {
        const std::string& hint = hintPresets[iHint];
        if (IsValidHint(hint) && RFindValueInArray(s_HintPresets, hint) == -1)
        {
            s_HintPresets.push_back(hint);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルからヒント情報プリセット配列を取得します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//!
//! @return 解析に成功したら true を返します。
//-----------------------------------------------------------------------------
static bool GetConfigHintPresets(const std::string& configStr)
{
    const size_t varIdx = RFindConfigVariable(configStr, "hint_presets");
    if (varIdx == std::string::npos)
    {
        return false;
    }
    RStringArray values;
    if (RGetConfigObjectValues(&values, configStr, varIdx, true) == 0)
    {
        return false;
    }
    AddHintPresets(values);
    return true;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルからフォーマットプリセットのラベルマップを取得します。
//!
//! @param[out] pFormatPresetLabelMap フォーマットプリセットのラベルマップを格納します。
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//!
//! @return 解析に成功したら true を返します。
//-----------------------------------------------------------------------------
static bool GetConfigFormatPresetLabelMap(
    RStringMap* pFormatPresetLabelMap,
    const std::string& configStr
)
{
    //-----------------------------------------------------------------------------
    // ラベルマップのオブジェクト定義を取得します。
    pFormatPresetLabelMap->clear();
    const size_t varIdx = RFindConfigVariable(configStr, "format_preset_labels");
    if (varIdx == std::string::npos)
    {
        return false;
    }
    RStringArray values;
    if (RGetConfigObjectValues(&values, configStr, varIdx, false) == 0)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // ラベルマップを作成します。
    for (size_t valueIdx = 0; valueIdx < values.size(); valueIdx += 2)
    {
        const std::string name  = RTrimString(values[valueIdx + 0]);
        const std::string label = RTrimString(values[valueIdx + 1]);
        if (!name.empty() && !label.empty())
        {
            (*pFormatPresetLabelMap)[name] = label;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief フォーマットプリセットのラベルマップを適用します。
//!
//! @param[in,out] pFormatPresetInfos フォーマットプリセット情報配列へのポインタです。
//! @param[in] formatPresetLabelMap フォーマットプリセットのラベルマップです。
//-----------------------------------------------------------------------------
static void ApplyFormatPresetLabelMap(
    RFormatPresetInfoArray* pFormatPresetInfos,
    const RStringMap& formatPresetLabelMap
)
{
    for (size_t presetIdx = 0; presetIdx < pFormatPresetInfos->size(); ++presetIdx)
    {
        RFormatPresetInfo& presetInfo = (*pFormatPresetInfos)[presetIdx];
        RStringMap::const_iterator labelMapIt = formatPresetLabelMap.find(presetInfo.m_Name);
        if (labelMapIt != formatPresetLabelMap.end())
        {
            presetInfo.m_Label = labelMapIt->second;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルを解析します。
//!
//! @param[out] pFormatPresetLabelMap フォーマットプリセットのラベルマップを格納します。
//! @param[in] globals グローバルデータです。
//! @param[in] configPath コンフィグファイルのパスです。
//! @param[in] isFromRealNumber 実数型から変換可能なフォーマットを取得するなら true、
//!                             整数型から変換可能なフォーマットを取得するなら false を指定します。
//!
//! @return 解析に成功したら true を返します。
//-----------------------------------------------------------------------------
static bool ParseConfigFile(
    RStringMap* pFormatPresetLabelMap,
    GPtr globals,
    const std::string& configPath,
    const bool isFromRealNumber
)
{
    //-----------------------------------------------------------------------------
    // コンフィグファイルをリードします。
    const std::string configStr = RReadConfigFile(configPath);

    //-----------------------------------------------------------------------------
    // 重み付け圧縮有効フラグを取得します。
    s_EnablesWeightedCompress = RGetEnablesWeightedCompress(configStr);

    //-----------------------------------------------------------------------------
    // フォーマットプリセットのラベルマップを取得します。
    GetConfigFormatPresetLabelMap(pFormatPresetLabelMap, configStr);

    //-----------------------------------------------------------------------------
    // フォーマットプリセットのオブジェクト定義を取得します。
    const size_t varIdx = RFindConfigVariable(configStr, "format_presets");
    if (varIdx == std::string::npos)
    {
        return false;
    }
    RStringArray values;
    if (RGetConfigObjectValues(&values, configStr, varIdx, false) == 0)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // フォーマットプリセット情報配列を作成します。
    for (size_t valueIdx = 0; valueIdx < values.size(); valueIdx += 2)
    {
        const std::string name = values[valueIdx];
        RStringArray words;
        const int wordCount = RSplitString(words, values[valueIdx + 1], ',');

        const std::string formatStr = RTrimString(words[0]);
        const FtxFormat format = RGetTextureFormatFromString(formatStr);

        std::string compSel = (wordCount >= 2) ? RTrimString(words[1]) : std::string();

        std::string hint    = (wordCount >= 3) ? RTrimString(words[2]) : std::string();

        int linearFlag      = (wordCount >= 4) ?
            RGetLinearFlagFromOptString(RTrimString(words[3])) : RLinearFlag::LINEAR_DEFAULT;
        if (linearFlag == RLinearFlag::LINEAR_INVALID)
        {
            linearFlag = RLinearFlag::LINEAR_DEFAULT;
        }
        if (RIsSrgbFetchTextureFormat(format)) // sRGB フェッチなら常に RGB 成分をリニア変換します。
        {
            if (linearFlag != RLinearFlag::LINEAR_DEFAULT)
            {
                linearFlag |= RLinearFlag::LINEAR_RGB;
            }
        }

        int initialSwizzle  = (wordCount >= 5) ?
            atoi(RTrimString(words[4]).c_str()) : RAddInfo::SWIZZLE_DEFAULT;
        if (initialSwizzle < RAddInfo::SWIZZLE_MIN ||
            initialSwizzle > RAddInfo::SWIZZLE_MAX)
        {
            initialSwizzle = RAddInfo::SWIZZLE_DEFAULT;
        }

        if (!name.empty()                                     &&
            format != FtxFormat_Invalid                       &&
            RIsConvertibleTextureFormat(format, isFromRealNumber))
        {
            const char* pCompSel = (IsValidCompSel(compSel)) ?
                compSel.c_str() : NULL;
            const char* pHint = (!hint.empty() && IsValidHint(hint)) ?
                hint.c_str() : NULL;
            s_FormatPresetInfos.push_back(RFormatPresetInfo(name, format,
                pCompSel, pHint, linearFlag, initialSwizzle));

            //RNoteTrace("format preset: %s: %s (%s) %x %d", name.c_str(), formatStr.c_str(), compSel.c_str(), linearFlag, initialSwizzle);
        }
    }

    //-----------------------------------------------------------------------------
    // ヒント情報プリセット配列を取得します。
    GetConfigHintPresets(configStr);

    R_UNUSED_VARIABLE(globals);
    return !s_FormatPresetInfos.empty();
}

//-----------------------------------------------------------------------------
//! @brief フォーマットプリセット情報配列をコンフィグファイルから取得します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] isFromRealNumber 実数型から変換可能なフォーマットを取得するなら true、
//!                             整数型から変換可能なフォーマットを取得するなら false を指定します。
//-----------------------------------------------------------------------------
static void GetFormatPresetInfos(GPtr globals, const bool isFromRealNumber)
{
    //-----------------------------------------------------------------------------
    // フォーマットプリセット情報配列をクリアします。
    s_FormatPresetInfos.clear();

    //-----------------------------------------------------------------------------
    // ヒント情報プリセット配列を空文字のみにします。
    s_HintPresets.clear();
    s_HintPresets.push_back("");

    //-----------------------------------------------------------------------------
    // コンフィグファイルを解析します。
    RStringMap formatPresetLabelMap;
    const std::string configFolderPath = RGetConfigFolderPath(true);
    if (!configFolderPath.empty())
    {
        // 標準のコンフィグファイルがなければサンプルのコンフィグファイルをコピーします。
        const std::string standardPath = configFolderPath + "ExportTexture.jsxinc";
        const std::string samplePath   = RGetConfigFolderPath(false) + "ExportTexture_Sample.jsxinc";

        if (!RFileExists(standardPath) && RFileExists(samplePath))
        {
            ::CopyFile(samplePath.c_str(), standardPath.c_str(), TRUE);
        }

        // 現在のコンフィグファイルを解析します。
        const std::string currentConfigPath =
            RGetConfigPathFromConfigName(s_ConfigName, configFolderPath);
        if (RFileExists(currentConfigPath))
        {
            ParseConfigFile(&formatPresetLabelMap, globals, currentConfigPath, isFromRealNumber);
        }
    }

    //-----------------------------------------------------------------------------
    // コンフィグファイルがない場合と、
    // ユーザー定義のプリセット情報に有効なものがない場合は、
    // デフォルトのプリセット情報配列を設定します。
    if (s_FormatPresetInfos.empty())
    {
        if (!isFromRealNumber)
        {
            s_FormatPresetInfos.push_back(RFormatPresetInfo("color_1bitmask", FtxFormat_Unorm_Bc1));
            s_FormatPresetInfos.push_back(RFormatPresetInfo("color_alpha"   , FtxFormat_Unorm_Bc3));
            s_FormatPresetInfos.push_back(RFormatPresetInfo("normal_map"    , FtxFormat_Snorm_Bc5));
            s_FormatPresetInfos.push_back(RFormatPresetInfo("grayscale"     , FtxFormat_Unorm_Bc4));
            //s_FormatPresetInfos.push_back(RFormatPresetInfo("表示用"        , FtxFormat_Unorm_Bc5));
            //s_FormatPresetInfos.push_back(RFormatPresetInfo("long_long_long_name_preset", FtxFormat_Unorm_Bc5));
        }
        else
        {
            s_FormatPresetInfos.push_back(RFormatPresetInfo("rgba16float", FtxFormat_Float_16_16_16_16));
            s_FormatPresetInfos.push_back(RFormatPresetInfo("rgba32float", FtxFormat_Float_32_32_32_32));
        }
    }

    //-----------------------------------------------------------------------------
    // フォーマットプリセットのラベルマップを適用します。
    ApplyFormatPresetLabelMap(&s_FormatPresetInfos, formatPresetLabelMap);

    //-----------------------------------------------------------------------------
    // フォーマットプリセットで使用されているヒント情報を追加します。
    for (int iPreset = 0; iPreset < static_cast<int>(s_FormatPresetInfos.size()); ++iPreset)
    {
        const std::string& hint = s_FormatPresetInfos[iPreset].m_Hint;
        if (hint != HintValueDefault && RFindValueInArray(s_HintPresets, hint) == -1)
        {
            s_HintPresets.push_back(hint);
        }
    }

    //-----------------------------------------------------------------------------
    // 法線用のヒント情報、現在のヒント情報がなければ強制的に追加します。
    RStringArray defaultHintPresets;
    defaultHintPresets.push_back(HintValueNormal);
    defaultHintPresets.push_back(g_AddInfo.m_Hint);
    AddHintPresets(defaultHintPresets);
}

//-----------------------------------------------------------------------------
//! @brief ダイアログの警告を表示します。
//!
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] format 書式です。
//-----------------------------------------------------------------------------
//static void ShowDialogWarning(HWND hDlg, const char* format, ...)
//{
//  char buf[1024];
//  va_list args;
//  va_start(args, format);
//  int size = vsnprintf_s(buf, _TRUNCATE, format, args);
//  va_end(args);
//  MessageBox(hDlg, buf, "Warning", MB_OK | MB_ICONWARNING);
//  R_UNUSED_VARIABLE(size);
//}

//-----------------------------------------------------------------------------
//! @brief ダイアログのリサイズの有効無効を設定します。
//-----------------------------------------------------------------------------
//static void SetResizeEnable(HWND hDlg, const bool enableFlag)
//{
//  long style = GetWindowLong(hDlg, GWL_STYLE);
//  if (enableFlag)
//  {
//      style |= WS_THICKFRAME;
//  }
//  else
//  {
//      style &= ~(long)WS_THICKFRAME;
//  }
//  SetWindowLong(hDlg, GWL_STYLE, style);
//}

//-----------------------------------------------------------------------------
//! @brief ダイアログの最大化の有効無効を設定します。
//-----------------------------------------------------------------------------
//static void SetMaximizeEnable(HWND hDlg, const bool enableFlag)
//{
//  long style = GetWindowLong(hDlg, GWL_STYLE);
//  if (enableFlag)
//  {
//      style |= WS_MAXIMIZEBOX;
//  }
//  else
//  {
//      style &= ~(long)WS_MAXIMIZEBOX;
//  }
//  SetWindowLong(hDlg, GWL_STYLE, style);
//}

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログのサイズを設定します。
//-----------------------------------------------------------------------------
static void SetFormatDialogSize(GPtr globals, HWND hDlg, const bool initPreviewScroll)
{
    //-----------------------------------------------------------------------------
    // ダイアログが最大化されていれば元に戻します。
    if (IsZoomed(hDlg))
    {
        ShowWindow(hDlg, SW_RESTORE);
    }

    //-----------------------------------------------------------------------------
    // プレビュー表示 ON なら、サイズ変更と最大化を可能にします。
    //SetResizeEnable(hDlg, globals->m_DisplaysPreview);
    //SetMaximizeEnable(hDlg, globals->m_DisplaysPreview);

    //-----------------------------------------------------------------------------
    // ダイアログサイズを設定します。
    RECT wRect;
    GetWindowRect(hDlg, &wRect);

    const int winW = (!globals->m_DisplaysPreview) ?
        s_FormatDialogNoPreviewW : s_FormatDialogPreviewMinW;
    const int winH = (!globals->m_DisplaysPreview) ?
        s_FormatDialogNoPreviewH : s_FormatDialogPreviewMinH;

    MoveWindow(hDlg, wRect.left, wRect.top, winW, winH, TRUE);

    //-----------------------------------------------------------------------------
    // set preview area
    if (globals->m_DisplaysPreview)
    {
        //s_Preview.SetAreaForDialog(hDlg);
        if (initPreviewScroll)
        {
            s_Preview.CenterScroll();
        }
        s_Preview.ClampScrollValue();
    }
}

//-----------------------------------------------------------------------------
//! @brief 成分選択のリストを更新します。
//-----------------------------------------------------------------------------
static void UpdateCompSelList(GPtr globals, HWND hDlg)
{
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        int iComp = (g_AddInfo.m_CompSel >> (3 - iRgba) * 8) & 0xff;
        iComp = (iComp < FtxComponent_Count) ? iComp : iRgba;
        HWND compSelLst = GetDlgItem(hDlg, s_CompSelControls[iRgba]);
        SendMessage(compSelLst, CB_SETCURSEL, iComp, 0L);
    }
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報のリストを更新します。
//-----------------------------------------------------------------------------
static void UpdateHintList(GPtr globals, HWND hDlg)
{
    int iHint = RFindValueInArray(s_HintPresets, g_AddInfo.m_Hint);
    if (iHint == -1)
    {
        iHint = 0;
        g_AddInfo.m_Hint = s_HintPresets[iHint];
    }
    SendMessage(GetDlgItem(hDlg, IDC_HINT_LST), CB_SETCURSEL, iHint, 0L);
}

//-----------------------------------------------------------------------------
//! @brief 初期スウィズル値のリストを更新します。
//-----------------------------------------------------------------------------
static void UpdateInitialSwizzleList(GPtr globals, HWND hDlg)
{
    SendMessage(GetDlgItem(hDlg, IDC_SWIZZLE_LST), CB_SETCURSEL,
        g_AddInfo.m_InitialSwizzle, 0L);
}

//-----------------------------------------------------------------------------
//! @brief データサイズ表示を更新します。
//-----------------------------------------------------------------------------
static void UpdateDataSize(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // フェースの幅・高さ・深さを求めます。
    int imageW = gStuff->imageSize32.h;
    int imageH = gStuff->imageSize32.v;
    int imageD = 1;

    if (g_AddInfo.m_Dimension == FtxDimension_CubeMap)
    {
        imageW = g_ImageStatus.m_CubeFaceW;
        imageH = g_ImageStatus.m_CubeFaceH;
        imageD = RImage::CUBE_FACE_COUNT;
    }
    else if (g_AddInfo.m_Dimension == FtxDimension_3d      ||
             g_AddInfo.m_Dimension == FtxDimension_1dArray ||
             g_AddInfo.m_Dimension == FtxDimension_2dArray)
    {
        imageW = g_ImageStatus.m_3DFaceW;
        imageH = g_ImageStatus.m_3DFaceH;
        imageD = g_ImageStatus.m_3DFaceD;
    }

    if (g_AddInfo.m_Dimension == FtxDimension_1d      ||
        g_AddInfo.m_Dimension == FtxDimension_1dArray)
    {
        imageH = 1;
    }

    //-----------------------------------------------------------------------------
    // ミップマップレベル数を求めます。
    const int maxMipLevel = RImage::GetMaxMipLevel(imageW, imageH);
    const int mipLevel = (g_AddInfo.m_MipLevel < maxMipLevel) ?
        g_AddInfo.m_MipLevel : maxMipLevel;

    //-----------------------------------------------------------------------------
    // データサイズを求めます。
    const int dataSize = static_cast<int>(g_AddInfo.GetDataSize(imageW, imageH, imageD));

    //-----------------------------------------------------------------------------
    // データサイズのテキストを設定します。
    const int KBYTES = 1024;
    const int sizeKB = (dataSize > KBYTES) ?
        static_cast<int>(ceil(static_cast<float>(dataSize) / KBYTES)) : 1;

    std::string sizeStr = RGetNumberString(sizeKB, "%5d") + " KB ( " +
        RGetNumberString(imageW) + " x " + RGetNumberString(imageH);
    if (imageD >= 2) sizeStr += " x " + RGetNumberString(imageD);
    sizeStr += ", " + RGetNumberString(mipLevel) + " level";
    if (mipLevel >= 2) sizeStr += "s";
    sizeStr += ", " + RGetBitsPerPixelString(g_AddInfo.m_Format) + " bits/pixel";
    sizeStr += ", " + RGetNumberString(dataSize) + " bytes )";

    SetDlgItemText(hDlg, IDC_DATA_SIZE, sizeStr.c_str());
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグ表示文字列を取得します。
//-----------------------------------------------------------------------------
static std::string GetLinearFlagDisplayString(const int linearFlag)
{
    static const int linearRgbaFlags[R_RGBA_COUNT] =
    {
        RLinearFlag::LINEAR_R,
        RLinearFlag::LINEAR_G,
        RLinearFlag::LINEAR_B,
        RLinearFlag::LINEAR_A,
    };
    static const std::string rgbaStr = "rgba";

    std::string str;
    if (linearFlag > 0)
    {
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            if ((linearFlag & linearRgbaFlags[iRgba]) != 0)
            {
                str += rgbaStr[iRgba];
            }
        }
    }
    return str;
}

//-----------------------------------------------------------------------------
//! @brief フォーマット変更時のコールバックです。
//-----------------------------------------------------------------------------
static void FormatDialogFormatCB(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // format preset
    bool hintEnable = true;
    bool swizzleEnable = true;

    HWND formatLst = GetDlgItem(hDlg, IDC_FORMAT_LST);
    EnableWindow(GetDlgItem(hDlg, IDC_FORMAT_LBL), !s_IsAllFormat);
    EnableWindow(formatLst, !s_IsAllFormat);
    if (s_IsAllFormat)
    {
        SendMessage(formatLst, LB_SETCURSEL, static_cast<WPARAM>(-1), 0L);
    }
    else
    {
        const RFormatPresetInfo& presetInfo = s_FormatPresetInfos[s_FormatPresetIdx];
        hintEnable    = (presetInfo.m_Hint == HintValueDefault);
        swizzleEnable = (presetInfo.m_InitialSwizzle == RAddInfo::SWIZZLE_DEFAULT);
    }

    //-----------------------------------------------------------------------------
    // all format
    EnableWindow(GetDlgItem(hDlg, IDC_ALL_FORMAT_LST), s_IsAllFormat);

    //-----------------------------------------------------------------------------
    // weighted compress
    EnableWindow(GetDlgItem(hDlg, IDC_WEIGHT_CMPR),
        (RIsBc123TextureFormat(g_AddInfo.m_Format) || RIsEtcTextureFormat(g_AddInfo.m_Format)) &&
        s_EnablesWeightedCompress);

    //-----------------------------------------------------------------------------
    // ミップマップ生成フィルタ
    const bool generatesMip = (g_AddInfo.m_MipLevel >= 2);
    EnableWindow(GetDlgItem(hDlg, IDC_MIP_GEN_FILTER_LBL), generatesMip);
    EnableWindow(GetDlgItem(hDlg, IDC_MIP_GEN_FILTER_LST), generatesMip);

    //-----------------------------------------------------------------------------
    // comp sel
    UpdateCompSelList(globals, hDlg);
    EnableWindow(GetDlgItem(hDlg, IDC_COMP_SEL_LBL), s_IsAllFormat);
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        EnableWindow(GetDlgItem(hDlg, s_CompSelControls[iRgba]), s_IsAllFormat);
    }

    //-----------------------------------------------------------------------------
    // ヒント情報
    UpdateHintList(globals, hDlg);
    EnableWindow(GetDlgItem(hDlg, IDC_HINT_LBL), hintEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_HINT_LST), hintEnable);

    //-----------------------------------------------------------------------------
    // リニア変換フラグ
    SetDlgItemText(hDlg, IDC_LINEAR_TXT,
        GetLinearFlagDisplayString(g_AddInfo.m_LinearFlag).c_str());
    //EnableWindow(GetDlgItem(hDlg, IDC_LINEAR_LBL), !s_IsAllFormat);
    //EnableWindow(GetDlgItem(hDlg, IDC_LINEAR_TXT), !s_IsAllFormat);

    //-----------------------------------------------------------------------------
    // 初期スウィズル値
    UpdateInitialSwizzleList(globals, hDlg);
    EnableWindow(GetDlgItem(hDlg, IDC_SWIZZLE_LBL), swizzleEnable);
    EnableWindow(GetDlgItem(hDlg, IDC_SWIZZLE_LST), swizzleEnable);

    //-----------------------------------------------------------------------------
    // data size
    UpdateDataSize(globals, hDlg);
}

//-----------------------------------------------------------------------------
//! @brief フォーマットプリセットを付加情報に適用します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] presetInfo フォーマットプリセット情報です。
//-----------------------------------------------------------------------------
static void ApplyFormatPresetToAddInfo(GPtr globals, const RFormatPresetInfo& presetInfo)
{
    g_AddInfo.m_Format = presetInfo.m_Format;
    g_AddInfo.m_CompSel = presetInfo.m_CompSel;
    g_AddInfo.m_Hint = (presetInfo.m_Hint != HintValueDefault) ?
        presetInfo.m_Hint : RGetDefaultHint(presetInfo.m_Format);
    g_AddInfo.m_LinearFlag = (presetInfo.m_LinearFlag != RLinearFlag::LINEAR_DEFAULT) ?
        presetInfo.m_LinearFlag : RGetDefaultLinearFlag(presetInfo.m_Format);

    g_AddInfo.m_InitialSwizzle = (presetInfo.m_InitialSwizzle != RAddInfo::SWIZZLE_DEFAULT) ?
        presetInfo.m_InitialSwizzle : 0;
}

//-----------------------------------------------------------------------------
//! @brief フォーマットプリセット選択時のコールバックです。
//-----------------------------------------------------------------------------
static void FormatDialogFormatPresetCB(GPtr globals, HWND hDlg)
{
    HWND formatLst = GetDlgItem(hDlg, IDC_FORMAT_LST);
    const int iPreset = static_cast<int>(SendMessage(formatLst, LB_GETCURSEL, 0, 0));
    if (iPreset == -1)
    {
        SendMessage(formatLst, LB_SETCURSEL, s_FormatPresetIdx, 0L);
    }
    else
    {
        s_FormatPresetIdx = iPreset;
    }
    const RFormatPresetInfo& presetInfo = s_FormatPresetInfos[s_FormatPresetIdx];
    ApplyFormatPresetToAddInfo(globals, presetInfo);

    // 全テクスチャフォーマットリストの選択位置を同期します。
    const bool isFromRealNumber = (gStuff->depth != 8);
    const int iCurFormat = FindFormatFromAllFormatArray(g_AddInfo.m_Format, isFromRealNumber);
    if (iCurFormat != -1)
    {
        SendMessage(GetDlgItem(hDlg, IDC_ALL_FORMAT_LST), CB_SETCURSEL, iCurFormat, 0L);
    }

    FormatDialogFormatCB(globals, hDlg);
}

//-----------------------------------------------------------------------------
//! @brief 全フォーマットリスト選択時のコールバックです。
//-----------------------------------------------------------------------------
static void FormatDialogAllFormatCB(GPtr globals, HWND hDlg)
{
    const bool isFromRealNumber = (gStuff->depth != 8);
    const FtxFormat* allFormats = GetAllFormatArray(nullptr, isFromRealNumber);
    const int iFormat = static_cast<int>(
        SendMessage(GetDlgItem(hDlg, IDC_ALL_FORMAT_LST), CB_GETCURSEL, 0, 0));
    g_AddInfo.m_Format  = allFormats[iFormat];
    g_AddInfo.m_CompSel = RGetDefaultCompSel(g_AddInfo.m_Format, g_AddInfo.m_Hint);
    g_AddInfo.m_LinearFlag = RGetDefaultLinearFlag(g_AddInfo.m_Format);

    // フォーマットプリセットリストの選択位置を同期します。
    const int iCurPreset = FindFormatPresetInfoByFormat(g_AddInfo.m_Format,
        g_AddInfo.m_CompSel, g_AddInfo.m_Hint, g_AddInfo.m_LinearFlag, g_AddInfo.m_InitialSwizzle);
    if (iCurPreset != -1)
    {
        s_FormatPresetIdx = iCurPreset;
    }

    FormatDialogFormatCB(globals, hDlg);

    UpdatePreviewBitmap(globals, hDlg);
}

//-----------------------------------------------------------------------------
//! @brief カスタム CenterDialog です。
//!        通常より左寄りにします。
//-----------------------------------------------------------------------------
//static void MyCenterDialog(HWND hDlg)
//{
//  int  nHeight;
//    int  nWidth;
//    int  nTitleBits;
//    RECT rcDialog;
//    RECT rcParent;
//    int  xOrigin;
//    int  yOrigin;
//    int  xScreen;
//    int  yScreen;
//    HWND hParent = GetParent(hDlg);
//
//    if  (hParent == NULL)
//        hParent = GetDesktopWindow();
//
//    GetClientRect(hParent, &rcParent);
//    ClientToScreen(hParent, (LPPOINT)&rcParent.left);
//    ClientToScreen(hParent, (LPPOINT)&rcParent.right);
//
//    // Center on Title: title bar has system menu, minimize,  maximize bitmaps
//    // Width of title bar bitmaps - assumes 3 of them and dialog has a sysmenu
//    nTitleBits = GetSystemMetrics(SM_CXSIZE);
//
//    // If dialog has no sys menu compensate for odd# bitmaps by sub 1 bitwidth
//  if (!(GetWindowLong(hDlg, GWL_STYLE) & WS_SYSMENU))
//  {
//      nTitleBits -= nTitleBits / 3;
//  }
//
//    GetWindowRect(hDlg, &rcDialog);
//    nWidth  = rcDialog.right  - rcDialog.left;
//    nHeight = rcDialog.bottom - rcDialog.top;
//
//    xOrigin = max(rcParent.right - rcParent.left - nWidth, 0) / 3 // changed
//            + rcParent.left - nTitleBits;
//    xScreen = GetSystemMetrics(SM_CXSCREEN);
//    if  (xOrigin + nWidth > xScreen)
//        xOrigin = max (0, xScreen - nWidth);
//
//  yOrigin = max(rcParent.bottom - rcParent.top - nHeight, 0) / 3
//            + rcParent.top;
//    yScreen = GetSystemMetrics(SM_CYSCREEN);
//    if  (yOrigin + nHeight > yScreen)
//        yOrigin = max(0 , yScreen - nHeight);
//
//    SetWindowPos(hDlg, NULL, xOrigin, yOrigin, nWidth, nHeight, SWP_NOZORDER);
//}

//-----------------------------------------------------------------------------
//! @brief 指定した数の半角スペースを含む文字列を返します。
//-----------------------------------------------------------------------------
static std::string GetSpaceString(const int count)
{
    static const std::string space10 = "          ";

    if (count <= 10)
    {
        return space10.substr(0, count);
    }
    else
    {
        std::string str;
        for (int ic = 0; ic < count; ++ic) str += " ";
        return str;
    }
}

//-----------------------------------------------------------------------------
//! @brief 必要ならリストボックスに水平スクロールバーを表示します。
//!        [参考] http://win32lab.com/tips/tips13.html
//!
//! @param[in] hLst リストボックスのハンドルです。
//! @param[in] hFont フォントのハンドルです。
//! @param[in] text 一番幅のある文字列を指定します。
//-----------------------------------------------------------------------------
static void SetHScrollBarForListBox(HWND hLst, HFONT hFont, const std::string& text)
{
    // デバイスコンテキストを取得してフォントを設定します。
    HDC hDc = GetDC(hLst);
    int iSave = SaveDC(hDc);
    SelectObject(hDc, hFont);

    // 文字列の幅を取得します。
    SIZE size;
    GetTextExtentPoint32(hDc, text.c_str(), static_cast<int>(text.size()), &size);
    UINT iListWidth = size.cx + GetSystemMetrics(SM_CXVSCROLL);

    // リストボックス内部の水平方向の広がりを設定します。
    SendMessage(hLst, LB_SETHORIZONTALEXTENT, (WPARAM)iListWidth, 0L);

    // デバイスコンテキストを元に戻します。
    RestoreDC(hDc, iSave);
    ReleaseDC(hLst, hDc);
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログのフォーマットプリセットリストを
//!        再構築します。
//!
//! @param[in] formatLst フォーマットリストのハンドルです。
//-----------------------------------------------------------------------------
static void RebuildFormatPresetList(HWND formatLst)
{
    //-----------------------------------------------------------------------------
    // プリセットのラベルとフォーマット名の最大長を取得します。
    int maxPresetLabelLen  = 0;
    int maxPresetFormatLen = 0;
    int maxPresetBpsLen    = 0;
    const int presetCount = static_cast<int>(s_FormatPresetInfos.size());
    for (int iPreset = 0; iPreset < presetCount; ++iPreset)
    {
        const RFormatPresetInfo& presetInfo = s_FormatPresetInfos[iPreset];
        const int labelLen  = static_cast<int>(presetInfo.m_Label.size());
        const int formatLen = static_cast<int>(RGetTextureFormatString(presetInfo.m_Format).size());
        const int bpsLen    = static_cast<int>(RGetBitsPerPixelString(presetInfo.m_Format).size());
        maxPresetLabelLen  = RMax(labelLen, maxPresetLabelLen);
        maxPresetFormatLen = RMax(formatLen, maxPresetFormatLen);
        maxPresetBpsLen    = RMax(bpsLen, maxPresetBpsLen);
    }

    //-----------------------------------------------------------------------------
    // プリセットのリストを空にします。
    while (SendMessage(formatLst, LB_GETCOUNT, 0, 0) > 0)
    {
        if (SendMessage(formatLst, LB_DELETESTRING, 0, 0) == LB_ERR)
        {
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // プリセットのリストに値を追加します。
    std::string maxLenItem;
    for (int iPreset = 0; iPreset < presetCount; ++iPreset)
    {
        const RFormatPresetInfo& presetInfo = s_FormatPresetInfos[iPreset];
        const std::string formatName = RGetTextureFormatString(presetInfo.m_Format);
        const std::string bpsStr = RGetBitsPerPixelString(presetInfo.m_Format);
        const std::string item = presetInfo.m_Label +
            GetSpaceString(maxPresetLabelLen - static_cast<int>(presetInfo.m_Label.size())) +
            " [" + formatName +
            GetSpaceString(maxPresetFormatLen - static_cast<int>(formatName.size())) +
            " (" + GetSpaceString(maxPresetBpsLen - static_cast<int>(bpsStr.size())) +
            bpsStr + " bits/pixel, " + RGetCompSelOptionString(presetInfo.m_CompSel) + ")]";
        if (item.size() > maxLenItem.size())
        {
            maxLenItem = item;
        }
        SendMessage(formatLst, LB_ADDSTRING, 0, reinterpret_cast<LPARAM>(item.c_str()));
    }

    // 必要なら水平スクロールバーを表示します。
    SetHScrollBarForListBox(formatLst, s_FormatFont, maxLenItem);
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報のリストを再構築します。
//-----------------------------------------------------------------------------
static void RebuildHintList(HWND hintLst)
{
    SendMessage(hintLst, CB_RESETCONTENT, 0, 0);
    for (int iHint = 0; iHint < static_cast<int>(s_HintPresets.size()); ++iHint)
    {
        SendMessage(hintLst, CB_ADDSTRING, 0,
            reinterpret_cast<LPARAM>(s_HintPresets[iHint].c_str()));
    }
}

//-----------------------------------------------------------------------------
//! @brief 重み付け圧縮を更新します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void UpdateWeightedCompress(GPtr globals, HWND hDlg)
{
    if (!s_EnablesWeightedCompress)
    {
        g_AddInfo.m_WeightedCompress = false;
    }
    CheckDlgButton(hDlg, IDC_WEIGHT_CMPR, (g_AddInfo.m_WeightedCompress) ? BST_CHECKED : BST_UNCHECKED);
}

//-----------------------------------------------------------------------------
//! @brief コンフィグ名選択時のコールバックです。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void ConfigChangedCB(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // 以前のプリセット名を記憶します。
    const std::string lastPresetName =
        s_FormatPresetInfos[s_FormatPresetIdx].m_Name;

    //-----------------------------------------------------------------------------
    // 現在のコンフィグ名を更新します。
    HWND configNameLst = GetDlgItem(hDlg, IDC_CONFIG_NAME_LST);
    const int iConfig = static_cast<int>(SendMessage(configNameLst, CB_GETCURSEL, 0, 0));
    s_ConfigName = s_AllConfigNames[iConfig];

    //-----------------------------------------------------------------------------
    // フォーマットプリセット情報配列をコンフィグファイルから取得します。
    const bool isFromRealNumber = (gStuff->depth != 8);
    GetFormatPresetInfos(globals, isFromRealNumber);

    //-----------------------------------------------------------------------------
    // フォーマットプリセットリストを再構築します。
    HWND formatLst = GetDlgItem(hDlg, IDC_FORMAT_LST);
    RebuildFormatPresetList(formatLst);

    //-----------------------------------------------------------------------------
    // ヒント情報のリストを再構築します。
    HWND hintLst = GetDlgItem(hDlg, IDC_HINT_LST);
    RebuildHintList(hintLst);

    //-----------------------------------------------------------------------------
    // 以前のプリセット名に対応するプリセットがあればそれを、
    // なければ現在のテクスチャフォーマット（と成分選択）に対応するプリセットを
    // 選択し、プリセットの成分選択を適用します。
    int iCurPreset = FindFormatPresetInfoByName(lastPresetName);
    if (iCurPreset == -1)
    {
        iCurPreset = FindFormatPresetInfoByFormat(g_AddInfo.m_Format,
            g_AddInfo.m_CompSel, g_AddInfo.m_Hint, g_AddInfo.m_LinearFlag, g_AddInfo.m_InitialSwizzle);
    }
    s_FormatPresetIdx = (iCurPreset != -1) ? iCurPreset : 0;
    if (!s_IsAllFormat)
    {
        SendMessage(formatLst, LB_SETCURSEL, s_FormatPresetIdx, 0L);
        FormatDialogFormatPresetCB(globals, hDlg);
    }
    else
    {
        FormatDialogAllFormatCB(globals, hDlg);
    }

    //-----------------------------------------------------------------------------
    // 重み付け圧縮を更新します。
    UpdateWeightedCompress(globals, hDlg);
}

//-----------------------------------------------------------------------------
//! @brief プレビューコントロールの可視性を設定します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void SetPreviewControlVisibility(GPtr globals, HWND hDlg)
{
    const int showCmd = (globals->m_DisplaysPreview) ? SW_SHOW : SW_HIDE;
    ShowWindow(GetDlgItem(hDlg, IDC_PREVIEW_RGB  ), showCmd);
    ShowWindow(GetDlgItem(hDlg, IDC_PREVIEW_ALPHA), showCmd);
    ShowWindow(GetDlgItem(hDlg, IDC_ZOOM_OUT     ), showCmd);
    ShowWindow(GetDlgItem(hDlg, IDC_ZOOM_IN      ), showCmd);
    ShowWindow(GetDlgItem(hDlg, IDC_ZOOM_VALUE   ), showCmd);
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログを初期化します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void InitFormatDialog(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // ダイアログのサイズを取得します。
    RECT dialogRect;
    GetWindowRect(hDlg, &dialogRect);

    // プレビューなしの場合 OK ボタン右端 + α の幅にします。
    RECT rect;
    GetWindowRect(GetDlgItem(hDlg, IDOK), &rect);
    s_FormatDialogNoPreviewW = rect.right - dialogRect.left + 16;

    // プレビューありの場合プレビューエリア右端 + α の幅にします。
    GetWindowRect(GetDlgItem(hDlg, IDC_PREVIEW_AREA), &rect);
    s_FormatDialogPreviewMinW = rect.right - dialogRect.left + 16;
    //s_FormatDialogNoPreviewW = s_FormatDialogPreviewMinW; // プレビューなしも同じ幅にするテストです。

    GetWindowRect(GetDlgItem(hDlg, IDC_DATA_SIZE_GRP), &rect);
    s_FormatDialogNoPreviewH  = rect.bottom - dialogRect.top + 16;
    s_FormatDialogPreviewMinH = s_FormatDialogNoPreviewH;

    //-----------------------------------------------------------------------------
    // get control position for resize
    RECT cRect;
    GetClientRect(hDlg, &cRect);

    GetWindowRect(GetDlgItem(hDlg, IDC_PREVIEW_RGB), &rect);
    ScreenToClient(hDlg, reinterpret_cast<LPPOINT>(&rect.left));
    s_PreviewRgbPosOfsY      = cRect.bottom - rect.top;

    GetWindowRect(GetDlgItem(hDlg, IDC_ZOOM_OUT), &rect);
    ScreenToClient(hDlg, reinterpret_cast<LPPOINT>(&rect.left));
    s_PreviewZoomBtnPosOfsY  = cRect.bottom - rect.top;

    GetWindowRect(GetDlgItem(hDlg, IDC_ZOOM_VALUE), &rect);
    ScreenToClient(hDlg, reinterpret_cast<LPPOINT>(&rect.left));
    s_PreviewZoomTextPosOfsY = cRect.bottom - rect.top;

    //-----------------------------------------------------------------------------
    // center dialog
    // WM_GETMINMAXINFO, WM_SIZE が発生
    CenterDialog(hDlg);
    //MyCenterDialog(hDlg);

    //-----------------------------------------------------------------------------
    // config
    RGetAllConfigNames(s_AllConfigNames);
    HWND configNameLst = GetDlgItem(hDlg, IDC_CONFIG_NAME_LST);
    const int configCount = static_cast<int>(s_AllConfigNames.size());
    for (int iConfig = 0; iConfig < configCount; ++iConfig)
    {
        SendMessage(configNameLst, CB_ADDSTRING, 0,
            reinterpret_cast<LPARAM>(s_AllConfigNames[iConfig].c_str()));
    }

    s_ConfigName = (g_AddInfo.m_ConfigName.empty()) ?
        R_STANDARD_CONFIG_NAME : g_AddInfo.m_ConfigName;
    int iCurConfig = RFindNoCaseStringInArray(s_AllConfigNames, s_ConfigName);
    if (iCurConfig == -1)
    {
        iCurConfig = 0;
    }
    s_ConfigName = s_AllConfigNames[iCurConfig];
    SendMessage(configNameLst, CB_SETCURSEL, iCurConfig, 0L);

    const bool configNameEnable = (configCount >= 2);
    EnableWindow(GetDlgItem(hDlg, IDC_CONFIG_NAME_LBL), configNameEnable);
    EnableWindow(configNameLst, configNameEnable);

    //-----------------------------------------------------------------------------
    // フォーマットプリセットリスト

    // 等幅フォントを設定します。
    LOGFONT logfont;
    ZeroMemory(&logfont, sizeof(logfont));
    logfont.lfHeight = 14;
    //logfont.lfUnderline = TRUE;
    logfont.lfCharSet = DEFAULT_CHARSET;
    strcpy_s(logfont.lfFaceName, "MS Gothic");
    s_FormatFont = CreateFontIndirect(&logfont);

    HWND formatLst = GetDlgItem(hDlg, IDC_FORMAT_LST);
    SendMessage(formatLst, WM_SETFONT, (WPARAM)s_FormatFont, 0);

    // フォーマットプリセット情報配列をコンフィグファイルから取得します。
    const bool isFromRealNumber = (gStuff->depth != 8);
    GetFormatPresetInfos(globals, isFromRealNumber);

    // フォーマットプリセットリストを再構築します。
    RebuildFormatPresetList(formatLst);

    // フォーマットが未確定ならフォーマットプリセットの先頭のフォーマットを適用します。
    if (!g_ImageStatus.m_IsFormatDecided)
    {
        ApplyFormatPresetToAddInfo(globals, s_FormatPresetInfos[0]);
        g_ImageStatus.m_IsFormatDecided = true;
    }

    // 現在のフォーマット（と成分選択）に対応するプリセットを選択します。
    const int iCurPreset = FindFormatPresetInfoByFormat(g_AddInfo.m_Format,
        g_AddInfo.m_CompSel, g_AddInfo.m_Hint, g_AddInfo.m_LinearFlag, g_AddInfo.m_InitialSwizzle);
    if (iCurPreset != -1)
    {
        SendMessage(formatLst, LB_SETCURSEL, iCurPreset, 0L);
    }
    s_FormatPresetIdx = (iCurPreset != -1) ? iCurPreset : 0;

    //-----------------------------------------------------------------------------
    // 全フォーマットリスト
    s_IsAllFormat = (iCurPreset == -1);
    CheckDlgButton(hDlg, IDC_ALL_FORMAT, s_IsAllFormat ? BST_CHECKED : BST_UNCHECKED);

    HWND allFormatLst = GetDlgItem(hDlg, IDC_ALL_FORMAT_LST);
    int allFormatCount;
    const FtxFormat* allFormats = GetAllFormatArray(&allFormatCount, isFromRealNumber);
    for (int iFormat = 0; iFormat < allFormatCount; ++iFormat)
    {
        const FtxFormat format = allFormats[iFormat];
        SendMessage(allFormatLst, CB_ADDSTRING, 0,
            reinterpret_cast<LPARAM>(RGetTextureFormatString(format).c_str()));
    }
    int iCurFormat = FindFormatFromAllFormatArray(g_AddInfo.m_Format, isFromRealNumber);
    if (iCurFormat == -1)
    {
        iCurFormat = 0;
    }
    SendMessage(allFormatLst, CB_SETCURSEL, iCurFormat, 0L);

    //-----------------------------------------------------------------------------
    // weighted compress
    UpdateWeightedCompress(globals, hDlg);

    //-----------------------------------------------------------------------------
    // dimension
    HWND dimensionLst = GetDlgItem(hDlg, IDC_DIMENSION_LST);
    s_Dimensions.clear();
    int iCurDimension = 1; // 2d
    const int allDimCount = sizeof(s_AllDimensions) / sizeof(s_AllDimensions[0]);
    for (int iAllDim = 0; iAllDim < allDimCount; ++iAllDim)
    {
        const FtxDimension dimension = s_AllDimensions[iAllDim];
        if ((dimension == FtxDimension_1d      && !g_ImageStatus.m_Is1DEnable  ) ||
            (dimension == FtxDimension_2d      && !g_ImageStatus.m_Is2DEnable  ) ||
            (dimension == FtxDimension_3d      && !g_ImageStatus.m_Is3DEnable  ) ||
            (dimension == FtxDimension_CubeMap && !g_ImageStatus.IsCubeEnable()) ||
            (dimension == FtxDimension_1dArray && !g_ImageStatus.m_Is3DEnable  ) ||
            (dimension == FtxDimension_2dArray && !g_ImageStatus.m_Is3DEnable  ))
        {
            continue;
        }
        SendMessage(dimensionLst, CB_ADDSTRING, 0,
            reinterpret_cast<LPARAM>(RGetTextureDimensionString(dimension).c_str()));
        if (dimension == g_AddInfo.m_Dimension)
        {
            iCurDimension = static_cast<int>(s_Dimensions.size());
        }
        s_Dimensions.push_back(dimension);
    }

    if (iCurDimension >= static_cast<int>(s_Dimensions.size()))
    {
        iCurDimension = 0;
    }
    SendMessage(dimensionLst, CB_SETCURSEL, iCurDimension, 0L);

    //-----------------------------------------------------------------------------
    // ミップマップ
    CheckDlgButton(hDlg, IDC_MIPMAP, (g_AddInfo.m_MipLevel >= 2) ? BST_CHECKED : BST_UNCHECKED);

    HWND mipGenFilterLst = GetDlgItem(hDlg, IDC_MIP_GEN_FILTER_LST);
    for (int iMipGenFilter = 0; iMipGenFilter < RMipGenFilter::MIP_GEN_FILTER_COUNT; ++iMipGenFilter)
    {
        SendMessage(mipGenFilterLst, CB_ADDSTRING, 0,
            reinterpret_cast<LPARAM>(RGetMipGenFilterString(
            static_cast<RMipGenFilter::MipGenFilter>(iMipGenFilter)).c_str()));
    }
    SendMessage(mipGenFilterLst, CB_SETCURSEL, g_AddInfo.m_MipGenFilter, 0L);

    //-----------------------------------------------------------------------------
    // comp sel
    static const char* const compSelItems[FtxComponent_Count] =
    {
        "r", "g", "b", "a", "0", "1"
    };

    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        HWND compSelLst = GetDlgItem(hDlg, s_CompSelControls[iRgba]);
        for (int iComp = 0; iComp < FtxComponent_Count; ++iComp)
        {
            SendMessage(compSelLst, CB_ADDSTRING, 0,
                reinterpret_cast<LPARAM>(compSelItems[iComp]));
        }
    }
    UpdateCompSelList(globals, hDlg);

    //-----------------------------------------------------------------------------
    // ヒント情報
    HWND hintLst = GetDlgItem(hDlg, IDC_HINT_LST);
    RebuildHintList(hintLst);
    UpdateHintList(globals, hDlg);

    //-----------------------------------------------------------------------------
    // リニア変換フラグ
    if (s_IsAllFormat)
    {
        g_AddInfo.m_LinearFlag = RGetDefaultLinearFlag(g_AddInfo.m_Format);
    }

    //-----------------------------------------------------------------------------
    // 初期スウィズル値
    HWND swizzleLst = GetDlgItem(hDlg, IDC_SWIZZLE_LST);
    for (int swizzle = RAddInfo::SWIZZLE_MIN; swizzle <= RAddInfo::SWIZZLE_MAX; ++swizzle)
    {
        SendMessage(swizzleLst, CB_ADDSTRING, 0,
            reinterpret_cast<LPARAM>(RGetNumberString(swizzle).c_str()));
    }
    UpdateInitialSwizzleList(globals, hDlg);

    //-----------------------------------------------------------------------------
    // 編集用コメント文
    SetDlgItemText(hDlg, IDC_COMMENT_TXT, g_AddInfo.m_CommentText.c_str());

    //-----------------------------------------------------------------------------
    // last format
    //const std::string lastFormat = (g_ImageStatus.m_LastFormat != FtxFormat_Invalid) ?
    //  RGetTextureFormatString(g_ImageStatus.m_LastFormat) : std::string("---");
    //SetWindowText(GetDlgItem(hDlg, IDC_LAST_FORMAT), lastFormat.c_str());

    //-----------------------------------------------------------------------------
    // preview
    #if 0
    static const int PREVIEW_W = 256 + 2;
    static const int PREVIEW_H = 256 + 2;

    RECT previewRect;
    GetWindowRect(GetDlgItem(hDlg, IDC_PREVIEW_AREA), &previewRect);
    ScreenToClient(hDlg, reinterpret_cast<LPPOINT>(&previewRect.left));
    previewRect.right  = previewRect.left + PREVIEW_W;
    previewRect.bottom = previewRect.top  + PREVIEW_H;

    s_Preview.InitMemory(hDlg, previewRect, gStuff->imageSize32.h, gStuff->imageSize32.v);
    s_Preview.CenterScroll();
    s_Preview.SetControlId(IDC_ZOOM_IN, IDC_ZOOM_OUT, IDC_ZOOM_VALUE);

    UpdatePreviewBitmap(globals, hDlg);

    // preview flag
    CheckDlgButton(hDlg, IDC_PREVIEW_FLAG,
        (globals->m_DisplaysPreview) ? BST_CHECKED : BST_UNCHECKED);

    // preview rgb & alpha
    CheckDlgButton(hDlg, IDC_PREVIEW_RGB,
        (s_Preview.m_DisplaysRgb) ? BST_CHECKED : BST_UNCHECKED);
    CheckDlgButton(hDlg, IDC_PREVIEW_ALPHA,
        (s_Preview.m_DisplaysAlpha) ? BST_CHECKED : BST_UNCHECKED);

    #else
    globals->m_DisplaysPreview = false;
    ShowWindow(GetDlgItem(hDlg, IDC_PREVIEW_FLAG), SW_HIDE);
    #endif

    // visibility
    SetPreviewControlVisibility(globals, hDlg);

    //-----------------------------------------------------------------------------
    // set dialog size
    SetFormatDialogSize(globals, hDlg, true);

    //-----------------------------------------------------------------------------
    // update control state
    FormatDialogFormatCB(globals, hDlg);

    //-----------------------------------------------------------------------------
    // focus OK button
    SetFocus(GetDlgItem(hDlg, IDOK));

    //-----------------------------------------------------------------------------
    // 各 UI にツールチップを追加します。
    const bool isJP = RIsJapaneseUI();
    RAddToolTip(hDlg, IDOK, (isJP) ? "セーブします" : "Saves");
    RAddToolTip(hDlg, IDCANCEL, (isJP) ? "キャンセルします" : "Cancels");

    const char* pConfigTip = (isJP) ? "使用するコンフィグ名を選択します" : "Selects the config name to be used";
    RAddToolTip(hDlg, IDC_CONFIG_NAME_LBL, pConfigTip);
    RAddToolTip(hDlg, IDC_CONFIG_NAME_LST, pConfigTip);

    const char* pFormatTip = (isJP) ?
        "フォーマットを指定します。\n選択可能な項目はコンフィグファイルで設定できます" :
        "Specifies the format. \nThe formats that can be selected here can be set in a config file";
    RAddToolTip(hDlg, IDC_FORMAT_LBL, pFormatTip);
    RAddToolTip(hDlg, IDC_FORMAT_LST, pFormatTip);
    RAddToolTip(hDlg, IDC_ALL_FORMAT, (isJP) ? "すべてのフォーマットから選択するなら ON にします" :
        "Selects from all formats if ON is selected");
    RAddToolTip(hDlg, IDC_ALL_FORMAT_LST, (isJP) ? "フォーマットを指定します" : "Specifies the format");
    RAddToolTip(hDlg, IDC_WEIGHT_CMPR, (isJP) ? "圧縮する際に、RGB の誤差に対して人間の目に合わせた重み付けをするなら ON にします" :
        "Applies a weight according to the human eye to RGB errors when using compression method if ON is selected");
    const char* pDimensionTip = (isJP) ? "次元を指定します" : "Specifies the dimension";
    RAddToolTip(hDlg, IDC_DIMENSION_LBL, pDimensionTip);
    RAddToolTip(hDlg, IDC_DIMENSION_LST, pDimensionTip);
    RAddToolTip(hDlg, IDC_MIPMAP, (isJP) ? "ミップマップを作成するなら ON にします" : "Generates a mipmap if ON is selected");
    const char* pMipGenFilterTip = (isJP) ?
        "ミップマップを生成する際のフィルタを指定します" : "Specifies the filter when generating a mipmap";
    RAddToolTip(hDlg, IDC_MIP_GEN_FILTER_LBL, pMipGenFilterTip);
    RAddToolTip(hDlg, IDC_MIP_GEN_FILTER_LST, pMipGenFilterTip);
    RAddToolTip(hDlg, IDC_COMP_SEL_LBL, (isJP) ? "RGBA の各チャンネルに対するカラー成分を選択します" :
        "Selects the color component corresponding to each R, G, B, and A channel");
    const std::string compCharStr = "RGBA";
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        const std::string compChar = compCharStr.substr(iRgba, 1);
        RAddToolTip(hDlg, s_CompSelControls[iRgba], (isJP) ?
            (compChar + " チャンネルに対するカラー成分を選択します").c_str() :
            ("Selects the color component corresponding to " + compChar + " channel").c_str());
    }
    const char* pHintTip = (isJP) ?
        "ヒント情報を指定します。\n選択可能な項目はコンフィグファイルで設定できます" :
        "Specifies hint information. \nThe item that can be selected here can be set in a config file";
    RAddToolTip(hDlg, IDC_HINT_LBL, pHintTip);
    RAddToolTip(hDlg, IDC_HINT_LST, pHintTip);
    const char* pLinearFlagTip = (isJP) ?
        "ガンマを sRGB からリニアに変換する RGBA 成分です。\nリニア変換はコンフィグファイルで設定します" :
        "RGBA components that have undergone linear conversion from sRGB to linear. \nLinear conversion is set in the config file";
    RAddToolTip(hDlg, IDC_LINEAR_LBL, pLinearFlagTip);
    RAddToolTip(hDlg, IDC_LINEAR_TXT, pLinearFlagTip);
    const char* pSwizzleTip = (isJP) ? "初期スウィズル値を指定します" : "Specifies the initial swizzle value";
    RAddToolTip(hDlg, IDC_SWIZZLE_LBL, pSwizzleTip);
    RAddToolTip(hDlg, IDC_SWIZZLE_LST, pSwizzleTip);
    const char* pCommentTip = (isJP) ? "ftx ファイルに対する編集用コメント文を指定します" :
        "Specifies comment text related to editing of the ftx file";
    RAddToolTip(hDlg, IDC_COMMENT_LBL, pCommentTip);
    RAddToolTip(hDlg, IDC_COMMENT_TXT, pCommentTip);

    RAddToolTip(hDlg, IDC_DATA_SIZE, (isJP) ?
        "現在のオプション設定におけるテクスチャの実データサイズ、幅、高さ、ミップマップのレベル数などの情報です" :
        "The actual data size, height, width, and number of mipmap levels for the texture under current option settings");
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログを終了します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
static bool EndFormatDialog(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // 編集用コメント文をチェックします。
    char commentBuf[512];
    GetDlgItemText(hDlg, IDC_COMMENT_TXT, commentBuf, sizeof(commentBuf));
    const std::string comment(commentBuf);
    //const std::string comment = RTrimString(commentBuf);
    if (comment.find_first_of("\"") != std::string::npos)
    {
        RShowError(globals, "Comment is wrong ( please don't use [double quote] )");
        return false;
    }
    g_AddInfo.m_CommentText = comment;

    //-----------------------------------------------------------------------------
    // 付加情報のコンフィグ名を更新します。
    g_AddInfo.m_ConfigName = (s_ConfigName == R_STANDARD_CONFIG_NAME) ?
        std::string() : s_ConfigName;

    return true;
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログのカーソルを設定します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] pos カーソル座標です。
//-----------------------------------------------------------------------------
static void SetFormatDialogCursor(GPtr globals, const POINTS& pos)
{
    if (globals->m_DisplaysPreview &&
        (s_Preview.m_IsDragging        ||
         s_Preview.m_IsShowingOriginal ||
         RIsPointInRect(s_Preview.m_ItemRect, pos)))
    {
        sPSUIHooks->SetCursor(kPICursorHand);
    }
    else
    {
        sPSUIHooks->SetCursor(kPICursorArrow);
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログのダイアログハンドルを返します。
//!        現在、ローカルプログレスダイアログ表示で使用します。
//-----------------------------------------------------------------------------
HWND RGetFormatDialogHandle()
{
    return s_FormatDialogHandle;
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログのダイアログプロシージャです。
//!
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] uMsg メッセージです。
//! @param[in] wParam メッセージの付加情報です。
//! @param[in] lParam メッセージの付加情報です。
//!
//! @return 処理した場合は TRUE を返します。
//-----------------------------------------------------------------------------
static BOOL WINAPI FormatDialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    static GPtr globals = NULL; // need to be static
    static POINTS dragStartPos;
    static int dragStartIx;
    static int dragStartIy;

    switch (uMsg)
    {
        case WM_INITDIALOG:
            //RNoteTrace("init start");
            s_FormatDialogHandle = hDlg;
            globals = reinterpret_cast<GPtr>(lParam);
            InitFormatDialog(globals, hDlg);
            //RNoteTrace("init end");
            //break; // break せずに WM_PAINT の処理もします。

        case WM_PAINT:
            if (globals->m_DisplaysPreview)
            {
                s_Preview.Paint(hDlg);
            }
            return FALSE;

        case WM_DESTROY:
            s_Preview.FreeMemory();
            s_FormatDialogHandle = NULL;
            break;

        case WM_COMMAND:
        {
            const short id     = LOWORD(wParam);
            const short notify = HIWORD(wParam);
            switch (id)
            {
                case IDOK:
                    if (EndFormatDialog(globals, hDlg))
                    {
                        DeleteObject(s_FormatFont);
                        EndDialog(hDlg, id);
                    }
                    break;

                case IDCANCEL:
                    DeleteObject(s_FormatFont);
                    EndDialog(hDlg, id);
                    break;

                case IDC_CONFIG_NAME_LST:
                    if (notify == CBN_SELCHANGE)
                    {
                        ConfigChangedCB(globals, hDlg);
                    }
                    break;

                case IDC_FORMAT_LST:
                    if (notify == CBN_SELCHANGE)
                    {
                        FormatDialogFormatPresetCB(globals, hDlg);
                    }
                    break;

                case IDC_ALL_FORMAT:
                    s_IsAllFormat = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                    if (s_IsAllFormat)
                    {
                        FormatDialogAllFormatCB(globals, hDlg);
                    }
                    else
                    {
                        FormatDialogFormatPresetCB(globals, hDlg);
                    }
                    break;

                case IDC_ALL_FORMAT_LST:
                    if (notify == CBN_SELCHANGE)
                    {
                        FormatDialogAllFormatCB(globals, hDlg);
                    }
                    break;

                case IDC_WEIGHT_CMPR:
                    g_AddInfo.m_WeightedCompress = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                    break;

                case IDC_DIMENSION_LST:
                    if (notify == CBN_SELCHANGE)
                    {
                        const int sel = static_cast<int>(SendMessage(GetDlgItem(hDlg, id),
                            CB_GETCURSEL, 0, 0));
                        g_AddInfo.m_Dimension = s_Dimensions[sel];
                        UpdateDataSize(globals, hDlg);
                    }
                    break;

                case IDC_MIPMAP:
                    g_AddInfo.m_MipLevel = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED) ?
                        RImage::MipCountMax : 1;
                    FormatDialogFormatCB(globals, hDlg);
                    break;

                case IDC_MIP_GEN_FILTER_LST:
                    if (notify == CBN_SELCHANGE)
                    {
                        const int sel = static_cast<int>(SendMessage(GetDlgItem(hDlg, id),
                            CB_GETCURSEL, 0, 0));
                        g_AddInfo.m_MipGenFilter = static_cast<RMipGenFilter::MipGenFilter>(sel);
                    }
                    break;

                case IDC_COMP_SEL_R:
                case IDC_COMP_SEL_G:
                case IDC_COMP_SEL_B:
                case IDC_COMP_SEL_A:
                    if (notify == CBN_SELCHANGE)
                    {
                        const int sel = static_cast<int>(SendMessage(GetDlgItem(hDlg, id),
                            CB_GETCURSEL, 0, 0));
                        int iRgba = 0;
                        if      (id == IDC_COMP_SEL_G) iRgba = 1;
                        else if (id == IDC_COMP_SEL_B) iRgba = 2;
                        else if (id == IDC_COMP_SEL_A) iRgba = 3;
                        const int shift = (3 - iRgba) * 8;
                        g_AddInfo.m_CompSel &= ~(0xff << shift);
                        g_AddInfo.m_CompSel |=  (sel  << shift);
                        //RNoteTrace("comp sel: %08x\n", g_AddInfo.m_CompSel);
                    }
                    break;

                case IDC_HINT_LST:
                    if (notify == CBN_SELCHANGE)
                    {
                        const int sel = static_cast<int>(SendMessage(GetDlgItem(hDlg, id),
                            CB_GETCURSEL, 0, 0));
                        g_AddInfo.m_Hint = s_HintPresets[sel];
                        if (s_IsAllFormat && RIsAstcTextureFormat(g_AddInfo.m_Format))
                        {
                            g_AddInfo.m_CompSel = RGetDefaultCompSel(g_AddInfo.m_Format, g_AddInfo.m_Hint);
                            UpdateCompSelList(globals, hDlg);
                        }
                    }
                    break;

                case IDC_SWIZZLE_LST:
                    if (notify == CBN_SELCHANGE)
                    {
                        const int sel = static_cast<int>(SendMessage(GetDlgItem(hDlg, id),
                            CB_GETCURSEL, 0, 0));
                        g_AddInfo.m_InitialSwizzle = sel;
                    }
                    break;

                case IDC_PREVIEW_FLAG:
                    globals->m_DisplaysPreview =
                        (IsDlgButtonChecked(hDlg, IDC_PREVIEW_FLAG) == BST_CHECKED);
                    SetPreviewControlVisibility(globals, hDlg);
                    UpdatePreviewBitmap(globals, hDlg);
                    SetFormatDialogSize(globals, hDlg, false);
                    break;

                case IDC_PREVIEW_RGB:
                    s_Preview.m_DisplaysRgb =
                        (IsDlgButtonChecked(hDlg, IDC_PREVIEW_RGB) == BST_CHECKED);
                    if (!s_Preview.m_DisplaysRgb && !s_Preview.m_DisplaysAlpha)
                    {
                        s_Preview.m_DisplaysAlpha = true;
                        CheckDlgButton(hDlg, IDC_PREVIEW_ALPHA, BST_CHECKED);
                    }
                    if (globals->m_DisplaysPreview)
                    {
                        s_Preview.UpdateWinBmp(hDlg, globals->m_pBitmapData);
                    }
                    break;

                case IDC_PREVIEW_ALPHA:
                    s_Preview.m_DisplaysAlpha =
                        (IsDlgButtonChecked(hDlg, IDC_PREVIEW_ALPHA) == BST_CHECKED);
                    if (!s_Preview.m_DisplaysRgb && !s_Preview.m_DisplaysAlpha)
                    {
                        s_Preview.m_DisplaysRgb = true;
                        CheckDlgButton(hDlg, IDC_PREVIEW_RGB, BST_CHECKED);
                    }
                    if (globals->m_DisplaysPreview)
                    {
                        s_Preview.UpdateWinBmp(hDlg, globals->m_pBitmapData);
                    }
                    break;

                case IDC_ZOOM_IN:
                    if (s_Preview.m_Zoom < s_Preview.m_MaxZoom)
                    {
                        s_Preview.ChangeZoom(hDlg, 1);
                    }
                    break;

                case IDC_ZOOM_OUT:
                    if (s_Preview.m_Zoom > s_Preview.m_MinZoom)
                    {
                        s_Preview.ChangeZoom(hDlg, -1);
                    }
                    break;

                case IDM_LOCAL_HELP:
                    RShowPluginHelp(NPS_PLUGIN_HELP_URL);
                    break;

                case IDM_HELP_INDEX:
                    RShowPluginHelp(NPS_HELP_INDEX_URL);
                    break;

                default:
                    break;
            }
            break;
        }

        case WM_HSCROLL:
        {
            //short code = LOWORD(wParam);
            //short pos = HIWORD(wParam); // SB_THUMBPOSITION と SB_THUMBTRACK のみ有効
            //int id = GetWindowLong(reinterpret_cast<HWND>(lParam), GWL_ID);
            break;
        }

        case WM_LBUTTONDOWN:
            dragStartPos = MAKEPOINTS(lParam);
            if (globals->m_DisplaysPreview && RIsPointInRect(s_Preview.m_ItemRect, dragStartPos))
            {
                // フォーカスをプレビューに移してホイールで拡大縮小ができるようにします。
                //SetFocus(GetDlgItem(hDlg, IDC_PREVIEW_AREA));
                s_Preview.m_IsDragging = s_Preview.IsScrollEnable();
                s_Preview.m_IsShowingOriginal = true;
                //s_Preview.UpdateWinBmp(hDlg, globals->m_pSrcBitmapData);

                SetFormatDialogCursor(globals, dragStartPos);
                if (s_Preview.m_IsDragging)
                {
                    // start drag
                    dragStartIx = s_Preview.m_Ix;
                    dragStartIy = s_Preview.m_Iy;
                    SetCapture(hDlg);
                    return FALSE;
                }
            }
            break;

        case WM_MOUSEMOVE:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (s_Preview.m_IsDragging)
            {
                s_Preview.ChangeScroll(hDlg,
                    dragStartIx - (pos.x - dragStartPos.x),
                    dragStartIy - (pos.y - dragStartPos.y));
            }
            SetFormatDialogCursor(globals, dragStartPos);
            break;
        }

        case WM_LBUTTONUP:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (s_Preview.m_IsDragging)
            {
                s_Preview.m_IsDragging = false;
                ReleaseCapture();
            }
            if (s_Preview.m_IsShowingOriginal)
            {
                s_Preview.m_IsShowingOriginal = false;
                s_Preview.UpdateWinBmp(hDlg, globals->m_pBitmapData);
            }
            SetFormatDialogCursor(globals, pos);
            break;
        }

        default:
            return FALSE;
            //return DefWindowProc(hDlg, uMsg, wParam, lParam); // 呼ぶとウィンドウタイトルが表示されなくなる
    }
    return TRUE;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ファイル形式オプションダイアログを表示します。
//!
//! @param[in,out] globals グローバルデータです。
//!
//! @return OK ボタンがクリックされたら true、
//!         Cancel ボタンがクリックされたら false を返します。
//-----------------------------------------------------------------------------
bool RDoFormatDialog(GPtr globals)
{
    HINSTANCE instance = GetDLLInstance();
    PlatformData* platform = reinterpret_cast<PlatformData*>(gStuff->platformData);
    INT_PTR result = DialogBoxParam(instance,
        MAKEINTRESOURCE(IDD_FORMAT_OPTS), reinterpret_cast<HWND>(platform->hwnd),
        reinterpret_cast<DLGPROC>(FormatDialogProc), reinterpret_cast<LPARAM>(globals));
    return (result == IDOK);
}

//-----------------------------------------------------------------------------
//! @brief アバウトダイアログを表示します。
//!
//! @param[in] about アバウトレコードです。
//-----------------------------------------------------------------------------
void RDoAboutDialog(AboutRecordPtr about)
{
    HINSTANCE instance = GetDLLInstance();
    PlatformData* platform = reinterpret_cast<PlatformData*>(about->platformData);
    RAboutParam aboutParam =
    {
        NPS_ABOUT_TITLE, NPS_ABOUT_MESSAGE,
        IDC_ABOUT_TEXT, IDC_HELP_INDEX, NPS_HELP_INDEX_URL
    };
    DialogBoxParam(instance,
        MAKEINTRESOURCE(IDD_ABOUT), reinterpret_cast<HWND>(platform->hwnd),
        reinterpret_cast<DLGPROC>(RAboutDialogProc), reinterpret_cast<LPARAM>(&aboutParam));
}

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

