﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// functions
// RAddInfo RImageStatus

//=============================================================================
// include
//=============================================================================
#include "NpsCommon.h"
#pragma warning(push)
    #pragma warning(disable: 4121) // メンバのアライメントは過剰にパッキングされています。
    #include "PIFormat.h"
#pragma warning(pop)

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//=============================================================================
//! @brief 付加情報のクラスです。
//=============================================================================
class RAddInfo
{
public:
    static const int SWIZZLE_DEFAULT = -1; //!< デフォルトのスウィズル値（指定なし）です。
    static const int SWIZZLE_MIN = 0; //!< スウィズル値の最小値です。
    static const int SWIZZLE_MAX = 7; //!< スウィズル値の最大値です。

    std::string m_ConfigName; //!< コンフィグ名です。
    std::string m_Hint; //!< ヒント情報です。
    int m_LinearFlag; //!< リニア変換フラグです。
    FtxDimension m_Dimension; //!< 次元です。
    FtxFormat m_Format; //!< テクスチャフォーマットです。
    int m_MipLevel; //!< ミップマップのレベル数です。
    RMipGenFilter::MipGenFilter m_MipGenFilter; //!< ミップマップ生成フィルタです。
    FtxCompSel m_CompSel; //!< 成分選択です。
    bool m_WeightedCompress; //!< bc1 / bc2 / bc3 / etc1 / etc2 フォーマット圧縮時の RGB 成分の誤差の重み付けです。
    int m_InitialSwizzle; //!< 初期スウィズル値です。
    char m_MergePath[MAX_PATH]; //!< マージする ftx ファイルのパスです。空文字ならマージしません。
    char m_OverwriteInputPath[MAX_PATH]; //!< 入力ファイルのパスの上書き指定です。空文字ならドキュメントのパスを使用します。
    std::string m_CreateInfo; //!< ftx の <create> 要素の内容です。
    RStringArray m_OriginalPaths; //!< ftx のオリジナルパス配列です。
    uint8_t* m_pPackedUserData; //!< ユーザーデータ群をパックしたデータへのポインタです。
    uint32_t m_PackedUserDataSize; //!< ユーザーデータ群をパックしたデータのサイズです。
    std::string m_CommentLabel; //!< 編集用コメントラベルです。
    std::string m_CommentCol; //!< 編集用コメントカラー文字列です。
    std::string m_CommentText; //!< 編集用コメント文です。
    std::string m_CropRect; //!< 切り抜き矩形です（X,Y,W,H の形式。空文字なら指定なし）。
    std::string m_ResizeWH; //!< サイズ変更後の幅と高さです（WxH の形式。空文字なら指定なし）。
    std::string m_ResizeFilter; //!< サイズ変更フィルタです（空文字なら指定なし）。
    bool m_ResizesInLinear; //!< リニア空間でサイズ変更計算するなら true です。
    std::string m_ToolData; //!< ftx の <tool_data> 要素の内容です。
    std::string m_UserToolData; //!< ftx の <user_tool_data> 要素の内容です。

    bool m_IsExportTexture; //!< テクスチャエクスポートプラグインから呼ばれたなら true です。
    bool m_IsConfigNameGot; //!< ライト時にリソースデータからコンフィグ名を取得したなら true です。
    std::string m_OutputOriginalPath; //!< 元画像ファイルの出力パスです（空文字なら出力しない）。

public:
    //! コンストラクタです。
    RAddInfo()
    : m_LinearFlag(RLinearFlag::LINEAR_NONE),
      m_Dimension(FtxDimension_2d),
      m_Format(FtxFormat_Invalid),
      m_MipLevel(1),
      m_MipGenFilter(RMipGenFilter::MIP_GEN_LINEAR),
      m_CompSel(FtxCompSelRgba),
      m_WeightedCompress(false),
      m_InitialSwizzle(0),
      m_pPackedUserData(NULL),
      m_PackedUserDataSize(0),
      m_ResizesInLinear(false),
      m_IsExportTexture(false),
      m_IsConfigNameGot(false)
    {
        //RNoteTrace("RAddInfo()");
        memset(m_MergePath, 0x00, sizeof(m_MergePath));
        memset(m_OverwriteInputPath, 0x00, sizeof(m_OverwriteInputPath));
    }

    //! デストラクタです。
    virtual ~RAddInfo()
    {
        //RNoteTrace("~RAddInfo()");
        RFreeAndClearArray(m_pPackedUserData);
    }

    //! @brief 現在の設定におけるテクスチャの実データサイズを取得します。
    //!        中間ファイルに出力されるデータはタイリングされるので、
    //!        この実データサイズより大きくなる場合があります。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //! @param[in] imageD 画像の深さです。
    //!
    //! @return テクスチャの実データサイズを返します。
    //!
    size_t GetDataSize(const int imageW, const int imageH, const int imageD) const;

    //! 他の ftx ファイルをマージするなら true を返します。
    bool Merges() const
    {
        return m_MergePath[0] != 0;
    }

    //! 入力ファイルのパスを上書きするなら true を返します。
    bool OverwritesInputPath() const
    {
        return m_OverwriteInputPath[0] != 0;
    }
};

//=============================================================================
//! @brief 画像状態のクラスです。
//=============================================================================
class RImageStatus
{
public:
    // resouce data
    int m_RdVersion; //!< 最後にライトした際のリソースデータのバージョンです。
    FtxDimension m_LastDimension; //!< 最後にライトした際の次元です。
    FtxFormat m_LastFormat; //!< 最後にライトした際のフォーマットです。
    int m_LastMipLevel; //!< 最後にライトした際のミップマップのレベル数です。

    // size
    bool m_Is1DEnable; //!< 1 次元テクスチャが可能なら true です。
    bool m_Is2DEnable; //!< 2 次元テクスチャが可能なら true です。
    bool m_Is3DEnable; //!< 3 次元および配列テクスチャが可能なら true です。
    bool m_IsCubeHCEnable; //!< 水平十字キューブマップが可能なら true です。
    bool m_IsCubeVCEnable; //!< 垂直十字キューブマップが可能なら true です。
    int m_CubeFaceW; //!< キューブマップの場合のフェースの幅です。
    int m_CubeFaceH; //!< キューブマップの場合のフェースの高さです。
    int m_3DFaceW; //!< 3 次元および配列テクスチャの場合のフェースの幅です。
    int m_3DFaceH; //!< 3 次元および配列テクスチャの場合のフェースの高さです。
    int m_3DFaceD; //!< 3 次元および配列テクスチャの場合のフェースの奥行きです。

    // color
    bool m_IsAllGray; //!< すべてのピクセルで R = G = B なら true です。
    RImage::TransparencyMode m_XpaMode; //!< 透明モードです。

    // option
    FtxFormat m_RecomFormat; //!< 画像状態から推奨するフォーマットです。
    bool m_IsFormatDecided; //!< 疑似リソースまたはスクリプトから取得したフォーマットで確定済みなら true です。

public:
    //! コンストラクタです。
    RImageStatus()
    : // resouce data
      m_RdVersion(0),
      m_LastDimension(FtxDimension_2d),
      m_LastFormat(FtxFormat_Invalid),
      m_LastMipLevel(1),

      // color
      m_IsAllGray(false),
      m_XpaMode(RImage::OPA),

      // option
      m_RecomFormat(FtxFormat_Unorm_8_8_8_8),
      m_IsFormatDecided(false)
    {
        //RNoteTrace("RImageStatus()");
    }

    //! デストラクタです。
    virtual ~RImageStatus()
    {
        //RNoteTrace("~RImageStatus()");
    }

    // キューブマップが可能なら true を返します。
    bool IsCubeEnable()
    {
        return (m_IsCubeHCEnable || m_IsCubeVCEnable);
    }
};

//=============================================================================
// ビットマップデータ関連の関数です。
//=============================================================================

//! @brief ビットマップデータを Photoshop に設定します。
//!        カラーインデックスは非対応です。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] pBitmapData ビットマップデータです。
//!
void SetBitmapDataToPhotoshop(GPtr globals, const void* pBitmapData);

//! @brief ビットマップデータを Photoshop から取得します。
//!
//! @param[out] pBitmapData ビットマップデータです。
//!                         この関数を呼ぶ前に確保しておきます。
//! @param[in] globals グローバルデータです。
//!
void GetBitmapDataFromPhotoshop(void* pBitmapData, GPtr globals);

//! @brief 透明なピクセルの RGB 値を補整します。
//!        レイヤーのピクセルの透明度をアルファとしてセーブする場合に、
//!        バイリニア補間による透明ピクセルの白色のにじみを軽減するための処理です。
//!
//! @param[in,out] pBitmapData ビットマップデータです。
//! @param[in] globals グローバルデータです。
//!
void AdjustTransparentPixelRgb(void* pBitmapData, GPtr globals);

//=============================================================================
// ドキュメント関連の関数です。
//=============================================================================

//! @brief ドキュメントの最終的なファイルパスを返します。
//!        別名で保存した場合に別名で保存したファイルパスを取得できます。
//!
//! @param[in] globals グローバルデータです。
//!
//! @return ドキュメントの最終的なファイルパスを返します。
//!
std::string RGetFinalFilePath(GPtr globals);

//=============================================================================
// チャンネル関連の関数です。
//=============================================================================

//! @brief ライト時に Photoshop 上に存在するアルファチャンネル数を返します。
//!        レイヤーの透明度プレーンはカウントしません。
//!
//! @param[in] globals グローバルデータです。
//!
//! @return アルファチャンネル数を返します。
//!
int RGetPsAlphaChanCount(GPtr globals);

//! @brief ライト時に Photoshop から取得するアルファプレーンのインデックスを返します。
//!        背景がなくレイヤーのみの画像の場合、アルファチャンネルがなければ
//!        レイヤーの透明度プレーンを返します。
//!
//! @param[in] globals グローバルデータです。
//!
//! @return アルファプレーンのインデックスを返します。
//!         アルファチャンネルもレイヤーの透明度プレーンもなければ 0 を返します。
//!
int RGetPsAlphaPlaneIndex(const GPtr globals);

//=============================================================================
// Photoshop のファイルシステム関連の関数です。
//=============================================================================

//! @brief ファイルサイズを取得します。
//!
//! @param[in] dataFork データフォークです。
//!
//! @return ファイルサイズを返します。
//!
size_t RGetFileSize(const intptr_t dataFork);

//! @brief 指定したバイト数リードします。
//!
//! @param[in] dataFork データフォークです。
//! @param[out] pBuffer リードしたデータを格納するバッファのポインタを指定します。
//! @param[in] size リードするバイト数を指定します。
//!
//! @return エラーコードを返します。
//!
OSErr RReadSome(const intptr_t dataFork, void* pBuffer, const size_t size);

//! @brief 指定したバイト数ライトします。
//!
//! @param[in] dataFork データフォークです。
//! @param[in] pBuffer ライトするデータを格納したバッファのポインタを指定します。
//! @param[in] size ライトするバイト数を指定します。
//!
//! @return エラーコードを返します。
//!
OSErr RWriteSome(const intptr_t dataFork, const void* pBuffer, const size_t size);

//=============================================================================
// Photoshop のレジストリ関連の関数です。
//=============================================================================
SPErr ReadRegistryParameters(GPtr globals);

SPErr WriteRegistryParameters(GPtr globals);

//=============================================================================
// 復帰情報関連の関数です。
//=============================================================================

//! @brief 復帰情報を設定します。
//!
//! @param[in,out] globals グローバルデータです。
//!
void RSetRevertInfo(GPtr globals);

//=============================================================================
// 付加情報関連の関数です。
//=============================================================================

//! @brief ファイル形式プラグイン用のリソースデータをドキュメントに設定します。
//!        リード時とライト時に呼ばれます。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] atWriteFlag リード時なら false、ライト時なら true を指定します。
//!
void SetFormatResourceData(GPtr globals, const bool atWriteFlag);


//! @brief ライトのために付加情報を設定します。
//!
//! @param[in,out] globals グローバルデータです。
//!
void SetAddInfoForWrite(GPtr globals);

//=============================================================================
// プレビュー関連の関数です。
//=============================================================================

void RUpdatePreviewDataBuf(GPtr globals);

//=============================================================================
// テクスチャ関連の関数です。
//=============================================================================

//! @brief テクスチャフォーマットに対応したデフォルトのヒント情報を取得します。
//!
//! @param[in] format テクスチャフォーマットです。
//!
//! @return ヒント情報を返します。
//!
std::string RGetDefaultHint(const FtxFormat format);

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

