﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include <windows.h>
#include <iostream>
#include <fstream>
#include <string>

using namespace std;

namespace {

//-----------------------------------------------------------------------------
//! @brief ファイルが存在なら true を返します。
//-----------------------------------------------------------------------------
bool RFileExists(const std::string& path)
{
    WIN32_FIND_DATAA ffd;
    HANDLE  hFindFile = FindFirstFileA(path.c_str(), &ffd);
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) // check is file
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//-----------------------------------------------------------------------------
std::string RGetEnvVar(const char* name)
{
    std::string ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が NULL に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != NULL)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを作成します。
//!
//! @param[out] pi プロセスの情報を格納します。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//! @param[in] getsStdOut 標準出力ハンドルを使用するなら true です。
//! @param[in] getsStdErr 標準エラーハンドルを使用するなら true です。
//! @param[in] hWriteOut 標準出力ハンドルを指定します。
//! @param[in] hWriteErr 標準エラーハンドルを指定します。
//!
//! @return 成功なら 0 以外を返します。
//-----------------------------------------------------------------------------
int CreateProcessExec(
    PROCESS_INFORMATION& pi,
    const char* cmd,
    const int showWindow = SW_HIDE,
    bool getsStdOut = false,
    bool getsStdErr = false,
    HANDLE hWriteOut = INVALID_HANDLE_VALUE,
    HANDLE hWriteErr = INVALID_HANDLE_VALUE
)
{
    //-----------------------------------------------------------------------------
    // init process information struct
    ZeroMemory(&pi, sizeof(PROCESS_INFORMATION));

    //-----------------------------------------------------------------------------
    // init startup info struct
    STARTUPINFOA si;
    ZeroMemory(&si, sizeof(si));
    si.cb          = sizeof(si);
    si.dwFlags     = STARTF_USESHOWWINDOW;
    si.wShowWindow = (WORD)showWindow;
    if (getsStdOut || getsStdErr)
    {
        si.dwFlags |= STARTF_USESTDHANDLES;
        if (getsStdOut)
        {
            si.hStdOutput = hWriteOut;
        }
        if (getsStdErr)
        {
            si.hStdError = hWriteErr;
        }
    }

    //-----------------------------------------------------------------------------
    // create process
    int retCode = CreateProcessA(
        NULL,   // lpApplicationName
        const_cast<char*>(cmd), // lpCommandLine
        NULL,   // lpProcessAttributes
        NULL,   // lpThreadAttributes
        (getsStdOut || getsStdErr), // bInheritHandles
        0,      // dwCreationFlags
        NULL,   // lpEnvironment
        NULL,   // lpCurrentDirectory
        &si,    // lpStartupInfo
        &pi);   // lpProcessInformation

    return retCode;
}

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを実行します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     NULL なら何も格納しません。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return コマンドの終了コードを返します。
//-----------------------------------------------------------------------------
int RExecProcess(int* pStatus, const char* cmd, const int showWindow)
{
    //-----------------------------------------------------------------------------
    // set status to success
    if (pStatus != NULL)
    {
        *pStatus = 1;
    }

    //-----------------------------------------------------------------------------
    // create process
    PROCESS_INFORMATION pi;
    if (CreateProcessExec(pi, cmd, showWindow) == 0)
    {
        cerr << "Error: Cannot create the process: " << cmd << endl;
        if (pStatus != NULL)
        {
            *pStatus = 0;
        }
        return 1;
    }

    //-----------------------------------------------------------------------------
    // wait until boot finish
    WaitForInputIdle(pi.hProcess, INFINITE);

    //-----------------------------------------------------------------------------
    // wait until command finish
    WaitForSingleObject(pi.hProcess, INFINITE);

    //-----------------------------------------------------------------------------
    // get return code
    DWORD exitCode;
        //                        0: 正常終了
        //  -532459699 (0xE0434F4D): 想定外の例外が発生
        // -1073741510 (0xC000013A): アプリケーションを強制的に終了した場合
    if (GetExitCodeProcess(pi.hProcess, &exitCode) == 0)
    {
        cerr << "Error: Cannot get the process exit code: " << cmd << endl;
        if (pStatus != NULL)
        {
            *pStatus = 0;
        }
        exitCode = 1;
    }

    //-----------------------------------------------------------------------------
    // close process handle
    CloseHandle(pi.hThread);
    CloseHandle(pi.hProcess);

    return exitCode;
}

//-----------------------------------------------------------------------------
//! @brief ロックファイルを作成します。
//!
//! @param[in] lockPath ロックファイルのパスです。
//! @param[in] exitCode コマンドの終了コードです。
//-----------------------------------------------------------------------------
void CreateLockFile(const std::string& lockPath, const int exitCode)
{
    std::ofstream ofs(lockPath);
    if (ofs)
    {
        ofs << "ExitCode: " << exitCode << endl;
    }
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief メイン関数です。
//-----------------------------------------------------------------------------
int WINAPI WinMain(
  HINSTANCE /*hInstance*/,
  HINSTANCE /*hPrevInstance*/,
  LPSTR /*lpCmdLine*/,
  int /*nCmdShow*/
)
{
    //-----------------------------------------------------------------------------
    // 環境変数からコマンドを取得します。
    const std::string cmd = RGetEnvVar("NINTENDO_PHOTOSHOP_PROCESS_CMD");
    if (cmd.empty())
    {
        cerr << "Error: Cannot get the command from environment variable" << endl;
        return 1;
    }

    //-----------------------------------------------------------------------------
    // 環境変数からウィンドウ表示指定を取得します。
    // 0: 非表示
    // 1: 表示
    // 2: 最小化
    int showWindow = SW_SHOW;
    const std::string showStr = RGetEnvVar("NINTENDO_PHOTOSHOP_PROCESS_SHOW");
    if (!showStr.empty())
    {
        const int showVal = atoi(showStr.c_str());
        if (showVal == 0)
        {
            showWindow = SW_HIDE;
        }
        else if (showVal == 2)
        {
            showWindow = SW_SHOWMINNOACTIVE; // SW_SHOWMINIMIZED
        }
    }

    //cerr << "cmd: [" << cmd << "] " << showWindow << endl;
    //MessageBoxA(NULL, ("[" + cmd + "] " + showStr).c_str(), "", MB_OK);

    //-----------------------------------------------------------------------------
    // コマンドを実行します。
    const int exitCode = RExecProcess(NULL, cmd.c_str(), showWindow);

    //-----------------------------------------------------------------------------
    // 環境変数でロックファイルが指定されていて、存在しなければ作成します。
    const std::string lockPath = RGetEnvVar("NINTENDO_PHOTOSHOP_PROCESS_LOCK");
    if (!lockPath.empty() && !RFileExists(lockPath))
    {
        CreateLockFile(lockPath, exitCode);
    }

    return exitCode;
}

