﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// NintendoImportCmd newSyntax ParseArgs
// DoIt ActionT
// ParseFmd ParseFsk ParseFvb ParseFma ParseFcl ParseFts ParseFtp ParseFsh ParseFsn
// ParseUserData
// ParseMaterial ParseSampler ConnectTexture
// ParseBone CreateMesh GetShapeData AdjustMeshNormal SetMeshHardEdge
// SetMeshSkinning CreateBlendShape
// CreateAnimCurve ConvertAnimCurveValueTangent

// ImpIntermediateFile
// ImpMaterial ImpSampler ImpTexData
// ImpBone ImpShapeData ImpKeyShape ImpVtxAttrib ImpMeshParam ImpUvSet ImpVtxMtx
// ImpTexPat ImpAnimKey ImpLodInfo

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
#define SET_UV_SET_NAME_FIX // nw4f_fix_ で始まる UV セット名を設定するなら定義します。

//=============================================================================
// include
//=============================================================================
#include "Common.h"

#define MNoVersionString
#define MNoPluginEntry
#include <maya/MFnPlugin.h>

//-----------------------------------------------------------------------------
// using namespace
using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
// nwImport ネームスペースを開始します。
//=============================================================================
namespace nwImport {

//=============================================================================
// NW 用の定数の定義です。
//=============================================================================
const std::string FmdbExtension = "fmdb"; //!< バイナリ形式の fmd ファイルの拡張子です。
const std::string FmdaExtension = "fmda"; //!< アスキー形式の fmd ファイルの拡張子です。
const std::string FtxbExtension = "ftxb"; //!< バイナリ形式の ftx ファイルの拡張子です。
const std::string FtxaExtension = "ftxa"; //!< アスキー形式の ftx ファイルの拡張子です。
const std::string FskbExtension = "fskb"; //!< バイナリ形式の fsk ファイルの拡張子です。
const std::string FskaExtension = "fska"; //!< アスキー形式の fsk ファイルの拡張子です。
const std::string FvbbExtension = "fvbb"; //!< バイナリ形式の fvb ファイルの拡張子です。
const std::string FvbaExtension = "fvba"; //!< アスキー形式の fvb ファイルの拡張子です。
const std::string FmabExtension = "fmab"; //!< バイナリ形式の fma ファイルの拡張子です。
const std::string FmaaExtension = "fmaa"; //!< アスキー形式の fma ファイルの拡張子です。
const std::string FclbExtension = "fclb"; //!< バイナリ形式の fcl ファイルの拡張子です。
const std::string FclaExtension = "fcla"; //!< アスキー形式の fcl ファイルの拡張子です。
const std::string FtsbExtension = "ftsb"; //!< バイナリ形式の fts ファイルの拡張子です。
const std::string FtsaExtension = "ftsa"; //!< アスキー形式の fts ファイルの拡張子です。
const std::string FtpbExtension = "ftpb"; //!< バイナリ形式の ftp ファイルの拡張子です。
const std::string FtpaExtension = "ftpa"; //!< アスキー形式の ftp ファイルの拡張子です。
const std::string FshbExtension = "fshb"; //!< バイナリ形式の fsh ファイルの拡張子です。
const std::string FshaExtension = "fsha"; //!< アスキー形式の fsh ファイルの拡張子です。
const std::string FsnbExtension = "fsnb"; //!< バイナリ形式の fsn ファイルの拡張子です。
const std::string FsnaExtension = "fsna"; //!< アスキー形式の fsn ファイルの拡張子です。

const double TOLERANCE_NRM = 1.0e-6; //!< 法線の誤差の許容値です。

const std::string DEFAULT_ROOT_BONE_NAME = "nw4f_root"; //!< エクスポートプラグインが追加するデフォルトのルートボーン名です。

const int LOD_INDEX_ALL = -1; //!< 全 LOD モデルをインポートする場合の LOD インデックスです。

#if (MAYA_API_VERSION >= 201600)
const MString ColorSpaceRaw = "Raw"; //!< Raw カラースペースです。
#endif

//-----------------------------------------------------------------------------
//! @brief std::vector を std::ostream に出力します（デバッグ用）。
//-----------------------------------------------------------------------------
template<typename T>
std::ostream& operator<<(std::ostream& os, const std::vector<T>& v)
{
    os << "[";
    for (size_t i = 0; i < v.size(); ++i)
    {
        if (i != 0) os << ", ";
        os << v[i];
    }
    os << "]";
    return os;
}

//-----------------------------------------------------------------------------
//! @brief ネームスペースを削除した文字列を返します。
//-----------------------------------------------------------------------------
static std::string RemoveNamespace(const std::string& src)
{
    const size_t iColon = src.find_last_of(':');
    if (iColon != std::string::npos &&
        iColon + 1 < src.size())
    {
        return std::string(src.c_str() + iColon + 1);
    }
    else
    {
        return src;
    }
}

//-----------------------------------------------------------------------------
//! @brief 文字列の末尾の数値を返します。
//!        末尾が数値でなければ 0 を返します。
//-----------------------------------------------------------------------------
static int GetStringSuffixNumber(const std::string& str)
{
    const int len = static_cast<int>(str.length());
    int nc = 0;
    for (int ic = len - 1; ic >= 0; --ic)
    {
        if ('0' <= str[ic] && str[ic] <= '9')
        {
            ++nc;
        }
        else
        {
            break;
        }
    }
    return (nc != 0) ? atoi(str.c_str() + len - nc) : 0;
}

//-----------------------------------------------------------------------------
//! @brief 文字列の末尾の数値を除いた文字列を返します。
//-----------------------------------------------------------------------------
static std::string GetNoSuffixNumberString(const std::string& str)
{
    const int len = static_cast<int>(str.length());
    int nc = 0;
    for (int ic = len - 1; ic >= 0; --ic)
    {
        if ('0' <= str[ic] && str[ic] <= '9')
        {
            ++nc;
        }
        else
        {
            break;
        }
    }
    return (nc != 0) ? str.substr(0, len - nc) : str;
}

//-----------------------------------------------------------------------------
//! @brief インポートのエラーを表示します。
//-----------------------------------------------------------------------------
void DisplayImportError(
    const std::string& formatJp,
    const std::string& formatEn,
    const RStringArray& args
)
{
    const std::string& format = (IsJapaneseUi() && !formatJp.empty()) ? formatJp : formatEn;
    const std::string msg = RReplaceArgumentsInString(format, args);
    MGlobal::displayError(msg.c_str());
}

inline void DisplayImportError(
    const std::string& formatJp,
    const std::string& formatEn
)
{
    DisplayImportError(formatJp, formatEn, RStringArray());
}

inline void DisplayImportError(
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0
)
{
    RStringArray args;
    args.push_back(arg0);
    DisplayImportError(formatJp, formatEn, args);
}

//-----------------------------------------------------------------------------
//! @brief インポートの警告を表示します。
//-----------------------------------------------------------------------------
void DisplayImportWarning(
    const std::string& formatJp,
    const std::string& formatEn,
    const RStringArray& args
)
{
    const std::string& format = (IsJapaneseUi() && !formatJp.empty()) ? formatJp : formatEn;
    const std::string msg = RReplaceArgumentsInString(format, args);
    MGlobal::displayWarning(msg.c_str());
}

inline void DisplayImportWarning(
    const std::string& formatJp,
    const std::string& formatEn
)
{
    DisplayImportWarning(formatJp, formatEn, RStringArray());
}

inline void DisplayImportWarning(
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0
)
{
    RStringArray args;
    args.push_back(arg0);
    DisplayImportWarning(formatJp, formatEn, args);
}

//-----------------------------------------------------------------------------
//! @brief sourceImages フォルダのパスを返します。
//-----------------------------------------------------------------------------
static std::string GetSourceImagesFolder()
{
    MString path;
    MGlobal::executeCommand("workspace -q -rd", path);
    MString imagesFolder;
    MGlobal::executeCommand("workspace -q -fileRuleEntry \"sourceImages\"", imagesFolder);
    if (imagesFolder == "" && RFolderExists((path + "sourceimages").asChar()))
    {
        imagesFolder = "sourceimages";
    }
    path += imagesFolder;
    return RGetFilePathWithEndingSlash(path.asChar());
}

//-----------------------------------------------------------------------------
//! @brief 指定した名前のノードのオブジェクトを返します。
//!        存在しなければ kNullObj を返します。
//!
//! @param[in] name ノード名です。
//!
//! @return オブジェクトを返します。
//-----------------------------------------------------------------------------
static MObject GetNodeObjectByName(const MString& name)
{
    MObject obj;
    MSelectionList slist;
    if (MGlobal::getSelectionListByName(name, slist) && !slist.isEmpty())
    {
        if (!slist.getDependNode(0, obj))
        {
            obj = MObject::kNullObj;
        }
    }
    return obj;
}

//-----------------------------------------------------------------------------
//! @brief 指定した名前のノードの DAG パスを返します。
//!        存在しなければ MDagPath() を返します。
//!
//! @param[in] name ノード名です。
//!
//! @return DAG パスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetDagPathByName(const MString& name)
{
    MDagPath dagPath;
    MSelectionList slist;
    if (MGlobal::getSelectionListByName(name, slist) && !slist.isEmpty())
    {
        if (!slist.getDagPath(0, dagPath))
        {
            return MDagPath();
        }
    }
    return dagPath;
}

//-----------------------------------------------------------------------------
//! @brief 選択されたノードのオブジェクトを返します。
//!        選択されていなければ kNullObj を返します。
//!        複数選択されている場合は最初に選択されたノードを返します。
//!
//! @return オブジェクトを返します。
//-----------------------------------------------------------------------------
static MObject GetSelectedNodeObject()
{
    MObject obj;
    MSelectionList slist;
    if (MGlobal::getActiveSelectionList(slist) && !slist.isEmpty())
    {
        if (!slist.getDependNode(0, obj))
        {
            obj = MObject::kNullObj;
        }
    }
    return obj;
}

//-----------------------------------------------------------------------------
//! @brief 選択されたノードの DAG パスを返します。
//!        選択されていなければ NULL パスを返します。
//!        複数選択されている場合は最初に選択されたノードを返します。
//!
//! @return DAG パスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetSelectedNodeDagPath()
{
    MDagPath dagPath;
    MSelectionList slist;
    if (MGlobal::getActiveSelectionList(slist) && !slist.isEmpty())
    {
        if (!slist.getDagPath(0, dagPath))
        {
            dagPath = MDagPath();
        }
    }
    return dagPath;
}

//-----------------------------------------------------------------------------
//! @brief DAG ノードが消去可能なら消去します。
//!
//! @param[in] dagPath DAG パスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus DeleteDagNodeIfPossible(const MDagPath& dagPath)
{
    MFnDagNode dagFn(dagPath);
    if (!dagFn.isLocked() && !dagFn.isDefaultNode() && !dagFn.isFromReferencedFile())
    {
        return MGlobal::executeCommand("delete " + dagPath.partialPathName());
    }
    else
    {
        return MS::kFailure;
    }
}

//-----------------------------------------------------------------------------
//! @brief プラグに接続されたファイルノード群を取得します。
//!
//! @param[in,out] fileObjs ファイルノード群を追加で格納します。
//! @param[in] plug プラグです。
//-----------------------------------------------------------------------------
static void GetFileNodesForPlug(MObjectArray& fileObjs, const MPlug& plug)
{
    MStatus status;
    MPlugArray plugArray;
    plug.connectedTo(plugArray, true, false);
    if (plugArray.length() != 0)
    {
        const MObject srcObj = plugArray[0].node();
        const MFn::Type srcType = srcObj.apiType();
        if (srcType == MFn::kFileTexture)
        {
            fileObjs.append(srcObj);
        }
        else if (srcType == MFn::kLayeredTexture)
        {
            MPlug inputsPlug = MFnDependencyNode(srcObj).findPlug("inputs");
            const int layerCount = inputsPlug.numElements();
            for (int iLayer = 0; iLayer < layerCount; ++iLayer)
            {
                MPlug inPlug = inputsPlug.elementByPhysicalIndex(iLayer, &status);
                if (!status)
                {
                    break;
                }
                MPlug colorPlug = inPlug.child(0);
                MPlugArray layerPlugArray;
                colorPlug.connectedTo(layerPlugArray, true, false);
                if (layerPlugArray.length() != 0)
                {
                    MObject layerObj = layerPlugArray[0].node();
                    if (layerObj.apiType() == MFn::kFileTexture)
                    {
                        fileObjs.append(layerObj);
                    }
                }
            }
        }
        else if (srcType == MFn::kBump)
        {
            GetFileNodesForPlug(fileObjs, MFnDependencyNode(srcObj).findPlug("bumpValue"));
        }
    }
}

//=============================================================================
// @brief インポート用バイナリデータ列のクラスです。
//=============================================================================
class ImpBinDataStream
{
public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, BYTE, STRING, WSTRING };

    char m_Identifier[8]; //!< 識別子です。
    int m_Type; //!< 値の型です。
    int m_Count; //!< 値の個数です。
    int m_Column; //!< アスキー出力時の列数です。
    int m_Size; //!< データサイズ（バイト数）です。文字列の場合は終端文字を含んだバイト数です。
    int m_Padding[2]; //!< 実データの先頭アドレスを 32 バイトアライメントにするためのパディングです。
    int m_Data[1]; //!< 実データです。実際は可変長です。

public:
    //! @brief バイナリデータ列へのポインタを取得します（static 関数）。
    //!
    //! @param[in] binTop 中間ファイルのバイナリ部分の先頭アドレスです。
    //! @param[in] streamIdx バイナリデータ列のインデックスです。
    //!
    //! @return バイナリデータ列へのポインタを返します。
    //!         インデックスが不正な場合は NULL を返します。
    //!
    static const ImpBinDataStream* Get(const void* binTop, const int streamIdx);
};

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの先頭からバイナリデータまでのオフセットを取得します。
//-----------------------------------------------------------------------------
int RGetBinaryOffset(const void* fileBuf, const int fileSize)
{
    int binOfs = fileSize;
    const uint8_t* buf = reinterpret_cast<const uint8_t*>(fileBuf);
    for (int iBuf = 0; iBuf < fileSize; ++iBuf)
    {
        if (buf[iBuf] == 0x00)
        {
            binOfs = RAlignValue(iBuf + 1, R_BINARY_ALIGNMENT);
            break;
        }
    }
    return binOfs;
}

//-----------------------------------------------------------------------------
//! @brief バイナリデータ列へのポインタを取得します（static 関数）。
//-----------------------------------------------------------------------------
const ImpBinDataStream* ImpBinDataStream::Get(const void* binTop, const int streamIdx)
{
    const ImpBinDataStream* streamTop = NULL;
    const uint8_t* src = reinterpret_cast<const uint8_t*>(binTop);
    src += 8; // "g3dstrma"
    const int streamCount = *reinterpret_cast<const int*>(src);
    src += sizeof(int);
    if (streamIdx < streamCount)
    {
        src += (4 + 4) * streamIdx; // address, size
        const int streamAddr = *reinterpret_cast<const int*>(src);
        src = reinterpret_cast<const uint8_t*>(binTop) + streamAddr;
        streamTop = reinterpret_cast<const ImpBinDataStream*>(src);
    }
    return streamTop;
}

//-----------------------------------------------------------------------------
//! @brief メモリを確保してファイルをリードします。
//!
//! @param[out] ppBuf 確保したメモリへのポインタを格納します。
//! @param[out] pSize ファイルサイズ（バイト数）を格納します。
//! @param[in] filePath ファイルのパスを指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static RStatus ReadFileToMemory(
    uint8_t** ppBuf,
    int* pSize,
    const std::string& filePath
)
{
    //-----------------------------------------------------------------------------
    // open file
    std::ifstream ifs(filePath.c_str(), std::ios::binary);
    if (!ifs)
    {
        // open failed
        *ppBuf = NULL;
        return RStatus(RStatus::FAILURE, "ファイルを開けません: {0}",
            "The file cannot be opened: {0}", filePath);
    }

    //-----------------------------------------------------------------------------
    // read file
    ifs.seekg(0, std::ios::end);
    *pSize = static_cast<int>(ifs.tellg());
    ifs.seekg(0, std::ios::beg);
    *ppBuf = new uint8_t[*pSize];
    ifs.read(reinterpret_cast<char*>(*ppBuf), *pSize);

    //-----------------------------------------------------------------------------
    // close file
    ifs.close();

    return RStatus::SUCCESS;
}

//=============================================================================
//! @brief インポート用中間ファイルのクラスです。
//=============================================================================
class ImpIntermediateFile
{
public:
    uint8_t* m_FileBuf; //!< ファイルバッファです。
    int m_FileSize; //!< ファイルサイズです。
    bool m_IsBinary; //!< バイナリ形式なら true です。
    int m_BinOfs; //!< 中間ファイルの先頭からバイナリデータまでのオフセットです。

    RXMLElement m_Xml; //!< ルートの XML 要素です。
    const RXMLElement* m_StreamsElem; //!< <stream_array> 要素です。存在しなければ NULL です。

public:
    //! @biref コンストラクタです。
    //!        ファイルをメモリに読み込んで XML を解析します。
    //!
    //! @param[out] status 処理結果を格納します。
    //! @param[in] path 中間ファイルのパスです。
    //!
    ImpIntermediateFile(RStatus& status, const std::string& path)
    : m_FileBuf(NULL),
      m_FileSize(0),
      m_IsBinary(false),
      m_BinOfs(0),
      m_StreamsElem(NULL)
    {
        status = ReadFileToMemory(&m_FileBuf, &m_FileSize, path);
        if (status)
        {
            m_IsBinary = (RGetLowerCaseString(path.substr(path.size() - 1)) == "b");
            m_BinOfs = (m_IsBinary) ? RGetBinaryOffset(m_FileBuf, m_FileSize) : m_FileSize;
            const std::string xmlStr(reinterpret_cast<const char*>(m_FileBuf), m_BinOfs);
            m_Xml.LoadDocument(NULL, xmlStr);
            if (!m_IsBinary)
            {
                const RXMLElement* rootElem = m_Xml.FindElement("nw4f_3dif", false);
                if (rootElem)
                {
                    for (size_t iChild = 0; iChild < rootElem->nodes.size(); ++iChild)
                    {
                        const RXMLElement* childElem = &rootElem->nodes[iChild];
                        if (childElem->name != "file_info")
                        {
                            m_StreamsElem = childElem->FindElement("stream_array", false);
                            break;
                        }
                    }
                }
            }
        }
    }

    //! デストラクタです。
    ~ImpIntermediateFile()
    {
        if (m_FileBuf != NULL)
        {
            delete[] m_FileBuf;
        }
    }

    //! データ列の数を返します。
    size_t GetStreamCount() const
    {
        if (!m_IsBinary)
        {
            return (m_StreamsElem != NULL) ? m_StreamsElem->nodes.size() : 0;
        }
        else
        {
            if (m_BinOfs == m_FileSize)
            {
                return 0;
            }
            else
            {
                const int count = *reinterpret_cast<const int*>(m_FileBuf + m_BinOfs + 8);
                return static_cast<size_t>(count);
            }
        }
    }

    //! @brief float 型のデータ列を取得します。
    //!
    //! @param[out] dst float 配列を格納します。
    //! @param[in] streamIdx データ列のインデックスです。
    //!
    void GetFloatStream(RFloatArray& dst, const int streamIdx) const;

    //! @brief int 型のデータ列を取得します。
    //!
    //! @param[out] dst int 配列を格納します。
    //! @param[in] streamIdx データ列のインデックスです。
    //!
    void GetIntStream(RIntArray& dst, const int streamIdx) const;
};

//-----------------------------------------------------------------------------
//! @brief 中間ファイルから float 型のデータ列を取得します。
//-----------------------------------------------------------------------------
void ImpIntermediateFile::GetFloatStream(RFloatArray& dst, const int streamIdx) const
{
    dst.clear();
    if (!m_IsBinary)
    {
        if (m_StreamsElem != NULL && streamIdx < static_cast<int>(m_StreamsElem->nodes.size()))
        {
            const RXMLElement* streamElem = &m_StreamsElem->nodes[streamIdx];
            if (streamElem->GetAttribute("type") == "float")
            {
                const int valCount = atoi(streamElem->GetAttribute("count").c_str());
                dst.resize(valCount);
                std::istringstream iss(streamElem->text);
                int iVal = 0;
                while (iss && iVal < valCount)
                {
                    iss >> dst[iVal++];
                }
            }
        }
    }
    else
    {
        const ImpBinDataStream* pBinStream =
            ImpBinDataStream::Get(m_FileBuf + m_BinOfs, streamIdx);
        if (pBinStream != NULL && pBinStream->m_Type == ImpBinDataStream::FLOAT)
        {
            const float* pSrc = reinterpret_cast<const float*>(pBinStream->m_Data);
            const int valCount = pBinStream->m_Size / sizeof(float);
            dst.resize(valCount);
            for (int iVal = 0; iVal < valCount; ++iVal)
            {
                dst[iVal] = *pSrc++;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルから int 型のデータ列を取得します。
//-----------------------------------------------------------------------------
void ImpIntermediateFile::GetIntStream(RIntArray& dst, const int streamIdx) const
{
    dst.clear();
    if (!m_IsBinary)
    {
        if (m_StreamsElem != NULL && streamIdx < static_cast<int>(m_StreamsElem->nodes.size()))
        {
            const RXMLElement* streamElem = &m_StreamsElem->nodes[streamIdx];
            if (streamElem->GetAttribute("type") == "int")
            {
                const int valCount = atoi(streamElem->GetAttribute("count").c_str());
                dst.resize(valCount);
                std::istringstream iss(streamElem->text);
                int iVal = 0;
                while (iss && iVal < valCount)
                {
                    iss >> dst[iVal++];
                }
            }
        }
    }
    else
    {
        const ImpBinDataStream* pBinStream =
            ImpBinDataStream::Get(m_FileBuf + m_BinOfs, streamIdx);
        if (pBinStream != NULL && pBinStream->m_Type == ImpBinDataStream::INT)
        {
            const int* pSrc = reinterpret_cast<const int*>(pBinStream->m_Data);
            const int valCount = pBinStream->m_Size / sizeof(int);
            dst.resize(valCount);
            for (int iVal = 0; iVal < valCount; ++iVal)
            {
                dst[iVal] = *pSrc++;
            }
        }
    }
}

//=============================================================================
//! @brief インポート用テクスチャデータのクラスです。
//=============================================================================
class ImpTexData
{
public:
    std::string m_Name; //!< 名前です。
    std::string m_FtxPath; //!< ftx ファイルのパスです。
    std::string m_ImagePath; //!< 画像ファイルのパスです。
    bool m_IsFloat; //!< データ型が浮動小数点数なら true、整数なら false です。

public:
    //! コンストラクタです。
    ImpTexData()
    : m_IsFloat(false)
    {
    }
};

//! @brief インポート用テクスチャデータ配列の定義です。
typedef std::vector<ImpTexData> ImpTexDataArray;

//=============================================================================
//! @brief インポート用サンプラのクラスです。
//=============================================================================
class ImpSampler
{
public:
    //! @brief テクスチャの関連付けに利用するヒント情報を表す列挙型です。
    enum Hint
    {
        ALBEDO,         //!< アルベドマップです。
        OPACITY,        //!< 不透明度マップです。
        EMISSION,       //!< エミッションマップです。
        NORMAL,         //!< 法線マップです。
        TANGENT,        //!< 接線マップです。
        SPECULAR,       //!< スペキュラマップです。
        REFLECTION,     //!< 反射マップです。
        EXTRA,          //!< エクストラマップです。
        HINT_COUNT      //!< ヒント情報の総数です。
    };

    Hint m_Hint; //!< テクスチャの関連付けに利用するヒント情報です。
    int m_HintIndex; //!< テクスチャの関連付けに利用するヒント情報のインデックスです。

    MObject m_TexObj; //!< テクスチャオブジェクト（file、envCube ノード）です。
    MObject m_PlaceObj; //!< place2dTexture ノードのオブジェクトです。
    bool m_HasAlpha; //!< file ノードがアルファ成分を持つなら true です。
    int m_UvHintIdx; //!< UV セットのヒント情報インデックスです。

public:
    //! コンストラクタです。
    ImpSampler(const Hint hint = ALBEDO, const int hintIndex = 0)
    : m_Hint(hint),
      m_HintIndex(hintIndex),
      m_HasAlpha(false),
      m_UvHintIdx(0)
    {
    }
};

//! @brief インポート用サンプラ配列の定義です。
typedef std::vector<ImpSampler> ImpSamplerArray;

//! @brief インポート用サンプラのポインタ配列の定義です。
typedef std::vector<ImpSampler*> ImpSamplerPtrArray;

//=============================================================================
//! @brief インポート用マテリアルのクラスです。
//=============================================================================
class ImpMaterial
{
public:
    std::string m_Name; //!< 名前です。
    bool m_CompressEnable; //!< 最適化で圧縮可能なら true です。
    MColor m_Diffuse;  //!< ディフューズカラーです。
    MColor m_Opacity; //!< 不透明カラーです。
    MColor m_Ambient;  //!< アンビエントカラーです。
    MColor m_Emission; //!< エミッションカラーです。
    MColor m_Specular; //!< スペキュラカラーです。
    bool m_HasSpecular; //!< スペキュラカラーを持つなら true です。

    MObject m_ShaderObj; //!< シェーダのオブジェクトです。
    MObject m_SGObj; //!< シェーディンググループのオブジェクトです。
    ImpSamplerArray m_Samplers; //!< サンプラ配列です。

public:
    //! コンストラクタです。
    ImpMaterial()
    : m_CompressEnable(true),

      m_Diffuse (1.0f, 1.0f, 1.0f),
      m_Opacity (1.0f, 1.0f, 1.0f),
      m_Ambient (0.0f, 0.0f, 0.0f),
      m_Emission(0.0f, 0.0f, 0.0f),
      m_Specular(0.0f, 0.0f, 0.0f),

      m_HasSpecular(true)
    {
    }
};

//! @brief インポート用マテリアル配列の定義です。
typedef std::vector<ImpMaterial> ImpMaterialArray;

//=============================================================================
//! @brief インポート用テクスチャパターンのクラスです。
//=============================================================================
class ImpTexPat
{
public:
    std::string m_Name; //!< テクスチャの名前（拡張子なし）です。
    int m_FrameExtension; //!< フレーム拡張子です。

public:
    //! コンストラクタです。
    explicit ImpTexPat(const std::string& name)
    : m_Name(name),
      m_FrameExtension(1)
    {
        const std::string ext = RGetExtensionFromFilePath(name);
        if (!ext.empty())
        {
            const char c = ext[0];
            if ('0' <= c && c <= '9')
            {
                m_FrameExtension = atoi(ext.c_str());
            }
        }
    }
};

//! @brief インポート用テクスチャパターン配列の定義です。
typedef std::vector<ImpTexPat> ImpTexPatArray;

//=============================================================================
//! @brief LOD 情報のクラスです。
//=============================================================================
class ImpLodInfo
{
public:
    bool m_UsesMayaLod; //!< Maya の lodGroup ノードを使用するなら true です。
    MDagPath m_GroupPath; //!< LOD グループの DAG パスです。
    MDagPathArray m_RootPaths; //!< 各 LOD レベルのルートの DAG パス配列です。
    RBoolArray m_ShapeFlags; //!< 各ボーンをシェイプツリーに含めるなら true となるフラグの配列です。
    bool m_UsesInfluenceTree; //!< インフルエンスツリーを使用するなら true です。
    RBoolArray m_InfluenceFlags; //!< 各ボーンをインフルエンスツリーに含めるなら true となるフラグの配列です。
    MDagPathArray m_BlendGroupPaths; //!< 各 LOD レベルのブレンドシェイプグループの DAG パス配列です。

public:
    //! コンストラクタです。
    ImpLodInfo()
    : m_UsesMayaLod(true),
      m_UsesInfluenceTree(false)
    {
    }

    //! ベースを含む LOD 数を返します。
    int GetLodCount() const
    {
        return RMax(static_cast<int>(m_RootPaths.length()), 1);
    }

    //! ボーンをシェイプツリーに含めるなら true を返します。
    bool GetShapeFlag(const size_t iBone) const
    {
        return (iBone < m_ShapeFlags.size()) ? m_ShapeFlags[iBone] : true;
    }

    //! ボーンをインフルエンスツリーに含めるなら true を返します。
    bool GetInfluenceFlag(const size_t iBone) const
    {
        return (iBone < m_InfluenceFlags.size()) ? m_InfluenceFlags[iBone] : false;
    }
};

//-----------------------------------------------------------------------------
//! @brief シェイプ配列要素からベースを含む LOD 数を取得します。
//!
//! @param[in] shapesElem シェイプ配列要素です（存在しなければ NULL）。
//!
//! @return ベースを含む LOD 数を返します。
//-----------------------------------------------------------------------------
static int GetLodCount(const RXMLElement* shapesElem)
{
    int lodCount = 1;
    if (shapesElem != NULL)
    {
        for (size_t iShape = 0; iShape < shapesElem->nodes.size(); ++iShape)
        {
            const RXMLElement* shapeElem = &shapesElem->nodes[iShape];
            const RXMLElement* meshsElem = shapeElem->FindElement("mesh_array", false); // 3.0.0 で追加
            if (meshsElem != NULL)
            {
                const int meshCount = static_cast<int>(meshsElem->nodes.size());
                if (meshCount > lodCount)
                {
                    lodCount = meshCount;
                }
            }
        }
    }
    return lodCount;
}

//-----------------------------------------------------------------------------
//! @brief LOD 情報を取得します。
//!
//! @param[in,out] lodInfo LOD 情報を格納します。
//! @param[in] shapesElem シェイプ配列要素です（存在しなければ NULL）。
//! @param[in] skeletonElem スケルトン要素です（存在しなければ NULL）。
//-----------------------------------------------------------------------------
static void GetLodInfo(
    ImpLodInfo& lodInfo,
    const RXMLElement* shapesElem,
    const RXMLElement* skeletonElem
)
{
    //-----------------------------------------------------------------------------
    // シェイプが使用しているボーン名の配列を取得します。
    RStringArray shapeBoneNames;
    if (shapesElem != NULL)
    {
        for (size_t iShape = 0; iShape < shapesElem->nodes.size(); ++iShape)
        {
            const RXMLElement* shapeElem = &shapesElem->nodes[iShape];
            const RXMLElement* infoElem = shapeElem->FindElement("shape_info");
            shapeBoneNames.push_back(infoElem->GetAttribute("bone_name"));
        }
    }

    //-----------------------------------------------------------------------------
    // スケルトン要素を解析します。
    if (skeletonElem != NULL)
    {
        const RXMLElement* bonesElem = skeletonElem->FindElement("bone_array");
        const size_t boneCount = bonesElem->nodes.size();
        RStringArray boneNames;
        RIntArray parentIdxs;
        lodInfo.m_ShapeFlags     = RBoolArray(boneCount, false);
        lodInfo.m_InfluenceFlags = RBoolArray(boneCount, false);

        //-----------------------------------------------------------------------------
        // ボーンについて 1 回目のループ
        for (size_t iBone = 0; iBone < boneCount; ++iBone)
        {
            const RXMLElement* boneElem = &bonesElem->nodes[iBone];

            const std::string boneName = boneElem->GetAttribute("name");
            boneNames.push_back(boneName);
            const std::string parentName = boneElem->GetAttribute("parent_name");
            const int iParent = RFindValueInArray(boneNames, parentName);
            parentIdxs.push_back(iParent);

            //-----------------------------------------------------------------------------
            // シェイプであるボーンの子孫もすべてシェイプツリーに含めます。
            //lodInfo.m_ShapeFlags[iBone] = (boneElem->GetAttribute("rigid_body") == "true");
            lodInfo.m_ShapeFlags[iBone] = (RFindValueInArray(shapeBoneNames, boneName) != -1);
            if (!lodInfo.m_ShapeFlags[iBone])
            {
                int iCur = iParent;
                while (iCur != -1)
                {
                    if (lodInfo.m_ShapeFlags[iCur])
                    {
                        lodInfo.m_ShapeFlags[iBone] = true;
                        break;
                    }
                    iCur = parentIdxs[iCur];
                }
            }

            //-----------------------------------------------------------------------------
            // インフルエンスであるボーンの子孫もすべてインフルエンスツリーに含めます。
            int smoothMtxIdx;
            int rigidMtxIdx;
            std::istringstream(boneElem->GetAttribute("matrix_index")) >> smoothMtxIdx >> rigidMtxIdx;
            lodInfo.m_InfluenceFlags[iBone] = (smoothMtxIdx != -1 || rigidMtxIdx != -1);
            if (!lodInfo.m_InfluenceFlags[iBone])
            {
                int iCur = iParent;
                while (iCur != -1)
                {
                    if (lodInfo.m_InfluenceFlags[iCur])
                    {
                        lodInfo.m_InfluenceFlags[iBone] = true;
                        break;
                    }
                    iCur = parentIdxs[iCur];
                }
            }
        }

        //-----------------------------------------------------------------------------
        // ボーンについて 2 回目のループ
        for (size_t iBone = 0; iBone < boneCount; ++iBone)
        {
            //-----------------------------------------------------------------------------
            // シェイプであるボーンの先祖もすべてシェイプツリーに含めます。
            const int iParent = parentIdxs[iBone];
            if (lodInfo.m_ShapeFlags[iBone])
            {
                int iCur = iParent;
                while (iCur != -1)
                {
                    lodInfo.m_ShapeFlags[iCur] = true;
                    iCur = parentIdxs[iCur];
                }
            }

            //-----------------------------------------------------------------------------
            // インフルエンスであるボーンの先祖もすべてインフルエンスツリーに含めます。
            if (lodInfo.m_InfluenceFlags[iBone])
            {
                lodInfo.m_UsesInfluenceTree = true;
                int iCur = iParent;
                while (iCur != -1)
                {
                    lodInfo.m_InfluenceFlags[iCur] = true;
                    iCur = parentIdxs[iCur];
                }
            }
        }
        //cerr << "lod info: " << endl << lodInfo.m_ShapeFlags << endl << lodInfo.m_InfluenceFlags << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief LOD レベルのルート名を返します。
//!
//! @param[in] iLod LOD インデックスです。
//!
//! @return LOD レベルのルート名を返します。
//-----------------------------------------------------------------------------
static std::string GetLodLevelRootName(const int iLod)
{
    return (iLod == 0) ? "Base" : RGetNumberString(iLod, "LOD%d");
}

//-----------------------------------------------------------------------------
//! @brief LOD グループと各 LOD レベルのルートを作成します。
//!
//! @param[in,out] lodInfo LOD 情報を格納します。
//! @param[in] lodCount ベースを含む LOD 数です。
//-----------------------------------------------------------------------------
static void CreateLodGroup(ImpLodInfo& lodInfo, const int lodCount)
{
    //-----------------------------------------------------------------------------
    // LOD グループを作成します。
    const std::string groupType = (lodInfo.m_UsesMayaLod) ? "lodGroup" : "transform";
    MFnDagNode dagFn;
    dagFn.create(groupType.c_str(), "lodGroup#");
    dagFn.getPath(lodInfo.m_GroupPath);
    //MObject lodGroupObj = lodInfo.m_GroupPath.node();
    const MString lodGroupName = lodInfo.m_GroupPath.partialPathName();

    //-----------------------------------------------------------------------------
    // 各 LOD レベルのルートを作成します。
    MString parentCmd = "parent ";
    for (int iLod = 0; iLod < lodCount; ++iLod)
    {
        const std::string rootName = GetLodLevelRootName(iLod);
        MDagPath levelRootPath;
        dagFn.create("transform", rootName.c_str());
        //dagFn.create("transform", rootName.c_str(), lodGroupObj);
            // ↑ m_UsesMayaLod の場合、作成時に親を指定すると Maya がクラッシュ
        dagFn.getPath(levelRootPath);
        lodInfo.m_RootPaths.append(levelRootPath);
        parentCmd += levelRootPath.partialPathName() + " ";
    }

    // 各 LOD レベルのルートを LOD グループの子にします。
    MGlobal::executeCommand(parentCmd + lodGroupName);

    // 各 LOD レベルのルートの名前を再設定します（指定した名前で作成できなかった場合のため）。
    for (int iLod = 0; iLod < lodCount; ++iLod)
    {
        const MDagPath& levelRootPath = lodInfo.m_RootPaths[iLod];
        const std::string rootName = GetLodLevelRootName(iLod);
        if (MFnDagNode(levelRootPath.node()).name() != rootName.c_str())
        {
            MGlobal::executeCommand("rename " + levelRootPath.partialPathName() + " " +
                rootName.c_str());
        }
    }

    //-----------------------------------------------------------------------------
    // Maya の lodGroup ノードのアトリビュートを設定します。
    if (lodInfo.m_UsesMayaLod)
    {
        const MString cameraShapeName = "perspShape";
        MGlobal::executeCommand("connectAttr " + cameraShapeName + ".worldMatrix " +
            lodGroupName + ".cameraMatrix");
        for (int iLod = 0; iLod < lodCount; ++iLod)
        {
            MGlobal::executeCommand("setAttr " + lodGroupName + ".displayLevel[" + iLod + "] 0");
        }
        MGlobal::executeCommand("setAttr " + lodGroupName + ".worldSpace 1");
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya の lodGroup ノードのしきい値を調整します。
//!
//! @param[in] lodInfo LOD 情報です。
//-----------------------------------------------------------------------------
static void AdjustLodGroupThreshold(const ImpLodInfo& lodInfo)
{
    // 参考 Maya2011/scripts/others/performSetupLod.mel
    const int lodCount = lodInfo.GetLodCount();
    if (lodCount >= 2 && lodInfo.m_UsesMayaLod)
    {
        double cameraDistance;
        MFnDependencyNode(lodInfo.m_GroupPath.node()).findPlug("distance").getValue(cameraDistance);

        const int thresholdCount = lodCount - 1;
        const int midElement = (thresholdCount >= 2) ? (thresholdCount - 2) / 2 : 0;

        MDoubleArray distances;
        for (int iThreshold = 0; iThreshold < thresholdCount; ++iThreshold)
        {
            distances.append(static_cast<double>((iThreshold + 1) * (iThreshold + 1)));
        }

        const double meanDist = (midElement + 1 < thresholdCount) ?
            (distances[midElement] + distances[midElement + 1]) / 2.0 :
            distances[midElement] / 2.0;
        const double scaling = cameraDistance / meanDist;

        const MString lodGroupName = lodInfo.m_GroupPath.partialPathName();
        for (int iThreshold = 0; iThreshold < thresholdCount; ++iThreshold)
        {
            MGlobal::executeCommand("setAttr " + lodGroupName + ".threshold[" + iThreshold + "] " +
                (distances[iThreshold] * scaling));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief プロセスログ配列を解析します。
//!        ボーン圧縮されていればスキンシェイプの別オブジェクト化を有効にします。
//!
//! @param[in,out] separatesSkinObject スキンシェイプをワールド下の別オブジェクトとして作成するなら true を格納します。
//! @param[in] processLogsElem プロセスログ配列です。
//-----------------------------------------------------------------------------
static void ParseProcessLogs(bool& separatesSkinObject, const RXMLElement* processLogsElem)
{
    for (size_t iLog = 0; iLog < processLogsElem->nodes.size(); ++iLog)
    {
        const RXMLElement* processLogElem = &processLogsElem->nodes[iLog];
        const std::string process = processLogElem->GetAttribute("process");
        if (process == "compress_bone_cull" ||
            process == "compress_bone_merge")
        {
            separatesSkinObject = true;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータを解析してカスタムアトリビュートを設定します。
//!
//! @param[in] nodeName 対象ノードのノード名です。
//! @param[in] userDataElem ユーザーデータ要素です。
//! @param[in] iData ユーザーデータのインデックスです。
//-----------------------------------------------------------------------------
static void ParseUserData(
    const MString& nodeName,
    const RXMLElement* userDataElem,
    const int iData
)
{
    const std::string name = userDataElem->GetAttribute("name");
    std::string dataStr = name + "\"";

    const RXMLElement* valueElem = &userDataElem->nodes[0];
    const int valCount = atoi(valueElem->GetAttribute("count").c_str());
    if (valueElem->name == "user_int"   ||
        valueElem->name == "user_float")
    {
        dataStr += std::string((valueElem->name == "user_int") ? "i" : "f") + " " +
            RGetNumberString(valCount);
        std::istringstream iss(valueElem->text);
        for (int iVal = 0; iVal < valCount; ++iVal)
        {
            std::string value;
            iss >> value;
            dataStr += " " + value;
        }
    }
    else
    {
        dataStr += std::string((valueElem->name == "user_string") ? "s" : "w") + " " +
            RGetNumberString(valCount) + " ";
        for (size_t iStr = 0; iStr < valueElem->nodes.size(); ++iStr)
        {
            if (iStr > 0)
            {
                dataStr += "\n";
            }
            dataStr += RDecodeXmlString(RGetShiftJisFromUtf8(valueElem->nodes[iStr].text));
        }
    }

    MGlobal::executeCommand("setAttr " + nodeName + ".nw4fUserData[" + iData + "] -type \"string\" \"" +
        REscapeString(dataStr).c_str() + "\"");
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータ配列を解析してカスタムアトリビュートを設定します。
//!
//! @param[in] nodeName 対象ノードのノード名です。
//! @param[in] userDatasElem ユーザーデータ配列要素です。
//-----------------------------------------------------------------------------
static void ParseUserDatas(
    const MString& nodeName,
    const RXMLElement* userDatasElem
)
{
    MGlobal::executeCommand(
        "if (!`attributeQuery -n " + nodeName + " -ex \"nw4fUserDataCount\"`)\n" +
        "{\n" +
        "    addAttr -ln \"nw4fUserDataCount\" -at \"short\" -dv 0 -h 1 " + nodeName + ";\n" +
        "}\n");
    MGlobal::executeCommand("setAttr " + nodeName + ".nw4fUserDataCount " +
        static_cast<int>(userDatasElem->nodes.size()));

    MGlobal::executeCommand(
        "if (`attributeQuery -n " + nodeName + " -ex \"nw4fUserData\"`)\n" +
        "{\n" +
        "    deleteAttr -attribute \"nw4fUserData\" " + nodeName + ";\n" +
        "}\n" +
        "addAttr -multi -ln \"nw4fUserData\" -dt \"string\" -h 1 " + nodeName + ";\n");

    for (size_t iData = 0; iData < userDatasElem->nodes.size(); ++iData)
    {
        ParseUserData(nodeName, &userDatasElem->nodes[iData], static_cast<int>(iData));
    }
}

//-----------------------------------------------------------------------------
//! @brief 名前でテクスチャデータを検索します。
//!
//! @param[in] texDatas テクスチャデータ配列です。
//! @param[in] name テクスチャ名です。
//!
//! @return インデックスを返します。見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindTexDataByName(const ImpTexDataArray& texDatas, const std::string& name)
{
    for (int iTexData = 0; iTexData < static_cast<int>(texDatas.size()); ++iTexData)
    {
        if (texDatas[iTexData].m_Name == name)
        {
            return iTexData;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルの情報を取得します。
//!
//! @param[out] format フォーマットを格納します。
//! @param[in] ftxPath ftx ファイルのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetFtxInfo(
    std::string& format,
    const std::string& ftxPath
)
{
    //-----------------------------------------------------------------------------
    // 出力を初期化します。
    format = "?";

    //-----------------------------------------------------------------------------
    // ファイルの先頭部分をリードします。
    std::ifstream ifs(ftxPath.c_str(), ios_base::binary);
    if (!ifs)
    {
        DisplayImportError("ファイルを開けません: {0}", "The file cannot be opened: {0}", ftxPath);
        return MS::kFailure;
    }

    const int READ_BUF_SIZE = 1024;
    char* pReadBuf = new char[READ_BUF_SIZE];
    ifs.read(pReadBuf, READ_BUF_SIZE);
    std::string header(pReadBuf, static_cast<size_t>(ifs.gcount()));
    ifs.close();
    delete[] pReadBuf;

    //-----------------------------------------------------------------------------
    // <texture_info> 要素を取得します。
    bool isValid = false;
    const size_t infoOfs = header.find("<texture_info");
    if (infoOfs != std::string::npos)
    {
        header = header.substr(infoOfs);
        const size_t infoEndOfs = header.find(">");
        if (infoEndOfs != std::string::npos)
        {
            header = header.substr(0, infoEndOfs + 1);
            //cerr << "ftx header: [" << header << "]" << endl;
            isValid = true;
        }
    }
    if (!isValid)
    {
        DisplayImportError("不正な ftx ファイルです: {0}", "ftx file is wrong: {0}", ftxPath);
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // <texture_info> 要素を解析します。
    format = RGetXmlAttr(header, "quantize_type");

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ画像ファイルを作成します。
//!
//! @param[out] texData テクスチャデータを格納します。
//! @param[in] texName テクスチャ名です。
//! @param[in] inputFolder 中間ファイルのフォルダのパスです。
//! @param[in] imagesFolder 画像ファイル出力フォルダのパスです。
//! @param[in] texcvtrPath 3D テクスチャーコンバーターのパスです。
//! @param[in] createsTexture テクスチャファイルを作成するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CreateTextureImageFile(
    ImpTexData& texData,
    const std::string& texName,
    const std::string& inputFolder,
    const std::string& imagesFolder,
    const std::string& texcvtrPath,
    const bool createsTexture
)
{
    //-----------------------------------------------------------------------------
    // テクスチャ名が空なら何もしません。
    if (texName.empty())
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // ftxb 優先で ftx ファイルを探します。
    const int FOLDER_COUNT = 2;
    const std::string ftxFolders[FOLDER_COUNT] =
    {
        inputFolder + "textures/",
        inputFolder
    };

    std::string ftxPath;
    for (int iFolder = 0; iFolder < FOLDER_COUNT; ++iFolder)
    {
        const std::string ftxPathTop = ftxFolders[iFolder] + texName;
        ftxPath = ftxPathTop + "." + FtxbExtension;
        if (RFileExists(ftxPath))
        {
            break;
        }
        ftxPath = ftxPathTop + "." + FtxaExtension;
        if (RFileExists(ftxPath))
        {
            break;
        }
    }
    const bool ftxExists = RFileExists(ftxPath);
    if (!ftxExists)
    {
        if (createsTexture)
        {
            DisplayImportError("ファイルを開けません: {0}", "The file cannot be opened: {0}",
                texName + "." + FtxbExtension);
            return MS::kFailure;
        }
    }
    //cerr << "ftx: " << ftxPath << endl;

    //-----------------------------------------------------------------------------
    // ftx ファイルの情報を取得します。
    if (ftxExists)
    {
        std::string format;
        GetFtxInfo(format, ftxPath);
        //MGlobal::displayInfo(MString("ftx info: ") + format.c_str());
        texData.m_IsFloat = (format.find("float_") == 0);
    }
    else
    {
        texData.m_IsFloat = (!RFileExists(imagesFolder + texName + ".tga") &&
            RFileExists(imagesFolder + texName + ".dds"));
    }

    //-----------------------------------------------------------------------------
    // ftx ファイルをテクスチャファイルに変換します。
    texData.m_Name = texName;
    texData.m_FtxPath = ftxPath;
    texData.m_ImagePath = imagesFolder + texName + (texData.m_IsFloat ? ".dds" : ".tga");

    if (ftxExists && createsTexture)
    {
        const std::string cmd = "\"" + texcvtrPath + "\" -s -o \"" + texData.m_ImagePath + "\" \"" +
            texData.m_FtxPath + "\"";

        std::string errMsg;
        const int exitCode = RExecProcessWithPipe(nullptr, nullptr, &errMsg, cmd.c_str(), SW_HIDE);
        //cerr << "texcvtr msg:" << endl << errMsg << endl;
        if (exitCode != 0)
        {
            DisplayImportError("ftx ファイルの変換に失敗しました: {0}", "Cannot convert ftx file: {0}",
                texData.m_FtxPath + "\n" + errMsg);
            return MS::kFailure;
        }
        MGlobal::displayInfo(MString("Created: ") + texData.m_ImagePath.c_str());
    }

    //-----------------------------------------------------------------------------
    // float 型テクスチャなら表示用の mll をロードします。
    if (texData.m_IsFloat)
    {
        MGlobal::executeCommand("loadPlugin -qt \"ddsFloatReader.mll\"");
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief Raw カラースペースが使用可能か判定します。
//!
//! @return Raw カラースペースが使用可能なら true を返します。
//-----------------------------------------------------------------------------
static bool CheckColorSpaceRaw()
{
#if (MAYA_API_VERSION >= 201600)
    MStringArray colorSpaces;
    MGlobal::executeCommand("colorManagementPrefs -q -inputSpaceNames", colorSpaces);
    return (YFindValueInArray(colorSpaces, ColorSpaceRaw) != -1);
#else
    return false;
#endif
}

//-----------------------------------------------------------------------------
//! @brief file ノードのカラースペースを設定します。
//!
//! @param[in] texFn file ノードのファンクションノードです。
//! @param[in] imagePath 画像ファイルのパスです。
//! @param[in] hint サンプラーのヒント情報です。
//! @param[in] isColorSpaceRawAvailable Raw カラースペースが使用可能なら true です。
//-----------------------------------------------------------------------------
static void SetFileColorSpace(
    const MFnDependencyNode& texFn,
    const std::string& imagePath,
    const ImpSampler::Hint hint,
    const bool isColorSpaceRawAvailable
)
{
#if (MAYA_API_VERSION >= 201600)
    MString colorSpace = ColorSpaceRaw;
    if (hint != ImpSampler::NORMAL || !isColorSpaceRawAvailable)
    {
        const MString name = RGetFileNameFromFilePath(imagePath).c_str();
        const MString cmd = "colorManagementFileRules -evaluate \"" + name + "\"";
        colorSpace = MGlobal::executeCommandStringResult(cmd);
    }
    texFn.findPlug("colorSpace").setString(colorSpace);
#else
    MString nodeName = texFn.name();
    R_UNUSED_VARIABLE(nodeName);
    R_UNUSED_VARIABLE(imagePath);
    R_UNUSED_VARIABLE(hint);
    R_UNUSED_VARIABLE(isColorSpaceRawAvailable);
#endif
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報でサンプラを検索します。
//!
//! @param[in] pSamplers サンプラのポインタ配列です。
//! @param[in] hint ヒント情報です。
//!
//! @return インデックスを返します。見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindSamplerByHint(const ImpSamplerPtrArray& pSamplers, const ImpSampler::Hint hint)
{
    for (int iSampler = 0; iSampler < static_cast<int>(pSamplers.size()); ++iSampler)
    {
        if (pSamplers[iSampler]->m_Hint == hint)
        {
            return iSampler;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief サンプラのポインタの比較関数です。
//!
//! @param[in] r1 サンプラ 1 のポインタです。
//! @param[in] r2 サンプラ 2 のポインタです。
//!
//! @return サンプラ 1 がサンプラ 2 より小さければ true を返します。
//-----------------------------------------------------------------------------
static bool RSamplerPtrLess(ImpSampler*& r1, ImpSampler*& r2)
{
    if (r1->m_Hint != r2->m_Hint)
    {
        return (r1->m_Hint < r2->m_Hint);
    }
    else
    {
        return (r1->m_HintIndex < r2->m_HintIndex);
    }
}

//-----------------------------------------------------------------------------
// サンプラの name と hint の値のデータです。
//-----------------------------------------------------------------------------
static const char* const s_ImpSamplerNameHintStrs[ImpSampler::HINT_COUNT][2] =
{
    { "_a", "albedo"     },
    { "_o", "opacity"    },
    { "_e", "emission"   },
    { "_n", "normal"     },
    { "_t", "tangent"    },
    { "_s", "specular"   },
    { "_r", "reflection" },
    { "_x", "extra"      },
};

//-----------------------------------------------------------------------------
//! @brief デフォルトのサンプラ名を取得します。
//!
//! @param[in] hint ヒント情報です。
//! @param[in] hintIndex ヒント情報のインデックスです。
//!
//! @return デフォルトのサンプラ名を返します。
//-----------------------------------------------------------------------------
static std::string GetDefaultSamplerName(const ImpSampler::Hint hint, const int hintIndex)
{
    return ((hint < ImpSampler::HINT_COUNT) ? s_ImpSamplerNameHintStrs[hint][0] : "") +
        RGetNumberString(hintIndex);
}

//-----------------------------------------------------------------------------
//! @brief サンプラを解析して、ftx ファイルを画像ファイルに変換し、file ノードを作成します。
//!
//! @param[in,out] samplers サンプラ配列です。
//! @param[in,out] texDatas テクスチャデータ配列です。
//! @param[in] samplerElem サンプラ要素です。
//! @param[in] orgSrtsElem オリジナル SRT 配列要素です。
//! @param[in] inputFolder 中間ファイルのフォルダのパスです。
//! @param[in] imagesFolder 画像ファイル出力フォルダのパスです。
//! @param[in] texcvtrPath 3D テクスチャーコンバーターのパスです。
//! @param[in] createsTexture テクスチャファイルを作成するなら true です。
//! @param[in] isColorSpaceRawAvailable Raw カラースペースが使用可能なら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseSampler(
    ImpSamplerArray& samplers,
    ImpTexDataArray& texDatas,
    const RXMLElement* samplerElem,
    const RXMLElement* orgSrtsElem,
    const std::string& inputFolder,
    const std::string& imagesFolder,
    const std::string& texcvtrPath,
    const bool createsTexture,
    const bool isColorSpaceRawAvailable
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // まだ取得していないテクスチャなら ftx ファイルを画像ファイルに変換します。
    const std::string texName = samplerElem->GetAttribute("tex_name");
    int iTexData = FindTexDataByName(texDatas, texName);
    if (iTexData == -1)
    {
        iTexData = static_cast<int>(texDatas.size());
        texDatas.push_back(ImpTexData());
        status = CreateTextureImageFile(texDatas.back(), texName,
            inputFolder, imagesFolder, texcvtrPath, createsTexture);
        CheckStatus(status);
    }
    const ImpTexData& texData = texDatas[iTexData];

    //-----------------------------------------------------------------------------
    // ヒント情報を取得します。
    const std::string samplerName = samplerElem->GetAttribute("name");
    const std::string hintStr = samplerElem->GetAttribute("hint");
    ImpSampler::Hint hint = ImpSampler::EXTRA;
    static const char* const hintNames[] =
    {
        "albedo",
        "opacity",
        "emission",
        "normal",
        "tangent",
        "specular",
        "reflection"
    };
    for (int iHint = 0; iHint <= ImpSampler::REFLECTION; ++iHint)
    {
        if (hintStr.find(hintNames[iHint]) == 0)
        {
            hint = static_cast<ImpSampler::Hint>(iHint);
            break;
        }
    }
    const int hintIndex = GetStringSuffixNumber(hintStr);

    //-----------------------------------------------------------------------------
    // サンプラを追加します。
    samplers.push_back(ImpSampler(hint, hintIndex));
    ImpSampler& sampler = samplers.back();

    //-----------------------------------------------------------------------------
    // file ノードを作成します。
    #if (MAYA_API_VERSION >= 201600)
    MGlobal::executeCommand("shadingNode -asTexture -isColorManaged file");
    #else
    MGlobal::executeCommand("shadingNode -asTexture file");
    #endif
    sampler.m_TexObj = GetSelectedNodeObject();
    MFnDependencyNode texFn(sampler.m_TexObj);
    texFn.findPlug("fileTextureName").setValue(texData.m_ImagePath.c_str());
    texFn.findPlug("fileHasAlpha").getValue(sampler.m_HasAlpha); // 画像ファイルにアルファがあってもすべて 0xff なら false
    SetFileColorSpace(texFn, texData.m_ImagePath, hint, isColorSpaceRawAvailable);
    //cerr << "fileHasAlpha: " << texData.m_ImagePath << ": " << sampler.m_HasAlpha << endl;

    //-----------------------------------------------------------------------------
    // file ノードにサンプラアトリビュートを設定します。
    if (hint == ImpSampler::EXTRA || samplerName != GetDefaultSamplerName(hint, hintIndex))
    {
        MGlobal::executeCommand("addAttr -ln \"nw4fSamplerName\" -dt \"string\" -h 1 " +
            texFn.name());
        MGlobal::executeCommand("setAttr " + texFn.name() + ".nw4fSamplerName -typ \"string\" \"" +
            samplerName.c_str() + "\"");
    }
    if (hint == ImpSampler::EXTRA && hintStr != samplerName)
    {
        MGlobal::executeCommand("addAttr -ln \"nw4fSamplerHint\" -dt \"string\" -h 1 " +
            texFn.name());
        MGlobal::executeCommand("setAttr " + texFn.name() + ".nw4fSamplerHint -typ \"string\" \"" +
            hintStr.c_str() + "\"");
    }

    //-----------------------------------------------------------------------------
    // place2dTexture ノードを作成します。
    MGlobal::executeCommand("shadingNode -asUtility place2dTexture");
    sampler.m_PlaceObj = GetSelectedNodeObject();
    MFnDependencyNode placeFn(sampler.m_PlaceObj);
    MGlobal::executeCommand("defaultNavigation -quiet 1 -connectToExisting -source " +
        placeFn.name() + " -destination " + texFn.name());

    const RXMLElement* wrapElem = samplerElem->FindElement("wrap");
    const std::string wrapU = wrapElem->GetAttribute("u");
    const std::string wrapV = wrapElem->GetAttribute("v");
    placeFn.findPlug("wrapU").setValue(wrapU != "clamp");
    placeFn.findPlug("wrapV").setValue(wrapV != "clamp");
    placeFn.findPlug("mirrorU").setValue(wrapU == "mirror" || wrapU == "mirror_once");
    placeFn.findPlug("mirrorV").setValue(wrapV == "mirror" || wrapV == "mirror_once");

    float scaleU = 1.0f;
    float scaleV = 1.0f;
    float rotate = 0.0f;
    float translateU = 0.0f;
    float translateV = 0.0f;
    if (orgSrtsElem != NULL)
    {
        for (size_t iSrt = 0; iSrt < orgSrtsElem->nodes.size(); ++iSrt)
        {
            const RXMLElement* orgSrtElem = &orgSrtsElem->nodes[iSrt];
            if (orgSrtElem->GetAttribute("hint") == hintStr)
            {
                std::istringstream(orgSrtElem->GetAttribute("scale")) >> scaleU >> scaleV;
                std::istringstream(orgSrtElem->GetAttribute("rotate")) >> rotate;
                std::istringstream(orgSrtElem->GetAttribute("translate")) >> translateU >> translateV;
                sampler.m_UvHintIdx = GetStringSuffixNumber(orgSrtElem->GetAttribute("uv_hint", "uv0", false)); // 3.4.0 で追加
                break;
            }
        }
    }
    placeFn.findPlug("repeatU").setValue(scaleU);
    placeFn.findPlug("repeatV").setValue(scaleV);
    placeFn.findPlug("rotateFrame").setValue(rotate * R_M_DEG_TO_RAD);
    placeFn.findPlug("translateFrameU").setValue(translateU);
    placeFn.findPlug("translateFrameV").setValue(translateV);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 名前でマテリアルを検索します。
//!
//! @param[in] materials マテリアル配列です。
//! @param[in] name マテリアル名です。
//!
//! @return インデックスを返します。見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindMaterialByName(const ImpMaterialArray& materials, const std::string& name)
{
    for (int iMat = 0; iMat < static_cast<int>(materials.size()); ++iMat)
    {
        if (materials[iMat].m_Name == name)
        {
            return iMat;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief サンプラのヒント情報に対応したマテリアルのプラグを取得します。
//!
//! @param[in] material マテリアルです。
//! @param[in] hint サンプラのヒント情報です。
//!
//! @return プラグを返します。存在しなければ NULL プラグを返します。
//-----------------------------------------------------------------------------
static MPlug GetMaterialPlugForHint(const ImpMaterial& material, const std::string& hint)
{
    std::string attr;
    if (hint == "albedo")
    {
        attr = "color";
    }
    else if (hint == "opacity")
    {
        attr = "transparency";
    }
    else if (hint == "emission")
    {
        attr = "incandescence";
    }
    else if (hint == "normal")
    {
        attr = "normalCamera";
    }
    else if (hint == "specular")
    {
        if (material.m_HasSpecular)
        {
            attr = "specularColor";
        }
    }
    else if (hint == "reflection")
    {
        if (material.m_HasSpecular)
        {
            attr = "reflectedColor";
        }
    }

    MPlug plug;
    if (!attr.empty())
    {
        const MFnDependencyNode shaderFn(material.m_ShaderObj);
        plug = shaderFn.findPlug(attr.c_str());
    }
    return plug;
}

//-----------------------------------------------------------------------------
//! @brief サンプラのヒント情報（インデックス付き）に対応したファイルノードを取得します。
//!
//! @param[in] material マテリアルです。
//! @param[in] hintStr サンプラのヒント情報（インデックス付き）です。
//!
//! @return ファイルノードを返します。存在しなければ kNullObj を返します。
//-----------------------------------------------------------------------------
static MObject GetFileNodeForHintString(const ImpMaterial& material, const std::string& hintStr)
{
    const std::string hint = GetNoSuffixNumberString(hintStr);
    const int hintIndex = GetStringSuffixNumber(hintStr);
    const MPlug plug = GetMaterialPlugForHint(material, hint);
    if (!plug.isNull())
    {
        MObjectArray fileObjs;
        GetFileNodesForPlug(fileObjs, plug);
        if (hintIndex < static_cast<int>(fileObjs.length()))
        {
            return fileObjs[hintIndex];
        }
    }
    return MObject::kNullObj;
}

//-----------------------------------------------------------------------------
//! @brief オリジナルマテリアルのカラーを取得します。
//!
//! @param[out] dst カラーを格納します。
//! @param[in] orgColorsElem オリジナルカラー要素です。
//! @param[in] hint ヒント情報です。
//-----------------------------------------------------------------------------
static void ParseOriginalColor(
    MColor& dst,
    const RXMLElement* orgColorsElem,
    const std::string& hint
)
{
    for (size_t iCol = 0; iCol < orgColorsElem->nodes.size(); ++iCol)
    {
        const RXMLElement* orgColorElem = &orgColorsElem->nodes[iCol];
        if (orgColorElem->GetAttribute("hint") == hint)
        {
            std::istringstream(orgColorElem->GetAttribute("color"))
                >> dst.r >> dst.g >> dst.b;
            break;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアルにテクスチャを接続します。
//!
//! @param[in] shaderFn シェーダのファンクションノードです。
//! @param[in] pMatSamplers マテリアルの全サンプラのポインタ配列です。
//! @param[in] hint 接続先のヒント情報です。
//! @param[in] color テクスチャに乗算するカラーです。
//! @param[in] alphaIsOpacity テクスチャのアルファ成分を不透明度として使用するなら true です。
//-----------------------------------------------------------------------------
static void ConnectTexture(
    const MFnDependencyNode& shaderFn,
    const ImpSamplerPtrArray& pMatSamplers,
    const ImpSampler::Hint hint,
    const MColor color,
    const bool alphaIsOpacity
)
{
    //-----------------------------------------------------------------------------
    // ヒント情報に対応するサンプラのポインタ配列を取得します。
    ImpSamplerPtrArray pSrcSamplers;
    for (size_t iSampler = 0; iSampler < pMatSamplers.size(); ++iSampler)
    {
        ImpSampler* pSampler = pMatSamplers[iSampler];
        if (pSampler->m_Hint == hint)
        {
            pSrcSamplers.push_back(pSampler);
        }
    }

    //-----------------------------------------------------------------------------
    // マテリアルにテクスチャを接続します。
    if (!pSrcSamplers.empty())
    {
        static const char* const dstPlugNames[] =
        {
            "color", // ALBEDO
            "transparency", // OPACITY
            "incandescence", // EMISSION
            "normalCamera", // NORMAL
            "", // TANGENT
            "specularColor", // SPECULAR
            "reflectedColor", // REFLECTION
            "translucenceDepth", // EXTRA
        };
        if (strlen(dstPlugNames[hint]) == 0)
        {
            return;
        }
        MString dstPlugName = shaderFn.name() + "." + dstPlugNames[hint];

        //-----------------------------------------------------------------------------
        // 法線マップなら bump2d ノードを作成します。
        if (hint == ImpSampler::NORMAL)
        {
            MGlobal::executeCommand("shadingNode -asUtility bump2d");
            MObject bump2dObj = GetSelectedNodeObject();
            MFnDependencyNode bump2dFn(bump2dObj);
            bump2dFn.findPlug("bumpInterp").setValue(1); // 接線空間法線
            MGlobal::executeCommand(MString("connectAttr ") +
                bump2dFn.name() + ".outNormal " + dstPlugName);
            dstPlugName = bump2dFn.name() + ".bumpValue";
        }

        //-----------------------------------------------------------------------------
        // カラーゲイン、アルファ値のゲインを設定します。
        const ImpSampler& firstSampler = *pSrcSamplers[0];
        MFnDependencyNode firstTexFn(firstSampler.m_TexObj);
        if (hint != ImpSampler::OPACITY)
        {
            const MPlug colorGainPlug = firstTexFn.findPlug("colorGain");
            colorGainPlug.child(0).setValue(color.r);
            colorGainPlug.child(1).setValue(color.g);
            colorGainPlug.child(2).setValue(color.b);
        }
        if ((hint == ImpSampler::ALBEDO && alphaIsOpacity && firstSampler.m_HasAlpha) ||
            hint == ImpSampler::OPACITY)
        {
            firstTexFn.findPlug("alphaGain").setValue(color.a);
        }

        //-----------------------------------------------------------------------------
        // マルチテクスチャなら layeredTexture ノードを作成します。
        MString srcTexName = firstTexFn.name();
        if (pSrcSamplers.size() >= 2)
        {
            MGlobal::executeCommand("shadingNode -asTexture layeredTexture");
            MObject layeredObj = GetSelectedNodeObject();
            MFnDependencyNode layeredFn(layeredObj);
            srcTexName = layeredFn.name();
            for (size_t iSampler = 0; iSampler < pSrcSamplers.size(); ++iSampler)
            {
                const ImpSampler& sampler = *pSrcSamplers[iSampler];
                MString inputPlugName = srcTexName + ".inputs[" + static_cast<int>(iSampler) + "]";
                MGlobal::executeCommand(MString("connectAttr ") +
                    MFnDependencyNode(sampler.m_TexObj).name() + ".outColor " +
                    inputPlugName + ".color");
                if (sampler.m_HasAlpha)
                {
                    MGlobal::executeCommand(MString("connectAttr ") +
                        MFnDependencyNode(sampler.m_TexObj).name() + ".outAlpha " +
                        inputPlugName + ".alpha");
                }
            }
        }

        //-----------------------------------------------------------------------------
        // マテリアルにテクスチャノードを接続します。
        MString srcPlugName = srcTexName;
        if (hint == ImpSampler::OPACITY)
        {
            srcPlugName += ".outTransparency";
        }
        else if (hint == ImpSampler::NORMAL || hint == ImpSampler::EXTRA)
        {
            srcPlugName += ".outAlpha";
        }
        else
        {
            srcPlugName += ".outColor";
        }
        MGlobal::executeCommand(MString("connectAttr ") + srcPlugName + " " + dstPlugName);

        if (pSrcSamplers.size() == 1 &&
            hint == ImpSampler::ALBEDO &&
            alphaIsOpacity           &&
            firstSampler.m_HasAlpha)
        {
            // ALBEDO テクスチャにアルファ成分があれば transparency にも接続します。
            MGlobal::executeCommand(MString("connectAttr ") + srcTexName + ".outTransparency " +
                    shaderFn.name() + ".transparency");
            //"defaultNavigation -quiet 1 -connectToExisting -source " +
            //  srcTexName + " -destination " + shaderFn.name();
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアルを解析してノードを作成します。
//!
//! @param[in,out] materials マテリアル配列です。
//! @param[in,out] texDatas テクスチャデータ配列です。
//! @param[in] matElem マテリアル要素です。
//! @param[in] orgMatsElem オリジナルマテリアル配列要素です。
//! @param[in] inputFolder 中間ファイルのフォルダのパスです。
//! @param[in] imagesFolder 画像ファイル出力フォルダのパスです。
//! @param[in] texcvtrPath 3D テクスチャーコンバーターのパスです。
//! @param[in] createsTexture テクスチャファイルを作成するなら true です。
//! @param[in] isColorSpaceRawAvailable Raw カラースペースが使用可能なら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseMaterial(
    ImpMaterialArray& materials,
    ImpTexDataArray& texDatas,
    const RXMLElement* matElem,
    const RXMLElement* orgMatsElem,
    const std::string& inputFolder,
    const std::string& imagesFolder,
    const std::string& texcvtrPath,
    const bool createsTexture,
    const bool isColorSpaceRawAvailable
)
{
    MStatus status;

    materials.push_back(ImpMaterial());
    ImpMaterial& material = materials.back();

    //-----------------------------------------------------------------------------
    // マテリアルの属性を取得します。
    const RXMLElement* infoElem = matElem->FindElement("material_info");
    //const RXMLElement* renderStateElem = matElem->FindElement("render_state");

    material.m_Name = matElem->GetAttribute("name");
    material.m_CompressEnable = (infoElem->GetAttribute("compress_enable") == "true");

    //-----------------------------------------------------------------------------
    // オリジナルマテリアルを取得します。
    const RXMLElement* orgMatElem = NULL;
    if (orgMatsElem != NULL)
    {
        for (size_t iOrg = 0; iOrg < orgMatsElem->nodes.size(); ++iOrg)
        {
            orgMatElem = &orgMatsElem->nodes[iOrg];
            if (orgMatElem->GetAttribute("mat_name") == material.m_Name)
            {
                const RXMLElement* orgColorsElem = orgMatElem->FindElement("original_color_array");
                ParseOriginalColor(material.m_Diffuse , orgColorsElem, "diffuse");
                ParseOriginalColor(material.m_Opacity , orgColorsElem, "opacity");
                ParseOriginalColor(material.m_Ambient , orgColorsElem, "ambient");
                ParseOriginalColor(material.m_Emission, orgColorsElem, "emission");
                ParseOriginalColor(material.m_Specular, orgColorsElem, "specular");
                break;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // シェーダノードを作成します。
    const bool isLambert1 = (material.m_Name == "lambert1");
    material.m_HasSpecular = !isLambert1;
    if (!isLambert1)
    {
        const MString shaderType = (material.m_HasSpecular) ? "blinn" : "lambert";
        MGlobal::executeCommand(MString("shadingNode -name \"") + material.m_Name.c_str() + "\" -asShader " + shaderType);
    }
    else
    {
        MGlobal::executeCommand("select lambert1");
    }
    material.m_ShaderObj = GetSelectedNodeObject();
    MFnLambertShader shaderFn(material.m_ShaderObj, &status);
    CheckStatusMsg(status, MString("Material cannot be created: ") + material.m_Name.c_str());

    shaderFn.setColor(material.m_Diffuse);
    shaderFn.setTransparency(MColor(1.0f, 1.0f, 1.0f) - material.m_Opacity);
    shaderFn.setAmbientColor(material.m_Ambient);
    shaderFn.setIncandescence(material.m_Emission);
    if (material.m_HasSpecular)
    {
        MFnReflectShader(material.m_ShaderObj).setSpecularColor(material.m_Specular);
    }

    //-----------------------------------------------------------------------------
    // シェーディンググループを作成します。
    MString sgName;
    if (!isLambert1)
    {
        MGlobal::executeCommand(MString("sets -renderable true -noSurfaceShader true -empty -name \"") + material.m_Name.c_str() + "SG\"", sgName);
    }
    else
    {
        sgName = "initialShadingGroup";
    }
    material.m_SGObj = GetNodeObjectByName(sgName);
    if (material.m_SGObj.isNull())
    {
        DisplayImportError("マテリアルを作成できません: {0}", "Material cannot be created: {0}",
            material.m_Name);
        return MS::kFailure;
    }

    if (!isLambert1)
    {
        MGlobal::executeCommand("connectAttr " + shaderFn.name() + ".outColor " + sgName + ".surfaceShader");
    }

    //-----------------------------------------------------------------------------
    // シェーディンググループのカスタムアトリビュートを設定します。
    if (!material.m_CompressEnable)
    {
        MGlobal::executeCommand("addAttr -ln \"nw4fNoCompressMat\" -at \"bool\" -dv 0 -h 1 " +
            sgName);
        MGlobal::executeCommand("setAttr " + sgName + ".nw4fNoCompressMat " +
            ((material.m_CompressEnable) ? "0" : "1"));
    }

    //-----------------------------------------------------------------------------
    // ユーザーデータを解析してカスタムアトリビュートを設定します。
    const RXMLElement* userDatasElem = matElem->FindElement("user_data_array", false);
    if (userDatasElem != NULL)
    {
        ParseUserDatas(sgName, userDatasElem);
    }

    //-----------------------------------------------------------------------------
    // サンプラを解析します。
    const RXMLElement* samplersElem = matElem->FindElement("sampler_array", false);
    if (samplersElem != NULL)
    {
        //-----------------------------------------------------------------------------
        // サンプラを解析して、ftx ファイルを画像ファイルに変換し、file ノードを作成します。
        const RXMLElement* orgSrtsElem = (orgMatElem != NULL) ?
            orgMatElem->FindElement("original_texsrt_array", false) : NULL;
        for (size_t iSampler = 0; iSampler < samplersElem->nodes.size(); ++iSampler)
        {
            status = ParseSampler(material.m_Samplers, texDatas,
                &samplersElem->nodes[iSampler], orgSrtsElem,
                inputFolder, imagesFolder, texcvtrPath, createsTexture, isColorSpaceRawAvailable);
            CheckStatus(status);
        }

        //-----------------------------------------------------------------------------
        // サンプラ配列をヒント情報でソートします。
        ImpSamplerPtrArray pMatSamplers;
        for (size_t iSampler = 0; iSampler < material.m_Samplers.size(); ++iSampler)
        {
            pMatSamplers.push_back(&material.m_Samplers[iSampler]);
        }
        std::sort(pMatSamplers.begin(), pMatSamplers.end(), RSamplerPtrLess);
        //for (size_t iSampler = 0; iSampler < pMatSamplers.size(); ++iSampler)
        //{
        //  const ImpSampler& sampler = *pMatSamplers[iSampler];
        //  cerr <<"sampler: " << sampler.m_Hint << ", " << sampler.m_HintIndex << endl;
        //}

        //-----------------------------------------------------------------------------
        // テクスチャをマテリアルに接続します。
        if (!pMatSamplers.empty())
        {
            MColor diffuseA = material.m_Diffuse;
            diffuseA.a = material.m_Opacity.r;
            const MColor whiteCol(1.0f, 1.0f, 1.0f, 1.0f);

            const bool alphaIsOpacity = (FindSamplerByHint(pMatSamplers, ImpSampler::OPACITY) == -1);
            ConnectTexture(shaderFn, pMatSamplers, ImpSampler::ALBEDO  , diffuseA           , alphaIsOpacity);
            ConnectTexture(shaderFn, pMatSamplers, ImpSampler::OPACITY , diffuseA           , false);
            ConnectTexture(shaderFn, pMatSamplers, ImpSampler::EMISSION, material.m_Emission, false);
            ConnectTexture(shaderFn, pMatSamplers, ImpSampler::NORMAL  , whiteCol           , false);
            ConnectTexture(shaderFn, pMatSamplers, ImpSampler::EXTRA   , whiteCol           , false);
            if (material.m_HasSpecular)
            {
                ConnectTexture(shaderFn, pMatSamplers, ImpSampler::SPECULAR  , material.m_Specular, false);
                ConnectTexture(shaderFn, pMatSamplers, ImpSampler::REFLECTION, whiteCol           , false);
            }
        }
    }

    return status;
} // NOLINT(impl/function_size)

//=============================================================================
//! @brief インポート用トランスフォームノードのクラスです。
//=============================================================================
class ImpTransformNode
{
public:
    std::string m_Name; //!< 名前です。

    MVector m_Scale; //!< スケールです。
    MVector m_Rotate; //!< 回転です。
    MVector m_Translate; //!< 移動です。

    MDagPath m_XformPath; //!< transform ノードの DAG パスです。
    int m_ParentIdx; //!< 親ノードの配列内インデックスです。
    bool m_IsJoint; //!< joint ノードとしてインポートするなら true、transform ノードなら false です。
    bool m_IsSkipped; //!< インポートをスキップしたなら true です。

public:
    //! コンストラクタです。
    ImpTransformNode()
    : m_Scale(MVector::one),
      m_Rotate(MVector::zero),
      m_Translate(MVector::zero),
      m_ParentIdx(-1),
      m_IsJoint(false),
      m_IsSkipped(false)
    {
    }
};

//=============================================================================
//! @brief インポート用ボーンのクラスです。
//=============================================================================
class ImpBone : public ImpTransformNode
{
public:
    //! @brief ビルボードモードを表す列挙型です。
    enum Billboard
    {
        BILLBOARD_NONE,     //!< ビルボードなしです。
        WORLD_VIEWVECTOR,   //!< Z 軸がカメラの視線と並行になります。
        WORLD_VIEWPOINT,    //!< Z 軸がカメラの方向を向きます。
        SCREEN_VIEWVECTOR,  //!< Y 軸がカメラの上方向を向き、Z 軸がカメラの視線と並行になります。
        SCREEN_VIEWPOINT,   //!< Y 軸がカメラの上方向を向き、Z 軸がカメラの方向を向きます。
        YAXIS_VIEWVECTOR,   //!< Y 軸のみ回転する状態で、Z 軸がカメラの視線と並行になります。
        YAXIS_VIEWPOINT,    //!< Y 軸のみ回転する状態で、Z 軸がカメラの方向を向きます。
        BILLBOARD_COUNT     //!< ビルボードモードの総数です。
    };

    // attr
    bool m_RigidBody; //!< 行列がリジッドボディの描画に使用されるなら true です。
    int m_SmoothMtxIdx; //!< スムーススキンの描画に使用する行列の行列パレットインデックスです。
    int m_RigidMtxIdx; //!< リジッドスキンの描画に使用する行列の行列パレットインデックスです。
    bool m_ScaleCompensate; //!< セグメントスケール補正が有効なら true です。
    Billboard m_Billboard; //!< ビルボード設定です。
    bool m_Visibility; //!< 可視性です。
    bool m_CompressEnable; //!< 最適化で圧縮可能なら true です。
    std::string m_Side; //!< ラベルのサイドです。
    std::string m_Type; //!< ラベルのタイプです。

    MMatrix m_BindGlobalMtx; //!< ワールドバインド行列です。
    MMatrix m_BindGlobalInvMtx; //!< ワールドバインド行列の逆行列です。

    bool m_HasShape; //!< シェイプを持つなら true です。
    bool m_HasSkinedShape; //!< スキンシェイプを持つなら true です。
    bool m_SeparatesSkinObject; //!< スキンシェイプをワールド下の別オブジェクトとして作成するなら true です。
    RIntArray m_ShapeElmIdxs; //!< シェイプ要素のインデックス配列です。
    RStringArray m_ShapeNames; //!< シェイプ名の配列です。
    MDagPath m_ShapeXformPath; //!< シェイプ用 transform ノードの DAG パスです。
    MDagPath m_ShapePath; //!< シェイプノードの DAG パスです。
    MDagPathArray m_LodXformPaths; //!< 各 LOD レベルの transform ノードの DAG パス配列です。
    MDagPathArray m_LodShapeXformPaths; //!< 各 LOD レベルのシェイプ用 transform ノードの DAG パス配列です。
    MDagPathArray m_LodShapePaths; //!< 各 LOD レベルのシェイプノードの DAG パス配列です。

public:
    //! コンストラクタです。
    ImpBone()
    :
      // attr
      m_RigidBody(false),
      m_SmoothMtxIdx(-1),
      m_RigidMtxIdx(-1),
      m_ScaleCompensate(false),
      m_Billboard(BILLBOARD_NONE),
      m_Visibility(true),
      m_CompressEnable(true),
      m_HasShape(false),
      m_HasSkinedShape(false),
      m_SeparatesSkinObject(false)
    {
    }
};

//! @brief インポート用ボーン配列の定義です。
typedef std::vector<ImpBone> ImpBoneArray;

//-----------------------------------------------------------------------------
//! @brief 名前でボーンを検索します。
//!
//! @param[in] bones ボーン配列です。
//! @param[in] name ボーン名です。
//!
//! @return インデックスを返します。見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindBoneByName(const ImpBoneArray& bones, const std::string& name)
{
    for (int iBone = 0; iBone < static_cast<int>(bones.size()); ++iBone)
    {
        if (bones[iBone].m_Name == name)
        {
            return iBone;
        }
    }
    return -1;
}

//=============================================================================
//! @brief インポート用 UV セットのクラスです。
//=============================================================================
class ImpUvSet
{
public:
    MString m_Name; //!< UV セット名です。
    MFloatArray m_Us; //!< mesh ノード中の U 配列です。
    MFloatArray m_Vs; //!< mesh ノード中の V 配列です。
    MIntArray m_Ids; //!< フェース頂点ごとの UV ID です。
};

//=============================================================================
//! @brief インポート用頂点属性のクラスです。
//=============================================================================
class ImpVtxAttrib
{
public:
    // attr
    int m_Count; //!< 頂点数です。
    bool m_IsFloat; //!< float 型なら true、int 型なら false です。
    int m_CompCount; //!< 成分数です。
    RFloatArray m_FloatValues; //!< float 型の値の配列です。
    RIntArray m_IntValues; //!< int 型の値の配列です。

public:
    //! コンストラクタです。
    ImpVtxAttrib()
    : m_Count(0),
      m_IsFloat(true),
      m_CompCount(0)
    {
    }

    //! float ベクトルを返します。
    MFloatVector GetFloatVector(const int iVtx) const
    {
        return MFloatVector(&m_FloatValues[iVtx * m_CompCount]);
    }

    //! ベクトルを返します。
    MVector GetVector(const int iVtx) const
    {
        return MVector(GetFloatVector(iVtx));
    }
};

//! @brief インポート用頂点属性配列の定義です。
typedef std::vector<ImpVtxAttrib> ImpVtxAttribArray;

//=============================================================================
//! @brief インポート用キーシェイプのクラスです。
//=============================================================================
class ImpKeyShape
{
public:
    static const int COL_MAX = 8; //!< 1 頂点あたりのカラーセットの最大数です。
    static const int UV_MAX = 8; //!< 1 頂点あたりの UV セットの最大数です。
    static const int WGT_MAX = 2; //!< 1 頂点あたりのブレンドウェイト属性の最大数です。

    std::string m_Name; //!< キーシェイプ名です。
    RStringArray m_AttribNames; //!< キーシェイプが使用する頂点属性名の配列です。

    ImpVtxAttrib m_AttribPos; //!< 頂点座標の頂点属性です。
    ImpVtxAttrib m_AttribNrm; //!< 法線の頂点属性です。
    ImpVtxAttrib m_AttribCols[COL_MAX]; //!< カラーの頂点属性配列です。
    ImpVtxAttrib m_AttribUvs[UV_MAX]; //!< UV の頂点属性配列です。
    ImpVtxAttrib m_AttribIdxs[WGT_MAX]; //!< ブレンドインデックスの頂点属性配列です。
    ImpVtxAttrib m_AttribWgts[WGT_MAX]; //!< ブレンドウェイトの頂点属性配列です。
};

//! @brief インポート用キーシェイプ配列の定義です。
typedef std::vector<ImpKeyShape> ImpKeyShapeArray;

//=============================================================================
//! @brief インポート用シェイプデータのクラスです。
//=============================================================================
class ImpShapeData
{
public:
    std::string m_Name; //!< シェイプ名です。
    int m_MatIdx; //!< マテリアルのインデックスです。
    std::string m_OrgBoneName; //!< 最適化前の描画に使用するボーン名です。
    int m_SkinCompCount; //!< スキニング用の頂点属性の成分数です。
    int m_FaceCount; //!< フェース数です。

    ImpKeyShapeArray m_KeyShapes; //!< キーシェイプ配列です。

public:
    //! コンストラクタです。
    ImpShapeData()
    : m_MatIdx(0),
      m_SkinCompCount(0),
      m_FaceCount(0)
    {
    }
};

//! @brief インポート用シェイプデータ配列の定義です。
typedef std::vector<ImpShapeData> ImpShapeDataArray;

//=============================================================================
//! @brief インポート用頂点が参照するボーンインデックスとウェイト値のクラスです。
//=============================================================================
class ImpVtxWeight
{
public:
    int m_BoneIdx; //!< ボーンインデックスです。
    float m_Weight; //!< ウェイト値です。

public:
    //! コンストラクタです。
    ImpVtxWeight(const int boneIdx = 0, const float weight = 1.0f)
    : m_BoneIdx(boneIdx),
      m_Weight(weight)
    {
    }

    //! 同値であれば true を返します。
    bool operator==(const ImpVtxWeight& rhs) const
    {
        return (m_BoneIdx == rhs.m_BoneIdx &&
                m_Weight  == rhs.m_Weight);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const ImpVtxWeight& rhs) const
    {
        return (m_BoneIdx != rhs.m_BoneIdx ||
                m_Weight  != rhs.m_Weight);
    }

    //! @brief 出力ストリームにボーンウェイト値を出力します（デバッグ用）。
    friend std::ostream& operator<<(std::ostream& os, const ImpVtxWeight& vw)
    {
        os << "[" << vw.m_BoneIdx << ", " << vw.m_Weight << "]";
        return os;
    }
};

//! @brief インポート用頂点が参照するボーンインデックスとウェイト値配列の定義です。
typedef std::vector<ImpVtxWeight> ImpVtxWeightArray;

//=============================================================================
//! @brief インポート用頂点行列のクラスです。
//=============================================================================
class ImpVtxMtx
{
public:
    ImpVtxWeightArray m_VtxWeights; //!< ボーンウェイト値配列です。

public:
    //! コンストラクタです（引数なし）。
    ImpVtxMtx()
    {
    }

    //! コンストラクタです（ボーンインデックスとウェイトを指定）。
    ImpVtxMtx(const int boneIdx, const float weight = 1.0f)
    {
        m_VtxWeights.push_back(ImpVtxWeight(boneIdx, weight));
    }

    int GetBoneCount() const { return static_cast<int>(m_VtxWeights.size()); }

    bool IsFull() const { return (GetBoneCount() == 1); }

    int GetBoneIndex(const int localBoneIdx) const
    {
        return m_VtxWeights[localBoneIdx].m_BoneIdx;
    }

    float GetWeight(const int localBoneIdx) const
    {
        return m_VtxWeights[localBoneIdx].m_Weight;
    }

    void Append(const int boneIdx, const float weight)
    {
        m_VtxWeights.push_back(ImpVtxWeight(boneIdx, weight));
    }

    //! 同値であれば true を返します。
    bool operator==(const ImpVtxMtx& rhs) const
    {
        return (RIsSameArray(m_VtxWeights, rhs.m_VtxWeights));
    }

    //! 同値でなければ true を返します。
    bool operator!=(const ImpVtxMtx& rhs) const
    {
        return (!RIsSameArray(m_VtxWeights, rhs.m_VtxWeights));
    }

    //! @brief 出力ストリームに頂点行列を出力します（デバッグ用）。
    friend std::ostream& operator<<(std::ostream& os, const ImpVtxMtx& vmt)
    {
        os << "[";
        for (size_t iLb = 0; iLb < vmt.m_VtxWeights.size(); ++iLb)
        {
            if (iLb != 0) os << ", ";
            os << vmt.m_VtxWeights[iLb];
        }
        os << "]";
        return os;
    }
};

//! @brief インポート用頂点行列配列の定義です。
typedef std::vector<ImpVtxMtx> ImpVtxMtxArray;

//=============================================================================
//! @brief インポート用 mesh ノードのブレンドシェイプターゲットのクラスです。
//=============================================================================
class ImpMeshTarget
{
public:
    std::string m_Name; //!< キーシェイプ名です。
    MPointArray m_VertexArray; //!< 頂点座標配列です。
    MVectorArray m_NrmArray; //!< 法線配列です。
    MColorArray m_ColArrays[ImpKeyShape::COL_MAX]; //!< カラー配列の配列です。
};

//! @brief インポート用 mesh ノードのブレンドシェイプターゲット配列の定義です。
typedef std::vector<ImpMeshTarget> ImpMeshTargetArray;

//=============================================================================
//! @brief インポート用 mesh ノードのパラメータのクラスです。
//=============================================================================
class ImpMeshParam
{
public:
    MPointArray m_VertexArray; //!< 頂点座標配列です。
    MIntArray m_VertexCounts; //!< 各フェースの頂点数配列です。
    MIntArray m_FaceConnects; //!< 各フェースを構成する頂点インデックス配列です。m_VertexCounts の値の合計と同じ長さです。

    MIntArray m_AllFaceList; //!< 全頂点フェースのフェースインデックス配列です。
    MIntArray m_AllVtxList; //!< 全頂点フェースの頂点インデックス配列です。m_AllFaceList と同じ長さです。

    MVectorArray m_NrmArray; //!< 法線配列です。
    MIntArray m_NrmFaceList; //!< 法線を設定する頂点フェースのフェースインデックス配列です。
    MIntArray m_NrmVtxList; //!< 法線を設定する頂点フェースの頂点インデックス配列です。m_NrmFaceList と同じ長さです。

    MColorArray m_ColArrays[ImpKeyShape::COL_MAX]; //!< カラー配列の配列です。
    ImpUvSet m_UvSets[ImpKeyShape::UV_MAX]; //!< UV セットの配列です。

    ImpVtxMtxArray m_VtxMtxs; //!< 頂点行列配列です（重複なし）。
    RIntArray m_VtxMtxIdxs; //!< 頂点行列のインデックス配列です。m_VertexArray と同じ長さです。
    RIntArray m_VertexShapeIdxs; //!< シェイプデータインデックス配列です。m_VertexArray と同じ長さです。
    RIntArray m_VertexSrcVtxIdxs; //!< シェイプデータ中の頂点インデックス配列です。m_VertexArray と同じ長さです。

    ImpMeshTargetArray m_MeshTargets; //!< ブレンドシェイプターゲット配列です。
    bool m_UpdatesPos; //!< ブレンドシェイプで頂点座標を更新するなら true です。

public:
    //! コンストラクタです。
    explicit ImpMeshParam(const int targetCount)
    : m_UpdatesPos(false)
    {
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            m_MeshTargets.push_back(ImpMeshTarget());
        }
    }
};

//-----------------------------------------------------------------------------
//! @brief ノード圧縮不可フラグのアトリビュートを追加・設定します。
//!
//! @param[in] nodeName transform ノード名です。
//! @param[in] compressEnable 圧縮可能なら true、圧縮不可なら false を設定します。
//-----------------------------------------------------------------------------
static void AddNoCompressNodeAttr(const MString& nodeName, const bool compressEnable)
{
    MGlobal::executeCommand("addAttr -ln \"nw4fNoCompressNode\" -at \"bool\" -dv 0 -h 1 " +
        nodeName);
    MGlobal::executeCommand("setAttr " + nodeName + ".nw4fNoCompressNode " +
        ((compressEnable) ? "0" : "1"));
}

//-----------------------------------------------------------------------------
//! @brief transform ノード名を加工してシェイプノード名を作成します。
//!
//! @param[in] xformName transform ノード名です。
//!
//! @return シェイプノード名を返します。
//-----------------------------------------------------------------------------
static MString CreateShapeNodeName(const MString& xformName)
{
    std::string name(xformName.asChar());
    int nc = 0;
    for (int ic = static_cast<int>(name.length()) - 1; ic >= 1; --ic)
    {
        if ('0' <= name[ic] && name[ic] <= '9')
        {
            ++nc;
        }
        else
        {
            break;
        }
    }
    if (nc != 0)
    {
        const int iNum = static_cast<int>(name.length()) - nc;
        name = name.substr(0, iNum) + "Shape" + name.substr(iNum);
    }
    else
    {
        name += "Shape";
    }
    return name.c_str();
}

//-----------------------------------------------------------------------------
//! @brief キーシェイプを解析します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in] keyShapeElem キーシェイプ要素です。
//-----------------------------------------------------------------------------
static void ParseKeyShape(ImpShapeData& shapeData, const RXMLElement* keyShapeElem)
{
    shapeData.m_KeyShapes.push_back(ImpKeyShape());
    ImpKeyShape& keyShape = shapeData.m_KeyShapes.back();

    keyShape.m_Name = keyShapeElem->GetAttribute("name");
    const RXMLElement* attribsElem = keyShapeElem->FindElement("target_attrib_array");
    for (size_t iAttrib = 0; iAttrib < attribsElem->nodes.size(); ++iAttrib)
    {
        const RXMLElement* attribElem = &attribsElem->nodes[iAttrib];
        keyShape.m_AttribNames.push_back(attribElem->GetAttribute("attrib_name"));
    }
    //cerr << "key shape: " << keyShape.m_Name << ": " << keyShape.m_AttribNames << endl;
}

//-----------------------------------------------------------------------------
//! @brief 行列インデックスからボーンインデックスに変換します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in] mtxIdxToBoneIdxs 行列インデックスからボーンインデックスへの変換テーブルです。
//-----------------------------------------------------------------------------
static void ConvertMtxIdxToBoneIdx(ImpShapeData& shapeData, const RIntArray& mtxIdxToBoneIdxs)
{
    const int tableCount = static_cast<int>(mtxIdxToBoneIdxs.size());
    for (int iHint = 0; iHint < ImpKeyShape::WGT_MAX; ++iHint)
    {
        ImpVtxAttrib& attrib = shapeData.m_KeyShapes[0].m_AttribIdxs[iHint];
        if (attrib.m_Count != 0)
        {
            for (size_t iVal = 0; iVal < attrib.m_IntValues.size(); ++iVal)
            {
                const int idx = attrib.m_IntValues[iVal];
                if (0 <= idx && idx < tableCount)
                {
                    attrib.m_IntValues[iVal] = mtxIdxToBoneIdxs[idx];
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief std::vector 配列の先頭をシフトします。
//!
//! @param[in,out] array std::vector 配列です。
//! @param[in] startIdx 新しい先頭のインデックスです。
//-----------------------------------------------------------------------------
template <typename T>
static void ShiftArrayStart(std::vector<T>& array, const int startIdx)
{
    if (0 < startIdx && startIdx < array.size())
    {
        const size_t dstCount = array.size() - startIdx;
        std::vector<T> dstArray(dstCount);
        for (size_t iVal = 0; iVal < dstCount; ++iVal)
        {
            dstArray[iVal] = array[startIdx + iVal];
        }
        array = dstArray;
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点バッファ配列要素から頂点属性インデックス配列を取得します。
//!
//! @param[out] vtxAttribIdxs 頂点属性インデックス配列です。
//! @param[in] buffersElem 頂点バッファ配列要素です。
//-----------------------------------------------------------------------------
static void GetVtxAttribIdxsFromVtxBuffers(
    RIntArray& vtxAttribIdxs,
    const RXMLElement* buffersElem
)
{
    for (size_t iBuf = 0; iBuf < buffersElem->nodes.size(); ++iBuf)
    {
        const RXMLElement* bufferElem = &buffersElem->nodes[iBuf];
        const RXMLElement* inputsElem = bufferElem->FindElement("input_array");
        for (size_t iInput = 0; iInput < inputsElem->nodes.size(); ++iInput)
        {
            const RXMLElement* inputElem = &inputsElem->nodes[iInput];
            const int iAttrib = atoi(inputElem->GetAttribute("attrib_index").c_str());
            RFindAppendValue(vtxAttribIdxs, iAttrib);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 最適化前のシェイプが描画に使用していたボーン名を取得します。
//!
//! @param[in] infoElem シェイプ情報要素です。
//! @param[in] shapeName シェイプ名です。
//!
//! @return 最適化前のシェイプが描画に使用していたボーン名を返します。
//-----------------------------------------------------------------------------
std::string GetShapeOriginalBoneName(
    const RXMLElement* infoElem,
    const std::string& shapeName
)
{
    std::string orgBoneName = infoElem->GetAttribute("original_bone_name", "", false); // 4.0.0 で追加
    const size_t commaIdx = orgBoneName.find(",");
    if (commaIdx != std::string::npos)
    {
        // original_bone_name 属性に複数のボーン名が格納されている場合は
        // 先頭のものを取得します。
        orgBoneName = RTrimString(orgBoneName.substr(0, commaIdx));
    }
    if (orgBoneName.empty())
    {
        const size_t sepIdx = shapeName.find("__");
        if (sepIdx != std::string::npos && sepIdx != 0)
        {
            // original_bone_name 属性が存在しないか空文字の場合、
            // シェイプ名が "BoneName__MaterialName" 形式なら
            // "__" より前の部分を original_bone_name として取得します。
            orgBoneName = shapeName.substr(0, sepIdx);
        }
    }
    return orgBoneName;
}

//-----------------------------------------------------------------------------
//! @brief シェイプデータを取得します。
//!
//! @param[in,out] shapeDatas シェイプデータ配列です。
//! @param[in,out] uvAttribNames 存在する UV セットの頂点属性名配列です。
//! @param[in,out] uvHintIdxs 存在する UV セットのヒント情報インデックス配列です。
//! @param[in] verticesElem 頂点配列要素です。
//! @param[in] srcLodIdx インポートするサブメッシュの LOD インデックスです。
//! @param[in] shapeElem シェイプ要素です。
//! @param[in] mtxIdxToBoneIdxs 行列インデックスからボーンインデックスへの変換テーブルです。
//! @param[in] materials マテリアル配列です。
//! @param[in] fmdFile fmd ファイルです。
//-----------------------------------------------------------------------------
static void GetShapeData(
    ImpShapeDataArray& shapeDatas,
    RStringArray& uvAttribNames,
    RIntArray& uvHintIdxs,
    const RXMLElement* verticesElem,
    const int srcLodIdx,
    const RXMLElement* shapeElem,
    const RIntArray& mtxIdxToBoneIdxs,
    const ImpMaterialArray& materials,
    const ImpIntermediateFile& fmdFile
)
{
    shapeDatas.push_back(ImpShapeData());
    ImpShapeData& shapeData = shapeDatas.back();

    //-----------------------------------------------------------------------------
    // シェイプの情報を取得します。
    shapeData.m_Name = shapeElem->GetAttribute("name");
    const RXMLElement* infoElem = shapeElem->FindElement("shape_info");
    shapeData.m_MatIdx = FindMaterialByName(materials, infoElem->GetAttribute("mat_name"));
    shapeData.m_OrgBoneName = GetShapeOriginalBoneName(infoElem, shapeData.m_Name);
    shapeData.m_SkinCompCount = atoi(infoElem->GetAttribute("vertex_skinning_count").c_str());

    //-----------------------------------------------------------------------------
    // キーシェイプを取得します。
    const RXMLElement* keyShapesElem = shapeElem->FindElement("key_shape_array", false);
    if (keyShapesElem != NULL)
    {
        for (size_t iKey = 0; iKey < keyShapesElem->nodes.size(); ++iKey)
        {
            ParseKeyShape(shapeData, &keyShapesElem->nodes[iKey]);
        }
    }
    else
    {
        shapeData.m_KeyShapes.push_back(ImpKeyShape());
    }

    //-----------------------------------------------------------------------------
    // LOD オフセットを取得します。
    const int iVertex = atoi(infoElem->GetAttribute("vertex_index").c_str());
    const RXMLElement* vertexElem = &verticesElem->nodes[iVertex];

    RIntArray lodOffsets;
    lodOffsets.push_back(0);
    const RXMLElement* lodOffsetElem = vertexElem->FindElement("lod_offset", false);
    if (lodOffsetElem != NULL)
    {
        std::istringstream ofsIss(lodOffsetElem->text);
        const int ofsCount = atoi(lodOffsetElem->GetAttribute("count").c_str());
        for (int iOfs = 0; iOfs < ofsCount; ++iOfs)
        {
            int ofs = 0;
            ofsIss >> ofs;
            lodOffsets.push_back(ofs);
        }
    }
    const int iLod = (srcLodIdx < lodOffsets.size()) ?
        srcLodIdx : static_cast<int>(lodOffsets.size()) - 1;
    const int lodOffset = lodOffsets[iLod];

    //-----------------------------------------------------------------------------
    // 頂点属性を取得します。
    const RXMLElement* attribsElem = vertexElem->FindElement("vtx_attrib_array");
    const RXMLElement* buffersElem = vertexElem->FindElement("vtx_buffer_array");
    RIntArray vtxAttribIdxs;
    GetVtxAttribIdxsFromVtxBuffers(vtxAttribIdxs, buffersElem);

    RBoolArray updatesColSets(ImpKeyShape::COL_MAX, false);
    RIntArray keyColSetsCounts(shapeData.m_KeyShapes.size(), 0); // キーシェイプごとのカラーセット数です。
    for (size_t iInput = 0; iInput < vtxAttribIdxs.size(); ++iInput)
    {
        const RXMLElement* attribElem = &attribsElem->nodes[vtxAttribIdxs[iInput]];
        const std::string name = attribElem->GetAttribute("name");
        const std::string hint = attribElem->GetAttribute("hint");
        const int hintIndex = GetStringSuffixNumber(hint);

        //-----------------------------------------------------------------------------
        // キーシェイプのインデックスを取得します。
        bool isKeyFound = false;
        size_t iKeyShape = 0;
        for (size_t iKey = 0; iKey < shapeData.m_KeyShapes.size(); ++iKey)
        {
            if (RFindValueInArray(shapeData.m_KeyShapes[iKey].m_AttribNames, name) != -1)
            {
                isKeyFound = true;
                iKeyShape = iKey;
                break;
            }
        }
        ImpKeyShape& keyShape = shapeData.m_KeyShapes[iKeyShape];

        //-----------------------------------------------------------------------------
        ImpVtxAttrib* pAttrib = NULL;
        if (hint.find("position") == 0)
        {
            pAttrib = &keyShape.m_AttribPos;
        }
        else if (hint.find("normal") == 0)
        {
            pAttrib = &keyShape.m_AttribNrm;
        }
        else if (hint.find("color") == 0)
        {
            int iColSet = hintIndex;
            if (iKeyShape >= 1)
            {
                int updateCount = 0;
                for (int iCS = 0; iCS < ImpKeyShape::COL_MAX; ++iCS)
                {
                    if (updatesColSets[iCS])
                    {
                        if (keyColSetsCounts[iKeyShape] == updateCount)
                        {
                            iColSet = iCS;
                            break;
                        }
                        ++updateCount;
                    }
                }
            }
            if (0 <= iColSet && iColSet < ImpKeyShape::COL_MAX)
            {
                pAttrib = &keyShape.m_AttribCols[iColSet];
                if (iKeyShape == 0 && isKeyFound)
                {
                    updatesColSets[iColSet] = true;
                }
                keyColSetsCounts[iKeyShape]++;
            }
        }
        else if (hint.find("uv") == 0)
        {
            if (uvHintIdxs.size() < ImpKeyShape::UV_MAX)
            {
                const int iUvSet = RFindAppendValue(uvHintIdxs, hintIndex);
                pAttrib = &keyShape.m_AttribUvs[iUvSet];
                if (iUvSet >= uvAttribNames.size())
                {
                    uvAttribNames.push_back(name);
                }
            }
        }
        else if (hint.find("blendindex") == 0)
        {
            if (0 <= hintIndex && hintIndex < ImpKeyShape::WGT_MAX)
            {
                pAttrib = &keyShape.m_AttribIdxs[hintIndex];
            }
        }
        else if (hint.find("blendweight") == 0)
        {
            if (0 <= hintIndex && hintIndex < ImpKeyShape::WGT_MAX)
            {
                pAttrib = &keyShape.m_AttribWgts[hintIndex];
            }
        }

        if (pAttrib != NULL)
        {
            pAttrib->m_Count = atoi(attribElem->GetAttribute("count").c_str());
            const std::string type = attribElem->GetAttribute("type");
            const int streamIdx = atoi(attribElem->GetAttribute("stream_index").c_str());
            pAttrib->m_IsFloat = (type.find("float") == 0);
            const char typeN = type[type.size() - 1];
            pAttrib->m_CompCount = ('1' <= typeN && typeN <= '9') ? typeN - '0' : 1;
            if (pAttrib->m_IsFloat)
            {
                fmdFile.GetFloatStream(pAttrib->m_FloatValues, streamIdx);
                ShiftArrayStart(pAttrib->m_FloatValues, lodOffset * pAttrib->m_CompCount);
            }
            else
            {
                fmdFile.GetIntStream(pAttrib->m_IntValues, streamIdx);
                ShiftArrayStart(pAttrib->m_IntValues, lodOffset * pAttrib->m_CompCount);
            }
        }
    }

    ConvertMtxIdxToBoneIdx(shapeData, mtxIdxToBoneIdxs);
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 頂点フェースリストに値が存在しなければ追加します。
//!
//! @param[in,out] faceList 頂点フェースのフェースインデックス配列です。
//! @param[in,out] vtxList 頂点フェースの頂点インデックス配列です。
//! @param[in] iFace フェースインデックスです。
//! @param[in] iVtx 頂点インデックスです。
//-----------------------------------------------------------------------------
//static int AppendFaceVtxList(
//  MIntArray& faceList,
//  MIntArray& vtxList,
//  const int iFace,
//  const int iVtx
//)
//{
//  const int fvCount = faceList.length();
//  for (int iFv = 0; iFv < fvCount; ++iFv)
//  {
//      if (faceList[iFv] == iFace && vtxList[iFv] == iVtx)
//      {
//          return;
//      }
//  }
//  faceList.append(iFace);
//  vtxList.append(iVtx);
//}

//-----------------------------------------------------------------------------
//! @brief mesh ノードの法線を調整します。
//!
//! @param[in] meshFn mesh ノードのファンクションセットです。
//! @param[in] meshPath mesh ノードの DAG パスです。
//-----------------------------------------------------------------------------
static void AdjustMeshNormal(MFnMesh& meshFn, const MDagPath& meshPath)
{
    //RTimeMeasure tm1;

    //-----------------------------------------------------------------------------
    // 不要な法線のロックを解除します。
#if 0
    MIntArray unlockFaceList;
    MIntArray unlockVtxList;

    // フェース頂点法線がフェース法線と同じならロック解除します。
    MItMeshVertex vIter(meshPath);
    for ( ; !vIter.isDone(); vIter.next())
    {
        //bool unlocked = false;
        //MVector averageNrm(MVector::zero);
        //MVectorArray fvNrms;
        MIntArray faceList;
        vIter.getConnectedFaces(faceList);
        const int fcCount = faceList.length();
        for (int iFc = 0; iFc < fcCount; ++iFc)
        {
            const int iFace = faceList[iFc];
            MVector faceNrm;
            meshFn.getPolygonNormal(iFace, faceNrm);
            //averageNrm += faceNrm;
            MVector fvNrm;
            meshFn.getFaceVertexNormal(iFace, vIter.index(), fvNrm);
            //fvNrms.append(fvNrm);
            if (fvNrm.isEquivalent(faceNrm, TOLERANCE_NRM))
            {
                unlockFaceList.append(iFace);
                unlockVtxList.append(vIter.index());
                //unlocked = true;
            }
        }

        // フェース頂点法線がフェース法線の平均と同じならロック解除します。
        // ソフトエッジの場合の Maya の法線計算は単純な平均でない？
        //averageNrm.normalize();
        //if (!unlocked)
        //{
        //  //cerr << "averageNrm: " << vIter.index() << ": " << averageNrm<<endl;
        //  int averageCount = 0;
        //  for (int iFc = 0; iFc < fcCount; ++iFc)
        //  {
        //      if (fvNrms[iFc].isEquivalent(averageNrm, TOLERANCE_NRM))
        //      {
        //          ++averageCount;
        //      }
        //  }
        //  if (averageCount == fcCount)
        //  {
        //      for (int iFc = 0; iFc < fcCount; ++iFc)
        //      {
        //          unlockFaceList.append(faceList[iFc]);
        //          unlockVtxList.append(vIter.index());
        //      }
        //  }
        //}
    }

    if (unlockFaceList.length() != 0)
    {
        meshFn.unlockFaceVertexNormals(unlockFaceList, unlockVtxList);
    }
#endif

    //-----------------------------------------------------------------------------
    // エッジの両側の頂点フェースの法線がそれぞれ同じならソフトエッジにします。
    MIntArray iSoftEdges;
    MItMeshEdge eIter(meshPath);
    for ( ; !eIter.isDone(); eIter.next())
    {
        MIntArray faceList;
        eIter.getConnectedFaces(faceList);
        if (faceList.length() == 2)
        {
            bool isSoft = true;
            for (int iV = 0; iV < 2; ++iV)
            {
                const int iVtx = eIter.index(iV);
                MVector nrm0;
                meshFn.getFaceVertexNormal(faceList[0], iVtx, nrm0);
                MVector nrm1;
                meshFn.getFaceVertexNormal(faceList[1], iVtx, nrm1);
                if (!nrm0.isEquivalent(nrm1, TOLERANCE_NRM))
                {
                    isSoft = false;
                    break;
                }
            }
            if (isSoft)
            {
                iSoftEdges.append(eIter.index());
            }
        }
    }

    if (iSoftEdges.length() != 0)
    {
        YSetEdgesSoftHard(meshPath, iSoftEdges, true);
        meshFn.syncObject();
    }

    //cerr << "adjust normal: " << meshPath.partialPathName() << ": " << tm1.GetMilliSec() << endl;
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードのフェース法線からハードエッジを設定します。
//!
//! @param[in] meshFn mesh ノードのファンクションセットです。
//! @param[in] meshPath mesh ノードの DAG パスです。
//-----------------------------------------------------------------------------
static void SetMeshHardEdge(MFnMesh& meshFn, const MDagPath& meshPath)
{
    const double COS_EP = 1.0e-5;
    MIntArray iHardEdges;
    MItMeshEdge eIter(meshPath);
    for ( ; !eIter.isDone(); eIter.next())
    {
        MIntArray faceList;
        eIter.getConnectedFaces(faceList);
        if (faceList.length() == 2)
        {
            // エッジに隣接するフェースの法線の角度差が一定以上ならハードエッジにします。
            MVector nrm0;
            meshFn.getPolygonNormal(faceList[0], nrm0);
            MVector nrm1;
            meshFn.getPolygonNormal(faceList[1], nrm1);
            nrm0.normalize();
            nrm1.normalize();
            const double dp = nrm0 * nrm1;
            if (dp < 0.0 + COS_EP) // 90 度以上
            {
                iHardEdges.append(eIter.index());
            }
        }
    }

    if (iHardEdges.length() != 0)
    {
        YSetEdgesSoftHard(meshPath, iHardEdges, false);
        meshFn.syncObject();
    }
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードの頂点座標を配列から許容値ありで検索します。
//!
//! @param[in] meshParam mesh ノードのパラメータです。
//! @param[in] shapeDatas シェイプデータ配列です。
//! @param[in] iShape 対象シェイプデータのインデックスです。
//! @param[in] iSrcVtx 中間ファイルの頂点インデックスです。
//! @param[in] pos オブジェクト空間に変換したベースシェイプの頂点座標です。
//! @param[in] iVmt 頂点行列インデックスです。
//!
//! @return インデックスを返します。見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
inline int FindMeshPos(
    const ImpMeshParam& meshParam,
    const ImpShapeDataArray& shapeDatas,
    const size_t iShape,
    const int iSrcVtx,
    const MPoint& pos,
    const int iVmt
)
{
    const ImpShapeData& shapeData = shapeDatas[iShape];
    const int targetCount = static_cast<int>(shapeData.m_KeyShapes.size()) - 1;

    const MPointArray& poss = meshParam.m_VertexArray;
    for (int iPos = 0; iPos < static_cast<int>(poss.length()); ++iPos)
    {
        if (poss[iPos].isEquivalent(pos) &&
            meshParam.m_VtxMtxIdxs[iPos] == iVmt)
        {
            if (targetCount >= 1)
            {
                //-----------------------------------------------------------------------------
                // ブレンドシェイプの場合、各ターゲットシェイプの 2 つの頂点座標が同じかどうかも
                // チェックします。
                bool isSamePos = true;
                const ImpShapeData& dstShapeData = shapeDatas[meshParam.m_VertexShapeIdxs[iPos]];
                if (dstShapeData.m_KeyShapes.size() == shapeData.m_KeyShapes.size())
                {
                    const int dstSrcVtxIdx = meshParam.m_VertexSrcVtxIdxs[iPos];
                    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
                    {
                        const ImpVtxAttrib& srcAttribPos = shapeData.m_KeyShapes[1 + iTarget].m_AttribPos;
                        const ImpVtxAttrib& dstAttribPos = dstShapeData.m_KeyShapes[1 + iTarget].m_AttribPos;
                        if (srcAttribPos.m_Count != 0 &&
                            dstAttribPos.m_Count != 0)
                        {
                            const MFloatVector srcPos = srcAttribPos.GetFloatVector(iSrcVtx);
                            const MFloatVector dstPos = dstAttribPos.GetFloatVector(dstSrcVtxIdx);
                            if (!dstPos.isEquivalent(srcPos))
                            {
                                isSamePos = false;
                                break;
                            }
                        }
                    }
                }
                else
                {
                    isSamePos = false;
                }

                if (isSamePos)
                {
                    return iPos;
                }
            }
            else
            {
                //-----------------------------------------------------------------------------
                // ブレンドシェイプでない場合
                return iPos;
            }
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードの頂点座標配列に値を追加します。
//!
//! @param[in,out] meshParam mesh ノードのパラメータです。
//! @param[in,out] iPosFromSrc 中間ファイルの頂点インデックスに対する mesh ノードの頂点座標インデックスです。
//! @param[in] bones ボーン配列です。
//! @param[in] iDstBone 対象ボーンのインデックスです。
//! @param[in] shapeDatas シェイプデータ配列です。
//! @param[in] iShape 対象シェイプデータのインデックスです。
//! @param[in] magnify 移動値や頂点座標に掛ける倍率です。
//! @param[in] iSrcVtx 中間ファイルの頂点インデックスです。
//!
//! @return mesh ノードの頂点座標インデックスを返します。
//-----------------------------------------------------------------------------
static int AppendMeshPos(
    ImpMeshParam& meshParam,
    RIntArray& iPosFromSrc,
    const ImpBoneArray& bones,
    const size_t iDstBone,
    const ImpShapeDataArray& shapeDatas,
    const size_t iShape,
    const double magnify,
    const int iSrcVtx
)
{
    int iDstPos = iPosFromSrc[iSrcVtx];
    if (iDstPos == -1)
    {
        const ImpShapeData& shapeData = shapeDatas[iShape];
        const ImpKeyShape& baseShape = shapeData.m_KeyShapes[0];
        MPoint pos = MPoint(baseShape.m_AttribPos.GetFloatVector(iSrcVtx)) * magnify;
        ImpVtxMtx vmt;
        if (shapeData.m_SkinCompCount == 0) // リジッドボディ
        {
            vmt.Append(static_cast<int>(iDstBone), 1.0f);
        }
        else if (shapeData.m_SkinCompCount == 1) // リジッドスキニング
        {
            const int iInf = baseShape.m_AttribIdxs[0].m_IntValues[iSrcVtx];
            pos *= bones[iInf].m_BindGlobalMtx * bones[iDstBone].m_BindGlobalInvMtx;
            vmt.Append(iInf, 1.0f);
        }
        else // スムーススキニング
        {
            pos *= bones[iDstBone].m_BindGlobalInvMtx;
            const int idxOfss[2] =
            {
                iSrcVtx * baseShape.m_AttribIdxs[0].m_CompCount,
                iSrcVtx * baseShape.m_AttribIdxs[1].m_CompCount
            };
            for (int iLb = 0; iLb < shapeData.m_SkinCompCount; ++iLb)
            {
                const int iHint = (iLb < 4) ? 0 : 1;
                const int iComp = (iLb < 4) ? iLb : iLb - 4;
                const int idxOfs = idxOfss[iHint];
                const float weight = baseShape.m_AttribWgts[iHint].m_FloatValues[idxOfs + iComp];
                if (weight != 0.0f)
                {
                    const int iInf = baseShape.m_AttribIdxs[iHint].m_IntValues[idxOfs + iComp];
                    vmt.Append(iInf, weight);
                }
            }
        }
        const int iVmt = RFindAppendValue(meshParam.m_VtxMtxs, vmt);

        iDstPos = FindMeshPos(meshParam, shapeDatas, iShape, iSrcVtx, pos, iVmt);
        if (iDstPos == -1)
        {
            iDstPos = static_cast<int>(meshParam.m_VertexArray.length());
            meshParam.m_VertexArray.append(pos);
            meshParam.m_VtxMtxIdxs.push_back(iVmt);
            meshParam.m_VertexShapeIdxs.push_back(static_cast<int>(iShape));
            meshParam.m_VertexSrcVtxIdxs.push_back(iSrcVtx);
        }
        iPosFromSrc[iSrcVtx] = iDstPos;
    }
    return iDstPos;
}

//-----------------------------------------------------------------------------
//! @brief オブジェクト空間に変換した法線を取得します。
//!
//! @param[in] bones ボーン配列です。
//! @param[in] iDstBone 対象ボーンのインデックスです。
//! @param[in] shapeData シェイプデータです。
//! @param[in] baseShape ベースシェイプのキーシェイプです。
//! @param[in] attribNrm 法線の頂点属性です。
//! @param[in] iSrcVtx 中間ファイルの頂点インデックスです。
//!
//! オブジェクト空間に変換した法線を返します。
//-----------------------------------------------------------------------------
inline MVector GetObjectSpaceNrm(
    const ImpBoneArray& bones,
    const size_t iDstBone,
    const ImpShapeData& shapeData,
    const ImpKeyShape& baseShape,
    const ImpVtxAttrib& attribNrm,
    const int iSrcVtx
)
{
    MVector nrm = attribNrm.GetVector(iSrcVtx);
    if (shapeData.m_SkinCompCount == 1)
    {
        const int iInf = baseShape.m_AttribIdxs[0].m_IntValues[iSrcVtx];
        nrm *= bones[iInf].m_BindGlobalMtx * bones[iDstBone].m_BindGlobalInvMtx;
        nrm.normalize();
    }
    else if (shapeData.m_SkinCompCount >= 2)
    {
        nrm *= bones[iDstBone].m_BindGlobalInvMtx;
        nrm.normalize();
    }
    return nrm;
}

//-----------------------------------------------------------------------------
//! @brief RGBA 4 成分の頂点カラーを取得します。
//!
//! @param[in] pValue 頂点属性の頂点カラー値へのポインタです。
//! @param[in] compCount 頂点属性の頂点カラーの成分数です。
//!
//! @return RGBA 4 成分の頂点カラーを返します。
//-----------------------------------------------------------------------------
static MColor GetRgbaVertexColor(const float* pValue, const int compCount)
{
    MColor col(0.0f, 0.0f, 0.0f, 1.0f);
    if (compCount == 1)
    {
        col.r = col.g = col.b = pValue[0];
    }
    else
    {
        for (int compIdx = 0; compIdx < compCount; ++compIdx)
        {
            col[compIdx] = pValue[compIdx];
        }
    }
    return col;
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードの頂点属性配列に値を追加します。
//!
//! @param[in,out] meshParam mesh ノードのパラメータです。
//! @param[in,out] iUvFromSrc 中間ファイルの頂点インデックスに対する mesh ノードの UV のインデックスです。
//! @param[in] bones ボーン配列です。
//! @param[in] iDstBone 対象ボーンのインデックスです。
//! @param[in] shapeData シェイプデータです。
//! @param[in] colCompCounts カラーセットごとの成分数です。
//! @param[in] uvSetCount UV セット数です。
//! @param[in] iSrcVtx 中間ファイルの頂点インデックスです。
//! @param[in] iDstFace mesh ノードのフェースインデックスです。
//! @param[in] iDstPos mesh ノードの頂点座標インデックスです。
//-----------------------------------------------------------------------------
static void AppendMeshVtxAttrib(
    ImpMeshParam& meshParam,
    RIntArray& iUvFromSrc,
    const ImpBoneArray& bones,
    const size_t iDstBone,
    const ImpShapeData& shapeData,
    const RIntArray& colCompCounts,
    const int uvSetCount,
    const int iSrcVtx,
    const int iDstFace,
    const int iDstPos
)
{
    const int targetCount = static_cast<int>(meshParam.m_MeshTargets.size());

    //-----------------------------------------------------------------------------
    // 法線
    const ImpKeyShape& baseShape = shapeData.m_KeyShapes[0];
    const ImpVtxAttrib& attribNrm = baseShape.m_AttribNrm;
    if (attribNrm.m_Count != 0)
    {
        const MVector nrm = GetObjectSpaceNrm(bones, iDstBone, shapeData, baseShape, attribNrm, iSrcVtx);
        meshParam.m_NrmArray.append(nrm);
        meshParam.m_NrmFaceList.append(iDstFace);
        meshParam.m_NrmVtxList.append(iDstPos);

        //-----------------------------------------------------------------------------
        // 各ターゲットシェイプの法線を追加します。
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            if (1 + iTarget < static_cast<int>(shapeData.m_KeyShapes.size()))
            {
                const ImpVtxAttrib& targetAttribNrm = shapeData.m_KeyShapes[1 + iTarget].m_AttribNrm;
                if (targetAttribNrm.m_Count != 0)
                {
                    const MVector targetNrm = GetObjectSpaceNrm(bones, iDstBone, shapeData, baseShape, targetAttribNrm, iSrcVtx);
                    ImpMeshTarget& target = meshParam.m_MeshTargets[iTarget];
                    target.m_NrmArray.append(targetNrm);
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // カラー
    for (int iColSet = 0; iColSet < ImpKeyShape::COL_MAX; ++iColSet)
    {
        if (colCompCounts[iColSet] != 0)
        {
            const ImpVtxAttrib& attribCol = baseShape.m_AttribCols[iColSet];
            MColor col(1.0f, 1.0f, 1.0f, 1.0f);
            if (attribCol.m_Count != 0)
            {
                const float* pCol = &attribCol.m_FloatValues[iSrcVtx * attribCol.m_CompCount];
                col = GetRgbaVertexColor(pCol, attribCol.m_CompCount);
            }
            meshParam.m_ColArrays[iColSet].append(col);

            //-----------------------------------------------------------------------------
            // 各ターゲットシェイプのカラーを追加します。
            for (int iTarget = 0; iTarget < targetCount; ++iTarget)
            {
                if (1 + iTarget < static_cast<int>(shapeData.m_KeyShapes.size()))
                {
                    const ImpVtxAttrib& targetAttribCol = shapeData.m_KeyShapes[1 + iTarget].m_AttribCols[iColSet];
                    if (targetAttribCol.m_Count != 0)
                    {
                        const float* pTargetCol = &targetAttribCol.m_FloatValues[iSrcVtx * targetAttribCol.m_CompCount];
                        const MColor targetCol = GetRgbaVertexColor(pTargetCol, targetAttribCol.m_CompCount);
                        ImpMeshTarget& target = meshParam.m_MeshTargets[iTarget];
                        target.m_ColArrays[iColSet].append(targetCol);
                    }
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // UV
    if (uvSetCount != 0)
    {
        int iDstUv = iUvFromSrc[iSrcVtx];
        if (iDstUv == -1)
        {
            iDstUv = meshParam.m_UvSets[0].m_Us.length();
            for (int iUvSet = 0; iUvSet < uvSetCount; ++iUvSet)
            {
                const ImpVtxAttrib& attribUv = baseShape.m_AttribUvs[iUvSet];
                ImpUvSet& uvSet = meshParam.m_UvSets[iUvSet];
                if (attribUv.m_Count != 0)
                {
                    const float* pUv = &attribUv.m_FloatValues[iSrcVtx * attribUv.m_CompCount];
                    uvSet.m_Us.append(pUv[0]);
                    uvSet.m_Vs.append(1.0f - pUv[1]);
                }
                else
                {
                    uvSet.m_Us.append(0.0f);
                    uvSet.m_Vs.append(0.0f);
                }
            }
            iUvFromSrc[iSrcVtx] = iDstUv;
        }
        for (int iUvSet = 0; iUvSet < uvSetCount; ++iUvSet)
        {
            meshParam.m_UvSets[iUvSet].m_Ids.append(iDstUv);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief uvChooser ノードを作成します。
//!
//! @param[in] shapePath シェイプの DAG パスです。
//! @param[in] iUvSet UV セットのインデックスです。
//! @param[in] sampler サンプラです。
//-----------------------------------------------------------------------------
static void CreateUvChooser(
    const MDagPath& shapePath,
    const int iUvSet,
    const ImpSampler& sampler
)
{
    //-----------------------------------------------------------------------------
    // uvChooser ノードが接続されていなければ作成します。
    MObject chooserObj;
    if (!sampler.m_PlaceObj.isNull())
    {
        MFnDependencyNode placeFn(sampler.m_PlaceObj);
        MPlug uvCoordPlug = placeFn.findPlug("uvCoord");
        MPlugArray plugArray;
        uvCoordPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            if (plugArray[0].node().apiType() == MFn::kUvChooser)
            {
                chooserObj = plugArray[0].node();
            }
        }
        else
        {
            MFnDependencyNode chooserFn;
            chooserObj = chooserFn.create("uvChooser");
            MGlobal::executeCommand("connectAttr " + chooserFn.name() + ".outUv " + uvCoordPlug.name());
            MGlobal::executeCommand("connectAttr " + chooserFn.name() + ".outVertexUvOne " +
                placeFn.name() + ".vertexUvOne");
            MGlobal::executeCommand("connectAttr " + chooserFn.name() + ".outVertexUvTwo " +
                placeFn.name() + ".vertexUvTwo");
            MGlobal::executeCommand("connectAttr " + chooserFn.name() + ".outVertexUvThree " +
                placeFn.name() + ".vertexUvThree");
            MGlobal::executeCommand("connectAttr " + chooserFn.name() + ".outVertexCameraOne " +
                placeFn.name() + ".vertexCameraOne");
        }
    }
    if (chooserObj.isNull())
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // シェイプと uvChooser ノードを接続します。
    const MFnDependencyNode chooserFn(chooserObj);
    MPlug uvSetsPlug = chooserFn.findPlug("uvSets");
    MGlobal::executeCommand("connectAttr " + shapePath.partialPathName() + ".uvSet[" + iUvSet + "].uvSetName " +
        uvSetsPlug.name() + "[" + uvSetsPlug.evaluateNumElements() + "]", true);
}

//-----------------------------------------------------------------------------
//! @brief blendShape ノードをヒストリーから取得します。
//!        存在しなければ kNullObj を返します。
//!
//! @param[in] shapePath シェイプの DAG パスです。
//!
//! @return blendShape ノードのオブジェクトを返します。
//-----------------------------------------------------------------------------
static MObject GetBlendShapeNodeFromHistory(const MDagPath& shapePath)
{
    MObject blendObj;
    MStringArray rets;
    MGlobal::executeCommand("ls -typ blendShape `listHistory " + shapePath.partialPathName() + "`", rets);
    if (rets.length() != 0)
    {
        blendObj = GetNodeObjectByName(rets[0]);
    }
    return  blendObj;
}

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプターゲットを作成して、ブレンドシェイプを設定します。
//!
//! @param[out] targetShapePaths ブレンドシェイプターゲットのシェイプノードの DAG パス配列です。
//! @param[in,out] lodInfo LOD 情報です。
//! @param[in] bone ボーンです。
//! @param[in] boneShapePath ボーンのシェイプノードの DAG パスです。
//! @param[in] meshParam mesh ノードのパラメータです。
//! @param[in] colCompCounts カラーセットごとの成分数です。
//! @param[in] srcLodIdx インポートするサブメッシュの LOD インデックスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CreateBlendShape(
    MDagPathArray& targetShapePaths,
    ImpLodInfo& lodInfo,
    const ImpBone& bone,
    const MDagPath& boneShapePath,
    const ImpMeshParam& meshParam,
    const RIntArray& colCompCounts,
    const int srcLodIdx
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // ベースシェイプの大きさを取得します。
    const MFnTransform baseXformFn(bone.m_XformPath);
    MBoundingBox box = baseXformFn.boundingBox(); // 親ノードの座標系
    box.transformUsing(bone.m_XformPath.exclusiveMatrix());
    double stepX = RRound((box.max().x - box.min().x) * 2.0);
    if (stepX == 0.0)
    {
        stepX = 1.0;
    }
    MVector curTranslate = baseXformFn.getTranslation(MSpace::kWorld);
    curTranslate.x += stepX;

    //-----------------------------------------------------------------------------
    // 各ブレンドシェイプターゲットを作成します。
    bool updatesNrm = false;
    bool updatesCol = false;
    MDagPathArray targetXformPaths;
    const int targetCount = static_cast<int>(meshParam.m_MeshTargets.size());
    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
    {
        //-----------------------------------------------------------------------------
        // transform ノードを作成します。
        const ImpMeshTarget& target = meshParam.m_MeshTargets[iTarget];
        MObject parentObj = MObject::kNullObj;
        if (lodInfo.GetLodCount() >= 2)
        {
            if (srcLodIdx >= static_cast<int>(lodInfo.m_BlendGroupPaths.length()))
            {
                MFnTransform blendGroupXformFn;
                blendGroupXformFn.create();
                blendGroupXformFn.setName(("Targets_" + GetLodLevelRootName(srcLodIdx)).c_str());
                MDagPath blendGroupPath;
                blendGroupXformFn.getPath(blendGroupPath);
                lodInfo.m_BlendGroupPaths.append(blendGroupPath);
            }
            parentObj = lodInfo.m_BlendGroupPaths[srcLodIdx].node();
        }
        MFnTransform xformFn;
        xformFn.create(parentObj);
        xformFn.setName(target.m_Name.c_str());
        MDagPath xformPath;
        xformFn.getPath(xformPath);
        xformFn.setTranslation(curTranslate, MSpace::kTransform);
        curTranslate.x += stepX;
        targetXformPaths.append(xformPath);

        //-----------------------------------------------------------------------------
        // mesh ノードを作成します。
        MFnMesh meshFn;
        meshFn.create(target.m_VertexArray.length(), meshParam.m_VertexCounts.length(),
            target.m_VertexArray, meshParam.m_VertexCounts,
            meshParam.m_FaceConnects, xformPath.node(), &status);
        CheckStatusMsg(status, "Cannot create blend shape target: " + xformPath.partialPathName());
        meshFn.setName(CreateShapeNodeName(xformFn.name()));
        MDagPath shapePath;
        meshFn.getPath(shapePath);
        targetShapePaths.append(shapePath);

        //-----------------------------------------------------------------------------
        // 法線を設定します。
        if (meshParam.m_NrmArray.length() != 0)
        {
            const MVectorArray* pNrmArray = &meshParam.m_NrmArray;
            if (target.m_NrmArray.length() == meshParam.m_NrmArray.length())
            {
                pNrmArray = &target.m_NrmArray;
                updatesNrm = true;
            }
            status = meshFn.setFaceVertexNormals(const_cast<MVectorArray&>(*pNrmArray),
                const_cast<MIntArray&>(meshParam.m_NrmFaceList),
                const_cast<MIntArray&>(meshParam.m_NrmVtxList), MSpace::kObject);
            CheckStatusMsg(status, "Cannot set normal: " + xformPath.partialPathName());
            AdjustMeshNormal(meshFn, shapePath);
        }
        else
        {
            SetMeshHardEdge(meshFn, shapePath);
        }

        //-----------------------------------------------------------------------------
        // カラーを設定します。
        MString firstColSetName;
        for (int iColSet = 0; iColSet < ImpKeyShape::COL_MAX; ++iColSet)
        {
            const MColorArray& colArray = target.m_ColArrays[iColSet];
            if (colCompCounts[iColSet] != 0 &&
                colArray.length() != 0 &&
                colArray.length() == meshParam.m_ColArrays[iColSet].length())
            {
                MString colSetName = MString("nw4f_color") + iColSet;
                const MFnMesh::MColorRepresentation rep = (colCompCounts[iColSet] == 3) ?
                    MFnMesh::kRGB : MFnMesh::kRGBA;
                meshFn.createColorSet(colSetName, NULL, false, rep); // 同じ名前がすでにあれば colSetName は変更されます。
                meshFn.setCurrentColorSetName(colSetName); // setFaceVertexColors の前に呼びます。
                status = meshFn.setFaceVertexColors(const_cast<MColorArray&>(colArray),
                    const_cast<MIntArray&>(meshParam.m_AllFaceList),
                    const_cast<MIntArray&>(meshParam.m_AllVtxList));
                CheckStatusMsg(status, MString("Cannot set color: ") + xformPath.partialPathName() + " (" + colSetName + ")");
                if (firstColSetName.length() == 0)
                {
                    firstColSetName = colSetName;
                }
            }
        }
        if (firstColSetName.length() > 0)
        {
            meshFn.setCurrentColorSetName(firstColSetName);
            meshFn.findPlug("displayColors").setValue(true);
            updatesCol = true;
        }

        //-----------------------------------------------------------------------------
        // mesh ノードを更新します。
        meshFn.updateSurface();
    }

    //-----------------------------------------------------------------------------
    // ブレンドシェイプを設定します。
    MString cmd = "blendShape ";
    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
    {
        cmd += targetXformPaths[iTarget].partialPathName() + " ";
    }
    cmd += boneShapePath.partialPathName();
    MGlobal::executeCommand(cmd);

    //-----------------------------------------------------------------------------
    // blendShape ノードのカスタムアトリビュートを設定します。
    MObject blendObj = GetBlendShapeNodeFromHistory(boneShapePath);
    if (!blendObj.isNull())
    {
        const MString blendName = MFnDependencyNode(blendObj).name();
        if (!meshParam.m_UpdatesPos)
        {
            MGlobal::executeCommand("addAttr -ln \"nw4fUpdatePos\" -at \"bool\" -dv 1 -h 1 " + blendName);
            MGlobal::executeCommand("setAttr " + blendName + ".nw4fUpdatePos " +
                ((meshParam.m_UpdatesPos) ? "1" : "0"));
        }
        if (!updatesNrm)
        {
            MGlobal::executeCommand("addAttr -ln \"nw4fUpdateNrm\" -at \"bool\" -dv 1 -h 1 " + blendName);
            MGlobal::executeCommand("setAttr " + blendName + ".nw4fUpdateNrm " +
                ((updatesNrm) ? "1" : "0"));
        }
        if (updatesCol)
        {
            MGlobal::executeCommand("addAttr -ln \"nw4fUpdateCol\" -at \"bool\" -dv 0 -h 1 " + blendName);
            MGlobal::executeCommand("setAttr " + blendName + ".nw4fUpdateCol " +
                ((updatesCol) ? "1" : "0"));
        }
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief mesh ノードにスキニングを設定します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] meshParam mesh ノードのパラメータです。
//! @param[in] bones ボーン配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus SetMeshSkinning(
    const MDagPath& meshPath,
    const ImpMeshParam& meshParam,
    const ImpBoneArray& bones
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 頂点行列に使用されているインフルエンスを取得します。
    RIntArray iInfs;
    for (size_t iVmt = 0; iVmt < meshParam.m_VtxMtxs.size(); ++iVmt)
    {
        const ImpVtxMtx& vmt = meshParam.m_VtxMtxs[iVmt];
        for (int iLb = 0; iLb < vmt.GetBoneCount(); ++iLb)
        {
            RFindAppendValue(iInfs, vmt.GetBoneIndex(iLb));
        }
    }

    //-----------------------------------------------------------------------------
    // インフルエンスをソートします（joint 優先、インデックス小優先）。
    const int infCount = static_cast<int>(iInfs.size());
    for (int iLb0 = 0; iLb0 < infCount - 1; ++iLb0)
    {
        for (int iLb1 = iLb0 + 1; iLb1 < infCount; ++iLb1)
        {
            const ImpBone& bone0 = bones[iInfs[iLb0]];
            const ImpBone& bone1 = bones[iInfs[iLb1]];
            if ((!bone0.m_IsJoint && bone1.m_IsJoint) ||
                (bone0.m_IsJoint == bone1.m_IsJoint && iInfs[iLb0] > iInfs[iLb1]))
            {
                RSwapValue(iInfs[iLb0], iInfs[iLb1]);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // ボーンインデックスをローカルボーンインデックスに変換するテーブルを作成します。
    RIntArray localBoneIdxs(bones.size(), 0);
    int jointCount = 0;
    for (int iLb = 0; iLb < infCount; ++iLb)
    {
        localBoneIdxs[iInfs[iLb]] = iLb;
        if (bones[iInfs[iLb]].m_IsJoint)
        {
            ++jointCount;
        }
    }
    //cerr <<"skin: " << meshPath.partialPathName() << ": " << iInfs << endl << localBoneIdxs << endl;

    //-----------------------------------------------------------------------------
    // スムースバインドします。
    MString dummyJoint;
    MString cmd = "skinCluster -normalizeWeights 1";
    if (jointCount != 0)
    {
        cmd += " -toSelectedBones";
        for (int iLb = 0; iLb < jointCount; ++iLb)
        {
            cmd += " " + bones[iInfs[iLb]].m_XformPath.partialPathName();
        }
    }
    else
    {
        MGlobal::executeCommand("createNode -skipSelect joint", dummyJoint);
        cmd += " -toSelectedBones " + dummyJoint;
    }
    cmd += " " + meshPath.partialPathName();
    //cerr << "bind: " << cmd << endl;
    MStringArray rets;
    MGlobal::executeCommand(cmd, rets);
    MString skinClusterName = rets[0];

    MObject skinObj = GetNodeObjectByName(skinClusterName);
    if (skinObj.isNull())
    {
        DisplayImportError("skinCluster ノードを取得できません: {0}", "Cannot get skinCluster node: {0}",
            meshPath.partialPathName().asChar());
        return MS::kFailure;
    }

    for (int iLb = jointCount; iLb < infCount; ++iLb)
    {
        cmd = "skinCluster -e -addInfluence " + bones[iInfs[iLb]].m_XformPath.partialPathName() +
            " " + skinClusterName;
        MGlobal::executeCommand(cmd);
    }

    if (jointCount == 0)
    {
        cmd = "skinCluster -e -removeInfluence " + dummyJoint + " " + skinClusterName;
        MGlobal::executeCommand(cmd);
        MGlobal::executeCommand("delete " + dummyJoint);
    }

    //-----------------------------------------------------------------------------
    // ウェイトを変更する間 skinCluster ノードを無効にします。
    MFnSkinCluster skinFn(skinObj);
    int nodeStateBak;
    MPlug statePlug = skinFn.findPlug("nodeState");
    statePlug.getValue(nodeStateBak);
    statePlug.setValue(static_cast<int>(1)); // HasNoEffect

    //-----------------------------------------------------------------------------
    // 正規化モードが「インタラクティブ」ならウェイトを変更する間「ポスト」にします。
    const int NORMALIZE_NONE        = 0;
    const int NORMALIZE_INTERACTIVE = 1;
    const int NORMALIZE_POST        = 2;

    int normalizeWeights;
    MPlug normalizePlug = skinFn.findPlug("normalizeWeights");
    normalizePlug.getValue(normalizeWeights);
    if (normalizeWeights == NORMALIZE_INTERACTIVE)
    {
        normalizePlug.setValue(NORMALIZE_POST);
    }

    //-----------------------------------------------------------------------------
    // 全頂点のウェイトを設定します。
    MFnSingleIndexedComponent allVtxCompFn;
    MObject allVtxComp = allVtxCompFn.create(MFn::kMeshVertComponent);
    allVtxCompFn.setCompleteData(static_cast<int>(meshParam.m_VtxMtxIdxs.size()));

    MIntArray allInfIdxs;
    for (int iInf = 0; iInf < infCount; ++iInf)
    {
        allInfIdxs.append(iInf);
    }

    MDoubleArray allWeights;
    for (size_t iVmt = 0; iVmt < meshParam.m_VtxMtxIdxs.size(); ++iVmt)
    {
        MDoubleArray weights(infCount, 0.0);
        double weightSum = 0.0;
        const ImpVtxMtx& vmt = meshParam.m_VtxMtxs[meshParam.m_VtxMtxIdxs[iVmt]];
        for (int iLb = 0; iLb < vmt.GetBoneCount(); ++iLb)
        {
            const double weight = vmt.GetWeight(iLb);
            if (weight != 0.0)
            {
                weights[localBoneIdxs[vmt.GetBoneIndex(iLb)]] = weight;
                weightSum += weight;
            }
        }

        // ウェイトの合計が 1.0 でなければ合計で各ウェイトを割ります。
        if (normalizeWeights != NORMALIZE_NONE &&
            weightSum > 0.0 &&
            !RIsSame(weightSum, 1.0))
        {
            //cerr << "weight sum not 1.0: vtx[" << vIter.index() << "]: " << weightSum << endl;
            const double invWeightSum = 1.0 / weightSum;
            for (int iInf = 0; iInf < infCount; ++iInf)
            {
                weights[iInf] *= invWeightSum;
            }
        }

        YAppendArrayToArray(allWeights, weights);
    }
    //cerr << "set weight: " << allInfIdxs << endl << allWeights << endl;

    status = skinFn.setWeights(meshPath, allVtxComp, allInfIdxs, allWeights,
            false, NULL);
    CheckStatusMsg(status, "Cannot set weights: " + meshPath.partialPathName());

    //-----------------------------------------------------------------------------
    // skinCluster ノードの有効状態とウェイトの正規化を元に戻します。
    statePlug.setValue(nodeStateBak);
    if (normalizeWeights == NORMALIZE_INTERACTIVE)
    {
        normalizePlug.setValue(normalizeWeights);
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 三角形ストリップのインデックス配列を三角形群のインデックス配列に変換します。
//!
//! @param[out] primIdxs 三角形群のインデックス配列を格納します。
//! @param[in] indexValues 三角形ストリップのインデックス配列です。
//! @param[in] vtxOffset 三角形ストリップのインデックス配列中のオフセットです。
//! @param[in] vtxCount 三角形ストリップの頂点数です。
//-----------------------------------------------------------------------------
static void ConvertTriangleStripToTriangles(
    RIntArray& primIdxs,
    const RIntArray& indexValues,
    const int vtxOffset,
    const int vtxCount
)
{
    int iVtxs[3];
    for (int iSrc = 0; iSrc < vtxCount; ++iSrc)
    {
        if (iSrc == 0)
        {
            iVtxs[0] = indexValues[vtxOffset + iSrc    ];
            iVtxs[1] = indexValues[vtxOffset + iSrc + 1];
            iVtxs[2] = indexValues[vtxOffset + iSrc + 2];
        }
        else
        {
            iVtxs[0] = iVtxs[1];
            iVtxs[1] = iVtxs[2];
            iVtxs[2] = indexValues[vtxOffset + iSrc];
        }

        if (iVtxs[0] != iVtxs[1] &&
            iVtxs[1] != iVtxs[2] &&
            iVtxs[2] != iVtxs[0])
        {
            primIdxs.push_back(iVtxs[0]);
            if ((iSrc & 1) == 0)
            {
                primIdxs.push_back(iVtxs[1]);
                primIdxs.push_back(iVtxs[2]);
            }
            else
            {
                primIdxs.push_back(iVtxs[2]);
                primIdxs.push_back(iVtxs[1]);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 不要なフェースなら true を返します。
//!
//! @param[in] vertexArray 頂点座標配列です。
//! @param[in] iDstPoss フェースの頂点インデックス配列です。
//! @param[in] cutsZeroFace ジオメトリ領域ゼロのフェースを省略するなら true です。
//!
//! @return 不要なフェースなら true を返します。
//-----------------------------------------------------------------------------
static bool IsNeedlessFace(
    const MPointArray& vertexArray,
    const RIntArray& iDstPoss,
    const bool cutsZeroFace
)
{
    //-----------------------------------------------------------------------------
    // 四角形以上の場合
    const int faceVtxCount = static_cast<int>(iDstPoss.size());
    if (faceVtxCount >= 4)
    {
        // 同じ頂点を 2 回以上使用
        for (int iV = 0; iV < faceVtxCount - 1; ++iV)
        {
            if (RFindValueInArray(iDstPoss, iDstPoss[iV], iV + 1) != -1)
            {
                return true;
            }
        }
        return false;
    }

    //-----------------------------------------------------------------------------
    // 三角形の場合

    // 同じ頂点を 2 回以上使用
    if (iDstPoss[0] == iDstPoss[1] ||
        iDstPoss[1] == iDstPoss[2] ||
        iDstPoss[2] == iDstPoss[0])
    {
        return true;
    }

    // ジオメトリ領域がゼロ（3 頂点が同一直線上）
    if (cutsZeroFace)
    {
        for (int iV = 0; iV < 3; ++iV)
        {
            const MPoint& p0 = vertexArray[iDstPoss[iV]];
            MVector v1 = vertexArray[iDstPoss[(iV + 1) % 3]] - p0;
            MVector v2 = vertexArray[iDstPoss[(iV + 2) % 3]] - p0;
            v1.normalize();
            v2.normalize();
            const double dp = v1 * v2;
            const double COS_EP = 1.0e-6;
            if (dp < -1.0 + COS_EP ||
                dp >  1.0 - COS_EP)
            {
                return true;
            }
        }
        // 面積を求めて比較する方法もあるが、
        // 頂点座標の範囲が様々なので許容誤差を決めにくい
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードを作成します。
//!
//! @param[in,out] bones ボーン配列です。
//! @param[in,out] lodInfo LOD 情報です。
//! @param[in] iDstBone 対象ボーンのインデックスです。
//! @param[in] magnify 移動値や頂点座標に掛ける倍率です。
//! @param[in] cutsZeroFace ジオメトリ領域ゼロのフェースを省略するなら true です。
//! @param[in] srcLodIdx インポートするサブメッシュの LOD インデックスです。
//! @param[in] verticesElem 頂点配列要素です。
//! @param[in] shapesElem シェイプ配列要素です。
//! @param[in] mtxIdxToBoneIdxs 行列インデックスからボーンインデックスへの変換テーブルです。
//! @param[in] materials マテリアル配列です。
//! @param[in] fmdFile fmd ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CreateMesh(
    ImpBoneArray& bones,
    ImpLodInfo& lodInfo,
    const size_t iDstBone,
    const double magnify,
    const bool cutsZeroFace,
    const int srcLodIdx,
    const RXMLElement* verticesElem,
    const RXMLElement* shapesElem,
    const RIntArray& mtxIdxToBoneIdxs,
    const ImpMaterialArray& materials,
    const ImpIntermediateFile& fmdFile
)
{
    MStatus status;

    const int lodCount = lodInfo.GetLodCount();

    //-----------------------------------------------------------------------------
    // 各シェイプのシェイプデータを取得します。
    ImpBone& bone = bones[iDstBone];
    std::vector<const RXMLElement*> boneShapeElems;
    const size_t boneShapeCount = bone.m_ShapeElmIdxs.size();
    for (size_t iShape = 0; iShape < boneShapeCount; ++iShape)
    {
        boneShapeElems.push_back(&shapesElem->nodes[bone.m_ShapeElmIdxs[iShape]]);
    }

    ImpShapeDataArray shapeDatas;
    RIntArray usedMatIdxs; // 使用されているマテリアルの重複のないインデックス配列です。
    int skinShapeCount = 0;
    int nrmShapeCount = 0;
    RIntArray colCompCounts(ImpKeyShape::COL_MAX, 0); // カラーセットごとの成分数です。
    RStringArray uvAttribNames;
    RIntArray uvHintIdxs;
    size_t maxKeyShapeCount = 0;
    for (size_t iShape = 0; iShape < boneShapeCount; ++iShape)
    {
        GetShapeData(shapeDatas, uvAttribNames, uvHintIdxs,
            verticesElem, srcLodIdx, boneShapeElems[iShape], mtxIdxToBoneIdxs, materials, fmdFile);

        const ImpShapeData& shapeData = shapeDatas.back();
        const ImpKeyShape& baseShape = shapeData.m_KeyShapes[0];
        bone.m_ShapeNames.push_back(shapeData.m_Name);
        RFindAppendValue(usedMatIdxs, shapeData.m_MatIdx);
        if (shapeData.m_SkinCompCount != 0)
        {
            ++skinShapeCount;
        }
        if (baseShape.m_AttribNrm.m_Count != 0)
        {
            ++nrmShapeCount;
        }
        for (int iColSet = 0; iColSet < ImpKeyShape::COL_MAX; ++iColSet)
        {
            const ImpVtxAttrib& attribCol = baseShape.m_AttribCols[iColSet];
            if (attribCol.m_CompCount > colCompCounts[iColSet])
            {
                colCompCounts[iColSet] = attribCol.m_CompCount;
            }
        }
        if (shapeData.m_KeyShapes.size() > maxKeyShapeCount)
        {
            maxKeyShapeCount = shapeData.m_KeyShapes.size();
        }
    }
    const int targetCount = (maxKeyShapeCount >= 1) ? static_cast<int>(maxKeyShapeCount) - 1 : 0;
    const int uvSetCount = static_cast<int>(uvHintIdxs.size());
    //cerr << "shape: " << bone.m_Name << ": " << bone.m_ShapeNames << ": " << uvSetCount << endl;

    //-----------------------------------------------------------------------------
    // 頂点属性の配列を作成します。
    ImpMeshParam meshParam(targetCount);
    for (size_t iShape = 0; iShape < boneShapeCount; ++iShape)
    {
        const RXMLElement* shapeElem = boneShapeElems[iShape];
        ImpShapeData& shapeData = shapeDatas[iShape];
        const ImpKeyShape& baseShape = shapeData.m_KeyShapes[0];
        const ImpVtxAttrib& attribPos = baseShape.m_AttribPos;
        if (attribPos.m_Count != 0)
        {
            const RXMLElement* meshsElem = shapeElem->FindElement("mesh_array", false); // 3.0.0 で追加
            const RXMLElement* meshElem;
            if (meshsElem != NULL)
            {
                const int iLod = (srcLodIdx < meshsElem->nodes.size()) ?
                    srcLodIdx : static_cast<int>(meshsElem->nodes.size()) - 1;
                meshElem = &meshsElem->nodes[iLod];
            }
            else
            {
                meshElem = shapeElem->FindElement("mesh");
            }
            const std::string meshMode = meshElem->GetAttribute("mode");

            const int iMeshStream = atoi(meshElem->GetAttribute("stream_index").c_str());
            RIntArray indexValues;
            fmdFile.GetIntStream(indexValues, iMeshStream);
            RIntArray iPosFromSrc(attribPos.m_Count, -1); // 中間ファイルの頂点インデックスに対する mesh ノードの頂点座標インデックスです。
            RIntArray iUvFromSrc(attribPos.m_Count, -1); // 中間ファイルの頂点インデックスに対する mesh ノードの UV のインデックスです。

            const RXMLElement* submeshesElem = meshElem->FindElement("submesh_array");
            for (size_t iSub = 0; iSub < submeshesElem->nodes.size(); ++iSub)
            {
                const RXMLElement* submeshElem = &submeshesElem->nodes[iSub];
                const int vtxOffset = atoi(submeshElem->GetAttribute("offset").c_str());
                const int vtxCount = atoi(submeshElem->GetAttribute("count").c_str());
                if (vtxCount < 3)
                {
                    continue; // 頂点数が 3 未満のフェースは追加しません。
                }

                //-----------------------------------------------------------------------------
                // プリミティブのインデックス配列を作成します。
                RIntArray primIdxs;
                if (meshMode == "triangles" || meshMode == "triangle_fan")
                {
                    primIdxs.resize(vtxCount);
                    for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
                    {
                         primIdxs[iVtx] = indexValues[vtxOffset + iVtx];
                    }
                }
                else // if (meshMode == "triangle_strip")
                {
                    ConvertTriangleStripToTriangles(primIdxs, indexValues, vtxOffset, vtxCount);
                }

                //-----------------------------------------------------------------------------
                // フェースについてループします。
                const int faceVtxCount = (meshMode == "triangle_fan") ? vtxCount : 3;
                const int primVtxCount = static_cast<int>(primIdxs.size());
                for (int iVtx = 0; iVtx < primVtxCount; iVtx += faceVtxCount)
                {
                    //-----------------------------------------------------------------------------
                    // 頂点座標配列に値を追加します。
                    const int beforeVertexCount = meshParam.m_VertexArray.length();
                    const size_t beforeVtxMtxCount = meshParam.m_VtxMtxs.size();
                    const size_t beforeVtxMtxIdxCount = meshParam.m_VtxMtxIdxs.size();
                    RIntArray iDstPoss(faceVtxCount);
                    for (int iV = 0; iV < faceVtxCount; ++iV)
                    {
                        const int iSrcVtx = primIdxs[iVtx + iV];
                        iDstPoss[iV] = AppendMeshPos(meshParam, iPosFromSrc,
                            bones, iDstBone, shapeDatas, iShape, magnify, iSrcVtx);
                    }

                    //-----------------------------------------------------------------------------
                    // 不要なフェースならスキップします。
                    if (IsNeedlessFace(meshParam.m_VertexArray, iDstPoss, cutsZeroFace))
                    {
                        // スキップするフェースについて追加した頂点を削除します。
                        if (static_cast<int>(meshParam.m_VertexArray.length()) > beforeVertexCount)
                        {
                            meshParam.m_VertexArray.setLength(beforeVertexCount);
                            for (size_t iPos = 0; iPos < iPosFromSrc.size(); ++iPos)
                            {
                                if (iPosFromSrc[iPos] >= beforeVertexCount)
                                {
                                    iPosFromSrc[iPos] = -1;
                                }
                            }
                        }
                        if (meshParam.m_VtxMtxs.size() > beforeVtxMtxCount)
                        {
                            meshParam.m_VtxMtxs.resize(beforeVtxMtxCount);
                        }
                        if (meshParam.m_VtxMtxIdxs.size() > beforeVtxMtxIdxCount)
                        {
                            meshParam.m_VtxMtxIdxs.resize(beforeVtxMtxIdxCount);
                        }
                        continue;
                    }

                    //-----------------------------------------------------------------------------
                    // フェースの各頂点属性配列に値を追加します。
                    const int iDstFace = meshParam.m_VertexCounts.length();
                    meshParam.m_VertexCounts.append(faceVtxCount);
                    ++shapeData.m_FaceCount;
                    for (int iV = 0; iV < faceVtxCount; ++iV)
                    {
                        const int iSrcVtx = primIdxs[iVtx + iV];
                        const int iDstPos = iDstPoss[iV];

                        meshParam.m_FaceConnects.append(iDstPos);
                        meshParam.m_AllFaceList.append(iDstFace);
                        meshParam.m_AllVtxList.append(iDstPos);

                        AppendMeshVtxAttrib(meshParam, iUvFromSrc,
                            bones, iDstBone, shapeData, colCompCounts, uvSetCount,
                            iSrcVtx, iDstFace, iDstPos);
                    }
                } // iVtx
            } // iSub
            //cerr << "iPosFromSrc: " << iPosFromSrc << endl;

            //-----------------------------------------------------------------------------
            // ブレンドシェイプターゲットの頂点属性を設定します。
            for (int iTarget = 0; iTarget < targetCount; ++iTarget)
            {
                ImpMeshTarget& target = meshParam.m_MeshTargets[iTarget];
                for (int iPos = static_cast<int>(target.m_VertexArray.length());
                    iPos < static_cast<int>(meshParam.m_VertexArray.length()); ++iPos)
                {
                    target.m_VertexArray.append(meshParam.m_VertexArray[iPos]);
                }
                if (1 + iTarget >= static_cast<int>(shapeData.m_KeyShapes.size()))
                {
                    continue;
                }
                const ImpKeyShape& keyShape = shapeData.m_KeyShapes[1 + iTarget];
                target.m_Name = keyShape.m_Name;
                if (keyShape.m_AttribPos.m_Count != attribPos.m_Count)
                {
                    continue;
                }
                meshParam.m_UpdatesPos = true;
                RBoolArray isPosCalced(target.m_VertexArray.length(), false);
                for  (int iSrcVtx = 0; iSrcVtx < keyShape.m_AttribPos.m_Count; ++iSrcVtx)
                {
                    const int iPos = iPosFromSrc[iSrcVtx];
                    if (iPos != -1 && !isPosCalced[iPos])
                    {
                        MPoint pos = MPoint(keyShape.m_AttribPos.GetFloatVector(iSrcVtx)) * magnify;
                        if (shapeData.m_SkinCompCount == 1)
                        {
                            const int iInf = baseShape.m_AttribIdxs[0].m_IntValues[iSrcVtx];
                            pos *= bones[iInf].m_BindGlobalMtx * bones[iDstBone].m_BindGlobalInvMtx;
                        }
                        else if (shapeData.m_SkinCompCount >= 2)
                        {
                            pos *= bones[iDstBone].m_BindGlobalInvMtx;
                        }
                        target.m_VertexArray[iPos] = pos;
                        isPosCalced[iPos] = true;
                    }
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 有効なフェースがなければ mesh ノードを作成しません。
    if (meshParam.m_VertexArray.length() == 0)
    {
        bone.m_HasShape = bone.m_HasSkinedShape = false;
        if (bone.m_CompressEnable)
        {
            AddNoCompressNodeAttr(bone.m_XformPath.partialPathName(), false);
        }
        //DisplayImportError("mesh ノードが不正です: {0}", "Mesh is wrong: {0}",
        //  bone.m_XformPath.partialPathName().asChar());
        //return MS::kFailure;
        DisplayImportWarning("mesh ノードが不正です: {0}", "Mesh is wrong: {0}",
            bone.m_XformPath.partialPathName().asChar());
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // シェイプ用 transform ノードが必要なら作成します。
    MDagPath shapeXformPath = bone.m_XformPath;
    if (lodCount >= 2 && srcLodIdx < static_cast<int>(bone.m_LodXformPaths.length()))
    {
        shapeXformPath = bone.m_LodXformPaths[srcLodIdx];
    }
    if (skinShapeCount != 0 && bone.m_SeparatesSkinObject)
    {
        std::string shapeXformName = shapeDatas[0].m_OrgBoneName;
        if (shapeXformName.empty())
        {
            shapeXformName = bone.m_Name + "Skin";
        }
        MObject parentObj = (lodCount >= 2) ?
            lodInfo.m_RootPaths[srcLodIdx].node() : MObject::kNullObj;
        MFnTransform xformFn;
        xformFn.create(parentObj, &status);
        CheckStatusMsg(status, "Cannot create mesh transform object: " + bone.m_XformPath.partialPathName());
        xformFn.setName(shapeXformName.c_str());
        xformFn.getPath(shapeXformPath);
        xformFn.set(MTransformationMatrix(bone.m_BindGlobalMtx));
    }

    //-----------------------------------------------------------------------------
    // mesh ノードを作成します。
    MFnMesh meshFn;
    meshFn.create(meshParam.m_VertexArray.length(), meshParam.m_VertexCounts.length(),
        meshParam.m_VertexArray, meshParam.m_VertexCounts,
        meshParam.m_FaceConnects, shapeXformPath.node(), &status);
    CheckStatusMsg(status, "Cannot create mesh object: " + shapeXformPath.partialPathName());
    meshFn.setName(CreateShapeNodeName(MFnDagNode(shapeXformPath.node()).name()));
    MDagPath boneShapePath;
    meshFn.getPath(boneShapePath);
    const MString xformName = shapeXformPath.partialPathName();
    meshFn.findPlug("visibility").setValue(bone.m_Visibility);

    if (lodCount == 1 || srcLodIdx == 0)
    {
        bone.m_ShapeXformPath = shapeXformPath;
        bone.m_ShapePath = boneShapePath;
    }
    bone.m_LodShapeXformPaths.append(shapeXformPath);
    bone.m_LodShapePaths.append(boneShapePath);

    //-----------------------------------------------------------------------------
    // 法線を設定します。
    if (nrmShapeCount != 0)
    {
        status = meshFn.setFaceVertexNormals(meshParam.m_NrmArray, meshParam.m_NrmFaceList, meshParam.m_NrmVtxList, MSpace::kObject);
        //if (!status)
        //{
        //  cerr << "set normal: " << meshParam.m_NrmArray.length() << "," << meshParam.m_NrmFaceList.length() << "," << meshParam.m_NrmVtxList.length() << "," << status << endl;
        //  cerr << meshParam.m_VertexCounts.length() << "," << meshParam.m_VertexCounts << endl;
        //  cerr << meshParam.m_FaceConnects << endl;
        //}
        CheckStatusMsg(status, "Cannot set normal: " + xformName);
        AdjustMeshNormal(meshFn, boneShapePath);
    }
    else
    {
        SetMeshHardEdge(meshFn, boneShapePath);
    }

    //-----------------------------------------------------------------------------
    // カラーを設定します。
    MString firstColSetName;
    for (int iColSet = 0; iColSet < ImpKeyShape::COL_MAX; ++iColSet)
    {
        if (colCompCounts[iColSet] != 0)
        {
            MString colSetName = MString("nw4f_color") + iColSet;
            const MFnMesh::MColorRepresentation rep = (colCompCounts[iColSet] == 3) ?
                MFnMesh::kRGB : MFnMesh::kRGBA;
            meshFn.createColorSet(colSetName, NULL, false, rep); // 同じ名前がすでにあれば colSetName は変更されます。
            meshFn.setCurrentColorSetName(colSetName); // setFaceVertexColors の前に呼びます。
            status = meshFn.setFaceVertexColors(meshParam.m_ColArrays[iColSet], meshParam.m_AllFaceList, meshParam.m_AllVtxList);
            CheckStatusMsg(status, MString("Cannot set color: ") + xformName + " (" + colSetName + ")");
            if (firstColSetName.length() == 0)
            {
                firstColSetName = colSetName;
            }
        }
    }

    if (firstColSetName.length() > 0)
    {
        meshFn.setCurrentColorSetName(firstColSetName);
        meshFn.findPlug("displayColors").setValue(true);
        //meshFn.findPlug("displayColorChannel").setValue("Diffuse"); // default is "Ambient+Diffuse"
    }

    //-----------------------------------------------------------------------------
    // UV を設定します。
    for (int iUvSet = 0; iUvSet < uvSetCount; ++iUvSet)
    {
        ImpUvSet& uvSet = meshParam.m_UvSets[iUvSet];
        if (iUvSet == 0)
        {
            uvSet.m_Name = "map1";
        }
        else
        {
            #ifdef SET_UV_SET_NAME_FIX
            uvSet.m_Name = MString("nw4f_fix") + uvHintIdxs[iUvSet];
            if (uvAttribNames[iUvSet] != RGetNumberString(uvHintIdxs[iUvSet], "_u%d"))
            {
                uvSet.m_Name += MString("_") + uvAttribNames[iUvSet].c_str();
            }
            #else
            uvSet.m_Name = MString("nw4f_uv") + uvHintIdxs[iUvSet];
            #endif
        }
        if (iUvSet == 0)
        {
            MString curName;
            meshFn.getCurrentUVSetName(curName);
            if (uvSet.m_Name != curName)
            {
                meshFn.renameUVSet(curName, uvSet.m_Name);
            }
        }
        else
        {
            status = meshFn.createUVSet(uvSet.m_Name); // 同じ名前がすでにあれば uvSetName は変更されます。
            CheckStatusMsg(status, MString("Cannot create UV set: ") + xformName + " (" + uvSet.m_Name + ")");
        }

        //MGlobal::displayInfo("set UV: " + uvSet.m_Name + ": " + uvSet.m_Us.length());
        status = meshFn.setUVs(uvSet.m_Us, uvSet.m_Vs, &uvSet.m_Name);
        CheckStatusMsg(status, MString("Cannot set UV: ") + xformName + " (" + uvSet.m_Name + ")");

        status = meshFn.assignUVs(meshParam.m_VertexCounts, uvSet.m_Ids, &uvSet.m_Name);
        CheckStatusMsg(status, MString("Cannot assign UV: ") + xformName + " (" + uvSet.m_Name + ")");
    }

    //-----------------------------------------------------------------------------
    // mesh ノードを更新します。
    meshFn.updateSurface();

    //-----------------------------------------------------------------------------
    // ブレンドシェイプターゲットを作成して、ブレンドシェイプを設定します。
    MDagPathArray targetShapePaths;
    if (targetCount > 0)
    {
        status = CreateBlendShape(targetShapePaths, lodInfo,
            bone, boneShapePath, meshParam, colCompCounts, srcLodIdx);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // マテリアルを割り当てます。
    //MGlobal::displayInfo("assign material: " + xformName);
    for (int iKeyShape = 0; iKeyShape < 1 + targetCount; ++iKeyShape)
    {
        const MDagPath shapePath = (iKeyShape == 0) ?
            boneShapePath : targetShapePaths[iKeyShape - 1];
        MDagPath xformPath = shapePath;
        xformPath.pop();
        if (usedMatIdxs.size() == 1) // 単一マテリアルの場合
        {
            const ImpMaterial& material = materials[usedMatIdxs[0]];
            MGlobal::executeCommand("sets -e -fe " + MFnDependencyNode(material.m_SGObj).name() + " " +
                shapePath.partialPathName());
        }
        else // 複数マテリアルの場合
        {
            int iFaceTop = 0;
            for (size_t iShape = 0; iShape < boneShapeCount; ++iShape)
            {
                const ImpShapeData& shapeData = shapeDatas[iShape];
                const ImpMaterial& material = materials[shapeData.m_MatIdx];
                MGlobal::executeCommand("sets -e -fe " + MFnDependencyNode(material.m_SGObj).name() + " " +
                    xformPath.partialPathName() + ".f[" + iFaceTop + ":" +
                    (iFaceTop + shapeData.m_FaceCount - 1) + "]");
                iFaceTop += shapeData.m_FaceCount;
            }
        }
    }
    meshFn.syncObject(); // MEL で mesh ノードを操作したので同期します。

    //-----------------------------------------------------------------------------
    // uvChooser ノードを作成します。
    bool isUvChoosed = false;
    for (size_t iMat = 0; iMat < usedMatIdxs.size(); ++iMat)
    {
        const ImpMaterial& material = materials[usedMatIdxs[iMat]];
        for (size_t iSampler = 0; iSampler < material.m_Samplers.size(); ++iSampler)
        {
            const ImpSampler& sampler = material.m_Samplers[iSampler];
            const int iUvSet = RFindValueInArray(uvHintIdxs, sampler.m_UvHintIdx);
            if (iUvSet >= 1)
            {
                CreateUvChooser(boneShapePath, iUvSet, sampler);
                isUvChoosed = true;
            }
        }
    }
    if (isUvChoosed)
    {
        meshFn.syncObject(); // MEL で mesh ノードを操作したので同期します。
    }

    //-----------------------------------------------------------------------------
    // スキニングを設定します。
    if (skinShapeCount != 0)
    {
        status = SetMeshSkinning(boneShapePath, meshParam, bones);
        CheckStatus(status);
        meshFn.syncObject(); // MEL で mesh ノードを操作したので同期します。
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 子ノードを持つボーンの名前の配列を取得します。
//!
//! @param[out] parentBoneNames 子ノードを持つボーンの名前の配列を格納します。
//! @param[in] bonesElem ボーン配列要素です。
//-----------------------------------------------------------------------------
static void GetParentBoneNames(RStringArray& parentBoneNames, const RXMLElement* bonesElem)
{
    parentBoneNames.clear();
    for (size_t iBone = 0; iBone < bonesElem->nodes.size(); ++iBone)
    {
        const std::string parentName = bonesElem->nodes[iBone].GetAttribute("parent_name");
        if (!parentName.empty())
        {
            RFindAppendValue(parentBoneNames, parentName);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief transform (joint) ノードを作成します。
//!
//! @param[in] bone ボーンです。
//! @param[in] pParent 親ボーンへのポインタです（存在しなければ NULL）。
//! @param[in] parentXformPath 親 transform ノードの DAG パスです。
//! @param[in] userDatasElem ユーザーデータ配列要素です。
//!
//! @return 作成した transform ノードの DAG パスを返します。
//-----------------------------------------------------------------------------
static MDagPath CreateTransformNode(
    const ImpBone& bone,
    const ImpBone* pParent,
    const MDagPath parentXformPath,
    const RXMLElement* userDatasElem
)
{
    MObject parentObj = (parentXformPath.isValid()) ? parentXformPath.node() : MObject::kNullObj;

    MFnDagNode dagFn;
    dagFn.create((bone.m_IsJoint) ? "joint" : "transform", bone.m_Name.c_str(), parentObj);
    MDagPath xformPath;
    dagFn.getPath(xformPath);
    MFnTransform xformFn(xformPath);
    const double scaleVals[] = { bone.m_Scale.x, bone.m_Scale.y, bone.m_Scale.z };
    xformFn.setScale(scaleVals);
    const MVector rotateRad = bone.m_Rotate * R_M_DEG_TO_RAD;
    xformFn.setRotation(MEulerRotation(rotateRad.x, rotateRad.y, rotateRad.z));
    xformFn.setTranslation(bone.m_Translate, MSpace::kTransform);
    //xformFn.findPlug("visibility").setValue(bone.m_Visibility);

    if (bone.m_IsJoint)
    {
        MPlug paPlug = xformFn.findPlug("preferredAngle");
        paPlug.child(0).setValue(rotateRad.x);
        paPlug.child(1).setValue(rotateRad.y);
        paPlug.child(2).setValue(rotateRad.z);
        xformFn.findPlug("segmentScaleCompensate").setValue(bone.m_ScaleCompensate);

        if (pParent != NULL && pParent->m_IsJoint)
        {
            const MString cmd = "connectAttr " + parentXformPath.partialPathName() + ".scale " +
                xformPath.partialPathName() + ".inverseScale";
            MGlobal::executeCommand(cmd);
        }

        int labelSide = 0; // center
        if (bone.m_Side == "left")
        {
            labelSide = 1;
        }
        else if (bone.m_Side == "right")
        {
            labelSide = 2;
        }
        else if (bone.m_Side == "none")
        {
            labelSide = 3;
        }
        xformFn.findPlug("side").setValue(labelSide);

        if (!bone.m_Type.empty())
        {
            const int labelType = YGetJointLabelTypeFromString(bone.m_Type);
            xformFn.findPlug("type").setValue(labelType);
            if (labelType == Y_JOINT_LABEL_TYPE_OTHER)
            {
                xformFn.findPlug("otherType").setValue(bone.m_Type.c_str());
            }
        }
    }
    //cerr << "bone: " << bone.m_Name << ": " << bone.m_ParentIdx << ", " << xformPath.partialPathName() << endl;

    //-----------------------------------------------------------------------------
    // transform (joint) ノードのカスタムアトリビュートを設定します。
    if (!bone.m_CompressEnable)
    {
        AddNoCompressNodeAttr(xformPath.partialPathName(), bone.m_CompressEnable);
    }

    //-----------------------------------------------------------------------------
    // ユーザーデータを解析してカスタムアトリビュートを設定します。
    if (userDatasElem != NULL)
    {
        ParseUserDatas(xformPath.partialPathName(), userDatasElem);
    }

    return xformPath;
}

//-----------------------------------------------------------------------------
//! @brief シェイプツリーに含めるボーンなら true を返します。
//!
//! @param[in] bones ボーン配列です。
//! @param[in] iDstBone 対象ボーンのインデックスです。
//! @param[in] lodInfo LOD 情報です。
//!
//! @return シェイプツリーに含めるボーンなら true を返します。
//-----------------------------------------------------------------------------
static bool IsShapeTreeBone(
    const ImpBoneArray& bones,
    const size_t iDstBone,
    const ImpLodInfo& lodInfo
)
{
    if (lodInfo.GetShapeFlag(iDstBone))
    {
        // 対象ボーンがインフルエンスツリーに含まれていて、
        // 対象ボーンまたは先祖ボーンのスキンシェイプを別オブジェクトとして作成するなら
        // シェイプツリーに含めません。
        const ImpBone& bone = bones[iDstBone];
        bool separatesSkinObject = bone.m_SeparatesSkinObject;
        if (!separatesSkinObject)
        {
            size_t iCur = bone.m_ParentIdx;
            while (iCur != -1)
            {
                if (bones[iCur].m_SeparatesSkinObject)
                {
                    separatesSkinObject = true;
                    break;
                }
                iCur = bones[iCur].m_ParentIdx;
            }
        }
        return !(separatesSkinObject && lodInfo.GetInfluenceFlag(iDstBone));
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ボーンを解析して transform (joint) ノードを作成します。
//!
//! @param[in,out] bones ボーン配列です。
//! @param[in] boneElem ボーン要素です。
//! @param[in] magnify 移動値や頂点座標に掛ける倍率です。
//! @param[in] isQuat 回転値がクォータニオンなら true、オイラー角なら false です。
//! @param[in] separatesSkinObject スキンシェイプをワールド下の別オブジェクトとして作成するなら true です。
//! @param[in] parentBoneNames 子ノードを持つボーンの名前の配列です。
//! @param[in] shapesElem シェイプ配列要素です。
//! @param[in] lodInfo LOD 情報です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseBone(
    ImpBoneArray& bones,
    const RXMLElement* boneElem,
    const double magnify,
    const bool isQuat,
    const bool separatesSkinObject,
    const RStringArray& parentBoneNames,
    const RXMLElement* shapesElem,
    const ImpLodInfo& lodInfo
)
{
    MStatus status;

    const size_t iDstBone = bones.size();
    bones.push_back(ImpBone());
    ImpBone& bone = bones.back();

    //-----------------------------------------------------------------------------
    // 属性を解析します。
    bone.m_Name = boneElem->GetAttribute("name");
    bone.m_ParentIdx = FindBoneByName(bones, boneElem->GetAttribute("parent_name"));
    const ImpBone* pParent = (bone.m_ParentIdx != -1) ? &bones[bone.m_ParentIdx] : NULL;
    if (pParent != NULL && pParent->m_IsSkipped)
    {
        bone.m_ParentIdx = -1;
        pParent = NULL;
    }

    std::istringstream(boneElem->GetAttribute("matrix_index")) >> bone.m_SmoothMtxIdx >> bone.m_RigidMtxIdx;
    std::istringstream(boneElem->GetAttribute("scale")) >>
        bone.m_Scale.x >> bone.m_Scale.y >> bone.m_Scale.z;
    const std::string rotateStr = boneElem->GetAttribute("rotate");
    if (isQuat)
    {
        MQuaternion quat;
        std::istringstream(rotateStr) >>
            quat.x >> quat.y >> quat.z >> quat.w;
        MEulerRotation euler = quat.asEulerRotation();
        if (euler.order != MEulerRotation::kXYZ)
        {
            euler.reorderIt(MEulerRotation::kXYZ);
        }
        bone.m_Rotate = euler.asVector() * R_M_RAD_TO_DEG;
    }
    else
    {
        std::istringstream(rotateStr) >>
            bone.m_Rotate.x >> bone.m_Rotate.y >> bone.m_Rotate.z;
    }
    std::istringstream(boneElem->GetAttribute("translate")) >>
        bone.m_Translate.x >> bone.m_Translate.y >> bone.m_Translate.z;
    bone.m_Translate *= magnify;
    bone.m_ScaleCompensate = (boneElem->GetAttribute("scale_compensate") == "true");
    bone.m_Visibility = (boneElem->GetAttribute("visibility") == "true");
    bone.m_CompressEnable = (boneElem->GetAttribute("compress_enable") == "true");
    bone.m_Side = boneElem->GetAttribute("side", "none", false); // 3.6.0 で追加
    bone.m_Type = boneElem->GetAttribute("type", "", false); // 3.6.0 で追加
    const bool isIdentMtx = (
        bone.m_Scale.isEquivalent(MVector::one) &&
        bone.m_Rotate.isEquivalent(MVector::zero) &&
        bone.m_Translate.isEquivalent(MVector::zero));
    const RXMLElement* userDatasElem = boneElem->FindElement("user_data_array", false);

    //-----------------------------------------------------------------------------
    // ボーンの子のシェイプ要素群を取得します。
    bone.m_HasSkinedShape = false;
    if (shapesElem != NULL)
    {
        const size_t shapeCount = shapesElem->nodes.size();
        for (size_t iShape = 0; iShape < shapeCount; ++iShape)
        {
            const RXMLElement* shapeElem = &shapesElem->nodes[iShape];
            const RXMLElement* infoElem = shapeElem->FindElement("shape_info");
            if (infoElem->GetAttribute("bone_name") == bone.m_Name)
            {
                bone.m_ShapeElmIdxs.push_back(static_cast<int>(iShape));
                const int skinCompCount = atoi(infoElem->GetAttribute("vertex_skinning_count").c_str());
                if (skinCompCount != 0)
                {
                    bone.m_HasSkinedShape = true;
                }
            }
        }
    }
    bone.m_HasShape = !bone.m_ShapeElmIdxs.empty();

    //-----------------------------------------------------------------------------
    // スキンシェイプを持つボーンがルートボーンで子ボーンを持つ場合と、
    // スキンシェイプを持つボーンかスキンのインフルエンスでもある場合は、
    // スキンシェイプを別オブジェクトにします。
    const bool hasChild = (RFindValueInArray(parentBoneNames, bone.m_Name) != -1);
    const bool isInfluence = (bone.m_SmoothMtxIdx != -1 || bone.m_RigidMtxIdx != -1);
    bone.m_SeparatesSkinObject = (
        separatesSkinObject   &&
        bone.m_HasSkinedShape &&
        ((bone.m_ParentIdx == -1 && hasChild) || isInfluence));
    const bool dstHasShape = bone.m_HasShape && !bone.m_SeparatesSkinObject;

    //-----------------------------------------------------------------------------
    // ノードタイプを決定します。
    bone.m_IsJoint = (!dstHasShape &&
        (bone.m_SmoothMtxIdx != -1 ||
         bone.m_RigidMtxIdx  != -1 ||
         bone.m_ScaleCompensate    ||
         bone.m_Side != "none"     ||
         !bone.m_Type.empty()));
    if (!dstHasShape && !bone.m_IsJoint && pParent != NULL)
    {
        // シェイプのないボーンの親が joint ノードなら matrix_index="-1 -1" でも joint ノードにします。
        bone.m_IsJoint = pParent->m_IsJoint;
    }

    //-----------------------------------------------------------------------------
    // インフルエンスツリー使用時はエクスポートプラグインが追加したルートボーンを省略します。
    if (lodInfo.m_UsesInfluenceTree &&
        bone.m_ParentIdx == -1 &&
        bone.m_Name == DEFAULT_ROOT_BONE_NAME &&
        isIdentMtx            &&
        !bone.m_IsJoint       &&
        !bone.m_HasShape      &&
        bone.m_CompressEnable &&
        !isInfluence          &&
        userDatasElem == NULL)
    {
        bone.m_XformPath = lodInfo.m_GroupPath;
        bone.m_IsSkipped = true;
        return status;
    }

    //-----------------------------------------------------------------------------
    // シェイプツリーの transform (joint) ノードを作成します。
    const int lodCount = lodInfo.GetLodCount();
    if (lodCount == 1 || IsShapeTreeBone(bones, iDstBone, lodInfo))
    {
        for (int iLod = 0; iLod < lodCount; ++iLod)
        {
            MDagPath parentXformPath;
            if (lodCount == 1)
            {
                if (pParent != NULL)
                {
                    parentXformPath = pParent->m_XformPath;
                }
            }
            else
            {
                if (pParent != NULL)
                {
                    parentXformPath = (iLod < static_cast<int>(pParent->m_LodXformPaths.length())) ?
                        pParent->m_LodXformPaths[iLod] : pParent->m_XformPath;
                }
                else
                {
                    parentXformPath = lodInfo.m_RootPaths[iLod];
                }
            }

            const MDagPath xformPath = CreateTransformNode(
                bone, pParent, parentXformPath, userDatasElem);
            if (iLod == 0)
            {
                bone.m_XformPath = xformPath;
            }
            bone.m_LodXformPaths.append(xformPath);
        }
    }

    //-----------------------------------------------------------------------------
    // インフルエンスツリーの transform (joint) ノードを作成します。
    if (lodCount >= 2 && lodInfo.m_InfluenceFlags[iDstBone])
    {
        MDagPath parentXformPath;
        if (pParent != NULL)
        {
            parentXformPath = pParent->m_XformPath;
        }
        bone.m_XformPath = CreateTransformNode(
            bone, pParent, parentXformPath, userDatasElem);
    }

    //-----------------------------------------------------------------------------
    // ワールドバインド行列を取得します。
    bone.m_BindGlobalMtx = bone.m_XformPath.inclusiveMatrix();
    bone.m_BindGlobalInvMtx = bone.m_BindGlobalMtx.inverse();

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 行列インデックスからボーンインデックスへの変換テーブルを作成します。
//!
//! @param[out] table 行列インデックスからボーンインデックスへの変換テーブルです。
//! @param[in] bones ボーン配列です。
//-----------------------------------------------------------------------------
static void CreateMtxIdxToBoneIdxTable(RIntArray& table, const ImpBoneArray& bones)
{
    // 行列パレットはスムーススキン用行列の後にリジッドスキン用行列が並ぶため、
    // 最大でボーン数の 2 倍の長さになります。
    table = RIntArray(bones.size() * 2, 0);
    for (size_t iBone = 0; iBone < bones.size(); ++iBone)
    {
        const ImpBone& bone = bones[iBone];
        if (bone.m_SmoothMtxIdx != -1)
        {
            table[bone.m_SmoothMtxIdx] = static_cast<int>(iBone);
        }
        if (bone.m_RigidMtxIdx != -1)
        {
            table[bone.m_RigidMtxIdx] = static_cast<int>(iBone);
        }
    }
}

//=============================================================================
//! @brief インポート用アニメーションキーのクラスです。
//=============================================================================
class ImpAnimKey
{
public:
    float m_Frame; //!< フレームです。
    float m_Value; //!< 値です。
    float m_InSlope; //!< 左側のスロープ値です。
    float m_OtSlope; //!< 右側のスロープ値です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] frame フレームです。
    //! @param[in] value 値です。
    //! @param[in] inSlope 左側のスロープ値です。
    //! @param[in] otSlope 右側のスロープ値です。
    //!
    ImpAnimKey(
        const float frame = 0.0f,
        const float value = 0.0f,
        const float inSlope = 0.0f,
        const float otSlope = 0.0f)
    : m_Frame(frame),
      m_Value(value),
      m_InSlope(inSlope),
      m_OtSlope(otSlope)
    {
    }
};

//! @brief インポート用アニメーションキー配列の定義です。
typedef std::vector<ImpAnimKey> ImpAnimKeyArray;

//-----------------------------------------------------------------------------
//! @brief 秒間フレーム数に対応した時間の作業単位を取得します。
//!
//! @param[in] fps 秒間フレーム数です。
//!
//! @return 時間の作業単位を返します。
//-----------------------------------------------------------------------------
static MTime::Unit GetTimeUnitFromFps(const float fps)
{
    return
        (fps == 15.0f) ? MTime::kGames     :
        (fps == 24.0f) ? MTime::kFilm      :
        (fps == 25.0f) ? MTime::kPALFrame  :
        (fps == 30.0f) ? MTime::kNTSCFrame :
        (fps == 48.0f) ? MTime::kShowScan  :
        (fps == 50.0f) ? MTime::kPALField  :
        (fps == 60.0f) ? MTime::kNTSCField :

        (RIsSame(fps, 1.0f / 3600.0f)) ? MTime::kHours        :
        (RIsSame(fps, 1.0f /   60.0f)) ? MTime::kMinutes      :
        (fps == 1.0f                 ) ? MTime::kSeconds      :
        (fps == 1000.0f              ) ? MTime::kMilliseconds :

        (fps ==    2.0f) ? MTime::k2FPS    :
        (fps ==    3.0f) ? MTime::k3FPS    :
        (fps ==    4.0f) ? MTime::k4FPS    :
        (fps ==    5.0f) ? MTime::k5FPS    :
        (fps ==    6.0f) ? MTime::k6FPS    :
        (fps ==    8.0f) ? MTime::k8FPS    :
        (fps ==   10.0f) ? MTime::k10FPS   :
        (fps ==   12.0f) ? MTime::k12FPS   :
        (fps ==   16.0f) ? MTime::k16FPS   :
        (fps ==   20.0f) ? MTime::k20FPS   :
        (fps ==   40.0f) ? MTime::k40FPS   :
        (fps ==   75.0f) ? MTime::k75FPS   :
        (fps ==   80.0f) ? MTime::k80FPS   :
        (fps ==  100.0f) ? MTime::k100FPS  :
        (fps ==  120.0f) ? MTime::k120FPS  :
        (fps ==  125.0f) ? MTime::k125FPS  :
        (fps ==  150.0f) ? MTime::k150FPS  :
        (fps ==  200.0f) ? MTime::k200FPS  :
        (fps ==  240.0f) ? MTime::k240FPS  :
        (fps ==  250.0f) ? MTime::k250FPS  :
        (fps ==  300.0f) ? MTime::k300FPS  :
        (fps ==  375.0f) ? MTime::k375FPS  :
        (fps ==  400.0f) ? MTime::k400FPS  :
        (fps ==  500.0f) ? MTime::k500FPS  :
        (fps ==  600.0f) ? MTime::k600FPS  :
        (fps ==  750.0f) ? MTime::k750FPS  :
        (fps == 1200.0f) ? MTime::k1200FPS :
        (fps == 1500.0f) ? MTime::k1500FPS :
        (fps == 2000.0f) ? MTime::k2000FPS :
        (fps == 3000.0f) ? MTime::k3000FPS :
        (fps == 6000.0f) ? MTime::k6000FPS :

        #if (MAYA_API_VERSION >= 201700)
        (RIsSame(fps, 23.976f, 0.01f)) ? MTime::k23_976FPS :
        (RIsSame(fps, 29.97f , 0.01f)) ? MTime::k29_97FPS  : // = k29_97DF
        (RIsSame(fps, 47.95f , 0.01f)) ? MTime::k47_952FPS :
        (RIsSame(fps, 59.94f , 0.01f)) ? MTime::k59_94FPS  :
        (fps == 44100.0f             ) ? MTime::k44100FPS  :
        (fps == 48000.0f             ) ? MTime::k48000FPS  :
        #endif

        MTime::uiUnit(); // いずれにも該当しなければ現在の単位を返します。
}

//-----------------------------------------------------------------------------
//! @brief 時間の作業単位と開始／終了フレームを設定します。
//!
//! @param[in] infoElem アニメーション情報要素です。
//!
//! @return 開始フレームを返します。
//-----------------------------------------------------------------------------
static int SetAnimFrameRange(const RXMLElement* infoElem)
{
    //-----------------------------------------------------------------------------
    // 時間の作業単位を設定します。
    const float dccFps = static_cast<float>(atof(infoElem->GetAttribute("dcc_fps").c_str()));
    MTime::setUIUnit(GetTimeUnitFromFps(dccFps));

    //-----------------------------------------------------------------------------
    // 開始フレームと終了フレームを設定します。
    const int startFrame = atoi(infoElem->GetAttribute("dcc_start_frame", "0" , false).c_str());
    int endFrame         = atoi(infoElem->GetAttribute("dcc_end_frame"  , "1" , false).c_str());
    const int frameCount = atoi(infoElem->GetAttribute("frame_count"    , "-1", false).c_str());
        // ↑ fsn ファイルの <scene_anim_info> には存在しません。
    if (frameCount >= 0)
    {
        endFrame = startFrame + frameCount;
    }
    if (endFrame == startFrame)
    {
        endFrame = startFrame + 1;
    }
    MGlobal::executeCommand(MString("playbackOptions -animationEndTime ") + endFrame);
    MGlobal::executeCommand(MString("playbackOptions -maxTime ") + endFrame);
    MGlobal::executeCommand(MString("playbackOptions -animationStartTime ") + startFrame);
    MGlobal::executeCommand(MString("playbackOptions -minTime ") + startFrame);
        // 開始フレームと終了フレームが同じ場合、
        // -minTime を先に設定すると -maxTime 設定時に開始フレームが 1 減るので、
        // -maxTime を先に設定します。
    return startFrame;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションのフレーム範囲を取得します。
//!
//! @param[out] startFrame 開始フレームを格納します。
//!
//! @return フレーム数を返します。
//-----------------------------------------------------------------------------
static int GetAnimFrameRange(int& startFrame)
{
    double startFrameD;
    double endFrameD;
    MGlobal::executeCommand("playbackOptions -q -animationStartTime", startFrameD);
    MGlobal::executeCommand("playbackOptions -q -animationEndTime", endFrameD);
    startFrame = static_cast<int>(startFrameD);
    int endFrame = static_cast<int>(endFrameD);
    if (endFrame < startFrame)
    {
        RSwapValue(startFrame, endFrame);
    }
    return endFrame - startFrame + 1;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブのインフィニティタイプを返します。
//-----------------------------------------------------------------------------
static MFnAnimCurve::InfinityType GetAnimCurveInfinityType(const std::string& wrap)
{
    if (wrap == "repeat"         ) return MFnAnimCurve::kCycle;
    if (wrap == "mirror"         ) return MFnAnimCurve::kOscillate;
    if (wrap == "relative_repeat") return MFnAnimCurve::kCycleRelative;
    return MFnAnimCurve::kConstant; // "clamp"
}

//-----------------------------------------------------------------------------
//! @brief 一定アニメーションの値を設定します。
//!
//! @param[in,out] dstPlug 対象プラグです。
//! @param[in] frame 開始フレームです。
//! @param[in] value 値です。
//! @param[in] setsConstantKey アニメーション値が一定でもキーを打つなら true です。
//-----------------------------------------------------------------------------
static MStatus SetConstantAnimValue(
    MPlug& dstPlug,
    const int frame,
    const double value,
    const bool setsConstantKey
)
{
    MStatus status;
    dstPlug.setValue(value);
    if (setsConstantKey)
    {
        MFnAnimCurve curveFn;
        curveFn.create(dstPlug, NULL, &status);
        CheckStatusMsg(status, "Cannot create anim curve: " + dstPlug.name() + ": " + status.errorString());
        curveFn.setIsWeighted(false);
        status = curveFn.addKeyframe(MTime(frame, MTime::uiUnit()), value);
        CheckStatusMsg(status, "Cannot add key frame: " + dstPlug.name() + ": " + status.errorString());
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブにキーを 1 つ追加します。
//!
//! @param[in,out] curveFn アニメーションカーブのファンクションノードです。
//! @param[in] time 時間です。既存のキーより後の時間である必要があります。
//! @param[in] value 値です。
//! @param[in] inSlope 入力スロープです。
//! @param[in] otSlope 出力スロープです。
//! @param[in] tanType 接線タイプです。
//-----------------------------------------------------------------------------
static void AddOneAnimKey(
    MFnAnimCurve& curveFn,
    const MTime& time,
    const double value,
    const float inSlope,
    const float otSlope,
    const MFnAnimCurve::TangentType tanType
)
{
    const int iKey = curveFn.numKeys();
    curveFn.addKey(time, value, tanType, tanType);
    float inY = inSlope;
    float otY = otSlope;
    const float inLen = sqrtf(1.0f + inY * inY);
    const float otLen = sqrtf(1.0f + otY * otY);
    const float inX = 1.0f / inLen;
    inY /= inLen;
    const float otX = 1.0f / otLen;
    otY /= otLen;
    if (RIsSame(inSlope, otSlope))
    {
        // デフォルトではイン接線とアウト接線が連動しているので、
        // インとアウトが同じなら、一方のみ設定で構いません。
        curveFn.setTangent(iKey, otX, otY, false, NULL, false);
    }
    else
    {
        curveFn.setTangentsLocked(iKey, false);
        curveFn.setTangent(iKey, inX, inY, true , NULL, false);
        curveFn.setTangent(iKey, otX, otY, false, NULL, false);
    }
}

//-----------------------------------------------------------------------------
//! @brief ターゲット要素からアニメーションカーブを作成します。
//!
//! @param[in,out] dstPlug 対象プラグです。
//! @param[in] targetElem ターゲット要素です。
//! @param[in] isOriginal オリジナルカーブ要素を解析するなら true です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] valueScale 値に乗算するスケールです。
//! @param[in] valueOfs 値に加算するオフセットです。
//! @param[in] setsConstantKey アニメーション値が一定でもキーを打つなら true です。
//! @param[in] intermediateFile 中間ファイルです。
//-----------------------------------------------------------------------------
static MStatus CreateAnimCurveFromElement(
    MPlug& dstPlug,
    const RXMLElement* targetElem,
    const bool isOriginal,
    const int startFrame,
    const double valueScale,
    const double valueOfs,
    const bool setsConstantKey,
    const ImpIntermediateFile& intermediateFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // アトリビュートを keyable にします。
    if (!dstPlug.isKeyable())
    {
        dstPlug.setKeyable(true);
    }

    //-----------------------------------------------------------------------------
    if (targetElem->nodes.empty())
    {
        //-----------------------------------------------------------------------------
        // カーブ要素がなければ固定値を設定します。
        const double value = atof(targetElem->GetAttribute("base_value").c_str()) * valueScale + valueOfs;
        status = SetConstantAnimValue(dstPlug, startFrame, value, setsConstantKey);
    }
    else
    {
        //-----------------------------------------------------------------------------
        // カーブ要素を取得して接線タイプを決定します。
        const char* hermiteName = (isOriginal) ? "original_hermite" : "hermite_curve";
        const char* linearName  = (isOriginal) ? "original_linear"  : "linear_curve" ;
        const char* stepName    = (isOriginal) ? "original_step"    : "step_curve"   ;

        MFnAnimCurve::TangentType tanType = MFnAnimCurve::kTangentFixed;
        const RXMLElement* curveElem = targetElem->FindElement(hermiteName, false);
        if (curveElem == NULL)
        {
            tanType = MFnAnimCurve::kTangentLinear;
            curveElem = targetElem->FindElement(linearName, false);
        }
        if (curveElem == NULL)
        {
            tanType = MFnAnimCurve::kTangentStep;
            curveElem = targetElem->FindElement(stepName, false);
        }
        if (curveElem == NULL)
        {
            return status;
        }

        //-----------------------------------------------------------------------------
        // キーのデータを取得します。
        const int keyCount = atoi(curveElem->GetAttribute("count").c_str());
        const int streamIdx = atoi(curveElem->GetAttribute("stream_index").c_str());
        RFloatArray keyValues;
        intermediateFile.GetFloatStream(keyValues, streamIdx);
        const int valuesPerKey = (tanType == MFnAnimCurve::kTangentFixed) ? 4 : 2;
        if (static_cast<int>(keyValues.size()) != valuesPerKey * keyCount)
        {
            DisplayImportWarning("アニメーションカーブデータが不正です: {0}", // 通常は発生しない
                "Animation curve data is wrong: {0}", dstPlug.name().asChar());
            return status;
        }

        //-----------------------------------------------------------------------------
        // アニメーションカーブを作成します。
        MFnAnimCurve curveFn;
        curveFn.create(dstPlug, NULL, &status);
        //curveFn.create(dstPlug, MFnAnimCurve::kAnimCurveTL, NULL, &status);
        CheckStatusMsg(status, "Cannot create anim curve: " + dstPlug.name() + ": " + status.errorString());
        curveFn.setIsWeighted(false);

        const double startFrameD = startFrame;
        const MTime::Unit timeUnit = MTime::uiUnit();
        const double framesPerSecond = MTime(1.0, MTime::kSeconds).as(timeUnit);
        const float slopeScale = static_cast<float>(valueScale * framesPerSecond);

        int iValue = 0;
        if (tanType == MFnAnimCurve::kTangentFixed)
        {
            for (int iKey = 0; iKey < keyCount; ++iKey, iValue += valuesPerKey)
            {
                AddOneAnimKey(curveFn,
                    MTime(startFrameD + keyValues[iValue], timeUnit),
                    keyValues[iValue + 1] * valueScale + valueOfs,
                    keyValues[iValue + 2] * slopeScale,
                    keyValues[iValue + 3] * slopeScale,
                    tanType);
            }
        }
        else // Linear or Step
        {
            MTimeArray times;
            MDoubleArray values;
            for (int iKey = 0; iKey < keyCount; ++iKey, iValue += valuesPerKey)
            {
                times.append(MTime(startFrameD + keyValues[iValue], timeUnit));
                values.append(keyValues[iValue + 1] * valueScale + valueOfs);
            }
            const MFnAnimCurve::TangentType inTanTyepForAdd =
                (tanType == MFnAnimCurve::kTangentStep) ?
                MFnAnimCurve::kTangentClamped : tanType;
            curveFn.addKeys(&times, &values, inTanTyepForAdd, tanType);
        }

        //-----------------------------------------------------------------------------
        // アニメーションカーブのインフィニティを設定します。
        curveFn.setPreInfinityType(GetAnimCurveInfinityType(curveElem->GetAttribute("pre_wrap")));
        curveFn.setPostInfinityType(GetAnimCurveInfinityType(curveElem->GetAttribute("post_wrap")));
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ベイクしたデータをすべてアニメーションキーとして設定します。
//!
//! @param[out] keys アニメーションキー配列です。
//! @param[in] fullValues アニメーション値配列です。
//-----------------------------------------------------------------------------
static void SetAnimKeysFromFullValues(
    ImpAnimKeyArray& keys,
    const RFloatArray& fullValues
)
{
    //-----------------------------------------------------------------------------
    // 全サブフレームについてキーを追加します。
    keys.clear();

    const int subFrameCount = static_cast<int>(fullValues.size());
    for (int iFrame = 0; iFrame < subFrameCount; ++iFrame)
    {
        keys.push_back(ImpAnimKey(static_cast<float>(iFrame), fullValues[iFrame]));
    }

    //-----------------------------------------------------------------------------
    // 前後のキーの値の差分からスロープ値に設定します。
    const int keyCount = static_cast<int>(keys.size());
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        ImpAnimKey& key = keys[iKey];
        if (keyCount == 1)
        {
            key.m_InSlope = 0.0f;
        }
        else if (iKey == 0)
        {
            key.m_InSlope = keys[iKey + 1].m_Value - key.m_Value;
            const float scale = keys[iKey + 1].m_Frame - key.m_Frame;
            if (scale != 0.0f)
            {
                key.m_InSlope /= scale;
            }
        }
        else if (iKey == keyCount - 1)
        {
            key.m_InSlope = key.m_Value - keys[iKey - 1].m_Value;
            const float scale = key.m_Frame - keys[iKey - 1].m_Frame;
            if (scale != 0.0f)
            {
                key.m_InSlope /= scale;
            }
        }
        else
        {
            key.m_InSlope = (keys[iKey + 1].m_Value - keys[iKey - 1].m_Value);
            const float scale = keys[iKey + 1].m_Frame - keys[iKey - 1].m_Frame;
            if (scale != 0.0f)
            {
                key.m_InSlope /= scale;
            }
        }
        key.m_OtSlope = key.m_InSlope;
    }
}

//-----------------------------------------------------------------------------
//! @brief 2 つのキーをエルミート補間した値を取得します。
//!
//! @param[in] key0 評価するフレームの直前のキーです。
//! @param[in] key1 評価するフレームの直後のキーです。
//! @param[in] frame 評価するフレームです。
//!
//! @return エルミート補間した値を返します。
//-----------------------------------------------------------------------------
static float GetHermiteInterp(
    const ImpAnimKey& key0,
    const ImpAnimKey& key1,
    const float frame
)
{
    if (key0.m_Frame == key1.m_Frame)
    {
        return key1.m_Value;
    }
    else
    {
        const float t1 = frame - key0.m_Frame;
        const float t2 = 1.0f / (key1.m_Frame - key0.m_Frame);
        const float v0 = key0.m_Value;
        const float v1 = key1.m_Value;
        const float s0 = key0.m_OtSlope;
        const float s1 = key1.m_InSlope;

        const float t1t1t2 = t1 * t1 * t2;
        const float t1t1t2t2 = t1t1t2 * t2;
        const float t1t1t1t2t2 = t1 * t1t1t2t2;
        const float t1t1t1t2t2t2 = t1t1t1t2t2 * t2;

        return v0 * (2.0f * t1t1t1t2t2t2 - 3.0f * t1t1t2t2 + 1.0f)
                + v1 * (-2.0f * t1t1t1t2t2t2 + 3.0f * t1t1t2t2)
                + s0 * (t1t1t1t2t2 - 2.0f * t1t1t2 + t1)
                + s1 * (t1t1t1t2t2 - t1t1t2);
    }
}

//-----------------------------------------------------------------------------
//! @brief キーが 2 つのアニメーションキー配列を最適化します。
//!
//! @param[in,out] keys アニメーションキー配列です。
//-----------------------------------------------------------------------------
static void OptimizeTwoAnimKeys(ImpAnimKeyArray& keys)
{
    if (keys.size() == 2)
    {
        if (RIsSame(keys[0].m_Value, keys[1].m_Value) &&
            //keys[0].m_Type == ImpAnimKey::HERMITE       &&
            //keys[1].m_Type == ImpAnimKey::HERMITE       &&
            RIsSame(keys[0].m_OtSlope, 0.0f)          &&
            RIsSame(keys[1].m_InSlope, 0.0f))
        {
            keys.resize(1); // 最初のキーだけ残します。
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションキー配列を最適化します。
//!
//! @param[in,out] keys アニメーションキー配列です。
//!                     全サブフレームの数だけキーが設定されている必要があります。
//! @param[in] fullValues ベイクしたデータ（全サブフレームにおける値の配列）です。
//! @param[in] tolerance 誤差の許容値です。
//-----------------------------------------------------------------------------
#define OPTIMIZE_KEY_2PASS_SW // 高速化のため 2 パスでキーを最適化するなら定義します。
const int UNIT_FRAME_RANGE = 1024; // 最適化する単位のフレーム範囲です。2 のべき乗にする必要があります。

static void OptimizeAnimKeys(
    ImpAnimKeyArray& keys,
    const RFloatArray& fullValues,
    float tolerance
)
{
    //-----------------------------------------------------------------------------
    // キー数をチェックします。2 つ以下なら専用の最適化をします。
    const int keyCount = static_cast<int>(keys.size());
    if (keyCount <= 2)
    {
        OptimizeTwoAnimKeys(keys);
        return;
    }

    //-----------------------------------------------------------------------------
    // 元のサブフレームごとのキー使用フラグ配列を 1 で初期化します。
    const int orgKeyCount = static_cast<int>(keys.size()); // 全サブフレームの数と同じである必要があります。
    RIntArray useFlags(orgKeyCount, 1);

    //-----------------------------------------------------------------------------
    // 最初のパスでは、最適化する単位のフレーム範囲ごとに不要なキーの使用フラグを 0 にします。
    int iKeyPrev = 0; // 使用する前のキーのインデックス（= サブフレームインデックス）です。
    for (int iKey = 1; iKey < orgKeyCount - 1; ++iKey)
    {
        #ifdef OPTIMIZE_KEY_2PASS_SW
        if ((iKey & (UNIT_FRAME_RANGE - 1)) == 0)
        {
            // 全フレーム範囲を一度に処理すると、
            // 不要なキーが多い場合に評価するフレーム範囲がどんどん広がるので、
            // 最適化する単位の境界部分のキーをかならず残して
            // 評価するフレーム範囲を制限します。
            iKeyPrev = iKey;
            continue;
        }
        #endif
        const int iKeyNext = iKey + 1;
        const ImpAnimKey& key0 = keys[iKeyPrev];
        const ImpAnimKey& key1 = keys[iKeyNext];
        float errMax = 0.0f;
        for (int iFrame = iKeyPrev + 1; iFrame < iKeyNext; ++iFrame)
        {
            const float floatFrame = static_cast<float>(iFrame);
            const float value = GetHermiteInterp(key0, key1, floatFrame);
            const float err = RAbs(fullValues[iFrame] - value);
            if (err > errMax)
            {
                errMax = err;
                if (errMax >= tolerance)
                {
                    break;
                }
            }
        }
        if (errMax == 0.0f || errMax < tolerance)
        {
            useFlags[iKey] = 0; // 誤差が 0 か許容値より小さければ使用フラグを 0 にします。
        }
        else
        {
            iKeyPrev = iKey;
        }
    }

    //-----------------------------------------------------------------------------
    // 2 番目のパスでは、最適化する単位の境界部分のキーが不要なら使用フラグを 0 にします。
    #ifdef OPTIMIZE_KEY_2PASS_SW
    for (int iKey = UNIT_FRAME_RANGE; iKey < orgKeyCount - 1; iKey += UNIT_FRAME_RANGE)
    {
        iKeyPrev = iKey - 1;
        while (iKeyPrev > 0 && !useFlags[iKeyPrev]) // 使用する前のキーを検索します。
        {
            --iKeyPrev;
        }
        int iKeyNext = iKey + 1;
        while (iKeyNext < orgKeyCount && !useFlags[iKeyNext]) // 使用する次のキーを検索します。
        {
            ++iKeyNext;
        }
        //cerr << "iKey: " << iKey << ": " << iKeyPrev << ", " << iKeyNext << endl;
        const ImpAnimKey& key0 = keys[iKeyPrev];
        const ImpAnimKey& key1 = keys[iKeyNext];
        float errMax = 0.0f;
        for (int iFrame = iKeyPrev + 1; iFrame < iKeyNext; ++iFrame)
        {
            const float floatFrame = static_cast<float>(iFrame);
            const float value = GetHermiteInterp(key0, key1, floatFrame);
            const float err = RAbs(fullValues[iFrame] - value);
            if (err > errMax)
            {
                errMax = err;
                if (errMax >= tolerance)
                {
                    break;
                }
            }
        }
        if (errMax == 0.0f || errMax < tolerance)
        {
            useFlags[iKey] = 0; // 誤差が 0 か許容値より小さければ使用フラグを 0 にします。
        }
    }
    #endif

    //-----------------------------------------------------------------------------
    // キー使用フラグが 0 でないキーから新しいキー配列を作成します。
    ImpAnimKeyArray newKeys;
    for (int iKey = 0; iKey < orgKeyCount; ++iKey)
    {
        if (useFlags[iKey])
        {
            newKeys.push_back(keys[iKey]);
        }
    }
    keys = newKeys;

    //-----------------------------------------------------------------------------
    // 残ったキー数が 2 つなら専用の最適化をします。
    OptimizeTwoAnimKeys(keys);
}

//-----------------------------------------------------------------------------
//! @brief アニメーション値配列からアニメーションカーブを作成します。
//!
//! @param[in,out] dstPlug 対象プラグです。
//! @param[in] values アニメーション値配列です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] valueScale 値に乗算するスケールです。
//! @param[in] tolerance 許容誤差です。
//! @param[in] setsConstantKey アニメーション値が一定でもキーを打つなら true です。
//-----------------------------------------------------------------------------
static MStatus CreateAnimCurveFromValues(
    MPlug& dstPlug,
    const RFloatArray& values,
    const int startFrame,
    const double valueScale,
    const double tolerance,
    const bool setsConstantKey
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // アトリビュートを keyable にします。
    if (!dstPlug.isKeyable())
    {
        dstPlug.setKeyable(true);
    }

    //-----------------------------------------------------------------------------
    // アニメーション値が一定なら値を設定します。
    const int frameCount = static_cast<int>(values.size());
    if (frameCount == 0)
    {
        return status;
    }
    const float toleranceF = static_cast<float>(tolerance);
    bool isConstant = true;
    const float firstValue = values[0];
    for (int iFrame = 1; iFrame < frameCount; ++iFrame)
    {
        const float value = values[iFrame];
        if (value != firstValue && RAbs(value - firstValue) >= toleranceF)
        {
            isConstant = false;
            break;
        }
    }

    if (isConstant)
    {
        const double value = firstValue * valueScale;
        return SetConstantAnimValue(dstPlug, startFrame, value, setsConstantKey);
    }

    //-----------------------------------------------------------------------------
    // ベイクしたデータをすべてアニメーションキーとして設定します。
    ImpAnimKeyArray keys;
    SetAnimKeysFromFullValues(keys, values);

    //-----------------------------------------------------------------------------
    // 不要なキーを削除します。
    OptimizeAnimKeys(keys, values, toleranceF);

    //-----------------------------------------------------------------------------
    // アニメーションカーブを作成します。
    MFnAnimCurve curveFn;
    curveFn.create(dstPlug, NULL, &status);
    CheckStatusMsg(status, "Cannot create anim curve: " + dstPlug.name() + ": " + status.errorString());
    curveFn.setIsWeighted(false);

    const double startFrameD = startFrame;
    const MTime::Unit timeUnit = MTime::uiUnit();
    const double framesPerSecond = MTime(1.0, MTime::kSeconds).as(timeUnit);
    const float slopeScale = static_cast<float>(valueScale * framesPerSecond);
    const MFnAnimCurve::TangentType tanType = MFnAnimCurve::kTangentFixed;

    const int keyCount = static_cast<int>(keys.size());
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        const ImpAnimKey& key = keys[iKey];
        AddOneAnimKey(curveFn,
            MTime(startFrameD + key.m_Frame, timeUnit),
            key.m_Value * valueScale,
            key.m_InSlope * slopeScale,
            key.m_OtSlope * slopeScale,
            tanType);
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 指定フレーム範囲でのプラグのアニメーション値を取得します。
//!
//! @param[out] values アニメーション値配列です。
//! @param[in] plug プラグです。
//! @param[in] startFrame 開始フレームです。
//! @param[in] frameCount フレーム数です。
//! @param[in] valueScale 取得した値に乗算するスケールです。
//! @param[in] valueOfs 取得した値に加算するオフセットです。
//-----------------------------------------------------------------------------
static void GetFullAnimValues(
    RFloatArray& values,
    const MPlug& plug,
    const int startFrame,
    const int frameCount,
    const float valueScale,
    const float valueOfs
)
{
    values.resize(frameCount);
    MPlugArray plugArray;
    plug.connectedTo(plugArray, true, false);
    if (plugArray.length() != 0)
    {
        MObject inObj = plugArray[0].node();
        const MFn::Type inType = inObj.apiType();
        if (inType == MFn::kAnimCurveTimeToAngular  ||
            inType == MFn::kAnimCurveTimeToDistance ||
            inType == MFn::kAnimCurveTimeToUnitless)
        {
            MFnAnimCurve curveFn(inObj);
            for (int iFrame = 0; iFrame < frameCount; ++iFrame)
            {
                MTime tm(startFrame + iFrame, MTime::uiUnit());
                values[iFrame] = static_cast<float>(curveFn.evaluate(tm)) * valueScale + valueOfs;
            }
        }
        else
        {
            for (int iFrame = 0; iFrame < frameCount; ++iFrame)
            {
                MTime tm(startFrame + iFrame, MTime::uiUnit());
                double dv;
                MDGContext ctx(tm);
                #if (MAYA_API_VERSION >= 20180000)
                MDGContextGuard contextGuard(ctx);
                plug.getValue(dv);
                #else
                plug.getValue(dv, ctx);
                #endif
                values[iFrame] = static_cast<float>(dv) * valueScale + valueOfs;
            }
        }
    }
    else
    {
        double dv;
        plug.getValue(dv);
        const float value = static_cast<float>(dv) * valueScale + valueOfs;
        for (int iFrame = 0; iFrame < frameCount; ++iFrame)
        {
            values[iFrame] = value;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブの値と接線変換用関数の定義です。
//-----------------------------------------------------------------------------
typedef double (*ConvertAnimCurveFuncPtr)(const double srcVal, const void* pUser);

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブの値と接線を変換します。
//!
//! @param[in] curveObj アニメーションカーブオブジェクトです。
//! @param[in] isHermite エルミート補間なら true です。
//! @param[in] convertFunc アニメーションカーブの値と接線変換用関数です。
//! @param[in] pUser アニメーションカーブの値と接線変換用関数に渡すユーザーデータのポインタです。
//-----------------------------------------------------------------------------
static void ConvertAnimCurveValueTangent(
    MObject& curveObj,
    const bool isHermite,
    ConvertAnimCurveFuncPtr convertFunc,
    const void* pUser
)
{
    const double framesPerSecond = MTime(1.0, MTime::kSeconds).as(MTime::uiUnit());
    MFnAnimCurve curveFn(curveObj);
    const int keyCount = curveFn.numKeys();
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        const double srcVal = curveFn.value(iKey);
        const double dstVal = convertFunc(srcVal, pUser);
        curveFn.setValue(iKey, dstVal);
        if (isHermite)
        {
            float inX;
            float inY;
            float otX;
            float otY;
            #if (MAYA_API_VERSION >= 20180000)
            MFnAnimCurve::TangentValue tanInX;
            MFnAnimCurve::TangentValue tanInY;
            MFnAnimCurve::TangentValue tanOtX;
            MFnAnimCurve::TangentValue tanOtY;
            curveFn.getTangent(iKey, tanInX, tanInY, true);
            curveFn.getTangent(iKey, tanOtX, tanOtY, false);
            inX = static_cast<float>(tanInX);
            inY = static_cast<float>(tanInY);
            otX = static_cast<float>(tanOtX);
            otY = static_cast<float>(tanOtY);
            #else
            curveFn.getTangent(iKey, inX, inY, true);
            curveFn.getTangent(iKey, otX, otY, false);
            #endif
            const float inSlope = (inX != 0.0f) ? inY / inX : 0.0f; // 中間ファイルの値 x fps
            const float otSlope = (otX != 0.0f) ? otY / otX : 0.0f; // 中間ファイルの値 x fps
            // 元の値にスロープを足した値を変換して、差分に戻します。
            const double inDst = convertFunc(srcVal - inSlope / framesPerSecond, pUser);
            const double otDst = convertFunc(srcVal + otSlope / framesPerSecond, pUser);
            inY = static_cast<float>((dstVal - inDst) * framesPerSecond);
            otY = static_cast<float>((otDst - dstVal) * framesPerSecond);
            const float inLen = sqrtf(1.0f + inY * inY);
            const float otLen = sqrtf(1.0f + otY * otY);
            inX = 1.0f / inLen;
            inY /= inLen;
            otX = 1.0f / otLen;
            otY /= otLen;
            if (curveFn.tangentsLocked(iKey))
            {
                curveFn.setTangent(iKey, otX, otY, false, NULL, false);
            }
            else
            {
                curveFn.setTangent(iKey, inX, inY, true , NULL, false);
                curveFn.setTangent(iKey, otX, otY, false, NULL, false);
            }
            //curveFn.setInTangentType (iKey, MFnAnimCurve::kTangentClamped); // 接線を変換せずにクランプにするテスト
            //curveFn.setOutTangentType(iKey, MFnAnimCurve::kTangentClamped); // 接線を変換せずにクランプにするテスト
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションを削除する MEL コマンドを文字列に追加します。
//!
//! @param[in,out] cmd MEL コマンド文字列です。
//! @param[in] plug プラグです。
//-----------------------------------------------------------------------------
static void AddAnimCurveClearCmd(MString& cmd, const MPlug& plug)
{
    //cerr << plug.name() << endl;
    MPlugArray plugArray;
    plug.connectedTo(plugArray, true, false);
    if (plugArray.length() != 0)
    {
        MObject inObj = plugArray[0].node();
        MFnDependencyNode inDepFn(inObj);
        if (inObj.hasFn(MFn::kAnimCurve)    &&
            !inDepFn.isLocked()             &&
            !inDepFn.isFromReferencedFile())
        {
            cmd += "delete " + inDepFn.name() + ";\n";
        }
        else
        {
            cmd += "disconnectAttr " + plugArray[0].name() + " " + plug.name() + ";\n";
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定したプラグのアニメーションを削除します。
//!
//! @param[in] plug プラグです。
//-----------------------------------------------------------------------------
static void ClearAnimCurve(const MPlug& plug)
{
    MString cmd;
    AddAnimCurveClearCmd(cmd, plug);
    if (cmd != "")
    {
        //cerr << "delete curve:" << endl << cmd << endl;
        MGlobal::executeCommand(cmd);
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した名前のボーンをシーンから取得して、SRT アニメーションを削除します。
//!
//! @param[in,out] bones ボーン配列です。
//! @param[in] boneName ボーン名です。
//! @param[in] selXforms 選択された transform ノード名の配列です。
//! @param[in] clearsAnim SRT アニメーションを削除するなら true です。
//-----------------------------------------------------------------------------
static MStatus GetSceneBoneByNameAndClearAnim(
    ImpBoneArray& bones,
    const std::string& boneName,
    const MStringArray& selXforms,
    const bool clearsAnim
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // ボーン名に対応した transform (joint) ノードを取得します。
    MDagPath xformPath;
    if (selXforms.length() != 0)
    {
        //-----------------------------------------------------------------------------
        // 選択されているノード以下を優先的に検索します。
        int iXform = YFindValueInArray(selXforms, boneName.c_str());
        if (iXform == -1)
        {
            for (int iXf = 0; iXf < static_cast<int>(selXforms.length()); ++iXf)
            {
                const std::string name = selXforms[iXf].asChar();
                const size_t cutLen = name.size() - boneName.size();
                if (cutLen > 0)
                {
                    if (name.rfind("|" + boneName) == cutLen - 1 ||
                        name.rfind(":" + boneName) == cutLen - 1)
                    {
                        MGlobal::displayInfo(MString("matched bone: ") + name.c_str());
                        iXform = iXf;
                        break;
                    }
                }
            }
        }
        if (iXform != -1)
        {
            xformPath = GetDagPathByName(selXforms[iXform]);
        }
    }

    if (!xformPath.isValid())
    {
        //-----------------------------------------------------------------------------
        // シーン全体から検索します。
        MStringArray xforms;
        MGlobal::executeCommand(MString("ls -typ transform \"") + boneName.c_str() + "\"", xforms);
        if (xforms.length() == 0)
        {
            MGlobal::executeCommand(MString("ls -typ transform \"*:") + boneName.c_str() + "\"", xforms);
        }
        if (xforms.length() == 0)
        {
            return status;
        }
        if (xforms.length() >= 2)
        {
            DisplayImportWarning("同名の transform ノードが複数存在します: {0}",
                "More than one transform node is found: {0}", boneName);
            //return status;
        }
        xformPath = GetDagPathByName(xforms[0]);
    }

    if (!xformPath.isValid())
    {
        return status;
    }

    //MSelectionList slist;
    //MGlobal::getSelectionListByName(boneName.c_str(), slist);
    //if (slist.length() == 0)
    //{
    //  MGlobal::getSelectionListByName(("*:" + boneName).c_str(), slist);
    //}
    //if (slist.length() == 0)
    //{
    //  return status;
    //}
    //MDagPath xformPath;
    //if (!slist.getDagPath(0, xformPath) ||
    //  !xformPath.node().hasFn(MFn::kTransform))
    //{
    //  return status;
    //}

    //-----------------------------------------------------------------------------
    // ボーンを追加します。
    bones.push_back(ImpBone());
    ImpBone& bone = bones.back();

    bone.m_Name = boneName;
    bone.m_XformPath = xformPath;

    unsigned int shapeCount = 0;
    if (xformPath.numberOfShapesDirectlyBelow(shapeCount))
    {
        for (unsigned int iShape = 0; iShape < shapeCount; ++iShape)
        {
            MDagPath shapePath = xformPath;
            if (shapePath.extendToShapeDirectlyBelow(iShape))
            {
                if (shapePath.node().apiType() == MFn::kMesh &&
                    !MFnDagNode(shapePath).isIntermediateObject())
                {
                    bone.m_HasShape = true;
                    bone.m_ShapeXformPath = xformPath;
                    bone.m_ShapePath = shapePath;
                    MStringArray rets;
                    MGlobal::executeCommand("ls -typ skinCluster `listHistory " + shapePath.partialPathName() + "`", rets);
                    bone.m_HasSkinedShape = (rets.length() != 0 && rets[0] != "");
                    break;
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 既存のアニメーションカーブを削除または接続解除します。
    if (clearsAnim)
    {
        const int PARAM_COUNT = 9;
        static const char* const attribNames[PARAM_COUNT] =
        {
            "scaleX",
            "scaleY",
            "scaleZ",
            "rotateX",
            "rotateY",
            "rotateZ",
            "translateX",
            "translateY",
            "translateZ",
        };

        MString cmd;
        MFnTransform xformFn(xformPath);
        for (int iParam = 0; iParam < PARAM_COUNT; ++iParam)
        {
            const MPlug plug = xformFn.findPlug(attribNames[iParam]);
            AddAnimCurveClearCmd(cmd, plug);
        }

        if (cmd.length() != 0)
        {
            //cerr << "delete curves:" << endl << cmd << endl;
            MGlobal::executeCommand(cmd);
        }
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 指定した名前のマテリアルをシーンから取得して配列に追加します。
//!
//! @param[in,out] materials マテリアル配列です。
//! @param[in] matName マテリアル名です。
//-----------------------------------------------------------------------------
static MStatus GetSceneMaterialByName(ImpMaterialArray& materials, const std::string& matName)
{
    MStringArray mats;
    MGlobal::executeCommand(MString("ls -typ lambert \"") + matName.c_str() + "\"", mats);
    if (mats.length() == 0)
    {
        MGlobal::executeCommand(MString("ls -typ lambert \"*:") + matName.c_str() + "\"", mats);
    }
    if (mats.length() != 0)
    {
        MObject shaderObj = GetNodeObjectByName(mats[0]);
        if (!shaderObj.isNull() && shaderObj.hasFn(MFn::kLambert))
        {
            materials.push_back(ImpMaterial());
            ImpMaterial& material = materials.back();

            MFnLambertShader shaderFn(shaderObj);
            material.m_Name = matName; // shaderFn.name() は namespace を含むので不可
            material.m_HasSpecular = shaderObj.hasFn(MFn::kReflect);
            material.m_ShaderObj = shaderObj;
            MPlugArray plugArray;
            shaderFn.findPlug("outColor").connectedTo(plugArray, false, true);
            if (plugArray.length() != 0)
            {
                MObject sgObj = plugArray[0].node();
                if (sgObj.apiType() == MFn::kShadingEngine)
                {
                    material.m_SGObj = sgObj;
                }
            }
            //cerr << "get scene material: " << matName << ": " << shaderFn.name() << ", " << MFnDependencyNode(material.m_SGObj).name() << endl;
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーションの対象アトリビュート名を取得します。
//-----------------------------------------------------------------------------
static std::string GetBoneAnimAttribName(const std::string& target)
{
    if (target == "scale_x") return "scaleX";
    if (target == "scale_y") return "scaleY";
    if (target == "scale_z") return "scaleZ";
    if (target == "rotate_x") return "rotateX";
    if (target == "rotate_y") return "rotateY";
    if (target == "rotate_z") return "rotateZ";
    if (target == "translate_x") return "translateX";
    if (target == "translate_y") return "translateY";
    if (target == "translate_z") return "translateZ";
    return "";
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーションを解析してアニメーションカーブを作成します。
//!
//! @param[in] boneAnimElem ボーンアニメーション要素です。
//! @param[in] magnify 移動値に掛ける倍率です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] bones ボーン配列です。
//! @param[in] usesAnimBin バイナリ出力フラグを反映するなら true です。
//! @param[in] fskFile fsk ファイルです。
//-----------------------------------------------------------------------------
static MStatus ParseBoneAnim(
    const RXMLElement* boneAnimElem,
    const double magnify,
    const int startFrame,
    const ImpBoneArray& bones,
    const bool usesAnimBin,
    const ImpIntermediateFile& fskFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 対応するボーンを検索します。
    const std::string boneName = boneAnimElem->GetAttribute("bone_name");
    const int iBone = FindBoneByName(bones, boneName);
    if (iBone == -1)
    {
        DisplayImportWarning("ボーンが見つかりません: {0}", "The bone cannot be found: {0}", boneName);
        return status;
    }
    const ImpBone& bone = bones[iBone];
    if (bone.m_IsSkipped)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // バイナリ出力フラグを反映します。
    bool binarizesS = true;
    bool binarizesR = true;
    bool binarizesT = true;
    if (usesAnimBin)
    {
        binarizesS = (boneAnimElem->GetAttribute("binarize_scale"    ) == "true");
        binarizesR = (boneAnimElem->GetAttribute("binarize_rotate"   ) == "true");
        binarizesT = (boneAnimElem->GetAttribute("binarize_translate") == "true");
    }
    const MString xformName = bone.m_XformPath.partialPathName();
    if (!binarizesS)
    {
        MGlobal::executeCommand("nnSetAnimationBinarization_Set_BinarizeScale "     + xformName + " 0");
    }
    if (!binarizesR)
    {
        MGlobal::executeCommand("nnSetAnimationBinarization_Set_BinarizeRotate "    + xformName + " 0");
    }
    if (!binarizesT)
    {
        MGlobal::executeCommand("nnSetAnimationBinarization_Set_BinarizeTranslate " + xformName + " 0");
    }
    if (!binarizesS && !binarizesR && !binarizesT)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 各アトリビュートにアニメーションカーブを接続します。
    MFnTransform xformFn(bone.m_XformPath);
    for (size_t iCurve = 0; iCurve < boneAnimElem->nodes.size(); ++iCurve)
    {
        const RXMLElement* targetElem = &boneAnimElem->nodes[iCurve];
        if (targetElem->name == "user_data_array")
        {
            continue;
        }

        const std::string targetName = targetElem->GetAttribute("target");
        if (
            (!binarizesS && targetName.find("scale"    ) == 0) ||
            (!binarizesR && targetName.find("rotate"   ) == 0) ||
            (!binarizesT && targetName.find("translate") == 0))
        {
            continue;
        }
        const std::string attribName = GetBoneAnimAttribName(targetName);
        if (!attribName.empty())
        {
            MPlug dstPlug = xformFn.findPlug(attribName.c_str());
            const double valueScale =
                (targetName.find("translate") == 0) ? magnify :
                ((targetName.find("rotate") == 0) ? R_M_DEG_TO_RAD : 1.0);
            //const MFnAnimCurve::AnimCurveType curveType =
            //  (targetName.find("scale"  ) == 0) ? MFnAnimCurve::kAnimCurveTU :
            //  (targetName.find("roatate") == 0) ? MFnAnimCurve::kAnimCurveTA :
            //  MFnAnimCurve::kAnimCurveTL;
            status = CreateAnimCurveFromElement(dstPlug, targetElem, false, startFrame,
                valueScale, 0.0, false, fskFile);
            CheckStatus(status);

            //-----------------------------------------------------------------------------
            // 各 LOD モデルにもアニメーションカーブを接続します。
            for (int iLod = 0; iLod < static_cast<int>(bone.m_LodXformPaths.length()); ++iLod)
            {
                const MDagPath lodXformPath = bone.m_LodXformPaths[iLod];
                if (!(lodXformPath == bone.m_XformPath))
                {
                    MPlug lodDstPlug = MFnTransform(lodXformPath).findPlug(attribName.c_str());
                    status = CreateAnimCurveFromElement(lodDstPlug, targetElem, false, startFrame,
                        valueScale, 0.0, false, fskFile);
                    CheckStatus(status);
                }
            }
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fsk ファイルを解析してアニメーションカーブを作成します。
//!
//! @param[in,out] bones ボーン配列です。
//! @param[in] fskPath fsk ファイルのパスです。
//! @param[in] magnify 移動値に掛ける倍率です。
//! @param[in] getsBones シーンからボーン配列を取得するなら true です。
//! @param[in] usesAnimBin バイナリ出力フラグを反映するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFsk(
    ImpBoneArray& bones,
    const std::string& fskPath,
    const double magnify,
    const bool getsBones,
    const bool usesAnimBin
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fsk ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile fskFile(rstat, fskPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = fskFile.m_Xml.FindElement("nw4f_3dif/skeletal_anim");
    const RXMLElement* infoElem = animElem->FindElement("skeletal_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // シーンからボーン配列を取得します。
    const RXMLElement* boneAnimsElem = animElem->FindElement("bone_anim_array", false);
    if (getsBones && boneAnimsElem != NULL)
    {
        MStringArray selXforms;
        MGlobal::executeCommand("ls -sl -dag -typ transform", selXforms);

        bones.clear();
        for (size_t iMember = 0; iMember < boneAnimsElem->nodes.size(); ++iMember)
        {
            const RXMLElement* boneAnimElem = &boneAnimsElem->nodes[iMember];
            const std::string boneName = boneAnimElem->GetAttribute("bone_name");
            GetSceneBoneByNameAndClearAnim(bones, boneName, selXforms, true);
        }
        if (bones.empty())
        {
            DisplayImportWarning("対応するボーンが 1 つもありません。", "No bone is matched.");
            return status;
        }
    }

    //-----------------------------------------------------------------------------
    // アニメーションのバイナリ出力設定プラグインの MEL を評価します。
    MGlobal::executeCommand("if (!`exists nnSetAnimationBinarization_Set_BinarizeScale`) "
        "eval \"source NintendoSetAnimationBinarization\";");

    //-----------------------------------------------------------------------------
    // bone_anim_array 要素を解析してアニメーションカーブを作成します。
    if (boneAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < boneAnimsElem->nodes.size(); ++iMember)
        {
            status = ParseBoneAnim(&boneAnimsElem->nodes[iMember],
                magnify, startFrame, bones, usesAnimBin, fskFile);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + fskPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ボーンビジビリティアニメーションを解析してアニメーションカーブを作成します。
//!
//! @param[in] boneVisAnimElem ボーンビジビリティアニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] bones ボーン配列です。
//! @param[in] usesAnimBin バイナリ出力フラグを反映するなら true です。
//! @param[in] fvbFile fvb ファイルです。
//-----------------------------------------------------------------------------
static MStatus ParseBoneVisAnim(
    const RXMLElement* boneVisAnimElem,
    const int startFrame,
    const ImpBoneArray& bones,
    const bool usesAnimBin,
    const ImpIntermediateFile& fvbFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 対応するボーンを検索します。
    const std::string boneName = boneVisAnimElem->GetAttribute("bone_name");
    const int iBone = FindBoneByName(bones, boneName);
    if (iBone == -1)
    {
        DisplayImportWarning("ボーンが見つかりません: {0}", "The bone cannot be found: {0}", boneName);
        return status;
    }
    const ImpBone& bone = bones[iBone];
    if (bone.m_IsSkipped || !bone.m_HasShape)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // バイナリ出力フラグを取得します。
    bool binarizesV = true;
    if (usesAnimBin)
    {
        binarizesV = (boneVisAnimElem->GetAttribute("binarize_visibility") == "true");
    }
    if (!binarizesV)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // シェイプの visibility アトリビュートに接続された既存のアニメーションカーブを削除または接続解除します。
    MFnDagNode dagFn(bone.m_ShapePath);
    MPlug plug = dagFn.findPlug("visibility");
    //cerr << "vis plug: " << plug.info() << ": " << plug.isKeyable() << endl;
    ClearAnimCurve(plug);

    //-----------------------------------------------------------------------------
    // シェイプの visibility アトリビュートにアニメーションカーブを接続します。
    status = CreateAnimCurveFromElement(plug, boneVisAnimElem, false, startFrame, 1.0, 0.0, true, fvbFile);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 各 LOD モデルにもアニメーションカーブを接続します。
    for (int iLod = 0; iLod < static_cast<int>(bone.m_LodShapePaths.length()); ++iLod)
    {
        const MDagPath lodShapePath = bone.m_LodShapePaths[iLod];
        if (!(lodShapePath == bone.m_ShapePath))
        {
            MPlug lodDstPlug = MFnDagNode(lodShapePath).findPlug("visibility");
            ClearAnimCurve(lodDstPlug);
            status = CreateAnimCurveFromElement(lodDstPlug, boneVisAnimElem, false, startFrame, 1.0, 0.0, true, fvbFile);
            CheckStatus(status);
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fvb ファイルを解析してアニメーションカーブを作成します。
//!
//! @param[in] bones fmd ファイルから取得したボーン配列です。
//! @param[in] fvbPath fvb ファイルのパスです。
//! @param[in] getsBones シーンからボーン配列を取得するなら true です。
//! @param[in] usesAnimBin バイナリ出力フラグを反映するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFvb(
    const ImpBoneArray& bones,
    const std::string& fvbPath,
    const bool getsBones,
    const bool usesAnimBin
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fvb ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile fvbFile(rstat, fvbPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = fvbFile.m_Xml.FindElement("nw4f_3dif/bone_visibility_anim");
    const RXMLElement* infoElem = animElem->FindElement("bone_visibility_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // シーンからボーン配列を取得します。
    const ImpBoneArray* pBones = &bones;
    ImpBoneArray fvbBones;
    const RXMLElement* boneVisAnimsElem = animElem->FindElement("bone_vis_bone_anim_array", false);
    if (getsBones && boneVisAnimsElem != NULL)
    {
        MStringArray selXforms;
        MGlobal::executeCommand("ls -sl -dag -typ transform", selXforms);

        for (size_t iMember = 0; iMember < boneVisAnimsElem->nodes.size(); ++iMember)
        {
            const RXMLElement* boneVisAnimElem = &boneVisAnimsElem->nodes[iMember];
            const std::string boneName = boneVisAnimElem->GetAttribute("bone_name");
            GetSceneBoneByNameAndClearAnim(fvbBones, boneName, selXforms, false);
        }
        if (fvbBones.empty())
        {
            DisplayImportWarning("対応するボーンが 1 つもありません。", "No bone is matched.");
            return status;
        }
        pBones = &fvbBones;
    }

    //-----------------------------------------------------------------------------
    // bone_vis_bone_anim_array 要素を解析してアニメーションカーブを作成します。
    if (boneVisAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < boneVisAnimsElem->nodes.size(); ++iMember)
        {
            status = ParseBoneVisAnim(&boneVisAnimsElem->nodes[iMember],
                startFrame, *pBones, usesAnimBin, fvbFile);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + fvbPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報からマテリアルアトリビュート名を取得します。
//!
//! @param[in] hint ヒント情報です。
//!
//! @return マテリアルアトリビュート名を返します。
//-----------------------------------------------------------------------------
static std::string GetMatColAttribNameForHint(const std::string& hint)
{
    if      (hint == "diffuse" ) return "color";
    else if (hint == "opacity" ) return "transparency";
    else if (hint == "ambient" ) return "ambientColor";
    else if (hint == "emission") return "incandescence";
    else if (hint == "specular") return "specularColor";
    else                         return "";
}

//-----------------------------------------------------------------------------
//! @brief シェーダパラメータマテリアルアニメーションを解析してカラーのアニメーションカーブを作成します。
//!
//! @param[in] matAnimElem シェーダパラメータマテリアルアニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] materials マテリアル配列です。
//! @param[in] fclFile fcl ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseShaderParamMatAnimColor(
    const RXMLElement* matAnimElem,
    const int startFrame,
    const ImpMaterialArray& materials,
    const ImpIntermediateFile& fclFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 対応するマテリアルを検索します。
    const std::string matName = matAnimElem->GetAttribute("mat_name");
    const int iMat = FindMaterialByName(materials, matName);
    if (iMat == -1)
    {
        DisplayImportWarning("マテリアルが見つかりません: {0}",
            "The material cannot be found: {0}", matName);
        return status;
    }
    const ImpMaterial& material = materials[iMat];

    //-----------------------------------------------------------------------------
    // マテリアルまたはマテリアルに接続された file ノードにアニメーションカーブを接続します。
    const MFnDependencyNode shaderFn(material.m_ShaderObj);
    const RXMLElement* paramsElem = matAnimElem->FindElement("param_anim_array");
    for (size_t iParam = 0; iParam < paramsElem->nodes.size(); ++iParam)
    {
        const RXMLElement* paramElem = &paramsElem->nodes[iParam];
        const std::string paramId = paramElem->GetAttribute("id");
        const std::string orgHint = paramElem->GetAttribute("original_hint");
        std::string matAttr = GetMatColAttribNameForHint(orgHint);
        if (matAttr.empty())
        {
            if      (paramId == "mat_color0") matAttr = "color";
            else if (paramId == "mat_color1") matAttr = "specularColor";
            else if (paramId == "amb_color0") matAttr = "ambientColor";
        }
        MPlug dstPlug = (!matAttr.empty()) ? shaderFn.findPlug(matAttr.c_str()) : MPlug();
        if (!dstPlug.isNull())
        {
            MObjectArray fileObjs;
            GetFileNodesForPlug(fileObjs, dstPlug);
            if (fileObjs.length() != 0)
            {
                dstPlug = MFnDependencyNode(fileObjs[0]).findPlug("colorGain");
            }

            for (size_t iCurve = 0; iCurve < paramElem->nodes.size(); ++iCurve)
            {
                const RXMLElement* targetElem = &paramElem->nodes[iCurve];
                const int iComp = atoi(targetElem->GetAttribute("component_index").c_str());
                if (iComp <= 2)
                {
                    //-----------------------------------------------------------------------------
                    // RGB カラーのアニメーションカーブを作成します。
                    MPlug compPlug = dstPlug.child(iComp);
                    ClearAnimCurve(compPlug);
                    status = CreateAnimCurveFromElement(compPlug, targetElem, false, startFrame, 1.0, 0.0, true, fclFile);
                    CheckStatus(status);
                }
                else if (paramId == "mat_color0" && iComp == 3)
                {
                    //-----------------------------------------------------------------------------
                    // 透明度（アルファ）のアニメーションカーブを作成します。
                    MPlug transparencyPlug = shaderFn.findPlug("transparency");
                    fileObjs.clear();
                    GetFileNodesForPlug(fileObjs, transparencyPlug);
                    if (fileObjs.length() != 0)
                    {
                        MPlug alphaGainPlug = MFnDependencyNode(fileObjs[0]).findPlug("alphaGain");
                        ClearAnimCurve(alphaGainPlug);
                        status = CreateAnimCurveFromElement(alphaGainPlug, targetElem, false, startFrame, 1.0, 0.0, true, fclFile);
                        CheckStatus(status);
                    }
                    else
                    {
                        for (int iRgb = 0; iRgb < R_RGB_COUNT; ++iRgb)
                        {
                            MPlug compPlug = transparencyPlug.child(iRgb);
                            ClearAnimCurve(compPlug);
                            // 透明度の場合 1 - アルファ を設定します。
                            status = CreateAnimCurveFromElement(compPlug, targetElem, false, startFrame, -1.0, 1.0, true, fclFile);
                            CheckStatus(status);
                        }
                    }
                }
            }
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief オリジナルマテリアルアニメーションを解析してカラーのアニメーションカーブを作成します。
//!
//! @param[in] matAnimElem オリジナルマテリアルアニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] materials マテリアル配列です。
//! @param[in] fclFile fma または fcl ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseOrgMatAnimColor(
    const RXMLElement* matAnimElem,
    const int startFrame,
    const ImpMaterialArray& materials,
    const ImpIntermediateFile& fclFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // オリジナルカラーアニメーション配列要素がなければ何もしません。
    const RXMLElement* orgColAnimsElem = matAnimElem->FindElement("original_color_anim_array", false);
    if (orgColAnimsElem == NULL)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 対応するマテリアルを検索します。
    const std::string matName = matAnimElem->GetAttribute("mat_name");
    const int iMat = FindMaterialByName(materials, matName);
    if (iMat == -1)
    {
        DisplayImportWarning("マテリアルが見つかりません: {0}",
            "The material cannot be found: {0}", matName);
        return status;
    }
    const ImpMaterial& material = materials[iMat];

    //-----------------------------------------------------------------------------
    // マテリアルまたはマテリアルに接続された file ノードにアニメーションカーブを接続します。
    const MFnDependencyNode shaderFn(material.m_ShaderObj);
    for (size_t iCol = 0; iCol < orgColAnimsElem->nodes.size(); ++iCol)
    {
        const RXMLElement* orgColAnimElem = &orgColAnimsElem->nodes[iCol];
        const std::string hint = orgColAnimElem->GetAttribute("hint");
        const std::string matAttr = GetMatColAttribNameForHint(hint);
        MPlug dstPlug = (!matAttr.empty()) ? shaderFn.findPlug(matAttr.c_str()) : MPlug();
        if (!dstPlug.isNull())
        {
            MObjectArray fileObjs;
            GetFileNodesForPlug(fileObjs, dstPlug);
            bool usesAlphaGain = false;
            if (fileObjs.length() != 0)
            {
                std::string fileAttr = "colorGain";
                if (hint == "opacity")
                {
                    usesAlphaGain = true;
                    fileAttr = "alphaGain";
                }
                dstPlug = MFnDependencyNode(fileObjs[0]).findPlug(fileAttr.c_str());
            }
            double valueScale = 1.0;
            double valueOfs = 0.0;
            if (hint == "opacity" && !usesAlphaGain)
            {
                valueScale = -1.0;
                valueOfs = 1.0;
            }

            for (size_t iCurve = 0; iCurve < orgColAnimElem->nodes.size(); ++iCurve)
            {
                const RXMLElement* targetElem = &orgColAnimElem->nodes[iCurve];
                const std::string target = targetElem->GetAttribute("target");
                const int iComp = (target == "color_r") ? 0 : ((target == "color_g") ? 1 : 2);
                MPlug compPlug = (usesAlphaGain) ? dstPlug : dstPlug.child(iComp);
                ClearAnimCurve(compPlug);
                status = CreateAnimCurveFromElement(compPlug, targetElem, true, startFrame, valueScale, valueOfs, true, fclFile);
                CheckStatus(status);
                if (usesAlphaGain) // alphaGain アトリビュートの場合、カーブは 1 つだけ接続
                {
                    break;
                }
            }
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fcl ファイルを解析してアニメーションカーブを作成します。
//!        現在、rmdl2nw4f.exe で変換した fcl ファイルのみ対応しています。
//!
//! @param[in] materials マテリアル配列です。
//! @param[in] fclPath fcl ファイルのパスです。
//! @param[in] getsMaterials シーンからマテリアル配列を取得するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFcl(
    const ImpMaterialArray& materials,
    const std::string& fclPath,
    const bool getsMaterials
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fcl ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile fclFile(rstat, fclPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = fclFile.m_Xml.FindElement("nw4f_3dif/shader_param_anim");
    const RXMLElement* infoElem = animElem->FindElement("shader_param_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // シーンからマテリアル配列を取得します。
    // <shader_param_mat_anim_array> がなければ
    // <original_material_anim_array> を取得します。
    const ImpMaterialArray* pMaterials = &materials;
    ImpMaterialArray fclMaterials;
    bool getsOrgMatAnim = false;
    const RXMLElement* matAnimsElem = animElem->FindElement("shader_param_mat_anim_array", false);
    if (matAnimsElem == NULL)
    {
        matAnimsElem = animElem->FindElement("original_material_anim_array", false);
        getsOrgMatAnim = true;
    }
    if (getsMaterials && matAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < matAnimsElem->nodes.size(); ++iMember)
        {
            const RXMLElement* matAnimElem = &matAnimsElem->nodes[iMember];
            const std::string matName = matAnimElem->GetAttribute("mat_name");
            GetSceneMaterialByName(fclMaterials, matName);
        }
        if (fclMaterials.empty())
        {
            DisplayImportWarning("対応するマテリアルが 1 つもありません。", "No material is matched.");
            return status;
        }
        pMaterials = &fclMaterials;
    }

    //-----------------------------------------------------------------------------
    // shader_param_mat_anim_array 要素を解析してアニメーションカーブを作成します。
    if (matAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < matAnimsElem->nodes.size(); ++iMember)
        {
            if (!getsOrgMatAnim)
            {
                status = ParseShaderParamMatAnimColor(&matAnimsElem->nodes[iMember], startFrame, *pMaterials, fclFile);
            }
            else
            {
                status = ParseOrgMatAnimColor(&matAnimsElem->nodes[iMember], startFrame, *pMaterials, fclFile);
            }
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + fclPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief シェーダパラメータマテリアルアニメーションを解析してテクスチャ SRT のアニメーションカーブを作成します。
//!
//! @param[in] matAnimElem シェーダパラメータマテリアルアニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] materials マテリアル配列です。
//! @param[in] ftsFile fts ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseShaderParamMatAnimTexSrt(
    const RXMLElement* matAnimElem,
    const int startFrame,
    const ImpMaterialArray& materials,
    const ImpIntermediateFile& ftsFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 対応するマテリアルを検索します。
    const std::string matName = matAnimElem->GetAttribute("mat_name");
    const int iMat = FindMaterialByName(materials, matName);
    if (iMat == -1)
    {
        DisplayImportWarning("マテリアルが見つかりません: {0}",
            "The material cannot be found: {0}", matName);
        return status;
    }
    const ImpMaterial& material = materials[iMat];

    //-----------------------------------------------------------------------------
    // マテリアルに接続された place2dTexture ノードにアニメーションカーブを接続します。
    const int PARAM_COUNT = 6;
    static const char* const attribNames[PARAM_COUNT] =
    {
        NULL, // モード値
        "repeatU",
        "repeatV",
        "rotateFrame",
        "translateFrameU",
        "translateFrameV",
    };

    const MFnDependencyNode shaderFn(material.m_ShaderObj);
    const RXMLElement* paramsElem = matAnimElem->FindElement("param_anim_array");
    for (size_t iParam = 0; iParam < paramsElem->nodes.size(); ++iParam)
    {
        const RXMLElement* paramElem = &paramsElem->nodes[iParam];
        const std::string paramId = paramElem->GetAttribute("id");
        const std::string orgHint = paramElem->GetAttribute("original_hint");
        const std::string paramType = paramElem->GetAttribute("type");
        if (paramType == "texsrt" ||
            paramType == "texsrt_ex")
        {
            MObject fileObj = GetFileNodeForHintString(material, orgHint);
            if (fileObj.isNull())
            {
                if (paramId.find("texmtx") == 0)
                {
                    const unsigned int iMtx = GetStringSuffixNumber(paramId);
                    MPlug dstPlug = shaderFn.findPlug("color"); // color 以外に接続する場合は未対応
                    MObjectArray fileObjs;
                    GetFileNodesForPlug(fileObjs, dstPlug);
                    if (iMtx < fileObjs.length())
                    {
                        fileObj = fileObjs[iMtx];
                    }
                }
            }
            if (fileObj.isNull())
            {
                continue;
            }
            MFnDependencyNode fileFn(fileObj);
            MPlugArray plugArray;
            fileFn.findPlug("wrapU").connectedTo(plugArray, true, false);
            if (plugArray.length() == 0 ||
                plugArray[0].node().apiType() != MFn::kPlace2dTexture)
            {
                continue;
            }
            MFnDependencyNode placeFn(plugArray[0].node());
            for (size_t iCurve = 0; iCurve < paramElem->nodes.size(); ++iCurve)
            {
                const RXMLElement* targetElem = &paramElem->nodes[iCurve];
                const int iComp = atoi(targetElem->GetAttribute("component_index").c_str());
                if (iComp < PARAM_COUNT && attribNames[iComp] != NULL)
                {
                    MPlug compPlug = placeFn.findPlug(attribNames[iComp]);
                    ClearAnimCurve(compPlug);
                    const double valueScale = (iComp == 3) ? R_M_DEG_TO_RAD : 1.0;
                    status = CreateAnimCurveFromElement(compPlug, targetElem, false, startFrame, valueScale, 0.0, true, ftsFile);
                    CheckStatus(status);
                }
            }
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief オリジナルマテリアルアニメーションを解析してテクスチャ SRT のアニメーションカーブを作成します。
//!
//! @param[in] matAnimElem オリジナルマテリアルアニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] materials マテリアル配列です。
//! @param[in] ftsFile fma または fts ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseOrgMatAnimTexSrt(
    const RXMLElement* matAnimElem,
    const int startFrame,
    const ImpMaterialArray& materials,
    const ImpIntermediateFile& ftsFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // オリジナルテクスチャ SRT アニメーション配列要素がなければ何もしません。
    const RXMLElement* orgTexSrtAnimsElem = matAnimElem->FindElement("original_texsrt_anim_array", false);
    if (orgTexSrtAnimsElem == NULL)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 対応するマテリアルを検索します。
    const std::string matName = matAnimElem->GetAttribute("mat_name");
    const int iMat = FindMaterialByName(materials, matName);
    if (iMat == -1)
    {
        DisplayImportWarning("マテリアルが見つかりません: {0}",
            "The material cannot be found: {0}", matName);
        return status;
    }
    const ImpMaterial& material = materials[iMat];

    //-----------------------------------------------------------------------------
    // マテリアルに接続された place2dTexture ノードにアニメーションカーブを接続します。
    const MFnDependencyNode shaderFn(material.m_ShaderObj);
    for (size_t iTexSrt = 0; iTexSrt < orgTexSrtAnimsElem->nodes.size(); ++iTexSrt)
    {
        const RXMLElement* orgTexSrtAnimElem = &orgTexSrtAnimsElem->nodes[iTexSrt];
        const MObject fileObj = GetFileNodeForHintString(material, orgTexSrtAnimElem->GetAttribute("hint"));
        if (!fileObj.isNull())
        {
            MFnDependencyNode fileFn(fileObj);
            MPlugArray plugArray;
            fileFn.findPlug("wrapU").connectedTo(plugArray, true, false);
            if (plugArray.length() == 0 ||
                plugArray[0].node().apiType() != MFn::kPlace2dTexture)
            {
                continue;
            }
            MFnDependencyNode placeFn(plugArray[0].node());
            for (size_t iCurve = 0; iCurve < orgTexSrtAnimElem->nodes.size(); ++iCurve)
            {
                const RXMLElement* targetElem = &orgTexSrtAnimElem->nodes[iCurve];
                const std::string target = targetElem->GetAttribute("target");
                std::string place2dAttr;
                if      (target == "scale_x"    ) place2dAttr = "repeatU";
                else if (target == "scale_y"    ) place2dAttr = "repeatV";
                else if (target == "rotate"     ) place2dAttr = "rotateFrame";
                else if (target == "translate_x") place2dAttr = "translateFrameU";
                else if (target == "translate_y") place2dAttr = "translateFrameV";
                if (!place2dAttr.empty())
                {
                    MPlug compPlug = placeFn.findPlug(place2dAttr.c_str());
                    ClearAnimCurve(compPlug);
                    const double valueScale = (target == "rotate") ? R_M_DEG_TO_RAD : 1.0;
                    status = CreateAnimCurveFromElement(compPlug, targetElem, true, startFrame, valueScale, 0.0, true, ftsFile);
                    CheckStatus(status);
                }
            }
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fts ファイルを解析してアニメーションカーブを作成します。
//!        現在、rmdl2nw4f.exe で変換した fts ファイルのみ対応しています。
//!
//! @param[in] materials マテリアル配列です。
//! @param[in] ftsPath fts ファイルのパスです。
//! @param[in] getsMaterials シーンからマテリアル配列を取得するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFts(
    const ImpMaterialArray& materials,
    const std::string& ftsPath,
    const bool getsMaterials
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fts ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile ftsFile(rstat, ftsPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = ftsFile.m_Xml.FindElement("nw4f_3dif/shader_param_anim");
    const RXMLElement* infoElem = animElem->FindElement("shader_param_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // シーンからマテリアル配列を取得します。
    // <shader_param_mat_anim_array> がなければ
    // <original_material_anim_array> を取得します。
    const ImpMaterialArray* pMaterials = &materials;
    ImpMaterialArray ftsMaterials;
    bool getsOrgMatAnim = false;
    const RXMLElement* matAnimsElem = animElem->FindElement("shader_param_mat_anim_array", false);
    if (matAnimsElem == NULL)
    {
        matAnimsElem = animElem->FindElement("original_material_anim_array", false);
        getsOrgMatAnim = true;
    }
    if (getsMaterials && matAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < matAnimsElem->nodes.size(); ++iMember)
        {
            const RXMLElement* matAnimElem = &matAnimsElem->nodes[iMember];
            const std::string matName = matAnimElem->GetAttribute("mat_name");
            GetSceneMaterialByName(ftsMaterials, matName);
        }
        if (ftsMaterials.empty())
        {
            DisplayImportWarning("対応するマテリアルが 1 つもありません。", "No material is matched.");
            return status;
        }
        pMaterials = &ftsMaterials;
    }

    //-----------------------------------------------------------------------------
    // shader_param_mat_anim_array 要素を解析してアニメーションカーブを作成します。
    if (matAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < matAnimsElem->nodes.size(); ++iMember)
        {
            if (!getsOrgMatAnim)
            {
                status = ParseShaderParamMatAnimTexSrt(&matAnimsElem->nodes[iMember], startFrame, *pMaterials, ftsFile);
            }
            else
            {
                status = ParseOrgMatAnimTexSrt(&matAnimsElem->nodes[iMember], startFrame, *pMaterials, ftsFile);
            }
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + ftsPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのテクスチャパターンアニメーションを解析してアニメーションカーブを作成します。
//!
//! @param[in] matAnimElem マテリアル単位アニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] materials マテリアル配列です。
//! @param[in] texPats テクスチャパターン配列です。
//! @param[in] ftpFile fma または ftp ファイルです。
//! @param[in] isFma fma ファイルなら true、ftp ファイルなら false を指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseTexPatMatAnim(
    const RXMLElement* matAnimElem,
    const int startFrame,
    const ImpMaterialArray& materials,
    const ImpTexPatArray& texPats,
    const ImpIntermediateFile& ftpFile,
    const bool isFma
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーション配列要素を取得します。
    const RXMLElement* texPatAnimsElem = (isFma) ?
        matAnimElem->FindElement("tex_pattern_anim_array", false) :
        matAnimElem;
    if (texPatAnimsElem == nullptr)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 対応するマテリアルを検索します。
    const std::string matName = matAnimElem->GetAttribute("mat_name");
    const int iMat = FindMaterialByName(materials, matName);
    if (iMat == -1)
    {
        DisplayImportWarning("マテリアルが見つかりません: {0}",
            "The material cannot be found: {0}", matName);
        return status;
    }
    const ImpMaterial& material = materials[iMat];

    //-----------------------------------------------------------------------------
    // マテリアルに接続された file ノードにアニメーションカーブを接続します。
    for (size_t iCurve = 0; iCurve < texPatAnimsElem->nodes.size(); ++iCurve)
    {
        const RXMLElement* targetElem = &texPatAnimsElem->nodes[iCurve];
        const MObject fileObj = GetFileNodeForHintString(material, targetElem->GetAttribute("hint"));
        if (!fileObj.isNull())
        {
            //-----------------------------------------------------------------------------
            // イメージの名前が name.number.ext の形式になっているかチェックします。
            MFnDependencyNode texFn(fileObj);
            MString filePathTmp;
            texFn.findPlug("fileTextureName").getValue(filePathTmp);
            const std::string filePath = filePathTmp.asChar();
            const std::string ext2 = RGetExtensionFromFilePath(RGetNoExtensionFilePath(filePath));
            if (ext2.empty() || (!GetNoSuffixNumberString(ext2).empty() && ext2 != "<f>"))
            {
                DisplayImportWarning("テクスチャーパターンアニメーションで使用するテクスチャーのファイル名が不正です: {0}",
                    "Texture image name is wrong for pattern animation: {0}", filePath);
                continue;
            }

            //-----------------------------------------------------------------------------
            // アニメーションカーブを作成します。
            MPlug fePlug = texFn.findPlug("frameExtension");
            ClearAnimCurve(fePlug);
            status = CreateAnimCurveFromElement(fePlug, targetElem, false, startFrame, 1.0, 0.0, true, ftpFile);
            CheckStatus(status);

            //-----------------------------------------------------------------------------
            // アニメーションカーブの値をフレーム拡張子に変換します。
            MPlugArray plugArray;
            fePlug.connectedTo(plugArray, true, false);
            if (plugArray.length() != 0)
            {
                MObject curveObj = plugArray[0].node();
                if (curveObj.apiType() == MFn::kAnimCurveTimeToUnitless)
                {
                    MFnAnimCurve curveFn(curveObj);
                    const int keyCount = curveFn.numKeys();
                    for (int iKey = 0; iKey < keyCount; ++iKey)
                    {
                        const int iv = static_cast<int>(curveFn.value(iKey));
                        if (0 <= iv && iv < static_cast<int>(texPats.size())) // 正常な ftp ファイルなら常に成立
                        {
                            curveFn.setValue(iKey, static_cast<double>(texPats[iv].m_FrameExtension));
                        }
                    }
                }
            }
            else // アニメーションカーブが接続されていない場合
            {
                int iv;
                fePlug.getValue(iv);
                if (0 <= iv && iv < static_cast<int>(texPats.size())) // 正常な ftp ファイルなら常に成立
                {
                    fePlug.setValue(texPats[iv].m_FrameExtension);
                }
            }

            //-----------------------------------------------------------------------------
            // フレーム拡張子の使用を ON にします。
            texFn.findPlug("useFrameExtension").setValue(true);
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターン配列からテクスチャデータを作成します。
//!
//! @param[in,out] pTexDatas 作成済みのテクスチャデータ配列へのポインタです。
//! @param[in] texPats テクスチャパターン配列です。
//! @param[in] inputFolder 中間ファイルのフォルダのパスです。
//! @param[in] createsTexture テクスチャファイルを作成するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CreateTextureDataForTexPats(
    ImpTexDataArray* pTexDatas,
    const ImpTexPatArray& texPats,
    const std::string& inputFolder,
    const bool createsTexture
)
{
    MStatus status;
    if (texPats.empty())
    {
        return status;
    }
    const std::string imagesFolder = GetSourceImagesFolder();
    const std::string texcvtrPath = GetNintendoMaya3dTextureConverterPath();
    if (!RFileExists(texcvtrPath.c_str()))
    {
        DisplayImportError("3D テクスチャーコンバーターが見つかりません: {0}",
            "3D texture converter cannot be found: {0}", texcvtrPath);
        return MS::kFailure;
    }
    ImpTexDataArray& texDatas = *pTexDatas;
    for (size_t texPatIdx = 0; texPatIdx < texPats.size(); ++texPatIdx)
    {
        const std::string texName = texPats[texPatIdx].m_Name;
        if (FindTexDataByName(texDatas, texName) == -1)
        {
            texDatas.push_back(ImpTexData());
            status = CreateTextureImageFile(texDatas.back(), texName,
                inputFolder, imagesFolder, texcvtrPath, createsTexture);
            CheckStatus(status);
        }
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief ftp ファイルを解析してアニメーションカーブを作成します。
//!        fmd ファイルで使用されていないテクスチャがあれば画像ファイルに変換します。
//!
//! @param[in,out] pTexDatas 作成済みのテクスチャデータ配列へのポインタです。
//! @param[in] materials マテリアル配列です。
//! @param[in] ftpPath ftp ファイルのパスです。
//! @param[in] getsMaterials シーンからマテリアル配列を取得するなら true です。
//! @param[in] createsTexture テクスチャファイルを作成するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFtp(
    ImpTexDataArray* pTexDatas,
    const ImpMaterialArray& materials,
    const std::string& ftpPath,
    const bool getsMaterials,
    const bool createsTexture
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // ftp ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile ftpFile(rstat, ftpPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = ftpFile.m_Xml.FindElement("nw4f_3dif/tex_pattern_anim");
    const RXMLElement* infoElem = animElem->FindElement("tex_pattern_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // シーンからマテリアル配列を取得します。
    const ImpMaterialArray* pMaterials = &materials;
    ImpMaterialArray ftpMaterials;
    const RXMLElement* matAnimsElem = animElem->FindElement("tex_pattern_mat_anim_array", false);
    if (getsMaterials && matAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < matAnimsElem->nodes.size(); ++iMember)
        {
            const RXMLElement* matAnimElem = &matAnimsElem->nodes[iMember];
            const std::string matName = matAnimElem->GetAttribute("mat_name");
            GetSceneMaterialByName(ftpMaterials, matName);
        }
        if (ftpMaterials.empty())
        {
            DisplayImportWarning("対応するマテリアルが 1 つもありません。", "No material is matched.");
            return status;
        }
        pMaterials = &ftpMaterials;
    }

    //-----------------------------------------------------------------------------
    // テクスチャパターン配列を取得します。
    ImpTexPatArray texPats;
    const RXMLElement* texPatsElem = animElem->FindElement("tex_pattern_array", false);
    if (texPatsElem != NULL)
    {
        for (size_t iMember = 0; iMember < texPatsElem->nodes.size(); ++iMember)
        {
            const RXMLElement* texPatElem = &texPatsElem->nodes[iMember];
            texPats.push_back(ImpTexPat(texPatElem->GetAttribute("tex_name")));
        }
    }

    //-----------------------------------------------------------------------------
    // tex_pattern_mat_anim_array 要素を解析してアニメーションカーブを作成します。
    if (matAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < matAnimsElem->nodes.size(); ++iMember)
        {
            status = ParseTexPatMatAnim(&matAnimsElem->nodes[iMember],
                startFrame, *pMaterials, texPats, ftpFile, false);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // fmd ファイルで使用されていないテクスチャがあれば画像ファイルに変換します。
    const std::string inputFolder = RGetFolderFromFilePath(ftpPath) + "/";
    status = CreateTextureDataForTexPats(pTexDatas, texPats, inputFolder, createsTexture);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + ftpPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fma ファイルを解析してアニメーションカーブを作成します。
//!        fmd ファイルで使用されていないテクスチャがあれば画像ファイルに変換します。
//!
//! @param[in,out] pTexDatas 作成済みのテクスチャデータ配列へのポインタです。
//! @param[in] materials マテリアル配列です。
//! @param[in] fmaPath fma ファイルのパスです。
//! @param[in] getsMaterials シーンからマテリアル配列を取得するなら true です。
//! @param[in] createsTexture テクスチャファイルを作成するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFma(
    ImpTexDataArray* pTexDatas,
    const ImpMaterialArray& materials,
    const std::string& fmaPath,
    const bool getsMaterials,
    const bool createsTexture
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fma ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile fmaFile(rstat, fmaPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = fmaFile.m_Xml.FindElement("nw4f_3dif/material_anim");
    const RXMLElement* infoElem = animElem->FindElement("material_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // シーンからマテリアル配列を取得します。
    const ImpMaterialArray* pMaterials = &materials;
    const ImpMaterialArray* pOrgMaterials = &materials;
    ImpMaterialArray fmaMaterials;
    ImpMaterialArray fmaOrgMaterials;
    const RXMLElement* matAnimsElem = animElem->FindElement("per_material_anim_array", false);
    const RXMLElement* orgMatAnimsElem = animElem->FindElement("original_per_material_anim_array", false);
    if (getsMaterials && matAnimsElem != nullptr)
    {
        for (size_t memberIdx = 0; memberIdx < matAnimsElem->nodes.size(); ++memberIdx)
        {
            const RXMLElement* matAnimElem = &matAnimsElem->nodes[memberIdx];
            const std::string matName = matAnimElem->GetAttribute("mat_name");
            GetSceneMaterialByName(fmaMaterials, matName);
        }
        pMaterials = &fmaMaterials;
    }
    if (getsMaterials && orgMatAnimsElem != nullptr)
    {
        for (size_t memberIdx = 0; memberIdx < orgMatAnimsElem->nodes.size(); ++memberIdx)
        {
            const RXMLElement* orgMatAnimElem = &orgMatAnimsElem->nodes[memberIdx];
            const std::string matName = orgMatAnimElem->GetAttribute("mat_name");
            GetSceneMaterialByName(fmaOrgMaterials, matName);
        }
        pOrgMaterials = &fmaOrgMaterials;
    }
    if (pMaterials->empty() && pOrgMaterials->empty())
    {
        DisplayImportWarning("対応するマテリアルが 1 つもありません。", "No material is matched.");
        return status;
    }

    //-----------------------------------------------------------------------------
    // テクスチャパターン配列を取得します。
    ImpTexPatArray texPats;
    const RXMLElement* texPatsElem = animElem->FindElement("tex_pattern_array", false);
    if (texPatsElem != nullptr)
    {
        for (size_t memberIdx = 0; memberIdx < texPatsElem->nodes.size(); ++memberIdx)
        {
            const RXMLElement* texPatElem = &texPatsElem->nodes[memberIdx];
            texPats.push_back(ImpTexPat(texPatElem->GetAttribute("tex_name")));
        }
    }

    //-----------------------------------------------------------------------------
    // マテリアル単位配列要素を解析してアニメーションカーブを作成します。
    if (matAnimsElem != nullptr)
    {
        for (size_t memberIdx = 0; memberIdx < matAnimsElem->nodes.size(); ++memberIdx)
        {
            status = ParseTexPatMatAnim(&matAnimsElem->nodes[memberIdx],
                startFrame, *pMaterials, texPats, fmaFile, true);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // オリジナルマテリアル単位配列要素を解析してアニメーションカーブを作成します。
    if (orgMatAnimsElem != nullptr)
    {
        for (size_t memberIdx = 0; memberIdx < orgMatAnimsElem->nodes.size(); ++memberIdx)
        {
            const RXMLElement* orgMatAnimElem = &orgMatAnimsElem->nodes[memberIdx];
            status = ParseOrgMatAnimColor(orgMatAnimElem, startFrame, *pOrgMaterials, fmaFile);
            CheckStatus(status);
            status = ParseOrgMatAnimTexSrt(orgMatAnimElem, startFrame, *pOrgMaterials, fmaFile);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // fmd ファイルで使用されていないテクスチャがあれば画像ファイルに変換します。
    const std::string inputFolder = RGetFolderFromFilePath(fmaPath) + "/";
    status = CreateTextureDataForTexPats(pTexDatas, texPats, inputFolder, createsTexture);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + fmaPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 頂点シェイプアニメーションの対象ボーンを検索します。
//!
//! @param[in] bones ボーン配列です。
//! @param[in] shapeName シェイプ名です。
//! @param[in] baseName ベースボーン名です。
//!
//! @return インデックスを返します。見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindBoneForVtxShapeAnim(
    const ImpBoneArray& bones,
    const std::string& shapeName,
    const std::string& baseName
)
{
    for (int iBone = 0; iBone < static_cast<int>(bones.size()); ++iBone)
    {
        if (RFindValueInArray(bones[iBone].m_ShapeNames, shapeName) != -1)
        {
            return iBone;
        }
    }
    return FindBoneByName(bones, baseName);
}

//-----------------------------------------------------------------------------
//! @brief シェイプに接続されたブレンドシェイプノードにアニメーションを設定します。
//!
//! @param[in] vtxShapeAnimElem 頂点シェイプアニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] shapePath シェイプノードの DAG パスです。
//! @param[in] fshFile fsh ファイルです。
//-----------------------------------------------------------------------------
static MStatus SetBlendShapeAnim(
    const RXMLElement* vtxShapeAnimElem,
    const int startFrame,
    const MDagPath& shapePath,
    const ImpIntermediateFile& fshFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // ブレンドシェイプノードを取得します。
    MObject blendObj = GetBlendShapeNodeFromHistory(shapePath);
    if (blendObj.isNull())
    {
        DisplayImportWarning("blendShape ノードが見つかりません: {0}",
            "blendShape node cannot be found: {0}", shapePath.partialPathName().asChar());
        return status;
    }

    //-----------------------------------------------------------------------------
    // ターゲット名とウェイトプラグのマップを取得します。
    std::map<std::string, MPlug> targetPlugs;
    MFnBlendShapeDeformer blendFn(blendObj);
    const int targetCount = blendFn.numWeights(); // ウェイトの要素数がターゲットの数となります。
    MPlug weightsPlug = blendFn.findPlug("weight");
    MIntArray weightIndexes; // 各ターゲットのウェイトインデックス配列です。
    blendFn.weightIndexList(weightIndexes);
    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
    {
        const int weightIndex = weightIndexes[iTarget];
        MPlug weightPlug = weightsPlug.elementByLogicalIndex(weightIndex);
        std::string targetName = weightPlug.partialName(false, false, false, true).asChar();
            // 第 4 引数 useAlias を true にすることで
            // ブレンドシェイプエディタで入力したエイリアス名を取得できます。
            // エイリアス名が設定されていなければ "w[0]" "w[1]" といったプラグ名が返ります。
        if (targetName.empty() || targetName.find('[') != std::string::npos)
        {
            MObjectArray targetObjs;
            blendFn.getTargets(shapePath.node(), weightIndex, targetObjs);
            if (targetObjs.length() == 0)
            {
                DisplayImportWarning("ブレンドシェイプのターゲットが見つかりません: {0}",
                    "Blend shape target cannot be found: {0}", weightPlug.info().asChar());
                continue;
            }
            MDagPath targetPath;
            MFnDagNode(targetObjs[0]).getPath(targetPath); // ターゲットのシェイプノードのパスを取得します。
            targetPath.pop();
            targetName = RemoveNamespace(MFnDependencyNode(targetPath.node()).name().asChar());
        }
        targetPlugs[targetName] = weightPlug;
    }
    //for (std::map<std::string, MPlug>::iterator it = targetPlugs.begin(); it != targetPlugs.end(); ++it)
    //{ cerr << "map: " << (*it).first << ", " << (*it).second.info() << endl; }

    //-----------------------------------------------------------------------------
    // 各ウェイトアトリビュートにアニメーションカーブを接続します。
    for (size_t iCurve = 0; iCurve < vtxShapeAnimElem->nodes.size(); ++iCurve)
    {
        const RXMLElement* targetElem = &vtxShapeAnimElem->nodes[iCurve];
        const std::string keyShapeName = targetElem->GetAttribute("key_shape_name");
        if (targetPlugs.find(keyShapeName) != targetPlugs.end())
        {
            MPlug weightPlug = targetPlugs[keyShapeName];
            if (!weightPlug.isNull())
            {
                ClearAnimCurve(weightPlug);
                status = CreateAnimCurveFromElement(weightPlug, targetElem, false, startFrame, 1.0, 0.0, false, fshFile);
                CheckStatus(status);
            }
        }
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief 頂点シェイプアニメーションを解析してアニメーションカーブを作成します。
//!
//! @param[in] vtxShapeAnimElem 頂点シェイプアニメーション要素です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] bones ボーン配列です。
//! @param[in] fshFile fsh ファイルです。
//-----------------------------------------------------------------------------
static MStatus ParseVtxShapeAnim(
    const RXMLElement* vtxShapeAnimElem,
    const int startFrame,
    const ImpBoneArray& bones,
    const ImpIntermediateFile& fshFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 対応するボーンを検索します。
    const std::string shapeName = vtxShapeAnimElem->GetAttribute("shape_name");
    const std::string baseName = vtxShapeAnimElem->GetAttribute("base_name");
    const int iBone = FindBoneForVtxShapeAnim(bones, shapeName, baseName);
    if (iBone == -1)
    {
        DisplayImportWarning("シェイプに対応するボーンが見つかりません: {0}",
            "The bone cannot be found for the shape: {0}", shapeName);
        return status;
    }
    const ImpBone& bone = bones[iBone];
    if (bone.m_IsSkipped)
    {
        return status;
    }
    if (!bone.m_HasShape)
    {
        DisplayImportWarning("ボーンの子に mesh ノードが見つかりません: {0}",
            "The mesh cannot be found under the bone: {0}", bone.m_Name);
        return status;
    }

    //-----------------------------------------------------------------------------
    // シェイプに接続されたブレンドシェイプノードにアニメーションを設定します。
    status = SetBlendShapeAnim(vtxShapeAnimElem, startFrame, bone.m_ShapePath, fshFile);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 各 LOD モデルに接続されたブレンドシェイプノードにもアニメーションを設定します。
    for (int iLod = 0; iLod < static_cast<int>(bone.m_LodShapePaths.length()); ++iLod)
    {
        const MDagPath lodShapePath = bone.m_LodShapePaths[iLod];
        if (!(lodShapePath == bone.m_ShapePath))
        {
            status = SetBlendShapeAnim(vtxShapeAnimElem, startFrame, lodShapePath, fshFile);
            CheckStatus(status);
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fsh ファイルを解析してブレンドシェイプのアニメーションカーブを作成します。
//!
//! @param[in] bones fmd ファイルから取得したボーン配列です。
//! @param[in] fshPath fsh ファイルのパスです。
//! @param[in] getsBones シーンからボーン配列を取得するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFsh(
    const ImpBoneArray& bones,
    const std::string& fshPath,
    const bool getsBones
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fsh ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile fshFile(rstat, fshPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = fshFile.m_Xml.FindElement("nw4f_3dif/shape_anim");
    const RXMLElement* infoElem = animElem->FindElement("shape_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // シーンからボーン配列を取得します。
    const ImpBoneArray* pBones = &bones;
    ImpBoneArray fshBones;
    const RXMLElement* vtxShapeAnimsElem = animElem->FindElement("vertex_shape_anim_array", false);
    if (getsBones && vtxShapeAnimsElem != NULL)
    {
        MStringArray selXforms;
        MGlobal::executeCommand("ls -sl -dag -typ transform", selXforms);

        for (size_t iMember = 0; iMember < vtxShapeAnimsElem->nodes.size(); ++iMember)
        {
            const RXMLElement* vtxShapeAnimElem = &vtxShapeAnimsElem->nodes[iMember];
            const std::string boneName = vtxShapeAnimElem->GetAttribute("base_name");
            GetSceneBoneByNameAndClearAnim(fshBones, boneName, selXforms, false);
        }
        if (fshBones.empty())
        {
            DisplayImportWarning("対応するボーンが 1 つもありません。", "No bone is matched.");
            return status;
        }
        pBones = &fshBones;
    }

    //-----------------------------------------------------------------------------
    // vertex_shape_anim_array 要素を解析してアニメーションカーブを作成します。
    if (vtxShapeAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < vtxShapeAnimsElem->nodes.size(); ++iMember)
        {
            status = ParseVtxShapeAnim(&vtxShapeAnimsElem->nodes[iMember], startFrame, *pBones, fshFile);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + fshPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 指定した名前のカメラをシーンから取得します。
//!
//! @param[in] camName カメラ名です。
//!
//! @return カメラの transform ノードの DAG パスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetSceneCameraByName(const std::string& camName)
{
    MStringArray xforms;
    MGlobal::executeCommand(MString("ls -typ transform \"") + camName.c_str() + "\"", xforms);
    if (xforms.length() == 0)
    {
        MGlobal::executeCommand(MString("ls -typ transform \"*:") + camName.c_str() + "\"", xforms);
    }
    if (xforms.length() >= 2)
    {
        DisplayImportWarning("同名の transform ノードが複数存在します: {0}",
            "More than one transform node is found: {0}", camName);
    }

    if (xforms.length() == 1)
    {
        MDagPath xformPath = GetDagPathByName(xforms[0]);
        if (xformPath.isValid())
        {
            MDagPath shapePath = xformPath;
            if (shapePath.extendToShape())
            {
                if (shapePath.node().apiType() == MFn::kCamera)
                {
                    return xformPath;
                }
            }
        }
    }
    return MDagPath();
}

//-----------------------------------------------------------------------------
//! @brief カメラの lookAt ノード、エイムロケータ、アップロケータを取得します。
//!
//! @param[in] lookAtPath lookAt ノードの DAG パスを格納します。
//! @param[in] aimPath aim ロケータの DAG パスを格納します。
//! @param[in] upPath up ロケータの DAG パスを格納します。
//! @param[in] xformPath transform ノードの DAG パスです。
//-----------------------------------------------------------------------------
static void GetCameraLookAtAim(
    MDagPath& lookAtPath,
    MDagPath& aimPath,
    MDagPath& upPath,
    const MDagPath& xformPath
)
{
    lookAtPath = aimPath = upPath = MDagPath();
    MPlugArray plugArray;
    MFnTransform(xformPath).findPlug("rx").connectedTo(plugArray, true, false);
    if (plugArray.length() != 0 &&
        plugArray[0].node().apiType() == MFn::kLookAt)
    {
        MDagPath::getAPathTo(plugArray[0].node(), lookAtPath);
        MFnDagNode lookAtFn(lookAtPath);
        MPlug targetPlug = lookAtFn.findPlug("target");
        if (targetPlug.numElements() == 1)
        {
            MPlug ttxPlug = targetPlug.elementByLogicalIndex(0).child(0).child(0);
            ttxPlug.connectedTo(plugArray, true, false);
            for (int iSrc = 0; iSrc < static_cast<int>(plugArray.length()); ++iSrc)
            {
                MObject obj = plugArray[iSrc].node();
                if (obj.apiType() == MFn::kTransform)
                {
                    MDagPath::getAPathTo(obj, aimPath);
                    break;
                }
            }
        }
        if (aimPath.isValid())
        {
            MPlug wumPlug = lookAtFn.findPlug("worldUpMatrix");
            wumPlug.connectedTo(plugArray, true, false);
            if (plugArray.length() > 0 &&
                plugArray[0].node().apiType() == MFn::kTransform)
            {
                MDagPath::getAPathTo(plugArray[0].node(), upPath);
            }
        }
    }
    //cerr << "lookAt: " << lookAtPath.partialPathName() << ", " << aimPath.partialPathName() << ", " << upPath.partialPathName() << endl;
}

//-----------------------------------------------------------------------------
//! @brief カメラのアニメーションを削除します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] shapePath シェイプノードの DAG パスです。
//! @param[in] lookAtPath lookAt ノードの DAG パスです。
//! @param[in] aimPath aim ロケータの DAG パスです。
//-----------------------------------------------------------------------------
static void ClearCameraAnim(
    const MDagPath& xformPath,
    const MDagPath& shapePath,
    const MDagPath& lookAtPath,
    const MDagPath& aimPath
)
{
    MFnTransform xformFn(xformPath);
    MFnCamera camFn(shapePath);

    MString cmd;
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("translateX"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("translateY"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("translateZ"));
    if (aimPath.isValid())
    {
        MFnTransform aimFn(aimPath);
        AddAnimCurveClearCmd(cmd, aimFn.findPlug("translateX"));
        AddAnimCurveClearCmd(cmd, aimFn.findPlug("translateY"));
        AddAnimCurveClearCmd(cmd, aimFn.findPlug("translateZ"));
    }
    else
    {
        AddAnimCurveClearCmd(cmd, xformFn.findPlug("rotateX"));
        AddAnimCurveClearCmd(cmd, xformFn.findPlug("rotateY"));
        AddAnimCurveClearCmd(cmd, xformFn.findPlug("rotateZ"));
    }
    if (lookAtPath.isValid())
    {
        MFnTransform lookAtFn(lookAtPath);
        AddAnimCurveClearCmd(cmd, lookAtFn.findPlug("twist"));
    }
    AddAnimCurveClearCmd(cmd, camFn.findPlug("horizontalFilmAperture"));
    AddAnimCurveClearCmd(cmd, camFn.findPlug("verticalFilmAperture"));
    AddAnimCurveClearCmd(cmd, camFn.findPlug("nearClipPlane"));
    AddAnimCurveClearCmd(cmd, camFn.findPlug("farClipPlane"));
    AddAnimCurveClearCmd(cmd, camFn.findPlug("orthographicWidth"));
    AddAnimCurveClearCmd(cmd, camFn.findPlug("focalLength"));
    if (cmd.length() != 0)
    {
        //cerr << "delete curves:" << endl << cmd << endl;
        MGlobal::executeCommand(cmd);
    }
}

//-----------------------------------------------------------------------------
//! @brief カメラの垂直方向の画角を焦点距離に変換します。
//!
//! @param[in] fovy 垂直方向の画角 [degree] です。
//! @param[in] verticalFilmAperture 垂直フィルムアパーチャ [in] です。
//!
//! @return 焦点距離 [mm] を返します。
//-----------------------------------------------------------------------------
static double GetFocalLengthFromFovy(const double fovy, const double verticalFilmAperture)
{
    // fovy = 2 * atan(verticalFilmAperture * 25.4 * 0.5 / focalLength) // 1in = 25.4mm
    // tan(persp_fovy * 0.5) = verticalFilmAperture * 12.7 / focalLength
    // focalLength = verticalFilmAperture * 12.7 / tan(persp_fovy * 0.5)
    const double rad = fovy * 0.5 * R_M_DEG_TO_RAD;
    return (rad != 0.0) ? verticalFilmAperture * 12.7 / tan(rad) : 35.0;
}

static double GetFocalLengthFromFovyFunc(const double srcVal, const void* pUser)
{
    return GetFocalLengthFromFovy(srcVal, *reinterpret_cast<const double*>(pUser));
}

//-----------------------------------------------------------------------------
//! @brief カメラの垂直方向の画角アニメーションを焦点距離アニメーションに変換します。
//!        アニメーションを接続した状態で呼びます。
//!
//! @param[in,out] focalPlug 焦点距離アトリビュートのプラグです。
//! @param[in] magnify 移動値に掛ける倍率です。
//! @param[in] verticalFilmAperture 垂直フィルムアパーチャ [in] です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ConvertCameraFovyAnim(
    MPlug& focalPlug,
    const double magnify,
    const double verticalFilmAperture
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 垂直方向の画角のアニメーション値を取得します。
    int startFrame;
    const int frameCount = GetAnimFrameRange(startFrame);
    RFloatArray focals;
    GetFullAnimValues(focals, focalPlug, startFrame, frameCount, 1.0f, 0.0f);

    //-----------------------------------------------------------------------------
    // 垂直方向の画角のアニメーションカーブを削除します。
    ClearAnimCurve(focalPlug);

    //-----------------------------------------------------------------------------
    // 垂直方向の画角を焦点距離に変換します。
    for (int iFrame = 0; iFrame < frameCount; ++iFrame)
    {
        focals[iFrame] = static_cast<float>(GetFocalLengthFromFovy(focals[iFrame], verticalFilmAperture));
    }

    //-----------------------------------------------------------------------------
    // 焦点距離のアニメーションカーブを作成します。
    const double TRS_TOL = 0.01;
    status = CreateAnimCurveFromValues(focalPlug, focals, startFrame, magnify, TRS_TOL, false);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief カメラアニメーションを解析してカメラとアニメーションカーブを作成します。
//!
//! @param[in] camAnimElem カメラアニメーション要素です。
//! @param[in] magnify 移動値に掛ける倍率です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] fsnFile fsn ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseCameraAnim(
    const RXMLElement* camAnimElem,
    const double magnify,
    const int startFrame,
    const ImpIntermediateFile& fsnFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 同名のカメラがシーン中に存在しなければ作成します。
    const std::string camName = camAnimElem->GetAttribute("camera_name");
    const bool isAim = (camAnimElem->GetAttribute("rotate_mode") == "aim");
    const bool isOrtho = (camAnimElem->GetAttribute("projection_mode") == "ortho");
    MDagPath xformPath = GetSceneCameraByName(camName);
    MDagPath shapePath;
    MDagPath lookAtPath;
    MDagPath aimPath;
    MDagPath upPath;
    if (xformPath.isValid())
    {
        //-----------------------------------------------------------------------------
        // シーン中に存在すればアニメーションを削除します。
        shapePath = xformPath;
        shapePath.extendToShape();
        GetCameraLookAtAim(lookAtPath, aimPath, upPath, xformPath);
        if ((!isAim && aimPath.isValid()) || upPath.isValid())
        {
            // シーン中のカメラのタイプが異なる場合はカメラを一度削除します。
            MDagPath delPath = (lookAtPath.isValid()) ? lookAtPath : xformPath;
            if (!DeleteDagNodeIfPossible(delPath))
            {
                DisplayImportWarning("既存のカメラのタイプが異なります。処理をスキップします: {0}",
                    "Camera type is different (skipped): {0}", xformPath.partialPathName().asChar());
                return status;
            }
            xformPath = shapePath = lookAtPath = aimPath = upPath = MDagPath();
        }
        else
        {
            ClearCameraAnim(xformPath, shapePath, lookAtPath, aimPath);
        }
    }

    if (!xformPath.isValid())
    {
        //-----------------------------------------------------------------------------
        // カメラを作成します。
        MString cmd;
        MGlobal::executeCommand((!isAim) ? "performCameraOnly 2" : "performCameraAim 2", cmd);
        MString ret;
        MGlobal::executeCommand(cmd, ret);
        xformPath = GetDagPathByName(ret);
        shapePath = xformPath;
        shapePath.extendToShape();
        MFnDependencyNode(xformPath.node()).setName(camName.c_str());
        if (isAim)
        {
            GetCameraLookAtAim(lookAtPath, aimPath, upPath, xformPath);
        }
    }

    //-----------------------------------------------------------------------------
    // アニメーションしないアトリビュートを設定します。
    MFnTransform xformFn(xformPath);
    if (!isAim)
    {
        xformFn.setRotationOrder(MTransformationMatrix::kZXY, true);
    }
    MFnCamera camFn(shapePath);
    camFn.setIsOrtho(isOrtho);
    const double verticalFilmAperture = 1.0;
    camFn.findPlug("verticalFilmAperture").setValue(verticalFilmAperture);
        // ↑水平フィルムアパーチャアトリビュートが aspect と 1:1 の関係になるようにします。
    const double magnifiedVfa = verticalFilmAperture * magnify;
    if (isOrtho)
    {
        // 正射投影幅アトリビュートが ortho_height と 1:1 の関係になるようにします。
        camFn.setFilmFit(MFnCamera::kVerticalFilmFit);
    }

    //-----------------------------------------------------------------------------
    // 各アトリビュートにアニメーションカーブを接続します。
    MFnTransform lookAtFn((lookAtPath.isValid()) ? lookAtPath : xformPath);
    MFnTransform aimFn((aimPath.isValid()) ? aimPath : xformPath);
    for (size_t iCurve = 0; iCurve < camAnimElem->nodes.size(); ++iCurve)
    {
        const RXMLElement* targetElem = &camAnimElem->nodes[iCurve];
        if (targetElem->name == "user_data_array")
        {
            ParseUserDatas(xformPath.partialPathName(), targetElem);
            continue;
        }

        const std::string targetName = targetElem->GetAttribute("target");
        MPlug dstPlug;
        if      (targetName == "position_x") dstPlug = xformFn.findPlug("translateX");
        else if (targetName == "position_y") dstPlug = xformFn.findPlug("translateY");
        else if (targetName == "position_z") dstPlug = xformFn.findPlug("translateZ");
        else if (targetName == "aim_x") dstPlug = aimFn.findPlug("translateX");
        else if (targetName == "aim_y") dstPlug = aimFn.findPlug("translateY");
        else if (targetName == "aim_z") dstPlug = aimFn.findPlug("translateZ");
        else if (targetName == "twist") dstPlug = lookAtFn.findPlug("twist");
        else if (targetName == "rotate_x") dstPlug = xformFn.findPlug("rotateX");
        else if (targetName == "rotate_y") dstPlug = xformFn.findPlug("rotateY");
        else if (targetName == "rotate_z") dstPlug = xformFn.findPlug("rotateZ");
        else if (targetName == "aspect") dstPlug = camFn.findPlug("horizontalFilmAperture");
        else if (targetName == "near"  ) dstPlug = camFn.findPlug("nearClipPlane");
        else if (targetName == "far"   ) dstPlug = camFn.findPlug("farClipPlane");
        else if (targetName == "ortho_height") dstPlug = camFn.findPlug("orthographicWidth");
        else if (targetName == "persp_fovy") dstPlug = camFn.findPlug("focalLength");
        if (!dstPlug.isNull())
        {
            double valueScale = 1.0;
            if (targetName.find("position_") == 0 ||
                targetName.find("aim_"     ) == 0 ||
                targetName == "near"              ||
                targetName == "far"               ||
                targetName == "ortho_height")
            {
                valueScale = magnify;
            }
            else if (targetName == "twist" ||
                     targetName.find("rotate_") == 0)
            {
                valueScale = R_M_DEG_TO_RAD;
            }
            // focalLength の最小値は 2.5 なので fovy は最初はラジアンではなく度数で処理します。
            status = CreateAnimCurveFromElement(dstPlug, targetElem, false, startFrame, valueScale, 0.0, false, fsnFile);
            CheckStatus(status);

            //-----------------------------------------------------------------------------
            // fovy を焦点距離に変換します。
            if (targetName == "persp_fovy")
            {
                MPlugArray plugArray;
                dstPlug.connectedTo(plugArray, true, false);
                if (plugArray.length() != 0)
                {
                    MObject curveObj = plugArray[0].node();
                    if (curveObj.apiType() == MFn::kAnimCurveTimeToUnitless)
                    {
                        // エルミート補間の接線を変換するより
                        // ベイクした値からアニメーションカーブを作成したほうが誤差が小さいので、
                        // エルミート補間以外の場合だけ、キーの値を変換します。
                        //
                        // 次のデータでテストした結果
                        // フレーム:     0      30      60
                        // 焦点距離:    35      50      45
                        //
                        // 30 フレームにおけるインポート後の焦点距離は
                        // 接線の変換だと   49.78
                        // ベイクだと       49.85
                        //
                        const bool isHermite = (targetElem->FindElement("hermite_curve", false) != NULL);
                        if (isHermite)
                        {
                            status = ConvertCameraFovyAnim(dstPlug, magnify, verticalFilmAperture);
                            CheckStatus(status);
                        }
                        else
                        {
                            ConvertAnimCurveValueTangent(curveObj,
                                isHermite, GetFocalLengthFromFovyFunc, &magnifiedVfa);
                        }
                    }
                }
                else // アニメーションカーブが接続されていない場合
                {
                    double fovy;
                    dstPlug.getValue(fovy);
                    dstPlug.setValue(GetFocalLengthFromFovy(fovy, magnifiedVfa));
                }
            }
        }
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 指定した名前のライトをシーンから取得します。
//!
//! @param[in] lgtName ライト名です。
//!
//! @return ライトの transform ノードの DAG パスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetSceneLightByName(const std::string& lgtName)
{
    MStringArray xforms;
    MGlobal::executeCommand(MString("ls -typ transform \"") + lgtName.c_str() + "\"", xforms);
    if (xforms.length() == 0)
    {
        MGlobal::executeCommand(MString("ls -typ transform \"*:") + lgtName.c_str() + "\"", xforms);
    }
    if (xforms.length() >= 2)
    {
        DisplayImportWarning("同名の transform ノードが複数存在します: {0}",
            "More than one transform node is found: {0}", lgtName);
    }

    if (xforms.length() == 1)
    {
        MDagPath xformPath = GetDagPathByName(xforms[0]);
        if (xformPath.isValid())
        {
            MDagPath shapePath = xformPath;
            if (shapePath.extendToShape())
            {
                if (shapePath.node().hasFn(MFn::kLight))
                {
                    return xformPath;
                }
            }
        }
    }
    return MDagPath();
}

//-----------------------------------------------------------------------------
//! @brief ライトのアニメーションを削除します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] shapePath シェイプノードの DAG パスです。
//-----------------------------------------------------------------------------
static void ClearLightAnim(const MDagPath& xformPath, const MDagPath& shapePath)
{
    MFnTransform xformFn(xformPath);
    MFnDependencyNode shapeFn(shapePath.node());
    const MFn::Type shapeType = shapePath.node().apiType();

    MString cmd;
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("visibility"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("translateX"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("translateY"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("translateZ"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("rotateX"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("rotateY"));
    AddAnimCurveClearCmd(cmd, xformFn.findPlug("rotateZ"));

    MPlug distAttnStartPlug = FindPlugQuiet(shapeFn, "NW4F_AttnStartDist", NULL);
    if (!distAttnStartPlug.isNull())
    {
        AddAnimCurveClearCmd(cmd, distAttnStartPlug);
    }
    MPlug distAttnEndPlug = FindPlugQuiet(shapeFn, "NW4F_AttnEndDist", NULL);
    if (!distAttnEndPlug.isNull())
    {
        AddAnimCurveClearCmd(cmd, distAttnEndPlug);
    }

    if (shapeType == MFn::kSpotLight)
    {
        AddAnimCurveClearCmd(cmd, shapeFn.findPlug("coneAngle"));
        AddAnimCurveClearCmd(cmd, shapeFn.findPlug("penumbraAngle"));
    }

    AddAnimCurveClearCmd(cmd, shapeFn.findPlug("colorR"));
    AddAnimCurveClearCmd(cmd, shapeFn.findPlug("colorG"));
    AddAnimCurveClearCmd(cmd, shapeFn.findPlug("colorB"));

    if (cmd.length() != 0)
    {
        //cerr << "delete curves:" << endl << cmd << endl;
        MGlobal::executeCommand(cmd);
    }
}

//-----------------------------------------------------------------------------
//! @brief ライトの方向／注視点アニメーションを回転アニメーションに変換します。
//!        transform ノードの回転アトリビュートに方向／注視点アニメーションを接続した状態で呼びます。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] shapePath シェイプノードの DAG パスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ConvertLightDirectionAnim(const MDagPath& xformPath, const MDagPath& shapePath)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // ライトの種類を取得します。
    const bool isSpot = (shapePath.node().apiType() == MFn::kSpotLight);

    //-----------------------------------------------------------------------------
    // 方向／注視点のアニメーション値を取得します。
    int startFrame;
    const int frameCount = GetAnimFrameRange(startFrame);

    MFnTransform xformFn(xformPath);
    MPlug txPlug = xformFn.findPlug("translateX");
    MPlug tyPlug = xformFn.findPlug("translateY");
    MPlug tzPlug = xformFn.findPlug("translateZ");
    MPlug rxPlug = xformFn.findPlug("rotateX");
    MPlug ryPlug = xformFn.findPlug("rotateY");
    MPlug rzPlug = xformFn.findPlug("rotateZ");

    RFloatArray txs;
    RFloatArray tys;
    RFloatArray tzs;
    if (isSpot)
    {
        GetFullAnimValues(txs, txPlug, startFrame, frameCount, 1.0f, 0.0f);
        GetFullAnimValues(tys, tyPlug, startFrame, frameCount, 1.0f, 0.0f);
        GetFullAnimValues(tzs, tzPlug, startFrame, frameCount, 1.0f, 0.0f);
    }
    RFloatArray rxs;
    RFloatArray rys;
    RFloatArray rzs;
    GetFullAnimValues(rxs, rxPlug, startFrame, frameCount, 1.0f, 0.0f);
    GetFullAnimValues(rys, ryPlug, startFrame, frameCount, 1.0f, 0.0f);
    GetFullAnimValues(rzs, rzPlug, startFrame, frameCount, 1.0f, 0.0f);

    //-----------------------------------------------------------------------------
    // 方向／注視点のアニメーションカーブを削除します。
    ClearAnimCurve(rxPlug);
    ClearAnimCurve(ryPlug);
    ClearAnimCurve(rzPlug);

    //-----------------------------------------------------------------------------
    // 方向／注視点を回転と距離に変換します。
    // Z 軸はエクスポート時と同じ向きになりますが、X 軸と Y 軸は保持されません。
    RFloatArray dists;
    dists.resize(frameCount);
    for (int iFrame = 0; iFrame < frameCount; ++iFrame)
    {
        MVector dir;
        float dist = 1.0f;
        if (isSpot)
        {
            dir = MVector(
                rxs[iFrame] - txs[iFrame],
                rys[iFrame] - tys[iFrame],
                rzs[iFrame] - tzs[iFrame]);
            dist = static_cast<float>(dir.length());
            dir.normalize();
        }
        else
        {
            dir = MVector(rxs[iFrame], rys[iFrame], rzs[iFrame]);
        }
        MQuaternion quat(MVector(0.0, 0.0, -1.0), dir);
        MEulerRotation euler = quat.asEulerRotation();
        if (euler.order != MEulerRotation::kXYZ)
        {
            euler.reorderIt(MEulerRotation::kXYZ);
        }
        rxs[iFrame] = static_cast<float>(euler.x);
        rys[iFrame] = static_cast<float>(euler.y);
        rzs[iFrame] = static_cast<float>(euler.z);
        dists[iFrame] = dist;
    }

    //-----------------------------------------------------------------------------
    // 回転のアニメーションカーブを作成します。
    const double ROT_TOL = 0.1 * R_M_DEG_TO_RAD;
    status = CreateAnimCurveFromValues(rxPlug, rxs, startFrame, 1.0, ROT_TOL, false);
    CheckStatus(status);
    status = CreateAnimCurveFromValues(ryPlug, rys, startFrame, 1.0, ROT_TOL, false);
    CheckStatus(status);
    status = CreateAnimCurveFromValues(rzPlug, rzs, startFrame, 1.0, ROT_TOL, false);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 回転順序を XYZ にします。
    xformFn.setRotationOrder(MTransformationMatrix::kXYZ, false);

    //-----------------------------------------------------------------------------
    // イルミネーションの中心のアニメーションカーブを作成します。
    if (isSpot)
    {
        const double TRS_TOL = 0.01;
        MPlug coiPlug = MFnDependencyNode(shapePath.node()).findPlug("centerOfIllumination");
        status = CreateAnimCurveFromValues(coiPlug, dists, startFrame, 1.0, TRS_TOL, false);
        CheckStatus(status);
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ライトの角度減衰アニメーションを変換します。
//!        円錐角度アトリビュートに angle_attn_start アニメーション、
//!        周縁部の角度アトリビュートに angle_attn_end アニメーションを接続した状態で呼びます。
//!
//! @param[in] shapePath シェイプノードの DAG パスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ConvertLightAngleAttnAnim(const MDagPath& shapePath)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 円錐角度と周縁部の角度のアニメーション値を取得します。
    int startFrame;
    const int frameCount = GetAnimFrameRange(startFrame);

    MFnDependencyNode shapeFn(shapePath.node());
    MPlug caPlug = shapeFn.findPlug("coneAngle");
    MPlug paPlug = shapeFn.findPlug("penumbraAngle");
    RFloatArray cas;
    RFloatArray pas;
    GetFullAnimValues(cas, caPlug, startFrame, frameCount, 1.0f, 0.0f);
    GetFullAnimValues(pas, paPlug, startFrame, frameCount, 1.0f, 0.0f);

    //-----------------------------------------------------------------------------
    // 周縁部の角度のアニメーションカーブを削除します。
    ClearAnimCurve(paPlug);

    //-----------------------------------------------------------------------------
    // angle_attn_end を周縁部の角度に変換します。
    for (int iFrame = 0; iFrame < frameCount; ++iFrame)
    {
        pas[iFrame] -= cas[iFrame] * 0.5f;
    }

    //-----------------------------------------------------------------------------
    // 周縁部の角度のアニメーションカーブを作成します。
    const double ROT_TOL = 0.1 * R_M_DEG_TO_RAD;
    status = CreateAnimCurveFromValues(paPlug, pas, startFrame, 1.0, ROT_TOL, false);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ライトアニメーションを解析してライトとアニメーションカーブを作成します。
//!
//! @param[in] lgtAnimElem ライトアニメーション要素です。
//! @param[in] magnify 移動値に掛ける倍率です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] fsnFile fsn ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseLightAnim(
    const RXMLElement* lgtAnimElem,
    const double magnify,
    const int startFrame,
    const ImpIntermediateFile& fsnFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // サポートしていないタイプのライトならスキップします。
    const std::string lgtName = lgtAnimElem->GetAttribute("light_name");
    const std::string lgtType = lgtAnimElem->GetAttribute("type");
    if (lgtType != "ambient"     &&
        lgtType != "directional" &&
        lgtType != "point"       &&
        lgtType != "spot")
    {
        DisplayImportWarning("サポートしていないタイプのライトです。処理をスキップします: {0}",
            "Unsupported light type (skipped): {0}", lgtName + " (" + lgtType + ")");
        return status;
    }

    //-----------------------------------------------------------------------------
    // 同名のライトがシーン中に存在しなければ作成します。
    MDagPath xformPath = GetSceneLightByName(lgtName);
    MDagPath shapePath;
    if (xformPath.isValid())
    {
        //-----------------------------------------------------------------------------
        // シーン中にライトが存在すればアニメーションを削除します。
        shapePath = xformPath;
        shapePath.extendToShape();
        const MFn::Type shapeType = shapePath.node().apiType();
        if ((lgtType == "ambient"     && shapeType != MFn::kAmbientLight    ) ||
            (lgtType == "directional" && shapeType != MFn::kDirectionalLight) ||
            (lgtType == "point"       && shapeType != MFn::kPointLight      ) ||
            (lgtType == "spot"        && shapeType != MFn::kSpotLight       ))
        {
            // シーン中のライトのタイプが異なる場合はライトを一度削除します。
            if (!DeleteDagNodeIfPossible(xformPath))
            {
                DisplayImportWarning("既存のライトのタイプが異なります。処理をスキップします: {0}",
                    "Light type is different (skipped): {0}", xformPath.partialPathName().asChar());
                return status;
            }
            xformPath = shapePath = MDagPath();
        }
        else
        {
            ClearLightAnim(xformPath, shapePath);
        }
    }

    if (!xformPath.isValid())
    {
        //-----------------------------------------------------------------------------
        // ライトを作成します。
        MString cmd0;
        if (lgtType == "directional")
        {
            cmd0 = "performDirectionalLight(2)";
        }
        else if (lgtType == "point")
        {
            cmd0 = "performPointLight(2)";
        }
        else if (lgtType == "spot")
        {
            cmd0 = "performSpotLight(2)";
        }
        else // ambient
        {
            cmd0 = "performAmbientLight(2)";
        }
        MString cmd1;
        MGlobal::executeCommand(cmd0, cmd1);
        MString ret;
        MGlobal::executeCommand(cmd1, ret); // ret は空文字
        xformPath = GetSelectedNodeDagPath();
        shapePath = xformPath;
        shapePath.extendToShape();
        MFnDependencyNode(xformPath.node()).setName(lgtName.c_str());
    }

    //-----------------------------------------------------------------------------
    // カスタムアトリビュートがなければ追加します。
    if (lgtType == "point" || lgtType == "spot")
    {
        MString shapeName = shapePath.partialPathName();
        MGlobal::executeCommand(
            "if (!`attributeQuery -n " + shapeName + " -ex \"NW4F_AttnStartDist\"`)\n" +
            "{\n" +
            "   addAttr -ln \"NW4F_AttnStartDist\" -at \"float\" -dv 0.0 -k true " + shapeName + ";\n" +
            "}\n" +
            "if (!`attributeQuery -n " + shapeName + " -ex \"NW4F_AttnEndDist\"`)\n" +
            "{\n" +
            "   addAttr -ln \"NW4F_AttnEndDist\" -at \"float\" -dv 10.0 -k true " + shapeName + ";\n" +
            "}\n");
    }

    //-----------------------------------------------------------------------------
    // 各アトリビュートにアニメーションカーブを接続します。
    bool usesDirection = false;
    bool usesAim = false;
    bool usesDistAttn = false;
    bool usesAngleAttn = false;
    MFnTransform xformFn(xformPath);
    MFnDependencyNode shapeFn(shapePath.node());
    for (size_t iCurve = 0; iCurve < lgtAnimElem->nodes.size(); ++iCurve)
    {
        const RXMLElement* targetElem = &lgtAnimElem->nodes[iCurve];
        if (targetElem->name == "user_data_array")
        {
            ParseUserDatas(xformPath.partialPathName(), targetElem);
            continue;
        }

        const std::string targetName = targetElem->GetAttribute("target");
        MPlug dstPlug;
        if      (targetName == "enable") dstPlug = xformFn.findPlug("visibility");
        else if (targetName == "position_x") dstPlug = xformFn.findPlug("translateX");
        else if (targetName == "position_y") dstPlug = xformFn.findPlug("translateY");
        else if (targetName == "position_z") dstPlug = xformFn.findPlug("translateZ");
        else if (targetName == "direction_x") dstPlug = xformFn.findPlug("rotateX");
        else if (targetName == "direction_y") dstPlug = xformFn.findPlug("rotateY");
        else if (targetName == "direction_z") dstPlug = xformFn.findPlug("rotateZ");
        else if (targetName == "aim_x") dstPlug = xformFn.findPlug("rotateX");
        else if (targetName == "aim_y") dstPlug = xformFn.findPlug("rotateY");
        else if (targetName == "aim_z") dstPlug = xformFn.findPlug("rotateZ");
        else if (targetName == "dist_attn_start") dstPlug = FindPlugQuiet(shapeFn, "NW4F_AttnStartDist", NULL);
        else if (targetName == "dist_attn_end"  ) dstPlug = FindPlugQuiet(shapeFn, "NW4F_AttnEndDist"  , NULL);
        else if (targetName == "angle_attn_start") dstPlug = shapeFn.findPlug("coneAngle");
        else if (targetName == "angle_attn_end") dstPlug = shapeFn.findPlug("penumbraAngle");
        else if (targetName == "color0_r") dstPlug = shapeFn.findPlug("colorR");
        else if (targetName == "color0_g") dstPlug = shapeFn.findPlug("colorG");
        else if (targetName == "color0_b") dstPlug = shapeFn.findPlug("colorB");
        if (!dstPlug.isNull())
        {
            double valueScale = 1.0;
            if (targetName.find("position_" ) == 0 ||
                targetName.find("aim_"      ) == 0 ||
                targetName.find("dist_attn_") == 0)
            {
                valueScale = magnify;
            }
            else if (targetName == "angle_attn_start")
            {
                valueScale = 2.0 * R_M_DEG_TO_RAD;
            }
            else if (targetName == "angle_attn_end")
            {
                valueScale = R_M_DEG_TO_RAD;
            }
            status = CreateAnimCurveFromElement(dstPlug, targetElem, false, startFrame, valueScale, 0.0, false, fsnFile);
            CheckStatus(status);

            if (targetName.find("direction_") == 0)
            {
                usesDirection = true;
            }
            else if (targetName.find("aim_") == 0)
            {
                usesAim = true;
            }
            else if (targetName.find("dist_attn_") == 0)
            {
                usesDistAttn = true;
            }
            else if (targetName.find("angle_attn_") == 0)
            {
                usesAngleAttn = true;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 方向／注視点アニメーションを回転アニメーションに変換します。
    if (usesDirection || usesAim)
    {
        status = ConvertLightDirectionAnim(xformPath, shapePath);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // 角度減衰アニメーションを変換します。
    if (usesAngleAttn)
    {
        status = ConvertLightAngleAttnAnim(shapePath);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // decayRate を設定します。
    if (lgtType == "point" || lgtType == "spot")
    {
        shapeFn.findPlug("decayRate").setValue((usesDistAttn) ? 1 : 0);
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief フォグのシェイプノードを取得します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//!
//! @return シェイプノードの DAG パスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetFogShapePath(const MDagPath& xformPath)
{
    MFnTransform xformFn(xformPath);
    for (int iChild = 0; iChild < static_cast<int>(xformFn.childCount()); ++iChild)
    {
        const MObject childObj = xformFn.child(iChild);
        if (childObj.apiType() == MFn::kEnvFogShape)
        {
            MDagPath shapePath;
            if (MDagPath::getAPathTo(childObj, shapePath))
            {
                return shapePath;
            }
        }
    }
    return MDagPath();
}

//-----------------------------------------------------------------------------
//! @brief 指定した名前のフォグをシーンから取得します。
//!
//! @param[in] fogName フォグ名です。
//!
//! @return フォグの transform ノードの DAG パスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetSceneFogByName(const std::string& fogName)
{
    MStringArray xforms;
    MGlobal::executeCommand(MString("ls -typ transform \"") + fogName.c_str() + "\"", xforms);
    if (xforms.length() == 0)
    {
        MGlobal::executeCommand(MString("ls -typ transform \"*:") + fogName.c_str() + "\"", xforms);
    }
    if (xforms.length() >= 2)
    {
        DisplayImportWarning("同名の transform ノードが複数存在します: {0}",
            "More than one transform node is found: {0}", fogName);
    }

    if (xforms.length() == 1)
    {
        MDagPath xformPath = GetDagPathByName(xforms[0]);
        if (xformPath.isValid() &&
            GetFogShapePath(xformPath).isValid())
        {
            return xformPath;
        }
    }
    return MDagPath();
}

//-----------------------------------------------------------------------------
//! @brief フォグの envFog ノードを取得します。
//!
//! @param[out] pSgObj 環境フォグのシェーディンググループのオブジェクトを格納します。
//! @param[in] shapePath シェイプの DAG パスです。
//!
//! @return envFog ノードのオブジェクトを返します。
//-----------------------------------------------------------------------------
static MObject GetEnvFog(MObject* pSgObj, const MDagPath& shapePath)
{
    MFnDependencyNode shapeFn(shapePath.node());
    MPlug instObjPlug = shapeFn.findPlug("instObjGroups");
    MPlugArray plugArray;
    instObjPlug[0].connectedTo(plugArray, false, true);
    if (plugArray.length() != 0)
    {
        MObject sgObj = plugArray[0].node();
        MFnDependencyNode(sgObj).findPlug("volumeShader").connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            MObject envFogObj = plugArray[0].node();
            if (envFogObj.apiType() == MFn::kEnvFogMaterial)
            {
                if (pSgObj != nullptr)
                {
                    *pSgObj = sgObj;
                }
                return envFogObj;
            }
        }
    }
    return MObject::kNullObj;
}

//-----------------------------------------------------------------------------
//! @brief フォグのアニメーションを削除します。
//!
//! @param[in] envFogObj envFog ノードのオブジェクトです。
//-----------------------------------------------------------------------------
static void ClearFogAnim(const MObject& envFogObj)
{
    MFnDependencyNode envFogFn(envFogObj);
    MString cmd;
    AddAnimCurveClearCmd(cmd, envFogFn.findPlug("fogNearDistance"));
    AddAnimCurveClearCmd(cmd, envFogFn.findPlug("fogFarDistance"));
    AddAnimCurveClearCmd(cmd, envFogFn.findPlug("colorR"));
    AddAnimCurveClearCmd(cmd, envFogFn.findPlug("colorG"));
    AddAnimCurveClearCmd(cmd, envFogFn.findPlug("colorB"));
    if (cmd.length() != 0)
    {
        //cerr << "delete curves:" << endl << cmd << endl;
        MGlobal::executeCommand(cmd);
    }
}

//-----------------------------------------------------------------------------
//! @brief フォグアニメーションを解析してライトとアニメーションカーブを作成します。
//!
//! @param[in] fogAnimElem フォグアニメーション要素です。
//! @param[in] magnify 移動値に掛ける倍率です。
//! @param[in] startFrame 開始フレームです。
//! @param[in] fsnFile fsn ファイルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFogAnim(
    const RXMLElement* fogAnimElem,
    const double magnify,
    const int startFrame,
    const ImpIntermediateFile& fsnFile
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 同名のフォグがシーン中に存在しなければ作成します。
    const std::string fogName = fogAnimElem->GetAttribute("fog_name");
    MDagPath xformPath = GetSceneFogByName(fogName);
    MDagPath shapePath;
    MObject envFogObj;
    MObject envFogSgObj;
    if (xformPath.isValid())
    {
        //-----------------------------------------------------------------------------
        // シーン中にフォグが存在すればアニメーションを削除します。
        shapePath = GetFogShapePath(xformPath);
        envFogObj = GetEnvFog(&envFogSgObj, shapePath);
        if (envFogObj.isNull())
        {
            DisplayImportWarning("環境フォグのマテリアル（envFog ノード）が存在しません。処理をスキップします: {0}",
                "No fog material (skipped): {0}", xformPath.partialPathName().asChar());
            return status;
        }
        ClearFogAnim(envFogObj);
    }

    if (!xformPath.isValid())
    {
        //-----------------------------------------------------------------------------
        // フォグを作成します。
        MObject renderObj = GetNodeObjectByName(":defaultRenderGlobals");
        MPlug fgPlug = MFnDependencyNode(renderObj).findPlug("fogGeometry");
        MPlugArray plugArray;
        fgPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            // アクティブなフォグの接続を解除します。
            MGlobal::executeCommand("disconnectAttr " + plugArray[0].name() + " " + fgPlug.name(), true);
        }
        MGlobal::executeCommand("defaultNavigation -createNew -destination \":defaultRenderGlobals.fogGeometry\"", true);
            // ↑作成されたフォグのシェイプノードが fgPlug に接続されます。
        fgPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() == 0)
        {
            return status;
        }
        MObject shapeObj = plugArray[0].node();
        MDagPath::getAPathTo(shapeObj, shapePath);
        xformPath = shapePath;
        xformPath.pop();
        MFnDependencyNode(xformPath.node()).setName(fogName.c_str());
        envFogObj = GetEnvFog(&envFogSgObj, shapePath);
    }

    //-----------------------------------------------------------------------------
    // 各アトリビュートにアニメーションカーブを接続します。
    MFnDependencyNode envFogFn(envFogObj);
    for (size_t iCurve = 0; iCurve < fogAnimElem->nodes.size(); ++iCurve)
    {
        const RXMLElement* targetElem = &fogAnimElem->nodes[iCurve];
        if (targetElem->name == "user_data_array")
        {
            ParseUserDatas(MFnDependencyNode(envFogSgObj).name(), targetElem);
            continue;
        }

        const std::string targetName = targetElem->GetAttribute("target");
        MPlug dstPlug;
        if      (targetName == "dist_attn_start") dstPlug = envFogFn.findPlug("fogNearDistance");
        else if (targetName == "dist_attn_end"  ) dstPlug = envFogFn.findPlug("fogFarDistance");
        else if (targetName == "color_r") dstPlug = envFogFn.findPlug("colorR");
        else if (targetName == "color_g") dstPlug = envFogFn.findPlug("colorG");
        else if (targetName == "color_b") dstPlug = envFogFn.findPlug("colorB");
        if (!dstPlug.isNull())
        {
            double valueScale = 1.0;
            if (targetName.find("dist_attn_") == 0)
            {
                valueScale = magnify;
            }
            status = CreateAnimCurveFromElement(dstPlug, targetElem, false, startFrame, valueScale, 0.0, false, fsnFile);
            CheckStatus(status);
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fsn ファイルを解析してカメラ／ライト／フォグとそのアニメーションカーブを作成します。
//!
//! @param[in] fsnPath fsn ファイルのパスです。
//! @param[in] magnify 移動値に掛ける倍率です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFsn(const std::string& fsnPath, const double magnify)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fsn ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile fsnFile(rstat, fsnPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 時間の作業単位と開始／終了フレームを設定します。
    const RXMLElement* animElem = fsnFile.m_Xml.FindElement("nw4f_3dif/scene_anim");
    const RXMLElement* infoElem = animElem->FindElement("scene_anim_info");
    const int startFrame = SetAnimFrameRange(infoElem);

    //-----------------------------------------------------------------------------
    // 選択をバックアップします。
    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    // カメラをインポートします。
    const RXMLElement* camAnimsElem = animElem->FindElement("camera_anim_array", false);
    if (camAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < camAnimsElem->nodes.size(); ++iMember)
        {
            status = ParseCameraAnim(&camAnimsElem->nodes[iMember], magnify, startFrame, fsnFile);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // ライトをインポートします。
    const RXMLElement* lgtAnimsElem = animElem->FindElement("light_anim_array" , false);
    if (lgtAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < lgtAnimsElem->nodes.size(); ++iMember)
        {
            status = ParseLightAnim(&lgtAnimsElem->nodes[iMember], magnify, startFrame, fsnFile);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // フォグをインポートします。
    const RXMLElement* fogAnimsElem = animElem->FindElement("fog_anim_array" , false);
    if (fogAnimsElem != NULL)
    {
        for (size_t iMember = 0; iMember < fogAnimsElem->nodes.size(); ++iMember)
        {
            status = ParseFogAnim(&fogAnimsElem->nodes[iMember], magnify, startFrame, fsnFile);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // 選択をリストアします。
    MGlobal::setActiveSelectionList(slist);

    //-----------------------------------------------------------------------------
    MGlobal::displayInfo(MString("Imported: ") + fsnPath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief fmd ファイルを解析してノードを作成します。
//!
//! @param[out] bones ボーン配列です。
//! @param[out] materials マテリアル配列です。
//! @param[out] texDatas テクスチャデータ配列です。
//! @param[in,out] lodInfo LOD 情報です。
//! @param[in] fmdPath fmd ファイルのパスです。
//! @param[in] magnify 移動値や頂点座標に掛ける倍率です。
//! @param[in] cutsZeroFace ジオメトリ領域ゼロのフェースを省略するなら true です。
//! @param[in] forcesSeparatesSkinObject 強制的にスキンシェイプをワールド下の別オブジェクトとして作成するなら true です。
//! @param[in] createsTexture テクスチャファイルを作成するなら true です。
//! @param[in] lodIdx インポートするサブメッシュの LOD インデックスです。
//! @param[in] showsProgress プログレス表示するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ParseFmd(
    ImpBoneArray& bones,
    ImpMaterialArray& materials,
    ImpTexDataArray& texDatas,
    ImpLodInfo& lodInfo,
    const std::string& fmdPath,
    const double magnify,
    const bool cutsZeroFace,
    const bool forcesSeparatesSkinObject,
    const bool createsTexture,
    const int lodIdx,
    const bool showsProgress
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // fmd ファイルをリードします。
    RStatus rstat;
    ImpIntermediateFile fmdFile(rstat, fmdPath);
    if (!rstat)
    {
        DisplayImportError(rstat.GetJapaneseMessage(), rstat.GetMessage());
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 各パスを取得します。
    const std::string inputFolder = RGetFolderFromFilePath(fmdPath) + "/";
    const std::string imagesFolder = GetSourceImagesFolder();
    const std::string texcvtrPath = GetNintendoMaya3dTextureConverterPath();
    if (!RFileExists(texcvtrPath.c_str()))
    {
        DisplayImportError("3D テクスチャーコンバーターが見つかりません: {0}",
            "3D texture converter cannot be found: {0}", texcvtrPath);
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // プロセスログを解析します。
    bool separatesSkinObject = forcesSeparatesSkinObject;
    const RXMLElement* modelElem = fmdFile.m_Xml.FindElement("nw4f_3dif/model");
    const RXMLElement* processLogsElem = modelElem->FindElement("process_log_array", false);
    if (processLogsElem != NULL)
    {
        ParseProcessLogs(separatesSkinObject, processLogsElem);
    }
    //cerr << "separatesSkinObject: " << separatesSkinObject << endl;

    //-----------------------------------------------------------------------------
    // 各要素を取得します。
    const RXMLElement* matsElem = modelElem->FindElement("material_array", false);
    const RXMLElement* shapesElem = modelElem->FindElement("shape_array", false);
    const RXMLElement* skeletonElem = modelElem->FindElement("skeleton", false);

    //-----------------------------------------------------------------------------
    // LOD グループと各レベルのルートを作成します。
    const int lodCount = GetLodCount(shapesElem);
    const bool usesLodGroup = (lodCount >= 2 && lodIdx == LOD_INDEX_ALL);
    if (!usesLodGroup && lodIdx >= lodCount)
    {
        DisplayImportWarning("LOD モデルが見つかりません: LOD{0}",
            "LOD level cannot be found: LOD{0}", RGetNumberString(lodIdx));
    }
    if (usesLodGroup)
    {
        GetLodInfo(lodInfo, shapesElem, skeletonElem);
        CreateLodGroup(lodInfo, lodCount);
    }

    //-----------------------------------------------------------------------------
    // プログレス表示を開始します。
    if (showsProgress)
    {
        int totalCount = (matsElem != NULL) ? static_cast<int>(matsElem->nodes.size()) : 0;
        totalCount += (shapesElem != NULL) ? static_cast<int>(shapesElem->nodes.size()) : 0;
        totalCount = (totalCount > 0) ? totalCount : 1;
        MGlobal::executeCommand(MString("progressBar -e -beginProgress -isInterruptable 0 -max ") +
            totalCount + " -st \"Importing fmd ...\" $gMainProgressBar");
    }

    //-----------------------------------------------------------------------------
    // マテリアルを解析してノードを作成します。
    if (matsElem != NULL)
    {
        const bool isColorSpaceRawAvailable = CheckColorSpaceRaw();
        const RXMLElement* orgMatsElem = modelElem->FindElement("original_material_array", false);
        for (size_t iMat = 0; iMat < matsElem->nodes.size(); ++iMat)
        {
            status = ParseMaterial(materials, texDatas,
                &matsElem->nodes[iMat], orgMatsElem, inputFolder, imagesFolder, texcvtrPath,
                createsTexture, isColorSpaceRawAvailable);
            CheckStatus(status);
            if (showsProgress)
            {
                MGlobal::executeCommand("progressBar -e -step 1 $gMainProgressBar");
            }
        }
    }

    //-----------------------------------------------------------------------------
    // スケルトンを解析して transform (joint) ノードを作成します。
    RIntArray mtxIdxToBoneIdxs;
    const RXMLElement* verticesElem = modelElem->FindElement("vertex_array", false);
    if (skeletonElem != NULL)
    {
        const RXMLElement* skeletonInfoElem = skeletonElem->FindElement("skeleton_info");
        const bool isQuat = (skeletonInfoElem->GetAttribute("rotate_mode") == "quaternion");
        const RXMLElement* bonesElem = skeletonElem->FindElement("bone_array");
        RStringArray parentBoneNames;
        GetParentBoneNames(parentBoneNames, bonesElem);
        for (size_t iBone = 0; iBone < bonesElem->nodes.size(); ++iBone)
        {
            status = ParseBone(bones, &bonesElem->nodes[iBone], magnify,
                isQuat, separatesSkinObject, parentBoneNames, shapesElem, lodInfo);
            CheckStatus(status);
        }
        CreateMtxIdxToBoneIdxTable(mtxIdxToBoneIdxs, bones);
    }

    //-----------------------------------------------------------------------------
    // シェイプを解析して mesh ノードを作成します。
    for (size_t iBone = 0; iBone < bones.size(); ++iBone)
    {
        ImpBone& bone = bones[iBone];
        if (bone.m_HasShape)
        {
            const int srcLodCount = (usesLodGroup) ? lodCount : 1;
            for (int iLod = 0; iLod < srcLodCount; ++iLod)
            {
                const int srcLodIdx = (usesLodGroup) ? iLod : lodIdx;
                status = CreateMesh(bones, lodInfo, iBone,
                    magnify, cutsZeroFace, srcLodIdx,
                    verticesElem, shapesElem, mtxIdxToBoneIdxs, materials, fmdFile);
                CheckStatus(status);
            }
            if (showsProgress)
            {
                MGlobal::executeCommand(MString("progressBar -e -step ") +
                    static_cast<int>(bone.m_ShapeElmIdxs.size()) + " $gMainProgressBar");
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 選択を解除します。
    MGlobal::clearSelectionList();

    //-----------------------------------------------------------------------------
    if (showsProgress)
    {
        MGlobal::executeCommand("progressBar -e -endProgress $gMainProgressBar");
    }
    MGlobal::displayInfo(MString("Imported: ") + fmdPath.c_str());

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief モデルの座標範囲によってデフォルトカメラのファークリッププレーンを調整します。
//!
//! @param[in] bones ボーン配列です。
//-----------------------------------------------------------------------------
static void AdjustDefaultCamera(const ImpBoneArray& bones)
{
    //-----------------------------------------------------------------------------
    // モデルの座標範囲を計算します。
    if (bones.empty())
    {
        return;
    }

    const ImpBone& bone = bones[0];
        // fmd ファイルをインポートした場合、ルートボーンは 1 つなので
        // 最初のボーンのバウンディングボックスが
        // モデル全体のバウンディングボックスとなります。
    MFnDagNode dagFn(bone.m_XformPath);
    MBoundingBox box = dagFn.boundingBox(); // 親ノードの座標系
    box.transformUsing(bone.m_XformPath.exclusiveMatrix());
    const MPoint allMin = box.min();
    const MPoint allMax = box.max();
    const double range = 2.0 * RMax(allMax.x - allMin.x, RMax(allMax.y - allMin.y, allMax.z - allMin.z));
    double farClip = 1000.0;
    while (farClip < range)
    {
        farClip *= 10.0;
    }
    //cerr << "adc: " << range << ", " << farClip << endl;

    //-----------------------------------------------------------------------------
    // カメラのアトリビュートを調整します。
    const int CAM_COUNT = 4;
    static const char* const camNames[CAM_COUNT] =
    {
        "perspShape", "topShape", "frontShape", "sideShape"
    };

    for (int iCam = 0; iCam < CAM_COUNT; ++iCam)
    {
        const MDagPath camPath = GetDagPathByName(camNames[iCam]);
        if (camPath.isValid() && camPath.node().hasFn(MFn::kCamera))
        {
            //cerr << "adc: " << camPath.partialPathName() << endl;
            MFnCamera camFn(camPath);
            if (camFn.farClippingPlane() < farClip)
            {
                camFn.setFarClippingPlane(farClip);
            }
        }
    }
}

//=============================================================================
// nwImport ネームスペースを終了します。
//=============================================================================
} // namespace nwImport

using namespace nwImport;

//=============================================================================
//! @brief Import コマンドのクラスです。
//!
//! Usage:
//!    NintendoImportCmd [flags] [fmd_file_path] [fsk_file_path] [fvb_file_path]
//!        [fma_file_path] [fcl_file_path] [fts_file_path] [ftp_file_path]
//!        [fsh_file_path] [fsn_file_path]
//! Flags:
//!    -newScene(-n) <boolean> 新規シーンに fmd ファイルをインポートするかどうかを指定します（デフォルト false）。
//!    -magnify(-m) <float> 移動値や頂点座標に掛ける倍率を指定します（デフォルト 1.0）。
//!    -cutZeroFace(-czf) <boolean> ジオメトリ領域ゼロのフェースを省略するかどうかを指定します（デフォルト true）。
//!    -separateSkinObject(-sso) <boolean> スキンシェイプをワールド下の別オブジェクトとして作成するかどうかを指定します（デフォルト true）。
//!    -createTexture(-ct) <boolean> テクスチャファイルを作成するかどうかを指定します（デフォルト true）。
//!    -adjustView (-av) <boolean> モデル全体が表示されるようにビューを調整するかどうかを指定します（デフォルト true）。
//!    -lodIndex(-loi) <integer> インポートするサブメッシュの LOD インデックスを指定します（-1 なら全レベル）（デフォルト -1）。
//!    -animBinarization(-ab) <boolean> アニメーションのバイナリ出力フラグを反映するかどうかを指定します（デフォルト true）。
//!    -progress(-prg) <boolean> プログレス表示するかどうかを指定します（デフォルト true）。
//=============================================================================
class NintendoImportCmd : public MPxCommand
{
public:
    NintendoImportCmd() {}
    virtual ~NintendoImportCmd() {}
    MStatus doIt(const MArgList& args);
    static MSyntax newSyntax();
    static void* creator() { return new NintendoImportCmd(); }

private:
    // option
    std::string m_FmdPath; //!< fmd ファイルのパスです。空文字ならインポートしません。
    std::string m_FskPath; //!< fsk ファイルのパスです。空文字ならインポートしません。
    std::string m_FvbPath; //!< fvb ファイルのパスです。空文字ならインポートしません。
    std::string m_FmaPath; //!< fma ファイルのパスです。空文字ならインポートしません。
    std::string m_FclPath; //!< fcl ファイルのパスです。空文字ならインポートしません。
    std::string m_FtsPath; //!< fts ファイルのパスです。空文字ならインポートしません。
    std::string m_FtpPath; //!< ftp ファイルのパスです。空文字ならインポートしません。
    std::string m_FshPath; //!< fsh ファイルのパスです。空文字ならインポートしません。
    std::string m_FsnPath; //!< fsn ファイルのパスです。空文字ならインポートしません。
    bool m_ClearsScene; //!< 新規シーンに fmd ファイルをインポートするなら true です。
    double m_Magnify; //!< 移動値や頂点座標に掛ける倍率です。
    double m_ToInternalMagnify; //!< (Maya 上の単位から cm へのスケール) x m_Magnify です。
    bool m_CutsZeroFace; //!< ジオメトリ領域ゼロのフェースを省略するなら true です。
    bool m_SeparatesSkinObject; //!< スキンシェイプをワールド下の別オブジェクトとして作成するなら true です。
    bool m_CreatesTexture; //!< テクスチャファイルを作成するなら true です。
    bool m_AdjustsView; //!< モデル全体が表示されるようにビューを調整するなら true です。
    int m_LodIdx; //!< インポートするサブメッシュの LOD インデックスです。
    bool m_UsesAnimBin; //!< アニメーションのバイナリ出力フラグを反映するなら true です。
    bool m_ShowsProgress; //!< プログレス表示するなら true です。

private:
    MStatus ParseArgs(const MArgList& args);
    MStatus Action();
};

//-----------------------------------------------------------------------------
//! @brief コマンドの文法を返します。
//-----------------------------------------------------------------------------
#define kNewSceneFlag               "n"
#define kNewSceneFlagL              "newScene"
#define kMagnifyFlag                "m"
#define kMagnifyFlagL               "magnify"
#define kCutsZeroFaceFlag           "czf"
#define kCutsZeroFaceFlagL          "cutZeroFace"
#define kSeparatesSkinObjectFlag    "sso"
#define kSeparatesSkinObjectFlagL   "separateSkinObject"
#define kCreatesTextureFlag         "ct"
#define kCreatesTextureFlagL        "createTexture"
#define kAdjustsViewFlag            "av"
#define kAdjustsViewFlagL           "adjustView"
#define kLodIndexFlag               "loi"
#define kLodIndexFlagL              "lodIndex"
#define kUsesAnimBinFlag            "ab"
#define kUsesAnimBinFlagL           "animBinarization"
#define kShowsProgressFlag          "prg"
#define kShowsProgressFlagL         "progress"

MSyntax NintendoImportCmd::newSyntax()
{
    MSyntax syntax;

    syntax.addFlag(kNewSceneFlag, kNewSceneFlagL, MSyntax::kBoolean);
    syntax.addFlag(kMagnifyFlag, kMagnifyFlagL, MSyntax::kDouble);
    syntax.addFlag(kCutsZeroFaceFlag, kCutsZeroFaceFlagL, MSyntax::kBoolean);
    syntax.addFlag(kSeparatesSkinObjectFlag, kSeparatesSkinObjectFlagL, MSyntax::kBoolean);
    syntax.addFlag(kCreatesTextureFlag, kCreatesTextureFlagL, MSyntax::kBoolean);
    syntax.addFlag(kAdjustsViewFlag, kAdjustsViewFlagL, MSyntax::kBoolean);
    syntax.addFlag(kLodIndexFlag, kLodIndexFlagL, MSyntax::kLong);
    syntax.addFlag(kUsesAnimBinFlag, kUsesAnimBinFlagL, MSyntax::kBoolean);
    syntax.addFlag(kShowsProgressFlag, kShowsProgressFlagL, MSyntax::kBoolean);
    syntax.setObjectType(MSyntax::kStringObjects);

    return syntax;
}

//-----------------------------------------------------------------------------
//! @brief コマンドの引数を解析します。
//!
//! @param[in] args 引数リストです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoImportCmd::ParseArgs(const MArgList& args)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // get data base
    MArgDatabase argData(syntax(), args, &status);
    if (!status)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 新規シーンフラグを取得します。
    m_ClearsScene = false;
    if (argData.isFlagSet(kNewSceneFlag))
    {
        status = argData.getFlagArgument(kNewSceneFlag, 0, m_ClearsScene);
        CheckStatusMsg(status, "Cannot get new scene");
    }

    //-----------------------------------------------------------------------------
    // 移動値や頂点座標に掛ける倍率を取得します。
    m_Magnify = 1.0;
    if (argData.isFlagSet(kMagnifyFlag))
    {
        status = argData.getFlagArgument(kMagnifyFlag, 0, m_Magnify);
        CheckStatusMsg(status, "Cannot get magnify");
        if (m_Magnify <= 0.0)
        {
            DisplayImportError("Magnify が不正です。", "Magnify is wrong.");
            return MS::kFailure;
        }
    }

    // API で移動値や頂点座標を扱う際は cm で指定するので、
    // 中間ファイルの値をそのまま Maya 上の単位に反映するための倍率を計算します。
    m_ToInternalMagnify = MDistance::uiToInternal(1.0) * m_Magnify;

    //-----------------------------------------------------------------------------
    // ジオメトリ領域ゼロのフェースの省略フラグを取得します。
    m_CutsZeroFace = true;
    if (argData.isFlagSet(kCutsZeroFaceFlag))
    {
        status = argData.getFlagArgument(kCutsZeroFaceFlag, 0, m_CutsZeroFace);
        CheckStatusMsg(status, "Cannot get cut zero face");
    }

    //-----------------------------------------------------------------------------
    // スキンシェイプをワールド下の別オブジェクトとして作成するフラグを取得します。
    m_SeparatesSkinObject = true;
    if (argData.isFlagSet(kSeparatesSkinObjectFlag))
    {
        status = argData.getFlagArgument(kSeparatesSkinObjectFlag, 0, m_SeparatesSkinObject);
        CheckStatusMsg(status, "Cannot get separate skin object");
    }

    //-----------------------------------------------------------------------------
    // テクスチャファイル作成フラグを取得します。
    m_CreatesTexture = true;
    if (argData.isFlagSet(kCreatesTextureFlag))
    {
        status = argData.getFlagArgument(kCreatesTextureFlag, 0, m_CreatesTexture);
        CheckStatusMsg(status, "Cannot get create texture");
    }

    //-----------------------------------------------------------------------------
    // モデル全体が表示されるようにビューを調整するフラグを取得します。
    m_AdjustsView = true;
    if (argData.isFlagSet(kAdjustsViewFlag))
    {
        status = argData.getFlagArgument(kAdjustsViewFlag, 0, m_AdjustsView);
        CheckStatusMsg(status, "Cannot get adjust view");
    }

    //-----------------------------------------------------------------------------
    // インポートするサブメッシュの LOD インデックスを取得します。
    m_LodIdx = LOD_INDEX_ALL;
    if (argData.isFlagSet(kLodIndexFlag))
    {
        status = argData.getFlagArgument(kLodIndexFlag, 0, m_LodIdx);
        CheckStatusMsg(status, "Cannot get LOD index");
        if (m_LodIdx != LOD_INDEX_ALL && m_LodIdx < 0)
        {
            DisplayImportError("LOD インデックスが不正です。", "LOD index is wrong.");
            return MS::kFailure;
        }
    }

    //-----------------------------------------------------------------------------
    // アニメーションのバイナリ出力フラグの反映フラグを取得します。
    m_UsesAnimBin = true;
    if (argData.isFlagSet(kUsesAnimBinFlag))
    {
        status = argData.getFlagArgument(kUsesAnimBinFlag, 0, m_UsesAnimBin);
        CheckStatusMsg(status, "Cannot get animation binarization");
    }

    //-----------------------------------------------------------------------------
    // プログレス表示フラグを取得します。
    m_ShowsProgress = true;
    if (argData.isFlagSet(kShowsProgressFlag))
    {
        status = argData.getFlagArgument(kShowsProgressFlag, 0, m_ShowsProgress);
        CheckStatusMsg(status, "Cannot get progress");
    }
    const bool isInteractive = (MGlobal::mayaState() == MGlobal::kInteractive);
    if (!isInteractive)
    {
        m_ShowsProgress = false;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルのパスを取得します。
    MStringArray filePaths;
    status = argData.getObjects(filePaths);
    CheckStatusMsg(status, "fmd file path is not specified");

    for (unsigned int iFile = 0; iFile < filePaths.length(); ++iFile)
    {
        const std::string filePath = filePaths[iFile].asChar();
        const std::string ext = RGetExtensionFromFilePath(filePath);
        if (filePath.empty())
        {
            // skip
        }
        else if (ext == FmdbExtension || ext == FmdaExtension)
        {
            if (!m_FmdPath.empty())
            {
                DisplayImportError("複数の fmd ファイルのパスが指定されています。",
                    "More than one fmd file paths are specified.");
                return MS::kFailure;
            }
            m_FmdPath = filePath;
        }
        else if (ext == FskbExtension || ext == FskaExtension)
        {
            if (!m_FskPath.empty())
            {
                DisplayImportError("複数の fsk ファイルのパスが指定されています。",
                    "More than one fsk file paths are specified.");
                return MS::kFailure;
            }
            m_FskPath = filePath;
        }
        else if (ext == FvbbExtension || ext == FvbaExtension)
        {
            if (!m_FvbPath.empty())
            {
                DisplayImportError("複数の fvb ファイルのパスが指定されています。",
                    "More than one fvb file paths are specified.");
                return MS::kFailure;
            }
            m_FvbPath = filePath;
        }
        else if (ext == FmabExtension || ext == FmaaExtension)
        {
            if (!m_FmaPath.empty())
            {
                DisplayImportError("複数の fma ファイルのパスが指定されています。",
                    "More than one fma file paths are specified.");
                return MS::kFailure;
            }
            m_FmaPath = filePath;
        }
        else if (ext == FclbExtension || ext == FclaExtension)
        {
            if (!m_FclPath.empty())
            {
                DisplayImportError("複数の fcl ファイルのパスが指定されています。",
                    "More than one fcl file paths are specified.");
                return MS::kFailure;
            }
            m_FclPath = filePath;
        }
        else if (ext == FtsbExtension || ext == FtsaExtension)
        {
            if (!m_FtsPath.empty())
            {
                DisplayImportError("複数の fts ファイルのパスが指定されています。",
                    "More than one fts file paths are specified.");
                return MS::kFailure;
            }
            m_FtsPath = filePath;
        }
        else if (ext == FtpbExtension || ext == FtpaExtension)
        {
            if (!m_FtpPath.empty())
            {
                DisplayImportError("複数の ftp ファイルのパスが指定されています。",
                    "More than one ftp file paths are specified.");
                return MS::kFailure;
            }
            m_FtpPath = filePath;
        }
        else if (ext == FshbExtension || ext == FshaExtension)
        {
            if (!m_FshPath.empty())
            {
                DisplayImportError("複数の fsh ファイルのパスが指定されています。",
                    "More than one fsh file paths are specified.");
                return MS::kFailure;
            }
            m_FshPath = filePath;
        }
        else if (ext == FsnbExtension || ext == FsnaExtension)
        {
            if (!m_FsnPath.empty())
            {
                DisplayImportError("複数の fsn ファイルのパスが指定されています。",
                    "More than one fsn file paths are specified.");
                return MS::kFailure;
            }
            m_FsnPath = filePath;
        }
        else
        {
            DisplayImportError("不正なファイルタイプです: {0}", "File type is wrong: {0}",
                filePath);
            return MS::kFailure;
        }
    }

    if (m_FmdPath.empty() &&
        m_FskPath.empty() &&
        m_FvbPath.empty() &&
        m_FmaPath.empty() &&
        m_FclPath.empty() &&
        m_FtsPath.empty() &&
        m_FtpPath.empty() &&
        m_FshPath.empty() &&
        m_FsnPath.empty())
    {
        DisplayImportError("中間ファイルのパスが指定されていません。",
            "Intermediate file path is not specified.");
        return MS::kFailure;
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief コマンド実行処理です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoImportCmd::Action() // ActionT
{
    MStatus status;
    //cerr << "NintendoImportCmd: " << m_FmdPath << ", " << m_FskPath << ", " << m_Magnify << endl;

    //-----------------------------------------------------------------------------
    // 処理終了待ちカーソルを表示します。
    YWaitCursor waitCursor;

    //-----------------------------------------------------------------------------
    // fmd ファイルをインポートします。
    ImpBoneArray bones;
    ImpMaterialArray materials;
    ImpTexDataArray texDatas;
    ImpLodInfo lodInfo;
    if (!m_FmdPath.empty())
    {
        if (m_ClearsScene)
        {
            MGlobal::executeCommand("file -f -new");
        }

        status = ParseFmd(bones, materials, texDatas, lodInfo, m_FmdPath,
            m_ToInternalMagnify, m_CutsZeroFace, m_SeparatesSkinObject, m_CreatesTexture,
            m_LodIdx, m_ShowsProgress);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fsk ファイルをインポートします。
    const bool getsBones = m_FmdPath.empty();
    if (!m_FskPath.empty())
    {
        status = ParseFsk(bones, m_FskPath, m_ToInternalMagnify, getsBones, m_UsesAnimBin);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fvb ファイルをインポートします。
    if (!m_FvbPath.empty())
    {
        status = ParseFvb(bones, m_FvbPath, getsBones, m_UsesAnimBin);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fma ファイルをインポートします。
    const bool getsMaterials = m_FmdPath.empty();
    if (!m_FmaPath.empty())
    {
        status = ParseFma(&texDatas, materials, m_FmaPath, getsMaterials, m_CreatesTexture);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fcl ファイルをインポートします。
    if (!m_FclPath.empty())
    {
        status = ParseFcl(materials, m_FclPath, getsMaterials);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fts ファイルをインポートします。
    if (!m_FtsPath.empty())
    {
        status = ParseFts(materials, m_FtsPath, getsMaterials);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // ftp ファイルをインポートします。
    if (!m_FtpPath.empty())
    {
        status = ParseFtp(&texDatas, materials, m_FtpPath, getsMaterials, m_CreatesTexture);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fsh ファイルをインポートします。
    if (!m_FshPath.empty())
    {
        status = ParseFsh(bones, m_FshPath, getsBones);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fsn ファイルをインポートします。
    if (!m_FsnPath.empty())
    {
        status = ParseFsn(m_FsnPath, m_ToInternalMagnify);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // モデルの座標範囲によってデフォルトカメラのファークリッププレーンを調整します。
    if (m_AdjustsView && !m_FmdPath.empty() && m_FsnPath.empty())
    {
        AdjustDefaultCamera(bones);
    }

    //-----------------------------------------------------------------------------
    // すべてのビューでオブジェクトをフレームに収め、
    // パースビューの表示を調整します。
    if (m_AdjustsView && MGlobal::mayaState() == MGlobal::kInteractive && m_FsnPath.empty())
    {
        MString cmd =
        "{\n"
        "   fitAllPanels -all;\n"
        "   string $localizedLabel = localizedPanelLabel(\"Persp View\");\n"
        "   string $panel = `getPanel -withLabel $localizedLabel`;\n"
        "   if ($panel != \"\")\n"
        "   {\n"
        "       modelEditor -e -displayAppearance \"smoothShaded\" -displayTextures 1\n"
        "           -displayLights \"default\" $panel;\n"
        "   }\n"
        "}\n";
        MGlobal::executeCommand(cmd);
    }

    //-----------------------------------------------------------------------------
    // Maya の lodGroup ノードのしきい値を調整します。
    AdjustLodGroupThreshold(lodInfo);

    //-----------------------------------------------------------------------------
    // 結果を設定します。
    //-----------------------------------------------------------------------------
    MString rootXformName;
    if (!bones.empty())
    {
        rootXformName = bones[0].m_XformPath.partialPathName();
    }
    setResult(rootXformName);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief コマンドを実行します。
//!
//! @param[in] args 引数リストです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoImportCmd::doIt(const MArgList& args) // DoIt
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // parse argument
    status = ParseArgs(args);
    if (!status)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // action
    status = Action();

    //-----------------------------------------------------------------------------
    // 処理失敗の場合にプログレス表示を終了します。
    if (!status && m_ShowsProgress)
    {
        MGlobal::executeCommand("progressBar -e -endProgress $gMainProgressBar");
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief インポートプラグインを登録します。
//!
//! @param[in,out] plugin プラグインオブジェクトです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus InitializeNintendoImport(MFnPlugin& plugin)
{
    MStatus status;

    status = plugin.registerCommand("NintendoImportCmd",
        NintendoImportCmd::creator,
        NintendoImportCmd::newSyntax);
    CheckStatusMsg(status, "register NintendoImportCmd");

    return status;
}

//-----------------------------------------------------------------------------
//! @brief インポートプラグインを登録解除します。
//!
//! @param[in,out] plugin プラグインオブジェクトです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus UninitializeNintendoImport(MFnPlugin& plugin)
{
    MStatus status;

    status = plugin.deregisterCommand("NintendoImportCmd");
    CheckStatusMsg(status, "deregister NintendoImportCmd");

    return status;
}

