﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// constants
// NintendoExportCmd DoIt ProcessT ExportOneModel GetData OutputFiles
// InitOptions ParseOptions CheckOptions CheckOverwrite MoveTmpFiles
// GetHierarchy SetYNodeConnection GetYNodeTransform GetYNodeBindTransform
// GetYNodeVertex GetYNodeShape
// GetShapePosMtxIdx GetShapeNormal GetShapeVtxColor GetShapeTexCoord
// GetShapeTangent GetShapeUserAttr
// GetEnvelope
// GetMaterial GetMatInfo
// GetTexNode GetTexSampler GetTexImage GetTexSrtAnim GetTexPatAnim
// Do3dOptimizer Do3dFormatter
// NintendoExportInfoCmd NintendoEvaluateModelCmd EvaluateOneModel

// GetFullAnimValue
// AnalyzeYNodeFullAnim GetYNodeKeyAnim
// GetColorFullAnimValue AnalyzeColorFullAnim GetColorKeyAnim
// GetTexSrtFullAnimValue AnalyzeTexSrtFullAnim GetTexSrtKeyAnim
// GetTexPatFullAnimValue AnalyzeTexPatFullAnim GetTexPatKeyAnim

// OutputFmdFile OutputFskFile OutputFvbFile
// OutputFmaFile OutputFclFile OutputFtsFile OutputFtpFile
// OutputTextureFiles ReadTexImageAndAddJobList
// OutputHeader OutputFooter
// OutputMaterials OutputSkeleton OutputShapes OutputShapeT

// YSceneT YModelT YNodeT YNodeCT ShapeDataT YVtxAttrInfo
// YMaterialT YMatInfo
// TexNode TexSrtAnimT TexPatAnimT

//=============================================================================
// 処理選択用マクロです。
//=============================================================================

// fixed
#define DO_OPTIMIZER_PARALLEL // 中間ファイルオプティマイザーを並列実行するなら定義します。
#define USE_FORMAT_CONVERTER_SW // 中間ファイルフォーマッターで中間ファイルを整形するなら定義します。
#define NO_CLAMP_FRAME_RANGE_SW // 出力フレーム範囲をアニメーション開始／終了の範囲でクランプしないなら定義します。
#define ADJUST_YNODE_ROT_XYZ_SW // ノードの回転を XYZ 3 軸同時に調整するなら定義します。
#define ADD_MODEL_ROOT_SW // ルートボーンが複数存在する場合にモデルルートを追加するなら定義します。
//#define OUT_CAM_LIT_AS_NULL_SW // カメラとライトを NULL ボーンとして出力するなら定義します。
//#define OUTPUT_ERROR_LOG // エラーログを出力するなら定義します。

//=============================================================================
// include
//=============================================================================

//-----------------------------------------------------------------------------
// my header
#include "Export.h"
#include "DccMaterial.h"
#include "DccShape.h"
#include "DccImage.h"
#include "DccOutput.h"
#include "Deform.h"
#include "Animation.h"
#include "Scene.h"
#include "CheckLod.h"

//-----------------------------------------------------------------------------
// maya header
#include <maya/MFnPlugin.h>

//-----------------------------------------------------------------------------
// using namespace
using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
// constants
//=============================================================================
namespace {

//-----------------------------------------------------------------------------
// version
const char* ExporterVersion = "4.5.0"; //!< Export プラグインのバージョンです。

//-----------------------------------------------------------------------------
// ファイル名
const char* const ExportConfigFileName = "Export.ini";

//-----------------------------------------------------------------------------
// 環境変数名
const char* const OptimizerExtraOptionFmdEnvVarName = "NINTENDO_EXPORT_OPTIMIZER_EXTRA_OPTION_FMD"; //!< fmd ファイルに対する中間ファイルオプティマイザーの追加オプション用環境変数名です。
const char* const OverwriteOptionStringVariableName = "nnExport_OverwriteOptionString"; //!< エクスポートオプション上書き用オプション変数名です。
const char* const BakingLoopScriptVariableName      = "nnExport_BakingLoopScript"; //!< Baking Loop Script 用オプション変数名です。

} // namespace

//-----------------------------------------------------------------------------
//! @brief ウェイトカーソルを開始します。
//-----------------------------------------------------------------------------
static void BeginWaitCursor()
{
    MGlobal::executeCommand("waitCursor -st 1");
}

//-----------------------------------------------------------------------------
//! @brief ウェイトカーソルを終了します。
//-----------------------------------------------------------------------------
static void EndWaitCursor()
{
    MGlobal::executeCommand("if (`waitCursor -q -st`) waitCursor -st 0;");
}

//-----------------------------------------------------------------------------
//! @brief 環境変数を 1 つ MEL で取得します。
//!
//! @param[in] name 環境変数名です。
//!
//! @return 値を返します。
//-----------------------------------------------------------------------------
static std::string GetEnvVarByMel(const std::string& name)
{
    const std::string cmd = "getenv \"" + name + "\"";
    MString value;
    MGlobal::executeCommand(cmd.c_str(), value);
    return value.asChar();
}

//-----------------------------------------------------------------------------
//! @brief 環境変数を 1 つ MEL で設定します。
//!
//! @param[in] name 環境変数名です。
//! @param[in] value 値です。
//! @param[in] setsOld 古い名前の環境変数も設定するなら true を指定します。
//!
//! @return 設定に成功すれば true を返します。
//!         値が長すぎる場合は空文字を設定して false を返します。
//-----------------------------------------------------------------------------
static bool SetEnvVarByMel(
    const std::string& name,
    const std::string& value,
    const bool setsOld
)
{
    static const std::string NINTENDO_PREFIX = "NINTENDO_";

    const std::string adjustedValue =
        (static_cast<int>(value.size()) < _MAX_ENV - 64) ? value : "";
    std::string cmd = "putenv \"" + name + "\" \"" + adjustedValue + "\"";
    MGlobal::executeCommand(cmd.c_str());

    if (setsOld && name.find(NINTENDO_PREFIX) == 0)
    {
        const std::string oldName = "NW4F_" + name.substr(NINTENDO_PREFIX.size());
        cmd = "putenv \"" + oldName + "\" \"" + adjustedValue + "\"";
        MGlobal::executeCommand(cmd.c_str());
    }

    return (value.size() == adjustedValue.size());
}

//-----------------------------------------------------------------------------
//! @brief Maya の文字列型オプション変数の値を取得します。
//!
//! @param[in] name オプション変数名です。
//!
//! @return オプション変数の値（存在しなければ空文字）を返します。
//-----------------------------------------------------------------------------
static std::string GetMayaOptionVariableString(const std::string& name)
{
    bool exists;
    MString value = MGlobal::optionVarStringValue(name.c_str(), &exists);
    return (exists) ? value.asChar() : "";
}

//-----------------------------------------------------------------------------
//! @brief Maya のオプション変数を削除します。
//!
//! @param[in] name オプション変数名です。
//-----------------------------------------------------------------------------
static void RemoveMayaOptionVariable(const std::string& name)
{
    MGlobal::executeCommand(("optionVar -remove " + name).c_str());
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーの追加オプション用環境変数をクリアします。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void ClearOptimizerExtraOptionEnvVar(const YExpOpt& yopt)
{
    if (!yopt.ExportsLodLevel())
    {
        SetEnvVarByMel(OptimizerExtraOptionFmdEnvVarName, "", false);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーの追加オプションを環境変数から取得します。
//!
//! @param[in,out] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void GetOptimizerExtraOption(YExpOpt& yopt)
{
    yopt.m_OptimizerExtraOptionFmd = RTrimString(GetEnvVarByMel(OptimizerExtraOptionFmdEnvVarName));
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプション上書き用オプション変数をクリアします。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void ClearOverwriteOptionStringVariable(const YExpOpt& yopt)
{
    if (!yopt.ExportsLodLevel())
    {
        RemoveMayaOptionVariable(OverwriteOptionStringVariableName);
    }
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプション上書き文字列をオプション変数から取得します。
//!
//! @return エクスポートオプション上書き文字列を返します。
//-----------------------------------------------------------------------------
static std::string GetOverwriteOptionString()
{
    return RTrimString(GetMayaOptionVariableString(OverwriteOptionStringVariableName));
}

//-----------------------------------------------------------------------------
//! @brief Baking Loop Script 用オプション変数をクリアします。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void ClearBakingLoopScriptVariable(const YExpOpt& yopt)
{
    if (!yopt.ExportsLodLevel())
    {
        RemoveMayaOptionVariable(BakingLoopScriptVariableName);
    }
}

//-----------------------------------------------------------------------------
//! @brief Baking Loop Script をオプション変数から取得します。
//!
//! @return Baking Loop Script を返します。
//-----------------------------------------------------------------------------
static std::string GetBakingLoopScript()
{
    return RTrimString(GetMayaOptionVariableString(BakingLoopScriptVariableName));
}

//-----------------------------------------------------------------------------
//! @brief サンプラのアトリビュートを取得します。
//!
//! @param[in,out] rscene シーンです。
//! @param[out] nameOverride サンプラの名前を格納します。
//! @param[out] hintOverride サンプラのヒント情報の文字列を格納します。
//! @param[in] texObj Maya のテクスチャオブジェクトです。
//! @param[in] checksError エラー判定するなら true を指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetSamplerAttribute(
    RScene& rscene,
    std::string& nameOverride,
    std::string& hintOverride,
    const MObject& texObj,
    const bool checksError
)
{
    MStatus status;
    MFnDependencyNode texFn(texObj);

    // 名前を取得します。
    nameOverride.clear();
    MPlug samplerNamePlug = FindPlugQuiet(texFn, "nw4fSamplerName", &status);
    if (status)
    {
        MString str;
        samplerNamePlug.getValue(str);
        nameOverride = RTrimString(str.asChar());
        if (checksError && !RIsValidElementNameString(nameOverride))
        {
            YShowError(&rscene, // Sampler name is wrong: %s: %s
                "サンプラの名前に禁止文字が含まれています: {0}: {1}",
                "The sampler name includes one or more prohibited characters: {0}: {1}",
                texFn.name().asChar(), nameOverride);
            return MS::kFailure;
        }
    }

    // 名前が空文字でない場合のみ、ヒント情報の文字列を取得します。
    hintOverride.clear();
    if (!nameOverride.empty())
    {
        MPlug samplerHintPlug = FindPlugQuiet(texFn, "nw4fSamplerHint", &status);
        if (status)
        {
            MString str;
            samplerHintPlug.getValue(str);
            hintOverride = RTrimString(str.asChar());
            if (checksError && !RIsValidElementNameString(hintOverride))
            {
                YShowError(&rscene, // Sampler hint is wrong: %s: %s
                    "サンプラのヒント情報に禁止文字が含まれています: {0}: {1}",
                    "The sampler hint includes one or more prohibited characters: {0}: {1}",
                    texFn.name().asChar(), hintOverride);
                return MS::kFailure;
            }
        }
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT 関連アトリビュート名を返します。
//!
//! @param[in] paramIdx パラメータのインデックスです。
//!
//! @return テクスチャ SRT 関連アトリビュート名を返します。
//-----------------------------------------------------------------------------
static const char* GetTexSrtAttrName(const int paramIdx)
{
    static const char* const attrNames[] =
    {
        "repeatU",
        "repeatV",
        "rotateFrame",
        "translateFrameU",
        "translateFrameV",
    };
    return attrNames[paramIdx];
}

//-----------------------------------------------------------------------------
//! @brief テクスチャノードのテクスチャ SRT 関連アトリビュートに
//!        place2dTexture ノード以外が接続されていればエラーを表示して false を返します。
//!
//! @param[in,out] rscene シーンです。
//! @param[in] texObj Maya のテクスチャオブジェクトです。
//!
//! @return place2dTexture ノード以外が接続されていれば false、
//!         接続されていなければ true を返します。
//-----------------------------------------------------------------------------
static bool CheckTexSrtInput(RScene& rscene, const MObject& texObj)
{
    MFnDependencyNode texFn(texObj);
    for (int iParam = 0; iParam < ROriginalTexsrt::PARAM_COUNT; ++iParam)
    {
        const MPlug plug = texFn.findPlug(GetTexSrtAttrName(iParam));
        MPlugArray plugs;
        plug.connectedTo(plugs, true, false);
        if (plugs.length() != 0 &&
            plugs[0].node().apiType() != MFn::kPlace2dTexture)
        {
            YShowError(&rscene, // place2dTexture node is not connected: %s
                "テクスチャー SRT アニメーションの出力が指定されていますが、file ノードの関連アトリビュートに place2dTexture ノード以外が接続されています: {0} \n"
                "file ノードに place2dTexture ノードを接続して、place2dTexture ノードのアトリビュートにアニメーションを設定してください。",
                "Texture SRT animation output is specified, but a node other than place2dTexture is connected to the file node-related attribute: {0} \n"
                "Connect the place2dTexture node to the file node, and set the animation in the attribute of the place2dTexture node.",
                plug.name().asChar());
            return false;
        }
    }
    return true;
}

//=============================================================================
//! @brief Maya 用のテクスチャノードのクラスです。
//=============================================================================
class TexNode
{
public:
    //-----------------------------------------------------------------------------
    // node
    MObject m_TexObj; //!< Maya のテクスチャオブジェクト（file、envCube ノード）です。
    MObject m_PlaceObj; //!< Maya の place2dTexture ノードです。
    MObject m_ChooserObj; //!< Maya の uvChooser ノードです。テクスチャが使用する UV セットを選択するノードです。
    bool m_IsSingleEnvCube; //!< 環境立方体の +X 面にキューブマップの DDS ファイルが接続されていれば true です。

    //-----------------------------------------------------------------------------
    // file

    //! @brief 画像ファイルのパス配列です。
    //!        環境立方体の場合 +X、-X、+Y、-Y、+Z、-Z 面の順に格納します。
    RStringArray m_FilePaths;

    //-----------------------------------------------------------------------------
    // サンプラ
    RSampler::Hint m_Hint; //!< テクスチャの関連付けに利用するヒント情報です。
    std::string m_SamplerNameOverride; //!< サンプラの名前のオーバーライドです。
    std::string m_SamplerHintOverride; //!< サンプラのヒント情報の文字列のオーバーライドです。

    //-----------------------------------------------------------------------------
    // static tex coord transform
    bool m_IsXformIdentity; //!< テクスチャ変換行列が単位行列なら true です。
    float m_ScaleU; //!< テクスチャ座標の U 方向のスケールです。
    float m_ScaleV; //!< テクスチャ座標の V 方向のスケールです。
    float m_Rotate; //! テクスチャ座標の回転です。
    float m_TranslateU; //! テクスチャ座標の U 方向の移動です。
    float m_TranslateV; //! テクスチャ座標の V 方向の移動です。

    //-----------------------------------------------------------------------------
    // anim
    MPlug m_TexSrtPlugs[ROriginalTexsrt::PARAM_COUNT]; //!< テクスチャ SRT パラメータのプラグ配列です。
    bool m_TexPatAnimFlag; //!< テクスチャパターンアニメーションが設定されていれば true です。

    //-----------------------------------------------------------------------------
    // internal
    std::string m_ObjName; //!< Maya のテクスチャオブジェクト名です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in,out] rscene シーンです。
    //! @param[in] texObj Maya のテクスチャオブジェクトです。
    //! @param[in] hint テクスチャの関連付けに利用するヒント情報です。
    //! @param[in] yopt エクスポートオプションです。
    //! @param[out] pstat 処理結果を格納します。
    //!
    TexNode(
        RScene& rscene,
        const MObject& texObj,
        const RSampler::Hint hint,
        const YExpOpt& yopt,
        MStatus* pStatus = NULL
    );

    //! @brief テクスチャ変換を適用します。
    //!
    //! @param[in,out] fu 元の U 座標を指定すると変換後の U 座標が格納されます。
    //! @param[in,out] fv 元の V 座標を指定すると変換後の V 座標が格納されます。
    //! @param[int] texSrtMode テクスチャ SRT の計算方式です。
    //!
    void ApplyTransform(
        float& fu,
        float& fv,
        ROriginalTexsrt::Mode texSrtMode
    ) const;

    //! 環境マップなら true を返します。
    bool IsEnvMap() const { return m_Hint == RSampler::REFLECTION; }
};

//! @brief Maya 用のテクスチャノード配列の定義です。
typedef std::vector<TexNode> TexNodeArray;

//-----------------------------------------------------------------------------
//! @brief Maya 用のテクスチャノードのコンストラクタです。
//-----------------------------------------------------------------------------
TexNode::TexNode(
    RScene& rscene,
    const MObject& texObj,
    const RSampler::Hint hint,
    const YExpOpt& yopt,
    MStatus* pStatus
)
:   m_TexObj(texObj),
    m_Hint(hint),
    m_TexPatAnimFlag(false)
{
    //-----------------------------------------------------------------------------
    // init return status
    if (pStatus != NULL)
    {
        *pStatus = MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // タイプをチェックします。
    m_ObjName = MFnDependencyNode(m_TexObj).name().asChar();
    const MFn::Type texType = texObj.apiType();
    const bool isEnvCube = (texType == MFn::kEnvCube);
    m_IsSingleEnvCube         = (isEnvCube && RIsSingleEnvCube(texObj, yopt.m_ProjectPath));
    const bool isCubeSeparate = (isEnvCube && !m_IsSingleEnvCube);

    //-----------------------------------------------------------------------------
    // ファイルパスを取得します。
    const int faceCount = (isCubeSeparate) ? RImage::CUBE_FACE_COUNT : 1;
    for (int iFace = 0; iFace < faceCount; ++iFace)
    {
        const MObject fileObj = (isCubeSeparate) ?
            GetEnvCubeFileTexture(m_TexObj, iFace) :
            GetMainFileTexture(m_TexObj);
        MFnDependencyNode texFn(fileObj);
        const std::string fileObjName = texFn.name().asChar();

        const std::string filePath = GetFileTextureName(fileObj, yopt.m_ProjectPath);
        if (filePath.empty())
        {
            if (!yopt.m_CheckElementFlag)
            {
                YShowError(&rscene, // Texture image name is wrong: %s
                    "file ノードのイメージの名前が空または不正です: {0}",
                    "The Image Name of the file node is empty or wrong: {0}",
                    fileObjName);
                return;
            }
            else
            {
                YShowWarning(&rscene, // Texture image name is wrong: %s
                    "file ノードのイメージの名前が空または不正です: {0}",
                    "The Image Name of the file node is empty or wrong: {0}",
                    fileObjName);
            }
        }
        m_FilePaths.push_back(filePath);

        // UV タイリング機能が使用されていればエラーにします。
        #if (MAYA_API_VERSION >= 201500)
        int uvTilingMode = 0;
        texFn.findPlug("uvTilingMode").getValue(uvTilingMode);
        if (uvTilingMode != 0)
        {
            YShowError(&rscene, // UV tiling is not supported: %s
                "UV タイリング機能には対応していません。file ノードの UV タイリングモードアトリビュートは「オフ」にしてください: {0}",
                "The UV tiling feature is not supported. Set the UV tiling mode attribute to off for file nodes: {0}",
                fileObjName);
            return;
        }
        #endif
    }

    //-----------------------------------------------------------------------------
    // サンプラの名前とヒント情報の文字列のオーバーライドを取得します。
    const MObject mainTexObj = GetMainFileTexture(m_TexObj);
    if (!GetSamplerAttribute(rscene, m_SamplerNameOverride, m_SamplerHintOverride, mainTexObj, true))
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // 環境マップでないテクスチャの情報を取得します。
    m_PlaceObj   = MObject::kNullObj;
    m_ChooserObj = MObject::kNullObj;
    if (!IsEnvMap())
    {
        //-----------------------------------------------------------------------------
        // テクスチャ SRT アニメーションを出力する場合、
        // file ノードのテクスチャ SRT 関連アトリビュートに
        // place2dTexture ノード以外が接続されていればエラーにします。
        if (yopt.ExportsTexSrtAnim())
        {
            if (!CheckTexSrtInput(rscene, mainTexObj))
            {
                return;
            }
        }

        //-----------------------------------------------------------------------------
        // file ノードの uvCoord に接続された place2dTexture ノードを取得します。
        MFnDependencyNode texFn(mainTexObj);
        MPlugArray plugArray;
        texFn.findPlug("uvCoord").connectedTo(plugArray, true, false);
        if (plugArray.length() != 0 &&
            plugArray[0].node().apiType() == MFn::kPlace2dTexture)
        {
            m_PlaceObj = plugArray[0].node();
        }

        //-----------------------------------------------------------------------------
        // uvChooser ノードを取得します。
        // place2dTexture ノードがあれば place2dTexture ノードの uvCoord、
        // なければ file ノードの uvCoord プラグの入力を調べます。
        MObject uvTargetObj = (!m_PlaceObj.isNull()) ? m_PlaceObj : mainTexObj;
        MPlug uvCoordPlug = MFnDependencyNode(uvTargetObj).findPlug("uvCoord");
        uvCoordPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() > 0 &&
            plugArray[0].node().apiType() == MFn::kUvChooser)
        {
            m_ChooserObj = plugArray[0].node();
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーションの存在フラグを取得します。
    if (!isCubeSeparate)
    {
        bool useFrameExtension;
        MFnDependencyNode(mainTexObj).findPlug("useFrameExtension").getValue(useFrameExtension);
        if (useFrameExtension)
        {
            m_TexPatAnimFlag = true;
        }
    }

    //-----------------------------------------------------------------------------
    // 開始フレームにおけるテクスチャ変換を取得します。

    // place2dTexture オブジェクトが存在すればそのオブジェクトから変換を取得します。
    MObject texXformObj = (!m_PlaceObj.isNull()) ? m_PlaceObj : mainTexObj;
    MFnDependencyNode texXformFn(texXformObj);

    // アニメーションのためにプラグを記録します。
    for (int iParam = 0; iParam < ROriginalTexsrt::PARAM_COUNT; ++iParam)
    {
        m_TexSrtPlugs[iParam] =
            texXformFn.findPlug(GetTexSrtAttrName(iParam));
    }

    // 変換値を取得します。
    m_TexSrtPlugs[ROriginalTexsrt::SCALE_X    ].getValue(m_ScaleU);
    m_TexSrtPlugs[ROriginalTexsrt::SCALE_Y    ].getValue(m_ScaleV);
    m_TexSrtPlugs[ROriginalTexsrt::ROTATE     ].getValue(m_Rotate);
    m_TexSrtPlugs[ROriginalTexsrt::TRANSLATE_X].getValue(m_TranslateU);
    m_TexSrtPlugs[ROriginalTexsrt::TRANSLATE_Y].getValue(m_TranslateV);

    // 変換値を補正します。
    m_Rotate *= static_cast<float>(R_M_RAD_TO_DEG);

    m_ScaleU     = RSnapToZero(m_ScaleU);
    m_ScaleV     = RSnapToZero(m_ScaleV);
    m_Rotate     = RSnapToZero(m_Rotate);
    m_TranslateU = RSnapToZero(m_TranslateU);
    m_TranslateV = RSnapToZero(m_TranslateV);

    // テクスチャ変換行列が単位行列かどうかのフラグを設定します。
    m_IsXformIdentity = (
        m_ScaleU     == 1.0f &&
        m_ScaleV     == 1.0f &&
        m_Rotate     == 0.0f &&
        m_TranslateU == 0.0f &&
        m_TranslateV == 0.0f);

    //-----------------------------------------------------------------------------
    // finish
    if (pStatus != NULL)
    {
        *pStatus = MS::kSuccess;
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief Maya 用のテクスチャノードのテクスチャ変換を適用します。
//-----------------------------------------------------------------------------
void TexNode::ApplyTransform(
    float& fu,
    float& fv,
    ROriginalTexsrt::Mode texSrtMode
) const
{
    //-----------------------------------------------------------------------------
    // no xform
    if (m_IsXformIdentity)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // set sin
    const float ang = static_cast<float>(m_Rotate * R_M_DEG_TO_RAD);
    const float sinR = static_cast<float>(sin(ang));
    const float cosR = static_cast<float>(cos(ang));

    //-----------------------------------------------------------------------------
    // transform UV
    if (texSrtMode == ROriginalTexsrt::MAYA)
    {
        // (ou ov) = (iu iv) [-pivot] [R] [+pivot] [T] [S]
        fu -= 0.5f;
        fv -= 0.5f;
        float gu = cosR * fu - sinR * fv;
        float gv = sinR * fu + cosR * fv;
        fu = (gu + 0.5f - m_TranslateU) * m_ScaleU;
        fv = (gv + 0.5f - m_TranslateV) * m_ScaleV;
    }
    else if (texSrtMode == ROriginalTexsrt::MAX)
    {
        // (ou ov) = (iu iv) [T] [-pivot] [R] [S] [+pivot]
        fu += -m_TranslateU - 0.5f;
        fv += -m_TranslateV - 0.5f;
        float gu = cosR * fu - sinR * fv;
        float gv = sinR * fu + cosR * fv;
        fu = gu * m_ScaleU + 0.5f;
        fv = gv * m_ScaleV + 0.5f;
    }
    else // SOFTIMAGE
    {
        // (ou ov) = (iu iv) [T] [R] [S]
        // [R] の回転方向は Maya および Max とは逆方向が正
        fu += -m_TranslateU;
        fv += -m_TranslateV;
        float gu =  cosR * fu + sinR * fv;
        float gv = -sinR * fu + cosR * fv;
        fu = gu * m_ScaleU;
        fv = gv * m_ScaleV;
    }
}

//=============================================================================
//! @brief Maya 用のテクスチャ SRT アニメーションのクラスです。
//=============================================================================
class TexSrtAnim // TexSrtAnimT
{
public:
    // anim
    MPlug m_AnimPlugs[ROriginalTexsrt::PARAM_COUNT]; //!< パラメータに対応する Maya のプラグ配列です。
    ChannelInput m_AnimChans[ROriginalTexsrt::PARAM_COUNT]; //!< パラメータに対応するチャンネル入力情報配列です。
    YAnimCurve m_Anims[ROriginalTexsrt::PARAM_COUNT]; //!< パラメータに対応するアニメーションカーブ配列です。

    // internal
    MObject m_PlaceObj; //!< Maya の place2dTexture ノードです。
    std::string m_PlaceNodeName; //!< Maya の place2dTexture ノード名です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] placeObj Maya の place2dTexture ノードです。
    //! @param[in] materialIndex マテリアルのインデックスです。
    //!
    explicit TexSrtAnim(const MObject& placeObj)
    : m_PlaceObj(placeObj)
    {
        m_PlaceNodeName = MFnDependencyNode(placeObj).name().asChar();
    }
};

//! @brief Maya 用のテクスチャ SRT アニメーション配列の定義です。
typedef std::vector<TexSrtAnim> TexSrtAnimArray;

//=============================================================================
//! @brief Maya 用のテクスチャパターンアニメーションのクラスです。
//=============================================================================
class TexPatAnim // TexPatAnimT
{
public:
    // texture list
    MIntArray m_UsedFes;    //!< 使用するフレーム拡張子の配列です。
    MIntArray m_TexImgIdxs; //!< m_UsedFes に対応するテクスチャイメージのモデル内のインデックス配列です。

    // anim
    YAnimCurve m_FeAnim;  //!< フレーム拡張子のアニメーションカーブです。
    YAnimCurve m_ImgAnim; //!< 出力するイメージのアニメーションカーブです。

    // internal
    MPlug m_FePlug;        //!< フレーム拡張子プラグです。
    bool m_ValidCurveFlag; //!< フレーム拡張子プラグに有効なアニメーションカーブ（animCurveTU）が接続されていれば true です。
    MObject m_InputObj;    //!< フレーム拡張子に接続されたノード（カーブ、エクスプレッション、その他）です。
    std::string m_ObjName; //!< Maya のテクスチャオブジェクト名です。
};

//! @brief Maya 用のテクスチャパターンアニメーション配列の定義です。
typedef std::vector<TexPatAnim> TexPatAnimArray;

//=============================================================================
//! @brief Maya 用のノードのクラスです。
//=============================================================================
class YNode : public RBone // YNodeT
{
public:
    //! @brief 種類を表す列挙型です。
    enum Kind
    {
        KindModelRoot,  //!< 出力用に追加したモデルルートです。
        KindTransform,  //!< トランスフォームです。
        KindMesh,       //!< メッシュです。
        KindJoint,      //!< ジョイントです。

        KindMax,        //!< 種類の数です。
        KindNone,       //!< 無効な種類です。
    };

    //-----------------------------------------------------------------------------
    // dag path
    MDagPath m_XformPath; //!< transform ノードの DAG パスです。
    MDagPath m_ShapePath; //!< シェイプノードの DAG パスです。
    MDagPath m_OrgShapePath; //!< オリジナルのシェイプノードの DAG パスです。

    //-----------------------------------------------------------------------------
    // kind

    //! @brief コンストラクタで指定されたオリジナルの種類です。
    //!        この値が変更されることはありません。
    Kind m_OrgKind;

    int m_Index; //!< 出力用インデックスです。
    int m_OrgIndex; //!< 内部インデックスです。

    std::string m_OrgName; //!< オリジナルの名前（transform ノード名）です。

    //! @brief プラグインが追加したノードなら true です。
    //! KindModelRoot タイプのデータは一度シーンをトラバースした後に状況によって追加されます。
    //! 追加されたノードは無効な DAG パスを持つので Maya 上のノードのように
    //! DAG パスを扱う処理ができません。
    //! このフラグで DAG パスを扱う処理ができるか判別します。
    bool m_AddedFlag;

    //! @brief ノードが Maya のジョイント（API タイプが MFn::kJoint）なら true です。
    //! ジョイントに固有なアトリビュート（orientation や segmentScaleCompensate）を取得できるか
    //! 判別するのに使用します。
    bool m_MayaJointFlag;

    //! @brief スキニングに影響するノードなら true です。
    // スキンクラスタからスキンの変形に使用されるインフルエンスオブジェクトかどうかを示すフラグ。
    // transform ノードの worldMatrix プラグが SkinClusterFilter に接続されていれば true となります。
    // このフラグはジョイント以外でもインフルエンスオブジェクトであれば true になります。
    //
    // 不要なノードの削除時に、インフルエンスオブジェクトを削除しないように
    // インフルエンスオブジェクトかどうかの判定に使用されます。
    bool m_SkinInfluenceFlag;

    //! @brief ジョイントが IK ハンドルによってコントロールされていれば true です。
    //! ノードが Maya のジョイントの場合のみ true になり得ます。
    bool m_IkControlledFlag;

    //! @brief ジオメトリ コンストレインが設定されていれば true です。現在使用されていません。
    bool m_SpecialConstraintFlag;

    bool m_ClusterHandleFlag; //!< クラスタハンドルであれば true です。

    //! @brief 最適化で圧縮可能なら true です（内部使用用）。
    //!        Maya 上で圧縮可能に設定されていてもルートのボーンなら false になります。
    bool m_InternalCompressEnable;

    bool m_IsSpecialUserDataNode; //!< ユーザーデータ取得のための特別なノードなら true です。

    //! @brief マージする親ノード数です。現在は 0 固定でマージ機能は使用しません。
    int m_MergeCount;

    //-----------------------------------------------------------------------------
    // connection

    //! @brief 階層構造の深さです。
    //! ルートノードに 0 が設定され、一段下位階層になるごとに +1 されていきます。
    //! 中間ファイルの Bone 配列に出力するときの順番を決定するのに使用されます。
    int m_Depth;

    //! @brief 子供の数です。
    //! 直下の下位階層にいくつのtransform ノードがあるのかを表します。
    int m_ChildCount;

    //! @brief 親ノードへのポインタです。
    //! ルートノードの場合は NULL が設定されます。
    YNode* m_pParent;

    //! @brief 先頭の子ノードへのポインタです。
    //! 複数の子供を持つ場合は先頭の子供のポインタが設定され、
    //! 以降の子供は mNext に設定されているリンクを辿ることで参照されます。
    YNode* m_pChild;

    //! @brief 同じ親を持つ次の兄弟ノードへのポインタです。
    //! 兄弟ノードのリンクで最後尾のノードでは NULL が設定されます。
    YNode* m_pNext;

    //! @brief 同じ親を持つ前の兄弟ノードへのポインタです。
    //! 兄弟ノードのリンクで先頭のノードでは NULL が設定されます。
    YNode* m_pPrev;

    //! @brief 出力用の親ノードへのポインタです。
    //! 現在 m_pParent と同じポインタが設定されます。
    YNode* m_pOutParent;

    //-----------------------------------------------------------------------------
    // transform

    MVector m_XformS; //!< transform ノードから取得したローカルのスケールです。
    MVector m_XformR; //!< transform ノードから取得したローカルの回転です。

    //! @brief transform ノードから取得したローカルの移動に Magnify オプションを掛けた値です。
    //! AdjustTranslate により mPivotMiddleOffset を考慮した値が設定されます。
    MVector m_XformT;

    MVector m_BindS; //!< バインドポーズでのスケールです。
    MVector m_BindR; //!< バインドポーズでの回転です。
    MVector m_BindT; //!< バインドポーズでの移動です。

    MVector m_OutXformS; //!< 出力用のスケールです。
    MVector m_OutXformR; //!< 出力用の回転です。
    MVector m_OutXformT; //!< 出力用の移動です。

    MVector m_OutBindS; //!< 出力用のバインドポーズでのスケールです。
    MVector m_OutBindR; //!< 出力用のバインドポーズでの回転です。
    MVector m_OutBindT; //!< 出力用のバインドポーズでの移動です。

    //! @brief ノードの変換行列計算用のスケール値です。
    //! ScaleCompensate を考慮した変換行列を計算する時に、親のスケールとして参照する値です。
    //! このスケールはノードのタイプごとに設定される値が変わるので、m_XformS とは別に保存されます。
    MVector m_CalcXformS;

    //! @brief 変換の継承フラグです。
    //! transform ノードの"トランスフォームの継承"アトリビュートの設定値です。
    //! スキンメッシュ以外では true に設定されている必要があります。
    bool m_InheritsXform;

    MPoint m_RotatePivot; //!< 回転ピボットです。m_TranslateOffset の計算に使用されます。

    //! @brief スケールピボットです。
    //! スキンのシェイプ行列やバインドポーズ行列に影響を与えます。
    //! また、mOrgPoss、mCurPoss はスケールピボットを引いて、
    //! ローカル座標の原点がスケールピボットと一致するようになっています。
    MPoint m_ScalePivot;

    MVector m_RotatePivotTrans; //!< 回転ピボットのオフセットです。m_TranslateOffset の計算に使用されます。

    //! @brief スケールピボットのオフセットです。
    //! mPivotMiddleOffset にスケールピボットのオフセットを含んだ値があるので、
    //! この変数は特に使用されていません。
    MVector m_ScalePivotTrans;

    //! @brief 移動成分のオフセットです。
    //! RotatePivotTrans と RotatePivot と親の ScalePivot(スキンモデル以外のノードの場合)をまとめたオフセット値です。
    //! 移動成分アニメーションの補正に使用されます。
    MVector m_TranslateOffset;

    //! @brief ピボットオフセットです。
    //! ScalePivit, ScalePivotTranslation, RotatePivot をまとめたオフセット値です。
    //! transform ノードの回転角度により回転して移動成分のオフセットとなります。
    //! AdjustTranslate で使用されます。
    MVector m_PivotMiddleOffset;

    //! @brief ピボットオフセットフラグです。
    //! m_PivotMiddleOffset にゼロ以外の値が設定されていれば true です。
    bool m_PivotMiddleOffsetFlag;

    //! @brief 回転順序です。
    //! 初期値には MEulerRotation::kXYZ が指定されます。
    MEulerRotation::RotationOrder m_RotateOrder;

    bool m_SpecialRotateOrderFlag; //!< 回転順序が MEulerRotation::kXYZ 以外なら true です。

    //! @brief 回転軸(RotateAxis)です。
    //! ノードの回転前の補正回転を表すオイラー角です。
    //! このパラメータは AdjustRotate において回転を補正するのに使用されます。
    MEulerRotation m_RotateAxis;

    //! @brief ジョイントの向きです。
    //! ノードがジョイントであった場合にのみ設定されます。
    //! ジョイント以外の場合は MEulerRotation::identity が設定されます。
    //! このパラメータは AdjustRotate において回転を補正するのに使用されます。
    MEulerRotation m_JointOrient;

    bool m_RotateAxisFlag; //!< m_RotateAxis が MEulerRotation::identity 以外なら true です。
    bool m_JointOrientFlag; //!< m_JointOrient が MEulerRotation::identity 以外なら true です。

    //! @brief 回転の補正フラグです。
    //! m_SpecialRotateOrderFlag、m_RotateAxisFlag、m_JointOrientFlag の内どれか一つでも true であれば true です。
    //! このフラグが true の場合、ベイクしたデータからアニメーションカーブを作成します。
    bool m_RotateAdjustFlag;

    bool m_OrgScaleCompensate; //!< セグメントスケール補正が有効なら true です（内部使用用）。

    //! @brief 移動・回転・スケールの制限フラグです。
    //! transform ノードの 制限情報 > 移動、回転、スケール の制限設定で、
    //! 最小か最大のどちらかでもチェックされていたら true です。
    //! アニメーションの変換で焼付けを行うかどうかの判断に使用されていましたが、
    //! 現在はコメントアウトされて反映されなくなっています。
    bool m_XformLimitFlag[RBone::PARAM_COUNT];

    bool m_SkinMtxFlag; //!< スムーススキンのインフルエンスオブジェクトとして行列が使用されるなら true です。現在は使用されていません。

    //! @brief ローカル変換行列です。
    //! ジョイントの場合はセグメントスケール補正も考慮した値が設定されます。
    MMatrix m_LocalMtx;

    MMatrix m_GlobalMtx; //!< ワールド変換行列です。
    MMatrix m_GlobalInvMtx; //!< ワールド変換行列の逆行列です。
    MMatrix m_BindLocalMtx; //!< ノードのローカルバインド行列です。
    MMatrix m_BindGlobalMtx; //!< ノードのワールドバインド行列です。
    MMatrix m_BindGlobalInvMtx; //!< ノードのワールドバインド行列の逆行列です。
    MMatrix m_BindNoScaleGlobalMtx; //!< スケールを除いたローカル変換行列です。現在は使用されていません。
    MMatrix m_BindNoScaleGlobalInvMtx; //!< スケールを除いたローカル変換行列の逆行列です。現在は使用されていません。
    MMatrix m_CalcLocalMtx; //!< アニメーションの計算のために一時的に現在のフレームでのローカル変換行列を格納する変数です。

    //-----------------------------------------------------------------------------
    // shape

    CObjectArray m_SGObjs; //!< シェイプの描画で使用するマテリアルを表すシェーディンググループのオブジェクト配列です。

    //! @brief シェーディンググループのコンポーネント配列です。
    //! m_SGObjs の要素に一対一で対応したシェーディンググループを使用するポリゴンのコンポーネントの配列です。
    //! 各要素のオブジェクトから MItMeshPolygon を作成し、ポリゴンの情報を取得できます。
    CObjectArray m_CompObjs;

    //-----------------------------------------------------------------------------
    // mesh

    int m_PosCount; //!< メッシュの頂点数です。MFnMesh::numVertices で取得した値です。
    int m_FaceCount; //!< メッシュのポリゴン数です。MFnMesh::numPolygons で取得した値です。

    //! @brief メッシュのオリジナルの頂点座標配列（ローカル座標）です。
    //! メッシュにデフォーマが設定されている場合でもそれらを無視した
    //! バインドポーズ時の頂点座標が設定されます。
    MPointArray m_OrgPoss;

    //! @brief メッシュの現在の頂点座標配列（ローカル座標）です。
    //! バインドポーズで出力する場合は m_OrgPoss と同じ値が設定されます。
    MPointArray m_CurPoss;

    //! @brief メッシュの出力用の頂点座標配列（ワールド座標）です。
    //! スムーススキニングの場合に出力されます。
    MPointArray m_GlobalPoss;

    //! @brief メッシュの出力用の頂点座標配列（影響するボーンのローカル座標）です。
    //! スキニングなしまたはリジッドスキニングの場合に出力されます。
    MPointArray m_InfPoss;

    //! @brief 各頂点の頂点行列のインデックス配列です。
    //! m_PosCount の長さを持ち、各要素は ymodel.m_VtxMtxs 内のインデックスを表します。
    MIntArray m_VtxMtxIdxs;

    MFloatVectorArray m_OrgNrms; //!< メッシュのオリジナルの法線配列（ローカル座標）です。
    MFloatVectorArray m_CurNrms; //!< メッシュの現在の法線配列（ローカル座標）です。

    //! @brief メッシュの頂点フェースごとの法線のインデックス配列です。
    //! m_OrgNrms 配列のインデックスとして使用されます。
    MIntArray m_OrgNrmIdxs;

    RPrimitive::PrimType m_PrimType; //!< プリミティブタイプです。

    MString m_MeshColSets[RPrimVtx::VTX_COL_MAX]; // !< 出力する頂点カラー属性のカラーセット名配列です。
    MString m_MeshUsrColSets[RPrimVtx::VTX_USR_MAX]; //!< ユーザー頂点属性のカラーセット名配列です。

    //! @brief デフォルトの UV セット名です。
    //!        メッシュが UV セットを持たない場合は "map1" が設定され、
    //!        UV セットを持つ場合は uvSet アトリビュートの先頭の UV セット名が設定されます。
    MString m_DefaultUvSet;

    //-----------------------------------------------------------------------------
    // deformer

    bool m_DeformerFlag; //!< メッシュがスキニングまたはブレンドシェイプで変形されるなら true です。
    bool m_IsSkinned; //!< メッシュがスキニングで変形されるなら true です。
    bool m_BlendShapeFlag; //!< メッシュがブレンドシェイプで変形されるなら true です。
    bool m_ForcesSmoothSkinning; //!< シェイプをすべてスムーススキニングとして出力するなら true です（1 頂点に影響するボーンが 1 つでも）。
    CObjectArray m_DeformerObjs; //!< メッシュに設定されているデフォーマオブジェクトの配列です。
    MObject m_SkinClusterObj; //!< メッシュに設定されている skinCluster ノードです。
    MObject m_BlendShapeObj; //!< メッシュに設定されている blendShape ノードです。
    int m_BlendShapeDataIndex; //!< ymodel.m_BlendShapeDatas の要素へのインデックスです。

    //-----------------------------------------------------------------------------
    // visibility
    MPlug m_XformVisibilityPlug; //!< ノードの可視性プラグ（transform ノードの visibility）です。
    MPlug m_ShapeVisibilityPlug; //!< シェイプの可視性プラグ（mesh ノードの visibility）です。
    bool m_CurXformVisibility; //!< 取得中のフレームでの transform ノードの可視性です。

    //-----------------------------------------------------------------------------
    // layer

    //! @brief レイヤのテクスチャリングフラグです。
    //! ノードがディスプレイレイヤに登録されていて、レイヤーの texturing アトリビュートが
    //! OFF なら false となります。現在は使用されていません。
    bool m_LayerTexturingFlag;

    //! @brief レイヤの再生フラグです。
    //! ノードがディスプレイレイヤに登録されていて、レイヤーの playback アトリビュートが
    //! OFF なら false となります。false の場合、ボーンのアニメーションを出力しません。
    bool m_LayerPlaybackFlag;

    //! @brief レイヤーの可視性フラグです。
    //! ノードがディスプレイレイヤに登録されていて、レイヤーの visibility アトリビュートが
    //! OFF なら false となります。false の場合、m_CurXformVisibility が true でも不可視となります。
    bool m_LayerVisibilityFlag;

    //-----------------------------------------------------------------------------
    // anim
    ChannelInput m_AnimChans[RBone::PARAM_COUNT]; //!< スケルタルアニメーションの各パラメーターのチャンネル入力情報配列です。
    YAnimCurve m_Anims[RBone::PARAM_COUNT]; //!< スケルタルアニメーションの各パラメーターのアニメーションカーブ配列です。
    float m_AnimScales[RBone::PARAM_COUNT]; //!< スケルタルアニメーションの各パラメーターに適用するスケール配列です。
    float m_AnimOfss[RBone::PARAM_COUNT]; //!< スケルタルアニメーションの各パラメーターに適用するオフセット配列です。

    YAnimCurve m_VisAnim; //!< ビジビリティアニメーションのアニメーションカーブです。
    bool m_VisibilityAnimFlag; //!< transform ノードの可視性にアニメーションが設定されているなら true です。
    bool m_ShapeVisibilityAnimFlag; //!< mesh ノードの可視性にアニメーションが設定されているなら true です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in,out] rscene シーンです。
    //! @param[in] xformPath transform ノードの DAG パスです。
    //! @param[in] kind 種類です。
    //! @param[in] compressEnable 最適化で圧縮可能なら true です。
    //! @param[in] ikControlled ジョイントが IK ハンドルによってコントロールされていれば true です。
    //! @param[in] clusterHandle クラスタハンドルであれば true です。
    //! @param[in] yopt エクスポートオプションです。
    //! @param[out] pStatus 処理結果を格納します。
    //!
    explicit YNode(
        RScene& rscene,
        const MDagPath& xformPath,
        const Kind kind,
        const bool compressEnable,
        const bool ikControlled,
        const bool clusterHandle,
        const YExpOpt& yopt,
        MStatus* pStatus = NULL
    );

    //! @brief Maya から取得した回転を出力用に補正します。
    //!
    //! @param[in,out] eulerRotate 回転です。
    //!
    void AdjustRotate(MEulerRotation& eulerRotate);

    //! @brief Maya から取得した移動を出力用に補正します。
    //!
    //! @param[in,out] translate 移動です。
    //! @param[in] eulerRotate AdjustRotate で補正した回転です。
    //!
    void AdjustTranslate(MVector& translate, const MEulerRotation& eulerRotate);

    //! 出力用の親ノード名を取得します。親ノードがなければ空文字を返します。
    std::string GetOutParentName() const
    {
        return (m_pOutParent != NULL) ? m_pOutParent->m_Name : "";
    }

    //! @brief ユーザーデータを取得します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //!
    void GetUserData(RScene* pScene)
    {
        if (!m_IsSpecialUserDataNode && m_XformPath.isValid())
        {
            ::GetUserData(m_UserDatas, *pScene, m_XformPath.node());
        }
    }

    //! @brief ソートのための比較演算子です。
    //!        深さが小さいほど順番が先になり、同じ深さならノード名のアルファベット順になります。
    friend bool operator<(const YNode& r1, const YNode& r2)
    {
        if (r1.m_Depth != r2.m_Depth)
        {
            return (r1.m_Depth < r2.m_Depth);
        }
        else
        {
            return (strcmp(r1.m_Name.c_str(), r2.m_Name.c_str()) < 0);
        }
    }
};

//! @brief Maya 用のノード配列の定義です。
typedef std::vector<YNode> YNodeArray;

//! @brief Maya 用のノードのポインタ配列の定義です。
typedef std::vector<YNode*> YNodePtrArray;

//! @brief Maya 用のノードのポインタ配列のイテレータの定義です。
typedef YNodePtrArray::iterator ItYNodePtr;

//-----------------------------------------------------------------------------
//! @brief Maya 用のノードのコンストラクタです。
//-----------------------------------------------------------------------------
YNode::YNode( // YNodeCT
    RScene& rscene,
    const MDagPath& xformPath,
    const Kind kind,
    const bool compressEnable,
    const bool ikControlled,
    const bool clusterHandle,
    const YExpOpt& yopt,
    MStatus* pStatus
)
:
    // copy argument
    m_XformPath(xformPath),
    m_OrgKind(kind),
    m_InternalCompressEnable(compressEnable),
    m_IkControlledFlag(ikControlled),
    m_ClusterHandleFlag(clusterHandle),

    // kind
    m_MayaJointFlag(false),
    m_SkinInfluenceFlag(false),
    m_SpecialConstraintFlag(false),
    m_IsSpecialUserDataNode(false),
    m_MergeCount(0),

    // xform
    m_InheritsXform(true),
    m_RotatePivot(MPoint::origin), m_ScalePivot(MPoint::origin),
    m_RotatePivotTrans(MVector::zero), m_ScalePivotTrans(MVector::zero),
    m_TranslateOffset(MVector::zero), m_PivotMiddleOffset(MVector::zero),
    m_PivotMiddleOffsetFlag(false),
    m_RotateOrder(MEulerRotation::kXYZ),
    m_SpecialRotateOrderFlag(false),
    m_RotateAxis(MEulerRotation::identity), m_JointOrient(MEulerRotation::identity),
    m_RotateAxisFlag(false), m_JointOrientFlag(false),
    m_RotateAdjustFlag(false),

    m_SkinMtxFlag(false),

    // mesh
    m_PrimType(RPrimitive::TRIANGLE_FAN),

    // deformer
    m_DeformerFlag(false),
    m_IsSkinned(false),
    m_BlendShapeFlag(false),
    m_ForcesSmoothSkinning(false),
    m_BlendShapeDataIndex(-1),

    // layer
    m_LayerTexturingFlag(true),
    m_LayerPlaybackFlag(true),
    m_LayerVisibilityFlag(true),

    // anim
    m_VisibilityAnimFlag(false),
    m_ShapeVisibilityAnimFlag(false),

    // connection
    m_pParent(NULL),
    m_pChild(NULL),
    m_pNext(NULL),
    m_pPrev(NULL)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 戻り値の初期設定
    if (pStatus != NULL)
    {
        *pStatus = MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 出力用の圧縮可能フラグを設定します。
    m_CompressEnable = compressEnable;

    //-----------------------------------------------------------------------------
    // xform の回転と移動に制限がかけられているかどうかのフラグを初期化します。
    // (初期状態では制限なしの false を設定します。)
    {
        for (int paramIdx = 0; paramIdx < RBone::PARAM_COUNT; ++paramIdx)
        {
            m_XformLimitFlag[paramIdx] = false;
        }
    }

    //-----------------------------------------------------------------------------
    // 追加フラグを設定します。
    // オリジナルのタイプが KindModelRoot の場合は、シーントラバース後の状況に
    // よって追加されます。このフラグはその追加されたノードであるかどうかを記録します。
    m_AddedFlag = (m_OrgKind == KindModelRoot);

    //-----------------------------------------------------------------------------
    // model root special
    if (m_OrgKind == KindModelRoot)
    {
        m_OrgName = m_Name = RBone::DefaultRootName;
        if (pStatus != NULL)
        {
            *pStatus = MS::kSuccess;
        }
        return;
    }

    //-----------------------------------------------------------------------------
    // set element
    m_MayaJointFlag = (m_XformPath.node().apiType() == MFn::kJoint);

    //-----------------------------------------------------------------------------
    // set name
    MFnDagNode dagFn(m_XformPath);
    m_OrgName = dagFn.name().asChar();
    m_Name = GetOutElementName(m_OrgName, yopt.m_RemoveNamespace);
    if (m_Name != m_OrgName)
    {
        //if (!yopt.m_CheckElementFlag)
        //{
        //  YShowWarning(&yscene, "", "Node name is changed: {0} -> {1}", // 現在は発生しない
        //      m_OrgName, m_Name);
        //}
    }

    //-----------------------------------------------------------------------------
    // get shape dag path
    if (m_OrgKind == KindMesh)
    {
        // 中間ファイルに出力するシェイプオブジェクトを決定します。
        int shapeCount = 0;
        int orgShapeCount = 0;
        MDagPath secondOrgShapePath; // 第２候補
        int childCount = dagFn.childCount();
        for (int ichild = 0; ichild < childCount; ++ichild)
        {
            MObject childObj = dagFn.child(ichild);
            if (childObj.apiType() == MFn::kMesh)
            {
                // インスタンス化されている可能性を考慮して
                // 直接の子供シェイプのパスを取得します。
                MDagPath childDagPath = GetDirectChildDagPath(
                    m_XformPath, childObj); // changed (2006/04/17)
                MFnDagNode childDagFn(childObj);
                if (!childDagFn.isIntermediateObject())
                {
                    // Mayaの中間オブジェクトでなければこのシェイプを
                    // 中間ファイルに出力します。
                    // 複数のシェイプが見つかった場合は最後に見つかった
                    // シェイプを出力します。
                    m_ShapePath = childDagPath;
                    ++shapeCount;
                }
                else
                {
                    // 子供のシェイプがMayaの中間オブジェクトであった場合は、
                    // "worldMesh"アトリビュートに接続されているオブジェクトを
                    // シェイプとして設定しようとします。
                    MPlug worldMeshPlug = childDagFn.findPlug("worldMesh");
                    if (worldMeshPlug.numElements() > 0 && worldMeshPlug[0].isConnected())
                    {
                        // "worldMesh"に接続されていたデータで名前に"Orig"が含まれています。
                        // 場合はオリジナルのシェイプとして記録しておく。
                        if (std::string(childDagFn.name().asChar()).find("Orig") != std::string::npos)
                        {
                            m_OrgShapePath = childDagPath;
                            ++orgShapeCount;
                            //g_Err << "YNode: org: " << m_OrgShapePath.partialPathName() << " -> " << m_OrgName << R_ENDL;
                        }
                        else
                        {
                            // Mayaの中間オブジェクトで名前に"Orig"も付かないものは
                            // 他にシェイプが見つからなかった場合に m_OrgShapePath
                            // に設定するため保存しておく。
                            secondOrgShapePath = childDagPath;
                        }
                    }
                }
            }
        }

        // m_OrgShapePath が設定されておらず、第二候補のシェイプがみつかっていた場合は、
        // その第二候補のシェイプを m_OrgShapePath として使用します。
        if (orgShapeCount == 0 && secondOrgShapePath.isValid())
        {
            // リファレンスしたノードにスキンを設定した場合など
            m_OrgShapePath = secondOrgShapePath;
            ++orgShapeCount;
            //g_Err << "YNode: 2nd org: " << m_OrgShapePath.partialPathName() << " -> " << m_ShapePath.partialPathName() << R_ENDL;
        }

        if (shapeCount == 0)
        {
            // KindMesh の判定で NodeHasEffectiveMeshChild() を
            // 使用しているので起こらない
            if (!yopt.m_CheckElementFlag)
            {
                YShowError(&rscene, // No effective shape // 通常は発生しない
                    "有効なシェイプが存在しません: {0}",
                    "No effective shape: {0}",
                    m_OrgName);
                return;
            }
        }
        else if (shapeCount > 1)
        {
            // 一つのトランスフォームノードが複数のシェイプを子供として参照している場合は
            // シェイプは一つしか出力されないので警告を表示します。
            if (!yopt.m_CheckElementFlag)
            {
                YShowWarning(&rscene, // There are multiple shape: %s
                    "transform ノードの下に複数の mesh ノードが存在します。出力される mesh ノードは 1 つだけです: {0}",
                    "Multiple mesh nodes exist below the transform node. Only one mesh node is output: {0}",
                    m_OrgName);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // transform ノードのアトリビュートを取得します。
    MEulerRotation r = MEulerRotation::identity;
    MVector rp = MVector::zero, sp = MVector::zero;
    MVector rpt = MVector::zero, spt = MVector::zero;
    MEulerRotation ra = MEulerRotation::identity;
    MEulerRotation jo = MEulerRotation::identity;

    MFnTransform xformFn(m_XformPath);

    // 親の変換を継承するかどうかを取得します。
    m_InheritsXform = IsInheritsTransform(m_XformPath);

    // transform ノードの変換を取得します。
    xformFn.getRotation(r); // to get rotate order
    // transform ノードのローカル座標系でのピボット情報を取得します。
    rp = xformFn.rotatePivot(MSpace::kTransform);
    sp = xformFn.scalePivot(MSpace::kTransform);
    rpt = xformFn.rotatePivotTranslation(MSpace::kTransform);
    spt = xformFn.scalePivotTranslation(MSpace::kTransform);
    ra = xformFn.rotateOrientation(MSpace::kTransform).asEulerRotation();

    // joint ノードなら、ジョイントの方向とラベル付けを取得します。
    if (m_MayaJointFlag)
    {
        MFnIkJoint jointFn(m_XformPath);
        jointFn.getOrientation(jo);

        int labelSide = 0;
        jointFn.findPlug("side").getValue(labelSide);
        switch (labelSide)
        {
        default:
        case 0: m_Side = SIDE_CENTER; break;
        case 1: m_Side = SIDE_LEFT  ; break;
        case 2: m_Side = SIDE_RIGHT ; break;
        case 3: m_Side = SIDE_NONE  ; break;
        }

        int labelType = Y_JOINT_LABEL_TYPE_NONE;
        jointFn.findPlug("type").getValue(labelType);
        if (labelType == Y_JOINT_LABEL_TYPE_OTHER)
        {
            MString otherType;
            jointFn.findPlug("otherType").getValue(otherType);
            m_Type = otherType.asChar();
            if (!RIsValidElementNameString(m_Type))
            {
                YShowError(&rscene, // Joint other type is wrong: %s
                    "joint ノードのその他のタイプに非 ASCII 文字やサポート外の記号が含まれています: {0} \n"
                    "「-（ハイフン）」「.（ドット）」「_（アンダーバー）」以外の記号は使用できません。",
                    "Non-ASCII text or other unsupported symbols are included in the Other Type of the joint node: {0} \n"
                    "Symbols other than the hyphen (-), underscore (_), and period (.) cannot be used.",
                    m_OrgName);
                return;
            }
        }
        else if (labelType != Y_JOINT_LABEL_TYPE_NONE)
        {
            m_Type = YGetJointLabelTypeString(static_cast<YJointLabelType>(labelType));
        }
    }

    // ピボットにエクスポートオプションの縮尺を掛けて保存します。
    m_RotatePivot      = rp  * yopt.m_InternalMagnify;
    m_ScalePivot       = sp  * yopt.m_InternalMagnify;
    m_RotatePivotTrans = rpt * yopt.m_InternalMagnify;
    m_ScalePivotTrans  = spt * yopt.m_InternalMagnify;

     // m_TranslateOffset は後( GetYNodeTransform() )で求めます。
    m_TranslateOffset = MVector::zero;
    //
    m_PivotMiddleOffset = (sp + spt - rp) * yopt.m_InternalMagnify;
    m_PivotMiddleOffsetFlag = !m_PivotMiddleOffset.isEquivalent(MVector::zero);

    // 回転順とその変更フラグを保存します。
    m_RotateOrder = r.order;
    m_SpecialRotateOrderFlag = (r.order != MEulerRotation::kXYZ);

    // 回転軸とその変更フラグを保存します。
    m_RotateAxis = ra;
    m_RotateAxisFlag = (ra != MEulerRotation::identity);

    // ジョイントの向きとその変更フラグを保存します。
    m_JointOrient = jo;
    m_JointOrientFlag = (jo != MEulerRotation::identity);
    // 回転順、回転軸、ジョイントの向きのどれかでもたっていたら回転の補正を行い、
    // アニメーションを強制的に焼き付けを行います。
    m_RotateAdjustFlag = (m_SpecialRotateOrderFlag ||
        m_RotateAxisFlag || m_JointOrientFlag);

    //-----------------------------------------------------------------------------
    // 回転の制限を取得します。
    static const char* limitMinStrs[] =
    {
        "msxe", "msye", "msze",
        "mrxe", "mrye", "mrze",
        "mtxe", "mtye", "mtze",
    };
    static const char* limitMaxStrs[] =
    {
        "xsxe", "xsye", "xsze",
        "xrxe", "xrye", "xrze",
        "xtxe", "xtye", "xtze",
    };
    for (int paramIdx = 0; paramIdx < RBone::PARAM_COUNT; ++paramIdx)
    {
        bool limitMin, limitMax;
        xformFn.findPlug(limitMinStrs[paramIdx]).getValue(limitMin);
        xformFn.findPlug(limitMaxStrs[paramIdx]).getValue(limitMax);
        // 制限が設定されているかどうかを保存します。
        m_XformLimitFlag[paramIdx] = (limitMin || limitMax);
    }

    //-----------------------------------------------------------------------------
    // 子ノードにジオメトリコンストレイントが設定されているか調べる
    int childCount = xformFn.childCount();
    for (int ichild = 0; ichild < childCount; ++ichild)
    {
        if (xformFn.child(ichild).apiType() == MFn::kGeometryConstraint)
        {
            m_SpecialConstraintFlag = true;
            #ifdef DEBUG_PRINT_SW
            cerr << "special constraint: " << m_OrgName << R_ENDL;
            #endif
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // get billboard
    MPlug billPlug = FindPlugQuiet(xformFn, "nw4fBillboard", &status);
    if (status)
    {
        int bill;
        billPlug.getValue(bill);
        if (0 <= bill && bill < RBone::BILLBOARD_COUNT)
        {
            m_Billboard = static_cast<RBone::Billboard>(bill);
        }
    }

    //-----------------------------------------------------------------------------
    // スケルタルアニメーションのバイナリ出力フラグを取得します。
    MPlug binarizeScalePlug = FindPlugQuiet(xformFn, "NW4F_BinarizeScale", &status);
    if (status)
    {
        binarizeScalePlug.getValue(m_BinarizesScale);
    }
    MPlug binarizeRotatePlug = FindPlugQuiet(xformFn, "NW4F_BinarizeRotate", &status);
    if (status)
    {
        binarizeRotatePlug.getValue(m_BinarizesRotate);
    }
    MPlug binarizeTranslatePlug = FindPlugQuiet(xformFn, "NW4F_BinarizeTranslate", &status);
    if (status)
    {
        binarizeTranslatePlug.getValue(m_BinarizesTranslate);
    }

    //-----------------------------------------------------------------------------
    // レイヤー設定を取得します。
    bool overrideFlag;
    // "overrideEnabled"アトリビュートの値を取得し、
    // 以下の３つが有効かどうかを判断します。
    xformFn.findPlug("overrideEnabled").getValue(overrideFlag);
    if (overrideFlag)
    {
        // テクスチャマッピングの解除設定
        xformFn.findPlug("overrideTexturing").getValue(m_LayerTexturingFlag);
        // アニメーションの再生 オン/オフ 設定
        xformFn.findPlug("overridePlayback").getValue(m_LayerPlaybackFlag);
        // 可視性
        xformFn.findPlug("overrideVisibility").getValue(m_LayerVisibilityFlag);
    }

    //-----------------------------------------------------------------------------
    // ノードの可視性を取得します。

    // "visibility"プラグを取得し、アニメーションが接続されているかどうかを確認します。
    m_XformVisibilityPlug = xformFn.findPlug("visibility");
    ChannelInput visChan(m_XformVisibilityPlug);
    m_VisibilityAnimFlag = visChan.IsEffective();

    if (m_OrgKind == KindMesh)
    {
        // transform ノードがメッシュの場合ば、
        // 参照しているシェイプの"visibility"プラグも取得し、そのプラグにアニメーション
        // が接続されているかどうか確認します。
        m_ShapeVisibilityPlug = MFnDependencyNode(m_ShapePath.node()).findPlug("visibility");
        ChannelInput shapeVisChan(m_ShapeVisibilityPlug);
        m_ShapeVisibilityAnimFlag = shapeVisChan.IsEffective();
    }
    //g_Err << "vis anim: " << m_OrgName << ": " << m_VisibilityAnimFlag << ", " << m_ShapeVisibilityAnimFlag << R_ENDL;

    //-----------------------------------------------------------------------------
    // get primitive type & strip enable
    if (m_OrgKind == KindMesh)
    {
        //MPlug primTypePlug = FindPlugQuiet(xformFn, "nw4fPrimType", &status);
        //if (status)
        //{
        //  int primType;
        //  primTypePlug.getValue(primType);
        //  if (primType == 1)
        //  {
        //      m_PrimType = RPrimitive::LINE_STRIP;
        //  }
        //  else if (primType == 2)
        //  {
        //      m_PrimType = RPrimitive::POINTS;
        //  }
        //  else
        //  {
        //      m_PrimType = RPrimitive::TRIANGLE_FAN;
        //  }
        //}
    }

    //-----------------------------------------------------------------------------
    // get deformer
    if (m_OrgKind == KindMesh)
    {
        // メッシュのデフォーマオブジェクト(SkinClusterやBlendShape)を取得します。
        FindDeformerNodes(m_DeformerObjs, m_ShapePath.node());
        const int deformerCount = static_cast<int>(m_DeformerObjs.size());
        for (int iDeformer = 0; iDeformer < deformerCount; ++iDeformer)
        {
            // デフォーマオブジェクトとそのタイプを取得します。
            MObject deformerObj = m_DeformerObjs[iDeformer];
            MFn::Type type = deformerObj.apiType();
            // デフォーマが無効に設定されていた場合に警告を表示し、
            // そのデフォーマは無視します。
            if (!IsNodeStateEffective(deformerObj))
            {
                if (!yopt.m_CheckElementFlag)
                {
                    YShowWarning(&rscene, // Deformer is not effective: %s -> %s
                        "デフォーマが無効（ノード状態アトリビュートが「エフェクトなし」）になっています: {0} -> {1} \n"
                        "デフォーマによるアニメーションは出力されません。",
                        "The deformer is disabled. (The Node State attribute is HasNoEffect.): {0} -> {1} \n"
                        "Animation using the deformer cannot be exported.",
                        MFnDependencyNode(deformerObj).name().asChar(), m_OrgName);
                }
                continue;
            }
            // デフォーマのタイプ別にオブジェクトを保存し、フラグを設定します。
            if (type == MFn::kSkinClusterFilter)
            {
                m_IsSkinned = true;
                m_SkinClusterObj = deformerObj;
            }
            else if (type == MFn::kBlendShape)
            {
                m_BlendShapeFlag = true;
                m_BlendShapeObj = deformerObj;
            }
        }
        // スキンかブレンドシェイプのどちらかでもデフォーマが適用されているかどうかの
        // フラグを設定します。
        m_DeformerFlag = m_IsSkinned || m_BlendShapeFlag;
    }

    //-----------------------------------------------------------------------------
    // スキンのジョイントとして使用されているか調べる

    // transform ノードの "worldMatrix" アトリビュートに接続されているオブジェクトを
    // 列挙し、そのオブジェクトの中に SkinClusterFilter が含まれているか確認します。
    // SkinClusterFilter が含まれていたらこのノードはスキンのジョイントとして使用
    // されているということなので、m_SkinInfluenceFlag に true を設定して印をついける。
    MPlugArray clusterPlugs;
    MPlug matrixPlug = xformFn.findPlug("worldMatrix");
    matrixPlug.elementByLogicalIndex(0).connectedTo(clusterPlugs, false, true);
    for (int iplug = 0; iplug < static_cast<int>(clusterPlugs.length()); ++iplug)
    {
        MFn::Type type = clusterPlugs[iplug].node().apiType();
        if (type == MFn::kSkinClusterFilter)
            //|| type == MFn::kJointCluster
        {
            m_SkinInfluenceFlag = true;
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // mesh ノードのアトリビュートを取得します。
    if (m_OrgKind == KindMesh)
    {
        //-----------------------------------------------------------------------------
        // 出力する頂点カラー属性のカラーセット名配列を取得します。
        if (!GetMeshVertexColorAttrs(m_MeshColSets, rscene, m_ShapePath))
        {
            return;
        }

        //-----------------------------------------------------------------------------
        // デフォルトの UV セット名を取得します。
        m_DefaultUvSet = GetDefaultUvSetName(m_ShapePath);

        //-----------------------------------------------------------------------------
        // get shading group
        GetShadingGroupsAndComponents(m_SGObjs, m_CompObjs, m_ShapePath);
    }

    //-----------------------------------------------------------------------------
    // finish
    if (pStatus != NULL)
    {
        *pStatus = MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // debug
//  g_Err << "YNode: " << m_XformPath.partialPathName() << " ";
//  if (m_ShapePath.isValid()) g_Err << m_ShapePath.partialPathName() << " ";
//  g_Err << R_ENDL;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief Maya から取得した回転を出力用に補正します。
//-----------------------------------------------------------------------------
void YNode::AdjustRotate(MEulerRotation& eulerRotate)
{
    if (m_RotateAdjustFlag)
    {
        if (m_RotateAxisFlag)
        {
            eulerRotate = m_RotateAxis * eulerRotate;
        }

        if (m_JointOrientFlag)
        {
            eulerRotate *= m_JointOrient;
        }

        if (eulerRotate.order != MEulerRotation::kXYZ)
        {
            eulerRotate.reorderIt(MEulerRotation::kXYZ);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya から取得した移動を出力用に補正します。
//-----------------------------------------------------------------------------
void YNode::AdjustTranslate(MVector& translate, const MEulerRotation& eulerRotate)
{
    translate += m_TranslateOffset;
    if (m_PivotMiddleOffsetFlag)
    {
        translate += m_PivotMiddleOffset.rotateBy(eulerRotate);
    }
}

//=============================================================================
//! @brief 頂点属性情報のクラスです。
//=============================================================================
class YVtxAttrInfo
{
public:
    bool m_ExistFlag; //!< 頂点属性が存在していれば true です。デフォルトは false です。
    int m_CompCount; //!< 成分数です。デフォルトは 3 です。
    bool m_IsRa; //!< RA の 2 成分を出力するなら true です。
    RPrimVtx::ValueType m_ValueType; //!< 値の型です。デフォルトは VALUE_FLOAT です。

public:
    //! コンストラクタです。
    YVtxAttrInfo()
    : m_ExistFlag(false),
      m_CompCount(3),
      m_IsRa(false),
      m_ValueType(RPrimVtx::VALUE_FLOAT)
    {
    }

    //! 同値であれば true を返します。
    bool operator==(const YVtxAttrInfo& rhs) const
    {
        return (
            m_ExistFlag == rhs.m_ExistFlag &&
            m_CompCount == rhs.m_CompCount &&
            m_IsRa      == rhs.m_IsRa      &&
            m_ValueType == rhs.m_ValueType);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const YVtxAttrInfo& rhs) const
    {
        return !(*this == rhs);
    }

    //! @brief 出力ストリームに頂点属性情報を出力します（デバッグ用）。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] info 頂点属性情報です。
    //!
    //! @return 出力ストリームを返します。
    //!
    friend std::ostream& operator<<(std::ostream& os, const YVtxAttrInfo& info)
    {
        os << info.m_ExistFlag << ' ' << info.m_CompCount << ' ' << info.m_IsRa << ' ' << info.m_ValueType;
        return os;
    }
};

//! @brief 頂点属性情報配列の定義です。
typedef std::vector<YVtxAttrInfo> YVtxAttrInfoArray;

//-----------------------------------------------------------------------------
//! @brief 頂点属性情報配列中に存在する頂点属性があるなら true を返します。
//!
//! @param[in] infos 頂点属性情報配列です。
//!
//! @return 存在する頂点属性が 1 つでもあれば true を返します。
//-----------------------------------------------------------------------------
static bool SomeVtxAttrExists(const YVtxAttrInfoArray& infos)
{
    for (int iInfo = 0; iInfo < static_cast<int>(infos.size()); ++iInfo)
    {
        if (infos[iInfo].m_ExistFlag)
        {
            return true;
        }
    }
    return false;
}

//=============================================================================
//! @brief シェイプデータのクラスです。
//=============================================================================
class ShapeData // ShapeDataT
{
public:
    //-----------------------------------------------------------------------------
    // node

    MDagPath m_MeshPath; //!< mesh ノードの DAG パスです。
    MObject m_CompObj; //!< シェイプに含まれるフェース群のコンポーネントです。

    //-----------------------------------------------------------------------------
    // attr

    //! @brief 出力用の名前です。
    //!        属するノードの出力名 + "__" + 適用されているマテリアルの出力名の形式です。
    std::string m_Name;

    //! @brief オリジナルの名前です。警告やエラー表示に使用します。
    //!        属するノード名 + "__" + 適用されているマテリアル名の形式です。
    std::string m_OrgName;

    int m_Index; //!< 出力用のインデックスです。
    int m_OrgIndex; //!< m_ShapeDatas 内のインデックスです。

    //! @brief シェイプが属するノードの m_pOutYNodes 内のインデックス配列です。
    MIntArray m_NodeIndexes;

    int m_MaterialIndex; //!< 適用されているマテリアルの m_YMaterials 内のインデックスです。
    int m_SortIndex; //!< ソート用のインデックスです。

    int m_Priority; //!< 描画優先度です。

    RPrimitive::PrimType m_PrimType; //!< プリミティブタイプです。

    ROrientedBB m_OrientedBB; //!< 有向バウンディングボックスです。

    int m_MaxBoneCountPerVtx; //!< 一頂点あたりに適用されるボーンの最大数です。

    RShape::SkinningMode m_SkinningMode; //!< スキニングモードです。

    //-----------------------------------------------------------------------------
    // component size
    MIntArray m_VtxCounts; //!< Maya 上のフェースごとの頂点数配列です。
    int m_TriangleCount; //!< 三角形数です。
    RIntArray m_TriPatterns; //!< Maya 上のフェースごとの三角形分割パターンです。

    //-----------------------------------------------------------------------------
    // vertex attr flag & size

    //! @brief 法線の出力フラグです。
    //!        適用されているマテリアルの YMaterial::m_VtxNrmFlag の設定値がコピーされます。
    bool m_VtxNrmFlag;

    //! @brief 接線を取得するテクスチャ座標のインデックス配列です。
    //!        -1 ならカレントの UV セットから接線を取得します。
    //!        接線を出力しない場合は長さ 0 です。
    //!        適用されているマテリアルの YMaterial::m_VtxTanTexIdxs の設定値がコピーされます。
    RIntArray m_VtxTanTexIdxs;

    //! @brief 接線を取得する UV セット名の配列の配列です。
    //!        ベースシェイプが先頭で、ベースシェイプ以外のキーシェイプが
    //!        存在すればその後に格納します。
    //!        接線を出力しない場合は空です。
    MStringArray m_TanUvSetNamess[RPrimVtx::VTX_TAN_MAX];

    YVtxAttrInfoArray m_VtxColInfos; //!< 頂点カラー属性情報配列です。

    int m_VtxTexCount; //!< UV セット数です。
    RStringArray m_UvAttribNames; //!< 各 UV セットの頂点属性名配列です。空文字ならヒント情報インデックスから決定します。
    RIntArray m_UvHintIdxs; //!< 各 UV セットのヒント情報インデックス配列です。

    YVtxAttrInfoArray m_VtxUsrInfos; //!< ユーザー頂点属性情報配列です。

    //-----------------------------------------------------------------------------
    // vertex attr table
    RVec3Array m_VtxPoss; //!< 頂点座標配列です。
    RVec3Array m_VtxNrms; //!< 法線配列です。
    RVec4Array m_VtxTanss[RPrimVtx::VTX_TAN_MAX]; //!< 接線配列の配列です。
    RVec4Array m_VtxBinss[RPrimVtx::VTX_TAN_MAX]; //!< 従法線配列の配列です。
    RVec4Array m_VtxColss[RPrimVtx::VTX_COL_MAX]; //!< 頂点カラー配列の配列です。
    RVec2Array m_VtxTexss[RPrimVtx::VTX_TEX_MAX]; //!< テクスチャ座標配列の配列です。
    RVec4Array m_VtxUsrss[RPrimVtx::VTX_USR_MAX]; //!< ユーザー頂点属性配列の配列です。

    std::vector<RVec3Array> m_TargetPosss; //!< ターゲットシェイプの頂点座標配列の配列です。
    std::vector<RVec3Array> m_TargetNrmss; //!< ターゲットシェイプの法線配列の配列です。
    std::vector<RVec4Array> m_TargetTansss[RPrimVtx::VTX_TAN_MAX]; //!< ターゲットシェイプの接線配列の配列の配列です。
    std::vector<RVec4Array> m_TargetBinsss[RPrimVtx::VTX_TAN_MAX]; //!< ターゲットシェイプの従法線配列の配列の配列です。

    //! @brief ターゲットシェイプの頂点カラー配列の配列の配列です。
    std::vector<RVec4Array> m_TargetColsss[RPrimVtx::VTX_COL_MAX];

    //-----------------------------------------------------------------------------
    // vertex attr indexes
    MIntArray m_VtxMtxIdxs; //!< 頂点行列インデックス配列です。
    MIntArray m_VtxPosIdxs; //!< 頂点座標インデックス配列です。
    MIntArray m_VtxNrmIdxs; //!< 法線インデックス配列です。
    MIntArray m_VtxTanIdxss[RPrimVtx::VTX_TAN_MAX]; //!< 接線インデックス配列の配列です。
    MIntArray m_VtxBinIdxss[RPrimVtx::VTX_TAN_MAX]; //!< 従法線インデックス配列の配列です。
    MIntArray m_VtxColIdxss[RPrimVtx::VTX_COL_MAX]; //!< 頂点カラーインデックス配列の配列です。
    MIntArray m_VtxTexIdxss[RPrimVtx::VTX_TEX_MAX]; //!< テクスチャ座標インデックス配列の配列です。
    MIntArray m_VtxUsrIdxss[RPrimVtx::VTX_USR_MAX]; //!< ユーザー頂点属性インデックス配列の配列です。

    //-----------------------------------------------------------------------------
    // ブレンドシェイプ
    RShapeUpdate m_ShapeUpdate; //!< ブレンドシェイプが更新する頂点属性の情報です。

    //-----------------------------------------------------------------------------
    // volume
    MPoint m_VolumeMin; //!< ボリュームの XYZ 成分の最小値です。
    MPoint m_VolumeMax; //!< ボリュームの XYZ 成分の最大値です。
    double m_VolumeR; //!< ボリュームの半径です。

public:
    //! コンストラクタです。
    ShapeData()
    : m_PrimType(RPrimitive::TRIANGLE_FAN)
    {
        // shape vertex attr information
        m_VtxColInfos = YVtxAttrInfoArray(RPrimVtx::VTX_COL_MAX);
        m_VtxUsrInfos = YVtxAttrInfoArray(RPrimVtx::VTX_USR_MAX);
    }

    //! メモリを解放します。
    void FreeMemory()
    {
        m_NodeIndexes.clear();

        m_VtxCounts.clear();
        m_TriangleCount = 0;
        m_TriPatterns.clear();

        m_VtxPoss.clear();
        m_VtxNrms.clear();

        m_VtxMtxIdxs.clear();
        m_VtxPosIdxs.clear();
        m_VtxNrmIdxs.clear();

        m_TargetPosss.clear();
        m_TargetNrmss.clear();

        for (int iTanSet = 0; iTanSet < RPrimVtx::VTX_TAN_MAX; ++iTanSet)
        {
            m_VtxTanss[iTanSet].clear();
            m_VtxBinss[iTanSet].clear();
            m_VtxTanIdxss[iTanSet].clear();
            m_VtxBinIdxss[iTanSet].clear();
            m_TargetTansss[iTanSet].clear();
            m_TargetBinsss[iTanSet].clear();
        }

        for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
        {
            m_VtxColss[iCol].clear();
            m_VtxColIdxss[iCol].clear();
            m_TargetColsss[iCol].clear();
        }
        for (int iTex = 0; iTex < RPrimVtx::VTX_TEX_MAX; ++iTex)
        {
            m_VtxTexss[iTex].clear();
            m_VtxTexIdxss[iTex].clear();
        }
        for (int iUa = 0; iUa < RPrimVtx::VTX_USR_MAX; ++iUa)
        {
            m_VtxUsrss[iUa].clear();
            m_VtxUsrIdxss[iUa].clear();
        }
    }
};

//! @brief シェイプデータ配列の定義です。
typedef std::vector<ShapeData> ShapeDataArray;

//! @brief シェイプデータのポインタ配列の定義です。
typedef std::vector<ShapeData*> ShapeDataPtrArray;

//=============================================================================
//! @brief マテリアル情報のクラスです。
//=============================================================================
class YMatInfo
{
public:
    MObject m_SGObj; //!< シェーディンググループのオブジェクトです。
    YVtxAttrInfoArray m_MatVtxColInfos; //!< 頂点カラー属性情報配列です。
    YVtxAttrInfoArray m_MatVtxUsrInfos; //!< ユーザー頂点属性情報配列です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] sgObj シェーディンググループのオブジェクトです。
    //!
    explicit YMatInfo(const MObject& sgObj)
    : m_SGObj(sgObj)
    {
        m_MatVtxColInfos = YVtxAttrInfoArray(RPrimVtx::VTX_COL_MAX);
        m_MatVtxUsrInfos = YVtxAttrInfoArray(RPrimVtx::VTX_USR_MAX);
    }
};

//! @brief マテリアル情報配列の定義です。
typedef std::vector<YMatInfo> YMatInfoArray;

//=============================================================================
//! @brief Maya 用のマテリアルのクラスです。
//=============================================================================
class YModel; // マテリアルから参照するための宣言です。

class YMaterial : public RMaterial // YMaterialT
{
public:
    //-----------------------------------------------------------------------------
    // internal node

    MObject m_SGObj; //!< シェーディンググループのオブジェクトです。
    MObject m_ShaderObj; //!< m_SGObj に接続されているシェーダオブジェクトです。
    MFn::Type m_ShaderType; //!< シェーダオブジェクトのタイプです。

    //! @brief 最初に処理した mesh ノードの DAG パスです。
    //!        UV linking のエラー表示用に使用されます。
    MDagPath m_FirstMeshPath;

    //-----------------------------------------------------------------------------
    // index & name

    int m_OrgIndex; //!< オリジナルのインデックスです。ymodel.m_YMaterials 内のインデックスです。
    int m_Index; //!< 出力用のインデックスです。ymodel.m_pOutYMaterials 内のインデックスです。
    std::string m_OrgName; //!< オリジナルのマテリアル名です。

    //-----------------------------------------------------------------------------
    // internal attr

    //! @brief レンダーステートの自動設定フラグです。
    //!        マテリアルの設定からレンダーステートを自動的に設定するなら true です。
    bool m_AutoRenderStateFlag;

    //! @brief デフォルトマテリアルフラグです。
    //!        シェーダが接続されていない時にプラグインが追加したデフォルトマテリアルなら true です。
    bool m_DefaultFlag;

    int m_Priority; //!< 描画優先度です。0（自動）、1（高優先度）- 255（低優先度）の範囲で設定されます。

    //-----------------------------------------------------------------------------
    // internal color

    float m_DiffuseCoeff; //!< ディフューズカラーに掛ける係数です。現在は常に 1.0 です。
    MObject m_ColorTexObj; //!< color プラグに接続されたテクスチャオブジェクトです。

    //! @brief マテリアルの transparency プラグを使用するなら true です。
    //!        transparency プラグにテクスチャが接続されていた場合、
    //!        そのテクスチャオブジェクトの alphaGain プラグを使用するので false となります。
    bool m_UseTransparencyPlug;
    MObject m_TransparencyTexObj; //!< transparency プラグに接続されたテクスチャオブジェクトです。

    bool m_SpecularPlugFlag; //!< specularColor プラグを使用するなら true です。

    //-----------------------------------------------------------------------------
    // vertex

    bool m_VtxNrmFlag; //!< 法線を出力するなら true です。現在常に true です。

    //! @brief 接線を取得するテクスチャ座標のインデックス配列です。
    //!        uvSetFamilyNames 配列の要素を指します。
    //!        -1 ならカレントの UV セットから接線を取得します。
    //!        接線を出力しない場合は長さ 0 です。
    RIntArray m_VtxTanTexIdxs;

    YVtxAttrInfoArray m_MatVtxColInfos; //!< 頂点カラーの情報配列です。

    int m_VtxTexCount; //!< UV セット数です。

    YVtxAttrInfoArray m_MatVtxUsrInfos; //!< ユーザー頂点データの情報配列です。

    //-----------------------------------------------------------------------------
    // texture

    //! @brief マテリアル内で何番目のテクスチャを法線マップに使用するかを示すインデックス配列です。
    //!        法線マップを使用しない場合は長さ 0 です。
    RIntArray m_NrmMapIdxs;

    MStringArray m_UvSetFamilyNames; //!< マテリアルが使用する UV セットのファミリー名配列です。
    RIntArray m_UvHintIdxs; //!< m_UvSetFamilyNames に対応する UV セットのヒント情報インデックス配列です。

    //! @brief 各サンプラが使用する UV セットの m_UvSetFamilyNames 内のインデックス配列です。
    //!        サンプラが UV セットを使用しない場合（反射マッピング）は -1 です。
    RIntArray m_SamplerUvSetIdxs;

    bool m_TexSrtAnimFlag; //!< テクスチャ SRT アニメーションが設定されているサンプラがあれば true です。
    bool m_TexPatAnimFlag; //!< テクスチャパターンアニメーションが設定されているサンプラがあれば true です。

    //-----------------------------------------------------------------------------
    // カラーアニメーション

    //! @brief カラーアニメーションが設定されていれば true です。
    bool m_ColAnimFlag;

    //! @brief カラーアニメーションのパラメータに対応する Maya のプラグ配列です。
    MPlug m_AnimPlugs[RMaterial::COLOR_PARAM_COUNT];

    //! @brief カラーアニメーションのパラメータに対応するチャンネル入力情報配列です。
    ChannelInput m_AnimChans[RMaterial::COLOR_PARAM_COUNT];

    //! @brief カラーアニメーションのパラメータに対応するアニメーションカーブ配列です。
    YAnimCurve m_Anims[RMaterial::COLOR_PARAM_COUNT];

public:
    YMaterial()
    : m_AutoRenderStateFlag(false),
      m_DefaultFlag(false),
      m_Priority(0),

      m_DiffuseCoeff(1.0f),
      m_UseTransparencyPlug(true),
      m_SpecularPlugFlag(false),

      m_VtxNrmFlag(true),
      m_VtxTexCount(0),

      m_TexSrtAnimFlag(false),
      m_TexPatAnimFlag(false),

      m_ColAnimFlag(false)
    {
        // material vertex attr information
        m_MatVtxColInfos = YVtxAttrInfoArray(RPrimVtx::VTX_COL_MAX);
        m_MatVtxUsrInfos = YVtxAttrInfoArray(RPrimVtx::VTX_USR_MAX);
    }

    bool HasTexture() const { return !m_Samplers.empty(); }
    bool IsSameElement(const YMaterial& other) const;

    //! @brief ユーザーデータを取得します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //!
    void GetUserData(RScene* pScene)
    {
        if (!m_SGObj.isNull())
        {
            ::GetUserData(m_UserDatas, *pScene, m_SGObj);
        }
    }
};

//! @brief Maya 用のマテリアル配列の定義です。
typedef std::vector<YMaterial> YMaterialArray;

//! @brief Maya 用のマテリアルのポインタ配列の定義です。
typedef std::vector<YMaterial*> YMaterialPtrArray;

//-----------------------------------------------------------------------------
//! @brief 他の Maya 用のマテリアルと要素の値が同じなら true を返します。
//!
//! @param[in] other 比較する他の Maya 用のマテリアルです。
//!
//! @return 要素の値が同じなら true を返します。
//-----------------------------------------------------------------------------
bool YMaterial::IsSameElement(const YMaterial& other) const
{
    const int samplerCount = static_cast<int>(m_Samplers.size());
    if (!RMaterial::IsSameElement(other) ||

        m_VtxNrmFlag  != other.m_VtxNrmFlag  ||
        m_VtxTexCount != other.m_VtxTexCount ||
        !RIsSameArray(m_VtxTanTexIdxs , other.m_VtxTanTexIdxs ) ||
        !RIsSameArray(m_MatVtxColInfos, other.m_MatVtxColInfos) ||
        !RIsSameArray(m_MatVtxUsrInfos, other.m_MatVtxUsrInfos) ||

        samplerCount != static_cast<int>(other.m_Samplers.size()))
    {
        return false;
    }

    // sampler
    for (int iSampler = 0; iSampler < samplerCount; ++iSampler)
    {
        const RSampler& sampler      = m_Samplers[iSampler];
        const RSampler& otherSampler = other.m_Samplers[iSampler];
        if (sampler                   != otherSampler                   ||
            sampler.m_TexSrtAnimIndex != otherSampler.m_TexSrtAnimIndex ||
            sampler.m_TexPatAnimIndex != otherSampler.m_TexPatAnimIndex)
        {
            return false;
        }
    }

    return true;
}

//=============================================================================
//! @brief Maya 用のモデルのクラスです。
//=============================================================================
class YScene; // モデルから参照するための宣言です。

class YModel // YModelT
{
public:
    //-----------------------------------------------------------------------------
    // scene
    YScene* m_pYScene; //!< シーンデータへのポインターです。

    //-----------------------------------------------------------------------------
    // ynode

    //! @brief Maya から取得したノードのポインタ配列です。
    //!        出力しないノードは取り除かれています。
    //!        SetYNodeConnection で出力順にソートされます。
    YNodePtrArray m_pYNodes;

    //! @brief 出力用の階層構造のノードのポインタ配列です。
    //!        現在 m_pYNodes と同じです。
    YNodePtrArray m_pOutYNodes;

    //! @brief 階層構造のルートノードのポインタ（m_pYNodes 内のポインタ）です。
    YNode* m_pRootYNode;

    RSkeleton m_Skeleton; //!< スケルトンです。

    YIkHandleArray m_IkHandles; //!< IK ハンドル配列です。

    //-----------------------------------------------------------------------------
    // shape
    ShapeDataArray m_ShapeDatas; //!< シェイプデータ配列です。

    //! @brief 出力用シェイプデータのポインタ配列です。
    //!        ymodel.m_ShapeDatas 内の要素へのポインタです。
    //!        ボーンインデックス順にソートされます。
    ShapeDataPtrArray m_pOutShapeDatas;

    int m_SmoothSkinningShapeCount; //!< スムーススキニングのシェイプ数です。
    int m_RigidSkinningShapeCount; //!< リジッドスキニングのシェイプ数です。
    int m_SmoothSkinningMtxCount; //!< スムーススキニングのシェイプ数です。
    int m_RigidSkinningMtxCount; //!< リジッドスキニングのシェイプ数です。
    int m_TotalTriangleCount; //!< 出力した三角形の総数です。
    int m_TotalIndexCount; //!< 出力した頂点インデックスの総数です。
    int m_TotalVertexCount; //!< 出力した頂点の総数です。
    int m_TotalProcessVertexCount; //!< 頂点シェーダが処理する頂点の総数です。

    //-----------------------------------------------------------------------------
    // vertex
    MPointArray m_GlobalPoss; //!< 全シェイプのモデル座標の配列です。

    RVtxMtxArray m_VtxMtxs; //!< モデルで使用されている頂点行列の配列です。

    //-----------------------------------------------------------------------------
    // material
    YMatInfoArray m_MatInfos; //!< モデルで使用されているマテリアル情報配列です。

    YMaterialArray m_YMaterials; //!< Maya から取得したマテリアル配列です。

    //! @brief 出力用マテリアルのポインタ配列です。
    //!        m_YMaterials 内の要素へのポインタです。
    //!        マテリアル名のアルファベット順にソートされます。
    YMaterialPtrArray m_pOutYMaterials;

    //-----------------------------------------------------------------------------
    // texture
    TexNodeArray m_TexNodes; //!< Maya のテクスチャノード配列です。

    RImageArray m_TexImgs; //!< 単一モデルのテクスチャイメージ配列です。

    //! @brief 単一モデルの出力用テクスチャイメージのポインタ配列です。
    //!        m_TexImgs 内の要素へのポインタです。
    //!        テクスチャ名のアルファベット順にソートされます。
    RImagePtrArray m_pOutTexImgs;

    TexSrtAnimArray m_TexSrtAnims; //!< テクスチャ SRT アニメーション配列です。
    TexPatAnimArray m_TexPatAnims; //!< テクスチャパターンアニメーション配列です。

    //-----------------------------------------------------------------------------
    // シェイプアニメーション
    YBlendShapeDataArray m_BlendShapeDatas; //!< ブレンドシェイプデータ配列です。
    YVertexShapeAnimArray m_VertexShapeAnims; //!< 頂点シェイプアニメーション配列です。

    //-----------------------------------------------------------------------------
    // geometry range
    bool m_VolumeCullingEnable;
    MPoint m_GlobalPosMin;
    MPoint m_GlobalPosMax;

    //-----------------------------------------------------------------------------
    // ユーザーデータ
    MDagPath m_UserDataXformPath; //!< モデルのユーザーデータ用 transform ノードの DAG パスです。
    MDagPath m_FskUserDataXformPath; //!< スケルタルアニメーションのユーザーデータ用 transform ノードの DAG パスです。

    //-----------------------------------------------------------------------------
    // 環境オブジェクト群
    YEnvObjs m_YEnvObjs; //!< 環境オブジェクト群です。

public:
    //! コンストラクタです。
    explicit YModel(YScene* pYScene)
    : m_pYScene(pYScene),
      m_SmoothSkinningShapeCount(0),
      m_RigidSkinningShapeCount(0),
      m_SmoothSkinningMtxCount(0),
      m_RigidSkinningMtxCount(0),
      m_TotalTriangleCount(0),
      m_TotalIndexCount(0),
      m_TotalVertexCount(0),
      m_TotalProcessVertexCount(0),
      m_VolumeCullingEnable(false)
    {
        //cerr << "YModel()" << endl;
    }

    //! デストラクタです。
    virtual ~YModel()
    {
        //cerr << "~YModel()" << endl;
        const int nodeCount = static_cast<int>(m_pYNodes.size());
        for (int iNode = 0; iNode < nodeCount; ++iNode)
        {
            delete m_pYNodes[iNode];
        }
    }

    //! シーンの参照を返します。
    YScene& GetScene() const { return *m_pYScene; }

    //! 環境オブジェクト群の参照を返します。
    const YEnvObjs& GetEnvObjs() const { return m_YEnvObjs; }

    //! 環境オブジェクト群の参照を返します。
    YEnvObjs& GetEnvObjs() { return m_YEnvObjs; }
};

//! @brief Maya 用のモデル配列の定義です。
typedef std::vector<YModel> YModelArray;

//=============================================================================
//! @brief Maya 用のシーンデータのクラスです。
//=============================================================================
class YScene : public RScene // YSceneT
{
public:
    //-----------------------------------------------------------------------------
    // export option
    YExpOpt m_ExpOpt; //!< エクスポートオプションです。

    //-----------------------------------------------------------------------------
    // texture
    RImageArray m_AllTexImgs; //!< 出力する全モデルで使用するテクスチャイメージ配列です。
    RImagePtrArray m_pOutAllTexImgs; //!< 出力する全モデルで使用するテクスチャイメージの出力用ポインタ配列です。
    RStringArray m_MergedTexImgNames; //!< マージされたテクスチャイメージ名配列です。

    //-----------------------------------------------------------------------------
    // file
    RFileMoveArray m_TmpFileMoves; //!< TEMP フォルダに出力した中間ファイルのファイル移動情報です。

    //-----------------------------------------------------------------------------
    // camera
    YCamera m_MainCamera;

    //-----------------------------------------------------------------------------
    // profile
    clock_t m_GetClocks; //!< データ取得のクロック数です。
    clock_t m_ModelClocks; //!< モデル出力のクロック数です。
    clock_t m_TexClocks; //!< テクスチャ出力のクロック数です。
    clock_t m_AnimClocks; //!< アニメーション出力のクロック数です。
    clock_t m_OptimizeClocks; //!< 最適化のクロック数です。
    clock_t m_FormatClocks; //!< フォーマット変換のクロック数です。

public:
    //! コンストラクタです。
    YScene()
    : m_GetClocks(0),
      m_ModelClocks(0),
      m_TexClocks(0),
      m_AnimClocks(0),
      m_OptimizeClocks(0),
      m_FormatClocks(0)
    {
        SetJapaneseUi(::IsJapaneseUi());
    }

    //! デストラクタです。
    virtual ~YScene()
    {
    }

    //! エクスポートオプションの参照を返します。
    const YExpOpt& GetOpt() const { return m_ExpOpt; }

    //! エクスポートオプションの参照を返します。
    YExpOpt& GetOpt() { return m_ExpOpt; }
};

//=============================================================================
//! @brief Export コマンドのクラスです。
//!
//! Usage:
//!    NintendoExportCmd [flags]
//! Flags:
//!    -option (-op) <option_string> エクスポートオプションを指定します。
//!    -force (-f) 指定すると出力ファイルがすでに存在していた場合に強制的に上書きします。
//!                指定しなければ上書き確認ダイアログを表示します。
//=============================================================================
class NintendoExportCmd : public MPxCommand
{
public:
    NintendoExportCmd()
    {
    }
    virtual ~NintendoExportCmd()
    {
    }
    MStatus doIt(const MArgList& args);
    static void* creator();
    static MSyntax newSyntax();

public:
    MString m_OptionString; //!< コマンドに渡されたエクスポートオプション文字列です。

    //! @brief 強制上書きフラグです。
    //!        true なら出力ファイルがすでに存在していた場合に上書き確認をしません。
    //!        false なら出力ファイルがすでに存在していた場合に上書き確認ダイアログを表示します。
    bool m_ForceOverwrite;

private:
    MStatus parseArgs(const MArgList& args);

    static volatile bool s_IsExporting; //!< Export コマンド実行中なら true です。
    static volatile int s_ExportCount; //!< プラグインをロードしてから Export した回数です。
};

volatile bool NintendoExportCmd::s_IsExporting = false;
volatile int NintendoExportCmd::s_ExportCount = 0;

//-----------------------------------------------------------------------------
//! @brief Export コマンドを作成します。
//-----------------------------------------------------------------------------
void* NintendoExportCmd::creator() { return new NintendoExportCmd(); }

//-----------------------------------------------------------------------------
//! @brief Export コマンドの文法を作成します。
//-----------------------------------------------------------------------------
#define kOptionFlag             "op"
#define kOptionFlagL            "option"
#define kForceOverwriteFlag     "f"
#define kForceOverwriteFlagL    "force"

MSyntax NintendoExportCmd::newSyntax()
{
    MSyntax syntax;

    syntax.addFlag(kOptionFlag, kOptionFlagL, MSyntax::kString);
    syntax.addFlag(kForceOverwriteFlag, kForceOverwriteFlagL);
    //syntax.setObjectType(MSyntax::kStringObjects, 1);

    return syntax;
}

//-----------------------------------------------------------------------------
//! @brief Export コマンドの引数を解析します。
//-----------------------------------------------------------------------------
MStatus NintendoExportCmd::parseArgs(const MArgList& args)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // init
    MArgDatabase argData(syntax(), args, &status);
    if (!status)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // set default
    m_OptionString = "";
    m_ForceOverwrite = false;

    //-----------------------------------------------------------------------------
    // get option
    if (argData.isFlagSet(kOptionFlag))
    {
        status = argData.getFlagArgument(kOptionFlag, 0, m_OptionString);
        CheckStatusMsg(status, "Cannot get option");
    }

    //-----------------------------------------------------------------------------
    // get check overwrite
    if (argData.isFlagSet(kForceOverwriteFlag))
    {
        m_ForceOverwrite = true;
    }

    //cerr << "opt str: " << m_OptionString << endl;
    return MS::kSuccess;
}

//=============================================================================
//! @brief Export プラグイン情報取得コマンドのクラスです。
//!
//! Usage:
//!    NintendoExportInfoCmd [flags]
//! Flags:
//!    -pluginVer(-p) Export プラグインのバージョンを取得します。
//!    -interVer(-i) 中間ファイルのバージョンを取得します。
//!    -dateTime(-dt) 現在の日時の文字列を取得します。
//=============================================================================
class NintendoExportInfoCmd : public MPxCommand
{
public:
    NintendoExportInfoCmd()
    {
    }
    virtual ~NintendoExportInfoCmd()
    {
    }
    MStatus doIt(const MArgList& args);
    static void* creator();
    static MSyntax newSyntax();
};

//-----------------------------------------------------------------------------
//! @brief Export プラグイン情報取得コマンドを作成します。
//-----------------------------------------------------------------------------
void* NintendoExportInfoCmd::creator() { return new NintendoExportInfoCmd(); }

//-----------------------------------------------------------------------------
//! @brief Export プラグイン情報取得コマンドの文法を作成します。
//-----------------------------------------------------------------------------
#define kPluginVerFlag      "p"
#define kPluginVerFlagL     "pluginVer"
#define kInterVerFlag       "i"
#define kInterVerFlagL      "interVer"
#define kDateTimeFlag       "dt"
#define kDateTimeFlagL      "dateTime"

MSyntax NintendoExportInfoCmd::newSyntax()
{
    MSyntax syntax;

    syntax.addFlag(kPluginVerFlag, kPluginVerFlagL);
    syntax.addFlag(kInterVerFlag , kInterVerFlagL);
    syntax.addFlag(kDateTimeFlag , kDateTimeFlagL);

    return syntax;
}

//-----------------------------------------------------------------------------
//! @brief Export プラグイン情報取得コマンドを実行します。
//!
//! @param[in] args 引数リストです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoExportInfoCmd::doIt(const MArgList& args)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // init
    MArgDatabase argData(syntax(), args, &status);
    if (!status)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // set result
    clearResult();

    if (argData.isFlagSet(kPluginVerFlag))
    {
        setResult(ExporterVersion);
    }
    else if (argData.isFlagSet(kInterVerFlag))
    {
        setResult(RExpOpt::GetRootElementVersion().c_str());
    }
    else if (argData.isFlagSet(kDateTimeFlag))
    {
        setResult(RGetDateTimeString().c_str());
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 出力前後のスクリプトにエクスポートオプションの値を伝達するための環境変数を設定します。
//!
//! @param[in] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void SetOptionEnvVarForPrePostExportScript(YScene& yscene, const YExpOpt& yopt)
{
    //-----------------------------------------------------------------------------
    // clear
    // Maya 2012 以前では Python の環境変数が
    // MEL の putenv "NAME" "" でクリアされない問題への対処です。
    std::string cmd = "import os\n";
    cmd += "os.environ[\"NINTENDO_EXPORT_OPTION_preset\"] = \"\"\n";
    cmd += "os.environ[\"NINTENDO_EXPORT_OPTION_merge_fmd_path\"] = \"\"\n";
    cmd += "os.environ[\"NINTENDO_EXPORT_OPTION_merge_anim_folder\"] = \"\"\n";
    cmd += "os.environ[\"NINTENDO_EXPORT_OPTION_merge_anim_name\"] = \"\"\n";
    cmd += "os.environ[\"NINTENDO_EXPORT_OPTION_comment\"] = \"\"\n";
    cmd += "os.environ[\"NW4F_EXPORT_OPTION_preset\"] = \"\"\n";
    cmd += "os.environ[\"NW4F_EXPORT_OPTION_merge_fmd_path\"] = \"\"\n";
    cmd += "os.environ[\"NW4F_EXPORT_OPTION_merge_anim_folder\"] = \"\"\n";
    cmd += "os.environ[\"NW4F_EXPORT_OPTION_merge_anim_name\"] = \"\"\n";
    cmd += "os.environ[\"NW4F_EXPORT_OPTION_comment\"] = \"\"\n";
    MStatus status = MGlobal::executePythonCommand(cmd.c_str());
    if (!status)
    {
        YShowError(&yscene, "", "Failed to clear option environment variables"); // 通常は発生しない
    }

    //-----------------------------------------------------------------------------
    // header
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tool_name",
        std::string("Nintendo Export for Maya ") + MGlobal::mayaVersion().asChar(), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tool_version", std::string(ExporterVersion), true);

    //-----------------------------------------------------------------------------
    // preset
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_preset", yopt.m_PresetName, true);

    //-----------------------------------------------------------------------------
    // output
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_export_target",
        std::string((yopt.m_Target == RExpOpt::EXPORT_TARGET_ALL) ? "all" : "selection"), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_lod_export", RBoolStr(yopt.m_ExportsLod), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_fmd", RBoolStr(yopt.m_MergeFmdFlag), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_fmd_path", yopt.m_MergeFmdPath, true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_ftx", RBoolStr(yopt.m_MergeFtxFlag), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_anim", RBoolStr(yopt.m_MergeAnimFlag), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_anim_folder", yopt.m_MergeAnimFolder, true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_anim_name", yopt.m_MergeAnimName, true);

    //-----------------------------------------------------------------------------
    // general
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_magnify", RGetFloatStringForSettingFile(yopt.m_Magnify), true);
    static const char* const texSrtModeStrs[] = { "maya", "3dsmax", "softimage" };
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_texsrt_mode", texSrtModeStrs[yopt.m_TexSrtMode], true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_remove_namespace", RBoolStr(yopt.m_RemoveNamespace), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_comment", REscapeString(yopt.m_CommentText), true);

    //-----------------------------------------------------------------------------
    // animation
    static const char* const frameRangeStrs[] = { "all", "playback", "range" };
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_frame_range", frameRangeStrs[yopt.m_FrameRange], true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_start_frame", RGetNumberString(yopt.m_StartFrame), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_end_frame", RGetNumberString(yopt.m_EndFrame), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_loop_anim", RBoolStr(yopt.m_LoopAnim), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_bake_all_anim", RBoolStr(yopt.m_BakeAllAnim), true);
    const double frameStep = (yopt.m_FramePrecision != 0) ? 1.0 / yopt.m_FramePrecision : 1.0;
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_frame_precision", RGetFloatStringForSettingFile(frameStep), true);

    //-----------------------------------------------------------------------------
    // optimization
    static const char* const compressBoneStrs[] = { "none", "cull", "merge", "unite", "unite_all" };
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_compress_bone", compressBoneStrs[yopt.m_CompressBone], true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_unite_child", RBoolStr(yopt.m_UniteChild), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_compress_material", RBoolStr(yopt.m_CompressMaterial), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_compress_shape", RBoolStr(yopt.m_CompressShape), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_divide_submesh", RBoolStr(yopt.m_DivideSubmesh), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_divide_submesh_mode", REscapeString(yopt.m_DivideSubmeshMode), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_poly_reduction", RBoolStr(yopt.m_PolyReduction), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_poly_reduction_mode", REscapeString(yopt.m_PolyReductionMode), true);

    //-----------------------------------------------------------------------------
    // animation bake tolerance
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tolerance_scale", RGetFloatStringForSettingFile(yopt.m_TolS), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tolerance_rotate", RGetFloatStringForSettingFile(yopt.m_TolR), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tolerance_translate", RGetFloatStringForSettingFile(yopt.m_TolT), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tolerance_color", RGetFloatStringForSettingFile(yopt.m_TolC), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tolerance_tex_scale", RGetFloatStringForSettingFile(yopt.m_TolTexS), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tolerance_tex_rotate", RGetFloatStringForSettingFile(yopt.m_TolTexR), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_tolerance_tex_translate", RGetFloatStringForSettingFile(yopt.m_TolTexT), true);

    //-----------------------------------------------------------------------------
    // animation quantization tolerance
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_quantize_tolerance_scale", RGetFloatStringForSettingFile(yopt.m_QuantTolS), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_quantize_tolerance_rotate", RGetFloatStringForSettingFile(yopt.m_QuantTolR), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_quantize_tolerance_translate", RGetFloatStringForSettingFile(yopt.m_QuantTolT), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_quantize_tolerance_tex_scale", RGetFloatStringForSettingFile(yopt.m_QuantTolTexS), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_quantize_tolerance_tex_rotate", RGetFloatStringForSettingFile(yopt.m_QuantTolTexR), true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_quantize_tolerance_tex_translate", RGetFloatStringForSettingFile(yopt.m_QuantTolTexT), true);
}

//-----------------------------------------------------------------------------
//! @brief 出力前後のスクリプトに情報を伝達するための環境変数をクリアします。
//!        Maya 2012 以前では Python の環境変数が
//!        MEL の putenv "NAME" "" でクリアされない問題への対処です。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void ClearEnvVarForPrePostExportScript(const YExpOpt& yopt)
{
    std::string cmd = "import os\n";
    for (int fileTypeIdx = RExpOpt::FMD; fileTypeIdx < RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const RExpOpt::FileType fileType = static_cast<RExpOpt::FileType>(fileTypeIdx);
        const std::string ext = yopt.GetNoFmaExtension(fileType);
        const std::string upperExt = RGetUpperCaseString(ext.substr(0, 3));
        cmd += "os.environ[\"NINTENDO_EXPORT_"      + upperExt + "_FILE_PATH\"] = \"\"\n";
        cmd += "os.environ[\"NINTENDO_EXPORT_TEMP_" + upperExt + "_FILE_PATH\"] = \"\"\n";
        cmd += "os.environ[\"NW4F_EXPORT_"      + upperExt + "_FILE_PATH\"] = \"\"\n";
        cmd += "os.environ[\"NW4F_EXPORT_TEMP_" + upperExt + "_FILE_PATH\"] = \"\"\n";
    }
    cmd += "os.environ[\"NINTENDO_EXPORT_FTX_FILE_NAMES\"] = \"\"\n";
    cmd += "os.environ[\"NINTENDO_EXPORT_USED_FTX_FILE_NAMES\"] = \"\"\n";
    cmd += "os.environ[\"NW4F_EXPORT_FTX_FILE_NAMES\"] = \"\"\n";
    cmd += "os.environ[\"NW4F_EXPORT_USED_FTX_FILE_NAMES\"] = \"\"\n";
    MStatus status = MGlobal::executePythonCommand(cmd.c_str());
    //cerr << "cevfppes: " << status << ": [" << cmd << "]" << endl;
    R_UNUSED_VARIABLE(status);
}

//-----------------------------------------------------------------------------
//! @brief 出力前後のスクリプトに情報を伝達するためのファイルパス関連の環境変数を設定します。
//-----------------------------------------------------------------------------
static void SetFilePathEnvVarForPrePostExportScript(const YExpOpt& yopt)
{
    //-----------------------------------------------------------------------------
    // 出力ファイル名（拡張子なし）、出力フォルダ、テクスチャ出力フォルダです。
    SetEnvVarByMel("NINTENDO_EXPORT_OUTPUT_FILE_NAME", yopt.m_OutFileName, true);
    SetEnvVarByMel("NINTENDO_EXPORT_OUTPUT_FOLDER"   , yopt.m_OutFolderPath, true);
    SetEnvVarByMel("NINTENDO_EXPORT_FTX_FOLDER"      , yopt.m_TexFolderPath, true);
    SetEnvVarByMel("NINTENDO_EXPORT_TEMP_OUTPUT_FOLDER",
        (yopt.m_UsesTmpFolderForScript) ? yopt.m_TmpOutFolderPath : "", true);
    SetEnvVarByMel("NINTENDO_EXPORT_TEMP_FTX_FOLDER"   ,
        (yopt.m_UsesTmpFolderForScript) ? yopt.m_TmpTexFolderPath : "", true);

    //-----------------------------------------------------------------------------
    // 各中間ファイルのフルパスです（出力しない場合は空文字）。
    for (int fileTypeIdx = RExpOpt::FMD; fileTypeIdx < RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const RExpOpt::FileType fileType = static_cast<RExpOpt::FileType>(fileTypeIdx);
        const std::string ext = yopt.GetNoFmaExtension(fileType);
        const std::string upperExt = RGetUpperCaseString(ext.substr(0, 3));
        const std::string name = "NINTENDO_EXPORT_" + upperExt + "_FILE_PATH";
        const std::string value = (yopt.m_OutFileFlag[fileType]) ?
            yopt.GetFilePath(fileType) : "";
        SetEnvVarByMel(name, value, true);

        const std::string tmpName = "NINTENDO_EXPORT_TEMP_" + upperExt + "_FILE_PATH";
        const std::string tmpValue = (yopt.m_UsesTmpFolderForScript && yopt.m_OutFileFlag[fileType]) ?
            yopt.GetTmpFilePath(fileType) : "";
        SetEnvVarByMel(tmpName, tmpValue, true);
    }

    const std::string fmaFileName = (yopt.UsesSingleFma() && yopt.ExportsMaterialAnim()) ?
        yopt.GetFileName(RExpOpt::FCL) : "";
    SetEnvVarByMel("NINTENDO_EXPORT_FMA_FILE_PATH",
        (!fmaFileName.empty() ? yopt.m_OutFolderPath    + fmaFileName : ""), false);
    SetEnvVarByMel("NINTENDO_EXPORT_TEMP_FMA_FILE_PATH",
        (!fmaFileName.empty() ? yopt.m_TmpOutFolderPath + fmaFileName : ""), false);
}

//-----------------------------------------------------------------------------
//! @brief 出力前後のスクリプトに情報を伝達するためのアニメーション関連の環境変数を設定します。
//-----------------------------------------------------------------------------
static void SetAnimEnvVarForPrePostExportScript(const YExpOpt& yopt)
{
    SetEnvVarByMel("NINTENDO_EXPORT_FRAME_COUNT", RGetNumberString(yopt.m_OutFrameCount), false);
    SetEnvVarByMel("NINTENDO_EXPORT_SUB_FRAME_COUNT", RGetNumberString(yopt.m_SubFrameCount), false);
}

//-----------------------------------------------------------------------------
//! @brief 出力前後のスクリプトに情報を伝達するための環境変数を設定します。
//!
//! @param[in] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//! @param[in] isPost 出力後のスクリプト用なら true を指定します。
//! @param[in] exportSucceeded 出力処理に成功していれば true を指定します。
//-----------------------------------------------------------------------------
static void SetEnvVarForPrePostExportScript(
    YScene& yscene,
    const YExpOpt& yopt,
    const bool isPost,
    const bool exportSucceeded
)
{
    //-----------------------------------------------------------------------------
    // 環境変数をクリアします。
    ClearEnvVarForPrePostExportScript(yopt);

    //-----------------------------------------------------------------------------
    // マルチエクスポートプラグインで出力中かどうかです。
    SetEnvVarByMel("NINTENDO_EXPORT_IS_MULTI_EXPORT", RGetNumberString(yopt.m_IsMultiExport ? 1 : 0), false);

    //-----------------------------------------------------------------------------
    // ファイルパス関連、アニメーション関連です。
    SetFilePathEnvVarForPrePostExportScript(yopt);
    SetAnimEnvVarForPrePostExportScript(yopt);

    //-----------------------------------------------------------------------------
    // ftx ファイルの出力指定が ON かどうかです。
    SetEnvVarByMel("NINTENDO_EXPORT_FTX_ENABLED", RGetNumberString(yopt.ExportsTexture() ? 1 : 0), true);

    //-----------------------------------------------------------------------------
    // G3D のコマンドラインツールフォルダです。
    SetEnvVarByMel("NINTENDO_EXPORT_G3D_TOOL_FOLDER", yopt.m_G3dToolPath, true);

    //-----------------------------------------------------------------------------
    // 出力に成功したかどうかです（出力後のスクリプト用）。
    const std::string succeededStr = (isPost) ? RGetNumberString(exportSucceeded ? 1 : 0) : "";
    SetEnvVarByMel("NINTENDO_EXPORT_SUCCEEDED", succeededStr, true);

    //-----------------------------------------------------------------------------
    // 出力した ftx ファイル名をスペースで区切って並べた文字列です。
    // （出力後のスクリプト用）（出力しない場合は空文字）
    std::string ftxNamesStr;
    if (isPost)
    {
        const std::string ext = yopt.GetExtension(RExpOpt::FTX);
        const int imgCount = static_cast<int>(yscene.m_pOutAllTexImgs.size());
        for (int iTexImg = 0; iTexImg < imgCount; ++iTexImg)
        {
            if (iTexImg > 0)
            {
                ftxNamesStr += " ";
            }
            const std::string ftxName =
                yscene.m_pOutAllTexImgs[iTexImg]->GetName() + "." + ext;
            ftxNamesStr += ftxName;
        }
    }
    SetEnvVarByMel("NINTENDO_EXPORT_FTX_FILE_NAMES", (yopt.ExportsTexture()) ? ftxNamesStr : "", true);
    SetEnvVarByMel("NINTENDO_EXPORT_USED_FTX_FILE_NAMES", ftxNamesStr, true); // fmd ファイルを出力していれば取得可能

    //-----------------------------------------------------------------------------
    // エクスポートオプションの値です。
    SetOptionEnvVarForPrePostExportScript(yscene, yopt);

    //-----------------------------------------------------------------------------
    // 環境変数展開前のエクスポートオプションの値です。
    SetEnvVarByMel("NINTENDO_EXPORT_ORG_OUTPUT_FILE_NAME"        , yopt.m_OrgOutFileName, false);
    SetEnvVarByMel("NINTENDO_EXPORT_ORG_OUTPUT_FOLDER"           , yopt.m_OrgOutFolderPath, false);
    SetEnvVarByMel("NINTENDO_EXPORT_ORG_OPTION_merge_fmd_path"   , yopt.m_OrgMergeFmdPath, false);
    SetEnvVarByMel("NINTENDO_EXPORT_ORG_OPTION_merge_anim_folder", yopt.m_OrgMergeAnimFolder, false);
    SetEnvVarByMel("NINTENDO_EXPORT_ORG_OPTION_merge_anim_name"  , yopt.m_OrgMergeAnimName, false);
}

//-----------------------------------------------------------------------------
//! @brief 出力前後のスクリプトに情報を伝達するための環境変数を更新します。
//!        環境変数やマクロ文字列が影響するオプションに関する環境変数のみ更新します。
//!
//! @param[in,out] yscene シーンです。
//! @param[in,out] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus UpdateEnvVarForPrePostExportScript(YScene& yscene, YExpOpt& yopt)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 環境変数やマクロ文字展開前のエクスポートオプションの値を取得します。
    yopt.m_OrgOutFileName     = GetEnvVarByMel("NINTENDO_EXPORT_ORG_OUTPUT_FILE_NAME");
    yopt.m_OrgOutFolderPath   = GetEnvVarByMel("NINTENDO_EXPORT_ORG_OUTPUT_FOLDER");
    yopt.m_OrgMergeFmdPath    = GetEnvVarByMel("NINTENDO_EXPORT_ORG_OPTION_merge_fmd_path");
    yopt.m_OrgMergeAnimFolder = GetEnvVarByMel("NINTENDO_EXPORT_ORG_OPTION_merge_anim_folder");
    yopt.m_OrgMergeAnimName   = GetEnvVarByMel("NINTENDO_EXPORT_ORG_OPTION_merge_anim_name");

    yopt.m_MergeFmdFlag       = (GetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_fmd" ) == "true");
    yopt.m_MergeAnimFlag      = (GetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_anim") == "true");

    //-----------------------------------------------------------------------------
    // 中間ファイルタイプごとの出力フラグを取得します。
    for (int fileTypeIdx = RExpOpt::FMD; fileTypeIdx < RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const RExpOpt::FileType fileType = static_cast<RExpOpt::FileType>(fileTypeIdx);
        const std::string ext = yopt.GetNoFmaExtension(fileType);
        const std::string name = "NINTENDO_EXPORT_" + RGetUpperCaseString(ext.substr(0, 3)) + "_FILE_PATH";
        yopt.m_OutFileFlag[fileType] = !GetEnvVarByMel(name.c_str()).empty();
        //cerr << "get out file flag: " << ext << ": " << yopt.m_OutFileFlag[fileType] << endl;
    }

    //-----------------------------------------------------------------------------
    // 環境変数やマクロ文字を展開します。
    status = yopt.ExpandOutputAndMergePath(&yscene, false);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 環境変数やマクロ文字を展開したエクスポートオプションの値を設定します。
    SetFilePathEnvVarForPrePostExportScript(yopt);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_fmd_path"   , yopt.m_MergeFmdPath, true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_anim_folder", yopt.m_MergeAnimFolder, true);
    SetEnvVarByMel("NINTENDO_EXPORT_OPTION_merge_anim_name"  , yopt.m_MergeAnimName, true);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief Export 情報メッセージを MGlobal::displayInfo で表示します。
//!
//! @param[in] msg メッセージです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void DisplayExportInfoSub(const std::string& msg, const YExpOpt& yopt)
{
    MGlobal::displayInfo(msg.c_str());
    R_UNUSED_VARIABLE(yopt);
}

//-----------------------------------------------------------------------------
//! @brief アニメーション中間ファイルのマージについての Export 情報を表示します。
//!
//! @param[in] yopt エクスポートオプションです。
//! @param[in] fileType ファイルタイプです。
//-----------------------------------------------------------------------------
static void DisplayExportInfoMergeAnim(
    const YExpOpt& yopt,
    const YExpOpt::FileType fileType
)
{
    bool outputs = yopt.m_OutFileFlag[fileType];
    if (yopt.UsesSingleFma() && yopt.IsFclFtsFtp(fileType))
    {
        // 全種類のマテリアルアニメーションを 1 つの fma ファイルに格納する場合、
        // fileType が FTS / FTP なら情報を表示しません。
        outputs = (fileType == YExpOpt::FCL) ? yopt.ExportsMaterialAnim() : false;
    }

    if (outputs && yopt.m_MergeAnimFlag)
    {
        const std::string mergeAnimPath = yopt.GetMergeAnimFilePath(fileType);
        if (RFileExists(mergeAnimPath))
        {
            const std::string ext = yopt.GetExtension(fileType);
            DisplayExportInfoSub("Merge " + ext.substr(0, 3) + " File  : " + mergeAnimPath, yopt);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief マージについての Export 情報を表示します。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void DisplayExportInfoMerge(const YExpOpt& yopt)
{
    if (yopt.MergesFmd())
    {
        DisplayExportInfoSub("Merge fmd File  : " + yopt.m_MergeFmdPath, yopt);
    }

    if (yopt.MergesFtx())
    {
        DisplayExportInfoSub("Merge ftx Folder: " + yopt.m_MergeFtxFolderPath, yopt);
    }

    DisplayExportInfoMergeAnim(yopt, YExpOpt::FSK);
    DisplayExportInfoMergeAnim(yopt, YExpOpt::FVB);
    DisplayExportInfoMergeAnim(yopt, YExpOpt::FCL);
    DisplayExportInfoMergeAnim(yopt, YExpOpt::FTS);
    DisplayExportInfoMergeAnim(yopt, YExpOpt::FTP);
    DisplayExportInfoMergeAnim(yopt, YExpOpt::FSH);
    DisplayExportInfoMergeAnim(yopt, YExpOpt::FSN);
}

//-----------------------------------------------------------------------------
//! @brief モデルの Export 情報のヘッダ部分を表示します。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void DisplayExportInfoModelHeader(const YExpOpt& yopt)
{
    const std::string remarkStr = "==========";
    std::string msg = remarkStr + " Nintendo Export " + remarkStr;
    DisplayExportInfoSub(msg, yopt);

    //-----------------------------------------------------------------------------
    // preset
    if (!yopt.m_PresetFilePath.empty())
    {
        DisplayExportInfoSub("Preset          : " + yopt.m_PresetFilePath, yopt);
    }

    //-----------------------------------------------------------------------------
    // file name & extensions
    msg = "File Name       : " + yopt.m_OutFileName + " (";
    bool isSingleFmaAdded = false;
    for (int fileTypeIdx = 0; fileTypeIdx < RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const RExpOpt::FileType fileType = static_cast<RExpOpt::FileType>(fileTypeIdx);
        if (yopt.m_OutFileFlag[fileType])
        {
            if (yopt.UsesSingleFma() && yopt.IsFclFtsFtp(fileType))
            {
                if (isSingleFmaAdded)
                {
                    continue;
                }
                isSingleFmaAdded = true;
            }
            const std::string ext = yopt.GetExtension(fileType);
            const std::string suffix = yopt.GetSuffix(fileType);
            const std::string extMsg = (!suffix.empty()) ? suffix + "." + ext : ext;
            msg += " " + extMsg;
        }
        if (fileType == RExpOpt::FMD && yopt.ExportsTexture())
        {
            msg += " " + yopt.GetExtension(RExpOpt::FTX);
        }
    }
    msg += " )";
    DisplayExportInfoSub(msg, yopt);

    //-----------------------------------------------------------------------------
    // folder
    DisplayExportInfoSub("Output Folder   : " + yopt.m_OutFolderPath, yopt);

    //-----------------------------------------------------------------------------
    // merge
    DisplayExportInfoMerge(yopt);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャの Export 情報を表示します。
//!
//! @param[in] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void DisplayExportInfoTexture(const YScene& yscene, const YExpOpt& yopt)
{
    if (yopt.ExportsTexture())
    {
        //-----------------------------------------------------------------------------
        // texture
        const int imgCount = static_cast<int>(yscene.m_pOutAllTexImgs.size());
        std::string msg = "Texture Count   : " + RGetNumberString(imgCount, "%3d");
        if (imgCount > 0)
        {
            msg += " ( ";
            for (int iTexImg = 0; iTexImg < imgCount; ++iTexImg)
            {
                std::string name = yscene.m_pOutAllTexImgs[iTexImg]->GetName();
                msg += name;
                if (iTexImg < imgCount - 1)
                {
                    msg += ", ";
                }
            }
            msg += " )";
        }
        DisplayExportInfoSub(msg, yopt);
    }
}

//-----------------------------------------------------------------------------
//! @brief マージした ftx ファイルの Export 情報を表示します。
//!
//! @param[in] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void DisplayExportInfoMergedFtx(const YScene& yscene, const YExpOpt& yopt)
{
    if (yopt.MergesFtx())
    {
        RStringArray mergedNames = yscene.m_MergedTexImgNames;
        std::sort(mergedNames.begin(), mergedNames.end()); // アルファベット順にソートします。
        const int imgCount = static_cast<int>(mergedNames.size());
        std::string msg = "Merge ftx Count : " + RGetNumberString(imgCount, "%3d");
        if (imgCount > 0)
        {
            msg += " ( ";
            for (int iTexImg = 0; iTexImg < imgCount; ++iTexImg)
            {
                msg += mergedNames[iTexImg];
                if (iTexImg < imgCount - 1)
                {
                    msg += ", ";
                }
            }
            msg += " )";
        }
        DisplayExportInfoSub(msg, yopt);
    }
}

//-----------------------------------------------------------------------------
//! @brief モデルの出力情報の結果部分を表示します。
//!
//! @param[in] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void DisplayExportInfoModelResult(const YScene& yscene, const YExpOpt& yopt)
{
    //std::string msg;

    //-----------------------------------------------------------------------------
    // node
    // 最適化後のサイズを取得できるまで保留
//  if (yopt.ExportsModelFile() || yopt.ExportsModelAnim())
//  {
//      msg = "Node Count     : " + RGetNumberString(ymodel.m_pOutYNodes.size());
//      DisplayExportInfoSub(msg, yopt);
//  }

    //-----------------------------------------------------------------------------
    // material
    // 最適化後のサイズを取得できるまで保留
//  if (yopt.ExportsModelFile() || yopt.ExportsModelAnim())
//  {
//      msg = "Material Count : " + RGetNumberString(ymodel.m_pOutYMaterials.size());
//      DisplayExportInfoSub(msg, yopt);
//  }

    //-----------------------------------------------------------------------------
    // texture
    DisplayExportInfoTexture(yscene, yopt);
    DisplayExportInfoMergedFtx(yscene, yopt);
}

//-----------------------------------------------------------------------------
//! @brief プロファイル情報を MGlobal::displayInfo で表示します。
//!
//! @param[in] name プロファイル名です。
//! @param[in] clocks 処理に掛かったクロック数です。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void DisplayProfileInfo(
    const std::string& name,
    const clock_t clocks,
    const YExpOpt& yopt
)
{
    if (yopt.m_DisplaysProfile)
    {
        const float sec = static_cast<float>(clocks) / CLOCKS_PER_SEC;
        const std::string msg = "PROFILE: " + name + ": " +
            RGetNumberString(RRound(sec * 1000.0f), "%6d") + "ms";
        DisplayExportInfoSub(msg, yopt);
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャイメージをテクスチャ名のアルファベット順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool TexImgPtrLess(RImage*& r1, RImage*& r2)
{
    return (strcmp(r1->GetName().c_str(), r2->GetName().c_str()) < 0);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャイメージの出力用ポインタ配列を設定し、
//!        テクスチャ名のアルファベット順にソートします。
//!
//! @param[out] pDstTexImgs テクスチャイメージの出力用ポインタ配列です。
//! @param[in] srcTexImgs テクスチャイメージ配列です。
//-----------------------------------------------------------------------------
static void SetOutTexImgPtrs(RImagePtrArray& pDstTexImgs, RImageArray& srcTexImgs)
{
    pDstTexImgs.clear();

    const int imgCount = static_cast<int>(srcTexImgs.size());
    if (imgCount > 0)
    {
        // 出力用ポインタ配列を設定します。
        for (int iTexImg = 0; iTexImg < imgCount; ++iTexImg)
        {
            pDstTexImgs.push_back(&srcTexImgs[iTexImg]);
        }

        // 出力用ポインタ配列をテクスチャ名のアルファベット順にソートします。
        std::sort(pDstTexImgs.begin(), pDstTexImgs.end(), TexImgPtrLess);
    }
}

//-----------------------------------------------------------------------------
//! @brief 全モデルのテクスチャイメージの出力用ポインタ配列を設定し、
//!        テクスチャ名のアルファベット順にソートします。
//!
//! @param[in] yscene シーンです。
//-----------------------------------------------------------------------------
static void SetAndSortOutAllTexImgs(YScene& yscene)
{
    SetOutTexImgPtrs(yscene.m_pOutAllTexImgs, yscene.m_AllTexImgs);
}

//-----------------------------------------------------------------------------
//! @brief 指定した DAG パスの親ノードが出力しない DAG パス配列に含まれていれば true を返します。
//!
//! @param[in] array 出力しない DAG パス配列です。
//! @param[in] dagPath 判定する DAG パスです。
//!
//! @return 親ノードが出力しない DAG パス配列に含まれていれば true を返します。
//-----------------------------------------------------------------------------
static bool HasNoOutputParent(const CDagPathArray& array, MDagPath dagPath)
{
    while (dagPath.pop())
    {
        if (dagPath.fullPathName() == "")
        {
            break;
        }
        if (std::find(array.begin(), array.end(), dagPath) != array.end())
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief DAG パス から YNode を検索します。
//!
//! @param[in] ymodel モデルです。
//! @param[in] path 検索対象の DAG パスです。
//!
//! @return m_pYNodes 内に引数 path で指定されたノードがあればをポインタを返します。
//!         なければ NULL を返します。
//-----------------------------------------------------------------------------
static YNode* FindYNodeByDagPath(const YModel& ymodel, const MDagPath& path)
{
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        const YNode& ynode = *ymodel.m_pYNodes[iNode];
        if (ynode.m_AddedFlag)
        {
            continue;
        }
        if (ynode.m_XformPath == path)
        {
            return ymodel.m_pYNodes[iNode];
        }
    }
    return NULL;
}

//-----------------------------------------------------------------------------
//! @brief find ynode by obj (not used)
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
//static YNode* FindYNodeByObj(const YModel& ymodel, const MObject& obj)
//{
//  const int nodeCount = ymodel.m_pYNodes.size();
//  for (int iNode = 0; iNode < nodeCount; ++iNode)
//  {
//      const YNode& ynode = *ymodel.m_pYNodes[iNode];
//      if (ynode.m_AddedFlag)
//      {
//          continue;
//      }
//      if (ynode.m_XformPath.node() == obj)
//      {
//          return ymodel.m_pYNodes[iNode];
//      }
//  }
//  return NULL;
//}

//-----------------------------------------------------------------------------
//! @brief print hierarchy (for debug)
//-----------------------------------------------------------------------------
static void PrintHierarcy(YNode* cur)
{
    for (int id = 0; id < cur->m_Depth; ++id)
    {
        cerr << "    ";
    }
    cerr << cur->m_OrgName << R_ENDL;
    if (cur->m_pChild != NULL)
    {
        PrintHierarcy(cur->m_pChild);
    }
    if (cur->m_pNext != NULL)
    {
        PrintHierarcy(cur->m_pNext);
    }
}

//-----------------------------------------------------------------------------
//! @brief m_pYNodes 内で全てのノードがユニークな名前を持つように名前を変更します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetUniqueYNodeNames(YModel& ymodel)
{
    const std::string bar = "_";

    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int iNode = 1; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];

        // YNodeの名前を取得します。
        const std::string baseName = ynode.m_Name;
        std::string name = baseName;
        int id = 1; // ユニークな名前を作成するための番号の初期値
        for (;;)
        {
            bool found = false;
            // m_pYNodes 配列の中から同じ名前を持つ YNode を数える
            for (int iOther = 0; iOther < iNode; ++iOther)
            {
                const YNode& other = *ymodel.m_pYNodes[iOther];
                // ここでは YNode が同一かを気にせずに名前が同じかどうかのみで判断します。
                if (other.m_Name == name)
                {
                    found = true;
                    break;
                }
            }
            // 見つかった場合は名前を "元の名前_番号" という形で変更し、
            // もう一度 m_pYNodes の中に同じ名前がないか確認します。
            if (found)
            {
                name = baseName + bar + RGetNumberString(id);
                ++id;
            }
            else
            {
                break;
            }
        }

        // id が 2 以上であった場合は重複する名前おを持つ YNode が存在していた
        // ということなので、YNode の名前を変更します。
        if (id > 1)
        {
            std::string orgPath = (!ynode.m_AddedFlag) ?
                ynode.m_XformPath.partialPathName().asChar() : ynode.m_OrgName;
            // 名前が変更された事について警告を表示します。
            if (yopt.m_WarnsNodeNameChanged && !yopt.m_CheckElementFlag)
            {
                YShowWarning(&yscene, // Node name is changed: %s -> %s
                    "同じ名前のノードが複数存在するため、中間ファイルに出力されるノード名が変更されました: {0} -> {1} \n"
                    "（この警告はコンフィグファイルで抑制できます）",
                    "Multiple nodes with the same name exist, so the names of the nodes exported to the intermediate file were changed: {0} -> {1} \n"
                    "(This warning can be suppressed using the config file.)",
                    orgPath, name);
            }
            ynode.m_Name = name;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief m_YMaterials 内で全てのマテリアルがユニークな名前を持つように名前を変更します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetUniqueMaterialNames(YModel& ymodel)
{
    //-----------------------------------------------------------------------------
    // set pointer array
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    const int matCount = static_cast<int>(ymodel.m_YMaterials.size());
    if (matCount == 0)
    {
        return;
    }
    // マテリアルのポインタを列挙します。
    YMaterialPtrArray matPtrs;
    matPtrs.resize(matCount);
    int iMat;
    for (iMat = 0; iMat < matCount; ++iMat)
    {
        matPtrs[iMat] = &ymodel.m_YMaterials[iMat];
    }
    // 頂点カラーなし、頂点カラー１つ、頂点カラー２つの順にソートするならここで
    // 現状は後から出力されるノードで使用されているマテリアルに
    // "_1", "_2" が付加されるというルールなのでソートしない

    //-----------------------------------------------------------------------------
    // set unique name
    const std::string bar = "_";
    for (iMat = 1; iMat < matCount; ++iMat)
    {
        YMaterial& mat = *matPtrs[iMat];
        const std::string baseName = mat.m_Name;
        std::string name = baseName;
        // マテリアル配列の中から変数 mat と同一の名前を持つマテリアルがあるか調べる
        int id = 1;
        for (;;)
        {
            bool found = false;
            for (int iOther = 0; iOther < iMat; ++iOther)
            {
                const YMaterial& other = *matPtrs[iOther];
                if (other.m_Name == name)
                {
                    found = true;
                    break;
                }
            }
            if (found)
            {
                // 見つかったら新しい名前の作成とidのインクリメントを行います。
                name = baseName + bar + RGetNumberString(id);
                ++id;
            }
            else
            {
                break;
            }
        }
        // idが 2 以上(重複した名前がある)の場合、上で求めた新しいマテリアル名を設定します。
        if (id > 1)
        {
            std::string sgName = "no shading group";
            if (!mat.m_SGObj.isNull())
            {
                sgName = MFnDependencyNode(mat.m_SGObj).name().asChar();
            }
            // マテリアル名が変更される場合は警告を表示します。
            if (!yopt.m_CheckElementFlag)
            {
                YShowWarning(&yscene, // Material name is changed: %s (%s) -> %s
                    "マテリアルの名前が重複しているため、中間ファイルに出力されるマテリアル名が変更されました: {0} ({1}) -> {2} \n"
                    "この警告は 1 つのマテリアルが複数のシェーディンググループに接続されている場合などに発生します。",
                    "A material name exported to the intermediate file was changed because two or more materials had the same name: {0} ({1}) -> {2} \n"
                    "This warning occurs when a single material has been connected to more than one shading group.",
                    mat.m_OrgName, sgName, name);
            }
            mat.m_Name = name;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief YNode ポインタの比較関数です。
//-----------------------------------------------------------------------------
static bool YNodePtrLess(YNode*& r1, YNode*& r2)
{
    return (*r1 < *r2);
}

//-----------------------------------------------------------------------------
//! @brief YNode を階層構造の深さ優先探索の順に並び替えます。
//!        再帰的に呼ばれます。
//!
//! @param[in,out] table YNode のポインタ配列です。
//! @param[in,out] cur 処理するノードです。
//! @param[in,out] index 配列中のインデックスです。
//-----------------------------------------------------------------------------
static void SortYNodeHierarchyVertically(
    YNodePtrArray& table,
    YNode* cur,
    int& index
)
{
    if (index >= static_cast<int>(table.size()))
    {
        cerr << "Failed to sort bones: " << index << " / " << table.size() << endl; // 通常は失敗しません。
        return;
    }

    // m_OrgIndex はテーブル内のインデックスを設定します。
    cur->m_OrgIndex = index;
    table[index] = cur;
    ++index;
    YNode* child = cur->m_pChild;
    if (child != NULL)
    {
        SortYNodeHierarchyVertically(table, child, index);
    }
    YNode* next = cur->m_pNext;
    if (next != NULL)
    {
        SortYNodeHierarchyVertically(table, next, index);
    }
}

//-----------------------------------------------------------------------------
//! @brief YNode のリンク情報を設定します。
//! m_pYNodes 内の YNode に対して親と兄弟ノードへのリンクを設定します。
//! そして、m_pYNodes 内のノードの並びを深さ優先探索順にソートします。
//! ソートが完了した後に m_pYNodes の内容を m_pOutYNodes にコピーします。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus SetYNodeConnection(YModel& ymodel)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // YNode配列に要素がなければ何もせずに終了します。
    int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    if (nodeCount == 0)
    {
        return MS::kSuccess;
    }

    // YNodeの親のポインタ(m_pParent)を設定します。
    int iNode;
    for (iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];
        ynode.m_pParent = NULL;
        // YNodeに親があるか調べる
        MFnDagNode dagFn(ynode.m_XformPath);
        int parentCount = dagFn.parentCount();
        if (parentCount > 0)
        {
            // YNodeに設定した transform ノードの DAG パスから親のパスを取得し、
            // そのパスに該当する YNode のポインタを取得します。。
            MDagPath parentPath = ynode.m_XformPath;
            parentPath.pop();
            ynode.m_pParent = FindYNodeByDagPath(ymodel, parentPath);
        }
    }

    //-----------------------------------------------------------------------------
    // 個々の YNode の階層構造上の深度を YNode::m_Depth に設定します。
    int rootCount = 0;
    for (iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];
        int depth = 0;
        // ルートノードまで親をたどっていき、階層構造上の深度を求めます。
        YNode* parent = ynode.m_pParent;
        while (parent != NULL)
        {
            ++depth;
            parent = parent->m_pParent;
        }
        ynode.m_Depth = depth;

        // 後で nw4f root を追加する必要があるかどうかを判断するために
        // ルートノードの数をカウントします。
        if (depth == 0)
        {
            ++rootCount;
        }
    }

    //-----------------------------------------------------------------------------
    // Maya 上でルートボーンが 2 つ以上存在する場合は
    // それらをグループ化するボーン（モデルルート）を追加します。
    #ifdef ADD_MODEL_ROOT_SW
    if (rootCount >= 2)
    {
        // モデルルートを作成します。
        MDagPath dummyPath;
        YNode* proot = new YNode(yscene, dummyPath, YNode::KindModelRoot, true, false, false, yopt);
        ymodel.m_pYNodes.push_back(proot);
        // 階層構造でルートとなっているノードをモデルルートの子供に設定します。
        nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
        for (iNode = 0; iNode < nodeCount; ++iNode)
        {
            YNode& ynode = *ymodel.m_pYNodes[iNode];
            if (ynode.m_OrgKind == YNode::KindModelRoot)
            {
                ynode.m_Depth = 0;
            }
            else
            {
                // 階層構造のトップにノードが追加された分、モデルルート以外の
                // すべてのノードの深度をインクリメントします。
                ++ynode.m_Depth;
                // 深度が 1 のノードはルートノードだったので、モデルルートを
                // 親として設定します。
                if (ynode.m_Depth == 1)
                {
                    ynode.m_pParent = proot;
                }
            }
        }
        // モデルルートが作成されたのでルートノード数を 1 に変更します。
        rootCount = 1;
    }
    #endif

    if (rootCount > 1)
    {
        // モデルルートを使うならエラーは起こりません。
        YShowError(&yscene, // There is more than one root node // 現在は発生しない
            "ルートノードが複数存在します。",
            "There is more than one root node.");
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // m_pYNodes 内の YNode にユニークな名前を設定します。
    SetUniqueYNodeNames(ymodel);

    //-----------------------------------------------------------------------------
    // 階層構造の深度と名前で m_pYNodes 内の YNode をソートします。 (hierarchy horizontally)
    std::sort(ymodel.m_pYNodes.begin(), ymodel.m_pYNodes.end(), YNodePtrLess);

    //-----------------------------------------------------------------------------
    // 兄弟のリンクを設定します。
    YNode* prev = NULL;
    // ルートノードに対して兄弟のリンクを設定します。
    for (iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode* cur = ymodel.m_pYNodes[iNode];
        if (cur->m_Depth == 0)
        {
            cur->m_pPrev = prev;
            cur->m_pNext = NULL;
            if (prev != NULL)
            {
                prev->m_pNext = cur;
            }
            prev = cur;
        }
    }

    // ノードの子供と兄弟のリンクを設定します。
    for (iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode* parent = ymodel.m_pYNodes[iNode];
        prev = NULL; // 前の兄弟ノード
        int childCount = 0; // 子供の数
        // parent ノードを親に持つノードに対して兄弟のリンクを設定します。
        for (int icur = 0; icur < nodeCount; ++icur)
        {
            YNode* cur = ymodel.m_pYNodes[icur];
            if (cur->m_pParent == parent)
            {
                // 兄弟のリンク情報を設定します。
                cur->m_pPrev = prev;
                cur->m_pNext = NULL;

                if (prev == NULL)
                {   // 最初の子ノードであれば親ノードの m_pChild にポインタを登録します。
                    parent->m_pChild = cur;
                }
                else
                {   // 前の兄弟ノードの m_pNext にノードのポインタを設定します。
                    prev->m_pNext = cur;
                }
                prev = cur;
                ++childCount;
            }
        }
        // 子ノードの数を設定します。
        parent->m_ChildCount = childCount;
    }

    //-----------------------------------------------------------------------------
    // YNode を階層構造の深さ優先探索順に並び替える
    if (nodeCount > 0)
    {
        // m_pYNodes は深度順に並び替えられているので、既に先頭のノードがルートノード
        // になっているはず。
        YNode* cur = ymodel.m_pYNodes[0];
        int index = 0;
        SortYNodeHierarchyVertically(ymodel.m_pYNodes, cur, index);
    }

    //-----------------------------------------------------------------------------
    // YNode のインデックスと階層構造のリンク情報などを出力用変数にコピーします。
    for (iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];
        ynode.m_Index = ynode.m_OrgIndex;
        ynode.m_pOutParent = ynode.m_pParent;
    }

    // ルートノードを m_pRootYNode に設定します。
    if (nodeCount > 0)
    {
        ymodel.m_pRootYNode = ymodel.m_pYNodes[0];
        // 出力時のノード圧縮処理上、ルートノードは圧縮不可に設定します。
        // 中間ファイルには m_CompressEnable のほうを出力します。
        ymodel.m_pRootYNode->m_InternalCompressEnable = false;
    }
    else
    {
        ymodel.m_pRootYNode = NULL;
    }

    // 出力用の YNode テーブルにコピーします。
    ymodel.m_pOutYNodes = ymodel.m_pYNodes;

    //-----------------------------------------------------------------------------
    // debug
    //PrintHierarcy(ymodel.m_pYNodes[0]);

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 不要なクラスターハンドルを削除します。
//!        クラスターハンドルのターゲットが出力されないノードなら
//!        クラスターハンドルも出力しません。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] noOutputPaths 出力されないノードの DAG パス配列です。
//-----------------------------------------------------------------------------
static void CutNeedlessClusterHandle(YModel& ymodel, const CDagPathArray& noOutputPaths)
{
    MStatus status;

    if (ymodel.m_pYNodes.size() == 0)
    {
        return;
    }

    for (;;)
    {
        ItYNodePtr pynodeIter;
        for (pynodeIter = ymodel.m_pYNodes.begin(); pynodeIter != ymodel.m_pYNodes.end(); ++pynodeIter)
        {
            YNode& ynode = **pynodeIter;
            if (!ynode.m_ClusterHandleFlag || !ynode.m_InternalCompressEnable)
            {
                continue;
            }
            //cerr << "cnch: " << ynode.getName() << R_ENDL;
            MDagPath dagPath = ynode.m_XformPath;
            MFnDagNode dagFn(dagPath);
            if (dagFn.childCount() != 1)
            {
                continue;
            }
            MObject shapeObj = dagFn.child(0);
            MFnDependencyNode shapeDepFn(shapeObj);
            MPlug xformPlug = FindPlugQuiet(shapeDepFn, "clusterTransforms", &status);
            if (!status)
            {
                continue;
            }
            MPlugArray plugArray;
            xformPlug[0].connectedTo(plugArray, false, true);
            if (plugArray.length() != 1)
            {
                continue;
            }
            MFnWeightGeometryFilter clusterFn(plugArray[0].node(), &status);
            if (!status)
            {
                continue;
            }
            MObjectArray targets;
            clusterFn.getOutputGeometry(targets);
            bool effectiveFlag = false;
            for (int iTarget = 0; iTarget < static_cast<int>(targets.length()); ++iTarget)
            {
                // changed (2006/02/09)
                MDagPathArray allPaths;
                MFnDagNode(targets[iTarget]).getAllPaths(allPaths);
                int pathCount = allPaths.length();
                for (int ipath = 0; ipath < pathCount; ++ipath)
                {
                    MDagPath targetPath = allPaths[ipath];
                    targetPath.pop(); // mesh -> transform
                    //cerr << targetPath.fullPathName() << R_ENDL;
                    if (std::find(noOutputPaths.begin(), noOutputPaths.end(), targetPath) == noOutputPaths.end())
                    {
                        // １つでもターゲットが出力しないノードに含まれなければ有効
                        effectiveFlag = true;
                        break;
                    }
                }
                if (effectiveFlag)
                {
                    break;
                }
            }
            if (!effectiveFlag)
            {
                #ifdef DEBUG_PRINT_SW
                cerr << "cut cluster handle: " << ynode.m_OrgName << R_ENDL;
                #endif
                break;
            }
        }
        if (pynodeIter != ymodel.m_pYNodes.end())
        {
            delete *pynodeIter;
            ymodel.m_pYNodes.erase(pynodeIter);
        }
        else
        {
            break;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 子ノードを持たない transform ノード を削除します。
//!        m_pYNodes 配列に列挙された YNode のうち、全ての子供が m_pYNodes
//!        に登録されていない YNode を m_pYNodes 配列から削除します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void CutNeedlessTransformNode(YModel& ymodel)
{
    MStatus status;

    // @@ ここの処理はかなり助長です。
    //    今後ノードが増えると処理負荷が指数関数的に高くなります。
    while (ymodel.m_pYNodes.size() > 0)
    {
        ItYNodePtr pynodeIter;
        // ymodel.m_pYNodes に列挙されたノード配列から削除可能なノードを探して削除します。
        for (pynodeIter = ymodel.m_pYNodes.begin(); pynodeIter != ymodel.m_pYNodes.end(); ++pynodeIter)
        {
            const YNode& ynode = **pynodeIter;
            // 以下の条件に当てはまる場合、トランスフォームを削除しません。
            //   - タイプがトランスフォーム以外である
            //   - プラグインにより削除されない設定がされている
            //   - スキンのインフルエンスオブジェクトである
            //   - クラスタハンドルである
            if (ynode.m_OrgKind != YNode::KindTransform ||
                !ynode.m_InternalCompressEnable ||
                ynode.m_SkinInfluenceFlag ||
                ynode.m_ClusterHandleFlag)
            {
                continue;
            }
            // DAGノードを作成、ノードの子供を走査します。
            MFnDagNode dagFn(ynode.m_XformPath);
            bool childExist = false;
            int childCount = dagFn.childCount();
            for (int ichild = 0; ichild < childCount; ++ichild)
            {
                MObject childObj = dagFn.child(ichild);
                // インスタンス化されている可能性を考慮して、
                // 直接の子供トなるノードの DAG パスを取得します。。
                MDagPath childDagPath = GetDirectChildDagPath(
                    ynode.m_XformPath, childObj); // changed (2006/02/09)
                // 出力される YNode 配列に子供が含まれていたら
                // childExist フラグを立てて子供の検索を中止します。
                if (FindYNodeByDagPath(ymodel, childDagPath) != NULL)
                {
                    childExist = true;
                    break;
                }
            }
            // 出力する子供が存在していなければノードの検索を中断します。
            if (!childExist)
            {
                #ifdef DEBUG_PRINT_SW
                cerr << "cut needless transform: " << ynode.m_OrgName << R_ENDL;
                #endif
                break;
            }
        }
        // 削除するノードがあれば検索が途中で打ち切られる
        // その場合はそのノードを削除してからもう一度最初から検索を行います。
        if (pynodeIter != ymodel.m_pYNodes.end())
        {
            delete *pynodeIter;
            ymodel.m_pYNodes.erase(pynodeIter);
        }
        else
        {
            break;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 階層構造を取得します。
//!        階層構造から出力するべきカメラ、ライト、メッシュ、ノードを列挙します。
//!        この関数内でエクスポートオプションで指定されたオブジェクトの出力対象
//!        の設定に従って出力するノードの選択を行います。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] getRootFlag アニメーションレンジ出力（現在未対応）のために、
//!                        Maya 上のルートノード（カメラ、ライト以外）を取得するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetHierarchy(YModel& ymodel, const bool getRootFlag)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    YEnvObjs& yenvObjs = ymodel.GetEnvObjs();

    #ifdef OUTPUT_ERROR_LOG
    MGlobal::doErrorLogEntry("// get hierarchy");
    #endif

    //-----------------------------------------------------------------------------
    // アニメーションレンジ出力のルートノード取得中でなければ、
    // 環境オブジェクトを取得します。
    const YExpOpt& yopt = yscene.GetOpt();
    const bool getEnvObjFlag = (!getRootFlag);

    //-----------------------------------------------------------------------------
    // 選択されたノードのみ出力する場合、選択された transform ノードと
    // その下の階層の transform ノードの DAG パス配列を取得します。
    RTimeMeasure tmHierarchy;
    CDagPathArray selectedXformPaths;
    if (yopt.m_Target == RExpOpt::EXPORT_TARGET_SELECTION)
    {
        MSelectionList slist;
        MGlobal::getActiveSelectionList(slist);
        GetSelectedTransformNodesBelow(&selectedXformPaths, slist);
    }

    //-----------------------------------------------------------------------------
    // IK ハンドルとそのハンドルが制御するジョイントの情報を取得します。
    GetIkHandleData(ymodel.m_IkHandles);
    // シーン中に存在している skinCluster ノードが参照しているインフルエンスオブジェクトを列挙します。
    CDagPathArray skinInfPaths;
    GetSkinInfluencePaths(skinInfPaths);

    //-----------------------------------------------------------------------------
    // シーン中の全 transform ノードについてループします。
    bool meshExists = false;
    CDagPathArray noOutputPaths; // 出力しない transform ノードの DAG パス配列です。
    MItDag dagIter(MItDag::kBreadthFirst, MFn::kTransform); // 幅優先探索
    for ( ; !dagIter.isDone(); dagIter.next())
    {
        MDagPath dagPath;
        dagIter.getPath(dagPath);

        //-----------------------------------------------------------------------------
        // 選択されたノードのみ出力する場合、選択されていなければスキップします。
        // selectedXformPaths についてループしていないのは、
        // 同名ノードの命名規則のために MItDag の幅優先探索の順番で取得するためです。
        if (yopt.m_Target == RExpOpt::EXPORT_TARGET_SELECTION &&
            FindDagPathInArray(selectedXformPaths, dagPath) == -1)
        {
            continue;
        }

        //-----------------------------------------------------------------------------
        // get function node
        MFnDagNode dagFn(dagPath, &status);
        if (!status)
        {
            YShowError(&yscene, "", "Attaching MFnDagNode"); // 通常は発生しない
            continue;
        }
        // パスからオブジェクトとオブジェクトの名前、タイプ名を取得します。
        const MObject curObj = dagFn.object();
        const std::string nodeName = dagPath.partialPathName().asChar();
        const std::string noNsName = GetOutElementName(dagFn.name().asChar(), true);
        const MFn::Type nodeType = curObj.apiType();

        //-----------------------------------------------------------------------------
        // グリッドデータ(ground plane)を参照している場合はスキップします。
        if (NodeHasSpecificChild(dagFn, MFn::kGroundPlane))
        {
            continue;
        }

        //-----------------------------------------------------------------------------
        // オブジェクトのフラグを取得します。

        // オブジェクトはフォグの環境設定オブジェクトかどうか
        bool envfogFlag = NodeHasSpecificChild(dagFn, MFn::kEnvFogShape);
        // オブジェクトはライトかどうか
        bool lightFlag = (!envfogFlag &&
            NodeHasSpecificFunctionChild(dagFn, MFn::kLight));

        //-----------------------------------------------------------------------------
        // 有効なノードかどうか
        // ライトであれば IsEffectiveNode() の結果に関係なく有効とします。
        bool effectiveFlag = IsEffectiveNode(curObj) || lightFlag;
        // ノードが可視かどうか
        //bool visibleFlag = IsVisibleNode(curObj);
        // dagPath が指定するオブジェクトが親階層を含めて可視かどうかを判断します。
        // hierarchyVisibleFlag  true=可視  false=不可視
        bool hierarchyVisibleFlag = IsVisibleNodeIncludeParent(dagPath);
        // テンプレートオブジェクトかどうかを調べる
        bool templateFlag = IsTemplateNode(curObj);
        // インフルエンスオブジェクト(スキンの変形に使用されるボーン)かどうかを調べる
        // skinInfPaths にはシーン中のインフルエンスオブジェクトが列挙されているので、
        // その配列の中にオブジェクトがあれば dagPath はインフルエンスオブジェクトです。
        bool skinInfFlag = (std::find(skinInfPaths.begin(), skinInfPaths.end(), dagPath) != skinInfPaths.end());
        // 最適化により削除される可能性があるオブジェクトかどうかを調べる
        // compressEnable  true=削除される可能性がある  false=削除されない
        bool compressEnable = !IsNoCuttingNode(curObj);

        // 出力フラグを設定します。
        bool outputFlag = effectiveFlag || skinInfFlag || !compressEnable;
        if (outputFlag && HasNoOutputParent(noOutputPaths, dagPath))
        {
            outputFlag = false;
        }

        //cerr << "gh: flag: " << nodeName << ", " << effectiveFlag << ", " << outputFlag << R_ENDL;
        #if 0
        // debug
        cerr << nodeName << " " << curObj.apiTypeStr() << " " << visibleFlag << R_ENDL;
        cerr << dagFn.parentCount() << " " << dagFn.childCount() << R_ENDL;
        if (dagFn.parentCount() > 0)
        {
            MObject parent = dagFn.parent(0);
            cerr << RTab() << "parent: " << MFnDagNode(parent).name() << " " << parent.apiTypeStr() << R_ENDL;
        }
        if (dagFn.childCount() > 0)
        {
            MObject child = dagFn.child(0);
            cerr << RTab() << "child: " << MFnDagNode(child).name() << " " << child.apiTypeStr() << R_ENDL;
        }
        #endif

        //-----------------------------------------------------------------------------
        // ユーザーデータ用 transform ノードを取得します。
        bool isSpecialUserDataNode = false;
        if (noNsName.find("nw4f_model_ud") == 0 && !ymodel.m_UserDataXformPath.isValid())
        {
            ymodel.m_UserDataXformPath = dagPath;
            isSpecialUserDataNode = true;
        }
        else if (noNsName.find("nw4f_fsk_ud") == 0 && !ymodel.m_FskUserDataXformPath.isValid())
        {
            ymodel.m_FskUserDataXformPath = dagPath;
            isSpecialUserDataNode = true;
        }

        //-----------------------------------------------------------------------------
        // get camera
        #ifndef OUT_CAM_LIT_AS_NULL_SW
        if (nodeType == MFn::kLookAt)   // skip camera group
        {
            continue;
        }
        #endif
        #if (MAYA_API_VERSION >= 200900)
        if (NodeHasSpecificChild(dagFn, MFn::kStereoCameraMaster)) // stereoRigCamera
        {
            noOutputPaths.push_back(dagPath); // ignore stereo camera & child
            continue;
        }
        #endif
        // ノードがカメラだった場合登録処理
        if (NodeHasSpecificFunctionChild(dagFn, MFn::kCamera))
        {
            // 立体視用のステレオカメラであった場合はカメラとして出力しません。
            if (IsStereoCameraChild(dagPath)) // for export selection
            {
                continue;
            }
            MDagPath shapePath = GetCameraShapePath(dagPath);
            MFnCamera camFn(shapePath, &status);
            if (status && getEnvObjFlag && outputFlag &&
                (hierarchyVisibleFlag || !compressEnable))
            {
                YCamera camera(yscene, dagPath, yopt, &status);
                if (status)
                {
                    yenvObjs.m_YCameras.push_back(camera);
                }
            }
            #ifndef OUT_CAM_LIT_AS_NULL_SW
            continue;
            #endif
        }

        #ifndef OUT_CAM_LIT_AS_NULL_SW
        // skip aim & up locator
        if (NodeHasSpecificParent(dagPath, MFn::kLookAt))
        {
            continue;
        }
        #endif

        //-----------------------------------------------------------------------------
        // get light
        if (lightFlag)
        {
            // エリアライト、ボリュームライトは出力しません。
            if (NodeHasSpecificChild(dagFn, MFn::kAreaLight) ||
                NodeHasSpecificChild(dagFn, MFn::kVolumeLight))
            {
                #ifdef DEBUG_PRINT_SW
                cerr << "skip: " << nodeName << R_ENDL;
                #endif
            }
            else if (getEnvObjFlag && outputFlag && !templateFlag)
            {
                YLight light(yscene, dagPath, yopt, &status);
                if (status)
                {
                    yenvObjs.m_YLights.push_back(light);
                }
            }
            #ifndef OUT_CAM_LIT_AS_NULL_SW
            continue;
            #endif
        }

        //-----------------------------------------------------------------------------
        // get environment fog
        if (envfogFlag)
        {
            if (getEnvObjFlag && outputFlag &&
                (hierarchyVisibleFlag || !compressEnable))
            {
                YFog fog(yscene, dagPath, yopt, &status);
                if (status)
                {
                    yenvObjs.m_YFogs.push_back(fog);
                }
            }
            #ifndef OUT_CAM_LIT_AS_NULL_SW
            continue;
            #endif
        }

        //-----------------------------------------------------------------------------
        // DAGノードのタイプを判別し、YNodeに種類を設定します。
        YNode::Kind kind = YNode::KindNone;
        // IKハンドルによりコントロースされているかどうか
        // true=コントロールされる  false=されない
        bool ikControlledFlag = false;
        // クラスタかどうか
        // true=クラスタである  fales=でない
        bool clusterHandleFlag = false;

        // コンストレイント系とエフェクターは中間ファイルに出力しない
        if (!outputFlag ||
            nodeType == MFn::kIkEffector ||
            nodeType == MFn::kPointConstraint ||
            nodeType == MFn::kAimConstraint ||
            nodeType == MFn::kOrientConstraint ||
            nodeType == MFn::kScaleConstraint ||
            nodeType == MFn::kPoleVectorConstraint)
        {
            // not add
        }
        else if (nodeType == MFn::kJoint)
        {
            // joint
            kind = YNode::KindJoint;
            // IKハンドルによってコントロールされるジョイントかどうかを取得します。
            if (FindIkHandleDataFromChainJoint(ymodel.m_IkHandles, dagPath) != -1)
            {
                ikControlledFlag = true;
            }
        }
        else if (NodeHasSpecificChild(dagFn, MFn::kMesh))
        {
            if (IsBlendShapeTarget(dagPath) && compressEnable)
            {
                // ブレンドシェイプのターゲットは圧縮可能なら出力しません。
            }
            else if (NodeHasEffectiveMeshChild(dagFn))
            {
                // 子ノードに有効な mesh ノードがあればメッシュとして出力します。
                kind = YNode::KindMesh;
            }
            else
            {
                // 子ノードに有効な mesh ノードがなければトランスフォームとして出力します。
                // （インフルエンスオブジェクトとして使用されている場合など）
                kind = YNode::KindTransform;
            }
        }
        else if (NodeHasSpecificChild(dagFn, MFn::kCluster))
        {
            // クラスタハンドルであればトランスフォームとして出力し、
            // クラスタハンドルフラグを ON にします。
            clusterHandleFlag = true;
            kind = YNode::KindTransform;
        }
        else if (NodeHasSpecificChild(dagFn, MFn::kNurbsCurve))
        {
            // Nurbsカーブはトランスフォームとして出力します。
            // curve (out as transform)
            //if (!yopt.m_CheckElementFlag)
            //{
            //  YShowWarning(&yscene, "", "Cannot export NURBS curve: {0}", nodeName); // 現在は発生しない
            //}
            if (!compressEnable || NodeHasEffectiveTransformChild(dagFn))
            {
                kind = YNode::KindTransform;
            }
        }
        else if (NodeHasSpecificChild(dagFn, MFn::kNurbsSurface))
        {
            // Nurbsサーフェースはトランスフォームとして出力します。
            // surface (out as transform)
            if (!yopt.m_CheckElementFlag)
            {
                YShowWarning(&yscene, // Cannot export NURBS surface: %s
                    "NURBS サーフェスは出力できません。ロケータとして出力します: {0}",
                    "NURBS surfaces cannot be exported. Export as a locator: {0}",
                    nodeName);
            }
            if (!compressEnable || NodeHasEffectiveTransformChild(dagFn))
            {
                kind = YNode::KindTransform;
            }
        }
        else if (NodeHasSpecificChild(dagFn, MFn::kSubdiv))
        {
            // サブディビジョンサーフェースはトランスフォームとして出力します。
            if (!yopt.m_CheckElementFlag)
            {
                YShowWarning(&yscene, // Cannot export Subdivision surface: %s
                    "サブディビジョンサーフェスは出力できません。ロケータとして出力します: {0}",
                    "Subdivision surfaces cannot be exported. Export as a locator: {0}",
                    nodeName);
            }
            if (!compressEnable || NodeHasEffectiveTransformChild(dagFn))
            {
                kind = YNode::KindTransform;
            }
        }
        else if (!compressEnable || skinInfFlag)
        {
            // 「Set No Compress Node」プラグインにより削除されない指定が
            // されたノードかスキンのインフルエンスオブジェクトであれば
            // トランスフォームとして出力します。
            kind = YNode::KindTransform;
        }
        else if (NodeHasSpecificChild(dagFn, MFn::kJoint)     ||
                 NodeHasSpecificChild(dagFn, MFn::kTransform) ||
                 NodeHasSpecificChild(dagFn, MFn::kLookAt)    ||
                 NodeHasSpecificChild(dagFn, MFn::kLodGroup))
        {
            // joint、transform、lookAt の子ノードを持ち、
            // その子ノードが出力されるなら、トランスフォームとして出力します。
            if (!compressEnable || NodeHasEffectiveTransformChild(dagFn))
            {
                kind = YNode::KindTransform;
            }
        }
        //cerr << "gh: kind: " << dagPath.partialPathName() << ", " << kind << R_ENDL;

        //-----------------------------------------------------------------------------
        // ノードを出力リストに登録します。
        if (kind != YNode::KindNone)
        {
            YNode* pynode = new YNode(yscene, dagPath, kind, compressEnable,
                ikControlledFlag, clusterHandleFlag, yopt, &status);
            if (!status)
            {
                //YShowError(&yscene, "", "Node is wrong: {0}", nodeName); // 現在は発生しない
                delete pynode;
                return MS::kFailure;
            }
            pynode->m_IsSpecialUserDataNode = isSpecialUserDataNode;
            ymodel.m_pYNodes.push_back(pynode);
            //cerr << "gh: append: " << nodeName << R_ENDL;
            if (kind == YNode::KindMesh)
            {
                meshExists = true;
            }
        }
        else
        {
            noOutputPaths.push_back(dagPath);
            //cerr << "gh: no output: " << nodeName << R_ENDL;
        }
    }

    //-----------------------------------------------------------------------------
    // cut needless cluster handle
    CutNeedlessClusterHandle(ymodel, noOutputPaths);

    //-----------------------------------------------------------------------------
    // 出力される子供がない YNode を m_pYNodes 配列から削除し、
    // 中間ファイルに出力されないようにします。
    CutNeedlessTransformNode(ymodel);

    //-----------------------------------------------------------------------------
    // get main camera
    MSelectionList cameraList;
    if (getEnvObjFlag && MGlobal::getSelectionListByName("persp", cameraList))
    {
        MDagPath xformPath;
        cameraList.getDagPath(0, xformPath);
        yscene.m_MainCamera = YCamera(yscene, xformPath, yopt, &status);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // モデルまたはモデルのアニメーションを出力する場合に
    // 出力されるノードがなければエラーまたは警告を表示します。
    if (yopt.ExportsModelFile() ||
        yopt.ExportsModelAnim())
    {
        if (ymodel.m_pYNodes.size() == 0)
        {
            if (getRootFlag || !yopt.m_CheckElementFlag)
            {
                YShowError(&yscene, // No effective node
                    "出力される有効なノードが 1 つもありません。",
                    "There are no active nodes to output.");
                return MS::kFailure;
            }
            else
            {
                YShowWarning(&yscene, // No effective node
                    "出力される有効なノードが 1 つもありません。",
                    "There are no active nodes to output.");
                return MS::kSuccess;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // モデルを出力する場合に出力されるポリゴンがなければエラーまたは警告を表示します。
    //if (yopt.ExportsModelFile() && !meshExists)
    //{
    //  if (!yopt.m_CheckElementFlag)
    //  {
    //      YShowError(&yscene, "", "No effective polygon"); // 現在は発生しない
    //      return MS::kFailure;
    //  }
    //  else
    //  {
    //      YShowWarning(&yscene, "", "No effective polygon"); // 現在は発生しない
    //  }
    //}

    //-----------------------------------------------------------------------------
    // set connection
    if (!getRootFlag)
    {
        // YNode の階層構造のリンクを設定し、
        // m_pYNodes 内の要素をソートします。
        status = SetYNodeConnection(ymodel);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // check scene element size
    if (!getRootFlag && yopt.ExportsSceneAnim())
    {
        CheckEnvObjCount(yenvObjs, yopt);
    }
    //cerr << "get hierarchy time: " << tmHierarchy.GetMilliSec() << endl;

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 出力用のノードの可視性を取得します。
//!        ノード自体の可視性の他に、ノードがメッシュであった場合はシェイプの可視性もチェックします。
//!        さらにノードの上位階層で不可視になっているノードがないかもチェックします。
//!
//! @param[in] ynode ノードです。
//!
//! @return 可視なら true を返します。
//-----------------------------------------------------------------------------
static bool GetOutVisibility(const YNode& ynode)
{
    //-----------------------------------------------------------------------------
    // transform ノードの可視性をチェックします。
    if (!ynode.m_CurXformVisibility || !ynode.m_LayerVisibilityFlag)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // ノードがメッシュの場合はシェイプの可視性もチェックします。
    if (ynode.m_OrgKind == YNode::KindMesh)
    {
        bool shapeVis;
        ynode.m_ShapeVisibilityPlug.getValue(shapeVis);
        if (!shapeVis)
        {
            return false;
        }
    }

    //-----------------------------------------------------------------------------
    // 上位階層で不可視に設定されているノードがないか確認します。
    const YNode* parent = ynode.m_pParent;
    while (parent != NULL)
    {
        if (!parent->m_CurXformVisibility)
        {
            return false;
        }
        parent = parent->m_pParent;
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief LOD レベルモデルのルートなら true を返します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//!
//! @return LOD レベルモデルのルートなら true を返します。
//-----------------------------------------------------------------------------
static bool IsLodLevelRoot(const MDagPath& xformPath)
{
    const std::string name = GetOutElementName(
        MFnDagNode(xformPath).name().asChar(), true);
    if (name.size() == 4 && name.find("LOD") == 0)
    {
        if ('1' <= name[3] && name[3] <= '9')
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief LOD レベルモデルのルートから LOD ベースモデルのルートの DAG パスを取得します。
//!
//! @param[in] levelRootPath LOD レベルモデルのルートの DAG パスです。
//!
//! @return LOD ベースモデルのルートの DAG パスを取得します。
//!         LOD レベルモデルの兄弟に存在しなければ無効なパスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetLodBaseRootFromLevelRoot(const MDagPath& levelRootPath)
{
    if (levelRootPath.length() >= 2)
    {
        MDagPath groupPath = levelRootPath;
        groupPath.pop();
        for (int childIdx = 0; childIdx < static_cast<int>(groupPath.childCount()); ++childIdx)
        {
            const MObject childObj = groupPath.child(childIdx);
            if (childObj.hasFn(MFn::kTransform) &&
                GetOutElementName(MFnDependencyNode(childObj).name().asChar(), true) == "Base")
            {
                return GetDirectChildDagPath(groupPath, childObj);
            }
        }
    }
    return MDagPath();
}

//-----------------------------------------------------------------------------
//! @brief モデルが LOD レベルモデルの transform ノードを含む場合に、
//!        LOD ベースモデルのルートの DAG パスを取得します。
//!
//! @param[in] ymodel モデルです。
//!
//! @return LOD ベースモデルのルートの DAG パスを返します。
//!         モデルが LOD レベルモデルの transform ノードを含まなければ無効なパスを返します。
//-----------------------------------------------------------------------------
static MDagPath GetLodBaseRootFromModel(const YModel& ymodel)
{
    MDagPath baseRootPath;
    for (size_t nodeIdx = 0; nodeIdx < ymodel.m_pYNodes.size(); ++nodeIdx)
    {
        const YNode& ynode = *ymodel.m_pYNodes[nodeIdx];
        if (!ynode.m_AddedFlag)
        {
            MDagPath parentPath = ynode.m_XformPath;
            if (parentPath.length() >= 3)
            {
                parentPath.pop();
                if (IsLodLevelRoot(parentPath))
                {
                    const MDagPath basePath = GetLodBaseRootFromLevelRoot(parentPath);
                    if (basePath.isValid())
                    {
                        if (!baseRootPath.isValid() || basePath.length() < baseRootPath.length())
                        {
                            baseRootPath = basePath;
                        }
                    }
                }
            }
        }
    }
    return baseRootPath;
}

//-----------------------------------------------------------------------------
//! @brief 指定した transform ノード以下のシェイプに影響する
//!        skinCluster オブジェクト群を配列に追加します。
//!        再帰的に呼ばれます。
//!
//! @param[in,out] pEffectiveSkinClusters 出力対象に影響する skinCluster オブジェクト配列へのポインタです。
//! @param[in] xformPath transform ノードの DAG パスです。
//-----------------------------------------------------------------------------
static void AppendEffectiveSkinClusters(
    MObjectArray* pEffectiveSkinClusters,
    const MDagPath& xformPath
)
{
    if (!IsEffectiveNode(xformPath.node()) &&
        !IsNoCuttingNode(xformPath.node()))
    {
        return;
    }

    for (int childIdx = 0; childIdx < static_cast<int>(xformPath.childCount()); ++childIdx)
    {
        const MObject childObj = xformPath.child(childIdx);
        if (childObj.hasFn(MFn::kTransform))
        {
            const MDagPath childPath = GetDirectChildDagPath(xformPath, childObj);
            AppendEffectiveSkinClusters(pEffectiveSkinClusters, childPath);
        }
        else if (childObj.apiType() == MFn::kMesh &&
            !MFnDagNode(childObj).isIntermediateObject())
        {
            CObjectArray deformerObjs;
            FindDeformerNodes(deformerObjs, childObj);
            for (size_t deformerIdx = 0; deformerIdx < deformerObjs.size(); ++deformerIdx)
            {
                const MObject& deformerObj = deformerObjs[deformerIdx];
                if (deformerObj.apiType() == MFn::kSkinClusterFilter &&
                    IsNodeStateEffective(deformerObj) &&
                    FindObjectInArray(*pEffectiveSkinClusters, deformerObj) == -1)
                {
                    //cerr << "effective skin cluster (append): " << MFnDependencyNode(deformerObj).name() << ": " << xformPath.partialPathName() << endl;
                    pEffectiveSkinClusters->append(deformerObj);
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 出力対象に影響する skinCluster オブジェクト群を取得します。
//!
//! @param[out] pEffectiveSkinClusters 出力対象に影響する skinCluster オブジェクト配列へのポインタです。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static void GetEffectiveSkinClusters(
    MObjectArray* pEffectiveSkinClusters,
    const YModel& ymodel
)
{
    const YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // 出力するノードに影響する skinCluster オブジェクト群を取得します。
    pEffectiveSkinClusters->clear();
    for (size_t nodeIdx = 0; nodeIdx < ymodel.m_pYNodes.size(); ++nodeIdx)
    {
        const YNode& ynode = *ymodel.m_pYNodes[nodeIdx];
        if (!ynode.m_SkinClusterObj.isNull())
        {
            //cerr << "effective skin cluster: " << MFnDependencyNode(ynode.m_SkinClusterObj).name() << ": " << ynode.m_XformPath.partialPathName() << endl;
            pEffectiveSkinClusters->append(ynode.m_SkinClusterObj);
        }
    }

    //-----------------------------------------------------------------------------
    // LOD エクスポートで、ベースモデル以外を出力中の場合、
    // ベースモデルに影響する skinCluster オブジェクト群も取得します。
    if (yopt.m_ExportsLod)
    {
        const MDagPath baseRootPath = GetLodBaseRootFromModel(ymodel);
        if (baseRootPath.isValid())
        {
            //cerr << "LOD base root: " << baseRootPath.partialPathName() << endl;
            for (int childIdx = 0; childIdx < static_cast<int>(baseRootPath.childCount()); ++childIdx)
            {
                const MObject childObj = baseRootPath.child(childIdx);
                if (childObj.hasFn(MFn::kTransform))
                {
                    const MDagPath childPath = GetDirectChildDagPath(baseRootPath, childObj);
                    AppendEffectiveSkinClusters(pEffectiveSkinClusters, childPath);
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief セグメントスケール補正を取得します。
//!        本来、fmd にはバインド時の値を出力するべきですが、
//!        取得する方法が不明なので現在の値を取得します。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetYNodeScaleCompensate(YModel& ymodel)
{
    YScene& yscene = ymodel.GetScene();

    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];
        ynode.m_OrgScaleCompensate = false;

        // ノードが joint ノードかつ親ノードも joint ノードなら
        // segmentScaleCompensate アトリビュートの値を取得します。
        if (ynode.m_MayaJointFlag && HasMayaJointParent(ynode.m_XformPath))
        {
            //if (ynode.m_SkinInfluenceFlag &&
            //  GetBindPoseScaleCompensate(ynode.m_OrgScaleCompensate, ynode.m_XformPath))
            //{
            //  // bind ssc
            //}
            //else
            {
                MFnDagNode dagFn(ynode.m_XformPath);
                dagFn.findPlug("segmentScaleCompensate").getValue(ynode.m_OrgScaleCompensate);
            }

            // セグメントスケール補正が ON なのに
            // 親のスケールが inverseScale に接続されていなければ警告します。
            if (ynode.m_OrgScaleCompensate &&
                !IsParentScaleConnectedToMayaJoint(ynode.m_XformPath))
            {
                YShowWarning(&yscene, // Parent scale is not connected to inverseScale: %s // ヘルプに誘導
                    "joint ノードで、親ノードのスケールが inverseScale アトリビュートに接続されていません: {0} \n"
                    "Maya 上でセグメントスケール補正が反映されないので、接続してください。"
                    "（詳細はプラグインのヘルプを見てください）",
                    "The scale of the parent node of a joint node is not connected to the inverseScale attribute: {0} \n"
                    "Because segment scale correction is not applied under Maya, make sure that you connect the parent node to the attribute. "
                    "(See the Plug-In Help for details.)",
                    ynode.m_OrgName);
            }
        }
        ynode.m_ScaleCompensate = ynode.m_OrgScaleCompensate; // 出力用の変数にコピーします。
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief get transform from global matrix sub
//-----------------------------------------------------------------------------
static void GetTransformFromGlobalMtxSub(
    MVector& s, MVector& r, MVector& t,
    const MMatrix& mtxGlobal,
    const bool scaleCompensate,
    const bool parentFlag,
    const MMatrix& mtxParentGlobalInv,
    const MVector& parentScale)
{
    MMatrix mtx = mtxGlobal;

    //-----------------------------------------------------------------------------
    // get local matrix
    if (parentFlag)
    {
        mtx = mtx * mtxParentGlobalInv;

        if (scaleCompensate)
        {
            // reset scale compensate
            double psx = parentScale.x;
            double psy = parentScale.y;
            double psz = parentScale.z;
            if (psx != 1.0)
            {
                mtx[0][0] *= psx;
                mtx[1][0] *= psx;
                mtx[2][0] *= psx;
            }
            if (psy != 1.0)
            {
                mtx[0][1] *= psy;
                mtx[1][1] *= psy;
                mtx[2][1] *= psy;
            }
            if (psz != 1.0)
            {
                mtx[0][2] *= psz;
                mtx[1][2] *= psz;
                mtx[2][2] *= psz;
            }
        }
    }

    GetTransformFromMtx(s, r, t, mtx);
}

//-----------------------------------------------------------------------------
//! @brief get transform from global matrix
//-----------------------------------------------------------------------------
static void GetTransformFromGlobalMtx(
    MVector& s, MVector& r, MVector& t,
    const MMatrix& mtxGlobal, const YNode& ynode)
{
    bool parentFlag = false;
    MMatrix mtxParentGlobalInv;
    MVector parentScale;
    if (ynode.m_pParent != NULL)
    {
        parentFlag = true;
        const YNode& parent = *ynode.m_pParent;
        mtxParentGlobalInv = parent.m_BindGlobalInvMtx;
        parentScale = parent.m_BindS;
    }

    GetTransformFromGlobalMtxSub(s, r, t, mtxGlobal,
        ynode.m_OrgScaleCompensate,
        parentFlag, mtxParentGlobalInv, parentScale);
}

//-----------------------------------------------------------------------------
//! @brief get skinned shape matrix (global)
//!
//! @param[in,out] yscene シーンです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetSkinedShapeMtx(MMatrix& mtx, YScene& yscene, const YNode& ynode)
{
    //-----------------------------------------------------------------------------
    // get geom matrix (global matrix at bind)
    MPlug gmPlug = MFnDependencyNode(ynode.m_SkinClusterObj).findPlug("geomMatrix");
    MObject gmObj;
    if (!gmPlug.getValue(gmObj))
    {
        YShowWarning(&yscene, // Get geom matrix failed: %s // 通常は発生しない
            "geomMatrix の取得に失敗しました: {0}",
            "Failed to get geomMatrix: {0}",
            gmPlug.info().asChar());
        return MS::kFailure;
    }
    mtx = MFnMatrixData(gmObj).matrix();

    //-----------------------------------------------------------------------------
    // adjust

    // magnify
    const YExpOpt& yopt = yscene.GetOpt();
    mtx[3][0] *= yopt.m_InternalMagnify;
    mtx[3][1] *= yopt.m_InternalMagnify;
    mtx[3][2] *= yopt.m_InternalMagnify;

    // Sp-1 * S * Sp * St * Rp-1 * R * Rp * Rt * T
    // ↓
    //        S * Sp * St * Rp-1 * R * Rp * Rt * T
    //
    // Sp-1 は下位ノードの translate および頂点座標で吸収
    MMatrix mtxSp;
    mtxSp[3][0] = ynode.m_ScalePivot.x;
    mtxSp[3][1] = ynode.m_ScalePivot.y;
    mtxSp[3][2] = ynode.m_ScalePivot.z;
    mtx = mtxSp * mtx;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief バインドポーズにおけるノードの変換を取得します。
//!
//! @param[out] s スケールを格納します。
//! @param[out] r 回転を格納します。
//! @param[out] t 移動を格納します。
//! @param[in,out] yscene シーンです。
//! @param[in] ynode ノードです。
//! @param[in] effectiveSkinClusters 出力対象に影響する skinCluster オブジェクト配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetYNodeBindTransform(
    MVector& s,
    MVector& r,
    MVector& t,
    YScene& yscene,
    const YNode& ynode,
    const MObjectArray& effectiveSkinClusters
)
{
    //-----------------------------------------------------------------------------
    // get bind global matrix
    const YExpOpt& yopt = yscene.GetOpt();
    MMatrix mtx;    // global
    if (!GetBindPoseMtx(mtx, yscene, ynode.m_XformPath, effectiveSkinClusters,
        yopt.m_InternalMagnify))
    {
        if (ynode.m_SkinInfluenceFlag)
        {
            YShowWarning(&yscene, // Bind pose is not found: %s // 通常は発生しない
                "バインドポーズの取得に失敗しました: {0}",
                "Failed to get bind pose: {0}",
                ynode.m_OrgName);
        }
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // adjust

    // joint ノードのときは Sp = Rp = (0, 0, 0) だが、
    // transform ノードを Add Influence した場合は次の補正が必要
    //
    // Sp-1 * S * Sp * St * Rp-1 * R * Rp * Rt * T
    // ↓
    //        S * Sp * St * Rp-1 * R * Rp * Rt * T
    //
    // Sp-1 は下位ノードの translate および頂点座標で吸収
    MMatrix mtxSp;
    mtxSp[3][0] = ynode.m_ScalePivot.x;
    mtxSp[3][1] = ynode.m_ScalePivot.y;
    mtxSp[3][2] = ynode.m_ScalePivot.z;
    mtx = mtxSp * mtx;

    //-----------------------------------------------------------------------------
    // get local transform
    // 親の逆行列を掛けてローカル行列を求めているので、
    // 親の m_ScalePivot による translate の補整は不要
    GetTransformFromGlobalMtx(s, r, t, mtx, ynode);

    // スキンに影響しているスケルトンの途中から
    // Export Selection した場合には未対応

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief set ynode visibility anim flag from parent
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetYNodeVisibilityAnimFlagFromParent(YModel& ymodel)
{
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];
        if (!ynode.m_VisibilityAnimFlag)
        {
            const YNode* parent = ynode.m_pParent;
            while (parent != NULL)
            {
                if (parent->m_VisibilityAnimFlag)
                {
                    ynode.m_VisibilityAnimFlag = true;
                    break;
                }
                parent = parent->m_pParent;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 開始フレームにおけるノードの変換と可視性を取得します。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetYNodeTransform(YModel& ymodel)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // check size
    const YExpOpt& yopt = yscene.GetOpt();
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    if (nodeCount == 0)
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // 出力対象に影響する skinCluster オブジェクト群を取得します。
    MObjectArray effectiveSkinClusters;
    GetEffectiveSkinClusters(&effectiveSkinClusters, ymodel);

    //-----------------------------------------------------------------------------
    // scale compensate の設定値を YNode にコピーします。
    GetYNodeScaleCompensate(ymodel);

    //-----------------------------------------------------------------------------
    // 各ノードの変換と可視性を取得します。
    int iNode;
    for (iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];

        //-----------------------------------------------------------------------------
        // check inherits transform
        if (!ynode.m_InheritsXform && !ynode.m_IsSkinned)
        {
            YShowWarning(&yscene, // Inherits transform of non-skinned object is off: %s
                "スキニングを設定したシェイプ以外の transform ノードのトランスフォームの継承アトリビュートが OFF になっています: {0} \n"
                "ON にしないと正しく表示されません。",
                "The Inherits Transform attribute of the transform node of the non-skinned shape is cleared: {0} \n"
                "Objects do not display correctly unless all of these settings are selected.",
                ynode.m_OrgName);
        }

        //-----------------------------------------------------------------------------
        // get translate offset
        // 親ノードの scale pivot が必要なのでここで求めます。
        MVector ofs = ynode.m_RotatePivotTrans + ynode.m_RotatePivot;
        if (ynode.m_pParent != NULL)
        {
            if (!ynode.m_IsSkinned || ynode.m_InheritsXform)
            {
                ofs = ofs - ynode.m_pParent->m_ScalePivot;
            }
        }
        ynode.m_TranslateOffset = ofs;

        //-----------------------------------------------------------------------------
        // get SRT
        MMatrix mtxEnvelopeGlobal;
        if (ynode.m_OrgKind == YNode::KindModelRoot)
        {
            // fixed (no transform)
            ynode.m_XformS = MVector(1.0, 1.0, 1.0);
            ynode.m_XformR = MVector(0.0, 0.0, 0.0);
            ynode.m_XformT = MVector(0.0, 0.0, 0.0);
        }
        else if (ynode.m_IsSkinned)
        {
            GetSkinedShapeMtx(mtxEnvelopeGlobal, yscene, ynode);
            GetTransformFromGlobalMtx(ynode.m_XformS, ynode.m_XformR, ynode.m_XformT,
                mtxEnvelopeGlobal, ynode);
            // 親の逆行列を掛けてローカル行列を求めているので、
            // 親の m_ScalePivot による translate の補整は不要
        }
        else
        {
            double scaleArray[3] = { 1.0, 1.0, 1.0 };
            MEulerRotation eulerRotate = MEulerRotation::identity;
            MVector translate = MVector::zero;

            MFnTransform xformFn(ynode.m_XformPath);
            xformFn.getScale(scaleArray);
            xformFn.getRotation(eulerRotate);
            translate = xformFn.getTranslation(MSpace::kTransform) * yopt.m_InternalMagnify;

            //-----------------------------------------------------------------------------
            // check shear
            double sh[3];
            xformFn.getShear(sh);
            if (!RIsSame(sh[0], 0.0) || !RIsSame(sh[1], 0.0) || !RIsSame(sh[2], 0.0))
            {
                YShowWarning(&yscene, // Shear is not zero: (ignored): %s
                    "transform ノードのシアアトリビュートに 0 でない値が入っているため正しく出力できません: {0} \n"
                    "「トランスフォームのフリーズ」を実行するか、シアアトリビュートを (0, 0, 0) にしてください。",
                    "Correct output is impossible because the Shear attribute of the transform node has a value other than 0: {0} \n"
                    "Make sure that you either execute Freeze Transformations or set the Shear attribute to (0, 0, 0).",
                    ynode.m_OrgName);
            }

            //-----------------------------------------------------------------------------
            // adjust
            ynode.AdjustRotate(eulerRotate);
            ynode.AdjustTranslate(translate, eulerRotate);

            //-----------------------------------------------------------------------------
            // set
            ynode.m_XformS = MVector(scaleArray[0], scaleArray[1], scaleArray[2]);
            ynode.m_XformR = MVector(eulerRotate.x * R_M_RAD_TO_DEG, eulerRotate.y * R_M_RAD_TO_DEG, eulerRotate.z * R_M_RAD_TO_DEG);
            ynode.m_XformT = translate;
        }

        //-----------------------------------------------------------------------------
        // get bind pose
        ynode.m_BindS = ynode.m_XformS;
        ynode.m_BindR = ynode.m_XformR;
        ynode.m_BindT = ynode.m_XformT;
        if (ynode.m_ClusterHandleFlag)
        {
            // クラスターハンドルのバインドポーズは単位行列
            ynode.m_BindS = MVector(1.0, 1.0, 1.0);
            ynode.m_BindR = MVector::zero;
            ynode.m_BindT = MVector::zero;
        }
        else if (!ynode.m_AddedFlag)
        {
            #if (MAYA_API_VERSION >= 200800)
            if (IsMultipleBindPoseUsed(ynode.m_XformPath, effectiveSkinClusters))
            {
                YShowError(&yscene, // Multiple bind poses are not supported: %s
                    "複数のバインドポーズが使用されています: {0} \n"
                    "スキンをデタッチし、各オブジェクトを共通のバインドポーズでバインドし直してください。",
                    "Multiple bind poses are in use: {0} \n"
                    "Detach the skin and use a common bind pose to rebind each object.",
                    ynode.m_OrgName);
                return MS::kFailure;
            }
            #endif
            GetYNodeBindTransform(ynode.m_BindS, ynode.m_BindR, ynode.m_BindT,
                yscene, ynode, effectiveSkinClusters);
        }

        //-----------------------------------------------------------------------------
        // set out xform
        ynode.m_OutXformS = ynode.m_XformS;
        ynode.m_OutXformR = ynode.m_XformR;
        ynode.m_OutXformT = ynode.m_XformT;
        ynode.m_OutBindS = ynode.m_BindS;
        ynode.m_OutBindR = ynode.m_BindR;
        ynode.m_OutBindT = ynode.m_BindT;

        //-----------------------------------------------------------------------------
        // set matrix
        if (ynode.m_OrgScaleCompensate && ynode.m_pParent != NULL)
        {
            CreateTransformMtxScaleCompensate(ynode.m_LocalMtx,
                ynode.m_XformS, ynode.m_XformR, ynode.m_XformT,
                ynode.m_pParent->m_XformS);
            CreateTransformMtxScaleCompensate(ynode.m_BindLocalMtx,
                ynode.m_BindS, ynode.m_BindR, ynode.m_BindT,
                ynode.m_pParent->m_BindS);
        }
        else
        {
            CreateTransformMtx(ynode.m_LocalMtx,
                ynode.m_XformS, ynode.m_XformR, ynode.m_XformT);
            CreateTransformMtx(ynode.m_BindLocalMtx,
                ynode.m_BindS, ynode.m_BindR, ynode.m_BindT);
        }
        const MVector noScale(1.0, 1.0, 1.0);
        MMatrix mtxBindNoScaleLocal;
        CreateTransformMtx(mtxBindNoScaleLocal,
                noScale, ynode.m_BindR, ynode.m_BindT);

        ynode.m_GlobalMtx = ynode.m_LocalMtx;
        ynode.m_BindGlobalMtx = ynode.m_BindLocalMtx;
        ynode.m_BindNoScaleGlobalMtx = mtxBindNoScaleLocal;
        if (ynode.m_pParent != NULL)
        {
            ynode.m_GlobalMtx *= ynode.m_pParent->m_GlobalMtx;
            ynode.m_BindGlobalMtx *= ynode.m_pParent->m_BindGlobalMtx;
            ynode.m_BindNoScaleGlobalMtx *= ynode.m_pParent->m_BindNoScaleGlobalMtx;
        }

        if (ynode.m_IsSkinned)
        {
            ynode.m_BindGlobalMtx = mtxEnvelopeGlobal;
        }

        ynode.m_GlobalInvMtx = ynode.m_GlobalMtx.inverse();
        ynode.m_BindGlobalInvMtx = ynode.m_BindGlobalMtx.inverse();
        ynode.m_BindNoScaleGlobalInvMtx = ynode.m_BindNoScaleGlobalMtx.inverse();

        //-----------------------------------------------------------------------------
        // check zero scale
            // removed (2008/01/25)
        //MVector outS = (yopt.m_FmdIsBindPose) ?
        //  ynode.m_OutBindS : ynode.m_OutXformS;
        //if (RIsSame(outS.x, 0.0) || RIsSame(outS.y, 0.0) || RIsSame(outS.z, 0.0))
        //{
        //  YShowError(&yscene, "ノードのスケールが 0 です: {0}", "Node scale is zero: {0}", ynode.m_OrgName); // 現在は発生しない
        //  return MS::kFailure;
        //}

        //-----------------------------------------------------------------------------
        // get visibility
        if (ynode.m_OrgKind == YNode::KindModelRoot)
        {
            ynode.m_CurXformVisibility = true;
        }
        else
        {
            ynode.m_XformVisibilityPlug.getValue(ynode.m_CurXformVisibility);
        }
        ynode.m_Visibility = GetOutVisibility(ynode);
    }

    //-----------------------------------------------------------------------------
    // set ynode visibility anim flag from parent
    SetYNodeVisibilityAnimFlagFromParent(ymodel);

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief モデルのノード群のユーザーデータを取得します。
//!
//! @param[in,out] pYModel モデルへのポインターです。
//-----------------------------------------------------------------------------
static void GetYNodeUserData(YModel* pYModel)
{
    YScene& yscene = pYModel->GetScene();
    const int nodeCount = static_cast<int>(pYModel->m_pYNodes.size());
    for (int nodeIdx = 0; nodeIdx < nodeCount; ++nodeIdx)
    {
        YNode& ynode = *pYModel->m_pYNodes[nodeIdx];
        ynode.GetUserData(&yscene);
    }
}

//-----------------------------------------------------------------------------
//! @brief UV セットファミリ名からインスタンスに対応した UV セット名を取得します。
//!        UV セットがインスタンスごとに設定されている場合、
//!        mesh ノードのインスタンス番号に対応する UV セット名を取得します。
//!
//! @param[in] uvSetFamilyName UV セットファミリ名です。
//! @param[in] meshPath mesh ノードの DAG パスです。
//!
//! @return インスタンスに対応した UV セット名を返します。
//!         存在しなければ空文字を返します。
//-----------------------------------------------------------------------------
static MString GetInstanceUvSetName(const MDagPath& meshPath, const MString& uvSetFamilyName)
{
    MFnMesh meshFn(meshPath);
    if (!meshFn.isUVSetPerInstance(uvSetFamilyName))
    {
        return uvSetFamilyName;
    }
    else // UV セットがインスタンスごとに設定されている場合
    {
        MStringArray members;
        meshFn.getUVSetsInFamily(uvSetFamilyName, members);
        const int memberCount = static_cast<int>(members.length());
        // UV セットファミリ中の UV セットのうち、
        // mesh ノードのインスタンス番号に対応したものを取得します。
        for (int iMember = 0; iMember < memberCount; ++iMember)
        {
            MIntArray instances;
            meshFn.getAssociatedUVSetInstances(members[iMember], instances);
            if (YFindValueInArray(instances, meshPath.instanceNumber()) != -1)
            {
                //cerr << "instance uv set: " << meshPath.partialPathName() << ": " << members[iMember] << " " << members << " " << instances << R_ENDL;
                return members[iMember];
            }
        }
        return MString();
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャノードに対応する UV セット名を取得します。
//!        UV セットがインスタンス間で共有されない場合、
//!        インスタンスに対応した UV セット名を返します。
//!
//! @param[out] uvSetFamilyName UV セットファミリ名を格納します。
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] ymodel モデルです。
//! @param[in] iTexNode m_TexNodes の添字として使用するテクスチャノードのインデックスです。
//! @param[in] defaultUvSet 標準の UV セット名です。
//!
//! @return インスタンスに対応した UV セット名を返します。UV セットが存在しない場合は
//!         引数 defaultUvSet に指定された名前を返します。
//-----------------------------------------------------------------------------
static MString GetUvSetNameForTexNode(
    MString& uvSetFamilyName,
    const MDagPath& meshPath,
    const YModel& ymodel,
    const int iTexNode,
    const MString& defaultUvSet
)
{
    MStatus status;

    // まずは標準の UV セット名を設定しておきます。
    // iTexNode でテクスチャが指定されていない、もしくは指定のテクスチャが存在していない
    // 場合は、この標準の UV セット名が返されます。
    MString uvset = defaultUvSet;
    uvSetFamilyName = defaultUvSet;

    // テクスチャが指定されていない場合は処理をスキップし、
    // 標準の UV セット名を返します。
    if (iTexNode != -1)
    {
        //-----------------------------------------------------------------------------
        // check uv chooser
        const TexNode& texNode = ymodel.m_TexNodes[iTexNode];
        if (!texNode.m_ChooserObj.isNull())
        {
            // uvChooser ノードの uvSets アトリビュートからメッシュに接続されています。
            // UV セットのプラグ配列を取得します。
            MFnDependencyNode depFn(texNode.m_ChooserObj);
            MPlug uvSetsPlug = depFn.findPlug("uvSets");
            // uvSets アトリビュートは配列になっているので、その配列要素の論理インデックス
            // を取得します。
            MIntArray idxs;
            uvSetsPlug.getExistingArrayAttributeIndices(idxs);
            // 得られた論理インデックスからプラグの要素を取得し、引数 meshPath で指定された
            // メッシュに接続されているプラグを検索します。
            const int uvSetsCount = static_cast<int>(idxs.length());
            for (int iElem = 0; iElem < uvSetsCount; ++iElem)
            {
                // 論理インデックスからプラグの要素を取得します。
                MPlug uvnPlug = uvSetsPlug.elementByLogicalIndex(idxs[iElem], &status);
                if (!status)
                {
                    break;
                }
                // 取得したプラグに出力先として接続されているプラグの配列を取得します。
                MPlugArray plugArray;
                uvnPlug.connectedTo(plugArray, true, false);
                if (plugArray.length() == 0)
                {
                    continue;
                }
                // 出力先のプラグがメッシュに接続されているのであれば、そのプラグから
                // UV セットの名前を取得します。
                if (plugArray[0].node() == meshPath.node())
                {
                    uvnPlug.getValue(uvset);
                    break;
                }
            }
        }
        // インスタンス単位の UV セットの場合、プラグから出力した名前は
        // "uvSet(n)" のようにインデックス付きの名前ですが、
        // n の値は mesh ノードのインスタンスに対応しているとは限りません。
        uvSetFamilyName = uvset;
        const int iParenthesis = uvSetFamilyName.index('(');
        if (iParenthesis > 0)
        {
            uvSetFamilyName = uvSetFamilyName.substring(0, iParenthesis - 1);
        }
        uvset = GetInstanceUvSetName(meshPath, uvSetFamilyName);
    }

    return uvset;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャのラップを取得します。
//!
//! @param[out] wrapU テクスチャの U 方向のラップを格納します。
//! @param[out] wrapV テクスチャの V 方向のラップを格納します。
//! @param[in] fileObj Maya の file ノードです。
//!
//! @return 処理結果を返します。現在常に成功します。
//-----------------------------------------------------------------------------
static MStatus GetTextureWrap(
    RSampler::Wrap& wrapU,
    RSampler::Wrap& wrapV,
    const MObject& fileObj
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // get wrap
    MFnDependencyNode texFn(fileObj);
    bool mayaWrapU, mayaWrapV;
    texFn.findPlug("wrapU").getValue(mayaWrapU);
    texFn.findPlug("wrapV").getValue(mayaWrapV);

    //-----------------------------------------------------------------------------
    // get mirror
    bool mirrorU, mirrorV;
    texFn.findPlug("mirrorU").getValue(mirrorU);
    texFn.findPlug("mirrorV").getValue(mirrorV);

    //-----------------------------------------------------------------------------
    // set wrap value
    wrapU = (mirrorU) ? RSampler::MIRROR :
        (mayaWrapU) ? RSampler::REPEAT : RSampler::CLAMP;
    wrapV = (mirrorV) ? RSampler::MIRROR :
        (mayaWrapV) ? RSampler::REPEAT : RSampler::CLAMP;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 同じテクスチャ名を持つ 2 つのテクスチャイメージの
//!        テクスチャファイルパスが同じか比較し、異なれば警告を表示します。
//!
//! @param[in,out] yscene シーンです。
//!
//! @return テクスチャファイルパスが同じなら kSuccess を返します。
//-----------------------------------------------------------------------------
static MStatus CheckTextureIdentical(YScene& yscene, const RImage& tex0, const RImage& tex1)
{
    const YExpOpt& yopt = yscene.GetOpt();
    if (!yopt.m_CheckElementFlag && tex0.GetFilePaths()[0] != tex1.GetFilePaths()[0])
    {
        YShowWarning(&yscene, // There are multiple texture files with the same name: %s
            "出力対象にファイル名が同じでフォルダまたは拡張子の異なるテクスチャーファイルが複数存在します: {0} \n"
            "出力される ftx ファイルは 1 つだけなので注意してください。\n"
            "    {1} \n"
            "    {2}",
            "Two or more texture files have the same filename as the export target, but are in different folders or have a different extension: {0} \n"
            "Note that only one ftx file will be exported. \n"
            "    {1} \n"
            "    {2}",
            tex0.GetName(), tex0.GetFilePaths()[0], tex1.GetFilePaths()[0]);
        return MS::kFailure;
    }
    else
    {
        return MS::kSuccess;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャイメージの情報を取得して m_TexImgs に追加します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[out] retIndex 追加したテクスチャイメージの m_TexImgs 内のインデックスを格納します。
//! @param[in] filePaths 画像ファイルのパス配列です。
//! @param[in] sampler サンプラです。
//! @param[in] curSwizzle 現在の初期スウィズル値です。
//! @param[in] isSingleEnvCube 環境立方体の +X 面にキューブマップの DDS ファイルが接続されていれば true です。
//! @param[in] orgMatName Maya のマテリアル名です（エラー表示用）。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetTexImage(
    YModel& ymodel,
    int& retIndex,
    const RStringArray& filePaths,
    const RSampler& sampler,
    const int curSwizzle,
    const bool isSingleEnvCube,
    const std::string& orgMatName
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    const bool checksTexFileError = !yopt.m_CheckElementFlag && yopt.ExportsTexture();

    //-----------------------------------------------------------------------------
    // 次元を決定します。
    bool checksCubeWh = false;
    int imageW = 1;
    int imageH = 1;
    if (sampler.m_Hint == RSampler::REFLECTION && filePaths.size() == 1)
    {
        // 反射マップの場合、画像の幅と高さを調べてキューブマップか判定します。
        checksCubeWh = true;
        if (!GetImageInfoForMaya(imageW, imageH,
            yscene, filePaths[0], !checksTexFileError))
        {
            checksCubeWh = false;
            if (checksTexFileError)
            {
                return MS::kFailure;
            }
        }
    }
    const RImage::Dimension dimension = (isSingleEnvCube) ?
        RImage::DIM_CUBE_INPUT :
        RImage::GetDimension(static_cast<int>(filePaths.size()), checksCubeWh, imageW, imageH);
    const std::string mainFilePath = (dimension == RImage::DIM_CUBE_SEPARATE) ?
        filePaths[RImage::CUBE_FACE_PZ] : filePaths[0];
    //cerr << "image dim: " << mainFilePath << ": " << dimension << endl;

    //-----------------------------------------------------------------------------
    // テクスチャイメージのヒント情報を決定します。
    const std::string imgHint = sampler.GetImageHintString();

    //-----------------------------------------------------------------------------
    // 既存のテクスチャイメージから同じ画像ファイルに対するデータを探します。
    const int texImgCount = static_cast<int>(ymodel.m_TexImgs.size());
    for (int iTexImg = 0; iTexImg < texImgCount; ++iTexImg)
    {
        const RImage& other = ymodel.m_TexImgs[iTexImg];
        if (RIsSameStringNoCase(other.GetMainFilePath(), mainFilePath))
        {
            if (checksTexFileError)
            {
                if (other.GetDimension()        != dimension        ||
                    other.GetFilePaths().size() != filePaths.size())
                {
                    if (other.GetDimension() == RImage::DIM_2D || dimension == RImage::DIM_2D)
                    {
                        YShowError(&yscene, // Texture is used for 2D and cube map: %s (%s)
                            "1 つのテクスチャーファイルが 2D とキューブマップの両方に使用されています。別ファイルにしてください: {0} ({1})",
                            "A single texture file is being used as both a 2D texture and a cube map. Use separate files: {0} ({1})",
                            mainFilePath, orgMatName);
                    }
                    else
                    {
                        YShowError(&yscene, // Texture is used for different type cube map: %s (%s)
                            "1 つのテクスチャーファイルが異なる種類のキューブマップ（環境立方体と水平十字キューブマップなど）に使用されています。別ファイルにしてください: {0} ({1})",
                            "A single texture file is being used for different types of cube maps (such as an environmental cube map and a horizontal cross cube map). Use separate files: {0} ({1})",
                            mainFilePath, orgMatName);
                    }
                    return MS::kFailure;
                }
                if (dimension == RImage::DIM_CUBE_SEPARATE)
                {
                    for (int iFile = 0; iFile < static_cast<int>(filePaths.size()); ++iFile)
                    {
                        if (!RIsSameStringNoCase(other.GetFilePaths()[iFile], filePaths[iFile]))
                        {
                            YShowError(&yscene, // Cube map image is not same: %s %s (%s)
                                "前面のテクスチャーファイルが同じで他の面のテクスチャーファイルが異なる環境立方体が存在します: {0} {1} ({2}) \n"
                                "環境立方体の場合、前面のテクスチャーのファイル名がテクスチャー名となるので、"
                                "前面のテクスチャーファイルを別ファイルにしてください。",
                                "A cube map exists with the same texture file for the front side but different texture files for the other sides: {0} {1} ({2}) \n"
                                "If the cube is used for environmental mapping, place the texture file for the front side "
                                "in a separate file because the filename of the front texture is used as the texture name.",
                                other.GetFilePaths()[iFile], filePaths[iFile], GetEnvCubeFaceAttrName(iFile));
                            return MS::kFailure;
                        }
                    }
                }
                if (other.GetHint() != imgHint)
                {
                    YShowError(&yscene, // Texture is used for different usage: %s (%s)
                        "1 つのテクスチャーファイルが異なる用途（アルベド用、不透明度用、法線マップ用、スペキュラ用など）に使用されています。別ファイルにしてください: {0} ({1})",
                        "A single texture file is being used for different purposes (such as for albedo, for transparency, for normal mapping, and for specular lighting). Use separate files: {0} ({1})",
                        mainFilePath, orgMatName);
                    return MS::kFailure;
                }
            }

            // 既存のテクスチャイメージの初期スウィズル値が小さければ更新します。
            if (other.GetInitialSwizzle() < curSwizzle)
            {
                ymodel.m_TexImgs[iTexImg].SetInitialSwizzle(curSwizzle);
            }

            retIndex = iTexImg;
            return MS::kSuccess;
        }
    }

    //-----------------------------------------------------------------------------
    // add tex img
    RImage texImg(texImgCount);

    texImg.SetHint(imgHint);
    texImg.SetLinearFlag(yopt.GetTexLinearFlag(imgHint));
    texImg.SetDimension(dimension);
    texImg.SetInitialSwizzle(curSwizzle);

    texImg.SetFilePaths(filePaths); // パスからテクスチャ名も設定されます。
    if (texImg.GetName() != texImg.GetOrgName())
    {
        YShowWarning(&yscene, // Texture name is changed: %s -> %s
            "テクスチャー名に禁止文字が含まれています。禁止文字は「_（アンダーバー）」に変換して出力されます: {0} -> {1}",
            "The texture name contains prohibited characters. Prohibited characters are converted to an underscore (_), and then exported.: {0} -> {1}",
            texImg.GetOrgName(), texImg.GetName());
    }

    ymodel.m_TexImgs.push_back(texImg);

    retIndex = texImgCount;

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのパスを取得します。
//!
//! @param[in] name テクスチャ名です。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static std::string GetFtxFilePath(const std::string& name, const YExpOpt& yopt)
{
    return std::string(RExpOpt::TexFolderName) + "/" +
        name + "." + yopt.GetExtension(RExpOpt::FTX);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャサンプラの属性を取得します。
//!
//! @param[in,out] sampler サンプラです。
//! @param[in,out] ymodel モデルです。
//! @param[in] texNode テクスチャノードです。
//! @param[in] curSwizzle 現在の初期スウィズル値です。
//! @param[in] orgMatName Maya のマテリアル名です（エラー表示用）。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetTexSampler(
    RSampler& sampler,
    YModel& ymodel,
    const TexNode& texNode,
    const int curSwizzle,
    const std::string& orgMatName
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // サンプラの名前とヒント情報の文字列のオーバーライドを設定します。
    sampler.m_NameOverride = texNode.m_SamplerNameOverride;
    sampler.m_HintOverride = texNode.m_SamplerHintOverride;

    //-----------------------------------------------------------------------------
    // テクスチャのイメージ情報を取得します。
    status = GetTexImage(ymodel, sampler.m_ImageIndex,
        texNode.m_FilePaths, sampler, curSwizzle, texNode.m_IsSingleEnvCube, orgMatName);
    CheckStatus(status);
    const RImage& texImg = ymodel.m_TexImgs[sampler.m_ImageIndex];

    sampler.m_TexName = texImg.GetName();
    sampler.m_TexPath = GetFtxFilePath(texImg.GetName(), yopt);

    //-----------------------------------------------------------------------------
    // テクスチャのラップ設定を取得します。
    sampler.m_WrapU = RSampler::CLAMP;
    sampler.m_WrapV = RSampler::CLAMP;
    if (!texNode.IsEnvMap() && !texNode.m_TexObj.isNull())
    {
        GetTextureWrap(sampler.m_WrapU, sampler.m_WrapV, texNode.m_TexObj);
    }

    //-----------------------------------------------------------------------------
    // テクスチャ SRT を設定します。
    ROriginalTexsrt& texSrt = sampler.m_OriginalTexsrt;
    texSrt.m_Mode        = static_cast<ROriginalTexsrt::Mode>(yopt.m_TexSrtMode);
    texSrt.m_Scale.x     = texNode.m_ScaleU;
    texSrt.m_Scale.y     = texNode.m_ScaleV;
    texSrt.m_Rotate      = texNode.m_Rotate;
    texSrt.m_Translate.x = texNode.m_TranslateU;
    texSrt.m_Translate.y = texNode.m_TranslateV;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief フレーム拡張子リストをカスタムアトリビュートから取得します。
//!
//! @param[out] feList フレーム拡張子リストを格納します。
//! @param[in,out] yscene シーンです。
//! @param[in] texFn file ノードのファンクションノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetFrameExtensionList(
    MIntArray& feList,
    YScene& yscene,
    const MFnDependencyNode& texFn
)
{
    MStatus status;
    const YExpOpt& yopt = yscene.GetOpt();

    MPlug feListPlug = FindPlugQuiet(texFn, "nw4fFeList", &status);
    if (status)
    {
        MString str;
        if (feListPlug.getValue(str) && str.length() != 0)
        {
            if (!GetArrayFromStringCutSame(feList, str.asChar()))
            {
                if (!yopt.m_CheckElementFlag)
                {
                    YShowError(&yscene, // Frame extension list is wrong: %s: %s
                        "テクスチャーパターンアニメーション用に出力するテクスチャーのリストが正しくありません: {0}: {1} \n"
                        "テクスチャーパターンアニメーション設定プラグインでリストを修正してください。",
                        "The list of textures to output for the texture pattern animation is incorrect: {0}: {1} \n"
                        "Use the Plug-In for Setting Texture Pattern Animations to correct the list.",
                        texFn.name().asChar(), str.asChar());
                }
                return MS::kFailure;
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーション用のテクスチャイメージを追加します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[out] usedFes 使用するフレーム拡張子の配列を格納します。
//! @param[out] texImgIdxs usedFes に対応するテクスチャイメージのモデル内のインデックス配列を格納します。
//! @param[in] filePre フレーム拡張子より前のファイルパスです（フォルダのパスも含みます）。
//! @param[in] fileExt フレーム拡張子より後のファイルパスです。
//! @param[in] feList フレーム拡張子リストです。
//! @param[in] sampler サンプラです。
//! @param[in] curSwizzle 現在の初期スウィズル値です。
//! @param[in] isSingleEnvCube 環境立方体の +X 面にキューブマップの DDS ファイルが接続されていれば true です。
//! @param[in] orgMatName Maya のマテリアル名です（エラー表示用）。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus AppendTexPatAnimTexImgs(
    YModel& ymodel,
    MIntArray& usedFes,
    MIntArray& texImgIdxs,
    const std::string& filePre,
    const std::string& fileExt,
    const MIntArray& feList,
    const RSampler& sampler,
    const int curSwizzle,
    const bool isSingleEnvCube,
    const std::string& orgMatName
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    const int texCount = feList.length();
    for (int iTex = 0; iTex < texCount; ++iTex)
    {
        const int fe = feList[iTex];
        std::string filePath = GetFilePathForTexPatAnim(filePre, fileExt, fe);
        if (filePath.length() == 0)
        {
            if (!yopt.m_CheckElementFlag && yopt.ExportsTexPatAnim())
            {
                const std::string requiredPath = filePre + RGetNumberString(fe) + fileExt;
                YShowWarning(&yscene, // Texture file for texture pattern animation is not found: %s
                    "テクスチャーパターンアニメーションで、フレーム拡張子に対応するテクスチャーファイルが存在しません: {0} \n"
                    "出力されるテクスチャーファイルのうち、フレーム拡張子の値が一番近いものがパターンアニメーションデータとして使用されます。",
                    "For the texture pattern animation, no texture file corresponds to Frame Extension: {0} \n"
                    "Of the texture files to export, the one closest to the Frame Extension value is used as the pattern animation data.",
                    requiredPath);
            }
            continue;
        }
        RStringArray filePaths;
        filePaths.push_back(filePath);

        int iTexImg;
        status = GetTexImage(ymodel, iTexImg, filePaths, sampler, curSwizzle, isSingleEnvCube, orgMatName);
        if (!status)
        {
            return (!yopt.m_CheckElementFlag) ? status : MS::kSuccess;
        }

        usedFes.append(fe);
        texImgIdxs.append(iTexImg);
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief フレーム拡張子に接続されたアニメーションカーブのキーの値を取得します。
//!
//! @param[in,out] curveValues アニメーションカーブのキーの値の配列（重複なし）です。
//-----------------------------------------------------------------------------
static void GetFrameExtensionCurveValues(MIntArray& curveValues, const MObject& curveObj)
{
    MFnAnimCurve curveFn(curveObj);
    const int keyCount = curveFn.numKeys();
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        const int fe = RRound(curveFn.value(iKey));
        if (YFindValueInArray(curveValues, fe) == -1)
        {
            curveValues.append(fe);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フレーム拡張子プラグの入力オブジェクトを取得します。
//!        アニメーションカーブが接続されていればキーの値も取得します。
//!
//! @param[out] directInputFlag オブジェクトが直接接続されていれば true、
//!                             blendWeighted や blendTwoAttr ノード経由で
//!                             接続されていれば false を格納します。
//! @param[out] curveValues アニメーションカーブのキーの値の配列（重複なし）を格納します。
//! @param[in] fePlug フレーム拡張子プラグです。
//!
//! @return フレーム拡張子プラグに接続されたオブジェクトを返します。
//-----------------------------------------------------------------------------
static MObject GetFrameExtensionInput(
    bool& directInputFlag,
    MIntArray& curveValues,
    const MPlug& fePlug
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // get input obj
    directInputFlag = true;
    MPlugArray plugArray;
    fePlug.connectedTo(plugArray, true, false);
    if (plugArray.length() == 0)
    {
        return MObject::kNullObj;
    }

    MObject inputObj = plugArray[0].node();
    MFn::Type type = inputObj.apiType();
    if (type == MFn::kCharacter)
    {
        //-----------------------------------------------------------------------------
        // character
        MPlugArray plugs;
        plugArray[0].connectedTo(plugs, true, false); // character は入力も同じプラグに接続されています。
        if (plugs.length() > 0)
        {
            inputObj = plugs[0].node();
            //cerr << "gfei: char: " << fePlug.name() << ": " << MFnDependencyNode(inputObj).name() << R_ENDL;
        }
    }
    else if (type == MFn::kBlendWeighted ||
             type == MFn::kBlendTwoAttr)
    {
        //-----------------------------------------------------------------------------
        // blend
        directInputFlag = false;
        MPlug inputPlug = MFnDependencyNode(inputObj).findPlug("input");
        MIntArray idxs;
        inputPlug.getExistingArrayAttributeIndices(idxs);
        for (int iElem = 0; iElem < static_cast<int>(idxs.length()); ++iElem)
        {
            MPlug inputNPlug = inputPlug.elementByLogicalIndex(idxs[iElem], &status);
            if (!status)
            {
                break;
            }
            MPlugArray plugs;
            inputNPlug.connectedTo(plugs, true, false);
            if (plugs.length() > 0)
            {
                MObject obj = plugs[0].node();
                //cerr << "gfei: blend: " << fePlug.name() << ": " << MFnDependencyNode(obj).name() << R_ENDL;
                if (obj.hasFn(MFn::kAnimCurve))
                {
                    // 接続されているすべてのカーブについて curveValues を取得します。
                    inputObj = obj;
                    GetFrameExtensionCurveValues(curveValues, inputObj);
                }
                else if (obj.apiType() == MFn::kExpression)
                {
                    return obj;
                }
            }
        }
        return inputObj; // すでに curveValues を取得済みなのでここで return
    }
    else if (!inputObj.hasFn(MFn::kAnimCurve) &&
        type != MFn::kExpression &&
        !IsAnimLayerBlendNode(inputObj))
    {
        MPlugArray plugs;
        plugArray[0].connectedTo(plugs, true, false);
        if (plugs.length() > 0)
        {
            inputObj = plugs[0].node();
        }
    }

    //-----------------------------------------------------------------------------
    // get curve values
    if (inputObj.hasFn(MFn::kAnimCurve))
    {
        GetFrameExtensionCurveValues(curveValues, inputObj);
    }

    return inputObj;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションを取得します。
//!
//! @param[in,out] sampler サンプラです。
//! @param[in,out] ymodel モデルです。
//! @param[in] texNode テクスチャノードです。
//! @param[in] curSwizzle 現在の初期スウィズル値です。
//! @param[in] orgMatName Maya のマテリアル名です（エラー表示用）。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetTexPatAnim(
    RSampler& sampler,
    YModel& ymodel,
    const TexNode& texNode,
    const int curSwizzle,
    const std::string& orgMatName
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    const bool checksErrorWarning = !yopt.m_CheckElementFlag && yopt.ExportsTexPatAnim();

    //-----------------------------------------------------------------------------
    // フレームオフセットが 0 かチェックします。
    MFnDependencyNode texFn(GetMainFileTexture(texNode.m_TexObj));
    const std::string texObjName = texFn.name().asChar();

    MPlug foPlug = texFn.findPlug("frameOffset");
    int frameOfs;
    foPlug.getValue(frameOfs);
    if (frameOfs != 0 || !IsAttributeConstantInt(foPlug, yopt))
    {
        if (checksErrorWarning)
        {
            YShowWarning(&yscene, // Frame offset is not zero: (ignored): %s
                "file ノードのフレームオフセットアトリビュートが 0 以外の値になっているフレームがあります: {0} \n"
                "テクスチャーパターンアニメーションが正しく出力されないので注意してください。",
                "The animation includes frames whose Frame Offset attribute for the file node is not 0: {0} \n"
                "Note that texture pattern animation is not exported correctly in this case.",
                texObjName);
        }
    }

    //-----------------------------------------------------------------------------
    // フレーム拡張子の前後のファイルパスを取得します。
    // 例えば、イメージの名前で指定したファイルパスが "D:/img/abc.1.tga" なら、
    //   filePre は "D:/img/abc."
    //   fileExt は ".tga"
    // となります。
    std::string filePath = texNode.m_FilePaths[0];
    int idot = static_cast<int>(filePath.find_last_of('.'));
    if (idot == static_cast<int>(std::string::npos))
    {
        if (checksErrorWarning)
        {
            YShowError(&yscene, // Texture image name is wrong: %s: %s
                "file ノードのイメージの名前が不正です: {0}: {1} \n"
                "テクスチャーパターンアニメーションで使用するテクスチャーのファイル名は \"name.#.ext\"（# は数値、ext は拡張子）の形式になっている必要があります。",
                "The Image Name of the file node is wrong: {0}: {1} \n"
                "The filename of the texture to be used for the texture pattern animation must be in the name.#.ext format (where # is a numeric value, and ext is a filename extension).",
                texObjName, filePath);
            return MS::kFailure;
        }
        else
        {
            return MS::kSuccess;
        }
    }
    const std::string fileExt = filePath.substr(idot, filePath.length() - idot); // include dot
    std::string filePre = filePath.substr(0, idot);
    idot = static_cast<int>(filePre.find_last_of('.'));
    if (idot == static_cast<int>(std::string::npos))
    {
        if (checksErrorWarning)
        {
            YShowError(&yscene, // Texture image name is wrong: %s: %s
                "file ノードのイメージの名前が不正です: {0}: {1} \n"
                "テクスチャーパターンアニメーションで使用するテクスチャーのファイル名は \"name.#.ext\"（# は数値、ext は拡張子）の形式になっている必要があります。",
                "The Image Name of the file node is wrong: {0}: {1} \n"
                "The filename of the texture to be used for the texture pattern animation must be in the name.#.ext format (where # is a numeric value, and ext is a filename extension).",
                texObjName, filePath);
            return MS::kFailure;
        }
        else
        {
            return MS::kSuccess;
        }
    }
    filePre = filePath.substr(0, idot + 1); // include dot

    //-----------------------------------------------------------------------------
    // フレーム拡張子リストをカスタムアトリビュートから取得します。
    MIntArray feList;
    if (!GetFrameExtensionList(feList, yscene, texFn))
    {
        if (!yopt.m_CheckElementFlag)
        {
            return MS::kFailure;
        }
        else
        {
            return MS::kSuccess;
        }
    }

    //-----------------------------------------------------------------------------
    // フレーム拡張子プラグに接続されたアニメーションカーブかエクスプレッションを取得します。
    MPlug fePlug = texFn.findPlug("frameExtension");
    bool directInputFlag;
    MIntArray curveValues;
    const MObject inputObj = GetFrameExtensionInput(directInputFlag, curveValues, fePlug);
    if (inputObj.isNull())
    {
        if (feList.length() != 0)
        {
            //-----------------------------------------------------------------------------
            // アニメーションが設定されていない場合でも
            // フレーム拡張子リストが設定されていれば
            // テクスチャイメージを追加します。
            SortArray(feList);
            MIntArray usedFes, texImgIdxs;
            status = AppendTexPatAnimTexImgs(ymodel, usedFes, texImgIdxs,
                filePre, fileExt, feList, sampler, curSwizzle, texNode.m_IsSingleEnvCube, orgMatName);
            if (!status)
            {
                return (!yopt.m_CheckElementFlag) ? status : MS::kSuccess;
            }
        }
        return MS::kSuccess;
    }

    const bool curveFlag = inputObj.hasFn(MFn::kAnimCurve);
    const bool expressionFlag = (inputObj.apiType() == MFn::kExpression);
    const bool animLayerFlag = IsAnimLayerBlendNode(inputObj);
    if (!curveFlag && !expressionFlag && !animLayerFlag)
    {
        ChannelInput feChan;
        feChan.GetInput(fePlug);
        if (!feChan.IsEffective())
        {
            return MS::kSuccess;
        }
    }
    if (animLayerFlag && !IsEffectiveAnimLayer(fePlug, inputObj))
    {
        //cerr << "anim layer is muted: " << fePlug.info() << endl;
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // エクスプレッションやアニメーションレイヤが接続されていた場合は、
    // フレーム拡張子リストが設定されているかチェックします。
    if (!curveFlag && feList.length() == 0)
    {
        if (expressionFlag)
        {
            if (checksErrorWarning)
            {
                YShowError(&yscene, // Frame extension list is not set for expression: %s
                    "テクスチャーパターンアニメーションでエクスプレッションを使用していますが、出力するテクスチャーのリストが設定されていません: {0} \n"
                    "エクスプレッションを使用する場合は常にテクスチャーパターンアニメーション設定プラグインで出力するテクスチャーのリストを設定してください。",
                    "Expressions are being used with texture pattern animation, but the list of textures to be output has not been set: {0} \n"
                    "When using expressions, always make sure that you set the list of textures to be output by the Plug-In for Setting Texture Pattern Animations.",
                    texObjName);
                return MS::kFailure;
            }
            else
            {
                return MS::kSuccess;
            }
        }
        else if (animLayerFlag)
        {
            if (checksErrorWarning)
            {
                YShowError(&yscene, // Frame extension list is not set for animation layer: %s
                    "テクスチャーパターンアニメーションでアニメーションレイヤを使用していますが、出力するテクスチャーのリストが設定されていません: {0} \n"
                    "アニメーションレイヤを使用する場合は常にテクスチャーパターンアニメーション設定プラグインで出力するテクスチャーのリストを設定してください。",
                    "Animation layers are being used with texture pattern animation, but the list of textures to be output has not been set: {0} \n"
                    "When using animation layers, always make sure that you set the list of textures to be output by the Plug-In for Setting Texture Pattern Animations.",
                    texObjName);
                return MS::kFailure;
            }
            else
            {
                return MS::kSuccess;
            }
        }
        else
        {
            if (checksErrorWarning)
            {
                YShowError(&yscene,
                    "テクスチャーパターンアニメーションで出力するテクスチャーのリストが設定されていません: {0} \n"
                    "直接キーを設定しない場合は常にテクスチャーパターンアニメーション設定プラグインで出力するテクスチャーのリストを設定してください。",
                    "The list of textures to be output for the texture pattern animation has not been set: {0} \n"
                    "When you do not set keys directly, always make sure that you set the list of textures to be output by the Plug-In for Setting Texture Pattern Animations.",
                    texObjName);
                return MS::kFailure;
            }
            else
            {
                return MS::kSuccess;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // フレーム拡張子リストにアニメーションカーブのキーの値を追加します。
    if (curveFlag)
    {
        const int valCount = curveValues.length();
        for (int ival = 0; ival < valCount; ++ival)
        {
            const int fe = curveValues[ival];
            if (YFindValueInArray(feList, fe) == -1)
            {
                feList.append(fe);
            }
        }
    }
    SortArray(feList);

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーションオブジェクトを作成します。
    TexPatAnim texPatAnim;
    texPatAnim.m_FePlug = fePlug;
    texPatAnim.m_ValidCurveFlag =
        (directInputFlag && inputObj.apiType() == MFn::kAnimCurveTimeToUnitless);
    texPatAnim.m_InputObj = inputObj;
    texPatAnim.m_ObjName = texObjName;

    //-----------------------------------------------------------------------------
    // フレーム拡張子リストに対応したテクスチャイメージを追加します。
    status = AppendTexPatAnimTexImgs(ymodel,
        texPatAnim.m_UsedFes, texPatAnim.m_TexImgIdxs,
        filePre, fileExt, feList, sampler, curSwizzle, texNode.m_IsSingleEnvCube, orgMatName);
    if (!status)
    {
        return (!yopt.m_CheckElementFlag) ? status : MS::kSuccess;
    }

    if (texPatAnim.m_UsedFes.length() == 0)
    {
        // フレーム拡張子リストに対応したファイルが 1 つも存在しない場合はエラーです。
        if (checksErrorWarning)
        {
            std::string requiredPath = filePre + "*" + fileExt;
            YShowError(&yscene, "ファイルを開けません: {0}", "The file cannot be opened: {0}", requiredPath);
            return MS::kFailure;
        }
        else
        {
            return MS::kSuccess;
        }
    }

    //-----------------------------------------------------------------------------
    // append
    sampler.m_TexPatAnimIndex = static_cast<int>(ymodel.m_TexPatAnims.size());
    ymodel.m_TexPatAnims.push_back(texPatAnim);

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT アニメーションを取得します。
//!        引数 texNode に渡したテクスチャノードにテクスチャ SRT アニメーションが
//!        設定されているか確認し、設定されていれば
//!        テクスチャ SRT アニメーションデータ(TexSrtAnim)を作成して ymodel.m_TexSrtAnims 配列に追加します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] texNode テクスチャノードです。
//!
//! @return 追加したアニメーションの ymodel.m_TexSrtAnims 内のインデックスを返します。
//!         テクスチャノードにアニメーションが設定されていなければ -1 を返します。
//-----------------------------------------------------------------------------
static int GetTexSrtAnim(YModel& ymodel, const TexNode& texNode)
{
    //-----------------------------------------------------------------------------
    // テクスチャ SRT アニメーションを作成します。
    TexSrtAnim texSrtAnim(texNode.m_PlaceObj);

    //-----------------------------------------------------------------------------
    // テクスチャ SRT パラメータにアニメーションが設定されているものが
    // あるかどうかを確認します。
    bool animExists = false;
    for (int paramIdx = 0; paramIdx < ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
    {
        texSrtAnim.m_AnimPlugs[paramIdx] = texNode.m_TexSrtPlugs[paramIdx];
        ChannelInput& chan = texSrtAnim.m_AnimChans[paramIdx];
        chan.GetInput(texSrtAnim.m_AnimPlugs[paramIdx]);
        if (chan.IsEffective())
        {
            animExists = true;
        }
        else
        {
            texSrtAnim.m_Anims[paramIdx].m_UseFlag = false;
        }
    }
    // アニメーションが設定されたパラメータがなければ -1 を返します。
    if (!animExists)
    {
        return -1;
    }

    //-----------------------------------------------------------------------------
    // データを ymodel.m_TexSrtAnims に追加し、インデックスを設定します。
    const int retIdx = static_cast<int>(ymodel.m_TexSrtAnims.size());
    ymodel.m_TexSrtAnims.push_back(texSrtAnim);

    return retIdx;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャノードを取得します。
//!        texObj で指定したテクスチャオブジェクトに対応するテクスチャノードが
//!        すでに m_TexNodes 内に存在すればそのインデックスを格納します。
//!        存在しなければ新規テクスチャノードを追加します。
//!
//! @param[in,out] sampler サンプラです。
//! @param[in,out] ymodel モデルです。
//! @param[in] texObj テクスチャオブジェクト（file、envCube）です。
//! @param[in] curSwizzle 現在の初期スウィズル値です。
//! @param[in] orgMatName Maya のマテリアル名です（エラー表示用）。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetTexNode(
    RSampler& sampler,
    YModel& ymodel,
    const MObject& texObj,
    const int curSwizzle,
    const std::string& orgMatName,
    const YExpOpt& yopt
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // 既存のテクスチャノードを検索します。
    sampler.m_TexNodeIndex = -1;
    const int texNodeCount = static_cast<int>(ymodel.m_TexNodes.size());
    for (int iTexNode = 0; iTexNode < texNodeCount; ++iTexNode)
    {
        const TexNode& texNode = ymodel.m_TexNodes[iTexNode];
        if (texNode.m_TexObj == texObj &&
            texNode.m_Hint   == sampler.m_Hint)
        {
            sampler.m_TexNodeIndex = iTexNode;
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャノードを新しく作成します。
    if (sampler.m_TexNodeIndex == -1)
    {
        TexNode texNode(yscene, texObj, sampler.m_Hint, yopt, &status);
        CheckStatus(status);
        sampler.m_TexNodeIndex = static_cast<int>(ymodel.m_TexNodes.size());
        ymodel.m_TexNodes.push_back(texNode);
    }
    TexNode& texNode = ymodel.m_TexNodes[sampler.m_TexNodeIndex];

    //-----------------------------------------------------------------------------
    // テクスチャサンプラの属性を取得します。
    status = GetTexSampler(sampler, ymodel, texNode, curSwizzle, orgMatName);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // テクスチャ配置オブジェクトが存在していればテクスチャ SRT アニメーションを取得します。
    if (!texNode.IsEnvMap() && !texNode.m_PlaceObj.isNull())
    {
        sampler.m_TexSrtAnimIndex = GetTexSrtAnim(ymodel, texNode);
    }

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーションを取得します。
    // 内部でサブテクスチャイメージが追加されます。
    if (texNode.m_TexPatAnimFlag)
    {
        status = GetTexPatAnim(sampler, ymodel, texNode, curSwizzle, orgMatName);
        CheckStatus(status);
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャオブジェクトにサンプラの名前が設定されているなら true を返します。
//!
//! @param[in,out] rscene シーンです。
//! @param[in] texObj Maya のテクスチャオブジェクトです。
//!
//! @return サンプラの名前が設定されているなら true を返します。
//-----------------------------------------------------------------------------
static bool TexObjHasSamplerName(RScene& rscene, const MObject& texObj)
{
    std::string nameOverride;
    std::string hintOverride;
    GetSamplerAttribute(rscene, nameOverride, hintOverride, texObj, false);
    return !nameOverride.empty();
}

//-----------------------------------------------------------------------------
//! @brief file ノードのテクスチャファイルが特定のヒント情報に使用されていれば true を返します。
//!
//! @param[in] ymodel モデルです。
//! @param[in] mat マテリアルです。
//! @param[in] fileObj Maya の file ノードです。
//! @param[in] hint テクスチャの関連付けに利用するヒント情報です。
//!
//! @return file ノードのテクスチャファイルが特定のヒント情報に使用されていれば true を返します。
//-----------------------------------------------------------------------------
static bool IsUsedTexuteForHint(
    const YModel& ymodel,
    const YMaterial& mat,
    const MObject& fileObj,
    const RSampler::Hint hint
)
{
    const YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    if (fileObj.apiType() == MFn::kFileTexture)
    {
        const std::string filePath = GetFileTextureName(fileObj, yopt.m_ProjectPath);
        if (!filePath.empty())
        {
            for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
            {
                const RSampler& sampler = mat.m_Samplers[iSampler];
                if (sampler.m_Hint == hint)
                {
                    const RImage& texImg = ymodel.m_TexImgs[sampler.m_ImageIndex];
                    if (RIsSameStringNoCase(texImg.GetMainFilePath(), filePath))
                    {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのテクスチャを取得します。
//!        引数 texObj で指定されたテクスチャオブジェクトを使用するようにマテリアルを設定します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in,out] mat マテリアルです。
//! @param[in,out] curSwizzle 現在の初期スウィズル値です。
//! @param[in] texObj テクスチャオブジェクトです。
//! @param[in] hint テクスチャの関連付けに利用するヒント情報です。
//!                 EXTRA の場合、サンプラの名前が設定されているテクスチャのみ取得します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetTextureForMaterial(
    YModel& ymodel,
    YMaterial& mat,
    int& curSwizzle,
    const MObject& texObj,
    const RSampler::Hint hint
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    if (texObj.apiType() == MFn::kLayeredTexture)
    {
        // テクスチャオブジェクトが layeredTexture ノードなら
        // 各レイヤーに接続された file、envCube ノードからテクスチャを取得します。
        MObjectArray multiTexObjs;
        const bool getsEnvCube = (hint == RSampler::REFLECTION);
        const int multiTexCount = GetLayeredTextureInputs(multiTexObjs, texObj, getsEnvCube);
        int hintIdx = 0;
        for (int iMultiTex = 0; iMultiTex < multiTexCount; ++iMultiTex)
        {
            const MObject& multiTexObj = multiTexObjs[iMultiTex];
            if (hint == RSampler::OPACITY && IsUsedTexuteForHint(ymodel, mat, multiTexObj, RSampler::ALBEDO))
            {
                continue;
            }

            if (hint != RSampler::EXTRA ||
                TexObjHasSamplerName(yscene, GetMainFileTexture(multiTexObj)))
            {
                RSampler sampler(hint, hintIdx);
                status = GetTexNode(sampler, ymodel, multiTexObj, curSwizzle, mat.m_OrgName, yopt);
                CheckStatus(status);
                mat.m_Samplers.push_back(sampler);
                ++hintIdx;
                ++curSwizzle;
            }
        }
    }
    else
    {
        // テクスチャオブジェクトが file、envCube ノードなら
        // 単一のテクスチャを取得します。
        if (hint == RSampler::OPACITY && IsUsedTexuteForHint(ymodel, mat, texObj, RSampler::ALBEDO))
        {
            return MS::kSuccess;
        }

        if (hint != RSampler::EXTRA ||
            TexObjHasSamplerName(yscene, GetMainFileTexture(texObj)))
        {
            RSampler sampler(hint, 0);
            status = GetTexNode(sampler, ymodel, texObj, curSwizzle, mat.m_OrgName, yopt);
            CheckStatus(status);
            mat.m_Samplers.push_back(sampler);
            ++curSwizzle;
        }
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 標準のマテリアルを追加します。
//! 標準マテリアルを作成し ymodel.m_YMaterials 配列に追加します。
//! 既に ymodel.m_YMaterials 内に標準マテリアルが作成されていたら、
//! そのマテリアルの情報を返します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[out] retIndex        標準マテリアルのインデックスを受け取る整数
//! @param[out] renderPriority  描画優先度を受け取る線数
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus AppendDefaultMaterial(YModel& ymodel, int& retIndex, int& renderPriority)
{
    //-----------------------------------------------------------------------------
    // 既存の標準マテリアルを検索します。
    const int matCount = static_cast<int>(ymodel.m_YMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const YMaterial& mat = ymodel.m_YMaterials[iMat];
        // 標準マテリアルが見つかれば、そのマテリアルからデータをコピーして返します。
        if (mat.m_DefaultFlag)
        {
            retIndex = iMat;
            renderPriority = mat.m_Priority;
            return MS::kSuccess;
        }
    }

    //-----------------------------------------------------------------------------
    // 標準のマテリアルを追加します。
    YMaterial mat;

    // 標準マテリアルであることを表すフラグを立てる
    mat.m_DefaultFlag = true;
    // マテリアルのインデックスを設定します。
    mat.m_OrgIndex = static_cast<int>(ymodel.m_YMaterials.size());
    mat.m_Index = mat.m_OrgIndex;
    // マテリアルの名前を設定します。
    mat.m_OrgName = "DummyMaterial";
    mat.m_Name = mat.m_OrgName;

    // ymodel.m_YMaterials に標準マテリアルのデータを追加します。
    ymodel.m_YMaterials.push_back(mat);

    // 追加された標準マテリアルのインデックスと描画優先度を返します。
    retIndex = mat.m_OrgIndex;
    renderPriority = mat.m_Priority;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 配列から UV セットのファミリー名を検索して存在すればインデックスを返し、
//!        存在しなければ名前を追加（設定）してそのインデックスを返します。
//!
//! @param[out] hintFamilyNames ヒント情報インデックスに対する UV セットのファミリー名配列です。
//! @param[out] hintUvSetNames  ヒント情報インデックスに対する UV セット名配列です。
//! @param[in] uvSetFamilyName  検索する UV セットのファミリー名です。
//! @param[in] uvSetName        uvSetFamilyName に対応する UV セット名です。
//!
//! @return hintFamilyNames 内のインデックスを返します。
//-----------------------------------------------------------------------------
static int FindAppendUvSetNames(
    MStringArray& hintFamilyNames,
    MStringArray& hintUvSetNames,
    const MString& uvSetFamilyName,
    const MString& uvSetName
)
{
    int iEmpty = -1;
    for (unsigned int iHint = 0; iHint < hintFamilyNames.length(); ++iHint)
    {
        if (hintFamilyNames[iHint] == uvSetFamilyName)
        {
            return iHint;
        }
        if (iEmpty == -1 && hintFamilyNames[iHint] == "")
        {
            iEmpty = iHint;
        }
    }

    if (iEmpty != -1)
    {
        hintFamilyNames[iEmpty] = uvSetFamilyName;
        hintUvSetNames[iEmpty] = uvSetName;
        return iEmpty;
    }
    else
    {
        hintFamilyNames.append(uvSetFamilyName);
        hintUvSetNames.append(uvSetName);
        return hintFamilyNames.length() - 1;
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアルが使用する UV セット一覧を取得します。
//!
//! @param[out] meshUvSetNames インスタンスを考慮した UV セット名配列を格納します。
//! @param[out] uvSetFamilyNames meshUvSetNames に対応する UV セットのファミリー名配列を格納します。
//! @param[out] uvAttribNames    meshUvSetNames に対応する UV セットの頂点属性名配列を格納します。
//! @param[out] uvHintIdxs       meshUvSetNames に対応する UV セットのヒント情報インデックス配列を格納します。
//! @param[out] samplerUvSetIdxs 各サンプラが使用する UV セットの meshUvSetNames 内のインデックス配列を格納します。
//!                              サンプラが UV セットを使用しない場合は -1 を格納します。
//! @param[in,out] ymodel モデルです。
//! @param[in] mat マテリアルです。
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] defaultUvSet UV セットが見つからなかった場合に設定される名前です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetUvSetsForMaterial(
    MStringArray& meshUvSetNames,
    MStringArray& uvSetFamilyNames,
    RStringArray& uvAttribNames,
    RIntArray& uvHintIdxs,
    RIntArray& samplerUvSetIdxs,
    YModel& ymodel,
    const YMaterial& mat,
    const MDagPath& meshPath,
    const MString& defaultUvSet
)
{
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // mesh ノードの全 UV セット名を取得します。
    MStringArray allUvSetFamilyNames;
    MFnMesh(meshPath).getUVSetFamilyNames(allUvSetFamilyNames);

    //-----------------------------------------------------------------------------
    // nw4f_fixN（N は整数）から始まる名前の UV セットを取得します。
    MStringArray hintFamilyNames; // ヒント情報インデックスに対する UV セットのファミリー名配列です。
    MStringArray hintUvSetNames;  // ヒント情報インデックスに対する UV セット名配列です。
        // ↑これらは、使用されていないヒント情報インデックスでは空文字です。
    const MString FIX_UV_SET_PREFIX = "nw4f_fix";
    for (unsigned int iUvSet = 0; iUvSet < allUvSetFamilyNames.length(); ++iUvSet)
    {
        const MString& uvSetFamilyName = allUvSetFamilyNames[iUvSet];
        if (uvSetFamilyName.indexW(FIX_UV_SET_PREFIX) == 0)
        {
            const MString uvSetName = GetInstanceUvSetName(meshPath, uvSetFamilyName);
            if (uvSetName == "")
            {
                continue;
            }
            MString idxStr = uvSetFamilyName.substring(FIX_UV_SET_PREFIX.length(), uvSetFamilyName.length() - 1);
            if (idxStr != "")
            {
                const int iUnder = idxStr.indexW("_");
                if (iUnder == 0 && idxStr.length() >= 2)
                {
                    // nw4f_fix_S の形式の場合、ヒント情報インデックスは固定しないのでスキップします。
                    continue;
                }
                else if (iUnder != -1)
                {
                    idxStr = idxStr.substring(0, iUnder - 1);
                }
            }
            const int iHint = (idxStr.isInt()) ? idxStr.asInt() : -1;
            if (iHint < 0 || iHint > RVtxAttrib::HINT_INDEX_MAX)
            {
                YShowWarning(&yscene, // UV set name is wrong: %s: %s
                    "「nw4f_fixN」から始まる名前の UV セットで、N の部分の値が 0 以上 254 以下になっていないため、ヒント情報を直接割り当てられません: {0}: {1} \n"
                    "または UV セット名が「nw4f_fix」、「nw4f_fix_」であり、頂点属性名が指定されていません。",
                    "You cannot directly assign hint information, because the value of N is not in the range from 0 to 254 for a UV set name that starts with nw4f_fixN: {0}: {1} \n"
                    "Alternatively, the UV set name is in the form nw4f_fix or nw4f_fix_, and a vertex attribute name is not specified.",
                    meshPath.partialPathName().asChar(), uvSetFamilyName.asChar());
                continue;
            }
            const unsigned int curLen = iHint + 1;
            if (hintFamilyNames.length() < curLen)
            {
                hintFamilyNames.setLength(curLen);
                hintUvSetNames.setLength(curLen);
            }
            if (hintFamilyNames[iHint] != "")
            {
                YShowError(&yscene, // UV set for same hint exists: %s: %s and %s
                    "同じヒント情報に対応する UV セットが複数存在します: {0}: {1} と {2} \n"
                    "たとえば 1 つのオブジェクトに「nw4f_fix3」と「nw4f_fix03」という名前の UV セットが存在するような状態です。"
                    "不要な UV セットを削除するか名前を変更してください。",
                    "More than one UV set corresponds to the same hint information: {0}: {1} and {2} \n"
                    "This error occurs, for example, if the UV sets named nw4f_fix3 and nw4f_fix03 both exist for the same object. "
                    "Either delete or change the names of unnecessary UV sets.",
                    meshPath.partialPathName().asChar(), hintFamilyNames[iHint].asChar(), uvSetFamilyName.asChar());
                return MS::kFailure;
            }
            hintFamilyNames[iHint] = uvSetFamilyName;
            hintUvSetNames[iHint] = uvSetName;
        }
    }

    //-----------------------------------------------------------------------------
    // サンプラで使用している UV セット一覧を取得します。
    RIntArray samplerHintIdxs;
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        const RSampler& sampler = mat.m_Samplers[iSampler];
        const TexNode& texNode = ymodel.m_TexNodes[sampler.m_TexNodeIndex];
        if (texNode.IsEnvMap()) // 環境マップはテクスチャ座標を使用しません。
        {
            samplerHintIdxs.push_back(-1);
            continue;
        }
        MString uvSetFamilyName;
        const MString uvSetName = GetUvSetNameForTexNode(uvSetFamilyName,
            meshPath, ymodel, sampler.m_TexNodeIndex, defaultUvSet);
        const int iHint = FindAppendUvSetNames(hintFamilyNames, hintUvSetNames,
            uvSetFamilyName, uvSetName);
        samplerHintIdxs.push_back(iHint);
    }

    //-----------------------------------------------------------------------------
    // nw4f_uv または nw4f_fix から始まる名前の UV セットで、サンプラで使用されていないものを追加します。
    // UV セット名のアルファベット順に追加します。
    const MString EXTRA_UV_SET_PREFIX = "nw4f_uv";
    MStringArray extraUvSetFamilyNames;
    for (unsigned int iUvSet = 0; iUvSet < allUvSetFamilyNames.length(); ++iUvSet)
    {
        const MString& uvSetFamilyName = allUvSetFamilyNames[iUvSet];
        if (uvSetFamilyName.indexW(EXTRA_UV_SET_PREFIX) == 0 ||
            uvSetFamilyName.indexW(FIX_UV_SET_PREFIX  ) == 0)
        {
            if (YFindValueInArray(hintFamilyNames, uvSetFamilyName) == -1)
            {
                extraUvSetFamilyNames.append(uvSetFamilyName);
            }
        }
    }

    if (extraUvSetFamilyNames.length() != 0)
    {
        SortArray(extraUvSetFamilyNames);
        for (unsigned int iUvSet = 0; iUvSet < extraUvSetFamilyNames.length(); ++iUvSet)
        {
            const MString& uvSetFamilyName = extraUvSetFamilyNames[iUvSet];
            const MString uvSetName = GetInstanceUvSetName(meshPath, uvSetFamilyName);
            if (uvSetName != "")
            {
                FindAppendUvSetNames(hintFamilyNames, hintUvSetNames,
                    uvSetFamilyName, uvSetName);
                //cerr << "extra uv set: " << uvSetFamilyName << ": " << uvSetName <<endl;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 空文字でない UV セット名を前から順に抽出します。
    MIntArray hintIdx2UvSetIdx(hintFamilyNames.length());
    for (unsigned int iHint = 0; iHint < hintFamilyNames.length(); ++iHint)
    {
        const MString& uvSetFamilyName = hintFamilyNames[iHint];
        if (uvSetFamilyName != "")
        {
            hintIdx2UvSetIdx[iHint] = uvSetFamilyNames.length();
            uvSetFamilyNames.append(uvSetFamilyName);
            meshUvSetNames.append(hintUvSetNames[iHint]);
            uvHintIdxs.push_back(iHint);

            //-----------------------------------------------------------------------------
            // UV セット名が nw4f_fix_S または nw4f_fixN_S の形式の場合、
            // S の部分を頂点属性の名前として取得します。
            std::string uvAttribName;
            if (uvSetFamilyName.indexW(FIX_UV_SET_PREFIX) == 0)
            {
                const MString idxStr = uvSetFamilyName.substring(FIX_UV_SET_PREFIX.length(), uvSetFamilyName.length() - 1);
                if (idxStr != "")
                {
                    const int iUnder = idxStr.indexW("_");
                    if (iUnder != -1 && iUnder + 1 < static_cast<int>(idxStr.length()))
                    {
                        const MString nameStr = idxStr.substring(iUnder + 1, idxStr.length() - 1);
                        uvAttribName = GetOutElementName(nameStr.asChar(), false);
                    }
                }
            }
            uvAttribNames.push_back(uvAttribName);
        }
    }

    //-----------------------------------------------------------------------------
    // 各サンプラが使用する UV セットのインデックス配列を変換します。
    for (int iSampler = 0; iSampler < static_cast<int>(samplerHintIdxs.size()); ++iSampler)
    {
        const int iHint = samplerHintIdxs[iSampler];
        const int iUvSet = (iHint != -1) ? hintIdx2UvSetIdx[iHint] : -1;
        samplerUvSetIdxs.push_back(iUvSet);
    }

    //-----------------------------------------------------------------------------
    // 出力する UV セット数をチェックします。
    // 制限はプラグインの都合なので、要望があれば増やします。
    if (uvSetFamilyNames.length() > RPrimVtx::VTX_TEX_MAX)
    {
        YShowError(&yscene, // The number of UV is over %d: %s
            "UV セット数が {0} を超えています: {1}",
            "The number of UV sets is over {0}: {1}",
            RGetNumberString(RPrimVtx::VTX_TEX_MAX), meshPath.partialPathName().asChar());
        return MS::kFailure;
    }

    //cerr << "meshUvSetNames  : " << meshUvSetNames << endl;
    //cerr << "uvSetFamilyNames: " << uvSetFamilyNames << endl;
    //RPrintArray(cerr, uvHintIdxs, "uvHintIdxs");
    //RPrintArray(cerr, samplerUvSetIdxs, "samplerUvSetIdxs");

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief サンプラと UV セットのリンクをチェックします。
//!
//! @param[in,out] yscene シーンです。
//! @param[in] mat マテリアルです。
//! @param[in] uvSetFamilyNames UV セットのファミリー名一覧です（現在は未使用）。
//! @param[in] uvHintIdxs uvSetFamilyNames に対応する UV セットのヒント情報インデックス配列です。
//! @param[in] samplerUvSetIdxs サンプラに対応する UV セットのインデックス配列です。
//!                             サンプラが UV セットを使用しない場合は -1 です。
//! @param[in] meshPath mesh ノードの DAG パスです（エラー表示用）。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CheckUvLinkingForMaterial(
    YScene& yscene,
    const YMaterial& mat,
    const MStringArray& uvSetFamilyNames,
    const RIntArray& uvHintIdxs,
    const RIntArray& samplerUvSetIdxs,
    const MDagPath& meshPath
)
{
    R_UNUSED_VARIABLE(uvSetFamilyNames); // 現状 uvSetName は異なっていても可

    bool matches = RIsSameArray(mat.m_SamplerUvSetIdxs, samplerUvSetIdxs);
    if (matches)
    {
        for (int iSampler = 0; iSampler < static_cast<int>(mat.m_SamplerUvSetIdxs.size()); ++iSampler)
        {
            const int iUvSet = mat.m_SamplerUvSetIdxs[iSampler];
            if (iUvSet != -1               &&
                iUvSet < uvHintIdxs.size() &&
                mat.m_UvHintIdxs[iUvSet] != uvHintIdxs[iUvSet])
            {
                matches = false;
                break;
            }
        }
    }

    if (!matches)
    {
        YShowError(&yscene, // UV linking is not identical: %s: %s and %s // ヘルプに誘導
            "テクスチャーと UV セットの組合せがオブジェクトによって異なります: {0}: {1} と {2} \n"
            "（詳細はプラグインのヘルプを見てください）",
            "The combination of textures and UV sets differs depending on the object: {0}: {1} and {2} \n"
            "(See the Plug-In Help for details.)",
            mat.m_OrgName, mat.m_FirstMeshPath.partialPathName().asChar(),
            meshPath.partialPathName().asChar());
        return MS::kFailure;
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief マテリアル情報のインデックスを取得します。
//!        引数 sgObj に指定されたシェーディンググループに関するマテリアル情報が
//!        ymodel.m_MatInfos 配列の何番目に存在しているのかを検索します。
//!
//! @param[in] ymodel モデルです。
//! @param[in] sgObj シェーディンググループのオブジェクトです。
//!
//! @return ymodel.m_MatInfos 配列内のインデックス を返します。
//!         配列内にマテリアル情報が見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
static int FindMatInfo(const YModel& ymodel, const MObject& sgObj)
{
    const int size = static_cast<int>(ymodel.m_MatInfos.size());
    for (int index = 0; index < size; ++index)
    {
        if (ymodel.m_MatInfos[index].m_SGObj == sgObj)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief ノード 1 つからマテリアル情報を取得します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] ynode ノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetMatInfoFromOneYNode(YModel& ymodel, const YNode& ynode)
{
    //-----------------------------------------------------------------------------
    // mesh ノードでなければ何も取得しません。
    if (ynode.m_OrgKind != YNode::KindMesh)
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // シェーディンググループについてループします。
    const int sgCount = static_cast<int>(ynode.m_SGObjs.size());
    for (int iSg = 0; iSg < sgCount; ++iSg)
    {
        //-----------------------------------------------------------------------------
        // 既存のマテリアル情報が同じシェーディンググループならそのマテリアル情報を更新、
        // 新しいシェーディンググループならマテリアル情報を追加します。
        const MObject sgObj = ynode.m_SGObjs[iSg];
        MObject comp = ynode.m_CompObjs[iSg]; // const にすると MItMeshPolygon で使用できません。
        int matInfoIdx = FindMatInfo(ymodel, sgObj);
        if (matInfoIdx == -1)
        {
            matInfoIdx = static_cast<int>(ymodel.m_MatInfos.size());
            ymodel.m_MatInfos.push_back(YMatInfo(sgObj));
        }
        YMatInfo& matInfo = ymodel.m_MatInfos[matInfoIdx];

        //-----------------------------------------------------------------------------
        // マテリアルを割り当てたフェース群に頂点カラー属性が設定されていれば情報を取得します。
        for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
        {
            const MString& colSet = ynode.m_MeshColSets[iCol];
            if (colSet.length() != 0 &&
                ComponentVertexColorExists(ynode.m_ShapePath, comp, colSet))
            {
                YVtxAttrInfo& colInfo = matInfo.m_MatVtxColInfos[iCol];
                const int compCount = GetColorSetComponentCount(nullptr, ynode.m_ShapePath, colSet);
                if (!colInfo.m_ExistFlag)
                {
                    colInfo.m_ExistFlag = true;
                    colInfo.m_CompCount = compCount;
                }
                else if (compCount > colInfo.m_CompCount)
                {
                    colInfo.m_CompCount = compCount;
                }
            }
        }

        //-----------------------------------------------------------------------------
        // マテリアルを割り当てたフェース群にユーザー頂点属性が設定されていれば情報を取得します。
        for (int iUa = 0; iUa < RPrimVtx::VTX_USR_MAX; ++iUa)
        {
            const MString& colSet = ynode.m_MeshUsrColSets[iUa];
            if (colSet.length() != 0 &&
                ComponentVertexColorExists(ynode.m_ShapePath, comp, colSet))
            {
                YVtxAttrInfo& usrInfo = matInfo.m_MatVtxUsrInfos[iUa];
                const int compCount = GetColorSetComponentCount(nullptr, ynode.m_ShapePath, colSet);
                if (!usrInfo.m_ExistFlag)
                {
                    usrInfo.m_ExistFlag = true;
                    usrInfo.m_CompCount = compCount;
                }
                else if (compCount > usrInfo.m_CompCount)
                {
                    usrInfo.m_CompCount = compCount;
                }
                //else if (usrInfo.m_CompCount != compCount)
                //{
                //  YShowError(&yscene, "", "Component type of user attribute is not identical: {0} ({1})", // 現在は発生しない
                //      ynode.m_OrgName, colSet.asChar());
                //  return MS::kFailure;
                //}
            }
        }

        //cerr << "mi: " << MFnDependencyNode(sgObj).name() << ": " << matInfo.m_MatVtxColInfos[0] << endl;
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief マテリアル情報を取得します。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetMatInfos(YModel& ymodel)
{
    MStatus status;
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        status = GetMatInfoFromOneYNode(ymodel, *ymodel.m_pYNodes[iNode]);
        CheckStatus(status);
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのカラーをプラグから取得します。
//!
//! @return マテリアルのカラーを返します。
//-----------------------------------------------------------------------------
static RVec3 GetMatColorFromPlug(const MPlug& plug, const float scale)
{
    MColor mc;
    plug.child(0).getValue(mc.r);
    plug.child(1).getValue(mc.g);
    plug.child(2).getValue(mc.b);
    return GetRVec3Color(mc * scale, false, Y_CLAMPS_COLOR);
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのカラーとテクスチャを取得します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in,out] mat マテリアルです。
//! @param[in,out] curSwizzle 現在の初期スウィズル値です。
//! @param[in] dstPlug マテリアルのプラグです。
//! @param[in] hint テクスチャの関連付けに利用するヒント情報です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetMatColorAndTexture(
    YModel& ymodel,
    YMaterial& mat,
    int& curSwizzle,
    const MPlug dstPlug,
    const RSampler::Hint hint
)
{
    //-----------------------------------------------------------------------------
    // マテリアルのプラグに接続されたテクスチャオブジェクトを取得します。
    // テクスチャが接続されていれば file ノードの colorGain をカラーのプラグとします。
    // 不透明カラーについては file ノードの alphaGain をカラーのプラグとします。
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    MPlug colorPlug = dstPlug;
    MObject texObj = FindTextureNode(yscene, dstPlug, yopt.m_CheckElementFlag);
    if (!texObj.isNull())
    {
        MObject mainFileObj = GetMainFileTexture(texObj);
        if (hint == RSampler::OPACITY)
        {
            colorPlug = MFnDependencyNode(mainFileObj).findPlug("alphaGain");
        }
        else
        {
            colorPlug = MFnDependencyNode(mainFileObj).findPlug("colorGain");
        }
    }

    //-----------------------------------------------------------------------------
    // 現在のカラーとカラーのプラグを取得します。
    RVec3* pMatCol = NULL;
    float matColScale = 1.0f;
    int iAnimPlug = -1;
    if (hint == RSampler::ALBEDO)
    {
        pMatCol = &mat.m_Diffuse;
        matColScale = mat.m_DiffuseCoeff;
        iAnimPlug = RMaterial::DIFFUSE_R;
        mat.m_ColorTexObj = texObj; // color と transparency のテクスチャの同一判定用に保持します。
    }
    else if (hint == RSampler::OPACITY)
    {
        mat.m_UseTransparencyPlug = texObj.isNull();
        if (mat.m_UseTransparencyPlug)
        {
            pMatCol = &mat.m_Opacity;
            iAnimPlug = RMaterial::OPACITY_R;
        }
        else
        {
            float floatAlpha;
            colorPlug.getValue(floatAlpha);
            #ifndef USE_HIGH_COLOR_SW
            floatAlpha = RClampValue(0.0f, 1.0f, floatAlpha);
            #endif
            mat.m_Opacity = RVec3(floatAlpha, floatAlpha, floatAlpha);
            mat.m_AnimPlugs[RMaterial::OPACITY_R] = colorPlug;
            mat.m_AnimPlugs[RMaterial::OPACITY_G] = colorPlug;
            mat.m_AnimPlugs[RMaterial::OPACITY_B] = colorPlug;
        }
        mat.m_TransparencyTexObj = texObj; // レンダーステート自動設定用に保持します。
    }
    else if (hint == RSampler::EMISSION)
    {
        pMatCol = &mat.m_Emission;
        iAnimPlug = RMaterial::EMISSION_R;
    }
    else if (hint == RSampler::SPECULAR)
    {
        pMatCol = &mat.m_Specular;
        iAnimPlug = RMaterial::SPECULAR_R;
    }

    if (pMatCol != NULL)
    {
        *pMatCol = GetMatColorFromPlug(colorPlug, matColScale);
        if (hint == RSampler::OPACITY)
        {
            *pMatCol = RVec3::kOne - *pMatCol;
        }
    }

    if (iAnimPlug != -1)
    {
        mat.m_AnimPlugs[iAnimPlug + 0] = colorPlug.child(0);
        mat.m_AnimPlugs[iAnimPlug + 1] = colorPlug.child(1);
        mat.m_AnimPlugs[iAnimPlug + 2] = colorPlug.child(2);
    }

    //-----------------------------------------------------------------------------
    // テクスチャを取得します。
    if (!texObj.isNull())
    {
        if (hint == RSampler::OPACITY && texObj == mat.m_ColorTexObj)
        {
            // color と transparency に同じテクスチャが接続されている場合は
            // transparency のテクスチャの <sampler> は出力しません。
        }
        else
        {
            MStatus status = GetTextureForMaterial(ymodel, mat, curSwizzle, texObj, hint);
            CheckStatus(status);
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief サンプラの名前とヒント情報の文字列に重複がないかチェックします。
//!
//! @param[in,out] rscene シーンです。
//! @param[in] mat マテリアルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CheckSamplerNameAndHintIsDuplicate(RScene& rscene, const YMaterial& mat)
{
    const int samplerCount = static_cast<int>(mat.m_Samplers.size());
    for (int iSampler = 1; iSampler < samplerCount; ++iSampler)
    {
        const RSampler& sampler = mat.m_Samplers[iSampler];
        const std::string name = sampler.GetName();
        const std::string hint = sampler.GetHintString();
        for (int iOther = 0; iOther < iSampler; ++iOther)
        {
            const RSampler& other = mat.m_Samplers[iOther];
            if (other.GetName() == name)
            {
                YShowError(&rscene, // Sampler name is duplicate: %s: %s
                    "同じマテリアルでサンプラの名前が重複しています: {0}: {1} \n"
                    "サンプラアトリビュート設定プラグインで重複しない名前を設定してください。",
                    "The same sampler name is used more than once by the same material: {0}: {1} \n"
                    "Use the Plug-In for Setting Sampler Attributes to set names that are not duplicates.",
                    mat.m_OrgName, name);
                return MS::kFailure;
            }
            if (other.GetHintString() == hint && !hint.empty())
            {
                YShowError(&rscene, // Sampler hint is duplicate: %s: %s
                    "同じマテリアルでサンプラのヒント情報が重複しています: {0}: {1} \n"
                    "サンプラアトリビュート設定プラグイン で重複しないヒント情報を設定してください。",
                    "The same sampler hint is used more than once by the same material: {0}: {1} \n"
                    "Use the Plug-In for Setting Sampler Attributes to set hints that are not duplicates.",
                    mat.m_OrgName, hint);
                return MS::kFailure;
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief シェーディンググループから Maya 用のマテリアルを取得します。
//!        ymodel.m_YMaterials に同じシェーディンググループのマテリアルが
//!        登録されていればそのマテリアルを返します。
//!        登録されていなければ新規マテリアルを追加して返します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[out] retMatIndex 取得したマテリアルの ymodel.m_YMaterials 内のインデックスを格納します。
//! @param[out] renderPriority 描画プライオリティを格納します。
//! @param[out] meshUvSetNames 出力する UV セットの UV セット名配列を格納します。
//! @param[out] uvAttribNames meshUvSetNames に対応する UV セットの頂点属性名配列を格納します。
//! @param[out] uvHintIdxs    meshUvSetNames に対応する UV セットのヒント情報インデックス配列を格納します。
//! @param[in] sgObj shadingEngine ノードです。
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] defaultUvSet 標準の UV セット名です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetMaterial(
    YModel& ymodel,
    int& retMatIndex,
    int& renderPriority,
    MStringArray& meshUvSetNames,
    RStringArray& uvAttribNames,
    RIntArray& uvHintIdxs,
    const MObject& sgObj,
    const MDagPath& meshPath,
    const MString& defaultUvSet
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // 指定されたシェーディンググループが取得済みのマテリアル配列に存在すれば
    // その情報を返します。
    const int matCount = static_cast<int>(ymodel.m_YMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const YMaterial& mat = ymodel.m_YMaterials[iMat];
        if (mat.m_SGObj == sgObj)
        {
            retMatIndex = iMat;
            renderPriority = mat.m_Priority;

            MStringArray uvSetFamilyNames;
            RIntArray samplerUvSetIdxs;
            status = GetUvSetsForMaterial(meshUvSetNames, uvSetFamilyNames,
                uvAttribNames, uvHintIdxs, samplerUvSetIdxs,
                ymodel, mat, meshPath, defaultUvSet);
            CheckStatus(status);

            // サンプラと UV セットのリンクをチェックします。
            status = CheckUvLinkingForMaterial(yscene, mat,
                uvSetFamilyNames, uvHintIdxs, samplerUvSetIdxs, meshPath);
            CheckStatus(status);

            return MS::kSuccess;
        }
    }

    //-----------------------------------------------------------------------------
    // シェーディンググループからシェーダオブジェクトを取得します。
    MFnDependencyNode sgFn(sgObj);
    MObject shaderObj = FindShaderObject(sgObj);
    if (shaderObj.isNull())
    {
        if (!yopt.m_CheckElementFlag)
        {
            YShowWarning(&yscene, // No shader: %s
                "シェーディンググループにマテリアルが接続されていません: {0} \n"
                "プラグインが自動的にデフォルトのマテリアル（DummyMaterial という名前）を中間ファイルに追加して出力します。",
                "No materials are connected to the shading group: {0} \n"
                "The plug-in automatically adds and exports the default material (named DummyMaterial) to the intermediate file.",
                sgFn.name().asChar());
        }
        // シェーダオブジェクトがない場合は標準のマテリアルを追加します。
        return AppendDefaultMaterial(ymodel, retMatIndex, renderPriority);
    }

    //-----------------------------------------------------------------------------
    // シェーダ名とシェーダタイプを取得します。
    MString shaderName = MFnDependencyNode(shaderObj).name();
    MFn::Type shaderType = shaderObj.apiType();

    // 以下のタイプのシェーダはサポートしないので、標準のマテリアルを追加して終了します。
    if (shaderType == MFn::kLayeredShader ||
        shaderType == MFn::kUseBackground ||
        shaderType == MFn::kOceanShader ||
        shaderType == MFn::kRampShader)
    {
        if (!yopt.m_CheckElementFlag)
        {
            YShowWarning(&yscene, // Shader is not supported: %s
                "エクスポートプラグインが対応していないシェーダが使用されています: {0} \n"
                "プラグインが自動的にデフォルトのマテリアル（DummyMaterial という名前）を中間ファイルに追加して出力します。",
                "A shader not supported by the Export plug-in is being used: {0} \n"
                "The plug-in automatically adds and exports the default material (named DummyMaterial) to the intermediate file.",
                shaderName.asChar());
        }
        return AppendDefaultMaterial(ymodel, retMatIndex, renderPriority);
    }

    // ランバートシェーダの機能を持たないシェーダはサポートしないので、
    // 標準マテリアルを追加して終了します。
    MFnLambertShader shaderFn(shaderObj, &status);
    if (!status)
    {
        if (!yopt.m_CheckElementFlag)
        {
            YShowWarning(&yscene, // Shader is not supported: %s
                "エクスポートプラグインが対応していないシェーダが使用されています: {0} \n"
                "プラグインが自動的にデフォルトのマテリアル（DummyMaterial という名前）を中間ファイルに追加して出力します。",
                "A shader not supported by the Export plug-in is being used: {0} \n"
                "The plug-in automatically adds and exports the default material (named DummyMaterial) to the intermediate file.",
                shaderName.asChar());
        }
        return AppendDefaultMaterial(ymodel, retMatIndex, renderPriority);
    }

    //-----------------------------------------------------------------------------
    // マテリアルを作成します。
    //-----------------------------------------------------------------------------
    YMaterial mat;

    //-----------------------------------------------------------------------------
    // Maya のシェーダの情報を記録します。
    mat.m_SGObj         = sgObj;
    mat.m_ShaderObj     = shaderObj;
    mat.m_ShaderType    = shaderType;
    mat.m_FirstMeshPath = meshPath;

    //-----------------------------------------------------------------------------
    // インデックスと名前を取得します。
    mat.m_OrgIndex = static_cast<int>(ymodel.m_YMaterials.size());
    mat.m_Index    = mat.m_OrgIndex;
    mat.m_OrgName  = shaderFn.name().asChar();
    mat.m_Name     = GetOutElementName(mat.m_OrgName, yopt.m_RemoveNamespace);

    //-----------------------------------------------------------------------------
    // マテリアル情報のインデックスを取得します。
    const int matInfoIdx = FindMatInfo(ymodel, sgObj);
    if (matInfoIdx != -1)
    {
        const YMatInfo& matInfo = ymodel.m_MatInfos[matInfoIdx];
        // 頂点カラーとユーザー頂点データに関する情報をマテリアルにコピーします。
        // m_MatVtxColInfos はテクスチャコンバイナの設定を出力する際に使用されます。
        mat.m_MatVtxColInfos = matInfo.m_MatVtxColInfos;
        mat.m_MatVtxUsrInfos = matInfo.m_MatVtxUsrInfos;
    }

    //-----------------------------------------------------------------------------
    // 最適化時の圧縮可能フラグを取得します。
    MPlug noCompressPlug = FindPlugQuiet(sgFn, "nw4fNoCompressMat", &status);
    if (status)
    {
        bool noCompressFlag;
        noCompressPlug.getValue(noCompressFlag);
        mat.m_CompressEnable = !noCompressFlag;
    }

    //-----------------------------------------------------------------------------
    // ポリゴンの表示面を取得します。
    // TODO: Maya 上で設定するならコメント解除。
    //MPlug displayFacePlug = FindPlugQuiet(sgFn, "nw4fDisplayFace", &status);
    //if (status)
    //{
    //  int displayFace;
    //  displayFacePlug.getValue(displayFace);
    //  mat.mRenderState.mDisplayFace =
    //      static_cast<RRenderState::DisplayFace>(displayFace);
    //}

    //-----------------------------------------------------------------------------
    // 法線の使用フラグを設定します。
    // DCC ツールからは常に法線を出力します。
    mat.m_VtxNrmFlag = true;

    //-----------------------------------------------------------------------------
    // ディフューズカラーを color アトリビュートから取得します。
    // color プラグにテクスチャが接続されている場合は
    // file ノードの colorGain アトリビュートの値をディフューズカラーとして出力します。

    //mat.m_DiffuseCoeff = shaderFn.diffuseCoeff();
    mat.m_DiffuseCoeff = 1.0f; // 現在 diffuse アトリビュートは反映しません。

    int curSwizzle = 0;
    status = GetMatColorAndTexture(ymodel, mat, curSwizzle, GetShaderColorPlug(shaderObj), RSampler::ALBEDO);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 不透明カラーを transparency アトリビュートから取得します。
    // transparency プラグにテクスチャが接続されている場合は
    // file ノードの alphaGain アトリビュートの値を不透明カラーとして出力します。
    const MString transparencyPlugName = (shaderType == MFn::kSurfaceShader) ?
        "outTransparency" : "transparency";
    status = GetMatColorAndTexture(ymodel, mat, curSwizzle, shaderFn.findPlug(transparencyPlugName), RSampler::OPACITY);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // アンビエントカラーを ambientColor アトリビュートから取得します。
    mat.m_Ambient = GetRVec3Color(shaderFn.ambientColor(), false, Y_CLAMPS_COLOR);
    MPlug ambientPlug = shaderFn.findPlug("ambientColor");
    mat.m_AnimPlugs[RMaterial::AMBIENT_R] = ambientPlug.child(0);
    mat.m_AnimPlugs[RMaterial::AMBIENT_G] = ambientPlug.child(1);
    mat.m_AnimPlugs[RMaterial::AMBIENT_B] = ambientPlug.child(2);

    //-----------------------------------------------------------------------------
    // エミッションカラーを incandescence アトリビュートから取得します。
    status = GetMatColorAndTexture(ymodel, mat, curSwizzle, shaderFn.findPlug("incandescence"), RSampler::EMISSION);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 法線マップが設定されていれば、そのテクスチャを法線マップとしてマテリアルに登録します。
    MObject nrmMapTexObj = FindNormalMapTextureNode(yscene, shaderObj, yopt.m_CheckElementFlag);
    if (!nrmMapTexObj.isNull())
    {
        const int nrmMapIdxStart = static_cast<int>(mat.m_Samplers.size());
        status = GetTextureForMaterial(ymodel, mat, curSwizzle, nrmMapTexObj, RSampler::NORMAL);
        CheckStatus(status);
        for (int iSampler = nrmMapIdxStart; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
        {
            mat.m_NrmMapIdxs.push_back(iSampler);
        }
    }

    //-----------------------------------------------------------------------------
    // 以下のシェーダタイプの場合はスペキュラと反射に関するアトリビュートを取得します。
    if (shaderType == MFn::kAnisotropy    ||
        shaderType == MFn::kBlinn         ||
        shaderType == MFn::kPhong         ||
        shaderType == MFn::kPhongExplorer)
    {
        //-----------------------------------------------------------------------------
        // スペキュラカラーを specularColor アトリビュートから取得します。
        mat.m_IsSpecularEnable = true; // スペキュラーを有効にします。
        mat.m_SpecularPlugFlag = true; // specularColor プラグを取得します。
        status = GetMatColorAndTexture(ymodel, mat, curSwizzle, shaderFn.findPlug("specularColor"), RSampler::SPECULAR);
        CheckStatus(status);

        //-----------------------------------------------------------------------------
        // reflectedColor アトリビュートに接続された環境マップのテクスチャを取得します。
        MObject reflectTexObj = FindEnvMapTextureNode(yscene, shaderFn.findPlug("reflectedColor"),
            yopt.m_ProjectPath, yopt.m_CheckElementFlag);
        if (!reflectTexObj.isNull())
        {
            status = GetTextureForMaterial(ymodel, mat, curSwizzle, reflectTexObj, RSampler::REFLECTION);
            CheckStatus(status);
        }

        //-----------------------------------------------------------------------------
        // 各シェーダタイプ固有のアトリビュートを取得します。
        MFnReflectShader reflectFn(shaderObj);
        if (shaderType == MFn::kAnisotropy) // 異方性シェーダ
        {
            mat.m_IsAnisotropic = true;
        }
        else if (shaderType == MFn::kBlinn)
        {
            //float eccentricity;
            //reflectFn.findPlug("eccentricity").getValue(eccentricity);
            //float specularRollOff;
            //reflectFn.findPlug("specularRollOff").getValue(specularRollOff);
            //mat.m_Shininess = (eccentricity < 0.03125f) ? 128.0f : 4.0f / eccentricity; // ?
        }
        else if (shaderType == MFn::kPhong)
        {
            // Phong シェーダの場合のみ余弦の累乗を取得します（現在は未使用）。
            reflectFn.findPlug("cosinePower").getValue(mat.m_Shininess);
            mat.m_Shininess *= 4.0f; // devkit/plug-ins/D3DViewportRenderer.cpp より
        }
        else if (shaderType == MFn::kPhongExplorer)
        {
            //float roughness;
            //reflectFn.findPlug("roughness").getValue(roughness);
            //mat.m_Shininess = roughness * ??;
        }
    }

    //-----------------------------------------------------------------------------
    // translucenceDepth アトリビュートに接続されたテクスチャをエクストラマップとして取得します。
    // エクストラマップは、サンプラの名前が設定されている場合のみ出力されます。
    MPlug translucenceDepthPlug = shaderFn.findPlug("translucenceDepth");
    status = GetMatColorAndTexture(ymodel, mat, curSwizzle, translucenceDepthPlug, RSampler::EXTRA);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // カラーアニメーションが設定されているかどうか判定します。
    for (int paramIdx = 0; paramIdx < RMaterial::COLOR_PARAM_COUNT; ++paramIdx)
    {
        mat.m_Anims[paramIdx].m_UseFlag = false;
        if (RMaterial::IsSpecular(paramIdx) && !mat.m_SpecularPlugFlag)
        {
            // スペキュラのないマテリアルの場合、スペキュラプラグの入力は取得しません。
        }
        else
        {
            ChannelInput& chan = mat.m_AnimChans[paramIdx];
            chan.GetInput(mat.m_AnimPlugs[paramIdx]);
            if (chan.IsEffective())
            {
                // アニメーションカーブ、ドリブンキー、エクスプレション、キャラクタなどが
                // プラグに接続されていればアニメーションが設定されているとみなします。
                mat.m_Anims[paramIdx].m_UseFlag = true;
                mat.m_ColAnimFlag = true;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 頂点テクスチャ座標を設定します。
    status = GetUvSetsForMaterial(meshUvSetNames, mat.m_UvSetFamilyNames,
        uvAttribNames, mat.m_UvHintIdxs, mat.m_SamplerUvSetIdxs,
        ymodel, mat, meshPath, defaultUvSet);
    CheckStatus(status);
    uvHintIdxs = mat.m_UvHintIdxs;

    mat.m_VtxTexCount = mat.m_UvSetFamilyNames.length();

    //-----------------------------------------------------------------------------
    // サンプラに UV セットのヒント情報インデックスを設定します。
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        const int iUvSet = mat.m_SamplerUvSetIdxs[iSampler];
        if (iUvSet != -1)
        {
            mat.m_Samplers[iSampler].m_OriginalTexsrt.m_UvHintIdx = mat.m_UvHintIdxs[iUvSet];
        }
    }

    //-----------------------------------------------------------------------------
    // 接線と従法線の出力フラグを設定します。
    for (int iNrmMap = 0; iNrmMap < static_cast<int>(mat.m_NrmMapIdxs.size()); ++iNrmMap)
    {
        const int vtxTanTexIdx = mat.m_SamplerUvSetIdxs[mat.m_NrmMapIdxs[iNrmMap]];
        if (RFindValueInArray(mat.m_VtxTanTexIdxs, vtxTanTexIdx) == -1)
        {
            if (static_cast<int>(mat.m_VtxTanTexIdxs.size()) >= RPrimVtx::VTX_TAN_MAX)
            {
                //break; // エラーにしないで VTX_TAN_MAX 個まで出力する場合は break します。
                YShowError(&yscene, // The number of UV set for tangent is over %d: %s
                    "接線（従法線）に使用する UV セット数が {0} を超えています: {1}",
                    "The number of UV sets for tangent is over {0}: {1}",
                    RGetNumberString(RPrimVtx::VTX_TAN_MAX), meshPath.partialPathName().asChar());
                return MS::kFailure;
            }
            mat.m_VtxTanTexIdxs.push_back(vtxTanTexIdx);
        }
    }

    if (mat.m_VtxTanTexIdxs.empty() && mat.m_IsAnisotropic)
    {
        mat.m_VtxTanTexIdxs.push_back((mat.m_VtxTexCount > 0) ? 0 : -1);
    }

    //-----------------------------------------------------------------------------
    // テクスチャ SRT アニメーションの情報を設定します。
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        const RSampler& sampler = mat.m_Samplers[iSampler];
        if (sampler.m_TexSrtAnimIndex != -1)
        {
            // 一つでも有効なインデックスがあれば
            // テクスチャ SRT アニメーション使用フラグを ON にします。
            mat.m_TexSrtAnimFlag = true;
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーションの情報を設定します。
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        const RSampler& sampler = mat.m_Samplers[iSampler];
        if (sampler.m_TexPatAnimIndex != -1)
        {
            // 一つでも有効なインデックスがあれば
            // テクスチャパターンアニメーション使用フラグを ON にします。
            mat.m_TexPatAnimFlag = true;
        }
    }

    //-----------------------------------------------------------------------------
    // レンダーステートを設定します。
    mat.m_AutoRenderStateFlag = false;
    mat.SetRenderStateMode(RRenderState::OPA);

    // TODO: Maya 上で設定するならコメント解除。
    //MPlug BlendModePlug = FindPlugQuiet(sgFn, "nw4fBlendMode", &status);
    //if (status)
    //{
    //  int blendMode;
    //  BlendModePlug.getValue(blendMode);
    //  --blendMode;
    //  if (0 <= blendMode && blendMode <= RMaterial::XLU)
    //  {
    //      mat.m_AutoRenderStateFlag = false;
    //      mat.SetRenderStateMode(static_cast<RMaterial::BlendMode>(blendMode));
    //  }
    //}

    // 自動設定が ON で透明度が 1.0 未満の場合は半透明に設定します。
    if (mat.m_AutoRenderStateFlag && mat.m_Opacity.x < 1.0f)
    {
        mat.SetRenderStateMode(RRenderState::XLU);
    }

    //-----------------------------------------------------------------------------
    // 描画優先度を取得します。
    // 描画優先度はプラグインによって nw4fRenderPriority アトリビュートに設定されます。
    // また、nw4fUseRenderPriority アトリビュートによって描画優先度を使用するかどうか
    // も設定されます。
    // ここでは nw4fUseRenderPriority に設定された値を確認し、true が設定されていたら
    // nw4fRenderPriority から優先度を取得します。false が設定されていたら優先度は 0 となります。
    mat.m_Priority = 0;
    MPlug useRenderPriorityPlug = FindPlugQuiet(sgFn, "nw4fUseRenderPriority", &status);
    if (status)
    {
        bool useRenderPriority;
        useRenderPriorityPlug.getValue(useRenderPriority);
        if (useRenderPriority)
        {
            mat.m_Priority = 1;
            MPlug renderPriorityPlug = FindPlugQuiet(sgFn, "nw4fRenderPriority", &status);
            if (status)
            {
                renderPriorityPlug.getValue(mat.m_Priority);
                mat.m_Priority = RClampValue(RShape::RENDER_PRIORITY_MIN, RShape::RENDER_PRIORITY_MAX, mat.m_Priority);
            }
        }
    }
    // マテリアルは圧縮されても render priority が違うことがあるのでここで代入
    renderPriority = mat.m_Priority;

    //-----------------------------------------------------------------------------
    // サンプラの名前とヒント情報の文字列に重複がないかチェックします。
    status = CheckSamplerNameAndHintIsDuplicate(yscene, mat);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // マテリアルを追加します。
    ymodel.m_YMaterials.push_back(mat);

    retMatIndex = mat.m_OrgIndex;

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief モデルのマテリアル群のユーザーデータを取得します。
//!
//! @param[in,out] pYModel モデルへのポインターです。
//-----------------------------------------------------------------------------
static void GetMaterialUserData(YModel* pYModel)
{
    YScene& yscene = pYModel->GetScene();
    const int matCount = static_cast<int>(pYModel->m_YMaterials.size());
    for (int matIdx = 0; matIdx < matCount; ++matIdx)
    {
        YMaterial& mat = pYModel->m_YMaterials[matIdx];
        mat.GetUserData(&yscene);
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点行列を追加します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] vtxMtx 頂点行列です。
//!
//! @return 追加された頂点行列の ymodel.m_VtxMtxs 内でのインデックスを返します。
//!         同じ内容の頂点行列が既に ymodel.m_VtxMtxs 内に存在してる場合、
//!         新たに追加せずに同じ内容の頂点行列のインデックスを返します。
//-----------------------------------------------------------------------------
static int AppendVtxMtx(YModel& ymodel, const RVtxMtx& vtxMtx)
{
    //-----------------------------------------------------------------------------
    // vtxMtx に渡されたデータが既に追加されていないか確認します。
    const int vmtCount = static_cast<int>(ymodel.m_VtxMtxs.size());
    for (int iVmt = 0; iVmt < vmtCount; ++iVmt)
    {
        // 既に追加済みのデータであればそのデータの配列内のインデックスを返します。
        if (ymodel.m_VtxMtxs[iVmt] == vtxMtx)
        {
            return iVmt;
        }
    }

    //-----------------------------------------------------------------------------
    // まだ追加されていないデータであれば、ymodel.m_VtxMtxs に追加します。
    ymodel.m_VtxMtxs.push_back(vtxMtx);

    // 追加されたデータのインデックスを返します。
    return static_cast<int>(ymodel.m_VtxMtxs.size()) - 1;
}

//-----------------------------------------------------------------------------
//! @brief 複数のインフルエンスにウェイトが分散している頂点がノードにあれば true を返します。
//!
//! @param[in] shapePath シェイプノードの DAG パスです。
//! @param[in] skinFn スキニングのファンクションセットです。
//!
//! @return 複数のインフルエンスにウェイトが分散している頂点がノードにあれば true を返します。
//-----------------------------------------------------------------------------
//static bool NodeHasMultiWeightedVertex(const MDagPath& shapePath, const MFnSkinCluster& skinFn)
//{
//  MItMeshVertex vIter(shapePath);
//  for ( ; !vIter.isDone(); vIter.next())
//  {
//      MFloatArray weights;
//      unsigned int infCount;
//      if (skinFn.getWeights(shapePath, vIter.vertex(), weights, infCount))
//      {
//          int boneCount = 0;
//          for (unsigned int iInf = 0; iInf < infCount; ++iInf)
//          {
//              const float weight = weights[iInf];
//              if (weight != 0.0f && RRound(weight * RVtxWeight::WEIGHT_MAX) != 0)
//              {
//                  ++boneCount;
//                  if (boneCount >= 2)
//                  {
//                      return true;
//                  }
//              }
//          }
//      }
//  }
//  return false;
//}

//-----------------------------------------------------------------------------
//! @brief スキンから頂点ウェイトと行列インデックスを取得します。
//!        引数 ynode が参照している skinCluster ノードから、同じく YNode
//!        が参照しているメッシュ用の頂点ウェイトと行列インデックスを取得します。
//!        取得した頂点ウェイトと行列インデックスの情報は ymodel.m_VtxMtxs に登録され、
//!        YNode::m_VtxMtxIdxs に ymodel.m_VtxMtxs 内のインデックスを設定します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in,out] ynode ノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetEnvelope(YModel& ymodel, YNode& ynode)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // スキンからインフルエンスオブジェクト（変形に影響を与える DAG ノード）を取得します。
    MFnSkinCluster skinFn(ynode.m_SkinClusterObj);
    MDagPathArray infPaths;
    skinFn.influenceObjects(infPaths, &status); // status は必ず必要

    // インフルエンスオブジェクトに対応する YNode を検索し、
    // その YNode のインデックスを取得します。
    MIntArray infIndexes; // インフルエンスオブジェクトに対応する YNode::m_OrgIndex の値です。
    const int infCount = infPaths.length();
    for (int iInf = 0; iInf < infCount; ++iInf)
    {
        // m_pYNodes 配列に格納されている DAG パス から YNode を検索します。
        const YNode* pinf = FindYNodeByDagPath(ymodel, infPaths[iInf]);
        if (pinf == NULL)
        {
            YShowError(&yscene, // Influence object is not outputted: %s -> %s
                "スキニングに影響するボーンが出力されていません: {0} -> {1} \n"
                "エクスポートオプションの Export Target が Selection の場合、スキニングに影響するボーンも選択してください。\n"
                "スキニングに影響するボーンのテンプレートアトリビュートが ON になっていないか、"
                "ディスプレイレイヤで非表示になっていないか確認してください。",
                "Bones influenced by skinning are not output: {0} -> {1} \n"
                "Select bones influenced by skinning if the export option Export Target is set to Selection. \n"
                "Confirm whether Template attribute is not ON for bones influenced by skinning "
                "or the bones are not hidden in the display layer.",
                infPaths[iInf].partialPathName().asChar(), ynode.m_OrgName);
            return MS::kFailure;
        }
        infIndexes.append(pinf->m_OrgIndex);
    }

    //-----------------------------------------------------------------------------
    // スムーススキニング強制フラグを取得します。
    //if (yopt.m_AdjustsSmoothSkinning)
    //{
    //  ynode.m_ForcesSmoothSkinning = NodeHasMultiWeightedVertex(ynode.m_ShapePath, skinFn);
    //}
    //cerr << "forces smooth skinning: " << ynode.m_ShapePath.partialPathName() << ": " << ynode.m_ForcesSmoothSkinning << endl;

    //-----------------------------------------------------------------------------
    // スキンから頂点ウェイトを取得します。
    MItMeshVertex vIter(ynode.m_ShapePath);
    for ( ; !vIter.isDone(); vIter.next())
    {
        // 頂点 (vIter.vertex()) のウェイト情報をスキンから取得します。
        // MFnSkinCluster::getWeights が返す float 配列は MFnSkinCluster::influenceObjects
        // により取得したインフルエンスオブジェクトと同じ数だけ要素があります。
        // 影響を及ぼすインフルエンスオブジェクトだけウェイト値が設定されそれ、
        // それ以外は 0.0 が設定されています。
        MFloatArray weights;
        unsigned int retInfCount;
        status = skinFn.getWeights(ynode.m_ShapePath, vIter.vertex(),
            weights, retInfCount);
        if (!status)
        {
            YShowError(&yscene, // Cannot get skin weight: %s.vtx[%d] // 通常は発生しない
                "スキニングのウェイト値を取得できません: {0}.vtx[{1}]",
                "Cannot get skin weight: {0}.vtx[{1}]",
                ynode.m_ShapePath.partialPathName().asChar(), RGetNumberString(vIter.index()));
            return MS::kFailure;
        }

        // 頂点ウェイトと行列インデックスの情報を取得します。
        RVtxMtx vmt;
        if (!yopt.m_ForceFullWeight)
        {
            float weightSum = 0.0f; // ウェイト値の合計
            for (int iInf = 0; iInf < infCount; ++iInf)
            {
                // 浮動小数点数のウェイト値を 0.0 ～ 1.0 の範囲から
                // 整数の 0 ～ 100 の範囲に変換します。
                // この時に四捨五入を行い変換誤差を補正します。
                weightSum += weights[iInf];
                int weight = RRound(weights[iInf] * RVtxWeight::WEIGHT_MAX);
                // ウェイト値を 0 ～ RVtxWeight::WEIGHT_MAX の範囲に収める
                weight = RClampValue(0, static_cast<int>(RVtxWeight::WEIGHT_MAX), weight);
                if (weight != 0)
                {
                    // ウェイトが 0 以外なら行列のインデックスとウェイト値を追加します。
                    vmt.Append(infIndexes[iInf], weight);
                }
            }
            if (weightSum > 0.0f)
            {
                // ウェイトの合計が 1.0 から 0.005 以上離れて入ればエラーとします。
                if (!RIsSame(weightSum, 1.0f, 0.005f))
                {
                    YShowError(&yscene, // Wrong weighted vertices exist: %s.vtx[%d]
                        "スキニングのウェイト値の合計が 1.0 にならない頂点が存在します。ウェイト値を修正してください: {0}.vtx[{1}] \n"
                        "なお、skinCluster ノードのウェイトの正規化アトリビュートがインタラクティブになっていれば、ウェイト値を変更した際に合計が 1.0 になるように自動的に調整されます。",
                        "Vertices have a total skinning weight that is less than 1.0. Make sure that you correct the weight value: {0}.vtx[{1}] \n"
                        "Note that if the Normalize Weights attribute of the skinCluster node is Interactive, the total is automatically adjusted to 1.0 when the weight value is changed.",
                        ynode.m_ShapePath.partialPathName().asChar(), RGetNumberString(vIter.index()));
                    return MS::kFailure;
                }

                // 頂点ウェイトを調整します。
                vmt.AdjustWeight();

                // スムーススキニング強制フラグを設定します。
                if (yopt.m_AdjustsSmoothSkinning && vmt.GetBoneCount() >= 2)
                {
                    ynode.m_ForcesSmoothSkinning = true;
                }

                // コンフィグで指定した数を超えるノードにウェイト値が分配されていればエラーとします。
                if (vmt.GetBoneCount() > yopt.m_MaxVertexSkinningCount)
                {
                    YShowError(&yscene, // The number of weighted influence is over %d: %s.vtx[%d]
                        "コンフィグファイルで指定した値（{0}）を超える数のノードにスキニングのウェイト値が分配されています: {1}.vtx[{2}]",
                        "The skinning weight is being distributed to more nodes than specified in the config file ({0}): {1}.vtx[{2}]",
                        RGetNumberString(yopt.m_MaxVertexSkinningCount),
                        ynode.m_ShapePath.partialPathName().asChar(), RGetNumberString(vIter.index()));
                    return MS::kFailure;
                }
            }
        }
        else
        {
            // フルウェイト（一番ウェイト値の大きいノードに 100%）にする場合の処理です。
            float weightMax = static_cast<float>(INT_MIN);
            int iNode = INT_MAX;
            float weightSum = 0.0f;
            // 取得したウェイトの中で一番値が大きなウェイトとその
            // インフルエンスオブジェクトを取得します。
            for (int iInf = 0; iInf < infCount; ++iInf)
            {
                float weight = weights[iInf];
                weightSum += weight;
                // 今までの最大のウェイト値よりも大きいか、
                // 同じウェイト値でもインフルエンスオブジェクトのインデックスが
                // 小さい場合はそちらを使用します。
                if (weight > weightMax ||
                    (weight == weightMax && infIndexes[iInf] < iNode))
                {
                    weightMax = weight;
                    iNode = infIndexes[iInf];
                }
            }
            if (weightSum > 0.0f)
            {
                // 最終的に残ったインフルエンスオブジェクトにウェイト 100% で設定します。
                vmt.Append(iNode, RVtxWeight::WEIGHT_MAX);
            }
        }

        if (vmt.GetBoneCount() == 0)
        {
            // 頂点を選択して Bind Skin した場合、
            // 選択されなかった頂点の重みはすべて 0 になります。
            // その場合は頂点の属するノードにフルウェイトにします。
            vmt.Append(ynode.m_OrgIndex, RVtxWeight::WEIGHT_MAX);
        }

        // 変換した１頂点分の頂点ウェイトと行列インデックスのデータを
        // m_VtxMtxs に登録し、戻ってきたインデックスを保存します。
        const int iVmt = AppendVtxMtx(ymodel, vmt);
        ynode.m_VtxMtxIdxs.append(iVmt);
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief is full weight at on node
//-----------------------------------------------------------------------------
//static bool IsFullWeightAtOneNode(const ShapeData& shapeData)
//{
//  int vtxCount = shapeData.m_VtxMtxIdxs.length();
//  if (vtxCount == 0) // 通常はあり得ない
//  {
//      return false;
//  }
//
//  int curMtx = shapeData.m_VtxMtxIdxs[0];
//  for (int iVtx = 1; iVtx < vtxCount; ++iVtx)
//  {
//      if (shapeData.m_VtxMtxIdxs[iVtx] != curMtx)
//      {
//          return false;
//      }
//  }
//
//  return ymodel.m_VtxMtxs[curMtx].IsFull();
//}

//-----------------------------------------------------------------------------
//! @brief 頂点座標の座標系を変換します。
//! スキニングなしの場合、変換しないでそのままの頂点座標を返します。
//! 特定のノードにフルウェイトの場合、影響するノードのローカル座標系に変換します。
//! 複数のノードにウェイトが分散している場合、モデル座標系に変換します。
//!
//! @param[out] globalPos モデル座標系に変換した頂点座標を格納します。
//! @param[in] src 元の頂点座標（シェイプノードのローカル座標）です。
//! @param[in] ymodel モデルです。
//! @param[in] ynode 頂点座標を含んでいるシェイプを持つノードです。
//! @param[in] vmt 頂点のウェイトと行列インデックスの情報です。
//!
//! @return 変換後の頂点座標を返します。
//-----------------------------------------------------------------------------
static MPoint GetCoordinateAdjustedPosition(
    MPoint& globalPos,
    const MPoint& src,
    const YModel& ymodel,
    const YNode& ynode,
    const RVtxMtx& vmt
)
{
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    MPoint dst = src;

    if (yopt.m_FmdIsBindPose)
    {
        globalPos = src * ynode.m_BindGlobalMtx;
    }
    else
    {
        globalPos = src * ynode.m_GlobalMtx;
    }

    if (vmt.IsFull())
    {
        if (vmt.GetBoneIndex(0) != ynode.m_OrgIndex)
        {
            dst = globalPos;
            const YNode& infNode = *ymodel.m_pYNodes[vmt.GetBoneIndex(0)];
            if (yopt.m_FmdIsBindPose)
            {
                dst *= infNode.m_BindGlobalInvMtx;
            }
            else
            {
                dst *= infNode.m_GlobalInvMtx;
            }
        }
    }
    else
    {
        dst = globalPos;
    }

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 法線の座標系をスキニングモードに合わせて変換します。
//!        スキニングなしの場合、変換しないでそのままの法線を返します。
//!        リジッドスキニングの場合、影響するノードのローカル座標系に変換します。
//!        スムーススキニングの場合、モデル座標系に変換します。
//!
//! @param[in] src 元の法線（シェイプノードのローカル座標）です。
//! @param[in] ymodel モデルです。
//! @param[in] ynode 法線を含んでいるシェイプを持つノードです。
//! @param[in] vmt 頂点のウェイトと行列インデックスの情報です。
//! @param[in] skinningMode スキニングモードです。
//!
//! @return 変換後の法線を返します。
//-----------------------------------------------------------------------------
static RVec3 GetCoordinateAdjustedNormal(
    const RVec3& src,
    const YModel& ymodel,
    const YNode& ynode,
    const RVtxMtx& vmt,
    const RShape::SkinningMode skinningMode
)
{
    MFloatVector dst = GetMFloatVector(src);

    // 法線に座標用の行列を掛ける際は
    // 逆行列の転置行列を掛ける必要があるので注意
    if (skinningMode != RShape::NO_SKINNING)
    {
        // グローバル座標に変換します。
        // （グローバル行列の逆行列の転置行列を掛ける）
        dst = MulNormalAndMtxPostNormalize(dst, ynode.m_BindGlobalInvMtx.transpose());
        //dst = MulNormalAndMtxPostNormalize(dst, ynode.m_BindNoScaleGlobalMtx);
        if (skinningMode != RShape::SMOOTH && vmt.IsFull())
        {
            // 影響するノードのローカル座標に変換します。
            // （影響するノードのグローバル行列の転置行列を掛ける）
            // M.inverse.transpose = M.transpose.inverse なので
            // M.inverse.transpose.inverse = M.transpose
            const YNode& infNode = *ymodel.m_pYNodes[vmt.GetBoneIndex(0)];
            dst = MulNormalAndMtxPostNormalize(dst, infNode.m_BindGlobalMtx.transpose());
            //dst = MulNormalAndMtxPostNormalize(dst, infNode.m_BindNoScaleGlobalInvMtx);
        }
    }
    else
    {
        dst.normalize();
    }

    return GetRVec3(dst);
}

//-----------------------------------------------------------------------------
//! @brief 接線（従法線）を量子化して精度を調整します。
//!
//! @param[in] tan 接線（従法線）です。
//!
//! @return 量子化して精度を調整した接線（従法線）を返します。
//-----------------------------------------------------------------------------
static RVec3 QuantizeTangent(const RVec3& tan)
{
    // Maya から取得した接線（従法線）は取得の度に最大 0.0001 程度誤差があるので、
    // 4096 段階に量子化します。
    const float PRECISION = 1 << 12;
    const float INV_PRECISION = 1.0f / PRECISION;
    RVec3 adjustedTan = tan;
    adjustedTan.x = RRound(adjustedTan.x * PRECISION) * INV_PRECISION;
    adjustedTan.y = RRound(adjustedTan.y * PRECISION) * INV_PRECISION;
    adjustedTan.z = RRound(adjustedTan.z * PRECISION) * INV_PRECISION;
    return adjustedTan;
}

//-----------------------------------------------------------------------------
//! @brief 接線と従法線の座標系をスキニングモードに合わせて変換し、
//!        接線、従法線、法線が右手座標系なら W 成分を 1、
//!        左手座標系なら W 成分を -1 にした 4 次元ベクトルを返します。
//!        接線または従法線の長さが 0 なら調整した値を設定します。
//!
//! @param[out] dstTan 変換後の接線を格納します。
//! @param[out] dstBin 変換後の従法線を格納します。
//! @param[in] oTan 元の接線  （オブジェクト座標）です。
//! @param[in] oBin 元の従法線（オブジェクト座標）です。
//! @param[in] oNrm 元の法線  （オブジェクト座標）です。
//! @param[in] ymodel モデルです。
//! @param[in] ynode 法線を含んでいるシェイプを持つノードです。
//! @param[in] vmt 頂点のウェイトと行列インデックスの情報です。
//! @param[in] skinningMode スキニングモードです。
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] iNodeVtx mesh ノード内の頂点インデックスです。
//! @param[in] iNodeVtxNext 隣接する頂点の mesh ノード内の頂点インデックスです。
//!
//! @return 接線または従法線の長さが 0 で調整した値を設定したなら true を返します。
//-----------------------------------------------------------------------------
static bool GetCoordinateAdjustedTangent(
    RVec4& dstTan,
    RVec4& dstBin,
    RVec3 oTan,
    RVec3 oBin,
    const RVec3& oNrm,
    const YModel& ymodel,
    const YNode& ynode,
    const RVtxMtx& vmt,
    const RShape::SkinningMode skinningMode,
    const MDagPath& meshPath,
    const int iNodeVtx,
    const int iNodeVtxNext
)
{
    bool isZeroTanBin = false;

    //-----------------------------------------------------------------------------
    // 接線の長さが 0 ならエッジ方向を接線とします。
    if (oTan == RVec3::kZero || !oTan.IsFinite())
    {
        MFnMesh meshFn(meshPath);
        MPoint p0;
        MPoint p1;
        meshFn.getPoint(iNodeVtx    , p0, MSpace::kObject);
        meshFn.getPoint(iNodeVtxNext, p1, MSpace::kObject);
        oTan = GetRVec3(p1 - p0);
        oTan.Normalize();
        oTan.SnapToZero();
        if (oTan == RVec3::kZero)
        {
            oTan = (!oNrm.IsEquivalent(RVec3::kXAxis) && !oNrm.IsEquivalent(RVec3::kXNegAxis)) ?
                RVec3::kXAxis : RVec3::kZAxis;
        }
        isZeroTanBin = true;
    }

    //-----------------------------------------------------------------------------
    // 従法線の長さが 0 なら法線と接線から従法線を計算します。
    if (oBin == RVec3::kZero || !oBin.IsFinite())
    {
        oBin = oNrm ^ oTan;
        oBin.Normalize();
        oBin.SnapToZero();
        if (oBin == RVec3::kZero)
        {
            oBin = (!oNrm.IsEquivalent(RVec3::kZAxis) && !oNrm.IsEquivalent(RVec3::kZNegAxis)) ?
                RVec3::kZNegAxis : RVec3::kXAxis;
        }
        isZeroTanBin = true;
    }

    //-----------------------------------------------------------------------------
    // Maya 2013 Extension2 以降では、デフォームされたモデルの接線（従法線）を取得するたびに
    // 値が微妙に異なる場合があるので、量子化して精度を調整します。
    if (ynode.m_DeformerFlag)
    {
        oTan = QuantizeTangent(oTan);
        oBin = QuantizeTangent(oBin);
    }

    //-----------------------------------------------------------------------------
    // 接線と従法線の座標系をスキニングモードに合わせて変換します。
    RVec3 tan3 = GetCoordinateAdjustedNormal(
        oTan, ymodel, ynode, vmt, skinningMode);
    tan3.SnapToZero();
    if (tan3 == RVec3::kZero)
    {
        tan3 = RVec3::kXAxis;
    }

    RVec3 bin3 = GetCoordinateAdjustedNormal(
        oBin, ymodel, ynode, vmt, skinningMode);
    bin3.SnapToZero();
    if (bin3 == RVec3::kZero)
    {
        bin3 = RVec3::kZNegAxis;
    }

    //-----------------------------------------------------------------------------
    // 座標系から接線と従法線の W 成分を設定します。
    const float tanW = (RIsRightHandTangent(oTan, oBin, oNrm)) ? 1.0f : -1.0f;
    dstTan = RVec4(tan3, tanW);
    dstBin = RVec4(bin3, tanW);

    return isZeroTanBin;
}

//-----------------------------------------------------------------------------
//! @brief 頂点座標属性の値の型を取得します。
//!
//! @param[in] shapeData シェイプデータです。
//!
//! @return 頂点座標属性の値の型を返します。
//-----------------------------------------------------------------------------
static RPrimVtx::ValueType GetPosAttrValueType(const ShapeData& shapeData)
{
    return (shapeData.m_ShapeUpdate.m_UpdatesPos) ?
        RPrimVtx::VALUE_FLOAT : RPrimVtx::VALUE_FLOAT; // 現状シェイプアニメーションの有無にかかわらず float です。
}

//-----------------------------------------------------------------------------
//! @brief シェイプのスキニングモードを取得します。
//!        シェイプの頂点に影響するボーン数からスキングモードを決定し、
//!        ShapeData::m_SkinningMode に設定します。
//!        また、スキニングに使用するボーンの行列パレットインデックスに 0 を設定します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in,out] ymodel モデルです。
//! @param[in] ynode シェイプが属するノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapeSkinningMode(
    ShapeData& shapeData,
    YModel& ymodel,
    const YNode& ynode
)
{
    //-----------------------------------------------------------------------------
    // 1 頂点に影響するボーン数の最大値とリジッドスキンが必要かどうかを調査します。
    shapeData.m_MaxBoneCountPerVtx = 0;
    int mainBoneIdx = -1; // リジッドスキンが必要か判断するためのボーンインデックスです。
    bool rigidSkinFlag = false; // リジッドスキンが必要なら true です。
    RIntArray boneUsedFlags(ymodel.m_pYNodes.size(), 0); // ボーンがシェイプで使用されているかどうかの配列です。

    MItMeshPolygon pIter(shapeData.m_MeshPath, shapeData.m_CompObj);
    for ( ; !pIter.isDone(); pIter.next())
    {
        const int vtxCount = pIter.polygonVertexCount();
        for (int iVtx = 0; iVtx < vtxCount; ++iVtx) // Maya front is counter-clockwise
        {
            // フェースの頂点に対応する頂点行列を参照します。
            const int iNodeVtx = pIter.vertexIndex(iVtx);
            const int iVmt = ynode.m_VtxMtxIdxs[iNodeVtx];
            const RVtxMtx& vmt = ymodel.m_VtxMtxs[iVmt];

            // 1 頂点に影響するボーン数の最大値を調査します。
            const int boneCount = vmt.GetBoneCount();
            if (boneCount > shapeData.m_MaxBoneCountPerVtx)
            {
                shapeData.m_MaxBoneCountPerVtx = boneCount;
            }

            // 影響するボーン数が 1 つの時にはリジッドスキンかスキン無しかの判定を行います。
            if (boneCount == 1)
            {
                // 頂点に影響するボーンを 1 つ抽出します。
                const int boneIdx = vmt.GetBoneIndex(0);
                if (mainBoneIdx == -1)
                {
                    mainBoneIdx = boneIdx;
                }
                else if (boneIdx != mainBoneIdx)
                {
                    // 抽出したボーンと異なるボーンを使用する頂点があった場合は
                    // 1 つのシェイプ内で複数のボーンにより変形されるということなので
                    // スキン無しにはできません。そのためリジッドスキンのフラグを ON にします。
                    rigidSkinFlag = true;
                }
            }

            // シェイプで使用されているボーンのフラグを ON にします。
            for (int iLocalBone = 0; iLocalBone < boneCount; ++iLocalBone)
            {
                boneUsedFlags[vmt.GetBoneIndex(iLocalBone)] = 1;
            }
        }
    }

    // シェイプ内で 1 つのボーンだけ使用する場合でも
    // シェイプの属するボーンでなければリジッドスキンとします。
    if (mainBoneIdx != -1 &&
        mainBoneIdx != ynode.m_OrgIndex)
    {
        rigidSkinFlag = true;
    }

    //-----------------------------------------------------------------------------
    // スキニングモードを決定します。
    if (shapeData.m_MaxBoneCountPerVtx >= 2 || (ynode.m_ForcesSmoothSkinning && rigidSkinFlag))
    {
        // 1 頂点につき 2 つ以上のボーンを使用する場合はスムーススキンです。
        shapeData.m_SkinningMode = RShape::SMOOTH;
        ++ymodel.m_SmoothSkinningShapeCount;
    }
    else if (rigidSkinFlag)
    {
        // 1 頂点について 1 つしかボーンを使用せず、
        // シェイプ内で 2 つ以上のボーンを使用する場合リジッドスキンです。
        shapeData.m_SkinningMode = RShape::RIGID;
        ++ymodel.m_RigidSkinningShapeCount;
    }
    else
    {
        // それ以外の場合はスキン無しです。
        shapeData.m_SkinningMode = RShape::NO_SKINNING;
    }

    //-----------------------------------------------------------------------------
    // スキニングに使用するボーンの行列パレットインデックスに 0 を設定します。
    // 実際の行列パレットインデックスは後で設定します。
    if (shapeData.m_SkinningMode == RShape::RIGID)
    {
        for (int iBone = 0; iBone < static_cast<int>(ymodel.m_pYNodes.size()); ++iBone)
        {
            if (boneUsedFlags[iBone] != 0)
            {
                ymodel.m_pYNodes[iBone]->m_RigidMtxIdx = 0;
            }
        }
    }
    else if (shapeData.m_SkinningMode == RShape::SMOOTH)
    {
        for (int iBone = 0; iBone < static_cast<int>(ymodel.m_pYNodes.size()); ++iBone)
        {
            if (boneUsedFlags[iBone] != 0)
            {
                ymodel.m_pYNodes[iBone]->m_SmoothMtxIdx = 0;
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief シェイプの頂点座標とそのインデックス、行列インデックスを取得します。
//!
//! @param[out] shapeData シェイプデータです。
//! @param[in,out] ymodel モデルです。
//! @param[in] ynode シェイプが属するノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapePosMtxIdx(
    ShapeData& shapeData,
    YModel& ymodel,
    const YNode& ynode
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // シェイプのスキニングモードを取得します。
    // また、スキニングに使用するボーンの行列パレットインデックスに 0 を設定します。
    GetShapeSkinningMode(shapeData, ymodel, ynode);

    //-----------------------------------------------------------------------------
    // ブレンドシェイプで変化するならターゲットシェイプごとの頂点座標配列を用意します。
    const YBlendShapeData* pBsData =
        (shapeData.m_ShapeUpdate.m_UpdatesPos && ynode.m_BlendShapeDataIndex != -1) ?
        &ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex] : NULL;
    int targetCount = 0;
    if (pBsData != NULL)
    {
        targetCount = static_cast<int>(pBsData->m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            shapeData.m_TargetPosss.push_back(RVec3Array());
        }
    }

    //-----------------------------------------------------------------------------
    // 頂点座標と行列インデックスを取得します。
    RVec3Array& vtxPoss = shapeData.m_VtxPoss; // シェイプの頂点座標配列
    RVec3Array calcObbPoss; // OBB 計算用頂点座標

    // スキニングなしなら、シェイプが属するボーンにおけるローカル座標、
    // リジッドスキニングなら、影響するボーンにおけるローカル座標、
    // スムーススキニングなら、バインドポーズでのワールド座標を取得します。
    const bool getsGlobalPos = (shapeData.m_SkinningMode == RShape::SMOOTH);

    // ノードの頂点インデックスに対するシェイプの頂点座標インデックス配列です。
    MIntArray nodeToShapePosIdxs(ynode.m_InfPoss.length(), -1);

    // シェイプに含まれるポリゴン群についてループします。
    MItMeshPolygon pIter(shapeData.m_MeshPath, shapeData.m_CompObj);
    for ( ; !pIter.isDone(); pIter.next())
    {
        RVec3Array faceVtxPoss; // 三角形分割パターン計算用の頂点座標配列です。
        const int vtxCount = pIter.polygonVertexCount();
        for (int iVtx = 0; iVtx < vtxCount; ++iVtx) // Maya front is counter-clockwise
        {
            //-----------------------------------------------------------------------------
            // 頂点座標
            const int iNodeVtx = pIter.vertexIndex(iVtx); // ノード中の頂点インデックス
            int iPos = nodeToShapePosIdxs[iNodeVtx];
            if (iPos == -1)
            {
                const MPoint* pPos = (getsGlobalPos) ?
                    &ynode.m_GlobalPoss[iNodeVtx] : &ynode.m_InfPoss[iNodeVtx];
                const RVec3 pos = GetRVec3(*pPos, true);

                if (pBsData == NULL)
                {
                    iPos = RFindValueInArray(vtxPoss, pos); // 同じ値なら圧縮します。
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 各ターゲットシェイプの頂点座標を追加します。
                    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
                    {
                        const YBlendShapeTarget& target = pBsData->m_Targets[iTarget];
                        const MPoint* pTargetPos = (getsGlobalPos) ?
                            &target.m_GlobalPoss[iNodeVtx] : &target.m_Poss[iNodeVtx];
                        shapeData.m_TargetPosss[iTarget].push_back(GetRVec3(*pTargetPos, true));
                    }
                }

                if (iPos == -1)
                {
                    iPos = static_cast<int>(vtxPoss.size());
                    vtxPoss.push_back(pos);
                    nodeToShapePosIdxs[iNodeVtx] = iPos;

                    // 「OBB 計算用頂点座標」は
                    // ・スキニングなしなら、シェイプが属するボーンにおけるローカル座標
                    // ・リジッド or スムーススキニングなら、バインドポーズでのワールド座標
                    const MPoint* calcCenterPosPtr =
                        (shapeData.m_SkinningMode == RShape::NO_SKINNING) ?
                        &ynode.m_InfPoss[iNodeVtx] : &ynode.m_GlobalPoss[iNodeVtx];
                    calcObbPoss.push_back(GetRVec3(*calcCenterPosPtr));
                }
            }
            shapeData.m_VtxPosIdxs.append(iPos);

            //-----------------------------------------------------------------------------
            // 三角形分割パターン計算用の頂点座標配列にグローバル座標を追加します。
            if (vtxCount >= 4)
            {
                faceVtxPoss.push_back(GetRVec3(ynode.m_GlobalPoss[iNodeVtx]));
            }

            //-----------------------------------------------------------------------------
            // ノードから行列インデックスを取得します。
            const int iVmt = ynode.m_VtxMtxIdxs[iNodeVtx];
            shapeData.m_VtxMtxIdxs.append(iVmt);

            //-----------------------------------------------------------------------------
            // スムーススキンを行う場合は変形に使用するボーンに対しスキニングに
            // 使用するフラグを設定します。
            if (shapeData.m_SkinningMode == RShape::SMOOTH)
            {
                const RVtxMtx& vmt = ymodel.m_VtxMtxs[iVmt];
                const int boneCount = vmt.GetBoneCount();
                for (int ibone = 0; ibone < boneCount; ++ibone)
                {
                    (*ymodel.m_pYNodes[vmt.GetBoneIndex(ibone)]).m_SkinMtxFlag = true;
                }
            }
        }

        //-----------------------------------------------------------------------------
        // 最適な三角形分割パターンを取得します。
        const int triPattern = (yopt.m_OptimizeTriangulation == RExpOpt::OPTIMIZE_TRIANGULATION_ANGLE) ?
            RGetAngleTriangulationPattern(faceVtxPoss) : 0;
        //const int triPattern = 0;
        shapeData.m_TriPatterns.push_back(triPattern);
    }

    //-----------------------------------------------------------------------------
    // get OBB
    shapeData.m_OrientedBB.CalculateByPca(calcObbPoss, true);

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 法線属性の値の型を取得します。
//!
//! @param[in] shapeData シェイプデータです。
//!
//! @return 法線属性の値の型を返します。
//-----------------------------------------------------------------------------
static RPrimVtx::ValueType GetNrmAttrValueType(const ShapeData& shapeData)
{
    return (shapeData.m_ShapeUpdate.m_UpdatesNrm) ?
        RPrimVtx::VALUE_FLOAT : RPrimVtx::VALUE_FLOAT; // 現状シェイプアニメーションの有無にかかわらず float です。
}

//-----------------------------------------------------------------------------
//! @brief Maya から取得した法線が不正なら調整します。
//!
//! @param[in,out] oNrm Maya から取得したオブジェクト座標の法線です。
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] iPoly 法線を含むフェースのインデックスです。
//!
//! @return 法線を調整したなら true を返します。
//-----------------------------------------------------------------------------
static bool AdjustInvalidNormal(
    RVec3& oNrm,
    const MDagPath& meshPath,
    const int iPoly
)
{
    if (oNrm == RVec3::kZero || !oNrm.IsFinite()) // 長さ 0 ならフェース法線を取得します。
    {
        MVector faceNrm;
        MFnMesh(meshPath).getPolygonNormal(iPoly, faceNrm, MSpace::kObject);
        oNrm = GetRVec3(faceNrm);
        oNrm.SnapToZero();
        if (oNrm == RVec3::kZero || !oNrm.IsFinite()) // フェース法線の長さも 0 なら固定値を設定します。
        {
            oNrm = R_ZERO_NRM_REPLACE;
        }
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプの法線を取得します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in] ymodel モデルです。
//! @param[in] ynode シェイプを参照するノードです。
//! @param[in] nodeNrms ノード中の法線の配列（オブジェクト座標）です。
//! @param[in] nodeNrmIdxs 頂点フェースごとの nodeNrms へのインデックスです。
//! @param[in] nodeNrmOfss フェースごとの最初の頂点フェースへのインデックスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapeNormal(
    ShapeData& shapeData,
    const YModel& ymodel,
    const YNode& ynode,
    const MFloatVectorArray& nodeNrms,
    const MIntArray& nodeNrmIdxs,
    const MIntArray& nodeNrmOfss
)
{
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // ブレンドシェイプで変化するならターゲットシェイプごとの法線配列を用意します。
    const YBlendShapeData* pBsData =
        (shapeData.m_ShapeUpdate.m_UpdatesNrm && ynode.m_BlendShapeDataIndex != -1) ?
        &ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex] : NULL;
    int targetCount = 0;
    if (pBsData != NULL)
    {
        targetCount = static_cast<int>(pBsData->m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            shapeData.m_TargetNrmss.push_back(RVec3Array());
        }
    }

    //-----------------------------------------------------------------------------
    // ブレンドシェイプのための、
    // Maya の頂点インデックスに対する出力法線インデックス配列を初期化します。
    // 出力法線インデックス配列は頂点単位の法線を使用するか（-1）、
    // 頂点フェース単位の法線を使用するか（-2）のフラグも兼ねています。
    MIntArray useVtxNrmIdxs;
    if (pBsData != NULL)
    {
        useVtxNrmIdxs = pBsData->m_UseVtxNrmFlags;
    }

    //-----------------------------------------------------------------------------
    // loop for face
    RVec3Array& vtxNrms = shapeData.m_VtxNrms;

    // ノードの法線インデックスに対するシェイプの法線インデックス配列です。
    MIntArray nodeToShapeNrmIdxs(nodeNrms.length(), -1);

    // 頂点座標が最初に出現した vtxNrms 内のインデックスです。
    // 検索を高速にするために使用します。
    RIntArray searchStartIdxs(shapeData.m_VtxPoss.size(), -1);
    //RIntArray searchStartIdxs(shapeData.m_VtxPoss.size(), 0); // 従来のようにすべて検索する場合

    int zeroNormalCount = 0; // ベースシェイプの長さ 0 法線の数です。
    int zeroNormalVtxIdx = -1; // ベースシェイプの長さ 0 法線を持つ最初の頂点のインデックスです。
    std::string zeroNormalTargetName; // 長さ 0 法線を持つターゲットシェイプ名です。
    int zeroNormalTargetVtxIdx = -1; // ターゲットシェイプの長さ 0 法線を持つ最初の頂点のインデックスです。
    int iVtxTotal = 0;
    MItMeshPolygon pIter(shapeData.m_MeshPath, shapeData.m_CompObj);
    for ( ; !pIter.isDone(); pIter.next())
    {
        //-----------------------------------------------------------------------------
        // loop for vertex / face
        const int vtxCount = pIter.polygonVertexCount();
        for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal) // Maya front is counter-clockwise
        {
            //-----------------------------------------------------------------------------
            // 配列から法線を取得します。
            const int iNodeVtx = pIter.vertexIndex(iVtx); // mesh ノード内の頂点インデックス
            const int nodeNrmOfs = nodeNrmOfss[pIter.index()] + iVtx; // nodeNrmIdxs 内の法線インデックス
            const int iNodeNrm = nodeNrmIdxs[nodeNrmOfs]; // mesh ノード内の法線インデックス
            RVec3 oNrm = GetRVec3(nodeNrms[iNodeNrm]);
            if (AdjustInvalidNormal(oNrm, shapeData.m_MeshPath, pIter.index()))
            {
                ++zeroNormalCount;
                if (zeroNormalVtxIdx == -1)
                {
                    zeroNormalVtxIdx = iNodeVtx;
                }
            }

            //-----------------------------------------------------------------------------
            // ブレンドシェイプで法線を更新しない場合、nodeToShapeNrmIdxs にインデックスが既に
            // 設定されていたら同じインデックスを使用するよう設定します。
            // これは同じ法線インデックスに対する処理を軽減するために行います。
            if (pBsData == NULL && nodeToShapeNrmIdxs[iNodeNrm] != -1)
            {
                shapeData.m_VtxNrmIdxs.append(nodeToShapeNrmIdxs[iNodeNrm]);
                continue;
            }

            //-----------------------------------------------------------------------------
            // ブレンドシェイプで法線を更新し、頂点単位の法線を使用する場合、
            // すでに頂点単位の法線を出力済みであれば、
            // その出力法線インデックスを設定します。
            if (pBsData != NULL && useVtxNrmIdxs[iNodeVtx] >= 0)
            {
                shapeData.m_VtxNrmIdxs.append(useVtxNrmIdxs[iNodeVtx]);
                continue;
            }

            //-----------------------------------------------------------------------------
            // 法線をスキニングモードに合わせた座標系に変換します。
            const int iVmt = shapeData.m_VtxMtxIdxs[iVtxTotal];
            const RVtxMtx& vmt = ymodel.m_VtxMtxs[iVmt];
            RVec3 nrm = GetCoordinateAdjustedNormal(
                oNrm, ymodel, ynode, vmt, shapeData.m_SkinningMode);
            nrm.SnapToZero();
            if (nrm == RVec3::kZero)
            {
                nrm = R_ZERO_NRM_REPLACE;
            }

            //-----------------------------------------------------------------------------
            // すでに追加済みの法線から同じ法線（誤差が許容値未満）を検索します。
            // ブレンドシェイプで法線が更新される場合はトポロジを Maya のオリジナルデータに
            // 合わせるため圧縮しません。
            int iNrm = -1;
            if (pBsData == NULL)
            {
                const int iPos = shapeData.m_VtxPosIdxs[iVtxTotal];
                const int searchStartIdx = searchStartIdxs[iPos]; // ここから検索します。
                if (searchStartIdx == -1)
                {
                    // 初めて出現した頂点座標なら、検索しないで検索開始インデックスを設定します。
                    searchStartIdxs[iPos] = static_cast<int>(vtxNrms.size());
                }
                else
                {
                    // すでに出現した頂点座標なら、検索開始インデックスから法線を検索します。
                    iNrm = RFindValueInArray(vtxNrms, nrm, R_VTX_NRM_TOLERANCE, searchStartIdx);
                }
            }
            else
            {
                //-----------------------------------------------------------------------------
                // 各ターゲットシェイプの法線を追加します。
                for (int iTarget = 0; iTarget < targetCount; ++iTarget)
                {
                    RVec3 targetNrm(nrm);
                    if (pBsData->m_IsMemberFaces[pIter.index()] != 0)
                    {
                        const YBlendShapeTarget& target = pBsData->m_Targets[iTarget];
                        MVector mnrm;
                        MFnMesh(target.m_ShapePath).getFaceVertexNormal(
                            pIter.index(), iNodeVtx, mnrm, MSpace::kObject);
                        RVec3 oTargetNrm = GetRVec3(mnrm);
                        if (AdjustInvalidNormal(oTargetNrm, target.m_ShapePath, pIter.index()))
                        {
                            if (zeroNormalTargetName.empty())
                            {
                                zeroNormalTargetName = target.m_ShapePath.partialPathName().asChar();
                                zeroNormalTargetVtxIdx = iNodeVtx;
                            }
                        }
                        targetNrm = GetCoordinateAdjustedNormal(
                            oTargetNrm, ymodel, ynode, vmt, shapeData.m_SkinningMode);
                        targetNrm.SnapToZero();
                        if (targetNrm == RVec3::kZero)
                        {
                            targetNrm = R_ZERO_NRM_REPLACE;
                        }
                    }
                    shapeData.m_TargetNrmss[iTarget].push_back(targetNrm);
                }
            }

            //-----------------------------------------------------------------------------
            // 見つからなければ法線を追加します。
            if (iNrm == -1)
            {
                iNrm = static_cast<int>(vtxNrms.size());
                vtxNrms.push_back(nrm);
            }

            //-----------------------------------------------------------------------------
            // 法線インデックスをシェイプに追加します。
            shapeData.m_VtxNrmIdxs.append(iNrm);

            //-----------------------------------------------------------------------------
            // ブレンドシェイプで法線を更新しない場合、nodeToShapeNrmIdxs に追加したインデックス値を
            // 記録しておき、同じ法線インデックスに対してこの値を使いまわすようにします。
            if (pBsData == NULL)
            {
                nodeToShapeNrmIdxs[iNodeNrm] = iNrm;
            }

            //-----------------------------------------------------------------------------
            // ブレンドシェイプで法線を更新し、頂点単位の法線を使用する設定だった場合、
            // 出力法線インデックスを記録します。
            if (pBsData != NULL && useVtxNrmIdxs[iNodeVtx] == YBlendShapeData::PER_VTX)
            {
                useVtxNrmIdxs[iNodeVtx] = iNrm;
            }
        }
    }

    const std::string zeroNrmWarnCompStr =
        (zeroNormalCount > 0) ? shapeData.m_MeshPath.partialPathName().asChar() +
            RGetNumberString(zeroNormalVtxIdx, ".vtx[%d]") +
            RGetNumberString(zeroNormalCount, " (%d vertex faces)") :
        (!zeroNormalTargetName.empty()) ? zeroNormalTargetName +
            RGetNumberString(zeroNormalTargetVtxIdx, ".vtx[%d]") :
        "";
    if (!zeroNrmWarnCompStr.empty())
    {
        YShowWarning(&yscene, // Zero normal exists: %s.vtx[%d] (%d vertex faces)
            "長さが 0 の法線が存在したため、フェース法線に置き換えて出力しました: {0} \n"
            "ライティングが正常に行えません。「頂点法線の設定」などで修正してください。",
            "A normal with length 0 was found. It was replaced with the face normal and exported: {0} \n"
            "Lighting effects will not work properly. Correct this problem, for example, by using Set Vertex Normal.",
            zeroNrmWarnCompStr);
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief カラーセット名から頂点属性の値の型を取得します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] colSet カラーセット名です。
//!
//! @return 頂点属性の値の型を返します。
//-----------------------------------------------------------------------------
static RPrimVtx::ValueType GetVtxAttrValueType(const MDagPath& meshPath, const MString& colSet)
{
    const MString familyName = GetColorSetFamilyName(meshPath, colSet);
    const int length = static_cast<int>(familyName.length());
    const int iUnder = familyName.rindex('_'); // 最後の _ を検索します。
    if (iUnder != -1 && (iUnder + 2 == length || iUnder + 3 == length))
    {
        int compCount = (iUnder + 3 == length) ?
            familyName.asChar()[iUnder + 1] - '0' : 1;
        compCount = (compCount == ColorSetComponentCountRa) ? 2 : compCount;
        if (1 <= compCount && compCount <= 4) // 成分数が正しいことを確認します。
        {
            const char t = familyName.asChar()[length - 1];
            if (t == 'i')
            {
                return RPrimVtx::VALUE_INT;
            }
            else if (t == 'u')
            {
                return RPrimVtx::VALUE_UINT;
            }
        }
    }
    return RPrimVtx::VALUE_FLOAT;
}

//-----------------------------------------------------------------------------
//! @brief 出力形式に変換した頂点カラーを返します。
//!
//! @param[in,out] shapeData シェイプデータです。
//!
//! @return 出力形式に変換した頂点カラーを返します。
//-----------------------------------------------------------------------------
static inline RVec4 GetOutVtxColor(const MColor& mc, const YVtxAttrInfo& colInfo)
{
    //-----------------------------------------------------------------------------
    // 値の型にあわせて調整します。
    RVec4 col = (!colInfo.m_IsRa) ?
        RVec4(mc.r, mc.g, mc.b, mc.a) :
        RVec4(mc.r, mc.a, 0.0f, 1.0f);
    if (colInfo.m_ValueType == RPrimVtx::VALUE_FLOAT)
    {
        col.SnapToZero();
    }
    else if (colInfo.m_ValueType == RPrimVtx::VALUE_INT)
    {
        for (int iRgba = 0; iRgba < colInfo.m_CompCount; ++iRgba)
        {
            col[iRgba] = static_cast<float>(RRound(col[iRgba]));
        }
    }
    else // VALUE_UINT
    {
        for (int iRgba = 0; iRgba < colInfo.m_CompCount; ++iRgba)
        {
            col[iRgba] = (col[iRgba] >= 0.0f) ?
                static_cast<float>(RRound(col[iRgba])) : 0.0f;
        }
    }

    //-----------------------------------------------------------------------------
    // 出力する成分以外は 1 にします（同じカラーを検索する際に比較されないように）。
    for (int iRgba = colInfo.m_CompCount; iRgba < R_RGBA_COUNT; ++iRgba)
    {
        col[iRgba] = 1.0f;
    }

    return col;
}

//-----------------------------------------------------------------------------
//! @brief シェイプの頂点カラーを取得します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[out] xluVtxFlag アルファが 1.0 未満の頂点カラーがあれば true を格納します。
//! @param[in] ymodel モデルです。
//! @param[in] ynode シェイプが属するノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapeVtxColor(
    ShapeData& shapeData,
    bool& xluVtxFlag,
    const YModel& ymodel,
    const YNode& ynode
)
{
    xluVtxFlag = false;
    const MColor defaultMC(0.0f, 0.0f, 0.0f, 1.0f); // black

    //-----------------------------------------------------------------------------
    // 出力するカラーセットについてループします。
    for (int iColSet = 0; iColSet < static_cast<int>(shapeData.m_VtxColInfos.size()); ++iColSet)
    {
        //-----------------------------------------------------------------------------
        // 頂点カラーが存在しなければスキップします。
        YVtxAttrInfo& colInfo = shapeData.m_VtxColInfos[iColSet];
        if (!colInfo.m_ExistFlag)
        {
            continue;
        }

        //-----------------------------------------------------------------------------
        // ブレンドシェイプで変化するならターゲットシェイプごとの頂点カラー配列を用意します。
        // また、ターゲットシェイプごとの全頂点カラーを Maya から取得します。
        const YBlendShapeData* pBsData =
            (shapeData.m_ShapeUpdate.m_UpdatesCols[iColSet] && ynode.m_BlendShapeDataIndex != -1) ?
            &ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex] : NULL;
        std::vector<RVec4Array>& targetColss = shapeData.m_TargetColsss[iColSet];
        std::vector<MColorArray> targetFaceVtxColss;
        int targetCount = 0;
        if (pBsData != NULL)
        {
            targetCount = static_cast<int>(pBsData->m_Targets.size());
            for (int iTarget = 0; iTarget < targetCount; ++iTarget)
            {
                const YBlendShapeTarget& target = pBsData->m_Targets[iTarget];
                targetColss.push_back(RVec4Array());
                targetFaceVtxColss.push_back(MColorArray());
                MFnMesh(target.m_ShapePath).getFaceVertexColors(
                    targetFaceVtxColss[iTarget], &target.m_ColSets[iColSet], &defaultMC);
            }
        }

        //-----------------------------------------------------------------------------
        // ブレンドシェイプのための、
        // Maya の頂点インデックスに対する出力頂点カラーインデックス配列を初期化します。
        // 出力頂点カラーインデックス配列は頂点単位の頂点カラーを使用するか（-1）、
        // 頂点フェース単位の頂点カラーを使用するか（-2）のフラグも兼ねています。
        MIntArray useVtxColIdxs;
        if (pBsData != NULL)
        {
            useVtxColIdxs = pBsData->m_UseVtxColFlagss[iColSet];
        }

        //-----------------------------------------------------------------------------
        // カラーセットに対応する出力データの参照を設定します。
        RVec4Array& vtxCols   = shapeData.m_VtxColss[iColSet];
        MIntArray& vtxColIdxs = shapeData.m_VtxColIdxss[iColSet]; // 頂点フェースごとの vtxCols へのインデックス配列です。
        const MString& colSet = ynode.m_MeshColSets[iColSet];

        //-----------------------------------------------------------------------------
        // 頂点カラーの値の型を取得します。
        colInfo.m_ValueType = GetVtxAttrValueType(shapeData.m_MeshPath, colSet);
        const float tolerance = RGetVertexColorTolerance(colInfo.m_ValueType);

        //-----------------------------------------------------------------------------
        // get opa flag
        //bool opaFlag = IsComponentVertexColorOpaque(
        //  shapeData.m_MeshPath, shapeData.m_CompObj, colSet);

        //-----------------------------------------------------------------------------
        // loop for face

        // 頂点座標が最初に出現した vtxCols 内のインデックスです。
        // 検索を高速にするために使用します。
        RIntArray searchStartIdxs(shapeData.m_VtxPoss.size(), -1);

        int iVtxTotal = 0;
        MItMeshPolygon pIter(shapeData.m_MeshPath, shapeData.m_CompObj);
        for ( ; !pIter.isDone(); pIter.next())
        {
            //-----------------------------------------------------------------------------
            // フェースの頂点カラーを取得します。
            int colCount = 0;
            pIter.numColors(colCount, &colSet);

            MColorArray vtxColors;
            if (colCount > 0)
            {
                pIter.getColors(vtxColors, &colSet);
                // 頂点カラーがない頂点に対しては（-1, -1, -1, -1) が入ります。
            }

            const int vtxCount = pIter.polygonVertexCount();
            for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal) // Maya front is counter-clockwise
            {
                const int iNodeVtx = pIter.vertexIndex(iVtx);

                //-----------------------------------------------------------------------------
                // ブレンドシェイプで頂点カラーを更新し、頂点単位の頂点カラーを使用する場合、
                // すでに頂点単位の頂点カラーを出力済みであれば、
                // その出力頂点カラーインデックスを設定します。
                if (pBsData != NULL && useVtxColIdxs[iNodeVtx] >= 0)
                {
                    vtxColIdxs.append(useVtxColIdxs[iNodeVtx]);
                    continue;
                }

                //-----------------------------------------------------------------------------
                // 頂点カラーを出力形式に変換します。
                MColor mc(defaultMC);
                if (colCount > 0 && vtxColors[iVtx] != Y_MCOLOR_NULL)
                {
                    mc = vtxColors[iVtx];
                }
                const RVec4 col = GetOutVtxColor(mc, colInfo);

                //-----------------------------------------------------------------------------
                // 1 つでも頂点カラーのアルファ値が 1.0 未満のモノがあれば
                // xluVtxFlag を true にします。
                if (col[3] < 1.0f)
                {
                    xluVtxFlag = true;
                }

                //-----------------------------------------------------------------------------
                // すでに追加済みの頂点カラーから同じ頂点カラー（誤差が許容値未満）を検索します。
                // ブレンドシェイプで頂点カラーが更新される場合はトポロジを Maya のオリジナルデータに
                // 合わせるため圧縮しません。
                int iCol = -1;
                if (pBsData == NULL)
                {
                    const int iPos = shapeData.m_VtxPosIdxs[iVtxTotal];
                    const int searchStartIdx = searchStartIdxs[iPos]; // ここから検索します。
                    if (searchStartIdx == -1)
                    {
                        // 初めて出現した頂点座標なら、検索しないで検索開始インデックスを設定します。
                        searchStartIdxs[iPos] = static_cast<int>(vtxCols.size());
                    }
                    else
                    {
                        // すでに出現した頂点座標なら、検索開始インデックスから頂点カラーを検索します。
                        iCol = RFindValueInArray(vtxCols, col, tolerance, searchStartIdx);
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 各ターゲットシェイプの頂点カラーを追加します。
                    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
                    {
                        RVec4 targetCol(col);
                        if (pBsData->m_IsMemberFaces[pIter.index()] != 0)
                        {
                            const YBlendShapeTarget& target = pBsData->m_Targets[iTarget];
                            int iTargetCol;
                            MFnMesh(target.m_ShapePath).getFaceVertexColorIndex(
                                pIter.index(), iVtx, iTargetCol, &target.m_ColSets[iColSet]);
                            targetCol = GetOutVtxColor(targetFaceVtxColss[iTarget][iTargetCol], colInfo);
                        }
                        targetColss[iTarget].push_back(targetCol);
                    }
                }

                //-----------------------------------------------------------------------------
                // 見つからなければ頂点カラーを追加します。
                if (iCol == -1)
                {
                    iCol = static_cast<int>(vtxCols.size());
                    vtxCols.push_back(col);
                }

                //-----------------------------------------------------------------------------
                // 頂点カラーのインデックスをシェイプに登録します。
                vtxColIdxs.append(iCol);

                //-----------------------------------------------------------------------------
                // ブレンドシェイプで頂点カラーを更新し、頂点単位の頂点カラーを使用する設定だった場合、
                // 出力頂点カラーインデックスを記録します。
                if (pBsData != NULL && useVtxColIdxs[iNodeVtx] == YBlendShapeData::PER_VTX)
                {
                    useVtxColIdxs[iNodeVtx] = iCol;
                }
            }
        }
        //cerr << "vtx col count: " << iColSet << ": " << vtxCols.size() << endl;
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief シェイプにユーザー頂点属性を設定します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in] ynode シェイプが属するノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapeUserAttr(ShapeData& shapeData, const YNode& ynode)
{
    //-----------------------------------------------------------------------------
    // loop for user attribute
    for (int iUa = 0; iUa < static_cast<int>(shapeData.m_VtxUsrInfos.size()); ++iUa)
    {
        // m_VtxUsrInfos の要素が設定されていなければ処理をスキップします。
        YVtxAttrInfo& usrInfo = shapeData.m_VtxUsrInfos[iUa];
        if (!usrInfo.m_ExistFlag)
        {
            continue;
        }
        RVec4Array& vtxUsrs = shapeData.m_VtxUsrss[iUa]; // ユーザー頂点属性配列
        MIntArray& vtxUsrIdxs = shapeData.m_VtxUsrIdxss[iUa]; // 頂点フェースごとの vtxUsrs へのインデックス
        const MString& colSet = ynode.m_MeshUsrColSets[iUa]; // ユーザー頂点属性の名前

        //-----------------------------------------------------------------------------
        // ユーザー頂点属性の量子化モードを取得します。
        usrInfo.m_ValueType = GetVtxAttrValueType(shapeData.m_MeshPath, colSet);
        const float tolerance = RGetVertexUserAttrTolerance(usrInfo.m_ValueType);

        //-----------------------------------------------------------------------------
        // mesh に対応するカラーセットがなければデフォルト値を設定します。
        MItMeshPolygon pIter(shapeData.m_MeshPath, shapeData.m_CompObj);
        if (colSet.length() == 0)
        {
            vtxUsrs.push_back(RVec4::kBlack);
            for ( ; !pIter.isDone(); pIter.next())
            {
                const int vtxCount = pIter.polygonVertexCount();
                for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
                {
                    vtxUsrIdxs.append(0);
                }
            }
            continue;
        }

        //-----------------------------------------------------------------------------
        // loop for face
        const MColor defMc = (usrInfo.m_CompCount == 1) ?
            MColor(0.0f, 0.0f, 0.0f, 0.0f) :
            MColor(0.0f, 0.0f, 0.0f, 1.0f);

        // 頂点座標が最初に出現した vtxUsrs 内のインデックスです。
        // 検索を高速にするために使用します。
        RIntArray searchStartIdxs(shapeData.m_VtxPoss.size(), -1);

        int iVtxTotal = 0;
        for ( ; !pIter.isDone(); pIter.next())
        {
            int colCount = 0;
            pIter.numColors(colCount, &colSet);
            MColorArray vtxColors;
            if (colCount > 0)
            {
                pIter.getColors(vtxColors, &colSet);
            }

            const int vtxCount = pIter.polygonVertexCount();
            for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal) // Maya front is counter-clockwise
            {
                MColor mc(defMc);
                if (colCount > 0 && vtxColors[iVtx] != Y_MCOLOR_NULL)
                {
                    mc = vtxColors[iVtx];
                }
                if (usrInfo.m_CompCount == 1)
                {
                    mc.r = mc.a;
                }
                RVec4 usr(mc.r, mc.g, mc.b, mc.a);
                usr.SnapToZero();

                //-----------------------------------------------------------------------------
                // すでに追加済みのユーザー頂点属性から同じユーザー頂点属性（誤差が許容値未満）を検索し、
                // 見つからなければ追加します。
                int iUsr = -1;
                const int iPos = shapeData.m_VtxPosIdxs[iVtxTotal];
                const int searchStartIdx = searchStartIdxs[iPos]; // ここから検索します。
                if (searchStartIdx == -1)
                {
                    // 初めて出現した頂点座標なら、検索しないで検索開始インデックスを設定します。
                    searchStartIdxs[iPos] = static_cast<int>(vtxUsrs.size());
                }
                else
                {
                    // すでに出現した頂点座標なら、検索開始インデックスからユーザー頂点属性を検索します。
                    iUsr = RFindValueInArray(vtxUsrs, usr, tolerance, searchStartIdx);
                }

                if (iUsr == -1)
                {
                    iUsr = static_cast<int>(vtxUsrs.size());
                    vtxUsrs.push_back(usr);
                }

                // テーブルにユーザー頂点属性のインデックスを登録します。
                vtxUsrIdxs.append(iUsr);
            }
        }
        //cerr << "vtx usr count: " << iUa << ": " << vtxUsrs.size() << endl;
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief シェイプのテクスチャ座標を取得します。
//!        引数 shapeData に渡されたシェイプが使用するテクスチャ座標を
//!        メッシュから取得し、m_VtxTexss と m_VtxTexIdxss に設定します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in,out] yscene シーンです。
//! @param[in] meshUvSetNames 出力する UV の UV セット名配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapeTexCoord(
    ShapeData& shapeData,
    YScene& yscene,
    const MStringArray& meshUvSetNames
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // loop for uvset
    MFnMesh meshFn(shapeData.m_MeshPath);
    const std::string meshName = shapeData.m_MeshPath.partialPathName().asChar(); // 警告に使用するメッシュ名です。
    for (int iTex = 0; iTex < shapeData.m_VtxTexCount; ++iTex)
    {
        //-----------------------------------------------------------------------------
        // 一つの UV セット名を取得します。
        const MString& uvSetName = meshUvSetNames[iTex];
        //cerr << "uvSet: " << meshName << ": " << uvSetName << R_ENDL;

        //-----------------------------------------------------------------------------
        // ポリゴン単位のループ

        // 警告表示フラグ、UV の取得に失敗した場合大量の警告が発生する可能性があるので、
        // 最初に表示した警告以降は表示しないようにします。
        bool getUvWarnFlag = false;

        RVec2Array& vtxTexs = shapeData.m_VtxTexss[iTex];

        // 頂点座標が最初に出現した vtxTexs 内のインデックスです。
        // 検索を高速にするために使用します。
        RIntArray searchStartIdxs(shapeData.m_VtxPoss.size(), -1);

        int iVtxTotal = 0;
        MItMeshPolygon pIter(shapeData.m_MeshPath, shapeData.m_CompObj);
        for ( ; !pIter.isDone(); pIter.next())
        {
            const int vtxCount = pIter.polygonVertexCount();
            for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal) // Maya front is counter-clockwise
            {
                //-----------------------------------------------------------------------------
                // 頂点フェースの UV を取得します。
                float fu, fv;
                status = meshFn.getPolygonUV(pIter.index(), iVtx,
                    fu, fv, &uvSetName);
                if (!status)
                {
                    // UV の取得に失敗したらテクスチャ座標に (0,0) を設定し、
                    // 最初の 1 つだけ警告を表示します。
                    fu = fv = 0.0f;
                    if (!getUvWarnFlag)
                    {
                        getUvWarnFlag = true;
                        YShowWarning(&yscene, // Cannot get UV: %s.f[%d] (%s)
                            "テクスチャーの貼られたフェースに UV 座標が設定されていません: {0}.f[{1}] ({2}) \n"
                            "フェースの各頂点の UV 座標を (0, 0) として出力します。",
                            "UV coordinates have not been set for the face to which the texture has been bound: {0}.f[{1}] ({2}) \n"
                            "UV coordinates for each vertex of the face are output as (0, 0).",
                            meshName, RGetNumberString(pIter.index()), uvSetName.asChar());
                    }
                }

                RVec2 tc(fu, 1.0f - fv); // V 座標を反転します。
                tc.SnapToZero(); // 極小な値を 0 に丸めます。

                //-----------------------------------------------------------------------------
                // すでに追加済みのテクスチャ座標から同じテクスチャ座標（誤差が許容値未満）を検索し、
                // 見つからなければ追加します。
                int iTc = -1;
                const int iPos = shapeData.m_VtxPosIdxs[iVtxTotal];
                const int searchStartIdx = searchStartIdxs[iPos]; // ここから検索します。
                if (searchStartIdx == -1)
                {
                    // 初めて出現した頂点座標なら、検索しないで検索開始インデックスを設定します。
                    searchStartIdxs[iPos] = static_cast<int>(vtxTexs.size());
                }
                else
                {
                    // すでに出現した頂点座標なら、検索開始インデックスから法線を検索します。
                    iTc = RFindValueInArray(vtxTexs, tc, R_VTX_TEX_TOLERANCE, searchStartIdx);
                }

                if (iTc == -1)
                {
                    // 重複したテクスチャ座標がなければ新しく追加します。
                    iTc = static_cast<int>(vtxTexs.size());
                    vtxTexs.push_back(tc);
                }

                // テクスチャ座標のインデックスを追加します。
                shapeData.m_VtxTexIdxss[iTex].append(iTc);
            }
        }
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief シェイプの接線を取得する UV セット名配列を設定します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in] meshUvSetNames 出力する UV の UV セット名配列です。
//! @param[in] ymodel モデルです。
//! @param[in] ynode シェイプが属するノードです。
//-----------------------------------------------------------------------------
static void SetShapeTangentUvSetNames(
    ShapeData& shapeData,
    const MStringArray& meshUvSetNames,
    YModel& ymodel,
    const YNode& ynode
)
{
    const int tanSetCount = static_cast<int>(shapeData.m_VtxTanTexIdxs.size());
    for (int iTanSet = 0; iTanSet < tanSetCount; ++iTanSet)
    {
        //-----------------------------------------------------------------------------
        // ベースシェイプのシェイプの接線を取得する UV セット名を設定します。
        // 接線に対応したテクスチャ座標が出力されない場合は
        // カレントの UV セットから取得します。
        MStringArray& tanUvSetNames = shapeData.m_TanUvSetNamess[iTanSet];
        const int baseVtxTanTexIdx = shapeData.m_VtxTanTexIdxs[iTanSet];
        const MString baseUvSetName =
            (0 <= baseVtxTanTexIdx && baseVtxTanTexIdx < shapeData.m_VtxTexCount) ?
            meshUvSetNames[baseVtxTanTexIdx] :
            MFnMesh(shapeData.m_MeshPath).currentUVSetName();
        tanUvSetNames.append(baseUvSetName);

        //-----------------------------------------------------------------------------
        // ブレンドシェイプで接線または従法線を更新するなら、
        // 各ターゲットの接線を取得する UV セット名を設定します。
        if (ynode.m_BlendShapeDataIndex != -1)
        {
            const YBlendShapeData& bsData =
                ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex];
            if (bsData.m_ShapeUpdate.m_UpdatesTan ||
                bsData.m_ShapeUpdate.m_UpdatesBin)
            {
                const int targetCount = static_cast<int>(bsData.m_Targets.size());
                for (int iTarget = 0; iTarget < targetCount; ++iTarget)
                {
                    const YBlendShapeTarget& target = bsData.m_Targets[iTarget];
                    MFnMesh targetMeshFn(target.m_ShapePath);
                    MStringArray allUvSetNames;
                    targetMeshFn.getUVSetNames(allUvSetNames);
                    // ベースシェイプと同じ名前の UV セットがあればそれを使用し、
                    // なければ現在の UV セットを使用します。
                    const MString targetUvSetName =
                        (YFindValueInArray(allUvSetNames, baseUvSetName) != -1) ?
                        baseUvSetName : targetMeshFn.currentUVSetName();
                    tanUvSetNames.append(targetUvSetName);
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプの接線と従法線を取得します。
//!
//! @param[in,out] shapeData シェイプデータです。
//! @param[in] ymodel モデルです。
//! @param[in] ynode ノードです。
//! @param[in] nodeNrms ノード中の法線の配列（オブジェクト座標系）です。
//! @param[in] nodeNrmIdxs 頂点フェースごとの nodeNrms へのインデックスです。
//! @param[in] nodeNrmOfss フェースごとの最初の頂点フェースへのインデックスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapeTangent(
    ShapeData& shapeData,
    const YModel& ymodel,
    const YNode& ynode,
    const MFloatVectorArray& nodeNrms,
    const MIntArray& nodeNrmIdxs,
    const MIntArray& nodeNrmOfss
)
{
    YScene& yscene = ymodel.GetScene();

    int zeroTanBinCount = 0; // ベースシェイプの長さ 0 接線（従法線）の数です。
    int zeroTanBinVtxIdx = -1; // ベースシェイプの長さ 0 接線（従法線）を持つ最初の頂点のインデックスです。
    std::string zeroTanBinTargetName; // 長さ 0 接線（従法線）を持つターゲットシェイプ名です。
    int zeroTanBinTargetVtxIdx = -1; // ターゲットシェイプの長さ 0 接線（従法線）を持つ最初の頂点のインデックスです。

    const int tanSetCount = static_cast<int>(shapeData.m_VtxTanTexIdxs.size());
    for (int iTanSet = 0; iTanSet < tanSetCount; ++iTanSet)
    {
        //-----------------------------------------------------------------------------
        // ブレンドシェイプで変化するならターゲットシェイプごとの接線（従法線）配列を用意します。
        std::vector<RVec4Array>& targetTanss = shapeData.m_TargetTansss[iTanSet];
        std::vector<RVec4Array>& targetBinss = shapeData.m_TargetBinsss[iTanSet];
        const bool updatesTanBin =
            shapeData.m_ShapeUpdate.m_UpdatesTan ||
            shapeData.m_ShapeUpdate.m_UpdatesBin;
        const YBlendShapeData* pBsData = (updatesTanBin && ynode.m_BlendShapeDataIndex != -1) ?
            &ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex] : NULL;
        int targetCount = 0;
        if (pBsData != NULL)
        {
            targetCount = static_cast<int>(pBsData->m_Targets.size());
            for (int iTarget = 0; iTarget < targetCount; ++iTarget)
            {
                if (shapeData.m_ShapeUpdate.m_UpdatesTan)
                {
                    targetTanss.push_back(RVec4Array());
                }
                if (shapeData.m_ShapeUpdate.m_UpdatesBin)
                {
                    targetBinss.push_back(RVec4Array());
                }
            }
        }

        //-----------------------------------------------------------------------------
        // mesh ノードから接線と従法線を取得します。
        MStringArray& tanUvSetNames = shapeData.m_TanUvSetNamess[iTanSet];
        MFnMesh meshFn(shapeData.m_MeshPath);
        MFloatVectorArray nodeTans, nodeBins; // 頂点フェースごとの接線と従法線です。
        MStatus tanStatus = meshFn.getTangents (nodeTans, MSpace::kObject, &tanUvSetNames[0]);
        MStatus binStatus = meshFn.getBinormals(nodeBins, MSpace::kObject, &tanUvSetNames[0]);
        if (!tanStatus || !binStatus)
        {
            YShowError(&yscene, // Cannot get tangents (binormals): %s (%s)
                "接線（従法線）を取得できません: {0} ({1})",
                "Cannot get tangents (binormals): {0} ({1})",
                ynode.m_OrgName, tanUvSetNames[0].asChar());
            return MS::kFailure;
        }

        //-----------------------------------------------------------------------------
        // ブレンドシェイプのための、
        // Maya の頂点インデックスに対する出力接線（従法線）インデックス配列を初期化します。
        // 出力接線（従法線）インデックス配列は頂点単位の接線（従法線）を使用するか（-1）、
        // 頂点フェース単位の接線（従法線）を使用するか（-2）のフラグも兼ねています。
        MIntArray useVtxTanIdxs;
        if (pBsData != NULL)
        {
            GetBlendShapeUseVtxTanFlags(useVtxTanIdxs, *pBsData, tanUvSetNames,
                nodeTans, nodeBins);
        }

        //-----------------------------------------------------------------------------
        // loop for face
        RVec4Array& vtxTans = shapeData.m_VtxTanss[iTanSet];
        RVec4Array& vtxBins = shapeData.m_VtxBinss[iTanSet];
        MIntArray& vtxTanIdxs = shapeData.m_VtxTanIdxss[iTanSet];
        MIntArray& vtxBinIdxs = shapeData.m_VtxBinIdxss[iTanSet];

        // 頂点座標が最初に出現した vtxTans、vtxBins 内のインデックスです。
        // 検索を高速にするために使用します。
        RIntArray tanSearchStartIdxs(shapeData.m_VtxPoss.size(), -1);
        RIntArray binSearchStartIdxs(shapeData.m_VtxPoss.size(), -1);

        int iVtxTotal = 0;
        MItMeshPolygon pIter(shapeData.m_MeshPath, shapeData.m_CompObj);
        for ( ; !pIter.isDone(); pIter.next())
        {
            //-----------------------------------------------------------------------------
            // loop for vertex / face
            const int vtxCount = pIter.polygonVertexCount();
            for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal) // Maya front is counter-clockwise
            {
                const int iNodeVtx = pIter.vertexIndex(iVtx); // mesh ノード内の頂点インデックス
                const int iNodeVtxNext = pIter.vertexIndex((iVtx + 1 < vtxCount) ? iVtx + 1 : 0);

                //-----------------------------------------------------------------------------
                // ブレンドシェイプで接線（従法線）を更新し、頂点単位の接線（従法線）を使用する場合、
                // すでに頂点単位の接線（従法線）を出力済みであれば、
                // その出力接線（従法線）インデックスを設定します。
                if (pBsData != NULL && useVtxTanIdxs[iNodeVtx] >= 0)
                {
                    vtxTanIdxs.append(useVtxTanIdxs[iNodeVtx]);
                    vtxBinIdxs.append(useVtxTanIdxs[iNodeVtx]);
                    continue;
                }

                //-----------------------------------------------------------------------------
                // 接線と従法線を配列から取得します。
                const int nodeNrmOfs = nodeNrmOfss[pIter.index()] + iVtx;
                RVec3 oNrm = GetRVec3(nodeNrms[nodeNrmIdxs[nodeNrmOfs]]);
                AdjustInvalidNormal(oNrm, shapeData.m_MeshPath, pIter.index());
                const unsigned int fvTanIdx = pIter.tangentIndex(iVtx);
                const RVec3 oTan = GetRVec3(nodeTans[fvTanIdx]);
                const RVec3 oBin = GetRVec3(nodeBins[fvTanIdx]);

                //-----------------------------------------------------------------------------
                // 接線（従法線）をスキニングモードに合わせた座標系に変換します。
                const int iVmt = shapeData.m_VtxMtxIdxs[iVtxTotal];
                const RVtxMtx& vmt = ymodel.m_VtxMtxs[iVmt];
                RVec4 tan;
                RVec4 bin;
                if (GetCoordinateAdjustedTangent(tan, bin, oTan, oBin, oNrm,
                    ymodel, ynode, vmt, shapeData.m_SkinningMode,
                    shapeData.m_MeshPath, iNodeVtx, iNodeVtxNext))
                {
                    ++zeroTanBinCount;
                    if (zeroTanBinVtxIdx == -1)
                    {
                        zeroTanBinVtxIdx = iNodeVtx;
                    }
                }

                //-----------------------------------------------------------------------------
                // すでに追加済みの接線（従法線）から同じもの（誤差が許容値未満）を検索します。
                const int iPos = shapeData.m_VtxPosIdxs[iVtxTotal];
                int iTan = -1;
                int iBin = -1;
                if (pBsData == NULL)
                {
                    const int tanSearchStartIdx = tanSearchStartIdxs[iPos]; // ここから検索します。
                    if (tanSearchStartIdx == -1)
                    {
                        // 初めて出現した頂点座標なら、検索しないで検索開始インデックスを設定します。
                        tanSearchStartIdxs[iPos] = static_cast<int>(vtxTans.size());
                    }
                    else
                    {
                        // すでに出現した頂点座標なら、検索開始インデックスから法線を検索します。
                        iTan = RFindValueInArray(vtxTans, tan, R_VTX_NRM_TOLERANCE, tanSearchStartIdx);
                    }

                    const int binSearchStartIdx = binSearchStartIdxs[iPos]; // ここから検索します。
                    if (binSearchStartIdx == -1)
                    {
                        // 初めて出現した頂点座標なら、検索しないで検索開始インデックスを設定します。
                        binSearchStartIdxs[iPos] = static_cast<int>(vtxBins.size());
                    }
                    else
                    {
                        // すでに出現した頂点座標なら、検索開始インデックスから法線を検索します。
                        iBin = RFindValueInArray(vtxBins, bin, R_VTX_NRM_TOLERANCE, binSearchStartIdx);
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 各ターゲットシェイプの接線（従法線）を追加します。
                    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
                    {
                        RVec4 targetTan(tan);
                        RVec4 targetBin(bin);
                        if (pBsData->m_IsMemberFaces[pIter.index()] != 0)
                        {
                            const YBlendShapeTarget& target = pBsData->m_Targets[iTarget];

                            MFnMesh targetMeshFn(target.m_ShapePath);
                            const MString targetUvSetName = tanUvSetNames[1 + iTarget]; // C4238 回避のため一度コピー

                            MVector mtan;
                            targetMeshFn.getFaceVertexTangent(
                                pIter.index(), iNodeVtx, mtan, MSpace::kObject, &targetUvSetName);

                            MVector mbin;
                            targetMeshFn.getFaceVertexBinormal(
                                pIter.index(), iNodeVtx, mbin, MSpace::kObject, &targetUvSetName);

                            MVector mnrm; // W 成分の計算に法線が必要なので取得します。
                            targetMeshFn.getFaceVertexNormal(
                                pIter.index(), iNodeVtx, mnrm, MSpace::kObject);
                            RVec3 oTargetNrm = GetRVec3(mnrm);
                            AdjustInvalidNormal(oTargetNrm, target.m_ShapePath, pIter.index());

                            const bool isZeroTanBinTarget = GetCoordinateAdjustedTangent(
                                targetTan, targetBin,
                                GetRVec3(mtan), GetRVec3(mbin), oTargetNrm,
                                ymodel, ynode, vmt, shapeData.m_SkinningMode,
                                target.m_ShapePath, iNodeVtx, iNodeVtxNext);

                            if (isZeroTanBinTarget)
                            {
                                if (zeroTanBinTargetName.empty())
                                {
                                    zeroTanBinTargetName = target.m_ShapePath.partialPathName().asChar();
                                    zeroTanBinTargetVtxIdx = iNodeVtx;
                                }
                            }
                        }
                        if (shapeData.m_ShapeUpdate.m_UpdatesTan)
                        {
                            targetTanss[iTarget].push_back(targetTan);
                        }
                        if (shapeData.m_ShapeUpdate.m_UpdatesBin)
                        {
                            targetBinss[iTarget].push_back(targetBin);
                        }
                    }
                }

                //-----------------------------------------------------------------------------
                // 見つからなければ接線（従法線）を追加します。
                if (iTan == -1)
                {
                    iTan = static_cast<int>(vtxTans.size());
                    vtxTans.push_back(tan);
                }

                if (iBin == -1)
                {
                    iBin = static_cast<int>(vtxBins.size());
                    vtxBins.push_back(bin);
                }

                //-----------------------------------------------------------------------------
                // 接線（従法線）インデックスをシェイプに追加します。
                vtxTanIdxs.append(iTan);
                vtxBinIdxs.append(iBin);

                //-----------------------------------------------------------------------------
                // ブレンドシェイプで接線（従法線）を更新し、頂点単位の接線（従法線）を使用する設定だった場合、
                // 出力接線（従法線）インデックスを記録します。
                if (pBsData != NULL && useVtxTanIdxs[iNodeVtx] == YBlendShapeData::PER_VTX)
                {
                    useVtxTanIdxs[iNodeVtx] = iTan; // ブレンドシェイプの場合 iTan と iBin は同じ
                }
            }
        }
    } // iTanSet

    const std::string zeroTanBinWarnCompStr =
        (zeroTanBinCount > 0) ? shapeData.m_MeshPath.partialPathName().asChar() +
            RGetNumberString(zeroTanBinVtxIdx, ".vtx[%d]") +
            RGetNumberString(zeroTanBinCount, " (%d vertex faces)") :
        (!zeroTanBinTargetName.empty()) ? zeroTanBinTargetName +
            RGetNumberString(zeroTanBinTargetVtxIdx, ".vtx[%d]") :
        "";
    if (!zeroTanBinWarnCompStr.empty())
    {
        YShowWarning(&yscene, // Zero tangent or binormal exists: %s.vtx[%d] (%d vertex faces)
            "長さが 0 の接線または従法線が存在したため、法線に直交するベクトルに置き換えて出力しました: {0} \n"
            "法線や UV 座標を修正してください",
            "A tangent or binormal vector with length 0 was found. It was replaced with a vector perpendicular to the normal and exported: {0} \n"
            "Be sure to correct the normal vector and UV coordinates.",
            zeroTanBinWarnCompStr);
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 各頂点フェースに対応する法線の mesh ノード内のインデックスを取得します。
//!
//! @param[out] array 法線インデックスを格納します。
//! @param[in] meshPath mesh ノードの DAG パスです。
//-----------------------------------------------------------------------------
static void GetVertexFaceNormalIdxs(MIntArray& array, const MDagPath& meshPath)
{
    MItMeshPolygon pIter(meshPath);
    for ( ; !pIter.isDone(); pIter.next())
    {
        const int vtxCount = pIter.polygonVertexCount();
        for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
        {
            const int iNrm = pIter.normalIndex(iVtx);
            array.append(iNrm);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプの頂点座標の座標系を調整します。
//!
//! @param[in,out] bsData ブレンドシェイプデータです。
//! @param[in] ymodel モデルです。
//! @param[in] ynode シェイプが属するノードです。
//-----------------------------------------------------------------------------
static void AdjustBlendShapePositionCoordinate(
    YBlendShapeData& bsData,
    const YModel& ymodel,
    const YNode& ynode
)
{
    if (bsData.m_ShapeUpdate.m_UpdatesPos)
    {
        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            target.m_GlobalPoss.clear();
            for (int iVtx = 0; iVtx < bsData.m_VtxCount; ++iVtx)
            {
                MPoint& pos = target.m_Poss[iVtx];
                const int iVmt = ynode.m_VtxMtxIdxs[iVtx];
                const RVtxMtx& vmt = ymodel.m_VtxMtxs[iVmt];
                MPoint globalPos;
                MPoint infPos = GetCoordinateAdjustedPosition(globalPos,
                    pos, ymodel, ynode, vmt);
                pos = infPos;
                target.m_GlobalPoss.append(globalPos);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ノードの頂点情報を取得します。
//!        頂点情報、スキンの頂点ウェイト、ブレンドシェイプの頂点情報を取得します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in,out] ynode ノードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetYNodeVertex(YModel& ymodel, YNode& ynode)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // get current pos & face size
    const YExpOpt& yopt = yscene.GetOpt();
    MDagPath meshPath = ynode.m_ShapePath;
    MFnMesh meshFn(meshPath);

    // メッシュの頂点数とポリゴン数を取得し、YNode に記録します。
    ynode.m_PosCount = meshFn.numVertices();
    ynode.m_FaceCount = meshFn.numPolygons();

    // ポリゴンが存在指定なければ関数は失敗します。
    if (ynode.m_FaceCount == 0)
    {
        YShowError(&yscene, // No polygon in mesh node: %s // 通常は発生しない
            "mesh ノードにポリゴンがありません: {0}",
            "No polygon in mesh node: {0}",
            ynode.m_OrgName);
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // get envelope
    if (ynode.m_IsSkinned)
    {
        // YNode がスキニングされたメッシュであればエンベロープ情報を変換します。
        status = GetEnvelope(ymodel, ynode);
        CheckStatus(status);
    }
    else
    {
        // スキニングしていない場合でも階層構造を表現するのにスキニング機能を使用するため、
        // 全てのメッシュに対して RVtxMtx を設定する必要があります。
        // そのため、YNode のインデックス(m_OrgIndex)を行列インデックスとしてダミーの
        // RVtxMtx を ymodel.m_VtxMtxs に追加し、追加した RVtxMtx のインデックスを全頂点に設定します。
        const int iVmt = AppendVtxMtx(ymodel, RVtxMtx(ynode.m_OrgIndex));
        for (int iVtx = 0; iVtx < ynode.m_PosCount; ++iVtx)
        {
            ynode.m_VtxMtxIdxs.append(iVmt);
        }
        ynode.m_RigidBody = true;
    }

    //-----------------------------------------------------------------------------
    // set deformer flag & check component size
    if (ynode.m_DeformerFlag)
    {
        // 出力するシェイプとオリジナルのシェイプで頂点とポリゴンの数が異なる場合はエラーとします。
        MFnMesh orgMeshFn(ynode.m_OrgShapePath);
        int orgPosCount = orgMeshFn.numVertices();
        if (orgPosCount != ynode.m_PosCount)
        {
            // スキニングまたはブレンドシェイプ使用時に発生
            YShowError(&yscene, // Vertex number is different from original mesh: %s
                "頂点数を変更する操作の結果、スキニングまたはブレンドシェイプを設定したオブジェクトの頂点数が"
                "オリジナル mesh ノード（～Orig）の頂点数と異なります: {0} \n"
                "スキニングの場合は、一度デタッチしてヒストリを削除してからバインドしなおすか、"
                "「編集 > 種類ごとに削除 > デフォーマ以外のヒストリ」を選択してください。\n"
                "       オリジナル = {1}, 現在 = {2}",
                "The number of vertices of an object for which skinning or blend shape animation is set differs "
                "from the number of vertices of the original mesh node (~Orig) as a result of an operation that changed the number of vertices: {0} \n"
                "For skinning, either rebind after detaching and deleting the history, "
                "or select Edit > Delete by Type > Non-Deformer History. \n"
                "       original = {1}, current = {2}",
                ynode.m_OrgName,
                RGetNumberString(orgPosCount), RGetNumberString(ynode.m_PosCount));
            return MS::kFailure;
        }
        int orgFaceCount = orgMeshFn.numPolygons();
        if (orgFaceCount != ynode.m_FaceCount && ynode.m_BlendShapeFlag)
        {
            // ブレンドシェイプ使用時のみ発生
            YShowError(&yscene, // Face number is different from original mesh: %s
                "ブレンドシェイプを設定したオブジェクトのフェース数がオリジナル mesh ノード（～Orig）のフェース数と異なっています: {0} \n"
                "       オリジナル = {1}, 現在 = {2}",
                "An object for which a blend shape has been set has a different number of faces than the original mesh node (~Orig): {0} \n"
                "       original = {1}, current = {2}",
                ynode.m_OrgName,
                RGetNumberString(orgFaceCount), RGetNumberString(ynode.m_FaceCount));
            return MS::kFailure;
        }
    }

    //-----------------------------------------------------------------------------
    // get original position & normal
    //-----------------------------------------------------------------------------

    //-----------------------------------------------------------------------------
    // disable deformer
    MIntArray deformerStates;
    // fmd ファイルにバインドポーズ状態のシェイプを出力する場合は
    // 一時的にデフォーマを無効にします。
    // Maya ではメッシュが変形済みの頂点座標を返すため、スキニングやモーフィングの
    // ソースメッシュとして変形前の状態を取得するために必要な処理です。
    if (yopt.m_FmdIsBindPose && ynode.m_DeformerFlag)
    {
        DisableDeformerStates(deformerStates, ynode.m_DeformerObjs);
        meshFn.setObject(meshPath); // ヒストリー変更後に必要
    }

    //-----------------------------------------------------------------------------
    // 頂点座標を取得して Magnify を掛けます。
    meshFn.getPoints(ynode.m_OrgPoss, MSpace::kObject);
    ScaleArray(ynode.m_OrgPoss, yopt.m_InternalMagnify);

    // 法線を取得します。
    meshFn.getNormals(ynode.m_OrgNrms, MSpace::kObject);

    GetVertexFaceNormalIdxs(ynode.m_OrgNrmIdxs, meshPath);

    //-----------------------------------------------------------------------------
    // 一時的に無効にしていたデフォーマの設定を元に戻します。
    if (yopt.m_FmdIsBindPose && ynode.m_DeformerFlag)
    {
        RestoreDeformerStates(ynode.m_DeformerObjs, deformerStates);
        meshFn.setObject(meshPath); // ヒストリー変更後に必要
    }

    //-----------------------------------------------------------------------------
    // get current position
    //-----------------------------------------------------------------------------

    //-----------------------------------------------------------------------------
    // 変形後の頂点座標を取得します。
    meshFn.getPoints(ynode.m_CurPoss, MSpace::kObject);
    ScaleArray(ynode.m_CurPoss, yopt.m_InternalMagnify);
    // デフォーマが設定されていメッシュをバインドポーズで出力する場合は
    // m_CurPoss に m_OrgPoss を上書きします。
    if (yopt.m_FmdIsBindPose && ynode.m_DeformerFlag)
    {
        ynode.m_CurPoss = ynode.m_OrgPoss;
    }

    //-----------------------------------------------------------------------------
    // トランスフォームにスケールピボットが設定されている場合は、
    // そのピボットが原点になるように頂点座標を調整します。
    int iVtx;
    for (iVtx = 0; iVtx < ynode.m_PosCount; ++iVtx)
    {
        ynode.m_OrgPoss[iVtx] -= ynode.m_ScalePivot;
        ynode.m_CurPoss[iVtx] -= ynode.m_ScalePivot;
    }

    //-----------------------------------------------------------------------------
    // ワールド座標系とボーンにより変形された頂点座標を求めます。
    // これはモデル(YModel)に設定されているシーンのバウンディングボリューム
    // (m_GlobalPosMin, m_GlobalPosMax) を求めるために必要となります。
    for (iVtx = 0; iVtx < ynode.m_PosCount; ++iVtx)
    {
        // この頂点を変換する頂点ウェイトと行列インデックスを求めます。
        const int iVmt = ynode.m_VtxMtxIdxs[iVtx];
        const RVtxMtx& vmt = ymodel.m_VtxMtxs[iVmt];

        // ワールド座標系の頂点座標(globalPos)とボーンによって変形された頂点座標
        // (infPos)を計算して求めます。
        MPoint globalPos;
        MPoint infPos = GetCoordinateAdjustedPosition(globalPos,
            ynode.m_CurPoss[iVtx], ymodel, ynode, vmt);

        // スキニングモードによって中間ファイルに出力する頂点の座標系が変わるため
        // この二つの頂点座標を保存しておく
        ynode.m_GlobalPoss.append(globalPos);
        ynode.m_InfPoss.append(infPos);

        // バウンディングボリュームを求めるための頂点座標を収集します。
        ymodel.m_GlobalPoss.append(globalPos);
    }

    //-----------------------------------------------------------------------------
    // 変形後の頂点座標を取得します。
    meshFn.getNormals(ynode.m_CurNrms, MSpace::kObject);
    // デフォーマが設定されていメッシュをバインドポーズで出力する場合は
    // m_CurNrms に m_OrgNrms を上書きします。
    if (yopt.m_FmdIsBindPose && ynode.m_DeformerFlag)
    {
        ynode.m_CurNrms = ynode.m_OrgNrms;
    }

    //-----------------------------------------------------------------------------
    // ブレンドシェイプデータを取得します。
    if (ynode.m_BlendShapeFlag)
    {
        YBlendShapeData bsData;
        bool isValid;
        status = GetBlendShapeData(bsData, isValid, yscene,
            ynode.m_Name, ynode.m_ShapePath, ynode.m_BlendShapeObj,
            ynode.m_OrgPoss, ynode.m_ScalePivot, ynode.m_MeshColSets, yopt);
        CheckStatus(status);
        // ブレンドシェイプが有効なら内部の頂点の座標系を変換し、
        // ymodel.m_BlendShapeDatas 配列に追加します。
        if (isValid)
        {
            AdjustBlendShapePositionCoordinate(bsData, ymodel, ynode);
            ynode.m_BlendShapeDataIndex = static_cast<int>(ymodel.m_BlendShapeDatas.size());
            ymodel.m_BlendShapeDatas.push_back(bsData);
        }
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 頂点シェイプアニメーション配列を設定します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetVertexShapeAnims(YModel& ymodel)
{
    const int shapeDataCount = static_cast<int>(ymodel.m_pOutShapeDatas.size());
    for (int iShape = 0; iShape < shapeDataCount; ++iShape)
    {
        const ShapeData& shapeData = *ymodel.m_pOutShapeDatas[iShape];
        if (shapeData.m_ShapeUpdate.UpdatesSome())
        {
            const YNode& ynode = *ymodel.m_pOutYNodes[shapeData.m_NodeIndexes[0]];
            const YBlendShapeData& bsData = ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex];
            if (bsData.m_OutTargetAnimCount != 0)
            {
                YVertexShapeAnim vertexShapeAnim(shapeData.m_Name, ynode.m_BlendShapeDataIndex);
                ymodel.m_VertexShapeAnims.push_back(vertexShapeAnim);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプの頂点カラー属性とユーザー頂点属性の成分数を取得します。
//!
//! @param[out] shapeData シェイプデータです。
//! @param[in] ynode シェイプが属するノードです。
//! @param[in] comp フェース群のコンポーネントです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShapeVertexAttrInfo(
    ShapeData& shapeData,
    const YNode& ynode,
    MObject& comp
)
{
    //-----------------------------------------------------------------------------
    // シェイプのフェース群に頂点カラー属性が設定されていれば情報を取得します。
    for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
    {
        const MString& colSet = ynode.m_MeshColSets[iCol];
        if (colSet.length() != 0 &&
            ComponentVertexColorExists(ynode.m_ShapePath, comp, colSet))
        {
            YVtxAttrInfo& colInfo = shapeData.m_VtxColInfos[iCol];
            colInfo.m_ExistFlag = true;
            colInfo.m_CompCount = GetColorSetComponentCount(&colInfo.m_IsRa, ynode.m_ShapePath, colSet);
        }
    }

    //-----------------------------------------------------------------------------
    // シェイプのフェース群にユーザー頂点属性が設定されていれば情報を取得します。
    for (int iUa = 0; iUa < RPrimVtx::VTX_USR_MAX; ++iUa)
    {
        const MString& colSet = ynode.m_MeshUsrColSets[iUa];
        if (colSet.length() != 0 &&
            ComponentVertexColorExists(ynode.m_ShapePath, comp, colSet))
        {
            YVtxAttrInfo& usrInfo = shapeData.m_VtxUsrInfos[iUa];
            usrInfo.m_ExistFlag = true;
            usrInfo.m_CompCount = GetColorSetComponentCount(&usrInfo.m_IsRa, ynode.m_ShapePath, colSet);
        }
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief シェイプの頂点要素を取得します。
//!        mesh ノードに適用されているシェーディンググループごとに
//!        フェース群をグループ化し、シェイプとして登録します。
//!        そして、シェイプに頂点属性の要素とインデックスを設定します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] ynode シェイプが属するノードです。
//! @param[in] getsMaterial マテリアルを取得するなら true を指定します。
//! @param[in] getsGeometry ジオメトリを取得するなら true を指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetYNodeShape(
    YModel& ymodel,
    const YNode& ynode,
    const bool getsMaterial,
    const bool getsGeometry
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // デフォーマの nodeState アトリビュートを変更した直後は
    // 正しい接線（従法線）を取得できません（Maya の不具合）。
    // 1 度法線の数を取得することによって Maya 内部の接線（従法線）情報を更新します。
    MDagPath meshPath = ynode.m_ShapePath;
    if (getsGeometry)
    {
        const int nrmCount = MFnMesh(meshPath).numNormals();
        R_UNUSED_VARIABLE(nrmCount);
    }

    //-----------------------------------------------------------------------------
    // ポリゴンの法線のインデックス情報を取得します。
    int nodeNrmCount = 0; // ノードの全法線数です。全フェースの頂点数を足しあわせた数になります。
    MIntArray nodeNrmOfss; // 各フェースの法線情報が全法線の何番目かを表すオフセット配列です。
    for (MItMeshPolygon pIter(meshPath); !pIter.isDone(); pIter.next())
    {
        nodeNrmOfss.append(nodeNrmCount);
        nodeNrmCount += pIter.polygonVertexCount();
    }

    //-----------------------------------------------------------------------------
    // シェーディンググループの数だけループして各シェーディンググループごとに
    // 頂点情報とポリゴン情報を ShapeData に設定します。

    // メッシュが使用するシェーディンググループの数を取得します。
    const int sgCount = static_cast<int>(ynode.m_SGObjs.size());
    if (sgCount == 0)
    {
        YShowError(&yscene, // No shading group: %s
            "オブジェクトにシェーディンググループが接続されていません。マテリアルを割り当ててください: {0}",
            "A shading group has not been connected to the object. Make sure that you allocate a material: {0}",
            ynode.m_OrgName);
        return MS::kFailure;
    }

    for (int iSg = 0; iSg < sgCount; ++iSg)
    {
        //-----------------------------------------------------------------------------
        // シェーディンググループのオブジェクトと、そのシェーディンググループを使用します。
        // ポリゴンのコンポーネント情報(シェーディンググループを使用するポリゴンの
        // インデックス配列オブジェクト)を YNode から取得します。
        const MObject sgObj = ynode.m_SGObjs[iSg];
        MObject comp = ynode.m_CompObjs[iSg]; // const にすると MItMeshPolygon で使用できない

        //-----------------------------------------------------------------------------
        // ポリゴンデータを作成します。

        // シェイプデータを作成し、初期情報を設定します。
        ShapeData shapeData;
        shapeData.m_OrgName  = ynode.m_OrgName;
        shapeData.m_MeshPath = meshPath;
        shapeData.m_CompObj  = comp;
        shapeData.m_PrimType = ynode.m_PrimType;

        // シェイプデータが作成された順番でインデックス付けを行います。
        shapeData.m_OrgIndex = static_cast<int>(ymodel.m_ShapeDatas.size());
        shapeData.m_Index    = shapeData.m_OrgIndex;
        shapeData.m_NodeIndexes.append(ynode.m_OrgIndex);

        //-----------------------------------------------------------------------------
        // get vertex size & pos volume & status & check holed polygon
        if (getsGeometry)
        {
            MPointArray polyPosArray; // ポリゴンの頂点座標を集めた配列です。
            shapeData.m_TriangleCount = 0;

            // シェーディンググループに属するポリゴンのみでバウンディングボリューム、
            // 頂点数、三角形数を求め、同時に穴あきポリゴンでないかチェックを行います。
            MItMeshPolygon pIter(meshPath, comp);
            for ( ; !pIter.isDone(); pIter.next())
            {
                const int vtxCount = pIter.polygonVertexCount();
                shapeData.m_VtxCounts.append(vtxCount);
                shapeData.m_TriangleCount += vtxCount - 2;

                // ポリゴンに穴が開いていないか確認。
                // 穴が開いているポリゴンがあればエラーとします。
                if (pIter.isHoled())
                {
                    YShowError(&yscene, // Holed polygon exists: (please triangulate): %s.f[%d]
                        "穴の空いたポリゴンが存在します。Maya 上で三角形分割してから出力してください: {0}.f[{1}]",
                        "A polygon with holes in it exists. Divide the polygon into triangles under Maya, and then export: {0}.f[{1}]",
                        ynode.m_OrgName, RGetNumberString(pIter.index()));
                    return MS::kFailure;
                }

                // メッシュの頂点座標を取得します。
                // ボリュームはスキン使用時でも
                // 本来属するノードのローカル座標で出力します。
                for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
                {
                    polyPosArray.append(ynode.m_CurPoss[pIter.vertexIndex(iVtx)]);
                }
            }
            // 取得した頂点座標配列からバウンディングボリュームを求めます。
            GetArrayRange(polyPosArray, shapeData.m_VolumeMin, shapeData.m_VolumeMax);
            // polyPosArray の中で原点から一番遠い座標と原点との距離を求めます。
            shapeData.m_VolumeR = GetPointArrayMaxLength(polyPosArray);
        }

        //-----------------------------------------------------------------------------
        // マテリアルを取得します。
        MStringArray meshUvSetNames; // マテリアルが圧縮された場合でも個別です。
        YMaterial* pMat = NULL;
        std::string matName;
        if (getsMaterial || getsGeometry)
        {
            status = GetMaterial(ymodel, shapeData.m_MaterialIndex, shapeData.m_Priority,
                meshUvSetNames, shapeData.m_UvAttribNames, shapeData.m_UvHintIdxs,
                sgObj, meshPath, ynode.m_DefaultUvSet);
            CheckStatus(status);
            YMaterial& mat = ymodel.m_YMaterials[shapeData.m_MaterialIndex];
            pMat = &mat;
            matName = mat.m_Name;

            //-----------------------------------------------------------------------------
            // シェイプデータのマテリアル関連パラメータを設定します。
            shapeData.m_OrgName = ynode.m_OrgName + "__" + mat.m_OrgName;
            shapeData.m_VtxNrmFlag    = mat.m_VtxNrmFlag;
            shapeData.m_VtxTanTexIdxs = mat.m_VtxTanTexIdxs;
            shapeData.m_VtxTexCount   = meshUvSetNames.length();
            //shapeData.m_VtxColInfos  = mat.m_MatVtxColInfos;
            //shapeData.m_VtxUsrInfos  = mat.m_MatVtxUsrInfos;

            if (!shapeData.m_VtxTanTexIdxs.empty())
            {
                SetShapeTangentUvSetNames(shapeData, meshUvSetNames, ymodel, ynode);
            }

            // 頂点カラーとユーザー頂点属性の情報を取得します。
            GetShapeVertexAttrInfo(shapeData, ynode, comp);
        }

        //-----------------------------------------------------------------------------
        // シェイプアニメーションにより変形させる要素を取得します。
        shapeData.m_ShapeUpdate.DisableAll();
        if (getsGeometry && ynode.m_BlendShapeDataIndex != -1)
        {
            // ブレンドシェイプデータの頂点属性更新情報をコピーします。
            const YBlendShapeData& bsData =
                ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex];
            shapeData.m_ShapeUpdate = bsData.m_ShapeUpdate;

            // シェイプに頂点カラーが存在しなければ頂点カラーの更新フラグを false にします。
            for (int iColSet = 0; iColSet < static_cast<int>(shapeData.m_VtxColInfos.size()); ++iColSet)
            {
                if (!shapeData.m_VtxColInfos[iColSet].m_ExistFlag)
                {
                    shapeData.m_ShapeUpdate.m_UpdatesCols[iColSet] = false;
                }
            }

            // シェイプのフェース群の頂点属性がブレンドシェイプで変化するか判定して
            // 頂点属性更新情報を設定します。
            SetBlendShapeUpdateForShape(shapeData.m_ShapeUpdate,
                bsData, comp, shapeData.m_TanUvSetNamess, false);
        }

        //-----------------------------------------------------------------------------
        // シェイプの頂点座標と行列インデックスを設定します。
        status = GetShapePosMtxIdx(shapeData, ymodel, ynode);
        CheckStatus(status);

        //-----------------------------------------------------------------------------
        // get vertex element
        if (getsGeometry)
        {
            //-----------------------------------------------------------------------------
            // 頂点カラーが設定されていたら頂点カラーを取得します。
            if (SomeVtxAttrExists(shapeData.m_VtxColInfos))
            {
                bool xluVtxFlag;
                GetShapeVtxColor(shapeData, xluVtxFlag, ymodel, ynode);
                // マテリアルが複数のシェイプに割り当てられていて
                // いずれかのシェイプの頂点カラーのみ透明／半透明である場合に
                // 対応するためにここで再設定
                if (pMat != NULL && pMat->m_AutoRenderStateFlag && xluVtxFlag)
                {
                    pMat->SetRenderStateMode(RRenderState::XLU);
                }
            }

            //-----------------------------------------------------------------------------
            // テクスチャ座標を取得します。
            if (shapeData.m_VtxTexCount > 0)
            {
                status = GetShapeTexCoord(shapeData, yscene, meshUvSetNames);
                CheckStatus(status);
            }

            //-----------------------------------------------------------------------------
            // 法線を取得します。
            if (shapeData.m_VtxNrmFlag)
            {
                status = GetShapeNormal(shapeData, ymodel, ynode, ynode.m_CurNrms,
                    ynode.m_OrgNrmIdxs, nodeNrmOfss);
                CheckStatus(status);
            }

            //-----------------------------------------------------------------------------
            // 接線と従法線を取得します。
            if (!shapeData.m_VtxTanTexIdxs.empty())
            {
                status = GetShapeTangent(shapeData, ymodel, ynode, ynode.m_CurNrms,
                    ynode.m_OrgNrmIdxs, nodeNrmOfss);
                CheckStatus(status);
            }

            //-----------------------------------------------------------------------------
            // check normal mapping vertex weight
            // TODO: オブジェクト空間法線マップの場合のみ警告するように修正
            //if (!mat.m_NrmMapIdxs.empty() && ynode.m_IsSkinned)
            //{
            //  if (!IsFullWeightAtOneNode(shapeData))
            //  {
            //      YShowWarning(&yscene, "", "Normal mapping polygon's vertex weight is wrong: {0} ({1})", // 現在は発生しない
            //          ynode.m_OrgName, mat.m_OrgName);
            //  }
            //}

            //-----------------------------------------------------------------------------
            // ユーザー頂点属性を取得します。
            if (SomeVtxAttrExists(shapeData.m_VtxUsrInfos))
            {
                GetShapeUserAttr(shapeData, ynode);
            }
        }

        //-----------------------------------------------------------------------------
        // シェイプを ymodel.m_ShapeDatas に追加します。
        ymodel.m_ShapeDatas.push_back(shapeData);
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief シェイプを取得します。
//!        シーングラフ内に存在するメッシュの頂点とインデックス情報を取得します。
//!        同時にマテリアル情報も変換します。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetShape(YModel& ymodel)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    //RTimeMeasure tm1;

    //-----------------------------------------------------------------------------
    // get ynode vertex
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    if (nodeCount == 0)
    {
        return MS::kSuccess;
    }
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];
        // YNode がメッシュであった場合に、頂点情報を変換します。
        // 同時にスキンのウェイト情報とブレンドシェイプの頂点情報も変換します。
        if (ynode.m_OrgKind == YNode::KindMesh)
        {
            status = GetYNodeVertex(ymodel, ynode);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // シーンのマテリアル情報を変換します。
    const bool getsMaterial =
        yopt.ExportsModelFile()    ||
        yopt.ExportsTexture()      ||
        yopt.ExportsMaterialAnim() ||
        yopt.ExportsShapeAnim();
    if (getsMaterial)
    {
        status = GetMatInfos(ymodel);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // 各ノードのシェイプを取得します。
    const bool getsGeometry = yopt.ExportsModelFile() || yopt.ExportsShapeAnim();
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pYNodes[iNode];
        if (ynode.m_OrgKind == YNode::KindMesh)
        {
            //-----------------------------------------------------------------------------
            // fmd ファイルにバインドポーズ状態のシェイプを出力する場合は
            // 一時的にデフォーマを無効にします。
            // Maya ではメッシュが変形済みの頂点座標を返すため、スキニングやモーフィングの
            // ソースメッシュとして変形前の状態を取得するために必要な処理です。

            MIntArray deformerStates; // 後で状態を戻すために保存したデフォーマのステートです。
            // デフォーマが適用されており、データをバインドポーズ状態で出力するかどうかをチェックします。
            const bool deformerDisableFlag =
                (yopt.m_FmdIsBindPose && ynode.m_DeformerFlag);
            // デフォーマを無効化します。
            if (deformerDisableFlag)
            {
                DisableDeformerStates(deformerStates, ynode.m_DeformerObjs);
            }

            //-----------------------------------------------------------------------------
            // シェイプデータを取得します。
            status = GetYNodeShape(ymodel, ynode, getsMaterial, getsGeometry);

            //-----------------------------------------------------------------------------
            // 無効化したデフォーマの設定を復帰させます。
            if (deformerDisableFlag)
            {
                RestoreDeformerStates(ynode.m_DeformerObjs, deformerStates);
            }

            //-----------------------------------------------------------------------------
            // check status
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // シーンのグローバルなボリュームを取得します。
    GetArrayRange(ymodel.m_GlobalPoss, ymodel.m_GlobalPosMin, ymodel.m_GlobalPosMax);
    //cerr << "get shape time: " << tm1.GetMilliSec() << endl;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ノードのキーアニメーションを取得します。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetYNodeKeyAnim(YModel& ymodel)
{
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int nodeIdx = 0; nodeIdx < nodeCount; ++nodeIdx)
    {
        YNode& ynode = *ymodel.m_pYNodes[nodeIdx];
        const float baseValues[RBone::PARAM_COUNT] = // スケルタルアニメーションの初期値配列です。
        {
            static_cast<float>(RSnapToZero(ynode.m_OutBindS.x)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindS.y)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindS.z)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindR.x)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindR.y)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindR.z)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindT.x)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindT.y)),
            static_cast<float>(RSnapToZero(ynode.m_OutBindT.z)),
        };
        for (int paramIdx = 0; paramIdx < RBone::PARAM_COUNT; ++paramIdx)
        {
            //-----------------------------------------------------------------------------
            // アニメーションカーブ使用フラグ（現在は未使用）を設定します。
            YAnimCurve& curve = ynode.m_Anims[paramIdx];
            const ChannelInput& chan = ynode.m_AnimChans[paramIdx];
            curve.UpdateUseFlag(baseValues[paramIdx]);
            if (!ynode.m_SkinInfluenceFlag)
            {
                // スキンに影響しないノードの場合、チャンネル入力が有効または
                // IK が設定されていればカーブ使用フラグを ON にします（IK は回転のみ）。
                if (chan.IsEffective() ||
                    (ynode.m_IkControlledFlag && RBone::IsRotate(paramIdx)))
                {
                    curve.m_UseFlag = true;
                }
            }

            //-----------------------------------------------------------------------------
            // アニメーションカーブデータを取得または作成します。
            if (!curve.m_ConstantFlag)
            {
                if (!curve.m_Baked)
                {
                    curve.GetKeys(chan.m_CurveObj, ynode.m_AnimScales[paramIdx], ynode.m_AnimOfss[paramIdx], yopt);
                }
                else
                {
                    #ifdef ADJUST_YNODE_ROT_XYZ_SW
                    // XYZ 3 軸同時に調整しているので単一軸の調整は不要
                    const bool continuousAngleFlag = false;
                    #else
                    const bool continuousAngleFlag = true;
                    #endif
                    curve.MakeKeys(GetFloatFrameFromSubFrame, &yopt, continuousAngleFlag);
                }
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるカラーアニメーション値を分析します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus AnalyzeColorFullAnim(YModel& ymodel, const YExpOpt& yopt)
{
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
        if (mat.m_ColAnimFlag)
        {
            for (int paramIdx = 0; paramIdx < RMaterial::COLOR_PARAM_COUNT; ++paramIdx)
            {
                YAnimCurve& curve = mat.m_Anims[paramIdx];
                if (curve.m_UseFlag)
                {
                    curve.m_Name      = mat.m_Name + "." + RMaterial::GetColorParamName(paramIdx);
                    curve.m_LoopFlag  = yopt.m_LoopAnim;
                    curve.m_AngleFlag = false;
                    curve.m_Tolerance = yopt.m_TolC;
                    curve.UpdateConstantFlag();
                }
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief カラーのキーアニメーションを取得します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetColorKeyAnim(YModel& ymodel, const YExpOpt& yopt)
{
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
        if (mat.m_ColAnimFlag)
        {
            for (int paramIdx = 0; paramIdx < RMaterial::COLOR_PARAM_COUNT; ++paramIdx)
            {
                YAnimCurve& curve = mat.m_Anims[paramIdx];
                if (curve.m_UseFlag && !curve.m_ConstantFlag)
                {
                    //-----------------------------------------------------------------------------
                    // check get key OK
                    bool forceMakeKey = yopt.m_BakeAllAnim;
                    float valueScale = 1.0f;
                    float valueOfs = 0.0f;
                    if (RMaterial::IsDiffuse(paramIdx))
                    {
                        valueScale = mat.m_DiffuseCoeff;
                    }
                    else if (RMaterial::IsOpacity(paramIdx) && mat.m_UseTransparencyPlug)
                    {
                        // opacity = 1 - transparency
                        valueScale = -1.0f;
                        valueOfs   =  1.0f;
                    }

                    const ChannelInput& chan = mat.m_AnimChans[paramIdx];
                    bool getKeyFlag = (
                        !forceMakeKey &&
                        chan.m_CurveFlag &&
                        !chan.m_CurveWeighted &&
                        IsValidAnimCurveValue(chan.m_CurveObj,
                            curve.m_FullValues, valueScale, valueOfs, yopt));

                    if (getKeyFlag)
                    {
                        //-----------------------------------------------------------------------------
                        // get keys (material)
                        curve.GetKeys(chan.m_CurveObj, valueScale, valueOfs, yopt);
                    }
                    else
                    {
                        //-----------------------------------------------------------------------------
                        // make keys (material)
                        //cerr << "make keys: " << curve.m_Name << endl;
                        curve.MakeKeys(GetFloatFrameFromSubFrame, &yopt, true);
                    }
                }
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief スケルタルアニメーション値に適用するスケールとオフセットを設定します。
//!
//! @param[in,out] pYNode ノードへのポインターです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void GetYNodeAnimScaleOffset(YNode* pYNode, const YExpOpt& yopt)
{
    float* valueScales = pYNode->m_AnimScales;
    float* valueOfss   = pYNode->m_AnimOfss;

    valueScales[RBone::SCALE_X] =
    valueScales[RBone::SCALE_Y] =
    valueScales[RBone::SCALE_Z] = 1.0f;
    valueOfss[RBone::SCALE_X] =
    valueOfss[RBone::SCALE_Y] =
    valueOfss[RBone::SCALE_Z] = 0.0f;

    valueScales[RBone::ROTATE_X] =
    valueScales[RBone::ROTATE_Y] =
    valueScales[RBone::ROTATE_Z] = static_cast<float>(R_M_RAD_TO_DEG);
    valueOfss[RBone::ROTATE_X] =
    valueOfss[RBone::ROTATE_Y] =
    valueOfss[RBone::ROTATE_Z] = 0.0f;

    valueScales[RBone::TRANSLATE_X] =
    valueScales[RBone::TRANSLATE_Y] =
    valueScales[RBone::TRANSLATE_Z] = static_cast<float>(yopt.m_InternalMagnify);
    valueOfss[RBone::TRANSLATE_X] = static_cast<float>(pYNode->m_TranslateOffset[0]);
    valueOfss[RBone::TRANSLATE_Y] = static_cast<float>(pYNode->m_TranslateOffset[1]);
    valueOfss[RBone::TRANSLATE_Z] = static_cast<float>(pYNode->m_TranslateOffset[2]);
}

//-----------------------------------------------------------------------------
//! @brief ノードについて Maya 上のキーを取得可能か判定します。
//!
//! @param[in] ynode ノードです。
//! @param[in] paramIdx パラメーターインデックスです。
//! @param[in] chan チャンネル入力情報です。
//! @param[in] fullValues ベイクしたデータです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return Maya 上のキーを取得可能なら true を返します。
//-----------------------------------------------------------------------------
static bool GetsYNodeMayaKey(
    const YNode& ynode,
    const int paramIdx,
    const ChannelInput& chan,
    const RFloatArray& fullValues,
    const YExpOpt& yopt
)
{
    if (yopt.m_BakeAllAnim ||
        !chan.m_CurveFlag ||
        chan.m_CurveWeighted ||
        //ynode.m_IkControlledFlag ||
        //ynode.m_SpecialConstraintFlag ||
        //ynode.m_XformLimitFlag[paramIdx] ||
        ynode.m_MergeCount > 0)
    {
        return false;
    }

    if (RBone::IsRotate(paramIdx))
    {
        if (ynode.m_RotateAdjustFlag || IsQuaternionAnimCurve(chan.m_CurveObj))
        {
            return false;
        }
    }
    else if (RBone::IsTranslate(paramIdx))
    {
        if (ynode.m_PivotMiddleOffsetFlag)
        {
            return false;
        }
    }
    return IsValidAnimCurveValue(chan.m_CurveObj, fullValues,
        ynode.m_AnimScales[paramIdx], ynode.m_AnimOfss[paramIdx], yopt);
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるスケルタルアニメーション値を分析します。
//!        回転の角度が連続的になるように調整します。
//!        アニメーション値が一定か判定します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus AnalyzeYNodeFullAnim(YModel& ymodel, const YExpOpt& yopt)
{
    static const char* const XformAttrNames[RBone::PARAM_COUNT] =
    {
        "sx", "sy", "sz",
        "rx", "ry", "rz",
        "tx", "ty", "tz",
    };

    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int nodeIdx = 0; nodeIdx < nodeCount; ++nodeIdx)
    {
        //-----------------------------------------------------------------------------
        // スケルタルアニメーションの各パラメーターについてアニメーション値が一定か判定し、
        // Maya 上のキーを取得可能か判定します。
        YNode& ynode = *ymodel.m_pYNodes[nodeIdx];
        YAnimCurve* curves = ynode.m_Anims;
        GetYNodeAnimScaleOffset(&ynode, yopt);
        for (int paramIdx = 0; paramIdx < RBone::PARAM_COUNT; ++paramIdx)
        {
            YAnimCurve& curve = curves[paramIdx];
            const bool isRotate = RBone::IsRotate(paramIdx);
            curve.m_Name = ynode.m_Name + "." + RBone::GetParamName(paramIdx);
            curve.m_LoopFlag = yopt.m_LoopAnim;
            curve.m_AngleFlag = isRotate;
            curve.m_Tolerance =
                (RBone::IsScale(paramIdx)) ? yopt.m_TolS :
                (isRotate                ) ? yopt.m_TolR :
                static_cast<float>(yopt.m_TolT * yopt.m_Magnify);
                // ↑ m_TolT は Magnify を掛ける前の値に対する許容値なので Magnify を掛けます。
            curve.UpdateConstantFlag();

            ChannelInput& chan = ynode.m_AnimChans[paramIdx];
            if (!ynode.m_AddedFlag)
            {
                const MPlug plug = MFnDagNode(ynode.m_XformPath).findPlug(XformAttrNames[paramIdx]);
                chan.GetInput(plug);
            }
            curve.m_Baked = (curve.m_ConstantFlag ||
                !GetsYNodeMayaKey(ynode, paramIdx, chan, curve.m_FullValues, yopt));
        }

        //-----------------------------------------------------------------------------
        // 回転 XYZ のいずれも Maya 上のキーを取得しない場合、
        // ベイクしたデータの回転の角度が連続的になるように調整します。
        #ifdef ADJUST_YNODE_ROT_XYZ_SW
        const bool hasRotateAnim =
            !curves[RBone::ROTATE_X].m_ConstantFlag ||
            !curves[RBone::ROTATE_Y].m_ConstantFlag ||
            !curves[RBone::ROTATE_Z].m_ConstantFlag;
        if (hasRotateAnim &&
            curves[RBone::ROTATE_X].m_Baked &&
            curves[RBone::ROTATE_Y].m_Baked &&
            curves[RBone::ROTATE_Z].m_Baked)
        {
            if (RMakeContinuousXyzAngleArray(
                curves[RBone::ROTATE_X].m_FullValues,
                curves[RBone::ROTATE_Y].m_FullValues,
                curves[RBone::ROTATE_Z].m_FullValues))
            {
                curves[RBone::ROTATE_X].UpdateConstantFlag();
                curves[RBone::ROTATE_Y].UpdateConstantFlag();
                curves[RBone::ROTATE_Z].UpdateConstantFlag();
                //cerr << "rotate anim adjusted: " << ynode.m_OrgName << R_ENDL;
            }
        }
        #endif
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief merge ynode transform
//-----------------------------------------------------------------------------
static void MergeYNodeTransform(const YNode& ynode,
    MVector& s, MVector& r, MVector& t)
{
    if (ynode.m_MergeCount == 0)
    {
        return;
    }

    MMatrix mtx = ynode.m_CalcLocalMtx;
    YNode* parent = ynode.m_pParent;    // m_pOutParent ではない
    for (int iparent = 0; iparent < ynode.m_MergeCount; ++iparent)
    {
        if (parent == NULL)
        {
            break;
        }
        mtx *= parent->m_CalcLocalMtx;
        parent = parent->m_pParent;
    }

    RVec3 rs, rr, rt;
    GetRMtx44(mtx).GetTransform(rs, rr, rt);
    s = GetMVector(rs);
    r = GetMVector(rr);
    t = GetMVector(rt);

    //MTransformationMatrix transMtx(mtx);
    //
    //double sb[3];
    //transMtx.getScale(sb, MSpace::kTransform);
    //s = MVector(sb);
    //
    //double rb[3];
    //MTransformationMatrix::RotationOrder rOrder;
    //transMtx.getRotation(rb, rOrder);
    //r = MVector(rb);
    //r *= R_M_RAD_TO_DEG;

    //t = transMtx.getTranslation(MSpace::kTransform);
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるカラーアニメーション値を取得します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void GetColorFullAnimValue(YModel& ymodel)
{
    const int matCount = static_cast<int>(ymodel.m_YMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        YMaterial& mat = ymodel.m_YMaterials[iMat];
        if (!mat.m_DefaultFlag && mat.m_ColAnimFlag)
        {
            // エクスポータが作成したデフォルトマテリアルでなく、
            // カラーアニメーションが設定されていればアニメーション値を取得します。
            for (int paramIdx = 0; paramIdx < RMaterial::COLOR_PARAM_COUNT; ++paramIdx)
            {
                float val = 0.0f;
                if (RMaterial::IsSpecular(paramIdx) && !mat.m_SpecularPlugFlag)
                {
                    val = 0.0f; // スペキュラのないマテリアルなら (0, 0, 0)
                }
                else
                {
                    // プラグからアニメーション値を取得します。
                    const MPlug& plug = mat.m_AnimPlugs[paramIdx];
                    plug.getValue(val);
                    if (RMaterial::IsDiffuse(paramIdx))
                    {
                        // ディフューズカラーなら DiffuseCoeff を掛けます。
                        val *= mat.m_DiffuseCoeff;
                    }
                    else if (RMaterial::IsOpacity(paramIdx) && mat.m_UseTransparencyPlug)
                    {
                        // 透明度かつ Maya の transparency 使用なら値を反転します。
                        val = 1.0f - val; // opacity = 1 - transparency
                    }
                }
                mat.m_Anims[paramIdx].m_FullValues.push_back(RSnapToZero(val));
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるテクスチャ SRT アニメーション値を取得します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void GetTexSrtFullAnimValue(YModel& ymodel)
{
    const int animCount = static_cast<int>(ymodel.m_TexSrtAnims.size());
    for (int iAnim = 0; iAnim < animCount; ++iAnim)
    {
        TexSrtAnim& texSrtAnim = ymodel.m_TexSrtAnims[iAnim];
        for (int paramIdx = 0; paramIdx < ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
        {
            const MPlug& plug = texSrtAnim.m_AnimPlugs[paramIdx];
            double val;
            plug.getValue(val);
            if (ROriginalTexsrt::IsRotate(paramIdx))
            {
                val *= R_M_RAD_TO_DEG;
            }
            texSrtAnim.m_Anims[paramIdx].m_FullValues.push_back(static_cast<float>(RSnapToZero(val)));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるテクスチャ SRT アニメーション値を分析します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus AnalyzeTexSrtFullAnim(YModel& ymodel, const YExpOpt& yopt)
{
    const int animCount = static_cast<int>(ymodel.m_TexSrtAnims.size());
    for (int iAnim = 0; iAnim < animCount; ++iAnim)
    {
        TexSrtAnim& texSrtAnim = ymodel.m_TexSrtAnims[iAnim];
        for (int paramIdx = 0; paramIdx < ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
        {
            YAnimCurve& curve = texSrtAnim.m_Anims[paramIdx];
            curve.m_Name = texSrtAnim.m_PlaceNodeName + "." + ROriginalTexsrt::GetParamName(paramIdx);
            curve.m_LoopFlag = yopt.m_LoopAnim;
            curve.m_AngleFlag = ROriginalTexsrt::IsRotate(paramIdx);

            //-----------------------------------------------------------------------------
            // set dst & tolerance
            if (ROriginalTexsrt::IsScale(paramIdx))
            {
                curve.m_Tolerance = yopt.m_TolTexS;
            }
            else if (ROriginalTexsrt::IsRotate(paramIdx))
            {
                curve.m_Tolerance = yopt.m_TolTexR;
            }
            else
            {
                curve.m_Tolerance = yopt.m_TolTexT;
            }

            //-----------------------------------------------------------------------------
            // check constant
            curve.UpdateConstantFlag();
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT のキーアニメーションを取得します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetTexSrtKeyAnim(YModel& ymodel, const YExpOpt& yopt)
{
    const int animCount = static_cast<int>(ymodel.m_TexSrtAnims.size());
    for (int iAnim = 0; iAnim < animCount; ++iAnim)
    {
        TexSrtAnim& texSrtAnim = ymodel.m_TexSrtAnims[iAnim];
        for (int paramIdx = 0; paramIdx < ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
        {
            YAnimCurve& curve = texSrtAnim.m_Anims[paramIdx];
            if (!curve.m_ConstantFlag)
            {
                //-----------------------------------------------------------------------------
                // check get key OK
                bool forceMakeKey = yopt.m_BakeAllAnim;
                float valueScale = 1.0f;
                float valueOfs = 0.0f;
                if (ROriginalTexsrt::IsRotate(paramIdx))
                {
                    valueScale = static_cast<float>(R_M_RAD_TO_DEG);
                }

                const ChannelInput& chan = texSrtAnim.m_AnimChans[paramIdx];
                bool getKeyFlag = (
                    !forceMakeKey &&
                    chan.m_CurveFlag &&
                    !chan.m_CurveWeighted &&
                    IsValidAnimCurveValue(chan.m_CurveObj,
                        curve.m_FullValues, valueScale, valueOfs, yopt));

                if (getKeyFlag)
                {
                    //-----------------------------------------------------------------------------
                    // get keys (tex srt)
                    curve.GetKeys(chan.m_CurveObj, valueScale, valueOfs, yopt);
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // make keys (tex srt)
                    //cerr << "make keys: " << curve.m_Name << R_ENDL;
                    curve.MakeKeys(GetFloatFrameFromSubFrame, &yopt, true);
                }
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるテクスチャパターンアニメーション値を取得します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void GetTexPatFullAnimValue(YModel& ymodel)
{
    const int animCount = static_cast<int>(ymodel.m_TexPatAnims.size());
    for (int iAnim = 0; iAnim < animCount; ++iAnim)
    {
        TexPatAnim& texPatAnim = ymodel.m_TexPatAnims[iAnim];
        int val;
        texPatAnim.m_FePlug.getValue(val);
        texPatAnim.m_FeAnim.m_FullValues.push_back(static_cast<float>(val));
    }
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるテクスチャパターンアニメーション値を分析します。
//!        参照されるテクスチャイメージの texPatIdxs に 0 を設定します。
//!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
//!
//! @param[in,out] texPatAnim テクスチャパターンアニメーションです。
//! @param[in,out] texPatIdxs モデルの全テクスチャイメージの
//!                           <tex_pattern_array> 内のインデックスです。
//!                           テクスチャパターンアニメーションで参照されていなければ -1 です。
//! @param[in] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus AnalyzeTexPatFullAnim(
    TexPatAnim& texPatAnim,
    MIntArray& texPatIdxs,
    const YModel& ymodel
)
{
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    texPatAnim.m_ImgAnim.m_Name = texPatAnim.m_ObjName + ".img";

    const RFloatArray& fullValues = texPatAnim.m_FeAnim.m_FullValues;
    //RPrintArray(cerr, texPatAnim.m_FeAnim.m_FullValues, "tp");

    const int texCount = texPatAnim.m_TexImgIdxs.length();
    bool isWarned = false;
    for (int iFrame = 0; iFrame < yopt.m_SubFrameCount; ++iFrame)
    {
        //-----------------------------------------------------------------------------
        // get index in fe list
        const int fe = RRound(fullValues[iFrame]);
        int iTexNear = 0;
        int diffMin = INT_MAX;
        int iTex;
        for (iTex = 0; iTex < texCount; ++iTex)
        {
            if (texPatAnim.m_UsedFes[iTex] == fe)
            {
                break;
            }
            const int diff = RAbs(texPatAnim.m_UsedFes[iTex] - fe);
            if (diff < diffMin)
            {
                diffMin = diff;
                iTexNear = iTex;
            }
        }
        if (iTex >= texCount)
        {
            iTex = iTexNear;
            if (!isWarned)
            {
                isWarned = true;
                YShowWarning(&yscene, // Frame extension is wrong: (use nearest): %s: %d -> %d
                    "特定フレームにおけるフレーム拡張子の値に対応するテクスチャーが出力されていません: {0}: {1} -> {2} \n"
                    "出力されるテクスチャーファイルのうち、フレーム拡張子の値が一番近いものがパターンアニメーションデータとして使用されます。\n"
                    "警告が出ないようにするには、アニメーションカーブを修正するか、テクスチャーパターンアニメーション設定プラグインで強制的に出力するテクスチャーを指定してください。",
                    "A texture corresponding to the value of Frame Extension has not been exported for a particular frame: {0}: {1} -> {2} \n"
                    "Of the texture files to export, the one closest to the Frame Extension value is used as the pattern animation data. \n"
                    "To stop this warning from appearing, either correct the animation curve or forcibly specify the texture to be output using the Plug-In for Setting Texture Pattern Animations.",
                    texPatAnim.m_ObjName,
                    RGetNumberString(fe), RGetNumberString(texPatAnim.m_UsedFes[iTexNear]));
            }
        }

        //-----------------------------------------------------------------------------
        // append tex img index
        const int iTexImg = texPatAnim.m_TexImgIdxs[iTex];
        texPatAnim.m_ImgAnim.m_FullValues.push_back(static_cast<float>(iTexImg));

        //-----------------------------------------------------------------------------
        // get tex used flags
        texPatIdxs[iTexImg] = 0; // 仮の値を格納します。
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションのキーの値を
//!        フレーム拡張子からテクスチャイメージのインデックスに変換します。
//!
//! @param[in,out] curve テクスチャパターンアニメーションのカーブです。
//! @param[in] texPatAnim テクスチャパターンアニメーションです。
//-----------------------------------------------------------------------------
static void ConvertTexPatAnimImgKey(
    YAnimCurve& curve,
    const TexPatAnim& texPatAnim
)
{
    //-----------------------------------------------------------------------------
    // create convert table
    int feMin = LONG_MAX;
    int feMax = LONG_MIN;
    const int texCount = texPatAnim.m_UsedFes.length();
    for (int iTex = 0; iTex < texCount; ++iTex)
    {
        const int fe = texPatAnim.m_UsedFes[iTex];
        if (fe < feMin)
        {
            feMin = fe;
        }
        if (fe > feMax)
        {
            feMax = fe;
        }
    }
    MIntArray feToImgIdx(feMax - feMin + 1, -1);
    for (int iTex = 0; iTex < texCount; ++iTex)
    {
        const int fe = texPatAnim.m_UsedFes[iTex];
        feToImgIdx[fe - feMin] = texPatAnim.m_TexImgIdxs[iTex];
    }
    //cerr << "fe -> img: [" << feMin << "," << feMax << "]: " << feToImgIdx << endl;

    //-----------------------------------------------------------------------------
    // convert all key
    const int keyCount = static_cast<int>(curve.m_Keys.size());
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        RAnimKey& key = curve.m_Keys[iKey];
        int fe = RRound(key.m_Value);
        if (fe < feMin || fe > feMax)
        {
            fe = feMin;
        }
        int imgIndex = feToImgIdx[fe - feMin];
        if (imgIndex == -1)
        {
            imgIndex = 0;
        }
        key.m_Value = static_cast<float>(imgIndex);
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンのキーアニメーションを取得します。
//!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
//!
//! @param[in,out] texPatAnim テクスチャパターンアニメーションです。
//! @param[in] texPatIdxs モデルの全テクスチャイメージの
//!                       <tex_pattern_array> 内のインデックスです。
//!                       テクスチャパターンアニメーションで参照されていなければ -1 です。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void GetTexPatKeyAnim(
    TexPatAnim& texPatAnim,
    const MIntArray& texPatIdxs,
    const YExpOpt& yopt
)
{
    //-----------------------------------------------------------------------------
    // 値が一定か判定します。
    YAnimCurve& curve = texPatAnim.m_ImgAnim;
    curve.m_Tolerance = 0.0f;
    curve.UpdateConstantFlag();

    if (!curve.m_ConstantFlag)
    {
        //-----------------------------------------------------------------------------
        // 値が一定でなければキーを取得または作成します。
        const bool getKeyFlag = (
            !yopt.m_BakeAllAnim &&
            texPatAnim.m_ValidCurveFlag &&
            IsStepAnimCurve(texPatAnim.m_InputObj) &&
            IsValidAnimCurveValue(texPatAnim.m_InputObj,
                texPatAnim.m_FeAnim.m_FullValues, 1.0f, 0.0f, yopt));

        if (getKeyFlag)
        {
            curve.GetStepKeys(texPatAnim.m_InputObj, yopt);
            ConvertTexPatAnimImgKey(curve, texPatAnim);
        }
        else
        {
            //cerr << "make keys: " << curve.m_Name << endl;
            curve.MakeStepKeys(GetFloatFrameFromSubFrame, &yopt);
        }

        //-----------------------------------------------------------------------------
        // キーの値をテクスチャイメージのインデックスから
        // <tex_pattern_array> 内のインデックスに変換します。
        const int keyCount = static_cast<int>(curve.m_Keys.size());
        for (int iKey = 0; iKey < keyCount; ++iKey)
        {
            RAnimKey& key = curve.m_Keys[iKey];
            int value = RRound(key.m_Value);
            if (value != -1)
            {
                value = texPatIdxs[value];
            }
            key.m_Value = static_cast<float>(value);
        }
    }

    //-----------------------------------------------------------------------------
    // 初期値をテクスチャイメージのインデックスから
    // <tex_pattern_array> 内のインデックスに変換します。
    int baseValue = RRound(curve.m_FullValues[0]);
    if (baseValue != -1)
    {
        baseValue = texPatIdxs[baseValue];
    }
    curve.m_FullValues[0] = static_cast<float>(baseValue);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションデータを作成します。
//!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
//!
//! @param[in,out] ymodel モデルです。
//! @param[out] texPatRefs モデルの全テクスチャパターンアニメーションで参照する
//!                        テクスチャ名の配列を格納します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CreateTexPatAnimData(YModel& ymodel, RStringArray& texPatRefs)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーションがなければ何もしません。
    const int texPatAnimCount = static_cast<int>(ymodel.m_TexPatAnims.size());
    if (texPatAnimCount == 0)
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // get full value & tex index list

    // init
    const int allTexImgCount = static_cast<int>(ymodel.m_TexImgs.size());
    MIntArray texPatIdxs(allTexImgCount, -1);

    // get full value & tex used flag
    for (int iAnim = 0; iAnim < texPatAnimCount; ++iAnim)
    {
        status = AnalyzeTexPatFullAnim(ymodel.m_TexPatAnims[iAnim], texPatIdxs, ymodel);
        CheckStatus(status);
    }

    // set out tex index
    int outTexImgCount = 0;
    for (int iTexImg = 0; iTexImg < allTexImgCount; ++iTexImg)
    {
        const RImage& texImg = *ymodel.m_pOutTexImgs[iTexImg];
        if (texPatIdxs[texImg.GetOrgIndex()] != -1)
        {
            texPatIdxs[texImg.GetOrgIndex()] = outTexImgCount;
            ++outTexImgCount;
        }
    }

    //-----------------------------------------------------------------------------
    // get key
    for (int iAnim = 0; iAnim < texPatAnimCount; ++iAnim)
    {
        TexPatAnim& texPatAnim = ymodel.m_TexPatAnims[iAnim];
        GetTexPatKeyAnim(texPatAnim, texPatIdxs, yopt);
    }

    //-----------------------------------------------------------------------------
    // create texture references
    if (outTexImgCount > 0)
    {
        for (int iTexImg = 0; iTexImg < allTexImgCount; ++iTexImg)
        {
            const RImage& texImg = *ymodel.m_pOutTexImgs[iTexImg];
            if (texPatIdxs[texImg.GetOrgIndex()] != -1)
            {
                texPatRefs.push_back(texImg.GetName());
            }
        }
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるアニメーション値を取得します。
//!        ノード、マテリアル、シェイプ、ライト、カメラが対象です。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetFullAnimValue(YModel& ymodel, const YExpOpt& yopt)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    YEnvObjs& yenvObjs = ymodel.GetEnvObjs();

    //-----------------------------------------------------------------------------
    // 後で時間を元に戻すため現在の時間を保存しておきます。
    const MTime orgFrame = MAnimControl::currentTime();

    //-----------------------------------------------------------------------------
    // Baking Loop Script を環境変数から取得します。
    const std::string bakingLoopScript = GetBakingLoopScript();
    if (!bakingLoopScript.empty())
    {
        SetAnimEnvVarForPrePostExportScript(yopt);
    }

    //-----------------------------------------------------------------------------
    // 全フレームのアトリビュート値を取得します。
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());

    for (int iFrame = 0; iFrame < yopt.m_SubFrameCount; ++iFrame)
    {
        //-----------------------------------------------------------------------------
        // アトリビュート値を取得する時間を設定します。
        const double floatFrame = yopt.m_StartFrame +
            static_cast<double>(iFrame) / yopt.m_FramePrecision;
        MAnimControl::setCurrentTime(MTime(floatFrame, yopt.m_UIUnitTime));

        //-----------------------------------------------------------------------------
        // Baking Loop Script を実行します。
        if (!bakingLoopScript.empty())
        {
            SetEnvVarByMel("NINTENDO_EXPORT_CURRENT_SUB_FRAME", RGetNumberString(iFrame), false);
            status = MGlobal::executeCommand(bakingLoopScript.c_str());
            if (!status)
            {
                YShowError(&yscene,
                    "ベイク処理のループで実行するスクリプト（MEL）でエラーが発生しました。（詳細はスクリプトエディタを見てください）",
                    "An error occurred with the baking loop script (MEL). (See the Script Editor for details.)");
                return status;
            }
        }

        //-----------------------------------------------------------------------------
        // ノード関連のアニメーション値を取得します。
        for (int iNode = 0; iNode < nodeCount; ++iNode)
        {
            YNode& ynode = *ymodel.m_pYNodes[iNode];

            //-----------------------------------------------------------------------------
            // デフォルト値を設定します。
            MVector scale(1.0, 1.0, 1.0);
            MVector rotate = MVector::zero;
            MVector translate = MVector::zero;
            ynode.m_CurXformVisibility = true;

            //-----------------------------------------------------------------------------
            // ノードの変換と可視性を取得します。

            if (ynode.m_OrgKind == YNode::KindModelRoot ||
                !ynode.m_LayerPlaybackFlag)
            {
                // ノードがモデルルートまたはレイヤの再生フラグが OFF なら
                // 開始フレームにおける値をコピーします。
                scale     = ynode.m_XformS;
                rotate    = ynode.m_XformR;
                translate = ynode.m_XformT;
                ynode.m_CurXformVisibility = ynode.m_Visibility;
            }
            else if (ynode.m_IsSkinned)
            {
                // スキンで変形されるメッシュの場合、
                // 変換は開始フレームにおける値をコピーし、
                // 可視性のアニメーション値のみ取得します。
                scale     = ynode.m_XformS;
                rotate    = ynode.m_XformR;
                translate = ynode.m_XformT;
                ynode.m_XformVisibilityPlug.getValue(ynode.m_CurXformVisibility);
            }
            else
            {
                // それ以外のノードは変換と可視性のアニメーション値を取得します。
                MFnTransform xformFn(ynode.m_XformPath);

                double scaleArray[3] = { 1.0, 1.0, 1.0 };
                MEulerRotation eulerRotate = MEulerRotation::identity;

                xformFn.getScale(scaleArray);
                xformFn.getRotation(eulerRotate);
                translate = xformFn.getTranslation(MSpace::kTransform) * yopt.m_InternalMagnify;
                ynode.m_XformVisibilityPlug.getValue(ynode.m_CurXformVisibility);

                // 回転と移動の成分を補正します。
                ynode.AdjustRotate(eulerRotate);
                ynode.AdjustTranslate(translate, eulerRotate);

                rotate = MVector(eulerRotate.x, eulerRotate.y, eulerRotate.z);
                rotate *= R_M_RAD_TO_DEG;

                scale = MVector(scaleArray);
            }

            //-----------------------------------------------------------------------------
            // ローカル変換行列を計算します。
            ynode.m_CalcXformS = scale;
            if (ynode.m_OrgScaleCompensate && ynode.m_pParent != NULL)
            {
                // SegmentScaleCompensate を考慮してローカル変換行列を計算します。
                CreateTransformMtxScaleCompensate(ynode.m_CalcLocalMtx,
                    scale, rotate, translate, ynode.m_pParent->m_CalcXformS);
            }
            else
            {
                // ローカル変換行列を計算します。
                CreateTransformMtx(ynode.m_CalcLocalMtx, scale, rotate, translate);
            }

            //-----------------------------------------------------------------------------
            // YNode の変換をマージしてワールド座標系の値を直接設定します。
            if (ynode.m_MergeCount > 0)
            {
                MergeYNodeTransform(ynode, scale, rotate, translate);
            }

            //-----------------------------------------------------------------------------
            // ノードの出力用の可視性を取得します。
            const bool visibility = GetOutVisibility(ynode);

            //-----------------------------------------------------------------------------
            // アニメーション値をアニメーションカーブのメンバ変数に追加します。
            ynode.m_Anims[RBone::SCALE_X].m_FullValues.push_back(static_cast<float>(RSnapToZero(scale.x)));
            ynode.m_Anims[RBone::SCALE_Y].m_FullValues.push_back(static_cast<float>(RSnapToZero(scale.y)));
            ynode.m_Anims[RBone::SCALE_Z].m_FullValues.push_back(static_cast<float>(RSnapToZero(scale.z)));
            ynode.m_Anims[RBone::ROTATE_X].m_FullValues.push_back(static_cast<float>(RSnapToZero(rotate.x)));
            ynode.m_Anims[RBone::ROTATE_Y].m_FullValues.push_back(static_cast<float>(RSnapToZero(rotate.y)));
            ynode.m_Anims[RBone::ROTATE_Z].m_FullValues.push_back(static_cast<float>(RSnapToZero(rotate.z)));
            ynode.m_Anims[RBone::TRANSLATE_X].m_FullValues.push_back(static_cast<float>(RSnapToZero(translate.x)));
            ynode.m_Anims[RBone::TRANSLATE_Y].m_FullValues.push_back(static_cast<float>(RSnapToZero(translate.y)));
            ynode.m_Anims[RBone::TRANSLATE_Z].m_FullValues.push_back(static_cast<float>(RSnapToZero(translate.z)));
            ynode.m_VisAnim.m_FullValues.push_back(static_cast<float>(visibility));
        }

        //-----------------------------------------------------------------------------
        // マテリアル関連のアニメーション値を取得します。
        if (yopt.ExportsColorAnim())
        {
            // マテリアルカラーのアニメーション値を取得します。
            GetColorFullAnimValue(ymodel);
        }

        if (yopt.ExportsTexSrtAnim())
        {
            // テクスチャ SRT アニメーション値を取得します。
            GetTexSrtFullAnimValue(ymodel);
        }

        if (yopt.ExportsTexPatAnim())
        {
            // テクスチャパターンアニメーション値を取得します。
            GetTexPatFullAnimValue(ymodel);
        }

        //-----------------------------------------------------------------------------
        // ブレンドシェイプ関連のアニメーション値を取得します。
        if (yopt.m_OutFileFlag[RExpOpt::FSH])
        {
            GetBlendShapeFullAnimValue(ymodel.m_BlendShapeDatas);
        }

        //-----------------------------------------------------------------------------
        // カメラ・ライト関連のアニメーション値を取得します。
        if (yopt.ExportsSceneAnim())
        {
            GetEnvObjFullAnimValue(yenvObjs, yopt);
        }
    }

    //-----------------------------------------------------------------------------
    // 現在時間を元に戻します。
    MAnimControl::setCurrentTime(orgFrame);

    //-----------------------------------------------------------------------------
    // 取得したアニメーション値を分析します。
    //-----------------------------------------------------------------------------

    //-----------------------------------------------------------------------------
    // ノード関連のアニメーション値を分析します。
    if (yopt.m_OutFileFlag[RExpOpt::FSK])
    {
        AnalyzeYNodeFullAnim(ymodel, yopt);
    }

    //-----------------------------------------------------------------------------
    // マテリアル関連のアニメーション値を分析します。
    if (yopt.ExportsColorAnim())
    {
        AnalyzeColorFullAnim(ymodel, yopt);
    }

    if (yopt.ExportsTexSrtAnim())
    {
        AnalyzeTexSrtFullAnim(ymodel, yopt);
    }

    //-----------------------------------------------------------------------------
    // ブレンドシェイプ関連のアニメーション値を分析します。
    if (yopt.m_OutFileFlag[RExpOpt::FSH])
    {
        AnalyzeBlendShapeFullAnim(ymodel.m_BlendShapeDatas, yopt);
    }

    //-----------------------------------------------------------------------------
    // カメラ・ライト関連のアニメーション値を分析します。
    if (yopt.ExportsSceneAnim())
    {
        AnalyzeEnvObjFullAnim(yenvObjs, yopt);
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief モデル全体の属性を設定します。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus SetYModelAttr(YModel& ymodel)
{
    //-----------------------------------------------------------------------------
    // ボーン行列のスケール計算方法を設定します。
    RSkeleton& skeleton = ymodel.m_Skeleton;
    skeleton.m_ScaleMode = RSkeleton::STANDARD;
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        const YNode& ynode = *ymodel.m_pYNodes[iNode];
        if (ynode.m_ScaleCompensate)
        {
            skeleton.m_ScaleMode = RSkeleton::MAYA;
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // ボリュームカリングを無効にします。
    ymodel.m_VolumeCullingEnable = false;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのポインタを名前でソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool YMaterialPtrNameLess(YMaterial*& r1, YMaterial*& r2)
{
    return (strcmp(r1->m_Name.c_str(), r2->m_Name.c_str()) < 0);
}

//-----------------------------------------------------------------------------
//! @brief マテリアルを名前でソートします。
//!        ymodel.m_YMaterials 配列を名前順でソートした結果を ymodel.m_pOutYMaterials 配列に格納します。
//!        そして YMaterial::m_Index に ymodel.m_pOutYMaterials へのインデックスを設定します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SortMaterialByName(YModel& ymodel)
{
    const int matCount = static_cast<int>(ymodel.m_YMaterials.size());
    if (matCount > 0)
    {
        // 出力用マテリアルのポインタ配列を初期化します。
        ymodel.m_pOutYMaterials.resize(matCount);
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            ymodel.m_pOutYMaterials[iMat] = &ymodel.m_YMaterials[iMat];
        }

        // 名前でソートします。
        std::sort(ymodel.m_pOutYMaterials.begin(), ymodel.m_pOutYMaterials.end(),
            YMaterialPtrNameLess);

        // ソート後に出力用のインデックスを設定します。
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            ymodel.m_pOutYMaterials[iMat]->m_Index = iMat;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプをソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool ShapeDataPtrSortIdxLess(ShapeData*& r1, ShapeData*& r2)
{
    return (r1->m_SortIndex < r2->m_SortIndex);
}

//-----------------------------------------------------------------------------
//! @brief シェイプをボーンとマテリアルのインデックスでソートします。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SortShapeByBoneMatIndex(YModel& ymodel)
{
    const int shapeCount = static_cast<int>(ymodel.m_ShapeDatas.size());
    if (shapeCount > 0)
    {
        // シェイプデータにソート用のインデックスを設定し、
        // 出力用シェイプデータのポインタ配列を初期化します。
        ymodel.m_pOutShapeDatas.resize(shapeCount);
        for (int iShape = 0; iShape < shapeCount; ++iShape)
        {
            ShapeData& shapeData = ymodel.m_ShapeDatas[iShape];
            shapeData.m_SortIndex =
                ymodel.m_pOutYNodes[shapeData.m_NodeIndexes[0]]->m_Index * 10000 +
                ymodel.m_YMaterials[shapeData.m_MaterialIndex].m_Index;
            ymodel.m_pOutShapeDatas[iShape] = &shapeData;
        }

        // ソートします。
        std::sort(ymodel.m_pOutShapeDatas.begin(), ymodel.m_pOutShapeDatas.end(),
            ShapeDataPtrSortIdxLess);

        // ソート後に出力用のインデックスを設定します。
        for (int iShape = 0; iShape < shapeCount; ++iShape)
        {
            ymodel.m_pOutShapeDatas[iShape]->m_Index = iShape;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 各シェイプがユニークな名前を持つように名前を調整します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetUniqueShapeNames(YModel& ymodel)
{
    const int shapeCount = static_cast<int>(ymodel.m_pOutShapeDatas.size());
    for (int shapeIdx = 1; shapeIdx < shapeCount; ++shapeIdx)
    {
        ShapeData& shapeData = *ymodel.m_pOutShapeDatas[shapeIdx];
        std::string name = shapeData.m_Name;
        int id = 1;
        for (;;)
        {
            bool isFound = false;
            for (int otherIdx = 0; otherIdx < shapeIdx; ++otherIdx)
            {
                const ShapeData& other = *ymodel.m_pOutShapeDatas[otherIdx];
                if (other.m_Name == name)
                {
                    isFound = true;
                    break;
                }
            }
            if (isFound)
            {
                name = shapeData.m_Name + RGetNumberString(id, "_%d");
                ++id;
            }
            else
            {
                break;
            }
        }

        if (id > 1)
        {
            shapeData.m_Name = name;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプ群に出力用の名前を設定します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetShapesOutName(YModel& ymodel)
{
    const int shapeCount = static_cast<int>(ymodel.m_pOutShapeDatas.size());
    for (int iShape = 0; iShape < shapeCount; ++iShape)
    {
        ShapeData& shapeData = *ymodel.m_pOutShapeDatas[iShape];
        const YNode& ynode = *ymodel.m_pOutYNodes[shapeData.m_NodeIndexes[0]];
        const YMaterial& mat = ymodel.m_YMaterials[shapeData.m_MaterialIndex];
        shapeData.m_Name = ynode.m_Name + "__" + mat.m_Name;
    }
    SetUniqueShapeNames(ymodel);
}

//-----------------------------------------------------------------------------
//! @brief スキニングに使用するボーンの行列パレットインデックスを設定します。
//!        行列パレットはスムーススキン用配列の後にリジッドスキン用配列が並ぶようにします。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetYNodeSkinningMtxIdx(YModel& ymodel)
{
    int mtxIdx = 0;
    const int nodeCount = static_cast<int>(ymodel.m_pOutYNodes.size());

    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pOutYNodes[iNode];
        if (ynode.m_SmoothMtxIdx != -1)
        {
            ynode.m_SmoothMtxIdx = mtxIdx++;
            ++ymodel.m_SmoothSkinningMtxCount;
        }
    }

    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pOutYNodes[iNode];
        if (ynode.m_RigidMtxIdx != -1)
        {
            ynode.m_RigidMtxIdx = mtxIdx++;
            ++ymodel.m_RigidSkinningMtxCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya から単一モデルのデータを取得します。
//! シーンの階層構造をトラバースし、エクスポートするオブジェクトを列挙します。
//! 次に列挙されたオブジェクトを Maya から YNode 等の出力形式に変換します。
//!
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetData(YModel& ymodel)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    YEnvObjs& yenvObjs = ymodel.GetEnvObjs();

    //-----------------------------------------------------------------------------
    // 現在の時間をアニメーションの開始時間に設定します。
    const MTime orgFrame = MAnimControl::currentTime();
    MAnimControl::setCurrentTime(MTime(yopt.m_DoubleStartFrame, yopt.m_UIUnitTime));

    //-----------------------------------------------------------------------------
    // get data
    //-----------------------------------------------------------------------------

    //-----------------------------------------------------------------------------
    // オプションで指定された出力対象を考慮して階層構造内を走査し、
    // 出力するノードやシェイプを列挙します。
    // 列挙されたデータは YNode に設定され m_pYNodes 配列に登録されます。
    const clock_t getStartClock = clock();
    status = GetHierarchy(ymodel, false);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 列挙された YNode の姿勢や可視性を変換します。
    status = GetYNodeTransform(ymodel);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // メッシュ等の形状情報を変換してモデルに出力するシェイプを登録します。
    status = GetShape(ymodel);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // 名前を一意な文字列となるよう設定します。
    SetUniqueMaterialNames(ymodel);
    SetUniqueEnvObjNames(yenvObjs, yscene, yopt);

    //-----------------------------------------------------------------------------
    // 中間ファイルに出力するデータを出力順にソートします。
    SetOutTexImgPtrs(ymodel.m_pOutTexImgs, ymodel.m_TexImgs);
    SortMaterialByName(ymodel);
    if (yopt.ExportsModelFile() || yopt.ExportsShapeAnim())
    {
        SortShapeByBoneMatIndex(ymodel);
    }
    SortEnvObjByName(yenvObjs);

    //-----------------------------------------------------------------------------
    // ボーンの描画行列フラグと
    // スキニングに使用するボーンの行列パレットインデックスを設定します。
    SetYNodeSkinningMtxIdx(ymodel);

    //-----------------------------------------------------------------------------
    // シェイプ群に出力用の名前を設定します。
    // マテリアルの出力用の名前が確定してから設定するためここで設定します。
    if (yopt.ExportsModelFile() || yopt.ExportsShapeAnim())
    {
        SetShapesOutName(ymodel);
    }

    //-----------------------------------------------------------------------------
    // profile
    yscene.m_GetClocks += clock() - getStartClock;

    //-----------------------------------------------------------------------------
    // get animation
    if (yopt.ExportsAnim())
    {
        const clock_t animStartClock = clock();

        //-----------------------------------------------------------------------------
        // 全フレームを焼き付けたフルアニメーションを取得します。
        status = GetFullAnimValue(ymodel, yopt);
        CheckStatus(status);

        //-----------------------------------------------------------------------------
        // YNode のキーフレームアニメーションを取得します。
        if (yopt.m_OutFileFlag[RExpOpt::FSK])
        {
            status = GetYNodeKeyAnim(ymodel);
            CheckStatus(status);
        }

        //-----------------------------------------------------------------------------
        // マテリアルのキーフレームアニメーションを取得します。
        if (yopt.ExportsColorAnim())
        {
            status = GetColorKeyAnim(ymodel, yopt);
            CheckStatus(status);
        }

        if (yopt.ExportsTexSrtAnim())
        {
            status = GetTexSrtKeyAnim(ymodel, yopt);
            CheckStatus(status);
        }

        //-----------------------------------------------------------------------------
        // ブレンドシェイプのキーフレームアニメーションを取得し、
        // 頂点シェイプアニメーション配列を設定します。
        if (yopt.m_OutFileFlag[RExpOpt::FSH])
        {
            status = GetBlendShapeKeyAnim(ymodel.m_BlendShapeDatas, yopt);
            CheckStatus(status);
            SetVertexShapeAnims(ymodel);
        }

        //-----------------------------------------------------------------------------
        // カメラとライト等のシーン要素のキーフレームアニメーションを取得します。
        if (yopt.ExportsSceneAnim())
        {
            status = GetEnvObjKeyAnim(yenvObjs, yopt);
            CheckStatus(status);
        }

        yscene.m_AnimClocks += clock() - animStartClock;
    }

    //-----------------------------------------------------------------------------
    // ユーザーデータを取得します。
    GetYNodeUserData(&ymodel);
    GetMaterialUserData(&ymodel);
    GetEnvObjUserData(&yenvObjs, &yscene);

    //-----------------------------------------------------------------------------
    // YModelにアトリビュートを設定します。
    SetYModelAttr(ymodel);

    //-----------------------------------------------------------------------------
    // 現在の時間をアニメーションの取得処理前の時間に戻します。
    MAnimControl::setCurrentTime(orgFrame);

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ファイル情報に出力する元ファイルのパスを取得します。
//!
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 元ファイルのパスを返します。
//-----------------------------------------------------------------------------
static std::string GetFileInfoSourceFilePath(const YExpOpt& yopt)
{
    std::string srcPath = MFileIO::currentFile().asChar();
    const std::string srcName = RGetFileNameFromFilePath(srcPath);
    if (srcName == "untitled" || srcName == "無題")
    {
        srcPath = "";
    }
    srcPath = RGetFileInfoSourceFilePath(srcPath, yopt.m_ProjectRootPath);
    return srcPath;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのヘッダを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc 要素のインデントに必要なタブの数です。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] ymodel モデルです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputHeader(
    std::ostream& os,
    const int tc,
    const RExpOpt::FileType fileType,
    const YModel& ymodel,
    const YExpOpt& yopt
)
{
    //-----------------------------------------------------------------------------
    // xml header
    os << RTab(tc) << "<?xml version=\"1.0\" encoding=\"utf-8\"?>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // begin intermediate file
    os << RTab(tc) << "<nw4f_3dif version=\"" << RExpOpt::GetRootElementVersion()
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // file info
    os << RTab(tc) << "<file_info>" << R_ENDL;
    MString toolName = "Nintendo Export for Maya " + MGlobal::mayaVersion();
    os << RTab(tc + 1) << "<create tool_name=\""
       << RGetUtf8FromShiftJis(REncodeXmlString(toolName.asChar()))
       << "\" tool_version=\"" << ExporterVersion << "\"" << R_ENDL;
    const std::string srcPath = GetFileInfoSourceFilePath(yopt);
    os << RTab(tc + 2) << "src_path=\"" << RGetUtf8FromShiftJis(REncodeXmlString(srcPath)) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "/>" << R_ENDL;
    os << RTab(tc) << "</file_info>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // begin root element
    os << RTab(tc) << "<" << yopt.GetTypeElementName(fileType)
       << " version=\"" << yopt.GetTypeElementVersion(fileType) << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // model info
    if (fileType == RExpOpt::FMD)
    {
        os << RTab(tc) << "<model_info" << R_ENDL;
        os << RTab(tc + 1) << "material_count=\"" << ymodel.m_pOutYMaterials.size() << "\"" << R_ENDL;
        os << RTab(tc + 1) << "bone_count=\"" << ymodel.m_pOutYNodes.size() << "\"" << R_ENDL;
        os << RTab(tc + 1) << "shape_count=\"" << ymodel.m_pOutShapeDatas.size() << "\"" << R_ENDL;
        os << RTab(tc + 1) << "smooth_skinning_shape=\"" << ymodel.m_SmoothSkinningShapeCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "rigid_skinning_shape=\"" << ymodel.m_RigidSkinningShapeCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "smooth_skinning_matrix=\"" << ymodel.m_SmoothSkinningMtxCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "rigid_skinning_matrix=\"" << ymodel.m_RigidSkinningMtxCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "total_process_vertex=\"" << ymodel.m_TotalProcessVertexCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "unite_pos_quantize=\"" << RBoolStr(false) << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_preset=\"" << RGetUtf8FromShiftJis(REncodeXmlString(yopt.m_PresetName)) << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_magnify=\"" << yopt.m_Magnify << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_start_frame=\"" << yopt.m_StartFrame << "\"" << R_ENDL;
        os << RTab(tc) << "/>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // animation info
    const bool isFma = (!yopt.m_UsesFclFtsFtp && yopt.IsFclFtsFtp(fileType));
    const char* animInfoElem = nullptr;
    if (isFma)
    {
        animInfoElem = "material_anim_info";
    }
    else
    {
        switch (fileType)
        {
        case RExpOpt::FSK: animInfoElem = "skeletal_anim_info"       ; break;
        case RExpOpt::FVB: animInfoElem = "bone_visibility_anim_info"; break;
        case RExpOpt::FCL: animInfoElem = "shader_param_anim_info"   ; break;
        case RExpOpt::FTS: animInfoElem = "shader_param_anim_info"   ; break;
        case RExpOpt::FTP: animInfoElem = "tex_pattern_anim_info"    ; break;
        case RExpOpt::FSH: animInfoElem = "shape_anim_info"          ; break;
        case RExpOpt::FSN: animInfoElem = "scene_anim_info"          ; break;
        default: break;
        }
    }

    if (animInfoElem != nullptr)
    {
        os << RTab(tc) << "<" << animInfoElem;
        if (fileType != RExpOpt::FSN)
        {
            os << " frame_count=\"" << yopt.m_OutFrameCount
               << "\" loop=\"" << RBoolStr(yopt.m_LoopAnim)
               << "\"" << R_ENDL;
        }
        else
        {
            os << R_ENDL;
        }
        os << RTab(tc + 1) << "frame_resolution=\"" << yopt.m_FramePrecision << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_preset=\"" << RGetUtf8FromShiftJis(REncodeXmlString(yopt.m_PresetName)) << "\"" << R_ENDL;
        if (fileType == RExpOpt::FSK ||
            fileType == RExpOpt::FSN)
        {
            os << RTab(tc + 1) << "dcc_magnify=\"" << yopt.m_Magnify << "\"" << R_ENDL;
        }
        os << RTab(tc + 1) << "dcc_start_frame=\"" << yopt.m_StartFrame << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_end_frame=\"" << yopt.m_EndFrame << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_fps=\"" << yopt.m_FramesPerSecond << "\"" << R_ENDL;
        os << RTab(tc + 1) << "bake_all=\"" << RBoolStr(yopt.m_BakeAllAnim) << "\"" << R_ENDL;

        if (fileType == RExpOpt::FSK)
        {
            ymodel.m_Skeleton.OutScaleRotateMode(os, tc + 1);
            os << RTab(tc + 1) << "bake_tolerance_scale=\""     << yopt.m_TolS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_rotate=\""    << yopt.m_TolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_translate=\"" << yopt.m_TolT << "\"" << R_ENDL;
        }
        else if (isFma                    ||
                 fileType == RExpOpt::FCL ||
                 fileType == RExpOpt::FTS)
        {
            os << RTab(tc + 1) << "bake_tolerance_color=\""         << yopt.m_TolC    << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_tex_scale=\""     << yopt.m_TolTexS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_tex_rotate=\""    << yopt.m_TolTexR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_tex_translate=\"" << yopt.m_TolTexT << "\"" << R_ENDL;
        }
        else if (fileType == RExpOpt::FSN)
        {
            os << RTab(tc + 1) << "bake_tolerance_rotate=\""    << yopt.m_TolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_translate=\"" << yopt.m_TolT << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_color=\""     << yopt.m_TolC << "\"" << R_ENDL;
        }

        if (fileType == RExpOpt::FSK)
        {
            os << RTab(tc + 1) << "quantize_tolerance_scale=\""     << yopt.m_QuantTolS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_rotate=\""    << yopt.m_QuantTolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_translate=\"" << yopt.m_QuantTolT << "\"" << R_ENDL;
        }
        else if (isFma                    ||
                 fileType == RExpOpt::FCL ||
                 fileType == RExpOpt::FTS)
        {
            os << RTab(tc + 1) << "quantize_tolerance_tex_scale=\""     << yopt.m_QuantTolTexS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_tex_rotate=\""    << yopt.m_QuantTolTexR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_tex_translate=\"" << yopt.m_QuantTolTexT << "\"" << R_ENDL;
        }
        else if (fileType == RExpOpt::FSN)
        {
            os << RTab(tc + 1) << "quantize_tolerance_rotate=\""    << yopt.m_QuantTolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_translate=\"" << yopt.m_QuantTolT << "\"" << R_ENDL;
        }

        os << RTab(tc) << "/>" << R_ENDL;
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのフッタを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc 要素のインデントに必要なタブの数です。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFooter(
    std::ostream& os,
    const int tc,
    const RExpOpt::FileType fileType,
    const YExpOpt& yopt
)
{
    //-----------------------------------------------------------------------------
    // 編集用コメント
    if (!yopt.m_CommentText.empty())
    {
        os << RTab(tc) << "<comment label=\"" << ""
           << "\" color=\"" << ""
           << "\" text=\"" << RGetUtf8FromShiftJis(REncodeXmlString(yopt.m_CommentText))
           << "\" />" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end root element
    os << RTab(tc) << "</" << yopt.GetTypeElementName(fileType) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end intermediate file
    os << RTab(tc) << "</nw4f_3dif>" << R_ENDL;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ出力後に設定可能なマテリアル属性を設定します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
static void SetMaterialAttrAfterOutTex(const YModel& ymodel)
{
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // loop for material
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];

        //-----------------------------------------------------------------------------
        // loop for sampler
        for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
        {
            RSampler& sampler = mat.m_Samplers[iSampler];
            const RImage& texImg = ymodel.m_TexImgs[sampler.m_ImageIndex];

            //-----------------------------------------------------------------------------
            // ミップマップならミップマップフィルタを変更します。
            if (texImg.GetMipLevel() >= 2)
            {
                sampler.m_FilterMip = (yopt.m_IsFilterMipLinear) ? RSampler::LINEAR : RSampler::POINT;
            }

            //-----------------------------------------------------------------------------
            // レンダーステートを設定します。
            if (mat.m_AutoRenderStateFlag)
            {
                if (sampler.IsAlbedo() &&
                    mat.m_TransparencyTexObj == mat.m_ColorTexObj)
                {
                    if (texImg.GetTransparencyMode() == RImage::MASK)
                    {
                        if (mat.GetRenderStateMode() == RRenderState::OPA)
                        {
                            mat.SetRenderStateMode(RRenderState::MASK);
                        }
                    }
                    else if (texImg.GetTransparencyMode() == RImage::XLU)
                    {
                        mat.SetRenderStateMode(RRenderState::XLU);
                    }
                }
                else if (sampler.IsOpacity())
                {
                    mat.SetRenderStateMode(RRenderState::XLU);
                }
            }

            //-----------------------------------------------------------------------------
            // キューブマップが貼られていたらマッピング方法を変更します。
            // TODO: キューブマップ用のシェーダを使用するように設定する？
            if (texImg.IsCubeMap())
            {
                //sampler.m_TexSrc.m_Semantic = RTexSrc::Normal;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアル群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <material_array> 要素のインデントに必要なタブの数です。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static void OutputMaterials(std::ostream& os, const int tc, const YModel& ymodel)
{
    //-----------------------------------------------------------------------------
    // set attr after out tex
    // テクスチャを出力するまで決定しないアトリビュートはここで設定します。
    SetMaterialAttrAfterOutTex(ymodel);

    //-----------------------------------------------------------------------------
    // out material array
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    if (matCount > 0)
    {
        os << RTab(tc) << "<material_array length=\"" << matCount << "\">" << R_ENDL;

        // loop for material
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
            mat.Out(os, tc + 1, iMat);
        }

        os << RTab(tc) << "</material_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief オリジナルマテリアル群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <original_material_array> 要素のインデントに必要なタブの数です。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static void OutOriginalMaterials(std::ostream& os, const int tc, const YModel& ymodel)
{
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    if (matCount > 0)
    {
        os << RTab(tc) << "<original_material_array length=\"" << matCount << "\">" << R_ENDL;

        // loop for material
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
            mat.OutOriginal(os, tc + 1, iMat);
        }

        os << RTab(tc) << "</original_material_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプにプリミティブを設定します。
//!
//! @param[in] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus SetRPrim(
    RShape& rshape,
    const ShapeData& shapeData,
    const YModel& ymodel
)
{
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // set vertex attr flag
    rshape.m_VtxAttrFlag[RPrimVtx::POS0] = true;
    rshape.m_VtxAttrFlag[RPrimVtx::NRM0] = shapeData.m_VtxNrmFlag;
    const int tanSetCount = static_cast<int>(shapeData.m_VtxTanTexIdxs.size());
    for (int iTanSet = 0; iTanSet < tanSetCount; ++iTanSet)
    {
        rshape.m_VtxAttrFlag[RPrimVtx::TAN0 + iTanSet] = true;
        rshape.m_VtxAttrFlag[RPrimVtx::BIN0 + iTanSet] = true;
    }
    const YVtxAttrInfoArray& colInfos = shapeData.m_VtxColInfos;
    for (int iCol = 0; iCol < static_cast<int>(colInfos.size()); ++iCol)
    {
        rshape.m_VtxAttrFlag[RPrimVtx::COL0 + iCol] = colInfos[iCol].m_ExistFlag;
    }
    for (int iTex = 0; iTex < shapeData.m_VtxTexCount; ++iTex)
    {
        rshape.m_VtxAttrFlag[RPrimVtx::TEX0 + iTex] = true;
    }
    rshape.m_VtxAttrFlag[RPrimVtx::IDX0] = (shapeData.m_SkinningMode != RShape::NO_SKINNING);
    rshape.m_VtxAttrFlag[RPrimVtx::WGT0] = (shapeData.m_SkinningMode == RShape::SMOOTH     );
    const YVtxAttrInfoArray& usrInfos = shapeData.m_VtxUsrInfos;
    for (int iUa = 0; iUa < static_cast<int>(usrInfos.size()); ++iUa)
    {
        rshape.m_VtxAttrFlag[RPrimVtx::USR0 + iUa] = usrInfos[iUa].m_ExistFlag;
    }

    //-----------------------------------------------------------------------------
    // loop for face
    int iVtxTotal = 0;
    const int faceCount = shapeData.m_VtxCounts.length();
    for (int iFace = 0; iFace < faceCount; ++iFace)
    {
        //cerr << "sp: " << iFace << endl;
        const int vtxCount = shapeData.m_VtxCounts[iFace];

        //-----------------------------------------------------------------------------
        // create rprim
        RPrimitive rprim(vtxCount);

        //-----------------------------------------------------------------------------
        // copy attr
        for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal)
        {
            RPrimVtx vtx;
            vtx.m_Mtx = shapeData.m_VtxMtxIdxs[iVtxTotal];
            vtx[RPrimVtx::POS0] = shapeData.m_VtxPosIdxs[iVtxTotal];
            vtx[RPrimVtx::NRM0] = (rshape.m_VtxAttrFlag[RPrimVtx::NRM0]) ?
                shapeData.m_VtxNrmIdxs[iVtxTotal] : -1;

            for (int iTanSet = 0; iTanSet < tanSetCount; ++iTanSet)
            {
                vtx[RPrimVtx::TAN0 + iTanSet] =
                    shapeData.m_VtxTanIdxss[iTanSet][iVtxTotal];
                vtx[RPrimVtx::BIN0 + iTanSet] =
                    shapeData.m_VtxBinIdxss[iTanSet][iVtxTotal];
            }

            for (int iCol = 0; iCol < static_cast<int>(colInfos.size()); ++iCol)
            {
                if (colInfos[iCol].m_ExistFlag)
                {
                    vtx[RPrimVtx::COL0 + iCol] =
                        shapeData.m_VtxColIdxss[iCol][iVtxTotal];
                }
            }

            for (int iTex = 0; iTex < shapeData.m_VtxTexCount; ++iTex)
            {
                vtx[RPrimVtx::TEX0 + iTex] =
                    shapeData.m_VtxTexIdxss[iTex][iVtxTotal];
            }

            for (int iUa = 0; iUa < static_cast<int>(usrInfos.size()); ++iUa)
            {
                if (usrInfos[iUa].m_ExistFlag)
                {
                    vtx[RPrimVtx::USR0 + iUa] =
                        shapeData.m_VtxUsrIdxss[iUa][iVtxTotal];
                }
            }

            rprim.SetPrimVtx(iVtx, vtx);
        }

        //-----------------------------------------------------------------------------
        // append
        if (shapeData.m_PrimType == RPrimitive::POINTS)
        {
            rshape.AppendPoints(rprim, ymodel.m_VtxMtxs);
        }
        else if (shapeData.m_PrimType == RPrimitive::LINES ||
                 shapeData.m_PrimType == RPrimitive::LINE_STRIP)
        {
            rshape.AppendLines(rprim, ymodel.m_VtxMtxs);
        }
        else
        {
            const bool forceTriangulate = true; // 現在は常に三角形分割して出力
            const int triPattern = shapeData.m_TriPatterns[iFace];
            if (!rshape.AppendPolygon(rprim, ymodel.m_VtxMtxs, forceTriangulate, triPattern))
            {
                YShowError(&yscene, // Polygon bone size is over: %s // 現在は発生しない
                    "１つのポリゴンに影響するボーン数が多すぎます: {0}",
                    "Polygon bone size is over: {0}",
                    shapeData.m_OrgName);
                return MS::kFailure;
            }
        }
    }
    //cerr << "prim: " << rshape.mRPrimTable.size() << endl;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief シェイプを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] vertices 頂点データ配列です。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//! @param[in] tc <shape> 要素のインデントに必要なタブの数です。
//! @param[in] shapeIdx シェイプのインデックスです。
//! @param[in] shapeData シェイプデータです。
//! @param[in] boneMtxIdxs ボーンに対する行列パレットインデックスの配列です。
//! @param[in,out] progShape シェイプ処理の進行状況です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputShape( // OutputShapeT
    std::ostream& os,
    RVertexArray& vertices,
    RDataStreamArray& dataStreams,
    YModel& ymodel,
    const int tc,
    const int shapeIdx,
    const ShapeData& shapeData,
    const RIntArray& boneMtxIdxs,
    RProgShape& progShape
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // RShape オブジェクトを作成します。
    const int mtxPalCount = RShape::GetStandardMtxPalCount(false, 0);
    //const int mtxPalCount = 2;
    //cerr << "mtx pal count: " << mtxPalCount << endl;
    RShape rshape(shapeData.m_SkinningMode, mtxPalCount,
        shapeData.m_OrgName.c_str(), &progShape);

    status = SetRPrim(rshape, shapeData, ymodel);
    CheckStatus(status);

    RStatus rstatus = rshape.SetAndOptimize(ymodel.m_VtxMtxs, boneMtxIdxs);
    status = GetMStatusDisplayError(&yscene, rstatus);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // シェイプ出力情報を設定します。
    const YNode& ynode = *ymodel.m_pOutYNodes[shapeData.m_NodeIndexes[0]];

    ROutShapeInfo outInfo;
    outInfo.m_Index = shapeIdx;
    outInfo.m_Name = shapeData.m_Name;
    outInfo.m_MatName = ymodel.m_YMaterials[shapeData.m_MaterialIndex].m_Name;
    outInfo.m_BoneName = ynode.m_Name;
    outInfo.m_OrientedBB = shapeData.m_OrientedBB;
    outInfo.m_pUvAttribNames = &shapeData.m_UvAttribNames;
    outInfo.m_pUvHintIdxs    = &shapeData.m_UvHintIdxs;

    outInfo.m_VtxInfos[RPrimVtx::POS0] = ROutShapeVtxInfo( // POS の array は RVec3Array のみ可
        3, GetPosAttrValueType(shapeData), &shapeData.m_VtxPoss);
    outInfo.m_VtxInfos[RPrimVtx::NRM0] = ROutShapeVtxInfo( // NRM の array は RVec3Array のみ可
        3, GetNrmAttrValueType(shapeData), &shapeData.m_VtxNrms);

    const int tanSetCount = static_cast<int>(shapeData.m_VtxTanTexIdxs.size());
    for (int iTanSet = 0; iTanSet < tanSetCount; ++iTanSet)
    {
        outInfo.m_VtxInfos[RPrimVtx::TAN0 + iTanSet] = ROutShapeVtxInfo( // TAN の array は RVec4Array のみ可
            4, GetNrmAttrValueType(shapeData), &shapeData.m_VtxTanss[iTanSet]); // TAN は NRM と同じ量子化
        outInfo.m_VtxInfos[RPrimVtx::BIN0 + iTanSet] = ROutShapeVtxInfo( // BIN の array は RVec4Array のみ可
            4, GetNrmAttrValueType(shapeData), &shapeData.m_VtxBinss[iTanSet]); // BIN は NRM と同じ量子化
    }

    for (int iCol = 0; iCol < static_cast<int>(shapeData.m_VtxColInfos.size()); ++iCol) // COL の array は RVec4Array のみ可
    {
        const YVtxAttrInfo& colInfo = shapeData.m_VtxColInfos[iCol];
        if (colInfo.m_ExistFlag)
        {
            outInfo.m_VtxInfos[RPrimVtx::COL0 + iCol] = ROutShapeVtxInfo(
                colInfo.m_CompCount, colInfo.m_ValueType, &shapeData.m_VtxColss[iCol]);
        }
    }

    for (int iTex = 0; iTex < shapeData.m_VtxTexCount; ++iTex) // TEX の array は RVec2Array のみ可
    {
        outInfo.m_VtxInfos[RPrimVtx::TEX0 + iTex] = ROutShapeVtxInfo(
            2, RPrimVtx::VALUE_FLOAT, &shapeData.m_VtxTexss[iTex]);
    }

    outInfo.m_VtxInfos[RPrimVtx::IDX0] = ROutShapeVtxInfo( // IDX は quantize のみ影響
        4, RPrimVtx::VALUE_UINT, NULL);
    outInfo.m_VtxInfos[RPrimVtx::WGT0] = ROutShapeVtxInfo( // WGT は quantize のみ影響
        4, RPrimVtx::VALUE_FLOAT, NULL);

    for (int iUa = 0; iUa < static_cast<int>(shapeData.m_VtxUsrInfos.size()); ++iUa) // USR の array は RVec4Array のみ可
    {
        const YVtxAttrInfo& usrInfo = shapeData.m_VtxUsrInfos[iUa];
        if (usrInfo.m_ExistFlag)
        {
            outInfo.m_VtxInfos[RPrimVtx::USR0 + iUa] = ROutShapeVtxInfo(
                usrInfo.m_CompCount, usrInfo.m_ValueType, &shapeData.m_VtxUsrss[iUa]);
        }
    }

    // キーシェイプの名前の配列を設定します。
    if (shapeData.m_ShapeUpdate.UpdatesSome())
    {
        const YBlendShapeData& bsData =
            ymodel.m_BlendShapeDatas[ynode.m_BlendShapeDataIndex];
        outInfo.m_KeyNames.push_back(bsData.m_BaseName);
        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            outInfo.m_KeyNames.push_back(bsData.m_Targets[iTarget].m_Name);
        }
    }

    // ベースシェイプ以外のキーシェイプの値の配列のポインタを配列に追加します。
    for (int iTarget = 0; iTarget < static_cast<int>(shapeData.m_TargetPosss.size()); ++iTarget)
    {
        outInfo.m_VtxInfos[RPrimVtx::POS0].m_pArrays.push_back(&shapeData.m_TargetPosss[iTarget]);
    }
    for (int iTarget = 0; iTarget < static_cast<int>(shapeData.m_TargetNrmss.size()); ++iTarget)
    {
        outInfo.m_VtxInfos[RPrimVtx::NRM0].m_pArrays.push_back(&shapeData.m_TargetNrmss[iTarget]);
    }

    for (int iTanSet = 0; iTanSet < tanSetCount; ++iTanSet)
    {
        const std::vector<RVec4Array>& targetTanss = shapeData.m_TargetTansss[iTanSet];
        const std::vector<RVec4Array>& targetBinss = shapeData.m_TargetBinsss[iTanSet];
        for (int iTarget = 0; iTarget < static_cast<int>(targetTanss.size()); ++iTarget)
        {
            outInfo.m_VtxInfos[RPrimVtx::TAN0 + iTanSet].m_pArrays.push_back(&targetTanss[iTarget]);
        }
        for (int iTarget = 0; iTarget < static_cast<int>(targetBinss.size()); ++iTarget)
        {
            outInfo.m_VtxInfos[RPrimVtx::BIN0 + iTanSet].m_pArrays.push_back(&targetBinss[iTarget]);
        }
    }

    for (int iCol = 0; iCol < static_cast<int>(shapeData.m_VtxColInfos.size()); ++iCol)
    {
        const std::vector<RVec4Array>& targetColss = shapeData.m_TargetColsss[iCol];
        for (int iTarget = 0; iTarget < static_cast<int>(targetColss.size()); ++iTarget)
        {
            outInfo.m_VtxInfos[RPrimVtx::COL0 + iCol].m_pArrays.push_back(&targetColss[iTarget]);
        }
    }

    //-----------------------------------------------------------------------------
    // RShape オブジェクトを出力します。
    //RTimeMeasure tm1;
    ROutShapeResult result;
    rstatus = rshape.Out(os, vertices, dataStreams, result, tc, outInfo);
    //cerr << "shape out: " << tm1.GetMilliSec() << endl;
    if (!rstatus)
    {
        // シェイプがノードから参照されている場合は、エラーメッセージにノードの名前を含める
        if (shapeData.m_NodeIndexes.length() > 0)
        {
            const std::string nodeMatStr =
                ymodel.m_pOutYNodes[shapeData.m_NodeIndexes[0]]->m_OrgName + " - " +
                ymodel.m_YMaterials[shapeData.m_MaterialIndex].m_OrgName;
            std::string msgJp = rstatus.GetJapaneseMessage();
            std::string msgEn = rstatus.GetMessage();
            if (!msgJp.empty())
            {
                msgJp += "\n" + nodeMatStr;
            }
            if (!msgEn.empty())
            {
                msgEn += "\n" + nodeMatStr;
            }
            rstatus = RStatus((!rstatus) ? RStatus::FAILURE : RStatus::SUCCESS,
                msgJp, msgEn);
        }
        status = GetMStatusDisplayError(&yscene, rstatus);
        CheckStatus(status);
    }

    ymodel.m_TotalTriangleCount      += result.m_TriangleCount;
    ymodel.m_TotalIndexCount         += result.m_IndexCount;
    ymodel.m_TotalVertexCount        += result.m_VertexCount;
    ymodel.m_TotalProcessVertexCount += result.m_ProcessVertexCount;

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief shape warning func
//-----------------------------------------------------------------------------
static void ShapeWarningFunc(const std::string& msg, void* pParam)
{
    YShowWarning(reinterpret_cast<YScene*>(pParam), "", msg); // 現在は発生しない
}

//-----------------------------------------------------------------------------
//! @brief シェイプ群と頂点データ群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//! @param[in] tc <shape_array> 要素のインデントに必要なタブの数です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputShapes(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    YModel& ymodel,
    const int tc
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // ボーンに対する行列パレットインデックスの配列を作成します。
    const int boneCount = static_cast<int>(ymodel.m_pYNodes.size());
    RIntArray smoothMtxIdxs(boneCount);
    RIntArray rigidMtxIdxs(boneCount);
    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        smoothMtxIdxs[iNode] = ymodel.m_pYNodes[iNode]->m_SmoothMtxIdx;
        rigidMtxIdxs[iNode]  = ymodel.m_pYNodes[iNode]->m_RigidMtxIdx;
    }

    //-----------------------------------------------------------------------------
    // init
    const int shapeDataCount = static_cast<int>(ymodel.m_pOutShapeDatas.size());
    if (shapeDataCount == 0)
    {
        return MS::kSuccess;
    }

    RProgShape progShape(ShapeWarningFunc, &yscene);

    std::ostringstream shapeOs;
    RInitOutStreamFormat(shapeOs);

    RVertexArray vertices;

    //-----------------------------------------------------------------------------
    // loop for shape data
    //RTimeMeasure tm1;
    for (int iShape = 0; iShape < shapeDataCount; ++iShape)
    {
        const ShapeData& shapeData = *ymodel.m_pOutShapeDatas[iShape];
        const RIntArray* pBoneMtxIdxs = (shapeData.m_SkinningMode == RShape::SMOOTH) ?
            &smoothMtxIdxs : &rigidMtxIdxs;
        status = OutputShape(shapeOs, vertices, dataStreams, ymodel,
            tc + 1, iShape, shapeData, *pBoneMtxIdxs, progShape);
        CheckStatus(status);
    }
    //cerr << "shapes out: " << tm1.GetMilliSec() << endl;

    //-----------------------------------------------------------------------------
    // out vertex array
    ROutArrayElement(os, 0, vertices, "vertex_array");

    //-----------------------------------------------------------------------------
    // out shape array
    os << RTab(tc) << "<shape_array length=\"" << shapeDataCount << "\">" << R_ENDL;
    os << shapeOs.str();
    os << RTab(tc) << "</shape_array>" << R_ENDL;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief スケルトンを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <skeleton> 要素のインデントに必要なタブの数です。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static void OutputSkeleton(std::ostream& os, const int tc, const YModel& ymodel)
{
    //-----------------------------------------------------------------------------
    // begin skeleton
    const int boneCount = static_cast<int>(ymodel.m_pOutYNodes.size());
    if (boneCount == 0)
    {
        return;
    }
    os << RTab(tc) << "<skeleton>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // skeleton info
    const RSkeleton& skeleton = ymodel.m_Skeleton;
    skeleton.OutInfo(os, tc + 1);

    //-----------------------------------------------------------------------------
    // loop for bone
    os << RTab(tc + 1) << "<bone_array length=\"" << boneCount << "\">" << R_ENDL;
    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pOutYNodes[iNode];
        ynode.m_Scale       = GetRVec3(ynode.m_OutBindS, true);
        ynode.m_Rotate      = GetRVec3(ynode.m_OutBindR, true).ShiftAngle();
        ynode.m_Translate   = GetRVec3(ynode.m_OutBindT, true);
        ynode.m_InvModelMtx = GetRMtx44(ynode.m_BindGlobalInvMtx.transpose()).SnapToZero();
        // ↑プラグイン内部での行列は座標を前から乗算する形式なので、転置してから出力します。

        ynode.Out(os, tc + 2, ynode.m_Index, ynode.GetOutParentName());
    }
    os << RTab(tc + 1) << "</bone_array>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end skeleton
    os << RTab(tc) << "</skeleton>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief fmd ファイル（モデルデータ）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFmdFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    YModel& ymodel
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    const int tc = 0;

    //-----------------------------------------------------------------------------
    // materials
    OutputMaterials(os, tc, ymodel);

    //-----------------------------------------------------------------------------
    // skeleton
    OutputSkeleton(os, tc, ymodel);

    //-----------------------------------------------------------------------------
    // shapes
    status = OutputShapes(os, dataStreams, ymodel, tc);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // original materials
    OutOriginalMaterials(os, tc, ymodel);

    //-----------------------------------------------------------------------------
    // モデルのユーザーデータ配列
    if (ymodel.m_UserDataXformPath.isValid())
    {
        RUserDataArray modelUserDatas;
        GetUserData(modelUserDatas, yscene, ymodel.m_UserDataXformPath.node());
        ROutArrayElement(os, tc, modelUserDatas, "user_data_array");
    }

    //-----------------------------------------------------------------------------
    // debug
    //const int nodeCount = ymodel.m_pOutYNodes.size();
    //for (int iNode = 0; iNode < nodeCount; ++iNode)
    //{
    //  const YNode& ynode = *ymodel.m_pOutYNodes[iNode];
    //  cerr << ynode.m_Name << " ";
    //  if (ynode.m_pOutParent != NULL)
    //  {
    //      cerr << ynode.m_pOutParent->m_Name << " ";
    //  }
    //  else
    //  {
    //      cerr << "-" << " ";
    //  }
    //  cerr << endl;
    //}

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] tc <bone_anim> 要素のインデントに必要なタブの数です。
//! @param[in] animIdx ボーンアニメーションのインデックスです。
//! @param[in] ynode ノードです。
//-----------------------------------------------------------------------------
static void OutBoneAnim(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tc,
    const int animIdx,
    const YNode& ynode
)
{
    //-----------------------------------------------------------------------------
    // begin bone anim
    os << RTab(tc) << "<bone_anim index=\"" << animIdx
       << "\" bone_name=\"" << RGetUtf8FromShiftJis(ynode.m_Name)
       << "\" parent_name=\"" << RGetUtf8FromShiftJis(ynode.GetOutParentName()) << "\"" << R_ENDL;
    ynode.OutMtxAttrib(os, tc + 1);
    os << RTab(tc + 1) << "scale_compensate=\"" << RBoolStr(ynode.m_ScaleCompensate) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "binarize_scale=\"" << RBoolStr(ynode.m_BinarizesScale) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "binarize_rotate=\"" << RBoolStr(ynode.m_BinarizesRotate) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "binarize_translate=\"" << RBoolStr(ynode.m_BinarizesTranslate) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "compress_enable=\"" << RBoolStr(ynode.m_CompressEnable) << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // bone anim targets
    for (int paramIdx = 0; paramIdx < RBone::PARAM_COUNT; ++paramIdx)
    {
        const YAnimCurve& curve = ynode.m_Anims[paramIdx];
        curve.Out(os, dataStreams, tc + 1, "bone_anim_target",
            RBone::GetParamName(paramIdx), false);
    }

    //-----------------------------------------------------------------------------
    // end bone anim
    os << RTab(tc) << "</bone_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief fsk ファイル（スケルタルアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFskFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const YModel& ymodel
)
{
    YScene& yscene = ymodel.GetScene();

    const int tc = 0;

    //-----------------------------------------------------------------------------
    // bone anim array
    const int boneCount = static_cast<int>(ymodel.m_pOutYNodes.size());
    os << RTab(tc) << "<bone_anim_array length=\"" << boneCount << "\">" << R_ENDL;
    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        OutBoneAnim(os, dataStreams, tc + 1, iNode, *ymodel.m_pOutYNodes[iNode]);
    }
    os << RTab(tc) << "</bone_anim_array>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // スケルタルアニメーションのユーザーデータ配列
    if (ymodel.m_FskUserDataXformPath.isValid())
    {
        RUserDataArray fskUserDatas;
        GetUserData(fskUserDatas, yscene, ymodel.m_FskUserDataXformPath.node());
        ROutArrayElement(os, tc, fskUserDatas, "user_data_array");
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ボーンビジビリティアニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] tc <bone_vis_bone_anim> 要素のインデントに必要なタブの数です。
//! @param[in] animIdx ボーンビジビリティアニメーションのインデックスです。
//! @param[in] ynode ノードです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void OutBoneVisAnim(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tc,
    const int animIdx,
    const YNode& ynode,
    const YExpOpt& yopt
)
{
    const YAnimCurve& curve = ynode.m_VisAnim;
    os << RTab(tc) << "<bone_vis_bone_anim index=\"" << animIdx
       << "\" bone_name=\"" << RGetUtf8FromShiftJis(ynode.m_Name)
       << "\" parent_name=\"" << RGetUtf8FromShiftJis(ynode.GetOutParentName()) << "\"" << R_ENDL;
    ynode.OutMtxAttrib(os, tc + 1);
    const bool binarizesBoneVis = (
        yopt.m_ExportsBoneVisAll        ||
        ynode.m_VisibilityAnimFlag      ||
        ynode.m_ShapeVisibilityAnimFlag);
    os << RTab(tc + 1) << "binarize_visibility=\"" << RBoolStr(binarizesBoneVis) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "compress_enable=\"" << RBoolStr(ynode.m_CompressEnable) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "base_value=\"" << curve.GetBaseValue() << "\"" << R_ENDL;
    if (curve.m_ConstantFlag)
    {
        os << RTab(tc) << "/>" << R_ENDL;
    }
    else
    {
        os << RTab(tc) << ">" << R_ENDL;

        curve.Out(os, dataStreams, tc + 1, false);

        os << RTab(tc) << "</bone_vis_bone_anim>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief fvb ファイル（ボーンビジビリティアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFvbFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const YModel& ymodel
)
{
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // ベイクしたデータからアニメーションカーブを作成します。
    const int boneCount = static_cast<int>(ymodel.m_pOutYNodes.size());
    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        YNode& ynode = *ymodel.m_pOutYNodes[iNode];
        YAnimCurve& curve = ynode.m_VisAnim;
        curve.m_Name = ynode.m_Name + ".v";
        curve.m_Tolerance = 0.0f;

        curve.UpdateConstantFlag();
        if (!curve.m_ConstantFlag)
        {
            curve.MakeStepKeys(GetFloatFrameFromSubFrame, &yopt);
        }
    }

    //-----------------------------------------------------------------------------
    // bone vis bone anim array
    const int tc = 0;
    os << RTab(tc) << "<bone_vis_bone_anim_array length=\"" << boneCount << "\">" << R_ENDL;
    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        OutBoneVisAnim(os, dataStreams, tc + 1, iNode, *ymodel.m_pOutYNodes[iNode], yopt);
    }
    os << RTab(tc) << "</bone_vis_bone_anim_array>" << R_ENDL;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief カラーアニメーションが設定されたマテリアルの数を返します。
//!
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static int GetColorAnimtedMaterialCount(const YModel& ymodel)
{
    int animatedCount = 0;
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
        if (mat.m_ColAnimFlag)
        {
            ++animatedCount;
        }
    }
    return animatedCount;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT アニメーションが設定されたマテリアルの数を返します。
//!
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static int GetTexSrtAnimtedMaterialCount(const YModel& ymodel)
{
    int animatedCount = 0;
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
        if (mat.m_TexSrtAnimFlag)
        {
            ++animatedCount;
        }
    }
    return animatedCount;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションが設定されたマテリアルの数を返します。
//!
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static int GetTexPatAnimtedMaterialCount(const YModel& ymodel)
{
    int animatedCount = 0;
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
        if (mat.m_TexPatAnimFlag)
        {
            ++animatedCount;
        }
    }
    return animatedCount;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのカラーアニメーション配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] tc カラーアニメーション配列要素のインデントに必要なタブの数です。
//! @param[in] mat マテリアルです。
//-----------------------------------------------------------------------------
static void OutputColorAnims(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tc,
    const YMaterial& mat
)
{
    const int RGB_COUNT = 3;
    static const char* const hintStrs[] =
    {
        "diffuse" ,
        "opacity" ,
        "ambient" ,
        "emission",
        "specular"
    };
    static const char* const targetStrs[] = { "color_r", "color_g", "color_b" };

    //-----------------------------------------------------------------------------
    // begin original color anim array
    int animatedColorCount = 0;
    for (int paramIdx = 0; paramIdx < RMaterial::COLOR_PARAM_COUNT; paramIdx += RGB_COUNT)
    {
        if (mat.m_Anims[paramIdx + 0].m_UseFlag ||
            mat.m_Anims[paramIdx + 1].m_UseFlag ||
            mat.m_Anims[paramIdx + 2].m_UseFlag)
        {
            ++animatedColorCount;
        }
    }
    os << RTab(tc) << "<original_color_anim_array length=\"" << animatedColorCount << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // loop for color attr
    int colorAnimIdx = 0;
    for (int paramIdx = 0; paramIdx < RMaterial::COLOR_PARAM_COUNT; paramIdx += RGB_COUNT)
    {
        if (mat.m_Anims[paramIdx + 0].m_UseFlag ||
            mat.m_Anims[paramIdx + 1].m_UseFlag ||
            mat.m_Anims[paramIdx + 2].m_UseFlag)
        {
            os << RTab(tc + 1) << "<original_color_anim index=\"" << colorAnimIdx
               << "\" hint=\"" << hintStrs[paramIdx / RGB_COUNT]
               << "\">" << R_ENDL;
            for (int rgbIdx = 0; rgbIdx < RGB_COUNT; ++rgbIdx)
            {
                const YAnimCurve& curve = mat.m_Anims[paramIdx + rgbIdx];
                if (curve.m_UseFlag)
                {
                    curve.Out(os, dataStreams, tc + 2, "original_color_anim_target",
                        targetStrs[rgbIdx], true);
                }
            }
            os << RTab(tc + 1) << "</original_color_anim>" << R_ENDL;
            ++colorAnimIdx;
        }
    }

    //-----------------------------------------------------------------------------
    // end original color anim array
    os << RTab(tc) << "</original_color_anim_array>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief fcl ファイル（カラーアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFclFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const YModel& ymodel
)
{
    MStatus status;
    const int tc = 0;

    const int animatedCount = GetColorAnimtedMaterialCount(ymodel);
    if (animatedCount != 0)
    {
        os << RTab(tc) << "<original_material_anim_array length=\"" << animatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
            if (mat.m_ColAnimFlag)
            {
                os << RTab(tc + 1) << "<original_material_anim index=\"" << matAnimIdx
                   << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputColorAnims(os, dataStreams, tc + 2, mat);
                os << RTab(tc + 1) << "</original_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << RTab(tc) << "</original_material_anim_array>" << R_ENDL;
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのテクスチャ SRT アニメーション配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] tc テクスチャ SRT アニメーション配列要素のインデントに必要なタブの数です。
//! @param[in] mat マテリアルです。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static void OutputTexSrtAnims(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tc,
    const YMaterial& mat,
    const YModel& ymodel
)
{
    //-----------------------------------------------------------------------------
    // begin original texsrt anim array
    int animatedTexSrtCount = 0;
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        if (mat.m_Samplers[iSampler].m_TexSrtAnimIndex != -1)
        {
            ++animatedTexSrtCount;
        }
    }
    os << RTab(tc) << "<original_texsrt_anim_array length=\"" << animatedTexSrtCount << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // loop for tex sampler
    int outAnimIdx = 0;
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        const int texSrtAnimIdx = mat.m_Samplers[iSampler].m_TexSrtAnimIndex;
        if (texSrtAnimIdx != -1)
        {
            const TexSrtAnim& texSrtAnim = ymodel.m_TexSrtAnims[texSrtAnimIdx];
            const RSampler& sampler = mat.m_Samplers[iSampler];
            os << RTab(tc + 1) << "<original_texsrt_anim index=\"" << outAnimIdx
               << "\" hint=\"" << sampler.GetHintString() << "\"" << R_ENDL;
            os << RTab(tc + 2) << "mode=\"" << sampler.m_OriginalTexsrt.GetModeString() << "\"" << R_ENDL;
            os << RTab(tc + 1) << ">" << R_ENDL;
            for (int paramIdx = 0; paramIdx < ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
            {
                const YAnimCurve& curve = texSrtAnim.m_Anims[paramIdx];
                if (curve.m_UseFlag)
                {
                    curve.Out(os, dataStreams, tc + 2, "original_texsrt_anim_target",
                        ROriginalTexsrt::GetParamName(paramIdx), true);
                }
            }
            os << RTab(tc + 1) << "</original_texsrt_anim>" << R_ENDL;
            ++outAnimIdx;
        }
    }

    //-----------------------------------------------------------------------------
    // end original texsrt anim array
    os << RTab(tc) << "</original_texsrt_anim_array>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief fts ファイル（テクスチャ SRT アニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFtsFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const YModel& ymodel
)
{
    MStatus status;
    const int tc = 0;

    const int animatedCount = GetTexSrtAnimtedMaterialCount(ymodel);
    if (animatedCount != 0)
    {
        os << RTab(tc) << "<original_material_anim_array length=\"" << animatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
            if (mat.m_TexSrtAnimFlag)
            {
                os << RTab(tc + 1) << "<original_material_anim index=\"" << matAnimIdx
                   << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputTexSrtAnims(os, dataStreams, tc + 2, mat, ymodel);
                os << RTab(tc + 1) << "</original_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << RTab(tc) << "</original_material_anim_array>" << R_ENDL;
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターン配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] ymodel モデルです。
//! @param[in] tc テクスチャパターン配列要素のインデントに必要なタブの数です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputTexPatterns(std::ostream& os, YModel& ymodel, const int tc)
{
    RStringArray texPatRefs;
    MStatus status = CreateTexPatAnimData(ymodel, texPatRefs);
    CheckStatus(status);

    const int texPatCount = static_cast<int>(texPatRefs.size());
    os << RTab(tc) << "<tex_pattern_array length=\"" << texPatCount
       << "\">" << R_ENDL;
    for (int texPatIdx = 0; texPatIdx < texPatCount; ++texPatIdx)
    {
        os << RTab(tc + 1) << "<tex_pattern"
           << " pattern_index=\"" << texPatIdx
           << "\" tex_name=\"" << RGetUtf8FromShiftJis(REncodeXmlString(texPatRefs[texPatIdx]))
           << "\" />" << R_ENDL;
    }
    os << RTab(tc) << "</tex_pattern_array>" << R_ENDL;
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのテクスチャパターンアニメーション配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] isFma fma ファイルなら true、ftp ファイルなら false を指定します。
//! @param[in] tc テクスチャパターンアニメーション配列要素のインデントに必要なタブの数です。
//! @param[in] mat マテリアルです。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
static void OutputTexPatAnims(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const bool isFma,
    const int tc,
    const YMaterial& mat,
    const YModel& ymodel
)
{
    int animatedSamplerCount = 0;
    for (size_t samplerIdx = 0; samplerIdx < mat.m_Samplers.size(); ++samplerIdx)
    {
        const RSampler& sampler = mat.m_Samplers[samplerIdx];
        if (sampler.m_TexPatAnimIndex != -1)
        {
            ++animatedSamplerCount;
        }
    }

    if (isFma)
    {
        os << RTab(tc) << "<tex_pattern_anim_array length=\""
           << animatedSamplerCount << "\">" << R_ENDL;
    }
    const int texPatAnimTc = (isFma) ? tc + 1 : tc;
    const char* texPatAnimElemName = (isFma) ?
        "pattern_anim" : "pattern_anim_target";
    for (size_t samplerIdx = 0; samplerIdx < mat.m_Samplers.size(); ++samplerIdx)
    {
        const RSampler& sampler = mat.m_Samplers[samplerIdx];
        if (sampler.m_TexPatAnimIndex != -1)
        {
            const TexPatAnim& texPatAnim = ymodel.m_TexPatAnims[sampler.m_TexPatAnimIndex];
            const YAnimCurve& curve = texPatAnim.m_ImgAnim;
            os << RTab(texPatAnimTc) << "<" << texPatAnimElemName
               << " sampler_name=\"" << sampler.GetName()
               << "\" hint=\"" << sampler.GetHintString()
               << "\" base_value=\"" << curve.GetBaseValue() << "\"";
            if (curve.m_ConstantFlag)
            {
                os << " />" << R_ENDL;
            }
            else
            {
                os << ">" << R_ENDL;
                curve.Out(os, dataStreams, texPatAnimTc + 1, false);
                os << RTab(texPatAnimTc) << "</" << texPatAnimElemName << ">" << R_ENDL;
            }
        }
    }
    if (isFma)
    {
        os << RTab(tc) << "</tex_pattern_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief ftp ファイル（テクスチャパターンアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFtpFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    YModel& ymodel
)
{
    MStatus status;
    const int tc = 0;

    const int animatedCount = GetTexPatAnimtedMaterialCount(ymodel);
    if (animatedCount != 0)
    {
        //-----------------------------------------------------------------------------
        // テクスチャパターン配列
        status = OutputTexPatterns(os, ymodel, tc);
        CheckStatus(status);

        //-----------------------------------------------------------------------------
        // tex pattern mat anim array
        os << RTab(tc) << "<tex_pattern_mat_anim_array length=\"" << animatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
            if (mat.m_TexPatAnimFlag)
            {
                os << RTab(tc + 1) << "<tex_pattern_mat_anim index=\"" << matAnimIdx
                   << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputTexPatAnims(os, dataStreams, false, tc + 2, mat, ymodel);
                os << RTab(tc + 1) << "</tex_pattern_mat_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << RTab(tc) << "</tex_pattern_mat_anim_array>" << R_ENDL;
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief fma ファイル（マテリアルアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//! @param[in] outputsColor カラーアニメーションを含めるなら true です。
//! @param[in] outputsTexSrt テクスチャ SRT アニメーションを含めるなら true です。
//! @param[in] outputsTexPat テクスチャパターンアニメーションを含めるなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFmaFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    YModel& ymodel,
    const bool outputsColor,
    const bool outputsTexSrt,
    const bool outputsTexPat
)
{
    MStatus status;
    const int tc = 0;

    //-----------------------------------------------------------------------------
    // テクスチャパターン配列
    const int texPatAnimatedCount =
        (outputsTexPat) ? GetTexPatAnimtedMaterialCount(ymodel) : 0;
    if (texPatAnimatedCount != 0)
    {
        status = OutputTexPatterns(os, ymodel, tc);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // マテリアル単位アニメーション配列
    const int matCount = static_cast<int>(ymodel.m_pOutYMaterials.size());
    const int animatedCount = texPatAnimatedCount;
    if (animatedCount != 0)
    {
        os << RTab(tc) << "<per_material_anim_array length=\"" << animatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        for (int matIdx = 0; matIdx < matCount; ++matIdx)
        {
            const YMaterial& mat = *ymodel.m_pOutYMaterials[matIdx];
            const bool hasTexPatAnim = (outputsTexPat && mat.m_TexPatAnimFlag);
            if (hasTexPatAnim)
            {
                os << RTab(tc + 1) << "<per_material_anim index=\"" << matAnimIdx
                   << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputTexPatAnims(os, dataStreams, true, tc + 2, mat, ymodel);
                os << RTab(tc + 1) << "</per_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << RTab(tc) << "</per_material_anim_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // オリジナルのマテリアル単位アニメーション配列
    const int orgAnimatedCount =
        ((outputsColor ) ? GetColorAnimtedMaterialCount(ymodel)  : 0) +
        ((outputsTexSrt) ? GetTexSrtAnimtedMaterialCount(ymodel) : 0);
    if (orgAnimatedCount != 0)
    {
        os << RTab(tc) << "<original_per_material_anim_array length=\"" << orgAnimatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        for (int matIdx = 0; matIdx < matCount; ++matIdx)
        {
            const YMaterial& mat = *ymodel.m_pOutYMaterials[matIdx];
            const bool hasColorAnim  = (outputsColor  && mat.m_ColAnimFlag   );
            const bool hasTexSrtAnim = (outputsTexSrt && mat.m_TexSrtAnimFlag);
            if (hasColorAnim || hasTexSrtAnim)
            {
                os << RTab(tc + 1) << "<original_per_material_anim index=\"" << matAnimIdx
                   << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                if (hasColorAnim)
                {
                    OutputColorAnims(os, dataStreams, tc + 2, mat);
                }
                if (hasTexSrtAnim)
                {
                    OutputTexSrtAnims(os, dataStreams, tc + 2, mat, ymodel);
                }
                os << RTab(tc + 1) << "</original_per_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << RTab(tc) << "</original_per_material_anim_array>" << R_ENDL;
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 出力する中間ファイルをオープンします。
//!
//! @param[in,out] ofs 出力ファイルストリームです。
//! @param[in,out] yscene シーンです。
//! @param[in] filePath 中間ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OpenOutputFile(
    std::ofstream& ofs,
    YScene& yscene,
    const std::string& filePath,
    const RExpOpt::FileType fileType
)
{
    #ifdef DEBUG_PRINT_SW
    const YExpOpt& yopt = yscene.GetOpt();
    const bool animFlag = (fileType < RExpOpt::FILE_TYPE_COUNT && fileType != RExpOpt::FMD);
    std::string dispPath = filePath;
    const std::string tmpPath = RGetTempFolderPath();
    if (!tmpPath.empty() && dispPath.find(tmpPath) == 0)
    {
        dispPath = "$TEMP/" + dispPath.substr(tmpPath.size(), dispPath.size() - 1);
    }
    cerr << "out " << yopt.GetExtension(fileType)
         << ": " << dispPath;
    if (animFlag)
    {
        cerr << " (" << yopt.m_OutFrameCount << " frames ["
             << yopt.m_StartFrame << " - " << yopt.m_EndFrame << "])";
        if (yopt.m_LoopAnim)
        {
            cerr << " (Loop)";
        }
    }
    cerr << R_ENDL;
    #else
    R_UNUSED_VARIABLE(fileType);
    #endif

    ofs.open(filePath.c_str(), ios_base::out | ios_base::binary);
    if (!ofs)
    {
        YShowError(&yscene,
            "ファイルを開けません。上書き禁止になっていないか確認してください: {0}",
            "The file cannot be opened. Confirm whether the file can be overwritten: {0}",
            filePath);
        return MS::kFailure;
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief マテリアル関連のアニメーション中間ファイルを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//! @param[in] fileType 中間ファイルタイプです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputOneMatAnimFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    YModel& ymodel,
    const RExpOpt::FileType fileType
)
{
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();
    if (yopt.UsesSingleFma())
    {
        return OutputFmaFile(os, dataStreams, ymodel,
            yopt.ExportsColorAnim(),
            yopt.ExportsTexSrtAnim(),
            yopt.ExportsTexPatAnim());
    }
    else if (yopt.UsesSeparatedFma())
    {
        switch (fileType)
        {
        case RExpOpt::FCL:
            return OutputFmaFile(os, dataStreams, ymodel, true , false, false);
        case RExpOpt::FTS:
            return OutputFmaFile(os, dataStreams, ymodel, false, true , false);
        case RExpOpt::FTP:
            return OutputFmaFile(os, dataStreams, ymodel, false, false, true );
        default:
            break;
        }
    }
    else
    {
        switch (fileType)
        {
        case RExpOpt::FCL:
            return OutputFclFile(os, dataStreams, ymodel);
        case RExpOpt::FTS:
            return OutputFtsFile(os, dataStreams, ymodel);
        case RExpOpt::FTP:
            return OutputFtpFile(os, dataStreams, ymodel);
        default:
            break;
        }
    }
    return MS::kFailure; // 通常はここに来ません。
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルを 1 つ出力します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] filePath 出力する中間ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputOneFile(
    YModel& ymodel,
    const std::string& filePath,
    const RExpOpt::FileType fileType
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    const YEnvObjs& yenvObjs = ymodel.GetEnvObjs();

    //-----------------------------------------------------------------------------
    // ファイルを開きます。
    const clock_t startClock = clock();
    const bool isBinary = yopt.IsBinaryFormat(fileType);
    std::ofstream ofs;
    status = OpenOutputFile(ofs, yscene, filePath, fileType);
    CheckStatus(status);
    std::ostream& os = ofs;

    ROutUtf8Bom(os);
    RInitOutStreamFormat(os);

    //-----------------------------------------------------------------------------
    // 各中間ファイルの内容を文字列ストリームに出力します。
    // （内容を出力しないと確定できないヘッダの情報があるため）
    std::ostringstream contentOs;
    RInitOutStreamFormat(contentOs);
    RDataStreamArray dataStreams;
    switch (fileType)
    {
    case RExpOpt::FMD:
        status = OutputFmdFile(contentOs, dataStreams, ymodel);
        break;
    case RExpOpt::FSK:
        status = OutputFskFile(contentOs, dataStreams, ymodel);
        break;
    case RExpOpt::FVB:
        status = OutputFvbFile(contentOs, dataStreams, ymodel);
        break;
    case RExpOpt::FCL:
    case RExpOpt::FTS:
    case RExpOpt::FTP:
        status = OutputOneMatAnimFile(contentOs, dataStreams, ymodel, fileType);
        break;
    case RExpOpt::FSH:
        status = OutputFshFile(contentOs, dataStreams,
            ymodel.m_VertexShapeAnims, ymodel.m_BlendShapeDatas);
        break;
    case RExpOpt::FSN:
        status = OutputFsnFile(contentOs, dataStreams, yenvObjs);
        break;
    default:
        break;
    }

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    OutputHeader(os, 0, fileType, ymodel, yopt);

    //-----------------------------------------------------------------------------
    // 各中間ファイルの内容を出力します。
    os << contentOs.str();

    //-----------------------------------------------------------------------------
    // アスキー形式の場合、ここでデータ列配列を出力します。
    if (status && !isBinary)
    {
        ROutDataStreams(os, dataStreams);
    }

    //-----------------------------------------------------------------------------
    // フッタを出力します。
    if (status)
    {
        OutputFooter(os, 0, fileType, yopt);
    }

    //-----------------------------------------------------------------------------
    // バイナリ形式の場合、ここでデータ列配列を出力します。
    if (status && isBinary)
    {
        ROutBinaryDataStreams(os, dataStreams, true);
    }

    //-----------------------------------------------------------------------------
    // ファイルを閉じます。
    ofs.close();

    //-----------------------------------------------------------------------------
    // プロファイル情報を設定します。
    const clock_t clocks = clock() - startClock;
    if (fileType == RExpOpt::FMD)
    {
        yscene.m_ModelClocks += clocks;
    }
    else if (RExpOpt::FSK <= fileType && fileType <= RExpOpt::FSN)
    {
        yscene.m_AnimClocks += clocks;
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief フォルダが存在しなければ作成します。
//!
//! @param[in] folderPath フォルダのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CreateFolderIfNotExist(RScene& rscene, const std::string& folderPath)
{
    MStatus status;

    if (!RFolderExists(folderPath))
    {
        #ifdef DEBUG_PRINT_SW
        cerr << "create: " << folderPath.c_str() << R_ENDL;
        #endif
        std::string cmd = "sysFile -md \"" + folderPath + "\"";
        int ret;
        MGlobal::executeCommand(cmd.c_str(), ret);
        if (!ret)
        {
            YShowError(&rscene, "フォルダを作成できません: {0}", "Cannot create folder: {0}", folderPath);
            return MS::kFailure;
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief フォルダが存在すれば削除します。
//!
//! @param[in] folderPath フォルダのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus RemoveFolderIfExist(const std::string& folderPath)
{
    if (RFolderExists(folderPath))
    {
        #ifdef DEBUG_PRINT_SW
        //cerr << "remove: " << folderPath.c_str() << R_ENDL;
        #endif
        if (!::RemoveDirectoryA(folderPath.c_str()))
        {
            return MS::kFailure;
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルをリードしてテクスチャイメージの属性を設定し、
//!        3D テクスチャーコンバーターのジョブリストに項目を追加します。
//!
//! @param[in,out] texImg テクスチャイメージです。
//! @param[in,out] cvtrOss 3D テクスチャーコンバーターのオプションの出力ストリームです。
//! @param[in,out] yscene シーンです。
//! @param[in] ftxPath 出力する ftx ファイルのパスです。空文字ならマージしません。
//! @param[in] mergePath マージする ftx ファイルのパスです。
//! @param[in] outFlag ftx ファイルを出力するなら true を指定します。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ReadTexImageAndAddJobList(
    RImage& texImg,
    std::ostringstream& cvtrOss,
    YScene& yscene,
    const std::string& ftxPath,
    const std::string& mergePath,
    const bool outFlag,
    const YExpOpt& yopt
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 画像ファイルをリードします。
    const RStringArray filePaths = texImg.GetFilePaths();
    status = ReadImageFileForMaya(texImg, yscene,
        filePaths,
        mergePath,
        yopt.m_PresetName,
        texImg.GetHint(),
        texImg.GetLinearFlag(),
        yopt.m_UsesSrgbFetch,
        texImg.GetDimension(),
        texImg.GetInitialSwizzle(),
        yopt.m_CommentText,
        yopt.m_CheckElementFlag || !outFlag); // ftx ファイルを出力しない場合はエラー表示を抑制します。

    //-----------------------------------------------------------------------------
    // ftx ファイルを出力しない場合でも、テクスチャファイルが存在しなければエラーを
    // 発生するならチェックします。
    if (!status && !outFlag && yopt.m_ChecksNoTextureFileErrorAlways && !yopt.m_CheckElementFlag)
    {
        for (size_t fileIdx = 0; fileIdx < filePaths.size(); ++fileIdx)
        {
            const std::string& filePath = filePaths[fileIdx];
            if (!RFileExists(filePath))
            {
                YShowError(&yscene, // Cannot open file: %s
                    "テクスチャーファイルが存在しません: {0}",
                    "The texture file cannot be found: {0}",
                    filePath);
                texImg.FreeMemory();
                return status;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 法線マップ用テクスチャならフォーマットと成分選択を設定します。
    if (texImg.GetHint() == RImage::HINT_NORMAL)
    {
        if (!texImg.IsFloat())
        {
            texImg.SetFormat(static_cast<RImage::Format>(yopt.m_NormalTextureFormat));
            texImg.SetCompSel(yopt.m_NormalTextureCompSel);
        }
    }

    //-----------------------------------------------------------------------------
    // プロジェクトのルートフォルダ、テクスチャの重み付け圧縮を設定します。
    texImg.SetProjectRootPath(yopt.m_ProjectRootPath);
    texImg.SetWeightedCompress(yopt.m_EnablesWeightedCompress);

    //-----------------------------------------------------------------------------
    // 3D テクスチャーコンバーターのジョブリストに項目を追加します。
    if (status && outFlag)
    {
        std::string cvtrOpt = texImg.GetConverterOption(ftxPath);
        cvtrOss << cvtrOpt << R_ENDL;
    }

    texImg.FreeMemory();

    return (outFlag) ? status : MS::kSuccess; // ftx ファイルを出力しない場合は常に成功とします。
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ中間ファイル群を出力します。
//!
//! @param[in] ymodel モデルです。
//! @param[in] outFlag 実際に出力する場合は true、
//!                    テクスチャデータを取得するだけの場合は false を指定します。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputTextureFiles(
    const YModel& ymodel,
    const bool outFlag,
    const YExpOpt& yopt
)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();

    //-----------------------------------------------------------------------------
    // モデルのテクスチャイメージ数をチェックします。
    const int texImgCount = static_cast<int>(ymodel.m_pOutTexImgs.size());
    if (texImgCount == 0)
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // テクスチャ出力フォルダを作成します。
    if (yopt.ExportsTexture())
    {
        status = CreateFolderIfNotExist(yscene, yopt.m_TmpTexFolderPath);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // 画像ファイルをリードして 3D テクスチャーコンバーターのジョブリストを作成します。
    YSetHelpLine("Making ... ftx");
    const clock_t startClock = clock();

    std::ostringstream cvtrOss;
    for (int iTexImg = 0; iTexImg < texImgCount; ++iTexImg)
    {
        //-----------------------------------------------------------------------------
        // 同じテクスチャ名を持つテクスチャイメージがすでに出力されているか判定します。
        RImage& texImg = *ymodel.m_pOutTexImgs[iTexImg];
        bool newTexFlag = true;
        const int otherCount = static_cast<int>(yscene.m_AllTexImgs.size());
        for (int iOther = 0; iOther < otherCount; ++iOther)
        {
            const RImage& other = yscene.m_AllTexImgs[iOther];
            if (other.GetName() == texImg.GetName())
            {
                newTexFlag = false;
                // 同じテクスチャ名を持つテクスチャイメージがすでに出力されていた場合、
                // テクスチャファイルパスを比較して異なれば警告を表示します。
                CheckTextureIdentical(yscene, texImg, other);
                break;
            }
        }
        //cerr << "out tex: " << texImg.GetName() << ": " << newTexFlag << R_ENDL;

        //-----------------------------------------------------------------------------
        // 画像ファイルをリードしてテクスチャイメージの属性を設定します。
        const std::string ftxName = texImg.GetName() + "." + yopt.GetExtension(RExpOpt::FTX);
        const std::string ftxPath = yopt.m_TmpTexFolderPath + ftxName;
        std::string mergePath = (yopt.m_MergeFtxFlag) ? yopt.m_MergeFtxFolderPath + ftxName : "";
        mergePath = (yopt.m_MergeFtxFlag && RFileExists(mergePath)) ? mergePath : "";

        status = ReadTexImageAndAddJobList(texImg, cvtrOss, yscene,
            ftxPath, mergePath, outFlag && newTexFlag, yopt);
        CheckStatus(status);

        //-----------------------------------------------------------------------------
        // マージする場合はテクスチャ名を記録します。
        if (newTexFlag && !mergePath.empty())
        {
            yscene.m_MergedTexImgNames.push_back(texImg.GetName());
        }

        //-----------------------------------------------------------------------------
        // 出力済みのテクスチャイメージを記録します。
        if (newTexFlag)
        {
            yscene.m_AllTexImgs.push_back(texImg);
        }

        //-----------------------------------------------------------------------------
        // ファイル移動情報を追加します。
        if (outFlag && newTexFlag)
        {
            yscene.m_TmpFileMoves.push_back(
                RFileMove(ftxPath, yopt.m_TexFolderPath + ftxName));
        }
    }

    //-----------------------------------------------------------------------------
    // マージ対象のテクスチャが 1 つも無い場合は警告します。
    // TODO: アニメーションレンジ出力に対応するなら全モデル出力後に判定
    if (yopt.MergesFtx() && yscene.m_MergedTexImgNames.empty())
    {
        YShowWarning(&yscene, // ftx file to merge is not found
            "Merge ftx File が ON ですが、マージ対象の ftx ファイルが 1 つも存在しません。マージしないで出力します。",
            "No ftx files to merge could be found even though the Merge ftx File is selected. Files are exported without merging.");
    }

    //-----------------------------------------------------------------------------
    // 3D テクスチャーコンバーターを実行します。
    const std::string cvtrStr = cvtrOss.str();
    if (!cvtrStr.empty())
    {
        //-----------------------------------------------------------------------------
        // 3D テクスチャーコンバーターの exe ファイルが存在するかチェックします。
        const std::string cvtrPath = GetNintendoMaya3dTextureConverterPath();
        if (!RFileExists(cvtrPath))
        {
            YShowError(&yscene, // 3D texture converter cannot be found: (at %s)
                "3D テクスチャーコンバーターが見つかりません: {0}",
                "3D texture converter cannot be found: {0}",
                cvtrPath);
            return MS::kFailure;
        }

        //-----------------------------------------------------------------------------
        // ジョブリストファイルを作成します。
        //cerr << "tex cvtrOss:" << endl << cvtrStr << endl;
        const std::string jobListFilePath = yopt.m_TmpOutFolderPath +
            "Nintendo_3dTextureConverter_" + RGetCurrentProcessIdString() + ".txt";
        std::ofstream ofs;
        ofs.open(jobListFilePath.c_str(), ios_base::out | ios_base::binary);
        if (!ofs)
        {
            YShowError(&yscene, "ファイルを開けません: {0}", "The file cannot be opened: {0}", jobListFilePath);
            return MS::kFailure;
        }
        ofs.write(cvtrStr.c_str(), cvtrStr.size());
        ofs.close();

        //-----------------------------------------------------------------------------
        // do
        std::string cmd = "\"" + RGetWindowsFilePath(cvtrPath) + "\"";
        cmd += " -s"; // 処理時間を表示する場合も 3D テクスチャーコンバーターの情報表示は抑制します。
        cmd += " --final-folder \"" + yopt.m_TexFolderPath + "\"";
        cmd += " --job-list \"" + jobListFilePath + "\"";
        if (yopt.m_DisablesOriginalImage)
        {
            cmd += " --disable-original-image";
        }
        if (!yopt.m_GpuEncoding.empty())
        {
            cmd += " --gpu-encoding " + yopt.m_GpuEncoding;
        }
        //cerr << "texcvtr: " << cmd << endl;
        if (yopt.m_DisplaysProfile) // 処理時間を表示する場合も 3D テクスチャーコンバーターのコマンドライン表示は抑制します。
        {
            //DisplayExportInfoSub(cmd, yopt);
        }

        std::string outMsg;
        std::string errMsg;
        const int exitCode = RExecProcessWithPipe(nullptr, &outMsg, &errMsg, cmd.c_str(), SW_HIDE);
        //cerr << "texcvtr msg:" << endl << errMsg << endl;

        //if (!outMsg.empty())
        //{
        //    DisplayExportInfoSub(outMsg, yopt);
        //}

        if (!errMsg.empty())
        {
            DisplayExportInfoSub(errMsg, yopt);
        }

        if (exitCode != 0)
        {
            YShowError(&yscene, // Cannot export ftx file
                "ftx ファイルを出力できませんでした。（詳細はスクリプトエディタを見てください）",
                "ftx files could not be exported. (See the Script Editor for details.)");
            status = MS::kFailure;
        }

        //-----------------------------------------------------------------------------
        // ジョブリストファイルを削除します。
        ::DeleteFileA(jobListFilePath.c_str());

        //-----------------------------------------------------------------------------
        // profile
        yscene.m_TexClocks += clock() - startClock;
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief check no output file
//!
//! @param[in,out] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//! @param[in] texCount 出力した ftx ファイル数です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CheckNoOutputFile(
    YScene& yscene,
    const YExpOpt& yopt,
    const int texCount
)
{
    if (!yopt.ExportsModelFile()   &&
        !yopt.ExportsAnim()        &&
        texCount == 0)
    {
        YShowError(&yscene, // No output file
            "中間ファイルの出力指定がすべて OFF になっているか、"
            "ftx ファイルの出力指定のみ ON で出力するテクスチャーがありません。",
            "All output specifications for the intermediate file are cleared, "
            "or only the output specification for the ftx file is selected but there is no texture to export.");
        DisplayExportInfoSub("---- No Output File ----", yopt);
        return MS::kFailure;
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief アニメーション中間ファイルの出力が指定されていて
//!        対応するアニメーションが設定されていない場合に警告を表示します。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void WarnNoAnim(const YExpOpt& yopt)
{
    R_UNUSED_VARIABLE(yopt);
//  if (yopt.m_OutFileFlag[RExpOpt::FSH])
//  {
//      if (GetOutShapeAnimCount(ymodel.m_VertexShapeAnims, ymodel.m_BlendShapeDatas) == 0 && // changed (2006/07/18)
//          !yopt.m_NoAnimWarnedFlag[RExpOpt::FSH])
//      {
//          YShowWarning(&yscene, "", "No shape animation"); // 現在は発生しない
//          yopt.m_NoAnimWarnedFlag[RExpOpt::FSH] = true;
//      }
//  }
}

//-----------------------------------------------------------------------------
//! @brief モデルについての各タイプの中間ファイル群を出力します。
//!
//! @param[out] fileCount 出力した中間ファイル数を格納します。
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus OutputFiles(int& fileCount, YModel& ymodel)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // check no output file
    status = CheckNoOutputFile(yscene, yopt, static_cast<int>(ymodel.m_pOutTexImgs.size()));
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // warn no anim
    WarnNoAnim(yopt);

    //-----------------------------------------------------------------------------
    // テクスチャ中間ファイル群を出力します。
    if (yopt.ExportsModelFile() ||
        yopt.ExportsTexture())
    {
        status = OutputTextureFiles(ymodel, yopt.ExportsTexture(), yopt);
        CheckStatus(status);

        fileCount += static_cast<int>(ymodel.m_pOutTexImgs.size());
    }

    //-----------------------------------------------------------------------------
    // テクスチャ以外の中間ファイル群を出力します。
    bool isSingleFmaOutput = false;
    for (int fileTypeIdx = 0; fileTypeIdx < RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const RExpOpt::FileType fileType = static_cast<RExpOpt::FileType>(fileTypeIdx);
        if (yopt.m_OutFileFlag[fileType])
        {
            if (yopt.UsesSingleFma() && yopt.IsFclFtsFtp(fileType))
            {
                if (isSingleFmaOutput)
                {
                    continue;
                }
                isSingleFmaOutput = true;
            }
            const std::string ext = yopt.GetExtension(fileType);
            const std::string suffix = yopt.GetSuffix(fileType);
            const std::string extMsg = (!suffix.empty()) ? suffix + "." + ext : ext;
            YSetHelpLine("Making ... %s", extMsg.c_str());
            const std::string tmpFilePath = yopt.GetTmpFilePath(fileType);
            status = OutputOneFile(ymodel, tmpFilePath, fileType);
            CheckStatus(status);
            yscene.m_TmpFileMoves.push_back(
                RFileMove(tmpFilePath, yopt.GetFilePath(fileType)));
            ++fileCount;
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーのアニメーション中間ファイルマージ引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザーの引数の文字列へのポインタです。
//! @param[in] yopt エクスポートオプションです。
//! @param[in] fileType ファイルタイプです。
//-----------------------------------------------------------------------------
static void Add3dOptimizerMergeAnimArg(
    std::string* pArgStr,
    const YExpOpt& yopt,
    const YExpOpt::FileType fileType
)
{
    if (yopt.m_MergeAnimFlag)
    {
        const std::string mergeAnimPath = yopt.GetMergeAnimFilePath(fileType);
        if (RFileExists(mergeAnimPath))
        {
            *pArgStr += " -m --merge-options=\"--merge-file='" + mergeAnimPath + "'\"";
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーのボーン圧縮引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザーの引数の文字列へのポインタです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void Add3dOptimizerCompressBoneArg(std::string* pArgStr, const YExpOpt& yopt)
{
    if (yopt.m_CompressBone == RExpOpt::COMPRESS_BONE_CULL)
    {
        *pArgStr += " --compress-bone-cull";
    }
    else if (yopt.m_CompressBone == RExpOpt::COMPRESS_BONE_MERGE)
    {
        *pArgStr += " --compress-bone-merge";
    }
    else if (yopt.m_CompressBone == RExpOpt::COMPRESS_BONE_UNITE)
    {
        *pArgStr += " --compress-bone-unite";
    }
    else if (yopt.m_CompressBone == RExpOpt::COMPRESS_BONE_UNITE_ALL)
    {
        *pArgStr += " --compress-bone-unite-all";
    }

    if (yopt.UnitesChildBone())
    {
        *pArgStr += " --compress-bone-unite-child";
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーのプリミティブ最適化引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザーの引数の文字列へのポインタです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void Add3dOptimizerPrimitiveArg(std::string* pArgStr, const YExpOpt& yopt)
{
    if (yopt.m_OptimizePrimitive != RExpOpt::OPTIMIZE_PRIMITIVE_NONE)
    {
        *pArgStr += " --optimize-primitive --optimize-primitive-options=\"--mode=";
        if (yopt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_DEFAULT)
        {
            *pArgStr += "Normal";
        }
        else if (yopt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_FORCE)
        {
            *pArgStr += "Force";
        }
        else if (yopt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_BRUTE_FORCE)
        {
            *pArgStr += "BruteForce";
        }
        else if (yopt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_FORSYTH)
        {
            *pArgStr += "Forsyth";
        }
        *pArgStr += "\"";
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーの量子化分析引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザーの引数の文字列へのポインタです。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static void Add3dOptimizerQuantizationAnalysisArg(std::string* pArgStr, const YExpOpt& yopt)
{
    if (yopt.m_QuantizationAnalysis)
    {
        *pArgStr += " --quantization-analysis";
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーの fmd 最適化引数群を取得します。
//!
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
static std::string Get3dOptimizerFmdArgs(const YExpOpt& yopt)
{
    const std::string fmdPath = yopt.GetTmpFilePath(RExpOpt::FMD);
    std::string argStr = " \"" + fmdPath + "\"";
    if (yopt.MergesFmd())
    {
        argStr += " -m --merge-options=\"--merge-file='" + yopt.m_MergeFmdPath + "'";
        if (yopt.m_SamplerMergePriority)
        {
            argStr += " --sampler-merge-priority";
        }
        if (yopt.m_BoneVisMergePriority)
        {
            argStr += " --bone-visibility-merge-priority";
        }
        argStr += "\"";
    }
    Add3dOptimizerCompressBoneArg(&argStr, yopt);
    if (yopt.m_CompressMaterial)
    {
        argStr += " --compress-material";
    }
    if (yopt.m_CompressShape)
    {
        argStr += " --compress-shape";
        if (yopt.m_CompressesIgnoringVertexSkinningCount)
        {
            argStr += " --compress-shape-options=\"";
            argStr += "--ignore-skinning-count";
            argStr += "\"";
        }
    }
    if (yopt.m_DivideSubmesh && !yopt.m_DivideSubmeshMode.empty())
    {
        argStr += " --divide-submesh --divide-submesh-options=\"" + yopt.m_DivideSubmeshMode + "\"";
    }
    if (yopt.m_PolyReduction && !yopt.m_PolyReductionMode.empty())
    {
        argStr += " --polygon-reduction --polygon-reduction-options=\"" + yopt.m_PolyReductionMode + "\"";
    }

    if (!yopt.m_ExportsLod)
    {
        Add3dOptimizerPrimitiveArg(&argStr, yopt);
        Add3dOptimizerQuantizationAnalysisArg(&argStr, yopt);
    }

    if (!yopt.m_DeleteNearVertex.empty())
    {
        argStr += " --delete-near-vertex --delete-near-vertex-options=\"" + yopt.m_DeleteNearVertex + "\"";
    }
    if (!yopt.m_OptimizerExtraOptionFmd.empty())
    {
        argStr += " " + yopt.m_OptimizerExtraOptionFmd;
    }
    argStr += " -o \"" + fmdPath + "\"";
    return argStr;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーのアニメーション最適化引数群を取得します。
//!
//! @param[in] yopt エクスポートオプションです。
//! @param[in] fileType ファイルタイプです。
//-----------------------------------------------------------------------------
static std::string Get3dOptimizerAnimArgs(
    const YExpOpt& yopt,
    const RExpOpt::FileType fileType
)
{
    const std::string filePath = yopt.GetTmpFilePath(fileType);
    std::string argStr = " \"" + filePath + "\"";
    Add3dOptimizerMergeAnimArg(&argStr, yopt, fileType);
    if (fileType == YExpOpt::FSK ||
        fileType == YExpOpt::FVB)
    {
        Add3dOptimizerCompressBoneArg(&argStr, yopt);
    }
    Add3dOptimizerQuantizationAnalysisArg(&argStr, yopt);
    argStr += " -o \"" + filePath + "\"";
    return argStr;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザーを実行します。
//!
//! @param[in,out] yscene シーンです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus Do3dOptimizer(YScene& yscene)
{
    //-----------------------------------------------------------------------------
    // 最適化が必要かチェックします。
    const YExpOpt& yopt = yscene.GetOpt();
    if (!yopt.Uses3dOptimizer())
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザーの exe ファイルが存在するかチェックします。
    std::string cvtrPath = yopt.m_G3dToolPath + "3dIntermediateFileOptimizer.exe";
    if (!RFileExists(cvtrPath))
    {
        cvtrPath = yopt.m_G3dToolPath + "NW4F_g3doptcvtr.exe";
    }
    if (!RFileExists(cvtrPath))
    {
        YShowError(&yscene, // 3D intermediate file optimizer cannot be found: (at %s) // 通常は発生しない
            "中間ファイルオプティマイザーが見つかりません: {0}",
            "The intermediate file optimizer cannot be found: {0}",
            cvtrPath);
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザーの引数を設定します。
    std::string cmdTop = "\"" + RGetWindowsFilePath(cvtrPath) + "\"";
    if (!yopt.m_DisplaysProfile)
    {
        cmdTop += " -s"; // silent
    }
    RStringArray cmds;

    if (yopt.OptimizesFmd() || yopt.MergesFmd())
    {
        cmds.push_back(cmdTop + Get3dOptimizerFmdArgs(yopt));
    }

    if (yopt.OptimizesFsk() ||
        (yopt.MergesFsk() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FSK))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FSK));
    }

    if (yopt.OptimizesFvb() ||
        (yopt.MergesFvb() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FVB))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FVB));
    }

    if (yopt.UsesSingleFma())
    {
        if (yopt.OptimizesFma() ||
            (yopt.MergesFma() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FCL))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FCL));
        }
    }
    else
    {
        if (yopt.OptimizesFcl() ||
            (yopt.MergesFcl() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FCL))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FCL));
        }

        if (yopt.OptimizesFts() ||
            (yopt.MergesFts() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FTS))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FTS));
        }

        if (yopt.OptimizesFtp() ||
            (yopt.MergesFtp() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FTP))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FTP));
        }
    }

    if (yopt.OptimizesFsh() ||
        (yopt.MergesFsh() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FSH))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FSH));
    }

    if (yopt.OptimizesFsn() ||
        (yopt.MergesFsn() && RFileExists(yopt.GetMergeAnimFilePath(YExpOpt::FSN))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(yopt, RExpOpt::FSN));
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザーを実行します。
    //cerr << "optimizer: " << RJoinStrings(cmds, "\n") << endl;
    const clock_t startClock = clock();
    #if 1
    if (yopt.OptimizesSome() && (yopt.MergesFmd() || yopt.MergesAnim()))
    {
        YSetHelpLine("Optimizing and Merging ...");
    }
    else if (yopt.OptimizesSome())
    {
        YSetHelpLine("Optimizing ...");
    }
    else
    {
        YSetHelpLine("Merging ...");
    }

    int exitCode = 0;
    std::string outMsg;
    std::string errMsg;
    std::string* pOutMsg = (yopt.m_DisplaysProfile) ? &outMsg : nullptr;
    //RTimeMeasure tm1;
    #ifdef DO_OPTIMIZER_PARALLEL
    if (yopt.m_DisplaysProfile)
    {
        for (size_t cmdIdx = 0; cmdIdx < cmds.size(); ++cmdIdx)
        {
            DisplayExportInfoSub(cmds[cmdIdx], yopt);
        }
    }
    exitCode = RExecProcessParallel(nullptr, pOutMsg, &errMsg, cmds, SW_HIDE);
    if (!outMsg.empty())
    {
        DisplayExportInfoSub(outMsg, yopt);
    }
    #else
    for (size_t cmdIdx = 0; cmdIdx < cmds.size(); ++cmdIdx)
    {
        const std::string& cmd = cmds[cmdIdx];
        if (yopt.m_DisplaysProfile)
        {
            DisplayExportInfoSub(cmd, yopt);
        }
        exitCode = RExecProcessWithPipe(nullptr, pOutMsg, &errMsg, cmd.c_str(), SW_HIDE);
        if (!outMsg.empty())
        {
            DisplayExportInfoSub(outMsg, yopt);
        }
        if (exitCode != 0)
        {
            break;
        }
    }
    #endif
    //cerr << "optimize time: " << tm1.GetMilliSec() << endl;

    if (!errMsg.empty())
    {
        DisplayExportInfoSub(errMsg, yopt);
        if (exitCode == 0)
        {
            YShowWarning(&yscene, // Optimization warning
                "中間ファイルオプティマイザーで警告が発生しました。（詳細はスクリプトエディタを見てください）",
                "A warning occurred in the intermediate file optimizer. (See the Script Editor for details.)");
        }
    }

    if (exitCode != 0)
    {
        YShowError(&yscene, // Optimization failed
            "中間ファイルオプティマイザーの処理が失敗しました。（詳細はスクリプトエディタを見てください）",
            "The intermediate file optimizer processing failed. (See the Script Editor for details.)");
        return MS::kFailure;
    }
    #endif

    //-----------------------------------------------------------------------------
    // profile
    yscene.m_OptimizeClocks += clock() - startClock;

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 中間ファイルフォーマッターを実行します。
//!        最適化オプションがすべて OFF かつコンフィグファイルが次のように
//!        なっている場合に実行されます。
//!        optimize_primitive="none"
//!        quantization_analysis="false"
//!        delete_near_vertex=""
//!
//! @param[in] yscene シーンです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus Do3dFormatter(YScene& yscene)
{
#ifdef USE_FORMAT_CONVERTER_SW
    //-----------------------------------------------------------------------------
    // 現在 fmd ファイルのみ処理します。
    // fmd ファイルを出力しない場合や
    // fmd ファイルを最適化する場合は何も実行しません。
    const YExpOpt& yopt = yscene.GetOpt();
    if (!yopt.Uses3dFormatter())
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルフォーマッターの exe ファイルが存在するかチェックします。
    const std::string g3dToolsPath = GetNintendoMaya3dToolsPath();
    std::string cvtrPath = g3dToolsPath + "3dIntermediateFileFormatter.exe";
    if (!RFileExists(cvtrPath))
    {
        cvtrPath = g3dToolsPath + "NW4F_g3dformat.exe";
    }
    if (!RFileExists(cvtrPath))
    {
        YShowError(&yscene, // 3D intermediate file formatter cannot be found: (at %s) // 通常は発生しない
            "中間ファイルフォーマッターが見つかりません: {0}",
            "The intermediate file formatter cannot be found: {0}",
            cvtrPath);
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルフォーマッターの引数を設定します。
    const std::string outFileNoExtPath = yopt.m_TmpOutFolderPath + yopt.m_OutFileName;

    std::string cmd = "\"" + RGetWindowsFilePath(cvtrPath) + "\"";
    if (!yopt.m_DisplaysProfile)
    {
        cmd += " -s"; // --silent
    }

    if (yopt.m_OutFileFlag[RExpOpt::FMD])
    {
        cmd += " \"" + outFileNoExtPath + "." + yopt.GetExtension(RExpOpt::FMD) + "\"";
    }

    //cerr << "if formatter: " << cmd.c_str() << endl;
    if (yopt.m_DisplaysProfile)
    {
        DisplayExportInfoSub(cmd, yopt);
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルフォーマッターを実行します。
    const clock_t startClock = clock();
    YSetHelpLine("Arranging intermediate file format ...");

    std::string outMsg;
    std::string errMsg;
    std::string* pOutMsg = (yopt.m_DisplaysProfile) ? &outMsg : nullptr;
    const int exitCode = RExecProcessWithPipe(nullptr, pOutMsg, &errMsg, cmd.c_str(), SW_HIDE);
    if (!outMsg.empty())
    {
        DisplayExportInfoSub(outMsg, yopt);
    }
    if (!errMsg.empty())
    {
        DisplayExportInfoSub(errMsg, yopt);
    }
    if (exitCode != 0)
    {
        YShowError(&yscene, // Cannot arrange intermediate file format
            "中間ファイルフォーマッターの処理が失敗しました。（詳細はスクリプトエディタを見てください）",
            "The intermediate file formatter processing failed. (See the Script Editor for details.)");
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // profile
    yscene.m_FormatClocks += clock() - startClock;
#endif
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 上書き確認ダイアログを表示します
//!
//! @param[in] overwriteFileNames すでに存在するファイル名の配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 上書きするなら true、キャンセルするなら false を返します。
//-----------------------------------------------------------------------------
static bool ConfirmOverwrite(RStringArray& overwriteFileNames, const YExpOpt& yopt)
{
    std::string cmd = "confirmDialog -t \"Export\" -ma \"left\" -m \"";
    std::string filePath = "\\\"" +
        yopt.m_OutFolderPath + yopt.m_OutFileName + ".*" + "\\\"";
    cmd += filePath;
    cmd += " already exists.\\n\\n";
    cmd += "Do you want to replace it ?\\n\\n";

    cmd += "------------------------------------------------\\n";
    for (size_t iFile = 0; iFile < overwriteFileNames.size(); ++iFile)
    {
        cmd += overwriteFileNames[iFile] + "\\n";
    }
    cmd += "------------------------------------------------\\n";

    cmd += "\" -b \"&Yes\" -b \"Cancel\" -db \"Cancel\" -ds \"Cancel\"";
    //cerr << "confirm: " << cmd << R_ENDL;
    MString ret;
    MGlobal::executeCommand(cmd.c_str(), ret);
    return (ret == "&Yes");
}

//-----------------------------------------------------------------------------
//! @brief 出力する中間ファイル群がすでに存在するか判定します。
//!
//! @param[out] overwriteFileNames すでに存在するファイル名の配列です。
//! @param[in] yscene シーンです。
//!
//! @return 1 つでもすでに存在するファイルがあれば true を返します。
//-----------------------------------------------------------------------------
static bool OutputFileExists(RStringArray& overwriteFileNames, const YScene& yscene)
{
    overwriteFileNames.clear();
    const YExpOpt& yopt = yscene.GetOpt();
    bool isSingleFmaChecked = false;
    for (int fileTypeIdx = 0; fileTypeIdx < RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const RExpOpt::FileType fileType = static_cast<RExpOpt::FileType>(fileTypeIdx);
        if (yopt.m_OutFileFlag[fileType])
        {
            if (yopt.UsesSingleFma() && yopt.IsFclFtsFtp(fileType))
            {
                if (isSingleFmaChecked)
                {
                    continue;
                }
                isSingleFmaChecked = true;
            }
            if (RFileExists(yopt.GetFilePath(fileType)))
            {
                overwriteFileNames.push_back(yopt.GetFileName(fileType));
            }
        }
    }
    return !overwriteFileNames.empty();
}

//-----------------------------------------------------------------------------
//! @brief 上書き確認します。
//!        出力先に同名ファイルが存在するなら確認ダイアログを表示します。
//!
//! @param[in] yscene シーンです。
//!
//! @return 上書きするなら true、キャンセルするなら false を返します。
//-----------------------------------------------------------------------------
static bool CheckOverwrite(const YScene& yscene)
{
    //-----------------------------------------------------------------------------
    // バッチモードでは上書き確認しません。
    const YExpOpt& yopt = yscene.GetOpt();
    if (MGlobal::mayaState() != MGlobal::kInteractive)
    {
        return true;
    }

    //-----------------------------------------------------------------------------
    // 出力先に同名ファイルが存在するなら確認ダイアログを表示します。
    RStringArray overwriteFileNames;
    if (OutputFileExists(overwriteFileNames, yscene))
    {
        return ConfirmOverwrite(overwriteFileNames, yopt);
    }
    else
    {
        return true;
    }
}

//-----------------------------------------------------------------------------
//! @brief 出力前のスクリプトを実行します。
//!
//! @param[in,out] yscene シーンです。
//! @param[in,out] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus DoPreExportScript(YScene& yscene, YExpOpt& yopt)
{
    static const char* const CANCEL_ENV = "NINTENDO_EXPORT_CANCEL_REQUEST";

    MStatus status;
    if (!yopt.m_PreExpScript.empty())
    {
        //-----------------------------------------------------------------------------
        // スクリプトに情報を伝達するための環境変数を設定します。
        SetEnvVarForPrePostExportScript(yscene, yopt, false, true);

        //-----------------------------------------------------------------------------
        // スクリプトを実行します。
        SetEnvVarByMel(CANCEL_ENV, "0", false);
        //cerr << "pre-export: [" << yopt.m_PreExpScript << "]" << endl;
        status = MGlobal::executeCommand(yopt.m_PreExpScript.c_str());
        if (!status)
        {
            YShowError(&yscene, // Pre-Export Script failed
                "出力前に実行するスクリプト（MEL）でエラーが発生しました。（詳細はスクリプトエディタを見てください）",
                "An error occurred with the pre-export script (MEL). (See the Script Editor for details.)");
            return status;
        }

        //-----------------------------------------------------------------------------
        // スクリプトから出力をキャンセルされたかチェックします。
        yopt.m_IsCancelledByScript = (GetEnvVarByMel(CANCEL_ENV) == "1");

        //-----------------------------------------------------------------------------
        // 出力ファイル（フォルダ）とマージファイル（フォルダ）のパスの
        // 環境変数やマクロ文字を展開してフルパスに変換します。
        // エクスポートオプションを上書きする場合は解析後に展開するので、ここでは展開しません。
        if (!yopt.m_IsCancelledByScript)
        {
            if (GetOverwriteOptionString().empty())
            {
                status = yopt.ExpandOutputAndMergePath(&yscene, true);
            }
        }
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief 出力後のスクリプトを実行します。
//!
//! @param[in] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//! @param[in] exportSucceeded 出力処理に成功していれば true を指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus DoPostExportScript(
    YScene& yscene,
    const YExpOpt& yopt,
    const bool exportSucceeded
)
{
    MStatus status;
    if (!yopt.m_PostExpScript.empty())
    {
        SetEnvVarForPrePostExportScript(yscene, yopt, true, exportSucceeded);
        //cerr << "post-export: [" << yopt.m_PostExpScript << "]" << endl;
        status = MGlobal::executeCommand(yopt.m_PostExpScript.c_str());
        if (!status)
        {
            YShowError(&yscene, // Post-Export Script failed
                "出力後に実行するスクリプト（MEL）でエラーが発生しました。（詳細はスクリプトエディタを見てください）",
                "An error occurred with the post-export script (MEL). (See the Script Editor for details.)");
        }
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief モデル 1 つのデータを取得して中間ファイル群を出力します。
//!
//! @param[out] ymodel モデルです。
//! @param[in] forceOverwrite true を指定すると、出力ファイルが存在していた場合に強制的に上書します。
//!                           false を指定すると、上書き確認ダイアログを表示します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ExportOneModel(YModel& ymodel, const bool forceOverwrite)
{
    MStatus status;
    YScene& yscene = ymodel.GetScene();
    YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // 出力前のスクリプトを実行します。
    status = DoPreExportScript(yscene, yopt);
    CheckStatus(status);
    if (yopt.m_IsCancelledByScript)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // エクスポートオプション上書き文字列を環境変数から取得して適用します。
    const std::string owOptStr = GetOverwriteOptionString();
    if (!owOptStr.empty())
    {
        status = yopt.Parse(&yscene, owOptStr.c_str(), true);
        CheckStatus(status);
        status = yopt.Check(&yscene);
        CheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // スクリプトの実行のみなら以降の処理をスキップします。
    if (yopt.m_IsScriptOnly)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // アプリケーションのプロジェクトのルートフォルダのパスをフルパスにします。
    if (!yopt.m_ProjectRootPath.empty())
    {
        yopt.m_ProjectRootPath = RGetFilePathWithEndingSlash(RGetFullFilePath(yopt.m_ProjectRootPath, true));
    }

    //-----------------------------------------------------------------------------
    // check overwrite
    // ファイルの上書き確認を行います。
    // 強制上書きフラグが設定されておらず、上書き確認で「いいえ」が選択されると
    // 出力をキャンセルします。
    if (!forceOverwrite && !CheckOverwrite(yscene))
    {
        yopt.m_CancelledFlag = true;
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザーの追加オプションを環境変数から取得します。
    GetOptimizerExtraOption(yopt);

    //-----------------------------------------------------------------------------
    // Windows カーソルを Wait 状態にします。
    BeginWaitCursor();

    //-----------------------------------------------------------------------------
    // 出力フォルダが存在しなければ作成します。
    status = CreateFolderIfNotExist(yscene, yopt.m_OutFolderPath);
    CheckStatus(status);

    status = CreateFolderIfNotExist(yscene, yopt.m_TmpOutFolderPath);
    CheckStatus(status);
    //cerr << "tmp out: " << m_TmpOutFolderPath << R_ENDL;

    //-----------------------------------------------------------------------------
    // エクスポートオプションのマージファイルが有効か判定します。
    status = yopt.CheckMergeFile(&yscene);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // モデルについての情報を表示します。
    DisplayExportInfoModelHeader(yopt);

    //-----------------------------------------------------------------------------
    // Maya のデータをモデルに設定します。
    // ここで中間ファイルに出力するべき Maya のオブジェクトを列挙し、
    // 中間形式のデータに変換します。
    if (status)
    {
        YSetHelpLine("Getting Data ...");
        status = GetData(ymodel);
    }

    //-----------------------------------------------------------------------------
    // GetData で列挙されたデータをタイプ別のデータを中間ファイルに出力します。
    int fileCount = 0;
    if (status)
    {
        status = OutputFiles(fileCount, ymodel);
    }

    //-----------------------------------------------------------------------------
    // シェーダパラメータアニメーションをアサインします。
    //if (status)
    //{
    //  status = AssignShaderParamAnim();
    //}

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザーを呼び出します。
    if (status)
    {
        status = Do3dOptimizer(yscene);
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルフォーマッターを実行します。
    if (status)
    {
        status = Do3dFormatter(yscene);
    }

    //-----------------------------------------------------------------------------
    // テクスチャイメージの出力用ポインタ配列を設定し、ソートします。
    SetAndSortOutAllTexImgs(yscene);

    //-----------------------------------------------------------------------------
    // モデルの出力情報の結果部分を表示します。
    if (status && fileCount > 0)
    {
        DisplayExportInfoModelResult(yscene, yopt);
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief テンポラリフォルダに出力した中間ファイル群を出力用フォルダに移動します。
//!
//! @param[in] yscene シーンです。
//! @param[in] status Export 処理の処理結果です。
//!                   失敗していた場合はテンポラリフォルダの中間ファイル群を削除します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus MoveTmpFiles(YScene& yscene, MStatus status)
{
    //-----------------------------------------------------------------------------
    // テンポラリフォルダを指定している場合、中間ファイル群の移動はしません。
    // 失敗していた場合はテンポラリフォルダの中間ファイル群を削除します。
    const YExpOpt& yopt = yscene.GetOpt();
    if (!yopt.m_SpecialTmpFolderPath.empty() && status)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // create tex folder
    if (status)
    {
        // テンポラリフォルダに textures フォルダが存在していれば出力先フォルダにも
        // textures フォルダを作成します。
        // これはエクスポートオプションでテクスチャの出力がチェックされていなくても、
        // マージ処理によりテンポラリフォルダにテクスチャが作成されてしまう場合の対処です。
        const bool exportTexture = RFolderExists(yopt.m_TmpTexFolderPath);
        if (yopt.ExportsTexture() || exportTexture)
        {
            if (yscene.m_AllTexImgs.size() > 0 || exportTexture)
            {
                status = CreateFolderIfNotExist(yscene, yopt.m_TexFolderPath);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // move or delete files
    const int moveCount = static_cast<int>(yscene.m_TmpFileMoves.size());
    for (int iMove = 0; iMove < moveCount; ++iMove)
    {
        const RFileMove& move = yscene.m_TmpFileMoves[iMove];
        if (status)
        {
            //cerr << "move: " << move.m_Src << " -> " << move.m_Dst << R_ENDL;
            if (yopt.m_UsesTmpFolderForScript && !RFileExists(move.m_Src))
            {
                DisplayExportInfoSub("Temp file cannot be found: " + move.m_Src, yopt);
            }
            else if (!move.Move())
            {
                YShowError(&yscene, // Cannot open file: %s (cannot move)
                    "ファイルを開けません（移動できません）。上書き禁止になっていないか確認してください: {0}",
                    "The file cannot be opened (moved). Confirm whether the file can be overwritten: {0}",
                    move.m_Dst);
                ::DeleteFileA(move.m_Src.c_str());
                status = MS::kFailure;
            }
        }
        else
        {
            if (yopt.m_DeletesTmpFileOnError)
            {
                //cerr << "del: " << move.m_Src << R_ENDL;
                ::DeleteFileA(move.m_Src.c_str());
            }
        }
    }

    //-----------------------------------------------------------------------------
    // delete tmp folder
    if (status || yopt.m_DeletesTmpFileOnError)
    {
        RemoveFolderIfExist(yopt.m_TmpTexFolderPath);
        RemoveFolderIfExist(yopt.m_TmpOutFolderPath);
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief Maya からデータを出力して中間ファイル群を出力する処理のメイン関数です。
//!
//! @param[in] yscene シーンです。
//! @param[in] forceOverwrite true を指定すると、出力ファイルが存在していた場合に強制的に上書します。
//!                           false を指定すると、上書き確認ダイアログを表示します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus Process(YScene& yscene, const bool forceOverwrite) // ProcessT
{
    MStatus status;
    const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // show feedback line
    MFeedbackLine::setShowFeedback(true);

    //-----------------------------------------------------------------------------
    // 出力処理を制御するための環境変数とオプション変数をクリアします。
    ClearOptimizerExtraOptionEnvVar(yopt);
    ClearOverwriteOptionStringVariable(yopt);
    ClearBakingLoopScriptVariable(yopt);

    //-----------------------------------------------------------------------------
    // export
    const time_t startTime = time(NULL);
    YModel* pYModel = new YModel(&yscene);
    status = ExportOneModel(*pYModel, forceOverwrite);
    delete pYModel;

    //-----------------------------------------------------------------------------
    // スクリプトから出力をキャンセルされたなら以降の処理をスキップします。
    if (yopt.m_IsCancelledByScript)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // profile
    if (status && !yopt.m_CancelledFlag && !yopt.m_IsScriptOnly)
    {
        DisplayProfileInfo("Get Data        ", yscene.m_GetClocks      , yopt);
        if (yopt.ExportsModelFile()) DisplayProfileInfo("Export Model    ", yscene.m_ModelClocks    , yopt);
        if (yopt.ExportsTexture()  ) DisplayProfileInfo("Export Texture  ", yscene.m_TexClocks      , yopt);
        if (yopt.ExportsModelAnim()) DisplayProfileInfo("Export Animation", yscene.m_AnimClocks     , yopt);
        if (yopt.Uses3dOptimizer() ) DisplayProfileInfo("Optimize        ", yscene.m_OptimizeClocks , yopt);
        if (yopt.Uses3dFormatter() ) DisplayProfileInfo("Convert Format  ", yscene.m_FormatClocks   , yopt);
    }

    //-----------------------------------------------------------------------------
    // テンポラリフォルダに出力した状態で出力後のスクリプトを実行します。
    // 出力処理に失敗した場合も実行します。
    if (yopt.m_UsesTmpFolderForScript && !yopt.m_CancelledFlag)
    {
        const MStatus scriptStatus = DoPostExportScript(yscene, yopt, static_cast<bool>(status));
        if (status)
        {
            status = scriptStatus;
        }
    }

    //-----------------------------------------------------------------------------
    // テンポラリフォルダに出力されたファイルを本来の出力フォルダに移動させます。
    // エラー発生時もテンポラリファイルを消すために実行します。
    if (yopt.m_UseTmpFolderFlag)
    {
        status = MoveTmpFiles(yscene, status);
    }

    //-----------------------------------------------------------------------------
    // 本来の出力フォルダに出力した状態で出力後のスクリプトを実行します。
    // 出力処理に失敗した場合も実行します。
    if (!yopt.m_UsesTmpFolderForScript && !yopt.m_CancelledFlag)
    {
        const MStatus scriptStatus = DoPostExportScript(yscene, yopt, static_cast<bool>(status));
        if (status)
        {
            status = scriptStatus;
        }
    }

    //-----------------------------------------------------------------------------
    // スクリプトの実行のみなら以降の処理をスキップします。
    if (yopt.m_IsScriptOnly)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // キャンセルされていなければ Window カーソルを Wait 状態から通常状態に戻します。
    if (!yopt.m_CancelledFlag)
    {
        EndWaitCursor();
    }

    //-----------------------------------------------------------------------------
    // エクスポートの終了処理

    // 処理時間を計測します。
    const time_t processTime = time(NULL) - startTime;
    // エクスポート結果をユーザーに通知します。
    const std::string timeStr = "Time = " +
        RGetNumberString(static_cast<int>(processTime) / 60) + ":" +
        RGetNumberString(static_cast<int>(processTime) % 60, "%02d");
    const std::string warnStr = "Warning = " + RGetNumberString(yscene.GetWarningCount());

    // normal export message
    std::string msg;
    if (status)
    {
        if (yopt.m_CancelledFlag)
        {
            msg = "Cancelled";
        }
        else
        {
            msg = "Finished ( " + timeStr + " ,  " + warnStr + " )";
        }
    }
    else
    {
        msg = "Stopped ( " + warnStr + " )";
    }

    YSetHelpLine(msg.c_str());
    if (status)
    {
        MGlobal::displayInfo(msg.c_str());
    }

    // LOD エクスポートでは処理を継続するので、ヘルプラインの Finished を消去します。
    if (yopt.m_ExportsLod && status)
    {
        YSetHelpLine("");
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief Maya のアニメーションの開始／終了フレームを
//!        エクスポートオプションの出力フレーム範囲として取得します。
//-----------------------------------------------------------------------------
void YExpOpt::GetMayaAnimationRange()
{
    MGlobal::executeCommand("playbackOptions -q -ast", m_DoubleStartFrame);
    MGlobal::executeCommand("playbackOptions -q -aet", m_DoubleEndFrame);
    m_StartFrame = static_cast<int>(m_DoubleStartFrame);
    m_EndFrame   = static_cast<int>(m_DoubleEndFrame);
}

//-----------------------------------------------------------------------------
//! @brief Maya のアニメーションの再生開始／再生終了フレームを
//!        エクスポートオプションの出力フレーム範囲として取得します。
//-----------------------------------------------------------------------------
void YExpOpt::GetMayaPlaybackRange()
{
    MGlobal::executeCommand("playbackOptions -q -min", m_DoubleStartFrame);
    MGlobal::executeCommand("playbackOptions -q -max", m_DoubleEndFrame);
    m_StartFrame = static_cast<int>(m_DoubleStartFrame);
    m_EndFrame   = static_cast<int>(m_DoubleEndFrame);
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションに初期値を設定します。
//-----------------------------------------------------------------------------
void YExpOpt::Init() // InitOptions
{
    //-----------------------------------------------------------------------------
    // project & scene name
    MString projectPathMS;
    MGlobal::executeCommand("workspace -q -rd", projectPathMS);
    m_ProjectPath = projectPathMS.asChar();

    MStringArray folders;
    projectPathMS.split('/', folders);
    //cerr << "folders: " << folders << R_ENDL;
    if (folders.length() >= 2)
    {
        m_ProjectName = folders[folders.length() - 1].asChar();
    }

    MString sceneNameMS;
    MGlobal::executeCommand("file -q -ns", sceneNameMS);
    m_SceneName = sceneNameMS.asChar();

    //-----------------------------------------------------------------------------
    // preset
    m_PresetName = "";
    m_PresetFilePath = "";

    //-----------------------------------------------------------------------------
    // output
    m_AnimRangeFlag = false;
    m_Target = EXPORT_TARGET_ALL;
    m_ExportsLod = false;
    m_OrgOutFileName = m_OutFileName = m_SceneName;
    m_OrgOutFolderPath = m_OutFolderPath = m_ProjectPath;
    m_SpecialTmpFolderPath = "";
    m_ModelName = m_OutFileName;

    // internal
    m_IsMultiExport = false;
    m_IsCancelledByScript = false;
    m_IsScriptOnly = false;
    m_TexFolderPath = m_OutFolderPath + RExpOpt::TexFolderName + "/";

    //-----------------------------------------------------------------------------
    // merge
    m_MergeFmdFlag = false;
    m_OrgMergeFmdPath = m_MergeFmdPath = "";
    m_MergesOutputFmd = false;
    m_MergeFtxFlag = false;
    m_MergeAnimFlag = false;
    m_OrgMergeAnimFolder = m_MergeAnimFolder = "";
    m_OrgMergeAnimName = m_MergeAnimName = "";
    m_MergesOutputAnim = false;

    // internal
    m_MergeFtxFolderPath = m_TexFolderPath;

    //-----------------------------------------------------------------------------
    // general
    m_Magnify = 1.0;
    m_InternalMagnify = MDistance::internalToUI(1.0) * m_Magnify;
    m_TexSrtMode = TEX_SRT_MODE_MAYA;
    m_RemoveNamespace = false;
    m_CommentText = "";

    //-----------------------------------------------------------------------------
    // animation
    m_FrameRange = FRAME_RANGE_ALL;
    GetMayaAnimationRange();
    m_SpecifiedStartFrame = 1;
    #ifdef NO_CLAMP_FRAME_RANGE_SW
    m_SpecifiedEndFrame = 100;
    #else
    m_SpecifiedEndFrame = 9999;
    #endif

    m_LoopAnim = false;
    m_BakeAllAnim = true;
    m_FramePrecision = 1;

    UpdateFrameCount();

    m_UIUnitTime = MTime::uiUnit();

    MTime oneSec(1.0, MTime::kSeconds);
    m_FramesPerSecond = static_cast<float>(oneSec.as(m_UIUnitTime));

    //-----------------------------------------------------------------------------
    // output file selection
    m_IsBinaryFormat = true;
    //m_IsBinaryFormat = false;
    ClearOutFileFlag();
    m_OutFileFlag[FMD] = true;
    m_OutFtxFlag = true;

    // internal
    ClearNoAnimWarnedFlag();

    //-----------------------------------------------------------------------------
    // optimization
    InitOptimization();

    // internal
    m_FmdIsBindPose = true;
    m_ForceFullWeight = false;
    m_OutAllTex = false;

    //-----------------------------------------------------------------------------
    // model

    //-----------------------------------------------------------------------------
    // bake tolerance
    m_TolS = 0.01f;
    m_TolR = 0.1f;
    m_TolT = 0.01f;

    m_TolC = 0.001f;

    m_TolTexS = 0.01f;
    m_TolTexR = 0.1f;
    m_TolTexT = 0.01f;

    //-----------------------------------------------------------------------------
    // quantization tolerance
    m_QuantTolS = 0.01f;
    m_QuantTolR = 0.2f;
    m_QuantTolT = 0.001f;

    m_QuantTolTexS = 0.01f;
    m_QuantTolTexR = 0.2f;
    m_QuantTolTexT = 0.01f;

    //-----------------------------------------------------------------------------
    // script
    m_PreExpScript = "";
    m_PostExpScript = "";

    //-----------------------------------------------------------------------------
    // コンフィグファイルで指定するオプション
    m_DisplaysProfile = false;
    m_ProjectRootPath = "";
    m_MaxVertexSkinningCount = VERTEX_SKINNING_COUNT_MAX;
    m_AdjustsSmoothSkinning = true; // changed (SIGLO-74055)
    m_CompressesIgnoringVertexSkinningCount = false;
    m_DisablesOriginalImage = false;
    m_UsesSrgbFetch = true; // changed (2012/11/15)
    m_NormalTextureFormat = RImage::Format_Snorm_Bc5;
    m_NormalTextureCompSel = "";
    m_EnablesWeightedCompress = false;
    m_GpuEncoding = "";
    m_IsFilterMipLinear = true;
    m_SamplerMergePriority = false;
    m_UsesFclFtsFtp = true;
    m_SeparatesFma = false;
    m_BoneVisMergePriority = false;
    m_ExportsBoneVisAll = true;
    m_ChecksPresetNoneError = false;
    m_ChecksWrongFileNameError = true;
    m_ChecksNoTextureFileErrorAlways = false;
    m_WarnsNodeNameChanged = true;
    m_UsesTmpFolderForScript = false;
    m_DeletesTmpFileOnError = true;
    m_UserCustomUiScript = "";
    m_MultiExportUserCustomUiScript = "";

    //-----------------------------------------------------------------------------
    // use data
    m_FileUserDatas.clear();

    //-----------------------------------------------------------------------------
    // 外部ツールのパスを取得します。
    m_G3dToolPath = GetNintendoMaya3dToolsPath();

    //-----------------------------------------------------------------------------
    // tmp folder
    m_UseTmpFolderFlag = false;
    m_TmpOutFolderPath = m_OutFolderPath;
    m_TmpTexFolderPath = m_TexFolderPath;

    //-----------------------------------------------------------------------------
    // internal flag
    m_CancelledFlag = false;
    m_CheckElementFlag = false;
    m_Arranges3dIfFormat = true;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief Export コマンドのオプション文字列からプリセット名を取得します。
//-----------------------------------------------------------------------------
static std::string GetPresetNameFromOptionString(const MString& optionString)
{
    static const std::string PRESET_TOP = "preset=";
    MStringArray optLines;
    optionString.split(';', optLines);
    for (int iOpt = static_cast<int>(optLines.length()) - 1; iOpt >= 0; --iOpt)
    {
        const std::string optLine = optLines[iOpt].asChar();
        if (optLine.find(PRESET_TOP) == 0)
        {
            return (optLine.size() > PRESET_TOP.size()) ?
                optLine.substr(PRESET_TOP.size()) : std::string();
        }
    }
    return std::string();
}

//-----------------------------------------------------------------------------
//! @brief Export コマンドのオプション文字列からプリセット無視フラグを取得します。
//!        プリセットを無視する場合でも中間ファイルの dcc_preset にはプリセット名が出力されます。
//-----------------------------------------------------------------------------
static bool IgnoresPresetFromOptionString(const MString& optionString)
{
    static const std::string IGNORE_TOP = "ignore_preset=";
    MStringArray optLines;
    optionString.split(';', optLines);
    for (int iOpt = static_cast<int>(optLines.length()) - 1; iOpt >= 0; --iOpt)
    {
        const std::string optLine = optLines[iOpt].asChar();
        if (optLine.find(IGNORE_TOP) == 0)
        {
            const std::string optValue = (optLine.size() > IGNORE_TOP.size()) ?
                optLine.substr(IGNORE_TOP.size()) : "false";
            return (optValue == "true" || optValue == "on" || optValue == "1");
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief Export コマンドのオプション文字列を解析して、エクスポートオプションを設定します。
//-----------------------------------------------------------------------------
MStatus YExpOpt::Parse( // ParseOptions
    RScene* pScene,
    const MString& optionString,
    const bool overwritesOption
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // コンフィグファイルを解析します。
    if (!overwritesOption)
    {
        const std::string configPath       = GetNintendoMayaConfigFolderPath(true)  + ExportConfigFileName;
        const std::string configSamplePath = GetNintendoMayaConfigFolderPath(false) + "Export_Sample.ini";

        // コンフィグファイルがなければサンプルのコンフィグファイルをコピーします。
        if (!RFileExists(configPath) && RFileExists(configSamplePath))
        {
            ::CopyFileA(configSamplePath.c_str(), configPath.c_str(), TRUE);
        }

        if (RFileExists(configPath))
        {
            RStatus rstatus = ParseConfigFile(configPath);
            status = GetMStatusDisplayError(pScene, rstatus);
            CheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // プリセットを解析します。
    MString optStr = optionString;
    if (!overwritesOption)
    {
        m_PresetName = GetPresetNameFromOptionString(optStr);
        //cerr << "preset: [" << m_PresetName << "]" << endl;
        if (!m_PresetName.empty())
        {
            m_PresetFilePath = GetNintendoMayaPresetsFolderPath() + m_PresetName + ".fdes";
            if (!IgnoresPresetFromOptionString(optStr))
            {
                //-----------------------------------------------------------------------------
                // check if preset file exists
                if (!RFileExists(m_PresetFilePath))
                {
                    YShowError(pScene, // Cannot open file: %s
                        "プリセットファイルを開けません: {0}",
                        "The preset file cannot be opened: {0}",
                        m_PresetFilePath);
                    return MS::kFailure;
                }

                //-----------------------------------------------------------------------------
                // get option string from preset file
                const std::string cmd = "nnExpDialog_GetOptionStringFromFdes \"" +
                    m_PresetFilePath + "\" 1";
                MString presetOptStr;
                if (!MGlobal::executeCommand(cmd.c_str(), presetOptStr))
                {
                    YShowError(pScene, // Preset file is wrong: %s
                        "プリセットファイルが不正です: {0}",
                        "Preset file is wrong: {0}",
                        m_PresetFilePath);
                    return MS::kFailure;
                }
                //cerr << "preset opt: " << presetOptStr << endl;
                optStr += ";" + presetOptStr;
            }
            else
            {
                //cerr << "ignore preset: " << m_PresetName << endl;
            }
        }
        else if (m_ChecksPresetNoneError)
        {
            YShowError(pScene,
                "プリセット None では出力できません。コンフィグファイルで禁止されています（error_preset_none）。",
                "You cannot export with the preset 'None'. It is prohibited by the config file (error_preset_none).");
            return MS::kFailure;
        }
    }

    //-----------------------------------------------------------------------------
    // get options
    if (optStr.length() > 0)
    {
        const bool overwritesLodLevel     = (overwritesOption && ExportsLodLevel());
        const bool overwritesLodBaseLevel = (overwritesOption && ExportsLodBaseLevel());
        MStringArray optLines;
        optStr.split(';', optLines);
        for (int iOpt = 0; iOpt < static_cast<int>(optLines.length()); ++iOpt)
        {
            const MString optLine = optLines[iOpt];
            const int iEqual = optLine.index('=');
            if (iEqual == -1)
            {
                continue;
            }
            const MString optName = optLine.substring(0, iEqual - 1);
            const MString optValue = (iEqual + 1 < static_cast<int>(optLine.length())) ?
                optLine.substring(iEqual + 1, optLine.length() - 1) : "";
            //cerr << "opt: [" << optLine << "] [" << optName << "] [" << optValue << "]" << endl;
            bool boolValue = (optValue == "true" || optValue == "on" || optValue == "1");

            //-----------------------------------------------------------------------------
            // 文字列型のオプション以外で値が空文字ならスキップします。
            if (optValue.length() == 0)
            {
                if (optName != "output_file_name"   &&
                    optName != "output_folder"      &&
                    optName != "special_tmp_folder" &&
                    optName != "merge_fmd_path"     &&
                    optName != "merge_anim_folder"  &&
                    optName != "merge_anim_name"    &&
                    optName != "comment"            &&
                    optName != "pre_export_script"  &&
                    optName != "post_export_script" &&
                    optName != "delete_near_vertex")
                {
                    continue;
                }
            }

            //-----------------------------------------------------------------------------
            // output
            if (optName == "process_mode")
            {
                //m_AnimRangeFlag = (optValue == "animation_range");
            }
            else if (optName == "export_target")
            {
                if (!overwritesLodLevel)
                {
                    m_Target = (optValue == "selection") ?
                        EXPORT_TARGET_SELECTION : EXPORT_TARGET_ALL;
                }
            }
            else if (optName == "lod_export")
            {
                if (!overwritesOption)
                {
                    m_ExportsLod = boolValue;
                }
            }
            else if (optName == "output_file_name")
            {
                if (!overwritesLodLevel)
                {
                    m_OrgOutFileName = m_OutFileName = RTrimString(optValue.asChar());
                }
            }
            else if (optName == "output_folder")
            {
                if (!overwritesLodLevel)
                {
                    m_OrgOutFolderPath = m_OutFolderPath = RTrimString(optValue.asChar());
                }
            }
            else if (optName == "special_tmp_folder")
            {
                if (!overwritesLodLevel)
                {
                    m_SpecialTmpFolderPath = RTrimString(optValue.asChar());
                }
            }

            //-----------------------------------------------------------------------------
            // merge
            else if (optName == "merge_fmd")
            {
                if (!overwritesLodLevel || overwritesLodBaseLevel)
                {
                    m_MergeFmdFlag = boolValue;
                }
            }
            else if (optName == "merge_fmd_path")
            {
                m_OrgMergeFmdPath = m_MergeFmdPath = RTrimString(optValue.asChar());
            }
            else if (optName == "merge_ftx")
            {
                m_MergeFtxFlag = boolValue;
            }
            else if (optName == "merge_anim")
            {
                m_MergeAnimFlag = boolValue;
            }
            else if (optName == "merge_anim_folder")
            {
                m_OrgMergeAnimFolder = m_MergeAnimFolder = RTrimString(optValue.asChar());
            }
            else if (optName == "merge_anim_name")
            {
                m_OrgMergeAnimName = m_MergeAnimName = RTrimString(optValue.asChar());
            }

            //-----------------------------------------------------------------------------
            // general
            else if (optName == "magnify")
            {
                m_Magnify = optValue.asDouble();
            }
            else if (optName == "texsrt_mode")
            {
                if (optValue == "maya")
                {
                    m_TexSrtMode = TEX_SRT_MODE_MAYA;
                }
                else if (optValue == "3dsmax")
                {
                    m_TexSrtMode = TEX_SRT_MODE_3DSMAX;
                }
                else if (optValue == "softimage")
                {
                    m_TexSrtMode = TEX_SRT_MODE_SOFTIMAGE;
                }
            }
            else if (optName == "remove_namespace")
            {
                m_RemoveNamespace = boolValue;
            }
            else if (optName == "comment")
            {
                m_CommentText = optValue.asChar();
            }

            //-----------------------------------------------------------------------------
            // animation
            else if (optName == "frame_range")
            {
                if (optValue == "all")
                {
                    m_FrameRange = FRAME_RANGE_ALL;
                }
                else if (optValue == "playback")
                {
                    m_FrameRange = FRAME_RANGE_PLAYBACK;
                }
                else if (optValue == "range")
                {
                    m_FrameRange = FRAME_RANGE_SPECIFY;
                }
            }
            else if (optName == "start_frame")
            {
                m_SpecifiedStartFrame = optValue.asInt();
            }
            else if (optName == "end_frame")
            {
                m_SpecifiedEndFrame = optValue.asInt();
            }
            else if (optName == "loop_anim")
            {
                m_LoopAnim = boolValue;
            }
            else if (optName == "bake_all_anim")
            {
                m_BakeAllAnim = boolValue;
            }
            else if (optName == "frame_precision")
            {
                m_FramePrecision = optValue.asInt();
            }

            //-----------------------------------------------------------------------------
            // output file selection
            else if (optName == "output_fmd")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FMD] = boolValue;
                }
            }
            else if (optName == "output_ftx")
            {
                if (!overwritesLodLevel || overwritesLodBaseLevel)
                {
                    m_OutFtxFlag = boolValue;
                }
            }
            else if (optName == "output_fsk")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FSK] = boolValue;
                }
            }
            else if (optName == "output_fvb")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FVB] = boolValue;
                }
            }
            else if (optName == "output_fcl")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FCL] = boolValue;
                }
            }
            else if (optName == "output_fts")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FTS] = boolValue;
                }
            }
            else if (optName == "output_ftp")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FTP] = boolValue;
                }
            }
            else if (optName == "output_fsh")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FSH] = boolValue;
                }
            }
            else if (optName == "output_fsn")
            {
                if (!overwritesLodLevel)
                {
                    m_OutFileFlag[FSN] = boolValue;
                }
            }

            //-----------------------------------------------------------------------------
            // optimization
            else if (optName == "compress_bone")
            {
                if (optValue == "none")
                {
                    m_CompressBone = COMPRESS_BONE_NONE;
                }
                else if (optValue == "cull")
                {
                    m_CompressBone = COMPRESS_BONE_CULL;
                }
                else if (optValue == "merge")
                {
                    m_CompressBone = COMPRESS_BONE_MERGE;
                }
                else if (optValue == "unite")
                {
                    m_CompressBone = COMPRESS_BONE_UNITE;
                }
                else if (optValue == "unite_all")
                {
                    m_CompressBone = COMPRESS_BONE_UNITE_ALL;
                }
            }
            else if (optName == "unite_child")
            {
                m_UniteChild = boolValue;
            }
            else if (optName == "compress_material")
            {
                m_CompressMaterial = boolValue;
            }
            else if (optName == "compress_shape")
            {
                m_CompressShape = boolValue;
            }
            else if (optName == "divide_submesh")
            {
                m_DivideSubmesh = boolValue;
            }
            else if (optName == "divide_submesh_mode")
            {
                m_DivideSubmeshMode = RConvertToSingleQuote(RTrimString(optValue.asChar()));
            }
            else if (optName == "poly_reduction")
            {
                if (!overwritesLodLevel || overwritesLodBaseLevel)
                {
                    m_PolyReduction = boolValue;
                }
            }
            else if (optName == "poly_reduction_mode")
            {
                const std::string polyReductionMode = RDecodeScriptString(RTrimString(optValue.asChar()));
                m_PolyReductionMode = RConvertToSingleQuote(polyReductionMode);
            }

            //-----------------------------------------------------------------------------
            // bake tolerance
            else if (optName == "tolerance_scale")
            {
                m_TolS = static_cast<float>(optValue.asDouble());
                if (m_TolS < 0.0f)
                {
                    m_TolS = 0.0f;
                }
            }
            else if (optName == "tolerance_rotate")
            {
                m_TolR = static_cast<float>(optValue.asDouble());
                if (m_TolR < 0.0f)
                {
                    m_TolR = 0.0f;
                }
            }
            else if (optName == "tolerance_translate")
            {
                m_TolT = static_cast<float>(optValue.asDouble());
                if (m_TolT < 0.0f)
                {
                    m_TolT = 0.0f;
                }
            }
            else if (optName == "tolerance_color")
            {
                m_TolC = static_cast<float>(optValue.asDouble());
                if (m_TolC < 0.0f)
                {
                    m_TolC = 0.0f;
                }
            }
            else if (optName == "tolerance_tex_scale")
            {
                m_TolTexS = static_cast<float>(optValue.asDouble());
                if (m_TolTexS < 0.0f)
                {
                    m_TolTexS = 0.0f;
                }
            }
            else if (optName == "tolerance_tex_rotate")
            {
                m_TolTexR = static_cast<float>(optValue.asDouble());
                if (m_TolTexR < 0.0f)
                {
                    m_TolTexR = 0.0f;
                }
            }
            else if (optName == "tolerance_tex_translate")
            {
                m_TolTexT = static_cast<float>(optValue.asDouble());
                if (m_TolTexT < 0.0f)
                {
                    m_TolTexT = 0.0f;
                }
            }

            //-----------------------------------------------------------------------------
            // quantization tolerance
            else if (optName == "quantize_tolerance_scale")
            {
                m_QuantTolS = static_cast<float>(optValue.asDouble());
                if (m_QuantTolS < 0.0f)
                {
                    m_QuantTolS = 0.0f;
                }
            }
            else if (optName == "quantize_tolerance_rotate")
            {
                m_QuantTolR = static_cast<float>(optValue.asDouble());
                if (m_QuantTolR < 0.0f)
                {
                    m_QuantTolR = 0.0f;
                }
            }
            else if (optName == "quantize_tolerance_translate")
            {
                m_QuantTolT = static_cast<float>(optValue.asDouble());
                if (m_QuantTolT < 0.0f)
                {
                    m_QuantTolT = 0.0f;
                }
            }
            else if (optName == "quantize_tolerance_tex_scale")
            {
                m_QuantTolTexS = static_cast<float>(optValue.asDouble());
                if (m_QuantTolTexS < 0.0f)
                {
                    m_QuantTolTexS = 0.0f;
                }
            }
            else if (optName == "quantize_tolerance_tex_rotate")
            {
                m_QuantTolTexR = static_cast<float>(optValue.asDouble());
                if (m_QuantTolTexR < 0.0f)
                {
                    m_QuantTolTexR = 0.0f;
                }
            }
            else if (optName == "quantize_tolerance_tex_translate")
            {
                m_QuantTolTexT = static_cast<float>(optValue.asDouble());
                if (m_QuantTolTexT < 0.0f)
                {
                    m_QuantTolTexT = 0.0f;
                }
            }

            //-----------------------------------------------------------------------------
            // script
            else if (optName == "pre_export_script")
            {
                if (!overwritesOption)
                {
                    m_PreExpScript = RDecodeScriptString(RTrimString(optValue.asChar()));
                }
            }
            else if (optName == "post_export_script")
            {
                if (!overwritesOption)
                {
                    m_PostExpScript = RDecodeScriptString(RTrimString(optValue.asChar()));
                }
            }

            //-----------------------------------------------------------------------------
            // user data
            //else if (optName == "user_string")
            //{
            //  GetOptionUserData(pScene, m_FileUserDatas, optValue.asChar(), RUserData::TYPE_STRING);
            //}
            //else if (optName == "user_int")
            //{
            //  GetOptionUserData(pScene, m_FileUserDatas, optValue.asChar(), RUserData::TYPE_INT);
            //}
            //else if (optName == "user_float")
            //{
            //  GetOptionUserData(pScene, m_FileUserDatas, optValue.asChar(), RUserData::TYPE_FLOAT);
            //}

            //-----------------------------------------------------------------------------
            // internal
            else if (optName == "multi_export")
            {
                if (!overwritesOption)
                {
                    m_IsMultiExport = boolValue;
                }
            }
            else if (optName == "script_only")
            {
                if (!overwritesOption)
                {
                    m_IsScriptOnly = boolValue;
                }
            }

            //-----------------------------------------------------------------------------
            // 上書きのみ対応
            else if (optName == "optimize_primitive")
            {
                if (overwritesOption)
                {
                    m_OptimizePrimitive =
                        (optValue == "none"       ) ? OPTIMIZE_PRIMITIVE_NONE        :
                        (optValue == "default"    ) ? OPTIMIZE_PRIMITIVE_DEFAULT     :
                        (optValue == "force"      ) ? OPTIMIZE_PRIMITIVE_FORCE       :
                        (optValue == "brute_force") ? OPTIMIZE_PRIMITIVE_BRUTE_FORCE :
                        (optValue == "forsyth"    ) ? OPTIMIZE_PRIMITIVE_FORSYTH     :
                        OPTIMIZE_PRIMITIVE_FORSYTH;
                }
            }
            else if (optName == "quantization_analysis")
            {
                if (overwritesOption)
                {
                    m_QuantizationAnalysis = boolValue;
                }
            }
            else if (optName == "delete_near_vertex")
            {
                if (overwritesOption)
                {
                    m_DeleteNearVertex = RConvertToSingleQuote(RTrimString(optValue.asChar()));
                }
            }
            else if (optName == "arrange_3dif_format")
            {
                if (overwritesOption)
                {
                    m_Arranges3dIfFormat = boolValue;
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 環境変数によるオプション変更です（デバッグ用）。
    if (RIsEnvVarNotZero("NINTENDO_MAYA_NO_OPTIMIZE_OP"))
    {
        m_OptimizePrimitive = OPTIMIZE_PRIMITIVE_NONE;
    }
    if (RIsEnvVarNotZero("NINTENDO_MAYA_NO_OPTIMIZE_QA"))
    {
        m_QuantizationAnalysis = false;
    }
    if (RIsEnvVarNotZero("NINTENDO_MAYA_NO_CONVERT_FORMAT"))
    {
        m_Arranges3dIfFormat = false;
    }

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 中間ファイルを一時的に出力するテンポラリフォルダのパスを取得します。
//!
//! @return テンポラリフォルダのパスを返します。
//!         %TEMP%/Nintendo_MayaPlugins_<PID> の形式です。
//-----------------------------------------------------------------------------
static std::string GetTempFolderPath()
{
    std::string tempFolderPath = RGetTempFolderPath();
    if (!tempFolderPath.empty())
    {
        tempFolderPath += "Nintendo_MayaPlugins_" +
            RGetCurrentProcessIdString() + "/";
    }
    return tempFolderPath;
}

//-----------------------------------------------------------------------------
//! @brief 出力する中間ファイル名を Maya 上で選択したノードから取得します。
//!
//! @param[out] name 中間ファイル名を格納します。
//! @param[in,out] pScene シーンデータへのポインターです。
//! @param[in] yopt エクスポートオプションです。
//! @param[in] checksSelectionError ノードが選択されていない場合にエラーにするなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetOutFileNameFromSelection(
    std::string& name,
    RScene* pScene,
    const YExpOpt& yopt,
    const bool checksSelectionError
)
{
    name = "";
    MDagPath nodePath;

    MSelectionList slist;
    MGlobal::getActiveSelectionList(slist);
    MItSelectionList sIter(slist, MFn::kTransform);
    for ( ; !sIter.isDone(); sIter.next())
    {
        MDagPath dagPath;
        sIter.getDagPath(dagPath);
        //cerr << "sel node: " << dagPath.partialPathName() << R_ENDL;
        if (name.size() == 0 || dagPath.length() < nodePath.length())
        {
            name = MFnDagNode(dagPath).name().asChar();
            nodePath = dagPath;
        }
    }

    if (name.size() == 0)
    {
        if (checksSelectionError)
        {
            YShowError(pScene, // Node for Output File Name is not selected
                "エクスポートオプションの Output File Name が @node になっていますが、ノードが選択されていません。",
                "Although @node is specified for the Output File Name export option, no node is selected.");
            return MS::kFailure;
        }
        else
        {
            name = RExpOpt::SpecialOutFileNameForNode;
            return MS::kSuccess;
        }
    }

    name = GetOutElementName(name, yopt.m_RemoveNamespace); // changed (2008/02/28)

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 出力ファイル（フォルダ）とマージファイル（フォルダ）のパスの
//!        環境変数やマクロ文字を展開してフルパスに変換します。
//-----------------------------------------------------------------------------
MStatus YExpOpt::ExpandOutputAndMergePath(RScene* pScene, const bool checksSelectionError)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 展開前のパスをコピーします。
    m_OutFileName     = m_OrgOutFileName;
    m_OutFolderPath   = m_OrgOutFolderPath;
    m_MergeFmdPath    = m_OrgMergeFmdPath;
    m_MergeAnimFolder = m_OrgMergeAnimFolder;
    m_MergeAnimName   = m_OrgMergeAnimName;

    //-----------------------------------------------------------------------------
    // 出力ファイル名を展開してチェックします。
    if (m_OutFileName.empty())
    {
        m_OutFileName = m_SceneName;
    }

    if (m_OutFileName == RExpOpt::SpecialOutFileNameForNode)
    {
        status = GetOutFileNameFromSelection(m_OutFileName, pScene, *this, checksSelectionError);
        CheckStatus(status);
        // checksSelectionError が false の場合、選択されたノードがなければ @node のままです。
    }

    if (m_OutFileName != RExpOpt::SpecialOutFileNameForNode)
    {
        if (!RIsValidFileNameString(m_OutFileName) ||
            (m_ChecksWrongFileNameError && !RIsValidElementNameString(m_OutFileName)))
        {
            YShowError(pScene, // Output File Name is wrong
                "Output File Name に非 ASCII 文字やサポート外の記号が含まれています。"
                "「-（ハイフン）」「.（ドット）」「_（アンダーバー）」以外の記号は使用できません。",
                "The Output File Name contains non-ASCII or other unsupported characters. "
                "Symbols other than the hyphen (-), underscore (_), and period (.) cannot be used.");
            return MS::kFailure;
        }
        string orgFileName = m_OutFileName;
        m_OutFileName = RAdjustNameChar(m_OutFileName);
        if (m_OutFileName != orgFileName)
        {
            YShowWarning(pScene, // Output File Name is changed: %s -> %s
                "Output File Name に禁止文字が含まれるため、出力されるファイル名が変更されました: {0} -> {1}",
                "The file name to export is changed because the Output File Name includes prohibited characters: {0} -> {1}",
                orgFileName, m_OutFileName);
        }
    }

    m_ModelName = m_OutFileName;

    //-----------------------------------------------------------------------------
    // 出力フォルダのパスを展開してチェックします。
    if (m_OutFolderPath.empty() ||
        !RIsValidFilePathString(m_OutFolderPath))
    {
        YShowError(pScene, // Output Folder is wrong
            "Output Folder が空であるか、禁止文字が含まれています。「;（セミコロン）」は使用できません。",
            "The Output Folder was left blank or contains prohibited characters. Semicolons (;) cannot be used.");
        return MS::kFailure;
    }
    m_OutFolderPath = RGetFilePathWithEndingSlash(
        YGetFullFilePath(m_OutFolderPath, m_ProjectPath));

    //-----------------------------------------------------------------------------
    // テクスチャ出力フォルダのパスを設定します。
    m_TexFolderPath = m_OutFolderPath + RExpOpt::TexFolderName + "/";

    //-----------------------------------------------------------------------------
    // マージする fmd ファイルのパスを設定します。
    m_MergesOutputFmd = m_MergeFmdPath.empty();
    if (m_MergeFmdFlag && m_MergesOutputFmd)
    {
        m_MergeFmdPath = m_OutFolderPath + GetFileName(FMD);
    }

    if (!m_MergeFmdPath.empty())
    {
        m_MergeFmdPath = YGetFullFilePath(m_MergeFmdPath, m_ProjectPath);
    }

    //-----------------------------------------------------------------------------
    // マージする ftx ファイルのあるフォルダのパスを設定します。
    m_MergeFtxFolderPath = (MergesFmd()) ?
        RGetFolderFromFilePath(m_MergeFmdPath) + "/" + RExpOpt::TexFolderName + "/" :
        m_TexFolderPath;

    //-----------------------------------------------------------------------------
    // マージするアニメーション中間ファイルのあるフォルダのパスとファイル名を設定します。
    m_MergesOutputAnim = m_MergeAnimFolder.empty();
    if (m_MergesOutputAnim)
    {
        m_MergeAnimFolder = m_OutFolderPath;
    }
    else
    {
        m_MergeAnimFolder = YGetFullFilePath(m_MergeAnimFolder, ""); // 環境変数を展開します。
    }
    m_MergeAnimFolder = RGetFilePathWithEndingSlash(m_MergeAnimFolder);

    if (m_MergeAnimName.empty())
    {
        m_MergeAnimName = m_OutFileName;
    }

    //-----------------------------------------------------------------------------
    // テンポラリ出力フォルダを設定します。
    if (!m_SpecialTmpFolderPath.empty())
    {
        m_SpecialTmpFolderPath = RGetFilePathWithEndingSlash(
            YGetFullFilePath(m_SpecialTmpFolderPath, m_ProjectPath));
    }

    m_TmpOutFolderPath = m_OutFolderPath;
    const std::string tmpPath = (!m_SpecialTmpFolderPath.empty()) ?
        m_SpecialTmpFolderPath : GetTempFolderPath();
    if (!tmpPath.empty())
    {
        m_UseTmpFolderFlag = true;
        m_TmpOutFolderPath = tmpPath;
    }
    m_TmpTexFolderPath = m_TmpOutFolderPath + RExpOpt::TexFolderName + "/";

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションが有効か判定します。
//-----------------------------------------------------------------------------
MStatus YExpOpt::Check(RScene* pScene) // CheckOptions
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 出力ファイル（フォルダ）とマージファイル（フォルダ）のパスの
    // 環境変数やマクロ文字を展開してフルパスに変換します。
    const bool checksSelectionError = m_PreExpScript.empty();
    status = ExpandOutputAndMergePath(pScene, checksSelectionError);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // プリセット名をチェックします。
    if (!RIsValidPresetNameString(m_PresetName))
    {
        YShowError(pScene, // Preset Name is wrong: %s
            "プリセット名が空であるか、禁止文字が含まれています: {0} \n"
            "プリセット名には、ファイル名に使用できない文字、非 ASCII 文字、「;（セミコロン）」は使用できません。\n"
            "また「None」というプリセット名を使用することもできません。",
            "The Preset Name field was left blank or contains prohibited characters: {0} \n"
            "Characters that cannot be used in filenames, in addition to other non-ASCII characters and semicolons (;), cannot be used in the Preset Name field. \n"
            "Also, None cannot be used as the name for a preset.",
            m_PresetName);
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // check magnify
    const double MAGNIFY_MIN = 0.001;
    const double MAGNIFY_MAX = 1000.0;
    if (m_Magnify < MAGNIFY_MIN)
    {
        m_Magnify = MAGNIFY_MIN;
    }
    else if (m_Magnify > MAGNIFY_MAX)
    {
        m_Magnify = MAGNIFY_MAX;
    }
    m_InternalMagnify = MDistance::internalToUI(1.0) * m_Magnify;

    //-----------------------------------------------------------------------------
    // set frame range
    if (m_FrameRange == FRAME_RANGE_ALL)
    {
        GetMayaAnimationRange();
    }
    else if (m_FrameRange == FRAME_RANGE_PLAYBACK)
    {
        GetMayaPlaybackRange();
    }
    else
    {
        if (m_SpecifiedStartFrame > m_SpecifiedEndFrame)
        {
            RSwapValue(m_SpecifiedStartFrame, m_SpecifiedEndFrame);
        }

        #ifdef NO_CLAMP_FRAME_RANGE_SW
        m_StartFrame = m_SpecifiedStartFrame;
        m_EndFrame   = m_SpecifiedEndFrame;
        m_DoubleStartFrame = static_cast<double>(m_StartFrame);
        m_DoubleEndFrame   = static_cast<double>(m_EndFrame);
        #else
        if (m_SpecifiedStartFrame > m_EndFrame)
        {
            YShowWarning(pScene, "", "Start frame is wrong: (ignored): {0}", // 現在は発生しない
                RGetNumberString(m_SpecifiedStartFrame));
        }
        else if (m_SpecifiedStartFrame >= m_StartFrame)
        {
            m_StartFrame = m_SpecifiedStartFrame;
            m_DoubleStartFrame = static_cast<double>(m_StartFrame);
        }
        if (m_SpecifiedEndFrame < m_StartFrame)
        {
            YShowWarning(pScene, "", "End frame is wrong: (ignored): {0}", // 現在は発生しない
                RGetNumberString(m_SpecifiedEndFrame));
        }
        else if (m_SpecifiedEndFrame <= m_EndFrame)
        {
            // 終了フレームは 9999 のように大きい値がデフォルトに
            // なっているので値の調整は必須
            m_EndFrame = m_SpecifiedEndFrame;
            m_DoubleEndFrame = static_cast<double>(m_EndFrame);
        }
        #endif
    }

    // check frame precision
    if (m_FramePrecision < 1)
    {
        m_FramePrecision = 1;
    }

    // update frame count
    UpdateFrameCount();

    //-----------------------------------------------------------------------------
    // check output file flag
    if (!m_IsScriptOnly       &&
        !ExportsModelFile()   &&
        !ExportsTexture()     &&
        !ExportsAnim())
    {
        YShowError(pScene, // No output file
            "中間ファイルの出力指定がすべて OFF になっています。",
            "All output specifications for the intermediate file are cleared.");
        return MS::kFailure;
    }

    m_FmdIsBindPose = true;

    //-----------------------------------------------------------------------------
    // test
    //cerr << "frame: " << m_StartFrame << " - " << m_EndFrame << R_ENDL;
    //cerr << "merge_fmd: " << m_MergeFmdFlag << " (" << m_MergeFmdPath.c_str() << ")" << R_ENDL;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションのマージファイルが有効か判定します。
//-----------------------------------------------------------------------------
MStatus YExpOpt::CheckMergeFile(RScene* pScene)
{
    //-----------------------------------------------------------------------------
    // マージする fmd ファイルをチェックします。
    if (m_MergeFmdFlag && ExportsModelFile())
    {
        if (m_MergesOutputFmd)
        {
            if (!RFileExists(m_MergeFmdPath))
            {
                m_MergeFmdFlag = false;
                YShowWarning(pScene, // fmd file to merge is not found: %s
                    "Merge fmd File が ON かつ fmd ファイルのパスが空欄ですが、マージ対象のファイルが見つかりません。"
                    "マージしないで出力します: {0}",
                    "The file to merge cannot be found even though the Merge fmd File is selected and the fmd file path is left blank. "
                    "Files are exported without merging: {0}",
                    m_MergeFmdPath);
            }
        }
        else
        {
            if (m_MergeFmdPath.empty()                  ||
                !RIsValidFilePathString(m_MergeFmdPath) ||
                !RFileExists(m_MergeFmdPath))
            {
                YShowError(pScene, // Path of fmd file to merge is wrong: %s
                    "Merge fmd File が ON かつ fmd ファイルのパスを指定していますが、"
                    "マージする fmd ファイルが存在しないか、パスに禁止文字が含まれます: {0} \n"
                    "パスには「;（セミコロン）」は使用できません。",
                    "The Merge fmd File is selected and also an fmd file path is specified, "
                    "but the fmd file to be merged does not exist or the path includes one or more prohibited characters: {0} \n"
                    "Semicolons (;) cannot be used in the path.",
                    m_MergeFmdPath);
                return MS::kFailure;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // マージするアニメーション中間ファイルをチェックします。
    if (MergesAnim())
    {
        if (!RIsValidFilePathString(m_MergeAnimFolder))
        {
            YShowError(pScene, // Merge Animation Folder is wrong
                "Merge Animation Folder に禁止文字が含まれています。「;（セミコロン）」は使用できません。",
                "The Merge Animation Folder contains prohibited characters. Semicolons (;) cannot be used.");
            return MS::kFailure;
        }
        if (!RIsValidFileNameString(m_MergeAnimName))
        {
            YShowError(pScene, // Merge Animation Name is wrong
                "Merge Animation Name に禁止文字が含まれています。「;（セミコロン）」は使用できません。",
                "The Merge Animation Name contains prohibited characters. Semicolons (;) cannot be used.");
            return MS::kFailure;
        }

        // マージ対象のアニメーション中間ファイルが 1 つも存在しない場合は警告またはエラーにします。
        bool mergeAnimFileExists = false;
        for (int fileTypeIdx = 0; fileTypeIdx < FILE_TYPE_COUNT; ++fileTypeIdx)
        {
            const FileType fileType = static_cast<FileType>(fileTypeIdx);
            if (fileType != FMD && m_OutFileFlag[fileType])
            {
                if (RFileExists(GetMergeAnimFilePath(fileType)))
                {
                    mergeAnimFileExists = true;
                    break;
                }
            }
        }
        if (!mergeAnimFileExists)
        {
            if (m_MergesOutputAnim)
            {
                m_MergeAnimFlag = false;
                YShowWarning(pScene, // Animation file to merge is not found
                    "Merge Animation Folder が ON かつフォルダが空欄ですが、マージ対象のアニメーション中間ファイルが 1 つも存在しません。"
                    "マージしないで出力します。",
                    "No animation intermediate files to merge can be found even though the Merge Animation Folder is enabled and a folder has not been specified. "
                    "Files are exported without merging.");
            }
            else
            {
                YShowError(pScene, // Animation file to merge is not found
                    "Merge Animation Folder が ON かつフォルダを指定していますが、"
                    "マージ対象のアニメーション中間ファイルが 1 つも存在しません。",
                    "No animation files to merge could be found "
                    "even though the Merge Animation Folder is enabled and a folder has been specified.");
                return MS::kFailure;
            }
        }
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief Export コマンドを実行します。
//!
//! @param[in] args 引数リストです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoExportCmd::doIt(const MArgList& args) // DoIt
{
    MStatus status;
    //cerr << "YModel size: " << sizeof(YModel) << endl;

    //-----------------------------------------------------------------------------
    // 2 重起動を防止します。
    if (MGlobal::mayaState() == MGlobal::kInteractive && s_IsExporting)
    {
        MGlobal::displayInfo("Skip double NW4F export");
        //s_IsExporting = false;
        return status;
    }
    s_IsExporting = true;

    //-----------------------------------------------------------------------------
    // ロケールを設定します。
    #if (_MSC_VER < 1900)
    setlocale(LC_CTYPE, "");
    #endif

    //-----------------------------------------------------------------------------
    // シーンデータを作成し、エラー出力ストリームを初期化します。
    YScene* pYScene = new YScene(); // 以降 delete するまで return できません。
    YScene& yscene = *pYScene;
    yscene.SetErrPtr(&cerr);
    RInitOutStreamFormat(yscene.Err());

    //-----------------------------------------------------------------------------
    // 引数を解析しエクスポートオプションを設定します。
    status = parseArgs(args);
    while (status) // エラー発生時に break するための while です。
    {
        YExpOpt& yopt = yscene.GetOpt();
        yopt.Init();
        status = yopt.Parse(&yscene, m_OptionString, false);

        //-----------------------------------------------------------------------------
        // start maya error log
        #ifdef OUTPUT_ERROR_LOG
        const std::string errorLogPath = RGetTempFolderPath() +
            "Nintendo_MayaPlugins_ExportLog_" + RGetCurrentProcessIdString() + ".txt";
        MGlobal::startErrorLogging(errorLogPath.c_str());
        cerr << "Logging: " << MGlobal::errorLogPathName() << R_ENDL;
        #endif

        //-----------------------------------------------------------------------------
        #ifdef DEBUG_PRINT_SW
        cerr << "---------------------------------------------------------- " << s_ExportCount << R_ENDL;
        #endif
        ++s_ExportCount;

        //-----------------------------------------------------------------------------
        // オプション設定の妥当性をチェックします。
        if (status)
        {
            status = yopt.Check(&yscene);
        }

        //-----------------------------------------------------------------------------
        // Maya のデータを変換し、中間ファイルに出力します。
        if (status)
        {
            status = Process(yscene, m_ForceOverwrite);
        }

        //-----------------------------------------------------------------------------
        // エラーか警告があれば Output Window を前面に表示させます。
        if (!status || yscene.GetWarningCount() != 0)
        {
            if (MGlobal::mayaState() == MGlobal::kInteractive)
            {
                yscene.Err() << R_ENDL;
                SetMayaOutputWindowForeground();
            }
        }

        //-----------------------------------------------------------------------------
        // close maya error log
        #ifdef OUTPUT_ERROR_LOG
        MGlobal::closeErrorLog();
        #endif

        break; // while から抜けます。
    }

    //-----------------------------------------------------------------------------
    // シーンデータを解放します。
    if (pYScene != NULL) delete pYScene;

    //-----------------------------------------------------------------------------
    // Export コマンド実行中フラグを false にします。
    s_IsExporting = false;

    return status;
}

//-----------------------------------------------------------------------------
//! @brief モデル 1 つを評価します。
//-----------------------------------------------------------------------------
static MStatus EvaluateOneModel(YModel& ymodel)
{
    MStatus status;
    //YScene& yscene = ymodel.GetScene();
    //const YExpOpt& yopt = yscene.GetOpt();

    //-----------------------------------------------------------------------------
    // 階層構造を取得します。
    status = GetHierarchy(ymodel, false);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // マテリアルを取得します。
    const int nodeCount = static_cast<int>(ymodel.m_pYNodes.size());
    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        const YNode& ynode = *ymodel.m_pYNodes[iNode];
        if (ynode.m_OrgKind == YNode::KindMesh)
        {
            const int sgCount = static_cast<int>(ynode.m_SGObjs.size());
            for (int iSg = 0; iSg < sgCount; ++iSg)
            {
                const MObject sgObj = ynode.m_SGObjs[iSg];
                int iMat;
                int priority;
                MStringArray meshUvSetNames;
                RStringArray uvAttribNames;
                RIntArray uvHintIdxs;
                status = GetMaterial(ymodel, iMat, priority,
                    meshUvSetNames, uvAttribNames, uvHintIdxs,
                    sgObj, ynode.m_ShapePath, ynode.m_DefaultUvSet);
                CheckStatus(status);
            }
        }
    }

    SetUniqueMaterialNames(ymodel);
    SortMaterialByName(ymodel);

    //-----------------------------------------------------------------------------
    // テクスチャデータをソートします。
    SetOutTexImgPtrs(ymodel.m_pOutTexImgs, ymodel.m_TexImgs);

    return status;
}

//=============================================================================
//! @brief モデル評価コマンドのクラスです。
//!
//! Usage:
//!    NintendoEvaluateModelCmd [flags]
//! Flags:
//!    -selection (-sl) 指定すると選択されたノード以下をモデルとして評価します。
//!                     指定しなければシーン全体をモデルとして評価します。
//!    -removeNamespace (-rn) 指定するとボーン名、マテリアル名などのネームスペース部分を削除します。
//!    -updateExportEnv (-uee) 指定すると Pre-Export Script で取得できる環境変数を更新します（モデルの評価はしません）。
//=============================================================================
class NintendoEvaluateModelCmd : public MPxCommand
{
public:
    NintendoEvaluateModelCmd()
    {
    }
    virtual ~NintendoEvaluateModelCmd()
    {
    }
    MStatus doIt(const MArgList& args);
    static void* creator();
    static MSyntax newSyntax();
};

//-----------------------------------------------------------------------------
//! @brief モデル評価コマンドを作成します。
//-----------------------------------------------------------------------------
void* NintendoEvaluateModelCmd::creator() { return new NintendoEvaluateModelCmd(); }

//-----------------------------------------------------------------------------
//! @brief モデル評価コマンドの文法を作成します。
//-----------------------------------------------------------------------------
#define kSelectionFlag          "sl"
#define kSelectionFlagL         "selection"
#define kRemoveNamespaceFlag    "rn"
#define kRemoveNamespaceFlagL   "removeNamespace"
#define kUpdateExportEnvFlag    "uee"
#define kUpdateExportEnvFlagL   "updateExportEnv"

MSyntax NintendoEvaluateModelCmd::newSyntax()
{
    MSyntax syntax;
    syntax.addFlag(kSelectionFlag, kSelectionFlagL);
    syntax.addFlag(kRemoveNamespaceFlag, kRemoveNamespaceFlagL);
    syntax.addFlag(kUpdateExportEnvFlag, kUpdateExportEnvFlagL);
    return syntax;
}

//-----------------------------------------------------------------------------
//! @brief モデル評価コマンドを実行します。
//-----------------------------------------------------------------------------
MStatus NintendoEvaluateModelCmd::doIt(const MArgList& args)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // ロケールを設定します。
    #if (_MSC_VER < 1900)
    setlocale(LC_CTYPE, "");
    #endif

    //-----------------------------------------------------------------------------
    // 引数を解析します。
    MArgDatabase argData(syntax(), args, &status);
    if (!status)
    {
        return status;
    }
    const bool isSelection      = argData.isFlagSet(kSelectionFlag);
    const bool removesNamespace = argData.isFlagSet(kRemoveNamespaceFlag);
    const bool updatesExportEnv = argData.isFlagSet(kUpdateExportEnvFlag);
    //cerr << "evaluate model: " << isSelection << " " << removesNamespace << endl;

    //-----------------------------------------------------------------------------
    // 評価用のシーンデータを作成します。
    YScene yscene;
    yscene.SetEvaluating(true);

    //-----------------------------------------------------------------------------
    // 評価用のエクスポートオプションを設定します。
    YExpOpt& yopt = yscene.GetOpt();
    yopt.Init();
    const std::string configPath = GetNintendoMayaConfigFolderPath(true) + ExportConfigFileName;
    if (RFileExists(configPath))
    {
        yopt.ParseConfigFile(configPath);
    }
    yopt.m_Target = (isSelection) ? RExpOpt::EXPORT_TARGET_SELECTION : RExpOpt::EXPORT_TARGET_ALL;
    yopt.m_RemoveNamespace = removesNamespace;
    yopt.m_WarnsNodeNameChanged = false;

    //-----------------------------------------------------------------------------
    // Pre-Export Script で取得できる環境変数を更新します。
    // その場合モデルの評価はしません。
    if (updatesExportEnv)
    {
        return UpdateEnvVarForPrePostExportScript(yscene, yopt);
    }

    //-----------------------------------------------------------------------------
    // モデルを評価します。
    YModel ymodel(&yscene);
    EvaluateOneModel(ymodel);

    //-----------------------------------------------------------------------------
    // ボーンに関する環境変数を設定します。
    std::string boneDagPathsStr;
    std::string boneNamesStr;
    for (size_t iNode = 0; iNode < ymodel.m_pOutYNodes.size(); ++iNode)
    {
        const YNode& ynode = *ymodel.m_pOutYNodes[iNode];
        if (iNode > 0)
        {
            boneDagPathsStr += " ";
            boneNamesStr    += " ";
        }
        boneDagPathsStr += (ynode.m_AddedFlag) ? "<null>" : ynode.m_XformPath.partialPathName().asChar();
        boneNamesStr += ynode.m_Name;
    }
    SetEnvVarByMel("NINTENDO_EVALUATE_MODEL_BONE_DAG_PATHS", boneDagPathsStr, true);
    SetEnvVarByMel("NINTENDO_EVALUATE_MODEL_BONE_NAMES", boneNamesStr, true);

    //-----------------------------------------------------------------------------
    // マテリアルに関する環境変数を設定します。
    std::string matSgsStr;
    std::string matNamesStr;
    for (size_t iMat = 0; iMat < ymodel.m_pOutYMaterials.size(); ++iMat)
    {
        const YMaterial& mat = *ymodel.m_pOutYMaterials[iMat];
        if (iMat > 0)
        {
            matSgsStr   += " ";
            matNamesStr += " ";
        }
        matSgsStr += (mat.m_DefaultFlag) ? "<null>" : MFnDependencyNode(mat.m_SGObj).name().asChar();
        matNamesStr += mat.m_Name;
    }
    SetEnvVarByMel("NINTENDO_EVALUATE_MODEL_MATERIAL_SGS", matSgsStr, true);
    SetEnvVarByMel("NINTENDO_EVALUATE_MODEL_MATERIAL_NAMES", matNamesStr, true);

    //-----------------------------------------------------------------------------
    // テクスチャに関する環境変数を設定します。
    std::string ftxFileNamesStr;
    for (size_t iTexImg = 0; iTexImg < ymodel.m_pOutTexImgs.size(); ++iTexImg)
    {
        const RImage& texImg = *ymodel.m_pOutTexImgs[iTexImg];
        if (iTexImg > 0)
        {
            ftxFileNamesStr += " ";
        }
        const std::string ftxName = texImg.GetName() + "." + yopt.GetExtension(RExpOpt::FTX);
        ftxFileNamesStr += ftxName;
    }
    SetEnvVarByMel("NINTENDO_EVALUATE_MODEL_FTX_FILE_NAMES", ftxFileNamesStr, true);

    //-----------------------------------------------------------------------------
    // test
    //cerr << "error  : " << yscene.GetErrorCount() << endl;
    //cerr << "warning: " << yscene.GetWarningCount() << endl;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief プラグインを初期化します。
//!
//! @param[in] obj プラグインのオブジェクトです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus initializePlugin(MObject obj)
{
    MStatus status;

    MString versionStr = ExporterVersion;
    #ifdef NDEBUG
    versionStr += " (Release)";
    #else
    versionStr += " (Debug)";
    #endif
    MFnPlugin plugin(obj, "NintendoWare", versionStr.asChar(), "1.0");

    //-----------------------------------------------------------------------------
    // export command
    status = plugin.registerCommand("NintendoExportCmd",
        NintendoExportCmd::creator,
        NintendoExportCmd::newSyntax);
    CheckStatusMsg(status, "register NintendoExportCmd");

    //-----------------------------------------------------------------------------
    // export info command
    status = plugin.registerCommand("NintendoExportInfoCmd",
        NintendoExportInfoCmd::creator,
        NintendoExportInfoCmd::newSyntax);
    CheckStatusMsg(status, "register NintendoExportInfoCmd");

    //-----------------------------------------------------------------------------
    // evaluate model command
    status = plugin.registerCommand("NintendoEvaluateModelCmd",
        NintendoEvaluateModelCmd::creator,
        NintendoEvaluateModelCmd::newSyntax);
    CheckStatusMsg(status, "register NintendoEvaluateModelCmd");

    //-----------------------------------------------------------------------------
    // check LOD
    status = NintendoCheckLodCmd::Register(plugin);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // インポート
    extern MStatus InitializeNintendoImport(MFnPlugin& plugin);
    status = InitializeNintendoImport(plugin);
    CheckStatus(status);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief プラグインを初期化解除します。
//!
//! @param[in] obj プラグインのオブジェクトです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus uninitializePlugin(MObject obj)
{
    MStatus status;
    MFnPlugin plugin(obj);

    //-----------------------------------------------------------------------------
    // export command
    status = plugin.deregisterCommand("NintendoExportCmd");
    CheckStatusMsg(status, "deregister NintendoExportCmd");

    //-----------------------------------------------------------------------------
    // export info command
    status = plugin.deregisterCommand("NintendoExportInfoCmd");
    CheckStatusMsg(status, "deregister NintendoExportInfoCmd");

    //-----------------------------------------------------------------------------
    // evaluate model command
    status = plugin.deregisterCommand("NintendoEvaluateModelCmd");
    CheckStatusMsg(status, "deregister NintendoEvaluateModelCmd");

    //-----------------------------------------------------------------------------
    // check LOD
    status = NintendoCheckLodCmd::Deregister(plugin);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // インポート
    extern MStatus UninitializeNintendoImport(MFnPlugin& plugin);
    status = UninitializeNintendoImport(plugin);
    CheckStatus(status);

    return status;
}

