﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// YBlendShapeData YBlendShapeTarget YVertexShapeAnim

//=============================================================================
// include
//=============================================================================
#include "DccShape.h"
#include "Animation.h"

//=============================================================================
//! @brief ブレンドシェイプターゲットのクラスです。
//=============================================================================
class YBlendShapeTarget
{
public:
    //-----------------------------------------------------------------------------
    // ターゲットシェイプ

    //! @brief ターゲット名（ウェイトプラグのエイリアス名）です。
    //!        ウェイトプラグにエイリアス名が設定されていなければ
    //!        ターゲットシェイプの transform ノード名が設定されます。
    //!        ターゲットシェイプもなければ固定の名前が設定されます。
    std::string m_Name;

    MDagPath m_ShapePath; //!< シェイプノードの DAG パスです。
    bool m_Visibility; //!< シェイプエディタでの可視性です。

    //! @brief ローカル座標系での頂点座標配列です。
    //!        デフォーマセットのメンバでない頂点はベースシェイプの頂点座標が格納されます。
    MPointArray m_Poss;

    //! @brief グローバル座標系での頂点座標配列です。
    //!        デフォーマセットのメンバでない頂点はベースシェイプの頂点座標が格納されます。
    MPointArray m_GlobalPoss;

    MString m_ColSets[nn::gfx::tool::dcc::RPrimVtx::VTX_COL_MAX]; //!< 出力する頂点カラー属性のカラーセット名配列です。

    //-----------------------------------------------------------------------------
    // アニメーション
    MPlug m_AnimPlug; //!< blendShape ノードのウェイトプラグ（weight[weightIndex]）です。
    MPlugArray m_GroupWeightPlugs; //!< 親（先祖）グループのウェイトプラグ配列です。
    YAnimCurve m_Anim; //!< ウェイト値のアニメーションカーブです。

public:
    //! @brief 現在のウェイト値を取得します。
    //!
    //! @return 現在のウェイト値を返します。
    //!
    float GetWeight() const;
};

//! @brief ブレンドシェイプターゲット配列の定義です。
typedef std::vector<YBlendShapeTarget> YBlendShapeTargetArray;

//=============================================================================
//! @brief ブレンドシェイプデータのクラスです。
//!        ベースシェイプとターゲットシェイプの情報を保持します。
//=============================================================================
class YBlendShapeData
{
public:
    //! @brief 使用する頂点属性のタイプを表す列挙型です。
    enum UseType
    {
        PER_VTX_FACE = -2, //!< 頂点フェース単位の頂点属性を使用します。
        PER_VTX = -1       //!< 頂点単位の法線を使用します。
    };

    //-----------------------------------------------------------------------------
    // ブレンドシェイプオブジェクト
    std::string m_Name; //!< blendShape ノードの名前です。
    MObject m_BlendShapeObj; //!< blendShape ノードです。
    nn::gfx::tool::dcc::RShapeUpdate m_ShapeUpdate; //!< シェイプアニメーションの頂点属性更新情報です。

    //-----------------------------------------------------------------------------
    // ベースシェイプ
    std::string m_BaseName; //!< ベースシェイプの名前（ボーンの出力名）です。
    MDagPath m_BasePath; //!< ベースシェイプのシェイプノードの DAG パスです。
    int m_VtxCount; //!< ベースシェイプの頂点数です。
    int m_FaceCount; //!< ベースシェイプのフェース数です。

    //! @brief ベースシェイプの出力する頂点カラー属性のカラーセット名配列です。
    //!        全ターゲットシェイプに対応するカラーセットが存在しなければ
    //!        空文字となります。
    MString m_BaseColSets[nn::gfx::tool::dcc::RPrimVtx::VTX_COL_MAX];

    //-----------------------------------------------------------------------------
    // ターゲットシェイプ
    YBlendShapeTargetArray m_Targets; //!< ブレンドシェイプターゲット配列です。

    //! @brief 出力するターゲットのウェイト値のアニメーションカーブ数です。
    //!        キーが打たれているか、値が 0 でないウェイト値のみ出力します。
    int m_OutTargetAnimCount;

    //-----------------------------------------------------------------------------
    // フラグ

    //! @brief ベースシェイプの各頂点がブレンドシェイプのメンバーかどうかのフラグ配列です。
    //!        ベースシェイプの頂点数と同じ長さの配列です。
    //!        頂点座標が変化するかどうかではなく、
    //!        ブレンドシェイプが接続されているセットオブジェクト内のコンポーネントに
    //!        その頂点が含まれているかどうかで判断されます。
    MIntArray m_IsMemberVtxs;

    //! @brief ベースシェイプの各フェースがブレンドシェイプのメンバーかどうかのフラグ配列です。
    //!        ベースシェイプのフェース数と同じ長さの配列です。
    //!        フェースの頂点が 1 つでもブレンドシェイプのメンバーなら
    //!        そのフェースはブレンドシェイプのメンバーとみなします。
    MIntArray m_IsMemberFaces;

    //! @brief ベースシェイプの各頂点座標がブレンドシェイプで変化するかどうかのフラグ配列です。
    //!        ベースシェイプの頂点数と同じ長さの配列です。
    MIntArray m_MovePosFlags;

    //! @brief 頂点単位の法線の使用フラグ配列です。
    //!        ベースシェイプの頂点数と同じ長さの配列です。
    //!        頂点が属するフェースすべてで同じ法線を使用するなら PER_VTX、
    //!        フェースごとに異なる法線を使用するなら PER_VTX_FACE が設定されます。
    MIntArray m_UseVtxNrmFlags;

    //! @brief 頂点単位の頂点カラーの使用フラグ配列の配列です。
    //!        使用フラグ配列はベースシェイプの頂点数と同じ長さの配列です。
    //!        頂点が属するフェースすべてで同じ頂点カラーを使用するなら PER_VTX、
    //!        フェースごとに異なる頂点カラーを使用するなら PER_VTX_FACE が設定されます。
    MIntArray m_UseVtxColFlagss[nn::gfx::tool::dcc::RPrimVtx::VTX_COL_MAX];
};

//! @brief ブレンドシェイプデータ配列の定義です。
typedef std::vector<YBlendShapeData> YBlendShapeDataArray;

//=============================================================================
//! @brief 頂点シェイプアニメーションのクラスです。
//=============================================================================
class YVertexShapeAnim
{
public:
    std::string m_ShapeName; //!< アニメーション対象 <shape> の名前です。
    int m_BlendShapeDataIndex; //!< ブレンドシェイプデータのインデックスです。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] shapeName アニメーション対象 <shape> の名前です。
    //! @param[in] blendShapeDataIndex ブレンドシェイプデータのインデックスです。
    //!
    YVertexShapeAnim(const std::string& shapeName, const int blendShapeDataIndex)
    : m_ShapeName(shapeName),
      m_BlendShapeDataIndex(blendShapeDataIndex)
    {
    }
};

//! @brief 頂点シェイプアニメーション配列の定義です。
typedef std::vector<YVertexShapeAnim> YVertexShapeAnimArray;

//=============================================================================
// グローバル関数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// デフォーマ全般

//! @brief シェイプのヒストリーに存在するデフォーマをすべて取得します。
//!
//! @param[out] nodes デフォーマオブジェクトの配列を格納します。
//! @param[in] shapeObj シェイプオブジェクトです。
//!
//! @return デフォーマが見つかったなら true を返します。
//!
bool FindDeformerNodes(CObjectArray& nodes, const MObject& shapeObj);

//! @brief デフォーマオブジェクト群を無効にします。
//!        nodeState アトリビュートに 1 (HasNoEffect) を設定し、評価されないようにします。
//!
//! @param[out] nodeStates デフォーマオブジェクト群の nodeState アトリビュートに
//!                        設定されていた値を格納します（リストア用）。
//! @param[in] nodes 設定を変更するデフォーマオブジェクトの配列です。
//!
void DisableDeformerStates(MIntArray& nodeStates, const CObjectArray& nodes);

//! @brief デフォーマオブジェクト群の有効状態を元に戻します。
//!
//! @param[in] nodes 設定を変更するデフォーマオブジェクトの配列です。
//! @param[in] nodeStates nodeState アトリビュートに設定する値です。
//!                       nodes と同じ長さである必要があります。
//!
void RestoreDeformerStates(const CObjectArray& nodes, const MIntArray& nodeStates);

//-----------------------------------------------------------------------------
// スキニング

//! @brief 指定したシェイプノードのスキニングに影響するオブジェクト（joint ノードも）
//!        をすべて取得します。
//!
//! @param[out] infPaths スキニングに影響するオブジェクトの DAG パス配列を格納します。
//! @param[in] shapePath シェイプノードの DAG です。
//!
//! @return スキニングに影響するオブジェクト数を返します。
//!
int GetSkinInfluencePathsForShape(CDagPathArray& infPaths, const MDagPath& shapePath);

//! @brief シーン中のスキニングが設定されたオブジェクトをすべて取得します。
//!
//! @param[out] skinedObjPaths スキニングが設定されたオブジェクトの DAG パス配列を格納します。
//!
void GetSkinedObjectPaths(CDagPathArray& skinedObjPaths);

//! @brief シーン中のスキニングのインフルエンスオブジェクト（joint ノード以外）
//!        をすべて取得します。
//!
//! @param[out] infPaths インフルエンスオブジェクト配列を格納します。
//!
void GetSkinInfluencePaths(CDagPathArray& infPaths);

//! @brief 複数のバインドポーズが使用されていれば true を返します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] effectiveSkinClusters 出力対象に影響する skinCluster オブジェクト配列です。
//!
//! @return 複数のバインドポーズが使用されていれば true を返します。
//!
bool IsMultipleBindPoseUsed(
    const MDagPath& xformPath,
    const MObjectArray& effectiveSkinClusters
);

//! @brief バインドポーズのノード行列を取得します。
//!
//! @param[out] mtx バインドポーズのノード行列を格納します。
//! @param[in,out] rscene シーンです。
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] effectiveSkinClusters 出力対象に影響する skinCluster オブジェクト配列です。
//! @param[in] magnify 移動値に掛ける倍率です。
//!
//! @return 処理結果を返します。
//!
MStatus GetBindPoseMtx(
    MMatrix& mtx,
    nn::gfx::tool::dcc::RScene& rscene,
    const MDagPath& xformPath,
    const MObjectArray& effectiveSkinClusters,
    const double magnify
);

//! @brief バインドポーズのセグメントスケール補正を取得します（未完成）。
//!
MStatus GetBindPoseScaleCompensate(bool& scaleCompensate, const MDagPath& xformPath);

//-----------------------------------------------------------------------------
// ブレンドシェイプ

//! @brief 指定した transform ノードの子のシェイプノードが
//!        ブレンドシェイプのターゲットシェイプとして参照されているなら true を返します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//!
//! @return ターゲットシェイプとして参照されているなら true を返します。
//!
bool IsBlendShapeTarget(const MDagPath& xformPath);

//! @brief ブレンドシェイプデータを取得します。
//!        ブレンドシェイプのベースとターゲットの頂点情報を取得し、
//!        更新される頂点属性を設定します。
//!
//! @param[out] bsData ブレンドシェイプデータを格納します。
//! @param[out] isValid ブレンドシェイプが有効なら true を格納します。
//!                     ブレンドシェイプのメンバーである頂点がない場合、
//!                     更新する頂点属性がない場合、
//!                     ターゲットシェイプがない場合は false を格納します。
//! @param[in,out] rscene シーンです。
//! @param[in] baseName ベースシェイプの名前（ボーンの出力名）です。
//! @param[in] baseShapePath ベースシェイプのシェイプノードの DAG パスです。
//! @param[in] blendShapeObj blendShape ノードです。
//! @param[in] orgPosArray ベースシェイプのローカル座標系での頂点座標配列です。
//! @param[in] pivot ベースシェイプの transform ノードのスケールピボットです。
//! @param[in] baseColSets ベースシェイプの出力する頂点カラー属性のカラーセット名配列です。
//!                        RPrimVtx::VTX_COL_MAX と同じ長さの配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//!
MStatus GetBlendShapeData(
    YBlendShapeData& bsData,
    bool& isValid,
    nn::gfx::tool::dcc::RScene& rscene,
    const std::string& baseName,
    const MDagPath& baseShapePath,
    const MObject& blendShapeObj,
    const MPointArray& orgPosArray,
    const MVector& pivot,
    const MString* baseColSets,
    const YExpOpt& yopt
);

//! @brief 指定したフェース群の頂点属性がブレンドシェイプで変化するか判定して
//!        シェイプアニメーションの頂点属性更新情報を設定します。
//!
//! @param[in,out] shapeUpdate シェイプアニメーションの頂点属性更新情報です。
//!                            変化しないと判明している頂点属性の更新フラグは
//!                            この関数を呼ぶ前に false にしておきます。
//! @param[in] bsData ブレンドシェイプデータです。
//! @param[in] comp フェース群のコンポーネントです。
//! @param[in] tanUvSetNamess 接線を取得する UV セット名の配列の配列です。
//!                           UV セット名の配列はベースシェイプが先頭で、ベースシェイプ以外のキーシェイプが
//!                           存在すればその後に格納します。
//!                           接線を出力しない場合、UV セット名の配列は空です。
//! @param[in] forcesVtxColor true なら頂点カラーが設定されていない場合に
//!                           (1, 1, 1, 1) が設定されているとして処理します。
//!                           false なら (0, 0, 0, 1) が設定されているとして処理します。
//!
void SetBlendShapeUpdateForShape(
    nn::gfx::tool::dcc::RShapeUpdate& shapeUpdate,
    const YBlendShapeData& bsData,
    MObject& comp,
    const MStringArray* tanUvSetNamess,
    const bool forcesVtxColor
);

//! @brief 頂点単位の接線（従法線）の使用フラグ配列を取得します。
//!        高速化のためにベースシェイプの接線（従法線）は
//!        出力用に取得したものを再利用します。
//!
//! @param[out] useVtxTanFlags 頂点単位の接線（従法線）の使用フラグ配列を格納します。
//!             ベースシェイプの頂点数と同じ長さの配列です。
//!             頂点が属するフェースすべてで同じ接線（従法線）を使用するなら PER_VTX、
//!             フェースごとに異なる接線（従法線）を使用するなら PER_VTX_FACE が設定されます。
//! @param[in] bsData ブレンドシェイプデータです。
//! @param[in] tanUvSetNames 接線を取得する UV セット名の配列です。
//!                          ベースシェイプが先頭で、ベースシェイプ以外のキーシェイプが
//!                          存在すればその後に格納します。
//!                          接線を出力しない場合は空です。
//! @param[in] baseTans ベースシェイプの全接線配列です。
//! @param[in] baseBins ベースシェイプの全従法線配列です。
//!
void GetBlendShapeUseVtxTanFlags(
    MIntArray& useVtxTanFlags,
    const YBlendShapeData& bsData,
    const MStringArray& tanUvSetNames,
    const MFloatVectorArray& baseTans,
    const MFloatVectorArray& baseBins
);

//! @brief 全サブフレームにおけるブレンドシェイプアニメーション値を取得します。
//!
//! @param[in,out] bsDatas ブレンドシェイプデータ配列です。
//!
void GetBlendShapeFullAnimValue(YBlendShapeDataArray& bsDatas);

//! @brief 全サブフレームにおけるブレンドシェイプアニメーション値を分析します。
//!
//! @param[in,out] bsDatas ブレンドシェイプデータ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
void AnalyzeBlendShapeFullAnim(YBlendShapeDataArray& bsDatas, const YExpOpt& yopt);

//! @brief ブレンドシェイプのキーアニメーションを取得します。
//!
//! @param[in,out] bsDatas ブレンドシェイプデータ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//!
MStatus GetBlendShapeKeyAnim(YBlendShapeDataArray& bsDatas, const YExpOpt& yopt);

//! @brief fsh ファイル（シェイプアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] vertexShapeAnims 頂点シェイプアニメーション配列です。
//! @param[in] bsDatas ブレンドシェイプデータ配列です。
//!
//! @return 処理結果を返します。
//!
MStatus OutputFshFile(
    std::ostream& os,
    nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
    const YVertexShapeAnimArray& vertexShapeAnims,
    const YBlendShapeDataArray& bsDatas
);

