﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// 定数 グローバル
// YIkHandle

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
#ifndef NDEBUG
    #define DEBUG_PRINT_SW
#endif

#define USE_HIGH_COLOR_SW // カラーの値が 0 ～ 1 の範囲外でもそのまま出力するなら定義します。

//=============================================================================
// 特定の警告を無効にします。
//=============================================================================
//#pragma warning (disable: 4786)

//=============================================================================
// dcc ライブラリのヘッダを include します。
//=============================================================================
#include "DccCommon.h"
#include "DccImage.h"

//=============================================================================
// C++ 標準ライブラリのヘッダを include します。
//=============================================================================
// dcc で必要になったら DccCommon.h に移動します。
//#include <cassert>

//=============================================================================
// Maya のヘッダを include します。
//=============================================================================
#define REQUIRE_IOSTREAM // Maya のクラスを std::ostream で表示する場合に定義します。

#pragma warning(push)
#pragma warning(disable: 4819) // ファイルは、現在のコード ページ (932) で表示できない文字を含んでいます。
    // ↑Maya 2017 Update 3 の不具合対策です。

#include <maya/MGlobal.h>
#include <maya/MStatus.h>
#include <maya/MTime.h>
#include <maya/MObjectArray.h>
#include <maya/MDagPath.h>
#include <maya/MDagPathArray.h>
#include <maya/MSelectionList.h>
#include <maya/MPlug.h>
#include <maya/MPlugArray.h>
#include <maya/MFeedbackLine.h>
#include <maya/M3dView.h>
#include <maya/MFileIO.h>
#if (MAYA_API_VERSION >= 20180000)
    #include <maya/MDGContextGuard.h>
#endif

#include <maya/MMatrix.h>
#include <maya/MFloatMatrix.h>
#include <maya/MPointArray.h>
#include <maya/MFloatPointArray.h>
#include <maya/MVectorArray.h>
#include <maya/MFloatVectorArray.h>
#include <maya/MColorArray.h>
#include <maya/MEulerRotation.h>
#include <maya/MQuaternion.h>
#include <maya/MDistance.h>
#include <maya/MBoundingBox.h>

#include <maya/MFnDagNode.h>
#include <maya/MFnAttribute.h>
#include <maya/MFnTransform.h>
#include <maya/MFnIkJoint.h>
#include <maya/MFnIkHandle.h>
#include <maya/MFnIntArrayData.h>
#include <maya/MFnMesh.h>
#include <maya/MFnMatrixData.h>
#include <maya/MFnSingleIndexedComponent.h>
#include <maya/MFnDoubleIndexedComponent.h>
#include <maya/MFnTripleIndexedComponent.h>
#include <maya/MFnSet.h>
#include <maya/MFnWeightGeometryFilter.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MFnBlendShapeDeformer.h>
#include <maya/MFnNurbsCurve.h>
#include <maya/MFnNurbsSurface.h>
#include <maya/MFnLambertShader.h>
#include <maya/MFnReflectShader.h>
#include <maya/MFnPhongShader.h>
#include <maya/MFnCamera.h>
#include <maya/MFnLight.h>

#include <maya/MItDag.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MItSelectionList.h>
#include <maya/MItGeometry.h>
#include <maya/MItMeshVertex.h>
#include <maya/MItMeshEdge.h>
#include <maya/MItMeshPolygon.h>

#include <maya/MAnimControl.h>
#include <maya/MAnimUtil.h>
#include <maya/MFnAnimCurve.h>

#include <maya/MPxFileTranslator.h>
#include <maya/MPxCommand.h>
#include <maya/MSyntax.h>
#include <maya/MArgDatabase.h>

#pragma warning(pop)

//-----------------------------------------------------------------------------
//! @brief Maya のクラスの配列型のカスタム定義です。
//-----------------------------------------------------------------------------

// MDagPathArray doesn't have operator=
typedef std::vector<MDagPath> CDagPathArray;

// MObjectArray doesn't have operator=
typedef std::vector<MObject> CObjectArray;

//=============================================================================
// 定数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// color
const MColor Y_MCOLOR_NULL(-1.0f, -1.0f, -1.0f, -1.0f);

#ifdef USE_HIGH_COLOR_SW
const bool Y_CLAMPS_COLOR = false; //!< カラーの値が 0 ～ 1 の範囲外のときにクランプするなら true です。
#else
const bool Y_CLAMPS_COLOR = true; //!< カラーの値が 0 ～ 1 の範囲外のときにクランプするなら true です。
#endif

//-----------------------------------------------------------------------------
// joint

//! @brief joint ラベルタイプを表す列挙型です。joint ノードの type アトリビュートの値に対応しています。
enum YJointLabelType
{
    Y_JOINT_LABEL_TYPE_NONE = 0,        //!<  0=なし
    Y_JOINT_LABEL_TYPE_ROOT,            //!<  1=ルート
    Y_JOINT_LABEL_TYPE_HIP,             //!<  2=ヒップ
    Y_JOINT_LABEL_TYPE_KNEE,            //!<  3=膝
    Y_JOINT_LABEL_TYPE_FOOT,            //!<  4=足
    Y_JOINT_LABEL_TYPE_TOE,             //!<  5=足指
    Y_JOINT_LABEL_TYPE_SPINE,           //!<  6=脊椎
    Y_JOINT_LABEL_TYPE_NECK,            //!<  7=首
    Y_JOINT_LABEL_TYPE_HEAD,            //!<  8=頭部
    Y_JOINT_LABEL_TYPE_COLLAR,          //!<  9=鎖骨
    Y_JOINT_LABEL_TYPE_SHOULDER,        //!< 10=肩
    Y_JOINT_LABEL_TYPE_ELBOW,           //!< 11=肘
    Y_JOINT_LABEL_TYPE_HAND,            //!< 12=手
    Y_JOINT_LABEL_TYPE_FINGER,          //!< 13=指
    Y_JOINT_LABEL_TYPE_THUMB,           //!< 14=親指
    Y_JOINT_LABEL_TYPE_PROP_A,          //!< 15=プロップ A
    Y_JOINT_LABEL_TYPE_PROP_B,          //!< 16=プロップ B
    Y_JOINT_LABEL_TYPE_PROP_C,          //!< 17=プロップ C
    Y_JOINT_LABEL_TYPE_OTHER,           //!< 18=その他
    Y_JOINT_LABEL_TYPE_INDEX_FINGER,    //!< 19=人差指
    Y_JOINT_LABEL_TYPE_MIDDLE_FINGER,   //!< 20=中指
    Y_JOINT_LABEL_TYPE_RING_FINGER,     //!< 21=薬指
    Y_JOINT_LABEL_TYPE_PINKY_FINGER,    //!< 22=小指
    Y_JOINT_LABEL_TYPE_EXTRA_FINGER,    //!< 23=追加指
    Y_JOINT_LABEL_TYPE_BIG_TOE,         //!< 24=追加足指
    Y_JOINT_LABEL_TYPE_INDEX_TOE,       //!< 25=足人差指
    Y_JOINT_LABEL_TYPE_MIDDLE_TOE,      //!< 26=足中指
    Y_JOINT_LABEL_TYPE_RING_TOE,        //!< 27=足薬指
    Y_JOINT_LABEL_TYPE_PINKY_TOE,       //!< 28=足小指
    Y_JOINT_LABEL_TYPE_FOOT_THUMB,      //!< 29=足親指
    Y_JOINT_LABEL_TYPE_COUNT            //!< joint ラベルタイプ数です（30）。
};

//-----------------------------------------------------------------------------
//! @brief 処理結果ステータスが成功でなければ return するマクロです。
//-----------------------------------------------------------------------------
#define CheckStatus(status)     \
{                               \
    if (status != MS::kSuccess) \
        return status;          \
}

#define CheckStatusMsg(status, msg)\
{                                   \
    if (status != MS::kSuccess) {   \
        MGlobal::displayError(msg); \
        return status;              \
    }                               \
}

//-----------------------------------------------------------------------------
//! @brief Maya 2011 のリフレッシュの不具合対策関数です。
//-----------------------------------------------------------------------------
inline void RefleshIfMaya2011()
{
    #if (MAYA_API_VERSION >= 201100)
    MGlobal::executeCommand("refresh");
    #endif
}

//=============================================================================
//! @brief 処理終了待ちカーソルのクラスです。
//=============================================================================
class YWaitCursor
{
public:
    YWaitCursor()  { MGlobal::executeCommand("waitCursor -st 1"); }
    ~YWaitCursor() { MGlobal::executeCommand("if (`waitCursor -q -st`) waitCursor -st 0;"); }
};

//=============================================================================
//! @brief IK ハンドルのクラスです。
//=============================================================================
class YIkHandle
{
public:
    MDagPath m_HandlePath; //!< IK ハンドルの DAG パスです。
    MDagPath m_StartPath; //!< 開始ジョイントの DAG パスです。
    MDagPath m_EndPath; //!< 終了ジョイントの DAG パスです。
    MDagPath m_EffPath; //!< エンド エフェクタの DAG パスです。
    CDagPathArray m_ChainPaths; //!< ジョイント チェーンに含まれるジョイントの DAG パス配列です。
};

typedef std::vector<YIkHandle> YIkHandleArray;

//=============================================================================
// カラー関連の関数です。
//=============================================================================

//! @brief 2 つの MColor の誤差が許容値未満なら true を返します。
bool IsEquivalentColor(const MColor& c0, const MColor& c1, const float tolerance = nn::gfx::tool::dcc::R_SAME_TOLERANCE_F);

//=============================================================================
// Maya のオブジェクトから dcc のオブジェクトへの変換関数です。
//=============================================================================

nn::gfx::tool::dcc::RVec3 GetRVec3(const MVector& v, const bool snapToZero = false);
nn::gfx::tool::dcc::RVec3 GetRVec3(const MFloatVector& v, const bool snapToZero = false);
nn::gfx::tool::dcc::RVec3 GetRVec3(const MPoint& v, const bool snapToZero = false);

//! @brief MColor を RVec3 に変換して返します。
//!
//! @param[in] mc Maya のカラーオブジェクトです。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//! @param[in] clamps [0, 1] の範囲にクランプするなら true を指定します。
//!
//! @return RVec3 型のカラーを返します。
//!
nn::gfx::tool::dcc::RVec3 GetRVec3Color(const MColor& mc, const bool snapToZero, const bool clamps);

//! @brief MColor を RVec4 に変換して返します。
//!
//! @param[in] mc Maya のカラーオブジェクトです。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//! @param[in] clamps [0, 1] の範囲にクランプするなら true を指定します。
//!
//! @return RVec4 型のカラーを返します。
//!
nn::gfx::tool::dcc::RVec4 GetRVec4Color(const MColor& mc, const bool snapToZero, const bool clamps);

nn::gfx::tool::dcc::RRgbaColor GetRRgbaColor(const MColor& mc);

nn::gfx::tool::dcc::RMtx44 GetRMtx44(const MMatrix& mm);

//=============================================================================
// dcc のオブジェクトから Maya のオブジェクトへの変換関数です。
//=============================================================================

MVector GetMVector(const nn::gfx::tool::dcc::RVec3& v);
MFloatVector GetMFloatVector(const nn::gfx::tool::dcc::RVec3& v);
MPoint GetMPoint(const nn::gfx::tool::dcc::RVec3& v);

MStatus GetMStatus(const nn::gfx::tool::dcc::RStatus& rstatus);
MStatus GetMStatusDisplayError(nn::gfx::tool::dcc::RScene* pScene, const nn::gfx::tool::dcc::RStatus& rstatus);

//=============================================================================
// 文字列関連の関数です。
//=============================================================================

//! @brief 整数値の文字列から MIntArray を取得します。
//!        同じ値はまとめられます。
//!
//! @param[in] array 整数値配列を格納します。
//! @param[in] str 整数値の文字列です。
//!                0 以上の値を半角スペースまたはカンマで区切って並べます。
//!                1,2,3,4,5 のように連続した値は 1-5 と記述できます。
//!
//! @return 処理結果を返します。
//!
MStatus GetArrayFromStringCutSame(MIntArray& array, const char* str);

//=============================================================================
// 名前関連の関数です。
//=============================================================================

//! @brief 出力用の要素名を返します。
//!
//! @param[in] src Maya から取得した名前です。
//! @param[in] removeNamespace ネームスペースを削除するなら true です。
//!
//! @return 出力用の要素名を返します。
//!
std::string GetOutElementName(const std::string& src, const bool removeNamespace);

//=============================================================================
// ファイルパス関連の関数です。
//=============================================================================

//! @brief ファイルパスをフルパスに変換します。
//!
//! @param[in] path 元のファイルパスです。
//! @param[in] rootFolder 元のファイルパスが相対パスの場合のルートフォルダのパスです。
//!
//! @return UNIX 形式のフルパスを返します。
//!
std::string YGetFullFilePath(const std::string& path, const std::string& rootFolder);

//! @brief プラグからファイルパスを取得してフルパスで返します。
//!
//! @param[in] plug プラグです。
//! @param[in] rootFolder 元のファイルパスが相対パスの場合のルートフォルダのパスです。
//!
//! @return UNIX 形式のフルパスを返します。
//!
MString GetFilePathAttribute(const MPlug& plug, const std::string& rootFolder);

//=============================================================================
// 配列関連の関数です。
//=============================================================================

//! @brief MIntArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int YFindValueInArray(const MIntArray& array, const int value, const int startIdx = 0);

//! @brief MFloatArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int YFindValueInArray(const MFloatArray& array, const float value, const int startIdx = 0);

//! @brief MDoubleArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int YFindValueInArray(const MDoubleArray& array, const double value, const int startIdx = 0);

//! @brief MStringArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int YFindValueInArray(const MStringArray& array, const MString& value, const int startIdx = 0);

//! @brief MColorArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int YFindValueInArray(const MColorArray& array, const MColor& value, const int startIdx = 0);

//! @brief MObjectArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int FindObjectInArray(const MObjectArray& array, const MObject& obj, const int startIdx = 0);

//! @brief MDagPathArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int FindDagPathInArray(const MDagPathArray& array, const MDagPath& path, const int startIdx = 0);

//! @brief CDagPathArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
int FindDagPathInArray(const CDagPathArray& array, const MDagPath& path, const int startIdx = 0);

int YFindArrayInArray(const MIntArray& base, const MIntArray& comp);
int YFindArrayInArray(const MFloatArray& base, const MFloatArray& comp);

//! @brief MIntArray に別の MIntArray の値をすべて追加します。
//!
void YAppendArrayToArray(MIntArray& dst, const MIntArray& src);

//! @brief MFloatArray に別の MFloatArray の値をすべて追加します。
//!
void YAppendArrayToArray(MFloatArray& dst, const MFloatArray& src);

//! @brief MDoubleArray に別の MDoubleArray の値をすべて追加します。
//!
void YAppendArrayToArray(MDoubleArray& dst, const MDoubleArray& src);

int YFindAppendValue(MIntArray& array, const int value);
int YFindAppendArray(MIntArray& dst, const MIntArray& src);

void GetArrayRange(const MIntArray& array, int& valueMin, int& valueMax);
void GetArrayRange(const MFloatArray& array, float& valueMin, float& valueMax);
void GetArrayRange(const MPointArray& array, MPoint& valueMin, MPoint& valueMax);

double GetPointArrayMaxLength(const MPointArray& array);

void ScaleArray(MPointArray& array, double scale);

void GetFixedPointArray(
    MIntArray& dst,
    const MFloatArray& src,
    int mul,
    int valueMin,
    int valueMax
);

//! @brief MIntArray を値の順にソートします。
//!
//! @param[in,out] array 整数値配列です。
//! @param[in] isDescending 降順なら true、昇順なら false を指定します。
//!
void SortArray(MIntArray& array, const bool isDescending = false);

//! @brief MStringArray をアルファベット順にソートします。
//!
//! @param[in,out] array 文字列配列です。
//! @param[in] isDescending 降順なら true、昇順なら false を指定します。
//!
void SortArray(MStringArray& array, const bool isDescending = false);

bool IsZeroVectorExist(const MFloatVectorArray& array);

//=============================================================================
// 数学関連の関数です。
//=============================================================================

//! @brief 変換行列を作成します。
//!
//! @param[out] m 変換行列を格納します。
//! @param[in] s スケールです。
//! @param[in] r オイラー角の回転角度（度数）です。
//! @param[in] t 移動です。
//!
void CreateTransformMtx(MMatrix& m, const MVector& s, const MVector& r, const MVector& t);

//! @brief セグメントスケール補正を考慮した変換行列を作成します。
//!        親のスケールを打ち消す変換行列を作成します。
//!
//! @param[out] m 変換行列を格納します。
//! @param[in] s スケールです。
//! @param[in] r オイラー角の回転角度（度数）です。
//! @param[in] t 移動です。
//! @param[in] ps 親のスケールです。
//!
void CreateTransformMtxScaleCompensate(
    MMatrix& m,
    const MVector& s,
    const MVector& r,
    const MVector& t,
    const MVector& ps
);

//! @brief 変換行列からスケール・回転・移動を取得します。
//!        変換行列は座標を左から掛けるタイプになっている必要があります。
//!
//! @param[out] scale スケールを格納します。
//! @param[out] rotate 回転を格納します。単位は度数です。回転順序は XYZ で固定です。
//! @param[out] translate 移動を格納します。
//! @param[in] mtxLocal 変換行列です。
//!
void GetTransformFromMtx(MVector& scale, MVector& rotate, MVector& translate, const MMatrix& mtxLocal);

void NormalizeMtx(MFloatMatrix& mf);

//! @brief 法線に行列を掛けた後に正規化します。
//!
//! @param[in] v 変換前の法線です。
//! @param[in] m 変換行列です。
//!
//! @return 変換後の法線を返します。
//!
MFloatVector MulNormalAndMtxPostNormalize(const MFloatVector& v, const MMatrix& m);

MVector GetAverageVertexNormal(const MDagPath& meshPath, const int ivtxNode);

//=============================================================================
// プラグ関連の関数です。
//=============================================================================

//! @brief オブジェクトからプラグを取得します。
//!        MFnDependencyNode::findPlug と違って、
//!        プラグが存在しなかった場合にエラーログにエラーを出力しません。
//!
//! @param[in] depFn オブジェクトのファンクションノードです。
//! @param[in] name アトリビュート名をロングネームで指定します。
//! @param[out] pStatus 処理結果を格納します。
//!
//! @return 指定されたアトリビュートのプラグを返します。
//!
MPlug FindPlugQuiet(
    const MFnDependencyNode& depFn,
    const MString& name,
    MStatus* pStatus
);

//! @brief 指定したアトリビュート名の子プラグのインデックスを取得します。
//!
//! @param[in] plug 親プラグです。
//! @param[in] name 子プラグのアトリビュート名です。
//!
//! @return 子プラグのインデックスを返します。
//!         存在しなければ -1 を返します。
//!
int FindChildPlugIndexByName(const MPlug& plug, const MString& name);

//! @brief 指定したアトリビュート名の子プラグを取得します。
//!
//! @param[in] plug 親プラグです。
//! @param[in] name 子プラグのアトリビュート名です。
//!
//! @return 子プラグを返します。
//!         存在しなければ無効なプラグを返します。
//!
MPlug FindChildPlugByName(const MPlug& plug, const MString& name);

//=============================================================================
// ノード関連の関数です。
//=============================================================================

//! @brief 親ノードが joint ノードなら true を返します。
//!
bool HasMayaJointParent(const MDagPath& dagPath);

//! @brief 親ノードのスケールが inverseScale アトリビュートに接続されていれば true を返します。
//!
bool IsParentScaleConnectedToMayaJoint(const MDagPath& dagPath);

//! @brief nodeState アトリビュートの値が有効（通常）なら true を返します。
//!
bool IsNodeStateEffective(const MObject& obj);

//! @brief ノードが圧縮不可なら true を返します。
//!
bool IsNoCuttingNode(const MObject& obj);

//! @brief ノードが有効（出力対象）なら true を返します。
//!        次の条件をすべて満たしていれば有効です。
//!        primaryVisibility が ON
//!        template が OFF
//!        intermediateObject が OFF
//!        overrideEnabled が OFF または overrideVisibility が ON
//!        overrideEnabled が OFF または overrideShading が ON
//!        overrideEnabled が OFF または overrideDisplayType が 1(Template) でない
//!
bool IsEffectiveNode(const MObject& obj);

//! @brief ノードが可視なら true を返します。
//!
bool IsVisibleNode(const MObject& obj);

//! @brief 親ノードも含めて DAG ノードが可視なら true を返します。
//!
bool IsVisibleNodeIncludeParent(const MDagPath& dagPath);

bool IsTemplateNode(const MObject& obj);

//! @brief ノードが指定されたタイプの子ノードを持つなら true を返します。
//!
//! @param[in] dagFn DAG ノードのファンクションノードです。
//! @param[in] type 子ノードのタイプです。
//!
//! @return 指定されたタイプの子ノードを持つなら true を返します。
//!
bool NodeHasSpecificChild(const MFnDagNode& dagFn, MFn::Type type);

//! @brief ノードが指定されたファンクションタイプの子ノードを持つなら true を返します。
//!
//! @param[in] dagFn DAG ノードのファンクションノードです。
//! @param[in] type ファンクションタイプです。
//!
//! @return 指定されたファンクションタイプの子ノードを持つなら true を返します。
//!
bool NodeHasSpecificFunctionChild(const MFnDagNode& dagFn, MFn::Type type);

//! @brief ノードが指定されたタイプの親ノードを持つなら true を返します。
//!
//! @param[in] dagFn DAG ノードのファンクションノードです。
//! @param[in] type 親ノードのタイプです。
//!
//! @return 指定されたタイプの親ノードを持つなら true を返します。
//!
bool NodeHasSpecificParent(MDagPath dagPath, MFn::Type type);

//! @brief ノードが有効なトランスフォームの子ノードを持つなら true を返します。
//!
//! @param[in] dagFn DAG ノードのファンクションノードです。
//!
//! @return 有効なトランスフォームの子ノードを持つなら true を返します。
//!
bool NodeHasEffectiveTransformChild(const MFnDagNode& dagFn);

//! @brief ノードが有効な mesh ノードの子ノードを持つなら true を返します。
//!
//! @param[in] dagFn DAG ノードのファンクションノードです。
//!
//! @return 有効な mesh ノードの子ノードを持つなら true を返します。
//!
bool NodeHasEffectiveMeshChild(const MFnDagNode& dagFn);

//! @brief 中間ファイルの文字列から joint ラベルタイプを取得します。
//!
//! @param[in] str joint ラベルタイプを表す文字列です。
//!
//! @return joint ラベルタイプを返します。
//!
YJointLabelType YGetJointLabelTypeFromString(const std::string& str);

//! @brief joint ラベルタイプを表す中間ファイルの文字列を取得します。
//!
//! @param[in] jointLabelType joint ラベルタイプです。
//!
//! @return joint ラベルタイプを表す文字列を返します。
//!
std::string YGetJointLabelTypeString(const YJointLabelType jointLabelType);

//=============================================================================
// DAG パス関連の関数です。
//=============================================================================

//! @brief インスタンスを考慮して指定した親ノードの直接の子の DAG パスを取得します。
//!
//! @param[in] parentPath 親ノードの DAG パスです。
//! @param[in] childObj 子ノードのオブジェクトです。
//!
//! @return 子ノードの DAG パスを返します。
//!         子ノードがインスタンス化されている場合、parentPath を親ノードとして
//!         持つ DAG パスを返します。
//!
MDagPath GetDirectChildDagPath(const MDagPath& parentPath, const MObject& childObj);

//=============================================================================
// 選択リスト関連の関数です。
//=============================================================================

//! @brief 選択された transform ノード群以下に含まれる transform ノード群の
//!        DAG パス配列を取得します。
//!
//! @param[out] pXformPaths transform ノードの DAG パス配列（重複なし）を格納します。
//! @param[in] slist 選択リストです。
//!
void GetSelectedTransformNodesBelow(CDagPathArray* pXformPaths, const MSelectionList& slist);

//! @brief 選択された transform ノード以下の階層も含む選択リストを取得します。
//!
//! @param[in,out] pSlist 選択リストへのポインターです。下位階層も含む選択リストを格納します。
//!                       DAG パスの階層レベルが浅い順にソートされます。
//!
void GetTreeSelectionList(MSelectionList* pSlist);

//=============================================================================
// マテリアル関連の関数です。
//=============================================================================

//! @brief シェーダオブジェクトからカラーのプラグを取得します。
MPlug GetShaderColorPlug(const MObject& shaderObj);

//! @brief シェーディンググループからシェーダオブジェクトを取得します。
MObject FindShaderObject(const MObject& sgObj);

//! @brief デフォルトの UV セット名を取得します。
//!        UV セットが複数ある場合、インデックスが最小のものを取得します。
//!
//! @param[in] shapePath シェイプノードの DAG パスです。
//!
//! @return デフォルトの UV セット名を返します。
//!
MString GetDefaultUvSetName(const MDagPath& shapePath);

//! @brief テクスチャーパターンアニメーション用のファイルパスを取得します。
//!
//! @param[in] filePre フレーム拡張子より前のファイルパスです（フォルダのパスも含みます）。
//! @param[in] fileExt フレーム拡張子より後のファイルパスです。
//! @param[in] fe フレーム拡張子です。
//!
//! @return フレーム拡張子に対応したファイルが存在すればファイルパスを返します。
//!         存在しなければ空文字を返します。
//!
std::string GetFilePathForTexPatAnim(
    const std::string& filePre,
    const std::string& fileExt,
    const int fe
);

//! @brief file ノードのファイルテクスチャー名（イメージの名前）を絶対パスで取得します。
//!        文字列 <f> は現在のフレーム拡張子に置き換えます。
//!
//! @param[in] fileObj file ノードオブジェクトです。
//! @param[in] rootFolder ファイルテクスチャー名が相対パスの場合のルートフォルダーのパスです。
//!
//! @return ファイルテクスチャー名（イメージの名前）の絶対パスを返します。
//!
std::string GetFileTextureName(const MObject& fileObj, const std::string& rootFolder);

//! @brief レイヤテクスチャに接続されたテクスチャノード群
//!        （file または envCube ノード）を取得します。
//!
//! @param[out] texObjs テクスチャノード配列を格納します。
//! @param[in] leyeredObj レイヤテクスチャオブジェクトです。
//! @param[in] getsEnvCube envCube ノードも取得するなら true です。
//!
//! @return 取得したテクスチャノード群の数を返します。
//!
int GetLayeredTextureInputs(
    MObjectArray& texObjs,
    const MObject& leyeredObj,
    const bool getsEnvCube
);

//! @brief レイヤテクスチャに接続された最初の file ノードを取得します。
//!
//! @param[in] leyeredObj レイヤテクスチャオブジェクトです。
//!
//! @return file ノードを返します。
//!
MObject GetFirstLayeredFileTexture(const MObject& layeredObj);

//! @brief 環境ボールに接続された file ノードを取得します。
//!
//! @param[in] envBallObj envBall ノードです。
//!
//! @return file ノードを返します。
//!         file ノードが接続されていなければ MObject::kNullObj を返します。
//!
MObject GetEnvBallFileTexture(const MObject& envBallObj);

//! @brief 環境立方体のフェースのアトリビュート名を返します。
//!
//! @param[in] faceIdx フェースのインデックスです。
//!
//! @return アトリビュート名を返します。
//!
std::string GetEnvCubeFaceAttrName(const int faceIdx);

//! @brief 環境立方体に接続された file ノードを取得します。
//!
//! @param[in] envCubeObj envCube ノードです。
//! @param[in] faceIdx フェースのインデックスです。
//!
//! @return file ノードを返します。
//!         指定したフェースに file ノードが接続されていなければ MObject::kNullObj を返します。
//!
MObject GetEnvCubeFileTexture(const MObject& envCubeObj, const int faceIdx);

//! @brief 環境立方体の +X 面にキューブマップの DDS ファイルが接続されていれば
//!        true を返します。
//!
//! @param[in] envCubeObj envCube ノードです。
//! @param[in] projectPath Maya のプロジェクトフォルダのパスです。
//!
//! @return 環境立方体の +X 面にキューブマップの DDS ファイルが接続されていれば
//!         true を返します。
//!
bool RIsSingleEnvCube(const MObject& envCubeObj, const std::string& projectPath);

//! @brief テクスチャオブジェクトのメインの file ノードを取得します。
//!
//! @param[in] texObj テクスチャオブジェクト（file、layeredTexture、envBall、envCube）です。
//!
//! @return file ノードを返します。
//!         texObj が file ノードでなく、
//!         texObj に file ノードが接続されていなければ MObject::kNullObj を返します。
//!
MObject GetMainFileTexture(const MObject& texObj);

//! @brief プラグに接続された有効なテクスチャオブジェクトを取得します。
//!        file ノードまたは、
//!        file ノードが接続された layeredTexture ノードのみ有効とみなします。
//!
//! @param[in,out] rscene シーンです。
//! @param[in] dstPlug プラグです。
//! @param[in] checkElementFlag 要素チェック中（エラー抑止）なら true を指定します。
//!
//! @return テクスチャオブジェクトを返します。
//!         有効なテクスチャが接続されていなければ MObject::kNullObj を返します。
//!
MObject FindTextureNode(
    nn::gfx::tool::dcc::RScene& rscene,
    const MPlug& dstPlug,
    const bool checkElementFlag
);

//! @brief 法線マップのテクスチャオブジェクトを取得します。
//!        file ノードまたは、
//!        file ノードが接続された layeredTexture ノードのみ有効とみなします。
//!
//! @param[in,out] rscene シーンです。
//! @param[in] shaderObj シェーダオブジェクトです。
//! @param[in] checkElementFlag 要素チェック中（エラー抑止）なら true を指定します。
//!
//! @return テクスチャオブジェクトを返します。
//!         有効なテクスチャが接続されていなければ MObject::kNullObj を返します。
//!
MObject FindNormalMapTextureNode(
    nn::gfx::tool::dcc::RScene& rscene,
    const MObject& shaderObj,
    const bool checkElementFlag
);

//! @brief 環境マップのテクスチャオブジェクトを取得します。
//!        file、layeredTexture、envBall、envCube ノードのみ有効とみなします。
//!        layeredTexture ノードには file または envCube ノードが接続されている必要があります。
//!
//! @param[in,out] rscene シーンです。
//! @param[in] dstPlug プラグです。
//! @param[in] projectPath Maya のプロジェクトフォルダのパスです。
//! @param[in] checkElementFlag 要素チェック中（エラー抑止）なら true を指定します。
//!
//! @return テクスチャオブジェクトを返します。
//!         有効なテクスチャが接続されていなければ MObject::kNullObj を返します。
//!
MObject FindEnvMapTextureNode(
    nn::gfx::tool::dcc::RScene& rscene,
    const MPlug& dstPlug,
    const std::string& projectPath,
    const bool checkElementFlag
);

//! @brief シェイプに適用されたシェーディンググループ群と
//!        それに対応するコンポーネント群を取得します。
//!        Maya API で取得したシェーディンググループ群に重複があれば、
//!        合成したコンポーネントオブジェクトを作成して返します。
//!
//! @param[out] outSgs シェーディンググループオブジェクト配列を格納します。
//! @param[out] outComps コンポーネントオブジェクト配列を格納します。
//! @param[in] shapePath シェイプの DAG パスです。
//!
//! @return 重複したシェーディンググループのコンポーネントを合成したなら
//!         true を返します。
//!
bool GetShadingGroupsAndComponents(
    CObjectArray& outSgs,
    CObjectArray& outComps,
    const MDagPath& shapePath
);

//=============================================================================
// transform ノード関連の関数です。
//=============================================================================

//! @brief ノードのローカル変換行列が単位行列で、
//!        スケール・回転・移動に何も接続されていなければ true を返します。
//!
bool IsTransformIdentity(const MDagPath& dagPath);

//! @brief 指定したノードの親ノードのローカル変換行列が単位行列で、
//!        親ノードのスケール・回転・移動に何も接続されていなければ true を返します。
//!        親ノードがない場合は常に true を返します。
//!
bool IsParentTransformIdentity(MDagPath dagPath);

//! @brief 親ノードの変換を継承するなら true を返します。
//!
bool IsInheritsTransform(const MDagPath& dagPath);

//! @brief 取得した回転を出力のために調整する必要があるなら true を返します。
//!
bool IsRotateNeededToAdjust(
    const MDagPath& dagPath,
    const MTransformationMatrix::RotationOrder targetRotationOrder = MTransformationMatrix::kXYZ
);

//! @brief 取得した移動を出力のために調整する必要があるなら true を返します。
//!
bool IsTranslateNeededToAdjust(const MDagPath& dagPath);

//! @brief transform ノードからスケール・回転・移動を取得します。
//!
void GetTransformFromNode(
    nn::gfx::tool::dcc::RVec3& scale,
    nn::gfx::tool::dcc::RVec3& rotate,
    nn::gfx::tool::dcc::RVec3& translate,
    const MDagPath& dagPath,
    const float magnify,
    const bool useWorldTransform,
    const bool snapToZero = false
);

//! @brief ワールド回転行列を取得します。
//!        [Ro R Jo] [Ro R Jo] ...
//!
MMatrix GetWorldRotateMtx(MDagPath dagPath);

//! @brief 親ノードのワールド回転行列を取得します。
//!        親ノードがない場合は常に単位行列を返します。
//!
MMatrix GetParentWorldRotateMtx(MDagPath dagPath);

//=============================================================================
// mesh ノード関連の関数です。
//=============================================================================

//-----------------------------------------------------------------------------
// normal

//! @brief エッジ群のソフト／ハードを設定します。
//!        MFnMesh::setEdgeSmoothing は重いので、MEL コマンド polySoftEdge を使用します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] iEdges 対象エッジのインデックス配列です。
//! @param[in] isSoft ソフトエッジにするなら true、ハードエッジにするなら false です。
//!
void YSetEdgesSoftHard(const MDagPath& meshPath, MIntArray& iEdges, const bool isSoft);

//-----------------------------------------------------------------------------
// color

//! @brief カラーセットのファミリー名を取得します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] colSet カラーセット名です。
//!
//! @return カラーセットのファミリー名を返します。
//!
MString GetColorSetFamilyName(const MDagPath& meshPath, const MString& colSet);

//! @brief mesh ノードのカレントのカラーセット名を取得します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//!
//! @return カレントのカラーセット名を返します。
//!         カラーセットが無い場合は空文字を返します。
//!
MString GetCurrentColorSetName(const MDagPath& meshPath);

//! @brief カラーセットの成分数を取得します。
//!        カラーセット名の末尾が _Ni、_Nu、_Nf (N = 1, 2, 3, 4) なら
//!        N を返します。
//!        N が 5 の場合は RA の 2 成分を出力するとみなして 2 を返します。
//!        それ以外の場合、カラーセットのコンポーネントタイプが RGB なら 3、
//!        RGBA または A なら 4 を返します。
//!
//! @param[out] pIsRa RA の 2 成分を出力するなら true を格納します。nullptr なら格納しません。
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] colSet カラーセット名です。
//!
//! @return カラーセットの成分数を返します。
//!
int GetColorSetComponentCount(bool* pIsRa, const MDagPath& meshPath, const MString& colSet);

//! @brief カラーセット名から頂点カラー属性のインデックスを取得します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] colSet カラーセット名です。
//!
//! @return 頂点カラー属性のインデックスを返します。
//!         カラーセット名がマルチ頂点カラー用の名前でなければ -1 を返します。
//!
int GetVertexColorAttrIndex(const MDagPath& meshPath, const MString& colSet);

//! @brief mesh ノードから出力する頂点カラー属性のカラーセット名配列を取得します。
//!
//! @param[out] meshColSets カラーセット名配列を格納します。
//! @param[in,out] rscene シーンです。
//! @param[in] meshPath mesh ノードの DAG パスです。
//!
//! @return 処理結果を返します。
//!
MStatus GetMeshVertexColorAttrs(
    MString* meshColSets,
    nn::gfx::tool::dcc::RScene& rscene,
    const MDagPath& meshPath
);

//! @brief mesh ノードのフェース群に頂点カラーが設定されていれば true を返します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] comp フェース群のコンポーネントです。
//! @param[in] colSet カラーセット名です。
//!
//! @return 頂点カラーが設定されていれば true を返します。
//!
bool ComponentVertexColorExists(
    const MDagPath& meshPath,
    MObject& comp,
    const MString& colSet
);

//! @brief mesh ノードのフェース群に設定されている頂点カラーがすべて不透明なら true を返します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] comp フェース群のコンポーネントです。
//! @param[in] colSet カラーセット名です。
//!
//! @return 頂点カラーがすべて不透明なら true を返します。
//!
bool IsComponentVertexColorOpaque(
    const MDagPath& meshPath,
    MObject& comp,
    const MString& colSet
);

//=============================================================================
// IK 関連の関数です。
//=============================================================================

//! @brief シーン内の全 IK ハンドルのデータを取得します。
//!
//! @param[out] ikHandleArray IK ハンドル配列を格納します。
//!
//! @return 処理結果を返します。
//!
MStatus GetIkHandleData(YIkHandleArray& ikHandleArray);

//! @brief ノードをコントロールしている IK ハンドルの配列内インデックスを取得します。
//!
//! @param[in] ikHandleArray IK ハンドル配列です。
//! @param[in] path ノードの DAG パスです。
//!
//! @return ノードをコントロールしている IK ハンドルの配列内インデックスを返します。
//!         ノードが IK ハンドルにコントロールされていない場合は -1 を返します。
//!
int FindIkHandleDataFromChainJoint(
    const YIkHandleArray& ikHandleArray,
    const MDagPath& path
);

//=============================================================================
// ユーザーデータ関連の関数です。
//=============================================================================

//! @brief Maya のノードに設定されているユーザーデータを取得します。
//!
//! @param[out] datas ユーザーデータ配列を格納します。
//! @param[in,out] rscene シーンです。
//! @param[in] nodeObj ノードのオブジェクトです。
//!
//! @return 処理結果を返します。
//!
MStatus GetUserData(nn::gfx::tool::dcc::RUserDataArray& datas, nn::gfx::tool::dcc::RScene& rscene, const MObject& nodeObj);

//=============================================================================
// イメージファイル関連の関数です。
//=============================================================================

//! @brief 画像ファイルの情報を取得します。
//!
//! @param[out] imageW 画像の幅を格納します。
//! @param[out] imageH 画像の高さを格納します。
//! @param[in,out] rscene シーンです。
//! @param[in] filePath 画像ファイルのフルパスを指定します。
//! @param[in] checkElementFlag 要素チェック中（エラー抑止）なら true を指定します。
//!
//! @return 処理結果を返します。
//!
MStatus GetImageInfoForMaya(
    int& imageW,
    int& imageH,
    nn::gfx::tool::dcc::RScene& rscene,
    const std::string& filePath,
    const bool checkElementFlag
);

//! @brief 画像ファイルをリードしてデータを取得します。
//!
//! @param[out] rimg テクスチャイメージです。
//! @param[in,out] rscene シーンです。
//! @param[in] filePaths 画像ファイルのパス配列です。
//! @param[in] mergePath マージする ftx ファイルのパスです。空文字ならマージしません。
//! @param[in] dccPreset DCC プリセット名です。
//! @param[in] hint ヒント情報です。
//! @param[in] linearFlag リニア変換フラグです。
//! @param[in] usesSrgbFetch sRGB フォーマットを使用するなら true です。
//! @param[in] dimension 次元です。
//! @param[in] initialSwizzle 初期スウィズル値です。
//! @param[in] comment 編集用コメント文です。
//! @param[in] checkElementFlag 要素チェック中（エラー抑止）なら true を指定します。
//!
//! @return 処理結果を返します。
//!
MStatus ReadImageFileForMaya(
    nn::gfx::tool::dcc::RImage& rimg,
    nn::gfx::tool::dcc::RScene& rscene,
    const nn::gfx::tool::dcc::RStringArray& filePaths,
    const std::string& mergePath,
    const std::string& dccPreset,
    const std::string& hint,
    const int linearFlag,
    const bool usesSrgbFetch,
    const nn::gfx::tool::dcc::RImage::Dimension dimension,
    const int initialSwizzle,
    const std::string& comment,
    const bool checkElementFlag
);

//=============================================================================
// Maya の UI 関連の関数です。
//=============================================================================

//! @brief プラグインの UI を日本語で表示するなら true を返します。
//!
bool IsJapaneseUi();

//! @brief Maya の Output Window をアクティブにして前面に表示します。
//!
void SetMayaOutputWindowForeground();

//=============================================================================
// エラーと警告関連の関数です。
//=============================================================================

//! @brief エラーを表示します。
//!
//! @param[in,out] pScene シーンデータへのポインターです。
//! @param[in] formatJp 日本語メッセージのフォーマット文字列です。
//! @param[in] formatEn 英語メッセージのフォーマット文字列です。
//! @param[in] args メッセージの引数配列です。
//!
void YShowError(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const nn::gfx::tool::dcc::RStringArray& args
);

//! @brief エラーを表示します（引数なし）。
inline void YShowError(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn
)
{
    YShowError(pScene, formatJp, formatEn, nn::gfx::tool::dcc::RStringArray());
}

//! @brief エラーを表示します（引数 1 つ）。
inline void YShowError(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0
)
{
    nn::gfx::tool::dcc::RStringArray args;
    args.push_back(arg0);
    YShowError(pScene, formatJp, formatEn, args);
}

//! @brief エラーを表示します（引数 2 つ）。
inline void YShowError(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0,
    const std::string& arg1
)
{
    nn::gfx::tool::dcc::RStringArray args;
    args.push_back(arg0);
    args.push_back(arg1);
    YShowError(pScene, formatJp, formatEn, args);
}

//! @brief エラーを表示します（引数 3 つ）。
inline void YShowError(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0,
    const std::string& arg1,
    const std::string& arg2
)
{
    nn::gfx::tool::dcc::RStringArray args;
    args.push_back(arg0);
    args.push_back(arg1);
    args.push_back(arg2);
    YShowError(pScene, formatJp, formatEn, args);
}

//! @brief 警告を表示します。
//!
//! @param[in,out] pScene シーンデータへのポインターです。
//! @param[in] formatJp 日本語メッセージのフォーマット文字列です。
//! @param[in] formatEn 英語メッセージのフォーマット文字列です。
//! @param[in] args メッセージの引数配列です。
//!
void YShowWarning(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const nn::gfx::tool::dcc::RStringArray& args
);

//! @brief 警告を表示します（引数なし）。
inline void YShowWarning(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn
)
{
    YShowWarning(pScene, formatJp, formatEn, nn::gfx::tool::dcc::RStringArray());
}

//! @brief 警告を表示します（引数 1 つ）。
inline void YShowWarning(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0
)
{
    nn::gfx::tool::dcc::RStringArray args;
    args.push_back(arg0);
    YShowWarning(pScene, formatJp, formatEn, args);
}

//! @brief 警告を表示します（引数 2 つ）。
inline void YShowWarning(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0,
    const std::string& arg1
)
{
    nn::gfx::tool::dcc::RStringArray args;
    args.push_back(arg0);
    args.push_back(arg1);
    YShowWarning(pScene, formatJp, formatEn, args);
}

//! @brief 警告を表示します（引数 3 つ）。
inline void YShowWarning(
    nn::gfx::tool::dcc::RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const std::string& arg0,
    const std::string& arg1,
    const std::string& arg2
)
{
    nn::gfx::tool::dcc::RStringArray args;
    args.push_back(arg0);
    args.push_back(arg1);
    args.push_back(arg2);
    YShowWarning(pScene, formatJp, formatEn, args);
}

//! @brief ヘルプラインにメッセージを表示します。
//!
void YSetHelpLine(const char* format, ...);

//=============================================================================
// Nintendo Maya プラグイン関連の関数です。
//=============================================================================

//! @brief Nintendo Maya プラグインルートパスを返します。
//!        環境変数が定義されていなければ "./" を返します。
//!
std::string GetNintendoMayaRootPath();

//! @brief Nintendo Maya プラグインが使用するグラフィックスツールフォルダのパスを返します。
//!
std::string GetNintendoMayaGraphicsToolsPath();

//! @brief Nintendo Maya プラグインが使用する 3D ツールフォルダのパスを返します。
//!
std::string GetNintendoMaya3dToolsPath();

//! @brief Nintendo Maya プラグインが使用する 3D テクスチャーコンバーターのパスを返します。
//!
std::string GetNintendoMaya3dTextureConverterPath();

//! @brief Nintendo Maya プラグインのコンフィグフォルダのパスを返します。
//!
//! @param[in] getsCofigEnv 専用の環境変数による指定を取得するなら true です。
//!
std::string GetNintendoMayaConfigFolderPath(const bool getsCofigEnv);

//! @brief Nintendo Maya プラグインのプリセットフォルダのパスを返します。
//!
std::string GetNintendoMayaPresetsFolderPath();

