﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// ChannelInput
// YAnimCurve YAnimKey

//=============================================================================
// include
//=============================================================================
#include "DccAnimation.h"
#include "Export.h"

//=============================================================================
//! @brief Maya のチャンネル入力情報のクラスです。
//=============================================================================
class ChannelInput
{
public:
    bool m_CurveFlag; //!< アニメーションカーブが設定されていれば true です。
    MObject m_CurveObj; //!< アニメーションカーブオブジェクト（animCurveT? ノード）です。
    bool m_CurveWeighted; //!< アニメーションカーブが接線ウェイトを使用するなら true です。
    bool m_DrivenKey; //!< ドリブンキーが設定されていれば true です。
    bool m_Expression; //!< エクスプレッションが設定されていれば true です。
    bool m_Character; //!< キャラクタが設定されていれば true です。
    bool m_AnimLayer; //!< アニメーションレイヤが設定されていれば true です。
    bool m_TimeEditor; //!< タイムエディタが設定されていれば true です。
    bool m_Constraint; //!< コンストレインが設定されていれば true です。

public:
    //! コンストラクタです（引数なし）。
    ChannelInput()
    : m_CurveFlag(false),
      m_CurveWeighted(false),
      m_DrivenKey(false),
      m_Expression(false),
      m_Character(false),
      m_AnimLayer(false),
      m_TimeEditor(false),
      m_Constraint(false)
    {
    }

    //! @brief コンストラクタです（プラグを指定）。
    //!
    //! @param[in] plug チャンネル入力情報を取得するプラグです。
    //!
    explicit ChannelInput(const MPlug& plug)
    {
        GetInput(plug);
    }

    //! プラグからチャンネル入力情報を取得します。
    //!
    //! @param[in] plug プラグです。
    //!
    void GetInput(const MPlug& plug);

    //! チャンネル入力が有効（アニメーション出力対象）なら true を返します。
    bool IsEffective() const
    {
        return (
            m_CurveFlag  ||
            m_DrivenKey  ||
            m_Expression ||
            m_Character  ||
            m_AnimLayer  ||
            m_TimeEditor ||
            m_Constraint);
    }
};

//! @brief チャンネル入力配列の定義です。
typedef std::vector<ChannelInput> ChannelInputArray;

//=============================================================================
//! @brief Maya 用のアニメーションキーのクラスです。
//=============================================================================
class YAnimKey
{
public:
    float m_Frame; //!< フレームです。
    float m_Value; //!< 値です。
    nn::gfx::tool::dcc::RAnimKey::TangentType m_InType; //!< 左側の接線タイプです。
    nn::gfx::tool::dcc::RAnimKey::TangentType m_OtType; //!< 右側の接線タイプです。
    float m_InSlope; //!< 左側のスロープ値です。
    float m_OtSlope; //!< 右側のスロープ値です。

public:
    //! コンストラクタです（引数なし）。
    YAnimKey()
    : m_Frame(0.0f),
      m_Value(0.0f),
      m_InType(nn::gfx::tool::dcc::RAnimKey::HERMITE),
      m_OtType(nn::gfx::tool::dcc::RAnimKey::HERMITE),
      m_InSlope(0.0f),
      m_OtSlope(0.0f)
    {
    }

    //! コンストラクタです（フレーム、値、左右共通のスロープ値を指定）。
    YAnimKey(const float frame, const float value, const float slope)
    : m_Frame(frame),
      m_Value(value),
      m_InType(nn::gfx::tool::dcc::RAnimKey::HERMITE),
      m_OtType(nn::gfx::tool::dcc::RAnimKey::HERMITE),
      m_InSlope(slope),
      m_OtSlope(slope)
    {
    }

    //! コンストラクタです（フレーム、値、左右それぞれのスロープ値を指定）。
    YAnimKey(
        const float frame,
        const float value,
        const float inSlope,
        const float otSlope)
    : m_Frame(frame),
      m_Value(value),
      m_InType(nn::gfx::tool::dcc::RAnimKey::HERMITE),
      m_OtType(nn::gfx::tool::dcc::RAnimKey::HERMITE),
      m_InSlope(inSlope),
      m_OtSlope(otSlope)
    {
    }

    //! 0 に近い値を 0 にします。
    void SnapToZero()
    {
        m_Frame   = nn::gfx::tool::dcc::RSnapToZero(m_Frame);
        m_Value   = nn::gfx::tool::dcc::RSnapToZero(m_Value);
        m_InSlope = nn::gfx::tool::dcc::RSnapToZero(m_InSlope);
        m_OtSlope = nn::gfx::tool::dcc::RSnapToZero(m_OtSlope);
    }

    //! 出力ストリームにデバッグ出力します。
    friend std::ostream& operator<<(std::ostream& os, const YAnimKey& key)
    {
        // debug print
        std::streamsize precBak = os.precision(3);
        os << key.m_Frame << " " << key.m_Value << " " << key.m_InSlope << " " << key.m_OtSlope << R_ENDL;
        os.precision(precBak);
        return os;
    }
};

//! @brief Maya 用のアニメーションキー配列の定義です。
typedef std::vector<YAnimKey> YAnimKeyArray;

//=============================================================================
//! @brief Maya 用のアニメーションカーブのクラスです。
//=============================================================================
class YAnimCurve : public nn::gfx::tool::dcc::RAnimCurve
{
public:
    //! コンストラクタです。
    YAnimCurve()
    {
    }

    //! @brief Maya からアニメーションキー配列を取得します。
    //!        呼ぶ前に m_LoopFlag, m_AngleFlag をセットしておく必要があります。
    //!
    //! @param[in] curveObj アニメーションカーブオブジェクトです。
    //! @param[in] valueScale 取得した値に乗算するスケールです。
    //! @param[in] valueOfs 取得した値に加算するオフセットです。
    //!
    void GetKeys(
        const MObject& curveObj,
        const float valueScale,
        const float valueOfs,
        const YExpOpt& yopt
    );

    //! @brief Maya からステップ補間用のアニメーションキー配列を取得します。
    //!
    //! @param[in] curveObj アニメーションカーブオブジェクトです。
    //!
    void GetStepKeys(const MObject& curveObj, const YExpOpt& yopt);
};

//! @brief Maya 用のアニメーションカーブ配列の定義です。
typedef std::vector<YAnimCurve> YAnimCurveArray;

//=============================================================================
// アニメーションカーブ関連の関数です。
//=============================================================================

//! @brief アニメーションカーブの回転補間がクォータニオン補間なら true を返します。
//!
//! @param[in] curveObj アニメーションカーブオブジェクトです。
//!
//! @return 回転補間がクォータニオン補間なら true を返します。
//!
bool IsQuaternionAnimCurve(const MObject& curveObj);

//! @brief アニメーションカーブが静的なら true を返します。
//!
//! @param[in] curveObj アニメーションカーブオブジェクトです。
//!
//! @return 静的なら true を返します。
//!
bool IsStaticAnimCurve(const MObject& curveObj);

//! @brief アニメーションカーブのキーがすべてステップ補間なら true を返します。
//!
//! @param[in] curveObj アニメーションカーブオブジェクトです。
//!
//! @return キーがすべてステップ補間なら true を返します。
//!
bool IsStepAnimCurve(const MObject& curveObj);

//! @brief 整数サブフレームから浮動小数点数フレームを取得します。
//!
//! @param[in] subFrame 整数サブフレームです。
//! @param[in] pParam YExpOpt オブジェクトへのポインタです。
//!
//! @return 浮動小数点数フレームを返します。
//!
float GetFloatFrameFromSubFrame(const int subFrame, const void* pParam);

//! @brief Maya のアニメーションカーブを評価した値が有効なら true を返します。
//!        アニメーションカーブを評価した値がベイクしたデータと等しく、
//!        接線タイプとインフィニティがランタイムで再生可能なら有効とみなします。
//!
//! @param[in] curveObj アニメーションカーブオブジェクトです。
//! @param[in] fullValues ベイクしたデータ（全サブフレームにおける値の配列）です。
//! @param[in] valueScale アニメーションカーブを評価した値に乗算するスケールです。
//! @param[in] valueOfs アニメーションカーブを評価した値に加算するオフセットです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return Maya のアニメーションカーブを評価した値が有効なら true を返します。
//!         有効な場合、アニメーションカーブから取得したキーを出力できます。
//!
bool IsValidAnimCurveValue(
    const MObject& curveObj,
    const nn::gfx::tool::dcc::RFloatArray& fullValues,
    const float valueScale,
    const float valueOfs,
    const YExpOpt& yopt
);

//! @brief int 型のアトリビュートの値が全サブフレームで一定なら true を返します。
//!
//! @param[in] plug プラグです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return int 型のアトリビュートが一定なら true を返します。
//!
bool IsAttributeConstantInt(const MPlug& plug, const YExpOpt& yopt);

//! @brief アニメーションレイヤのブレンドノードなら true を返します。
//!
//! @param[in] obj オブジェクトです。
//!
//! @return アニメーションレイヤのブレンドノードなら true を返します。
//!
bool IsAnimLayerBlendNode(const MObject& obj);

//! @brief アニメーションレイヤが有効なら true を返します。
//!        ウェイトが 0 でないレイヤにキーがあれば有効とみなします。
//!        ミュートされたレイヤ、ソロの対象外のレイヤはウェイト 0 となります。
//!
//! @param[in] plug 対象アトリビュートのプラグです。
//! @param[in] blendObj アニメーションレイヤブレンドオブジェクトです。
//!
//! @return アニメーションレイヤが有効なら true を返します。
//!
bool IsEffectiveAnimLayer(const MPlug& plug, const MObject& blendObj);

