﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityCommon.h"

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

/******************************************************************************
    classes
******************************************************************************/
/**
ユーザーへ通知されるメッセージをハンドルするクラス
\par 解説：
　中間モジュールからのメッセージ出力処理をこのインターフェイスへ集約し、
各DCCツールごとのメッセージ出力機構へ処理を委譲するためのクラス。
DCCツールごとのコードと結合度を低くしたいため処理の委譲をコールバックで実装しています。
 */
class RLogger
{
public:
    /**
    メッセージのタイプを表す列挙子
    - kError	エラー
    - kWarning	警告
    - kInfo		一般情報
    */
    enum RMessageType
    {
        kError = 0,
        kWarning,
        kInfo,
        kDump
    };

    /**
    メッセージの種類を表す列挙子
    */
    enum RMessageDetail
    {
        kLogERR_FindCS,
        kLogERR_FindOptimizer,
        kLogERR_GetTextureSize,
        kLogERR_OpenFile,
        kLogERR_StartCS,
        kLogERR_HoledPolygon,
        kLogERR_InfluenceObj,
        kLogERR_MultipleBindPoses,
        kLogERR_NoEffectiveNode,
        kLogERR_NoEffectivePolygon,
        kLogERR_NoOutputFile,
        kLogERR_NoShadingGroup,
        kLogERR_NodeForOutputFileName,
        kLogERR_OutputFileName,
        kLogERR_OutputFolder,
        kLogERR_SameUserAttrib,
        kLogERR_TextureImageName,
        kLogERR_TextureSize,
        kLogERR_TGA_File,
        kLogERR_NumberOfTexture,
        kLogERR_UV_Linking,
        kLogERR_VertexNumberIsDifferent,
        kLogERR_WrongWeightedVertices,
        kLogERR_PolygonBoneSizeIsOver,
        kLogERR_MaxReservedUniformRegistersIsWrong,
        kLogERR_InternalError,
        kLogERR_PathOfMergeIsWrong,
        kLogERR_TheNumberOfVerticesExceedsTheLimitation,

        kLogERR_CantExportFtxFile,
        kLogERR_TextureConverterCantBeFound,
        kLogERR_FtxFileToMergeIsNotFound,
        kLogERR_Optimization,

        kLogERR_MergeAnimationFolderIsWrong,
        kLogERR_MergeAnimationNameIsWrong,
        kLogERR_AnimationFileToMergeIsNotFound,

        kLogERR_SamplerNameIsWrong,
        kLogERR_SamplerHintIsWrong,
        kLogERR_SamplerNameIsDuplicate,
        kLogERR_SamplerHintIsDuplicate,

        kLogERR_CommentIsWrong,

        kLogERR_UvSetSameHintExists,

        kLogERR_END,
        ///////////////////////////////////////////////
        kLogWRN_NotConnectedToFrame,
        kLogWRN_ExportNURBS,
        kLogWRN_ExportSubdivision,
        kLogWRN_GetUV,
        kLogWRN_DeformerIsNotEffective,
        kLogWRN_FileTexIsNotConnected,
        kLogWRN_FrameOffsetIsNotZero,
        kLogWRN_InheritsTransformIsOff,
        kLogWRN_MaterialNameIsChanged,
        kLogWRN_NoShader,
        kLogWRN_NodeNameIsChanged,
        kLogWRN_FileNameIsChanged,
        kLogWRN_TexIsIgnored,
        kLogWRN_ProjectionMappingIsNotSupported,
        kLogWRN_PSD_File,
        kLogWRN_NonSupportedShader,
        kLogWRN_ShearIsNotZero,
        kLogWRN_StencilMappingIsNotSupported,
        kLogWRN_TexFileForTexAnimIsNotFound,
        kLogWRN_TexNameIsChanged,
        kLogWRN_MultipleShape,
        kLogWRN_MultipleTexWithSameName,
        kLogWRN_ZeroNormalExist,
        kLogWRN_TexFileIsNotFound,
        kLogWRN_NonSupportedTex,
        kLogWRN_NonSupportedTexFormat,
        kLogWRN_NumberOfTexture,
        kLogWRN_MapChannelDoesNotExist,
        kLogWRN_OptimizationFailed,
        kLogWRN_GeneratingTangentFailed,
        kLogWRN_BillboardAndConvertToModel,
        kLogWRN_NonBoneBetweenBones,
        kLogWRN_Optimization,
        kLogWRN_ZeroTangentExists,
        kLogWRN_AnimationFileToMergeIsNotFound,
        kLogWRN_FtxFileToMergeIsNotFound,
        kLogWRN_MorpherMustBeUnderSkinModifier,
        kLogWRN_MorpherTargetTopologyIsDifferentFromOriginalMesh,
        kLogWRN_UvSetNameIsWrong,
        kLogWRN_END,

        kLogINF_END,
    };


    /**
    ロギングコールバック
    */
    typedef RStatus (*LogMessageCallback)( const std::string& message, RLogger::RMessageType type );
public:
    /**
    デストラクタ
    */
    ~RLogger(void);

    /**
    　メッセージロギングメソッド
    \par 解説：
    　規定のメッセージをユーザーへ通知します。
    通知方法は各DCCツールにより異なり、コールバックの形で実装されます。
    このメソッドは規定のメッセージを登録されたコールバックへ処理を委譲するだけです。
    */
    RStatus logMessagebyID(RMessageDetail id, std::string option = "");
    static RStatus LogMessagebyID(RMessageDetail id, std::string option = "");

    /**
    　メッセージロギングメソッド
    \par 解説：
    　メッセージをユーザーへ通知します。
    通知方法は各DCCツールにより異なり、コールバックの形で実装されます。
    このメソッドは登録されたコールバックへ処理を委譲するだけです。
    */
    RStatus logMessage( const std::string&	message, RMessageType type = kError );
    static RStatus LogMessage( const std::string&	message, RMessageType type = kError );
    static RStatus LogMessageFormat( const char* format, ... );
    /**
    　ロギングコールバック設定
    \par 解説：
    　LogMessageから呼び出されるコールバックを設定します。
    コールバックは下記の形式です。
    typedef nn::gfx::tool::dcc::RStatus (*LogMessageCallback)( std::string message, RLogger::RMessageType type );
    */
    void setLoggingCallback( LogMessageCallback	callback ){ mCallback = callback; }
    static void SetLoggingCallback( LogMessageCallback	callback ){ get()->setLoggingCallback(callback); }

    /**
    シングルトンアクセッサ
    \par 解説：
    インスタンス取得メソッド。
    インスタンスが作成されていない状態で呼ばれると
    内部でインスタンスを作成します。
    */
    static RLogger* get(void);

    /**
    Warning数をクリア
    */
    void clearWarningCount(){ mWarningCount = 0; }
    static void ClearWarningCount(){ get()->clearWarningCount(); }

    /**
    Warning数を取得
    */
    int getWarningCount() const { return mWarningCount; }
    static int GetWarningCount() { return get()->getWarningCount(); }

private:
    /**
    コンストラクタ
    \par 解説：
    直接実体化されないようにデフォルトコンストラクタは
    直接アクセスを禁止しておく。
    */
    RLogger(void) : mCallback(nullptr), mWarningCount(0) {};

    /**
    メッセージ設定時に呼び出されるコールバック
    \par 解説：
    　メッセージを設定されると呼び出されます。
    */
    LogMessageCallback	mCallback;

    /**
    自分自身のインスタンス。
    \par 解説：
    Singletonインスタンス。
    */
    static RLogger*	mInstance;

    /**
    Warning数
    */
    int mWarningCount;

};



/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
