﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityCommon.h"
#include "DccUtilityMaterial.h"

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

/*!--------------------------------------------------------------------------*
    @brief ライト情報を出力するためのクラスです。
    ライト情報の中間ファイルへの出力には RSceneLights を使用します。
 *---------------------------------------------------------------------------*/
class RLight
{
public:
    //! ライトの種類です。
    enum LightKind
    {
        Ambient,		//!< アンビエントライトです。
        Directional,	//!< ディレクショナルライトです。
        Point,			//!< ポイントライトです。
        Spot			//!< スポットライトです。
    };

public:
    // シーンの拡大率
    static float mMagnify;
    //! シーン全体の拡大・縮小率を設定する
    static void SetMagnify(float magnify)
    {
        mMagnify = magnify;
    }

    //! コンストラクタです。
    RLight();

    //!	@brief ライトアニメーションの出力準備をします。
    //!	アニメーションカーブデータを確認し、出力する必要のあるデータを判断します。
    //!	OutLightAnim を呼ぶ前に一度呼び出す必要があります。
    //!	@param[in] loop ループ再生フラグ
    //!	@param[in] toleranceTrans 移動アニメーションの許容誤差
    //!	@param[in] toleranceRotate 回転アニメーションの許容誤差
    //!	@param[in] toleranceScale スケールアニメーションの許容誤差
    //!	@param[in] toleranceColor カラーアニメーションの許容誤差
    void PrepareAnimations( const nn::gfx::tool::dcc::RExpOpt& rOpt );

    //! @brief <light_anim> 要素を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount <light_anim> 要素のインデントに必要なタブの数です。
    //! @param[in] index <light_anim_array> 内でのインデックスです。
    //!
    void OutAnim(
        std::ostream& os,
        nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
        const int tabCount,
        const int index,
        const nn::gfx::tool::dcc::RExpOpt& rOpt) const;

    //! 名前を指定します。
    void SetName(const char* name) { mName = name; }
    //! 名前を取得します。
    const char* GetName() const { return mName.c_str(); }

    //! ライトタイプを指定します。
    void SetLightKind(LightKind kind) { mLightKind = kind; }

    //! ライトタイプを取得します(FDC)。
    u_int GetLightType() const
    {
        u_int ltype;
        switch(mLightKind)
        {
        case Spot:
            ltype = nn::gfx::tool::dcc::RLight::SPOT;	break;
        case Directional:
            ltype = nn::gfx::tool::dcc::RLight::DIRECTIONAL;	break;
        case Point:
            ltype = nn::gfx::tool::dcc::RLight::POINT;	break;
        case Ambient:
        default:
            ltype = nn::gfx::tool::dcc::RLight::AMBIENT; break;
        }
        return ltype;
    }

    static bool IsRotateAttr(const int iattr)
    {
        return (nn::gfx::tool::dcc::RLight::ANGLE_ATTN_START == iattr || nn::gfx::tool::dcc::RLight::ANGLE_ATTN_END == iattr);
    }

    // アニメーション
    void SetVisibleAnim(bool visible)
    {
        mAnims[nn::gfx::tool::dcc::RLight::ENABLE].m_FullValues.push_back(visible? 1.0f: 0.0f);
    }

    void SetTransAnim(nn::gfx::tool::dcc::RVec3 pos)
    {
        pos *= mMagnify;
        mAnims[nn::gfx::tool::dcc::RLight::POSITION_X].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.x));
        mAnims[nn::gfx::tool::dcc::RLight::POSITION_Y].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.y));
        mAnims[nn::gfx::tool::dcc::RLight::POSITION_Z].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.z));
    }

    void SetDirectionAnim(nn::gfx::tool::dcc::RVec3 dir)
    {
        mAnims[nn::gfx::tool::dcc::RLight::DIRECTION_X].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(dir.x));
        mAnims[nn::gfx::tool::dcc::RLight::DIRECTION_Y].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(dir.y));
        mAnims[nn::gfx::tool::dcc::RLight::DIRECTION_Z].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(dir.z));
    }

    void SetAimAnim(nn::gfx::tool::dcc::RVec3 aim)
    {
        mAnims[nn::gfx::tool::dcc::RLight::AIM_X].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(aim.x));
        mAnims[nn::gfx::tool::dcc::RLight::AIM_Y].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(aim.y));
        mAnims[nn::gfx::tool::dcc::RLight::AIM_Z].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(aim.z));
    }

    void SetColor0Anim(nn::gfx::tool::dcc::RVec3 color)
    {
        mAnims[nn::gfx::tool::dcc::RLight::COLOR0_R].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(color.x));
        mAnims[nn::gfx::tool::dcc::RLight::COLOR0_G].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(color.y));
        mAnims[nn::gfx::tool::dcc::RLight::COLOR0_B].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(color.z));
    }

    void SetColor1Anim(nn::gfx::tool::dcc::RVec3 color)
    {
        mAnims[nn::gfx::tool::dcc::RLight::COLOR1_R].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(color.x));
        mAnims[nn::gfx::tool::dcc::RLight::COLOR1_G].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(color.y));
        mAnims[nn::gfx::tool::dcc::RLight::COLOR1_B].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(color.z));
    }

    void SetDistAttenAnim(float start, float end)
    {
        mAnims[nn::gfx::tool::dcc::RLight::DIST_ATTN_START].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(start));
        mAnims[nn::gfx::tool::dcc::RLight::DIST_ATTN_END].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(end));
    }

    void SetAngleAttenAnim(float start, float end)
    {
        mAnims[nn::gfx::tool::dcc::RLight::ANGLE_ATTN_START].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(start));
        mAnims[nn::gfx::tool::dcc::RLight::ANGLE_ATTN_END].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(end));
    }

    void EnableDistAttenAnim(bool flag)
    {
        mIsDistAttnAnimEnable = flag;
    }

    /**
    アニメーションカーブインスタンスを取得する。
    */
    nn::gfx::tool::dcc::RAnimCurve* GetAnimCurve( const unsigned int attr )
    {
        return &mAnims[attr];
    }

public:

    std::string mName;
    LightKind mLightKind;

    nn::gfx::tool::dcc::RAnimCurve mAnims[nn::gfx::tool::dcc::RLight::PARAM_COUNT];
    bool mIsDistAttnAnimEnable;
};


/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
