﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NintendoExportMax.h"

namespace Dcc = nn::gfx::tool::dcc;
using namespace nn::gfx::tool::dcc::utility;

////////////////////////////////////////////////////////////////////////////
//
/*
 * PBlock2からコントロールを取得
 * 2012からメソッド名が変わったので、この関数で取得する。
 */
static Control* getControlFromPBlockByIndex(IParamBlock2* pblock, int index)
{
    if(!pblock)
    {
        return NULL;
    }

#if (MAX_VERSION_MAJOR >= 14) // max2012以降
    return pblock->GetControllerByIndex(index);
#else
    return pblock->GetController(index);
#endif

}

// サブアニメーションの列挙（テスト用）
static void showSubAnims( Animatable* anim, int tc = 0)
{
    MSTR cname, sname;
    anim->GetClassName(cname);
    int numSubAnim = anim->NumSubs();
    DebugPrint(_M_TEXT("%sClass(%s) NumSub:%d\n"), Dcc::RTab(tc), cname.data(), numSubAnim);
    if(numSubAnim > 0)
    {
        for (int i = 0; i < anim->NumSubs(); ++i)
        {
            sname = anim->SubAnimName(i);
            DebugPrint(_M_TEXT("%sSubAnim(%d) %s\n"), Dcc::RTab(tc), i, sname.data());
            showSubAnims(anim->SubAnim(i), tc + 1);
        }
    }
    //throw std::exception("The method or operation is not implemented.");
}

// ノードリストに子ノードを再帰的に追加
static void AddChildrenRecursive(INodeTab& selNodes, INode* node)
{
    if(!node) return;

    selNodes.AppendNode(node);

    INode* child;
    for (int idx=0; idx < node->NumberOfChildren(); idx++)
    {
        child = node->GetChildNode(idx);
        if(child)
        {
            selNodes.AppendNode(child);
            AddChildrenRecursive(selNodes, child);
        }
    }
}

// IFLファイルかどうか
static bool isIFL(Texmap* tex)
{
    TCHAR path[2048], name[2048], ext[2048];

    if(tex && tex->ClassID() == Class_ID(BMTEX_CLASS_ID,0))
    {
        BitmapTex* bmap = static_cast<BitmapTex*>(tex);
        BMMSplitFilename(bmap->GetMapName(), path, name, ext);
#ifndef UNICODE
        if(_strcmpi(ext, ".ifl") == 0)
        {
            return true;
        }
#else
        if(_wcsicmp(ext, L".ifl") == 0)
        {
            return true;
        }
#endif

    }

    return false;
}


// マテリアル内にテクスチャパターンアニメーションがあるかどうか
static bool HasTexPatAnim(Mtl * mtl)
{

    if(!mtl) return false;

    if(mtl->IsMultiMtl())
    {
        int numSubMtl = mtl->NumSubMtls();
        for(int i =0; i < numSubMtl; i++)
        {
            Mtl* subMtl = mtl->GetSubMtl(i);
            if(HasTexPatAnim(subMtl))
            {
                return true;
            }
        }

        return false;
    }
    else
    {
        // マテリアルの全てのテクスチャを列挙する
        for(int i = 0; i < mtl->NumSubTexmaps(); i++)
        {
            if(!mtl->SubTexmapOn(i)) continue;
            Texmap* tex = mtl->GetSubTexmap(i);
            if(tex->ClassID() == Class_ID(BMTEX_CLASS_ID,0) ) // 通常のビットマップ
            {
                if(isIFL(tex))
                {
                    return true;
                }
            }
            else if(tex->ClassID() == Class_ID(COMPOSITE_CLASS_ID,0) ) // 合成マップ
            {
                // max2009以降では、1つのレイヤーにカラーテクスチャとマスクテクスチャが含まれる。
                int numSubmap = tex->NumSubTexmaps();
                for(int j = 0; j < numSubmap; j += 2)
                {
                    Texmap* stex = tex->GetSubTexmap(j);
                    if(isIFL(stex))
                    {
                        return true;
                    }
                }
            }
        }
    }

    return false;

}

// 異方性反射シェーダのクラスID（sdkのヘッダにないため）
#define ANISOSHADER_CLASS_ID     0x2857f460

// マテリアル内にカラーアニメーションがあるかどうか
static bool HasMtlColorAnim(Mtl* mtl)
{
    if(!mtl) return false;

    if(mtl->IsMultiMtl())
    {
        // マルチサブオブジェクトの場合
        int numSubMtl = mtl->NumSubMtls();
        for(int i =0; i < numSubMtl; i++)
        {
            Mtl* subMtl = mtl->GetSubMtl(i);
            if(HasMtlColorAnim(subMtl))
            {
                return true;
            }
        }

        return false;
    }
    else
    {
        // 標準マテリアルのみサポート
        if(mtl->ClassID() == Class_ID(DMTL_CLASS_ID,0))
        {
            StdMat2* stdmtl = static_cast<StdMat2*>(mtl);

            // マテリアルのパラメータブロックから透明度のコントローラを取り出し、
            // キーが有れば強制的にアニメーション出力するフラグを立てる
            IParamBlock2* pbext = mtl->GetParamBlock(1);
            if(pbext)
            {
                Control* c;
                c = getControlFromPBlockByIndex(pbext, 1);
                if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                {
                    return true;
                }
            }

            // シェーダのパラメータブロックから、各カラーのキー情報を取得
            Shader* shader = stdmtl->GetShader();
            if(shader)
            {
                // シェーダに応じてディフューズ、スペキュラ、エミット量、
                // エミットカラーのパラメータIDをセット
                int pid_amb, pid_dif, pid_spec, pid_emit, pid_emitcol;
                if(shader->ClassID() == Class_ID(BLINNClassID,0) ||
                    shader->ClassID() == Class_ID(PHONGClassID,0)
                    )
                {
                    pid_amb = 0;
                    pid_dif = 1;
                    pid_spec = 2;
                    pid_emit = 7;
                    pid_emitcol = 8;
                }
                else if(shader->ClassID() == Class_ID(ANISOSHADER_CLASS_ID,0))
                {
                    pid_amb = 0;
                    pid_dif = 1;
                    pid_spec = 2;
                    pid_emit = 6;
                    pid_emitcol = 3;
                }
                else
                {
                    return false;
                }


                // シェーダのpblockからコントローラを取得
                IParamBlock2* pblock = shader->GetParamBlock(0);
                if(pblock)
                {
                    Control* c;
                    c = getControlFromPBlockByIndex(pblock, pid_amb);
                    if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                    {
                        return true;
                    }
                    c = getControlFromPBlockByIndex(pblock, pid_dif);
                    if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                    {
                        return true;
                    }
                    c = getControlFromPBlockByIndex(pblock, pid_spec);
                    if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                    {
                        return true;
                    }

                    if(mtl->GetSelfIllumColorOn()) //自己照明カラー使用時
                    {
                        c = getControlFromPBlockByIndex(pblock, pid_emitcol);
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            return true;
                        }
                    }
                    else
                    {
                        c = getControlFromPBlockByIndex(pblock, pid_emit);
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            return true;
                        }
                    }
                }
            }
        }

    }
    return false;
}

// ビットマップにSRTアニメーションが含まれるかどうか
static bool HasSRTAnim(Texmap* tex)
{

    if(tex && tex->ClassID() == Class_ID(BMTEX_CLASS_ID,0) ) // 通常のビットマップ
    {
        BitmapTex* bmap = static_cast<BitmapTex*>(tex);
        StdUVGen* uv = bmap->GetUVGen();
        if(uv)
        {
            // 直接UV座標のコントローラが取得できないので、サブアニメーション経由で取得
            int numsub = uv->NumSubs();
            if(numsub > 0)
            {
                MSTR snmae = uv->SubAnimName(0);
                Animatable* anim = uv->SubAnim(0);

                int numasub = anim->NumSubs();
                enum
                {
                    texOffU = 0,
                    texOffV = 1,
                    texSclU = 2,
                    texSclV = 3,
                    texRot  = 6,
                };

                // 間違って変なアニメーションにアクセスしないことを期待…
                if(numasub >= 13)
                {
                    Animatable* sa;

                    sa = anim->SubAnim(texSclU);
                    if(sa->NumKeys() > 0 || sa->IsAnimated())
                    {
                        return true;
                    }

                    sa = anim->SubAnim(texSclV);
                    if(sa->NumKeys() > 0 || sa->IsAnimated())
                    {
                        return true;
                    }

                    sa = anim->SubAnim(texRot);
                    if(sa->NumKeys() > 0 || sa->IsAnimated())
                    {
                        return true;
                    }

                    sa = anim->SubAnim(texOffU);
                    if(sa->NumKeys() > 0 || sa->IsAnimated())
                    {
                        return true;
                    }

                    sa = anim->SubAnim(texOffV);
                    if(sa->NumKeys() > 0 || sa->IsAnimated())
                    {
                        return true;
                    }
                }
            }
        }
    }

    return false;
}

// マテリアル内にテクスチャSRTアニメーションがあるかどうか
static bool HasTexSRTAnim( Mtl* mtl )
{
    if(!mtl) return false;

    if(mtl->IsMultiMtl())
    {
        // マルチサブオブジェクトの場合
        int numSubMtl = mtl->NumSubMtls();
        for(int i =0; i < numSubMtl; i++)
        {
            Mtl* subMtl = mtl->GetSubMtl(i);
            if(HasTexSRTAnim(subMtl))
            {
                return true;
            }
        }

        return false;
    }
    else
    {
        // マテリアルの全てのテクスチャを列挙する
        for(int i = 0; i < mtl->NumSubTexmaps(); i++)
        {
            if(!mtl->SubTexmapOn(i)) continue;
            Texmap* tex = mtl->GetSubTexmap(i);
            if(tex->ClassID() == Class_ID(BMTEX_CLASS_ID,0) ) // 通常のビットマップ
            {
                if(HasSRTAnim(tex))
                {
                    return true;
                }
            }
            else if(tex->ClassID() == Class_ID(COMPOSITE_CLASS_ID,0) ) // 合成マップ
            {
                // max2009以降では、1つのレイヤーにカラーテクスチャとマスクテクスチャが含まれる。
                int numSubmap = tex->NumSubTexmaps();
                for(int j = 0; j < numSubmap; j += 2)
                {
                    Texmap* stex = tex->GetSubTexmap(j);
                    if(HasSRTAnim(stex))
                    {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}
// ノードのSRTにアニメーションがあるかどうか
// ローカル座標のみ
static bool HasNodeSRTAnim(INode* node)
{
    if(!node) return false;
    Control* c = node->GetTMController();
    if(c)
    {
        if(c->IsAnimated() || c->NumKeys() > 0)
        {
            return true;
        }
        else
        {
            // subアニメ
            int numSub = c->NumSubs();
            for(int subIdx = 0; subIdx < numSub; subIdx++)
            {
                Animatable* anim = c->SubAnim(subIdx);
                if(anim)
                {
                    if(anim->IsAnimated() || anim->NumKeys() > 0)
                    {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}

// ノードのSRTにアニメーションがあるかどうか
// グローバル座標での状態を調べるため、親をたどって調べる
static bool HasNodeGlobalSRTAnim(INode* node)
{
    if(!node) return false;
    if(HasNodeSRTAnim(node))
    {
        return true;
    }
    else
    {
        INode* parent = node->GetParentNode();
        if(parent)
        {
            return HasNodeGlobalSRTAnim(parent);
        }
    }
    return false;
}

//
enum CAM_PARAM
{
    CAM_PB_FOV = 0,
    CAM_PB_TDIST,
    CAM_PB_HITHER,
    CAM_PB_YON,
    CAM_PB_NRANGE,
    CAM_PB_FRANGE,
};



//MCHAR* SearchAnim( BOOL selected )
CStr SearchAnim( BOOL selected )
{
    Interface11* ip11 = GetCOREInterface11();


    INodeTab searchNodes;

    if(selected) // 選択をエクスポート
    {
        INodeTab selNodes;
        ip11->GetSelNodeTab(selNodes);
        if(selNodes.Count() == 0)
        {
            // 有効なノードが一つも選択されない場合はアニメーションなし
            return "";
        }
        else
        {
            // 選択されたノードを順番に追加
            for (int idx=0; idx < selNodes.Count(); idx++)
            {
                INode* selnode = selNodes[idx];
                if(selnode)
                {
                    AddChildrenRecursive(searchNodes, selnode);
                }
            }
        }
    }
    else // 全てエクスポート対象にする
    {
        INode* sceneRoot = ip11->GetRootNode();
        int numRootChildren = sceneRoot->NumberOfChildren();
        if(numRootChildren == 0)
        {
            // 有効なノードが一つない場合はアニメーションなし
            return "";
        }
        else
        {
            AddChildrenRecursive(searchNodes, ip11->GetRootNode());
        }
    }

    // 検索するノードがなければアニメーションなし
    if(searchNodes.Count() == 0)
    {
        return "";
    }

    // アニメーションフラグを用意
    bool animFlags[Dcc::RExpOpt::FILE_TYPE_COUNT];

    for(int i = 0; i < Dcc::RExpOpt::FILE_TYPE_COUNT; i++)
    {	animFlags[i] = false;	}

    // 周囲光アニメーションのチェック
    {
        Control* c = ip11->GetAmbientController();
        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
        {
            animFlags[Dcc::RExpOpt::FSN] = true;
        }
    }

    //fogアニメーションのチェック
    /* fogは無効に
    {
        int numAtmos = ip11->NumAtmospheric();
        for (int i = 0; i < numAtmos; ++i)
        {
            Atmospheric* atmos = ip11->GetAtmospheric(i);
            //DebugPrint("atmos(%d): %s\n", i, atmos->GetName().data());

            // 環境効果が非アクティブなら無視
            if(atmos->Active(0) == FALSE) continue;
            // フォグ以外は無視
            if(atmos->ClassID() != Class_ID(FOG_CLASS_ID, 0)) continue;

            StdFog* fog = static_cast<StdFog*>(atmos);

            // フォグにパラメータがないときは無視（念のため）
            if(fog->NumSubs() == 0) continue;

            IParamBlock2* pb = static_cast<IParamBlock2*>(fog->SubAnim(0));
            showSubAnims(fog->SubAnim(0));

            //カラーアニメーションをチェック
            Animatable* sa = pb->SubAnim(0);
            if(sa && ((sa->NumKeys() > 0) || sa->IsAnimated()))
            {
                animFlags[Dcc::RExpOpt::FSN]	= true;
                break;
            }
        }
    }
    */

    // 各ノードのアニメーションをチェック
    for(int i = 0; i < searchNodes.Count(); i++)
    {
        INode* node = searchNodes[i];
        if(!node)
        {	continue;	}

        const MCHAR* nodename =node->GetName();

        ObjectState os = node->EvalWorldState(0);

        // ライトのアニメーション
        if (os.obj && os.obj->SuperClassID() == LIGHT_CLASS_ID)
        {
            if(!animFlags[Dcc::RExpOpt::FSN])
            {

                // ライトはノードの階層構造ではなくグローバル座標で出力されるため、
                // 親をたどってアニメーションの有無を調べる。
                if(HasNodeGlobalSRTAnim(node))
                {
                    animFlags[Dcc::RExpOpt::FSN] = true;
                }
                else
                {
                    GenLight* maxlit = (GenLight*)os.obj;
                    LightState ls;
                    Interval valid = FOREVER;
                    // サポートするライトかどうか？
                    maxlit->EvalLightState(0, valid, &ls);
                    if(ls.type == OMNI_LGT || ls.type == SPOT_LGT || ls.type == DIRECT_LGT )
                    {
                        Control *c = maxlit->GetColorControl();
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            animFlags[Dcc::RExpOpt::FSN]	= true;
                        }
                        c = maxlit->GetHotSpotControl();
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            animFlags[Dcc::RExpOpt::FSN]	= true;
                        }
                        c = maxlit->GetFalloffControl();
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            animFlags[Dcc::RExpOpt::FSN]	= true;
                        }
                        // 強度は直接取れないので、パラメータブロックから取得
                        IParamBlock2* pb = static_cast<IParamBlock2*>(maxlit->SubAnim(0));
                        if(pb)
                        {
                            Animatable* sa = pb->SubAnim(1);
                            if(sa && ((sa->NumKeys() > 0) || sa->IsAnimated()))
                            {
                                animFlags[Dcc::RExpOpt::FSN]	= true;
                                break;
                            }
                        }
                        /*
                        c = getControlFromPBlockByIndex(pb, 1); //即値…
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            animFlags[Dcc::RExpOpt::FSN]	= true;
                        }
                        */
                        showSubAnims(maxlit->SubAnim(0));
                    }
                    // ターゲットを調べる
                    if(node->GetTarget())
                    {
                        INode* target = node->GetTarget();
                        if(HasNodeGlobalSRTAnim(target))
                        {
                            animFlags[Dcc::RExpOpt::FSN] = true;
                        }
                    }
                }
            }
        }
        // カメラのアニメーション
        else if (os.obj && os.obj->SuperClassID() == CAMERA_CLASS_ID)
        {
            if(!animFlags[Dcc::RExpOpt::FSN])
            {
                GenCamera* cam = (GenCamera*)os.obj;
                Interval valid = FOREVER;

                // カメラはノードの階層構造ではなくグローバル座標で出力されるため、
                // 親をたどってアニメーションの有無を調べる。
                if(HasNodeGlobalSRTAnim(node))
                {
                    animFlags[Dcc::RExpOpt::FSN] = true;
                }
                else
                {
                    IParamBlock2* pb = static_cast<IParamBlock2*>(cam->SubAnim(0));

                    // 各パラメータのアニメーションをチェック
                    for (int iparam = CAM_PB_FOV; iparam <= CAM_PB_FRANGE; ++iparam)
                    {
                        if(iparam == CAM_PB_TDIST) continue; //TDISTは無視

                        Animatable* sa = pb->SubAnim(iparam);
                        if(sa && ((sa->NumKeys() > 0) || sa->IsAnimated()))
                        {
                            animFlags[Dcc::RExpOpt::FSN] = true;
                            break;
                        }
                    }

                    showSubAnims(cam->SubAnim(0));

                    // ターゲットを調べる
                    if(!animFlags[Dcc::RExpOpt::FSN] && node->GetTarget())
                    {
                        INode* target = node->GetTarget();
                        if(HasNodeGlobalSRTAnim(target))
                        {
                            animFlags[Dcc::RExpOpt::FSN] = true;
                        }
                        else
                        {
                            Control* tmc = node->GetTMController();
                            // コントローラがルックアップの時
                            if(tmc->ClassID() == Class_ID(LOOKAT_CONTROL_CLASS_ID, 0))
                            {
                                // ロール角を取得
                                Control* roll = tmc->GetRollController();
                                if(roll)
                                {
                                    if(roll->NumKeys() > 0 || roll->IsAnimated())
                                    {
                                        animFlags[Dcc::RExpOpt::FSN] = true;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        // ターゲット
        else if (os.obj && os.obj->ClassID().PartA() == TARGET_CLASS_ID)
        {
            // ターゲットは子がある時だけアニメーションフラグを立てる
            if(!animFlags[Dcc::RExpOpt::FSK])
            {
                if(node->NumberOfChildren() > 0 && HasNodeSRTAnim(node))
                {
                    animFlags[Dcc::RExpOpt::FSK] = true;
                }
            }
        }
        // その他一般的なノードのアニメーション
        else
        {
            // srt anim
            if(!animFlags[Dcc::RExpOpt::FSK])
            {
                if(HasNodeSRTAnim(node))
                {
                    animFlags[Dcc::RExpOpt::FSK] = true;
                }
            }

            // shape anim
            if(!animFlags[Dcc::RExpOpt::FSH])
            {
                Object* obj = node->GetObjectRef();

                while(!animFlags[Dcc::RExpOpt::FSH] && obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
                {
                    IDerivedObject* derivedObj = static_cast<IDerivedObject*>(obj);

                    for(int i = 0; i < derivedObj->NumModifiers(); i++)
                    {
                        Modifier* mod = derivedObj->GetModifier(i);
                        if(!mod || !mod->IsEnabled()) continue;

                        if(mod->ClassID() == Class_ID(MR3_CLASS_ID))
                        {
                            animFlags[Dcc::RExpOpt::FSH] = true;
                        }
                    }
                    obj = derivedObj->GetObjRef();
                }
            }
        }

        // FVB ボーンのビジビリティアニメーション
        if(!animFlags[Dcc::RExpOpt::FVB])
        {
            Control* vc = node->GetVisController();
            BOOL hidden = node->IsNodeHidden();
            if(vc && !hidden)
            {
                if(vc && (vc->NumKeys() > 0) || vc->IsAnimated())
                {
                    animFlags[Dcc::RExpOpt::FVB] = true;
                }
            }
        }

        // ノードに割り当てられたマテリアルをチェック
        Mtl* mtl= node->GetMtl();
        if(mtl)
        {
            // FTP テクスチャパターンアニメーション
            if(!animFlags[Dcc::RExpOpt::FTP])
            {
                animFlags[Dcc::RExpOpt::FTP] = HasTexPatAnim(mtl);
            }

            // FCL マテリアルカラーアニメーション
            if(!animFlags[Dcc::RExpOpt::FCL])
            {
                animFlags[Dcc::RExpOpt::FCL] = HasMtlColorAnim(mtl);
            }

            // FTS テクスチャSRT アニメーション
            if(!animFlags[Dcc::RExpOpt::FTS])
            {
                animFlags[Dcc::RExpOpt::FTS] = HasTexSRTAnim(mtl);
            }

        }
    }

    std::string retStr = "";
    for(int i = 0; i < Dcc::RExpOpt::FILE_TYPE_COUNT; i++)
    {
        if(animFlags[i])
        {
            std::string outStr = "";
            switch(i)
            {
            case Dcc::RExpOpt::FSK:
                outStr = "FSK"; break;
            case Dcc::RExpOpt::FVB:
                outStr = "FVB"; break;

            case Dcc::RExpOpt::FCL:
                outStr = "FCL"; break;
            case Dcc::RExpOpt::FTS:
                outStr = "FTS"; break;
            case Dcc::RExpOpt::FTP:
                outStr = "FTP"; break;
            case Dcc::RExpOpt::FSN:
                outStr = "FSN"; break;
            case Dcc::RExpOpt::FSH:
                outStr = "FSH"; break;
            /*
            case Dcc::RExpOpt::Fcm:
                outStr = "Fcm"; break;
            case Dcc::RExpOpt::Flt:
                outStr = "Flt"; break;
            */

            default:
                outStr = "ERR"; break;
            }
            if(retStr.size() > 0)
            {
                retStr += ",";
            }
            retStr += outStr;
        }
    }

    //MCHAR* retChar = new MCHAR[retStr.size() + 1];
    //strcpy_s(retChar, retStr.size() + 1,retStr.c_str());
    CStr ret = retStr.c_str();

    return ret;
}
