﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/g3d.h>
#include <nn/g3d/g3d_Configuration.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_SkeletalAnimObj.h>
#include <nn/g3d/g3d_BoneVisibilityAnimObj.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/g3d/g3d_ShapeAnimObj.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/util/util_BitArray.h>
#include <nn/util/util_MatrixApi.h>

#include <nn/vfx/vfx_Random.h>
#include <nn/vfx/vfx_Misc.h>

#include "detail/util_FileSystem.h"
#include <Descriptor.h>
#include <CombModelPreview.h>

#include <bfresPlane.h>
#include <bfresSphere.h>
#include <bntxLinear.h>


namespace detail {

#if defined( NN_BUILD_CONFIG_OS_WIN )

#include <Windows.h>

// モデル、テクスチャ、シェーダ読み込みフォルダ
char g_BaseDirectory[_MAX_PATH];


nn::vfx::detail::Mutex  m_Mutex;

#define SHADER_SOURCE( ... ) #__VA_ARGS__

//---------------------------------------------------------------------------
// 頂点シェーダソースコード
//---------------------------------------------------------------------------
static const char VERTEX_SHADER_SOURCE[] =
    "#version 440 \n"
    SHADER_SOURCE(
        layout( location = 0 ) in vec4  i_Position;
layout( location = 1 ) in vec3  i_Normal;
layout( location = 2 ) in vec4  i_Tangent;
layout( location = 3 ) in vec4  i_Binormal;
layout( location = 4 ) in vec2  i_UV;
layout( location = 5 ) in ivec4 i_Index;
layout( location = 6 ) in vec4  i_Weight;

layout( location = 0 ) out Raster
{
    vec3 pos_w;
    vec3 normal;
    vec3 normal_w;
    vec3 tangent_w;
    vec3 binormal_w;
    vec3 eye2pos_w;
    vec4 color;
    vec4 uv;
    vec4 projection;
} OUT;

layout( std140 ) uniform Skeleton
{
    vec4 mtxPalette[ 3 * 128 ];
};

layout( std140 ) uniform Shape
{
    vec4 shapeMtx[ 3 ];
    int vtxSkinCount;
};

layout( std140 ) uniform View
{
    vec4 projMtx[ 4 ];
    vec4 cameraMtx[ 4 ];
    vec4 cameraPos;
    vec4 cameraVec;
};

out gl_PerVertex
{
    vec4 gl_Position;
};

mat4 sysTexCoordTransformMatrix =
    mat4( 0.5, 0.0, 0.0, 0.0,
        0.0, 0.5, 0.0, 0.0,
        0.0, 0.0, 0.5, 0.0,
        0.5, 0.5, 0.5, 1.0 );

void main()
{
    vec4 pos_w = vec4( 0, 0, 0, 1 );
    vec3 nrm_w = vec3( 0, 0, 0 );

    // 分岐によるパフォーマンス度外視の実装。
    if ( vtxSkinCount == 0 )
    {
        pos_w.x = dot( shapeMtx[ 0 ], i_Position );
        pos_w.y = dot( shapeMtx[ 1 ], i_Position );
        pos_w.z = dot( shapeMtx[ 2 ], i_Position );
        nrm_w.x = dot( shapeMtx[ 0 ].xyz, i_Normal );
        nrm_w.y = dot( shapeMtx[ 1 ].xyz, i_Normal );
        nrm_w.z = dot( shapeMtx[ 2 ].xyz, i_Normal );

        OUT.tangent_w.x = dot( shapeMtx[ 0 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.y = dot( shapeMtx[ 1 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.z = dot( shapeMtx[ 2 ].xyz, i_Tangent.xyz );
        OUT.binormal_w.x = dot( shapeMtx[ 0 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.y = dot( shapeMtx[ 1 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.z = dot( shapeMtx[ 2 ].xyz, i_Binormal.xyz );
    }
    else if ( vtxSkinCount == 1 )
    {
        int mtxIndex = i_Index.x * 3;
        pos_w.x = dot( mtxPalette[ mtxIndex + 0 ], i_Position );
        pos_w.y = dot( mtxPalette[ mtxIndex + 1 ], i_Position );
        pos_w.z = dot( mtxPalette[ mtxIndex + 2 ], i_Position );
        nrm_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Normal );
        nrm_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Normal );
        nrm_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Normal );
        OUT.tangent_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Tangent.xyz );
        OUT.binormal_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Binormal.xyz );
    }
    else
    {
        for ( int i = 0; i < vtxSkinCount; ++i )
        {
            int mtxIndex = i_Index[ i ] * 3;
            float weight = i_Weight[ i ];
            pos_w.x += weight * dot( mtxPalette[ mtxIndex + 0 ], i_Position );
            pos_w.y += weight * dot( mtxPalette[ mtxIndex + 1 ], i_Position );
            pos_w.z += weight * dot( mtxPalette[ mtxIndex + 2 ], i_Position );
            nrm_w.x += weight * dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Normal );
            nrm_w.y += weight * dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Normal );
            nrm_w.z += weight * dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Normal );
            OUT.tangent_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Tangent.xyz );
            OUT.tangent_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Tangent.xyz );
            OUT.tangent_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Tangent.xyz );
            OUT.binormal_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Binormal.xyz );
            OUT.binormal_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Binormal.xyz );
            OUT.binormal_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Binormal.xyz );
        }
    }

    nrm_w = normalize( nrm_w );

    OUT.pos_w = pos_w.xyz;

    vec4 pos_v;
    pos_v.x = dot( cameraMtx[ 0 ], pos_w );
    pos_v.y = dot( cameraMtx[ 1 ], pos_w );
    pos_v.z = dot( cameraMtx[ 2 ], pos_w );
    pos_v.w = 1;

    gl_Position.x = dot( projMtx[ 0 ], pos_v );
    gl_Position.y = dot( projMtx[ 1 ], pos_v );
    gl_Position.z = dot( projMtx[ 2 ], pos_v );
    gl_Position.w = dot( projMtx[ 3 ], pos_v );

    OUT.projection = sysTexCoordTransformMatrix * gl_Position;

    OUT.normal_w = nrm_w;
    OUT.normal.x = dot( cameraMtx[ 0 ].xyz, nrm_w );
    OUT.normal.y = dot( cameraMtx[ 1 ].xyz, nrm_w );
    OUT.normal.z = dot( cameraMtx[ 2 ].xyz, nrm_w );

    OUT.uv.xy = i_UV;
    OUT.uv.zw = i_UV;

    vec3 up = vec3( 0, 1, 0 );
    float rate = dot( nrm_w, up ) * 0.5 + 0.5;
    OUT.color.rgb = rate * vec3( 0.7, 0.7, 0.9 ) + ( 1 - rate ) * vec3( 0.3, 0.2, 0.2 );
    OUT.color.a = pow( 1 - abs( OUT.normal.z ), 2 ) * 0.2;


    OUT.eye2pos_w = pos_w.xyz - cameraPos.xyz;
}
);

//---------------------------------------------------------------------------
// フラグメントシェーダソースコード
//---------------------------------------------------------------------------
static const char FRAGMENT_SHADER_SOURCE[] =
    "#version 440 \n"
    SHADER_SOURCE(
        layout( location = 0 ) in Raster
{
    vec3 pos_w;
    vec3 normal;
    vec3 normal_w;
    vec3 tangent_w;
    vec3 binormal_w;
    vec3 eye2pos_w;
    vec4 color;
    vec4 uv;
    vec4 projection;
} IN;

layout( location = 0 ) out vec4 o_Color;

uniform sampler2D nwAlbedoTexture0;

layout( std140 ) uniform Material
{
    vec4 offset;
};

void main()
{
    o_Color.r = max( IN.normal_w.y, 0.2 );
    o_Color.g = max( IN.normal_w.y, 0.2 );
    o_Color.b = max( IN.normal_w.y, 0.2 );
    o_Color.a = 1;
}
);

//---------------------------------------------------------------------------
// フラグメントシェーダ　ソースコードヘッダ
//---------------------------------------------------------------------------
static const char FRAGMENT_SHADER_HEADER[] =
    "#version 440\n"
    "mat4 IdentityMatrix = \n"
    "mat4( 0.5, 0.0, 0.0, 0.0, \n"
    "      0.0, 0.5, 0.0, 0.0, \n"
    "      0.0, 0.0, 0.5, 0.0, \n"
    "      0.5, 0.5, 0.5, 1.0 );\n"
    "#define M_PI 3.14159265358979323846264\n"
    "layout( location = 0 ) in Raster\n"
    "{\n"
    "    vec3 pos_w;\n"
    "    vec3 normal;\n"
    "    vec3 normal_w;\n"
    "    vec3 tangent_w;"
    "    vec3 binormal_w;"
    "    vec3 eye2pos_w;\n"
    "    vec4 color;\n"
    "    vec4 uv;\n"
    "    vec4 projection;\n"
    "} IN;\n"
    "layout( std140 ) uniform Material\n"
    "{\n"
    "    vec4 nwParam;\n"
    "    vec4 nwParam1;\n"
    "    vec4 nwParam2;\n"
    "    vec4 nwParam3;\n"
    "};\n"
    "layout( std140 ) uniform View\n"
    "{\n"
    "    mat4 projMtx;\n"
    "    mat4 cameraMtx;\n"
    "    vec4 cameraPos;\n"
    "    vec4 cameraVec;\n"
    "};\n"
    "float CalcFresnelRate( vec3 eyeVec, vec3 worldNormal )\n"
    "{\n"
    "#define fresnelAlphaMin         0.0\n"
    "#define fresnelAlphaMax         1.0\n"
    "    vec3  worldDir = normalize( eyeVec );\n"
    "    float rotDiff = abs( dot( worldNormal, worldDir.xyz ) );\n"
    "    float fresnel = smoothstep( fresnelAlphaMin, fresnelAlphaMax, rotDiff );\n"
    "    return fresnel;\n"
    "}\n"
    "#define _USE_NN_VFX\n"
    "layout( location = 0 ) out vec4 nwOutputAlbedoColor;\n"
    "#define OUTPUT_COLOR nwOutputAlbedoColor\n"
    "uniform sampler2D nwAlbedoTexture0;\n"
    "uniform sampler2D nwAlbedoTexture1;\n"
    "uniform sampler2D nwAlbedoTexture2;\n"
    "uniform sampler2D nwAlbedoTexture3;\n"
    "uniform sampler2D nwColorBufferTexture;\n"
    "uniform sampler2D nwDepthBufferTexture;\n"
    "#define sysTextureSampler0 nwAlbedoTexture0\n"
    "#define sysTextureSampler1 nwAlbedoTexture1\n"
    "#define sysTextureSampler2 nwAlbedoTexture2\n"
    "#define sysFrameBufferTexture nwColorBufferTexture\n"
    "#define sysDepthBufferTexture nwDepthBufferTexture\n"
    "#define nwTextureCoord0 IN.uv.xy\n"
    "#define nwTextureCoord1 IN.uv.xy\n"
    "#define nwTextureCoord2 IN.uv.xy\n"
    "#define nwTextureCoord3 IN.uv.xy\n"
    "#define sysTexCoord01Vary IN.uv\n"
    "#define sysTexCoord2Vary IN.uv.xy\n"
    "#define sysLifeRatio nwParam.x\n"
    "#define sysFragCoordVary IN.projection\n"
    "#define sysWorldPositionVary IN.pos_w\n"
    "#define sysWorldNormalVary IN.normal_w\n"
    "#define sysWorldTangentVary IN.tangent_w\n"
    "#define sysWorldBinormalVary IN.binormal_w\n"
    "#define v_outWorldNormal IN.normal_w\n"
    "#define sysViewMatrix cameraMtx\n"
    "#define v_outPrimitiveColor IN.color\n"
    "#define viewMat cameraMtx\n"
    "#define sysShaderAnim nwParam.y\n"
    "#define sysShaderAnimY nwParam.z\n"
    "#define sysShaderAnimZ nwParam.w\n"
    "#define emitterMatrixRT IdentityMatrix\n"
    "#define sysEmitterRTMatrix IdentityMatrix\n"
    "#define eyePos cameraPos\n"
    "#define eyeVec cameraVec\n"
    "#define sysEyeVecVary eye2pos_w\n"
    "#define v_outColor0 nwParam1\n"
    "#define v_outColor1 nwParam2\n"
    "#define sysParticleRandom nwParam3\n"
    "#define sysFragRandomVary nwParam3\n"
    "#define sysProjectionMatrix projMtx\n"
    SHADER_SOURCE(
float       nwTimeRatio;                          // nw_def コンパイル通し用
uniform sampler3D sysCurlNoiseTextureArray;       // カールノイズテクスチャアレイサンプラ―( コンパイル通し用 )
float CalcAlphaProcess( float inAlpha ) { return inAlpha; }
void FinalAdjustmentFragmentColor() {}
float GetDepthValue( vec4 position ) { return 0.0; }
float GetDepthValueFromTexture( vec4 projection ) { return 0.0; }
);




//---------------------------------------------------------------------------
//  テクスチャ
//---------------------------------------------------------------------------
Texture::Texture() NN_NOEXCEPT
{
    m_TextureFace = 0;
    m_pDevice = nullptr;
    m_pResTextureFile[ 0 ] = nullptr;
    m_pResTextureFile[ 1 ] = nullptr;
    m_TextureBinaryPtr[ 0 ] = nullptr;
    m_TextureBinaryPtr[ 1 ] = nullptr;
}

Texture::~Texture() NN_NOEXCEPT
{
    if ( m_pResTextureFile[ 0 ] )
    {
        m_pResTextureFile[ 0 ]->Finalize( m_pDevice );
    }
    if ( m_pResTextureFile[ 1 ] )
    {
        m_pResTextureFile[ 1 ]->Finalize( m_pDevice );
    }

    if ( m_TextureBinaryPtr[ 0 ] )
    {
        detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( m_TextureBinaryPtr[ 0 ] );
    }
    if ( m_TextureBinaryPtr[ 1 ] )
    {
        detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( m_TextureBinaryPtr[ 1 ] );
    }

}

bool Texture::Initialize( nn::gfx::Device* pDevice, void* pBntxBinary ) NN_NOEXCEPT
{
    // こちらで初期化されたテクスチャは、リロードできない。

    Finalize( pDevice );
    m_TextureFace = 1 - m_TextureFace;

    m_TextureBinaryPtr[ m_TextureFace ] = reinterpret_cast<uint8_t *>( pBntxBinary );

    m_pResTextureFile[ m_TextureFace ] = nn::gfx::ResTextureFile::ResCast( m_TextureBinaryPtr[ m_TextureFace ] );
    NN_SDK_ASSERT_NOT_NULL( m_pResTextureFile );

    m_pResTextureFile[ m_TextureFace ]->Initialize( pDevice );

    int textureCount = m_pResTextureFile[ m_TextureFace ]->GetTextureDic()->GetCount();
    for ( int i = 0; i < textureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = m_pResTextureFile[ m_TextureFace ]->GetResTexture( i );
        resTexture->Initialize( pDevice );
    }

    return true;
}

bool Texture::Initialize( nn::gfx::Device* pDevice, const char* pFtxbFilePath, const char* pOutputDir ) NN_NOEXCEPT
{
    if ( !pFtxbFilePath )
    {
        return false;
    }
    if ( pDevice )
    {
        m_pDevice = pDevice;
    }
    if ( !m_pDevice )
    {
        return false;
    }

    Finalize( pDevice );
    m_TextureFace = 1 - m_TextureFace;

    size_t textureDataSize;

    char command[ 1024 ];

    GetCurrentDirectoryA( _MAX_PATH, command );

    // テクスチャコンバート
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    sprintf_s( command, "%%NINTENDO_SDK_ROOT%%\\Tools\\Graphics\\3dTools\\3dTextureConverter.exe %s -o %s\\combinerPreviewTexture.bntx --tile-mode NX", pFtxbFilePath, pOutputDir );
#else
    sprintf_s( command, "%%NINTENDO_SDK_ROOT%%\\Tools\\Graphics\\3dTools\\3dTextureConverter.exe %s -o %s\\combinerPreviewTexture.bntx --tile-mode linear", pFtxbFilePath, pOutputDir );
    //sprintf_s( command, "3dTools\\3dTextureConverter.exe %s -o %s\\combinerPreviewTexture.bntx --tile-mode linear", pFtxbFilePath, pOutputDir );
#endif

    system( command );

    // テクスチャロード
    sprintf_s( command, "%s\\combinerPreviewTexture.bntx", pOutputDir );
    bool resRead = detail::VfxViewerUtilFileSystem::GetInstance().Read( &m_TextureBinaryPtr[ m_TextureFace ], &textureDataSize, command, 1024 * 64 );
    if ( !resRead )
    {
        return false;
    }

    m_pResTextureFile[ m_TextureFace ] = nn::gfx::ResTextureFile::ResCast( m_TextureBinaryPtr[ m_TextureFace ] );
    NN_SDK_ASSERT_NOT_NULL( m_pResTextureFile );

    m_pResTextureFile[ m_TextureFace ]->Initialize( pDevice );

    int textureCount = m_pResTextureFile[ m_TextureFace ]->GetTextureDic()->GetCount();
    for ( int i = 0; i < textureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = m_pResTextureFile[ m_TextureFace ]->GetResTexture( i );
        resTexture->Initialize( pDevice );
    }

    return true;
}

void Texture::UpdateDescriptorSlot( detail::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator )
{
    if ( !m_pResTextureFile[ m_TextureFace ] )
    {
        return;
    }

    int textureCount = m_pResTextureFile[ m_TextureFace ]->GetTextureDic()->GetCount();
    for ( int i = 0; i < textureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = m_pResTextureFile[ m_TextureFace ]->GetResTexture( i );
        const nn::gfx::TextureView* textureView = resTexture->GetTextureView();
        pTextureDescriptorIndexAllocator->AllocateTransientTextureDescriptorSlot( &m_DescriptorSlot[ m_TextureFace ], *textureView, NULL );
    }
}

void Texture::Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    if ( m_pResTextureFile[ m_TextureFace ] )
    {
        m_pResTextureFile[ m_TextureFace ]->Finalize( pDevice );
        m_pResTextureFile[ m_TextureFace ] = nullptr;
        m_DescriptorSlot[ m_TextureFace ].Invalidate();
        detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( m_TextureBinaryPtr[ m_TextureFace ] );
        m_TextureBinaryPtr[ m_TextureFace ] = nullptr;
    }
}



//---------------------------------------------------------------------------
//  シェーダ
//---------------------------------------------------------------------------
Shader::Shader() NN_NOEXCEPT
{
    m_pDevice = nullptr;
    m_IsInitialized[ 0 ] = false;
    m_IsInitialized[ 1 ] = false;
    m_ShaderFace = 0;
}

Shader::~Shader() NN_NOEXCEPT
{
    if ( m_IsInitialized[ 0 ] )
    {
        m_Shader[ 0 ].Finalize( m_pDevice );
    }
    if ( m_IsInitialized[ 1 ] )
    {
        m_Shader[ 1 ].Finalize( m_pDevice );
    }
}

bool Shader::Initialize( nn::gfx::Device* pDevice, nn::gfx::ShaderStage shaderStage, const char* shaderCode ) NN_NOEXCEPT
{
    if ( pDevice )
    {
        m_pDevice = pDevice;
    }
    if ( !m_pDevice )
    {
        return false;
    }

    Finalize( m_pDevice );
    m_ShaderFace = 1 - m_ShaderFace;

    // シェーダ生成
    nn::gfx::Shader::InfoType info;
    info.SetDefault();
    info.SetSeparationEnabled( true );

    nn::gfx::ShaderCode psCode;
    psCode.codeSize = static_cast< uint32_t >( strlen( shaderCode ) );
    psCode.pCode = shaderCode;

    info.SetShaderCodePtr( shaderStage, &psCode );
    info.SetSourceFormat( nn::gfx::ShaderSourceFormat_Glsl );
    info.SetCodeType( nn::gfx::ShaderCodeType_Source );
    nn::gfx::ShaderInitializeResult result = m_Shader[ m_ShaderFace ].Initialize( m_pDevice, info );
    NN_SDK_ASSERT( result == nn::gfx::ShaderInitializeResult_Success, "nn::gfx::Shader::Initialize() failed (%d)\n", result );
    NN_UNUSED( result );

    // TODO:シェーダコンパイルエラー時の処理

    m_IsInitialized[ m_ShaderFace ] = true;

    return true;
}

void Shader::Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    if ( m_IsInitialized[ m_ShaderFace ] )
    {
        m_Shader[ m_ShaderFace ].Finalize( pDevice );
        m_IsInitialized[ m_ShaderFace ] = false;
    }
}



//---------------------------------------------------------------------------
//  シェーダ
//---------------------------------------------------------------------------
Model::Model() NN_NOEXCEPT
{
    m_pDevice = nullptr;
    m_pG3dModelResFile = nullptr;
    m_pVertexStateArray = nullptr;
    m_pVertexStateBuffer = nullptr;
    m_pModelObj = nullptr;
    m_pHeap = nullptr;
    m_CurrentFace = 0;
    m_RotateY = 0.0f;
}


//---------------------------------------------------------------------------
//  初期化処理を行います。
//---------------------------------------------------------------------------
bool Model::Initialize( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pHeap ) NN_NOEXCEPT
{
    m_pDevice = pDevice;
    m_pHeap = pHeap;

    // デフォルトシェーダを初期化
    m_VertexShader.Initialize( m_pDevice, nn::gfx::ShaderStage_Vertex, VERTEX_SHADER_SOURCE );
    m_FragmentShader.Initialize( m_pDevice, nn::gfx::ShaderStage_Pixel, FRAGMENT_SHADER_SOURCE );

    return true;
}

//---------------------------------------------------------------------------
//  破棄を行います。
//---------------------------------------------------------------------------
void Model::Finalize() NN_NOEXCEPT
{

}

//---------------------------------------------------------------------------
//  モデルバイナリをセットアップします。
//---------------------------------------------------------------------------
bool Model::SetupG3dModelBinary( void* pBfresG3dBinary )
{
    NN_SDK_ASSERT( nn::g3d::ResFile::IsValid( pBfresG3dBinary ) );
    m_pG3dModelResFile = nn::g3d::ResFile::ResCast( pBfresG3dBinary );
    NN_SDK_ASSERT( m_pG3dModelResFile->GetModelCount() == 1 );
    m_pG3dModelResFile->Setup( m_pDevice );

    nn::g3d::ResModel* pResModel = m_pG3dModelResFile->GetModel( 0 );

    // 頂点ステートを初期化
    {
        const char* attribName[] = { "_p0", "_n0", "_t0", "_b0", "_u0", "_i0", "_w0" };
        const char* attribId[] = { "i_Position", "i_Normal", "i_Tangent", "i_Binormal", "i_UV", "i_Index", "i_Weight" };
        const int attribCount = sizeof( attribName ) / sizeof( char* );
        const int bufferCount = 16;

        // VertexStateは ResVertex と1対1で対応します。
        // 複数の ResShape が ResVertex を共有している場合があります。
        int numVertex = pResModel->GetVertexCount();
        size_t vertexStateArraySize = sizeof( nn::gfx::VertexState ) * numVertex;
        m_pVertexStateArray = static_cast< nn::gfx::VertexState* >( m_pHeap->Alloc( vertexStateArraySize, 4 ) );
        NN_ASSERT_NOT_NULL( m_pVertexStateArray );

        for ( int idxVertex = 0; idxVertex < numVertex; ++idxVertex )
        {
            nn::gfx::VertexState* pVertexState = new( &m_pVertexStateArray[ idxVertex ] ) nn::gfx::VertexState;
            nn::gfx::VertexState::InfoType info;
            info.SetDefault();
            nn::g3d::ResVertex* pResVertex = pResModel->GetVertex( idxVertex );

            // 頂点属性を設定
            int idxElement = 0;
            nn::gfx::VertexAttributeStateInfo vertexAttribs[ attribCount ];
            for ( int idxAttrib = 0; idxAttrib < attribCount; ++idxAttrib )
            {
                const char* name = attribName[ idxAttrib ];
                vertexAttribs[ idxAttrib ].SetDefault();
                nn::g3d::ResVertexAttr* pResVertexAttr = pResVertex->FindVertexAttr( name );
                if ( pResVertexAttr )
                {
                    vertexAttribs[ idxElement ].SetBufferIndex( pResVertexAttr->GetBufferIndex() );
                    vertexAttribs[ idxElement ].SetFormat( pResVertexAttr->GetFormat() );
                    vertexAttribs[ idxElement ].SetOffset( pResVertexAttr->GetOffset() );
                    int slotAttr = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, attribId[ idxAttrib ] );
                    vertexAttribs[ idxElement ].SetShaderSlot( slotAttr );
                    if ( slotAttr != -1 )
                    {
                        idxElement++;
                    }
                }
            }

            // 頂点バッファ情報を設定
            int numVertexBuffer = pResVertex->GetVertexBufferCount();
            nn::gfx::VertexBufferStateInfo bufferStateInfo[ bufferCount ];
            NN_ASSERT( numVertexBuffer < bufferCount );
            for ( int idxBuffer = 0; idxBuffer < numVertexBuffer; ++idxBuffer )
            {
                bufferStateInfo[ idxBuffer ].SetDefault();
                bufferStateInfo[ idxBuffer ].SetStride( pResVertex->GetVertexBufferStride( idxBuffer ) );
            }

            // 頂点属性と頂点バッファ情報を頂点ステートに設定
            info.SetVertexAttributeStateInfoArray( vertexAttribs, idxElement );
            info.SetVertexBufferStateInfoArray( bufferStateInfo, numVertexBuffer );

            size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize( info );
            m_pVertexStateBuffer = m_pHeap->Alloc( memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
            NN_ASSERT_NOT_NULL( m_pVertexStateBuffer );
            pVertexState->SetMemory( m_pVertexStateBuffer, memorySize );
            pVertexState->Initialize( m_pDevice, info, NULL );
            pResVertex->SetUserPtr( pVertexState );
        }
    }

    // モデルを初期化
    {
        m_pModelObj = new( m_pHeap->Alloc( sizeof( nn::g3d::ModelObj ), nn::g3d::ModelObj::Alignment_Buffer ) ) nn::g3d::ModelObj();
        NN_SDK_ASSERT_NOT_NULL( m_pModelObj );
        nn::g3d::ModelObj::Builder builder( m_pG3dModelResFile->GetModel( 0 ) );

        builder.MaterialBufferingCount( 2 );
        builder.SkeletonBufferingCount( 2 );
        builder.ShapeBufferingCount( 2 );
        builder.ViewCount( 2 );
        builder.CalculateMemorySize();
        size_t bufferSize = builder.GetWorkMemorySize();
        void* pBuffer = m_pHeap->Alloc( bufferSize, nn::g3d::SkeletalAnimObj::Alignment_Buffer );
        NN_SDK_ASSERT_NOT_NULL( pBuffer );
        bool success = builder.Build( m_pModelObj, pBuffer, bufferSize );
        NN_SDK_ASSERT( success );

        // Uniform Block のバッファは所定のアライメントが必要です。
        size_t blockBufferSize = m_pModelObj->CalculateBlockBufferSize( m_pDevice );
        ptrdiff_t blockBufferOffset = -1;
        blockBufferOffset = CombinerPreviewer::GetInstance()->AllocateFromMemoryPool( blockBufferSize, m_pModelObj->GetBlockBufferAlignment( m_pDevice ) );
        success = m_pModelObj->SetupBlockBuffer( m_pDevice, CombinerPreviewer::GetInstance()->GetMemoryPool(), blockBufferOffset, blockBufferSize );
        NN_ASSERT( success );
    }

    // 原点で計算処理
    nn::util::Matrix4x3fType drawMatrix;
    nn::util::MatrixIdentity( &drawMatrix );
    m_pModelObj->CalculateWorld( drawMatrix );

    return true;
}

//---------------------------------------------------------------------------
//  G3dモデル破棄を行います。
//---------------------------------------------------------------------------
void Model::FinalizeG3dModelBinary() NN_NOEXCEPT
{
    if ( m_pVertexStateArray != NULL )
    {
        m_pHeap->Free( m_pVertexStateArray );
        m_pVertexStateArray = NULL;
    }
    if ( m_pVertexStateBuffer != NULL )
    {
        m_pHeap->Free( m_pVertexStateBuffer );
        m_pVertexStateBuffer = NULL;
    }

    m_pModelObj->CleanupBlockBuffer( m_pDevice );

    m_pG3dModelResFile->Cleanup( m_pDevice );

    m_pHeap->Free( m_pModelObj->GetBufferPtr() );
    m_pHeap->Free( m_pModelObj );
}



//---------------------------------------------------------------------------
//  モデルファイルをロードします。
//---------------------------------------------------------------------------
bool Model::LoadG3dModelBinary( const char* pFmdbFilePath, const char* pOutputDir )
{
    if ( !m_VertexShader.GetGfxShader() ) return false;
    if ( !m_FragmentShader.GetGfxShader() ) return false;

    if ( !pFmdbFilePath )
    {
        return false;
    }

    // bfresコンバート
    char command[ 1024 ];
    sprintf_s( command, "%%NINTENDO_SDK_ROOT%%\\Tools\\Graphics\\3dTools\\3dBinaryConverter.exe -o %s\\combinerPreviewModel.bfres %s -i", pOutputDir, pFmdbFilePath );
    system( command );

    // bfres ロード
    uint8_t*    pBfresBinary = nullptr;
    size_t      pBfresBinarySize;
    sprintf_s( command, "%s\\combinerPreviewModel.bfres", pOutputDir );
    bool resRead = detail::VfxViewerUtilFileSystem::GetInstance().Read( &pBfresBinary, &pBfresBinarySize, command, 1024 * 64 );
    if ( !resRead )
    {
        return false;
    }

    SetupG3dModelBinary( pBfresBinary );

    return true;
}

//---------------------------------------------------------------------------
//! @brief  ブロックの更新処理を行います。
//---------------------------------------------------------------------------
void Model::CalcBlock( const nn::util::Matrix4x3fType* viewMtx, float addRotateY )
{
    if ( !m_pModelObj )
    {
        return;
    }

    // バッファの更新
    m_CurrentFace = 1 - m_CurrentFace;

    m_RotateY += addRotateY;

    nn::util::Matrix4x3fType drawMatrix;
    nn::util::MatrixIdentity( &drawMatrix );
    nn::util::Vector3fType rotate;
    nn::util::VectorSet( &rotate, 0.0f, m_RotateY, 0.0f );
    nn::util::MatrixSetRotateXyz( &drawMatrix, rotate );
    m_pModelObj->CalculateWorld( drawMatrix );

    m_pModelObj->CalculateSkeleton( m_CurrentFace );
    m_pModelObj->CalculateMaterial( m_CurrentFace );
    m_pModelObj->CalculateShape( m_CurrentFace );
    m_pModelObj->CalculateView( m_CurrentFace, *viewMtx, m_CurrentFace );
}

//---------------------------------------------------------------------------
// フラグメントシェーダの更新
//---------------------------------------------------------------------------
void Model::UpdateFragmentShader( const char* fshFilePath )
{
    if ( !fshFilePath )
    {
        // パス指定が無い場合は、初期シェーダに戻す
        m_FragmentShader.Initialize( nullptr, nn::gfx::ShaderStage_Pixel, FRAGMENT_SHADER_SOURCE );
        return;
    }

    if ( strlen( fshFilePath ) == 0 )
    {
        return;
    }

    uint8_t* pCodeData;
    size_t codeSize;
    bool resRead = detail::VfxViewerUtilFileSystem::GetInstance().Read( &pCodeData, &codeSize, fshFilePath );
    if ( !resRead )
    {
        return;
    }

    size_t headerSize = strlen( FRAGMENT_SHADER_HEADER );
    void* codeBuffer = m_pHeap->Alloc( codeSize + 4098 );
    if ( !codeBuffer )
    {
        return;
    }

    strncpy( reinterpret_cast< char* >( codeBuffer ), FRAGMENT_SHADER_HEADER, headerSize );
    strncpy( reinterpret_cast< char* >( codeBuffer ) + headerSize, reinterpret_cast< const char* >( pCodeData ), codeSize );
    reinterpret_cast< char* >( codeBuffer )[ headerSize + codeSize ] = 0;
    m_FragmentShader.Initialize( nullptr, nn::gfx::ShaderStage_Pixel, reinterpret_cast< const char* >( codeBuffer ) );
    m_pHeap->Free( codeBuffer );
    detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( pCodeData );
}

//---------------------------------------------------------------------------
//! @brief  描画処理を行います。
//---------------------------------------------------------------------------
void Model::Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::GpuBuffer* pGpuBuffer,
    const nn::util::Matrix4x4fType* projectionMatrix, const nn::util::Matrix4x3fType* viewMatrix, const nn::util::Vector3fType* camPos )
{
    if ( !m_pModelObj )
    {
        return;
    }

    // シェーダーを設定
    pCommandBuffer->SetShader( m_VertexShader.GetGfxShader(), nn::gfx::ShaderStageBit_Vertex );
    pCommandBuffer->SetShader( m_FragmentShader.GetGfxShader(), nn::gfx::ShaderStageBit_Pixel );

    int slotSkeleton = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Skeleton" );
    int slotShape = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Shape" );
    int slotVertexView = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "View" );
    int slotFragmentView = m_FragmentShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "View" );


    // ユニフォームブロック設定
    const nn::g3d::SkeletonObj* pSkeletonObj = m_pModelObj->GetSkeleton();
    nn::gfx::GpuAddress gpuAddress;
    if ( pSkeletonObj->GetMtxBlockSize() > 0 )
    {
        pSkeletonObj->GetMtxBlock( m_CurrentFace )->GetGpuAddress( &gpuAddress );
        pCommandBuffer->SetConstantBuffer( slotSkeleton, nn::gfx::ShaderStage_Vertex, gpuAddress, pSkeletonObj->GetMtxBlockSize() );
    }

    // ビュー
    ViewBlock* pViewBlockBuffer = reinterpret_cast< ViewBlock* >( pGpuBuffer->Allocate( sizeof( ViewBlock ) ) );
    if ( pViewBlockBuffer )
    {
        pViewBlockBuffer->Set( *viewMatrix, *projectionMatrix, *camPos );
        pGpuBuffer->GetGpuAddress( &gpuAddress, pViewBlockBuffer );
        if ( slotVertexView != -1 )
        {
            pCommandBuffer->SetConstantBuffer( slotVertexView, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof( ViewBlock ) );
        }
        if ( slotFragmentView != -1 )
        {
            pCommandBuffer->SetConstantBuffer( slotFragmentView, nn::gfx::ShaderStage_Pixel, gpuAddress, sizeof( ViewBlock ) );
        }
    }

    int shapeCount = m_pModelObj->GetShapeCount();
    for ( int idxShape = 0; idxShape < shapeCount; ++idxShape )
    {
        const nn::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape( idxShape );

        if ( !m_pModelObj->IsShapeVisible( idxShape ) ) // ビジビリティの制御を行います。
        {
            continue;
        }

        // シェイプ
        pShapeObj->GetShapeBlock( 0, 0 )->GetGpuAddress( &gpuAddress );
        pCommandBuffer->SetConstantBuffer( slotShape, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof( nn::g3d::ShapeBlock ) );

        // 頂点属性を設定
        int idxVertex = pShapeObj->GetVertexIndex();
        nn::g3d::ResModel* pResModel = const_cast< nn::g3d::ResModel* >( m_pModelObj->GetResource() );
        nn::g3d::ResVertex* pResVertex = pResModel->GetVertex( idxVertex );
        nn::gfx::VertexState* pVertexState = static_cast< nn::gfx::VertexState* >( pResVertex->GetUserPtr() );
        pCommandBuffer->SetVertexState( pVertexState );

        // バッファを設定
        int numVertexBuffer = pResVertex->GetVertexBufferCount();
        for ( int idxBuffer = 0; idxBuffer < numVertexBuffer; ++idxBuffer )
        {
            pResVertex->GetVertexBuffer( idxBuffer )->GetGpuAddress( &gpuAddress );
            pCommandBuffer->SetVertexBuffer( idxBuffer, gpuAddress,
                pResVertex->GetVertexBufferStride( idxBuffer ),
                pResVertex->GetVertexBufferInfo( idxBuffer )->GetSize() );
        }

        // 描画コール
        pShapeObj->GetResMesh()->Draw( pCommandBuffer, 1 );
    }
}






//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
CombinerPreviewer::CombinerPreviewer()
{
    m_pDevice = nullptr;
    m_pHeap = nullptr;
    m_TextureReload = false;
    m_TextureReset = false;
    m_PreviewShaderReload = false;
    m_OutputShaderReload = false;
    m_pBinaryLoadBuffer = nullptr;

    m_pMemoryPoolPtr = nullptr;
    m_MemoryPoolOffset = 0;

    m_ShaderParameter[ 0 ] = 0.0f;
    m_ShaderParameter[ 1 ] = 0.0f;
    m_ShaderParameter[ 2 ] = 0.0f;
    m_ShaderParameter[ 3 ] = 0.0f;

    m_VectorParameter0.x = 1.0f;
    m_VectorParameter0.y = 1.0f;
    m_VectorParameter0.z = 1.0f;
    m_VectorParameter0.w = 1.0f;

    m_VectorParameter1.x = 1.0f;
    m_VectorParameter1.y = 1.0f;
    m_VectorParameter1.z = 1.0f;
    m_VectorParameter1.w = 1.0f;

    m_VectorParameter2.x = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();
    m_VectorParameter2.y = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();
    m_VectorParameter2.z = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();
    m_VectorParameter2.w = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();

    memset( m_TextureFilePath[ 0 ], 0, _MAX_PATH );
    memset( m_TextureFilePath[ 1 ], 0, _MAX_PATH );
    memset( m_TextureFilePath[ 2 ], 0, _MAX_PATH );
    memset( m_TextureFilePath[ 3 ], 0, _MAX_PATH );
    memset( m_OutputFshFilePath,    0, _MAX_PATH );
    memset( m_PreviewFshFilePath,   0, _MAX_PATH );
    memset( m_OutputModelFilePath,  0, _MAX_PATH );
    memset( m_WorkingDir,           0, _MAX_PATH );

}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
CombinerPreviewer::~CombinerPreviewer() {}

//---------------------------------------------------------------------------
//! @brief  初期化処理を行います。
//---------------------------------------------------------------------------
bool CombinerPreviewer::Initialize( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pHeap )
{
    m_pDevice = pDevice;
    m_pHeap = pHeap;

    m_ReloadTextureIndex = -1;
    m_ResetTextureIndex = CombinerPreviewer_TextureCountMax;
    m_ShaderReset = false;

    m_Model[ 0 ].Initialize( pDevice, pHeap );
    m_Model[ 1 ].Initialize( pDevice, pHeap );

    g_Time = 0;

    // テンポラリフォルダ
    WIN32_FIND_DATA fd = { 0 };

    GetCurrentDirectoryA( _MAX_PATH, g_BaseDirectory );
    SetCurrentDirectoryA( g_BaseDirectory );
    if ( FindFirstFile( L"ScreenCap", &fd ) == INVALID_HANDLE_VALUE )
    {
        CreateDirectory( L"ScreenCap", NULL );
    }
    SetCurrentDirectory( L"ScreenCap" );

    strcat_s( g_BaseDirectory, _MAX_PATH, "\\ScreenCap" );

    // メモリプールの初期化
    static const uint64_t cpuUncachedMemoryPoolSize = 4 * 1024 * 1024; // ダブり
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( m_pDevice, memoryPoolInfo );
    m_pMemoryPoolPtr = m_pHeap->Alloc( cpuUncachedMemoryPoolSize, memoryPoolGranularity );
    memoryPoolInfo.SetPoolMemory( m_pMemoryPoolPtr, cpuUncachedMemoryPoolSize );
    m_MemoryPool.Initialize( m_pDevice, memoryPoolInfo );

    // 共通テクスチャサンプラ初期化
    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    m_Sampler.Initialize( m_pDevice, samplerInfo );

    // 共通テクスチャ
    {
        void* buffer = m_pHeap->Alloc( sizeof( bntxLinear ), 1024 * 64 );
        NN_SDK_ASSERT_NOT_NULL( buffer );
        memcpy( buffer, bntxLinear, sizeof( bntxLinear ) );
        m_CommonTexture.Initialize( m_pDevice, buffer );
    }

    // バイナリロード用バッファ
    const size_t binaryLoadBufferSize = 1024 * 1024 * 16;
    m_pBinaryLoadBuffer = pHeap->Alloc( binaryLoadBufferSize );
    m_HeapHandle = nn::lmem::CreateExpHeap( m_pBinaryLoadBuffer, binaryLoadBufferSize, nn::lmem::CreationOption_NoOption );
    detail::VfxViewerUtilFileSystem::GetInstance().SetHeap( m_HeapHandle );

    return true;
}

//---------------------------------------------------------------------------
//! @brief  メモリプールから領域を確保します
//---------------------------------------------------------------------------
ptrdiff_t CombinerPreviewer::AllocateFromMemoryPool( size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/ )
{
    static const uint64_t cpuUncachedMemoryPoolSize = 4 * 1024 * 1024;

    ptrdiff_t offset = nn::util::align_up( m_MemoryPoolOffset, alignment );
    NN_ASSERT( cpuUncachedMemoryPoolSize - offset >= size, "AllocateMemoryPool failed.\n" );
    m_MemoryPoolOffset = offset + size;
    return offset;
}




//---------------------------------------------------------------------------
//! @brief  初期状態モデルファイルをロードします。
//---------------------------------------------------------------------------
bool CombinerPreviewer::InitializeG3dModel()
{
    void* buffer = m_pHeap->Alloc( sizeof( bfresSphere ), 1024 * 64 );
    NN_SDK_ASSERT_NOT_NULL( buffer );
    memcpy( buffer, bfresSphere, sizeof( bfresSphere ) );
    m_Model[ 0 ].SetupG3dModelBinary( buffer );

    buffer = m_pHeap->Alloc( sizeof( bfresPlane ), 1024 * 64 );
    NN_SDK_ASSERT_NOT_NULL( buffer );
    memcpy( buffer, bfresPlane, sizeof( bfresPlane ) );
    m_Model[ 1 ].SetupG3dModelBinary( buffer );

    return true;
}


//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewer::UpdateShaderParameter( const char* name, float value )
{
    if ( !strcmp( name, "m_ShaderParameter[ 0 ]" ) )
    {
        m_ShaderParameter[ 0 ] = value;
    }
    else if ( !strcmp( name, "m_ShaderParameter[ 1 ]" ) )
    {
        m_ShaderParameter[ 1 ] = value;
    }
    else if ( !strcmp( name, "m_ShaderParameter[ 2 ]" ) )
    {
        m_ShaderParameter[ 2 ] = value;
    }
    else if ( !strcmp( name, "m_ShaderParameter[ 3 ]" ) )
    {
        m_ShaderParameter[ 3 ] = value;
    }
    else if ( !strcmp( name, "m_VectorParameter0.x" ) )
    {
        m_VectorParameter0.x = value;
    }
    else if ( !strcmp( name, "m_VectorParameter0.y" ) )
    {
        m_VectorParameter0.y = value;
    }
    else if ( !strcmp( name, "m_VectorParameter0.z" ) )
    {
        m_VectorParameter0.z = value;
    }
    else if ( !strcmp( name, "m_VectorParameter0.w" ) )
    {
        m_VectorParameter0.w = value;
    }
    else if ( !strcmp( name, "m_VectorParameter1.x" ) )
    {
        m_VectorParameter1.x = value;
    }
    else if ( !strcmp( name, "m_VectorParameter1.y" ) )
    {
        m_VectorParameter1.y = value;
    }
    else if ( !strcmp( name, "m_VectorParameter1.z" ) )
    {
        m_VectorParameter1.z = value;
    }
    else if ( !strcmp( name, "m_VectorParameter1.w" ) )
    {
        m_VectorParameter1.w = value;
    }
    else if ( !strcmp( name, "m_VectorParameter2.x" ) )
    {
        m_VectorParameter2.x = value;
    }
    else if ( !strcmp( name, "m_VectorParameter2.y" ) )
    {
        m_VectorParameter2.y = value;
    }
    else if ( !strcmp( name, "m_VectorParameter2.z" ) )
    {
        m_VectorParameter2.z = value;
    }
    else if ( !strcmp( name, "m_VectorParameter2.w" ) )
    {
        m_VectorParameter2.w = value;
    }
}
//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewer::ReloadTextureFilePath( const char* name, const char* ftxbFilePath )
{
    if ( !strcmp( name, "nwAlbedoTexture0" ) )
    {
        ReloadTextureFilePath( 0, ftxbFilePath );
    }
    else if ( !strcmp( name, "nwAlbedoTexture1" ) )
    {
        ReloadTextureFilePath( 1, ftxbFilePath );
    }
    else if ( !strcmp( name, "nwAlbedoTexture2" ) )
    {
        ReloadTextureFilePath( 2, ftxbFilePath );
    }
    else if ( !strcmp( name, "nwAlbedoTexture3" ) )
    {
        ReloadTextureFilePath( 3, ftxbFilePath );
    }
    else
    {
        return;
    }
}
//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewer::ReceiveUpdateParameter( const char* receiveData )
{
    int nameSize;
    memcpy( &nameSize, receiveData, sizeof( uint32_t ) );
    receiveData += sizeof( uint32_t );
    char name[ _MAX_PATH ];
    memset( name, 0, nameSize );
    memcpy( name, receiveData, nameSize );
    receiveData += nameSize;

    int valueTypeSize;
    memcpy( &valueTypeSize, receiveData, sizeof( uint32_t ) );
    receiveData += sizeof( uint32_t );
    char valueType[ _MAX_PATH ];
    memset( valueType, 0, valueTypeSize );
    memcpy( valueType, receiveData, valueTypeSize );
    receiveData += valueTypeSize;


    if ( !strcmp( valueType, "file" ) )
    {
        int filePathSize;
        memcpy( &filePathSize, receiveData, sizeof( uint32_t ) );
        receiveData += sizeof( uint32_t );

        char filePath[ _MAX_PATH ];
        memset( filePath, 0, filePathSize );
        memcpy( filePath, receiveData, filePathSize );

        ReloadTextureFilePath( name, filePath );
    }
    else if ( !strcmp( valueType, "float" ) )
    {
        float value;
        memcpy( &value, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        UpdateShaderParameter( name, value );
    }
    else if ( !strcmp( valueType, "vec2" ) )
    {
        float valueX;
        memcpy( &valueX, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        float valueY;
        memcpy( &valueY, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        name[ nameSize - 1 ] = '.';
        name[ nameSize ] = 'x';
        name[ nameSize + 1 ] = '\0';
        UpdateShaderParameter( name, valueX );
        name[ nameSize ] = 'y';
        UpdateShaderParameter( name, valueY );
    }
    else if ( !strcmp( valueType, "vec3" ) )
    {
        float valueX;
        memcpy( &valueX, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        float valueY;
        memcpy( &valueY, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        float valueZ;
        memcpy( &valueZ, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        name[ nameSize - 1 ] = '.';
        name[ nameSize ] = 'x';
        name[ nameSize + 1 ] = '\0';
        UpdateShaderParameter( name, valueX );
        name[ nameSize ] = 'y';
        UpdateShaderParameter( name, valueY );
        name[ nameSize ] = 'z';
        UpdateShaderParameter( name, valueZ );
    }
    else if ( !strcmp( valueType, "vec4" ) )
    {
        float valueX;
        memcpy( &valueX, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        float valueY;
        memcpy( &valueY, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        float valueZ;
        memcpy( &valueZ, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        float valueW;
        memcpy( &valueW, receiveData, sizeof( float_t ) );
        receiveData += sizeof( float_t );

        name[ nameSize - 1 ] = '.';
        name[ nameSize ] = 'x';
        name[ nameSize + 1 ] = '\0';
        UpdateShaderParameter( name, valueX );
        name[ nameSize ] = 'y';
        UpdateShaderParameter( name, valueY );
        name[ nameSize ] = 'z';
        UpdateShaderParameter( name, valueZ );
        name[ nameSize ] = 'w';
        UpdateShaderParameter( name, valueW );
    }
}

//---------------------------------------------------------------------------
//! @brief  ブロックの更新処理を行います。
//---------------------------------------------------------------------------
void CombinerPreviewer::CalcBlock( const nn::util::Matrix4x3fType* viewMtx, float addY )
{
    g_Time += 1;
    g_Time = g_Time % 100;

    m_Mutex.Lock();

    // フラグメントシェーダリロード
    UpdateFragmentShader();

    // テクスチャのリロード
    UpdateTexture();

    // テクスチャのリセット
    if ( m_ResetTextureIndex != CombinerPreviewer_TextureCountMax )
    {
        if ( m_ResetTextureIndex == -1 )
        {
            for ( int i = 0; i < CombinerPreviewer_TextureCountMax; i++ )
            {
                m_Texture[ i ].Finalize( m_pDevice );
            }
        }
        else
        {
            m_Texture[ m_ResetTextureIndex ].Finalize( m_pDevice );
        }

        m_TextureReset = true;
        m_ResetTextureIndex = CombinerPreviewer_TextureCountMax;
    }

    // モデルの差し替え
    if ( strlen( m_OutputModelFilePath ) > 0 )
    {
        m_Model[ CombinerPreviewer_Output ].FinalizeG3dModelBinary();
        m_Model[ CombinerPreviewer_Output ].LoadG3dModelBinary( m_OutputModelFilePath, m_WorkingDir );
        memset( m_OutputModelFilePath, 0, _MAX_PATH );
    }

    m_Mutex.Unlock();

    m_Model[ CombinerPreviewer_Output ].CalcBlock( viewMtx, addY );
    m_Model[ CombinerPreviewer_Preview ].CalcBlock( viewMtx, 0.0f );
}


//---------------------------------------------------------------------------
//! @brief  描画処理を行います。
//---------------------------------------------------------------------------
void CombinerPreviewer::Draw( int idenx, nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::GpuBuffer* pGpuBuffer,
    const nn::util::Matrix4x4fType* projectionMatrix, const nn::util::Matrix4x3fType* viewMatrix, const nn::util::Vector3fType* camPos,
    nn::gfx::DescriptorSlot colorBufferDescSlot, nn::gfx::DescriptorSlot depthBufferDescSlot )
{
    m_TextureReload = false;
    m_TextureReset = false;
    m_PreviewShaderReload = false;
    m_OutputShaderReload = false;

    int slotMaterial = m_Model[ idenx ].GetPixelShaderInterfaceSlot( nn::gfx::ShaderInterfaceType_ConstantBuffer, "Material" );

    // マテリアル
    if ( slotMaterial != -1 )
    {
        nn::util::Float4* pMaterialBuffer = reinterpret_cast< nn::util::Float4* >( pGpuBuffer->Allocate( sizeof( nn::util::Float4 ) * 4 ) );
        if ( pMaterialBuffer )
        {
            nn::gfx::GpuAddress gpuAddress;
            pGpuBuffer->GetGpuAddress( &gpuAddress, pMaterialBuffer );

            if ( idenx == CombinerPreviewer_Output )
            {
                pMaterialBuffer->x = ( float )g_Time / 100.f;
            }
            else
            {
                pMaterialBuffer->x = m_ShaderParameter[ 0 ];
            }
            pMaterialBuffer->y = m_ShaderParameter[ 1 ];
            pMaterialBuffer->z = m_ShaderParameter[ 2 ];
            pMaterialBuffer->w = m_ShaderParameter[ 3 ];
            pMaterialBuffer++;

            pMaterialBuffer->x = m_VectorParameter0.x;
            pMaterialBuffer->y = m_VectorParameter0.y;
            pMaterialBuffer->z = m_VectorParameter0.z;
            pMaterialBuffer->w = m_VectorParameter0.w;
            pMaterialBuffer++;

            pMaterialBuffer->x = m_VectorParameter1.x;
            pMaterialBuffer->y = m_VectorParameter1.y;
            pMaterialBuffer->z = m_VectorParameter1.z;
            pMaterialBuffer->w = m_VectorParameter1.w;
            pMaterialBuffer++;

            pMaterialBuffer->x = m_VectorParameter2.x;
            pMaterialBuffer->y = m_VectorParameter2.y;
            pMaterialBuffer->z = m_VectorParameter2.z;
            pMaterialBuffer->w = m_VectorParameter2.w;

            pCommandBuffer->SetConstantBuffer( slotMaterial, nn::gfx::ShaderStage_Pixel, gpuAddress, sizeof( nn::util::Float4 ) * 4 );
        }
    }

    // テクスチャ設定
    const char* samplerId[] = { "nwAlbedoTexture0", "nwAlbedoTexture1", "nwAlbedoTexture2", "nwAlbedoTexture3" };
    const int samplerCount = sizeof( samplerId ) / sizeof( char* );

    for ( int idxTex = 0; idxTex < samplerCount; ++idxTex )
    {
        const char* name = samplerId[ idxTex ];
        int slotSampler = m_Model[ idenx ].GetPixelShaderInterfaceSlot( nn::gfx::ShaderInterfaceType_Sampler, name );
        if ( slotSampler != -1 )
        {
            if ( m_Texture[ idxTex ].IsReady() )
            {
                pCommandBuffer->SetTextureAndSampler( slotSampler, nn::gfx::ShaderStage_Pixel,
                    m_Texture[ idxTex ].GetDescriptorSlot(), m_SamplerDescriptorSlot );
            }
            else
            {
                pCommandBuffer->SetTextureAndSampler( slotSampler, nn::gfx::ShaderStage_Pixel,
                    m_CommonTexture.GetDescriptorSlot(), m_SamplerDescriptorSlot );

            }
        }
    }

    // カラーバッファテクスチャ設定
    {
        int slotSampler = m_Model[ idenx ].GetPixelShaderInterfaceSlot( nn::gfx::ShaderInterfaceType_Sampler, "nwColorBufferTexture" );
        if ( slotSampler != -1 && colorBufferDescSlot.IsValid() )
        {
            pCommandBuffer->SetTextureAndSampler( slotSampler, nn::gfx::ShaderStage_Pixel, colorBufferDescSlot, m_SamplerDescriptorSlot );
        }
    }

    // デプスバッファテクスチャ設定
    {
        int slotSampler = m_Model[ idenx ].GetPixelShaderInterfaceSlot( nn::gfx::ShaderInterfaceType_Sampler, "nwDepthBufferTexture" );
        if ( slotSampler != -1 && depthBufferDescSlot.IsValid() )
        {
            pCommandBuffer->SetTextureAndSampler( slotSampler, nn::gfx::ShaderStage_Pixel, depthBufferDescSlot, m_SamplerDescriptorSlot );
        }
    }

    // 描画
    m_Model[ idenx ].Draw( pCommandBuffer, pGpuBuffer, projectionMatrix, viewMatrix, camPos );
}


//---------------------------------------------------------------------------
//! @brief  ディスクリプタスロットの更新を行います。
//---------------------------------------------------------------------------
void CombinerPreviewer::UpdateDescriptorSlot( detail::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator,
    detail::SamplerDescriptorIndexAllocator* pSamplerDescriptorIndexAllocator )
{
    for ( int i = 0; i < CombinerPreviewer_TextureCountMax; i++ )
    {
        m_Texture[ i ].UpdateDescriptorSlot( pTextureDescriptorIndexAllocator );
    }

    m_CommonTexture.UpdateDescriptorSlot( pTextureDescriptorIndexAllocator );

    pSamplerDescriptorIndexAllocator->AllocateTransientSamplerDescriptorSlot( &m_SamplerDescriptorSlot, m_Sampler, nullptr );
}


//---------------------------------------------------------------------------
// フラグメントシェーダの更新
//---------------------------------------------------------------------------
void CombinerPreviewer::UpdateFragmentShader()
{
    if ( m_ShaderReset )
    {
        m_Model[ CombinerPreviewer_Output ].UpdateFragmentShader( nullptr );
        m_Model[ CombinerPreviewer_Preview ].UpdateFragmentShader( nullptr );
        m_ShaderReset = false;
        return;
    }

    if ( strlen( m_OutputFshFilePath ) > 0 )
    {
        m_Model[ CombinerPreviewer_Output ].UpdateFragmentShader( m_OutputFshFilePath );
        memset( m_OutputFshFilePath, 0, _MAX_PATH );
        m_OutputShaderReload = true;
    }

    if ( strlen( m_PreviewFshFilePath ) > 0 )
    {
        m_Model[ CombinerPreviewer_Preview ].UpdateFragmentShader( m_PreviewFshFilePath );
        memset( m_PreviewFshFilePath, 0, _MAX_PATH );
        m_PreviewShaderReload = true;
    }
}


//---------------------------------------------------------------------------
// テクスチャの更新
//---------------------------------------------------------------------------
void CombinerPreviewer::UpdateTexture()
{
    if ( m_ReloadTextureIndex != -1 )
    {
        m_Texture[ m_ReloadTextureIndex ].Initialize( m_pDevice, m_TextureFilePath[ m_ReloadTextureIndex ], g_BaseDirectory );
        m_TextureReload = true;

        m_ReloadTextureIndex = -1;
    }
}

//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewer::ReloadTextureFilePath( int index, const char* ftxbFilePath )
{
    if ( index > 4 ) return;
    m_Mutex.Lock();

    if ( strlen( ftxbFilePath ) == 0 )
    {
        ResetTexture( index );
    }
    else
    {
        m_ReloadTextureIndex = index;
        strcpy_s(m_TextureFilePath[m_ReloadTextureIndex], _MAX_PATH, ftxbFilePath);
    }

    m_Mutex.Unlock();
}

//---------------------------------------------------------------------------
// 出力シェーダリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewer::SetReloadOutputShaderFilePath( const char* fshFilePath )
{
    m_Mutex.Lock();
    if ( strlen( fshFilePath ) == 0 )
    {
        ResetShader();
        m_OutputShaderReload = true;
    }
    else
    {
        NN_SDK_ASSERT_RANGE( static_cast< int >( strlen( fshFilePath ) ), 1, _MAX_PATH );
        strncpy_s( m_OutputFshFilePath, fshFilePath, _MAX_PATH );
    }
    m_Mutex.Unlock();
}

//---------------------------------------------------------------------------
// 出力シェーダ描画用モデルの差し替え
//---------------------------------------------------------------------------
bool CombinerPreviewer::LoadG3dModelBinary( const char* pFmdbFilePath, const char* pOutputDir )
{
    m_Mutex.Lock();
    if ( strlen( m_OutputModelFilePath ) == 0 )
    {
        strcpy_s( m_OutputModelFilePath, _MAX_PATH, pFmdbFilePath );
        strcpy_s( m_WorkingDir, _MAX_PATH, pOutputDir );
    }
    m_Mutex.Unlock();

    return true;
}

//---------------------------------------------------------------------------
// シェーダを初期状態にリセットします
//---------------------------------------------------------------------------
void CombinerPreviewer::ResetShader()
{
    m_ShaderReset = true;
}

//---------------------------------------------------------------------------
// ノード経過ビューシェーダリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewer::SetReloadPreviewShaderFilePath( const char* fshFilePath )
{
    m_Mutex.Lock();
    NN_SDK_ASSERT_RANGE( static_cast< int >( strlen( fshFilePath ) ), 1, _MAX_PATH );
    strncpy_s( m_PreviewFshFilePath, fshFilePath, _MAX_PATH );
    m_Mutex.Unlock();
}

//---------------------------------------------------------------------------
//! @brief  コンバイナプレビューモデルの生成。
//---------------------------------------------------------------------------
void CombinerPreviewer::InitializeCombinerPreviewer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pHeap )
{
    void* buffer = pHeap->Alloc( sizeof( CombinerPreviewer ) );
    g_pCombinerPreviewer = new ( buffer ) CombinerPreviewer();

    g_pCombinerPreviewer->Initialize( pDevice, pHeap );
    g_pCombinerPreviewer->InitializeG3dModel();
}

//---------------------------------------------------------------------------
//! @brief  コンバイナプレビューモデルの破棄。
//---------------------------------------------------------------------------
void CombinerPreviewer::FinalizeCombinerPreviewer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pHeap )
{
    NN_UNUSED( pDevice );
    g_pCombinerPreviewer->~CombinerPreviewer();

    pHeap->Free( g_pCombinerPreviewer );
}

#endif

CombinerPreviewer* CombinerPreviewer::g_pCombinerPreviewer = nullptr;


} // namespace detail
