﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <Camera.h>

static void     AngleToVec_( nn::util::Vector3fType* pOutLookVec, nn::util::Vector3fType* pOutUpVec, float angleH, float angleV );
static void     VecToAngle_( const nn::util::Vector3fType* vec, float* pOutAngleH, float* pOutAngleV, float* pOutDistance );

//---------------------------------------------------------------------------
//! @briefprivate        sin, cos の値より角度を取得する関数
//!
//! @param[in]    sin     sin 値
//! @param[in]    cos     cos 値
//!
//! @return       2πを 0x100000000 とする値
//---------------------------------------------------------------------------
static inline float GetAngle_( float sin, float cos )
{
    float angle = nn::util::AsinEst( sin );

    if ( cos < 0 )
    {
        angle = nn::util::FloatPi - angle;
    }

    return angle;
}

//---------------------------------------------------------------------------
Camera::Camera()
    : m_Distance    ( 4.0f )
    , m_AngleV      ( 0.0f )
    , m_AngleH      ( 0.0f )
    , m_Twist       ( 0.0f )
    , m_DistanceMax ( 1000.0f )
    , m_DistanceMin ( 0.1f )
    , m_TwistReset  ( 0.0f )
    , m_RecalcFlg   ( RECALC_OBJECT )
{
    nn::util::VectorSet( &m_LookAtPos, 0.0f, 0.0f, 0.0f );
    nn::util::VectorSet( &m_Pos, 1.0f, 0.0f, 0.0f );
}


//---------------------------------------------------------------------------
void Camera::SetDirection( Camera* dirObj )
{
    NN_SDK_ASSERT_NOT_NULL( dirObj );
    dirObj->Recalc_();

    // アングルと位置を設定する
    m_AngleV     = dirObj->m_AngleV;
    m_AngleH     = dirObj->m_AngleH;
    m_Pos        = dirObj->m_Pos;
    m_LookAtPos  = dirObj->m_LookAtPos;
    m_Distance   = dirObj->m_Distance;
    m_LookVec    = dirObj->m_LookVec;
    m_UpVec      = dirObj->m_UpVec;
    m_RecalcFlg  = RECALC_NONE;
}

//---------------------------------------------------------------------------
void Camera::SetPos( float x, float y, float z )
{
    RecalcIf_( RECALC_TARGET );

    nn::util::VectorSet( &m_Pos, x, y, z );

    m_RecalcFlg   = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
void Camera::SetPos( const nn::util::Vector3fType& vec )
{
    RecalcIf_( RECALC_TARGET );

    m_Pos  = vec;

    m_RecalcFlg  = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
const nn::util::Vector3fType* Camera::GetPos() const
{
    RecalcIf_( RECALC_OBJECT );
    return &m_Pos;
}


//---------------------------------------------------------------------------
void Camera::GetPos( nn::util::Vector3fType* pos ) const
{
    NN_SDK_ASSERT_NOT_NULL( pos );

    *pos = *(this->GetPos());
}


//---------------------------------------------------------------------------
void Camera::SetLookAtPos( float x, float y, float z )
{
    RecalcIf_( RECALC_OBJECT );

    nn::util::VectorSet( &m_LookAtPos, x, y, z );

    m_RecalcFlg   = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
void Camera::SetLookAtPos( const nn::util::Vector3fType& vec )
{
    RecalcIf_( RECALC_OBJECT );

    m_LookAtPos = vec;

    m_RecalcFlg = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
const nn::util::Vector3fType* Camera::GetLookAtPos() const
{
    RecalcIf_( RECALC_TARGET );

    return &m_LookAtPos;
}

//---------------------------------------------------------------------------
void Camera::GetLookAtPos( nn::util::Vector3fType* pPos ) const
{
    NN_SDK_ASSERT_NOT_NULL( pPos );

    *pPos = *(this->GetLookAtPos());
}


//---------------------------------------------------------------------------
void Camera::SetUpVec( float x, float y, float z )
{
    RecalcIf_( RECALC_OBJECT );

    nn::util::VectorSet( &m_UpVec, x, y, z );

//    m_RecalcFlg   = RECALC_ANGLE;        // アングルを再計算
//    ClampDistance_();
}


//---------------------------------------------------------------------------
void Camera::SetUpVec( const nn::util::Vector3fType& vec )
{
    RecalcIf_( RECALC_OBJECT );
    m_UpVec = vec;
//    m_RecalcFlg = RECALC_ANGLE;        // アングルを再計算
//    ClampDistance_();
}


//---------------------------------------------------------------------------
const nn::util::Vector3fType* Camera::GetUpVec() const
{
    RecalcIf_( RECALC_OBJECT );
    RecalcIf_( RECALC_TARGET );

    return &m_UpVec;
}

//---------------------------------------------------------------------------
void Camera::GetUpVec( nn::util::Vector3fType* pVec ) const
{
    NN_SDK_ASSERT_NOT_NULL( pVec );

    *pVec = *(this->GetLookAtPos());
}

//---------------------------------------------------------------------------
float Camera::GetDistance() const
{
    RecalcIf_( RECALC_ANGLE );

    return m_Distance;
}


//---------------------------------------------------------------------------
void Camera::SetDistance( float dist )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    m_Distance = dist;
    if ( m_Distance > m_DistanceMax )
    {
        m_Distance = m_DistanceMax;
    }
    else if ( m_Distance < m_DistanceMin )
    {
        m_Distance = m_DistanceMin;
    }
    m_RecalcFlg = RECALC_OBJECT;
}


//---------------------------------------------------------------------------
void Camera::AddDistanceByMovingObj( float length )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    m_Distance += length;

    if ( m_Distance > m_DistanceMax )
    {
        m_Distance = m_DistanceMax;
    }
    else if ( m_Distance < m_DistanceMin )
    {
        m_Distance = m_DistanceMin;
    }

    m_RecalcFlg = RECALC_OBJECT;
}


//---------------------------------------------------------------------------
void Camera::AddDistanceByMovingTarget( float length )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_OBJECT );

    m_Distance += length;

    if ( m_Distance > m_DistanceMax )
    {
        m_Distance = m_DistanceMax;
    }
    else if ( m_Distance < m_DistanceMin )
    {
        m_Distance = m_DistanceMin;
    }

    m_RecalcFlg = RECALC_TARGET;
}


//---------------------------------------------------------------------------
void Camera::Move( float x, float y, float z )
{
    // NOTE:
    //   並行移動なのでアングルを再計算する必要はありません。
    //   また再計算フラグは保存されるので、
    //   座標値の加算前に再計算しておく必要もありません。

    nn::util::Vector3fType v;
    nn::util::VectorSet( &v, x, y, z );
    nn::util::VectorAdd( &m_Pos, m_Pos, v );
    nn::util::VectorAdd( &m_LookAtPos, m_LookAtPos, v );
}

//---------------------------------------------------------------------------
void Camera::Rotate( float angleV, float angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_OBJECT );

    SetAngleV_( m_AngleV + angleV );
    SetAngleH_( m_AngleH + angleH );

    m_RecalcFlg = RECALC_TARGET;
}

//---------------------------------------------------------------------------
void Camera::Revolve( float angleV, float angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    SetAngleV_( m_AngleV + angleV );
    SetAngleH_( m_AngleH + angleH );

    m_RecalcFlg = RECALC_OBJECT;
}

//---------------------------------------------------------------------------
void Camera::Rotate360( float angleV, float angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_OBJECT );

    m_AngleV += static_cast<uint32_t>(angleV);
    m_AngleH += static_cast<uint32_t>(angleH);

    m_RecalcFlg = RECALC_TARGET;
}


//---------------------------------------------------------------------------
void Camera::Revolve360( float angleV, float angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    m_AngleV += angleV;
    m_AngleH += angleH;

    m_RecalcFlg = RECALC_OBJECT;
}

//---------------------------------------------------------------------------
void Camera::RecalcIf_( uint32_t recalcFlg ) const
{
    if ( m_RecalcFlg != recalcFlg )
    {
        return;
    }

    Recalc_();
}

//---------------------------------------------------------------------------
void Camera::Recalc_() const
{
    if ( m_RecalcFlg == RECALC_NONE )
    {
        return;
    }

    switch ( m_RecalcFlg )
    {
    //---------------------------------------------
    // ターゲット座標と角度からカメラ座標を求める
    case RECALC_OBJECT:
        RecalcPos_();
        break;
    //---------------------------------------------
    // カメラ座標と角度からターゲット座標を求める
    case RECALC_TARGET:
        RecalcLookAtPos_();
        break;
    //---------------------------------------------
    // カメラ座標とターゲット座標から角度と距離を求める
    case RECALC_ANGLE:
        RecalcAngle_();
        break;
    }

    //---------------------------------------------
    // ツイストを再計算後の姿勢に対して適用する
    if (m_Twist != 0.0f)
    {
        nn::util::Matrix4x3fType rotMtx;
        nn::util::MatrixIdentity( &rotMtx );
        nn::util::Vector3fType rotRad;
        nn::util::VectorSet(&rotRad, 0.0f, 0.0f, m_Twist);
        nn::util::MatrixSetRotateXyz( &rotMtx, rotRad );

        nn::util::VectorTransform( &m_UpVec, m_UpVec, rotMtx );
    }

    m_RecalcFlg = RECALC_NONE;
}

//---------------------------------------------------------------------------
void Camera::RecalcAngle_() const
{
    nn::util::VectorSubtract( &m_LookVec, m_Pos, m_LookAtPos );
    VecToAngle_( &m_LookVec, &m_AngleH, &m_AngleV, &m_Distance );
    SetAngleV_( m_AngleV );
    AngleToVec_( nullptr, &m_UpVec, m_AngleH, m_AngleV );
}


//---------------------------------------------------------------------------
void Camera::RecalcPos_() const
{
    AngleToVec_( &m_LookVec, &m_UpVec, m_AngleH, m_AngleV );

    nn::util::Vector3fType v;
    nn::util::VectorMultiply( &v, m_LookVec, m_Distance );
    nn::util::VectorAdd( &m_Pos, m_LookAtPos, v );
}


//---------------------------------------------------------------------------
void Camera::RecalcLookAtPos_() const
{
    AngleToVec_( &m_LookVec, &m_UpVec, m_AngleH, m_AngleV );

    nn::util::Vector3fType v;
    nn::util::VectorMultiply( &v, m_LookVec, m_Distance );
    nn::util::VectorSubtract( &m_LookAtPos, m_Pos, v );
}


//---------------------------------------------------------------------------
void Camera::Dump() const
{
    NN_SDK_LOG("(%f, %f, %f) -> (%f, %f, %f)\n", nn::util::VectorGetX( m_Pos ), nn::util::VectorGetY( m_Pos ), nn::util::VectorGetZ( m_Pos ),
                                                 nn::util::VectorGetX( m_LookAtPos ), nn::util::VectorGetY( m_LookAtPos ), nn::util::VectorGetZ( m_LookAtPos ) );
    NN_SDK_LOG( "angleH = 0x%x, angleV = 0x%x\n", m_AngleH, m_AngleV );
}


//---------------------------------------------------------------------------
//! @brief        水平垂直のアングル情報からベクトル値への変換を行ないます。
//!
//! @param[out]   lookVec 視線べクトルが返されるポインタ
//! @param[out]   upVec   アップベクトルが返されるポインタ
//! @param[in]    angleH  水平角度
//! @param[in]    angleV  垂直角度
//---------------------------------------------------------------------------
static void AngleToVec_( nn::util::Vector3fType* pOutLookVec, nn::util::Vector3fType* pOutUpVec, float angleH, float angleV )
{
    float sinV, cosV, sinH, cosH;
    nn::util::SinCosEst( &sinV, &cosV, angleV );
    nn::util::SinCosEst( &sinH, &cosH, angleH );

    if ( pOutLookVec != nullptr )
    {
        nn::util::VectorSet( pOutLookVec, -( cosV * sinH ), -( sinV ), -( cosV * cosH ) );
    }

    if ( pOutUpVec != nullptr )
    {
        nn::util::VectorSet( pOutUpVec, -( sinV * sinH ), ( cosV ), -( sinV * cosH ) );
    }
}


//---------------------------------------------------------------------------
//! @brief        ベクトルデータから水平垂直角度への変換を行ないます。
//!
//! @param[in]    vec      視線ベクトル
//! @param[out]   angleH   計算された水平角度が返されるポインタ
//! @param[out]   angleV   計算された垂直角度が返されるポインタ
//! @param[out]   distance 計算されたベクトル距離が返されるポインタ
//---------------------------------------------------------------------------
static void VecToAngle_( const nn::util::Vector3fType* pVec, float* pOutAngleH, float* pOutAngleV, float* pOutDistance )
{
    // ベクトルの長さを取得
    *pOutDistance = nn::util::VectorLength( *pVec );

    // 水平方向の角度を計算
    nn::util::Vector3fType srcVec, dstVec;

    nn::util::VectorMultiply( &srcVec, *pVec, -1.0f );
    nn::util::VectorSetY( &srcVec, 0 );

    if ( nn::util::VectorGetX( srcVec ) != 0 || nn::util::VectorGetZ( srcVec ) != 0 )
    {
        nn::util::VectorNormalize( &dstVec, srcVec );
        *pOutAngleH = GetAngle_( nn::util::VectorGetX( dstVec ), nn::util::VectorGetZ( dstVec ) );
    }

    // 垂直方向の角度を計算
    float hDistance = nn::util::VectorLength( srcVec );
    nn::util::VectorSet( &srcVec, hDistance, -nn::util::VectorGetY( *pVec ), 0 );

    if ( nn::util::VectorGetX( srcVec ) != 0 || nn::util::VectorGetY( srcVec ) != 0 )
    {
        nn::util::VectorNormalize( &dstVec, srcVec );
        *pOutAngleV = GetAngle_( nn::util::VectorGetY( dstVec ), nn::util::VectorGetX( dstVec ) );
    }
}


//---------------------------------------------------------------------------
void Camera::Reset()
{
    m_Pos       = m_PosReset;
    m_LookAtPos = m_LookAtPosReset;
    m_Twist     = m_TwistReset;
    m_RecalcFlg = RECALC_ANGLE;
}

//---------------------------------------------------------------------------
void Camera::Preset()
{
    RecalcIf_( RECALC_OBJECT );
    RecalcIf_( RECALC_TARGET );

    m_PosReset       = m_Pos;
    m_LookAtPosReset = m_LookAtPos;
    m_TwistReset     = m_Twist;
}

//---------------------------------------------------------------------------
void Camera::PresetTwist()
{
    m_TwistReset = m_Twist;
}

//---------------------------------------------------------------------------
void Camera::ClampDistance_()
{
    RecalcIf_( RECALC_TARGET );
    RecalcIf_( RECALC_OBJECT );

    nn::util::VectorSubtract( &m_LookVec, m_Pos, m_LookAtPos );

    float distance = nn::util::VectorLength( m_LookVec );

    if ( distance < m_DistanceMin )
    {
        if ( distance == 0.0f )
        {
            m_AngleV = 0;
            m_AngleH = 0;
        }
        else
        {
            RecalcIf_( RECALC_ANGLE );
        }
        m_Distance   = m_DistanceMin;
        m_RecalcFlg = RECALC_OBJECT;
    }
    else if ( distance > m_DistanceMax )
    {
        RecalcIf_( RECALC_ANGLE );
        m_Distance   = m_DistanceMax;
        m_RecalcFlg = RECALC_OBJECT;
    }
}

//---------------------------------------------------------------------------
void Camera::MoveViewHorizontal( float right, float forward, const Camera& camera )
{
    // カメラの水平視線ベクトルを元に移動する
    float viewAngleH = static_cast<float>( camera.GetAngleH() );
    Recalc_();

    float sinH, cosH;
    nn::util::SinCosEst( &sinH, &cosH, viewAngleH );
    nn::util::Vector3fType vUp   = NN_UTIL_VECTOR_3F_INITIALIZER( 0.0f, 1.0f, 0.0f );
    nn::util::Vector3fType vLook = NN_UTIL_VECTOR_3F_INITIALIZER( sinH, 1.0f, cosH );

    nn::util::Vector3fType vRight;
    nn::util::VectorCross( &vRight, vUp, vLook );

    nn::util::Vector3fType moveVec;
    nn::util::Vector3fType v1;
    nn::util::Vector3fType v2;
    nn::util::VectorMultiply( &v1, vRight, right );
    nn::util::VectorMultiply( &v2, vLook, forward );
    nn::util::VectorAdd( &moveVec, v1, v2 );
    nn::util::VectorSetY( &moveVec, 0 );

    nn::util::VectorAdd( &m_LookAtPos, m_LookAtPos, moveVec );
    nn::util::VectorAdd( &m_Pos, m_Pos, moveVec );
}


//---------------------------------------------------------------------------
void Camera::MoveView( float right, float forward, float up, const Camera& camera )
{
    // カメラの水平視線ベクトルを元に移動する
    float viewAngleH = static_cast<float>( camera.GetAngleH() );
    float viewAngleV = static_cast<float>( camera.GetAngleV() );

    float sinH, cosH;
    float sinV, cosV;
    nn::util::Vector3fType moveVec;

    Recalc_();

    nn::util::SinCosEst( &sinH, &cosH, viewAngleH );
    nn::util::SinCosEst( &sinV, &cosV, viewAngleV );

    nn::util::Vector3fType vLook = NN_UTIL_VECTOR_3F_INITIALIZER( sinH * cosV,  sinV, cosH * cosV  );
    nn::util::Vector3fType vUp   = NN_UTIL_VECTOR_3F_INITIALIZER( -sinH * sinV, cosV, -cosH * sinV );

    nn::util::Vector3fType vRight;
    nn::util::VectorCross( &vRight, vUp, vLook );

    nn::util::Vector3fType v[3];
    nn::util::VectorMultiply( &v[0], vRight, right );
    nn::util::VectorMultiply( &v[1], vLook, forward );
    nn::util::VectorMultiply( &v[2], vUp, up );
    nn::util::VectorAdd( &moveVec, v[0], v[1] );
    nn::util::VectorAdd( &moveVec, moveVec, v[2] );

    nn::util::VectorAdd( &m_LookAtPos, m_LookAtPos, moveVec );
    nn::util::VectorAdd( &m_Pos, m_Pos, moveVec );
}


//---------------------------------------------------------------------------
void Camera::MoveByEvent( const Event& refEvent, const Camera* pCamera /* = nullptr */ )
{
    float distance = this->GetDistance();
    Event event = refEvent;

    if (pCamera == nullptr) { pCamera = this; }

    // 初期位置にリセット
    if ( event.Flags() & Event::POSITION_RESET )
    {
        this->Reset();
    }

    // REVERSE_ON_VIEW_CAMERAが有効でビューカメラの場合には
    // 移動方向を反転する。
    if ( event.Flags() & Event::REVERSE_ON_VIEW_CAMERA )
    {
        if ( this == pCamera )
        {
            event.SetRotateH( -event.RotateH() );
            event.SetRotateV( -event.RotateV() );

            nn::util::Vector3fType move;
            nn::util::VectorMultiply( &move, event.Move(), -1.0f );
            event.SetMove( move );
        }
    }

    // 回転
    if ( event.RotateH() != 0.0f || event.RotateV() != 0.0f )
    {
        float rotV = event.RotateV() * 0.1f;
        float rotH = event.RotateH() * 0.1f;

        if ( event.Flags() & Event::ROTATE_BASED_ON_TARGET )
        {
            if ( event.Flags() & Event::ROTATE_VERTICAL_360 )
            {
                // 注視点を中心に360度公転
                this->Revolve360( rotV, rotH );
            }
            else
            {
                // 注視点を中心に公転
                this->Revolve( rotV, rotH );
            }
        }
        else
        {
            if ( event.Flags() & Event::ROTATE_VERTICAL_360 )
            {
                // オブジェクト中心に360度自転
                this->Rotate360( -rotV, -rotH );
            }
            else
            {
                // オブジェクト中心に自転
                this->Rotate( -rotV, -rotH );
            }
        }
    }

    // 並行移動
    const float MOVE_SPEED = 60.0f / 0x800; // 体感でなんとなくこの値

    if ( event.Flags() & Event::MOVE_ALONG_AXIS )
    {
        if ( event.MoveRight() != 0.0f || event.MoveUp() != 0.0f || event.MoveForward() != 0.0f )
        {
            float moveX = - distance * event.MoveRight()   * MOVE_SPEED;
            float moveY =   distance * event.MoveUp()      * MOVE_SPEED;
            float moveZ = - distance * event.MoveForward() * MOVE_SPEED;
            this->Move( moveX, moveY, moveZ );
        }
    }
    else if ( event.Flags() & Event::MOVE_ALONG_VIEW )
    {
        if ( event.MoveForward() != 0.0f || event.MoveRight() != 0.0f || event.MoveUp() != 0.0f )
        {
            float moveForward = distance * event.MoveForward() * MOVE_SPEED;
            float moveRight   = distance * event.MoveRight()   * MOVE_SPEED;
            float moveUp      = distance * event.MoveUp()      * MOVE_SPEED;
            this->MoveView( moveRight, moveForward, moveUp, *pCamera );
        }
    }
    else
    {
        if ( event.MoveForward() != 0.0f || event.MoveRight() != 0.0f )
        {
            float moveForward = distance * event.MoveForward() * MOVE_SPEED;
            float moveRight   = distance * event.MoveRight()   * MOVE_SPEED;

            this->MoveViewHorizontal( moveRight, moveForward, *pCamera );
        }
        if ( event.MoveUp() != 0.0f )
        {
            float moveY = distance * event.MoveUp() * MOVE_SPEED;
            this->Move( 0, moveY, 0 );
        }
    }

    // 距離の拡縮
    if ( event.Zoom() != 0.0f )
    {
        float expand = distance * event.Zoom() * MOVE_SPEED;
        this->AddDistanceByMovingObj( expand );
    }

    // ツイスト
    if ( event.Twist() != 0.0f )
    {
        const float TWIST_SPEED = 360.0f / 100; // 100フレームで一回転程度
        float twist = this->GetTwist();
        twist += event.Twist() * TWIST_SPEED;
        this->SetTwist( twist );
    }
}


//---------------------------------------------------------------------------
void Camera::GetTargetMtx_( nn::util::Matrix4x3fType* pMatrix ) const
{
    NN_SDK_ASSERT( pMatrix != nullptr );
    Recalc_();
    nn::util::MatrixLookAtRightHanded( pMatrix, m_Pos, m_LookAtPos, m_UpVec );
}


//---------------------------------------------------------------------------
void Camera::MakeCameraEventClassic( Event* pEvents, nns::nac::Pad* pPad, nns::nac::Mouse* pMouse, bool isMouseAvailable, bool isAltModified, float frameSpeed)
{
    NN_SDK_ASSERT_NOT_NULL( pPad );

    float move_rate = frameSpeed;
    bool mouseAvailable = ( pMouse && isMouseAvailable );

    const float PAD_STICK_RATE = ( pPad->IsNpadDataAvailable() ) ? 30.f : 100.0f;
    const float MOUSE_MOVE_RATE = 4.0f;
    const float MOUSE_ZOOM_RATE = 0.1f;
    const float WHEEL_ZOOM_RATE = 1.0f;

    // カメラのリセット
    if ( pPad->IsHold( nns::nac::Pad::MASK_START ) )
    {
        pEvents->EnableFlags( Camera::Event::POSITION_RESET );
    }

    // メインスティック/マウス左ドラッグ
    const int8_t STICK_MAX = 56;
    const int8_t SUBSTICK_MAX = 44;
    float stickX = 0.f;
    float stickY = 0.f;

    if ( mouseAvailable && pMouse->IsHold( nns::nac::Mouse::MASK_LBUTTON ) )
    {
        // マウス操作
        if ( isAltModified )
        {
            stickX = - pMouse->GetPointerDiffX() * MOUSE_MOVE_RATE;
            stickY = - pMouse->GetPointerDiffY() * MOUSE_MOVE_RATE;
        }
    }
    else
    {
        // パッド操作
        stickX =   pPad->GetLeftStickX() * PAD_STICK_RATE;
        stickY = - pPad->GetLeftStickY() * PAD_STICK_RATE;
    }

    if ( stickX != 0.f || stickY != 0.f )
    {
        // 注視点を中心に公転
        pEvents->EnableFlags( Camera::Event::ROTATE_BASED_ON_TARGET );
        pEvents->SetRotateH( stickX / STICK_MAX * move_rate );
        pEvents->SetRotateV( stickY / STICK_MAX * move_rate );
    }

    // サブスティック/マウス中ドラッグ
    float substickX = 0.f;
    float substickY = 0.f;

    if ( mouseAvailable && pMouse->IsHold( nns::nac::Mouse::MASK_MBUTTON ) )
    {
        // マウス操作
        if ( isAltModified )
        {
            substickX = pMouse->GetPointerDiffX() * MOUSE_MOVE_RATE / 3.f;
            substickY = pMouse->GetPointerDiffY() * MOUSE_MOVE_RATE / 3.f;
        }
    }
    else
    {
        // パッド操作
        substickX = - pPad->GetRightStickX() * PAD_STICK_RATE;
        substickY =   pPad->GetRightStickY() * PAD_STICK_RATE;
    }

    if ( substickX != 0.f || substickY != 0.f )
    {
        if ( pPad->IsHold( nns::nac::Pad::MASK_L )
        || ( pMouse && pMouse->IsHold( nns::nac::Mouse::MASK_MBUTTON ) /* && !isCtrlModified */ ) )
        {
            // カメラ平面水平移動
            pEvents->SetMoveRight( substickX / SUBSTICK_MAX * move_rate );
            pEvents->SetMoveUp( substickY / SUBSTICK_MAX * move_rate );

            pEvents->EnableFlags( Camera::Event::MOVE_ALONG_VIEW );
        }
        else
        {
            // XZ平面水平移動
            pEvents->SetMoveRight( substickX / SUBSTICK_MAX * move_rate );
            pEvents->SetMoveForward( substickY / SUBSTICK_MAX * move_rate );
        }
    }

    // ズーム
    if ( ! pPad->IsHold( nns::nac::Pad::MASK_L ) )
    {
        if ( ( pPad->IsHold( nns::nac::Pad::MASK_X ) && pPad->IsHold( nns::nac::Pad::MASK_Y ) ) ||
             ( pPad->IsHold( nns::nac::Pad::MASK_UP ) && pPad->IsHold( nns::nac::Pad::MASK_LEFT ) ) )
        {
            // 両方押されている場合は何もしない。
        }
        else if ( pPad->IsHold( nns::nac::Pad::MASK_Y ) || pPad->IsHold( nns::nac::Pad::MASK_LEFT ) )
        {
            // パッドズームイン操作
            pEvents->EnableFlags( Camera::Event::EXPAND_BASED_ON_TARGET );
            pEvents->SetZoom( move_rate );
        }
        else if ( pPad->IsHold( nns::nac::Pad::MASK_X ) || pPad->IsHold( nns::nac::Pad::MASK_UP ) )
        {
            // パッドズームアウト操作
            pEvents->EnableFlags( Camera::Event::EXPAND_BASED_ON_TARGET );
            pEvents->SetZoom( -move_rate );
        }
        else if ( mouseAvailable && pMouse->GetWheel() != 0 )
        {
            // マウス操作（ホイール回転）
            pEvents->EnableFlags( Camera::Event::EXPAND_BASED_ON_TARGET );
            pEvents->SetZoom( -move_rate * WHEEL_ZOOM_RATE * pMouse->GetWheel() );
        }
        else if ( mouseAvailable && pMouse->IsHold( nns::nac::Mouse::MASK_RBUTTON ) )
        {
            // マウス操作（右ボタン）
            if ( isAltModified )
            {
                float differenceX = pMouse->GetPointerDiffX();
                float differenceY = pMouse->GetPointerDiffY();
                float distance = ( ::std::fabsf(differenceX) - ::std::fabsf(differenceY) >= 0 ) ? differenceX : differenceY;

                pEvents->EnableFlags( Camera::Event::EXPAND_BASED_ON_TARGET );
                pEvents->SetZoom( -move_rate * MOUSE_ZOOM_RATE * distance );
            }
        }
    }
}

void Camera::UpdateCamera( nns::nac::Pad* pPad, nns::nac::Mouse* pMouse, bool isMouseAvailable, bool isAltModified )
{
    Event events;

    MakeCameraEventClassic( &events, pPad, pMouse, isMouseAvailable, isAltModified );
    MoveByEvent( events );
}
