﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d.h>
#include <nns/gfx/gfx_GpuBuffer.h>
#include <nns/nac/nac_MemoryAllocator.h>

namespace detail      {

//---------------------------------------------------------------------------
//! @briefprivate   Matrix4x4fType <- Matrix4x3fType の簡略化。
//! @param[out]     pOutMatrix  出力用 nn::util::Matrix4x4fType
//! @param[in]      source      入力となる nn::util::Matrix4x3fType
//---------------------------------------------------------------------------
inline void Matrix4x3fTo4x4f( nn::util::Matrix4x4fType* pOutMatrix, const nn::util::Matrix4x3fType& source )
{
    nn::util::Vector3fType axisX, axisY, axisZ, axisW;
    nn::util::MatrixGetAxisX( &axisX, source );
    nn::util::MatrixGetAxisY( &axisY, source );
    nn::util::MatrixGetAxisZ( &axisZ, source );
    nn::util::MatrixGetAxisW( &axisW, source );

    nn::util::MatrixSet( pOutMatrix,
        nn::util::VectorGetX( axisX ), nn::util::VectorGetY( axisX ), nn::util::VectorGetZ( axisX ), 0,
        nn::util::VectorGetX( axisY ), nn::util::VectorGetY( axisY ), nn::util::VectorGetZ( axisY ), 0,
        nn::util::VectorGetX( axisZ ), nn::util::VectorGetY( axisZ ), nn::util::VectorGetZ( axisZ ), 0,
        nn::util::VectorGetX( axisW ), nn::util::VectorGetY( axisW ), nn::util::VectorGetZ( axisW ), 1 );
}

#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------
//! @brief テクスチャクラスです。
//------------------------------------------------
class Texture
{
    NN_DISALLOW_COPY( Texture );
public:
    Texture() NN_NOEXCEPT;

    ~Texture() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  初期化を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, const char* pFtxbFilePath,const char* pOutputDir ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  初期化を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, void* pBntxBinary ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ディスクリプタスロットの更新を行います。
    //---------------------------------------------------------------------------
    void UpdateDescriptorSlot( detail::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator );

    //---------------------------------------------------------------------------
    //! @brief  破棄を行います。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    nn::gfx::DescriptorSlot GetDescriptorSlot() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( m_pResTextureFile[ m_TextureFace ] );
        return m_DescriptorSlot[ m_TextureFace ];
    }

    //---------------------------------------------------------------------------
    //! @brief テクスチャが利用可能な状態か確認します。
    //---------------------------------------------------------------------------
    bool IsReady() const NN_NOEXCEPT
    {
        return m_DescriptorSlot[ m_TextureFace ].IsValid();
    }

    //---------------------------------------------------------------------------
    //! @brief テクスチャを利用不可状態に設定します。
    //---------------------------------------------------------------------------
    void Invalidate() NN_NOEXCEPT
    {
        return m_DescriptorSlot[ m_TextureFace ].Invalidate();
    }


private:
    nn::gfx::Device*            m_pDevice;
    int                         m_TextureFace;
    nn::gfx::ResTextureFile*    m_pResTextureFile[ 2 ];
    nn::gfx::DescriptorSlot     m_DescriptorSlot[ 2 ];
    uint8_t*                    m_TextureBinaryPtr[ 2 ];
};


//------------------------------------------------
//! @brief  Shader
//------------------------------------------------
class Shader
{
    NN_DISALLOW_COPY( Shader );
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    Shader() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    ~Shader() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  初期化を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, nn::gfx::ShaderStage shaderStage, const char* shaderCode ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  破棄を行います。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  シェーダを取得します。
    //---------------------------------------------------------------------------
    nn::gfx::Shader* GetGfxShader() NN_NOEXCEPT
    {
        return &m_Shader[ m_ShaderFace ];
    }

private:
    nn::gfx::Device*            m_pDevice;
    int                         m_ShaderFace;
    nn::gfx::Shader             m_Shader[ 2 ];
    bool                        m_IsInitialized[ 2 ];
};



//------------------------------------------------
//! @brief  Model
//------------------------------------------------
class Model
{
    NN_DISALLOW_COPY( Model );
public:

    struct ViewBlock
    {
        nn::util::FloatColumnMajor4x4  projectionMatrix;
        nn::util::FloatColumnMajor4x4  viewMatrix;
        nn::util::Float4    cameraPosition;
        nn::util::Float4    cameraVector;

        void Set(
            const nn::util::Matrix4x3fType& view,
            const nn::util::Matrix4x4fType& projection,
            const nn::util::Vector3fType&   cameraPos ) NN_NOEXCEPT
        {
            // Projection Matrix
            nn::util::MatrixStore( &projectionMatrix, projection );

            // View Matrix(3x4から4x3に変換)
            nn::util::Matrix4x4fType tempView4x4f;
            nn::util::Matrix4x4fType tempViewT4x4f;
            detail::Matrix4x3fTo4x4f( &tempView4x4f, view );

            // View Matrix
            nn::util::MatrixStore( &viewMatrix, tempView4x4f );

            // Look Direction
            nn::util::MatrixTranspose( &tempViewT4x4f, tempView4x4f );
            cameraVector.x = tempViewT4x4f._m[ 0 ][ 2 ];
            cameraVector.y = tempViewT4x4f._m[ 1 ][ 2 ];
            cameraVector.z = tempViewT4x4f._m[ 2 ][ 2 ];
            cameraVector.w = 0;

            // Eye Position
            nn::util::VectorStore( cameraPosition.v, cameraPos );
            cameraPosition.w = 0;
        }
    };

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    Model() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    ~Model() NN_NOEXCEPT
    {
        m_pG3dModelResFile->Cleanup( m_pDevice );
    }

    //---------------------------------------------------------------------------
    //! @brief  初期化を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator*  pHeap ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  破棄を行います。
    //---------------------------------------------------------------------------
    void Finalize() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  フラグメントシェーダの更新
    //---------------------------------------------------------------------------
    void UpdateFragmentShader( const char* fshFilePath );


    int GetVertexShaderInterfaceSlot( nn::gfx::ShaderInterfaceType shaderInterfaceType, const char* slotName )
    {
        return m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, shaderInterfaceType, slotName );
    }

    int GetPixelShaderInterfaceSlot( nn::gfx::ShaderInterfaceType shaderInterfaceType, const char* slotName )
    {
        return m_FragmentShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, shaderInterfaceType, slotName );
    }

    //---------------------------------------------------------------------------
    //! @brief  モデルバイナリをセットアップします。
    //---------------------------------------------------------------------------
    bool SetupG3dModelBinary( void* pBfresG3dBinary );

    //---------------------------------------------------------------------------
    //! @brief  モデルファイルをロードします。
    //---------------------------------------------------------------------------
    bool LoadG3dModelBinary( const char* pFmdbFilePath, const char* pOutputDir );

    //---------------------------------------------------------------------------
    //! @brief  G3dモデル破棄を行います。
    //---------------------------------------------------------------------------
    void FinalizeG3dModelBinary() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ブロックの更新処理を行います。
    //---------------------------------------------------------------------------
    void CalcBlock( const nn::util::Matrix4x3fType* viewMtx, float addRotateY );

    //---------------------------------------------------------------------------
    //! @brief  描画処理を行います。
    //---------------------------------------------------------------------------
    void Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::GpuBuffer* pGpuBuffer,
        const nn::util::Matrix4x4fType* projectionMatrix, const nn::util::Matrix4x3fType* viewMatrix, const nn::util::Vector3fType* camPos );

private:
    nn::gfx::Device*            m_pDevice;
    nns::nac::MemoryAllocator*  m_pHeap;

    int                         m_CurrentFace;

    nn::g3d::ResFile*           m_pG3dModelResFile;
    nn::gfx::VertexState*       m_pVertexStateArray;
    void*                       m_pVertexStateBuffer;
    nn::g3d::ModelObj*          m_pModelObj;

    Shader                      m_VertexShader;
    Shader                      m_FragmentShader;

    float                       m_RotateY;
};



//------------------------------------------------------------------------------
//  コンバイナプレビューモデルクラス
//------------------------------------------------------------------------------
class CombinerPreviewer
{
public:
    enum
    {
        CombinerPreviewer_Output  = 0,  // 出力シェーダ
        CombinerPreviewer_Preview = 1,  // ノードプレビューシェーダ
        CombinerPreviewer_Max     = 2,
    };

    enum
    {
        CombinerPreviewer_TextureCountMax = 4,  // 最大テクスチャ枚数
    };

private:
    struct ViewBlock
    {
        nn::util::FloatColumnMajor4x4  projectionMatrix;
        nn::util::FloatColumnMajor4x4  viewMatrix;
        nn::util::Float4    cameraPosition;
        nn::util::Float4    cameraVector;

        void Set(
            const nn::util::Matrix4x3fType& view,
            const nn::util::Matrix4x4fType& projection,
            const nn::util::Vector3fType&   cameraPos ) NN_NOEXCEPT
        {
            // Projection Matrix
            nn::util::MatrixStore( &projectionMatrix, projection );

            // View Matrix(3x4から4x3に変換)
            nn::util::Matrix4x4fType tempView4x4f;
            nn::util::Matrix4x4fType tempViewT4x4f;
            detail::Matrix4x3fTo4x4f( &tempView4x4f, view );

            // View Matrix
            nn::util::MatrixStore( &viewMatrix, tempView4x4f );

            // Look Direction
            nn::util::MatrixTranspose( &tempViewT4x4f, tempView4x4f );
            cameraVector.x = tempViewT4x4f._m[ 0 ][ 2 ];
            cameraVector.y = tempViewT4x4f._m[ 1 ][ 2 ];
            cameraVector.z = tempViewT4x4f._m[ 2 ][ 2 ];
            cameraVector.w = 0;

            // Eye Position
            nn::util::VectorStore( cameraPosition.v, cameraPos );
            cameraPosition.w = 0;
        }
    };


    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    CombinerPreviewer();

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    ~CombinerPreviewer();

public:
    //---------------------------------------------------------------------------
    //! @brief  初期化処理を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator*  pHeap );

    //---------------------------------------------------------------------------
    //! @brief  メモリプールから領域を確保します
    //---------------------------------------------------------------------------
    ptrdiff_t AllocateFromMemoryPool( size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/ );

    //---------------------------------------------------------------------------
    //! @brief  メモリプールから取得します
    //---------------------------------------------------------------------------
    nn::gfx::MemoryPool* GetMemoryPool()
    {
        return &m_MemoryPool;
    }

    //---------------------------------------------------------------------------
    //! @brief  初期状態モデルファイルをセットアップします。
    //---------------------------------------------------------------------------
    bool InitializeG3dModel();

    //---------------------------------------------------------------------------
    //! @brief  ブロックの更新処理を行います。
    //---------------------------------------------------------------------------
    void CalcBlock( const nn::util::Matrix4x3fType* viewMtx, float addRotateY );

    //---------------------------------------------------------------------------
    //! @brief  描画処理を行います。
    //---------------------------------------------------------------------------
    void Draw( int idenx, nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::GpuBuffer* pGpuBuffer,
        const nn::util::Matrix4x4fType* projectionMatrix, const nn::util::Matrix4x3fType* viewMatrix, const nn::util::Vector3fType* camPos,
        nn::gfx::DescriptorSlot colorBufferDescSlot, nn::gfx::DescriptorSlot depthBufferDescSlot );

    //---------------------------------------------------------------------------
    //! @brief  ディスクリプタスロットの更新を行います。
    //---------------------------------------------------------------------------
    void UpdateDescriptorSlot( detail::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator,
                               detail::SamplerDescriptorIndexAllocator* pSamplerDescriptorIndexAllocator );

    //---------------------------------------------------------------------------
    //! @brief  テクスチャのリロードが行われたかどうか。
    //---------------------------------------------------------------------------
    bool IsTextureReload() const
    {
        return m_TextureReload;
    }

    //---------------------------------------------------------------------------
    //! @brief  テクスチャのリセットが行われたかどうか。
    //---------------------------------------------------------------------------
    bool IsTextureReset() const
    {
        return m_TextureReset;
    }

    //---------------------------------------------------------------------------
    //! @brief  プレビューシェーダのリロードが行われたかどうか。
    //---------------------------------------------------------------------------
    bool IsPreviewShaderReload() const
    {
        return m_PreviewShaderReload;
    }

    //---------------------------------------------------------------------------
    //! @brief  出力シェーダのリロードが行われたかどうか。
    //---------------------------------------------------------------------------
    bool IsOutputShaderReload() const
    {
        return m_OutputShaderReload;
    }


    //---------------------------------------------------------------------------
    //! @brief  出力シェーダ用 モデルファイルをロードします。
    //---------------------------------------------------------------------------
    bool LoadG3dModelBinary( const char* pFmdbFilePath, const char* pOutputDir );

    //---------------------------------------------------------------------------
    // フラグメントシェーダの更新
    //---------------------------------------------------------------------------
    void UpdateFragmentShader();

    //---------------------------------------------------------------------------
    // テクスチャの更新
    //---------------------------------------------------------------------------
    void UpdateTexture();

    //---------------------------------------------------------------------------
    // シェーダを初期状態にリセットします
    //---------------------------------------------------------------------------
    void ResetShader();

    //---------------------------------------------------------------------------
    // テクスチャを初期状態にリセットします
    //---------------------------------------------------------------------------
    void ResetTexture( int index )
    {
        NN_SDK_ASSERT_RANGE( index, -1, CombinerPreviewer_TextureCountMax );
        m_ResetTextureIndex = index;
    }

    //---------------------------------------------------------------------------
    // 出力シェーダリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void SetReloadOutputShaderFilePath( const char* fshFilePath );

    //---------------------------------------------------------------------------
    // ノード経過ビューシェーダリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void SetReloadPreviewShaderFilePath( const char* fshFilePath );

    //---------------------------------------------------------------------------
    // テクスチャリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void ReloadTextureFilePath( int index, const char* ftxbFilePath );

    //---------------------------------------------------------------------------
    // テクスチャリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void ReloadTextureFilePath( const char* name, const char* ftxbFilePath );

    //---------------------------------------------------------------------------
    // テクスチャリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void UpdateShaderParameter( const char* name, float value );

    //---------------------------------------------------------------------------
    // 更新データを受信し適用します。
    //---------------------------------------------------------------------------
    void ReceiveUpdateParameter( const char* receiveData );

public:
    //---------------------------------------------------------------------------
    //! @brief  インスタンス取得。
    //---------------------------------------------------------------------------
    static CombinerPreviewer* GetInstance()
    {
        return g_pCombinerPreviewer;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンバイナプレビューモデルの生成。
    //---------------------------------------------------------------------------
    static void InitializeCombinerPreviewer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator );

    //---------------------------------------------------------------------------
    //! @brief  コンバイナプレビューモデルの破棄。
    //---------------------------------------------------------------------------
    static void FinalizeCombinerPreviewer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator );

private:
    nn::gfx::Device*            m_pDevice;
    nns::nac::MemoryAllocator*  m_pHeap;
    bool                        m_TextureReload;
    bool                        m_TextureReset;
    bool                        m_PreviewShaderReload;
    bool                        m_OutputShaderReload;

    void*                       m_pBinaryLoadBuffer;
    nn::lmem::HeapHandle        m_HeapHandle;

    void*                       m_pMemoryPoolPtr;
    nn::gfx::MemoryPool         m_MemoryPool;
    ptrdiff_t                   m_MemoryPoolOffset;

    Texture                     m_Texture[ CombinerPreviewer_TextureCountMax ];

    nn::gfx::Sampler            m_Sampler;
    nn::gfx::DescriptorSlot     m_SamplerDescriptorSlot;

    float                       m_ShaderParameter[ 4 ];
    nn::util::Float4            m_VectorParameter0;
    nn::util::Float4            m_VectorParameter1;
    nn::util::Float4            m_VectorParameter2;

    bool                        m_ShaderReset;
    int                         m_ReloadTextureIndex;
    int                         m_ResetTextureIndex;
    char                        m_TextureFilePath[ CombinerPreviewer_TextureCountMax ][ _MAX_PATH ];
    char                        m_OutputFshFilePath[ _MAX_PATH ];
    char                        m_PreviewFshFilePath[ _MAX_PATH ];

    Model                       m_Model[ CombinerPreviewer_Max ];
    void*                       m_ModelBinary[ CombinerPreviewer_Max ];
    char                        m_OutputModelFilePath[ _MAX_PATH ];
    char                        m_WorkingDir[ _MAX_PATH ];
    Texture                     m_CommonTexture;
    void*                       m_CommonTextureBinary;

    int                         g_Time;

    static CombinerPreviewer*   g_pCombinerPreviewer;
};
#else
class CombinerPreviewer
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    CombinerPreviewer() {}

    //---------------------------------------------------------------------------
    // シェーダリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void SetReloadShaderFilePath( const char* fshFilePath )
    {
        NN_UNUSED( fshFilePath );
        return;
    }

    //---------------------------------------------------------------------------
    //! @brief  ブロックの更新処理を行います。
    //---------------------------------------------------------------------------
    void CalcBlock( const nn::util::Matrix4x3fType* viewMatrix, float addRotateY )
    {
        NN_UNUSED( viewMatrix );
        NN_UNUSED( addRotateY );
    }

    //---------------------------------------------------------------------------
    //! @brief  描画処理を行います。
    //---------------------------------------------------------------------------
    void Draw( int idenx, nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::GpuBuffer* pGpuBuffer,
        const nn::util::Matrix4x4fType* projectionMatrix, const nn::util::Matrix4x3fType* viewMatrix, const nn::util::Vector3fType* camPos,
        nn::gfx::DescriptorSlot colorBufferDescSlot, nn::gfx::DescriptorSlot depthBufferDescSlot )
    {
        NN_UNUSED( idenx );
        NN_UNUSED( pCommandBuffer );
        NN_UNUSED( pGpuBuffer );
        NN_UNUSED( projectionMatrix );
        NN_UNUSED( viewMatrix );
        NN_UNUSED( camPos );
        NN_UNUSED( colorBufferDescSlot );
        NN_UNUSED( depthBufferDescSlot );
        return;
    }

    //---------------------------------------------------------------------------
    //! @brief  インスタンス取得。
    //---------------------------------------------------------------------------
    static CombinerPreviewer* GetInstance()
    {
        return g_pCombinerPreviewer;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンバイナプレビューモデルの生成。
    //---------------------------------------------------------------------------
    static void InitializeCombinerPreviewer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator )
    {
        return;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンバイナプレビューモデルの破棄。
    //---------------------------------------------------------------------------
    static void FinalizeCombinerPreviewer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator )
    {
        return;
    }

    //---------------------------------------------------------------------------
    //! @brief  ディスクリプタスロットの更新を行います。
    //---------------------------------------------------------------------------
    bool IsReload() const
    {
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief  ディスクリプタスロットの更新を行います。
    //---------------------------------------------------------------------------
    void UpdateDescriptorSlot( detail::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator,
        detail::SamplerDescriptorIndexAllocator* pSamplerDescriptorIndexAllocator )
    {
        return;
    }

    //---------------------------------------------------------------------------
    //! @brief  テクスチャのリロードが行われたかどうか。
    //---------------------------------------------------------------------------
    bool IsTextureReload() const
    {
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief  プレビューシェーダのリロードが行われたかどうか。
    //---------------------------------------------------------------------------
    bool IsPreviewShaderReload() const
    {
        return false;
    }

private:
    static CombinerPreviewer* g_pCombinerPreviewer;
};
#endif

} // namespace detail
