﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <util/types.h>
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <Windows.h>
#include <eh.h>

#pragma warning(push)
#pragma warning(disable:4091)
#include <dbghelp.h>
#pragma warning(pop)

namespace nw { namespace g3d { namespace tool {
namespace util {

//---------------------------------------------------------------------------
//! @brief 時間を表すクラスです。
//!
//! @details
//!  このクラスの内部では、64bit 整数としてナノ秒単位で時間を表しています。
//!  このクラスのインスタンスは From* 関数を使うことで、各単位で表された整数値から作成することができます。
//!  単位があいまいになるのを防ぐため、整数からこの型への暗黙的な変換は用意されていませんが、
//!  0 はこの型への暗黙的な変換をすることができます。
//---------------------------------------------------------------------------
class TimeSpan
{
private:

    typedef const class ZeroOnlyTag {} * ZeroOnly;

public:

    //---------------------------------------------------------------------------
    //! @brief        時間を 0 で初期化するコンストラクタです。
    //!
    //! @details
    //!  引数を省略した場合、デフォルトコンストラクタとなり、時間 0 で初期化されます。
    //!  引数つきのこのコンストラクタは、時間を指定するべき場所で 0 を指定できるようにするために用意されています。
    //!  0 以外の引数を与えることはできません。
    //!
    //! @param[in]    zeroOnly 引数を省略するか、0 を指定してください。
    //---------------------------------------------------------------------------
    TimeSpan(ZeroOnly zeroOnly = 0) : m_NanoSeconds(0) { (void)zeroOnly; }

    //---------------------------------------------------------------------------
    //! @brief        ナノ秒数から TimeSpan オブジェクトを生成します。
    //!
    //! @param[in]    nanoSeconds   時間の長さをナノ秒数で指定します。
    //!
    //! @return       TimeSpan      指定された時間の長さを表す TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    static TimeSpan FromNanoSeconds(s64 nanoSeconds) { TimeSpan ret; ret.m_NanoSeconds = nanoSeconds; return ret; }

    //---------------------------------------------------------------------------
    //! @brief        マイクロ秒数から TimeSpan オブジェクトを生成します。
    //!
    //! @param[in]    microSeconds  時間の長さをマイクロ秒数で指定します。
    //!
    //! @return       TimeSpan      指定された時間の長さを表す TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    static TimeSpan FromMicroSeconds(s64 microSeconds) { return FromNanoSeconds(microSeconds * 1000); }

    //---------------------------------------------------------------------------
    //! @brief        ミリ秒数から TimeSpan オブジェクトを生成します。
    //!
    //! @param[in]    milliSeconds  時間の長さをミリ秒数で指定します。
    //!
    //! @return       TimeSpan      指定された時間の長さを表す TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    static TimeSpan FromMilliSeconds(s64 milliSeconds) { return FromNanoSeconds(milliSeconds * 1000 * 1000); }

    //---------------------------------------------------------------------------
    //! @brief        秒数から TimeSpan オブジェクトを生成します。
    //!
    //! @param[in]    seconds       時間の長さを秒数で指定します。
    //!
    //! @return       TimeSpan      指定された時間の長さを表す TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    static TimeSpan FromSeconds(s64 seconds) { return FromNanoSeconds(seconds * 1000 * 1000 * 1000); }

    //---------------------------------------------------------------------------
    //! @brief        分数から TimeSpan オブジェクトを生成します。
    //!
    //! @param[in]    minutes       時間の長さを分数で指定します。
    //!
    //! @return       TimeSpan      指定された時間の長さを表す TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    static TimeSpan FromMinutes(s64 minutes) { return FromNanoSeconds(minutes * 1000 * 1000 * 1000 * 60); }

    //---------------------------------------------------------------------------
    //! @brief        時間数から TimeSpan オブジェクトを生成します。
    //!
    //! @param[in]    hours         時間の長さを時数で指定します。
    //!
    //! @return       TimeSpan      指定された時間の長さを表す TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    static TimeSpan FromHours(s64 hours) { return FromNanoSeconds(hours * 1000 * 1000 * 1000 * 60 * 60); }

    //---------------------------------------------------------------------------
    //! @brief        日数から TimeSpan オブジェクトを生成します。
    //!
    //! @param[in]    days         時間(日数)を指定します。
    //!
    //! @return       TimeSpan      指定された時間の長さを表す TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    static TimeSpan FromDays(s64 days) { return FromNanoSeconds(days * 1000 * 1000 * 1000 * 60 * 60 * 24); }

    //---------------------------------------------------------------------------
    //! @brief        時間を日数で取得します。
    //!
    //! @return       s64           この TimeSpan オブジェクトが表す時間の長さを日数に換算した値を返します。
    //---------------------------------------------------------------------------
    s64 GetDays() const { return m_NanoSeconds / (1000LL * 1000 * 1000 * 60 * 60 * 24); }

    //---------------------------------------------------------------------------
    //! @brief        時間を時数で取得します。
    //!
    //! @return       s64           この TimeSpan オブジェクトが表す時間の長さを時数に換算した値を返します。
    //---------------------------------------------------------------------------
    s64 GetHours() const { return m_NanoSeconds / (1000LL * 1000 * 1000 * 60 * 60); }

    //---------------------------------------------------------------------------
    //! @brief        時間を分数で取得します。
    //!
    //! @return       s64           この TimeSpan オブジェクトが表す時間の長さを分数に換算した値を返します。
    //---------------------------------------------------------------------------
    s64 GetMinutes() const { return m_NanoSeconds / (1000LL * 1000 * 1000 * 60); }

    //---------------------------------------------------------------------------
    //! @brief        時間を秒数で取得します。
    //!
    //! @return       s64           この TimeSpan オブジェクトが表す時間の長さを秒数に換算した値を返します。
    //---------------------------------------------------------------------------
    s64 GetSeconds() const { return m_NanoSeconds / (1000 * 1000 * 1000); }
    //---------------------------------------------------------------------------
    //! @brief        時間をミリ秒数で取得します。
    //!
    //! @return       TimeSpan      この TimeSpan オブジェクトが表す時間の長さをミリ秒数に換算した値を返します。
    //---------------------------------------------------------------------------
    s64 GetMilliSeconds() const { return m_NanoSeconds / (1000 * 1000); }
    //---------------------------------------------------------------------------
    //! @brief        時間をマイクロ秒数で取得します。
    //!
    //! @return       TimeSpan      この TimeSpan オブジェクトが表す時間の長さをマイクロ秒数に換算した値を返します。
    //---------------------------------------------------------------------------
    s64 GetMicroSeconds() const { return m_NanoSeconds / 1000; }
    //---------------------------------------------------------------------------
    //! @brief        時間をナノ秒数で取得します。
    //!
    //! @return       TimeSpan      この TimeSpan オブジェクトが表す時間の長さをナノ秒数に換算した値を返します。
    //---------------------------------------------------------------------------
    s64 GetNanoSeconds() const { return m_NanoSeconds; }

    friend bool operator==(const TimeSpan& lhs, const TimeSpan& rhs) { return lhs.m_NanoSeconds == rhs.m_NanoSeconds; }
    friend bool operator!=(const TimeSpan& lhs, const TimeSpan& rhs) { return !(lhs == rhs); }
    friend bool operator< (const TimeSpan& lhs, const TimeSpan& rhs) { return lhs.m_NanoSeconds <  rhs.m_NanoSeconds; }
    friend bool operator> (const TimeSpan& lhs, const TimeSpan& rhs) { return rhs < lhs; }
    friend bool operator<=(const TimeSpan& lhs, const TimeSpan& rhs) { return !(lhs > rhs); }
    friend bool operator>=(const TimeSpan& lhs, const TimeSpan& rhs) { return !(lhs < rhs); }

    TimeSpan& operator+=(const TimeSpan& rhs) { this->m_NanoSeconds += rhs.m_NanoSeconds; return *this; }
    friend TimeSpan operator+(const TimeSpan& lhs, const TimeSpan& rhs) { TimeSpan ret(lhs); return ret += rhs; }

    TimeSpan& operator-=(const TimeSpan& rhs) { this->m_NanoSeconds -= rhs.m_NanoSeconds; return *this; }
    friend TimeSpan operator-(const TimeSpan& lhs, const TimeSpan& rhs) { TimeSpan ret(lhs); return ret -= rhs; }

private:

    s64 m_NanoSeconds;
};

//---------------------------------------------------------------------------
//! @brief システムチックを扱う為のクラスです。
//!
//! @details
//!    システムチックは、CPU の 1 クロックの時間を表します。
//!    変換コンストラクタ・変換演算子を使うことで、実際の時間を表す @ref TimeSpan との相互変換ができます。
//!    システムが立ち上がってからのチック数を取得するには @ref GetSystemCurrent を使います。
//---------------------------------------------------------------------------
class Tick
{
public:
    //---------------------------------------------------------------------------
    //! @brief      システムが起動してからのチック数を返します。
    //!
    //! @return     システムの Tick 値。
    //---------------------------------------------------------------------------
    static Tick GetSystemCurrent();

    //---------------------------------------------------------------------------
    //! @brief チック数を取ってオブジェクトを初期化するコンストラクタです。
    //!
    //! @param[in] tick チック数
    //---------------------------------------------------------------------------
    explicit Tick(s64 tick = 0) : m_Tick(tick) {}

    //---------------------------------------------------------------------------
    //! @brief TimeSpan を取ってオブジェクトを初期化するコンストラクタです。
    //!
    //! @param[in] span 初期化する時間の値
    //---------------------------------------------------------------------------
    /* implicit */ Tick(TimeSpan span) : m_Tick( NanoSecondsToTick( span.GetNanoSeconds() ))
    {
    }

    //---------------------------------------------------------------------------
    //! @brief 64bit のチック値に変換します。
    //---------------------------------------------------------------------------
    operator s64() const { return m_Tick; }

    //---------------------------------------------------------------------------
    //! @brief @ref TimeSpan の値に変換します。
    //---------------------------------------------------------------------------
    operator TimeSpan() const
    {
        return TimeSpan::FromNanoSeconds( TickToNanoSeconds( m_Tick ) );
    }

    //---------------------------------------------------------------------------
    //! @brief @ref TimeSpan の値に変換します。
    //!
    //! @return チックと同じ時間を表す @ref TimeSpan オブジェクトを返します。
    //---------------------------------------------------------------------------
    TimeSpan ToTimeSpan() const
    {
        return *this;
    }

    Tick& operator+=(Tick rhs) { this->m_Tick += rhs.m_Tick; return *this; }
    Tick& operator-=(Tick rhs) { this->m_Tick -= rhs.m_Tick; return *this; }

    Tick operator+(Tick rhs) const { Tick ret(*this); return ret += rhs; }
    Tick operator-(Tick rhs) const { Tick ret(*this); return ret -= rhs; }

    Tick& operator+=(TimeSpan rhs)
    {
        const s64 tick = NanoSecondsToTick( rhs.GetNanoSeconds() );
        this->m_Tick += tick;
        return *this;
    }

    Tick operator+(TimeSpan rhs) const { Tick ret(*this); return ret += rhs; }

private:
    static s64 GetFrequency();

    static const s64 NanoSecondsToTick( s64 nanoSeconds )
    {
        return ( (nanoSeconds * (TICKS_PER_SECOND / 125000)) / 8000 );
    }

    static const s64 TickToNanoSeconds( s64 tick )
    {
        s64 spanAbs = tick > 0 ? tick : -tick;
        if ( spanAbs < LLONG_MAX / 1000 / 1000 / 1000 )
        {
            return tick * 1000 * 1000 * 1000 / TICKS_PER_SECOND;
        }
        else if ( spanAbs < LLONG_MAX / 1000 / 1000 )
        {
            return tick * 1000 * 1000 / TICKS_PER_SECOND * 1000;
        }
        else if ( spanAbs < LLONG_MAX / 1000 )
        {
            return tick * 1000 / TICKS_PER_SECOND * 1000 * 1000;
        }
        else
        {
            return tick / TICKS_PER_SECOND * 1000 * 1000 * 1000;
        }
    }

    // 1秒間のチック数を表す定数です。
    static const s64 TICKS_PER_SECOND;

    s64 m_Tick;
};

} // namespace util

} // namespace tool
} // namespace g3d
} // namespace nw
