﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/Texture.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    texture_info, dimension,
    S11N_DEFINE_ENUM_ID(1d),
    S11N_DEFINE_ENUM_ID(2d),
    S11N_DEFINE_ENUM_ID(3d),
    S11N_DEFINE_ENUM_ID(cube),
    S11N_DEFINE_ENUM_ID(1d_array),
    S11N_DEFINE_ENUM_ID(2d_array),
    S11N_DEFINE_ENUM_ID(cube_array)
    );

G3DIF_DEFINE_ENUM_TABLE(
    texture_info, quantize_type,
    S11N_DEFINE_ENUM_ID(unorm_8),
    S11N_DEFINE_ENUM_ID(uint_8),
    S11N_DEFINE_ENUM_ID(snorm_8),
    S11N_DEFINE_ENUM_ID(sint_8),
    S11N_DEFINE_ENUM_ID(unorm_4_4),
    S11N_DEFINE_ENUM_ID(unorm_16),
    S11N_DEFINE_ENUM_ID(uint_16),
    S11N_DEFINE_ENUM_ID(snorm_16),
    S11N_DEFINE_ENUM_ID(sint_16),
    S11N_DEFINE_ENUM_ID(float_16),
    S11N_DEFINE_ENUM_ID(unorm_8_8),
    S11N_DEFINE_ENUM_ID(uint_8_8),
    S11N_DEFINE_ENUM_ID(snorm_8_8),
    S11N_DEFINE_ENUM_ID(sint_8_8),
    S11N_DEFINE_ENUM_ID(unorm_5_6_5),
    S11N_DEFINE_ENUM_ID(unorm_5_5_5_1),
    S11N_DEFINE_ENUM_ID(unorm_4_4_4_4),
    S11N_DEFINE_ENUM_ID(unorm_1_5_5_5),
    S11N_DEFINE_ENUM_ID(uint_32),
    S11N_DEFINE_ENUM_ID(sint_32),
    S11N_DEFINE_ENUM_ID(float_32),
    S11N_DEFINE_ENUM_ID(unorm_16_16),
    S11N_DEFINE_ENUM_ID(uint_16_16),
    S11N_DEFINE_ENUM_ID(snorm_16_16),
    S11N_DEFINE_ENUM_ID(sint_16_16),
    S11N_DEFINE_ENUM_ID(float_16_16),
    S11N_DEFINE_ENUM_ID(float_11_11_10),
    S11N_DEFINE_ENUM_ID(unorm_10_10_10_2),
    S11N_DEFINE_ENUM_ID(uint_10_10_10_2),
    S11N_DEFINE_ENUM_ID(unorm_8_8_8_8),
    S11N_DEFINE_ENUM_ID(uint_8_8_8_8),
    S11N_DEFINE_ENUM_ID(snorm_8_8_8_8),
    S11N_DEFINE_ENUM_ID(sint_8_8_8_8),
    S11N_DEFINE_ENUM_ID(srgb_8_8_8_8),
    S11N_DEFINE_ENUM_ID(unorm_2_10_10_10),
    S11N_DEFINE_ENUM_ID(uint_2_10_10_10),
    S11N_DEFINE_ENUM_ID(uint_32_32),
    S11N_DEFINE_ENUM_ID(sint_32_32),
    S11N_DEFINE_ENUM_ID(float_32_32),
    S11N_DEFINE_ENUM_ID(unorm_16_16_16_16),
    S11N_DEFINE_ENUM_ID(uint_16_16_16_16),
    S11N_DEFINE_ENUM_ID(snorm_16_16_16_16),
    S11N_DEFINE_ENUM_ID(sint_16_16_16_16),
    S11N_DEFINE_ENUM_ID(float_16_16_16_16),
    S11N_DEFINE_ENUM_ID(uint_32_32_32_32),
    S11N_DEFINE_ENUM_ID(sint_32_32_32_32),
    S11N_DEFINE_ENUM_ID(float_32_32_32_32),
    S11N_DEFINE_ENUM_ID(unorm_bc1),
    S11N_DEFINE_ENUM_ID(srgb_bc1),
    S11N_DEFINE_ENUM_ID(unorm_bc2),
    S11N_DEFINE_ENUM_ID(srgb_bc2),
    S11N_DEFINE_ENUM_ID(unorm_bc3),
    S11N_DEFINE_ENUM_ID(srgb_bc3),
    S11N_DEFINE_ENUM_ID(unorm_bc4),
    S11N_DEFINE_ENUM_ID(snorm_bc4),
    S11N_DEFINE_ENUM_ID(unorm_bc5),
    S11N_DEFINE_ENUM_ID(snorm_bc5)
);

G3DIF_DEFINE_ENUM_TABLE(
    texture_info, tile_mode,
    S11N_DEFINE_ENUM_ID(linear),
    S11N_DEFINE_ENUM_ID(linear_aligned),
    S11N_DEFINE_ENUM_ID(1d_tiled_thin1),
    S11N_DEFINE_ENUM_ID(1d_tiled_thick),
    S11N_DEFINE_ENUM_ID(2d_tiled_thin1),
    S11N_DEFINE_ENUM_ID(2d_tiled_thin2),
    S11N_DEFINE_ENUM_ID(2d_tiled_thin4),
    S11N_DEFINE_ENUM_ID(2d_tiled_thick),
    S11N_DEFINE_ENUM_ID(2b_tiled_thin1),
    S11N_DEFINE_ENUM_ID(2b_tiled_thin2),
    S11N_DEFINE_ENUM_ID(2b_tiled_thin4),
    S11N_DEFINE_ENUM_ID(2b_tiled_thick),
    S11N_DEFINE_ENUM_ID(3d_tiled_thin1),
    S11N_DEFINE_ENUM_ID(3d_tiled_thick),
    S11N_DEFINE_ENUM_ID(3b_tiled_thin1),
    S11N_DEFINE_ENUM_ID(3b_tiled_thick)
    );

G3DIF_DEFINE_ENUM_TABLE(
    texture_info, comp_sel,
    S11N_DEFINE_ENUM_ID(r),
    S11N_DEFINE_ENUM_ID(g),
    S11N_DEFINE_ENUM_ID(b),
    S11N_DEFINE_ENUM_ID(a),
    S11N_DEFINE_ENUM_ID(0),
    S11N_DEFINE_ENUM_ID(1)
    );

void elem_texture_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        dimension << pElem;
        quantize_type << pElem;
        width << pElem;
        height << pElem;
        depth << pElem;
        mip_level << pElem;
        size << pElem;
        comp_sel << pElem;
        // スペース区切り
        std::vector<std::string> sep;
        util::Split(sep, comp_sel.value, " ");

        for (int i = 0; i < 4; ++i)
        {
            enum_comp_sel sel;
            Read(sep[i].c_str(), sel);
            component_selector[i] = sel;
        }

        linear << pElem;
        util::Split(sep, linear.value, " ");

        for (int i = 0; i < 4; ++i)
        {
            bool flag;
            Read(sep[i].c_str(), flag);
            linear_flag[i] = flag;
        }

        tile_mode << pElem;
        swizzle << pElem;
        alignment << pElem;
        pitch << pElem;
        mip_offset << pElem;
        stream_index << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_texture::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        texture_info << pElem->Child(elem_texture_info::Id());
        stream_array << pElem->Child(elem_stream::IdArray());
        user_data_array << pElem->Child(elem_user_data::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_texture::PostProcess()
{
    // 参照の解決。
    try
    {
        int stream_array_size = static_cast<int>(stream_array.size());

        if (stream_array_size <= texture_info.stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d texture.stream_index: %d",
                stream_array_size, texture_info.stream_index.value);
        }
        // テキストデータを解析します。
        elem_stream& xmlStream = stream_array[texture_info.stream_index.value];
        stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
        stream.count = xmlStream.count.value;
        stream.type = static_cast<StreamType>(xmlStream.type.value);
    }
    CATCH_THROW_XML_ERROR()

    // ユーザーデータの解析
    for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
    {
        iter->PostProcess(stream_array);
    }
}

void elem_texture::PostBinaryProcess(void* data)
{
    try
    {
        StreamArray streamArray;
        AnalizeBinaryData(streamArray, data);

        int stream_array_size = static_cast<int>(streamArray.streamChunk.size());

        if (stream_array_size <= texture_info.stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d texture.stream_index: %d",
                stream_array_size, texture_info.stream_index.value);
        }

        void* rawdata = CopyRawData(streamArray.streamChunk[texture_info.stream_index.value]);

        stream.rawdata.reset(rawdata, free);
        stream.count = streamArray.streamChunk[texture_info.stream_index.value].count;
        stream.type = streamArray.streamChunk[texture_info.stream_index.value].type;

        for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
        {
            iter->PostBinaryProcess(streamArray);
        }
    }
    CATCH_THROW_XML_ERROR()
}

void
elem_texture::CheckData(int /*flag*/)
{
    try
    {
        if (texture_info.quantize_type.value == elem_texture_info::type_8_8_8_8_srgb ||
            texture_info.quantize_type.value == elem_texture_info::type_bc1_srgb ||
            texture_info.quantize_type.value == elem_texture_info::type_bc2_srgb ||
            texture_info.quantize_type.value == elem_texture_info::type_bc3_srgb)
        {
            if (!texture_info.linear_flag[0] ||
                !texture_info.linear_flag[1] ||
                !texture_info.linear_flag[2])
            {
                THROW_ERROR(ERRCODE_XML_INVALID_TEXTURE_FORMAT, "Invalid a \"linear\" texture format.");
            }
        }
    }
    CATCH_THROW_XML_ERROR()
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
