﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/Model.h>
#include <algorithm>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

//--------------------------------------------------------------------------------------------------

void elem_model_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        unite_pos_quantize << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_model::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        version << pElem;
        model_info << pElem->Child(elem_model_info::Id());
        skeleton << pElem->Child(elem_skeleton::Id());
        vertex_array << pElem->Child(elem_vertex::IdArray());
        shape_array << pElem->Child(elem_shape::IdArray());
        material_array << pElem->Child(elem_material::IdArray());
        stream_array << pElem->Child(elem_stream::IdArray());
        user_data_array << pElem->Child(elem_user_data::IdArray());
        tool_data << pElem->Child(elem_tool_data::Id());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_model::PostProcess()
{
    try
    {
        // 参照の解決。
        int shapeIndex = 0;
        for (auto iterShape = shape_array.begin(); iterShape != shape_array.end(); ++iterShape, ++shapeIndex)
        {
            iterShape->index = shapeIndex;
            iterShape->PostProcess(stream_array);
        }

        int vertexIndex = 0;
        for (auto iterVertex = vertex_array.begin(); iterVertex != vertex_array.end(); ++iterVertex, ++vertexIndex)
        {
            iterVertex->index = vertexIndex;
            iterVertex->PostProcess(stream_array);
        }

        // ユーザーデータの解析
        for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
        {
            iter->PostProcess(stream_array);
        }

        for (auto iterMat = material_array.begin(); iterMat != material_array.end(); ++iterMat)
        {
            for (auto iter = iterMat->user_data_array.begin(); iter != iterMat->user_data_array.end(); ++iter)
            {
                iter->PostProcess(stream_array);
            }
        }

        for (auto iterBone = skeleton->bone_array.begin(); iterBone != skeleton->bone_array.end(); ++iterBone)
        {
            for (auto iter = iterBone->user_data_array.begin(); iter != iterBone->user_data_array.end(); ++iter)
            {
                iter->PostProcess(stream_array);
            }
        }

        for (auto iterShape = shape_array.begin(); iterShape != shape_array.end(); ++iterShape)
        {
            for (auto iter = iterShape->user_data_array.begin(); iter != iterShape->user_data_array.end(); ++iter)
            {
                iter->PostProcess(stream_array);
            }
        }

        for (auto iterMat = material_array.begin(); iterMat != material_array.end(); ++iterMat)
        {
            iterMat->Expand();
        }
    }
    CATCH_THROW_XML_ERROR()

    skeleton->ResolveSkeleton();
    ResolveVertexAttrib();
    ResolveShapeAttrib();
    ResolvePosition();
    ResolveBoneIndex();
    ResolveShapeMaterial();
    ResolveSamplerIndex();
    ResolveUniteQuantizeType();
}

void elem_model::PostBinaryProcess(void* data)
{
    try
    {
        StreamArray streamArray;
        AnalizeBinaryData(streamArray, data);

        // 参照の解決。
        int shapeIndex = 0;
        for (auto iterShape = shape_array.begin(); iterShape != shape_array.end(); ++iterShape, ++shapeIndex)
        {
            iterShape->index = shapeIndex;
            iterShape->PostBinaryProcess(streamArray);
        }
        int vertexIndex = 0;
        for (auto iterVertex = vertex_array.begin(); iterVertex != vertex_array.end(); ++iterVertex, ++vertexIndex)
        {
            iterVertex->index = vertexIndex;
            iterVertex->PostBinaryProcess(streamArray);
        }

        // ユーザーデータの解析
        for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
        {
            iter->PostBinaryProcess(streamArray);
        }

        for (auto iterMat = material_array.begin(); iterMat != material_array.end(); ++iterMat)
        {
            for (auto iter = iterMat->user_data_array.begin(); iter != iterMat->user_data_array.end(); ++iter)
            {
                iter->PostBinaryProcess(streamArray);
            }
        }

        for (auto iterBone = skeleton->bone_array.begin(); iterBone != skeleton->bone_array.end(); ++iterBone)
        {
            for (auto iter = iterBone->user_data_array.begin(); iter != iterBone->user_data_array.end(); ++iter)
            {
                iter->PostBinaryProcess(streamArray);
            }
        }

        for (auto iterShape = shape_array.begin(); iterShape != shape_array.end(); ++iterShape)
        {
            for (auto iter = iterShape->user_data_array.begin(); iter != iterShape->user_data_array.end(); ++iter)
            {
                iter->PostBinaryProcess(streamArray);
            }
        }

        for (auto iterMat = material_array.begin(); iterMat != material_array.end(); ++iterMat)
        {
            iterMat->Expand();
        }
    }
    CATCH_THROW_XML_ERROR()

    skeleton->ResolveSkeleton();
    ResolveVertexAttrib();
    ResolveShapeAttrib();
    ResolvePosition();
    ResolveBoneIndex();
    ResolveShapeMaterial();
    ResolveSamplerIndex();
    ResolveUniteQuantizeType();
}

// vertexbuffer->input_array の attrib_index を元に vtx_attrib を関連付け
void elem_model::ResolveVertexAttrib()
{
    for (auto iterVertex = vertex_array.begin(); iterVertex != vertex_array.end(); ++iterVertex)
    {
        auto& vtx_attrib_array = iterVertex->vtx_attrib_array;
        auto& vtx_buffer_array = iterVertex->vtx_buffer_array;
        try
        {
            int vtx_attrib_size = static_cast<int>(vtx_attrib_array.size());
            for (auto iter = vtx_buffer_array.begin(); iter != vtx_buffer_array.end(); ++iter)
            {
                auto& input_array = iter->input_array;
                for (auto iterInput = input_array.begin(); iterInput != input_array.end(); ++iterInput)
                {
                    if (vtx_attrib_size <= iterInput->attrib_index.value)
                    {
                        THROW_TRANSLATED_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Identifier_OutOfRange",
                            "vtx_buffer_array->attrib_index", iterInput->attrib_index.value, "vtx_attrib_array", vtx_attrib_size);
                    }
                    iterInput->attrib = &vtx_attrib_array[iterInput->attrib_index.value];
                }
            }
        }
        CATCH_THROW_XML_ERROR()
    }
}

// vertex の lodOffset を各 mesh に持たせる
// vertexCount の解決
// keyshape の attribIndex と hint の解決
void elem_model::ResolveShapeAttrib()
{
    try
    {
        for (auto iterShape = shape_array.begin(); iterShape != shape_array.end(); ++iterShape)
        {
            if (iterShape->shape_info.vertex_index.value >= static_cast<int>(vertex_array.size()))
            {
                THROW_TRANSLATED_ERROR( ERRCODE_XML_SHAPE_INVALID_VERTEX_INDEX, "Identifier_OutOfRange",
                    "vertex_index", iterShape->shape_info.vertex_index.value,
                    "vertex_array", static_cast<int>(vertex_array.size())
                );
            }

            elem_vertex& el_vertex = vertex_array[iterShape->shape_info.vertex_index.value];

            if (iterShape->mesh_array.size() >= 2)
            {
                // mesh が 2 以上存在している場合は mesh_array の数 - 1 が lodOffset の数になっている必要がある。
                if (!el_vertex.lod_offset ||
                    (iterShape->mesh_array.size() - 1 != el_vertex.lod_offset->lodOffset.size()))
                {
                    THROW_TRANSLATED_ERROR( ERRCODE_XML_INVALID_LOD_OFFSET, "Identifier_InvalidLodOffset", iterShape->name.value.c_str());
                }

                // lod_offset を shape に持たせる。
                for (int idxMesh = 1; idxMesh < static_cast<int>( iterShape->mesh_array.size() ); ++idxMesh)
                {
                    iterShape->mesh_array[idxMesh].offset = el_vertex.lod_offset->lodOffset[idxMesh - 1];
                }
            }

            for (auto iterVtxAttrib = el_vertex.vtx_attrib_array.cbegin();
                iterVtxAttrib != el_vertex.vtx_attrib_array.cend();
                ++iterVtxAttrib)
            {
                iterShape->shape_info.vertex_count = iterVtxAttrib->count.value;
            }

            size_t targetCount = (iterShape->key_shape_array.size() > 0) ?
                iterShape->key_shape_array.begin()->target_attrib_array.size() : 0;
            for (auto iterKeyShape = iterShape->key_shape_array.begin();
                iterKeyShape != iterShape->key_shape_array.end();
                ++iterKeyShape)
            {
                // 全てのキーシェイプの数は同じになっている必要がある。
                if (targetCount != iterKeyShape->target_attrib_array.size())
                {
                    THROW_TRANSLATED_ERROR( ERRCODE_XML_INVALID_TARGET_ATTRIB_COUNT, "Identifier_InvalidValue", "target_attrib count");
                }

                for (auto targetAttrib = iterKeyShape->target_attrib_array.begin();
                    targetAttrib != iterKeyShape->target_attrib_array.end();
                    ++targetAttrib)
                {
                    int attribIndex = 0;
                    for (auto iterAttrib = el_vertex.vtx_attrib_array.cbegin();
                        iterAttrib != el_vertex.vtx_attrib_array.cend();
                        ++iterAttrib, ++attribIndex)
                    {
                        if (targetAttrib->attrib_name.value == iterAttrib->name.value)
                        {
                            targetAttrib->attrib_index = attribIndex;
                            targetAttrib->attrib_hint = iterAttrib->hint.value;
                            break;
                        }
                    }

                    if (targetAttrib->attrib_index == -1)
                    {
                        THROW_TRANSLATED_ERROR( ERRCODE_XML_TARGET_ATTRIB_NAME_NOT_FOUND, "Identifier_NotFoundKeyAttribute", targetAttrib->attrib_name.value.c_str());
                    }
                }
            }
        }
    }
    CATCH_THROW_XML_ERROR()
}

// 頂点属性 "position" を vertex に vec3 として格納
void elem_model::ResolvePosition()
{
    try
    {
        for (auto iterVertex = vertex_array.begin(); iterVertex != vertex_array.end(); ++iterVertex)
        {
            auto& vtx_attrib_array = iterVertex->vtx_attrib_array;
            for (auto iter = vtx_attrib_array.begin(); iter != vtx_attrib_array.end(); ++iter)
            {
                if (iter->hint.value.find("position") != std::string::npos)
                {
                    if (iter->type.value != elem_vtx_attrib::type_float3)
                    {
                        THROW_TRANSLATED_ERROR( ERRCODE_XML_INVALID_ATTRIBUTE_TYPE, "Identifier_PositionMustBeFloat3");
                    }

                    // 頂点を追加
                    const float* positionPtr = static_cast<float*>(iter->stream.rawdata.get());
                    iterVertex->positions.reserve(iter->count.value / 3);
                    for (int i = 0; i < iter->count.value; ++i)
                    {
                        const float* posPtr = positionPtr + i * 3;
                        util::Vec3_t vec;
                        vec.x = *posPtr;
                        vec.y = *(posPtr + 1);
                        vec.z = *(posPtr + 2);
                        iterVertex->positions.push_back(vec);
                    }

                    break;
                }
            }
        }
    }
    CATCH_THROW_XML_ERROR()
}

struct AttribStream
{
    int count;
    int elemSize;
    const void* stream;
};

// shape と bone の関連付け
// 頂点属性の blendindex/blendweight を元に boneindex,weight のペアのリストを vertex->boneIndices に設定
// 頂点毎の blend に使用するboneindex のリストを Vertex->boneLists[] に設定
void
elem_model::ResolveBoneIndex()
{
    try
    {
        // ランタイムでのバウンディング計算のために、シェイプが参照するボーンインデクスを関連付けます。
        for (auto iterShape = shape_array.begin(); iterShape != shape_array.end(); ++iterShape)
        {
            for (int boneIndex = 0; boneIndex < static_cast<int>(skeleton->bone_array.size()); ++boneIndex)
            {
                if (iterShape->shape_info.bone_name.value == skeleton->bone_array[boneIndex].name.value)
                {
                    iterShape->shape_info.bone_index = boneIndex;
                    break;
                }
            }

            if (iterShape->shape_info.bone_index == -1)
            {
                THROW_TRANSLATED_ERROR(ERRCODE_XML_BONE_NAME_NOT_FOUND, "Identifier_NotFoundBone", iterShape->shape_info.bone_name.value, iterShape->name.value.c_str());
            }
        }

        // 頂点毎の blendIndex と blendWeight のペアを Vertex に関連付けます。
        const int sizeMatrixPalette = static_cast<int>(skeleton->matrixPalette.size());
        for (auto iterVertex = vertex_array.begin(); iterVertex != vertex_array.end(); ++iterVertex)
        {
            // まずは index と weight を収集する。
            auto& vtx_attrib_array = iterVertex->vtx_attrib_array;

            std::vector<AttribStream> indexStreames;
            std::vector<AttribStream> weightStreames;
            indexStreames.resize(vtx_attrib_array.size());
            weightStreames.resize(vtx_attrib_array.size());

            int vertexCount = 0;
            for (int i = 0; i < static_cast<int>(vtx_attrib_array.size()); ++i)
            {
                AttribStream stream;
                stream.count = 0;
                stream.elemSize = 0;
                stream.stream = nullptr;
                indexStreames[i] = stream;
                weightStreames[i] = stream;
            }

            for (auto iter = vtx_attrib_array.begin(); iter != vtx_attrib_array.end(); ++iter)
            {
                if (iter->hint.value.find("blendindex") != std::string::npos)
                {
                    int index = 0;
                    std::string indexStr = iter->hint.value.substr(sizeof("blendindex") - 1);
                    util::TryParse(index, indexStr);
                    AttribStream& stream = indexStreames[index];
                    stream.count = iter->stream.count;
                    stream.stream = iter->stream.rawdata.get();
                    stream.elemSize = static_cast<int>(iter->type.value) % 4 + 1;

                    // 型の大きさを調べる
                    vertexCount = iter->count.value;
                }
                else if (iter->hint.value.find("blendweight") != std::string::npos)
                {
                    int index = 0;
                    std::string indexStr = iter->hint.value.substr(sizeof("blendweight") - 1);
                    util::TryParse(index, indexStr);
                    AttribStream& stream = weightStreames[index];
                    stream.count = iter->stream.count;
                    stream.stream = iter->stream.rawdata.get();
                    stream.elemSize = static_cast<int>(iter->type.value) % 4 + 1;
                }
            }

            // 頂点数だけ bone リストを用意する。
            iterVertex->boneLists.resize(vertexCount);
            for (auto iter = iterVertex->boneLists.begin(), endIter = iterVertex->boneLists.end(); iter != endIter; ++iter)
            {
                // とりあえず、8ボーンで予約する。
                iter->reserve(8);
            }

            // 頂点を追加
            for (int attribIndex = 0; attribIndex < static_cast<int>(vtx_attrib_array.size()); ++attribIndex)
            {
                AttribStream& indexStream = indexStreames[attribIndex];
                AttribStream& weightStream = weightStreames[attribIndex];
                const int* indexPtr = static_cast<const int*>(indexStream.stream);
                const float* weightPtr = static_cast<const float*>(weightStream.stream);

                for (int i = 0; i < vertexCount; ++i)
                {
                    auto& boneList = iterVertex->boneLists[i];

                    for (int j = 0; j < indexStream.elemSize; ++j)
                    {
                        int indexIndex = i * indexStream.elemSize + j;
                        int boneIndex = indexPtr[indexIndex];
                        boneList.emplace_back(boneIndex);
                        // weight が存在しない場合はフルウェイト。
                        float boneWeight = weightStream.count ? weightPtr[indexIndex] : 1.0f;
                        auto found = std::find_if(iterVertex->boneIndices.begin(), iterVertex->boneIndices.end(), util::RefBoneWeightFinder(boneIndex));
                        if (found == iterVertex->boneIndices.cend())
                        {
                            if (boneIndex >= sizeMatrixPalette)
                            {
                                THROW_TRANSLATED_ERROR(ERRCODE_VERTEX_INVALID_BONE_INDEX,
                                    "Identifier_OutOfRangeWeightBone",
                                    boneIndex, std::distance(vertex_array.begin(), iterVertex));
                            }
                            util::RefBoneWeight refBoneWeight;
                            refBoneWeight.index = boneIndex;
                            refBoneWeight.weight = boneWeight;
                            iterVertex->boneIndices.push_back(refBoneWeight);
                        }
                        else
                        {
                            found->weight = std::max(found->weight, boneWeight);
                        }
                    }
                }
            }
        }
    }
    CATCH_THROW_XML_ERROR()
}

// shape が参照するマテリアルを index 関連付け
// マテリアルの mesh_adjancy を shape_info に適用
void
elem_model::ResolveShapeMaterial()
{
    try
    {
        for (auto iterShape = shape_array.begin(); iterShape != shape_array.end(); ++iterShape)
        {
            for (int matIndex = 0; matIndex < static_cast<int>(material_array.size()); ++matIndex)
            {
                if (iterShape->shape_info.mat_name.value == material_array[matIndex].name.value)
                {
                    iterShape->shape_info.mat_index = matIndex;
                    break;
                }
            }

            if (iterShape->shape_info.mat_index == -1)
            {
                THROW_TRANSLATED_ERROR(ERRCODE_XML_MATERIAL_NAME_NOT_FOUND, "Identifier_NotFoundValueIn", "mat_name", "shape", iterShape->name.value.c_str());
            }

            const elem_material& material = material_array[iterShape->shape_info.mat_index];
            if (material.material_info.mesh_adjacency.value)
            {
                iterShape->shape_info.mesh_adjacency = true;
            }
        }
    }
    CATCH_THROW_XML_ERROR()
}

// SamplerAssign 文字列から sampleIndex を解決
void
elem_model::ResolveSamplerIndex()
{
    try
    {
        for (auto iterMaterial = material_array.begin(); iterMaterial != material_array.end(); ++iterMaterial)
        {
            if (iterMaterial->shader_assign)
            {
                for (auto iterSamplerAssign = iterMaterial->shader_assign->sampler_assign_array.begin();
                    iterSamplerAssign != iterMaterial->shader_assign->sampler_assign_array.end();
                    ++iterSamplerAssign)
                {
                    int samplerIndex = 0;
                    for (auto iterSampler = iterMaterial->sampler_array.cbegin();
                        iterSampler != iterMaterial->sampler_array.cend();
                        ++iterSampler, ++samplerIndex)
                    {
                        if (iterSampler->name.value == iterSamplerAssign->sampler_name.value)
                        {
                            iterSamplerAssign->sampler_index = samplerIndex;
                            break;
                        }
                    }

                    // SamplerAssign が空文字でない場合に参照先が見つからなかったらエラー
                    if (iterSamplerAssign->sampler_index == -1 && !iterSamplerAssign->sampler_name.value.empty())
                    {
                        THROW_TRANSLATED_ERROR(ERRCODE_XML_SAMPLER_NAME_NOT_FOUND, "Identifier_NotFoundValueIn", iterSamplerAssign->sampler_name.value.c_str(), "material", iterMaterial->name.value.c_str())
                    }
                }
            }
        }
    }
    CATCH_THROW_XML_ERROR()
}

// 頂点position属性の unite_quantize_type の解決を行う
void elem_model::ResolveUniteQuantizeType()
{
    // ポジションを統一するために最大の量子化型を調べる。
    if (model_info.unite_pos_quantize.value)
    {
        bool yetAnalyzed = false;
        nw::g3d::tool::g3dif::elem_vtx_attrib::enum_quantize_type unite_quantize_type = nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_none;
        std::vector<nw::g3d::tool::g3dif::elem_vtx_attrib*> unite_vtx_attrib_array;

        for (auto iterVertex = vertex_array.begin(); iterVertex != vertex_array.end(); ++iterVertex)
        {
            for (auto vtx_attrib = iterVertex->vtx_attrib_array.begin(); vtx_attrib != iterVertex->vtx_attrib_array.end(); ++vtx_attrib)
            {
                if (vtx_attrib->semantic_name == nw::g3d::tool::g3dif::elem_vtx_attrib::position)
                {
                    if (!yetAnalyzed)
                    {
                        unite_quantize_type = vtx_attrib->quantize_type.value;
                        yetAnalyzed = true;
                    }
                    else
                    {
                        unite_quantize_type = nw::g3d::tool::g3dif::elem_vtx_attrib::Compare(unite_quantize_type, vtx_attrib->quantize_type.value);
                    }
                    unite_vtx_attrib_array.emplace_back(&(*vtx_attrib));
                    break;
                }
            }
        }

        // 収集した結果を書き込む
        for (auto unite_vtx_attrib = unite_vtx_attrib_array.begin(); unite_vtx_attrib != unite_vtx_attrib_array.end(); ++unite_vtx_attrib)
        {
            (*unite_vtx_attrib)->quantize_type.value = unite_quantize_type;
        }
    }
}

//--------------------------------------------------------------------------------------------------
void
elem_model::CheckData(int flag)
{
    for (auto iterMaterial = material_array.cbegin(); iterMaterial != material_array.cend(); ++iterMaterial)
    {
        if (iterMaterial->shader_assign)
        {
            for (auto iterRenderInfo = iterMaterial->shader_assign->render_info_array.cbegin(); iterRenderInfo != iterMaterial->shader_assign->render_info_array.cend(); ++iterRenderInfo)
            {
                bool inconsistentDataSize = false;
                if ((iterRenderInfo->type.value == elem_render_info::type_int) ||
                    (iterRenderInfo->type.value == elem_render_info::type_float))
                {
                    if (static_cast<int>(iterRenderInfo->dataSize / sizeof(float)) != iterRenderInfo->count.value)
                    {
                        inconsistentDataSize = true;
                    }
                }
                else if (iterRenderInfo->type.value == elem_render_info::type_string)
                {
                    if (static_cast<int>( iterRenderInfo->stringArray.size() ) != iterRenderInfo->count.value)
                    {
                        inconsistentDataSize = true;
                    }
                }
                if (inconsistentDataSize)
                {
                    THROW_TRANSLATED_ERROR(ERRCODE_XML_RENDER_INFO_COUNT, "Identifier_InvalidRenderInfo", iterRenderInfo->name.value.c_str(), iterMaterial->name.value.c_str());
                }
            }
        }
        else
        {
            if (util::CheckFlag(flag, CHECK_SHADER_ASSIGN))
            {
                THROW_TRANSLATED_ERROR(ERRCODE_XML_SHADER_ASSIGN_NOT_FOUND, "Identifier_NotFoundValueIn", "shader_assign", "material", iterMaterial->name.value.c_str());
            }
        }
    }

    for (auto iterVertex = vertex_array.cbegin(); iterVertex != vertex_array.cend(); ++iterVertex)
    {
        if (iterVertex->vtx_attrib_array.size() <= 0)
        {
            continue;
        }

        int attribCount = iterVertex->vtx_attrib_array.front().count.value;
        // vtx_attrib の count をチェックする。
        for (auto iterAttrib = iterVertex->vtx_attrib_array.cbegin(); iterAttrib != iterVertex->vtx_attrib_array.cend(); ++iterAttrib)
        {
            if (attribCount != iterAttrib->count.value)
            {
                THROW_TRANSLATED_ERROR( ERRCODE_XML_INVALID_VERTEX_ATTRIB_COUNT, "Identifier_InvalidVertexAttribute", iterAttrib->name.value.c_str(), iterAttrib->count.value);
            }
        }
    }
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
