﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/Material.h>
#include <util/UtilMath.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    shader_param, type,
    S11N_DEFINE_ENUM_ID(bool),
    S11N_DEFINE_ENUM_ID(bool2),
    S11N_DEFINE_ENUM_ID(bool3),
    S11N_DEFINE_ENUM_ID(bool4),
    S11N_DEFINE_ENUM_ID(int),
    S11N_DEFINE_ENUM_ID(int2),
    S11N_DEFINE_ENUM_ID(int3),
    S11N_DEFINE_ENUM_ID(int4),
    S11N_DEFINE_ENUM_ID(uint),
    S11N_DEFINE_ENUM_ID(uint2),
    S11N_DEFINE_ENUM_ID(uint3),
    S11N_DEFINE_ENUM_ID(uint4),
    S11N_DEFINE_ENUM_ID(float),
    S11N_DEFINE_ENUM_ID(float2),
    S11N_DEFINE_ENUM_ID(float3),
    S11N_DEFINE_ENUM_ID(float4),
    S11N_DEFINE_ENUM_ID(float2x2),
    S11N_DEFINE_ENUM_ID(float2x3),
    S11N_DEFINE_ENUM_ID(float2x4),
    S11N_DEFINE_ENUM_ID(float3x2),
    S11N_DEFINE_ENUM_ID(float3x3),
    S11N_DEFINE_ENUM_ID(float3x4),
    S11N_DEFINE_ENUM_ID(float4x2),
    S11N_DEFINE_ENUM_ID(float4x3),
    S11N_DEFINE_ENUM_ID(float4x4),
    S11N_DEFINE_ENUM_ID(srt2d),
    S11N_DEFINE_ENUM_ID(srt3d),
    S11N_DEFINE_ENUM_ID(texsrt),
    S11N_DEFINE_ENUM_ID(texsrt_ex)
    );

G3DIF_DEFINE_ENUM_CAST_TABLE(
    shader_param, type, size,
    1, 2, 3, 4,
    1, 2, 3, 4,
    1, 2, 3, 4,
    1, 2, 3, 4,
    4, 6, 8,
    6, 9, 12,
    8, 12, 16,
    5, 9,
    6, 7
    );

G3DIF_DEFINE_ENUM_TABLE(
    render_info, type,
    S11N_DEFINE_ENUM_ID(int),
    S11N_DEFINE_ENUM_ID(float),
    S11N_DEFINE_ENUM_ID(string)
    );

G3DIF_DEFINE_ENUM_TABLE(
    render_state, mode,
    S11N_DEFINE_ENUM_ID(custom),
    S11N_DEFINE_ENUM_ID(opaque),
    S11N_DEFINE_ENUM_ID(mask),
    S11N_DEFINE_ENUM_ID(translucent)
    );

G3DIF_DEFINE_ENUM_TABLE(
    render_state, display_face,
    S11N_DEFINE_ENUM_ID(both),
    S11N_DEFINE_ENUM_ID(front),
    S11N_DEFINE_ENUM_ID(back),
    S11N_DEFINE_ENUM_ID(none)
);

G3DIF_DEFINE_ENUM_TABLE(
    depth_test, func,
    S11N_DEFINE_ENUM_ID(never),
    S11N_DEFINE_ENUM_ID(equal),
    S11N_DEFINE_ENUM_ID(less),
    S11N_DEFINE_ENUM_ID(lequal),
    S11N_DEFINE_ENUM_ID(greater),
    S11N_DEFINE_ENUM_ID(gequal),
    S11N_DEFINE_ENUM_ID(nequal),
    S11N_DEFINE_ENUM_ID(always)
);

G3DIF_DEFINE_ENUM_TABLE(
    alpha_test, func,
    S11N_DEFINE_ENUM_ID(never),
    S11N_DEFINE_ENUM_ID(equal),
    S11N_DEFINE_ENUM_ID(less),
    S11N_DEFINE_ENUM_ID(lequal),
    S11N_DEFINE_ENUM_ID(greater),
    S11N_DEFINE_ENUM_ID(gequal),
    S11N_DEFINE_ENUM_ID(nequal),
    S11N_DEFINE_ENUM_ID(always)
);

G3DIF_DEFINE_ENUM_TABLE(
    color_blend, func,
    S11N_DEFINE_ENUM_ID(zero),
    S11N_DEFINE_ENUM_ID(one),
    S11N_DEFINE_ENUM_ID(src_color),
    S11N_DEFINE_ENUM_ID(one_minus_src_color),
    S11N_DEFINE_ENUM_ID(src_alpha),
    S11N_DEFINE_ENUM_ID(one_minus_src_alpha),
    S11N_DEFINE_ENUM_ID(dst_color),
    S11N_DEFINE_ENUM_ID(one_minus_dst_color),
    S11N_DEFINE_ENUM_ID(dst_alpha),
    S11N_DEFINE_ENUM_ID(one_minus_dst_alpha),
    S11N_DEFINE_ENUM_ID(const_color),
    S11N_DEFINE_ENUM_ID(one_minus_const_color),
    S11N_DEFINE_ENUM_ID(const_alpha),
    S11N_DEFINE_ENUM_ID(one_minus_const_alpha),
    S11N_DEFINE_ENUM_ID(src1_color),
    S11N_DEFINE_ENUM_ID(one_minus_src1_color),
    S11N_DEFINE_ENUM_ID(src1_alpha),
    S11N_DEFINE_ENUM_ID(one_minus_src1_alpha),
    S11N_DEFINE_ENUM_ID(src_alpha_saturate)
);

G3DIF_DEFINE_ENUM_TABLE(
    color_blend, op,
    S11N_DEFINE_ENUM_ID(add),
    S11N_DEFINE_ENUM_ID(src_minus_dst),
    S11N_DEFINE_ENUM_ID(dst_minus_src),
    S11N_DEFINE_ENUM_ID(min),
    S11N_DEFINE_ENUM_ID(max)
);

G3DIF_DEFINE_ENUM_TABLE(
    logical_blend, op,
    S11N_DEFINE_ENUM_ID(clear),
    S11N_DEFINE_ENUM_ID(set),
    S11N_DEFINE_ENUM_ID(copy),
    S11N_DEFINE_ENUM_ID(inv_copy),
    S11N_DEFINE_ENUM_ID(no_op),
    S11N_DEFINE_ENUM_ID(inv),
    S11N_DEFINE_ENUM_ID(and),
    S11N_DEFINE_ENUM_ID(nand),
    S11N_DEFINE_ENUM_ID(or),
    S11N_DEFINE_ENUM_ID(nor),
    S11N_DEFINE_ENUM_ID(xor),
    S11N_DEFINE_ENUM_ID(equiv),
    S11N_DEFINE_ENUM_ID(rev_and),
    S11N_DEFINE_ENUM_ID(inv_and),
    S11N_DEFINE_ENUM_ID(rev_or),
    S11N_DEFINE_ENUM_ID(inv_or)
);

G3DIF_DEFINE_ENUM_TABLE(
    render_state, blend_mode,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(color),
    S11N_DEFINE_ENUM_ID(logic)
);

G3DIF_DEFINE_ENUM_TABLE(
    wrap, mode,
    S11N_DEFINE_ENUM_ID(repeat),
    S11N_DEFINE_ENUM_ID(mirror),
    S11N_DEFINE_ENUM_ID(clamp),
    S11N_DEFINE_ENUM_ID(mirror_once),
);

G3DIF_DEFINE_ENUM_TABLE(
    filter, mode,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(point),
    S11N_DEFINE_ENUM_ID(linear)
);

G3DIF_DEFINE_ENUM_TABLE(
    filter, max_aniso,
    S11N_DEFINE_ENUM_ID(aniso_1),
    S11N_DEFINE_ENUM_ID(aniso_2),
    S11N_DEFINE_ENUM_ID(aniso_4),
    S11N_DEFINE_ENUM_ID(aniso_8),
    S11N_DEFINE_ENUM_ID(aniso_16)
);

namespace {

const elem_shader_param::ParamFormat s_ParamFormat[] = {
    { 1, 1 }, { 1, 2 }, { 1, 3 }, { 1, 4 },
    { 1, 1 }, { 1, 2 }, { 1, 3 }, { 1, 4 },
    { 1, 1 }, { 1, 2 }, { 1, 3 }, { 1, 4 },
    { 1, 1 }, { 1, 2 }, { 1, 3 }, { 1, 4 },
    { 2, 2 }, { 2, 3 }, { 2, 4 },
    { 3, 2 }, { 3, 3 }, { 3, 4 },
    { 4, 2 }, { 4, 3 }, { 4, 4 }
};

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
const elem_shader_param::ParamFormat& elem_shader_param::GetShaderParamFormat( elem_shader_param::enum_type type )
{
    return s_ParamFormat[type];
}

elem_shader_param::enum_type elem_shader_param::GetType(std::string str)
{
    elem_shader_param::enum_type type;
    Read(str.c_str(), type);
    return type;
}

std::string elem_shader_param::GetType(elem_shader_param::enum_type type)
{
    return elem_shader_param::tbl_type[type].str;
}

size_t elem_shader_param::GetSrcParamSize(elem_shader_param::enum_type type)
{
    return tbl_shader_param_type_size[type] * sizeof(float);
}


std::pair<void*, size_t> elem_shader_param::TextToValue(const char* text, enum_type type)
{
    void* pData = nullptr;
    int count = 0;
    if (type == type_srt2d)
    {
        // s(2) + r(1) + t(2)
        count = 5;
    }
    else if (type == type_texsrt ||
             type == type_texsrt_ex)
    {
        // mode(1) + s(2) + r(1) + t(2)
        count = 6;
    }
    else if (type == type_srt3d)
    {
        // s(3) + r(3) + t(3)
        count = 9;
    }
    else
    {
        const int numFormat = sizeof(s_ParamFormat) / sizeof(ParamFormat);
        const ParamFormat& format = s_ParamFormat[type % numFormat];
        if ((type / (numFormat * 4)) == 0)
        {
            count = format.row * format.col;
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_XML_UNKNOWN_ENUM, "Identifier_InvalidValue", "shader param format");
        }
    }

    if (type <= type_uint4)
    {
        pData = ReadArray<int>(text, count);
    }
    else
    {
        pData = ReadArray<float>(text, count);
    }

    if (type == type_texsrt ||
        type == type_texsrt_ex)
    {
        // mode を float から uint へ変換する
        u32 mode = static_cast<u32>(*static_cast<float*>(pData));
        memcpy(pData, &mode, sizeof(u32));
    }

    return std::make_pair(pData, count * sizeof(float));
}

std::pair<void*, size_t> elem_render_info::TextToValue(const char* text, int count, enum_type type)
{
    void* pData = nullptr;
    if (type == type_int)
    {
        pData = ReadArray<int>(text, count);
    }
    else if (type == type_float)
    {
        pData = ReadArray<float>(text, count);
    }
    else
    {
        THROW_TRANSLATED_ERROR(ERRCODE_XML_UNKNOWN_ENUM, "Identifier_InvalidValue", "shader param format.");
    }

    return std::make_pair(pData, count * sizeof(float));
}

void elem_shader_param::ValueToSrcParam(void* pDst, elem_shader_param::enum_type type)
{
    // 回転値を degree から radian に変換します。
    if (type == elem_shader_param::type_srt2d)
    {
        float* rotateValue = util::AddOffset<float>(pDst, ROTATE_2D_COMPONENT_INDEX * sizeof(float));
        *rotateValue = util::DegToRad(*rotateValue);
    }
    else if (type == elem_shader_param::type_texsrt ||
             type == elem_shader_param::type_texsrt_ex)
    {
        float* rotateValue = util::AddOffset<float>(pDst, ROTATE_2D_COMPONENT_INDEX_SRT * sizeof(float));
        *rotateValue = util::DegToRad(*rotateValue);
    }
    else if (type == g3dif::elem_shader_param::type_srt3d)
    {
        float* rotateValueX = util::AddOffset<float>(pDst, ROTATE_3D_COMPONENT_INDEX_X * sizeof(float));
        *rotateValueX = util::DegToRad(*rotateValueX);

        float* rotateValueY = util::AddOffset<float>(pDst, ROTATE_3D_COMPONENT_INDEX_Y * sizeof(float));
        *rotateValueY = util::DegToRad(*rotateValueY);

        float* rotateValueZ = util::AddOffset<float>(pDst, ROTATE_3D_COMPONENT_INDEX_Z * sizeof(float));
        *rotateValueZ = util::DegToRad(*rotateValueZ);
    }
}
//--------------------------------------------------------------------------------------------------

void elem_shader_param::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        type << pElem;
        depend << pElem;

        textData = pElem->Text();
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------

void elem_attrib_assign::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        attrib_name << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------

void elem_shader_option::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        value << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_sampler_assign::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        sampler_name << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shader_assign::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        shading_model << pElem;
        shader_archive << pElem;
        revision << pElem;

        render_info_array << pElem->Child(elem_render_info::IdArray());
        shader_option_array << pElem->Child(elem_shader_option::IdArray());
        sampler_assign_array << pElem->Child(elem_sampler_assign::IdArray());
        shader_param_array << pElem->Child(elem_shader_param::IdArray());
        attrib_assign_array << pElem->Child(elem_attrib_assign::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------

void elem_depth_test::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        enable << pElem;
        write << pElem;
        func << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_alpha_test::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        enable << pElem;
        func << pElem;
        value << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_color_blend::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        rgb_src_func << pElem;
        rgb_dst_func << pElem;
        rgb_op << pElem;
        alpha_src_func << pElem;
        alpha_dst_func << pElem;
        alpha_op << pElem;
        const_color << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_logical_blend::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        op << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_render_state::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        mode << pElem;
        display_face << pElem;
        blend_mode << pElem;

        depth_test << pElem->Child(elem_depth_test::Id());
        alpha_test << pElem->Child(elem_alpha_test::Id());
        color_blend << pElem->Child(elem_color_blend::Id());
        logical_blend << pElem->Child(elem_logical_blend::Id());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_render_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        type << pElem;
        count << pElem;

        // データへのポインタを入れておく
        textData = pElem->Text();
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------

void elem_wrap::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        u << pElem;
        v << pElem;
        w << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_filter::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        mag << pElem;
        min << pElem;
        mip << pElem;
        max_aniso << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_lod::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        min << pElem;
        max << pElem;
        bias << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_sampler::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        tex_name << pElem;

        wrap << pElem->Child(elem_wrap::Id());
        filter << pElem->Child(elem_filter::Id());
        lod << pElem->Child(elem_lod::Id());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_material_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        visibility << pElem;
        mesh_adjacency << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------

void elem_material::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;

        material_info << pElem->Child(elem_material_info::Id());
        render_state << pElem->Child(elem_render_state::Id());
        sampler_array << pElem->Child(elem_sampler::IdArray());
        shader_assign << pElem->Child(elem_shader_assign::Id());
        user_data_array << pElem->Child(elem_user_data::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_material::Expand()
{
    if (shader_assign)
    {
        for (auto render_info = shader_assign->render_info_array.begin(); render_info != shader_assign->render_info_array.end(); ++render_info)
        {
            if (render_info->type.value == elem_render_info::type_string)
            {
                util::Split(render_info->stringArray, render_info->textData, " \t\n");
            }
            else
            {
                std::pair<void*, size_t> data;
                try
                {
                    data = elem_render_info::TextToValue(render_info->textData, render_info->count.value, render_info->type.value);
                }
                CATCH_THROW_WITH_MSG("Error occurred in render_info \"%hs\" in material \"%hs\"",
                                     render_info->name.value.c_str(),
                                     this->name.value.c_str() );
                render_info->dataSize = data.second;
                render_info->rawdata.reset(data.first, free);
            }
        }

        for (auto param = shader_assign->shader_param_array.begin();
            param != shader_assign->shader_param_array.end();
            ++param)
        {
            std::pair<void*, size_t> data;
            try
            {
                data = elem_shader_param::TextToValue(param->textData, param->type.value);
            }
            CATCH_THROW_WITH_MSG("Error occurred in shader_param \"%hs\" in material \"%hs\"",
                                 param->id.value.c_str(),
                                 this->name.value.c_str() );
            param->dataSize = data.second;
            param->rawdata.reset(data.first, free);
        }
    }
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
