﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <g3dif/Serialization.h>
#include <g3dif/UserData.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

class ScaleAdjuster
{
public:
    static float ZeroToMin(float src)
    {
        return fabs(src) < s_MinScale ? ( src < 0.0f ? -s_MinScale : s_MinScale ) : src;
    }

    template <typename T>
    static float ZeroToMin(float src, float scale, float offset)
    {
        const float maxValue = static_cast<float>(std::numeric_limits<T>::max());
        const float minValue = static_cast<float>(std::numeric_limits<T>::lowest());
        float d_float = util::Clamp((src - offset) / scale, minValue, maxValue);
        T d_T = util::Round<T>(d_float);
        float key = static_cast<float>(d_T) * scale + offset;
        if (fabs(key) < s_MinScale)
        {
            src = ( s_MinScale > scale ?
                ( src < 0.0f ? -s_MinScale : s_MinScale ) :
                ( src < 0.0f ? -scale : scale ) );
        }
        return src;
    }
private:
    static const float s_MinScale;
};

class elem_bone
{
public:
    G3DIF_DEFINE_ENUM(billboard,
        none,
        world_viewvector,
        world_viewpoint,
        screen_viewvector,
        screen_viewpoint,
        yaxis_viewvector,
        yaxis_viewpoint
    );

    G3DIF_DEFINE_ENUM(side,
        side_none,
        side_left,
        side_right,
        side_center
    );

    elem_bone()
        : parent_index(-1)
        , name()
        , parent_name()
        , matrix_index()
        , scale()
        , rotate()
        , translate()
        , scale_compensate(false)
        , billboard(none)
        , visibility(true)
        , side(side_none)
        , inv_model_matrix()
        , user_data_array()
    {
        scale->x = scale->y = scale->z = rotate->w = 1.0f;
        rotate->x = rotate->y = rotate->z = translate->x = translate->y = translate->z = 0.0f;
    }

    G3DIF_DEFINE_ELEM_ARRAY(bone);

    int parent_index;

    G3DIF_DEFINE_ATTRIB(std::string, name);
    G3DIF_DEFINE_ATTRIB(std::string, parent_name);
    G3DIF_DEFINE_ATTRIB(IntStream, matrix_index);
    G3DIF_DEFINE_ATTRIB(util::Vec3_t, scale);
    G3DIF_DEFINE_ATTRIB(util::Quat_t, rotate);
    G3DIF_DEFINE_ATTRIB(util::Vec3_t, translate);
    G3DIF_DEFINE_ATTRIB(bool, scale_compensate);
    G3DIF_DEFINE_ATTRIB(enum_billboard, billboard);
    G3DIF_DEFINE_ATTRIB(bool, visibility);
    G3DIF_DEFINE_ATTRIB(enum_side, side);

    util::Optional<util::Mtx34_t> inv_model_matrix;
    std::vector<elem_user_data> user_data_array;
};

class elem_skeleton_info
{
public:
    G3DIF_DEFINE_ENUM(scale_mode,
        standard,
        maya,
        softimage
    );

    G3DIF_DEFINE_ENUM(rotate_mode,
        euler_xyz,
        quaternion
        );

    elem_skeleton_info()
        : scale_enable(true)
        , scale_mode(elem_skeleton_info::standard)
        , rotate_mode(elem_skeleton_info::euler_xyz)
        , motion_mirroring_enable(false)
    {
    }

    G3DIF_DEFINE_ELEM(skeleton_info);

    G3DIF_DEFINE_ATTRIB(bool, scale_enable);
    G3DIF_DEFINE_ATTRIB(enum_scale_mode, scale_mode);
    G3DIF_DEFINE_ATTRIB(enum_rotate_mode, rotate_mode);
    G3DIF_DEFINE_ATTRIB(bool, motion_mirroring_enable);
};

class elem_skeleton
{
public:
    struct MatrixPalette
    {
        int matrixIndex;
        int boneIndex;
    };

    elem_skeleton()
        : skeleton_info()
        , bone_array()
        , matrixPalette()
    {
    }

    G3DIF_DEFINE_ELEM(skeleton);

    void ResolveSkeleton();

    elem_skeleton_info skeleton_info;
    std::vector<elem_bone> bone_array;

    std::vector<MatrixPalette> matrixPalette;
};

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
