﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <unordered_map>
#include <sstream>
#include <util/UtilString.h>
#include <util/UtilFlag.h>
#include <util/UtilTime.h>
#include <util/UtilSerialization.h>
#include <g3dif/Identifier.h>

#define G3DIF_DEFINE_ELEM(name) \
    static const nw::g3d::tool::util::LiteralStr& Id() { return nw::g3d::tool::g3dif::tbl_identifier[nw::g3d::tool::g3dif::id_##name]; }\
    void operator<<(const nw::g3d::tool::util::XMLElement* pElem) \


#define G3DIF_DEFINE_ELEM_ARRAY(name) \
    static const nw::g3d::tool::util::LiteralStr& Id() { return nw::g3d::tool::g3dif::tbl_identifier[nw::g3d::tool::g3dif::id_##name]; }\
    static const nw::g3d::tool::util::LiteralStr& IdArray() { return nw::g3d::tool::g3dif::tbl_identifier[nw::g3d::tool::g3dif::id_##name##_array]; }\
    mutable int index; \
    void operator<<(const nw::g3d::tool::util::XMLElement* pElem) \

#define G3DIF_DEFINE_ATTRIB(type, name) nw::g3d::tool::util::Attribute<type, nw::g3d::tool::g3dif::id_##name> name


#define CATCH_THROW_XML_ERROR() \
    catch (nw::g3d::tool::util::Exception&) \
    { \
        PRINT_ERROR_LOG("<%hs>", Id().str); \
        PRINT_TRACE(); \
        throw; \
    } \


#define G3DIF_DEFINE_ENUM(name, ...) \
    enum enum_##name \
    { \
        __VA_ARGS__, \
        num_##name \
    }; \
    static const util::LiteralStr tbl_##name[]; \
    friend void Read(const char* str, enum_##name& value); \
    friend const util::LiteralStr& Id(enum_##name value) \


#define G3DIF_DEFINE_ENUM_TABLE(type, name, ...) \
    const util::LiteralStr elem_##type::tbl_##name[] = { \
        __VA_ARGS__ \
    }; \
    void Read(const char* str, elem_##type::enum_##name& value) \
    { \
    value = elem_##type::enum_##name(FindEnum( \
        elem_##type::tbl_##name, elem_##type::num_##name, str)); \
    } \
    const util::LiteralStr& Id(elem_##type::enum_##name value) \
    { \
        if (value < 0 || elem_##type::num_##name <= value) \
        { \
            THROW_ERROR(ERRCODE_XML_UNKNOWN_ENUM, "Unknown enum value. %d", value); \
        } \
        return elem_##type::tbl_##name[value]; \
    } \
    static_assert(sizeof(elem_##type::tbl_##name) == \
        sizeof(util::LiteralStr) * elem_##type::num_##name, \
        "Invalid table size.") \

#define G3DIF_DEFINE_ENUM_CAST_TABLE(type, name, table, ...)                                         \
    namespace {                                                                                      \
    const u32 tbl_##type##_##name##_##table[] = {                                                    \
    __VA_ARGS__                                                                                      \
};                                                                                                   \
}                                                                                                    \
    u32 ToEnum_##table(g3dif::elem_##type::enum_##name value)                                        \
{                                                                                                    \
    if (!(0 <= value || value < g3dif::elem_##type::num_##name))                                     \
    {                                                                                                \
        THROW_ERROR(ERRCODE_XML_UNKNOWN_ENUM, "Unknown enum value. %d", value);                      \
    }                                                                                                \
    return tbl_##type##_##name##_##table[value];                                                     \
}                                                                                                    \
    static_assert(sizeof(tbl_##type##_##name##_##table) / sizeof(u32) == g3dif::elem_##type::num_##name, \
    "Invalid table size.")


#define G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, value)                                            \
    outStream.write(reinterpret_cast<const char*>(&value), sizeof(value))                            \

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

using util::s8;
using util::s16;
using util::s32;
using util::s64;

using util::u8;
using util::u16;
using util::u32;
using util::u64;

using util::bit8;
using util::bit16;
using util::bit32;
using util::bit64;

using util::uint;
using util::char16;

enum
{
    STREAM_BINARY_ALIGNMENT     = 32,
    STREAM_CHUNK_ALIGNMENT      = 32,
    STREAM_CHUNK_SIZE_ALIGNMENT = 32
};

// 中間ファイルのバージョンはここで指定します。
// g3d_defs.h のバージョンを上げた場合必ず更新してください。

struct Version
{
    int major;
    int minor;
    int micro;
};

// 全拡張子共通のコンテナバージョン
#define NN_G3D_INTERMEDIATE_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_VERSION_MICRO 0

// 各拡張子別のバージョン
// モデル
#define NN_G3D_INTERMEDIATE_FMD_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FMD_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FMD_VERSION_MICRO 0
// マテリアルアニメーション
#define NN_G3D_INTERMEDIATE_FMA_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FMA_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FMA_VERSION_MICRO 0
// シェイプアニメーション
#define NN_G3D_INTERMEDIATE_FSH_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FSH_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FSH_VERSION_MICRO 0
// スケルタルアニメーション
#define NN_G3D_INTERMEDIATE_FSK_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FSK_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FSK_VERSION_MICRO 0
// シーンアニメーション
#define NN_G3D_INTERMEDIATE_FSN_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FSN_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FSN_VERSION_MICRO 0
// シェーダーパラメーターアニメーション
#define NN_G3D_INTERMEDIATE_FSP_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FSP_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FSP_VERSION_MICRO 0
// テクスチャーパターンアニメーション
#define NN_G3D_INTERMEDIATE_FTP_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FTP_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FTP_VERSION_MICRO 0
// ボーンビジビリティーアニメーション
#define NN_G3D_INTERMEDIATE_FVB_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FVB_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FVB_VERSION_MICRO 0
// マテリアルビジビリティーアニメーション
#define NN_G3D_INTERMEDIATE_FVM_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FVM_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FVM_VERSION_MICRO 0

// シェーダー設定
#define NN_G3D_INTERMEDIATE_FSC_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FSC_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FSC_VERSION_MICRO 0
// シェーダー定義
#define NN_G3D_INTERMEDIATE_FSD_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FSD_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FSD_VERSION_MICRO 0
// シェーダーバリエーション
#define NN_G3D_INTERMEDIATE_FSV_VERSION_MAJOR 4
#define NN_G3D_INTERMEDIATE_FSV_VERSION_MINOR 0
#define NN_G3D_INTERMEDIATE_FSV_VERSION_MICRO 0

enum StreamType
{
    StreamTypeFloat     = 0,
    StreamTypeInt       = 1,
    StreamTypeByte      = 2,
    StreamTypeString    = 3,
    StreamTypeWString   = 4
};

const u64 STREAM_HEADER_ORDER = 0x616D727473643367;
const u64 STREAM_CHUNK_HEADER_ORDER = 0x20206D6165727473;

struct ChunkOffset
{
    u32 offset;
    u32 size;
};

struct StreamChunk
{
    u64 header;
    StreamType type;
    s32 count;
    u32 column;
    u32 size;

    u64 padding;

    void* data;
};

struct StreamArray
{
    u64 header;
    u32 chunkCount;
    std::vector<ChunkOffset> chunkOffset;

    // StreamChunk が 32 バイトで揃うように padding

    std::vector<StreamChunk> streamChunk;
};

struct IntStream
{
    std::shared_ptr<int> data;
    u32 size;
};

struct UIntStream
{
    std::shared_ptr<unsigned int> data;
    size_t size;
};

template <typename T>
inline
T Parse(const char* nptr, char** endptr = nullptr)
{
    static_assert(false, "Undefined type.");
    return T();
}

template <>
inline
float Parse(const char* nptr, char** endptr)
{
    float value = static_cast<float>(std::strtod(nptr, endptr));

    if ((endptr != NULL) && (**endptr == 'f' || **endptr == 'F'))
    {
        ++(*endptr);
    }

    return value;
}

template <>
inline
int Parse(const char* nptr, char** endptr)
{
    return std::strtol(nptr, endptr, 10);
}

template <>
inline
unsigned int Parse(const char* nptr, char** endptr)
{
    return std::strtoul(nptr, endptr, 10);
}

template <>
inline
IntStream Parse(const char* nptr, char** endptr)
{
    (void)endptr;

    IntStream stream;
    std::vector<std::string> v;
    util::Split(v, std::string(nptr), " ");
    stream.size = static_cast<u32>(v.size());

    int* pData = static_cast<int*>(malloc(stream.size * sizeof(int)));
    for (int i = 0; i < static_cast<int>(stream.size); ++i)
    {
        std::istringstream is(v[i]);
        is >> pData[i];
    }
    stream.data.reset(pData, free);

    return stream;
}

template <>
inline
UIntStream Parse(const char* nptr, char** endptr)
{
    (void)endptr;

    UIntStream stream;
    std::vector<std::string> v;
    util::Split(v, std::string(nptr), " ");
    stream.size = v.size();

    unsigned int* pData = static_cast<unsigned int*>(malloc(stream.size * sizeof(unsigned int)));
    for (int i = 0; i < static_cast<int>(stream.size); ++i)
    {
        std::istringstream is(v[i]);
        is >> pData[i];
    }
    stream.data.reset(pData, free);

    return stream;
}

template <>
inline
unsigned char Parse(const char* nptr, char** endptr)
{
    return static_cast<unsigned char>(std::strtoul(nptr, endptr, 16));
}

template <typename T>
inline
T* ReadArray(const char* str, int count, T* pData = nullptr)
{
    if (str == nullptr)
    {
        str = "";
    }
    if(pData == nullptr)
    {
        pData = static_cast<T*>(malloc(count * sizeof(T)));
    }
    int index = 0;
    T value;
    char* endptr = nullptr;

    while (index < count)
    {
        value = Parse<T>(str, &endptr);
        if (str == endptr)
        {
            break;
        }
        str = endptr;
        pData[index++] = value;
    }
    // 値が足りなかった場合は例外を投げる。
    if (index < count)
    {
        THROW_ERROR(ERRCODE_XML_STREAM_TOO_SHORT,
            "Too short stream. Expected: %d Actual: %d", count, index);
    }
    return pData;
}

template <typename T>
inline
T* ReadArray(const void* data, int count, T* pData = nullptr)
{
    T* pData = reinterpret_cast<T*>(malloc(count * sizeof(T)));
    memcpy(pData, data, count);
    return pData;
}

template <typename T>
inline
void Read(const char* str, T& value)
{
    value = Parse<T>(str);
}

template <>
inline
void Read(const char* str, bool& value)
{
    if (str)
    {
        // 先頭のみで判定。他のパターンは必要なら追加。
        if (str[0] == 't')
        {
            value = true;
            return;
        }
    }
    value = false;
}

template <>
inline
void Read(const char* str, std::string& value)
{
    value = str ? str : "";
}

template <> inline void Read(const char* str, util::Vec2_t& value) { ReadArray(str, 2, value.a); }
template <> inline void Read(const char* str, util::Vec3_t& value) { ReadArray(str, 3, value.a); }
template <> inline void Read(const char* str, util::Vec4_t& value) { ReadArray(str, 4, value.a); }
template <> inline void Read(const char* str, util::Quat_t& value) { ReadArray(str, 4, value.a); }
template <> inline void Read(const char* str, util::Mtx22_t& value) { ReadArray(str, 4, value.a); }
template <> inline void Read(const char* str, util::Mtx23_t& value) { ReadArray(str, 6, value.a); }
template <> inline void Read(const char* str, util::Mtx33_t& value) { ReadArray(str, 9, value.a); }
template <> inline void Read(const char* str, util::Mtx34_t& value) { ReadArray(str, 12, value.a); }
template <> inline void Read(const char* str, util::Mtx44_t& value) { ReadArray(str, 16, value.a); }

// テキストとして読み込んだデータを適切な型に変換してコピーします。
std::shared_ptr<void> AnalizeAndCopyData(const char* textData, int count, StreamType type);

void ExpandChoice(std::vector<std::pair<std::string, std::string>>& choiceArray, const std::string& choice);
void ExpandChoiceUint(std::vector<uint32_t>& choiceUintArray, const std::vector<std::pair<std::string, std::string>>& choiceArray);
int IncludeValue(const std::vector<std::pair<std::string, std::string>>& choiceArray, const std::string& value);

//--------------------------------------------------------------------------------------------------

inline std::string GetBoolString(bool val)
{
    return val ? "true" : "false";
}

inline bool GetBoolValue(std::string str)
{
    return (str == "true") ? true : false;
}

template <typename T>
void operator<<(std::vector<T>& elem_array, const util::XMLElement* pArray)
{
    if (pArray == nullptr)
    {
        return;
    }

    // 要素数を取得する。
    std::vector<const util::XMLElement*> nodeArray;
    for (const util::XMLElement* pElem = pArray->Child(T::Id()); pElem;
        pElem = pElem->NextSibling(T::Id()))
    {
        nodeArray.push_back(pElem);
    }

    if (nodeArray.empty())
    {
        return;
    }

    elem_array.resize(nodeArray.size());
    int index = 0;
    for (auto iter = nodeArray.cbegin(); iter != nodeArray.cend(); ++iter, ++index)
    {
        elem_array[index] << *iter;
    }
}

template <typename T>
inline
void operator<<(util::Optional<T>& elem, const util::XMLElement* pElem)
{
    if (pElem)
    {
        elem.Validate();
        elem.Get() << pElem;
    }
}

template <typename T>
inline
void operator<<(util::Optional<std::vector<T>>& elem_array, const util::XMLElement* pArray)
{
    if (pArray)
    {
        // 要素数を取得する。
        std::vector<const util::XMLElement*> nodeArray;
        for (const util::XMLElement* pElem = pArray->Child(T::Id()); pElem;
        pElem = pElem->NextSibling(T::Id()))
        {
            nodeArray.push_back(pElem);
        }

        if (nodeArray.empty())
        {
            return;
        }
        elem_array.Validate();
        auto& elem = elem_array.Get();
        elem.resize(nodeArray.size());
        int index = 0;
        for (auto iter = nodeArray.cbegin(); iter != nodeArray.cend(); ++iter, ++index)
        {
            elem[index] << *iter;
        }
    }
}

template <typename T, int identifier>
inline
void operator<<(util::Attribute<T, identifier>& attrib, const util::XMLElement* pElem)
{
    const util::LiteralStr& id = tbl_identifier[identifier];
    const char* str = pElem->Attribute(id);
    if (str == nullptr)
    {
        THROW_ERROR(ERRCODE_XML_ATTRIBUTE_NOT_FOUND, "%hs attribute is not found. Please add %hs attribute.", id.str, id.str);
    }

    try
    {
        Read(str, attrib.value);
    }
    catch (util::Exception&)
    {
        PRINT_ERROR_LOG("attribute: %hs", id.str);
        PRINT_TRACE();
        throw;
    }
}

// 中間ファイルバージョン読み込み
template <int identifier>
inline
void operator<<(util::Attribute<Version, identifier>& attrib, const util::XMLElement* pElem)
{
    const util::LiteralStr& id = tbl_identifier[identifier];
    const char* str = pElem->Attribute(id);
    if (str == nullptr)
    {
        THROW_ERROR(ERRCODE_XML_ATTRIBUTE_NOT_FOUND, "%hs attribute is not found. Please add %hs attribute.", id.str, id.str);
    }

    try
    {
        std::vector<std::string> v;
        util::Split(v, str, ".");

        // 中間ファイルのバージョン番号が不正な値
        if (v.size() != 3)
        {
            THROW_ERROR(ERRCODE_XML_INVALID_VERSION,
                "Version number (%s) is invalid.",
                str);
        }

        Read(v[0].c_str(), attrib.value.major);
        Read(v[1].c_str(), attrib.value.minor);
        Read(v[2].c_str(), attrib.value.micro);
    }
    catch (util::Exception&)
    {
        PRINT_ERROR_LOG("attribute: %hs", id.str);
        PRINT_TRACE();
        throw;
    }
}

inline
void VerifyElement(const util::XMLElement* pElem, const util::LiteralStr& id)
{
    if (pElem == nullptr)
    {
        THROW_ERROR(ERRCODE_XML_ELEMENT_NOT_FOUND, "<%hs> element is not found. Please add %hs element.", id.str, id.str);
    }
}

inline
int FindEnum(const util::LiteralStr table[], int count, const char* str)
{
    if (str)
    {
        for (int i = 0; i < count; ++i)
        {
            if (0 == strncmp(str, table[i].str, table[i].len + 1))
            {
                return i;
            }
        }
    }
    else
    {
        str = "";
    }
    THROW_ERROR(ERRCODE_XML_UNKNOWN_ENUM, "Unknown enum string. %hs", str);
    return -1;
}

class elem_nw4f_3dif
{
public:
    elem_nw4f_3dif()
        : version()
    {
    }

    G3DIF_DEFINE_ELEM(nw4f_3dif);

    G3DIF_DEFINE_ATTRIB(Version, version);
};

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
