﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using nw.g3d.nw4f_3dif;
using Nintendo.Foundation.IO;

namespace nw.g3d.toollib
{
    // シンプルフィルタツール
    public abstract class SimpleFilterTool<TParam> where TParam : SimpleFilterToolParams, new()
    {
        // コンストラクタ
        public SimpleFilterTool(string stringResourcePath, bool requireXsdCheck = false)
        {
            this.StringResourcePath = stringResourcePath;
            this.RequireXsdCheck = requireXsdCheck;

            // エラー時にポーズしない
            G3dToolUtility.EnableErrorPause = false;
        }

        // 実行
        protected void Run()
        {
            // 全体計測
            Stopwatch swRun = new Stopwatch();
            swRun.Start();
            {
                // 正常終了できなければ、異常終了
                Environment.ExitCode = 1;

                // ローカライズ処理
                Strings.Initialize(this.StringResourcePath);
                G3dLocalization.SetCulture();

                // 初期化
                Stopwatch swInitialize = new Stopwatch();
                swInitialize.Start();
                {
                    if (!Initialize())
                    {
                        Environment.ExitCode = 0;
                        return;
                    }
                }
                swInitialize.Stop();
                WriteToolProgress("SimpleFilterTool_Initialize",
                    this.FilePaths.Count, swInitialize.ElapsedMilliseconds);

                // 処理
                Process(this.ProcessFile);
                Environment.ExitCode = 0;
            }
            swRun.Stop();
            WriteToolProgress("SimpleFilterTool_Finish",
                this.FilePaths.Count, swRun.ElapsedMilliseconds);
        }

        //---------------------------------------------------------------------
        // 初期化
        private bool Initialize()
        {
            TParam param;
            var setting = new CommandLineParserSettings()
            {
                ApplicationDescription = SimpleFilterToolParams.DescriptionConverter("ApplicationDescription", null),
                // 例外処理に任せる
                ErrorWriter = x => { },
            };

            try
            {
                if (!(new CommandLineParser(setting).ParseArgs(Environment.GetCommandLineArgs().Skip(1), out param)))
                {
                    return false;
                }
            }
            catch
            {
                throw;
            }

            ProcessProgramArgument(param);

            foreach (var path in param.Paths)
            {
                G3dToolUtility.CollectPath(
                    this.FilePaths, path, this.SearchPattern, CheckFilePath);
            }

            return true;
        }

        protected virtual void ProcessProgramArgument(TParam param)
        {
            // 進捗表示切り替え
            this.Silent = param.Silent;

            // 並列処理数
            G3dParallel.Job = param.Job;

            var simpleFilterToolParamsNotRequireXsdCheck = param as SimpleFilterToolParamsNotRequireXsdCheck;
            if (simpleFilterToolParamsNotRequireXsdCheck != null)
            {
                if (!simpleFilterToolParamsNotRequireXsdCheck.DisableXsdCheck)
                {
                    this.XsdBasePath = G3dToolUtility.GetXsdBasePath();
                }
            }
            else
            {
                this.XsdBasePath = G3dToolUtility.GetXsdBasePath();
                if (this.XsdBasePath == null)
                {
                    TlStrings.Throw("SimpleFilterTool_Error_XsdNotFound");
                }
            }

            // 有効な拡張子
            if (!string.IsNullOrEmpty(param.EnableExtension))
            {
                this.EnableExtensions = G3dToolUtility.GetExtensions(param.EnableExtension);
            }

            // 無効な拡張子
            if (!string.IsNullOrEmpty(param.DisableExtension))
            {
                this.DisableExtensions = G3dToolUtility.GetExtensions(param.DisableExtension);
            }
        }

        // ファイルパスのチェック
        protected virtual bool CheckFilePath(string filePath)
        {
            // 有効な拡張子チェック
            if (this.EnableExtensions != null)
            {
                string extension = Path.GetExtension(filePath);
                bool enable = false;
                foreach (string enableExtension in this.EnableExtensions)
                {
                    if (extension == enableExtension) { enable = true; }
                }
                if (!enable) { return false; }
            }

            // 無効な拡張子チェック
            if (this.DisableExtensions != null)
            {
                string extension = Path.GetExtension(filePath);
                foreach (string disableExtension in this.DisableExtensions)
                {
                    if (extension == disableExtension) { return false; }
                }
            }
            return true;
        }

        //---------------------------------------------------------------------
        // 処理
        protected virtual void Process(ProcessFileDelegate processFile)
        {
            ExceptionListException allExceptions = new ExceptionListException();

            G3dParallel.For(0, this.FilePaths.Count, delegate(int i)
                {
#if !DEBUG
                    try
#endif
                    {
                        Stopwatch swProcessFile = new Stopwatch();
                        swProcessFile.Start();
                        {
                            processFile(i, this.FilePaths[i]);
                        }
                        swProcessFile.Stop();
                        WriteToolProgress("SimpleFilterTool_ProcessFile",
                            swProcessFile.ElapsedMilliseconds, this.FilePaths[i]);
                    }
#if !DEBUG
                    catch (Exception exp)
                    {
                        lock (allExceptions)
                        {
                            allExceptions.Exceptions.Add(
                                new Exception(this.FilePaths[i], exp));
                        }
                    }
#endif
                });

            if (allExceptions.Exceptions.Count > 0) { throw allExceptions; }
        }

        // ファイル処理のデリゲート
        protected delegate void ProcessFileDelegate(int index, string filePath);

        // ファイル処理
        protected abstract void ProcessFile(int index, string filePath);

        //---------------------------------------------------------------------
        // ツール進捗の書き出し
        private void WriteToolProgress(string id, params object[] args)
        {
            if (this.Silent) { return; }
            string progress = TlStrings.Get(id, args);
            // Console、Debug はスレッドセーフ
            Console.WriteLine(progress);
            Debug.WriteLine(progress);
        }

        // 進捗の書き出し
        protected void WriteProgress(string id, params object[] args)
        {
            if (this.Silent) { return; }
            string progress = Strings.Get(id, args);
            // Console、Debug はスレッドセーフ
            Console.WriteLine(progress);
            Debug.WriteLine(progress);
        }

        //---------------------------------------------------------------------
        protected string XsdBasePath { get; set; }
        protected readonly List<string> FilePaths = new List<string>();

        private string StringResourcePath { get; set; }
        private bool RequireXsdCheck { get; set; }
        private bool Silent { get; set; }
        private int SingleCore { get; set; }
        private string SearchPattern { get; set; }
        private List<string> EnableExtensions { get; set; }
        private List<string> DisableExtensions { get; set; }
    }
}
