﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Windows.Forms;

namespace nw.g3d.toollib
{
    // 例外リストを持つ例外
    public class ExceptionListException : Exception
    {
        public readonly List<Exception> Exceptions = new List<Exception>();
    }

    public static class G3dToolUtility
    {
        // 実行関数
        public delegate void RunMethod();

        // エラー発生時のポーズ
        public static bool EnableErrorPause { get; set; } = true;

        // 実行
        public static void Run(RunMethod runMethod)
        {
            Thread.GetDomain().UnhandledException +=
                G3dToolUtility.CatchUnhandledException;
#if !DEBUG
            try
            {
                runMethod();
            }
            catch (Exception exception)
            {
                // 例外の表示
                G3dToolUtility.DumpException(exception);
            }
#else
            // デバッグ版では何もしない
            runMethod();
#endif

            if ((Environment.ExitCode == 1) && G3dToolUtility.EnableErrorPause)
            {
                Console.ReadKey();
            }
        }

        // 未処理例外のキャッチ
        private static void CatchUnhandledException(
            object sender, UnhandledExceptionEventArgs args)
        {
#if !DEBUG
            G3dToolUtility.DumpException(args.ExceptionObject as Exception);
            if (G3dToolUtility.EnableErrorPause) { Console.ReadKey(); }
            Environment.Exit(1);
#endif
        }

        private static string GetErrorLogFolderPath()
        {
            return Environment.ExpandEnvironmentVariables($"%LOCALAPPDATA%/Nintendo/{Assembly.GetEntryAssembly().GetName().Name}");
        }

        public static DateTime RetrieveLinkerTimestamp(string filePath)
        {
            const int PeHeaderOffset = 60;
            const int LinkerTimestampOffset = 8;
            byte[] bytes = new byte[2048];
            System.IO.Stream stream = null;

            try
            {
                stream = new System.IO.FileStream(filePath, System.IO.FileMode.Open, System.IO.FileAccess.Read);
                stream.Read(bytes, 0, 2048);
            }
            finally
            {
                if (stream != null)
                {
                    stream.Close();
                }
            }

            int i = System.BitConverter.ToInt32(bytes, PeHeaderOffset);
            int secondsSince1970 = System.BitConverter.ToInt32(bytes, i + LinkerTimestampOffset);
            DateTime dt = new DateTime(1970, 1, 1, 0, 0, 0);
            dt = dt.AddSeconds(secondsSince1970);
            dt = dt.AddHours(TimeZone.CurrentTimeZone.GetUtcOffset(dt).Hours);
            return dt;
        }

        private static void WriteErrorLogFile(Exception exception)
        {
            string errorLogFolderPath = GetErrorLogFolderPath();
            try
            {
                System.IO.Directory.CreateDirectory(errorLogFolderPath);
            }
            catch
            {
                return;
            }

            DateTime now = DateTime.Now;
            string errorLogFileName = string.Format("Error-{0:D4}{1:D2}{2:D2}_{3:D2}{4:D2}_{5:D2}.log",
                now.Year, now.Month, now.Day, now.Hour, now.Minute, now.Second);
            string errorLogFilePath = System.IO.Path.Combine(errorLogFolderPath, errorLogFileName);

            StringBuilder errorDetailMessage = new StringBuilder();
            var versionInfo = FileVersionInfo.GetVersionInfo(Assembly.GetEntryAssembly().Location);
            var exeTimestamp = RetrieveLinkerTimestamp(Assembly.GetEntryAssembly().Location);
            errorDetailMessage.AppendLine(string.Format($"Date: {now}"));
            errorDetailMessage.AppendLine();
            errorDetailMessage.AppendLine(string.Format("Version: {0}({1})", versionInfo.ProductVersion, exeTimestamp));
            errorDetailMessage.AppendLine();
            errorDetailMessage.AppendLine(string.Format("Machine Name: {0}", Environment.MachineName));
            errorDetailMessage.AppendLine();
            errorDetailMessage.AppendLine("Error Message:");
            errorDetailMessage.AppendLine(CreateExceptionMessage(exception, true));

            System.IO.StreamWriter streamWriter = new System.IO.StreamWriter(errorLogFilePath);
            streamWriter.Write(errorDetailMessage.ToString());
            streamWriter.Close();

            Console.WriteLine(TlStrings.Get("ErrorDetailSaved", errorLogFilePath));
        }

        // 例外の表示とファイルへの書き出し
        private static void DumpException(Exception exp)
        {
            string consoleMessage = CreateExceptionMessage(exp, false);
            Console.Error.Write(consoleMessage);

            WriteErrorLogFile(exp);
        }

        private static string CreateExceptionMessage(Exception exp, bool withStackTrace)
        {
            StringBuilder exceptionText = new StringBuilder();
            ExceptionListException exceptionList = exp as ExceptionListException;
            if (exceptionList != null)
            {
                foreach (Exception exception in exceptionList.Exceptions)
                {
                    exceptionText.AppendLine(CreateExceptionMessage(exception, withStackTrace));
                }
            }
            else
            {
                var aggregateException = exp as AggregateException;
                if (aggregateException != null)
                {
                    foreach (Exception exception in aggregateException.InnerExceptions)
                    {
                        exceptionText.AppendLine(CreateExceptionMessage(exception, withStackTrace));
                    }
                }
                else
                {
                    while (exp != null)
                    {
                        exceptionText.AppendLine(exp.Message);
                        if (withStackTrace)
                        {
                            exceptionText.AppendLine(exp.StackTrace);
                        }
                        exp = exp.InnerException;
                    }
                }
            }

            return exceptionText.ToString();
        }

        // スキーマベースパスの取得
        public static string GetXsdBasePath()
        {
            return System.IO.Path.Combine(GetG3dToolRootPath(), "3dIntermediateFileXsd");
        }

        public static string GetG3dToolRootPath()
        {
            string sdkRootPath = FindSdkRootPath();
            return System.IO.Path.Combine(sdkRootPath, "Tools/Graphics/3dTools");
        }

        private static string FindSdkRootPath()
        {
            string sdkRootPath = Path.GetDirectoryName((Assembly.GetEntryAssembly()??Assembly.GetCallingAssembly()).Location);
            if (!System.IO.Path.IsPathRooted(sdkRootPath))
            {
                return null;
            }

            do
            {
                sdkRootPath = System.IO.Path.Combine(sdkRootPath, "../");
            } while (!System.IO.File.Exists(System.IO.Path.Combine(sdkRootPath, "SigloRootMark")) &&
                     !System.IO.File.Exists(System.IO.Path.Combine(sdkRootPath, "NintendoSdkRootMark")));

            return sdkRootPath;
        }

        // パスの収集
        public static void CollectPath(List<string> result,
            string path, string searchPattern, Predicate<string> match)
        {
            bool found = false;
            if (path.Any(x => x == '?' || x == '*'))
            {
                // ワイルドカード指定のときはエラーにはしない
                found = true;
                foreach (var item in Glob(path))
                {
                    if (File.Exists(item))
                    {
                        if (match(item)) { result.Add(item); }
                    }
                }
            }
            else
            {
                if (File.Exists(path))
                {
                    found = true;
                    // 重複除去をいれる？やるならフルパス変換して比較
                    // result に詰める値は変更したくない
                    if (match(path)) { result.Add(path); }
                }

                // ファイル名と同じディレクトリがあれば、両方処理する
                if (Directory.Exists(path))
                {
                    found = true;
                    if (searchPattern == null) { searchPattern = "*"; }
                    foreach (string filePath in Directory.EnumerateFiles(
                        path, searchPattern, SearchOption.AllDirectories))
                    {
                        if (match(filePath)) { result.Add(filePath); }
                    }
                }
            }

            if (!found) { TlStrings.Throw("SimpleFilterTool_Error_FileNotFound", path); }
        }

        /// <summary>
        /// アルファベットの組み合わせを探します。
        /// * を特定のアルファベットで置き換えるために、元文字列で使われていないアルファベットの組み合わせを探すために使用します。
        /// </summary>
        /// <returns>アルファベットの組み合わせを返します。</returns>
        private static IEnumerable<string> EnumerateStrings()
        {
            for (char c = 'a'; c <= 'z'; c++)
            {
                yield return c.ToString();
            }

            foreach (var s in EnumerateStrings())
            {
                for (char c = 'a'; c <= 'z'; c++)
                {
                    yield return s + c.ToString();
                }
            }
        }

        // ワイルドカード(* と ?)の展開
        public static IEnumerable<string> Glob(string pattern)
        {
            foreach (var s in EnumerateStrings())
            {
                // GetFullPath を使って正規化するために一度 * と ? を文字列に置き換える
                if (!pattern.Contains(s))
                {
                    // s は小文字からなり、A, B は大文字なので一意に元に戻せる。
                    var replaced = pattern.Replace("*", s + "A").Replace("?", s + "B");
                    var path = Path.GetFullPath(replaced).Replace(s + "A", "*").Replace(s + "B", "?");
                    return EnumerateEntries(SplitPath(path), ".");
                }
            }

            Nintendo.Foundation.Contracts.Assertion.Fail($"Can not replace * or ? with any alphabets. The pattern is \"{pattern}\"");
            return Enumerable.Empty<string>();
        }

        // basePath から entries をたどって得られるパス全体の取得
        private static IEnumerable<string> EnumerateEntries(IEnumerable<string> entries, string basePath)
        {
            if (entries.Any())
            {
                if (!Directory.Exists(basePath))
                {
                    return Enumerable.Empty<string>();
                }

                var first = entries.First();

                // Skip が繰り返されるので少し非効率
                var remains = entries.Skip(1);

                if (first.Any(x => x == '?' || x == '*'))
                {
                    string[] paths;
                    try
                    {
                        paths = Directory.EnumerateFileSystemEntries(basePath != string.Empty ? basePath : ".", first).ToArray();
                    }
                    catch
                    {
                        return Enumerable.Empty<string>();
                    }

                    return paths.SelectMany(x => EnumerateEntries(remains, x));
                }
                else
                {
                    return EnumerateEntries(remains, Path.Combine(basePath, first));
                }
            }
            else
            {
                return Enumerable.Repeat(basePath, 1);
            }
        }

        // ディレクトリに分解
        private static IEnumerable<string> SplitPath(string pattern)
        {
            if (!string.IsNullOrEmpty(pattern))
            {
                // 少し非効率だけど標準機能に頼る
                var directory = Path.GetDirectoryName(pattern);
                var file = Path.GetFileName(pattern);
                if (string.IsNullOrEmpty(directory) || string.IsNullOrEmpty(file))
                {
                    return Enumerable.Repeat(pattern, 1);
                }
                else
                {
                    return SplitPath(directory).Concat(Enumerable.Repeat(file, 1));
                }
            }
            else
            {
                return Enumerable.Empty<string>();
            }
        }

        // 拡張子リストの取得
        public static List<string> GetExtensions(string extensions)
        {
            List<string> result = new List<string>();
            foreach (string extension in extensions.Split(','))
            {
                string extTemp = extension;
                if (extTemp == string.Empty) { continue; }
                if (extTemp[0] != '.') { extTemp = '.' + extTemp; }
                result.Add(extTemp);
            }
            return result;
        }
    }
}
