﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <ShdrExtractor.h>
#include <util/UtilXMLParser.h>
#include <util/UtilBuffer.h>
#include <g3dif/ShaderDefinition.h>
#include <sstream>
#include <nn/nn_Macro.h>

namespace nn { namespace g3dTool {
namespace {

const std::string uniform_tbl_shader_names[] =
{
    "bool",
    "bvec2",
    "bvec3",
    "bvec4",
    "int",
    "ivec2",
    "ivec3",
    "ivec4",
    "uint",
    "uvec2",
    "uvec3",
    "uvec4",
    "float",
    "vec2",
    "vec3",
    "vec4",
    "float2x2",
    "float2x3",
    "float2x4",
    "float3x2",
    "float3x3",
    "float3x4",
    "float4x2",
    "float4x3",
    "float4x4",
    "srt2d",
    "srt3d",
    "texsrt",
    "texsrt_ex"
};

const std::string uniform_tbl_logical_names[] =
{
    "bool",
    "bool2",
    "bool3",
    "bool4",
    "int",
    "int2",
    "int3",
    "int4",
    "uint",
    "uint2",
    "uint3",
    "uint4",
    "float",
    "float2",
    "float3",
    "float4",
    "float2x2",
    "float2x3",
    "float2x4",
    "float3x2",
    "float3x3",
    "float3x4",
    "float4x2",
    "float4x3",
    "float4x4",
    "srt2d",
    "srt3d",
    "texsrt",
    "texsrt_ex"
};

const std::string sampler_tbl_names[] =
{
    "isampler1D",
    "isampler2D",
    "isampler2DRect",
    "isampler3D",
    "isamplerCube",
    "isamplerBuffer",
    "isampler2DMS",
    "isampler1DArray",
    "isampler2DArray",
    "isamplerCubeArray",
    "isampler2DMSArray",
    "usampler1D",
    "usampler2D",
    "usampler2DRect",
    "usampler3D",
    "usamplerCube",
    "usamplerBuffer",
    "usampler2DMS",
    "usampler1DArray",
    "usampler2DArray",
    "usamplerCubeArray",
    "usampler2DMSArray",
    "sampler1D",
    "sampler2D",
    "sampler2DRect",
    "sampler3D",
    "samplerCube",
    "samplerBuffer",
    "sampler2DMS",
    "sampler1DArray",
    "sampler2DArray",
    "samplerCubeArray",
    "sampler2DMSArray",
    "sampler1DShadow",
    "sampler2DShadow",
    "sampler2DRectShadow",
    "samplerCubeShadow",
    "sampler1DArrayShadow",
    "sampler2DArrayShadow",
    "samplerCubeArrayShadow"
};

const std::string attrib_tbl_shader_names[] =
{
    "int",
    "ivec2",
    "ivec3",
    "ivec4",
    "uint",
    "uvec2",
    "uvec3",
    "uvec4",
    "float",
    "vec2",
    "vec3",
    "vec4"
};

const std::string attrib_tbl_logical_names[] =
{
    "int",
    "int2",
    "int3",
    "int4",
    "uint",
    "uint2",
    "uint3",
    "uint4",
    "float",
    "float2",
    "float3",
    "float4"
};


const nw::g3d::tool::util::StringRef stage_names[] =
{
    "vertex_shader",
    "geometry_shader",
    "fragment_shader"
};


std::vector<std::string> textblock_auto_id_array;

// uniform の型を shader_param の type に変換します。
nw::g3d::tool::g3dif::elem_shader_param::enum_type GetUniformType(std::string id, std::string type)
{
    if (type.empty())
    {
        // 配列型の uniform 変数は type のアノテーションを省略することができません。
        THROW_TRANSLATED_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_ANNOTATION, "Identifier_NoAnnotationForArrayUniform", id.c_str());
    }

    // float4x4 などは自動的に付かない
    for (int i = 0; i < nw::g3d::tool::g3dif::elem_shader_param::num_type; ++i)
    {
        if (uniform_tbl_shader_names[i] == type)
        {
            return static_cast<nw::g3d::tool::g3dif::elem_shader_param::enum_type>(i);
        }
    }

    for (int i = 0; i < nw::g3d::tool::g3dif::elem_shader_param::num_type; ++i)
    {
        if (uniform_tbl_logical_names[i] == type)
        {
            return static_cast<nw::g3d::tool::g3dif::elem_shader_param::enum_type>(i);
        }
    }

    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_ANNOTATION, "Identifier_InvalidAnnotaionType", type.c_str(), "uniform", id.c_str());

    // なんらかの値を返す必要がある。
    return nw::g3d::tool::g3dif::elem_shader_param::type_bool;
}

// sampler の type に変換します。
nw::g3d::tool::g3dif::elem_sampler_var::enum_type GetSamplerType(std::string id, std::string type)
{
    for (int i = 0; i < nw::g3d::tool::g3dif::elem_sampler_var::num_type; ++i)
    {
        if (sampler_tbl_names[i] == type)
        {
            return static_cast<nw::g3d::tool::g3dif::elem_sampler_var::enum_type>(i);
        }
    }

    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_ANNOTATION, "Identifier_InvalidAnnotaionType", type.c_str(), "sampler", id.c_str());

    // なんらかの値を返す必要がある。
    return nw::g3d::tool::g3dif::elem_sampler_var::sampler_1d;
}

// attrib の type に変換します。
nw::g3d::tool::g3dif::elem_vtx_attrib::enum_type GetAttribType(std::string id, std::string type)
{
    for (int i = 0; i < nw::g3d::tool::g3dif::elem_vtx_attrib::num_type; ++i)
    {
        if (attrib_tbl_shader_names[i] == type)
        {
            return static_cast<nw::g3d::tool::g3dif::elem_vtx_attrib::enum_type>(i);
        }
    }

    for (int i = 0; i < nw::g3d::tool::g3dif::elem_vtx_attrib::num_type; ++i)
    {
        if (attrib_tbl_logical_names[i] == type)
        {
            return static_cast<nw::g3d::tool::g3dif::elem_vtx_attrib::enum_type>(i);
        }
    }

    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_ANNOTATION, "Identifier_InvalidAnnotaionType", type.c_str(), "attribute", id.c_str());

    // なんらかの値を返す必要がある。
    return nw::g3d::tool::g3dif::elem_vtx_attrib::type_int;
}

//--------------------------------------------------------------------------------------------------

const StringRef remoteIds[] =
{
    id_attrib_id,
    id_option_id,
    id_block_id,
    id_uniform_id,
    id_sampler_id,
    id_renderinfo,
    id_textblock,
    id_interleave,
    id_streamout,
    id_revision,
    id_group_id,
    id_layout,
    id_page
};
const int REMOTE_ID_COUNT = sizeof(remoteIds) / sizeof( StringRef );

const StringRef INDENT = "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"; // とりあえず。必要なら増やす。

struct Indent
{
    Indent(int indent) : indent(indent) {}

    int indent;
};

inline
std::ostream& operator<<(std::ostream& lhs, const Indent& rhs)
{
    assert(rhs.indent < sizeof(INDENT) - 1);
    lhs.write(INDENT.Data(), rhs.indent);
    return lhs;
}

template <typename T>
struct AttrManip
{
    AttrManip(int indent, const StringRef& name, const T& value)
        : indent(indent), name(name), value(value)
    {
    }

    int indent;
    const StringRef& name;
    const T& value;

private:
    AttrManip<T> &operator =(const AttrManip &);
};

template <typename T>
inline
std::ostream& operator<<(std::ostream& lhs, const AttrManip<T>& rhs)
{
    if (rhs.indent > 0)
    {
        lhs << std::endl;
        lhs << Indent(rhs.indent);
    }
    else
    {
        lhs << ' ';
    }
    lhs << rhs.name << '=' << '"' << rhs.value << '"';
    return lhs;
}

template <typename T>
inline
AttrManip<T> Attr(int indent, const StringRef& name, const T& value)
{
    return AttrManip<T>(indent, name, value);
}

inline
void Advance(
    Extractor::context_type::iterator_type& iter,
    Extractor::context_type::iterator_type& end)
{
    do
    {
        try
        {
            ++iter;
        }
        catch (boost::wave::preprocess_exception& e)
        {
            std::stringstream ss;
            ss << e.file_name() << "(" << e.line_no() << "): "
                << e.description();

            // マクロの多重定義の警告は無視する。
            if (e.get_errorcode() == boost::wave::preprocess_exception::macro_redefinition)
            {
                continue;
            }
            else if (e.get_errorcode() == boost::wave::preprocess_exception::missing_matching_if ||
                e.get_errorcode() == boost::wave::preprocess_exception::bad_include_file)
            {
                // if-else-endif の数が一致しない
                // include ファイルが開けない
                THROW_ERROR(ERRCODE_SHADER_CONVERTER_GLSL_SYNTAX_ERROR, "GLSL syntax error.\n%hs", ss.str().c_str());
            }

            PRINT_WARNING(ERRCODE_SHADER_CONVERTER_GLSL_SYNTAX_ERROR, "GLSL syntax warning\n%hs",ss.str().c_str());

            if (!e.is_recoverable())
            {
                throw;
            }
            continue; // 復帰可能な例外の場合は再試行。
        }
        break; // 例外が発生しなければ続行。
    } while (iter != end);
}

inline
size_t GetAnnotHeaderLen(const char* str)
{
    const int skip = 2; // コメント開始記号の分をスキップ。
    const char* iter = str;
    for (int i = 0; i < skip && *iter; ++i, ++iter)
    {
    }
    for (; isspace(*iter); ++iter)
    {
    }
    if (strncmp(iter, "@@", 2)) // '@' が2つ連続した時点でアノテーションとみなす。"@@@" は "@@" + "@"
    {
        return 0;
    }
    // アノテーションのヘッダー部のサイズを返す "// @@ " の場合は 5
    return iter + skip - str;
}

//--------------------------------------------------------------------------------------------------

template <typename Annot>
inline
void Resolve(std::vector<Annot>& annotVec, const nw::g3d::tool::util::XMLElement* pElem)
{
    for (;pElem; pElem = pElem->NextSibling())
    {
        const char* name = pElem->Attribute(id_name);
        if (name == nullptr)
        {
            // シンボル名の取得に失敗
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }

        for (auto iter = annotVec.cbegin(), end = annotVec.cend(); iter != end; ++iter)
        {
            if (name == iter->GetAttributeValue(id_name))
            {
                // シンボル名の二重定義
                THROW_ERROR_INTERNAL(ERRCODE_SHADER_CONVERTER_REDIFINED_SYMBOL,
                    "%hs, line %hs: %hs is redifined.",
                    pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), name);
            }
        }

        // 無効な属性が配下に無いかチェック
        if (const char* pAnnotationName = Annot::GetInvalidXmlAnnotationName(pElem))
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION, "Identifier_InvalidAnnotationAttribute",
                pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), pAnnotationName, name);
        }

        // id が存在しない属性は追加しない
        const char* id = pElem->Attribute(id_id);
        int index = pElem->AttributeIndex(id_id);
        if (id == nullptr)
        {
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_ID_NOT_FOUND,
                "%hs, line %hs: Not found id in an annotation of the symbol(%hs).",
                pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), name);
        }
        else if (index != 1)
        {
            // id は必ず name の次の属性でなければならない (アノテーションの先頭で宣言される必要がある)。
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), "id", id);
        }
        else
        {
            annotVec.emplace_back(pElem, name);
        }
    }
}

template <typename TBlock>
inline
void ResolveBlock(std::vector<TBlock>& annotVec, const nw::g3d::tool::util::XMLElement* pElem)
{
    for (;pElem; pElem = pElem->NextSibling())
    {
        const char* name = pElem->Attribute(id_name);
        if (name == nullptr)
        {
            // シンボル名の取得に失敗
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }

        for (auto iter = annotVec.cbegin(), end = annotVec.cend(); iter != end; ++iter)
        {
            if (name == iter->GetAttributeValue(id_name))
            {
                // シンボル名の二重定義
                THROW_ERROR_INTERNAL(ERRCODE_SHADER_CONVERTER_REDIFINED_SYMBOL,
                    "%hs, line %hs: %hs is redifined.",
                    pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), name);
            }
        }

        // 無効な属性が配下に無いかチェック
        if (const char* pAnnotationName = TBlock::GetInvalidXmlAnnotationName(pElem))
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION, "Identifier_InvalidAnnotationAttribute",
                pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), pAnnotationName, name);
        }

        // id が存在しない属性は追加しない
        const StringRef* pBlockInfoName = &id_block_info;
        const StringRef* pBlockTypeName = &id_uniform;
        const nw::g3d::tool::util::XMLElement* pBlockInfo = pElem->Child( *pBlockInfoName );
        if (pBlockInfo == nullptr)
        {
            pBlockInfoName = &id_ssbo_block_info;
            pBlockTypeName = &id_buffer;
            pBlockInfo = pElem->Child( *pBlockInfoName );
        }
        if (pBlockInfo == nullptr)
        {
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_ID_NOT_FOUND, "Not found id in annotation: %hs.", name);
        }
        const char* id = pBlockInfo->Attribute(id_id);
        if (id == nullptr)
        {
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_ID_NOT_FOUND,
                "%hs, line %hs: Not found id in an annotation of the symbol(%hs).",
                pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), name);
        }
        else if (pBlockInfo->AttributeIndex(id_id) != 0)
        {
            // id は必ず最初の属性でなければならない。
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                pBlockInfo->Attribute(id_file_info), pBlockInfo->Attribute(id_line_info), "id", id);
        }
        else
        {
            annotVec.emplace_back(pElem, name, *pBlockInfoName);

            // ユニフォームブロック内の変数を追加する
            Resolve(annotVec.back().uniforms, pElem->Child(*pBlockTypeName));
        }
    }
}

template <typename Annot>
inline
auto SearchById(std::vector<Annot>& annotVec, const StringRef& id)
    -> decltype(std::find_if(annotVec.begin(), annotVec.end(), nullptr))
{
    return std::find_if(annotVec.begin(), annotVec.end(), [&](Annot& annot)
    {
        return annot.GetAttributeValue(id_id) == id;
    });
}

template <typename AnnotDB, typename Annot>
inline
bool ExistsAnyStage(AnnotDB* annotDBs, std::vector<Annot> AnnotDB::*annotVec, const StringRef& id)
{
    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
    {
        if (SearchById(annotDBs[stage].*annotVec, id) != ( annotDBs[stage].*annotVec ).end())
        {
            return true;
        }
    }
    return false;
}

template <typename AnnotDB, typename AnnotParent, typename Annot>
inline
bool ExistsAnyStage(AnnotDB* annotDBs, std::vector<AnnotParent> AnnotDB::*annotParentVec,
    std::vector<Annot> AnnotParent::*annotVec, const StringRef& id)
{
    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
    {
        for (auto parent = ( annotDBs[stage].*annotParentVec ).begin();
            parent != ( annotDBs[stage].*annotParentVec ).end(); ++parent)
        {
            if (SearchById((*parent).*annotVec, id) != ( (*parent).*annotVec ).end())
            {
                return true;
            }
        }
    }
    return false;
}

template <typename Annot>
inline
bool ResolveById(std::vector<Annot>& annotVec,
    const nw::g3d::tool::util::XMLElement* pElem, const StringRef& id)
{
    auto iter = SearchById(annotVec, id);
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
        return true;
    }

    return false;
}

// AnnotRenderInfo, AnnotTextBlock, AnnotInterleave, AnnotGroup の場合の特殊化
template <>
inline
bool ResolveById<AnnotRenderInfo>(std::vector<AnnotRenderInfo>& annotVec,
const nw::g3d::tool::util::XMLElement* pElem, const StringRef& renderinfo)
{
    auto iter = std::find_if(annotVec.begin(), annotVec.end(), [&](AnnotRenderInfo& annot)
    {
        return annot.GetAttributeValue(id_renderinfo) == renderinfo;
    });
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
    }
    else
    {
        annotVec.emplace_back(pElem);
    }
    return true;
}

template <>
inline
bool ResolveById<AnnotTextBlock>(std::vector<AnnotTextBlock>& annotVec,
    const nw::g3d::tool::util::XMLElement* pElem, const StringRef& textblock)
{
    static int systemId = 0;
    auto iter = std::find_if(annotVec.begin(), annotVec.end(), [&](AnnotTextBlock& annot)
    {
        return annot.GetAttributeValue(id_textblock) == textblock;
    });
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
    }
    else
    {
        annotVec.emplace_back(pElem);
    }
    return true;
}

template <>
inline
bool ResolveById<AnnotInterleave>(std::vector<AnnotInterleave>& annotVec,
const nw::g3d::tool::util::XMLElement* pElem, const StringRef& interleave)
{
    auto iter = std::find_if(annotVec.begin(), annotVec.end(), [&](AnnotInterleave& annot)
    {
        return annot.GetAttributeValue(id_interleave) == interleave;
    });
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
    }
    else
    {
        annotVec.emplace_back(pElem);
    }
    return true;
}

template <>
inline
bool ResolveById<AnnotGroup>(std::vector<AnnotGroup>& annotVec,
const nw::g3d::tool::util::XMLElement* pElem, const StringRef& group)
{
    auto iter = std::find_if(annotVec.begin(), annotVec.end(), [&](AnnotGroup& annot)
    {
        return annot.GetAttributeValue(id_group_id) == group;
    });
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
    }
    else
    {
        annotVec.emplace_back(pElem);
    }
    return true;
}

template <>
inline
bool ResolveById<AnnotStreamout>(std::vector<AnnotStreamout>& annotVec,
const nw::g3d::tool::util::XMLElement* pElem, const StringRef& streamout)
{
    auto iter = std::find_if(annotVec.begin(), annotVec.end(), [&](AnnotStreamout& annot)
    {
        return annot.GetAttributeValue(id_streamout) == streamout;
    });
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
    }
    else
    {
        annotVec.emplace_back(pElem);
    }
    return true;
}

template <>
inline
bool ResolveById<AnnotRevision>(std::vector<AnnotRevision>& annotVec,
const nw::g3d::tool::util::XMLElement* pElem, const StringRef& revision)
{
    auto iter = std::find_if(annotVec.begin(), annotVec.end(), [&](AnnotRevision& annot)
    {
        return annot.GetAttributeValue(id_revision) == revision;
    });
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
    }
    else
    {
        annotVec.emplace_back(pElem);
    }
    return true;
}

template <>
inline
bool ResolveById<AnnotPage>(std::vector<AnnotPage>& annotVec,
const nw::g3d::tool::util::XMLElement* pElem, const StringRef& page)
{
    auto iter = std::find_if(annotVec.begin(), annotVec.end(), [&](AnnotPage& annot)
    {
        return annot.GetAttributeValue(id_page) == page;
    });
    if (iter != annotVec.end())
    {
        iter->MergeAttr(pElem);
    }
    else
    {
        annotVec.emplace_back(pElem);
    }
    return true;
}

template <typename Symbol, typename Annot>
inline
void MergeToSymbol(ShaderStage stage,
    std::vector<Symbol>& dst, const std::vector<Annot>& src,
    bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const Annot& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](Symbol& symbol)
        {
            return symbol.GetSymbolValue(id_id) == annot.GetAttributeValue(id_id);
        });

        if (found == dst.end())
        {
            // 登録されていない場合
            dst.emplace_back(stage, annot);
        }
        else if (!found->strName[stage].empty())
        {
            // 同じ変数へのアノテーションが重複して定義されていた場合
            THROW_ERROR(ERRCODE_SHADER_ANNOTATION_ID_REDEFINED,
                "%hs, line %hs: Annotation(%hs) was redefined.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_id).Str().c_str());
        }
        else if (found->CompareAttr(annot))
        {
            // アノテーション設定は同じだが、ステージ間で別名で定義されているシンボルがある。
            found->strName[stage].assign(annot.GetAttributeValue(id_name).Str());
        }
        else if (overrideAnnotation)
        {
            // vs -> gs -> fs -> cs の順でアノテーションを上書きする。
            found->Override(annot);
            found->strName[stage].assign(annot.GetAttributeValue(id_name).Str());
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_id).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_id).Str().c_str());
        }
    });
}

// SysmbolRenderInfo, SymbolInterleave, SymbolStreamout, SymbolRevision, SymbolGroup の場合の特殊化
template <>
inline
void MergeToSymbol<SymbolRenderInfo, AnnotRenderInfo>(ShaderStage /*stage*/,
std::vector<SymbolRenderInfo>& dst, const std::vector<AnnotRenderInfo>& src,
bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const AnnotRenderInfo& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](SymbolRenderInfo& symbol)
        {
            return symbol.GetSymbolValue(id_renderinfo) == annot.GetAttributeValue(id_renderinfo);
        });
        if (found == dst.end())
        {
            dst.emplace_back(annot);
        }
        else if (found->CompareAttr(annot))
        {
            // do nothing.
        }
        else if (overrideAnnotation)
        {
            found->Override(annot);
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_renderinfo).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_renderinfo).Str().c_str());
        }
    });
}

template <>
inline
void MergeToSymbol<SymbolTextBlock, AnnotTextBlock>(ShaderStage /*stage*/,
    std::vector<SymbolTextBlock>& dst, const std::vector<AnnotTextBlock>& src,
    bool overrideAnnotation)
{
    // id 以外の要素が 1 つでも異なっている場合、dst に追加する。
    // id が異なっている場合でも、id 以外の要素が一致していれば追加しない。
    for (int srcIndex = 0; srcIndex < src.size(); srcIndex++)
    {
        const AnnotTextBlock& annot = src[srcIndex];
        bool match = false;
        for (int dstIndex = 0; dstIndex < dst.size(); dstIndex++)
        {
            const SymbolTextBlock& symbol = dst[dstIndex];
            if (symbol.GetSymbolValue(id_text) == annot.GetAttributeValue(id_text) && symbol.GetSymbolValue(id_group) == annot.GetAttributeValue(id_group) && symbol.GetSymbolValue(id_order) == annot.GetAttributeValue(id_order))
            {
                match = true;
                break;
            }
        }
        if (!match)
        {
            dst.emplace_back(annot);
        }
    }
    NN_UNUSED(overrideAnnotation);
}

template <>
inline
void MergeToSymbol<SymbolInterleave, AnnotInterleave>(ShaderStage /*stage*/,
std::vector<SymbolInterleave>& dst, const std::vector<AnnotInterleave>& src,
bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const AnnotInterleave& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](SymbolInterleave& symbol)
        {
            return symbol.GetSymbolValue(id_interleave) == annot.GetAttributeValue(id_interleave);
        });
        if (found == dst.end())
        {
            dst.emplace_back(annot);
        }
        else if (found->CompareAttr(annot))
        {
            // do nothing.
        }
        else if(overrideAnnotation)
        {
            found->Override(annot);
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_interleave).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_interleave).Str().c_str());
        }
    });
}

template <>
inline
void MergeToSymbol<SymbolStreamout, AnnotStreamout>(ShaderStage /*stage*/,
std::vector<SymbolStreamout>& dst, const std::vector<AnnotStreamout>& src,
bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const AnnotStreamout& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](SymbolStreamout& symbol)
        {
            return symbol.GetSymbolValue(id_streamout) == annot.GetAttributeValue(id_streamout);
        });
        if (found == dst.end())
        {
            dst.emplace_back(annot);
        }
        else if (found->CompareAttr(annot))
        {
            // do nothing.
        }
        else if (overrideAnnotation)
        {
            found->Override(annot);
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_streamout).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_streamout).Str().c_str());
        }
    });
}

template <>
inline
void MergeToSymbol<SymbolRevision, AnnotRevision>(ShaderStage /*stage*/,
std::vector<SymbolRevision>& dst, const std::vector<AnnotRevision>& src,
bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const AnnotRevision& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](SymbolRevision& symbol)
        {
            return symbol.GetSymbolValue(id_revision) == annot.GetAttributeValue(id_revision);
        });
        if (found == dst.end())
        {
            dst.emplace_back(annot);
        }
        else if (found->CompareAttr(annot))
        {
            // do nothing.
        }
        else if (overrideAnnotation)
        {
            found->Override(annot);
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_revision).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_revision).Str().c_str());
        }
    });
}

template <>
inline
void MergeToSymbol<SymbolLayout, AnnotLayout>(ShaderStage /*stage*/,
std::vector<SymbolLayout>& dst, const std::vector<AnnotLayout>& src,
bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const AnnotLayout& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](SymbolLayout& symbol)
        {
            return symbol.CompareAttr(annot);
        });
        if (found == dst.end())
        {
            dst.emplace_back(annot);
        }
        else if(overrideAnnotation)
        {
            found->Override(annot);
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_qualifier).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_qualifier).Str().c_str());
        }
    });
}

template <>
inline
void MergeToSymbol<SymbolGroup, AnnotGroup>(ShaderStage /*stage*/,
std::vector<SymbolGroup>& dst, const std::vector<AnnotGroup>& src,
bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const AnnotGroup& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](SymbolGroup& symbol)
        {
            return symbol.GetSymbolValue(id_group_id) == annot.GetAttributeValue(id_group_id);
        });
        if (found == dst.end())
        {
            dst.emplace_back(annot);
        }
        else if (found->CompareAttr(annot))
        {
            // do nothing.
        }
        else if (overrideAnnotation)
        {
            found->Override(annot);
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_group_id).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_group_id).Str().c_str());
        }
    });
}

template<>
inline
void MergeToSymbol<SymbolPage, AnnotPage>(ShaderStage /*stage*/,
std::vector<SymbolPage>& dst, const std::vector<AnnotPage>& src,
bool overrideAnnotation)
{
    std::for_each(src.begin(), src.end(), [&](const AnnotPage& annot)
    {
        auto found = std::find_if(dst.begin(), dst.end(), [&](SymbolPage& symbol)
        {
            return symbol.GetSymbolValue(id_page) == annot.GetAttributeValue(id_page);
        });
        if (found == dst.end())
        {
            dst.emplace_back(annot);
        }
        else if (found->CompareAttr(annot))
        {
            // do nothing
        }
        else if (overrideAnnotation)
        {
            found->Override(annot);
            PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                "Annotation(&hs) is overridden by other stages.", annot.GetAttributeValue(id_page).Str().c_str());
        }
        else
        {
            THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_page).Str().c_str());
        }
    });
}

template <typename TSymbol>
inline void CheckGroupReference(const std::vector<SymbolGroup>& groups,
                                const std::vector<TSymbol>& symbols)
{
    for (auto symbol = symbols.cbegin(); symbol != symbols.cend(); ++symbol)
    {
        if (!symbol->GetSymbolValue(id_group).empty())
        {
            if (std::find_if(groups.cbegin(), groups.cend(), [&](const SymbolGroup& group)
            {
                return symbol->GetSymbolValue(id_group) == group.GetSymbolValue(id_group_id);
            }) == groups.cend())
            {
                PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_ID_NOT_FOUND,
                    "Group(%hs) is not exist.", symbol->GetSymbolValue(id_group).c_str());
            }
        }
    };
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

using namespace boost::wave;

struct Extractor::State
{
    State()
        : fileDB(nullptr)
        , includePaths()
        , forceIncludes()
        , glslVersion()
        , annot()
        , stringPool()
        , annotDB()
        , codePage(UINT_MAX)
        , pCtx(nullptr)
        , lastMacro()
    {
    }

    ~State()
    {
        ClearSettings();
        ClearResults();
    }

    void ClearSettings();
    void ClearResults();

    // 例外の場合でも 必ず Extract 終了前に呼び出される必要がある。
    void ClearTempInfo();

    // Clear で初期化される。
    FileDB* fileDB;
    std::unordered_map<std::string, std::string> fixed_macro_map;
    std::vector<std::string> includePaths;
    std::vector<std::string> forceIncludes;
    std::string glslVersion;

    std::stringstream annot;
    std::unique_ptr<char[]> stringPool;   // ソースから抽出したコメントを XML として展開するバッファ。
                                        // 解析後はそのまま文字列プールとして利用される。
    AnnotDB annotDB;
    SymbolDB symbolDB; // 外から与えた方がいいかもしれない

    uint32_t codePage;

    // Extract 毎に初期化される。
    context_type* pCtx;
    context_type::iterator_type srcBegin; // var の探索用に残しておく。
    token_type lastMacro;
    token_type blockName;
    token_type blockVar;
    token_type ssboBlockName;
    token_type ssboBlockVar;
    std::vector<token_type> tokens;

    // エラー情報用に現在位置を保持する。
    boost::wave::util::file_position_type current_position;
};

void Extractor::State::ClearSettings()
{
    this->fileDB = nullptr;
    this->includePaths.clear();
    this->forceIncludes.clear();
    this->codePage = UINT_MAX;

    this->fixed_macro_map.clear();
}

void Extractor::State::ClearResults()
{
    this->annot.str("");
    this->stringPool.reset();
    this->annotDB.Clear();
    this->symbolDB.Clear();
}

void Extractor::State::ClearTempInfo()
{
    this->pCtx = nullptr;
    this->srcBegin = context_type::iterator_type();
    this->lastMacro = this->blockName= this->blockVar = token_type();
    this->tokens.clear();
}

void Extractor::AnnotDB::Clear()
{
    options.clear();
    attribs.clear();
    samplers.clear();
    for (auto block = blocks.begin(); block != blocks.end(); ++block)
    {
        block->uniforms.clear();
    }
    blocks.clear();
    for (auto block = ssboBlocks.begin(); block != ssboBlocks.end(); ++block)
    {
        block->uniforms.clear();
    }
    ssboBlocks.clear();
    renderInfos.clear();
    textBlocks.clear();
    interleaves.clear();
    groups.clear();
    streamouts.clear();
    revisions.clear();
    layouts.clear();
    pages.clear();
}

void Extractor::SymbolDB::Clear()
{
    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
    {
        path[stage].clear();
    }
    options.clear();
    attribs.clear();
    samplers.clear();
    for (auto block = blocks.begin(); block != blocks.end(); ++block)
    {
        block->uniforms.clear();
    }
    blocks.clear();
    for (auto ssboBlock = ssboBlocks.begin(); ssboBlock != ssboBlocks.end(); ++ssboBlock)
    {
        ssboBlock->uniforms.clear();
    }
    ssboBlocks.clear();
    renderInfos.clear();
    textBlocks.clear();
    interleaves.clear();
    groups.clear();
    streamouts.clear();
    revisions.clear();
    layouts.clear();
    pages.clear();
}

//--------------------------------------------------------------------------------------------------

Extractor::Extractor() : m_pState(new State())
{
}

void Extractor::ClearSettings()
{
    m_pState->ClearSettings();
}

void Extractor::ClearResults()
{
    m_pState->ClearResults();
}

void Extractor::ClearMacro()
{
    m_pState->fixed_macro_map.clear();
}

bool Extractor::AddFixedMacro(const StringRef& macro)
{
    // MACRO=value を分解する。
    auto posEq = macro.Str().find_first_of('=');

    if (posEq != std::string::npos)
    {
        return AddFixedMacro(macro.Str().substr(0, posEq), macro.Str().data() + posEq + 1);
    }
    else
    {
        return AddFixedMacro(macro.Str(), "1");
    }
}

bool Extractor::AddFixedMacro(const StringRef& name, const StringRef& value)
{
    auto comp = m_pState->fixed_macro_map.find(name.Str());
    if (comp == m_pState->fixed_macro_map.cend())
    {
        m_pState->fixed_macro_map.insert(std::make_pair(nw::g3d::tool::util::EraseNull(name.Str()), value.Str()));
        return true;
    }
    else
    {
        return false;
    }
}

void Extractor::AddIncludePath(const StringRef& path)
{
    m_pState->includePaths.push_back(path.Str());
}

void Extractor::AddForceInclude(const StringRef& path)
{
    m_pState->forceIncludes.push_back(path.Str());
}

void Extractor::AddGlslVersion(const StringRef& path)
{
    m_pState->glslVersion = path.Str();
}

void Extractor::SetCodePage(uint32_t codePage)
{
    m_pState->codePage = codePage;
}

uint32_t Extractor::GetCodePage() const
{
    return m_pState->codePage;
}

FileDB* Extractor::GetFileDB()
{
    return m_pState->fileDB;
}

StringRef Extractor::LoadFile(const StringRef& fullpath, FileDB* fileDB, uint32_t codePage)
{
    std::string path(fullpath.Data(), fullpath.Len());
    std::replace(path.begin(), path.end(), '/', '\\');
    auto src = fileDB->find(path);
    if (src == fileDB->end())
    {
        // read in the file
        std::ifstream instream(nw::g3d::tool::util::Path::ToWideChar(path.c_str()).c_str(), std::ios::in | std::ios::binary);
        if (!instream.is_open()) {
            THROW_ERROR(ERRCODE_OPEN_FILE, "Cannot open source file(%hs).", path.c_str());
        }
        instream.unsetf(std::ios::skipws);

        static const u8 BOM[] = { 0xEF, 0xBB, 0xBF };
        char buf[3];
        instream.read(buf, sizeof(buf));
        instream.seekg(0, std::ios::beg);

        // UTF-8 の BOM が付いていればその分をとばす。
        std::string file;
        bool hasBOM = false;
        if (0 == memcmp(BOM, buf, sizeof(BOM)))
        {
            auto iter = std::istreambuf_iterator<char>(instream.rdbuf());
            file.assign(++++++iter, std::istreambuf_iterator<char>()); // BOM をスキップ。
            hasBOM = true;
        }
        else
        {
            file.assign(std::istreambuf_iterator<char>(instream.rdbuf()),
                std::istreambuf_iterator<char>());
        }

        // BOM がある場合はUTF8以外のコードページが指定されると例外
        if (hasBOM && (codePage != UINT_MAX && codePage != CP_UTF8))
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_SOURCEFILE_CODE_PAGE, "Discrepant character code between source file and a code_page option: %hs.", fullpath.Str().c_str());
        }

        // BOM がない場合はコードページが指定されてないと例外
        if (!hasBOM && codePage == UINT_MAX)
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_SOURCEFILE_CODE_PAGE, "Please set a code_page option in case of using character code other than utf8.");
        }

        if (codePage != UINT_MAX && codePage != CP_UTF8)
        {
            // ワイド文字に変換
            int wcsLen = MultiByteToWideChar(codePage, 0, file.data(), static_cast<int>(file.length()), nullptr, 0);
            std::unique_ptr<wchar_t> wcs(new wchar_t[wcsLen]);
            wcsLen = MultiByteToWideChar(codePage, 0,
                file.data(), static_cast<int>(file.length()), wcs.get(), wcsLen);

            // utf-8 に変換
            int mbsLen = WideCharToMultiByte(CP_UTF8, 0,
                wcs.get(), wcsLen, nullptr, 0, nullptr, nullptr);
            file.resize(mbsLen);
            mbsLen = WideCharToMultiByte(CP_UTF8, 0,
                wcs.get(), wcsLen, &file[0], mbsLen, nullptr, nullptr);
        }

        const char* c_str = file.c_str();
        size_t len = strlen(c_str);
        // file の末尾が \r\n でない場合に処理を行う
        const char* eof = &file.data()[len - 2];
        if (*eof != 0x0d || *(eof + 1) != 0x0a)
        {
            file = c_str + nw::g3d::tool::util::CRLF;
        }

        // CRLF を LF に変換する。
        file = nw::g3d::tool::util::ReplaceAll(file, nw::g3d::tool::util::CRLF, nw::g3d::tool::util::LF);

        auto result = fileDB->insert(std::make_pair(path,
            std::make_pair(static_cast<int>(fileDB->size()), std::string())));
        if (!result.second)
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }
        src = result.first;
        src->second.second.swap(file);
    }

    return src->second.second;
}

StringRef Extractor::ModifySource(StringRef src, std::string path)
{
    State& state = *m_pState;
    FileDB* fileDB = state.fileDB;

    // forceInclude がある場合はソースファイルの先頭に include を付加する
    // forceInclude が解決されたソースは fileDB のファイルパス名に .nw_modify がつく。
    if (state.forceIncludes.size() != 0)
    {
        std::stringstream ss;
        ss << path << NW_MODIFY_FILE;
        auto found = fileDB->find(ss.str());

        if (found != fileDB->end())
        {
            // 既に modify が登録されている
            return found->second.second;
        }

        auto result = fileDB->insert(std::make_pair(ss.str(),
            std::make_pair(static_cast<int>(fileDB->size()), std::string())));
        auto& modifySource = result.first->second.second;
        modifySource.assign(src.Data(), src.Len());

        for (auto include = state.forceIncludes.cbegin(); include != state.forceIncludes.cend(); ++include)
        {
            std::stringstream stringStream;
            stringStream << "#include <" << *include << ">" << std::endl;
            modifySource.insert(0, stringStream.str());
        }

        if (!result.second)
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }

        return modifySource;
    }
    else
    {
        return src;
    }
}

void Extractor::ExtractInternal(context_type* pCtx, ShaderStage stage)
{
    State& state = *m_pState;

    // boost::wave の動的設定
    pCtx->set_language(static_cast<language_support>(
        support_cpp
        //| support_option_long_long
        | support_option_variadics
        | support_option_insert_whitespace
        | support_option_preserve_comments
        | support_option_no_character_validation
        //|support_option_convert_trigraphs
        //|support_option_single_line
        | support_option_prefer_pp_numbers
        //| support_option_emit_line_directives
        //| support_option_include_guard_detection
        | support_option_emit_pragma_directives));

    // マクロを一旦削除する。
    pCtx->reset_macro_definitions();

    // fsc 及び コンバートオプションで指定したマクロを追加
    std::for_each(state.fixed_macro_map.cbegin(), state.fixed_macro_map.cend(), [&](const std::pair<std::string, std::string>& macro) {
        // 外から与えるマクロはシステム定義マクロとして上書きさせない。
        // MACRO=value の形に変換する。
        std::stringstream ss;
        ss << macro.first << '=' << macro.second;
        pCtx->add_macro_definition(ss.str(), true);
    });

    // 自身のステージに対応したマクロを追加
    static const char* tblStageMacro[] =
    {
        "NN_G3D_VERTEX_SHADER=1",
        "NN_G3D_GEOMETRY_SHADER=1",
        "NN_G3D_FRAGMENT_SHADER=1",
        "NN_G3D_COMPUTE_SHADER=1"
    };
    static_assert(ARRAYSIZE(tblStageMacro) == ShaderStage_StageCount, "Invalid table size");
    pCtx->add_macro_definition(tblStageMacro[stage], true);
}

void Extractor::Extract(const StringRef& fullpath, ShaderStage stage, FileDB* fileDB)
{
    // フックから参照するためにステートとして保持する。
    State& state = *m_pState;

    // 抽出用の各種情報を初期化する。
    state.fileDB = fileDB;

    if (stage >= ShaderStage_StageCount)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error. Invalid enum ShaderStage.");
    }

    state.annot << '<' << id_shader
        << Attr(0, id_stage, stage)
        << Attr(0, id_path, fullpath)
        << " >" << std::endl;

    std::string path(fullpath.Data(), fullpath.Len());
    // シェーダーソースを読み込む
    StringRef src = LoadFile(fullpath, state.fileDB, state.codePage);
    if (src.Empty())
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }

    std::string errLog = "";
    try {
        // forceInclude の #include を src に追加
        StringRef modifySource = ModifySource(src, path);

        //  The preprocessor iterator shouldn't be constructed directly. It is
        //  to be generated through a wave::context<> object. This wave:context<>
        //  object is to be used additionally to initialize and define different
        //  parameters of the actual preprocessing (not done here).
        //
        //  The preprocessing of the input stream is done on the fly behind the
        //  scenes during iteration over the context_type::iterator_type stream.
        context_type ctx(modifySource.Begin(), modifySource.End(), path.c_str(), *this);

        // ctx の初期設定とマクロの追加
        ExtractInternal(&ctx, stage);

        state.pCtx = &ctx;
        state.srcBegin = ctx.begin();
        auto context_end = ctx.end();
        // state.tokens.reserve();
        for (auto iter = state.srcBegin, end = context_end; iter != end; Advance(iter, end))
        {
            state.tokens.emplace_back(*iter);
        }
        int braceDepth = 0;
        // id を省略した textblock の自動 id
        static int textblockAutoId = 0;
        for (auto iter = state.tokens.cbegin(), end = state.tokens.cend(); iter != end; ++iter)
        {
            state.current_position = iter->get_position();
            if (T_POUND == *iter)
            {
                // プリプロセッサディレクティブをスキップ
                for (++iter; T_NEWLINE != *iter && iter != end; ++iter)
                {
                    // 何もしない
                }
            }
            else if (T_POUND_POUND == *iter)
            {
                THROW_ERROR(ERRCODE_SHADER_CONVERTER_GLSL_SYNTAX_ERROR, "Syntax error.");
            }
            else if (T_CPPCOMMENT == *iter)
            {
                // C++ スタイルコメント "// "
                const char* comment = iter->get_value().c_str();
                size_t len = GetAnnotHeaderLen(comment);
                if (len == 0)
                {
                    continue; // ただのコメント。
                }
                comment += len;
                if (iter->get_position().get_column() == 1)
                {
                    std::string line(comment);
                    // id が無い textblock に自動 id を付加する
                    InsertTextblockAutoId(&line, &textblockAutoId);
                    // 他の行のマクロまたは変数に対するアノテーション
                    ExtractAny(line.c_str());
                }
                else if (state.blockName.is_valid() &&
                    state.blockName.get_position().get_line() == iter->get_position().get_line() &&
                    state.blockName.get_position().get_file() == iter->get_position().get_file())
                {
                    // Uniform Block に対するアノテーション
                    ExtractBlock(comment);
                }
                else if (state.ssboBlockName.is_valid() &&
                    state.ssboBlockName.get_position().get_line() == iter->get_position().get_line() &&
                    state.ssboBlockName.get_position().get_file() == iter->get_position().get_file())
                {
                    // SsboBlock に対するアノテーションの解析
                    ExtractSsboBlock(comment);
                }
                else
                {
                    // 変数に対するアノテーション
                    ExtractVar(static_cast<int>(std::distance(state.tokens.cbegin(), iter)), comment);
                }
            }
            else if (T_CCOMMENT == *iter)
            {
                // C スタイルコメント "/* */"
                const char* comment = iter->get_value().c_str();
                size_t len = GetAnnotHeaderLen(comment);
                if (len == 0)
                {
                    continue; // ただのコメント。
                }
                auto head = comment + len;
                auto cend = comment + iter->get_value().length() - strlen("*/");
                std::string line;
                std::string group_name;
                for (auto citer = head; citer != cend; ++citer)
                {
                    // スペースをスキップする。
                    if (isspace(*head))
                    {
                        head = citer;
                        continue;
                    }
                    // グループの帯域定義の抽出
                    if (*citer == ':' && *(citer+1) == '\n')
                    {
                        auto tail = citer + 1;
                        line.assign(head, tail);
                        head = tail + 1;
                        if (group_name.empty())
                        {
                            std::stringstream ss;
                            ss << id_group.Str() << "=\"" << line.substr(0, line.size() - 1) << "\"";
                            group_name = ss.str();
                        }
                        continue;
                    }

                    ptrdiff_t diff = citer - head;
                    const int MIN_ANNOTATION_ID_SIZE = static_cast<const int >(id_uniform_id.Len());
                    if (diff < MIN_ANNOTATION_ID_SIZE)
                    {
                        // 文字数が足りない場合はスキップ
                        continue;
                    }

                    // */ が出現したらアノテーションの抽出を行う。
                    if ((citer + 1) == cend)
                    {
                        const int CLOSED_PARENTHESIS = 2;
                        const char* tailStr = citer + 1;
                        if (strncmp(tailStr, "*/", CLOSED_PARENTHESIS) == 0)
                        {
                            auto tail = citer;
                            line.assign(head, tail);
                            head = tail + 1;
                            if (!line.empty())
                            {
                                if (!group_name.empty() &&
                                    line.find("group=") == std::string::npos)
                                {
                                    // グループの定義が記述されていない場合は
                                    // グループの帯域定義を追記する。
                                    line.append(" ");
                                    line.append(group_name);
                                }
                                // id が無い textblock に自動 id を付加する
                                InsertTextblockAutoId(&line, &textblockAutoId);
                                ExtractAny(line.c_str());
                            }
                        }
                        continue;
                    }
                    // *_id= や interleave=, renderinfo= が出現したらアノテーションの抽出を行う。
                    const char* tailStr = nullptr;
                    for (int i = 0; i < REMOTE_ID_COUNT; ++i)
                    {
                        const StringRef id = remoteIds[i];
                        size_t id_len = id.Len();
                        const char* idStr = citer - id_len;
                        if ((strncmp(idStr, id.Str().c_str(), id_len) == 0) &&
                            (*(idStr + id_len) == '='))
                        {
                            tailStr = idStr;
                            break;
                        }
                        // "textblock+半角スペース"が出現した場合もアノテーションの抽出を行う。
                        if (strncmp(idStr, "textblock ", 10) == 0)
                        {
                            tailStr = idStr;
                            break;
                        }
                    }

                    if (tailStr != nullptr)
                    {
                        auto tail = tailStr - 1;
                        if (tail <= head)
                        {
                            continue;
                        }

                        line.assign(head, tail);
                        head = tail + 1;
                        if (!line.empty())
                        {
                            if (!group_name.empty() &&
                                line.find("group=") == std::string::npos)
                            {
                                // グループの定義が記述されていない場合は
                                // グループの帯域定義を追記する。
                                line.append(" ");
                                line.append(group_name);
                            }
                            // id が無い textblock に自動 id を付加する
                            InsertTextblockAutoId(&line, &textblockAutoId);
                            ExtractAny(line.c_str());
                        }
                    }
                }
                while (isspace(*head) && head != cend)
                {
                    ++head;
                }
                line.assign(head, cend);
                if (!line.empty())
                {
                    // 1 行で書いた場合はここに来る。
                    if (iter->get_position().get_column() == 1)
                    {
                        // 複数行アノテーションの最後の1行。
                        ExtractAny(line.c_str());
                    }
                    else if (state.blockName.is_valid() &&
                        state.blockName.get_position().get_line() == iter->get_position().get_line() &&
                        state.blockName.get_position().get_file() == iter->get_position().get_file())
                    {
                        // Uniform Block に対するアノテーション
                        ExtractBlock(line.c_str());
                    }
                    else
                    {
                        // 変数に対するアノテーション。
                        ExtractVar(static_cast<int>(std::distance(state.tokens.cbegin(), iter)), line.c_str());
                    }
                }
            }
            else if (T_IDENTIFIER == *iter && id_uniform == iter->get_value().c_str())		//!< "uniform"でtokenを引っかける
            {
                BeginBlock(static_cast<int>(std::distance(state.tokens.cbegin(), iter)));
            }
            else if (T_IDENTIFIER == *iter && id_buffer == iter->get_value().c_str())		//!< "uniform"でtokenを引っかける
            {
                BeginSsboBlock(static_cast<int>(std::distance(state.tokens.cbegin(), iter)));
            }
            else if (T_LEFTBRACE == *iter)
            {
                ++braceDepth;
            }
            else if (T_RIGHTBRACE == *iter)
            {
                --braceDepth;
                if (braceDepth < 0)
                {
                    THROW_ERROR(ERRCODE_SHADER_CONVERTER_GLSL_SYNTAX_ERROR,
                        "syntax error. '}' not matched. %hs (l.%d)", iter->get_position().get_file().c_str(),
                        iter->get_position().get_line());
                }
                if (braceDepth == 0 && state.blockName.is_valid())
                {
                    state.blockName = state.blockVar = token_type();
                    EndBlock();
                }
                if (braceDepth == 0 && state.ssboBlockName.is_valid())
                {
                    state.ssboBlockName = state.ssboBlockVar = token_type();
                    EndSsboBlock();
                }
            }
            else if (T_IDENTIFIER == *iter && id_layout == iter->get_value().c_str())
            {
                ExtractLayout(static_cast<int>(std::distance(state.tokens.cbegin(), iter)));
            }
        }
    }
    catch( boost::wave::cpp_exception const& exception )
    {
        std::stringstream ss;
        ss << exception.file_name() << "(" << exception.line_no() << "): " << exception.description();
        errLog = ss.str();
    }
    catch (boost::wave::cpplexer::lexing_exception const& exception)
    {
        std::stringstream ss;
        ss << exception.file_name() << "(" << exception.line_no() << "): " << exception.description();
        errLog = ss.str();
    }
    catch( std::exception const& exception )
    {
        std::stringstream ss;
        ss << state.current_position.get_file() << "(" <<
            state.current_position.get_line() << "): "  << "exception: " << exception.what();
        errLog = ss.str();
    }
    catch( ... )
    {
        std::stringstream ss;
        ss << state.current_position.get_file() << "(" <<
            state.current_position.get_line() << "): " << "unexpected exception";
        errLog = ss.str();
    }
    state.annot << "</" << id_shader << '>' << std::endl;

    // wave の iterator を Extract 終了前にクリアしておかないと、メモリ破壊を引き起こす。
    state.ClearTempInfo();

    if (!errLog.empty())
    {
        THROW_ERROR(ERRCODE_SHADER_CONVERTER_EXTRACTION_ERROR, "Extraction error.\n%hs", errLog.c_str());
    }
}

void Extractor::ExtractOption(const char* str)
{
    CheckAnnotationSyntax(str);

    State& state = *m_pState;
    std::stringstream& annot = state.annot;

    bool has_params, is_predefined;
    context_type::position_type pos;
    std::vector<token_type> parameters;
    context_type::token_sequence_type definition;
    auto value = state.lastMacro.get_value();
    bool success = state.pCtx->get_macro_definition(value, has_params ,is_predefined,
        pos, parameters, definition);
    if (!success)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }

    annot << '<' << id_option << Attr(0, id_name, value);
    annot << std::endl; // 区切る。
    annot << str; // 改行が混ざるが気にしない。
    annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str() << '\"'
        << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line() << '\"'
        << " />" << std::endl;
}

void Extractor::BeginBlock(int tokenIndex)
{
    State& state = *m_pState;

    auto begin = state.tokens.cbegin();
    auto end = state.tokens.cend();
    std::advance(begin, tokenIndex + 1);
    auto iter = begin;

    // uniform block かどうかの確認
    for (; iter != end; ++iter)
    {
        if (T_SEMICOLON == *iter)
        {
            return; // UniformBlock でない。
        }
        else if (T_LEFTBRACE == *iter)
        {
            break;
        }
    }
    if (iter == end)
    {
        return;
    }

    // uniform block 内部の構造体を確認
    int braceDepth = 1;
    for (++iter; iter != end; ++iter)
    {
        if (T_LEFTBRACE == *iter)
        {
            ++braceDepth;
        }
        else if (T_RIGHTBRACE == *iter)
        {
            --braceDepth;
            if (braceDepth == 0)
            {
                break;
            }
        }
    }
    if (braceDepth > 0)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Broken Uniform Block.");
    }

    // uniform block の変数
    for (++iter; iter != end; ++iter)
    {
        if (T_IDENTIFIER == *iter)
        {
            state.blockVar = *iter;
            break;
        }
        else if (T_SEMICOLON == *iter)
        {
            break;
        }
    }

    // 最初から uniform block 名を探す
    for (iter = begin; iter != end; ++iter)
    {
        if (T_IDENTIFIER == *iter)
        {
            state.blockName = *iter;
            break;
        }
        else if (T_LEFTBRACE == *iter)
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Broken Uniform Block.");
        }
    }

    state.annot << '<' << id_block
        << Attr(0, id_name, state.blockName.get_value());
    if (state.blockVar.is_valid())
    {
        state.annot << Attr(0, id_var, state.blockVar.get_value());
    }
    state.annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str() << '\"'
        << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line() << '\"';
    state.annot << " >" << std::endl;
}

void Extractor::BeginSsboBlock(int tokenIndex)
{
    State& state = *m_pState;

    auto begin = state.tokens.cbegin();
    auto end = state.tokens.cend();
    std::advance(begin, tokenIndex + 1);
    auto iter = begin;

    // ssbo block かどうかの確認
    for (; iter != end; ++iter)
    {
        if (T_SEMICOLON == *iter)
        {
            return; // UniformBlock でない。
        }
        else if (T_LEFTBRACE == *iter)
        {
            break;
        }
    }
    if (iter == end)
    {
        return;
    }

    // ssbo block 内部の構造体を確認
    int braceDepth = 1;
    for (++iter; iter != end; ++iter)
    {
        if (T_LEFTBRACE == *iter)
        {
            ++braceDepth;
        }
        else if (T_RIGHTBRACE == *iter)
        {
            --braceDepth;
            if (braceDepth == 0)
            {
                break;
            }
        }
    }
    if (braceDepth > 0)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Broken Uniform Block.");
    }

    // ssbo block の変数
    for (++iter; iter != end; ++iter)
    {
        if (T_IDENTIFIER == *iter)
        {
            state.ssboBlockVar = *iter;
            break;
        }
        else if (T_SEMICOLON == *iter)
        {
            break;
        }
    }

    // 最初から ssbo block 名を探す
    for (iter = begin; iter != end; ++iter)
    {
        if (T_IDENTIFIER == *iter)
        {
            state.ssboBlockName = *iter;
            break;
        }
        else if (T_LEFTBRACE == *iter)
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Broken Shader Storage Block.");
        }
    }

    state.annot << '<' << id_ssbo_block
        << Attr(0, id_name, state.ssboBlockName.get_value());
    if (state.ssboBlockVar.is_valid())
    {
        state.annot << Attr(0, id_var, state.ssboBlockVar.get_value());
    }
    state.annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str() << '\"'
        << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line() << '\"';
    state.annot << " >" << std::endl;
}

void Extractor::EndBlock()
{
    m_pState->annot << "</" << id_block << ">" << std::endl;
}

void Extractor::EndSsboBlock()
{
    m_pState->annot << "</" << id_ssbo_block << ">" << std::endl;
}

void Extractor::ExtractLayout(int tokenIndex)
{
    State& state = *m_pState;

    auto begin = state.tokens.cbegin();
    auto end = state.tokens.cend();
    std::advance(begin, tokenIndex + 1);
    auto iter = begin;

    enum LayoutKind
    {
        LAYOUT_IN,
        LAYOUT_OUT,
        LAYOUT_UNIFORM
    };

    LayoutKind kind = LAYOUT_IN;

    // layout の種類を判別
    for (; iter != end; ++iter)
    {
        if (T_SEMICOLON == *iter)
        {
            return; // in, out, uniform ではない。
        }
        else if (id_uniform == iter->get_value().c_str())
        {
            // uniform/uniform_block
            kind = LAYOUT_UNIFORM;
            break;
        }
        else if (id_in == iter->get_value().c_str())
        {
            kind = LAYOUT_IN;
            break;
        }
        else if (id_out == iter->get_value().c_str())
        {
            kind = LAYOUT_OUT;
            break;
        }
    }
    // とりあえず、LAYOUT_IN 以外は不要
    if (iter == end || kind != LAYOUT_IN)
    {
        return;
    }

    const char* qualifier = nullptr;
    // () の文字列を取得
    for (iter = begin; iter != end; ++iter)
    {
        if (T_IDENTIFIER == *iter)
        {
            qualifier = iter->get_value().c_str();
            break;
        }
        else if (T_RIGHTPAREN == *iter)
        {
            THROW_ERROR(ERRCODE_INTERNAL, "Broken Layout-in.");
        }
    }

    if (qualifier == id_triangles_adjacency)
    {
        std::stringstream& annot = state.annot;
        annot << '<' << id_at << ' ' << Attr(0, id_layout, id_in);
        annot << Attr(0, id_qualifier, qualifier);
        annot << std::endl; // 区切る。
        annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str() << '\"'
            << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line() << '\"'
            << " />" << std::endl;
    }
}

void Extractor::ExtractBlock(const char* annotation)
{
    CheckAnnotationSyntax(annotation);

    State& state = *m_pState;
    std::stringstream& annot = state.annot;

    annot << '<' << id_block_info << std::endl; // 区切る。
    annot << annotation; // 改行が混ざるが気にしない。
    // ユーザ定義で上書きするために後から追加する。
    annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str() << '\"'
        << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line() << '\"'
        << " />" << std::endl;
}

void Extractor::ExtractSsboBlock(const char* annotation)
{
    CheckAnnotationSyntax(annotation);

    State& state = *m_pState;
    std::stringstream& annot = state.annot;

    annot << '<' << id_ssbo_block_info << std::endl; // 区切る。
    annot << annotation; // 改行が混ざるが気にしない。
    // ユーザ定義で上書きするために後から追加する。
    annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str() << '\"'
        << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line() << '\"'
        << " />" << std::endl;
}

void Extractor::ExtractVar(int tokenIndex, const char* annotation)
{
    CheckAnnotationSyntax(annotation);

    State& state = *m_pState;
    token_type type, name, length, qualifier;

    // 逆イテレーターを使って走査する
    auto rbegin = state.tokens.crbegin();
    auto rend = state.tokens.crend();
    std::advance(rbegin, state.tokens.size() - tokenIndex - 1);
    auto riter = rbegin;
    for (++riter; riter != rend; ++riter)
    {
        if (T_SEMICOLON == *riter)
        {
            break;
        }
    }
    if (riter == rend)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }
    for (++riter; riter != rend; ++riter)
    {
        if (T_IDENTIFIER == *riter)
        {
            name = *riter;
            break;
        }
        else if (T_PP_NUMBER == *riter)
        {
            // 数値リテラルは配列長とみなす。
            // support_option_prefer_pp_numbers 有効の場合は T_INTLIT ではなく T_PP_NUMBER を使う。
            length = *riter;
        }
        else if (T_SEMICOLON == *riter)
        {
            break;
        }

    }
    if (!name.is_valid())
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "The variable name(%hs) is not found.", annotation); // 変数名が取得できない
    }
    for (++riter; riter != rend; ++riter)
    {
        if (T_IDENTIFIER == *riter || T_FLOAT == *riter || T_INT == *riter || T_BOOL == *riter)
        {
            type = *riter;
            break;
        }
    }
    if (!type.is_valid())
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "The type name(%hs) is not found.", annotation); // 型名が取得できない
    }
    bool uniform = true;
    bool sampler = false;
    if ( !( state.blockName.is_valid() || state.ssboBlockName.is_valid() ) )
    {
        // Uniform Block 外
        for (++riter; riter != rend; ++riter)
        {
            if (T_SEMICOLON == *riter)
            {
                break;
            }
            else if (T_IDENTIFIER == *riter)
            {
                qualifier = *riter;
                break;
            }
        }
        if (!qualifier.is_valid())
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Qualifier is not found."); // 修飾子が取得できない
        }
        if (id_uniform == qualifier.get_value().c_str())
        {
            // samplerX, isamplerX, usamplerX
            const char* str = type.get_value().c_str();
            if (0 == strncmp(str, "sampler", strlen("sampler")) ||
                0 == strncmp(str + 1, "sampler", strlen("sampler")))
            {
                sampler = true;
            }
        }
        else
        {
            uniform = false;
        }
    }

    std::stringstream& annot = state.annot;

    if (sampler)
    {
        annot << '<' << id_sampler;
    }
    else if (uniform && state.blockName.is_valid())
    {
        annot << '<' << id_uniform;
    }
    else if (uniform && state.ssboBlockName.is_valid())
    {
        annot << '<' << id_buffer;
    }
    else
    {
        annot << '<' << id_attrib;
    }

    annot << Attr(0, id_name, name.get_value());

    annot << std::endl; // 区切る。
    annot << annotation; // 改行が混ざるが気にしない。
    if (length.is_valid())
    {
        annot << Attr(0, id_length, length.get_value());
    }
    else
    {
        // ユーザ定義で上書きするために後から追加する。
        annot << Attr(0, id_type, type.get_value());
    }
    annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str() << '\"'
        << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line() << '\"'
        << " />" << std::endl;
}

void Extractor::InsertTextblockAutoId(std::string* annotation, int* id)
{
    // textblock の特別処理。id が省略されている、もしくは空文字の場合は、場合は、id を自動で振る。
    // textblock の後ろにスペースが入っている場合と、空文字が入力された場合
    size_t searchIndex = annotation->find("textblock ", 0);
    while (searchIndex != std::string::npos)
    {
        std::stringstream textblockId;
        textblockId << "=\"textblock_auto_id" << (*id)++ << "\"";
        annotation->insert(searchIndex + 9, textblockId.str().c_str());
        searchIndex = annotation->find("textblock ", searchIndex);
    }
    searchIndex = annotation->find("textblock=\"\"", 0);
    while (searchIndex != std::string::npos)
    {
        std::stringstream textblockId;
        textblockId << "textblock_auto_id" << (*id)++;
        annotation->insert(searchIndex + 11, textblockId.str().c_str());
        searchIndex = annotation->find("textblock=\"\"", searchIndex);
    }
}

void Extractor::ExtractAny(const char* annotation)
{
    CheckAnnotationSyntax(annotation);
    // TODO: var="" に対応する変数の型を取得
    m_pState->annot << '<' << id_at << ' ';
    // 属性としての group と id としての group を区別
    if (isspace(*annotation))
    {
        ++annotation;
    }
    if (strncmp(annotation, "group=", 6) == 0)
    {
        m_pState->annot << "group_id=";
        annotation += 6;
    }
    m_pState->annot << annotation; // 改行が混ざるが気にしない。
    m_pState->annot << ' ' << id_file_info << "=\"" << m_pState->current_position.get_file().c_str()
        << '\"' << ' ' << id_line_info << "=\"" << m_pState->current_position.get_line()
        << '\"' << " />" << std::endl;
}

void Extractor::CheckAnnotationSyntax(const char* annotation)
{
    size_t length = strlen(annotation) + 8;
    std::unique_ptr<char[]> check(new char[length]);
    sprintf_s(check.get(), length, "<an %s />", annotation);
    nw::g3d::tool::util::XMLParser parser;
    if (!parser.ParseTree(check.get(), length))
    {
        THROW_ERROR(ERRCODE_XML_CONVERT_PARSE_FAILURE,
            "%hs, line %d: annotation syntax error.\n\t%hs",
            m_pState->current_position.get_file().c_str(), m_pState->current_position.get_line(), annotation);
    }
}

void Extractor::OnDefinedMacro(const token_type& token)
{
    if (m_pState->pCtx) // predefined macros を回避するため初期化後のみ処理。
    {
        m_pState->lastMacro = token;
    }
}

void Extractor::OnSkippedComment(const token_type& token)
{
    if (T_CPPCOMMENT == token &&
        m_pState->lastMacro.is_valid() &&
        m_pState->lastMacro.get_position().get_line() == token.get_position().get_line() &&
        m_pState->lastMacro.get_position().get_file() == token.get_position().get_file())
    {
        const char* str = token.get_value().c_str();
        size_t len = GetAnnotHeaderLen(str);
        if (len == 0)
        {
            return;
        }
        m_pState->current_position = token.get_position();
        ExtractOption(str + len);
    }
}

//--------------------------------------------------------------------------------------------------

void Extractor::Merge(bool unifiedAnnotation)
{
    State& state = *m_pState;

    std::unique_ptr<char[]>& stringPool = state.stringPool;
    std::stringstream& annotString = state.annot;
    size_t length = static_cast<size_t>(annotString.tellp()); // 書き込み位置を文字列長とみなす
    {
        static const StringRef tagOpen = "<annotation>\n";
        static const StringRef tagClose = "</annotation>\n";
        stringPool.reset(new char[length + tagOpen.Len() + tagClose.Len() + 1]);
        char* str = stringPool.get();
        memcpy(str, tagOpen.Data(), tagOpen.Len());
        str += tagOpen.Len();
        annotString.seekg(0, std::ios::beg);
        annotString.rdbuf()->sgetn(str, length);
        str += length;
        memcpy(str, tagClose.Data(), tagClose.Len());
        str += tagClose.Len();
        str[0] = '\0';
        length += tagOpen.Len() + tagClose.Len();
    }

    // 構築済みのアノテーション xml をパースする
    nw::g3d::tool::util::XMLParser parser;
    std::string xmlStr(stringPool.get());
    bool success = parser.ParseTree(stringPool.get(), length);
    if (!success)
    {
        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_XML_SYNTAX_ERROR, "Shader annotation syntax error.");
        PRINT_ERROR_LOG("Internal xml source.", xmlStr.c_str());
        PRINT_ERROR_LOG("%hs", xmlStr.c_str());
    }
    const nw::g3d::tool::util::XMLElement* pRoot = parser.Root();

    // ステージ別のアノテーションの値を記録するデーターベース。
    // marge時に不正な属性値の検査と、遠隔指定のマージを行う。基本的に遠隔指定の後勝ち。
    AnnotDB annotDB[ShaderStage_StageCount];

    // ステージ別の AnnnotDB を１つにまとめるデーターベース。
    // ステージ間で宣言が異なる場合は、エラーとしたり、上書き可能オプションがついていたら上書き処理を行う。
    SymbolDB& symbolDB = state.symbolDB;

    // option, attribute, sampler, uniformblockとuniform変数, ssboと内部変数のアノテーションを AnnotDB に記録
    for (const nw::g3d::tool::util::XMLElement* pShader= pRoot->Child(id_shader);
        pShader; pShader = pShader->NextSibling())
    {
        ShaderStage stage = static_cast<ShaderStage>(atoi(pShader->Attribute(id_stage)));
        AnnotDB& annot = annotDB[stage];
        // 各ステージのシェーダーソースのパスをシンボルに記録
        symbolDB.path[stage] = pShader->Attribute(id_path);

        // 不正なアノテーションが無いか、変数が多重定義されていないかを確認して AnnotDB に追加
        Resolve(annot.options, pShader->Child(id_option));
        if (stage == ShaderStage_Vertex)
        {
            Resolve(annot.attribs, pShader->Child(id_attrib));
        }
        Resolve(annot.samplers, pShader->Child(id_sampler));
        ResolveBlock(annot.blocks, pShader->Child(id_block));
        ResolveBlock(annot.ssboBlocks, pShader->Child(id_ssbo_block));
    }

    // アノテーションで遠隔指定したものの解決、基本的に設定を上書きする。
    // 列の先頭から記述されているアノテーションはここで処理される。
    for (const nw::g3d::tool::util::XMLElement* pShader = pRoot->Child(id_shader);
        pShader; pShader = pShader->NextSibling())
    {
        ShaderStage stage = static_cast<ShaderStage>(atoi(pShader->Attribute(id_stage)));
        AnnotDB& annot = annotDB[stage];

        // 遠隔指定の場合は id_at がついている at の先頭の id が遠隔指定の id になっているかをチェックする
        // *_id は必ず最初の属性でなければならない。
        for (const nw::g3d::tool::util::XMLElement* pAt = pShader->Child(id_at); pAt; pAt = pAt->NextSibling())
        {
            // at 要素の最初のアトリビュート名です。
            const char* pAttribName = pAt->AttributeId( 0 );
            bool isInvalidName = true;
            for(int idxIdName = 0; idxIdName < REMOTE_ID_COUNT; ++idxIdName)
            {
                if(remoteIds[idxIdName] == pAttribName)
                {
                    isInvalidName = false;
                    break;
                }
            }
            if(isInvalidName)
            {
                THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION,
                    "%hs, line %hs: %hs is invalid annotation name or %hs should not be defined at the head of the annotation.",
                    pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), pAttribName, pAttribName);
            }
        }

        for (const nw::g3d::tool::util::XMLElement* pAt = pShader->Child(id_at); pAt; pAt = pAt->NextSibling())
        {
            if (const char* id = pAt->Attribute(id_option_id))
            {
                if (pAt->AttributeIndex(id_option_id) != 0)
                {
                    // *_id は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "id", id);
                }

                if (!ResolveById(annot.options, pAt, id))		//!< 遠隔指定に含まれる id 名から group 名等の属性を埋めていく。
                {
                    if (!ExistsAnyStage(annotDB, &AnnotDB::options, id))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_REMOTE_ID_NOT_DEFINED,
                            "%hs, line %hs: option_id(%hs) is undefined.",
                            pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), id);
                    }
                }
                continue;
            }
            if (stage == ShaderStage_Vertex)
            {
                if (const char* id = pAt->Attribute(id_attrib_id))
                {
                    if (pAt->AttributeIndex(id_attrib_id) != 0)
                    {
                        // *_id は必ず最初の属性でなければならない。
                        THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                            pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "id", id);
                    }
                    if (!ResolveById(annot.attribs, pAt, id))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_REMOTE_ID_NOT_DEFINED,
                            "%hs, line %hs: attrib_id(%hs) is undefined.",
                            pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), id);
                    }
                    continue;
                }
            }
            if (const char* id = pAt->Attribute(id_sampler_id))
            {
                if (pAt->AttributeIndex(id_sampler_id) != 0)
                {
                    // *_id は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "id", id);
                }
                if (!ResolveById(annot.samplers, pAt, id))
                {
                    if (!ExistsAnyStage(annotDB, &AnnotDB::samplers, id))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_REMOTE_ID_NOT_DEFINED,
                            "%hs, line %hs: sampler_id(%hs) is undefined.",
                            pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), id);
                    }
                }
                continue;
            }
            if (const char* id = pAt->Attribute(id_block_id))
            {
                if (pAt->AttributeIndex(id_block_id) != 0)
                {
                    // *_id は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "id", id);
                }
                if (!ResolveById(annot.blocks, pAt, id))
                {
                    if (!ExistsAnyStage(annotDB, &AnnotDB::blocks, id))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_REMOTE_ID_NOT_DEFINED,
                            "%hs, line %hs: block_id(%hs) is undefined.",
                            pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), id);
                    }
                }
                continue;
            }
            if (const char* id = pAt->Attribute(id_uniform_id))
            {
                if (pAt->AttributeIndex(id_uniform_id) != 0)
                {
                    // *_id は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "id", id);
                }
                success = false;
                for (auto block = annot.blocks.begin(); block != annot.blocks.end(); ++block)
                {
                    if (ResolveById(block->uniforms, pAt, id))
                    {
                        success = true;
                        break;
                    }
                }
                if (!success)
                {
                    if (!ExistsAnyStage(annotDB, &AnnotDB::blocks, &AnnotBlock::uniforms, id))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_REMOTE_ID_NOT_DEFINED,
                            "%hs, line %hs: uniform_id(%hs) is undefined on the stage(%hs).",
                            pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), id, stage_names[stage].Str().c_str());
                    }
                }
                continue;
            }
            if (const char* textblock = pAt->Attribute(id_textblock))
            {
                if (pAt->AttributeIndex(id_textblock) != 0)
                {
                    // textblock は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "textblock", textblock);
                }
                ResolveById(annot.textBlocks, pAt, textblock);
                continue;
            }
            if (const char* renderinfo = pAt->Attribute(id_renderinfo))
            {
                if (pAt->AttributeIndex(id_renderinfo) != 0)
                {
                    // renderinfo は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "renderinfo", renderinfo);
                }
                ResolveById(annot.renderInfos, pAt, renderinfo);
                continue;
            }
            if (const char* interleave = pAt->Attribute(id_interleave))
            {
                if (pAt->AttributeIndex(id_interleave) != 0)
                {
                    // interleave は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "interleave", interleave);
                }
                ResolveById(annot.interleaves, pAt, interleave);
                continue;
            }
            if (const char* group = pAt->Attribute(id_group_id))
            {
                // group= は group_id= に置換されている
                if (pAt->AttributeIndex(id_group_id) != 0)
                {
                    // group は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "group", group);
                }
                ResolveById(annot.groups, pAt, group);
                continue;
            }
            if (const char* streamout = pAt->Attribute(id_streamout))
            {
                if (pAt->AttributeIndex(id_streamout) != 0)
                {
                    // streamout は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "streamout", streamout);
                }
                ResolveById(annot.streamouts, pAt, streamout);
                continue;
            }
            if (const char* revision = pAt->Attribute(id_revision))
            {
                if (pAt->AttributeIndex(id_revision) != 0)
                {
                    // revision は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "revision", revision);
                }
                ResolveById(annot.revisions, pAt, revision);
                continue;
            }
            if (const char* layout = pAt->Attribute(id_layout))
            {
                if (pAt->AttributeIndex(id_layout) != 0)
                {
                    // layout は必ず最初の属性でなければならない。
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "layout", layout);
                }
                annot.layouts.emplace_back(AnnotLayout(pAt));
                continue;
            }
            if (const char* page = pAt->Attribute(id_page))
            {
                if (pAt->AttributeIndex(id_page) != 0)
                {
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_ID_LOCATION, "Identifier_InvalidAnnotationPosition",
                        pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), "page", page);
                }
                ResolveById(annot.pages, pAt, page);
                continue;
            }

            const char* id = pAt->AttributeId(0);
            const char* value = pAt->Attribute(id);
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_ANNOTATION,
                "%hs, line %hs: Invalid annotation id(%hs=\"%hs\").",
                pAt->Attribute(id_file_info), pAt->Attribute(id_line_info), id, value);
        }

        // textblock の特殊処理。id のマージが全て終わった後に行う。
        // 同じ id が複数存在する場合、その他の属性値が異なれば別の TextBlock として追加する。
        static int textBlockIdIndex = 0;
        for (const nw::g3d::tool::util::XMLElement* pAt = pShader->Child(id_at); pAt; pAt = pAt->NextSibling())
        {
            if (const char* textblock = pAt->Attribute(id_textblock))
            {
                auto iter = std::find_if(annot.textBlocks.begin(), annot.textBlocks.end(), [&](AnnotTextBlock& annot)
                {
                    return annot.GetAttributeValue(id_textblock) == textblock;
                });
                if (iter != annot.textBlocks.end())
                {
                    // 同じ id で他の属性も同じものが既にある場合は追加しない。同じ id でも属性が異なる場合は追加する。
                    bool match = false;
                    AnnotTextBlock newTextBlock = AnnotTextBlock(pAt);
                    for (int textblockIndex = 0; textblockIndex < annot.textBlocks.size(); textblockIndex++)
                    {
                        AnnotTextBlock& iter2 = annot.textBlocks[textblockIndex];
                        if (newTextBlock.GetAttributeValue(id_textblock) == iter2.GetAttributeValue(id_textblock) && newTextBlock.GetAttributeValue(id_group) == iter2.GetAttributeValue(id_group) && newTextBlock.GetAttributeValue(id_order) == iter2.GetAttributeValue(id_order))
                        {
                            match = true;
                            break;
                        }
                    }
                    if (!match)
                    {
                        // id を作成してリストに追加
                        std::stringstream id;
                        id << iter->GetAttributeValue(id_textblock).Str() << "_" << textBlockIdIndex;
                        textblock_auto_id_array.push_back(id.str().c_str());

                        // 全ステージの検索が終わって、自動 id リストが確定するまで仮 id を入れておく(vector のアドレスが変わるため)
                        newTextBlock.SetAttributeValue(id_textblock, "tmp_id");
                        newTextBlock.SetAttributeValue(id_text, iter->GetAttributeValue(id_text));
                        annot.textBlocks.push_back(std::move(newTextBlock));
                        textBlockIdIndex++;
                    }
                }
            }
        }
    }
    // ステージ別のアノテーション情報を AnnotDB に記録完了

    // textblock の仮 id の解決
    int autoIdIndex = 0;
    for (int idxStage = 0; idxStage < ShaderStage_StageCount; ++idxStage)
    {
        AnnotDB& annot = annotDB[idxStage];
        for (int textblockIndex = 0; textblockIndex < annot.textBlocks.size(); textblockIndex++)
        {
            AnnotTextBlock& textblock = annot.textBlocks[textblockIndex];
            if (strcmp(textblock.GetAttributeValue(id_textblock).Str().c_str(), "tmp_id") == 0)
            {
                textblock.SetAttributeValue(id_textblock, textblock_auto_id_array[autoIdIndex++]);
            }
        }
    }

    // AnnnotDB を SymbolDB に入れていく。SymbolDB はシンボル名(変数名)をステージ別で保持する
    for (int idxStage = 0; idxStage < ShaderStage_StageCount; ++idxStage)
    {
        ShaderStage stage = static_cast<ShaderStage>(idxStage);
        AnnotDB& annotation = annotDB[idxStage];
        MergeToSymbol(stage, symbolDB.options, annotation.options, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.attribs, annotation.attribs, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.samplers, annotation.samplers, unifiedAnnotation);
        auto buildBlock = [&](const AnnotBlock& annot, auto& blocks)
        {
            auto found = std::find_if(blocks.begin(), blocks.end(), [&](SymbolBlock& symbol)
            {
                return symbol.GetSymbolValue(id_id) == annot.GetAttributeValue(id_id);
            });
            if (found == blocks.end())
            {
                blocks.emplace_back(SymbolBlock(stage, annot));
            }
            else if (found->CompareAttr(annot))
            {
                // アノテーション設定がすべて等しい場合、ユニフォームブロック名とユニフォームブロック変数名、ユニフォーム変数名はステージ間で異なっていてもよい。
                found->strName[stage].assign(annot.GetAttributeValue(id_name).Str());
                found->strVar[stage].assign(annot.GetAttributeValue(id_var).Str());
                auto iter = annot.uniforms.begin();
                for (auto uniform = found->uniforms.begin(); uniform != found->uniforms.end(); ++uniform, ++iter)
                {
                    uniform->strName[stage].assign(iter->GetAttributeValue(id_name).Str());
                }
            }
            else if (unifiedAnnotation)
            {
                // アノテーション設定が異なる場合 & --unified-annotation 有効時は vs -> gs -> fs -> cs の順でアノテーションを上書きする。
                found->Override(annot);
                found->strName[stage].assign(annot.GetAttributeValue(id_name).Str());
                found->strVar[stage].assign(annot.GetAttributeValue(id_var).Str());
                auto iter = annot.uniforms.begin();
                for (auto uniform = found->uniforms.begin(); uniform != found->uniforms.end(); ++uniform, ++iter)
                {
                    uniform->strName[stage].assign(iter->GetAttributeValue(id_name).Str());
                }
                PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT,
                    "%hs, line %hs: Annotation(%hs) is overridden by other stages.",
                    annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_id).Str().c_str());
            }
            else
            {
                THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_STAGE_INCONSISTENT, "Identifier_InconsistentStageAnnotaion",
                     annot.GetAttributeValue(id_file_info).Str().c_str(), annot.GetAttributeValue(id_line_info).Str().c_str(), annot.GetAttributeValue(id_id).Str().c_str());
            }
        };
        auto buildUniformBlock = [&]( const AnnotBlock& annot )
        {
            buildBlock( annot, symbolDB.blocks );
        };
        auto buildSsboBlock = [&]( const AnnotSsboBlock& annot )
        {
            buildBlock( annot, symbolDB.ssboBlocks );
        };
        std::for_each(annotation.blocks.begin(), annotation.blocks.end(), buildUniformBlock);
        std::for_each(annotation.ssboBlocks.begin(), annotation.ssboBlocks.end(), buildSsboBlock);

        MergeToSymbol(stage, symbolDB.renderInfos, annotation.renderInfos, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.textBlocks, annotation.textBlocks, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.interleaves, annotation.interleaves, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.groups, annotation.groups, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.streamouts, annotation.streamouts, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.revisions, annotation.revisions, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.layouts, annotation.layouts, unifiedAnnotation);
        MergeToSymbol(stage, symbolDB.pages, annotation.pages, unifiedAnnotation);
    }

    // グループの名前対応チェック
    CheckGroupReference(symbolDB.groups, symbolDB.options);
    CheckGroupReference(symbolDB.groups, symbolDB.samplers);
    std::for_each(symbolDB.blocks.cbegin(), symbolDB.blocks.cend(), [&](const SymbolBlock& block)
    {
        CheckGroupReference(symbolDB.groups, block.uniforms);
    });
    std::for_each(symbolDB.ssboBlocks.cbegin(), symbolDB.ssboBlocks.cend(), [&](const SymbolSsboBlock& block)
    {
        CheckGroupReference(symbolDB.groups, block.uniforms);
    });
    CheckGroupReference(symbolDB.groups, symbolDB.renderInfos);
    CheckGroupReference(symbolDB.groups, symbolDB.textBlocks);
    CheckGroupReference(symbolDB.groups, symbolDB.groups);

    // ページの名前対応チェック
    for (auto group = symbolDB.groups.cbegin(); group != symbolDB.groups.cend(); ++group)
    {
        if (!group->GetSymbolValue(id_page).empty())
        {
            if (std::find_if(symbolDB.pages.cbegin(), symbolDB.pages.cend(),
                [&](const SymbolPage& page)
            {
                return group->GetSymbolValue(id_page) == page.GetSymbolValue(id_page);
            }) == symbolDB.pages.cend())
            {
                PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_ID_NOT_FOUND,
                    "Page(%hs) is not exist.", group->GetSymbolValue(id_page).c_str());
            }
        }
    }

    // 依存パラメータの名前対応チェック
    auto getMaterialBlock = [&](const auto& block){ return block.GetSymbolValue(id_type) == "material"; };
    auto materialBlock = std::find_if(symbolDB.blocks.cbegin(), symbolDB.blocks.cend(), getMaterialBlock);
    auto materialSsboBlock = std::find_if(symbolDB.ssboBlocks.cbegin(), symbolDB.ssboBlocks.cend(), getMaterialBlock);

    // システムブロックはssbo, uboの重複はできません。
    if( ( materialBlock != symbolDB.blocks.cend() ) &&  ( materialSsboBlock != symbolDB.ssboBlocks.cend() ) )
    {
        THROW_ERROR( ERRCODE_XML_SYSTEM_BLOCK_DUPLICATION, "System block duplication: %hs", materialBlock->GetSymbolValue(id_id).c_str() );
    }

    auto checkBlockDependError = [&]( const auto& materialBlock )
    {
        std::unique_ptr<bool[]> depends(new bool[materialBlock->uniforms.size()]);
        std::fill_n(depends.get(), materialBlock->uniforms.size(), false);
        for (auto uniform = materialBlock->uniforms.cbegin();
            uniform != materialBlock->uniforms.cend(); ++uniform)
        {
            if (!uniform->GetSymbolValue(id_depend).empty())
            {
                auto found = std::find_if(materialBlock->uniforms.cbegin(),
                    materialBlock->uniforms.cend(), [&](const SymbolUniform& uniform2)
                {
                    return uniform2.GetSymbolValue(id_id) == uniform->GetSymbolValue(id_depend);
                });
                if (found == materialBlock->uniforms.cend())
                {
                    PRINT_WARNING(ERRCODE_SHADER_ANNOTATION_ID_NOT_FOUND,
                        "Depend(%hs) is not exist.", uniform->GetSymbolValue(id_depend).c_str());
                }
                int index = static_cast<int>(std::distance(materialBlock->uniforms.cbegin(), found));
                if (index != std::distance(materialBlock->uniforms.cbegin(), uniform))
                {
                    if (depends[index])
                    {
                        PRINT_WARNING(ERRCODE_UNSUPPORTED,
                            "Overlap Dependency is not supported (%hs).", uniform->GetSymbolValue(id_depend).c_str());
                    }
                    depends[index] = true;
                }
            }
        }
    };
    if( materialBlock != symbolDB.blocks.cend() )
    {
        checkBlockDependError( materialBlock );
    }
    else if( materialSsboBlock != symbolDB.ssboBlocks.cend() )
    {
        checkBlockDependError( materialSsboBlock );
    }

    // TODO: 情報の不足をチェック
}

//--------------------------------------------------------------------------------------------------
template<typename TElemBlock, typename TElemVar, typename TSymbolBlock>
void BuildBlock( std::vector<TElemBlock>& shadingModelBlockVarArray, const std::vector<TSymbolBlock>& blocks )
{
    for (auto block = blocks.cbegin(); block != blocks.cend(); ++block)
    {
        TElemBlock el_block_var;
        el_block_var.id = block->GetSymbolValue(id_id);
        if (block->GetSymbolValue(id_type).empty())
        {
            el_block_var.type = TElemBlock::none;
        }
        else
        {
            el_block_var.type = TElemBlock::GetType(block->GetSymbolValue(id_type));
        }
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (!block->strName[stage].empty())
            {
                el_block_var.shader_symbol[stage].Validate();
                el_block_var.shader_symbol[stage]->name = block->strName[stage];
            }
        }
        // ユニフォームブロック変数が存在する場合は、ユニフォーム変数の頭に prefix を付ける。
        std::string prefix[ShaderStage_StageCount];
        for (int iStage = 0; iStage < ShaderStage_StageCount; ++iStage)
        {
            if (block->strVar[iStage].empty())
            {
                prefix[iStage] = "";
            }
            else
            {
                prefix[iStage] = block->strVar[iStage];
                prefix[iStage] += ".";
            }
        }

        for (auto uniform = block->uniforms.cbegin(); uniform != block->uniforms.cend(); ++uniform)
        {
            TElemVar el_uniform_var;
            el_uniform_var.id = uniform->GetSymbolValue(id_id);
            el_uniform_var.hint = uniform->GetSymbolValue(id_hint);

            el_uniform_var.type = GetUniformType(uniform->GetSymbolValue(id_id), uniform->GetSymbolValue(id_type));

            // block_var の type が option の場合に uniform の type は必ず "int" or "bool" である必要がある。
            // TODO: elem_ssbo_block_varの時はエラー.
            if (el_block_var.type.value == nw::g3d::tool::g3dif::elem_block_var::option)
            {
                if (el_uniform_var.type.value != nw::g3d::tool::g3dif::elem_shader_param::type_bool &&
                    el_uniform_var.type.value != nw::g3d::tool::g3dif::elem_shader_param::type_int &&
                    el_uniform_var.type.value != nw::g3d::tool::g3dif::elem_shader_param::type_uint)
                {
                    THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_UNIFORM,
                        "%hs, line %hs: A uniform contained in a option type uniform block must be \"int\" or \"bool\" values.",
                        uniform->GetSymbolValue(id_file_info).c_str(), uniform->GetSymbolValue(id_line_info).c_str());
                }
            }

            if (!uniform->GetSymbolValue(id_default).empty())
            {
                // デフォルト値の数についてエラーチェック
                std::vector<std::string> defaultValueArray;
                nw::g3d::tool::util::Split(defaultValueArray, uniform->GetSymbolValue(id_default), " \t");

                for (auto value = defaultValueArray.begin(); value != defaultValueArray.end(); ++value)
                {
                    if (!nw::g3d::tool::util::ModifyFloatValue(*value))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_DEFAULT,
                            "%hs, line %hs: Invalid expression of the default parameter on the annotation(%hs).",
                            uniform->GetSymbolValue(id_file_info).c_str(), uniform->GetSymbolValue(id_line_info).c_str(), uniform->GetSymbolValue(id_id).c_str());
                    }
                }

                if (static_cast<int>(defaultValueArray.size()) != nw::g3d::tool::g3dif::elem_uniform_var::GetDefaultCount(el_uniform_var.type.value))
                {
                    THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_DEFAULT,
                        "%hs, line %hs: Invalid a default parameter count on the annotation(%hs).",
                        uniform->GetSymbolValue(id_file_info).c_str(), uniform->GetSymbolValue(id_line_info).c_str(), uniform->GetSymbolValue(id_id).c_str());
                }
                el_uniform_var.default_value.value = nw::g3d::tool::util::ModifyBlank(defaultValueArray);
            }
            else
            {
                el_uniform_var.default_value = nw::g3d::tool::g3dif::elem_uniform_var::GetDefault(el_uniform_var.type.value);
            }

            if (uniform->GetSymbolValue(id_nostrip).empty())
            {
                el_uniform_var.nostrip = false;
            }
            else
            {
                el_uniform_var.nostrip = nw::g3d::tool::g3dif::GetBoolValue(uniform->GetSymbolValue(id_nostrip));
            }

            el_uniform_var.converter = uniform->GetSymbolValue(id_converter);

            el_uniform_var.depend = uniform->GetSymbolValue(id_depend);

            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (!uniform->strName[stage].empty())
                {
                    el_uniform_var.shader_symbol[stage].Validate();
                    el_uniform_var.shader_symbol[stage]->name = prefix[stage];
                    el_uniform_var.shader_symbol[stage]->name.value += uniform->strName[stage];
                }
            }

            if (!uniform->GetSymbolValue(id_label).empty())
            {
                el_uniform_var.ui_label.Validate();
                el_uniform_var.ui_label->value = uniform->GetSymbolValue(id_label);
            }

            if (!uniform->GetSymbolValue(id_comment).empty())
            {
                el_uniform_var.ui_comment.Validate();
                el_uniform_var.ui_comment->value = uniform->GetSymbolValue(id_comment);
            }

            if (!uniform->GetSymbolValue(id_group).empty())
            {
                el_uniform_var.ui_group.Validate();
                el_uniform_var.ui_group->group_name = uniform->GetSymbolValue(id_group);
            }

            if (!uniform->GetSymbolValue(id_order).empty())
            {
                el_uniform_var.ui_order.Validate();
                el_uniform_var.ui_order->value = uniform->GetSymbolValue(id_order);
            }

            if (!uniform->GetSymbolValue(id_item).empty())
            {
                el_uniform_var.ui_item.Validate();
                el_uniform_var.ui_item->value = uniform->GetSymbolValue(id_item);
            }

            if (!uniform->GetSymbolValue(id_min).empty())
            {
                el_uniform_var.ui_min.Validate();
                el_uniform_var.ui_min->value = uniform->GetSymbolValue(id_min);
            }

            if (!uniform->GetSymbolValue(id_max).empty())
            {
                el_uniform_var.ui_max.Validate();
                el_uniform_var.ui_max->value = uniform->GetSymbolValue(id_max);
            }

            if (!uniform->GetSymbolValue(id_visible).empty())
            {
                el_uniform_var.ui_visible.Validate();
                el_uniform_var.ui_visible->value = uniform->GetSymbolValue(id_visible);
            }

            if (!uniform->GetSymbolValue(id_default_min).empty())
            {
                el_uniform_var.ui_default_min.Validate();
                el_uniform_var.ui_default_min->value = uniform->GetSymbolValue(id_default_min);
            }

            if (!uniform->GetSymbolValue(id_default_max).empty())
            {
                el_uniform_var.ui_default_max.Validate();
                el_uniform_var.ui_default_max->value = uniform->GetSymbolValue(id_default_max);
            }

            el_block_var.uniform_var_array.push_back(el_uniform_var);
        }
        shadingModelBlockVarArray.push_back(el_block_var);
    }
}

void Extractor::WriteDefinition(nw::g3d::tool::g3dif::elem_shading_model& el_shading_model, FileDB* /*fileDB*/) const
{
    SymbolDB& symbolDB = m_pState->symbolDB;

    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
    {
        const StringRef& path = symbolDB.path[stage];
        if (!path.Empty())
        {
            el_shading_model.shader_stage[stage].Validate();
            el_shading_model.shader_stage[stage]->path = path.Str();
        }
    }

    for (auto macro = m_pState->fixed_macro_map.cbegin(); macro != m_pState->fixed_macro_map.cend(); ++macro)
    {
        nw::g3d::tool::g3dif::elem_macro el_macro;

        el_macro.name = macro->first;
        el_macro.value = macro->second;

        el_shading_model.macro_array.push_back(el_macro);
    }

    for (auto option = symbolDB.options.cbegin(); option != symbolDB.options.cend(); ++option)
    {
        // option enable="false" の場合にオプションから排除する。
        if (option->GetSymbolValue(id_enable) == "false")
        {
            continue;
        }

        // マクロ名と重複していた場合はエラーにする。
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (!option->strName[stage].empty())
            {
                auto& optionName = option->strName[stage];
                auto result = m_pState->fixed_macro_map.find(optionName);
                if (result != m_pState->fixed_macro_map.cend())
                {
                    // NOTE: マクロ名と重複する場合はそもそも抽出できないので、ここにはこない。
                    THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_NAME, "Option name(%hs) overlap macro definition.", optionName.c_str());
                }
            }
        }

        nw::g3d::tool::g3dif::elem_option_var el_option_var;

        el_option_var.id = option->GetSymbolValue(id_id);

        if (!option->GetSymbolValue(id_choice).empty())
        {
            el_option_var.choice = option->GetSymbolValue(id_choice);
        }
        else
        {
            el_option_var.choice->assign("bool");
        }

        if (!option->GetSymbolValue(id_default).empty())
        {
            el_option_var.default_value = option->GetSymbolValue(id_default);
        }

        if (!option->GetSymbolValue(id_type).empty())
        {
            el_option_var.type = nw::g3d::tool::g3dif::elem_option_var::GetType(option->GetSymbolValue(id_type));
        }

        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (!option->strName[stage].empty())
            {
                el_option_var.shader_symbol[stage].Validate();
                el_option_var.shader_symbol[stage]->name = option->strName[stage];
            }
        }

        if (!option->GetSymbolValue(id_label).empty())
        {
            el_option_var.ui_label.Validate();
            el_option_var.ui_label->value = option->GetSymbolValue(id_label);
        }

        if (!option->GetSymbolValue(id_comment).empty())
        {
            el_option_var.ui_comment.Validate();
            el_option_var.ui_comment->value = option->GetSymbolValue(id_comment);
        }

        if (!option->GetSymbolValue(id_group).empty())
        {
            el_option_var.ui_group.Validate();
            el_option_var.ui_group->group_name = option->GetSymbolValue(id_group);
        }

        if (!option->GetSymbolValue(id_order).empty())
        {
            el_option_var.ui_order.Validate();
            el_option_var.ui_order->value = option->GetSymbolValue(id_order);
        }

        if (!option->GetSymbolValue(id_visible).empty())
        {
            el_option_var.ui_visible.Validate();
            el_option_var.ui_visible->value = option->GetSymbolValue(id_visible);
        }

        el_shading_model.option_var_array.push_back(el_option_var);
    }

    for (auto attrib = symbolDB.attribs.cbegin(); attrib != symbolDB.attribs.cend(); ++attrib)
    {
        nw::g3d::tool::g3dif::elem_attrib_var el_attrib_var;
        el_attrib_var.id = attrib->GetSymbolValue(id_id);
        el_attrib_var.hint = attrib->GetSymbolValue(id_hint);
        el_attrib_var.type = GetAttribType(attrib->GetSymbolValue(id_id), attrib->GetSymbolValue(id_type));

        if (!attrib->strName[ShaderStage_Vertex].empty())
        {
            el_attrib_var.vertex_symbol.name = attrib->strName[ShaderStage_Vertex];
        }
        else
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal Error");
        }

        if (!attrib->GetSymbolValue(id_label).empty())
        {
            el_attrib_var.ui_label.Validate();
            el_attrib_var.ui_label->value = attrib->GetSymbolValue(id_label);
        }

        if (!attrib->GetSymbolValue(id_comment).empty())
        {
            el_attrib_var.ui_comment.Validate();
            el_attrib_var.ui_comment->value = attrib->GetSymbolValue(id_comment);
        }

        if (!attrib->GetSymbolValue(id_group).empty())
        {
            el_attrib_var.ui_group.Validate();
            el_attrib_var.ui_group->group_name = attrib->GetSymbolValue(id_group);
        }

        if (!attrib->GetSymbolValue(id_editable).empty())
        {
            el_attrib_var.ui_editable.Validate();
            el_attrib_var.ui_editable->value = attrib->GetSymbolValue(id_editable);
        }

        if (!attrib->GetSymbolValue(id_order).empty())
        {
            el_attrib_var.ui_order.Validate();
            el_attrib_var.ui_order->value = attrib->GetSymbolValue(id_order);
        }

        el_shading_model.attrib_var_array.push_back(el_attrib_var);
    }

    for (auto sampler = symbolDB.samplers.cbegin(); sampler != symbolDB.samplers.cend(); ++sampler)
    {
        nw::g3d::tool::g3dif::elem_sampler_var el_sampler_var;
        el_sampler_var.id = sampler->GetSymbolValue(id_id);
        el_sampler_var.hint = sampler->GetSymbolValue(id_hint);
        el_sampler_var.type = GetSamplerType(sampler->GetSymbolValue(id_id), sampler->GetSymbolValue(id_type));

        el_sampler_var.alt = sampler->GetSymbolValue(id_alt);

        if (sampler->GetSymbolValue(id_nostrip).empty())
        {
            el_sampler_var.nostrip = false;
        }
        else
        {
            el_sampler_var.nostrip = nw::g3d::tool::g3dif::GetBoolValue(sampler->GetSymbolValue(id_nostrip));
        }

        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (!sampler->strName[stage].empty())
            {
                el_sampler_var.shader_symbol[stage].Validate();
                el_sampler_var.shader_symbol[stage]->name = sampler->strName[stage];
            }
        }

        if (!sampler->GetSymbolValue(id_label).empty())
        {
            el_sampler_var.ui_label.Validate();
            el_sampler_var.ui_label->value = sampler->GetSymbolValue(id_label);
        }

        if (!sampler->GetSymbolValue(id_comment).empty())
        {
            el_sampler_var.ui_comment.Validate();
            el_sampler_var.ui_comment->value = sampler->GetSymbolValue(id_comment);
        }

        if (!sampler->GetSymbolValue(id_group).empty())
        {
            el_sampler_var.ui_group.Validate();
            el_sampler_var.ui_group->group_name = sampler->GetSymbolValue(id_group);
        }

        if (!sampler->GetSymbolValue(id_order).empty())
        {
            el_sampler_var.ui_order.Validate();
            el_sampler_var.ui_order->value = sampler->GetSymbolValue(id_order);
        }

        if (!sampler->GetSymbolValue(id_visible).empty())
        {
            el_sampler_var.ui_visible.Validate();
            el_sampler_var.ui_visible->value = sampler->GetSymbolValue(id_visible);
        }

        el_shading_model.sampler_var_array.push_back(el_sampler_var);
    }

    BuildBlock<nw::g3d::tool::g3dif::elem_block_var, nw::g3d::tool::g3dif::elem_uniform_var, SymbolBlock>( el_shading_model.block_var_array, symbolDB.blocks );
    BuildBlock<nw::g3d::tool::g3dif::elem_ssbo_block_var, nw::g3d::tool::g3dif::elem_ssbo_uniform_var, SymbolBlock>( el_shading_model.ssbo_block_var_array, symbolDB.ssboBlocks );

    for (auto render_info = symbolDB.renderInfos.cbegin(); render_info != symbolDB.renderInfos.cend(); ++render_info)
    {
        nw::g3d::tool::g3dif::elem_render_info_slot el_render_info_slot;
        el_render_info_slot.name = render_info->GetSymbolValue(id_renderinfo);

        // 省略時は string として扱う
        if (render_info->GetSymbolValue(id_type).empty())
        {
            el_render_info_slot.type = nw::g3d::tool::g3dif::elem_render_info_slot::type_string;
        }
        else
        {
            el_render_info_slot.type = nw::g3d::tool::g3dif::elem_render_info_slot::GetType(render_info->GetSymbolValue(id_type));
        }

        if (render_info->GetSymbolValue(id_count).empty())
        {
            el_render_info_slot.count = std::string("1");
        }
        else
        {
            el_render_info_slot.count = render_info->GetSymbolValue(id_count);
        }

        if (render_info->GetSymbolValue(id_optional).empty())
        {
            el_render_info_slot.optional = true;
        }
        else
        {
            el_render_info_slot.optional = nw::g3d::tool::g3dif::GetBoolValue(render_info->GetSymbolValue(id_optional));
        }

        el_render_info_slot.choice = render_info->GetSymbolValue(id_choice);
        {
            // デフォルト値のエラーチェック
            std::vector<std::string> defaultValueArray;
            nw::g3d::tool::util::Split(defaultValueArray, render_info->GetSymbolValue(id_default), " \t");

            if (el_render_info_slot.type.value == nw::g3d::tool::g3dif::elem_render_info_slot::type_float)
            {
                for (auto value = defaultValueArray.begin(); value != defaultValueArray.end(); ++value)
                {
                    if (!nw::g3d::tool::util::ModifyFloatValue(*value))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_DEFAULT,
                            "%hs, line %hs: Invalid expression of the default parameter on the annotation(%hs).",
                            render_info->GetSymbolValue(id_file_info).c_str(), render_info->GetSymbolValue(id_line_info).c_str(), render_info->GetSymbolValue(id_renderinfo).c_str());
                    }
                }
            }
            else if (el_render_info_slot.type.value == nw::g3d::tool::g3dif::elem_render_info_slot::type_int)
            {
                for (auto value = defaultValueArray.begin(); value != defaultValueArray.end(); ++value)
                {
                    if (!nw::g3d::tool::util::ModifyIntValue(*value))
                    {
                        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_DEFAULT,
                            "%hs, line %hs: Invalid expression of the default parameter on the annotation(%hs).",
                            render_info->GetSymbolValue(id_file_info).c_str(), render_info->GetSymbolValue(id_line_info).c_str(), render_info->GetSymbolValue(id_renderinfo).c_str());
                    }
                }
            }

            el_render_info_slot.default_value = nw::g3d::tool::util::ModifyBlank(defaultValueArray);
        }

        if (!render_info->GetSymbolValue(id_label).empty())
        {
            el_render_info_slot.ui_label.Validate();
            el_render_info_slot.ui_label->value = render_info->GetSymbolValue(id_label);
        }

        if (!render_info->GetSymbolValue(id_comment).empty())
        {
            el_render_info_slot.ui_comment.Validate();
            el_render_info_slot.ui_comment->value = render_info->GetSymbolValue(id_comment);
        }

        if (!render_info->GetSymbolValue(id_group).empty())
        {
            el_render_info_slot.ui_group.Validate();
            el_render_info_slot.ui_group->group_name = render_info->GetSymbolValue(id_group);
        }

        if (!render_info->GetSymbolValue(id_order).empty())
        {
            el_render_info_slot.ui_order.Validate();
            el_render_info_slot.ui_order->value = render_info->GetSymbolValue(id_order);
        }

        if (!render_info->GetSymbolValue(id_item).empty())
        {
            el_render_info_slot.ui_item.Validate();
            el_render_info_slot.ui_item->value = render_info->GetSymbolValue(id_item);
        }

        if (!render_info->GetSymbolValue(id_visible).empty())
        {
            el_render_info_slot.ui_visible.Validate();
            el_render_info_slot.ui_visible->value = render_info->GetSymbolValue(id_visible);
        }

        el_shading_model.render_info_slot_array.push_back(el_render_info_slot);
    }

    for (auto textblock = symbolDB.textBlocks.cbegin(); textblock != symbolDB.textBlocks.cend(); ++textblock)
    {
        nw::g3d::tool::g3dif::elem_textblock el_textblock;
        el_textblock.id = textblock->GetSymbolValue(id_textblock);
        el_textblock.text = textblock->GetSymbolValue(id_text);

        if (!textblock->GetSymbolValue(id_group).empty())
        {
            el_textblock.ui_group.Validate();
            el_textblock.ui_group->group_name = textblock->GetSymbolValue(id_group);
        }

        if (!textblock->GetSymbolValue(id_order).empty())
        {
            el_textblock.ui_order.Validate();
            el_textblock.ui_order->value = textblock->GetSymbolValue(id_order);
        }

        el_shading_model.textblock_array.push_back(el_textblock);
    }

    {
        bool useDeleteVertex = false;
        for (auto interleave = symbolDB.interleaves.cbegin(); interleave != symbolDB.interleaves.cend(); ++interleave)
        {
            if (interleave->GetSymbolValue(id_interleave).empty())
            {
                if (interleave->GetSymbolValue(id_binarize) == "false")
                {
                    useDeleteVertex = true;
                }
                continue;
            }

            nw::g3d::tool::g3dif::elem_interleave el_interleave;
            el_interleave.id_set = interleave->GetSymbolValue(id_interleave);

            if (interleave->GetSymbolValue(id_binarize).empty())
            {
                el_interleave.binarize.value = "true";
            }
            else
            {
                el_interleave.binarize = interleave->GetSymbolValue(id_binarize);
            }

            el_shading_model.interleave_array.push_back(el_interleave);
        }

        nw::g3d::tool::g3dif::elem_interleave el_empty_interleave;
        el_empty_interleave.id_set.value = "";
        el_empty_interleave.binarize.value = useDeleteVertex ? "false" : "true";
        el_shading_model.interleave_array.push_back(el_empty_interleave);
    }

    for (auto group = symbolDB.groups.cbegin(); group != symbolDB.groups.cend(); ++group)
    {
        nw::g3d::tool::g3dif::elem_group el_group;
        el_group.name = group->GetSymbolValue(id_group_id);
        el_group.condition = nw::g3d::tool::util::EscapeString(group->GetSymbolValue(id_condition));

        if (!group->GetSymbolValue(id_label).empty())
        {
            el_group.ui_label.Validate();
            el_group.ui_label->value = group->GetSymbolValue(id_label);
        }

        if (!group->GetSymbolValue(id_comment).empty())
        {
            el_group.ui_comment.Validate();
            el_group.ui_comment->value = group->GetSymbolValue(id_comment);
        }

        if (!group->GetSymbolValue(id_order).empty())
        {
            el_group.ui_order.Validate();
            el_group.ui_order->value = group->GetSymbolValue(id_order);
        }

        if (!group->GetSymbolValue(id_group).empty())
        {
            el_group.ui_group.Validate();
            el_group.ui_group->group_name = group->GetSymbolValue(id_group);
        }

        el_group.page_name = group->GetSymbolValue(id_page);

        el_shading_model.group_array.push_back(el_group);
    }

    for (auto page = symbolDB.pages.cbegin(); page != symbolDB.pages.cend(); ++page)
    {
        nw::g3d::tool::g3dif::elem_page el_page;
        el_page.name = page->GetSymbolValue(id_page);

        if (!page->GetSymbolValue(id_label).empty())
        {
            el_page.ui_label.Validate();
            el_page.ui_label->value = page->GetSymbolValue(id_label);
        }

        if (!page->GetSymbolValue(id_comment).empty())
        {
            el_page.ui_comment.Validate();
            el_page.ui_comment->value = page->GetSymbolValue(id_comment);
        }

        if (!page->GetSymbolValue(id_order).empty())
        {
            el_page.ui_order.Validate();
            el_page.ui_order->value = page->GetSymbolValue(id_order);
        }

        el_shading_model.page_array.push_back(el_page);
    }

    if (symbolDB.streamouts.size() > 1)
    {
        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_STREAMOUT, "Invalid streamout annotation count.");
    }
    else if (symbolDB.streamouts.size() == 1)
    {
        el_shading_model.streamout.Validate();
        el_shading_model.streamout->varying = symbolDB.streamouts[0].GetSymbolValue(id_streamout);
    }

    if (symbolDB.revisions.size() > 1)
    {
        THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_REVISION, "Invalid revision annotation count.");
    }
    else if (symbolDB.revisions.size() == 1)
    {
        int value = 0;
        if (!nw::g3d::tool::util::TryParse(value, symbolDB.revisions[0].GetSymbolValue(id_revision)))
        {
            THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_REVISION,
                "%hs, line %hs: Invalid revision annotation value: %hs.",
                symbolDB.revisions[0].GetSymbolValue(id_file_info).c_str(), symbolDB.revisions[0].GetSymbolValue(id_line_info).c_str(), symbolDB.revisions[0].GetSymbolValue(id_revision).c_str());
        }
        el_shading_model.revision = value;
    }
}

void Extractor::WriteLog(std::ostream& stream) const
{
    auto& annot = m_pState->annot;
    annot.seekg(0, std::ios::beg);
    stream << annot.rdbuf();
}

} // namespace g3dTool
} // namespace nn
