﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/ShaderDefinition.h>
#include <ShdrAnnotation.h>
#include <util/UtilSerialization.h>
#include <util/UtilXMLParser.h>

namespace nn { namespace g3dTool {

using nw::g3d::tool::util::XMLElement;

//--------------------------------------------------------------------------------------------------

void AnnotationBase::MergeAttr(const XMLElement* pElem)
{
    if (const char* pAttribName = GetInvalidXmlAnnotationName(m_IdArray, m_IdCount, pElem))
    {
        THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION, "Identifier_InvalidAnnotationAttribute",
            pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), pAttribName, pElem->AttributeId(0));
    }

    for (int index = 0; index < m_IdCount; ++index)
    {
        const StringRef& id = m_IdArray[index];
        if (id == id_name, id == id_id || id == id_var || id == id_file_info || id == id_line_info)
        {
            continue;
        }
        if (const char* attributeValue = pElem->Attribute(id))
        {
            if (id == id_group)
            {
                if (!m_ValueArray[index].Empty())
                {
                    std::string idValue;
                    for (int findIndex = 0; findIndex < m_IdCount; ++findIndex)
                    {
                        const StringRef& findId = m_IdArray[findIndex];
                        if (findId == id_id || findId == id_renderinfo)
                        {
                            idValue = m_ValueArray[findIndex].Str();
                            break;
                        }
                    }
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION, "Identifier_InvalidAnnotationDuplicatedGroup",
                        pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), idValue.c_str(), attributeValue, m_ValueArray[index].Str().c_str());
                }
            }
            m_ValueArray[index] = attributeValue;
        }
    }
}

const char* AnnotationBase::GetInvalidXmlAnnotationName(const StringRef* pIdArray, int idCount, const XMLElement* pElem)
{
    int idxAnnotation = 0;
    while (const char* pAnnotationName = pElem->AttributeId(idxAnnotation))
    {
        if (idxAnnotation == 0 )
        {
            // 遠隔操作の場合の id アノテーションは無視する
            if (pAnnotationName == id_option_id || pAnnotationName == id_attrib_id ||
                pAnnotationName == id_sampler_id || pAnnotationName == id_uniform_id ||
                pAnnotationName == id_block_id)
            {
                ++idxAnnotation;
                continue;
            }
        }
        bool isFound = false;
        for (int i = 0; i < idCount; ++i)
        {
            if (pIdArray[i] == pAnnotationName)
            {
                isFound = true;
                break;
            }
        }

        // サンプラーの場合、テクスチャ合成アノテーションが書かれていても無視する
        if (!isFound)
        {
            bool isSampler = true;
            for (int idIndex = 0; idIndex < idCount; ++idIndex)
            {
                if (pIdArray[idIndex] != g_AnnotationSamplerIdList[idIndex])
                {
                    isSampler = false;
                    break;
                }
            }
            if (isSampler)
            {
                static const StringRef annotaionMergeTextureIdList[] =
                {
                    "comp_sel",
                    "source_r",
                    "source_g",
                    "source_b",
                    "source_a",
                    "format",
                    "linear",
                };
                for (int i = 0; i < sizeof(annotaionMergeTextureIdList) / sizeof(const StringRef); ++i)
                {
                    if (annotaionMergeTextureIdList[i] == pAnnotationName)
                    {
                        isFound = true;
                        break;
                    }
                }
            }
        }

        if (isFound)
        {
            ++idxAnnotation;
            continue;
        }
        return pAnnotationName;
    }
    return nullptr;
}

// AnnotBlock の例外処理
void AnnotBlock::MergeAttr(const XMLElement* pElem)
{
    if (const char* pAttribName = GetInvalidXmlAnnotationName(pElem))
    {
        THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION, "Identifier_InvalidAnnotationAttribute",
            pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), pAttribName, pElem->AttributeId(0));
    }

    if (const char* attr = pElem->Attribute(id_type))
    {
        SetAttributeValue(id_type, attr);
    }
}

// AnnotBlock の例外処理
const char* AnnotBlock::GetInvalidXmlAnnotationName(const XMLElement* pElem)
{
    int idxAnnotation = 1;
    if (pElem->Child(id_block_info))
    {
        while (const char* pAnnotationName = pElem->Child(id_block_info)->AttributeId(idxAnnotation))
        {
            if (!((id_id == pAnnotationName) || (id_type == pAnnotationName) ||
                (id_file_info == pAnnotationName) || (id_line_info == pAnnotationName)))
            {
                return pAnnotationName;
            }
            ++idxAnnotation;
        }
    }
    return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationBlockIdList, sizeof(g_AnnotationBlockIdList) / sizeof(StringRef), pElem);
}

// AnnotTextBlock の例外処理
void AnnotTextBlock::MergeAttr(const XMLElement* pElem)
{
    if (const char* pAttribName = GetInvalidXmlAnnotationName(pElem))
    {
        THROW_TRANSLATED_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION, "Identifier_InvalidAnnotationAttribute",
            pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), pAttribName, pElem->AttributeId(0));
    }
    // textblock は text のみをマージする。複数箇所も同じ id で別の text が定義されていたらエラーとする。
    if (const char* attr = pElem->Attribute(id_text))
    {
        if (!GetAttributeValue(id_text).Empty())
        {
            THROW_ERROR(ERRCODE_SHADER_ANNOTATION_INVALID_ANNOTATION,
                "%hs, line %hs: %hs text is overwrited. '%hs' and '%hs'.",
                pElem->Attribute(id_file_info), pElem->Attribute(id_line_info), GetAttributeValue(id_id).Str().c_str(), attr, GetAttributeValue(id_text).Str().c_str());
        }
        SetAttributeValue(id_text, attr);
    }
}

//--------------------------------------------------------------------------------------------------

bool SymbolBase::CompareAttr(const AnnotationBase& rhs) const
{
    // 自身の Symbol の内容と AnnotaionBase の内容を比較する
    for (int index = 0; index < m_IdCount; ++index)
    {
        const StringRef& id = m_IdArray[index];
        if (id == id_name || id == id_var || id == id_file_info || id == id_line_info)
        {
            continue;
        }
        if (GetSymbolValue(id) != rhs.GetAttributeValue(id).Str())
        {
            return false;
        }
    }
    return true;
}

void SymbolBase::Override(const AnnotationBase& annot)
{
    // 自身の symbol の内容を AnnotaionBase の内容で上書きする
    for (int index = 0; index < m_IdCount; ++index)
    {
        const StringRef& id = m_IdArray[index];
        if (id == id_name || id == id_var || id == id_file_info || id == id_line_info)
        {
            continue;
        }
        if (!annot.GetAttributeValue(id).Empty())
        {
            m_ValueArray[index] = annot.GetAttributeValue(id).Str();
        }
    }
}

// SymbolBlock の例外処理
bool SymbolBlock::CompareAttr(const AnnotBlock& rhs) const
{
    if (uniforms.size() != rhs.uniforms.size())
    {
        return false;
    }
    int uniform_index = 0;
    for (auto uniform = uniforms.cbegin(); uniform != uniforms.cend(); ++uniform, ++uniform_index)
    {
        if (!uniform->CompareAttr(rhs.uniforms[uniform_index]))
        {
            return false;
        }
    }

    return SymbolBase::CompareAttr(rhs);
}

// SymbolBlock の例外処理
void SymbolBlock::Override(const AnnotBlock& annot)
{
    int uniform_index = 0;
    for (auto uniform = uniforms.begin(); uniform != uniforms.end(); ++uniform, ++uniform_index)
    {
        uniform->Override(annot.uniforms[uniform_index]);
    }

    SymbolBase::Override(annot);
}

} // namespace g3dTool
} // namespace nn
