﻿using Microsoft.CodeAnalysis.CSharp.Scripting;
using Microsoft.CodeAnalysis.Scripting;
using Nintendo.G3dTool.Entities;
using Nintendo.ToolFoundation.CommandLine;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;

namespace TextureCompositor
{
    public class ScriptPreprocessGlobals
    {
        public PreprocessMaterialArg PreprocessMaterialArg { get; internal set; }
        public ConvertTextureArg ConvertTextureArg { get; internal set; }
    }

    public static class Program
    {
        internal static CommandLineOptions CommandLineOptions { get; set; }

        public static void Main(string[] args)
        {
#if DEBUG != true
            try
#endif
            {
                CommandLineOptions = new CommandLineOptions(args);
                Logger.Instance.IsSilent = CommandLineOptions.IsSilent;
                GlslAnalyzationUtility.IsSilent = CommandLineOptions.IsSilent;
                string xsdBasePath = G3dToolUtility.GetXsdBasePath();

                // シェーダ定義ファイルの読み込み
                List<IntermediateFile> shaderDefFiles = new List<IntermediateFile>();
                foreach (string fsdPath in CommandLineOptions.InputShaderDefinitionPaths)
                {
                    var shaderDefFile = IfReadUtility.ReadIntermediateFile(fsdPath, null);
                    shaderDefFiles.Add(shaderDefFile);
                }

                // マテリアル中間ファイルの処理
                foreach (string fmtPath in CommandLineOptions.InputMaterialPaths)
                {
                    var matFile = IfReadUtility.ReadIntermediateFile(fmtPath, xsdBasePath);
                    var mat = matFile.GetRootEntity<Material>();
                    if (!mat.IsShaderAssigned)
                    {
                        throw new Exception($"Shader is not assigned for material \"{mat.Name}\"");
                    }

                    CompositeSamplers(
                        mat,
                        shaderDefFiles,
                        CommandLineOptions.OutputFolder,
                        matFile.Path,
                        PreprocessMaterial,
                        ConvertTexture);

                    string outputFilePath = System.IO.Path.Combine(
                        CommandLineOptions.OutputFolder,
                        System.IO.Path.GetFileName(matFile.Path));

                    Logger.Instance.WriteDebugMessageLine($"出力ファイル: {outputFilePath}");
                    IfWriteUtility.WriteIntermediateFile(matFile, outputFilePath, xsdBasePath);
                }

                // モデル中間ファイルの処理
                foreach (var fmdPath in CommandLineOptions.InputModelPaths)
                {
                    var modelFile = IfReadUtility.ReadIntermediateFile(fmdPath, xsdBasePath);
                    CompositeSamplers(
                        modelFile,
                        shaderDefFiles,
                        CommandLineOptions.OutputFolder,
                        PreprocessMaterial,
                        ConvertTexture);
                }
            }
#if DEBUG != true
            catch (Exception exception)
            {
                Logger.Instance.WriteErrorMessageLine($"{exception.Message}\n{exception.StackTrace}");
                var innerException = exception.InnerException;
                while (innerException != null)
                {
                    Logger.Instance.WriteErrorMessageLine($"{innerException.Message}\n{innerException.StackTrace}");
                    innerException = innerException.InnerException;
                    Environment.ExitCode = 1;
                }
            }
#endif
        }

        public delegate void TextureConversionDelegate(TextureConverterOptions args, Material material);
        public delegate void MaterialEditingDelegate(MaterialEditingInfo info);

        public class MaterialEditingInfo
        {
            public Material Material { get; internal set; }
            public ShaderDefinition ShaderDefinition { get; internal set; }
        }

        private static void CompositeSamplers(
            IntermediateFile modelFile,
            IEnumerable<IntermediateFile> shaderDefFiles,
            string outputFolder,
            MaterialEditingDelegate materialEditing,
            TextureConversionDelegate convertTexture)
        {
            var model = modelFile.GetRootEntity<Model>();
            foreach (var material in model.Materials)
            {
                CompositeSamplers(
                    material,
                    shaderDefFiles,
                    outputFolder,
                    modelFile.Path,
                    materialEditing,
                    convertTexture);
            }

            string outputFilePath = System.IO.Path.Combine(
                outputFolder,
                System.IO.Path.GetFileName(modelFile.Path));

            Logger.Instance.WriteDebugMessageLine($"出力ファイル: {outputFilePath}");
            IfWriteUtility.WriteIntermediateFile(modelFile, outputFilePath, G3dToolUtility.GetXsdBasePath());
        }

        private static void CompositeSamplers(
            Material material,
            IEnumerable<IntermediateFile> shaderDefFiles,
            string outputFolder,
            string inputFilePath,
            MaterialEditingDelegate materialEditing,
            TextureConversionDelegate convertTexture)
        {
            if (!material.IsShaderAssigned)
            {
                throw new Exception($"Shader is not assigned for material \"{material.Name}\"");
            }

            var shaderDefFile = shaderDefFiles.FirstOrDefault(x => x.Name == material.ShaderAssign.ShaderArchive);
            if (shaderDefFile == null)
            {
                throw new Exception($"ShaderDefinition \"{material.ShaderAssign.ShaderArchive}\" was not found in given shader definitions");
            }

            var shaderDef = shaderDefFile.GetRootEntity<ShaderDefinition>();

            // C# スクリプトで外部からマテリアルを置き換える(合成パターンを決めるためのオプション書き換え処理)
            if (materialEditing != null)
            {
                MaterialEditingInfo info = new MaterialEditingInfo();
                info.Material = material;
                info.ShaderDefinition = shaderDef;
                materialEditing(info);
            }

            {
                List<string> texRefPaths = new List<string>();
                texRefPaths.Add(System.IO.Path.Combine(
                        System.IO.Path.GetDirectoryName(inputFilePath), "textures"));
                texRefPaths.Add(System.IO.Path.Combine(outputFolder, "textures"));

                CompositeSamplerInfo info = GlslAnalyzationUtility.ExtractCompositeSamplerInfo(shaderDef, material);
                if (info != null)
                {
                    // 合成情報が存在するので、サンプラーやテクスチャーの合成を行う
                    var compositor = new TextureSamplerCompositor();
                    compositor.OutputFolderPath = outputFolder;
                    compositor.TextureReferencePaths.AddRange(texRefPaths);

                    compositor.CompositeMaterialSamplers(
                        material,
                        info,
                        shaderDefFile.GetRootEntity<ShaderDefinition>());
                }

                // 各サンプラーのテクスチャーをリサイズ、圧縮
                foreach (var sampler in material.Samplers)
                {
                    string texPath = TextureUtility.FindTexturePath(sampler.TexName, texRefPaths);
                    TextureConverterOptions args = new TextureConverterOptions();
                    args.Input = texPath;
                    args.Output = Path.Combine(outputFolder, "textures", $"{sampler.TexName}.ftxb");
                    if (info != null)
                    {
                        var samplerInfo = info.CompositeSamplers.FirstOrDefault(x => x.SamplerName == sampler.Name);
                        if (samplerInfo != null)
                        {
                            args.CompSel = samplerInfo.CompSel;
                            args.Format = samplerInfo.Format;
                            args.Linear = samplerInfo.Linear;
                        }
                    }

                    if (convertTexture != null)
                    {
                        convertTexture(args, material);
                    }
                }
            }
        }

        public static void ConvertTexture(TextureConverterOptions args, Material material)
        {
            if (string.IsNullOrEmpty(CommandLineOptions.TextureConvertScriptPath))
            {
                // スクリプトがない場合はデフォルトの設定でコンバート
                TextureUtility.ConvertTexture(args.ToString());
            }
            else
            {
                string scriptPath = CommandLineOptions.TextureConvertScriptPath;
                string scriptCode = File.ReadAllText(scriptPath);

                // TODO: シンボル名をちゃんと解析して判定する
                string ConvertTextureSymbolName = "ConvertTexture";
                if (scriptCode.RemoveCommentOut().Contains(ConvertTextureSymbolName))
                {
                    scriptCode += $"\n{ConvertTextureSymbolName}({ConvertTextureSymbolName}Arg);";
                    ScriptPreprocessGlobals globals = new ScriptPreprocessGlobals();
                    globals.ConvertTextureArg = new ConvertTextureArg();
                    globals.ConvertTextureArg.TextureConverterOptions = args;
                    globals.ConvertTextureArg.Material = material;
                    globals.ConvertTextureArg.TextureConverterPath = TextureUtility.Get3dTextureConverterPath();
                    ExecuteCsharpScript(scriptCode, scriptPath, globals);
                }
            }
        }

        public static void PreprocessMaterial(MaterialEditingInfo info)
        {
            if (string.IsNullOrEmpty(CommandLineOptions.MaterialEditingScriptPath))
            {
                // スクリプトの指定がない場合は何もしない
                return;
            }
            else
            {
                string scriptPath = CommandLineOptions.MaterialEditingScriptPath;
                string scriptCode = File.ReadAllText(scriptPath);

                // TODO: シンボル名をちゃんと解析して判定する
                string MaterialPreprocessSymbolName = "PreprocessMaterial";
                if (scriptCode.RemoveCommentOut().Contains(MaterialPreprocessSymbolName))
                {
                    scriptCode += $"\n{MaterialPreprocessSymbolName}({MaterialPreprocessSymbolName}Arg);";
                    ScriptPreprocessGlobals globals = new ScriptPreprocessGlobals();
                    globals.PreprocessMaterialArg = new PreprocessMaterialArg();
                    globals.PreprocessMaterialArg.Material = info.Material;
                    globals.PreprocessMaterialArg.ShaderDefinition = info.ShaderDefinition;
                    ExecuteCsharpScript(scriptCode, scriptPath, globals);
                }
            }
        }

        private static string RemoveCommentOut(this string source)
        {
            StringBuilder result = new StringBuilder();
            foreach (string line in source.Split(new string[] { Environment.NewLine }, StringSplitOptions.None))
            {
                int commentStartIndex = line.IndexOf("//");
                if (commentStartIndex < 0)
                {
                    result.AppendLine(line);
                }
                else if (commentStartIndex == 0)
                {
                    result.AppendLine(line.Substring(0, commentStartIndex));
                }
                else
                {
                    result.AppendLine();
                }
            }

            return result.ToString();
        }

        private static void ExecuteCsharpScript(string scriptPath, object scriptGlobalObj)
        {
            string scriptCode = File.ReadAllText(scriptPath);
            ExecuteCsharpScript(scriptCode, scriptPath, scriptGlobalObj);
        }

        private static void ExecuteCsharpScript(string scriptCode, string scriptPath, object scriptGlobalObj)
        {
            string scriptFolder = Path.GetDirectoryName(scriptPath);
            var sourceResolver = ScriptSourceResolver.Default
                .WithBaseDirectory(scriptFolder);
            var metadataResolver = ScriptMetadataResolver.Default
                .WithBaseDirectory(scriptFolder);
            var scriptOptions = ScriptOptions.Default
                .WithSourceResolver(sourceResolver)
                .WithMetadataResolver(metadataResolver)
                .WithReferences(Utility.EnumerateAssemblies());

            var script = CSharpScript.Create(scriptCode, options: scriptOptions, globalsType: scriptGlobalObj.GetType());
            try
            {
                script.RunAsync(scriptGlobalObj).Wait();
            }
            catch (Exception exception)
            {
                StringBuilder errorMessage = new StringBuilder();
                Exception currentException = exception;
                do
                {
                    errorMessage.AppendLine($"{currentException.Message}");
                    errorMessage.AppendLine($"{currentException.StackTrace}");
                    errorMessage.AppendLine();
                    currentException = currentException.InnerException;
                } while (currentException != null);

                throw new Exception($"{CommandLineOptions.MaterialEditingScriptPath}の実行に失敗しました。\n{errorMessage.ToString()}");
            }
        }
    }
}
