﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace nw.g3d.nw4f_3dif
{
    /// <summary>
    /// 配列ヒントの更新
    /// </summary>
    public static class G3dArrayHintUpdater
    {
        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="model">更新するモデル</param>
        public static void Update(modelType model)
        {
            UpdateRootElement(model);

            if (model.material_array != null)
            {
                model.material_array.UpdateHint();
                foreach (materialType material in model.material_array.Enumerate())
                {
                    Update(material);
                }
            }

            if (model.skeleton.bone_array != null)
            {
                model.skeleton.bone_array.UpdateHint();
                foreach (boneType bone in model.skeleton.bone_array.Enumerate())
                {
                    Update<user_dataType>(bone.user_data_array);
                }
            }

            if (model.vertex_array != null)
            {
                model.vertex_array.UpdateHint();
                foreach (vertexType vertex in model.vertex_array.Enumerate())
                {
                    Update<vtx_attribType>(vertex.vtx_attrib_array);
                    if (vertex.vtx_buffer_array != null)
                    {
                        vertex.vtx_buffer_array.UpdateHint();
                        foreach (vtx_bufferType vtx_buffer in vertex.vtx_buffer_array.Enumerate())
                        {
                            Update<inputType>(vtx_buffer.input_array);
                        }
                    }
                }
            }

            if (model.shape_array != null)
            {
                model.shape_array.UpdateHint();
                foreach (shapeType shape in model.shape_array.Enumerate())
                {
                    shape.mesh_array.UpdateHint();
                    foreach (meshType mesh in shape.mesh_array.Enumerate())
                    {
                        Update<submeshType>(mesh.submesh_array);
                    }
                    if (shape.key_shape_array != null)
                    {
                        shape.key_shape_array.UpdateHint();
                        foreach (key_shapeType key_shape in shape.key_shape_array.Enumerate())
                        {
                            Update<target_attribType>(key_shape.target_attrib_array);
                        }
                    }
                }
            }

            if (model.original_material_array != null)
            {
                model.original_material_array.UpdateHint();
                foreach (original_materialType original_material in
                    model.original_material_array.Enumerate())
                {
                    Update<original_colorType>(original_material.original_color_array);
                    Update<original_texsrtType>(original_material.original_texsrt_array);
                }
            }
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="material">更新するマテリアル</param>
        public static void Update(materialType material)
        {
            Update<samplerType>(material.sampler_array);
            shader_assignType shader_assign = material.shader_assign;
            if (shader_assign != null)
            {
                Update<render_infoType>(shader_assign.render_info_array);
                Update<shader_optionType>(shader_assign.shader_option_array);
                Update<sampler_assignType>(shader_assign.sampler_assign_array);
                Update<shader_paramType>(shader_assign.shader_param_array);
                Update<attrib_assignType>(shader_assign.attrib_assign_array);
            }
            Update<user_dataType>(material.user_data_array);
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="texture">更新するテクスチャ</param>
        public static void Update(textureType texture)
        {
            UpdateRootElement(texture);
            Update<original_imageType>(texture.original_image_array);
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="skeletal_anim">更新するスケルタルアニメーション</param>
        public static void Update(skeletal_animType skeletal_anim)
        {
            UpdateRootElement(skeletal_anim);
            Update<bone_animType>(skeletal_anim.bone_anim_array);
            if (skeletal_anim.bone_anim_array != null)
            {
                foreach (var boneAnim in skeletal_anim.bone_anim_array.Enumerate())
                {
                    Update<user_dataType>(boneAnim.user_data_array);
                }
            }
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="material_anim">更新するマテリアルアニメーション</param>
        public static void Update(material_animType material_anim)
        {
            UpdateRootElement(material_anim);

            Update<tex_patternType>(material_anim.tex_pattern_array);

            if (material_anim.per_material_anim_array != null)
            {
                material_anim.per_material_anim_array.UpdateHint();

                foreach (var per_material_anim in material_anim.per_material_anim_array.Enumerate())
                {
                    Update<param_animType>(per_material_anim.shader_param_anim_array);
                    Update<pattern_anim_targetType>(per_material_anim.tex_pattern_anim_array);

                    // ビジビリティアニメーション部分は G3dArray ではないので処理は不要
                }
            }

            if (material_anim.original_per_material_anim_array != null)
            {
                material_anim.original_per_material_anim_array.UpdateHint();
                foreach (var original_per_material_anim in material_anim.original_per_material_anim_array.Enumerate())
                {
                    Update<original_color_animType>(
                        original_per_material_anim.original_color_anim_array);
                    Update<original_texsrt_animType>(
                        original_per_material_anim.original_texsrt_anim_array);
                }
            }
        }
        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="shader_param_anim">更新するシェーダーパラメーターアニメーション</param>
        public static void Update(shader_param_animType shader_param_anim)
        {
            UpdateRootElement(shader_param_anim);

            if (shader_param_anim.shader_param_mat_anim_array != null)
            {
                shader_param_anim.shader_param_mat_anim_array.UpdateHint();
                foreach (shader_param_mat_animType shader_param_mat_anim in
                    shader_param_anim.shader_param_mat_anim_array.Enumerate())
                {
                    Update<param_animType>(shader_param_mat_anim.param_anim_array);
                }
            }

            if (shader_param_anim.original_material_anim_array != null)
            {
                shader_param_anim.original_material_anim_array.UpdateHint();
                foreach (original_material_animType original_material_anim in
                    shader_param_anim.original_material_anim_array.Enumerate())
                {
                    Update<original_color_animType>(
                        original_material_anim.original_color_anim_array);
                    Update<original_texsrt_animType>(
                        original_material_anim.original_texsrt_anim_array);
                }
            }
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="tex_pattern_anim">更新するテクスチャパターンアニメーション</param>
        public static void Update(tex_pattern_animType tex_pattern_anim)
        {
            UpdateRootElement(tex_pattern_anim);
            Update<tex_patternType>(tex_pattern_anim.tex_pattern_array);
            Update<tex_pattern_mat_animType>(tex_pattern_anim.tex_pattern_mat_anim_array);
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="bone_visibility_anim">更新するボーンビジビリティアニメーション</param>
        public static void Update(bone_visibility_animType bone_visibility_anim)
        {
            UpdateRootElement(bone_visibility_anim);
            Update<bone_vis_bone_animType>(bone_visibility_anim.bone_vis_bone_anim_array);
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="mat_visibility_anim">更新するマテリアルビジビリティアニメーション</param>
        public static void Update(mat_visibility_animType mat_visibility_anim)
        {
            UpdateRootElement(mat_visibility_anim);
            Update<mat_vis_mat_animType>(mat_visibility_anim.mat_vis_mat_anim_array);
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="shape_anim">更新するシェイプアニメーション</param>
        public static void Update(shape_animType shape_anim)
        {
            UpdateRootElement(shape_anim);
            Update<vertex_shape_animType>(shape_anim.vertex_shape_anim_array);
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="scene_anim">更新するシーンアニメーション</param>
        public static void Update(scene_animType scene_anim)
        {
            UpdateRootElement(scene_anim);

            Update<camera_animType>(scene_anim.camera_anim_array);
            Update<light_animType>(scene_anim.light_anim_array);
            Update<fog_animType>(scene_anim.fog_anim_array);

            //if (scene_anim.camera_anim_array != null)
            //{
            //    scene_anim.camera_anim_array.UpdateHint();
            //    foreach (camera_animType camera_anim in
            //        scene_anim.camera_anim_array.Enumerate())
            //    {
            //        Update<user_dataType>(camera_anim.user_data_array);
            //    }
            //}

            //if (scene_anim.light_anim_array != null)
            //{
            //    scene_anim.light_anim_array.UpdateHint();
            //    foreach (light_animType light_anim in
            //        scene_anim.light_anim_array.Enumerate())
            //    {
            //        Update<user_dataType>(light_anim.user_data_array);
            //    }
            //}

            //if (scene_anim.fog_anim_array != null)
            //{
            //    scene_anim.fog_anim_array.UpdateHint();
            //    foreach (fog_animType fog_anim in
            //        scene_anim.fog_anim_array.Enumerate())
            //    {
            //        Update<user_dataType>(fog_anim.user_data_array);
            //    }
            //}
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="shader_config">更新するシェーダー設定</param>
        public static void Update(shader_configType shader_config)
        {
            UpdateRootElement(shader_config);
            Update<include_pathType>(shader_config.include_path_array);
            Update<force_include_fileType>(shader_config.force_include_file_array);
            if (shader_config.shader_array != null)
            {
                shader_config.shader_array.UpdateHint();
                foreach (shaderType shader in shader_config.shader_array.Enumerate())
                {
                    Update<macroType>(shader.macro_array);
                    Update<variationType>(shader.variation_array);
                }
            }
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="shader_definition">更新するシェーダー定義</param>
        public static void Update(shader_definitionType shader_definition)
        {
            UpdateRootElement(shader_definition);
            Update<force_includeType>(shader_definition.force_include_array);
            if (shader_definition.shading_model_array != null)
            {
                shader_definition.shading_model_array.UpdateHint();
                foreach (shading_modelType shading_model in
                    shader_definition.shading_model_array.Enumerate())
                {
                    Update<macroType>(shading_model.macro_array);
                    Update<option_varType>(shading_model.option_var_array);
                    Update<attrib_varType>(shading_model.attrib_var_array);
                    Update<sampler_varType>(shading_model.sampler_var_array);
                    if (shading_model.block_var_array != null)
                    {
                        shading_model.block_var_array.UpdateHint();
                        foreach (block_varType block_var in
                            shading_model.block_var_array.Enumerate())
                        {
                            Update<uniform_varType>(block_var.uniform_var_array);
                        }
                    }
                    Update<render_info_slotType>(shading_model.render_info_slot_array);
                    Update<interleaveType>(shading_model.interleave_array);
                    Update<groupType>(shading_model.group_array);
                    if (shading_model.shader_storage_block_var_array != null)
                    {
                        shading_model.shader_storage_block_var_array.UpdateHint();
                        foreach (shader_storage_block_varType bufferVar in
                            shading_model.shader_storage_block_var_array.Enumerate())
                        {
                            Update<buffer_varType>(bufferVar.buffer_var_array);
                        }
                    }
                }
            }
            Update<shader_srcType>(shader_definition.shader_src_array);
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="shader_variation">更新するシェーダー設定</param>
        public static void Update(shader_variationType shader_variation)
        {
            UpdateRootElement(shader_variation);
            if (shader_variation.target_shader_array != null)
            {
                shader_variation.target_shader_array.UpdateHint();

                foreach (target_shaderType target_shader in
                    shader_variation.target_shader_array.Enumerate())
                {
                    if (target_shader.shader_program_array != null)
                    {
                        target_shader.shader_program_array.UpdateHint();

                        foreach (shader_programType shader_program in
                            target_shader.shader_program_array.Enumerate())
                        {
                            Update<optionType>(shader_program.option_array);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 配列ヒントの更新
        /// </summary>
        /// <param name="combiner">更新するコンバイナーシェーダー</param>
        public static void Update(combiner_shaderType combiner)
        {
            UpdateRootElement(combiner);
            if (combiner.material != null)
            {
                Update(combiner.material);
            }

            Update<connectionType>(combiner.connection_array);
            Update<blockType>(combiner.block_array);
            Update<uniformType>(combiner.uniform_array);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// ルート要素の配列ヒント更新
        /// </summary>
        /// <param name="rootElement">ルート要素</param>
        private static void UpdateRootElement(IG3dRootElement rootElement)
        {
            Update<process_logType>(rootElement.process_log_array);

            Update<user_dataType>(rootElement.user_data_array);

            Update<streamType>(rootElement.stream_array);
        }

        /// <summary>
        /// 配列ヒント更新
        /// </summary>
        /// <typeparam name="TType">配列の型</typeparam>
        /// <param name="arrayElement">配列要素</param>
        private static void Update<TType>(G3dArrayElement<TType> arrayElement)
            where TType : IG3dIndexHintElement
        {
            if (arrayElement == null) { return; }
            arrayElement.UpdateHint();
        }
    }
}
