﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    /// <summary>
    /// WingedEdge データ構造
    /// </summary>
    public partial class WingedEdgeMesh
    {
        /// <summary>
        /// 指定のエッジをベクトルで取得
        /// </summary>
        /// <param name="wem">WingedEdgeMeshデータ</param>
        /// <param name="e0">対象となるエッジ</param>
        /// <returns></returns>
        public static Vector3 GetEdgeVector3(WingedEdgeMesh wem, WingedEdge e0)
        {
            try
            {
                if (e0 == null)
                {
                    throw ExcepHandle.CreateException("GetEdgeVector:input winged edge is null.");
                }

                Mesh mesh = wem.mMesh;
                MeshSourceBase positions = mesh.GetSource(SourceType.Position, null);

                var v0 = new Vector3();
                v0.SetZero();
                var v1 = new Vector3();
                v1.SetZero();
                MeshSourceDblCtrler.GetAsVec3(positions, e0.Vertex0, ref v0);
                MeshSourceDblCtrler.GetAsVec3(positions, e0.Vertex1, ref v1);
                return v1 - v0;
            }
            catch
            {
                return null;
            }
        }

        public static Vector2 GetEdgeVector2(WingedEdgeMesh wem, WingedEdge e0)
        {
            try
            {
                if (e0 == null)
                {
                    throw ExcepHandle.CreateException("GetEdgeVector:input winged edge is null.");
                }

                Mesh mesh = wem.mMesh;
                MeshSourceBase positions = mesh.GetSource(e0.SpaceKey.type, e0.SpaceKey.name);

                var v0 = new Vector2();
                v0.SetZero();
                var v1 = new Vector2();
                v1.SetZero();
                MeshSourceDblCtrler.GetAsVec2(positions, e0.Vertex0, ref v0);
                MeshSourceDblCtrler.GetAsVec2(positions, e0.Vertex1, ref v1);
                return v1 - v0;
            }
            catch
            {
                return null;
            }
        }

        /// <summary>
        /// エッジ間の角度を0-180で取得
        /// </summary>
        /// <param name="wem">WingedEdgeMeshデータ</param>
        /// <param name="e0">対象となるエッジ</param>
        /// <param name="e1">対象となるエッジ</param>
        /// <returns>エッジ間の角度（0-180）</returns>
        public static double ComputeAngle(WingedEdgeMesh wem, WingedEdge e0, WingedEdge e1)
        {
            try
            {
                if (e0 == null || e1 == null)
                {
                    return double.MaxValue;
                }
                if (!e0.SpaceKey.Equals(e1.SpaceKey))
                {
                    return double.MaxValue;
                }

                if (e0.SpaceKey.type == SourceType.Position)
                {
                    Vector3 d0 = GetEdgeVector3(wem, e0);
                    Vector3 d1 = GetEdgeVector3(wem, e1);
                    // 何れかの頂点を起点に真逆を向くように調整
                    if (e0.Vertex0 == e1.Vertex1)
                    {
                        d1 = -d1;
                    }
                    if (e0.Vertex1 == e1.Vertex0)
                    {
                        d0 = -d0;
                    }
                    // 0-180 で角度を取得
                    return d0.GetAngle(d1,　false);
                }
                else if (e0.SpaceKey.type == SourceType.Texture)
                {
                    Vector2 d0 = GetEdgeVector2(wem, e0);
                    Vector2 d1 = GetEdgeVector2(wem, e1);
                    // 何れかの頂点を起点に真逆を向くように調整
                    if (e0.Vertex0 == e1.Vertex1)
                    {
                        d1 = -d1;
                    }
                    if (e0.Vertex1 == e1.Vertex0)
                    {
                        d0 = -d0;
                    }
                    // 0-180 で角度を取得
                    return d0.GetAngle(d1, false);
                }
                return 0.0;
            }
            catch
            {
                return double.NaN;
            }
        }

        /// <summary>
        /// エッジ間の角度を0-90で取得
        /// </summary>
        /// <param name="wem">WingedEdgeMeshデータ</param>
        /// <param name="e0">対象となるエッジ</param>
        /// <param name="e1">対象となるエッジ</param>
        /// <returns>エッジ間の角度（0-180）</returns>
        public static double Old_ComputeAngle(WingedEdgeMesh wem, WingedEdge e0, WingedEdge e1)
        {
            try
            {
                if (e0 == null || e1 == null)
                {
                    return double.MaxValue;
                }
                if (!e0.SpaceKey.Equals(e1.SpaceKey))
                {
                    return double.MaxValue;
                }

                if (e0.SpaceKey.type == SourceType.Position)
                {
                    Vector3 d0 = GetEdgeVector3(wem, e0);
                    Vector3 d1 = GetEdgeVector3(wem, e1);
                    return d0.GetAngle(d1);
                }
                else if (e0.SpaceKey.type == SourceType.Texture)
                {
                    Vector2 d0 = GetEdgeVector2(wem, e0);
                    Vector2 d1 = GetEdgeVector2(wem, e1);
                    return d0.GetAngle(d1, false);
                }
                return 0.0;
            }
            catch
            {
                return double.NaN;
            }
        }
    }
}
