﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    internal class VtxTriMeshPF
    {
        private Mesh mMesh = null;
        internal ResizableList<int> VAdjOffset { get; set; } = new ResizableList<int>();
        internal ResizableList<int> VAdjNum { get; set; } = new ResizableList<int>();
        internal ResizableList<int> VAdjFBuff { get; set; } = new ResizableList<int>();
        internal ResizableList<int> VAdjVBuff { get; set; } = new ResizableList<int>();

        /// <summary>
        /// 頂点の隣接フェースリスト、隣接頂点リストを構成する
        /// </summary>
        /// <param name="mesh">メッシュ</param>
        internal void Create(Mesh mesh)
        {
            try
            {
                Destroy();
                mMesh = mesh;

                int faceN = mMesh.FaceN;
                int vtxN = mMesh.VertexN;
                int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;

                VAdjOffset.Resize(vtxN, 0);
                VAdjNum.Resize(vtxN, 0);

                // 初期カウントアップ
                for (int f = 0; f < faceN; f++)
                {
                    int vnum = mMesh.mVnums[f];
                    int vofs = mMesh.mVoffs[f];

                    for (int v = 0; v < vnum; v++)
                    {
                        int vid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                        VAdjNum[vid]++;
                    }
                }

                // オフセット設定
                int total = 0;
                for (int v = 0; v < vtxN; v++)
                {
                    VAdjOffset[v] = total;
                    total += VAdjNum[v];
                }

                // 実際にバッファを設定する
                VAdjFBuff.Resize(total, int.MaxValue);
                VAdjVBuff.Resize(total, int.MaxValue);
                for (int v = 0; v < vtxN; v++)
                {
                    VAdjNum[v] = 0;
                }

                for (int f = 0; f < faceN; f++)
                {
                    int vnum = mMesh.mVnums[f];
                    int vofs = mMesh.mVoffs[f];

                    for (int v = 0; v < vnum; v++)
                    {
                        int v0 = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                        int v1 = mMesh.mVbuffs[(vofs + (v + 1) % 3) * sourceN + posOfs];
                        int tvofs = VAdjOffset[v0];
                        VAdjFBuff[tvofs + VAdjNum[v0]] = f;
                        VAdjVBuff[tvofs + VAdjNum[v0]] = v1;
                        VAdjNum[v0]++;
                    }
                }
            }
            catch (Exception ex)
            {
                Destroy();
                VAdjOffset = null;
                VAdjNum = null;
                VAdjFBuff = null;
                VAdjVBuff = null;
                throw ExcepHandle.CreateException("Construction error" + mesh.MeshName, ex);
            }
        }

        /// <summary>
        /// エッジを縮退させて、フェースを削除します。
        /// </summary>
        /// <param name="keepID">保持する頂点ID</param>
        /// <param name="killID">削除する頂点ID</param>
        /// <param name="newpos">新しい位置</param>
        /// <returns></returns>
        internal int ShrinkEdge(int keepID, int killID, VectorN newpos)
        {
            try
            {
                int result = 0;
                int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;
                int anm = VAdjNum[killID];
                int aof = VAdjOffset[killID];

                // とりあえず位置座標のみを更新する
                MeshSource<double> poslist = mMesh.GetSource(SourceType.Position, null) as MeshSource<double>;
                if (poslist == null)
                {
                    throw ExcepHandle.CreateException("The value type of position stream is not double/float.");
                }

                MeshSourceDblCtrler.SetAsVec3(poslist, keepID, newpos[0], newpos[1], newpos[2]);

                for (int a = 0; a < anm; a++)
                {
                    int afid = VAdjFBuff[aof + a];
                    int vnum = mMesh.mVnums[afid];
                    int vofs = mMesh.mVoffs[afid];

                    for (int v = 0; v < vnum; v++)
                    {
                        int pid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                        // 削除頂点IDに一致する場合
                        if (pid == killID)
                        {
                            mMesh.mVbuffs[(vofs + v) * sourceN + posOfs] = keepID;
                        }
                    }
                }

                // 縮退するフェースを洗い出して消す
                for (int a = 0; a < anm; a++)
                {
                    int afid = VAdjFBuff[aof + a];
                    int vnum = mMesh.mVnums[afid];
                    int vofs = mMesh.mVoffs[afid];

                    int p0 = mMesh.mVbuffs[(vofs + 0) * sourceN + posOfs];
                    int p1 = mMesh.mVbuffs[(vofs + 1) * sourceN + posOfs];
                    int p2 = mMesh.mVbuffs[(vofs + 2) * sourceN + posOfs];

                    // 縮退エッジがあったので、消す
                    if (p0 == p1 || p1 == p2 || p2 == p0)
                    {
                        mMesh.InvalidateFace(afid);
                    }
                }

                return result;
            }
            catch (Exception ex)
            {
                string message = string.Format("Fail to Shrink edge [{0},{1}].", keepID, killID);
                throw ExcepHandle.CreateException(message, ex);
            }
        }

        /// <summary>
        /// 開放
        /// </summary>
        internal void Destroy()
        {
            VAdjOffset.Clear();
            VAdjNum.Clear();
            VAdjFBuff.Clear();
            VAdjVBuff.Clear();
        }
    }
}
