﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace nw4f.tinymathlib
{
    public class Vector2
    {
        private double[] mData = new double[2] { 0, 0 };

        // prop X
        public double x
        {
            get { return mData[0]; }
            set { mData[0] = value; }
        }
        // prop Y
        public double y
        {
            get { return mData[1]; }
            set { mData[1] = value; }
        }

        /// <summary>
        /// インデクサ
        /// </summary>
        public double this[int index]
        {
            get { return mData[index]; }
            set { mData[index] = value; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Vector2(double ax, double ay)
        {
            mData[0] = ax;
            mData[1] = ay;
        }
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Vector2()
        {
            mData[0] = 0;
            mData[1] = 0;
        }

        public VectorN ToVectorN()
        {
            var ret = new VectorN(2);
            ret[0] = x;
            ret[1] = y;
            return ret;
        }

        /// <summary>
        /// Zeroで初期化
        /// </summary>
        public void Set(Vector2 rhs)
        {
            Array.Copy(rhs.mData, mData, 2);
        }
        /// <summary>
        /// ゼロ初期化
        /// </summary>
        public void SetZero()
        {
            x = 0;
            y = 0;
        }
        /// <summary>
        /// 外積を取得
        /// </summary>
        public double Cross(Vector2 rhs)
        {
            return x * rhs.y - y * rhs.x;
        }

        /// <summary>
        /// 内積を取得
        /// </summary>
        public double Dot(Vector2 rhs)
        {
            return x * rhs.x + y * rhs.y;
        }

        /// <summary>
        /// ノルムを計算する
        /// </summary>
        /// <returns>ノルム</returns>
        public double Norm()
        {
            return Math.Sqrt(Dot(this));
        }

        /// <summary>
        /// 正規化
        /// </summary>
        public void Normalize()
        {
            double norm = Norm();
            if (norm > 0)
            {
                mData[0] = mData[0] / norm;
                mData[1] = mData[1] / norm;
            }
        }

        /// <summary>
        /// 乗算
        /// </summary>
        /// <param name="lhs">掛けられるベクタ</param>
        /// <param name="rhs">掛ける数</param>
        /// <returns>乗算結果</returns>
        public static Vector2 operator *(Vector2 lhs, double rhs)
        {
            Vector2 result = new Vector2();
            result.x = lhs.x * rhs;
            result.y = lhs.y * rhs;
            return result;
        }

        /// <summary>
        /// 減算
        /// </summary>
        /// <param name="rhs">引く数</param>
        /// <returns>結果</returns>
        public static Vector2 operator -(Vector2 lhs, Vector2 rhs)
        {
            Vector2 result = new Vector2();
            for (int i = 0; i < 2; i++)
            {
                result.mData[i] = lhs.mData[i] - rhs.mData[i];
            }
            return result;
        }

        /// <summary>
        /// 加算
        /// </summary>
        /// <param name="rhs">引く数</param>
        /// <returns>結果</returns>
        public static Vector2 operator +(Vector2 lhs, Vector2 rhs)
        {
            Vector2 result = new Vector2();
            for (int i = 0; i < 2; i++)
            {
                result.mData[i] = lhs.mData[i] + rhs.mData[i];
            }
            return result;
        }

        /// <summary>
        /// 単項マイナス演算子
        /// </summary>
        public static Vector2 operator -(Vector2 lhs)
        {
            Vector2 result = new Vector2();
            result.x = -lhs.x;
            result.y = -lhs.y;
            return result;
        }

        /// <summary>
        /// ベクトル同士がなす角度
        /// </summary>
        /// <param name="d1"></param>
        /// <returns>度数で取得</returns>
        public double GetAngle(Vector2 d1, bool bContrast = true)
        {
            double dot = 0;
            if (bContrast)
            {
                dot = Math.Abs(Dot(d1));
            }
            else
            {
                dot = Dot(d1);
            }

            if (Norm() <= double.Epsilon || d1.Norm() <= double.Epsilon)
            {
                return 0.0f;
            }
            dot /= Norm();
            dot /= d1.Norm();

            if (dot >= 1.0)
            {
                dot = 1.0f - double.Epsilon;
            }

            dot = Math.Acos(dot);
            return dot * 180.0 / (Math.PI);
        }
    }
}
