﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace nw4f.tinymathlib
{
    public class IMatrix
    {
        /// <summary>
        /// 2x2行列の行列式
        /// </summary>
        public static double det2x2(double a, double b, double c, double d)
        {
            return a * d - b * c;
        }

        /// <summary>
        /// 3x3行列の行列式
        /// </summary>
        public static double det3x3(
                            double a1, double a2, double a3,
                            double b1, double b2, double b3,
                            double c1, double c2, double c3)
        {
            return a1 * det2x2(b2, b3, c2, c3) -
                    b1 * det2x2(a2, a3, c2, c3) +
                    c1 * det2x2(a2, a3, b2, b3);
        }
    }
    /// <summary>
    /// 44行列
    /// </summary>
    public class Matrix44
    {
        private double[,] mData = new double[4, 4];

        /// <summary>
        /// インデクサ
        /// </summary>
        public double this[int c, int r]
        {
            get { return mData[c, r]; }
            set { mData[c, r] = value; }
        }

        /// <summary>
        /// Zeroで初期化
        /// </summary>
        public void SetZero()
        {
            for (int c = 0; c < 4; c++)
            {
                for (int r = 0; r < 4; r++)
                {
                    mData[c, r] = 0.0f;
                }
            }
        }

        /// <summary>
        /// 単位行列を取得する
        /// </summary>
        public static Matrix44 Identity
        {
            get
            {
                Matrix44 result = new Matrix44();
                result.SetZero();
                result[0, 0] = 1.0f;
                result[1, 1] = 1.0f;
                result[2, 2] = 1.0f;
                result[3, 3] = 1.0f;
                return result;
            }
        }

        /// <summary>
        /// 行列の乗算
        /// </summary>
        /// <param name="lhs">行列（左側）</param>
        /// <param name="rhs">ベクタ（右側）</param>
        /// <returns>結果</returns>
        public static Matrix44 operator *(Matrix44 lhs, Matrix44 rhs)
        {
            Matrix44 result = new Matrix44();
            for (int i = 0; i < 4; i++)
            {
                for (int j = 0; j < 4; j++)
                {
                    result[i, j] = lhs[i, 0] * rhs[0, j] + lhs[i, 1] * rhs[1, j] + lhs[i, 2] * rhs[2, j] + lhs[i, 3] * rhs[3, j];
                }
            }
            return result;
        }

        /// <summary>
        /// 行列とベクタの乗算（左から掛ける）
        /// </summary>
        /// <param name="lhs">行列（左側）</param>
        /// <param name="rhs">ベクタ（右側）</param>
        /// <returns>結果</returns>
        public static Vector4 operator *(Matrix44 lhs, Vector4 rhs)
        {
            Vector4 result = new Vector4();
            for (int i = 0; i < 4; i++)
            {
                result[i] = lhs[i, 0] * rhs[0] + lhs[i, 1] * rhs[1] + lhs[i, 2] * rhs[2] + lhs[i, 3] * rhs[3];
            }
            return result;
        }

        /// <summary>
        /// 行列式を取得
        /// </summary>
        /// <returns></returns>
        private double GetDeterminant()
        {
            double a1, a2, a3, a4, b1, b2, b3, b4, c1, c2, c3, c4, d1, d2, d3, d4;

            a1 = mData[0, 0];
            b1 = mData[0, 1];
            c1 = mData[0, 2];
            d1 = mData[0, 3];

            a2 = mData[1, 0];
            b2 = mData[1, 1];
            c2 = mData[1, 2];
            d2 = mData[1, 3];

            a3 = mData[2, 0];
            b3 = mData[2, 1];
            c3 = mData[2, 2];
            d3 = mData[2, 3];

            a4 = mData[3, 0];
            b4 = mData[3, 1];
            c4 = mData[3, 2];
            d4 = mData[3, 3];

            return a1 * IMatrix.det3x3(b2, b3, b4, c2, c3, c4, d2, d3, d4) -
                    b1 * IMatrix.det3x3(a2, a3, a4, c2, c3, c4, d2, d3, d4) +
                    c1 * IMatrix.det3x3(a2, a3, a4, b2, b3, b4, d2, d3, d4) -
                    d1 * IMatrix.det3x3(a2, a3, a4, b2, b3, b4, c2, c3, c4);
        }

        /// <summary>
        /// 44行列の逆行列の公式の随伴行列の部分
        /// </summary>
        /// <returns>随伴行列</returns>
        private Matrix44 GetAdjoint()
        {
            Matrix44 result = new Matrix44();

            double a1, a2, a3, a4, b1, b2, b3, b4;
            double c1, c2, c3, c4, d1, d2, d3, d4;

            /* assign to individual variable names to aid  */
            /* selecting correct values  */

            a1 = mData[0, 0];
            b1 = mData[0, 1];
            c1 = mData[0, 2];
            d1 = mData[0, 3];

            a2 = mData[1, 0];
            b2 = mData[1, 1];
            c2 = mData[1, 2];
            d2 = mData[1, 3];

            a3 = mData[2, 0];
            b3 = mData[2, 1];
            c3 = mData[2, 2];
            d3 = mData[2, 3];

            a4 = mData[3, 0];
            b4 = mData[3, 1];
            c4 = mData[3, 2];
            d4 = mData[3, 3];

            result.mData[0, 0] = IMatrix.det3x3(b2, b3, b4, c2, c3, c4, d2, d3, d4);
            result.mData[1, 0] = -IMatrix.det3x3(a2, a3, a4, c2, c3, c4, d2, d3, d4);
            result.mData[2, 0] = IMatrix.det3x3(a2, a3, a4, b2, b3, b4, d2, d3, d4);
            result.mData[3, 0] = -IMatrix.det3x3(a2, a3, a4, b2, b3, b4, c2, c3, c4);

            result.mData[0, 1] = -IMatrix.det3x3(b1, b3, b4, c1, c3, c4, d1, d3, d4);
            result.mData[1, 1] = IMatrix.det3x3(a1, a3, a4, c1, c3, c4, d1, d3, d4);
            result.mData[2, 1] = -IMatrix.det3x3(a1, a3, a4, b1, b3, b4, d1, d3, d4);
            result.mData[3, 1] = IMatrix.det3x3(a1, a3, a4, b1, b3, b4, c1, c3, c4);

            result.mData[0, 2] = IMatrix.det3x3(b1, b2, b4, c1, c2, c4, d1, d2, d4);
            result.mData[1, 2] = -IMatrix.det3x3(a1, a2, a4, c1, c2, c4, d1, d2, d4);
            result.mData[2, 2] = IMatrix.det3x3(a1, a2, a4, b1, b2, b4, d1, d2, d4);
            result.mData[3, 2] = -IMatrix.det3x3(a1, a2, a4, b1, b2, b4, c1, c2, c4);

            result.mData[0, 3] = -IMatrix.det3x3(b1, b2, b3, c1, c2, c3, d1, d2, d3);
            result.mData[1, 3] = IMatrix.det3x3(a1, a2, a3, c1, c2, c3, d1, d2, d3);
            result.mData[2, 3] = -IMatrix.det3x3(a1, a2, a3, b1, b2, b3, d1, d2, d3);
            result.mData[3, 3] = IMatrix.det3x3(a1, a2, a3, b1, b2, b3, c1, c2, c3);

            return result;
        }

        /// <summary>
        /// 逆行列を求める
        /// </summary>
        /// <param name="success">行列式が小さくて、求まらない場合はfalseが格納される</param>
        /// <returns>逆行列</returns>
        public Matrix44 GetInverse(out bool success)
        {
            Matrix44 result = GetAdjoint();
            double det = GetDeterminant();

            if (Math.Abs(det) < double.Epsilon)
            {
                success = false;
                return Matrix44.Identity;
            }

            double invdet = 1.0f / det;

            for (int i = 0; i < 4; ++i)
            {
                for (int j = 0; j < 4; ++j)
                {
                    result[i, j] = result[i, j] * invdet;
                }
            }

            success = true;
            return result;
        }
    }
}
