﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class PropMapBase
    {
        protected Mesh Mesh { get; set; }
        protected ResizableList<int> OrigMap { get; set; } = new ResizableList<int>();
        protected SourceKey SourceKey { get; set; } = new SourceKey(SourceType.Position, null);

        public virtual void Init(Mesh mesh, SourceType type, string name)
        {
            Mesh = mesh;
            SourceKey.type = type;
            SourceKey.name = name;

            CreateOrigMap();
        }

        private void CreateOrigMap()
        {
            int faceN = Mesh.FaceN;
            int sn = Mesh.SourceN;
            int so = Mesh.GetSourceOfset(SourceKey.type, SourceKey.name);

            OrigMap.Resize(faceN * 3, int.MaxValue);
            int tid = 0;
            for (int fi = 0; fi < faceN; fi++)
            {
                int vo = Mesh.mVoffs[fi];
                int vn = 3;

                for (int vi = 0; vi < vn; vi++)
                {
                    int sid = Mesh.mVbuffs[(vo + vi) * sn + so];
                    OrigMap[tid++] = sid;
                }
            }
        }
    }

    /// <summary>
    /// プロパティを完全にバラバラになる様に別ＩＤにする
    /// </summary>
    public class SplitPropManager : PropMapBase
    {
        private Dictionary<int, List<int>> mInjectionMap = new Dictionary<int, List<int>>();

        private void CreateInjectionMap()
        {
            int faceN = Mesh.FaceN;
            int sn = Mesh.SourceN;
            int so = Mesh.GetSourceOfset(SourceKey.type, SourceKey.name);
            MeshSourceBase source = Mesh.GetSource(SourceKey.type, SourceKey.name);
            ResizableList<ResizableList<object>> newSouece = new ResizableList<ResizableList<object>>();

            //mInjectionMap.Resize(faceN * 3, int.MaxValue);
            newSouece.Resize(faceN * 3, null);

            // インデックスの再割り当てを行いつつ、コピー元のソースの配列を作成しなおす
            int tid = 0;
            for (int fi = 0; fi < faceN; fi++)
            {
                int vo = Mesh.mVoffs[fi];
                int vn = 3;

                for (int vi = 0; vi < vn; vi++)
                {
                    int orig = Mesh.mVbuffs[(vo + vi) * sn + so];

                    newSouece[tid] = new ResizableList<object>();
                    newSouece[tid].Resize(source.Stride, 0.0);
                    for (int sti = 0; sti < source.Stride; sti++)
                    {
                        newSouece[tid][sti] = source.GetRowValue(orig * source.Stride + sti);
                    }
                    Mesh.mVbuffs[(vo + vi) * sn + so] = tid;

                    if (!mInjectionMap.ContainsKey(orig))
                    {
                        mInjectionMap.Add(orig, new List<int>());
                    }
                    mInjectionMap[orig].Add(tid);
                    tid++;
                }
            }

            // ソースの値もセットしなおします。
            source.SetLength(faceN * 3, false);
            int stride = source.Stride;
            for (int si = 0; si < faceN * 3; si++)
            {
                for (int sti = 0; sti < stride; sti++)
                {
                    source.SetRowValue(stride * si + sti, newSouece[si][sti]);
                }
            }
        }

        public void Run()
        {
            CreateInjectionMap();
        }

        /// <summary>
        /// 逆写像を求める
        /// </summary>
        /// <returns></returns>
        public List<int> GetReverseMap()
        {
            ResizableList<int> revMap = new ResizableList<int>();
            revMap.Resize(OrigMap.Count, int.MaxValue);
            foreach (var valpair in mInjectionMap)
            {
                int orig = valpair.Key;
                foreach (var mapv in valpair.Value)
                {
                    revMap[mapv] = orig;
                }
            }
            RevMapCheck(revMap);
            return revMap;
        }

        /// <summary>
        /// 逆写像が、ちゃんと作れているか？を確認する
        /// </summary>
        /// <param name="revMap"></param>
        [Conditional("DEBUG")]
        private static void RevMapCheck(ResizableList<int> revMap)
        {
            foreach (var val in revMap)
            {
                if (val == int.MaxValue)
                {
                    throw new Exception(string.Format("Find no reference index at {0}", revMap.IndexOf(val)));
                }
            }
        }
    }
}
