﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class SparseOctree<T, TNodeType> where TNodeType : SparseOctree<T, TNodeType>.Node, new()
    {
        /// <summary>
        /// ノードクラス
        /// </summary>
        public class Node
        {
            public Node()
            {
                ParentNode = null;
                ChildNodes = new ResizableList<Node>();
                Objects = new ResizableList<T>();
            }

            public bool IsChildrenNullOrEmpty()
            {
                return ChildNodes == null || !ChildNodes.Any();
            }

            public void AddObject(T obj)
            {
                if (Objects == null) { throw new Exception(); }

                if (!Objects.Contains(obj))
                {
                    Objects.Add(obj);
                }
            }

            public void RemoveObject(T obj)
            {
                if (Objects == null) { throw new Exception(); }
                if (Objects.Contains(obj))
                {
                    Objects.Remove(obj);
                }
            }

            public void ClearObjects()
            {
                Objects.Clear();
            }

            public int Level { get; set; }
            public BoundingBox CellSize { get; set; }

            public Node ParentNode { get; set; }
            public ResizableList<Node> ChildNodes { get; set; }
            public ResizableList<T> Objects { get; set; }
        }

        public TNodeType RootNode { get; set; }
        public uint MaxLevel { get; set; }

        /// <summary>
        /// 八分木の初期化
        /// </summary>
        /// <param name="tl"></param>
        /// <param name="rb"></param>
        public void Initialize(Vector3 tl, Vector3 br, uint maxLevel)
        {
            RootNode = new TNodeType
            {
                Level = 0,
                ParentNode = null,
                CellSize = new BoundingBox()
            };
            RootNode.CellSize.Min.Set(tl);
            RootNode.CellSize.Max.Set(br);
            MaxLevel = maxLevel;
        }

        /// <summary>
        /// 八分木の初期化
        /// </summary>
        /// <param name="bb"></param>
        public void Initialize(BoundingBox bb, uint maxLevel)
        {
            Initialize(bb.Min, bb.Max, maxLevel);
        }

        /// <summary>
        /// ノードを分離する
        /// </summary>
        /// <param name="node"></param>
        public bool SplitNode(Node node)
        {
            // これ以上のスプリットはできない
            if (MaxLevel <= node.Level) { return false; }

            node.ChildNodes = new ResizableList<Node>();
            node.ChildNodes.Resize(8, null);

            var tl = node.CellSize.Min;
            var br = node.CellSize.Max;

            var sizeX = (br.x - tl.x) / 2.0;
            var sizeY = (br.y - tl.y) / 2.0;
            var sizeZ = (br.z - tl.z) / 2.0;
            for (var i = 0; i < 8; i++)
            {
                var z = i / 4;
                var y = (i % 4) / 2;
                var x = (i % 4) % 2;

                node.ChildNodes[i] = new Node
                {
                    Level = node.Level + 1,
                    ParentNode = node,
                    CellSize = new BoundingBox
                    {
                        Min =
                        {
                            x = x * sizeX + tl.x,
                            y = y * sizeY + tl.y,
                            z = z * sizeZ + tl.z
                        },
                        Max =
                        {
                            x = (x + 1) * sizeX + tl.x,
                            y = (y + 1) * sizeY + tl.y,
                            z = (z + 1) * sizeZ + tl.z
                        }
                    }
                };
            }
            // スプリット成功
            return true;
        }

        /// <summary>
        /// 指定のノードと、BBの衝突判定
        /// </summary>
        /// <param name="node">検査対象のノード</param>
        /// <param name="bb">判定したいBB</param>
        /// <returns></returns>
        public static bool HitTest(Node node, BoundingBox bb)
        {
            return node.CellSize.Intersect(bb);
        }

        /// <summary>
        /// 指定のBBと交差しているセルを全部取得する
        /// </summary>
        public IEnumerable<Node> WindowQuery(BoundingBox bb, bool bottomOnly)
        {
            var result = new List<Node>();
            WindowQuery(result, RootNode, bb);
            return bottomOnly ? result.Where(o => o.IsChildrenNullOrEmpty()) : result;
        }

        /// <summary>
        /// 子ノードまで行って、交差してるノードをすべて取得する
        /// </summary>
        /// <param name="result"></param>
        /// <param name="node"></param>
        /// <param name="bb"></param>
        private static void WindowQuery(ICollection<Node> result, Node node, BoundingBox bb)
        {
            if (!HitTest(node, bb)) { return; }
            result.Add(node);

            if (node.IsChildrenNullOrEmpty()) { return; }
            foreach (var child in node.ChildNodes)
            {
                WindowQuery(result, child, bb);
            }
        }
        /// <summary>
        /// 指定オブジェクトを含んでいるノードをすべて取得する
        /// </summary>
        /// <param name="result"></param>
        /// <param name="node"></param>
        /// <param name="obj"></param>
        public void FindNode(ICollection<Node> result, Node node, T obj)
        {
            if (node == null) { return; }
            // オブジェクトがノードに含まれているので取得
            if (node.Objects.Contains(obj))
            {
                result.Add(node);
            }
            if (node.IsChildrenNullOrEmpty()) { return; }
            foreach (var child in node.ChildNodes)
            {
                FindNode(result, child, obj);
            }
        }
        /// <summary>
        /// 全てのノードから、指定レベルのノードを再帰的に取得する
        /// </summary>
        /// <param name="result"></param>
        /// <param name="node"></param>
        /// <param name="level"></param>
        public void GetLevelNodes(ICollection<Node> result, Node node, int level)
        {
            if (node == null) { return; }
            // オブジェクトがノードに含まれているので取得
            if (node.Level == level)
            {
                result.Add(node);
            }

            if (node.IsChildrenNullOrEmpty()) { return; }
            foreach (var child in node.ChildNodes)
            {
                GetLevelNodes(result, child, level);
            }
        }

        /// <summary>
        /// 指定の位置にオブジェクトを追加
        /// 最下層ノードにのみ追加を行うかどうか？を設定する
        /// </summary>
        /// <param name="obj"></param>
        /// <param name="bb"></param>
        /// <param name="bottomOnly"></param>
        public void AddObject(T obj, BoundingBox bb, bool bottomOnly = true)
        {
            if (obj == null) { return; }
            var nodes = WindowQuery(bb, bottomOnly);
            foreach (var node in nodes)
            {
                node.AddObject(obj);
            }
        }

        /// <summary>
        /// オブジェクトの削除を行う
        /// </summary>
        /// <param name="obj"></param>
        public void RemoveObject(T obj)
        {
            var result = new List<Node>();
            FindNode(result, RootNode, obj);
            foreach (var node in result)
            {
                node.RemoveObject(obj);
            }
        }

        /// <summary>
        /// 指定レベルのノードをすべて取得
        /// </summary>
        /// <param name="level"></param>
        /// <returns></returns>
        public IEnumerable<Node> GetLevelNodes(int level)
        {
            var result = new List<Node>();
            // 再帰的に指定レベルのノードを取得する
            GetLevelNodes(result, RootNode, level);
            return result;
        }
    }
}
