﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    /// <summary>
    /// 空間ハッシュクラス
    /// </summary>
    public class SpacialHash
    {
        private const int P1 = 73856093;
        private const int P2 = 19349663;
        private const int P3 = 83492791;

        // function lists
        public static long hash_func(long x, long y, long z)
        {
            var hash = (x * P1);
            hash += (y * P2);
            hash += (z * P3);
            return hash;
        }

        /// <summary>
        /// 頂点位置を１００倍して、小数点以下第二位程度まで
        /// 有効にした上でハッシュを計算します        /// </summary>
        /// <param name="dx"></param>
        /// <param name="dy"></param>
        /// <param name="dz"></param>
        /// <param name="scale"></param>
        /// <returns></returns>
        public static long hash_func(double dx, double dy, double dz, double scale = 1e4)
        {
            var x = (long)(dx * scale);
            var y = (long)(dy * scale);
            var z = (long)(dz * scale);
            return hash_func(x, y, z);
        }

        /// <summary>
        /// 頂点位置を１００倍して、小数点以下第二位程度まで
        /// 有効にした上でハッシュを計算します
        /// </summary>
        /// <param name="pos"></param>
        /// <returns></returns>
        public static long hash_func(Vector3 pos, double scale = 1e4)
        {
            pos = pos * scale;
            return hash_func((long)pos.x, (long)pos.y, (long)pos.z);
        }
    }

    /// <summary>
    /// 空間ハッシュリスト
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class SpacialHashList<T>
    {
        private int mTableSize = 1;
        private ResizableList<ResizableList<T>> mTable = null;

        public SpacialHashList()
        {
            Count = 1;
        }

        /// <summary>
        /// インデクサ
        /// </summary>
        public ResizableList<T> this[int index]
        {
            get { return mTable[index]; }
            set { mTable[index] = value; }
        }

        /// <summary>
        /// ハッシュテーブルのサイズを取得します
        /// </summary>
        public int Count
        {
            get { return mTableSize; }
            set
            {
                mTableSize = value;
                mTable = new ResizableList<ResizableList<T>>();
                mTable.Resize(mTableSize, null);
                for (int i = 0; i < mTableSize; i++)
                {
                    mTable[i] = new ResizableList<T>();
                }
            }
        }

        public void Clear()
        {
            for (int i = 0; i < mTableSize; i++)
            {
                mTable[i].Clear();
            }
        }

        /// <summary>
        /// 指定の座標から計算される空間ハッシュのリストを取得します
        /// </summary>
        /// <param name="p"></param>
        /// <returns></returns>
        public List<T> GetList(Vector3 p)
        {
            int h = (int)SpacialHash.hash_func(p) % mTableSize;
            return mTable[h];
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <returns></returns>
        public List<T> GetList(int x, int y, int z)
        {
            var h = (int)SpacialHash.hash_func(x, y, z) % mTableSize;
            return mTable[h];
        }

        /// <summary>
        /// 指定の座標から、ハッシュを計算しオブジェクトを追加
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <param name="obj"></param>
        public void Add(int x, int y, int z, T obj)
        {
            var h = (int)SpacialHash.hash_func(x, y, z) % mTableSize;

            mTable[h].Add(obj);
        }

        /// <summary>
        /// 指定の座標から、ハッシュを計算しオブジェクトを追加
        /// </summary>
        /// <param name="p">位置座標</param>
        /// <param name="obj">追加するオブジェクト</param>
        public void Add(Vector3 p, T obj)
        {
            var h = (int)SpacialHash.hash_func(p) % mTableSize;
            mTable[h].Add(obj);
        }
    }
}
