﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using MS.Internal.Xml.XPath;
using nw4f.tinymathlib;
using System.Diagnostics;
using System.Linq;

namespace nw4f.meshlib
{
    public partial class QemSimp
    {
        /// <summary>
        /// 境界線かどうか？を取得します
        /// </summary>
        /// <param name="val">境界</param>
        /// <param name="mask"></param>
        /// <returns></returns>
        public bool IsBorder(int val, QemVertexType mask = QemVertexType.ALL)
        {
            return (val & (int)mask) != 0;
        }

        /// <summary>
        /// 内点同士が結合されたエッジであるかどうか？について、検査します
        /// </summary>
        /// <param name="flag0"></param>
        /// <param name="flag1"></param>
        /// <param name="mask"></param>
        /// <returns></returns>
        private bool isInnerEdge(int flag0, int flag1, QemVertexType mask = QemVertexType.ALL)
        {
            bool innner = !IsBorder(flag0, mask) && !IsBorder(flag1, mask);
            return innner;
        }

        /// <summary>
        /// 指定タイプの頂点同士が結合されたエッジかをチェックします
        /// </summary>
        /// <param name="flag0"></param>
        /// <param name="flag1"></param>
        /// <param name="mask"></param>
        /// <returns></returns>
        private bool IsBorderEdge(int flag0, int flag1, QemVertexType mask = QemVertexType.ALL)
        {
            bool innner = IsBorder(flag0, mask) && IsBorder(flag1, mask);
            return innner;
        }

        /// <summary>
        /// 指定頂点に接地しているエッジであるかどうか？をチェックします
        /// </summary>
        /// <param name="flag0"></param>
        /// <param name="flag1"></param>
        /// <param name="mask"></param>
        /// <returns></returns>
        private bool isOnBorderEdge(int flag0, int flag1, QemVertexType mask = QemVertexType.ALL)
        {
            bool onBorder = IsBorder(flag0, mask) || IsBorder(flag1, mask);
            return onBorder;
        }

        /// <summary>
        /// 周辺の面の法線を比較して、指定の角度以上かどうか？を判定
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="threshold"></param>
        /// <returns></returns>
        private bool IsShrinkableFeaturedEdgeAngle(WingedEdge edge, double threshold, ref double angle)
        {
            var awn0 = new Vector3();
            var totalArea0 = 0.0;
            var awn1 = new Vector3();
            var totalArea1 = 0.0;
            // 面積重み付き
            awn0.SetZero();
            awn1.SetZero();
            foreach (var tid in VtxTriMesh.VTL[edge.Vertex0].TriList)
            {
                if (tid != int.MaxValue)
                {
                    awn0 += (mMesh.mFaceNormal[tid] * mMesh.mFaceArea[tid]);
                    totalArea0 += mMesh.mFaceArea[tid];
                }
            }

            foreach (var tid in VtxTriMesh.VTL[edge.Vertex1].TriList)
            {
                if (tid != int.MaxValue)
                {
                    awn1 += (mMesh.mFaceNormal[tid] * mMesh.mFaceArea[tid]);
                    totalArea1 += mMesh.mFaceArea[tid];
                }
            }
            // 一方がゼロベクトルであれば判定不能
            // 法線を持ってない場合にあり得る
            if (awn0.Norm() < double.Epsilon || awn1.Norm() < double.Epsilon ||
                totalArea0 < double.Epsilon || totalArea1 < double.Epsilon)
            {
                return true;
            }

            awn0 = awn0 / totalArea0;
            awn1 = awn1 / totalArea1;
            awn0.Normalize();
            awn1.Normalize();

            angle = awn0.GetAngle(awn1, false);
            // 上限であれば動く
            if (threshold >= 180.0 || threshold < 0)
            {
                return true;
            }
            if (angle <= threshold)
            {
                return true;
            }
            return false;
        }

        internal List<WingedEdge> GetNeighborOpenEdges(WingedEdge edge, int side, bool getOpen = true)
        {
            List<WingedEdge> borders = new List<WingedEdge>();

            ResizableList<VtxWnEdge> typeVEL = WingEdedgeMesh.GetVtxWEList(edge.SpaceKey.type, edge.SpaceKey.name);

            var vtxId = side == 0 ? edge.Vertex0 : edge.Vertex1;
            //　重み計算の為に角度を求めたい
            foreach (var nedge in typeVEL[vtxId].EdgeList)
            {
                if (edge.IsVerticesEqual(nedge))
                {
                    continue;
                }
                if (nedge.Face0 == int.MaxValue)
                {
                    continue;
                }
                if (getOpen && nedge.IsOpen)
                {
                    borders.Add(nedge);
                }
            }
            return borders;
        }

        /// <summary>
        /// 旧関数、不要になれば関数と参照箇所すべてを削除する
        ///
        /// 縮退可能な、OpenEdgeかを判断する
        /// UV開放端と、開放端の双方が対象
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="threshold"></param>
        /// <param name="angle"></param>
        /// <returns></returns>
        private bool Old_IsShrinkableOpenEdgeAngle(WingedEdge edge, double threshold, ref double angle)
        {
            // 下限であれば動かさない
            if (threshold < double.Epsilon)
            {
                return false;
            }
            // 上限であれば動く
            if (threshold >= 90.0)
            {
                return true;
            }

            List<WingedEdge> borders = new List<WingedEdge>();
            borders.AddRange(GetNeighborOpenEdges(edge, 0));
            if (edge.IsOpen)
            {
                borders.Add(edge);
            }
            borders.AddRange(GetNeighborOpenEdges(edge, 1));
            if (borders.Count <= 1)
            {
                return true;
            }

            int bn = borders.Count;
            angle = double.MaxValue;
            for (int i = 0; i < bn - 1; i++)
            {
                angle = Math.Min(angle, WingedEdgeMesh.Old_ComputeAngle(WingEdedgeMesh, borders[i], borders[i + 1]));
            }
            if (angle <= (threshold + 1e-5))
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// 縮退可能な、OpenEdgeかを判断する
        /// UV開放端と、開放端の双方が対象
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="threshold"></param>
        /// <param name="angle"></param>
        /// <returns></returns>
        private bool IsShrinkableOpenEdgeAngle(WingedEdge edge, double threshold, ref double angle)
        {
            // 下限であれば,全エッジが保護角ではない
            if (threshold < double.Epsilon)
            {
                return true;
            }
            // 上限であれば全エッジが保護角になる
            if (threshold >= 180.0)
            {
                return false;
            }

            List<WingedEdge> borders = new List<WingedEdge>();
            borders.AddRange(GetNeighborOpenEdges(edge, 0));
            if (edge.IsOpen)
            {
                borders.Add(edge);
            }
            borders.AddRange(GetNeighborOpenEdges(edge, 1));
            if (borders.Count <= 1)
            {
                return true;
            }

            int bn = borders.Count;
            angle = double.MaxValue;
            for (int i = 0; i < bn - 1; i++)
            {
                angle = Math.Min(angle, WingedEdgeMesh.ComputeAngle(WingEdedgeMesh, borders[i], borders[i + 1]));
            }

            // 指定の劣角以上の場合のみ簡略化が可能
            if (angle > threshold + 1e-5)
            {
                return true;
            }
            return false;
        }

        private List<WingedEdge> GetNeighborFeatEdges(WingedEdge edge, int side, int[] border)
        {
            List<WingedEdge> borders = new List<WingedEdge>();

            ResizableList<VtxWnEdge> typeVEL = WingEdedgeMesh.GetVtxWEList(edge.SpaceKey.type, edge.SpaceKey.name);

            var vtxId = side == 0 ? edge.Vertex0 : edge.Vertex1;
            //　重み計算の為に角度を求めたい
            foreach (var nedge in typeVEL[vtxId].EdgeList)
            {
                if (edge.IsVerticesEqual(nedge))
                {
                    continue;
                }
                if (nedge.Face0 == int.MaxValue)
                {
                    continue;
                }
                if (IsBorderEdge(border[nedge.Vertex0], border[nedge.Vertex1], QemVertexType.FeatEdge))
                {
                    borders.Add(nedge);
                }
            }
            return borders;
        }

        /// <summary>
        /// 縮退可能な特徴エッジかを判断する
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="threshold"></param>
        /// <param name="border"></param>
        /// <param name="angle"></param>
        /// <returns></returns>
        private bool IsShrinkableFeatEdgeAngle(WingedEdge edge, double threshold, int[] border, ref double angle)
        {
            // 下限であれば動かさない
            if (threshold < double.Epsilon)
            {
                return false;
            }
            // 上限であれば動く
            if (threshold >= 90.0)
            {
                return true;
            }

            List<WingedEdge> borders = new List<WingedEdge>();
            borders.AddRange(GetNeighborFeatEdges(edge, 0, border));
            if (IsBorderEdge(border[edge.Vertex0], border[edge.Vertex1], QemVertexType.FeatEdge))
            {
                borders.Add(edge);
            }
            borders.AddRange(GetNeighborFeatEdges(edge, 1, border));
            if (borders.Count <= 1)
            {
                return true;
            }

            int bn = borders.Count;
            angle = double.MaxValue;
            for (int i = 0; i < bn - 1; i++)
            {
                angle = Math.Min(angle, WingedEdgeMesh.ComputeAngle(WingEdedgeMesh, borders[i], borders[i + 1]));
            }

            if (angle <= threshold)
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// 指定のエッジがサブメッシュ境界そのものであるか？をチェックする
        /// </summary>
        /// <param name="edge">チェックしたいエッジ</param>
        /// <returns> true: サブメッシュ境界</returns>
        private bool IsSubmeshBorderEdge(WingedEdge edge)
        {
            if (edge == null)
            {
                throw ExcepHandle.CreateException("Argument is NULL");
            }
            if (LockSubMeshBoder)
            {
                if (edge.Face0 == int.MaxValue || edge.Face1 == int.MaxValue) { return false; }
                else if (mMesh.mFaceSubMeshID[edge.Face0] != mMesh.mFaceSubMeshID[edge.Face1]) { return true; }
            }
            return false;
        }

        /// <summary>
        /// シェイプの境界であるか？をチェックします
        /// </summary>
        /// <param name="edge">チェックしたいエッジ</param>
        /// <returns> true: シェイプ境界 </returns>
        private bool IsShapeBorderEdge(WingedEdge edge)
        {
            if (edge == null)
            {
                throw ExcepHandle.CreateException("Argument is NULL");
            }

            if (edge.Face0 == int.MaxValue || edge.Face1 == int.MaxValue) { return false; }
            else if (mMesh._FaceMaterialShapeID[edge.Face0] != mMesh._FaceMaterialShapeID[edge.Face1]) { return true; }
            return false;
        }

        /// <summary>
        /// サブメッシュ境界そのものか？サブメッシュ境界に設置しているか？を判定する
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="vertexborder"></param>
        /// <returns></returns>
        private int JudgeSubMeshBorderEdge(
            WingedEdge edge,
            int[] vertexborder)
        {
            //--------------------------------------
            // サブメッシュ境界エッジそのものは不動
            if (IsSubmeshBorderEdge(edge))
            {
                return (int)QemEdgeType.NonManifold;
            }
            //-------------------------------------------------
            // サブメッシュの境界に接地している
            if (isOnBorderEdge(vertexborder[edge.Vertex0], vertexborder[edge.Vertex1], QemVertexType.SubMesh))
            {
                // 反対側の頂点もサブメッシュ境界の場合は、削減できないエッジ
                bool b0 = IsBorder(vertexborder[edge.Vertex0], QemVertexType.SubMesh);
                bool b1 = IsBorder(vertexborder[edge.Vertex1], QemVertexType.SubMesh);
                int otherSide = edge.Vertex1;
                if (b1 && !b0)
                {
                    otherSide = edge.Vertex0;
                }
                else if (!b1 && b0)
                {
                    otherSide = edge.Vertex1;
                }
                else
                {
                    return (int)QemEdgeType.NonManifold;
                }

                // オープンエッジとShape境界にも接続がある場合は、ロックエッジにする
                // これをやらないと穴が開く
                if (
                    (vertexborder[otherSide] == (int)QemVertexType.Exclude) ||
                    (vertexborder[otherSide] == (int)QemVertexType.OpenEdge) ||
                    (vertexborder[otherSide] == (int)QemVertexType.FeatEdge))
                {
                    return (int)QemEdgeType.NonManifold;
                }
                return (int)QemEdgeType.TouchedHard;
            }
            return (int)QemEdgeType.Interior;
        }

        /// <summary>
        /// Shape Edge の状態チェックを行います
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="vertexborder"></param>
        /// <param name="cangle"></param>
        /// <returns></returns>
        private int JudgeShapeEdgeType
        (
            WingedEdge edge,
            int[] vertexborder,
            ref double cangle)
        {
            int f0 = edge.Face0;
            int f1 = edge.Face1;

            int reslt = 0;
            //--------------------------------------
            // そもそも縮退不可能なエッジ
            if (f0 == int.MaxValue && f1 == int.MaxValue)
            {
                return (int)QemEdgeType.NonManifold;
            }
            //--------------------------------------
            // 除外対象なので不動エッジ
            if (IsBorder(vertexborder[edge.Vertex0], QemVertexType.Exclude) &&
                IsBorder(vertexborder[edge.Vertex1], QemVertexType.Exclude))
            {
                return (int)QemEdgeType.NonManifold;
            }
            //--------------------------------------
            //非多様体エッジに接地しているエッジは触らない
            if (isOnBorderEdge(vertexborder[edge.Vertex0], vertexborder[edge.Vertex1], QemVertexType.Nonmanifold))
            {
                return (int)QemEdgeType.NonManifold;
            }

            // T字エッジに引っかかってる時は動かせない
            if (isOnBorderEdge(vertexborder[edge.Vertex0], vertexborder[edge.Vertex1], QemVertexType.Junction))
            {
                return (int)QemEdgeType.NonManifold;
            }

            if (isOnBorderEdge(vertexborder[edge.Vertex0], vertexborder[edge.Vertex1], QemVertexType.Corner))
            {
                if (IsBorderEdge(vertexborder[edge.Vertex0], vertexborder[edge.Vertex1], QemVertexType.Corner))
                {
                    return (int)QemEdgeType.NonManifold;
                }
                reslt |= (int)QemEdgeType.TouchedHard;
            }
            // 一端が除外対象の場合は除外側への縮退が可能
            if (isOnBorderEdge(vertexborder[edge.Vertex0], vertexborder[edge.Vertex1], QemVertexType.Exclude))
            {
                reslt |= (int)QemEdgeType.TouchedHard;
            }
            // サブメッシュ境界とサブメッシュ境界に接地しているエッジの判定
            reslt |= JudgeSubMeshBorderEdge(edge, vertexborder);

            //開放端
            if (edge.Boder)
            {
                // エッジそのものが、開放端でありながら、両端はプロパティハードエッジ
                //角度によって、縮退の可否を決める
                //新パラメータが有効
                if (LockOpenEdgeAngle > 0)
                {
                    if (!IsShrinkableOpenEdgeAngle(edge, LockOpenEdgeAngle, ref cangle))
                    {
                        return (int)QemEdgeType.NonManifold;
                    }
                }
                // 旧パラメータ有効
                else
                {
                    if (!Old_IsShrinkableOpenEdgeAngle(edge, OldLockOpenEdgeAngle, ref cangle))
                    {
                        return (int)QemEdgeType.NonManifold;
                    }
                }

                //角度によって、縮退の可否を決める
                reslt |= (int)QemEdgeType.HardEdge;
            }
            else if (isOnBorderEdge(vertexborder[edge.Vertex0], vertexborder[edge.Vertex1], QemVertexType.OpenEdge))
            {
                if (IsBorder(vertexborder[edge.Vertex0], QemVertexType.OpenEdge) &&
                    IsBorder(vertexborder[edge.Vertex1], QemVertexType.OpenEdge))
                {
                    reslt |= (int)QemEdgeType.NonManifold;
                }
                if (IsBorder(vertexborder[edge.Vertex0], QemVertexType.OpenEdge) &&
                    IsBorder(vertexborder[edge.Vertex1], QemVertexType.FeatEdge))
                {
                    reslt |= (int)QemEdgeType.NonManifold;
                }
                else if (IsBorder(vertexborder[edge.Vertex1], QemVertexType.OpenEdge) &&
                    IsBorder(vertexborder[edge.Vertex0], QemVertexType.FeatEdge))
                {
                    reslt |= (int)QemEdgeType.NonManifold;
                }
                else if (!IsBorder(vertexborder[edge.Vertex0], QemVertexType.OpenEdge) ||
                         !IsBorder(vertexborder[edge.Vertex1], QemVertexType.OpenEdge))
                {
                    reslt |= (int)QemEdgeType.TouchedHard;
                }
            }

            return reslt;
        }

        /// <summary>
        /// 一本のＵＶ空間エッジについて、エッジ種別を判定する
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="vertexborder"></param>
        /// <param name="cangle"></param>
        /// <returns></returns>
        private int JudgeSingleUVEdgeType
            (
            WingedEdge edge,
            int[] vertexborder,
            ref double cangle)
        {
            int reslt = 0;

            WingedEdge pe = edge.ParentEdges[0];
            if (pe == null)
            {
                throw ExcepHandle.CreateException("Null reference");
            }

            if (edge.IsOpen)
            {
                // 新パラメータがセットされている
                if (LockUvHardEdgeAngle > 0)
                {
                    if (!IsShrinkableOpenEdgeAngle(edge, LockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    else
                    {
                        reslt |= (int)QemEdgeType.HardEdge;
                    }
                }
                else
                {
                    if (!Old_IsShrinkableOpenEdgeAngle(edge, OldLockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    else
                    {
                        reslt |= (int)QemEdgeType.HardEdge;
                    }
                }
            }
            else if (IsBorderEdge(vertexborder[pe.Vertex0], vertexborder[pe.Vertex1], QemVertexType.UVEdge))
            {
                reslt |= (int)QemEdgeType.NonManifold;
            }
            else if (isOnBorderEdge(vertexborder[pe.Vertex0], vertexborder[pe.Vertex1], QemVertexType.UVEdge))
            {
                // 新パラメータがセットされている
                if (LockUvHardEdgeAngle > 0)
                {
                    if (!IsShrinkableOpenEdgeAngle(edge, LockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    else
                    {
                        reslt |= (int)QemEdgeType.TouchedHard;
                    }
                }
                else
                {
                    if (!Old_IsShrinkableOpenEdgeAngle(edge, OldLockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    else
                    {
                        reslt |= (int)QemEdgeType.TouchedHard;
                    }
                }
            }
            return reslt;
        }

        /// <summary>
        /// ＵＶエッジのペア（ないし、単体）に対してエッジの種別の判定を行います。
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="vertexborder"></param>
        /// <param name="cangle"></param>
        /// <returns></returns>
        private int JudgeUVEdgeType
            (
            ResizableList<WingedEdge> edgeList,
            int[] vertexborder,
            ref double cangle)
        {
            int reslt = 0;
            if (edgeList.Count == 0)
            {
                return reslt;
            }
            if (edgeList.Count == 1)
            {
                reslt |= JudgeSingleUVEdgeType(edgeList[0], vertexborder, ref cangle);
            }
            else
            {
                if (LockUvHardEdgeAngle > 0)
                {
                    if (!IsShrinkableOpenEdgeAngle(edgeList[0], LockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    else if (!IsShrinkableOpenEdgeAngle(edgeList[1], LockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    reslt |= (int)QemEdgeType.PropertyEdge;
                }
                else
                {
                    if (!Old_IsShrinkableOpenEdgeAngle(edgeList[0], OldLockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    else if (!Old_IsShrinkableOpenEdgeAngle(edgeList[1], OldLockUvHardEdgeAngle, ref cangle))
                    {
                        reslt |= (int)QemEdgeType.NonManifold;
                    }
                    reslt |= (int)QemEdgeType.PropertyEdge;
                }
            }
            return reslt;
        }

        private int JugeFeatureEdgeType(WingedEdge edge, int[] border, double[] cangles)
        {
            int result = (int)QemEdgeType.Interior;
            // 特徴稜線かどうか？を周辺法線から計算
            if (FeatureFaceAngle > 0.0f && FeatureFaceAngle <= 180.0f)
            {
                if (IsBorderEdge(border[edge.Vertex0], border[edge.Vertex1], QemVertexType.FeatEdge))
                {
                    if (IsShrinkableFeatEdgeAngle(edge, FeatureEdgeLockAngle, border, ref cangles[(int)SourceType.Normal]))
                    {
                        result = result | (int)QemEdgeType.HardEdge;
                    }
                }
                else if (isOnBorderEdge(border[edge.Vertex0], border[edge.Vertex1], QemVertexType.FeatEdge))
                {
                    result = result | (int)QemEdgeType.TouchedHard;
                }
            }
            else if (OldFeatureFaceAngle >= 0.0f && OldFeatureFaceAngle < 180.0f)
            {
                if (IsBorderEdge(border[edge.Vertex0], border[edge.Vertex1], QemVertexType.FeatEdge))
                {
                    if (IsShrinkableFeatEdgeAngle(edge, FeatureEdgeLockAngle, border, ref cangles[(int)SourceType.Normal]))
                    {
                        result = result | (int)QemEdgeType.HardEdge;
                    }
                }
                else if (isOnBorderEdge(border[edge.Vertex0], border[edge.Vertex1], QemVertexType.FeatEdge))
                {
                    result = result | (int)QemEdgeType.TouchedHard;
                }
            }
            return result;
        }

        /// <summary>
        /// これに置き換える
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="vertexborder"></param>
        /// <param name="ecnfl0"></param>
        /// <param name="ecnfl1"></param>
        /// <param name="wangle"></param>
        /// <param name="cangle"></param>
        /// <returns></returns>
        private int JudgeEdgeType(WingedEdge edge, List<int[]> vertexborder, double[] cangles)
        {
            double cangle = 0;
            int result = (int)QemEdgeType.Interior;
            result = result | JudgeShapeEdgeType(edge, vertexborder[0], ref cangles[(int)SourceType.Position]);
            // 既にシェイプのエッジタイプ判定の段階で、削れないエッジであることが分かった場合は以降は不要
            if ((result & (int)QemEdgeType.NonManifold) == (int)QemEdgeType.NonManifold)
            {
                return result;
            }
            var ci = 0;
            foreach (var childContainer in edge.ChildEdges)
            {
                if (childContainer.Key.type == SourceType.Texture && IsUVEnabled)
                {
                    result |= JudgeUVEdgeType(childContainer.Value, vertexborder[ci], ref cangle);
                    cangles[(int)SourceType.Texture] = Math.Max(cangle, cangles[(int)SourceType.Texture]);
                }
                ci++;
                break;
            }
            result |= JugeFeatureEdgeType(edge, vertexborder[0], cangles);
            return result;
        }
    }
}
