﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class PropMapTaransform
    {
        private Mesh mMesh;
        private Mesh mOrigMesh;
        private SourceKey mKey = new SourceKey(SourceType.Position, null);

        private ClosedPropManager mCloseMan = new ClosedPropManager();
        private SplitPropManager mSplitMan = new SplitPropManager();

        /// <summary>
        /// テスト関数
        /// 実行時に、未参照のプロパティが存在する事を確認する。
        /// 逆写像を走らせる際の未参照プロパティの存在がエラーではないか？を確認する為に使用
        /// </summary>
        [Conditional("DEBUG")]
        private void CheckHasUnusedProps()
        {
            var valueRefFlags = new ResizableList<ResizableList<bool>>();

            valueRefFlags.Resize(mMesh.SourceN, null);

            for (int si = 0; si < mMesh.SourceN; si++)
            {
                valueRefFlags[si] = new ResizableList<bool>();
                valueRefFlags[si].Resize(mMesh.Sources[si].Count, false);
            }

            for (int fi = 0; fi < mMesh.FaceN; fi++)
            {
                var vn = mMesh.VNums[fi];
                var vo = mMesh.VOffset[fi];

                for (var vi = 0; vi < vn; vi++)
                {
                    for (int si = 0; si < mMesh.SourceN; si++)
                    {
                        var index = mMesh.VBuffer[(vo + vi) * mMesh.SourceN + si];
                        valueRefFlags[si][index] = true;
                    }
                }
            }

            foreach (var valueRefFlag in valueRefFlags)
            {
                if (valueRefFlag.Any(o => o == false))
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"mesh has unreferenced values in source >> this is not error.");
                }
            }
        }

        /// <summary>
        /// 処理を実行する
        /// </summary>
        /// <param name="mesh">対象メッシュ</param>
        /// <param name="type">変換を行うソースタイプ</param>
        /// <param name="name">変換を行うソース名</param>
        public void Run(Mesh mesh, SourceType type, string name, bool cleanUp)
        {
            mMesh = mesh;
            CheckHasUnusedProps();
            mOrigMesh = mesh.Clone();
            mKey.type = type;
            mKey.name = name;

            mSplitMan.Init(mMesh, type, name);
            mSplitMan.Run();
            mCloseMan.Init(mMesh, type, name);
            mCloseMan.Run();

            mMesh.RemoveInvalidFaces();
            mMesh.RemoveUnusedVertex();
        }

        /// <summary>
        /// 与えられたメッシュのインデックスの入れ替え
        /// </summary>
        /// <param name="mesh"></param>
        public bool Revert(Mesh mesh, bool createNewSource)
        {
            MeshSourceBase source = mesh.GetSource(mKey.type, mKey.name);
            if (source == null)
            {
                throw ExcepHandle.CreateException("Source can not be found.");
            }

            int faceN = mesh.FaceN;
            int sn = mesh.SourceN;
            int so = mesh.GetSourceOfset(mKey.type, mKey.name);

            //------------------------------------------------------------
            // Face インデックスの入れ替え
            //------------------------------------------------------------
            Dictionary<int, List<int>> revMap1 = mCloseMan.GetReverseMap();
            for (int fi = 0; fi < faceN; fi++)
            {
                int vo = mesh.mVoffs[fi];
                int vn = mesh.mVnums[fi];
                for (int vi = 0; vi < vn; vi++)
                {
                    int sid = mesh.mVbuffs[(vo + vi) * sn + so];
                    // 簡略化の結果重複が出るので、完全に合理的にはばらせない
                    // 従って、やむなく一つに纏める
                    if (revMap1.ContainsKey(sid))
                    {
                        mesh.mVbuffs[(vo + vi) * sn + so] = revMap1[sid][0];
                    }
                }
            }

            //------------------------------------------------------------
            List<int> revMap2 = mSplitMan.GetReverseMap();
            for (int fi = 0; fi < faceN; fi++)
            {
                int vo = mesh.mVoffs[fi];
                int vn = mesh.mVnums[fi];
                for (int vi = 0; vi < vn; vi++)
                {
                    int sid = mesh.mVbuffs[(vo + vi) * sn + so];
                    mesh.mVbuffs[(vo + vi) * sn + so] = revMap2[sid];
                }
            }

            //------------------------------------------------------------
            // Source Value の更新
            //------------------------------------------------------------
            if (!createNewSource)
            {
                MeshSourceBase orig_source = mOrigMesh.GetSource(mKey.type, mKey.name);
                source.SetLength(orig_source.RowCount, true);
                for (int i = 0; i < orig_source.RowCount; i++)
                {
                    source.SetRowValue(i, orig_source.GetRowValue(i));
                }
                orig_source.Clone();
            }
            else
            {
                // 最大ソースＩＤをピック
                int revSourceN = 0;
                foreach (var rvm1 in revMap1)
                {
                    // SplitMan で全単射化してあるので、全てを逆検索しないと失敗する
                    foreach (var tmpid in rvm1.Value)
                    {
                        int dis = revMap2[tmpid];
                        revSourceN = Math.Max(revSourceN, dis);
                    }
                }
                revSourceN++;

                revMap1 = mCloseMan.GetReverseMap();
                ResizableList<ResizableList<object>> tmpSource = new ResizableList<ResizableList<object>>();
                tmpSource.Resize(revSourceN, null);
                int stride = source.Stride;
                foreach (var rvm1 in revMap1)
                {
                    int sid = rvm1.Key;
                    // SplitMan で全単射化してあるので、全てを逆検索しないと失敗する
                    foreach (var tmpid in rvm1.Value)
                    {
                        int dis = revMap2[tmpid];
                        tmpSource[dis] = new ResizableList<object>();
                        for (int si = 0; si < stride; si++)
                        {
                            object v = source.GetRowValue(sid * stride + si);
                            tmpSource[dis].Add(v);
                        }
                    }
                }

                //------------------------------------------------------------
                bool allPropValueReferenced = true;
                source.SetLength(revSourceN, false);
                for (int i = 0; i < tmpSource.Count; i++)
                {
                    for (int si = 0; si < stride; si++)
                    {
                        if (tmpSource[i] == null)
                        {
                            allPropValueReferenced = false;
                        }
                        else
                        {
                            source.SetRowValue(i * stride + si, tmpSource[i][si]);
                        }
                    }
                }
                // 注
                // falseの場合は、未参照の値がソース中に存在する事になる（ただし、それがエラーとは言えない様です)
                // 頂点ストリーム中に、未参照の値が眠っているけどデータは正しく構成されている、という状態
                return allPropValueReferenced;
            }
            return true;
        }
    }
}
