﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class XYZImporter
    {
        /// <summary>
        /// 指定の点列を、ファイルにXYZ フォーマットで保存します
        /// </summary>
        /// <param name="points"></param>
        /// <param name="pFileName"></param>
        public static void SavePointsToXYZ(List<Vector3> points, List<Vector3> normals, string pFileName)
        {
            if (points == null) { return; }
            if (normals != null)
            {
                if (points.Count != normals.Count)
                {
                    throw new Exception("points and normals count is diiferent");
                }
            }

            using (StreamWriter writer = new StreamWriter(pFileName))
            {
                int i = 0;
                foreach (var point in points)
                {
                    if (normals != null)
                    {
                        writer.WriteLine("{0} {1} {2} {3} {4} {5}", point.x, point.y, point.z, normals[i].x, normals[i].y, normals[i].z);
                    }
                    else
                    {
                        writer.WriteLine("{0} {1} {2}", point.x, point.y, point.z);
                    }
                    i++;
                }
            }
        }

        public static void SavePointsToXYZ(List<Vector2> points, string pFileName)
        {
            if (points == null) { return; }
            using (StreamWriter writer = new StreamWriter(pFileName))
            {
                int i = 0;
                foreach (var point in points)
                {
                    writer.WriteLine("{0} {1} {2}", point.x, point.y, 0.0);
                    i++;
                }
            }
        }
    }

    public class ObjImporter
    {
        //----------------------------------------------------------------------------------
        public static Mesh LoadMeshFromObj(string pFileName)
        {
            try
            {
                var pMesh = new Mesh();
                pMesh.MeshName = System.IO.Path.GetFileNameWithoutExtension(pFileName);

                MeshSourceBase vertex = pMesh.AddSource("pos", SourceType.Position, 3);
                MeshSourceBase texcoord = null;
                MeshSourceBase normals = null;
                //MeshSourceBase etcetera = pMesh.AddSource("test", SourceType.EtceteraInt, 1);

                var coord = new List<string>();
                using (StreamReader reader = File.OpenText(pFileName))
                {
                    while (reader.Peek() >= 0)
                    {
                        // ファイルを 1 行ずつ読み込む
                        string stBuffer = reader.ReadLine();
                        // 読み込んだものを追加で格納する
                        coord.Add(stBuffer);
                    }
                }

                // prefix sum
                int vtxN = 0;
                int nmlN = 0;
                int uvN = 0;
                int colN = 0;
                int faceN = 0;

                foreach (var line in coord)
                {
                    if (string.IsNullOrEmpty(line))
                    {
                        continue;
                    }

                    if (line[0] == '#') //if it is a comment (the first character is #)
                    {
                        continue; //we don't care about that
                    }
                    else if (line[0] == 'v' && line[1] == ' ') //if vector
                    {
                        vtxN++;
                    }
                    else if (line[0] == 'v' && line[1] == 't') //if normal vector
                    {
                        uvN++;
                    }
                    else if (line[0] == 'v' && line[1] == 'n') //if tex vector
                    {
                        nmlN++;
                    }
                    else if (line[0] == 'f') //if face
                    {
                        faceN++;
                    }
                }

                int propN = 0;
                if (vtxN > 0)
                {
                    propN++;
                }
                if (uvN > 0)
                {
                    propN++;
                    texcoord = pMesh.AddSource("uv", SourceType.Texture, 2);
                }
                if (nmlN > 0)
                {
                    propN++;
                    normals = pMesh.AddSource("normal", SourceType.Normal, 3);
                }
                if (colN > 0)
                {
                    propN++;
                }

                // etcetera.SetLength(vtxN,false);
                vertex.SetLength(vtxN, false);

                if (normals != null)
                {
                    normals.SetLength(nmlN, false);
                }
                if (texcoord != null)
                {
                    texcoord.SetLength(uvN, false);
                }

                //propN = pMesh.Sources.Count;
                pMesh.ResizeFaces(faceN, faceN * 3 * propN);

                vtxN = 0;
                nmlN = 0;
                uvN = 0;
                colN = 0;
                faceN = 0;

                int vofs = 0;

                char[] delim = new char[]
                    {
                        ' ', '\t'
                    };
                int vid = 0;
                //go through all of the elements of coord, and decide what kind of element is that
                foreach (var line in coord)
                {
                    if (string.IsNullOrEmpty(line))
                    {
                        continue;
                    }
                    if (line[0] == '#') //if it is a comment (the first character is #)
                    {
                        continue; //we don't care about that
                    }
                    else if (line[0] == 'v' && line[1] == ' ') //if vector
                    {
                        double tmpx, tmpy, tmpz;
                        var tokens = line.Split(delim);
                        int index = 1;
                        if (string.IsNullOrEmpty(tokens[1]))
                        {
                            index++;
                        }
                        tmpx = Convert.ToDouble(tokens[index++]);
                        tmpy = Convert.ToDouble(tokens[index++]);
                        tmpz = Convert.ToDouble(tokens[index++]);

                        MeshSourceDblCtrler.SetAsVec3(vertex, vtxN, tmpx, tmpy, tmpz);
                        //etcetera.SetRowValue( vtxN, (double )vtxN);
                        vtxN++;
                    }
                    else if (line[0] == 'v' && line[1] == 'n') //if normal vector
                    {
                        double tmpx, tmpy, tmpz;
                        var tokens = line.Split(delim);
                        int index = 1;
                        if (string.IsNullOrEmpty(tokens[1]))
                        {
                            index++;
                        }
                        tmpx = Convert.ToDouble(tokens[index++]);
                        tmpy = Convert.ToDouble(tokens[index++]);
                        tmpz = Convert.ToDouble(tokens[index++]);
                        if (normals != null)
                        {
                            MeshSourceDblCtrler.SetAsVec3(normals, nmlN, tmpx, tmpy, tmpz);
                        }
                        nmlN++;
                    }
                    else if (line[0] == 'v' && line[1] == 't') //if normal vector
                    {
                        double tmpx, tmpy;
                        var tokens = line.Split(delim);
                        int index = 1;
                        if (string.IsNullOrEmpty(tokens[1]))
                        {
                            index++;
                        }
                        tmpx = Convert.ToDouble(tokens[index++]);
                        tmpy = Convert.ToDouble(tokens[index++]);
                        if (texcoord != null)
                        {
                            MeshSourceDblCtrler.SetAsVec2(texcoord, uvN, tmpx, tmpy);
                        }
                        uvN++;
                    }
                    else if (line[0] == 'f') //if face
                    {
                        var tokens = line.Split(delim);
                        pMesh.mVnums[faceN] = 0;
                        pMesh.mVoffs[faceN] = vofs;
                        for (int vi = 1; vi < (int)tokens.Length; vi++)
                        {
                            if (string.IsNullOrEmpty(tokens[vi]))
                            {
                                continue;
                            }

                            var Indices = tokens[vi].Split('/');
                            int ii = 0;
                            foreach (var index in Indices)
                            {
                                pMesh.mVbuffs[vid * propN + ii] = Convert.ToInt32(index) - 1;
                                ii++;
                            }
                            //pMesh.mVbuffs[vid*propN + 3] = pMesh.mVbuffs[vid*propN + 0];
                            vid++;
                            pMesh.mVnums[faceN]++;
                        }
                        vofs += pMesh.mVnums[faceN];
                        faceN++;
                    }
                }
                return pMesh;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        //--------------------------------------------------------------------------------------------
        public static void SaveFaceGroupInfo(Mesh rMesh, string pFileName)
        {
            using (StreamWriter writer = new StreamWriter(pFileName))
            {
                ResizableList<int> faceGroup = rMesh.FaceMaterialShapeId;
                ResizableList<int> faceSubmesh = rMesh.mFaceSubMeshID;
                for (int i = 0; i < rMesh.FaceN; i++)
                {
                    writer.Write("{0} {1} {2}\n", i, faceGroup[i], faceSubmesh[i]);
                }
            }
        }

        //--------------------------------------------------------------------------------------------
        public static void SaveMeshToObj(Mesh rMesh, string pFileName)
        {
            if (pFileName == null)
            {
                pFileName = @"C:\depot\model\out\" + Environment.TickCount.ToString() + ".obj";
            }

            Mesh pMesh = rMesh;

            using (StreamWriter writer = new StreamWriter(pFileName))
            {
                MeshSourceBase position = pMesh.GetSource(SourceType.Position, null);
                MeshSourceBase normal = pMesh.GetSource(SourceType.Normal, null);
                MeshSourceBase uv = pMesh.GetSource(SourceType.Texture, null);

                int vtxN = position.Count;
                int nmlN = (normal == null) ? 0 : normal.Count;
                int uvN = (uv == null) ? 0 : uv.Count;
                int faceN = pMesh.FaceN;

                //-----------------------------------------------------------
                Vector3 tmpV3 = new Vector3();
                Vector2 tmpV2 = new Vector2();
                for (int vi = 0; vi < vtxN; vi++)
                {
                    MeshSourceDblCtrler.GetAsVec3(position, vi, ref tmpV3);
                    writer.WriteLine("v {0} {1} {2}", tmpV3.x, tmpV3.y, tmpV3.z);
                }

                //-----------------------------------------------------------
                for (int vi = 0; vi < nmlN; vi++)
                {
                    MeshSourceDblCtrler.GetAsVec3(normal, vi, ref tmpV3);
                    writer.WriteLine("vn {0} {1} {2}", tmpV3.x, tmpV3.y, tmpV3.z);
                }

                //-----------------------------------------------------------
                for (int vi = 0; vi < uvN; vi++)
                {
                    MeshSourceDblCtrler.GetAsVec2(uv, vi, ref tmpV2);
                    writer.WriteLine("vt {0} {1}", tmpV2.x, tmpV2.y);
                }

                //-----------------------------------------------------------
                int sn = pMesh.Sources.Count;
                int po = pMesh.GetSourceOfset(SourceType.Position, null);
                int no = pMesh.GetSourceOfset(SourceType.Normal, null);
                int to = pMesh.GetSourceOfset(SourceType.Texture, null);

                for (int fi = 0; fi < faceN; fi++)
                {
                    try
                    {
                        int vn = pMesh.mVnums[fi];
                        int vo = pMesh.mVoffs[fi];

                        writer.Write("f");
                        for (int vi = 0; vi < vn; vi++)
                        {
                            int vid = pMesh.mVbuffs[(vo + vi) * sn + po] + 1;
                            //int nid = pMesh.mVbuffs[(vo + vi)*sn + no] + 1;
                            int nid = (no != int.MaxValue) ? pMesh.mVbuffs[(vo + vi) * sn + no] + 1 : int.MaxValue;
                            int tid = (to != int.MaxValue) ? pMesh.mVbuffs[(vo + vi) * sn + to] + 1 : int.MaxValue;

                            if (vid > vtxN)
                            {
                                throw new Exception("Index over flow");
                            }
                            if (nid > nmlN && nid != int.MaxValue)
                            {
                                throw new Exception("Index over flow");
                            }
                            if (tid > uvN && tid != int.MaxValue)
                            {
                                throw new Exception("Index over flow");
                            }

                            if (tid != int.MaxValue)
                            {
                                writer.Write(" {0}/{1}/{2}", vid, tid, nid);
                            }
                            else if (nid != int.MaxValue)
                            {
                                writer.Write(" {0}/{1}", vid, nid);
                            }
                            else
                            {
                                writer.Write(" {0}", vid);
                            }
                        }
                        writer.Write(Environment.NewLine);
                    }
                    catch
                    {
                        throw;
                    }
                }
            }
        }

        //--------------------------------------------------------------------------------------------
        public static void SaveUvAtlasToObj(Mesh rMesh, string pFileName)
        {
            Mesh pMesh = rMesh;

            using (StreamWriter writer = new StreamWriter(pFileName))
            {
                MeshSourceBase uv = pMesh.GetSource(SourceType.Texture, null);

                int vtxN = uv.Count;
                int faceN = pMesh.FaceN;

                //-----------------------------------------------------------
                Vector2 tmpV2 = new Vector2();
                for (int vi = 0; vi < vtxN; vi++)
                {
                    MeshSourceDblCtrler.GetAsVec2(uv, vi, ref tmpV2);
                    writer.WriteLine("v {0} {1} {2}", tmpV2.x, tmpV2.y, 0.0);
                }

                //-----------------------------------------------------------
                int sn = pMesh.Sources.Count;
                int po = pMesh.GetSourceOfset(SourceType.Texture, null);

                for (int fi = 0; fi < faceN; fi++)
                {
                    try
                    {
                        int vn = pMesh.mVnums[fi];
                        int vo = pMesh.mVoffs[fi];

                        writer.Write("f");
                        for (int vi = 0; vi < vn; vi++)
                        {
                            int vid = pMesh.mVbuffs[(vo + vi) * sn + po] + 1;

                            if (vid > vtxN)
                            {
                                throw new Exception("Index over flow");
                            }

                            writer.Write(" {0}", vid);
                        }
                        writer.Write(Environment.NewLine);
                    }
                    catch
                    {
                        throw;
                    }
                }
            }
        }
    }
}
