﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using nw4f.meshlib;
using nw4f.tinymathlib;
using System.Diagnostics;
using nw.g3d.iflib;

// ReSharper disable PossibleIntendedRethrow
// ReSharper disable TooWideLocalVariableScope
// ReSharper disable RedundantAssignment
// ReSharper disable SuggestUseVarKeywordEvident
// ReSharper disable CompareOfFloatsByEqualityOperator
// ReSharper disable RedundantCast

namespace nw4f.meshlib
{
    public class MultiQemSimp
    {
        private QemSimp.QemSimpParam mParam = new QemSimp.QemSimpParam();
        private ResizableList<Mesh> mCloneMeshList = new ResizableList<Mesh>();
        private ICollection<Mesh> mSourceCollection = null;
        private bool mbForceRoundPos = false;

        /// <summary>
        /// 位置座標を、強制的に元のメッシュの境界で丸めたい場合
        /// </summary>
        public bool ForceRoundPos
        {
            get { return mbForceRoundPos; }
            set { mbForceRoundPos = value; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="list">メッシュリスト</param>
        /// <param name="param">リダクションパラメーター</param>
        public MultiQemSimp(ICollection<Mesh> list, QemSimp.QemSimpParam param)
        {
            mParam = param;
            mSourceCollection = list;
            mCloneMeshList = new ResizableList<Mesh>(list);
        }

        public Mesh PreExec(bool isForceMerge)
        {
            if (mParam == null) { return null; }
            if (mCloneMeshList == null) { return null; }

            //--------------------------------------------------------
            // 重複検知を行って重複がある場合には、nullで終了
            // momo : 部分結合・分離を行う場合には、検知結果を使って、このクラスの拡張で対応する
            if (!isForceMerge)
            {
                DupShapeDetector detector = new DupShapeDetector(mCloneMeshList);
                var dret = detector.Execute();
                if (dret)
                {
                    var ul = detector.UniqueDuplicateList;
                    // もとに戻す
                    mSourceCollection.Clear();
                    foreach (var mesh in mCloneMeshList)
                    {
                        mSourceCollection.Add(mesh);
                    }
                    // 重複オブジェクトを出力して、警告
                    foreach (var list in ul)
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_DupChecker_DuplicateFound_Error"));
                        foreach (var mi in list)
                        {
                            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog(mCloneMeshList[mi].MeshName + " ");
                        }
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog(".\r\n");
                    }
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_DupChecker_DuplicateFound_Warning"));
                    return null;
                }
            }

            if ((mParam.PropUseflag & (int)QemUseProp.BWT) != 0)
            {
                foreach (var mesh in mCloneMeshList)
                {
                    var blendWeightList = mesh.GetSources(SourceType.BlendWeight);
                    foreach (var source in blendWeightList)
                    {
                        ClosedPropManager pmap = new ClosedPropManager();
                        pmap.Init(mesh, source.Type, source.Name);
                        pmap.Run();
                    }
                }
            }

            // Mergeの前に法線縮退を行っておかないと、正しく分離できなくて失敗する
            for (int i = 0; i < mCloneMeshList.Count; i++)
            {
                mCloneMeshList[i].ShrinkVertexNormals(null);
            }
            //--------------------------------------------------------
            // merge all input shapes if they can merge.
            ShapeMerger marger = new ShapeMerger(mCloneMeshList);
            Mesh margedMesh = marger.Execute();
            return margedMesh;
        }

        public PolygonReductionProfiler.ProfileResult Profile(PolygonReductionProfiler.ProfileOptions profilerOption, bool isForceMerge)
        {
            if (mParam == null) { return null; }
            if (mCloneMeshList == null) { return null; }

            Mesh margedMesh = PreExec(isForceMerge);
            if (margedMesh == null)
            {
                return null;
            }

            var profiler = new PolygonReductionProfiler();
            return profiler.Profile(margedMesh, mParam, profilerOption);
        }

        /// <summary>
        /// リダクションを実行する
        /// リストを結合して、リダクションし、分割する
        /// </summary>
        /// <returns>分割されたメッシュリスト</returns>
        public List<Mesh> Execute(bool isForceMerge = false)
        {
            if (mParam == null) { return null; }
            if (mCloneMeshList == null) { return null; }
            //--------------------------------------------------------
            // 実行前の順番を記録
            Dictionary<string, int> KeydIndexList = new Dictionary<string, int>();
            for (int i = 0; i < mCloneMeshList.Count; i++)
            {
                Mesh mesh = mCloneMeshList[i];
                mesh.ComputeBounds();
                KeydIndexList[mesh.MeshName] = i;
            }

            Mesh margedMesh = PreExec(isForceMerge);
            if (margedMesh == null)
            {
                mSourceCollection.Clear();
                foreach (var mesh in mCloneMeshList)
                {
                    //mergeに失敗した場合、ここで法線の分離をしないとコンバートに失敗する
                    var rearranger = new FaceAttributeRearrange(mesh);
                    rearranger.Run();
                    mSourceCollection.Add(mesh);
                }
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Process", "ShapeMerge"));
                return null;
            }
            else
            {
                QemSimp qemSimp = new QemSimp();
                qemSimp.Params = mParam;

                // simplify merged mesh.
                if (qemSimp.Initialize(margedMesh) == false)
                {
                    throw ExcepHandle.CreateException("qemSimp.Initialize failed");
                }

                bool[] moved;
                bool[] deleted;
                qemSimp.Execute(out moved, out deleted);

                // splitting simplified mesh as shapes.
                ShapeSplitter splitter = new ShapeSplitter(margedMesh, mCloneMeshList);
                Dictionary<string, Mesh> dictioary = splitter.Execute();
                if (dictioary == null)
                {
                    throw ExcepHandle.CreateException("ShapeSplitter.Execute failed");
                }

                // 元々貰った順番に入れ替えて返す準備
                var tmpList = new nw4f.meshlib.Mesh[KeydIndexList.Count];
                foreach (var executedMesh in dictioary)
                {
                    int index = KeydIndexList[executedMesh.Value.MeshName];
                    tmpList[index] = executedMesh.Value;
                }

                // 全ての処理を終えた後に、バウンディングオブジェクトが一致しているか
                // などのチェックを行います
                List<Mesh> meshes = new List<Mesh>();
                for (int i = 0; i < tmpList.Length; i++)
                {
                    try
                    {
                        //強制的にUVを丸める。特に境界付近では動きがち
                        mCloneMeshList[i].ComputeBounds();
                        if (mbForceRoundPos)
                        {
                            tmpList[i].ForceRoundingPosInBBoxes(mCloneMeshList[i].BBoxes);
                        }

                        tmpList[i].ForceRoundingUvInBSquares(mCloneMeshList[i].UVBSquares);

                        tmpList[i].ComputeBounds();
                        if (mbForceRoundPos)
                        {
                            Mesh.TestCompareBBoxes(mCloneMeshList[i].BBoxes, tmpList[i].BBoxes);
                        }
                        meshes.Add(tmpList[i]);
                    }
                    catch
                    {
                        throw;
                    }
                }
                return meshes;
            }
        }
    }
}
