﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib;
using nw4f.tinymathlib;
using Vector2 = nw4f.tinymathlib.Vector2;
using Vector3 = nw4f.tinymathlib.Vector3;
using Vector4 = nw4f.tinymathlib.Vector4;

namespace nw4f.meshlib
{
    // [頂点インデックス][法線ID][UV ID][種類(Tangent / Binormal)]
    using VertexTangentsAndBinormals = ResizableList<Dictionary<int, Dictionary<int, Dictionary<SourceType, Vector3>>>>;

    /// <summary>
    /// メッシュクラス
    /// </summary>
    public partial class Mesh
    {
        public static double minimalDistForProp { get; set; } = 0.0001;

        /// <summary>
        /// サブメッシュの情報
        /// </summary>
        public class SubMeshInfo
        {
            public static bool IsNull(SubMeshInfo info)
            {
                return info.meshIndex <= 0 && info.subMeshindex <= 0 && info.subMeshOffset <= 0 &&
                       info.subMeshCount <= 0;
            }

            public static SubMeshInfo Null
            {
                get
                {
                    return new SubMeshInfo() { meshIndex = -1, subMeshindex = 0, subMeshCount = 0, subMeshOffset = 0 };
                }
            }

            public int meshIndex { get; internal set; }
            public int subMeshindex { get; internal set; }
            public int subMeshOffset { get; internal set; }
            public int subMeshCount { get; internal set; }
        }

        /// <summary>
        /// Shapeグループの情報（NWでいうところのShape単位）
        /// </summary>
        public class MaterialShapeInfo
        {
            public MaterialShapeInfo()
            {
                IsExcludeForSimplification = false;
            }

            public int Index { get; set; }
            public string ShapeName { get; set; }
            public bool IsExcludeForSimplification { get; set; }
            public List<Tuple<SourceType, string>> refSources { get; set; } = new List<Tuple<SourceType, string>>();
        }

        /// <summary>
        /// 連結フェース情報
        /// </summary>
        public class ConnectedShapeInfo : MaterialShapeInfo
        {
            /// <summary>
            /// 連結形状が所属してる、Shapeのリスト
            /// </summary>
            public List<int> MaterialShapeIndeces { get; set; }
        }

        /// <summary>
        /// メッシュの名前(識別用、任意利用)
        /// </summary>
        private string mName = "name";

        private BoundingBox mTotalBoudingBox = new BoundingBox();

        /// <summary>
        /// 境界ボリューム
        /// </summary>
        private Dictionary<int, BoundingBox> mBBoxes = new Dictionary<int, BoundingBox>();

        public Dictionary<int, BoundingBox> BBoxes
        {
            get { return mBBoxes; }
        }

        /// <summary>
        /// UVの境界ボリューム
        /// </summary>
        private Dictionary<int, List<BoundingSquare>> mUVBSquares = new Dictionary<int, List<BoundingSquare>>();

        public Dictionary<int, List<BoundingSquare>> UVBSquares
        {
            get { return mUVBSquares; }
        }

        /// <summary>
        /// 位置・テクスチャなどのソースのリスト
        /// </summary>
        private List<MeshSourceBase> _sources = new List<MeshSourceBase>();

        /// <summary>
        /// 各フェースの頂点数を保持
        /// </summary>
        public ResizableList<int> mVnums { get; set; } = new ResizableList<int>();

        /// <summary>
        /// 各フェースの頂点バッファへのオフセット( アドレス
        /// </summary>
        public ResizableList<int> mVoffs { get; set; } = new ResizableList<int>();

        /// <summary>
        /// 各フェースの頂点バッファ
        /// </summary>
        public ResizableList<int> mVbuffs { get; set; } = new ResizableList<int>();

        /// <summary>
        /// 面法線( フェース付属プロパティ )
        /// </summary>
        internal ResizableList<Vector3> mFaceNormal { get; set; } = new ResizableList<Vector3>();

        /// <summary>
        /// 面の色( フェース付属プロパティ )
        /// </summary>
        internal ResizableList<Vector4> mFaceColor { get; set; } = new ResizableList<Vector4>();

        /// <summary>
        /// 面の有効無効を指定するフラグ( フェース付属プロパティ )
        /// </summary>
        private ResizableList<bool> mFaceInvalid = new ResizableList<bool>();

        /// <summary>
        /// 面毎のサブメッシュ( フェース付属プロパティ )
        /// </summary>
        internal ResizableList<int> mFaceSubMeshID { get; set; } = new ResizableList<int>();

        /// <summary>
        /// 面毎のShape等の任意の組み分け設定( フェース付属プロパティ )
        /// </summary>
        internal ResizableList<int> _FaceMaterialShapeID { get; set; } = new ResizableList<int>();

        /// <summary>
        /// 面毎の連結形状単位での組み分け設定( フェース付属プロパティ )
        /// </summary>
        internal ResizableList<int> _FaceConnectedShapeID { get; set; } = new ResizableList<int>();

        /// <summary>
        /// 各面積( フェース付属プロパティ )
        /// </summary>
        internal ResizableList<double> mFaceArea { get; set; } = new ResizableList<double>();

        internal ResizableList<int> mSubmeshCounter { get; set; } = new ResizableList<int>();
        internal ResizableList<int> _MaterialShapeFaceCounter { get; set; } = new ResizableList<int>();
        internal ResizableList<int> _ConnectedShapeFaceCounter { get; set; }  = new ResizableList<int>();

        /// <summary>
        /// 接線、従法線、法線が右手座標系 (UV ワインド順序が時計回り、前向き) であるかどうかを取得します。
        /// </summary>
        /// <param name="tangent"></param>
        /// <param name="binormal"></param>
        /// <param name="normal"></param>
        /// <returns>右手座標系の場合に true 、左手座標系の場合に false を返します。</returns>
        private bool IsRightHandTangent(Vector3 tangent, Vector3 binormal, Vector3 normal)
        {
            return tangent.Cross(binormal).Dot(normal) >= 0.0f;
        }

        public Mesh()
        {
            CreateFaceConnectedMeshInfo();
        }

        /// <summary>
        /// バウンディングボックスのサイズを比較する
        /// </summary>
        /// <param name="left"></param>
        /// <param name="right"></param>
        [Conditional("DEBUG")]
        internal static void TestCompareBBoxes(Dictionary<int, BoundingBox> left, Dictionary<int, BoundingBox> right)
        {
            if (left.Count != right.Count)
            {
                throw new Exception("TestCompareBBoxes:: the count of bouding box is different");
            }

            int sizeN = left.Count;

            for (int i = 0; i < sizeN; i++)
            {
                KeyValuePair<int, BoundingBox> lnode = left.ElementAt(i);
                KeyValuePair<int, BoundingBox> rnode = right.ElementAt(i);
                if (!lnode.Value.Inside(rnode.Value))
                {
                    throw new Exception("TestCompareBBoxes :: the right is bigger than the left.");
                }
            }
        }

        /// <summary>
        /// バウンディングボックスのサイズを比較する
        /// </summary>
        /// <param name="left"></param>
        /// <param name="right"></param>
        [Conditional("DEBUG")]
        internal static void TestCompareUvBSquares(Dictionary<int, List<BoundingSquare>> left,
            Dictionary<int, List<BoundingSquare>> right)
        {
            if (left.Count != right.Count)
            {
                throw new Exception("TestCompareUvBSquares:: the count of bouding box(2D) list is different.");
            }

            int sizeN = left.Count;

            for (int i = 0; i < sizeN; i++)
            {
                KeyValuePair<int, List<BoundingSquare>> lnode = left.ElementAt(i);
                KeyValuePair<int, List<BoundingSquare>> rnode = right.ElementAt(i);

                if (lnode.Value.Count != rnode.Value.Count)
                {
                    throw new Exception("TestCompareUvBSquares::  the count of bouding box(2D) list is different.");
                }

                int sizeN2 = lnode.Value.Count;

                for (int j = 0; j < sizeN2; j++)
                {
                    BoundingSquare lSquare = lnode.Value.ElementAt(j);
                    BoundingSquare rSquare = rnode.Value.ElementAt(j);

                    if (!lSquare.Inside(rSquare))
                    {
                        throw new Exception("TestCompareUvBSquares :: right is bigger than the left.");
                    }
                }
            }
        }

        /// <summary>
        /// 各フェースのFaceShapeIdを取得。（Shapeなどの分離に
        /// </summary>
        public ResizableList<int> FaceMaterialShapeId
        {
            get { return _FaceMaterialShapeID; }
        }

        /// <summary>
        /// 各フェースのFaceShapeIdを取得。（Shapeなどの分離に
        /// </summary>
        public ResizableList<int> FaceConnectedShapeID
        {
            get { return _FaceConnectedShapeID; }
        }

        /// <summary>
        /// 各フェースのSubMeshIDを取得。（SubMeshなどの分離に
        /// </summary>
        public ResizableList<int> FaceSubMeshID
        {
            get { return mFaceSubMeshID; }
        }

        /// <summary>
        /// サブメッシュ情報
        /// Sequentialに並んでないのでDictionary構造にする
        /// </summary>
        internal SortedDictionary<int, Mesh.SubMeshInfo> mSubMeshInfo { get; set; } = new SortedDictionary<int, Mesh.SubMeshInfo>();

        /// <summary>
        /// Shape情報
        /// </summary>
        internal ResizableList<MaterialShapeInfo> _MaterialShapeInfoList { get; set; } = new ResizableList<MaterialShapeInfo>();

        /// <summary>
        /// サブメッシュ情報
        /// </summary>
        public SortedDictionary<int, Mesh.SubMeshInfo> SubMeshInformationList
        {
            get { return mSubMeshInfo; }
        }
        /// <summary>
        /// ConnectedShape情報
        /// </summary>
        internal ResizableList<ConnectedShapeInfo> ConnectedShapeInfomationList { get; set; }

        /// <summary>
        /// サブメッシュ情報
        /// </summary>
        public ResizableList<MaterialShapeInfo> MaterialShapeInformationList
        {
            get { return _MaterialShapeInfoList; }
            set { _MaterialShapeInfoList = value; }
        }

        /// <summary>
        ///
        /// </summary>
        /// <returns></returns>
        public BoundingBox GetTotalBoundingBox()
        {
            return mTotalBoudingBox;
        }

        /// <summary>
        /// ソースにアクセス
        /// </summary>
        public string MeshName
        {
            get { return mName; }
            set { mName = value; }
        }

        /// <summary>
        /// ソースにアクセス
        /// </summary>
        public List<MeshSourceBase> Sources
        {
            get { return _sources; }
        }

        /// <summary>
        /// 各フェースの頂点数のリスト
        /// </summary>
        public ResizableList<int> VNums
        {
            get { return mVnums; }
        }

        /// <summary>
        /// 各フェースの頂点数オフセットリストを取得
        /// </summary>
        public ResizableList<int> VOffset
        {
            get { return mVoffs; }
        }

        /// <summary>
        /// 各フェースの頂点バッファを取得
        /// </summary>
        public ResizableList<int> VBuffer
        {
            get { return mVbuffs; }
        }

        /// <summary>
        /// 全フェース数を取得
        /// </summary>
        public int FaceN
        {
            get { return mVnums.Count; }
        }

        /// <summary>
        /// 全頂点数を取得
        /// </summary>
        public int VertexN
        {
            get
            {
                MeshSourceBase poslist = GetSource(SourceType.Position, null);
                return poslist.Count;
            }
        }

        /// <summary>
        /// 全フェース数を取得
        /// </summary>
        public int SourceN
        {
            get { return Sources.Count; }
        }

        /// <summary>
        /// バウンディングボックス
        /// </summary>
        public BoundingBox GetBoundingBox(int fgid)
        {
            if (!mBBoxes.ContainsKey(fgid))
            {
                throw ExcepHandle.CreateException(fgid.ToString() + " is not a key of mBBoxes.");
            }
            if (!mBBoxes.ContainsKey(fgid))
            {
                throw ExcepHandle.CreateException("Bouding box doesnt exit for fgid : " + fgid.ToString());
            }
            return mBBoxes[fgid];
        }

        /// <summary>
        /// 指定のテクスチャ座標の境界ボリュームを取得
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>境界ボリューム</returns>
        public BoundingSquare GetUvBound(int fgid, int index)
        {
            if (!mUVBSquares.ContainsKey(fgid))
            {
                throw ExcepHandle.CreateException(fgid.ToString() + " is not a key of mUVBSquares.");
            }

            if (mUVBSquares[fgid].Count > index)
            {
                return mUVBSquares[fgid][index];
            }
            return null;
        }

        /// <summary>
        /// 指定のUV空間内での境界を取得
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public BoundingSquare GetTotalUvBound(int index)
        {
            BoundingSquare ret = new BoundingSquare();
            bool hit = false;
            ret.Reset();
            foreach (var uvbs in mUVBSquares)
            {
                if (uvbs.Value.Count > index)
                {
                    hit = true;
                    BoundingSquare tmp = uvbs.Value[index];
                    ret.Update(tmp.Min);
                    ret.Update(tmp.Max);
                }
            }
            if (!hit)
            {
                ret.Max.SetZero();
                ret.Min.SetZero();
            }
            return ret;
        }

        /// <summary>
        /// 指定のキーのShapeIndexを取得します
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public MaterialShapeInfo GetShapeInfo(string key)
        {
            foreach (var shapeInfo in _MaterialShapeInfoList)
            {
                if (shapeInfo.ShapeName == key)
                {
                    return shapeInfo;
                }
            }
            return null;
        }

        /// <summary>
        /// 一つのサブメッシュでメッシュ全体が設定されているか？
        /// </summary>
        /// <returns></returns>
        public bool CoveredMeshbyASubMesh()
        {
            if (mSubMeshInfo.Count > 1) { return false; }
            if (mSubMeshInfo.Count == 0) { return true; }

            if (mSubMeshInfo.First().Value.subMeshCount == FaceN) { return true; }
            throw ExcepHandle.CreateException("Submesh information is not legal.");
        }

        /// <summary>
        /// サブメッシュの情報を設定します
        /// </summary>
        /// <param name="offset">サブメッシュの開始フェースインデックス</param>
        /// <param name="count">所属するフェースの数</param>
        public void SetSubMeshInfo(int subMeshId, int offset, int count)
        {
            int sizeN = mFaceSubMeshID.Count;
            if (offset >= sizeN)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Index_Over_Flow", "SetSubMeshInfo"));
                return;
            }

            if ((offset + count) > sizeN)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Index_Over_Flow", "SetSubMeshInfo"));
                return;
            }

            // サブメッシュのIDを設定します
            for (int i = offset; i < offset + count; i++)
            {
                mFaceSubMeshID[i] = subMeshId;
            }
            mSubMeshInfo[subMeshId] = new SubMeshInfo();
            mSubMeshInfo[subMeshId].subMeshOffset = offset;
            mSubMeshInfo[subMeshId].subMeshCount = count;
            mSubMeshInfo[subMeshId].subMeshindex = subMeshId;
        }

        /// <summary>
        /// サブメッシュの情報を取得します
        /// </summary>
        /// <param name="subMeshId">指定のサブメッシュのインデックス</param>
        /// <param name="offset">オフセット</param>
        /// <param name="count">所属するフェースの数</param>
        public void GetSubMeshInfo(int subMeshId, ref int offset, ref int count)
        {
            foreach (var smi in mSubMeshInfo)
            {
                if (smi.Value.subMeshindex == subMeshId)
                {
                    offset = smi.Value.subMeshOffset;
                    count = smi.Value.subMeshCount;
                    break;
                }
            }
        }

        /// <summary>
        /// サブメッシュに含まれるフェースの数を取得
        /// </summary>
        public void GetSubMeshCount()
        {
            ReComputeSubMeshArray();

            mSubmeshCounter = new ResizableList<int>();
            mSubmeshCounter.Resize(mSubMeshInfo.Count, 0);
            foreach (var smi in mSubMeshInfo)
            {
                mSubmeshCounter[smi.Key] = smi.Value.subMeshCount;
            }
        }

        /// <summary>
        /// MaterialShapeを追加する
        /// </summary>
        /// <param name="index"></param>
        /// <param name="name"></param>
        public void AddMaterialShapeInfo(int index, string shapename, List<MeshSourceBase> sources)
        {
            MaterialShapeInfo newSI = new MaterialShapeInfo();
            newSI.Index = index;
            newSI.ShapeName = shapename;
            foreach (var meshSourceBase in sources)
            {
                newSI.refSources.Add(Tuple.Create(meshSourceBase.Type, meshSourceBase.Name));
            }
            MaterialShapeInformationList.Add(newSI);
        }

        /// <summary>
        /// 既定のShapeIDを設定する
        /// </summary>
        public void CoveredMeshbyMaterialShape(int index)
        {
            for (int i = 0; i < FaceN; i++)
            {
                _FaceMaterialShapeID[i] = _MaterialShapeInfoList[index].Index;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void GetMaterialShapeFaceCounter()
        {
            int maxN = 0;
            foreach (var fgi in _FaceMaterialShapeID)
            {
                maxN = Math.Max(maxN, fgi);
            }
            _MaterialShapeFaceCounter = new ResizableList<int>();
            _MaterialShapeFaceCounter.Resize(maxN + 1, 0);
            foreach (var fgi in _FaceMaterialShapeID)
            {
                _MaterialShapeFaceCounter[fgi]++;
            }
        }

        /// <summary>
        /// 指定FaceGroupのFaceを一括で取得する
        /// </summary>
        /// <param name="dst"></param>
        /// <param name="faceGroupList"></param>
        public void GetFacesFromGroupIndex(int dst, List<int> faceGroupList)
        {
            int fi = 0;
            foreach (var fgid in _FaceMaterialShapeID)
            {
                if (fgid == dst)
                {
                    faceGroupList.Add(fi);
                }
                fi++;
            }
        }

        /// <summary>
        /// 各ポリゴンのプロパティをスワップする
        /// </summary>
        /// <param name="dst"></param>
        /// <param name="src"></param>
        private void SwapFaceProps(int dst, int src)
        {
            try
            {
                Vector3 tmpv3 = mFaceNormal[dst];
                Vector4 tmpv4 = mFaceColor[dst];
                bool tmpb = mFaceInvalid[dst];
                int tmpI32 = mFaceSubMeshID[dst];

                mFaceNormal[dst] = mFaceNormal[src];
                mFaceNormal[src] = tmpv3;

                mFaceColor[dst] = mFaceColor[src];
                mFaceColor[src] = tmpv4;

                mFaceInvalid[dst] = mFaceInvalid[src];
                mFaceInvalid[src] = tmpb;

                mFaceSubMeshID[dst] = mFaceSubMeshID[src];
                mFaceSubMeshID[src] = tmpI32;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Swap face prop failed:" + ex);
            }
        }

        /// <summary>
        /// 指定のサブメッシュ以外を全て消去
        /// </summary>
        /// <param name="smid"></param>
        public void SetAllFaceInvalidWithoutSubmesh(int smid)
        {
            for (int fi = 0; fi < FaceN; fi++)
            {
                if (mFaceSubMeshID[fi] != smid)
                {
                    mFaceInvalid[fi] = true;
                }
            }
        }

        /// <summary>
        /// ソースの追加
        /// </summary>
        /// <param name="name">名前</param>
        /// <param name="type">種別</param>
        /// <param name="stride">１データ長、例えばVector3なら３</param>
        /// <returns>追加したリソース</returns>
        public MeshSourceBase AddSource(string name, SourceType type, int stride)
        {
            try
            {
                MeshSourceBase source = null;
                switch (type)
                {
                    case SourceType.EtceteraInt:
                        source = new MeshSource<int>(name, type, stride);
                        _sources.Add(source);
                        break;
                    case SourceType.EtceteraDouble:
                        source = new MeshSource<double>(name, type, stride);
                        _sources.Add(source);
                        break;
                    default:
                        source = new MeshSource<double>(name, type, stride);
                        _sources.Add(source);
                        break;
                }

                // UVを加えた場合は、境界矩形も同時に追加する
                if (type == SourceType.Texture)
                {
                    int shCnt = 0;
                    foreach (var shapeInfo in _MaterialShapeInfoList)
                    {
                        shCnt = shapeInfo.Index;

                        int key = shCnt;
                        if (!mUVBSquares.ContainsKey(key))
                        {
                            mUVBSquares.Add(key, new List<BoundingSquare>());
                        }
                        mUVBSquares[key].Add(new BoundingSquare());
                        shCnt++;
                    }
                }
                return source;
            }
            catch (Exception exp)
            {
                throw exp;
            }
        }

        /// <summary>
        /// 指定のソースを削除する
        /// </summary>
        /// <param name="name"></param>
        /// <param name="type"></param>
        public void RemobeSource(SourceType type, string name)
        {
            _sources.RemoveAll(o => (o.Name == name || name == null) && o.Type == type);
        }

        /// <summary>
        /// 同一タイプのソースをリストで取得します
        /// </summary>
        /// <param name="type">取得したいソースの種別</param>
        /// <returns> ソースリスト </returns>
        public MeshSourceBase GetSource(SourceType type, string name)
        {
            try
            {
                foreach (MeshSourceBase source in _sources)
                {
                    if (source.Type == type)
                    {
                        if (name == null || name == source.Name)
                        {
                            return source;
                        }
                    }
                }
                return null;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// 同一タイプのソースをリストで取得します
        /// </summary>
        /// <param name="type">取得したいソースの種別</param>
        /// <returns> ソースリスト </returns>
        public int GetPropDimension()
        {
            try
            {
                int dim = 0;
                foreach (MeshSourceBase source in _sources)
                {
                    dim += source.Stride;
                }

                return dim;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// 同一タイプのソースをリストで取得します
        /// </summary>
        /// <param name="type">取得したいソースの種別</param>
        /// <returns> ソースリスト </returns>
        public List<MeshSourceBase> GetSources(SourceType type)
        {
            try
            {
                List<MeshSourceBase> result = new List<MeshSourceBase>();

                foreach (MeshSourceBase source in _sources)
                {
                    if (source.Type == type)
                    {
                        result.Add(source);
                    }
                }

                return result;
            }
            catch (Exception ex)
            {
                ExcepHandle.CreateException(ex);
                throw;
            }
        }

        /// <summary>
        /// 指定されたソースのインデックスを取得します
        /// </summary>
        /// <param name="type">種類</param>
        /// <param name="name">名前</param>
        /// <returns>インデックス</returns>
        public int GetSourceOfset(SourceType type, string name)
        {
            try
            {
                int index = 0;
                foreach (MeshSourceBase source in _sources)
                {
                    if (source.Type == type)
                    {
                        if (name == null || name == source.Name)
                        {
                            return index;
                        }
                    }
                    index++;
                }
                return int.MaxValue;
            }
            catch (Exception exp)
            {
                throw exp;
            }
        }

        /// <summary>
        /// 指定のフェースを無効なフェースとして処理
        /// </summary>
        /// <param name="fid">無効にするフェースID</param>
        public void InvalidateFace(int fid)
        {
            if (!mFaceInvalid[fid])
            {
                int smid = mFaceSubMeshID[fid];
                if (smid < mSubmeshCounter.Count && smid != int.MaxValue)
                {
                    if (mSubmeshCounter[smid] > 0)
                    {
                        mSubmeshCounter[smid]--;
                    }
                    mFaceSubMeshID[fid] = int.MaxValue;
                }

                if (_FaceMaterialShapeID.Count > fid)
                {
                    // MaterialShape情報を更新
                    int msId = _FaceMaterialShapeID[fid];
                    if (msId < _MaterialShapeFaceCounter.Count && msId != int.MaxValue)
                    {
                        if (_MaterialShapeFaceCounter[msId] > 0)
                        {
                            _MaterialShapeFaceCounter[msId]--;
                        }
                        _FaceMaterialShapeID[fid] = int.MaxValue;
                    }
                }
                if (_FaceConnectedShapeID.Count > fid)
                {
                    // ConnectedShape情報を更新
                    int csId = _FaceConnectedShapeID[fid];
                    if (csId < _ConnectedShapeFaceCounter.Count && csId != int.MaxValue)
                    {
                        if (_ConnectedShapeFaceCounter[csId] > 0)
                        {
                            _ConnectedShapeFaceCounter[csId]--;
                        }
                        _FaceConnectedShapeID[fid] = int.MaxValue;
                    }
                }
            }
            mFaceInvalid[ fid] = true;
        }

        public void ValidateFace(int fid)
        {
            mFaceInvalid[fid] = false;
        }

        /// <summary>
        /// 指定のフェースが無効かどうか？を取得します
        /// </summary>
        /// <param name="fid">フェースID </param>
        /// <returns>true : 無効なフェース false : 有効なフェース</returns>
        public bool IsInvalidateFace(int fid)
        {
            return mFaceInvalid[fid];
        }

        /// <summary>
        /// メッシュの表面積を計算して取得します
        /// </summary>
        public double ComputeSurfaceArea()
        {
            double area = 0.0;
            int fn = FaceN;
            for (int f = 0; f < fn; f++)
            {
                if (!IsInvalidateFace(f))
                {
                    area += GetFaceArea(f);
                }
            }
            return area;
        }

        /// <summary>
        /// 指定のフェースの面積を取得します
        /// </summary>
        /// <param name="f"></param>
        /// <returns></returns>
        public double GetFaceArea(int f)
        {
            return mFaceArea[f];
        }

        /// <summary>
        /// 三点からなる面の法線を計算
        /// </summary>
        public static Vector3 CalculateTriangleNormal(Vector3 pos1, Vector3 pos2, Vector3 pos3)
        {
            var normals = new Vector3[3];

            var checkNormal = new Vector3(0, 0, 0);
            normals[0] = (pos1 - pos3).Cross(pos2 - pos1);
            checkNormal += normals[0];

            normals[1] = (pos3 - pos2).Cross(pos1 - pos3);
            checkNormal += normals[1];

            normals[2] = (pos2 - pos1).Cross(pos3 - pos2);
            checkNormal += normals[2];

            var sumNormal = normals.Aggregate((sum, next) =>
            {
                // 法線が逆向きな場合を補正
                if (checkNormal.Dot(next) < 0.0)
                {
                    next = -next;
                }
                return sum + next;
            });

            sumNormal.Normalize();

            return sumNormal;
        }

        /// <summary>
        /// 指定の面の法線と面積を計算する
        /// </summary>
        public Vector3 CalculateTriangleNormal(int fid)
        {
            var ids = new int[3];
            var pos = new Vector3[3];
            GetFaceVtxIndices(fid, ref ids);
            for (var vi = 0; vi < 3; vi++)
            {
                GetVertexPos(ids[vi], ref pos[vi]);
            }
            var cross = CalculateTriangleNormal(pos[0], pos[1], pos[2]);
            mFaceNormal[fid] = cross;
            return mFaceNormal[fid];
        }

        /// <summary>
        /// 指定のフェースの面積を計算します
        /// </summary>
        /// <param name="f"></param>
        /// <returns></returns>
        public double ComputeFaceArea(int f)
        {
            if (mVnums[f] != 3)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Illegal_Area_Computation"));
            }

            VectorN[] plist = new VectorN[mVnums[f]];
            int dim = GetPropDimension();
            for (int i = 0; i < mVnums[f]; i++)
            {
                plist[i] = new VectorN((uint)dim);
            }

            GetFaceVertex(f, ref plist);

            Vector3 pos0 = new Vector3(plist[0][0], plist[0][1], plist[0][2]);
            Vector3 pos1 = new Vector3(plist[1][0], plist[1][1], plist[1][2]);
            Vector3 pos2 = new Vector3(plist[2][0], plist[2][1], plist[2][2]);

            Vector3 e1 = pos1 - pos0;
            Vector3 e2 = pos2 - pos0;
            Vector3 n = e2.Cross(e1);
            mFaceArea[f] = n.Norm() * 0.5;
            return mFaceArea[f];
        }

        /// <summary>
        ///  値が一致する法線を同一IDになる様にまとめる
        ///
        /// (注)メッシュ全域でやると影響が大きいので、あまり使わない方が良いかも。
        /// その代わりに、VTriMeshに周辺近傍での一致マップ関数がある
        /// </summary>
        /// <returns></returns>
        public int[] ComputeNormalCoincidentMap()
        {
            MeshSourceBase nmlSource = GetSource(SourceType.Normal, null);
            var coincidentNmlIndex = new int[nmlSource.Count];
            for (int ni = 0; ni < nmlSource.Count; ni++)
            {
                coincidentNmlIndex[ni] = ni;
            }

            var hashMap = new Dictionary<long, List<int>>();

            // 総当たり検索O(n^2)では重たいのでハッシュを作る
            for (int ni = 0; ni < nmlSource.Count; ni++)
            {
                Vector3 tmpI = new Vector3();
                MeshSourceDblCtrler.GetAsVec3(nmlSource, ni, ref tmpI);
                var hash = SpacialHash.hash_func(tmpI);

                if (!hashMap.ContainsKey(hash))
                {
                    hashMap.Add(hash, new List<int>());
                }
                hashMap[hash].Add(ni);
            }

            // 重複がある場合は、一つにまとめる
            foreach (var kvp in hashMap)
            {
                var normalIndeces = kvp.Value;
                if (normalIndeces.Count() > 1)
                {
                    normalIndeces.Sort();
                    var repIndex = normalIndeces.First();
                    foreach (var normalIndex in normalIndeces)
                    {
                        coincidentNmlIndex[normalIndex] = repIndex;
                    }
                }
            }

            return coincidentNmlIndex;
        }

        /// <summary>
        /// 各面の各頂点ごとに分解してバラバラにした法線を同一値を持つものに纏めなおす
        /// インデックス縮退はさせても、ストリームの内容は変更しない
        /// </summary>
        /// <param name="vtxFaceIterator"></param>
        public void ShrinkVertexNormals(VtxTriMesh vtxFaceIterator)
        {
            if (vtxFaceIterator == null)
            {
                vtxFaceIterator = new VtxTriMesh();
                vtxFaceIterator.Create(this);
            }

            int sn = Sources.Count;
            int po = GetSourceOfset(SourceType.Position, null);
            int no = GetSourceOfset(SourceType.Normal, null);
            MeshSourceBase posSource = GetSource(SourceType.Position, null);
            MeshSourceBase nmlSource = GetSource(SourceType.Normal, null);
            if (nmlSource == null || posSource == null) { return; }

            // 頂点の周辺ポリゴンごとで面法線の計算を行う
            for (int vtxIndex = 0; vtxIndex < VertexN; vtxIndex++)
            {
                var coincidentNmlIndex = vtxFaceIterator.ComouteNormalCoincedentMap(vtxIndex);
                // 頂点ごとの法線をスムーシングされてる範囲を取得
                foreach (var faceId in vtxFaceIterator.VTL[vtxIndex].TriList)
                {
                    int vn = mVnums[faceId];
                    int vo = mVoffs[faceId];

                    var hitIndex = -1;
                    Vector3[] pos = new Vector3[vn];
                    for (int triVi = 0; triVi < vn; triVi++)
                    {
                        var localVtxIndex = VBuffer[(sn * (vo + triVi)) + po];
                        pos[triVi] = new Vector3();
                        MeshSourceDblCtrler.GetAsVec3(posSource, localVtxIndex, ref pos[triVi]);
                        if (localVtxIndex == vtxIndex)
                        {
                            hitIndex = triVi;
                        }
                    }
                    if (hitIndex != -1)
                    {
                        var ni = coincidentNmlIndex[VBuffer[(sn * (vo + hitIndex)) + no]];
                        VBuffer[(sn * (vo + hitIndex)) + no] = ni;
                    }
                }
            }
        }

        /// <summary>
        /// 頂点法線を計算しなおす
        /// 簡略化に備えて、各面の各頂点ごとに分解する
        /// ストリーム長の変更は行わない
        /// </summary>
        /// <param name="vtxFaceIterator"></param>
        public void RecomputeVertexNormals(VtxTriMesh vtxFaceIterator)
        {
            if (vtxFaceIterator == null)
            {
                vtxFaceIterator = new VtxTriMesh();
                vtxFaceIterator.Create(this);
            }

            int nFaces = FaceN;

            MeshSourceBase posSource = GetSource(SourceType.Position, null);
            MeshSourceBase nmlSource = GetSource(SourceType.Normal, null);
            if (nmlSource == null || posSource == null) { return; }

            // 重複解除する分、バッファを拡大する
            var newNormals = new Vector3[nFaces * 3];
            var newNormalIndeces = new int[nFaces * 3];

            for (int i = 0; i < nFaces * 3; i++)
            {
                newNormalIndeces[i] = i;
                newNormals[i] = new Vector3();
            }

            // 頂点の周辺ポリゴンごとで面法線の計算を行う
            for (int vtxIndex = 0; vtxIndex < VertexN; vtxIndex++)
            {
                vtxFaceIterator.ComputeVtxNormals(vtxIndex, null);
            }
        }

        /// <summary>
        /// 全面法線と面積の計算を行います
        /// </summary>
        public void RecomputeFaceNormalsAndArea()
        {
            int nFaces = FaceN;
            int maxN = Sources.Count;
            MeshSourceBase PosSource = GetSource(SourceType.Position, null);
            int posOffs = GetSourceOfset(SourceType.Position, null);

            if (mFaceNormal.Count < nFaces)
            {
                mFaceNormal.Resize(nFaces, default(Vector3));
            }

            Vector3 crs = new Vector3();
            for (int fi = 0; fi < nFaces; fi++)
            {
                int vn = mVnums[fi];
                int vo = mVoffs[fi];
                Vector3[] v = new Vector3[vn];
                for (int vi = 0; vi < vn; vi++)
                {
                    int pi = (maxN * (vo + vi)) + posOffs;
                    int posIdx = mVbuffs[pi];
                    v[vi] = new Vector3();
                    MeshSourceDblCtrler.GetAsVec3(PosSource, posIdx, ref v[vi]);
                }
                if (vn >= 3)
                {
                    Vector3 e1 = v[1] - v[0];
                    Vector3 e2 = v[2] - v[0];

                    crs.SetZero();
                    if (e1.Norm() > double.Epsilon && e2.Norm() > double.Epsilon)
                    {
                        crs = e1.Cross(e2);
                        mFaceArea[fi] = crs.Norm() * 0.5;
                    }
                    mFaceNormal[fi] = Mesh.CalculateTriangleNormal(v[0], v[1], v[2]);
                }
            }
        }

        /// <summary>
        /// 法線を正規化
        /// </summary>
        public void NormalizeNormals(bool bForceRound = false)
        {
            List<MeshSourceBase> normals = GetSources(SourceType.Normal);
            int counter = 0;
            List<int> indeces = new List<int>();

            foreach (var normal in normals)
            {
                int Count = normal.Count;
                Vector3 tmp = new Vector3();
                for (int i = 0; i < Count; i++)
                {
                    MeshSourceDblCtrler.GetAsVec3(normal, i, ref tmp);
                    if (bForceRound)
                    {
                        if (tmp.Norm() < 1e-2)
                        {
                            tmp = new Vector3(0, 1, 0);
                            counter++;
                            indeces.Add(i);
                        }
                        else
                        {
                            tmp.Normalize();
                        }
                    }
                    else
                    {
                        tmp.Normalize();
                    }
                    MeshSourceDblCtrler.SetAsVec3(normal, i, tmp.x, tmp.y, tmp.z);
                }
            }
            if (counter > 0)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Force_Normalize_Notify", counter));
                var str = "index : ";
                indeces.ForEach(o => str += o.ToString() + ",");
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog(str.TrimEnd(','));
            }
        }

        /// <summary>
        /// 様々な境界ボリュームを計算します
        /// </summary>
        public void ComputeBounds()
        {
            ComputeMeshBounding();
            ComputeTexCoordBV();
        }

        /// <summary>
        /// メッシュの境界ボリュームを計算します
        /// </summary>
        public void ComputeMeshBounding()
        {
            if (_MaterialShapeInfoList.Count == 0)
            {
                throw ExcepHandle.CreateException("Shape information was not created.");
            }
            int shCnt = 0;
            mBBoxes.Clear();
            foreach (var shapeInfo in _MaterialShapeInfoList)
            {
                shCnt = shapeInfo.Index;
                if (!mBBoxes.ContainsKey(shCnt))
                {
                    mBBoxes.Add(shCnt, new BoundingBox());
                }
                ComputeMeshBounding(shCnt);
                shCnt++;
            }

            mTotalBoudingBox.Reset();
            foreach (var boundingBox in mBBoxes)
            {
                BoundingBox bb = boundingBox.Value;

                if (bb != null)
                {
                    if (bb.IsValid())
                    {
                        mTotalBoudingBox.Update(bb.Min);
                        mTotalBoudingBox.Update(bb.Max);
                    }
                }
            }
            PrintBoundingBox();
        }

        /// <summary>
        /// 指定のFaceGroupの境界を計算します
        /// </summary>
        /// <param name="fgid"></param>
        public void ComputeMeshBounding(int fgid)
        {
            MeshSourceBase posSource = GetSource(SourceType.Position, null);

            int po = GetSourceOfset(SourceType.Position, null);
            int sN = Sources.Count;

            bool[] visited = new bool[posSource.Count];
            for (int i = 0; i < posSource.Count; i++)
            {
                visited[i] = false;
            }

            Vector3 tmp = new Vector3();
            int faceN = FaceN;
            // バウンディングボックスを初期化
            mBBoxes[fgid].Reset();
            for (int fi = 0; fi < faceN; fi++)
            {
                if (_FaceMaterialShapeID[fi] == fgid)
                {
                    int vo = mVoffs[fi];
                    int vn = mVnums[fi];
                    for (int vi = 0; vi < vn; vi++)
                    {
                        int pid = mVbuffs[(vo + vi) * sN + po];
                        if (!visited[pid])
                        {
                            MeshSourceDblCtrler.GetAsVec3(posSource, pid, ref tmp);
                            mBBoxes[fgid].Update(tmp);
                            visited[pid] = true;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// テクスチャ座標の境界ボリュームを計算します
        /// </summary>
        /// <param name="minCoord"> 最小座標 </param>
        /// <param name="maxCoord"> 最大座標 </param>
        public void ComputeTexCoordBV()
        {
            if (_MaterialShapeInfoList.Count == 0)
            {
                throw ExcepHandle.CreateException("Shape information was not created.");
            }
            // UVが存在しません
            List<MeshSourceBase> uvSourceList = GetSources(SourceType.Texture);
            if (uvSourceList == null) { return; }

            int uvsizeN = uvSourceList.Count;
            int shCnt = 0;
            mUVBSquares.Clear();
            foreach (var shapeInfo in _MaterialShapeInfoList)
            {
                shCnt = shapeInfo.Index;
                if (!mUVBSquares.ContainsKey(shCnt))
                {
                    mUVBSquares.Add(shCnt, new List<BoundingSquare>());
                    for (int ui = 0; ui < uvsizeN; ui++)
                    {
                        mUVBSquares[shCnt].Add(new BoundingSquare());
                    }
                }
                ComputeTexCoordBV(shCnt);
            }
        }

        /// <summary>
        /// FaceGroup毎にテクスチャ境界を計算します
        /// </summary>
        /// <param name="fgid"></param>
        protected void ComputeTexCoordBV(int fgid)
        {
            if (!mUVBSquares.ContainsKey(fgid))
            {
                throw ExcepHandle.CreateException("Bounding volume List doesen't exist.");
            }

            // UVが存在しません
            List<MeshSourceBase> uvSourceList = GetSources(SourceType.Texture);
            if (uvSourceList == null) { return; }

            int sN = Sources.Count;
            int faceN = FaceN;
            int uvsizeN = uvSourceList.Count;
            if (mUVBSquares[fgid].Count < uvsizeN)
            {
                throw ExcepHandle.CreateException("Bounding volume List is short.");
            }

            for (int index = 0; index < uvsizeN; index++)
            {
                MeshSourceBase uvSource = uvSourceList[index];
                var quantizeLimit = uvSource.GetQuantizeLimit();
                int uvo = GetSourceOfset(SourceType.Texture, uvSource.Name);
                Vector2 tmp = new Vector2();

                bool[] visited = new bool[uvSource.Count];
                for (int i = 0; i < uvSource.Count; i++)
                {
                    visited[i] = false;
                }

                // 境界を初期化
                mUVBSquares[fgid][index].Reset();
                for (int fi = 0; fi < faceN; fi++)
                {
                    if (_FaceMaterialShapeID[fi] == fgid)
                    {
                        int vo = mVoffs[fi];
                        int vn = mVnums[fi];
                        for (int vi = 0; vi < vn; vi++)
                        {
                            int ui = mVbuffs[(vo + vi) * sN + uvo];

                            if (!visited[ui])
                            {
                                MeshSourceDblCtrler.GetAsVec2(uvSource, ui, ref tmp);
                                mUVBSquares[fgid][index].Update(tmp);
                                visited[ui] = true;
                            }
                        }
                    }
                }
                // 丸めておくことで、誤差による影響を抑えたい
                mUVBSquares[fgid][index].Min = quantizeLimit.Round(mUVBSquares[fgid][index].Min);
                mUVBSquares[fgid][index].Max = quantizeLimit.Round(mUVBSquares[fgid][index].Max);
                mUVBSquares[fgid][index].Round();
            }
        }

        /// <summary>
        /// BiNormalを指定のUVに対して作成
        /// </summary>
        /// <param name="uv"></param>
        /// <returns></returns>
        public MeshSourceBase CreateBiNormal(MeshSourceBase uv)
        {
            var bin = AddSource(uv.Name + "bin", SourceType.Binormal, 3);
            bin.SetLength(uv.Count, false);
            return bin;
        }

        /// <summary>
        /// タンジェントを、指定のUVに対して作成
        /// </summary>
        /// <param name="uv"></param>
        /// <returns></returns>
        public MeshSourceBase CreateTangent(MeshSourceBase uv)
        {
            // 従法線、接戦を追加
            var tangent = AddSource(uv.Name + "tan", SourceType.Tangent, 3);
            tangent.SetLength(uv.Count, false);
            return tangent;
        }

        /// <summary>
        /// 頂点 - UVで計算したタンジェントを設定する
        /// </summary>
        /// <param name="vertexTangentsAndBinormals"></param>
        private void SetTangentBasis(VertexTangentsAndBinormals vertexTangentsAndBinormals)
        {
            var uvMaps = GetSources(SourceType.Texture);
            var uvMapN = uvMaps.Count;
            if (uvMaps.Count == 0)
            {
                return;
            }
            var tangents = GetSources(SourceType.Tangent);
            var binormals = GetSources(SourceType.Binormal);

            // 頂点バッファ情報の取得
            int sn = Sources.Count;
            var poslist = GetSource(SourceType.Position, null) as MeshSource<double>;
            if (poslist == null)
            {
                return;
            }

            int no = GetSourceOfset(SourceType.Normal, null);
            int po = GetSourceOfset(SourceType.Position, null);
            var nmllist = GetSource(SourceType.Normal, null) as MeshSource<double>;
            for (var fi = 0; fi < FaceN; fi++)
            {
                var vn = mVnums[fi];
                var vo = mVoffs[fi];
                for (int vi = 0; vi < vn; vi++)
                {
                    // 頂点法線を取得
                    int vid = VBuffer[(vo + vi) * sn + po];
                    int nmlid = VBuffer[(vo + vi) * sn + no];
                    var nml = new Vector3();
                    MeshSourceDblCtrler.GetAsVec3(nmllist, nmlid, ref nml);
                    // UV毎にアジャスト
                    for (int ui = 0; ui < uvMapN; ui++)
                    {
                        MeshSourceBase uv = uvMaps[ui];

                        int so = GetSourceOfset(SourceType.Texture, uv.Name);
                        int uvid = VBuffer[(vo + vi) * sn + so];

                        if (!vertexTangentsAndBinormals[vid].ContainsKey(nmlid))
                        {
                            throw new Exception("tangent list does not have the key : " + uvid.ToString());
                        }
                        if (!vertexTangentsAndBinormals[vid][nmlid].ContainsKey(uvid))
                        {
                            throw new Exception("tangent list does not have the key : " + uvid.ToString());
                        }

                        // Mayaの計算では、直行化をここでする必要はない模様
                        // それぞれ別個の面について直交化をして足し合わせるだけでOK
                        Vector3 tmpTan = vertexTangentsAndBinormals[vid][nmlid][uvid][SourceType.Tangent];
                        if (tmpTan.IsNormZero())
                        {
                            var shapeIndex = FaceMaterialShapeId[fi];
                            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Force_Tangent_Notify", MaterialShapeInformationList.Find(shapeInfo => shapeInfo.Index == shapeIndex).ShapeName, vid));
                            Nintendo.Foundation.Contracts.Ensure.Operation.True(!nml.IsNormZero(), _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_ReCalculateTangentException", vid));

                            Vector3 axisY = new Vector3(0, 1, 0);
                            if (Vector3.IsEqual(nml, axisY) || Vector3.IsEqual(nml, new Vector3(0, -1, 0)))
                            {
                                Vector3 negativeAxisZ = new Vector3(0, 0, -1);
                                tmpTan = negativeAxisZ.Cross(nml);
                            }
                            else
                            {
                                tmpTan = axisY.Cross(nml);
                            }
                        }

                        var tan = tmpTan;
                        tan.Normalize();

                        Vector3 tmpBin = vertexTangentsAndBinormals[vid][nmlid][uvid][SourceType.Binormal];
                        if (tmpBin.IsNormZero())
                        {
                            // 不正なケースなので、法線と接線と直行するベクトルを格納する
                            tmpBin = nml.Cross(tan);
                        }

                        tmpBin.Normalize();

                        // UV がミラーされている場合のフラグを設定
                        var w = IsRightHandTangent(tan, tmpBin, nml) ? 1.0 : -1.0;

                        var bin = nml.Cross(tan) * w;
                        bin.Normalize();

                        // いずれか一方しか設定されてないケースもある
                        if (tangents.Count > ui)
                        {
                            var tangent = tangents[ui] as MeshSource<double>;
                            int to = GetSourceOfset(SourceType.Tangent, tangent.Name);
                            MeshSourceDblCtrler.SetAsVec4(tangent, mVbuffs[(vo + vi) * sn + to], tan[0], tan[1], tan[2], w);
                        }
                        if (binormals.Count > ui)
                        {
                            var binormal = binormals[ui] as MeshSource<double>;
                            int bo = GetSourceOfset(SourceType.Binormal, binormal.Name);
                            MeshSourceDblCtrler.SetAsVec4(binormal, mVbuffs[(vo + vi) * sn + bo], bin[0], bin[1], bin[2], w);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 頂点ごとに各条件で必要となる接線と従法線を計算する。
        /// 頂点の加算方式で必要となる値が異なる
        /// </summary>
        /// <param name="vertexTangentsAndBinormals"></param>
        private void ComputePerVtxTangentBasis(out VertexTangentsAndBinormals vertexTangentsAndBinormals)
        {
            var uvMaps = GetSources(SourceType.Texture);
            var uvMapN = uvMaps.Count;

            // バッファ作成
            vertexTangentsAndBinormals = new VertexTangentsAndBinormals();

            if (uvMaps.Count == 0)
            {
                return;
            }
            vertexTangentsAndBinormals.Resize(VertexN, null);

            // 頂点バッファ情報の取得
            int sn = Sources.Count;
            int po = GetSourceOfset(SourceType.Position, null);
            int no = GetSourceOfset(SourceType.Normal, null);
            var poslist = GetSource(SourceType.Position, null) as MeshSource<double>;
            var nmllist = GetSource(SourceType.Normal, null) as MeshSource<double>;
            if (poslist == null)
            {
                return;
            }

            // 同一頂点を利用する面をピックしてまとめる
            VtxTriMesh vtm = new VtxTriMesh();
            vtm.Create(this);

            for (var vindex = 0; vindex < VertexN; vindex++)
            {
                // 同じ頂点を共有する面でループ
                vertexTangentsAndBinormals[vindex] = new Dictionary<int, Dictionary<int, Dictionary<SourceType, Vector3>>>();

                foreach (var faceIdx in vtm.VTL[vindex].TriList)
                {
                    var vn = mVnums[faceIdx];
                    var vo = mVoffs[faceIdx];
                    var pos = new Vector3[vn];
                    var normals = new Vector3[vn];
                    var localCoincidentVid = -1;
                    //---------------------------------------
                    // 頂点位置取得
                    for (int vi = 0; vi < vn; vi++)
                    {
                        pos[vi] = new Vector3();
                        int vid = VBuffer[(vo + vi) * sn + po];
                        MeshSourceDblCtrler.GetAsVec3(poslist, vid, ref pos[vi]);

                        if (vid == vindex)
                        {
                            localCoincidentVid = vi;
                        }
                    }

                    // 法線取得
                    for (int vi = 0; vi < vn; vi++)
                    {
                        int nmlid = VBuffer[(vo + vi) * sn + no];
                        normals[vi] = new Vector3();
                        MeshSourceDblCtrler.GetAsVec3(nmllist, nmlid, ref normals[vi]);
                    }

                    //---------------------------------------
                    // UV毎にtangentを計算する
                    for (int ui = 0; ui < uvMapN; ui++)
                    {
                        MeshSourceBase uv = uvMaps[ui];
                        // UV座標を取得
                        Vector2[] uvs = new Vector2[vn];
                        int uvo = GetSourceOfset(SourceType.Texture, uv.Name);
                        for (int vi = 0; vi < vn; vi++)
                        {
                            uvs[vi] = new Vector2();
                            int uvid = VBuffer[(vo + vi) * sn + uvo];
                            MeshSourceDblCtrler.GetAsVec2(uv, uvid, ref uvs[vi]);
                        }

                        if (localCoincidentVid != -1)
                        {
                            // 接線計算
                            var id0 = 0;
                            var id1 = 1;
                            var id2 = 2;

                            var e1 = pos[id1] - pos[id0];
                            var e2 = pos[id2] - pos[id0];
                            var cross = e1.Cross(e2);
                            cross.Normalize();

                            var uv1 = uvs[id1] - uvs[id0];
                            var uv2 = uvs[id2] - uvs[id0];
                            var r = uv1.Cross(uv2);

                            if (Math.Abs(r) > 1e-5)
                            {
                                r = 1.0 / r;
                            }
                            else
                            {
                                r = 1.0;
                            }

                            int nmlid = VBuffer[(vo + localCoincidentVid) * sn + no];
                            var nml = new Vector3();
                            MeshSourceDblCtrler.GetAsVec3(nmllist, nmlid, ref nml);
                            int uvid = VBuffer[(vo + localCoincidentVid) * sn + uvo];

                            // UV 毎にタンジェントを求める
                            // -- Shellの違いに対応
                            if (!vertexTangentsAndBinormals[vindex].ContainsKey(nmlid))
                            {
                                vertexTangentsAndBinormals[vindex].Add(nmlid, new Dictionary<int, Dictionary<SourceType, Vector3>>());
                            }

                            if (!vertexTangentsAndBinormals[vindex][nmlid].ContainsKey(uvid))
                            {
                                vertexTangentsAndBinormals[vindex][nmlid].Add(uvid, new Dictionary<SourceType, Vector3>(2));
                                vertexTangentsAndBinormals[vindex][nmlid][uvid][SourceType.Tangent] = new Vector3(0, 0, 0);
                                vertexTangentsAndBinormals[vindex][nmlid][uvid][SourceType.Binormal] = new Vector3(0, 0, 0);
                            }

                            // タンジェント
                            {
                                // グラムシュミット直交化
                                Vector3 tan = (e1 * uv2.y - e2 * uv1.y) * r;
                                tan -= nml * (nml.Dot(tan));
                                if (tan.IsNormZero())
                                {
                                    continue;
                                }

                                tan.Normalize();
                                vertexTangentsAndBinormals[vindex][nmlid][uvid][SourceType.Tangent] += tan;
                            }

                            // バイノーマル
                            // NOTE: 現在は、こちらでバイノーマルの再計算を行い、後段の SetTangentBasis() で
                            // タンジェントとバイノーマルの符号情報を生成するために利用します。
                            // 将来的には、 UV のミラー境界が保護されるため (符号は変化しない) 、オリジナルのメッシュから w 成分を取得すれば問題ありません。
                            {
                                Vector3 bin = (e1 * uv2.x - e2 * uv1.x) * r;
                                bin -= nml * (nml.Dot(bin));
                                if (bin.IsNormZero())
                                {
                                    continue;
                                }

                                bin.Normalize();
                                vertexTangentsAndBinormals[vindex][nmlid][uvid][SourceType.Binormal] += bin;
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 接平面空間の基底ベクトルを再計算する
        /// </summary>
        public void ComputeTangentBasis()
        {
            var uvMaps = GetSources(SourceType.Texture);
            if (uvMaps.Count == 0)
            {
                return;
            }

            // テンポラリバッファ作成
            VertexTangentsAndBinormals perVertexTangentsAndBinormals;

            // 万一、未使用頂点があった場合にエラーになるのを回避するため
            // 強制的に長さ１のベクトルを代入しておく(使用される頂点は、上書きされるので影響はない)
            ResetTangentBasis();

            // 頂点ごとの加算方式を確認する
            ComputePerVtxTangentBasis(out perVertexTangentsAndBinormals);
            SetTangentBasis(perVertexTangentsAndBinormals);
        }

        /// <summary>
        /// 接線、従法線が未使用の値を持つ場合の回避用初期化関数（基本的には、この関数を使わない場合でも問題ないはずだけど、念のため）
        /// </summary>
        private void ResetTangentBasis()
        {
            var tangents  = GetSources(SourceType.Tangent);

            foreach (var tangent in tangents)
            {
                var count = tangent.Count;
                for (int i = 0; i < count; i++)
                {
                    MeshSourceDblCtrler.SetAsVec4( tangent, i, 0.0, 1.0, 0.0, 1.0);
                }
            }

            var binormals = GetSources(SourceType.Binormal);
            foreach (var binormal in binormals)
            {
                var count = binormal.Count;
                for (int i = 0; i < count; i++)
                {
                    MeshSourceDblCtrler.SetAsVec4( binormal, i, 0.0, 0.0, 1.0, 1.0);
                }
            }
        }

        /// <summary>
        /// 指定のフェースが縮退しているかどうか？
        /// </summary>
        /// <param name="fid"></param>
        /// <returns></returns>
        public bool IsFaceDegenerate(int fid, double threshold = double.Epsilon)
        {
            try
            {
                int vn = mVnums[fid];
                int vo = mVoffs[fid];
                int sn = Sources.Count;
                int[] vid = new int[vn];
                int so = 0;
                Vector3[] plist = new Vector3[vn];

                // 各頂点の値を取得して、頂点データとして返す
                for (int v = 0; v < vn; v++)
                {
                    plist[v] = new Vector3();
                    so = this.GetSourceOfset(SourceType.Position, null);
                    MeshSource<double> source = GetSource(SourceType.Position, null) as MeshSource<double>;
                    if (source == null)
                    {
                        throw ExcepHandle.CreateException("The value type of position stream is not double/float");
                    }
                    {
                        vid[v] = mVbuffs[(vo + v) * sn + so];
                        for (int st = 0; st < source.Stride; st++)
                        {
                            plist[v][st] = source.GetRowValue(vid[v] * source.Stride + st);
                        }
                    }
                }

                for (int vi = 0; vi < vn - 1; vi++)
                {
                    for (int vj = vi + 1; vj < vn; vj++)
                    {
                        Vector3 t = plist[vi] - plist[vj];
                        if (vid[vi] == vid[vj] || t.Norm() <= threshold)
                        {
                            return true;
                        }
                    }
                }
            }
            catch (Exception exp)
            {
                var message = "fail to IsFaceDegenerate " + fid;
                throw ExcepHandle.CreateException(message + exp);
            }
            return false;
        }

        /// <summary>
        /// 各フェースの頂点座標を取得する
        /// </summary>
        /// <param name="fid"></param>
        /// <param name="plist"></param>
        public void GetFaceVertex(int fid, ref VectorN[] plist)
        {
            try
            {
                int vn = mVnums[fid];
                int vo = mVoffs[fid];
                int sn = Sources.Count;
                int di = 0;
                int so = 0;

                if (plist.Length < vn)
                {
                    throw ExcepHandle.CreateException("The length of output array is shorter than vertex count.");
                }

                // 各頂点の値を取得して、頂点データとして返す
                for (int v = 0; v < vn; v++)
                {
                    di = 0;
                    so = 0;
                    uint _dim = plist[v].Dim;
                    foreach (MeshSourceBase source in Sources)
                    {
                        if (_dim < (di + source.Stride))
                        {
                            break;
                        }
                        int vid = mVbuffs[(vo + v) * sn + so];
                        int _stride = source.Stride;
                        for (int st = 0; st < _stride; st++, di++)
                        {
                            if (plist[v].Dim <= di)
                            {
                                throw ExcepHandle.CreateException(
                                    "The length of output array is shorter than dimension order.");
                            }
                            if (source.Type == SourceType.EtceteraInt)
                            {
                                plist[v][di] = (int)source.GetRowValue(vid * source.Stride + st);
                            }
                            else
                            {
                                int index = vid * source.Stride + st;
                                if (source.RowCount <= index || index < 0)
                                {
                                    throw ExcepHandle.CreateException("index ; out of bounds.");
                                }
                                plist[v][di] = (double)source.GetRowValue(index);
                            }
                        }
                        so++;
                    }
                }
            }
            catch (Exception exp)
            {
                var message = "fail to IsFaceDegenerate " + fid;
                throw ExcepHandle.CreateException(message + exp);
            }
        }

        /// <summary>
        /// 面を構成するインデックスを取得
        /// </summary>
        /// <param name="fid"></param>
        /// <param name="idlist"></param>
        public void GetFaceVtxIndices(int fid, ref int[] idlist, SourceType stype = SourceType.Position,
            string sname = null)
        {
            int vn = mVnums[fid];
            int vo = mVoffs[fid];
            int sn = Sources.Count;
            int po = GetSourceOfset(stype, sname);

            if (idlist == null)
            {
                idlist = new int[vn];
            }
            if (idlist.Length < vn)
            {
                throw ExcepHandle.CreateException("The length of output array is shorter than vertex count.");
            }
            // 各頂点の値を取得して、頂点データとして返す
            for (int v = 0; v < vn; v++)
            {
                idlist[v] = mVbuffs[(vo + v) * sn + po];
            }
        }

        /// <summary>
        /// 各フェースの頂点座標を取得する
        /// </summary>
        /// <param name="vid"> 頂点ID　</param>
        /// <param name="vertex">　頂点データを格納する </param>
        public void GetVertexPos(int vid, ref Vector3 vertex)
        {
            try
            {
                if (vertex == null)
                {
                    vertex = new Vector3();
                }
                foreach (MeshSourceBase source in Sources)
                {
                    if (source.Type == SourceType.Position)
                    {
                        if (source.Stride != 3) { continue; }
                        for (int st = 0; st < source.Stride; st++)
                        {
                            if (source.Type == SourceType.EtceteraInt)
                            {
                                vertex[st] = (int)source.GetRowValue(vid * source.Stride + st);
                            }
                            else
                            {
                                vertex[st] = (double)source.GetRowValue(vid * source.Stride + st);
                            }
                        }
                    }
                }
            }
            catch (Exception exp)
            {
                var message = "fail to IsFaceDegenerate " + vid;
                throw ExcepHandle.CreateException(message + exp);
            }
        }

        public Vector3 GetVertexPos(int vid)
        {
            var ret = new Vector3();
            GetVertexPos(vid, ref ret);
            return ret;
        }

        /// <summary>
        /// 頂点を設定する
        /// </summary>
        /// <param name="wedge"></param>
        /// <param name="side"></param>
        /// <param name="newpos"></param>
        public void SetVertexInfo(WingedEdge wedge, int side, VectorN newpos)
        {
            try
            {
                int vid = 0;
                int fid = 0;

                if (side == 0)
                {
                    vid = wedge.Vertex0;
                    fid = wedge.Face0;
                    if (fid == int.MaxValue)
                    {
                        fid = wedge.Face1;
                    }
                }
                else if (side == 1)
                {
                    vid = wedge.Vertex1;
                    fid = wedge.Face1;
                    if (fid == int.MaxValue)
                    {
                        fid = wedge.Face0;
                    }
                }

                int sn = Sources.Count;
                int po = GetSourceOfset(SourceType.Position, null);
                // ただのスコープ
                {
                    int vo = mVoffs[fid];
                    int di = 0;
                    int so = 0;
                    bool hit = false;
                    for (int i = 0; i < 3; i++)
                    {
                        if (mVbuffs[(vo + i) * sn + po] == vid)
                        {
                            hit = true;
                            foreach (MeshSourceBase source in Sources)
                            {
                                if (newpos.Dim < (di + source.Stride))
                                {
                                    break;
                                }
                                vid = mVbuffs[(vo + i) * sn + so];
                                for (int st = 0; st < source.Stride; st++, di++)
                                {
                                    if (newpos.Dim <= di)
                                    {
                                        throw ExcepHandle.CreateException(
                                            "The length of output array is shorter than stride.");
                                    }
                                    source.SetRowValue(vid * source.Stride + st, newpos[di]);
                                }
                                so++;
                            }
                            break;
                        }
                    }
                    if (!hit)
                    {
                        throw ExcepHandle.CreateException("Failed to set vertex information.");
                    }
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// 孤立頂点の削除
        /// </summary>
        /// <returns></returns>
        public int RemoveUnusedVertex()
        {
            ResizableList<int> usedFlag = null;
            return RemoveUnusedVertex(ref usedFlag);
        }

        /// <summary>
        /// 孤立頂点の削除
        /// </summary>
        /// <param name="usedFlag">削除前の各頂点の使用状況</param>
        /// <returns>削除頂点数</returns>
        public int RemoveUnusedVertex(ref ResizableList<int> usedFlag)
        {
            try
            {
                int faceN = FaceN;
                int sourceN = Sources.Count;

                int[] vertexNList = new int[sourceN];

                usedFlag = new ResizableList<int>();

                for (int s = 0; s < sourceN; s++)
                {
                    vertexNList[s] = Sources[s].Count;
                }

                ResizableList<ResizableList<bool>> vtxUsage = new ResizableList<ResizableList<bool>>();
                vtxUsage.Resize(sourceN, null);
                for (int s = 0; s < sourceN; s++)
                {
                    vtxUsage[s] = new ResizableList<bool>();
                    vtxUsage[s].Resize(vertexNList[s], false);
                    vertexNList[s] = Sources[s].Count;
                }

                // どの頂点が使われているか？をプロパティ毎にリストアップする
                for (int fi = 0; fi < faceN; fi++)
                {
                    int vn = mVnums[fi];
                    int vnoffs = mVoffs[fi];
                    for (int vi = 0; vi < vn; vi++)
                    {
                        for (int sid = 0; sid < sourceN; ++sid)
                        {
                            // skip unused vert indices
                            if (vertexNList[sid] == 0)
                            {
                                continue;
                            }
                            int vid = sourceN * (vnoffs + vi) + sid;
                            vtxUsage[sid][mVbuffs[vid]] = true;
                        }
                    }
                }

                usedFlag.Resize(vertexNList[0], int.MaxValue);
                for (int i = 0; i < vertexNList[0]; i++)
                {
                    if (vtxUsage[0][i])
                    {
                        usedFlag[i] = i;
                    }
                }

                // 移動テーブルリストを作成
                ResizableList<ResizableList<int>> transTable = new ResizableList<ResizableList<int>>();
                transTable.Resize(sourceN, null);
                for (int vid = 0; vid < sourceN; ++vid)
                {
                    transTable[vid] = new ResizableList<int>();
                    transTable[vid].Resize(vertexNList[vid], int.MaxValue);
                }

                // 新しいインデックスを割り当てる、
                // 未使用データにはMaxを入れる
                ResizableList<int> newIdx = new ResizableList<int>();
                newIdx.Resize(sourceN, 0);
                for (int s = 0; s < sourceN; ++s)
                {
                    // skip unused vert indices
                    if (vertexNList[s] == 0)
                    {
                        continue;
                    }

                    for (int i = 0; i < vertexNList[s]; ++i)
                    {
                        if (vtxUsage[s][i])
                        {
                            int stride = Sources[s].Stride;
                            MeshSourceBase _dblsource = Sources[s];
                            for (int sti = 0; sti < stride; sti++)
                            {
                                var tmpvalue = _dblsource.GetRowValue(i * stride + sti);
                                _dblsource.SetRowValue(newIdx[s] * stride + sti, tmpvalue);
                            }
                            transTable[s][i] = newIdx[s]++;
                        }
                        else
                        {
                            transTable[s][i] = int.MaxValue;
                        }
                    }
                }

                // 不要な頂点は消す
                // バッファ最適化
                for (int s = 0; s < sourceN; ++s)
                {
                    Sources[s].SetLength(newIdx[s], false);
                }

                // フェースごとのインデックスの再割当て
                for (int fi = 0; fi < faceN; fi++)
                {
                    int vn = mVnums[fi];
                    int vnoffs = mVoffs[fi];
                    for (int vi = 0; vi < vn; vi++)
                    {
                        for (int sid = 0; sid < sourceN; ++sid)
                        {
                            // skip unused vert indices
                            if (vertexNList[sid] == 0)
                            {
                                continue;
                            }
                            int pi = sourceN * (vnoffs + vi) + sid;
                            mVbuffs[pi] = transTable[sid][mVbuffs[pi]];
                        }
                    }
                }
                return vertexNList[0] - newIdx[0];
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// 不要となったフェースを除去して綺麗にする
        /// </summary>
        public void RemoveInvalidFaces()
        {
            try
            {
                int nFaces = FaceN;
                bool bHasFaceNmls = this.mFaceNormal.Count > 0;

                int nSourceN = Sources.Count;
                int it = 0;
                int id = 0;
                int vnum = 0;
                for (; it < nFaces; it++)
                {
                    if (!IsInvalidateFace(it))
                    {
                        if (bHasFaceNmls)
                        {
                            mFaceNormal[id] = mFaceNormal[it];
                        }
                        mVnums[id] = mVnums[it];
                        mFaceInvalid[id] = mFaceInvalid[it];

                        if (mFaceArea.Count > it)
                        {
                            mFaceArea[id] = mFaceArea[it];
                        }
                        if (mFaceColor.Count > it)
                        {
                            mFaceColor[id] = mFaceColor[it];
                        }
                        if (_FaceMaterialShapeID.Count > it)
                        {
                            _FaceMaterialShapeID[id] = _FaceMaterialShapeID[it];
                        }
                        if (mFaceNormal.Count > it)
                        {
                            mFaceNormal[id] = mFaceNormal[it];
                        }
                        if (mFaceSubMeshID.Count > it)
                        {
                            mFaceSubMeshID[id] = mFaceSubMeshID[it];
                        }

                        int of0 = mVoffs[id];
                        int of1 = mVoffs[it];
                        // 頂点バッファのコピー
                        for (int vi = 0; vi < mVnums[it]; vi++)
                        {
                            for (int si = 0; si < nSourceN; si++)
                            {
                                mVbuffs[(of0 + vi) * nSourceN + si] = mVbuffs[(of1 + vi) * nSourceN + si];
                            }
                        }
                        vnum += mVnums[id];
                        id++;
                    }
                }
                if (bHasFaceNmls)
                {
                    mFaceNormal.Resize(id, default(Vector3));
                }

                mVnums.Resize(id, 0);
                mFaceArea.Resize(id, default(double));
                mFaceColor.Resize(id, default(Vector4));
                _FaceMaterialShapeID.Resize(id, default(int));
                mFaceInvalid.Resize(id, false);
                mFaceNormal.Resize(id, default(Vector3));
                mFaceSubMeshID.Resize(id, default(int));

                mVbuffs.Resize(vnum * nSourceN, 0);

                mVoffs.Resize(id, 0);
                for (int i = 0; i < id; i++)
                {
                    mVoffs[i] = 0;
                }
                int sum = 0;
                for (int i = 0; i < id; i++)
                {
                    mVoffs[i] = sum;
                    sum += mVnums[i];
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// クローンを作成する
        /// </summary>
        /// <returns></returns>
        public Mesh Clone()
        {
            Mesh result = new Mesh();
            result._sources = new List<MeshSourceBase>();
            for (int i = 0; i < _sources.Count; i++)
            {
                result._sources.Add(_sources[i].Clone());
            }

            //--------------------------------------------
            foreach (var boundingBox in mBBoxes)
            {
                result.mBBoxes.Add(boundingBox.Key, new BoundingBox());
                result.mBBoxes[boundingBox.Key].Update(boundingBox.Value.Min);
                result.mBBoxes[boundingBox.Key].Update(boundingBox.Value.Max);
            }

            //--------------------------------------------
            foreach (var uvbSquare in mUVBSquares)
            {
                int key = uvbSquare.Key;
                result.mUVBSquares.Add(key, new List<BoundingSquare>());
                for (int i = 0; i < result.mUVBSquares[key].Count; i++)
                {
                    BoundingSquare bs = new BoundingSquare();
                    bs.Update(uvbSquare.Value[i].Min);
                    bs.Update(uvbSquare.Value[i].Max);
                    result.mUVBSquares[key].Add(bs);
                }
            }

            //--------------------------------------------
            result.mVnums = new ResizableList<int>(mVnums);
            result.mVoffs = new ResizableList<int>(mVoffs);
            result.mVbuffs = new ResizableList<int>(mVbuffs);

            result.mFaceArea = new ResizableList<double>(mFaceArea);
            result.mFaceColor = new ResizableList<Vector4>(mFaceColor);
            result._FaceMaterialShapeID = new ResizableList<int>(_FaceMaterialShapeID);
            result.mFaceInvalid = new ResizableList<bool>(mFaceInvalid);
            result.mFaceNormal = new ResizableList<Vector3>(mFaceNormal);
            result.mFaceSubMeshID = new ResizableList<int>(mFaceSubMeshID);

            result.mSubMeshInfo = new SortedDictionary<int, Mesh.SubMeshInfo>(mSubMeshInfo);
            result._MaterialShapeInfoList = new ResizableList<MaterialShapeInfo>(_MaterialShapeInfoList);

            result.mName = mName;
            return result;
        }

        /// <summary>
        /// フェースの数を設定する
        /// </summary>
        /// <param name="faceN">フェース数</param>
        /// <param name="buffNum">バッファサイズ</param>
        public void ResizeFaces(int faceN, int buffNum)
        {
            try
            {
                mVnums.Resize(faceN, int.MaxValue);
                mVoffs.Resize(faceN, int.MaxValue);
                mVbuffs.Resize(buffNum, int.MaxValue);

                mFaceArea.Resize(faceN, 0.0);
                mFaceColor.Resize(faceN, null);
                _FaceMaterialShapeID.Resize(faceN, 0);
                mFaceInvalid.Resize(faceN, false);
                mFaceNormal.Resize(faceN, null);
                mFaceSubMeshID.Resize(faceN, -1);
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// サブメッシュ情報を縮小する
        /// </summary>
        /// <param name="subMeshInfo"></param>
        private void ShrinkSubmeshInfo(ref SortedDictionary<int, ResizableList<int>> subMeshInfo)
        {
            var convertMap = new Dictionary<int, int>();
            var newInfo = new SortedDictionary<int, ResizableList<int>>();
            var totalSubmesh = 0;
            foreach (var sumesh in subMeshInfo)
            {
                convertMap.Add(sumesh.Key, totalSubmesh);
                newInfo.Add(totalSubmesh, new ResizableList<int>());
                totalSubmesh++;
            }

            // 減少したサブメッシュ情報に合わせて、貼り直しを行う
            foreach (var sumesh in subMeshInfo)
            {
                for (int fi = 0; fi < FaceN; fi++)
                {
                    if (mFaceSubMeshID[fi] == sumesh.Key)
                    {
                        mFaceSubMeshID[fi] = convertMap[sumesh.Key];
                        newInfo[convertMap[sumesh.Key]].Add(fi);
                    }
                }
            }
            subMeshInfo = newInfo;
        }

        /// <summary>
        /// サブメッシュを再度配置し直します
        /// </summary>
        public bool ReComputeSubMeshArray(bool forceRecompute = false)
        {
            if (FaceN == 0)
            {
                return false;
            }

            try
            {
                var oldSubMeshN = mSubMeshInfo.Count;
                var subMeshN = 0;
                var counter = new Dictionary<int, int>();
                var sumeshes = new SortedDictionary<int, ResizableList<int>>();

                // サブメッシュ毎に設定されている、フェースリストと
                // サブメッシュ毎のカウンタを作成する
                for (int fi = 0; fi < FaceN; fi++)
                {
                    if (mFaceSubMeshID[fi] < 0)
                    {
                        throw ExcepHandle.CreateException(@"Invalid sub mesh index is found.");
                    }

                    if (!sumeshes.ContainsKey(mFaceSubMeshID[fi]))
                    {
                        sumeshes.Add(mFaceSubMeshID[fi], new ResizableList<int>());
                        counter.Add(mFaceSubMeshID[fi], 1);
                    }
                    else
                    {
                        counter[mFaceSubMeshID[fi]]++;
                    }
                }

                subMeshN = sumeshes.Keys.Count();
                // サブメッシュ情報が設定されていない場合は
                // デフォルトサブメッシュ情報をメッシュ全体に適用する
                if (subMeshN == 0)
                {
                    SetSubMeshInfo(0, 0, FaceN);
                    return true;
                }

                int chkSum = 0;
                foreach (var cnt in counter)
                {
                    chkSum += cnt.Value;
                }

                if (chkSum != FaceN)
                {
                    throw ExcepHandle.CreateException("Existing faces have no submesh info.");
                }

                //-------------------------------
                // 並べ替えを開始
                //-------------------------------

                //-------------------------------
                // 各フェースに、どのサブメッシュが割り当てられてるかを取得
                // バラバラになっていても、大丈夫なようにします
                foreach (var sumesh in sumeshes)
                {
                    for (int fi = 0; fi < FaceN; fi++)
                    {
                        if (mFaceSubMeshID[fi] == sumesh.Key)
                        {
                            sumesh.Value.Add(fi);
                        }
                    }
                }
                //-------------------------------
                if (oldSubMeshN != subMeshN)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Decrease_Submeh_Poly", oldSubMeshN, subMeshN));
                    if (forceRecompute)
                    {
                        ShrinkSubmeshInfo(ref sumeshes);
                    }
                    else
                    {
                        return false;
                    }
                }

                //-------------------------------
                // 入れ替え用の入れ物
                var newBuffs = new ResizableList<int>(mVbuffs);

                var newFaceArea = new ResizableList<double>(mFaceArea);
                var newFaceColor = new ResizableList<Vector4>(mFaceColor);
                var newFaceGroupID = new ResizableList<int>(_FaceMaterialShapeID);
                var newFaceInvalid = new ResizableList<bool>(mFaceInvalid);
                var newFaceNormal = new ResizableList<Vector3>(mFaceNormal);
                var newFaceSubMeshID = new ResizableList<int>(mFaceSubMeshID);

                //-------------------------------
                // インデックス張替え開始
                int sN = Sources.Count;
                int totalFN = 0;
                foreach (var sumesh in sumeshes)
                {
                    // sumesh には、飛び番で値が入っている可能性がある
                    int sizeN = sumesh.Value.Count;
                    for (int fi = 0; fi < sizeN; fi++)
                    {
                        int faceId = sumesh.Value[fi]; // ランダムでOK
                        int reId = (fi + totalFN); // シーケンシャルな番号
                        int vo1 = mVoffs[faceId];
                        int vo2 = mVoffs[reId];

                        if (mVnums[reId] != mVnums[faceId])
                        {
                            throw ExcepHandle.CreateException("Some polygons have verteces over 3.");
                        }

                        for (int vi = 0; vi < mVnums[faceId]; vi++)
                        {
                            for (int si = 0; si < sN; si++)
                            {
                                int s = (vi + vo1) * sN + si;
                                int d = (vi + vo2) * sN + si;
                                if (mVbuffs.Count <= s || mVbuffs.Count <= d)
                                {
                                    throw ExcepHandle.CreateException("Index error at ReComputeSubMeshArray");
                                }
                                newBuffs[d] = mVbuffs[s];
                            }
                        }

                        // インデックス張替え
                        newFaceSubMeshID[reId] = mFaceSubMeshID[faceId];

                        // フェースに従属するプロパティの並べ替えも同時に行います
                        newFaceArea[reId] = mFaceArea[faceId];
                        newFaceColor[reId] = mFaceColor[faceId];
                        newFaceGroupID[reId] = _FaceMaterialShapeID[faceId];
                        newFaceInvalid[reId] = mFaceInvalid[faceId];
                        newFaceNormal[reId] = mFaceNormal[faceId];
                    }
                    totalFN += sizeN;
                }

                mVbuffs = newBuffs;
                //------------------------
                // 入れ替え
                mFaceArea = newFaceArea;
                mFaceColor = newFaceColor;
                _FaceMaterialShapeID = newFaceGroupID;
                mFaceInvalid = newFaceInvalid;
                mFaceNormal = newFaceNormal;
                mFaceSubMeshID = newFaceSubMeshID;

                //---------------------------
                // サブメッシュ情報の作り変え
                int sum = 0;
                mSubMeshInfo = new SortedDictionary<int, SubMeshInfo>();
                foreach (var sumesh in sumeshes)
                {
                    var smi = sumesh.Key;

                    if (sumeshes.ContainsKey(smi) && !mSubMeshInfo.ContainsKey(smi))
                    {
                        mSubMeshInfo.Add(smi, new SubMeshInfo()
                        {
                            subMeshindex = smi,
                            subMeshOffset = sum,
                            subMeshCount = sumeshes[smi].Count
                        });
                        sum += sumeshes[smi].Count;
                    }
                    else
                    {
                        if (!mSubMeshInfo.ContainsKey(smi))
                        {
                            mSubMeshInfo.Add(smi, SubMeshInfo.Null);
                        }
                    }
                }
                //-------------------------
                CheckSubMeshArray();
                return true;
            }
            catch (Exception ex)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(false, ex.Message);
                throw;
            }
        }

        /// <summary>
        /// 強制的に頂点座標を、境界ボリュームの中に収める
        /// </summary>
        /// <param name="bboxes"></param>
        public void ForceRoundingPosInBBoxes(Dictionary<int, BoundingBox> bboxes)
        {
            if (bboxes == null)
            {
                throw ExcepHandle.CreateException("Bounding box is null.");
            }
            if (_MaterialShapeInfoList.Count != bboxes.Count)
            {
                throw ExcepHandle.CreateException("Count is not equal to shape count.");
            }

            int sN = SourceN;
            MeshSourceBase pos = GetSource(SourceType.Position, null);
            int po = GetSourceOfset(SourceType.Position, null);
            Vector3 tmppos = new Vector3();

            for (int fi = 0; fi < FaceN; fi++)
            {
                int fgid = _FaceMaterialShapeID[fi];
                if (fgid >= bboxes.Count)
                {
                    throw ExcepHandle.CreateException("Illegal bounding box is found.");
                }

                BoundingBox bbox = bboxes.ElementAt(fgid).Value;

                int vo = mVoffs[fi];
                for (int vi = 0; vi < 3; vi++)
                {
                    int vid = mVbuffs[(vo + vi) * sN + po];

                    MeshSourceDblCtrler.GetAsVec3(pos, vid, ref tmppos);
                    tmppos[0] = Math.Max(bbox.Min[0], tmppos[0]);
                    tmppos[1] = Math.Max(bbox.Min[1], tmppos[1]);
                    tmppos[2] = Math.Max(bbox.Min[2], tmppos[2]);

                    tmppos[0] = Math.Min(bbox.Max[0], tmppos[0]);
                    tmppos[1] = Math.Min(bbox.Max[1], tmppos[1]);
                    tmppos[2] = Math.Min(bbox.Max[2], tmppos[2]);

                    MeshSourceDblCtrler.SetAsVec3(pos, vid, tmppos.x, tmppos.y, tmppos.z);
                }
            }
        }

        /// <summary>
        /// UVを指定の境界の中に納める
        /// </summary>
        /// <param name="bsquares"></param>
        public void ForceRoundingUvInBSquares(Dictionary<int, List<BoundingSquare>> bsquares)
        {
            if (bsquares == null)
            {
                throw ExcepHandle.CreateException("Bounding squares is null.");
            }
            if (_MaterialShapeInfoList.Count != bsquares.Count)
            {
                throw ExcepHandle.CreateException("Count is not equal to shape count.");
            }

            int sN = SourceN;
            List<MeshSourceBase> uvs = GetSources(SourceType.Texture);
            List<int> uvofs = new List<int>();
            Vector2 tmpuv = new Vector2();

            foreach (var uv in uvs)
            {
                int ofs = GetSourceOfset(uv.Type, uv.Name);
                uvofs.Add(ofs);
            }

            for (int fi = 0; fi < FaceN; fi++)
            {
                int fgid = _FaceMaterialShapeID[fi];
                List<BoundingSquare> boundings = null;
                if (!bsquares.ContainsKey(fgid))
                {
                    if (bsquares.Count <= fgid)
                    {
                        throw ExcepHandle.CreateException("Cannnot find bouding for texture.");
                    }
                    boundings = bsquares.Values.ElementAt(fgid);
                }
                else
                {
                    boundings = bsquares[fgid];
                }

                int vo = mVoffs[fi];
                for (int vi = 0; vi < 3; vi++)
                {
                    int uvi = 0;
                    foreach (var uo in uvofs)
                    {
                        int uvid = mVbuffs[(vo + vi) * sN + uo];
                        MeshSourceBase uv = uvs[uvi];
                        BoundingSquare bsquare = boundings[uvi];

                        MeshSourceDblCtrler.GetAsVec2(uv, uvid, ref tmpuv);
                        tmpuv[0] = Math.Max(bsquare.Min[0], tmpuv[0]);
                        tmpuv[1] = Math.Max(bsquare.Min[1], tmpuv[1]);
                        tmpuv[0] = Math.Min(bsquare.Max[0], tmpuv[0]);
                        tmpuv[1] = Math.Min(bsquare.Max[1], tmpuv[1]);
                        MeshSourceDblCtrler.SetAsVec2(uv, uvid, tmpuv.x, tmpuv.y);
                        uvi++;
                    }
                }
            }
        }

        /// <summary>
        /// 面積ゼロのメッシュを削除
        /// </summary>
        /// <returns></returns>
        public int CleanDegenerateFaces()
        {
            int result = 0;

            try
            {
                for (int f = 0; f < FaceN; f++)
                {
                    if (IsFaceDegenerate(f))
                    {
                        InvalidateFace(f);
                        result++;
                    }
                }
                return result;
            }
            catch
            {
                throw;
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="mesh"></param>
        /// <returns></returns>
        public bool EqualityTest(Mesh mesh)
        {
            if (FaceN != mesh.FaceN) { return false; }
            if (VertexN != mesh.VertexN) { return false; }
            if (SourceN != mesh.SourceN) { return false; }

            for (int i = 0; i < SourceN; i++)
            {
                if (!Sources[i].EqualityTest(mesh.Sources[i])) { return false; }
            }

            for (int fi = 0; fi < FaceN; fi++)
            {
                if (mVoffs[fi] != mesh.mVoffs[fi]) { return false; }
                if (mVnums[fi] != mesh.mVnums[fi]) { return false; }
            }
            for (int i = 0; i < mVbuffs.Count; i++)
            {
                if (mVbuffs[i] != mesh.mVbuffs[i]) { return false; }
            }
            return true;
        }

        /// <summary>
        /// メッシュ内の平均エッジ長を取得
        /// </summary>
        /// <returns></returns>
        public double ComputeAverageEdgeLength()
        {
            if (FaceN == 0)
            {
                return 0.0;
            }
            double result = 0.0;
            for (int fi = 0; fi < FaceN; fi++)
            {
                VectorN[] plist = new VectorN[mVnums[fi]];
                int dim = GetPropDimension();
                for (int i = 0; i < mVnums[fi]; i++)
                {
                    plist[i] = new VectorN((uint)dim);
                }

                GetFaceVertex(fi, ref plist);

                Vector3 pos0 = new Vector3(plist[0][0], plist[0][1], plist[0][2]);
                Vector3 pos1 = new Vector3(plist[1][0], plist[1][1], plist[1][2]);
                Vector3 pos2 = new Vector3(plist[2][0], plist[2][1], plist[2][2]);

                Vector3 e1 = pos1 - pos0;
                Vector3 e2 = pos2 - pos1;
                Vector3 e3 = pos0 - pos2;

                result += e1.Norm();
                result += e2.Norm();
                result += e3.Norm();
            }
            result /= ((double)FaceN * 3.0);
            return result;
        }

        /// <summary>
        /// 指定のサブメッシュのみを出力
        /// </summary>
        public Mesh GetSubMesh(int subMeshIndex, int connectedShapeId = -1, int materialShapeId = -1)
        {
            Mesh result = Clone();
            var smInfo = mSubMeshInfo[subMeshIndex];

            for (var fi = 0; fi < FaceN; fi++)
            {
                if (FaceSubMeshID[fi] != smInfo.subMeshindex)
                {
                    result.InvalidateFace(fi);
                }
                if (FaceMaterialShapeId[fi] != materialShapeId && materialShapeId != -1)
                {
                    result.InvalidateFace(fi);
                }
            }
            result.RemoveInvalidFaces();
            result.RemoveUnusedVertex();
            return result;
        }

        /// <summary>
        /// 指定のFaceGroup(=MaterialShape)に切り出したメッシュを取得
        /// </summary>
        /// <param name="fgname_part"></param>
        /// <returns></returns>
        public Mesh GetMaterialShapeMesh(string fgname_part)
        {
            Mesh result = Clone();

            var fginfo = MaterialShapeInformationList.Find(o => o.ShapeName.Contains(fgname_part));
            if (fginfo == null)
            { return null; }

            for (var fi = 0; fi < FaceN; fi++)
            {
                if (FaceMaterialShapeId[fi] != fginfo.Index)
                {
                    result.InvalidateFace(fi);
                }
            }
            result.RemoveInvalidFaces();
            result.RemoveUnusedVertex();
            return result;
        }

        /// <summary>
        /// 連結形状単位で、メッシュを切り出す
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public Mesh GetConnectedShapeMesh(int id)
        {
            Mesh result = Clone();
            var fginfo = ConnectedShapeInfomationList.ElementAt(id);
            if (fginfo == null)
            {
                return null;
            }

            for (var fi = 0; fi < FaceN; fi++)
            {
                if (FaceConnectedShapeID[fi] != fginfo.Index)
                {
                    result.InvalidateFace(fi);
                }
            }
            result.RemoveInvalidFaces();
            result.RemoveUnusedVertex();
            return result;
        }

        /// <summary>
        /// 連結Shape情報を格納する構造を確保する
        /// </summary>
        public void CreateFaceConnectedMeshInfo()
        {
            ConnectedShapeInfomationList = new ResizableList<ConnectedShapeInfo>();
            _FaceConnectedShapeID = new ResizableList<int>();
            _FaceConnectedShapeID.Resize(FaceN, 0);
        }

        /// <summary>
        /// 連結Shape毎に所属する面数のカウンタを取得する
        /// </summary>
        public void GetConnectedShapeFaceCounter()
        {
            int maxN = 0;
            foreach (var fci in FaceConnectedShapeID)
            {
                maxN = Math.Max(maxN, fci);
            }
            _ConnectedShapeFaceCounter = new ResizableList<int>();
            _ConnectedShapeFaceCounter.Resize(maxN + 1, 0);
            foreach (var fci in FaceConnectedShapeID)
            {
                _ConnectedShapeFaceCounter[fci]++;
            }
        }

        /// <summary>
        /// MaterialShapeを追加する
        /// </summary>
        /// <param name="index"></param>
        /// <param name="name"></param>
        public void AddConnectedShapeInfo(int index)
        {
            var newSI = new ConnectedShapeInfo { Index = index, MaterialShapeIndeces = new ResizableList<int>() };
            ConnectedShapeInfomationList.Add(newSI);
        }

        /// <summary>
        /// 既定のShapeIDを設定する
        /// </summary>
        public void CoveredMeshbyConnectedShape(int index, int materialShapeId)
        {
            ConnectedShapeInfomationList[index].MaterialShapeIndeces.Add(materialShapeId);
            for (int i = 0; i < FaceN; i++)
            {
                FaceConnectedShapeID[i] = ConnectedShapeInfomationList[index].Index;
            }
        }
    }
}
